// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief e+ e- > light hadrons
  class BESIII_2008_I779017 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2008_I779017);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(UnstableParticles(Cuts::abspid==411 ||
				Cuts::abspid==421 ||
				Cuts::abspid==431 ||
				Cuts::abspid==4122||
				Cuts::abspid==4232||
				Cuts::abspid==4132||
				Cuts::abspid==4332), "UFS");
      declare(FinalState(), "FS");

      // Book histograms
      book(_c[0], "/TMP/sigma");
      book(_c[1], "/TMP/energy");
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // get the axis, direction of incoming positron
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      _c[1]->fill((beams.first.momentum()+beams.second.momentum()).mass());
      const FinalState& fs = apply<FinalState>(event, "FS");
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      if(!ufs.particles().empty()) vetoEvent;
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
	nCount[p.pid()] += 1;
	++ntotal;
      }
      // mu+mu- + photons
      if(nCount[-13]==1 and nCount[13]==1 &&
	 ntotal==2+nCount[22])
	vetoEvent;
      else if(nCount[-11]==1 and nCount[11]==1 &&
	      ntotal==2+nCount[22])
	vetoEvent;
      // everything else
      else
	_c[0]->fill();
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      double fact = crossSection()/sumOfWeights()/nanobarn;
      double sig_h = _c[0]->val()*fact;
      double err_h = _c[0]->err()*fact;
      double energy = _c[1]->val()/ sumOfWeights();
      Scatter2D temphisto(refData(1, 1, 1));
      Scatter2DPtr cross;
      book(cross, 1, 1, 1);
      double deltaE=1e30;
      unsigned int ipoint=100000000;
      for (size_t b = 0; b < temphisto.numPoints(); b++) {
	double test = abs(temphisto.point(b).x()-energy);
	if(test<deltaE) {
	  deltaE=test;
	  ipoint=b;
	}
      }
      if(deltaE>0.01) ipoint=-1;
      for (size_t b = 0; b < temphisto.numPoints(); b++) {
	const double x  = temphisto.point(b).x();
	pair<double,double> ex = temphisto.point(b).xErrs();
	if (b!=ipoint)
	  cross  ->addPoint(x, 0., ex, make_pair(0.,.0));
	else
	  cross  ->addPoint(x, sig_h, ex, make_pair(err_h,err_h));
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    CounterPtr _c[2];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2008_I779017);

}
