module Msf
  ###
  #
  # This module exposes methods for querying a remote LDAP service
  #
  ###
  module Exploit::Remote::LDAP
    module ActiveDirectory
      include Msf::Exploit::Remote::LDAP
      include Msf::Exploit::Remote::LDAP::EntryCache
      include Msf::Exploit::Remote::LDAP::ActiveDirectory::SecurityDescriptorMatcher

      LDAP_CAP_ACTIVE_DIRECTORY_OID = '1.2.840.113556.1.4.800'.freeze
      LDAP_SERVER_SD_FLAGS_OID = '1.2.840.113556.1.4.801'.freeze
      OWNER_SECURITY_INFORMATION = 0x1
      GROUP_SECURITY_INFORMATION = 0x2
      DACL_SECURITY_INFORMATION = 0x4
      SACL_SECURITY_INFORMATION = 0x8

      # Query the remote server via the provided LDAP connection to determine if it's an Active Directory LDAP server.
      # More specifically, this ensures that it reports active directory capabilities and the whoami extension.
      #
      # @param Net::LDAP::Connection ldap_connection
      # @rtype Boolean
      def is_active_directory?(ldap)
        root_dse = ldap.search(
          ignore_server_caps: true,
          base: '',
          scope: Net::LDAP::SearchScope_BaseObject,
          attributes: %i[ supportedCapabilities supportedExtension ]
        )&.first

        return false unless root_dse[:supportedCapabilities].map(&:to_s).include?(LDAP_CAP_ACTIVE_DIRECTORY_OID)

        return false unless root_dse[:supportedExtension].include?(Net::LDAP::WhoamiOid)

        true
      end

      # Build a control blob that will fetch all security descriptor data but the SACL. This often enables reading a
      # security descriptor's DACL without the need for elevated permissions.
      #
      # @rtype String
      def adds_build_ldap_sd_control(owner: true, group: true, dacl: true, sacl: false)
        # Set the value of LDAP_SERVER_SD_FLAGS_OID flag so everything but
        # the SACL flag is set, as we need administrative privileges to retrieve
        # the SACL from the ntSecurityDescriptor attribute on Windows AD LDAP servers.
        #
        # Note that without specifying the LDAP_SERVER_SD_FLAGS_OID control in this manner,
        # the LDAP searchRequest will default to trying to grab all possible attributes of
        # the ntSecurityDescriptor attribute, hence resulting in an attempt to retrieve the
        # SACL even if the user is not an administrative user.
        #
        # Now one may think that we would just get the rest of the data without the SACL field,
        # however in reality LDAP will cause that attribute to just be blanked out if a part of it
        # cannot be retrieved, so we just will get nothing for the ntSecurityDescriptor attribute
        # in these cases if the user doesn't have permissions to read the SACL.
        flags = 0
        flags |= OWNER_SECURITY_INFORMATION if owner
        flags |= GROUP_SECURITY_INFORMATION if group
        flags |= DACL_SECURITY_INFORMATION if dacl
        flags |= SACL_SECURITY_INFORMATION if sacl
        control_values = [flags].map(&:to_ber).to_ber_sequence.to_s.to_ber
        [LDAP_SERVER_SD_FLAGS_OID.to_ber, true.to_ber, control_values].to_ber_sequence
      end

      # Query LDAP and obtain all members of a particular group. In this context, "members" are either users or groups.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @param [String] group_dn The DN of the group to obtain members for.
      # @param [String] base_dn An optional base search DN.
      # @param [Boolean] inherited Whether or not to include entities that are members by inheritance.
      # @param [String] object_class An optional object class for filtering. This is typically either 'user' or 'group'.
      def adds_query_group_members(ldap, group_dn, base_dn: nil, inherited: true, object_class: nil)
        return enum_for(:adds_query_group_members, ldap, group_dn, base_dn: base_dn, inherited: inherited, object_class: object_class) unless block_given?
        results = 0

        member_filter = "memberOf#{inherited ? ':1.2.840.113556.1.4.1941:' : ''}=#{ldap_escape_filter(group_dn)}"

        # Get the member's primaryGroupID
        group = adds_get_object_by_dn(ldap, group_dn)
        if group && group[:objectSID]
          group_sid = Rex::Proto::MsDtyp::MsDtypSid.read(group[:objectSID].first)
          # if we have a group RID, filter on that when the object has it as it's primaryGroupId to include those groups too
          member_filter = "|(#{member_filter})(primaryGroupId=#{group_sid.rid})"
        end

        filters = []
        filters << "objectClass=#{ldap_escape_filter(object_class)}" if object_class
        filters << member_filter

        ldap.search(
          base: base_dn || ldap.base_dn,
          controls: [adds_build_ldap_sd_control],
          filter: "(&#{filters.map { "(#{_1})" }.join})",
          return_result: false # make sure we're streaming because this could be a lot of data
        ) do |ldap_entry|
          yield ldap_entry
          results += 1
        end

        unless ldap.get_operation_result.code == 0
          raise "LDAP Error: #{ldap.get_operation_result.message}"
        end

        results
      end

      # Query LDAP and obtain all groups a particular entity is a member of. In this context, "members" are either users or groups.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @param [String] member_dn The DN of the member to obtain groups for.
      # @param [String] base_dn An optional base search DN.
      # @param [Boolean] inherited Whether or not to include groups that are inherited.
      def adds_query_member_groups(ldap, member_dn, base_dn: nil, inherited: true)
        return enum_for(:adds_query_member_groups, ldap, member_dn, base_dn: base_dn, inherited: inherited) unless block_given?
        results = 0

        # Get the member's primaryGroupId
        member = adds_get_object_by_dn(ldap, member_dn)
        if member && member[:objectSid] && member[:primaryGroupId] && !member[:primaryGroupId].empty?
          # if it's found, calculate the SID of the primary group and query it, the primary group is typically 'Domain Users'
          # and is *not* included in the member query
          member_sid = Rex::Proto::MsDtyp::MsDtypSid.read(member[:objectSid].first)
          primary_group_sid = "#{member_sid.to_s.rpartition('-').first}-#{member[:primaryGroupId].first}"
          primary_group = adds_get_object_by_sid(ldap, primary_group_sid)
          yield primary_group if primary_group
        end

        filters = []
        filters << "objectClass=group"
        filters << "member#{inherited ? ':1.2.840.113556.1.4.1941:' : ''}=#{ldap_escape_filter(member_dn)}"

        ldap.search(
          base: base_dn || ldap.base_dn,
          controls: [adds_build_ldap_sd_control],
          filter: "(&#{filters.map { "(#{_1})" }.join})",
          return_result: false
        ) do |ldap_entry|
          yield ldap_entry
          results += 1
        end

        unless ldap.get_operation_result.code == 0
          raise "LDAP Error: #{ldap.get_operation_result.message}"
        end

        results
      end

      # Obtain a particular entity by its distinguished name (DN).
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @param [String] object_dn The full distinguished name of the object to retrieve.
      # @return Returns nil when the object was not found.
      # @rtype [Net::LDAP::Entry,nil]
      def adds_get_object_by_dn(ldap, object_dn)
        object = ldap_entry_cache.get_by_dn(object_dn)
        return nil if ldap_entry_cache.missing_entry?(object)
        return object if object

        object = ldap.search(base: object_dn, controls: [adds_build_ldap_sd_control], scope: Net::LDAP::SearchScope_BaseObject)&.first
        validate_query_result!(ldap.get_operation_result.table)

        if object
          ldap_entry_cache << object
        else
          ldap_entry_cache.mark_missing_by_dn(object_dn)
        end
        object
      end

      # Obtain a particular entity by its sAMAccountName.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @param [String] object_samaccountname The sAMAccountName of the object to retrieve.
      # @return Returns nil when the object was not found.
      # @rtype [Net::LDAP::Entry,nil]
      def adds_get_object_by_samaccountname(ldap, object_samaccountname)
        object = ldap_entry_cache.get_by_samaccountname(object_samaccountname)
        return nil if ldap_entry_cache.missing_entry?(object)
        return object if object

        filter = "(sAMAccountName=#{ldap_escape_filter(object_samaccountname)})"
        begin
          object = ldap.search(base: ldap.base_dn, controls: [adds_build_ldap_sd_control], filter: filter)&.first
        rescue Net::LDAP::Error => e
          elog('ldap search error for sAMAccountName', error: e)
          return nil
        end
        validate_query_result!(ldap.get_operation_result.table, filter)

        if object
          ldap_entry_cache << object
        else
          ldap_entry_cache.mark_missing_by_samaccountname(object_samaccountname)
        end
        object
      end

      # Obtain a particular entity by its SID.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @param [String] object_sid The SID of the object to retrieve.
      # @return Returns nil when the object was not found.
      # @rtype [Net::LDAP::Entry,nil]
      def adds_get_object_by_sid(ldap, object_sid)
        object_sid = Rex::Proto::MsDtyp::MsDtypSid.new(object_sid)
        object = ldap_entry_cache.get_by_sid(object_sid)
        return nil if ldap_entry_cache.missing_entry?(object)
        return object if object

        filter = "(objectSID=#{ldap_escape_filter(object_sid.to_s)})"
        object = ldap.search(base: ldap.base_dn, controls: [adds_build_ldap_sd_control], filter: filter)&.first
        validate_query_result!(ldap.get_operation_result.table, filter)

        if object
          ldap_entry_cache << object
        else
          ldap_entry_cache.mark_missing_by_sid(object_sid)
        end
        object
      end

      # Get the LDAP object that describes the current user.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @rtype [Net::LDAP::Entry]
      def adds_get_current_user(ldap)
        whoami = @ldap_whoami = (@ldap_whoami || ldap.ldapwhoami.to_s)
        our_domain, _, our_username = whoami.delete_prefix('u:').partition('\\')
        # todo: this is probably going to have issues if our user is from a domain that the target server is not the
        # authority of
        adds_get_object_by_samaccountname(ldap, our_username)
      end

      # Get the AD DS domain info for the current server.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @rtype [Hash]
      def adds_get_domain_info(ldap)
        domain_object = ldap.search(base: ldap.base_dn, filter: '(objectClass=domain)', return_result: true)&.first
        return nil unless domain_object

        ldap_entry_cache << domain_object
        domain_sid = Rex::Proto::MsDtyp::MsDtypSid.read(domain_object[:objectSid].first)
        domain_behavior_version = domain_object['msds-behavior-version'].first

        root_dse = ldap.search(
          base: '',
          scope: Net::LDAP::SearchScope_BaseObject,
          attributes: %i[configurationNamingContext]
        )&.first
        return nil unless root_dse

        xrefs = ldap.search(
          base: root_dse[:configurationNamingContext].first,
          filter: "(&(objectCategory=crossref)(nETBIOSName=*)(nCName=#{ldap.base_dn}))"
        )
        return nil unless xrefs&.length == 1

        xref = xrefs.first
        ldap_entry_cache << xref

        {
          domain_behavior_version: domain_behavior_version.to_i,
          netbios_name: xref[:nETBIOSName].first.to_s,
          dns_name: xref[:dNSRoot].first.to_s,
          sid: domain_sid
        }
      end

      # Query the LDAP server to retrieve all Certificate Authority (Enterprise CA) servers in the domain.
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @return [Array<Hash>] An array of hashes, where each hash contains the `:name` and `:dNSHostName` of a CA server.
      # @rtype [Array<Hash>]
      def adds_get_ca_servers(ldap)
        base_dn = "CN=Enrollment Services,CN=Public Key Services,CN=Services,CN=Configuration,#{@base_dn}"
        filter = '(objectClass=pKIEnrollmentService)'
        attributes = ['cn', 'dNSHostName']
        ca_servers = []

        ldap.search(base: base_dn, filter: filter, attributes: attributes) do |entry|
          ca_servers << {
            name: entry[:cn]&.first,
            dNSHostName: entry[:dNSHostName]&.first
          }
        end

        ca_servers
      end

      # Determine if a security descriptor will grant the permissions identified by *matcher* to the
      # *test_sid*.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @param [Rex::Proto::MsDtyp::MsDtypSecurityDescriptor] security_descriptor The security descriptor object to
      #   evaluate.
      # @param [SecurityDescriptorMatcher::Base] matcher An object that will match ACEs that allow or deny the desired permissions.
      # @param [Rex::Proto::MsDtyp::MsDtypSid] test_sid The SID to check for access.
      # @param [Rex::Proto::MsDtyp::MsDtypSid] self_sid The SID of the object who owns the security_descriptor. This is
      #   typically the objectSid LDAP attribute and is used when the security descriptor references the special 'SELF'
      #   entity.
      def adds_sd_grants_permissions?(ldap, security_descriptor, matcher, test_sid: nil, self_sid: nil)
        unless test_sid
          current_user = adds_get_current_user(ldap)
          raise RuntimeError.new('No SID was specified and the current user could not be identified.') unless current_user

          test_sid = Rex::Proto::MsDtyp::MsDtypSid.read(current_user[:objectSid].first)
        end

        test_member_sids = nil

        dacl_aces = []
        # because deny entries take precedence, process them first
        dacl_aces += security_descriptor.dacl.aces.select { |ace| Rex::Proto::MsDtyp::MsDtypAceType.deny?(ace.header.ace_type) }
        dacl_aces += security_descriptor.dacl.aces.select { |ace| Rex::Proto::MsDtyp::MsDtypAceType.allow?(ace.header.ace_type) }

        dacl_aces.each do |ace|
          # Uncomment this if you need to debug ACE evaluation
          # ldap_object = adds_get_object_by_sid(ldap, ace.body.sid)
          # $stderr.puts  "ACE:"
          # $stderr.puts  "  Type:        #{Rex::Proto::MsDtyp::MsDtypAceType.name(ace.header.ace_type)}"
          # $stderr.puts  "  Permissions: #{ace.body.access_mask.permissions.map(&:to_s).join(', ')}"
          # $stderr.write "  SID:         #{ace.body.sid}"
          # $stderr.puts (ldap_object && ldap_object[:sAMAccountName].first) ? " (#{ldap_object[:sAMAccountName].first})" : ""
          # $stderr.puts "  Object:      #{ace.body.object_type}" if Rex::Proto::MsDtyp::MsDtypAceType.has_object?(ace.header.ace_type)

          next if matcher.ignore_ace?(ace)

          case ace.body.sid
          when Rex::Proto::Secauthz::WellKnownSids::SECURITY_WORLD_SID
            matcher.apply_ace!(ace)
          when Rex::Proto::Secauthz::WellKnownSids::SECURITY_AUTHENTICATED_USER_SID
            matcher.apply_ace!(ace)
          when Rex::Proto::Secauthz::WellKnownSids::SECURITY_PRINCIPAL_SELF_SID
            matcher.apply_ace!(ace) if self_sid == test_sid
          when Rex::Proto::Secauthz::WellKnownSids::SECURITY_CREATOR_OWNER_SID
            matcher.apply_ace!(ace) if security_descriptor.owner_sid == test_sid
          when Rex::Proto::Secauthz::WellKnownSids::SECURITY_CREATOR_GROUP_SID
            matcher.apply_ace!(ace) if security_descriptor.group_sid == test_sid
          when test_sid
            matcher.apply_ace!(ace)
          when Rex::Proto::Secauthz::WellKnownSids::SECURITY_LOCAL_SYSTEM_SID
            # the SECURITY_LOCAL_SYSTEM_SID won't be found if looked up in the next block and if it's not the SID we're checking for, it doesn't apply anyways so just skip it
            next
          else
            ldap_object = adds_get_object_by_sid(ldap, ace.body.sid)
            next unless ldap_object && ldap_object[:objectClass].include?('group')

            member_sids = adds_query_group_members(ldap, ldap_object.dn, inherited: false).map { |member| Rex::Proto::MsDtyp::MsDtypSid.read(member[:objectSid].first) }
            if member_sids.include?(test_sid)
              matcher.apply_ace!(ace)
              next
            end

            if test_member_sids.nil?
              test_obj = adds_get_object_by_sid(ldap, test_sid)
              test_member_sids = adds_query_member_groups(ldap, test_obj.dn, inherited: true).map { |member| Rex::Proto::MsDtyp::MsDtypSid.read(member[:objectSid].first) }.to_set
              if test_obj[:objectClass].include?('user') && test_sid.rid != Rex::Proto::Secauthz::WellKnownSids::DOMAIN_USER_RID_GUEST
                test_member_sids << Rex::Proto::Secauthz::WellKnownSids::SECURITY_AUTHENTICATED_USER_SID
                test_member_sids << Rex::Proto::Secauthz::WellKnownSids::DOMAIN_ALIAS_SID_USERS
              end
            end

            matcher.apply_ace!(ace) if member_sids.any? { |member_sid| test_member_sids.include?(member_sid) }
          end

          break if matcher.satisfied?
        end

        matcher.matches?
      end

      # Determine if a security descriptor will grant the permissions identified by *matcher* to the
      # *test_sid*. For this to work, the authenticated user typically needs "Read permissions", and "Read general
      # information" from the advanced "Permission Entry" form in Active Directory. The more generic, "Read properties"
      # permission will also do the trick.
      #
      # @param [Net::LDAP::Connection] ldap The LDAP connection to use for querying.
      # @param [Net::LDAP::Entry] obj The LDAP object to test. The security descriptor will be taken from the
      #   nTSecurityDescriptor attribute.
      # @param [#call] matcher An object that will match ACEs that allow or deny the desired permissions.
      # @param [Rex::Proto::MsDtyp::MsDtypSid] test_sid The SID to check for access.
      def adds_obj_grants_permissions?(ldap, obj, matcher, test_sid: nil)
        unless obj[:nTSecurityDescriptor].first
          raise RuntimeError.new('The nTSecurityDescriptor can not be read from the object.')
        end

        security_descriptor = Rex::Proto::MsDtyp::MsDtypSecurityDescriptor.read(obj[:nTSecurityDescriptor].first)
        self_sid = nil
        if obj[:objectSid]&.first
          self_sid = Rex::Proto::MsDtyp::MsDtypSid.read(obj[:objectSid].first)
        end

        adds_sd_grants_permissions?(ldap, security_descriptor, matcher, test_sid: test_sid, self_sid: self_sid)
      end
    end
  end
end
