# frozen_string_literal: true

require 'spec_helper'
require 'net/ldap'
require 'msf/core/exploit/remote/ldap/entry_cache'

RSpec.describe Msf::Exploit::Remote::LDAP::EntryCache::LDAPEntryCache do
  subject(:cache) { described_class.new(max_size: 1000) }

  let(:dn) { 'cn=User One,dc=example,dc=com' }
  let(:samaccountname) { 'user1' }
  let(:sid) { 'S-1-5-21-111111111-222222222-333333333-4444' }

  describe '#get_by_dn' do
    it 'returns a missing sentinel when a DN miss is cached' do
      cache.mark_missing_by_dn(dn)
      expect(cache.get_by_dn(dn)).to eq described_class::MissingEntry
    end
  end

  describe '#get_by_samaccountname' do
    it 'returns a cached entry when present' do
      entry = Net::LDAP::Entry.new(dn)
      entry[:sAMAccountName] = [samaccountname]
      cache << entry

      expect(cache.get_by_samaccountname(samaccountname)).to eq entry
    end

    it 'returns a missing sentinel when a sAMAccountName miss is cached' do
      cache.mark_missing_by_samaccountname(samaccountname)
      expect(cache.get_by_samaccountname(samaccountname)).to eq described_class::MissingEntry
    end
  end

  describe '#get_by_sid' do
    it 'returns a cached entry when present' do
      entry = Net::LDAP::Entry.new(dn)
      entry[:objectSid] = [Rex::Proto::MsDtyp::MsDtypSid.new(sid).to_binary_s]
      cache << entry

      expect(cache.get_by_sid(sid)).to eq entry
    end

    it 'returns a missing sentinel when a SID miss is cached' do
      cache.mark_missing_by_sid(sid)
      expect(cache.get_by_sid(sid)).to eq described_class::MissingEntry
    end
  end
end
