<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\Config\Tests\Loader;

use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Config\Exception\FileLoaderImportCircularReferenceException;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\Config\FileLocatorInterface;
use Symfony\Component\Config\Loader\FileLoader;
use Symfony\Component\Config\Loader\LoaderResolver;

class FileLoaderTest extends TestCase
{
    public function testImportWithFileLocatorDelegation()
    {
        $locatorMockForAdditionalLoader = $this->createStub(FileLocatorInterface::class);
        $locatorMockForAdditionalLoader
            ->method('locate')
            ->willReturn(
                ['path/to/file1'],
                ['path/to/file1', 'path/to/file2'],
                ['path/to/file1', 'path/to/file2'],
                ['path/to/file1'],
                ['path/to/file1', 'path/to/file2']
            );

        $fileLoader = new TestFileLoader(new FileLocator());
        $fileLoader->setSupports(false);
        $fileLoader->setCurrentDir('.');

        $additionalLoader = new TestFileLoader($locatorMockForAdditionalLoader);
        $additionalLoader->setCurrentDir('.');

        $fileLoader->setResolver($loaderResolver = new LoaderResolver([$fileLoader, $additionalLoader]));

        // Default case
        $this->assertSame('path/to/file1', $fileLoader->import('my_resource'));

        // Check first file is imported if not already loading
        $this->assertSame('path/to/file1', $fileLoader->import('my_resource'));

        // Check second file is imported if first is already loading
        $fileLoader->addLoading('path/to/file1');
        $this->assertSame('path/to/file2', $fileLoader->import('my_resource'));

        // Check exception throws if first (and only available) file is already loading
        try {
            $fileLoader->import('my_resource');
            $this->fail('->import() throws a FileLoaderImportCircularReferenceException if the resource is already loading');
        } catch (\Exception $e) {
            $this->assertInstanceOf(FileLoaderImportCircularReferenceException::class, $e, '->import() throws a FileLoaderImportCircularReferenceException if the resource is already loading');
        }

        // Check exception throws if all files are already loading
        try {
            $fileLoader->addLoading('path/to/file2');
            $fileLoader->import('my_resource');
            $this->fail('->import() throws a FileLoaderImportCircularReferenceException if the resource is already loading');
        } catch (\Exception $e) {
            $this->assertInstanceOf(FileLoaderImportCircularReferenceException::class, $e, '->import() throws a FileLoaderImportCircularReferenceException if the resource is already loading');
        }
    }

    public function testImportWithGlobLikeResource()
    {
        $locatorMock = $this->createMock(FileLocatorInterface::class);
        $locatorMock->expects($this->once())->method('locate')->willReturn('');
        $loader = new TestFileLoader($locatorMock);

        $this->assertSame('[foo]', $loader->import('[foo]'));
    }

    public function testImportWithGlobLikeResourceWhichContainsSlashes()
    {
        $locatorMock = $this->createMock(FileLocatorInterface::class);
        $locatorMock->expects($this->once())->method('locate')->willReturn('');
        $loader = new TestFileLoader($locatorMock);

        $this->assertNull($loader->import('foo/bar[foo]'));
    }

    public function testImportWithGlobLikeResourceWhichContainsMultipleLines()
    {
        $loader = new TestFileLoader(new FileLocator());

        $this->assertSame("foo\nfoobar[foo]", $loader->import("foo\nfoobar[foo]"));
    }

    public function testImportWithGlobLikeResourceWhichContainsSlashesAndMultipleLines()
    {
        $loader = new TestFileLoader(new FileLocator());

        $this->assertSame("foo\nfoo/bar[foo]", $loader->import("foo\nfoo/bar[foo]"));
    }

    public function testImportWithNoGlobMatch()
    {
        $locatorMock = $this->createMock(FileLocatorInterface::class);
        $locatorMock->expects($this->once())->method('locate')->willReturn('');
        $loader = new TestFileLoader($locatorMock);

        $this->assertNull($loader->import('./*.abc'));
    }

    public function testImportWithSimpleGlob()
    {
        $loader = new TestFileLoader(new FileLocator(__DIR__));

        $this->assertSame(__FILE__, strtr($loader->import('FileLoaderTest.*'), '/', \DIRECTORY_SEPARATOR));
    }

    #[DataProvider('importWithExcludeProvider')]
    public function testImportWithExclude(string $include, string $exclude, int $expectedCount)
    {
        $loader = new TestFileLoader(new FileLocator(__DIR__.'/../Fixtures'));
        $loadedFiles = $loader->import($include, null, false, null, $exclude);
        $this->assertCount($expectedCount, $loadedFiles);
        $this->assertNotContains('ExcludeFile.txt', $loadedFiles);
    }

    #[DataProvider('excludeTrailingSlashConsistencyProvider')]
    public function testExcludeTrailingSlashConsistency(string $exclude)
    {
        $loader = new TestFileLoader(new FileLocator(__DIR__.'/../Fixtures'));
        $loadedFiles = $loader->import('ExcludeTrailingSlash/*', null, false, null, $exclude);
        $this->assertCount(2, $loadedFiles);
        $this->assertNotContains('baz.txt', $loadedFiles);
    }

    public static function importWithExcludeProvider(): iterable
    {
        yield ['Include/*', __DIR__.'/../Fixtures/Include/{ExcludeFile.txt}', 2];
        yield ['Include/', __DIR__.'/../Fixtures/Include/{ExcludeFile.txt}', 4];
        yield ['Include', __DIR__.'/../Fixtures/Include/{ExcludeFile.txt}', 4];
        yield ['Include/**/*', __DIR__.'/../Fixtures/Include/{ExcludeFile.txt}', 4];
        yield ['Include/*', __DIR__.'/../Fixtures/Include/{Exclude*.txt}', 2];
        yield ['Include/', __DIR__.'/../Fixtures/Include/{Exclude*.txt}', 4];
        yield ['Include', __DIR__.'/../Fixtures/Include/{Exclude*.txt}', 4];
        yield ['Include/**/*', __DIR__.'/../Fixtures/Include/{Exclude*.txt}', 4];
        yield ['Include/', __DIR__.'/../Fixtures/Include/**/{ExcludeFile.txt}', 3];
        yield ['Include', __DIR__.'/../Fixtures/Include/**/{ExcludeFile.txt}', 3];
        yield ['Include/**/*', __DIR__.'/../Fixtures/Include/**/{ExcludeFile.txt}', 3];
        yield ['Include/', __DIR__.'/../Fixtures/Include/**/{Exclude*.txt}', 3];
        yield ['Include', __DIR__.'/../Fixtures/Include/**/{Exclude*.txt}', 3];
        yield ['Include/**/*', __DIR__.'/../Fixtures/Include/**/{Exclude*.txt}', 3];
    }

    public static function excludeTrailingSlashConsistencyProvider(): iterable
    {
        yield [__DIR__.'/../Fixtures/Exclude/ExcludeToo/'];
        yield [__DIR__.'/../Fixtures/Exclude/ExcludeToo'];
        yield [__DIR__.'/../Fixtures/Exclude/ExcludeToo/*'];
        yield [__DIR__.'/../Fixtures/*/ExcludeToo'];
        yield [__DIR__.'/../Fixtures/*/ExcludeToo/'];
        yield [__DIR__.'/../Fixtures/Exclude/ExcludeToo/*'];
        yield [__DIR__.'/../Fixtures/Exclude/ExcludeToo/AnotheExcludedFile.txt'];
    }
}

class TestFileLoader extends FileLoader
{
    private bool $supports = true;

    public function load(mixed $resource, ?string $type = null): mixed
    {
        return $resource;
    }

    public function supports(mixed $resource, ?string $type = null): bool
    {
        return $this->supports;
    }

    public function addLoading(string $resource): void
    {
        self::$loading[$resource] = true;
    }

    public function setSupports(bool $supports): void
    {
        $this->supports = $supports;
    }
}
