#!/usr/bin/env python
#
# This file is part of pacman-mirrors.
#
# pacman-mirrors is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# pacman-mirrors is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with pacman-mirrors.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Esclapion <esclapion@manjaro.org>
#          philm <philm@manjaro.org>
#          Ramon Buldó <rbuldo@gmail.com>
#          Hugo Posnic <huluti@manjaro.org>
#          Frede Hundewadt <echo ZmhAbWFuamFyby5vcmcK | base64 -d>

"""Pacman-Mirrors Main Module"""

import importlib.util
import sys
import logging

from pacman_mirrors.functions import util
from pacman_mirrors.builder import common, fasttrack, interactive
from pacman_mirrors.config import configuration as conf
from pacman_mirrors.functions import cliFn
from pacman_mirrors.functions import config_setup
from pacman_mirrors.functions import defaultFn
from pacman_mirrors.functions import fileFn
from pacman_mirrors.functions import httpFn
from pacman_mirrors.functions import outputFn
from pacman_mirrors.mirrors.mirror import Mirror
from pacman_mirrors.translation import i18n

try:
    importlib.util.find_spec("gi.repository.Gtk")
except ImportError:
    GTK_AVAILABLE = False
else:
    GTK_AVAILABLE = True
_ = i18n.language.gettext


class PacmanMirrors:
    def __init__(self):
        self.config = {"config_file": conf.CONFIG_FILE}
        self.continent = False
        self.custom = False
        self.default = False
        self.fasttrack = None
        self.tz = False
        self.interactive = False
        self.max_wait_time = 2
        self.mirrors = Mirror()
        self.network = True
        self.no_display = False
        self.no_mirrorList = False
        self.no_status = False
        self.quiet = False
        self.selected_countries = []
        self.tty = False
        self.arm = False

    def run(self):
        """
        # Setup config: returns the config dictionary and true/false on custom
        # Check working folders
        # Check for static config
        # Parse commandline
        # Check network
        # aarch64 check
        # sanitize config
        # Update mirror pool
        # Check if mirrorList is not to be touched - normal exit
        # Handle missing network
        # Load default mirror pool
        # Build mirror list
        """
        # load configuration
        (self.config, self.custom) = config_setup.setup_config(self)

        # ensure workdir exist
        fileFn.create_dir(self.config["work_dir"])

        # check for static configuration
        if self.config["static"]:
            # static setup
            self.custom = False
            if fileFn.check_file(self.config["custom_file"]):
                fileFn.delete_file(self.config["custom_file"])

            # write enterprise mirror list
            util.msg("Static configuration found")
            mirrors = util.build_static_from_config(self, self.config["static"])
            outputFn.write_static_mirror_pool(self, mirrors)
            outputFn.write_pacman_mirror_list(self, mirrors)
            exit(0)

        ##########################################################
        # default manjaro setup
        cliFn.parse_command_line(self, gtk_available=GTK_AVAILABLE)

        # check network connection
        self.network = httpFn.check_internet_connection(tty=self.tty)

        # check for arm architecture
        util.aarch64_check(self, write=True)

        # sanitize the configuration
        if not config_setup.sanitize_config(config=self.config):
            sys.exit(2)

        # download mirror pool if internet connection is available
        if self.network:
            httpFn.download_mirror_pool(
                config=self.config, tty=self.tty, quiet=self.quiet
            )

        # conditional exit
        # used in pacman hook to avoid changing mirror pool and mirror list on updates
        if self.no_mirrorList:
            sys.exit(0)

        # conditional reset CLI configuration
        # if internet connection is down
        if not self.network:
            if not self.quiet:
                util.internet_message(tty=self.tty)
            self.config["method"] = "random"
            self.fasttrack = False

        # Load a configured mirror pool
        defaultFn.load_config_mirror_pool(self)

        # Decide which type of mirrorlist to create Fasttrack/Interactive/Default
        if self.fasttrack:
            fasttrack.build_mirror_list(self, limit=self.fasttrack)
        elif self.interactive:
            interactive.build_mirror_list(self)
        else:
            common.build_mirror_list(self)


def start():
    app = PacmanMirrors()
    app.run()


if __name__ == "__main__":
    start()
