﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/ImageLoggingConfiguration.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class ImportDiskImageRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API ImportDiskImageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ImportDiskImage"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the image resource that's created from the import.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ImportDiskImageRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The semantic version to attach to the image that's created during the import
   * process. This version follows the semantic version syntax.</p>
   */
  inline const Aws::String& GetSemanticVersion() const { return m_semanticVersion; }
  inline bool SemanticVersionHasBeenSet() const { return m_semanticVersionHasBeenSet; }
  template <typename SemanticVersionT = Aws::String>
  void SetSemanticVersion(SemanticVersionT&& value) {
    m_semanticVersionHasBeenSet = true;
    m_semanticVersion = std::forward<SemanticVersionT>(value);
  }
  template <typename SemanticVersionT = Aws::String>
  ImportDiskImageRequest& WithSemanticVersion(SemanticVersionT&& value) {
    SetSemanticVersion(std::forward<SemanticVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description for your disk image import.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  ImportDiskImageRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The operating system platform for the imported image. Allowed values include
   * the following: <code>Windows</code>.</p>
   */
  inline const Aws::String& GetPlatform() const { return m_platform; }
  inline bool PlatformHasBeenSet() const { return m_platformHasBeenSet; }
  template <typename PlatformT = Aws::String>
  void SetPlatform(PlatformT&& value) {
    m_platformHasBeenSet = true;
    m_platform = std::forward<PlatformT>(value);
  }
  template <typename PlatformT = Aws::String>
  ImportDiskImageRequest& WithPlatform(PlatformT&& value) {
    SetPlatform(std::forward<PlatformT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The operating system version for the imported image. Allowed values include
   * the following: <code>Microsoft Windows 11</code>.</p>
   */
  inline const Aws::String& GetOsVersion() const { return m_osVersion; }
  inline bool OsVersionHasBeenSet() const { return m_osVersionHasBeenSet; }
  template <typename OsVersionT = Aws::String>
  void SetOsVersion(OsVersionT&& value) {
    m_osVersionHasBeenSet = true;
    m_osVersion = std::forward<OsVersionT>(value);
  }
  template <typename OsVersionT = Aws::String>
  ImportDiskImageRequest& WithOsVersion(OsVersionT&& value) {
    SetOsVersion(std::forward<OsVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or Amazon Resource Name (ARN) for the IAM role you create that
   * grants Image Builder access to perform workflow actions to import an image from
   * a Microsoft ISO file.</p>
   */
  inline const Aws::String& GetExecutionRole() const { return m_executionRole; }
  inline bool ExecutionRoleHasBeenSet() const { return m_executionRoleHasBeenSet; }
  template <typename ExecutionRoleT = Aws::String>
  void SetExecutionRole(ExecutionRoleT&& value) {
    m_executionRoleHasBeenSet = true;
    m_executionRole = std::forward<ExecutionRoleT>(value);
  }
  template <typename ExecutionRoleT = Aws::String>
  ImportDiskImageRequest& WithExecutionRole(ExecutionRoleT&& value) {
    SetExecutionRole(std::forward<ExecutionRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the infrastructure configuration resource
   * that's used for launching the EC2 instance on which the ISO image is built.</p>
   */
  inline const Aws::String& GetInfrastructureConfigurationArn() const { return m_infrastructureConfigurationArn; }
  inline bool InfrastructureConfigurationArnHasBeenSet() const { return m_infrastructureConfigurationArnHasBeenSet; }
  template <typename InfrastructureConfigurationArnT = Aws::String>
  void SetInfrastructureConfigurationArn(InfrastructureConfigurationArnT&& value) {
    m_infrastructureConfigurationArnHasBeenSet = true;
    m_infrastructureConfigurationArn = std::forward<InfrastructureConfigurationArnT>(value);
  }
  template <typename InfrastructureConfigurationArnT = Aws::String>
  ImportDiskImageRequest& WithInfrastructureConfigurationArn(InfrastructureConfigurationArnT&& value) {
    SetInfrastructureConfigurationArn(std::forward<InfrastructureConfigurationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>uri</code> of the ISO disk file that's stored in Amazon S3.</p>
   */
  inline const Aws::String& GetUri() const { return m_uri; }
  inline bool UriHasBeenSet() const { return m_uriHasBeenSet; }
  template <typename UriT = Aws::String>
  void SetUri(UriT&& value) {
    m_uriHasBeenSet = true;
    m_uri = std::forward<UriT>(value);
  }
  template <typename UriT = Aws::String>
  ImportDiskImageRequest& WithUri(UriT&& value) {
    SetUri(std::forward<UriT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Define logging configuration for the image build process.</p>
   */
  inline const ImageLoggingConfiguration& GetLoggingConfiguration() const { return m_loggingConfiguration; }
  inline bool LoggingConfigurationHasBeenSet() const { return m_loggingConfigurationHasBeenSet; }
  template <typename LoggingConfigurationT = ImageLoggingConfiguration>
  void SetLoggingConfiguration(LoggingConfigurationT&& value) {
    m_loggingConfigurationHasBeenSet = true;
    m_loggingConfiguration = std::forward<LoggingConfigurationT>(value);
  }
  template <typename LoggingConfigurationT = ImageLoggingConfiguration>
  ImportDiskImageRequest& WithLoggingConfiguration(LoggingConfigurationT&& value) {
    SetLoggingConfiguration(std::forward<LoggingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags that are attached to image resources created from the import.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  ImportDiskImageRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  ImportDiskImageRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  ImportDiskImageRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_semanticVersion;

  Aws::String m_description;

  Aws::String m_platform;

  Aws::String m_osVersion;

  Aws::String m_executionRole;

  Aws::String m_infrastructureConfigurationArn;

  Aws::String m_uri;

  ImageLoggingConfiguration m_loggingConfiguration;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_nameHasBeenSet = false;
  bool m_semanticVersionHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_platformHasBeenSet = false;
  bool m_osVersionHasBeenSet = false;
  bool m_executionRoleHasBeenSet = false;
  bool m_infrastructureConfigurationArnHasBeenSet = false;
  bool m_uriHasBeenSet = false;
  bool m_loggingConfigurationHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
