﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisanalyticsv2/KinesisAnalyticsV2Request.h>
#include <aws/kinesisanalyticsv2/KinesisAnalyticsV2_EXPORTS.h>
#include <aws/kinesisanalyticsv2/model/Input.h>

#include <utility>

namespace Aws {
namespace KinesisAnalyticsV2 {
namespace Model {

/**
 */
class AddApplicationInputRequest : public KinesisAnalyticsV2Request {
 public:
  AWS_KINESISANALYTICSV2_API AddApplicationInputRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddApplicationInput"; }

  AWS_KINESISANALYTICSV2_API Aws::String SerializePayload() const override;

  AWS_KINESISANALYTICSV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of your existing application to which you want to add the streaming
   * source.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  AddApplicationInputRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current version of your application. You must provide the
   * <code>ApplicationVersionID</code> or the <code>ConditionalToken</code>.You can
   * use the <a>DescribeApplication</a> operation to find the current application
   * version.</p>
   */
  inline long long GetCurrentApplicationVersionId() const { return m_currentApplicationVersionId; }
  inline bool CurrentApplicationVersionIdHasBeenSet() const { return m_currentApplicationVersionIdHasBeenSet; }
  inline void SetCurrentApplicationVersionId(long long value) {
    m_currentApplicationVersionIdHasBeenSet = true;
    m_currentApplicationVersionId = value;
  }
  inline AddApplicationInputRequest& WithCurrentApplicationVersionId(long long value) {
    SetCurrentApplicationVersionId(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <a>Input</a> to add.</p>
   */
  inline const Input& GetInput() const { return m_input; }
  inline bool InputHasBeenSet() const { return m_inputHasBeenSet; }
  template <typename InputT = Input>
  void SetInput(InputT&& value) {
    m_inputHasBeenSet = true;
    m_input = std::forward<InputT>(value);
  }
  template <typename InputT = Input>
  AddApplicationInputRequest& WithInput(InputT&& value) {
    SetInput(std::forward<InputT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;
  bool m_applicationNameHasBeenSet = false;

  long long m_currentApplicationVersionId{0};
  bool m_currentApplicationVersionIdHasBeenSet = false;

  Input m_input;
  bool m_inputHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisAnalyticsV2
}  // namespace Aws
