/*=========================================================================
*
*  Copyright NumFOCUS
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*         http://www.apache.org/licenses/LICENSE-2.0.txt
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*=========================================================================*/
#ifndef sitkFastMarchingImageFilter_h
#define sitkFastMarchingImageFilter_h

/*
 * WARNING: DO NOT EDIT THIS FILE!
 * THIS FILE IS AUTOMATICALLY GENERATED BY THE SIMPLEITK BUILD PROCESS.
 * Please look at sitkImageFilterTemplate.h.in to make changes.
 */

#include <memory>

#include "sitkBasicFilters.h"
#include "sitkImageFilter.h"

namespace itk::simple {

    /**\class FastMarchingImageFilter
\brief Solve an Eikonal equation using Fast Marching.

Fast marching solves an Eikonal equation where the speed is always non-negative and depends on the position only. Starting from an initial position on the front, fast marching systematically moves the front forward one grid point at a time.

Updates are performed using an entropy satisfy scheme where only "upwind" neighborhoods are used. This implementation of Fast Marching uses a std::priority_queue to locate the next proper grid position to update.

Fast Marching sweeps through N grid points in (N log N) steps to obtain the arrival time value as the front propagates through the grid.

Implementation of this class is based on Chapter 8 of "Level Set Methods and Fast Marching Methods", J.A. Sethian, Cambridge Press, Second edition, 1999.

This class is templated over the level set image type and the speed image type. The initial front is specified by two containers: one containing the known points and one containing the trial points. Alive points are those that are already part of the object, and trial points are considered for inclusion. In order for the filter to evolve, at least some trial points must be specified. These can for instance be specified as the layer of pixels around the alive points.

The speed function can be specified as a speed image or a speed constant. The speed image is set using the method SetInput() . If the speed image is nullptr, a constant speed function is used and is specified using method the SetSpeedConstant() .

If the speed function is constant and of value one, fast marching results in an approximate distance function from the initial alive points. FastMarchingImageFilter is used in the ReinitializeLevelSetImageFilter object to create a signed distance function from the zero level set.

The algorithm can be terminated early by setting an appropriate stopping value. The algorithm terminates when the current arrival time being processed is greater than the stopping value.

There are two ways to specify the output image information ( LargestPossibleRegion, Spacing, Origin): (a) it is copied directly from the input speed image or (b) it is specified by the user. Default values are used if the user does not specify all the information.

The output information is computed as follows. If the speed image is nullptr or if the OverrideOutputInformation is set to true, the output information is set from user specified parameters. These parameters can be specified using methods SetOutputRegion() , SetOutputSpacing() , SetOutputDirection() , and SetOutputOrigin() . Else if the speed image is not nullptr, the output information is copied from the input speed image.

For an alternative implementation, see itk::FastMarchingImageFilter .

Possible Improvements: In the current implementation, std::priority_queue only allows taking nodes out from the front and putting nodes in from the back. To update a value already on the heap, a new node is added to the heap. The defunct old node is left on the heap. When it is removed from the top, it will be recognized as invalid and not used. Future implementations can implement the heap in a different way allowing the values to be updated. This will generally require some sift-up and sift-down functions and an image of back-pointers going from the image to heap in order to locate the node which is to be updated.

\see FastMarchingImageFilterBase 


\see LevelSetTypeDefault
\sa itk::simple::FastMarching for the procedural interface
\sa itk::FastMarchingImageFilter for the Doxygen on the original ITK class.
     */
    class SITKBasicFilters_EXPORT FastMarchingImageFilter : public ImageFilter {
    public:
      using Self = FastMarchingImageFilter;

      /** Destructor */
      virtual ~FastMarchingImageFilter();

      /** Default Constructor that takes no arguments and initializes
       * default parameters */
      FastMarchingImageFilter();

      /** Define the pixels types supported by this filter */
      using PixelIDTypeList = BasicPixelIDTypeList;
\

      /**
       * Set the container of Trial Points representing the initial front. Trial points are represented as a VectorContainer of LevelSetNodes.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetTrialPoints ( std::vector< std::vector<unsigned int> > TrialPoints ) { this->m_TrialPoints = std::move(TrialPoints); return *this; }

      /**
       * Get the container of Trial Points representing the initial front.
       */
      std::vector< std::vector< unsigned int > > GetTrialPoints() const { return this->m_TrialPoints; }
      /** \brief Add TrialPoints point */
      SITK_RETURN_SELF_TYPE_HEADER AddTrialPoint( std::vector< unsigned int > point ) { this->m_TrialPoints.push_back(std::move(point)); return *this;}
      /** \brief Remove all TrialPoints points */
      SITK_RETURN_SELF_TYPE_HEADER ClearTrialPoints( ) { this->m_TrialPoints.clear(); return *this;}
\

      /**
       * Set/Get the Normalization Factor for the Speed Image . The values in the Speed Image is divided by this factor. This allows the use of images with integer pixel types to represent the speed.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetNormalizationFactor ( double NormalizationFactor ) { this->m_NormalizationFactor = NormalizationFactor; return *this; }

      /**
       * Set/Get the Normalization Factor for the Speed Image . The values in the Speed Image is divided by this factor. This allows the use of images with integer pixel types to represent the speed.
       */
      double GetNormalizationFactor() const { return this->m_NormalizationFactor; }\

      /**
       * Set the Fast Marching algorithm Stopping Value. The Fast Marching algorithm is terminated when the value of the smallest trial point is greater than the stopping value.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetStoppingValue ( double StoppingValue ) { this->m_StoppingValue = StoppingValue; return *this; }

      /**
       * Get the Fast Marching algorithm Stopping Value.
       */
      double GetStoppingValue() const { return this->m_StoppingValue; }\

      /**
       * \brief Set the initial seed values for corresponding trial points.
       */
      SITK_RETURN_SELF_TYPE_HEADER SetInitialTrialValues ( std::vector<double> InitialTrialValues ) { this->m_InitialTrialValues = InitialTrialValues; return *this; }

      /**
       */
      std::vector<double> GetInitialTrialValues() const { return this->m_InitialTrialValues; }

      /** Name of this class */
      std::string GetName() const { return std::string ("FastMarchingImageFilter"); }

      /** Print ourselves out */
      std::string ToString() const;


      /** Execute the filter on the input image */

      Image Execute ( const Image& image1 );

    private:

      /** Setup for member function dispatching */

      using MemberFunctionType = Image (Self::*)( const Image& image1 );
      template <class TImageType> Image ExecuteInternal ( const Image& image1 );


      friend struct detail::MemberFunctionAddressor<MemberFunctionType>;

      std::unique_ptr<detail::MemberFunctionFactory<MemberFunctionType> > m_MemberFactory;


      std::vector< std::vector<unsigned int> >  m_TrialPoints{std::vector< std::vector<unsigned int > >()};

      double  m_NormalizationFactor{1.0};

      double  m_StoppingValue{std::numeric_limits<double>::max()/2.0};

      std::vector<double>  m_InitialTrialValues{std::vector<double>()};


    };

    /**\
     * \brief Solve an Eikonal equation using Fast Marching.
     *
     * This function directly calls the execute method of FastMarchingImageFilter
     * in order to support a procedural API
     *
     * \sa itk::simple::FastMarchingImageFilter for the object oriented interface
     * @{
     */

     SITKBasicFilters_EXPORT Image FastMarching ( const Image& image1, std::vector< std::vector<unsigned int> > trialPoints = std::vector< std::vector<unsigned int > >(), double normalizationFactor = 1.0, double stoppingValue = std::numeric_limits<double>::max()/2.0, std::vector<double> initialTrialValues = std::vector<double>() );

     /** @} */
}
#endif
