#! /usr/bin/env nix-shell
#! nix-shell -i bash -p coreutils git -I nixpkgs=.

# This script uses the data pulled with
# maintainers/scripts/haskell/hydra-report.hs get-report to produce a list of
# failing builds that get written to the hackage2nix config. Then
# hackage-packages.nix gets regenerated and transitive-broken packages get
# marked as dont-distribute in the config as well.
# This should disable builds for most failing jobs in the haskell-updates jobset.

set -euo pipefail

do_commit=false
mark_broken_list_flags=""

for arg in "$@"; do
    case "$arg" in
        --do-commit)
            do_commit=true
            ;;
        --no-request-logs)
            mark_broken_list_flags="$mark_broken_list_flags $arg"
            ;;
        *)
            echo "$0: unknown flag: $arg"
            exit 100
            ;;
    esac
done

broken_config="pkgs/development/haskell-modules/configuration-hackage2nix/broken.yaml"

tmpfile=$(mktemp)
trap "rm ${tmpfile}" 0

echo "Remember that you need to manually run 'maintainers/scripts/haskell/hydra-report.hs get-report' sometime before running this script."
echo "Generating a list of broken builds and displaying for manual confirmation ..."
maintainers/scripts/haskell/hydra-report.hs mark-broken-list $mark_broken_list_flags | sort -i > "$tmpfile"

$EDITOR "$tmpfile"

tail -n +3 "$broken_config" >> "$tmpfile"

cat > "$broken_config" << EOF
broken-packages:
  # These packages don't compile.
EOF

# clear environment here to avoid things like allowing broken builds in
sort -iu "$tmpfile" >> "$broken_config"
clear="env -u HOME -u NIXPKGS_CONFIG"
$clear maintainers/scripts/haskell/regenerate-hackage-packages.sh
evalline=$(maintainers/scripts/haskell/hydra-report.hs eval-info)

if $do_commit; then
git add $broken_config
git add pkgs/development/haskell-modules/configuration-hackage2nix/transitive-broken.yaml
git add pkgs/development/haskell-modules/hackage-packages.nix
git commit -F - << EOF
haskellPackages: mark builds failing on hydra as broken

This commit has been generated by maintainers/scripts/haskell/mark-broken.sh based on
$evalline
from the haskell-updates jobset on hydra under https://hydra.nixos.org/jobset/nixpkgs/haskell-updates
EOF
fi
