{
  config,
  lib,
  options,
  pkgs,
  ...
}:
let

  host = config.networking.fqdnOrHostName;

  cfg = config.services.smartd;
  opt = options.services.smartd;

  nm = cfg.notifications.mail;
  ns = cfg.notifications.systembus-notify;
  nw = cfg.notifications.wall;
  nx = cfg.notifications.x11;

  smartdNotify = pkgs.writeScript "smartd-notify.sh" ''
    #! ${pkgs.runtimeShell}
    ${lib.optionalString nm.enable ''
      {
      ${pkgs.coreutils}/bin/cat << EOF
      From: smartd on ${host} <${nm.sender}>
      To: ${nm.recipient}
      Subject: $SMARTD_SUBJECT

      $SMARTD_FULLMESSAGE
      EOF

      ${pkgs.smartmontools}/sbin/smartctl -a -d "$SMARTD_DEVICETYPE" "$SMARTD_DEVICE"
      } | ${nm.mailer} -i "${nm.recipient}"
    ''}
    ${lib.optionalString ns.enable ''
      ${pkgs.dbus}/bin/dbus-send --system \
        / net.nuetzlich.SystemNotifications.Notify \
        "string:Problem detected with disk: $SMARTD_DEVICESTRING" \
        "string:Warning message from smartd is: $SMARTD_MESSAGE"
    ''}
    ${lib.optionalString nw.enable ''
      {
      ${pkgs.coreutils}/bin/cat << EOF
      Problem detected with disk: $SMARTD_DEVICESTRING
      Warning message from smartd is:

      $SMARTD_MESSAGE
      EOF
      } | ${pkgs.util-linux}/bin/wall 2>/dev/null
    ''}
    ${lib.optionalString nx.enable ''
      export DISPLAY=${nx.display}
      {
      ${pkgs.coreutils}/bin/cat << EOF
      Problem detected with disk: $SMARTD_DEVICESTRING
      Warning message from smartd is:

      $SMARTD_FULLMESSAGE
      EOF
      } | ${pkgs.xorg.xmessage}/bin/xmessage -file - 2>/dev/null &
    ''}
  '';

  notifyOpts = lib.optionalString (nm.enable || nw.enable || nx.enable) (
    "-m <nomailer> -M exec ${smartdNotify} " + lib.optionalString cfg.notifications.test "-M test "
  );

  smartdConf = pkgs.writeText "smartd.conf" ''
    # Autogenerated smartd startup config file
    DEFAULT ${notifyOpts}${cfg.defaults.monitored}

    ${lib.concatMapStringsSep "\n" (d: "${d.device} ${d.options}") cfg.devices}

    ${lib.optionalString cfg.autodetect "DEVICESCAN ${notifyOpts}${cfg.defaults.autodetected}"}
  '';

  smartdDeviceOpts =
    { ... }:
    {

      options = {

        device = lib.mkOption {
          example = "/dev/sda";
          type = lib.types.str;
          description = "Location of the device.";
        };

        options = lib.mkOption {
          default = "";
          example = "-d sat";
          type = lib.types.separatedString " ";
          description = "Options that determine how smartd monitors the device.";
        };

      };

    };

in

{
  ###### interface

  options = {

    services.smartd = {

      enable = lib.mkEnableOption "smartd daemon from `smartmontools` package";

      autodetect = lib.mkOption {
        default = true;
        type = lib.types.bool;
        description = ''
          Whenever smartd should monitor all devices connected to the
          machine at the time it's being started (the default).

          Set to false to monitor the devices listed in
          {option}`services.smartd.devices` only.
        '';
      };

      extraOptions = lib.mkOption {
        default = [ ];
        type = lib.types.listOf lib.types.str;
        example = [
          "-A /var/log/smartd/"
          "--interval=3600"
        ];
        description = ''
          Extra command-line options passed to the `smartd`
          daemon on startup.

          (See `man 8 smartd`.)
        '';
      };

      notifications = {

        mail = {
          enable = lib.mkOption {
            default = config.services.mail.sendmailSetuidWrapper != null;
            defaultText = lib.literalExpression "config.services.mail.sendmailSetuidWrapper != null";
            type = lib.types.bool;
            description = "Whenever to send e-mail notifications.";
          };

          sender = lib.mkOption {
            default = "root";
            example = "example@domain.tld";
            type = lib.types.str;
            description = ''
              Sender of the notification messages.
              Acts as the value of `email` in the emails' `From: ...` field.
            '';
          };

          recipient = lib.mkOption {
            default = "root";
            type = lib.types.str;
            description = "Recipient of the notification messages.";
          };

          mailer = lib.mkOption {
            default = "/run/wrappers/bin/sendmail";
            type = lib.types.path;
            description = ''
              Sendmail-compatible binary to be used to send the messages.

              You should probably enable
              {option}`services.postfix` or some other MTA for
              this to work.
            '';
          };
        };

        systembus-notify = {
          enable = lib.mkOption {
            default = false;
            type = lib.types.bool;
            description = ''
              Whenever to send systembus-notify notifications.

              WARNING: enabling this option (while convenient) should *not* be done on a
              machine where you do not trust the other users as it allows any other
              local user to DoS your session by spamming notifications.

              To actually see the notifications in your GUI session, you need to have
              `systembus-notify` running as your user, which this
              option handles by enabling {option}`services.systembus-notify`.
            '';
          };
        };

        wall = {
          enable = lib.mkOption {
            default = true;
            type = lib.types.bool;
            description = "Whenever to send wall notifications to all users.";
          };
        };

        x11 = {
          enable = lib.mkOption {
            default = config.services.xserver.enable;
            defaultText = lib.literalExpression "config.services.xserver.enable";
            type = lib.types.bool;
            description = "Whenever to send X11 xmessage notifications.";
          };

          display = lib.mkOption {
            default = ":${toString config.services.xserver.display}";
            defaultText = lib.literalExpression ''":''${toString config.services.xserver.display}"'';
            type = lib.types.str;
            description = "DISPLAY to send X11 notifications to.";
          };
        };

        test = lib.mkOption {
          default = false;
          type = lib.types.bool;
          description = "Whenever to send a test notification on startup.";
        };

      };

      defaults = {
        monitored = lib.mkOption {
          default = "-a";
          type = lib.types.separatedString " ";
          example = "-a -o on -s (S/../.././02|L/../../7/04)";
          description = ''
            Common default options for explicitly monitored (listed in
            {option}`services.smartd.devices`) devices.

            The default value turns on monitoring of all the things (see
            `man 5 smartd.conf`).

            The example also turns on SMART Automatic Offline Testing on
            startup, and schedules short self-tests daily, and long
            self-tests weekly.
          '';
        };

        autodetected = lib.mkOption {
          default = cfg.defaults.monitored;
          defaultText = lib.literalExpression "config.${opt.defaults.monitored}";
          type = lib.types.separatedString " ";
          description = ''
            Like {option}`services.smartd.defaults.monitored`, but for the
            autodetected devices.
          '';
        };
      };

      devices = lib.mkOption {
        default = [ ];
        example = [
          { device = "/dev/sda"; }
          {
            device = "/dev/sdb";
            options = "-d sat";
          }
        ];
        type = with lib.types; listOf (submodule smartdDeviceOpts);
        description = "List of devices to monitor.";
      };

    };

  };

  ###### implementation

  config = lib.mkIf cfg.enable {

    assertions = [
      {
        assertion = cfg.autodetect || cfg.devices != [ ];
        message = "smartd can't run with both disabled autodetect and an empty list of devices to monitor.";
      }
    ];

    systemd.services.smartd = {
      description = "S.M.A.R.T. Daemon";
      wantedBy = [ "multi-user.target" ];
      serviceConfig = {
        Type = "notify";
        ExecStart = "${pkgs.smartmontools}/sbin/smartd ${lib.concatStringsSep " " cfg.extraOptions} --no-fork --configfile=${smartdConf}";
      };
    };

    services.systembus-notify.enable = lib.mkDefault ns.enable;

  };

}
