{ lib
, stdenv
, python3
, fetchFromGitHub
, rustPlatform
, SDL2
, libiconv
, darwin
}:

python3.pkgs.buildPythonApplication rec {
  pname = "pyxel";
  version = "2.0.7";
  pyproject = true;

  disabled = python3.pythonOlder "3.7";

  src = fetchFromGitHub {
    owner = "kitao";
    repo = "pyxel";
    rev = "v${version}";
    hash = "sha256-5Jrwfi79HbS4hh+eMwI49Rsk4jrAdAuDhNpUT2cEvDo=";
  };

  patches = [
    ./never-bundle-sdl2.patch
    ./update-bindgen-f16-support.patch # can be removed once rust-bindgen gets a new release
  ];

  cargoRoot = "crates/pyxel-wrapper";

  # Lockfile is generated by applying patches with `git apply`
  # and then running `cargo generate-lockfile` in `crates/pyxel-wrapper`
  cargoDeps = rustPlatform.importCargoLock {
    lockFile = ./Cargo.lock;
    outputHashes = {
      "bindgen-0.69.1" = "sha256-1967EmuyWgmrKmhwAcW49dlmuWPNuKjuRr5/u7ZKpXQ=";
    };
  };

  postPatch = ''
    cp ${./Cargo.lock} crates/pyxel-wrapper/Cargo.lock
  '';

  nativeBuildInputs = with rustPlatform; [
    cargoSetupHook
    maturinBuildHook
    bindgenHook
  ];

  buildInputs = [ SDL2 ] ++ lib.optionals stdenv.isDarwin [
    libiconv
    darwin.apple_sdk.frameworks.IOKit
  ];

  env.NIX_CFLAGS_COMPILE = "-I${lib.getDev SDL2}/include/SDL2";

  # Tests can't use the display
  dontCheck = true;

  pythonImportsCheck = [
    "pyxel"
    "pyxel.pyxel_wrapper"
  ];

  meta = {
    changelog = "https://github.com/kitao/pyxel/tree/${src.rev}/CHANGELOG.md";
    description = "A retro game engine for Python";
    homepage = "https://github.com/kitao/pyxel";
    license = lib.licenses.mit;
    mainProgram = "pyxel";
    maintainers = with lib.maintainers; [ tomasajt ];
    platforms = lib.platforms.linux ++ lib.platforms.darwin;
  };
}
