{ version, sha256, patches ? [] }:

{ lib, stdenv, buildPackages, fetchurl, perl, xz, libintl, bash
, gnulib, gawk, freebsd, libiconv

# we are a dependency of gcc, this simplifies bootstraping
, interactive ? false, ncurses, procps
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

let
  inherit (lib) getDev getLib optional optionals optionalString;
  crossBuildTools = stdenv.hostPlatform != stdenv.buildPlatform;
in

stdenv.mkDerivation {
  pname = "texinfo${optionalString interactive "-interactive"}";
  inherit version;

  src = fetchurl {
    url = "mirror://gnu/texinfo/texinfo-${version}.tar.xz";
    inherit sha256;
  };

  patches = patches ++ optional crossBuildTools ./cross-tools-flags.patch;

  postPatch = ''
    patchShebangs tp/maintain
  ''
  # This patch is needed for IEEE-standard long doubles on
  # powerpc64; it does not apply cleanly to texinfo 5.x or
  # earlier.  It is merged upstream in texinfo 6.8.
  + lib.optionalString (version == "6.7") ''
    patch -p1 -d gnulib < ${gnulib.passthru.longdouble-redirect-patch}
  '';

  # ncurses is required to build `makedoc'
  # this feature is introduced by the ./cross-tools-flags.patch
  NATIVE_TOOLS_CFLAGS = if crossBuildTools then "-I${getDev buildPackages.ncurses}/include" else null;
  NATIVE_TOOLS_LDFLAGS = if crossBuildTools then "-L${getLib buildPackages.ncurses}/lib" else null;

  strictDeps = true;
  enableParallelBuilding = true;

  # A native compiler is needed to build tools needed at build time
  depsBuildBuild = [ buildPackages.stdenv.cc perl ];

  buildInputs = [ xz.bin bash libintl ]
    ++ optionals stdenv.isSunOS [ libiconv gawk ]
    ++ optional interactive ncurses;

  configureFlags = [ "PERL=${buildPackages.perl}/bin/perl" ]
    # Perl XS modules are difficult to cross-compile and texinfo has pure Perl
    # fallbacks.
    # Also prevent the buildPlatform's awk being used in the texindex script
    ++ optionals crossBuildTools [ "--enable-perl-xs=no" "TI_AWK=${gawk}/bin/awk" ]
    ++ lib.optional stdenv.isSunOS "AWK=${gawk}/bin/awk";

  installFlags = [ "TEXMF=$(out)/texmf-dist" ];
  installTargets = [ "install" "install-tex" ];

  nativeCheckInputs = [ procps ]
    ++ optionals stdenv.buildPlatform.isFreeBSD [ freebsd.locale ];

  doCheck = interactive
    && !stdenv.isDarwin
    && !stdenv.isSunOS; # flaky

  checkFlags = lib.optionals (!stdenv.hostPlatform.isMusl && lib.versionOlder version "7") [
    # Test is known to fail on various locales on texinfo-6.8:
    #   https://lists.gnu.org/r/bug-texinfo/2021-07/msg00012.html
    "XFAIL_TESTS=test_scripts/layout_formatting_fr_icons.sh"
  ];

  postFixup = optionalString crossBuildTools ''
    for f in "$out"/bin/{pod2texi,texi2any}; do
      substituteInPlace "$f" \
        --replace ${buildPackages.perl}/bin/perl ${perl}/bin/perl
    done
  '';

  meta = with lib; {
    description = "The GNU documentation system";
    homepage = "https://www.gnu.org/software/texinfo/";
    changelog = "https://git.savannah.gnu.org/cgit/texinfo.git/plain/NEWS";
    license = licenses.gpl3Plus;
    platforms = platforms.all;
    maintainers = with maintainers; [ vrthra oxij ];
    # see comment above in patches section
    broken = stdenv.hostPlatform.isPower64 && lib.strings.versionOlder version "6.0";

    longDescription = ''
      Texinfo is the official documentation format of the GNU project.
      It was invented by Richard Stallman and Bob Chassell many years
      ago, loosely based on Brian Reid's Scribe and other formatting
      languages of the time.  It is used by many non-GNU projects as
      well.

      Texinfo uses a single source file to produce output in a number
      of formats, both online and printed (dvi, html, info, pdf, xml,
      etc.).  This means that instead of writing different documents
      for online information and another for a printed manual, you
      need write only one document.  And when the work is revised, you
      need revise only that one document.  The Texinfo system is
      well-integrated with GNU Emacs.
    '';
    branch = version;
  };
}
