{ lib, stdenv
, fetchurl
, makeWrapper
, xz
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

stdenv.mkDerivation rec {
  pname = "gzip";
  version = "1.13";

  src = fetchurl {
    url = "mirror://gnu/gzip/${pname}-${version}.tar.xz";
    hash = "sha256-dFTraTXbF8ZlVXbC4bD6vv04tNCTbg+H9IzQYs6RoFc=";
  };

  outputs = [ "out" "man" "info" ];

  enableParallelBuilding = true;

  nativeBuildInputs = [ xz.bin makeWrapper ];

  makeFlags = [
    "SHELL=/bin/sh"
    "GREP=grep"
    # gzip 1.12 doesn't build `zless` unless it can find `less`, but we
    # can avoid having `less` as a build input if we just override these.
    "ZLESS_MAN=zless.1"
    "ZLESS_PROG=zless"
  ];

  # Many gzip executables are shell scripts that depend upon other gzip
  # executables being in $PATH.  Rather than try to re-write all the
  # internal cross-references, just add $out/bin to PATH at the top of
  # all the executables that are shell scripts.
  preFixup = ''
    sed -i '1{;/#!\/bin\/sh/aPATH="'$out'/bin:$PATH"
    }' $out/bin/*
  ''
  # run gzip with "-n" when $GZIP_NO_TIMESTAMPS (set by stdenv's setup.sh) is set to stop gzip from adding timestamps
  # to archive headers: https://github.com/NixOS/nixpkgs/issues/86348
  # if changing so that there's no longer a .gzip-wrapped then update copy in make-bootstrap-tools.nix
  + ''
    wrapProgram $out/bin/gzip \
      --add-flags "\''${GZIP_NO_TIMESTAMPS:+-n}"
  '';

  meta = {
    homepage = "https://www.gnu.org/software/gzip/";
    description = "GNU zip compression program";

    longDescription =
      ''gzip (GNU zip) is a popular data compression program written by
        Jean-loup Gailly for the GNU project.  Mark Adler wrote the
        decompression part.

        We developed this program as a replacement for compress because of
        the Unisys and IBM patents covering the LZW algorithm used by
        compress.  These patents made it impossible for us to use compress,
        and we needed a replacement.  The superior compression ratio of gzip
        is just a bonus.
      '';

    platforms = lib.platforms.all;

    license = lib.licenses.gpl3Plus;

    mainProgram = "gzip";
  };
}
