/* The top-level package collection of nixpkgs.
 * It is sorted by categories corresponding to the folder names in the /pkgs
 * folder. Inside the categories packages are roughly sorted by alphabet, but
 * strict sorting has been long lost due to merges. Please use the full-text
 * search of your editor. ;)
 * Hint: ### starts category names.
 */
{ lib, noSysDirs, config, overlays }:
res: pkgs: super:

with pkgs;

{
  # A module system style type tag
  #
  # Allows the nixpkgs fixpoint, usually known as `pkgs` to be distinguished
  # nominally.
  #
  #     pkgs._type == "pkgs"
  #     pkgs.pkgsStatic._type == "pkgs"
  #
  # Design note:
  # While earlier stages of nixpkgs fixpoint construction are supertypes of this
  # stage, they're generally not usable in places where a `pkgs` is expected.
  # (earlier stages being the various `super` variables that precede
  # all-packages.nix)
  _type = "pkgs";

  # A stdenv capable of building 32-bit binaries.
  # On x86_64-linux, it uses GCC compiled with multilib support; on i686-linux,
  # it's just the plain stdenv.
  stdenv_32bit = lowPrio (if stdenv.hostPlatform.is32bit then stdenv else multiStdenv);

  stdenvNoCC = stdenv.override (
    { cc = null; hasCC = false; }

    // lib.optionalAttrs (stdenv.hostPlatform.isDarwin && (stdenv.hostPlatform != stdenv.buildPlatform)) {
      # TODO: This is a hack to use stdenvNoCC to produce a CF when cross
      # compiling. It's not very sound. The cross stdenv has:
      #   extraBuildInputs = [ targetPackages.darwin.apple_sdks.frameworks.CoreFoundation ]
      # and uses stdenvNoCC. In order to make this not infinitely recursive, we
      # need to exclude this extraBuildInput.
      extraBuildInputs = [];
    }
  );

  mkStdenvNoLibs = stdenv: let
    bintools = stdenv.cc.bintools.override {
      libc = null;
      noLibc = true;
    };
  in stdenv.override {
    cc = stdenv.cc.override {
      libc = null;
      noLibc = true;
      extraPackages = [];
      inherit bintools;
    };
    allowedRequisites =
      lib.mapNullable (rs: rs ++ [ bintools ]) (stdenv.allowedRequisites or null);
  };

  stdenvNoLibs =
    if stdenvNoCC.hostPlatform != stdenvNoCC.buildPlatform && (stdenv.hostPlatform.isDarwin || stdenv.hostPlatform.useLLVM or false)
    then
      # We cannot touch binutils or cc themselves, because that will cause
      # infinite recursion. So instead, we just choose a libc based on the
      # current platform. That means we won't respect whatever compiler was
      # passed in with the stdenv stage argument.
      #
      # TODO It would be much better to pass the `stdenvNoCC` and *unwrapped*
      # cc, bintools, compiler-rt equivalent, etc. and create all final stdenvs
      # as part of the stage. Then we would never be tempted to override a later
      # thing to to create an earlier thing (leading to infinite recursion) and
      # we also would still respect the stage arguments choices for these
      # things.
      overrideCC stdenv buildPackages.llvmPackages.clangNoCompilerRt
    else mkStdenvNoLibs stdenv;

  gccStdenvNoLibs = mkStdenvNoLibs gccStdenv;
  clangStdenvNoLibs = mkStdenvNoLibs clangStdenv;

  # For convenience, allow callers to get the path to Nixpkgs.
  path = ../..;

  ### Helper functions.
  inherit lib config overlays;

  # do not import 'appendToName' to get consistent package-names with the same
  # set of package-parameters: https://github.com/NixOS/nixpkgs/issues/68519
  inherit (lib) lowPrio hiPrio makeOverridable;

  inherit (lib) recurseIntoAttrs;

  # This is intended to be the reverse of recurseIntoAttrs, as it is
  # defined now it exists mainly for documentation purposes, but you
  # can also override this with recurseIntoAttrs to recurseInto all
  # the Attrs which is useful for testing massive changes. Ideally,
  # every package subset not marked with recurseIntoAttrs should be
  # marked with this.
  inherit (lib) dontRecurseIntoAttrs;

  stringsWithDeps = lib.stringsWithDeps;

  ### Evaluating the entire Nixpkgs naively will fail, make failure fast
  AAAAAASomeThingsFailToEvaluate = throw ''
    Please be informed that this pseudo-package is not the only part
    of Nixpkgs that fails to evaluate. You should not evaluate
    entire Nixpkgs without some special measures to handle failing
    packages, like using pkgs/top-level/release-attrpaths-superset.nix.
  '';

  tests = callPackages ../test { };

  defaultPkgConfigPackages =
    # We don't want nix-env -q to enter this, because all of these are aliases.
    dontRecurseIntoAttrs (
      import ./pkg-config/defaultPkgConfigPackages.nix pkgs
    ) // { __attrsFailEvaluation = true; };

  ### Nixpkgs maintainer tools

  nix-generate-from-cpan = callPackage ../../maintainers/scripts/nix-generate-from-cpan.nix { };

  nixpkgs-lint = callPackage ../../maintainers/scripts/nixpkgs-lint.nix { };

  common-updater-scripts = callPackage ../common-updater/scripts.nix { };

  vimPluginsUpdater = callPackage ../applications/editors/vim/plugins/updater.nix {
    inherit (python3Packages) buildPythonApplication ;
  };

  genericUpdater = callPackage ../common-updater/generic-updater.nix { };

  _experimental-update-script-combinators = callPackage ../common-updater/combinators.nix { };

  directoryListingUpdater = callPackage ../common-updater/directory-listing-updater.nix { };

  gitUpdater = callPackage ../common-updater/git-updater.nix { };

  httpTwoLevelsUpdater = callPackage ../common-updater/http-two-levels-updater.nix { };

  unstableGitUpdater = callPackage ../common-updater/unstable-updater.nix { };

  nix-update-script = callPackage ../common-updater/nix-update.nix { };

  ### Push NixOS tests inside the fixed point

  # See also allTestsForSystem in nixos/release.nix
  nixosTests = import ../../nixos/tests/all-tests.nix {
    inherit pkgs;
    system = stdenv.hostPlatform.system;
    callTest = config: config.test;
  } // {
    # for typechecking of the scripts and evaluation of
    # the nodes, without running VMs.
    allDrivers = import ../../nixos/tests/all-tests.nix {
      inherit pkgs;
      system = stdenv.hostPlatform.system;
      callTest = config: config.test.driver;
    };
    __attrsFailEvaluation = true;
  };

  ### BUILD SUPPORT

  autoreconfHook = callPackage (
    { makeSetupHook, autoconf, automake, gettext, libtool }:
    makeSetupHook {
      name = "autoreconf-hook";
      propagatedBuildInputs = [ autoconf automake gettext libtool ];
    } ../build-support/setup-hooks/autoreconf.sh
  ) { };

  autoreconfHook264 = autoreconfHook.override {
    autoconf = autoconf264;
    automake = automake111x;
  };

  autoreconfHook269 = autoreconfHook.override {
    autoconf = autoconf269;
  };
  autoreconfHook271 = autoreconfHook.override {
    autoconf = autoconf271;
  };

  autorestic = callPackage ../tools/backup/autorestic { };

  autoPatchelfHook = makeSetupHook {
    name = "auto-patchelf-hook";
    propagatedBuildInputs = [ bintools ];
    substitutions = {
      pythonInterpreter = "${python3.withPackages (ps: [ ps.pyelftools ])}/bin/python";
      autoPatchelfScript = ../build-support/setup-hooks/auto-patchelf.py;
    };
  } ../build-support/setup-hooks/auto-patchelf.sh;

  tomato-c = callPackage ../applications/misc/tomato-c { };

  appflowy = callPackage ../applications/office/appflowy { };

  appimageTools = callPackage ../build-support/appimage { };

  appindicator-sharp = callPackage ../development/libraries/appindicator-sharp { };

  bindle = callPackage ../servers/bindle {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  stripJavaArchivesHook = makeSetupHook {
    name = "strip-java-archives-hook";
    propagatedBuildInputs = [ strip-nondeterminism ];
  } ../build-support/setup-hooks/strip-java-archives.sh;

  ensureNewerSourcesHook = { year }: makeSetupHook {
    name = "ensure-newer-sources-hook";
  } (writeScript "ensure-newer-sources-hook.sh" ''
      postUnpackHooks+=(_ensureNewerSources)
      _ensureNewerSources() {
        '${findutils}/bin/find' "$sourceRoot" \
          '!' -newermt '${year}-01-01' -exec touch -h -d '${year}-01-02' '{}' '+'
      }
    '');

  # addDriverRunpath is the preferred package name, as this enables
  # many more scenarios than just opengl now.
  addDriverRunpath = callPackage ../build-support/add-driver-runpath { };

  # addOpenGLRunpath should be added to aliases.nix after the 24.05 branch-off.
  # Post 24.11 branch-off, this should throw an error in aliases.nix.
  addOpenGLRunpath = callPackage ../build-support/add-opengl-runpath { };

  quickgui = callPackage ../applications/virtualization/quickgui { };

  adcli = callPackage ../os-specific/linux/adcli { };

  alda = callPackage ../development/interpreters/alda { };

  align = callPackage ../tools/text/align { };

  alire = callPackage ../development/tools/build-managers/alire { };

  althttpd = callPackage ../servers/althttpd { };

  amqpcat = callPackage ../development/tools/amqpcat { };

  anders = callPackage ../applications/science/logic/anders { };

  ankisyncd = callPackage ../servers/ankisyncd { };

  ariang = callPackage ../servers/ariang { };

  ani-cli = callPackage ../applications/video/ani-cli { };

  dra-cla = callPackage ../applications/video/dra-cla { };

  anime-downloader = callPackage ../applications/video/anime-downloader { };

  aocd = with python3Packages; toPythonApplication aocd;

  aesfix = callPackage ../tools/security/aesfix { };

  aeskeyfind = callPackage ../tools/security/aeskeyfind { };

  alterx = callPackage ../tools/security/alterx { };

  altserver-linux = callPackage ../tools/misc/altserver-linux { };

  asciicam = callPackage ../applications/video/asciicam { };

  asitop = pkgs.python3Packages.callPackage ../os-specific/darwin/asitop { };

  asnmap = callPackage ../tools/security/asnmap { };

  astrolog = callPackage ../applications/science/astronomy/astrolog { };

  astronomer = callPackage ../tools/misc/astronomer { };

  atkinson-hyperlegible = callPackage ../data/fonts/atkinson-hyperlegible { };

  _0x =  callPackage ../tools/misc/0x { };

  automatic-timezoned = callPackage ../tools/system/automatic-timezoned { };

  cve = with python3Packages; toPythonApplication cvelib;

  db-rest = callPackage ../servers/db-rest { };

  fiche = callPackage ../servers/fiche { };

  fishnet = callPackage ../servers/fishnet { };

  fission = callPackage ../development/tools/fission { };

  apko = callPackage ../development/tools/apko { };

  melange = callPackage ../development/tools/melange { };

  activate-linux = callPackage ../applications/misc/activate-linux { };

  ansi = callPackage ../development/tools/ansi { };

  ares-rs = callPackage ../tools/security/ares-rs { };

  arti = callPackage ../tools/security/arti {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  asn1c = callPackage ../development/compilers/asn1c { };

  authz0 = callPackage ../tools/security/authz0 { };

  auth0-cli = callPackage ../tools/admin/auth0-cli { };

  atomic-operator = callPackage ../tools/security/atomic-operator { };

  atomic-swap = callPackage ../applications/blockchains/atomic-swap { };

  avro-tools = callPackage ../development/tools/avro-tools { };

  bacnet-stack = callPackage ../tools/networking/bacnet-stack { };

  bada-bib = callPackage ../applications/science/misc/bada-bib { };

  badlion-client = callPackage ../games/badlion-client {};

  banana-accounting = callPackage ../applications/office/banana-accounting { };

  beebeep = libsForQt5.callPackage ../applications/office/beebeep { };

  beeper = callPackage ../applications/networking/instant-messengers/beeper { };

  bakelite = callPackage ../tools/backup/bakelite { };

  bazecor = callPackage ../applications/misc/bazecor { };

  bearer = callPackage ../development/tools/bearer { };

  benthos = callPackage ../development/tools/benthos { };

  beyond-identity = callPackage ../tools/security/beyond-identity { };

  bibtex-tidy = callPackage ../tools/typesetting/bibtex-tidy { };

  bililiverecorder = callPackage ../servers/bililiverecorder { };

  binbloom = callPackage ../tools/security/binbloom { };

  bingo = callPackage ../development/tools/bingo { };

  bin2c = callPackage ../development/tools/bin2c { };

  binserve = callPackage ../servers/binserve {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  blst = callPackage ../development/libraries/blst { };

  bloom = qt6Packages.callPackage ../development/tools/bloom { };

  bloodhound-py = with python3Packages; toPythonApplication bloodhound-py;

  bodyclose = callPackage ../development/tools/bodyclose { };

  bootstrap-studio = callPackage ../development/web/bootstrap-studio { };

  breakpad = callPackage ../development/misc/breakpad { };

  brev-cli = callPackage ../development/misc/brev-cli { };

  buf = callPackage ../development/tools/buf { };

  cbfmt = callPackage ../development/tools/cbfmt { };

  cfn-nag = callPackage ../development/tools/cfn-nag { };

  circumflex = callPackage ../applications/networking/circumflex { };

  citron = callPackage ../tools/misc/citron { };

  cxx-rs = callPackage ../development/libraries/cxx-rs { };

  elfcat = callPackage ../tools/misc/elfcat { };

  # Zip file format only allows times after year 1980, which makes e.g. Python
  # wheel building fail with:
  # ValueError: ZIP does not support timestamps before 1980
  ensureNewerSourcesForZipFilesHook = ensureNewerSourcesHook { year = "1980"; };

  updateAutotoolsGnuConfigScriptsHook = makeSetupHook {
    name = "update-autotools-gnu-config-scripts-hook";
    substitutions = { gnu_config = gnu-config; };
  } ../build-support/setup-hooks/update-autotools-gnu-config-scripts.sh;

  gogUnpackHook = makeSetupHook {
    name = "gog-unpack-hook";
    propagatedBuildInputs = [ innoextract file-rename ]; }
    ../build-support/setup-hooks/gog-unpack.sh;

  buildEnv = callPackage ../build-support/buildenv { }; # not actually a package

  buildFHSEnv = buildFHSEnvBubblewrap;
  buildFHSEnvChroot = callPackage ../build-support/build-fhsenv-chroot { }; # Deprecated; use buildFHSEnv/buildFHSEnvBubblewrap
  buildFHSEnvBubblewrap = callPackage ../build-support/build-fhsenv-bubblewrap { };

  buildMaven = callPackage ../build-support/build-maven.nix { };

  c64-debugger = callPackage ../applications/emulators/c64-debugger { };

  camunda-modeler = callPackage ../applications/misc/camunda-modeler { };

  checkpointBuildTools = callPackage ../build-support/checkpoint-build.nix {};

  caroline = callPackage ../development/libraries/caroline { };

  cartridges = callPackage ../applications/misc/cartridges { };

  castget = callPackage ../applications/networking/feedreaders/castget { };

  catatonit = callPackage ../applications/virtualization/catatonit { };

  catppuccin-catwalk = callPackage ../development/tools/misc/catppuccin-catwalk { };

  catppuccin-kde = callPackage ../data/themes/catppuccin-kde { };

  catppuccin-kvantum = callPackage ../data/themes/catppuccin-kvantum { };

  catppuccin-papirus-folders = callPackage ../data/icons/catppuccin-papirus-folders { };

  btdu = callPackage ../tools/misc/btdu { };

  ccal = callPackage ../tools/misc/ccal { };

  celeste-classic-pm = pkgs.celeste-classic.override {
    practiceMod = true;
  };

  cereal = cereal_1_3_0;
  cereal_1_3_0 = callPackage ../development/libraries/cereal/1.3.0.nix { };
  cereal_1_3_2 = callPackage ../development/libraries/cereal/1.3.2.nix { };

  cdecrypt = callPackage ../development/tools/cdecrypt { };

  certgraph = callPackage ../tools/security/certgraph { };

  certinfo = callPackage ../tools/security/certinfo { };

  certsync = callPackage ../tools/security/certsync { };

  cewl = callPackage ../tools/security/cewl { };

  chatgpt-cli = callPackage ../tools/misc/chatgpt-cli { };

  chatgpt-retrieval-plugin = callPackage ../servers/chatgpt-retrieval-plugin { };

  mswatch = callPackage ../applications/networking/mailreaders/mswatch { };

  chef-cli = callPackage ../tools/misc/chef-cli { };

  checkov = callPackage ../development/tools/analysis/checkov {
    python3 = python311;
  };

  checkpwn = callPackage ../tools/misc/checkpwn { };

  banana-vera = callPackage ../development/tools/analysis/banana-vera { };

  ciel = callPackage ../tools/package-management/ciel { };

  circt = callPackage ../development/compilers/circt { };

  classicube = callPackage ../games/classicube { };

  clj-kondo = callPackage ../development/tools/clj-kondo { };

  cloak = callPackage ../applications/misc/cloak {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  cotp = callPackage ../applications/misc/cotp {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  cmark = callPackage ../development/libraries/cmark { };

  cmark-gfm = callPackage ../development/libraries/cmark-gfm { };

  cm256cc = callPackage ../development/libraries/cm256cc {  };

  cnquery = callPackage ../tools/security/cnquery { };

  cocogitto = callPackage ../development/tools/cocogitto {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  conftest = callPackage ../development/tools/conftest { };

  coldsnap = callPackage ../tools/admin/coldsnap {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  colemak-dh = callPackage ../data/misc/colemak-dh { };

  collision = callPackage ../applications/misc/collision { };

  colmena = callPackage ../tools/admin/colmena { };

  colorz = callPackage ../tools/misc/colorz { };

  colorpanes = callPackage ../tools/misc/colorpanes { };

  colorpicker = callPackage ../tools/misc/colorpicker { };

  colorstorm = callPackage ../applications/misc/colorstorm { };

  comedilib = callPackage ../development/libraries/comedilib {  };

  commix = callPackage ../tools/security/commix { };

  comodoro = callPackage ../applications/misc/comodoro { };

  compdb = callPackage ../tools/misc/compdb { };

  conserve = callPackage ../tools/backup/conserve { };

  conserver = callPackage ../tools/misc/conserver { };

  containerlab = callPackage ../tools/networking/containerlab { };

  containerpilot = callPackage ../applications/networking/cluster/containerpilot { };

  coolercontrol = recurseIntoAttrs (callPackage ../applications/system/coolercontrol { });

  confetty = callPackage ../applications/misc/confetty { };

  confy = callPackage ../applications/misc/confy { };

  coordgenlibs  = callPackage ../development/libraries/coordgenlibs { };

  copilot-cli = callPackage ../tools/admin/copilot-cli { };

  cp210x-program = callPackage ../tools/misc/cp210x-program { };

  cp437 = callPackage ../tools/misc/cp437 { };

  cpm-cmake = callPackage ../development/tools/cpm-cmake { };

  crackle = callPackage ../tools/networking/crackle { };

  crackql = callPackage ../tools/security/crackql { };

  crow-translate = libsForQt5.callPackage ../applications/misc/crow-translate { };

  dae = callPackage ../tools/networking/dae { };

  darling = callPackage ../applications/emulators/darling { };

  databricks-sql-cli = python3Packages.callPackage ../applications/misc/databricks-sql-cli { };

  dbip-country-lite = callPackage ../data/misc/dbip-country-lite { };

  dcgm = callPackage ../os-specific/linux/dcgm { };

  dhallDirectoryToNix = callPackage ../build-support/dhall/directory-to-nix.nix { };

  dhallPackageToNix = callPackage ../build-support/dhall/package-to-nix.nix { };

  dhallToNix = callPackage ../build-support/dhall/to-nix.nix { };

  dinghy = with python3Packages; toPythonApplication dinghy;

  djgpp = djgpp_i586;
  djgpp_i586 = callPackage ../development/compilers/djgpp { targetArchitecture = "i586"; stdenv = gccStdenv; };
  djgpp_i686 = lowPrio (callPackage ../development/compilers/djgpp { targetArchitecture = "i686"; stdenv = gccStdenv; });

  djhtml = python3Packages.callPackage ../development/tools/djhtml { };

  deadnix = callPackage ../development/tools/deadnix { };

  dec-decode = callPackage ../development/tools/dec-decode { };

  dnf-plugins-core = with python3Packages; toPythonApplication dnf-plugins-core;

  dnf4 = python3Packages.callPackage ../development/python-modules/dnf4/wrapper.nix { };

  dnf5 = callPackage ../tools/package-management/dnf5 { };

  documenso = callPackage ../applications/office/documenso { };

  domine = callPackage ../tools/misc/domine { };

  dsq = callPackage ../tools/misc/dsq { };

  dt = callPackage ../tools/text/dt { };

  dtv-scan-tables = callPackage ../data/misc/dtv-scan-tables { };

  dufs = callPackage ../servers/http/dufs { };

  dynein = callPackage ../development/tools/database/dynein {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ea = callPackage ../tools/misc/ea { };

  each = callPackage ../tools/text/each { };

  ebusd = callPackage ../servers/ebusd { };

  eclipse-mat = callPackage ../development/tools/eclipse-mat { };

  ebpf-verifier = callPackage ../tools/networking/ebpf-verifier {
    catch2 = catch2_3;
  };

  edgedb = callPackage ../tools/networking/edgedb {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  edge-runtime = callPackage ../development/web/edge-runtime { };

  efficient-compression-tool = callPackage ../tools/compression/efficient-compression-tool { };

  eludris = callPackage ../tools/misc/eludris {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  enochecker-test = with python3Packages; callPackage ../development/tools/enochecker-test { };

  enumer = callPackage ../tools/misc/enumer { };

  evans = callPackage ../development/tools/evans { };

  evhz = callPackage ../tools/misc/evhz { };

  expand-response-params = callPackage ../build-support/expand-response-params { };

  expressvpn = callPackage ../applications/networking/expressvpn { };

  faq = callPackage ../development/tools/faq { };

  femtolisp = callPackage ../development/interpreters/femtolisp { };

  fh = callPackage ../tools/nix/fh { };

  figma-agent = callPackage ../applications/graphics/figma-agent { };

  figma-linux = callPackage ../applications/graphics/figma-linux { };

  firefly-desktop = callPackage ../applications/misc/firefly-desktop { };

  flaca = callPackage ../applications/graphics/flaca { };

  frece = callPackage ../development/tools/frece { };

  frida-tools = callPackage ../tools/security/frida-tools { };

  frink = callPackage ../development/tools/frink { };

  frugal = callPackage ../development/tools/frugal { };

  frugally-deep = callPackage ../development/libraries/frugally-deep { };

  functiontrace-server = callPackage ../development/tools/functiontrace-server { };

  gendef = callPackage ../development/tools/gendef { };

  glade = callPackage ../development/tools/glade { };

  glamoroustoolkit = callPackage ../development/tools/glamoroustoolkit { };

  gobble = callPackage ../tools/X11/gobble { };

  goda = callPackage ../development/tools/goda { };

  gokrazy = callPackage ../development/misc/gokrazy { };

  gojq = callPackage ../development/tools/gojq { };

  govulncheck = callPackage ../tools/security/govulncheck { };

  gpick = callPackage ../tools/misc/gpick { };

  gradescope-submit = callPackage ../tools/misc/gradescope-submit { };

  gridlock = callPackage ../tools/nix/gridlock { };
  inherit (gridlock) nyarr;

  hwatch = callPackage ../tools/misc/hwatch { };

  hobbes = callPackage ../development/tools/hobbes { };

  honeycomb-refinery = callPackage ../servers/tracing/honeycomb/refinery { };

  html5validator = python3Packages.callPackage ../applications/misc/html5validator { };

  inspec = callPackage ../tools/misc/inspec { };

  lshw-gui = lshw.override { withGUI = true; };

  kdePackages = callPackage ../kde { };

  buildcatrust = with python3.pkgs; toPythonApplication buildcatrust;

  probe-rs = callPackage ../development/tools/rust/probe-rs {
    inherit (darwin.apple_sdk.frameworks) AppKit;
    inherit (darwin) DarwinTools;
  };

  probe-run = callPackage ../development/tools/rust/probe-run {
    inherit (darwin.apple_sdk.frameworks) AppKit IOKit;
    inherit (darwin) DarwinTools;
  };

  mix2nix = callPackage ../development/tools/mix2nix { };

  n98-magerun = callPackage ../development/tools/misc/n98-magerun { };

  n98-magerun2 = callPackage ../development/tools/misc/n98-magerun2 { };

  prisma-engines = callPackage ../development/tools/database/prisma-engines {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  pridecat = callPackage ../tools/misc/pridecat { };

  pridefetch = callPackage ../tools/misc/pridefetch { };

  proto-contrib = callPackage ../development/tools/proto-contrib { };

  protoc-gen-dart = callPackage ../development/tools/protoc-gen-dart { };

  protoc-gen-doc = callPackage ../development/tools/protoc-gen-doc { };

  protoc-gen-entgrpc = callPackage ../development/tools/protoc-gen-entgrpc { };

  protoc-gen-go = callPackage ../development/tools/protoc-gen-go { };

  protoc-gen-go-grpc = callPackage ../development/tools/protoc-gen-go-grpc { };

  protoc-gen-go-vtproto = callPackage ../development/tools/protoc-gen-go-vtproto { };

  protoc-gen-grpc-web = callPackage ../development/tools/protoc-gen-grpc-web {
    protobuf = protobuf_21;
  };

  protoc-gen-connect-go = callPackage ../development/tools/protoc-gen-connect-go { };

  protoc-gen-prost = callPackage ../development/tools/protoc-gen-prost { };

  protoc-gen-prost-crate = callPackage ../development/tools/protoc-gen-prost-crate { };

  protoc-gen-prost-serde = callPackage ../development/tools/protoc-gen-prost-serde { };

  protoc-gen-tonic = callPackage ../development/tools/protoc-gen-tonic { };

  protoc-gen-twirp = callPackage ../development/tools/protoc-gen-twirp { };

  protoc-gen-twirp_php = callPackage ../development/tools/protoc-gen-twirp_php { };

  protoc-gen-twirp_swagger = callPackage ../development/tools/protoc-gen-twirp_swagger { };

  protoc-gen-twirp_typescript = callPackage ../development/tools/protoc-gen-twirp_typescript { };

  protoc-gen-validate = callPackage ../development/tools/protoc-gen-validate { };

  protolint = callPackage ../development/tools/protolint { };

  ptags = callPackage ../development/tools/misc/ptags { };

  resolve-march-native = callPackage ../development/tools/resolve-march-native { };

  riot-redis = callPackage ../development/tools/riot-redis { };

  sea-orm-cli = callPackage ../development/tools/sea-orm-cli { };

  vcpkg-tool = callPackage ../by-name/vc/vcpkg-tool/package.nix {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
    fmt = fmt_10;
  };

  r3ctl = qt5.callPackage ../tools/misc/r3ctl { };

  ptouch-print = callPackage ../misc/ptouch-print { };

  demoit = callPackage ../servers/demoit { };

  deviceTree = callPackage ../os-specific/linux/device-tree { };

  enum4linux = callPackage ../tools/security/enum4linux { };

  enum4linux-ng = callPackage ../tools/security/enum4linux-ng { };

  enumerepo = callPackage ../tools/security/enumerepo {};

  erosmb = callPackage ../tools/security/erosmb { };

  eslint_d = callPackage ../development/tools/eslint_d { };

  oauth2c = callPackage ../tools/security/oauth2c { };

  octodns = python3Packages.callPackage ../tools/networking/octodns { };

  octodns-providers = recurseIntoAttrs {
    bind = python3Packages.callPackage ../tools/networking/octodns/providers/bind { };
    gandi = python3Packages.callPackage ../tools/networking/octodns/providers/gandi { };
    hetzner = python3Packages.callPackage ../tools/networking/octodns/providers/hetzner { };
    powerdns = python3Packages.callPackage ../tools/networking/octodns/providers/powerdns { };
  };

  octosuite = callPackage ../tools/security/octosuite { };

  octosql = callPackage ../tools/misc/octosql { };

  onesixtyone = callPackage ../tools/security/onesixtyone { };

  oletools = with python3.pkgs; toPythonApplication oletools;

  ots = callPackage ../tools/security/ots {  };

  credential-detector = callPackage ../tools/security/credential-detector { };

  creds = callPackage ../tools/security/creds { };

  credslayer = callPackage ../tools/security/credslayer { };

  device-tree_rpi = callPackage ../os-specific/linux/device-tree/raspberrypi.nix { };

  devour = callPackage ../tools/X11/devour { };

  didyoumean = callPackage ../tools/misc/didyoumean {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  didu = callPackage ../tools/misc/didu { };

  diffPlugins = (callPackage ../build-support/plugins.nix {}).diffPlugins;

  dieHook = makeSetupHook {
    name = "die-hook";
  } ../build-support/setup-hooks/die.sh;

  archiver = callPackage ../applications/misc/archiver { };

  digitalbitbox = libsForQt5.callPackage ../applications/misc/digitalbitbox {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  gretl = callPackage ../applications/science/math/gretl {
    inherit (darwin.apple_sdk.frameworks) Accelerate;
  };

  grsync = callPackage ../applications/misc/grsync { };

  dockerTools = callPackage ../build-support/docker {
    writePython3 = buildPackages.writers.writePython3;
  } // { __attrsFailEvaluation = true; };

  fakeNss = callPackage ../build-support/fake-nss { };

  tarsum = callPackage ../build-support/docker/tarsum.nix { };

  snapTools = throw "snapTools was removed because makeSnap produced broken snaps and it was the only function in snapTools. See https://github.com/NixOS/nixpkgs/issues/100618 for more details."; # 2024-03-04;

  nix-prefetch-docker = callPackage ../build-support/docker/nix-prefetch-docker.nix { };

  docker-ls = callPackage ../tools/misc/docker-ls { };

  docker-slim = callPackage ../applications/virtualization/docker-slim { };

  doc2go = callPackage ../development/tools/doc2go {
    buildGoModule = buildGo122Module;
  };

  docker-sync = callPackage ../tools/misc/docker-sync { };

  undocker = callPackage ../tools/misc/undocker { };

  dockle = callPackage ../development/tools/dockle { };

  docui = callPackage ../tools/misc/docui { };

  done = callPackage ../applications/misc/done { };

  dotfiles = callPackage ../applications/misc/dotfiles { };

  # Dotnet

  csharp-ls = callPackage ../development/tools/language-servers/csharp-ls { };

  dotnetCorePackages = recurseIntoAttrs (callPackage ../development/compilers/dotnet {});

  dotnet-sdk_6 = dotnetCorePackages.sdk_6_0;
  dotnet-sdk_7 = dotnetCorePackages.sdk_7_0;
  dotnet-sdk_8 = dotnetCorePackages.sdk_8_0;

  dotnet-runtime_6 = dotnetCorePackages.runtime_6_0;
  dotnet-runtime_7 = dotnetCorePackages.runtime_7_0;
  dotnet-runtime_8 = dotnetCorePackages.runtime_8_0;

  dotnet-aspnetcore_6 = dotnetCorePackages.aspnetcore_6_0;
  dotnet-aspnetcore_7 = dotnetCorePackages.aspnetcore_7_0;
  dotnet-aspnetcore_8 = dotnetCorePackages.aspnetcore_8_0;

  dotnet-sdk = dotnetCorePackages.sdk_6_0;
  dotnet-runtime = dotnetCorePackages.runtime_6_0;
  dotnet-aspnetcore = dotnetCorePackages.aspnetcore_6_0;

  buildDotnetModule = callPackage ../build-support/dotnet/build-dotnet-module { };
  nuget-to-nix = callPackage ../build-support/dotnet/nuget-to-nix { };
  mkNugetSource = callPackage ../build-support/dotnet/make-nuget-source { };
  mkNugetDeps = callPackage ../build-support/dotnet/make-nuget-deps { };

  buildDotnetGlobalTool = callPackage ../build-support/dotnet/build-dotnet-global-tool { };

  fsautocomplete = callPackage ../development/tools/fsautocomplete { };

  pbm = callPackage ../tools/admin/pbm { };

  fable = callPackage ../development/tools/fable { };

  dotnetenv = callPackage ../build-support/dotnet/dotnetenv {
    dotnetfx = dotnetfx40;
  };

  buildDotnetPackage = callPackage ../build-support/dotnet/build-dotnet-package { };
  fetchNuGet = callPackage ../build-support/dotnet/fetchnuget { };
  dotnetbuildhelpers = callPackage ../build-support/dotnet/dotnetbuildhelpers { };

  dumb-init = callPackage ../applications/virtualization/dumb-init { };

  umoci = callPackage ../applications/virtualization/umoci { };

  dippi = callPackage ../tools/graphics/dippi { };

  dispad = callPackage ../tools/X11/dispad { };

  diswall = callPackage ../applications/networking/diswall { };

  dupeguru = callPackage ../applications/misc/dupeguru { };

  dump1090 = callPackage ../applications/radio/dump1090 { };

  dwarf2json = callPackage ../tools/misc/dwarf2json { };

  ebook2cw = callPackage ../applications/radio/ebook2cw { };

  qdmr = libsForQt5.callPackage ../applications/radio/qdmr { };

  edwin = callPackage ../data/fonts/edwin { };

  edwood = callPackage ../applications/editors/edwood { };

  etBook = callPackage ../data/fonts/et-book { };

  fntsample = callPackage ../tools/misc/fntsample { };

  fetchutils = callPackage ../tools/misc/fetchutils { };

  fet-sh = callPackage ../tools/misc/fet-sh { };

  fetchbower = callPackage ../build-support/fetchbower {
    inherit (nodePackages) bower2nix;
  };

  fetchbzr = callPackage ../build-support/fetchbzr { };

  fetchcvs = if stdenv.buildPlatform != stdenv.hostPlatform
    # hack around splicing being crummy with things that (correctly) don't eval.
    then buildPackages.fetchcvs
    else callPackage ../build-support/fetchcvs { };

  fetchdarcs = callPackage ../build-support/fetchdarcs { };

  fetchdocker = callPackage ../build-support/fetchdocker { };

  fetchDockerConfig = callPackage ../build-support/fetchdocker/fetchDockerConfig.nix { };

  fetchDockerLayer = callPackage ../build-support/fetchdocker/fetchDockerLayer.nix { };

  fetchfossil = callPackage ../build-support/fetchfossil { };

  fetchgit = (callPackage ../build-support/fetchgit {
    git = buildPackages.gitMinimal;
    cacert = buildPackages.cacert;
    git-lfs = buildPackages.git-lfs;
  }) // { # fetchgit is a function, so we use // instead of passthru.
    tests = pkgs.tests.fetchgit;
  };

  fetchgitLocal = callPackage ../build-support/fetchgitlocal { };

  fetchmtn = callPackage ../build-support/fetchmtn (config.fetchmtn or {});

  fetchMavenArtifact = callPackage ../build-support/fetchmavenartifact { };

  fetchpijul = callPackage ../build-support/fetchpijul { };

  inherit (callPackages ../build-support/node/fetch-yarn-deps { })
    fixup-yarn-lock
    prefetch-yarn-deps
    fetchYarnDeps;

  find-cursor = callPackage ../tools/X11/find-cursor { };

  flare-floss = callPackage ../tools/security/flare-floss { };

  flare-signal = callPackage ../applications/networking/instant-messengers/flare-signal { };

  prefer-remote-fetch = import ../build-support/prefer-remote-fetch;

  global-platform-pro = callPackage ../development/tools/global-platform-pro { };

  gomi = callPackage ../tools/misc/gomi { };

  go-ios = callPackage ../tools/misc/go-ios { };

  graph-easy = callPackage ../tools/graphics/graph-easy { };

  mangal = callPackage ../applications/misc/mangal { };

  mermaid-filter = callPackage ../tools/misc/mermaid-filter { };

  graphw00f = callPackage ../tools/security/graphw00f { };

  graphite-cursors = callPackage ../data/icons/graphite-cursors { };

  opendrop = python3Packages.callPackage ../tools/networking/opendrop { };

  owl = callPackage ../tools/networking/owl { };

  mutmut = callPackage ../development/tools/mutmut { };

  packcc = callPackage ../development/tools/packcc { };

  packer = callPackage ../development/tools/packer { };

  packr = callPackage ../development/libraries/packr { };

  pacup = callPackage ../tools/package-management/pacup { };

  patcher9x = callPackage ../development/tools/patcher9x { };

  perseus-cli = callPackage ../development/tools/perseus-cli {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  pet = callPackage ../development/tools/pet { };

  pe-bear = libsForQt5.callPackage ../applications/misc/pe-bear { };

  pkger = callPackage ../development/libraries/pkger { };

  tapview = callPackage ../development/tools/tapview { };

  run = callPackage ../development/tools/run { };

  ruler = callPackage ../tools/security/ruler { };

  magika = with python3Packages; toPythonApplication magika;

  mblock-mlink = callPackage ../development/tools/mblock-mlink { };

  mod = callPackage ../development/tools/mod { };

  mongosh = callPackage ../development/tools/mongosh { };

  mya = callPackage ../applications/misc/mya { };

  mysql-shell = callPackage ../development/tools/mysql-shell {
    inherit (darwin) cctools DarwinTools;
    inherit (darwin.apple_sdk.frameworks) CoreServices;
    antlr = antlr4_10;
    boost = boost177; # Configure checks for specific version.
    icu =  icu73;
    protobuf = protobuf_24;
  };

  mysql-shell-innovation = callPackage ../development/tools/mysql-shell/innovation.nix {
    inherit (darwin) cctools DarwinTools;
    inherit (darwin.apple_sdk.frameworks) CoreServices;
    antlr = antlr4_10;
    icu =  icu73;
    protobuf = protobuf_24;
  };

  broadlink-cli = callPackage ../tools/misc/broadlink-cli { };

  fetchpatch = callPackage ../build-support/fetchpatch {
    # 0.3.4 would change hashes: https://github.com/NixOS/nixpkgs/issues/25154
    patchutils = buildPackages.patchutils_0_3_3;
  } // {
    tests = pkgs.tests.fetchpatch;
    version = 1;
  };

  fetchpatch2 = callPackage ../build-support/fetchpatch {
    patchutils = buildPackages.patchutils_0_4_2;
  } // {
    tests = pkgs.tests.fetchpatch2;
    version = 2;
  };

  fetchs3 = callPackage ../build-support/fetchs3 { };

  fetchtorrent = callPackage ../build-support/fetchtorrent { };

  fetchsvn = if stdenv.buildPlatform != stdenv.hostPlatform
    # hack around splicing being crummy with things that (correctly) don't eval.
    then buildPackages.fetchsvn
    else callPackage ../build-support/fetchsvn { };

  fetchsvnrevision = import ../build-support/fetchsvnrevision runCommand subversion;

  fetchsvnssh = callPackage ../build-support/fetchsvnssh { };

  fetchhg = callPackage ../build-support/fetchhg { };

  fetchFirefoxAddon = callPackage ../build-support/fetchfirefoxaddon { }
    // {
      tests = pkgs.tests.fetchFirefoxAddon;
    };

  fetchNextcloudApp = callPackage ../build-support/fetchnextcloudapp { };

  # `fetchurl' downloads a file from the network.
  fetchurl = if stdenv.buildPlatform != stdenv.hostPlatform
    then buildPackages.fetchurl # No need to do special overrides twice,
    else makeOverridable (import ../build-support/fetchurl) {
      inherit lib stdenvNoCC buildPackages;
      inherit cacert;
      curl = buildPackages.curlMinimal.override (old: rec {
        # break dependency cycles
        fetchurl = stdenv.fetchurlBoot;
        zlib = buildPackages.zlib.override { fetchurl = stdenv.fetchurlBoot; };
        pkg-config = buildPackages.pkg-config.override (old: {
          pkg-config = old.pkg-config.override {
            fetchurl = stdenv.fetchurlBoot;
          };
        });
        perl = buildPackages.perl.override { inherit zlib; fetchurl = stdenv.fetchurlBoot; };
        openssl = buildPackages.openssl.override {
          fetchurl = stdenv.fetchurlBoot;
          buildPackages = {
            coreutils = buildPackages.coreutils.override {
              fetchurl = stdenv.fetchurlBoot;
              inherit perl;
              xz = buildPackages.xz.override { fetchurl = stdenv.fetchurlBoot; };
              gmpSupport = false;
              aclSupport = false;
              attrSupport = false;
            };
            inherit perl;
          };
          inherit perl;
        };
        libssh2 = buildPackages.libssh2.override {
          fetchurl = stdenv.fetchurlBoot;
          inherit zlib openssl;
        };
        # On darwin, libkrb5 needs bootstrap_cmds which would require
        # converting many packages to fetchurl_boot to avoid evaluation cycles.
        # So turn gssSupport off there, and on Windows.
        # On other platforms, keep the previous value.
        gssSupport =
          if stdenv.isDarwin || stdenv.hostPlatform.isWindows
            then false
            else old.gssSupport or true; # `? true` is the default
        libkrb5 = buildPackages.libkrb5.override {
          fetchurl = stdenv.fetchurlBoot;
          inherit pkg-config perl openssl;
          keyutils = buildPackages.keyutils.override { fetchurl = stdenv.fetchurlBoot; };
        };
        nghttp2 = buildPackages.nghttp2.override {
          fetchurl = stdenv.fetchurlBoot;
          inherit pkg-config;
          enableApp = false; # curl just needs libnghttp2
          enableTests = false; # avoids bringing `cunit` and `tzdata` into scope
        };
      });
    };

  fetchRepoProject = callPackage ../build-support/fetchrepoproject { };

  fetchipfs = import ../build-support/fetchipfs {
    inherit curl stdenv;
  };

  fetchit = callPackage ../applications/networking/cluster/fetchit { };

  fetchzip = callPackage ../build-support/fetchzip { }
    // {
      tests = pkgs.tests.fetchzip;
    };

  fetchDebianPatch = callPackage ../build-support/fetchdebianpatch { }
    // {
      tests = pkgs.tests.fetchDebianPatch;
    };

  fetchCrate = callPackage ../build-support/rust/fetchcrate.nix { };

  fetchFromGitea = callPackage ../build-support/fetchgitea { };

  fetchFromGitHub = callPackage ../build-support/fetchgithub { };

  fetchFromBitbucket = callPackage ../build-support/fetchbitbucket { };

  fetchFromSavannah = callPackage ../build-support/fetchsavannah { };

  fetchFromSourcehut = callPackage ../build-support/fetchsourcehut { };

  fetchFromGitLab = callPackage ../build-support/fetchgitlab { };

  fetchFromGitiles = callPackage ../build-support/fetchgitiles { };

  fetchFrom9Front = callPackage ../build-support/fetch9front { };

  fetchFromRepoOrCz = callPackage ../build-support/fetchrepoorcz { };

  fetchgx = callPackage ../build-support/fetchgx { };

  fetchPypi = callPackage ../build-support/fetchpypi { };

  fetchPypiLegacy = callPackage ../build-support/fetchpypilegacy { };

  resolveMirrorURLs = {url}: fetchurl {
    showURLs = true;
    inherit url;
  };

  installShellFiles = callPackage ../build-support/install-shell-files { };

  lazydocker = callPackage ../tools/misc/lazydocker { };

  ld-is-cc-hook = makeSetupHook { name = "ld-is-cc-hook"; }
    ../build-support/setup-hooks/ld-is-cc-hook.sh;

  libgflow = callPackage ../development/libraries/libgflow { };
  libgtkflow3 = callPackage ../development/libraries/libgtkflow3 { };
  libgtkflow4 = callPackage ../development/libraries/libgtkflow4 { };

  libredirect = callPackage ../build-support/libredirect { };

  livecaptions = callPackage ../applications/misc/livecaptions { };

  madonctl = callPackage ../applications/misc/madonctl { };

  copyDesktopItems = makeSetupHook {
    name = "copy-desktop-items-hook";
  } ../build-support/setup-hooks/copy-desktop-items.sh;

  makeDesktopItem = callPackage ../build-support/make-desktopitem { };

  copyPkgconfigItems = makeSetupHook {
    name = "copy-pkg-config-items-hook";
  } ../build-support/setup-hooks/copy-pkgconfig-items.sh;

  makePkgconfigItem = callPackage ../build-support/make-pkgconfigitem { };

  makeDarwinBundle = callPackage ../build-support/make-darwin-bundle { };

  makeAutostartItem = callPackage ../build-support/make-startupitem { };

  makeImpureTest = callPackage ../build-support/make-impure-test.nix;

  makeInitrd = callPackage ../build-support/kernel/make-initrd.nix; # Args intentionally left out

  makeInitrdNG = callPackage ../build-support/kernel/make-initrd-ng.nix;
  makeInitrdNGTool = callPackage ../build-support/kernel/make-initrd-ng-tool.nix { };

  makeWrapper = makeShellWrapper;

  makeShellWrapper = makeSetupHook {
    name = "make-shell-wrapper-hook";
    propagatedBuildInputs = [ dieHook ];
    substitutions = {
      # targetPackages.runtimeShell only exists when pkgs == targetPackages (when targetPackages is not  __raw)
      shell = if targetPackages ? runtimeShell then targetPackages.runtimeShell else throw "makeWrapper/makeShellWrapper must be in nativeBuildInputs";
    };
    passthru = {
      tests = tests.makeWrapper;
    };
  } ../build-support/setup-hooks/make-wrapper.sh;

  makeBinaryWrapper = callPackage ../build-support/setup-hooks/make-binary-wrapper { };

  compressFirmwareXz = callPackage ../build-support/kernel/compress-firmware.nix { type = "xz"; };

  compressFirmwareZstd = callPackage ../build-support/kernel/compress-firmware.nix { type = "zstd"; };

  makeModulesClosure = { kernel, firmware, rootModules, allowMissing ? false }:
    callPackage ../build-support/kernel/modules-closure.nix {
      inherit kernel firmware rootModules allowMissing;
    };

  mkBinaryCache = callPackage ../build-support/binary-cache { };

  mkShell = callPackage ../build-support/mkshell { };
  mkShellNoCC = mkShell.override { stdenv = stdenvNoCC; };

  mokutil = callPackage ../tools/security/mokutil { };

  mpsolve = libsForQt5.callPackage ../applications/science/math/mpsolve { };

  nixBufferBuilders = import ../build-support/emacs/buffer.nix { inherit lib writeText; inherit (emacs.pkgs) inherit-local; };

  nix-gitignore = callPackage ../build-support/nix-gitignore { };

  numworks-epsilon = callPackage ../applications/science/math/numworks-epsilon { };

  ociTools = callPackage ../build-support/oci-tools { };

  inherit (
    callPackages ../build-support/setup-hooks/patch-rc-path-hooks { }
  ) patchRcPathBash patchRcPathCsh patchRcPathFish patchRcPathPosix;

  pathsFromGraph = ../build-support/kernel/paths-from-graph.pl;

  pruneLibtoolFiles = makeSetupHook { name = "prune-libtool-files"; }
    ../build-support/setup-hooks/prune-libtool-files.sh;

  closureInfo = callPackage ../build-support/closure-info.nix { };

  sensible-utils = callPackage ../tools/misc/sensible-utils { };

  serverspec = callPackage ../tools/misc/serverspec { };

  setupSystemdUnits = callPackage ../build-support/setup-systemd-units.nix { };

  shortenPerlShebang = makeSetupHook {
    name = "shorten-perl-shebang-hook";
    propagatedBuildInputs = [ dieHook ];
  } ../build-support/setup-hooks/shorten-perl-shebang.sh;

  singularity-tools = callPackage ../build-support/singularity-tools { };

  stacktile = callPackage ../tools/wayland/stacktile { };

  sirula = callPackage ../tools/wayland/sirula { };

  sitelen-seli-kiwen = callPackage ../data/fonts/sitelen-seli-kiwen { };

  srcOnly = callPackage ../build-support/src-only { };

  subnetcalc = callPackage ../tools/networking/subnetcalc { };

  substitute = callPackage ../build-support/substitute/substitute.nix { };

  substituteAll = callPackage ../build-support/substitute/substitute-all.nix { };

  substituteAllFiles = callPackage ../build-support/substitute-files/substitute-all-files.nix { };

  replaceDependency = callPackage ../build-support/replace-dependency.nix { };

  nukeReferences = callPackage ../build-support/nuke-references {
    inherit (darwin) signingUtils;
  };

  referencesByPopularity = callPackage ../build-support/references-by-popularity { };

  removeReferencesTo = callPackage ../build-support/remove-references-to {
    inherit (darwin) signingUtils;
  };

  # No callPackage.  In particular, we don't want `img` *package* in parameters.
  vmTools = makeOverridable (import ../build-support/vm) { inherit pkgs lib; };

  releaseTools = callPackage ../build-support/release { };

  inherit (lib.systems) platforms;

  setJavaClassPath = makeSetupHook {
    name = "set-java-classpath-hook";
  } ../build-support/setup-hooks/set-java-classpath.sh;

  fixDarwinDylibNames = makeSetupHook {
    name = "fix-darwin-dylib-names-hook";
    substitutions = { inherit (binutils) targetPrefix; };
    meta.platforms = lib.platforms.darwin;
  } ../build-support/setup-hooks/fix-darwin-dylib-names.sh;

  writeDarwinBundle = callPackage ../build-support/make-darwin-bundle/write-darwin-bundle.nix { };

  desktopToDarwinBundle = makeSetupHook {
    name = "desktop-to-darwin-bundle-hook";
    propagatedBuildInputs = [ writeDarwinBundle librsvg imagemagick python3Packages.icnsutil ];
  } ../build-support/setup-hooks/desktop-to-darwin-bundle.sh;

  keepBuildTree = makeSetupHook {
    name = "keep-build-tree-hook";
  } ../build-support/setup-hooks/keep-build-tree.sh;

  moveBuildTree = makeSetupHook {
    name = "move-build-tree-hook";
  } ../build-support/setup-hooks/move-build-tree.sh;

  enableGCOVInstrumentation = makeSetupHook {
    name = "enable-gcov-instrumentation-hook";
  } ../build-support/setup-hooks/enable-coverage-instrumentation.sh;

  makeGCOVReport = makeSetupHook {
    name = "make-gcov-report-hook";
    propagatedBuildInputs = [ lcov enableGCOVInstrumentation ];
  } ../build-support/setup-hooks/make-coverage-analysis-report.sh;

  makeHardcodeGsettingsPatch = callPackage ../build-support/make-hardcode-gsettings-patch { };

  # intended to be used like nix-build -E 'with import <nixpkgs> { }; enableDebugging fooPackage'
  enableDebugging = pkg: pkg.override { stdenv = stdenvAdapters.keepDebugInfo pkg.stdenv; };

  findXMLCatalogs = makeSetupHook {
    name = "find-xml-catalogs-hook";
  } ../build-support/setup-hooks/find-xml-catalogs.sh;

  wrapGAppsHook3 = wrapGAppsNoGuiHook.override {
    isGraphical = true;
  };

  wrapGAppsHook4 = wrapGAppsNoGuiHook.override {
    isGraphical = true;
    gtk3 = __splicedPackages.gtk4;
  };

  wrapGAppsNoGuiHook = callPackage ../build-support/setup-hooks/wrap-gapps-hook {
    makeWrapper = makeBinaryWrapper;
  };

  separateDebugInfo = makeSetupHook {
    name = "separate-debug-info-hook";
  } ../build-support/setup-hooks/separate-debug-info.sh;

  setupDebugInfoDirs = makeSetupHook {
    name = "setup-debug-info-dirs-hook";
  } ../build-support/setup-hooks/setup-debug-info-dirs.sh;

  useOldCXXAbi = makeSetupHook {
    name = "use-old-cxx-abi-hook";
  } ../build-support/setup-hooks/use-old-cxx-abi.sh;

  iconConvTools = callPackage ../build-support/icon-conv-tools { };

  validatePkgConfig = makeSetupHook
    { name = "validate-pkg-config"; propagatedBuildInputs = [ findutils pkg-config ]; }
    ../build-support/setup-hooks/validate-pkg-config.sh;

  patchPpdFilesHook = callPackage ../build-support/setup-hooks/patch-ppd-files { };

  #package writers
  writers = callPackage ../build-support/writers { };

  # lib functions depending on pkgs
  inherit (import ../pkgs-lib {
    # The `lib` variable in this scope doesn't include any applied lib overlays,
    # `pkgs.lib` does.
    inherit (pkgs) lib;
    inherit pkgs;
  }) formats;

  testers = callPackage ../build-support/testers { };

  ### TOOLS

  _3llo = callPackage ../tools/misc/3llo { };

  _3mux = callPackage ../tools/misc/3mux { };

  _1password = callPackage ../applications/misc/1password { };

  _1password-gui = callPackage ../applications/misc/1password-gui { };

  _1password-gui-beta = callPackage ../applications/misc/1password-gui { channel = "beta"; };

  _6tunnel = callPackage ../tools/networking/6tunnel { };

  _7zz = darwin.apple_sdk_11_0.callPackage ../tools/archivers/7zz { };

  _9pfs = callPackage ../tools/filesystems/9pfs { };

  _9ptls = callPackage ../os-specific/linux/9ptls { };

  aaa = callPackage ../tools/misc/aaa { };

  aardvark-dns = callPackage ../tools/networking/aardvark-dns { };

  a2ps = callPackage ../tools/text/a2ps { };

  abbreviate = callPackage ../tools/text/abbreviate { };

  abcm2ps = callPackage ../tools/audio/abcm2ps { };

  abcmidi = callPackage ../tools/audio/abcmidi { };

  abduco = callPackage ../tools/misc/abduco { };

  acct = callPackage ../tools/system/acct { };

  accuraterip-checksum = callPackage ../tools/audio/accuraterip-checksum { };

  acme-dns = callPackage ../servers/dns/acme-dns/default.nix { };

  acme-sh = callPackage ../tools/admin/acme-sh { };

  acousticbrainz-client = callPackage ../tools/audio/acousticbrainz-client { };

  alsaequal = callPackage ../tools/audio/alsaequal { };

  acpica-tools = callPackage ../tools/system/acpica-tools { };

  amdgpu_top = callPackage ../tools/system/amdgpu_top { };

  acquire = with python3Packages; toPythonApplication acquire;

  act = callPackage ../development/tools/misc/act { };

  actdiag = with python3.pkgs; toPythonApplication actdiag;

  actkbd = callPackage ../tools/system/actkbd { };

  adafruit-ampy = callPackage ../tools/misc/adafruit-ampy { };

  addic7ed-cli = callPackage ../tools/misc/addic7ed-cli { };

  addlicense = callPackage ../tools/misc/addlicense { };

  adenum  = callPackage ../tools/security/adenum { };

  adlplug = callPackage ../applications/audio/adlplug {
    inherit (darwin.apple_sdk.frameworks) Foundation Cocoa Carbon CoreServices ApplicationServices CoreAudio CoreMIDI AudioToolbox Accelerate CoreImage IOKit AudioUnit QuartzCore WebKit DiscRecording CoreAudioKit;
    jack = libjack2;
  };
  opnplug = adlplug.override {
    type = "OPN";
  };

  arc_unpacker = callPackage ../tools/archivers/arc_unpacker { };

  adminer = callPackage ../servers/adminer { };

  akkoma = callPackage ../servers/akkoma {
    elixir = elixir_1_16;
    beamPackages = beamPackages.extend (self: super: { elixir = elixir_1_16; });
  };
  akkoma-frontends = recurseIntoAttrs {
    akkoma-fe = callPackage ../servers/akkoma/akkoma-fe { };
    admin-fe = callPackage ../servers/akkoma/admin-fe {
      nodejs = nodejs_18;
      yarn = yarn.override { nodejs = nodejs_18; };
    };
  };
  akkoma-emoji = recurseIntoAttrs {
    blobs_gg = callPackage ../servers/akkoma/emoji/blobs_gg.nix { };
  };

  advancecomp = callPackage ../tools/compression/advancecomp { };

  aefs = callPackage ../tools/filesystems/aefs { };

  aegisub = callPackage ../by-name/ae/aegisub/package.nix ({
    boost = boost179;
    luajit = luajit.override { enable52Compat = true; };
    wxGTK = wxGTK32;
  } // (config.aegisub or {}));

  aerc = callPackage ../applications/networking/mailreaders/aerc { };

  aerospike = callPackage ../servers/nosql/aerospike { };

  aespipe = callPackage ../tools/security/aespipe { };

  aescrypt = callPackage ../tools/misc/aescrypt { };

  aether-lv2 = callPackage ../applications/audio/aether-lv2 { };

  acme-client = callPackage ../tools/networking/acme-client {
    stdenv = gccStdenv;
  };

  adrgen = callPackage ../tools/misc/adrgen { };

  adriconf = callPackage ../tools/graphics/adriconf { };

  amass = callPackage ../tools/networking/amass { };

  afew = callPackage ../applications/networking/mailreaders/afew { };

  afetch = callPackage ../tools/misc/afetch { };

  afio = callPackage ../tools/archivers/afio { };

  honggfuzz = callPackage ../tools/security/honggfuzz {
    clang = clang_16;
    llvm = llvm_16;
  };

  aflplusplus = callPackage ../tools/security/aflplusplus {
    clang = clang_15;
    llvm = llvm_15;
    llvmPackages = llvmPackages_15;
    wine = null;
  };

  ledfx = callPackage ../applications/audio/ledfx { };

  libdislocator = callPackage ../tools/security/aflplusplus/libdislocator.nix { };

  afpfs-ng = callPackage ../tools/filesystems/afpfs-ng { };

  afsctool = callPackage ../tools/filesystems/afsctool {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  agate = callPackage ../servers/gemini/agate {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  agda-pkg = callPackage ../development/tools/agda-pkg { };

  agi = callPackage ../tools/graphics/agi { };

  agola = callPackage ../development/tools/continuous-integration/agola { };

  agrep = callPackage ../tools/text/agrep { };

  aha = callPackage ../tools/text/aha { };

  ahcpd = callPackage ../tools/networking/ahcpd { };

  aichat = callPackage ../tools/misc/aichat { };

  aide = callPackage ../tools/security/aide { };

  ain = callPackage ../tools/networking/ain { };

  aioblescan = with python3Packages; toPythonApplication aioblescan;

  aiodnsbrute = callPackage ../tools/security/aiodnsbrute { };

  aircrack-ng = callPackage ../tools/networking/aircrack-ng { };

  airgeddon = callPackage ../tools/networking/airgeddon { };

  apache-airflow = callPackage ../servers/apache-airflow { };

  apachetomcatscanner = callPackage ../tools/security/apachetomcatscanner { };

  airsonic = callPackage ../servers/misc/airsonic { };

  airspy = callPackage ../applications/radio/airspy { };

  airspyhf = callPackage ../applications/radio/airspyhf { };

  airwindows-lv2 = callPackage ../applications/audio/airwindows-lv2 { };

  aixlog = callPackage ../development/libraries/aixlog { };

  aj-snapshot  = callPackage ../applications/audio/aj-snapshot { };

  ajour = callPackage ../tools/games/ajour {
    inherit (gnome) zenity;
    inherit (plasma5Packages) kdialog;
  };

  akku = callPackage ../tools/package-management/akku { };

  albert = qt6Packages.callPackage ../applications/misc/albert { };

  alice-lg = callPackage ../servers/alice-lg{ };

  alice-tools = callPackage ../tools/games/alice-tools {
    withGUI = false;
  };

  alice-tools-qt5 = libsForQt5.callPackage ../tools/games/alice-tools { };

  alice-tools-qt6 = qt6Packages.callPackage ../tools/games/alice-tools { };

  allure = callPackage ../development/tools/allure { };

  aquosctl = callPackage ../tools/misc/aquosctl { };

  arch-install-scripts = callPackage ../tools/misc/arch-install-scripts { };

  argocd-vault-plugin = callPackage ../tools/security/argocd-vault-plugin { };

  arubaotp-seed-extractor = callPackage ../tools/security/arubaotp-seed-extractor { };

  auditwheel = with python3Packages; toPythonApplication auditwheel;

  amidst = callPackage ../tools/games/minecraft/amidst { };

  asar = callPackage ../tools/compression/asar { };

  askalono = callPackage ../tools/misc/askalono { };

  asleap = callPackage ../tools/networking/asleap { };

  awsbck = callPackage ../tools/backup/awsbck {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  balena-cli = callPackage ../tools/admin/balena-cli { };

  basez = callPackage ../tools/misc/basez { };

  btc-rpc-explorer = callPackage ../tools/misc/btc-rpc-explorer { };

  butler = callPackage ../by-name/bu/butler/package.nix {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  carbon-now-cli = callPackage ../tools/typesetting/carbon-now-cli { };

  cf-vault = callPackage ../tools/admin/cf-vault { };

  cfspeedtest = callPackage ../tools/networking/cfspeedtest { };

  cfonts = callPackage ../tools/misc/cfonts { };

  bikeshed = python3Packages.callPackage ../applications/misc/bikeshed { };

  cie-middleware-linux = callPackage ../tools/security/cie-middleware-linux { };

  cidrgrep = callPackage ../tools/text/cidrgrep { };

  clematis = callPackage ../tools/misc/clematis { };

  clubhouse-cli = callPackage ../tools/misc/clubhouse-cli { };

  colorless = callPackage ../tools/misc/colorless { };

  configurable-http-proxy = callPackage ../tools/networking/configurable-http-proxy { };

  crypto-tracker = callPackage ../tools/misc/crypto-tracker { };

  ejs = callPackage ../development/tools/ejs { };

  ejson2env = callPackage ../tools/admin/ejson2env { };

  davinci-resolve = callPackage ../applications/video/davinci-resolve { };

  davinci-resolve-studio = callPackage ../applications/video/davinci-resolve { studioVariant = true; };

  dehinter = with python3Packages; toPythonApplication dehinter;

  dolbybcsoftwaredecode = callPackage ../applications/audio/dolbybcsoftwaredecode { };

  donkey = callPackage ../tools/security/donkey { };

  dwarfs = callPackage ../tools/filesystems/dwarfs { };

  dysk = callPackage ../tools/filesystems/dysk { };

  fscan = callPackage ../tools/security/fscan { };

  copier = callPackage ../tools/misc/copier { };

  gabutdm = callPackage ../applications/networking/gabutdm { };

  gamemode = callPackage ../tools/games/gamemode {
    libgamemode32 = pkgsi686Linux.gamemode.lib;
  };

  gamescope = callPackage ../by-name/ga/gamescope/package.nix {
    enableExecutable = true;
    enableWsi = false;

    wlroots = wlroots_0_17;
  };

  gamescope-wsi = callPackage ../by-name/ga/gamescope/package.nix {
    enableExecutable = false;
    enableWsi = true;

    wlroots = wlroots_0_17;
  };

  gay = callPackage ../tools/misc/gay {  };

  elkhound = callPackage ../development/tools/elkhound { };

  ethq = callPackage ../os-specific/linux/ethq { };

  font-v = with python3Packages; toPythonApplication font-v;

  fontbakery = with python3Packages; toPythonApplication fontbakery;

  weidu = callPackage ../tools/games/weidu { };

  weylus = callPackage ../applications/graphics/weylus  {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Carbon Cocoa VideoToolbox;
  };

  gam = callPackage ../tools/admin/gam { };

  gen6dns = callPackage ../tools/networking/gen6dns { };

  gen-license = callPackage ../development/tools/gen-license {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  github-copilot-cli = callPackage ../tools/misc/github-copilot-cli { };

  # This is to workaround gfal2-python broken against Python 3.12 or later.
  # TODO: Remove these lines after solving the breakage.
  gfal2-util = callPackage ../by-name/gf/gfal2-util/package.nix (lib.optionalAttrs python3Packages.gfal2-python.meta.broken {
    python3Packages = python311Packages;
  });

  gfshare = callPackage ../tools/security/gfshare { };

  gh-actions-cache = callPackage ../tools/misc/gh-actions-cache { };

  gh-cal = callPackage ../tools/misc/gh-cal {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  gh-dash = callPackage ../tools/misc/gh-dash { };

  gh-markdown-preview = callPackage ../tools/misc/gh-markdown-preview { };

  ghostie = callPackage ../tools/misc/ghostie { };

  glooctl = callPackage ../applications/networking/cluster/glooctl { };

  gobgp = callPackage ../tools/networking/gobgp { };

  gof5 = callPackage ../tools/networking/gof5 { };

  goflow = callPackage ../tools/networking/goflow { };

  gofu = callPackage ../applications/misc/gofu { };

  godspeed = callPackage ../tools/networking/godspeed { };

  goodhosts = callPackage ../tools/networking/goodhosts { };

  goose = callPackage ../tools/misc/goose { };

  gp-saml-gui = python3Packages.callPackage ../tools/networking/gp-saml-gui { };

  grizzly = callPackage ../tools/misc/grizzly { };

  guestfs-tools = callPackage ../tools/virtualization/guestfs-tools {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  fabs = callPackage ../tools/backup/fabs { };

  fm = callPackage ../applications/file-managers/fm { };

  fwbuilder = libsForQt5.callPackage ../tools/security/fwbuilder { };

  hblock = callPackage ../tools/networking/hblock { };

  headsetcontrol = callPackage ../tools/audio/headsetcontrol { };

  headset-charge-indicator = callPackage ../tools/audio/headset-charge-indicator { };

  heh = callPackage ../applications/editors/heh { };

  hexdiff = callPackage ../tools/misc/hexdiff { };

  hexo-cli = callPackage ../development/tools/hexo-cli { };

  hostmux = callPackage ../tools/misc/hostmux { };

  hsd = callPackage ../tools/misc/hsd { };

  httm = darwin.apple_sdk_11_0.callPackage ../tools/filesystems/httm { };

  hyperpotamus = callPackage ../tools/misc/hyperpotamus { };

  immich-cli = callPackage ../tools/misc/immich-cli { };

  inherit (callPackages ../tools/networking/ivpn/default.nix {}) ivpn ivpn-service;

  jobber = callPackage ../tools/system/jobber { };

  kanata = callPackage ../tools/system/kanata { };

  kanata-with-cmd = kanata.override { withCmd = true; };

  kaufkauflist = callPackage ../applications/misc/kaufkauflist { };

  kdocker = libsForQt5.callPackage ../tools/X11/kdocker { };

  kicli = callPackage ../tools/misc/kicli { };

  ksmbd-tools = callPackage ../os-specific/linux/ksmbd-tools { };

  ksnip = libsForQt5.callPackage ../tools/misc/ksnip { };

  kstart = callPackage ../tools/security/kstart { };

  kubevirt = callPackage ../tools/virtualization/kubevirt { };

  lektor = callPackage ../tools/misc/lektor { };

  licenseclassifier = callPackage ../development/tools/misc/licenseclassifier { };

  license-cli = callPackage ../tools/misc/license-cli { };

  license-generator = callPackage ../tools/misc/license-generator { };

  linux-router = callPackage ../tools/networking/linux-router { };

  linux-router-without-wifi = linux-router.override { useWifiDependencies = false; };

  makehuman = libsForQt5.callPackage ../applications/misc/makehuman { };

  markdownlint-cli = callPackage ../tools/text/markdownlint-cli { };

  markdownlint-cli2 = callPackage ../tools/text/markdownlint-cli2 { };

  mbidled = callPackage ../tools/networking/mbidled { };

  mcaselector = callPackage ../tools/games/minecraft/mcaselector {
    jre = jre.override {
      enableJavaFX = true;
    };
  };

  metapixel = callPackage ../tools/graphics/metapixel { };

  memos = callPackage ../servers/memos { };

  midimonster = callPackage ../tools/audio/midimonster { };

  midi-trigger = callPackage ../applications/audio/midi-trigger { };

  mnc = callPackage ../tools/misc/mnc { };

  mgmt = callPackage ../applications/system/mgmt { };

  mkosi = python3Packages.callPackage ../tools/virtualization/mkosi { inherit systemd; };

  mkosi-full = mkosi.override { withQemu = true; };

  monica = callPackage ../servers/web-apps/monica { };

  mpremote = python3Packages.callPackage ../tools/misc/mpremote { };

  mprocs = callPackage ../tools/misc/mprocs { };

  mpy-utils = python3Packages.callPackage ../tools/misc/mpy-utils { };

  mymcplus = python3Packages.callPackage ../tools/games/mymcplus { };

  near-cli = callPackage ../tools/misc/near-cli { };

  networkd-notify = python3Packages.callPackage ../tools/networking/networkd-notify {
    systemd = pkgs.systemd;
  };

  node-glob = callPackage ../tools/misc/node-glob { };

  nominatim = callPackage ../servers/nominatim {
    postgresql = postgresql_14;
  };

  npm-check-updates = callPackage ../tools/package-management/npm-check-updates { };

  ntpd-rs = darwin.apple_sdk_11_0.callPackage ../tools/networking/ntpd-rs {
    inherit (darwin.apple_sdk_11_0.frameworks) Security;
  };

  ocs-url = libsForQt5.callPackage ../tools/misc/ocs-url { };

  openbugs = pkgsi686Linux.callPackage ../applications/science/machine-learning/openbugs { };

  openusd = python3Packages.openusd.override {
    withTools = true;
    withUsdView = true;
  };

  osquery = callPackage ../tools/system/osquery { };

  paperview = callPackage ../tools/X11/paperview { };

  pferd = callPackage ../tools/misc/pferd { };

  polygon-cli = callPackage ../tools/networking/polygon-cli { };

  pricehist = python3Packages.callPackage ../tools/misc/pricehist { };

  proycon-wayout = callPackage ../tools/wayland/proycon-wayout { };

  q = callPackage ../tools/networking/q { };

  qFlipper = libsForQt5.callPackage ../tools/misc/qflipper { };

  quich = callPackage ../tools/misc/quich { } ;

  rabtap = callPackage ../tools/networking/rabtap { } ;

  redfang = callPackage ../tools/networking/redfang { };

  ronin = callPackage ../tools/security/ronin { };

  scarab = callPackage ../tools/games/scarab { };

  sdbus-cpp = callPackage ../development/libraries/sdbus-cpp { };

  basu = callPackage ../development/libraries/basu { };

  sdlookup = callPackage ../tools/security/sdlookup { };

  sdkmanager = with python3Packages; toPythonApplication sdkmanager;

  sgrep = callPackage ../tools/text/sgrep { };

  shaperglot = with python3Packages; toPythonApplication shaperglot;

  shell-genie = callPackage  ../applications/misc/shell-genie { };

  sloth = callPackage ../tools/misc/sloth { };

  snagboot = python3.pkgs.callPackage  ../applications/misc/snagboot { };

  simple-dlna-browser = callPackage ../tools/networking/simple-dlna-browser { };

  sitespeed-io = callPackage ../tools/networking/sitespeed-io { };

  slipstream = callPackage ../tools/games/slipstream {
    jdk = jdk8;
  };

  sorted-grep = callPackage ../tools/text/sorted-grep { };

  smbmap = callPackage ../tools/security/smbmap { };

  smbscan = callPackage ../tools/security/smbscan { };

  spectre-cli = callPackage ../tools/security/spectre-cli { };

  speedtest-go = callPackage ../tools/networking/speedtest-go { };

  speedtest-rs = callPackage ../tools/networking/speedtest-rs { };

  stargazer = callPackage ../servers/gemini/stargazer {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  steamtinkerlaunch = callPackage ../tools/games/steamtinkerlaunch { };

  supercronic = callPackage ../tools/system/supercronic { };

  supermin = callPackage ../tools/virtualization/supermin {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  sx-go = callPackage ../tools/security/sx-go { };

  systeroid = callPackage ../tools/system/systeroid { };

  tailwindcss = callPackage ../development/tools/tailwindcss { };

  tauon = callPackage ../applications/audio/tauon { };

  tere = callPackage ../tools/misc/tere { };

  termusic = darwin.apple_sdk_11_0.callPackage ../applications/audio/termusic {
    inherit (darwin.apple_sdk_11_0.frameworks) AppKit CoreAudio CoreGraphics Foundation IOKit MediaPlayer Security;
  };

  tfk8s = callPackage ../tools/misc/tfk8s { };

  thumbs = callPackage ../tools/misc/thumbs { };

  tnat64 = callPackage ../tools/networking/tnat64 { };

  toast = callPackage ../development/tools/toast { };

  topicctl = callPackage ../tools/misc/topicctl { };

  transmission-rss = callPackage ../tools/networking/transmission-rss { };

  trimage = callPackage ../applications/graphics/trimage { inherit (qt5) wrapQtAppsHook; };

  ttchat = callPackage ../tools/misc/ttchat { };

  ufolint = with python3Packages; toPythonApplication ufolint;

  ukmm = callPackage ../tools/games/ukmm { };

  unflac = callPackage ../tools/audio/unflac { };

  valeronoi = qt6Packages.callPackage ../tools/misc/valeronoi { };

  veikk-linux-driver-gui = libsForQt5.callPackage ../tools/misc/veikk-linux-driver-gui { };

  ventoy-full = ventoy.override {
    withCryptsetup = true;
    withXfs = true;
    withExt4 = true;
    withNtfs = true;
  };

  vitess = callPackage ../development/tools/database/vitess { };

  vopono = callPackage ../tools/networking/vopono { };

  vprof = with python3Packages; toPythonApplication vprof;

  vrc-get = callPackage ../tools/misc/vrc-get {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  vrrtest = callPackage ../tools/video/vrrtest { };

  winbox = callPackage ../tools/admin/winbox {
    wine = wineWowPackages.stable;
  };

  wtwitch = callPackage ../tools/video/wtwitch { };

  wwcd = callPackage ../tools/misc/wwcd { };

  writedisk = callPackage ../tools/misc/writedisk { };

  xcd = callPackage ../tools/misc/xcd { };

  xpaste = callPackage ../tools/text/xpaste { };

  xrootd = callPackage ../tools/networking/xrootd {
    # Workaround systemd static build breakage
    systemd = if systemd.meta.broken then null else systemd;
  };

  yabridge = callPackage ../tools/audio/yabridge {
    wine = wineWowPackages.staging;
  };

  yabridgectl = callPackage ../tools/audio/yabridgectl {
    wine = wineWowPackages.staging;
  };

  yafetch = callPackage ../tools/misc/yafetch {
    stdenv = clangStdenv;
  };

  yarn-lock-converter = callPackage ../tools/package-management/yarn-lock-converter { };

  zsh-history-to-fish = callPackage ../tools/misc/zsh-history-to-fish { };

  archi = callPackage ../tools/misc/archi { };

  breitbandmessung = callPackage ../applications/networking/breitbandmessung { };

  ### APPLICATIONS/VERSION-MANAGEMENT

  deepgit = callPackage ../applications/version-management/deepgit { };

  git = callPackage ../applications/version-management/git {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
    perlLibs = [perlPackages.LWP perlPackages.URI perlPackages.TermReadKey];
    smtpPerlLibs = [
      perlPackages.libnet perlPackages.NetSMTPSSL
      perlPackages.IOSocketSSL perlPackages.NetSSLeay
      perlPackages.AuthenSASL perlPackages.DigestHMAC
    ];
  };

  # The full-featured Git.
  gitFull = git.override {
    svnSupport = true;
    guiSupport = true;
    sendEmailSupport = true;
    withSsh = true;
    withLibsecret = !stdenv.isDarwin;
  };

  # Git with SVN support, but without GUI.
  gitSVN = lowPrio (git.override { svnSupport = true; });

  git-doc = lib.addMetaAttrs {
    description = "Additional documentation for Git";
    longDescription = ''
      This package contains additional documentation (HTML and text files) that
      is referenced in the man pages of Git.
    '';
  } gitFull.doc;

  gitMinimal = git.override {
    withManual = false;
    pythonSupport = false;
    perlSupport = false;
    withpcre2 = false;
  };

  bfg-repo-cleaner = callPackage ../applications/version-management/bfg-repo-cleaner { };

  bit = callPackage ../applications/version-management/bit { };

  bitbucket-server-cli = callPackage ../applications/version-management/bitbucket-server-cli { };

  bump2version = python3Packages.callPackage ../applications/version-management/bump2version { };

  cgit = callPackage ../applications/version-management/cgit { };

  cgit-pink = callPackage ../applications/version-management/cgit/pink.nix { };

  commit-formatter = callPackage ../applications/version-management/commit-formatter { };

  commitlint = nodePackages."@commitlint/cli";

  conform = callPackage ../applications/version-management/conform {
    buildGoModule = buildGo122Module;
  };

  datalad = callPackage ../applications/version-management/datalad { };

  darcs-to-git = callPackage ../applications/version-management/darcs-to-git { };

  degit = callPackage ../applications/version-management/degit { };

  delta = darwin.apple_sdk_11_0.callPackage ../applications/version-management/delta { };

  diff-so-fancy = callPackage ../applications/version-management/diff-so-fancy { };

  gex = callPackage ../applications/version-management/gex {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  gfold = callPackage ../applications/version-management/gfold {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  gita = python3Packages.callPackage ../applications/version-management/gita { };

  gitoxide = darwin.apple_sdk_11_0.callPackage ../applications/version-management/gitoxide {
    inherit (darwin.apple_sdk_11_0.frameworks) Security SystemConfiguration;
  };

  gg-scm = callPackage ../applications/version-management/gg { };

  github-cli = gh;
  gh = callPackage ../applications/version-management/gh { };

  ghorg = callPackage ../applications/version-management/ghorg { };

  ghq = callPackage ../applications/version-management/ghq { };

  ghr = callPackage ../applications/version-management/ghr { };

  git-absorb = callPackage ../applications/version-management/git-absorb {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  git-aggregator = callPackage ../applications/version-management/git-aggregator { };

  git-annex-metadata-gui = libsForQt5.callPackage ../applications/version-management/git-annex-metadata-gui {
    inherit (python3Packages) buildPythonApplication pyqt5 git-annex-adapter;
  };

  git-annex-remote-dbx = callPackage ../applications/version-management/git-annex-remote-dbx {
    inherit (python3Packages)
    buildPythonApplication
    dropbox
    annexremote
    humanfriendly;
  };

  git-annex-remote-googledrive = callPackage ../applications/version-management/git-annex-remote-googledrive {
    inherit (python3Packages)
    buildPythonApplication
    annexremote
    drivelib
    gitpython
    tenacity
    humanfriendly;
  };

  git-annex-remote-rclone = callPackage ../applications/version-management/git-annex-remote-rclone { };

  git-annex-utils = callPackage ../applications/version-management/git-annex-utils { };

  git-appraise = callPackage ../applications/version-management/git-appraise { };

  git-archive-all = python3.pkgs.callPackage ../applications/version-management/git-archive-all { };

  git-backdate = callPackage ../applications/version-management/git-backdate { };

  git-backup = callPackage ../applications/version-management/git-backup {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  git-big-picture = callPackage ../applications/version-management/git-big-picture { };

  git-bars = callPackage ../applications/version-management/git-bars { };

  git-branchless = callPackage ../applications/version-management/git-branchless {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  git-branchstack = python3.pkgs.callPackage ../applications/version-management/git-branchstack { };

  git-bug = callPackage ../applications/version-management/git-bug { };

  git-bug-migration = callPackage ../applications/version-management/git-bug-migration { };

  git-cache = callPackage ../applications/version-management/git-cache { };

  git-chglog = callPackage ../applications/version-management/git-chglog { };

  git-cinnabar = callPackage ../applications/version-management/git-cinnabar {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  git-cliff = callPackage ../applications/version-management/git-cliff {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  git-codeowners = callPackage ../applications/version-management/git-codeowners { };

  git-codereview = callPackage ../applications/version-management/git-codereview { };

  git-cola = callPackage ../applications/version-management/git-cola { };

  git-crecord = callPackage ../applications/version-management/git-crecord { };

  git-credential-keepassxc = darwin.apple_sdk_11_0.callPackage ../applications/version-management/git-credential-keepassxc {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration Foundation;
  };

  git-credential-manager = callPackage ../applications/version-management/git-credential-manager { };

  git-credential-oauth = callPackage ../applications/version-management/git-credential-oauth { };

  git-crypt = callPackage ../applications/version-management/git-crypt { };

  git-delete-merged-branches = callPackage ../applications/version-management/git-delete-merged-branches { };

  git-dive = callPackage ../applications/version-management/git-dive { };

  git-extras = callPackage ../applications/version-management/git-extras { };

  git-fame = callPackage ../applications/version-management/git-fame { };

  git-fast-export = callPackage ../applications/version-management/fast-export { };

  git-fire = callPackage ../applications/version-management/git-fire { };

  git-ftp = callPackage ../applications/version-management/git-ftp { };

  git-gone = callPackage ../applications/version-management/git-gone {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  git-graph = callPackage ../applications/version-management/git-graph { };

  git-hound = callPackage ../applications/version-management/git-hound { };

  git-hub = callPackage ../applications/version-management/git-hub { };

  git-ignore = callPackage ../applications/version-management/git-ignore { };

  git-imerge = python3Packages.callPackage ../applications/version-management/git-imerge { };

  git-interactive-rebase-tool = callPackage ../applications/version-management/git-interactive-rebase-tool {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  git-lfs = lowPrio (callPackage ../applications/version-management/git-lfs { });

  git-mit = callPackage ../applications/version-management/git-mit { };

  git-my = callPackage ../applications/version-management/git-my { };

  git-machete = python3Packages.callPackage ../applications/version-management/git-machete { };

  git-nomad = callPackage ../applications/version-management/git-nomad { };

  git-octopus = callPackage ../applications/version-management/git-octopus { };

  git-open = callPackage ../applications/version-management/git-open { };

  git-privacy = callPackage ../applications/version-management/git-privacy { };

  git-ps-rs = callPackage ../development/tools/git-ps-rs {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  git-publish = python3Packages.callPackage ../applications/version-management/git-publish { };

  git-quick-stats = callPackage ../applications/version-management/git-quick-stats { };

  git-quickfix = callPackage ../applications/version-management/git-quickfix {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  git-radar = callPackage ../applications/version-management/git-radar { };

  git-recent = callPackage ../applications/version-management/git-recent {
    util-linux = if stdenv.isLinux then util-linuxMinimal else util-linux;
  };

  git-relevant-history = callPackage ../applications/version-management/git-relevant-history { };

  git-remote-codecommit = python3Packages.callPackage ../applications/version-management/git-remote-codecommit { };

  gitRepo = git-repo;
  git-repo = callPackage ../applications/version-management/git-repo { };

  git-repo-updater = python3Packages.callPackage ../applications/version-management/git-repo-updater { };

  git-review = python3Packages.callPackage ../applications/version-management/git-review { };

  git-remote-gcrypt = callPackage ../applications/version-management/git-remote-gcrypt { };

  git-remote-hg = callPackage ../applications/version-management/git-remote-hg { };

  git-reparent = callPackage ../applications/version-management/git-reparent { };

  git-secret = callPackage ../applications/version-management/git-secret { };

  git-secrets = callPackage ../applications/version-management/git-secrets { };

  git-series = callPackage ../applications/version-management/git-series { };

  git-sizer = callPackage ../applications/version-management/git-sizer { };

  git-stack = callPackage ../applications/version-management/git-stack {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  git-stree = callPackage ../applications/version-management/git-stree { };

  git-subrepo = callPackage ../applications/version-management/git-subrepo { };

  git-subtrac = callPackage ../applications/version-management/git-subtrac { };

  git-sync = callPackage ../applications/version-management/git-sync { };

  git-team = callPackage ../applications/version-management/git-team { };

  git-test = callPackage ../applications/version-management/git-test { };

  git-town = callPackage ../applications/version-management/git-town { };

  git-trim = darwin.apple_sdk_11_0.callPackage ../applications/version-management/git-trim {
    inherit (darwin.apple_sdk_11_0.frameworks) IOKit CoreFoundation Security;
  };

  git-up = callPackage ../applications/version-management/git-up {
    pythonPackages = python3Packages;
  };

  git-vanity-hash = callPackage ../applications/version-management/git-vanity-hash { };

  git-vendor = callPackage ../applications/version-management/git-vendor { };

  git-when-merged = callPackage ../applications/version-management/git-when-merged { };

  git-workspace = callPackage ../applications/version-management/git-workspace {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  git2cl = callPackage ../applications/version-management/git2cl { };

  gitbatch = callPackage ../applications/version-management/gitbatch { };

  gitflow = callPackage ../applications/version-management/gitflow { };

  gitfs = callPackage ../tools/filesystems/gitfs { };

  gitless = callPackage ../applications/version-management/gitless { };

  gitlint = python3Packages.callPackage ../applications/version-management/gitlint { };

  gitls = callPackage ../applications/version-management/gitls { };

  gitmux = callPackage ../applications/version-management/gitmux { };

  gitnuro = callPackage ../applications/version-management/gitnuro { };

  gitnr = callPackage ../applications/version-management/gitnr { };

  gitsign = callPackage ../applications/version-management/gitsign { };

  gitstats = callPackage ../applications/version-management/gitstats { };

  gitstatus = callPackage ../applications/version-management/gitstatus { };

  gitty = callPackage ../applications/version-management/gitty { };

  gittyup = libsForQt5.callPackage ../applications/version-management/gittyup { };

  gitweb = callPackage ../applications/version-management/gitweb { };

  glab = callPackage ../applications/version-management/glab { };

  glitter = callPackage ../applications/version-management/glitter { };

  globalping-cli = callPackage ../tools/networking/globalping-cli { };

  gst = callPackage ../applications/version-management/gst { };

  guilt = callPackage ../applications/version-management/guilt { };

  gut = callPackage ../applications/version-management/gut { };

  hred = callPackage ../development/tools/hred { };

  hub = callPackage ../applications/version-management/hub { };

  hut = callPackage ../applications/version-management/hut { };

  josh = callPackage ../applications/version-management/josh { };

  lab = callPackage ../applications/version-management/lab { };

  labctl = callPackage ../tools/networking/labctl { };

  legit = callPackage ../applications/version-management/legit { };

  legit-web = callPackage ../applications/version-management/legit-web { };

  lucky-commit = callPackage ../applications/version-management/lucky-commit {
    inherit (darwin.apple_sdk.frameworks) OpenCL;
  };

  merge-fmt = callPackage ../applications/version-management/merge-fmt {
    inherit (ocamlPackages) buildDunePackage cmdliner base stdio;
   };

  pass-git-helper = python3Packages.callPackage ../applications/version-management/pass-git-helper { };

  qgit = qt5.callPackage ../applications/version-management/qgit { };

  rs-git-fsmonitor = callPackage ../applications/version-management/rs-git-fsmonitor { };

  scmpuff = callPackage ../applications/version-management/scmpuff { };

  silver-platter = python3Packages.callPackage ../applications/version-management/silver-platter { };

  stgit = callPackage ../applications/version-management/stgit { };

  subgit = callPackage ../applications/version-management/subgit { };

  svn-all-fast-export = libsForQt5.callPackage ../applications/version-management/svn-all-fast-export { };

  svn2git = callPackage ../applications/version-management/svn2git {
    git = gitSVN;
  };

  thicket = callPackage ../applications/version-management/thicket { };

  tig = callPackage ../applications/version-management/tig { };

  top-git = callPackage ../applications/version-management/topgit { };

  transcrypt = callPackage ../applications/version-management/transcrypt { };

  ungit = callPackage ../applications/version-management/ungit { };

  inherit (haskellPackages) git-annex;

  inherit (haskellPackages) git-brunch;

  git-autofixup = perlPackages.GitAutofixup;

  ghrepo-stats = with python3Packages; toPythonApplication ghrepo-stats;

  git-filter-repo = with python3Packages; toPythonApplication git-filter-repo;

  git-revise = with python3Packages; toPythonApplication git-revise;

  ### APPLICATIONS/EMULATORS

  _86Box = callPackage ../applications/emulators/86box { };

  _86Box-with-roms = _86Box.override {
    unfreeEnableRoms = true;
    unfreeEnableDiscord = true;
  };

  attract-mode = callPackage ../applications/emulators/attract-mode { };

  basiliskii = callPackage ../applications/emulators/basiliskii { };

  box64 = callPackage ../applications/emulators/box64 {
    hello-x86_64 = if stdenv.hostPlatform.isx86_64 then
      hello
    else
      pkgsCross.gnu64.hello;
  };

  caprice32 = callPackage ../applications/emulators/caprice32 { };

  ccemux = callPackage ../applications/emulators/ccemux { };

  cdemu-client = callPackage ../applications/emulators/cdemu/client.nix { };

  cdemu-daemon = callPackage ../applications/emulators/cdemu/daemon.nix { };

  cemu = callPackage ../applications/emulators/cemu { };

  cen64 = callPackage ../applications/emulators/cen64 { };

  citations = callPackage ../applications/misc/citations { };

  webfontkitgenerator = callPackage ../applications/misc/webfontkitgenerator { };

  collapseos-cvm = callPackage ../applications/emulators/collapseos-cvm { };

  coltrane = callPackage ../applications/misc/coltrane { };

  craftos-pc = callPackage ../applications/emulators/craftos-pc { };

  darcnes = callPackage ../applications/emulators/darcnes { };

  desmume = callPackage ../applications/emulators/desmume { };

  dgen-sdl = callPackage ../applications/emulators/dgen-sdl { };

  dlx = callPackage ../applications/emulators/dlx { };

  dosbox = callPackage ../applications/emulators/dosbox {
    inherit (darwin.apple_sdk.frameworks ) OpenGL;
    SDL = if stdenv.isDarwin then SDL else SDL_compat;
  };

  dosbox-x = darwin.apple_sdk_11_0.callPackage ../applications/emulators/dosbox-x {
    inherit (darwin.apple_sdk_11_0.frameworks) AudioUnit Carbon Cocoa;
  };

  duckstation = qt6Packages.callPackage ../applications/emulators/duckstation { };

  dynamips = callPackage ../applications/emulators/dynamips { };

  emu2 = callPackage ../applications/emulators/emu2 { };

  fceux = libsForQt5.callPackage ../applications/emulators/fceux { };

  firebird-emu = libsForQt5.callPackage ../applications/emulators/firebird-emu { };

  flycast = callPackage ../applications/emulators/flycast { };

  flix = callPackage ../development/compilers/flix { };

  fsrx = callPackage ../tools/misc/fsrx { };

  fuc = callPackage ../tools/misc/fuc { };

  fuse-emulator = callPackage ../applications/emulators/fuse-emulator { };

  fusesoc = python3Packages.callPackage ../tools/package-management/fusesoc { };

  fw = callPackage ../tools/misc/fw { };

  g810-led = callPackage ../misc/g810-led { };

  gcdemu = callPackage ../applications/emulators/cdemu/gui.nix { };

  gensgs = pkgsi686Linux.callPackage ../applications/emulators/gens-gs { };

  goldberg-emu = callPackage ../applications/emulators/goldberg-emu {
    protobuf = protobuf_21;
  };

  gopsuinfo = callPackage ../tools/system/gopsuinfo { };

  gxemul = callPackage ../applications/emulators/gxemul { };

  hatari = callPackage ../applications/emulators/hatari { };

  hostapd-mana = callPackage ../tools/networking/hostapd-mana { };

  image-analyzer = callPackage ../applications/emulators/cdemu/analyzer.nix { };

  kega-fusion = pkgsi686Linux.callPackage ../applications/emulators/kega-fusion { };

  lambda-delta = callPackage ../applications/emulators/lambda-delta { };

  libdsk = callPackage ../applications/emulators/libdsk { };

  libmirage = callPackage ../applications/emulators/cdemu/libmirage.nix { };

  ludusavi = callPackage ../applications/backup/ludusavi { };

  maiko = callPackage ../applications/emulators/maiko { };

  mainsail = callPackage ../applications/misc/mainsail { };

  mailctl = (haskellPackages.callPackage ../tools/networking/mailctl {}).overrideScope (final: prev: {
    # Dependency twain requires an older version of http2, and we cannot mix
    # versions of transitive dependencies.
    http2 = final.http2_3_0_3;
    warp = final.warp_3_3_30;
  });

  mame = libsForQt5.callPackage ../applications/emulators/mame { };

  mame-tools = lib.addMetaAttrs {
    description = mame.meta.description + " (tools only)";
  } (lib.getOutput "tools" mame);

  mupen64plus = callPackage ../applications/emulators/mupen64plus { };

  np2kai = callPackage ../applications/emulators/np2kai { };

  nuked-md = callPackage ../applications/emulators/nuked-md { };

  oberon-risc-emu = callPackage ../applications/emulators/oberon-risc-emu { };

  openmsx = callPackage ../applications/emulators/openmsx { };

  packwiz = callPackage ../tools/games/minecraft/packwiz { };

  pcem = callPackage ../applications/emulators/pcem { };

  pcsx2 = qt6Packages.callPackage ../applications/emulators/pcsx2 { };

  pcsxr = callPackage ../applications/emulators/pcsxr { };

  ppsspp-sdl = let
    argset = {
      ffmpeg = ffmpeg_4;
      glew = glew.override { enableEGL = argset.forceWayland; };
      enableQt = false;
      enableVulkan = true;
      forceWayland = false;
    };
  in
    ppsspp.override argset;

  ppsspp-sdl-wayland = let
    argset = {
      ffmpeg = ffmpeg_4;
      glew = glew.override { enableEGL = argset.forceWayland; };
      enableQt = false;
      enableVulkan = false; # https://github.com/hrydgard/ppsspp/issues/13845
      forceWayland = true;
    };
  in
    ppsspp.override argset;

  ppsspp-qt = let
    argset = {
      ffmpeg = ffmpeg_4;
      glew = glew.override { enableEGL = argset.forceWayland; };
      enableQt = true;
      enableVulkan = false; # https://github.com/hrydgard/ppsspp/issues/11628
      forceWayland = false;
    };
  in
    ppsspp.override argset;

  proton-caller = callPackage ../applications/emulators/proton-caller { };

  punes = libsForQt5.callPackage ../applications/emulators/punes { };

  punes-qt6 = qt6Packages.callPackage ../applications/emulators/punes { };

  py65 = with python3.pkgs; toPythonApplication py65;

  retrofe = callPackage ../applications/emulators/retrofe { };

  ripes = qt6Packages.callPackage ../applications/emulators/ripes { };

  rmg-wayland = callPackage ../by-name/rm/rmg/package.nix {
    withWayland = true;
  };

  rpcemu = callPackage ../applications/emulators/rpcemu { };

  ruffle = callPackage ../applications/emulators/ruffle { };

  sameboy = callPackage ../applications/emulators/sameboy { };

  simh = callPackage ../applications/emulators/simh { };

  simplenes = callPackage ../applications/emulators/simplenes { };

  snes9x-gtk = snes9x.override {
    withGtk = true;
  };

  tamatool = callPackage ../applications/emulators/tamatool { };

  termtekst = callPackage ../applications/emulators/termtekst { };

  tilem = callPackage ../applications/emulators/tilem { };

  tiny8086 = callPackage ../applications/emulators/tiny8086 { };

  tinyemu = callPackage ../applications/emulators/tinyemu { };

  uae = callPackage ../applications/emulators/uae { };

  vbam = callPackage ../applications/emulators/vbam { };

  vice = callPackage ../applications/emulators/vice { };

  winetricks = callPackage ../applications/emulators/wine/winetricks.nix {
    inherit (gnome) zenity;
  };

  xcpc = callPackage ../applications/emulators/xcpc { };

  yapesdl = callPackage ../applications/emulators/yapesdl { };

  zsnes = pkgsi686Linux.callPackage ../applications/emulators/zsnes { };
  zsnes2 = pkgsi686Linux.callPackage ../applications/emulators/zsnes/2.x.nix { };

  ### APPLICATIONS/EMULATORS/BSNES

  ares = darwin.apple_sdk_11_0.callPackage ../applications/emulators/bsnes/ares { };

  bsnes-hd = darwin.apple_sdk_11_0.callPackage ../applications/emulators/bsnes/bsnes-hd { };

  higan = callPackage ../applications/emulators/bsnes/higan { };

  ### APPLICATIONS/EMULATORS/DOLPHIN-EMU

  dolphin-emu = qt6Packages.callPackage ../applications/emulators/dolphin-emu {
    inherit (darwin.apple_sdk_11_0.frameworks) CoreBluetooth ForceFeedback IOBluetooth IOKit OpenGL VideoToolbox;
    inherit (darwin) moltenvk;
  };

  dolphin-emu-primehack = qt5.callPackage ../applications/emulators/dolphin-emu/primehack.nix {
    inherit (darwin.apple_sdk.frameworks) CoreBluetooth ForceFeedback IOKit OpenGL;
    fmt = fmt_8;
  };

  ### APPLICATIONS/EMULATORS/RETROARCH

  retroarchBare = qt5.callPackage ../applications/emulators/retroarch { };

  retroarchFull = retroarch.override {
    cores = builtins.filter
      # Remove cores not supported on platform
      (c: c ? libretroCore && (lib.meta.availableOn stdenv.hostPlatform c))
      (builtins.attrValues libretro);
  };

  wrapRetroArch = { retroarch, settings ? {} }:
    callPackage ../applications/emulators/retroarch/wrapper.nix
      { inherit retroarch settings; };

  retroarch = wrapRetroArch {
    retroarch = retroarchBare;
    settings = {
      assets_directory = "${retroarch-assets}/share/retroarch/assets";
      joypad_autoconfig_dir = "${retroarch-joypad-autoconfig}/share/libretro/autoconfig";
      libretro_info_path = "${libretro-core-info}/share/retroarch/cores";
    };
  };

  retroarch-assets = callPackage ../applications/emulators/retroarch/retroarch-assets.nix { };

  retroarch-joypad-autoconfig = callPackage ../applications/emulators/retroarch/retroarch-joypad-autoconfig.nix { };

  libretranslate = with python3.pkgs; toPythonApplication libretranslate;

  libretro = recurseIntoAttrs
    (callPackage ../applications/emulators/retroarch/cores.nix {
      retroarch = retroarchBare;
    });

  libretro-core-info = callPackage ../applications/emulators/retroarch/libretro-core-info.nix { };

  kodi-retroarch-advanced-launchers =
    callPackage ../applications/emulators/retroarch/kodi-advanced-launchers.nix { };

  # Aliases kept here because they are easier to use
  x16-emulator = x16.emulator;
  x16-rom = x16.rom;
  x16-run = x16.run;

  yabause = libsForQt5.callPackage ../applications/emulators/yabause {
    freeglut = null;
    openal = null;
  };

  ### APPLICATIONS/FILE-MANAGERS

  browsr = callPackage ../applications/file-managers/browsr { };

  cfm = callPackage ../applications/file-managers/cfm { };

  clex = callPackage ../applications/file-managers/clex { };

  clifm = callPackage ../applications/file-managers/clifm { };

  doublecmd = callPackage ../by-name/do/doublecmd/package.nix {
    inherit (qt5) wrapQtAppsHook;
  };

  felix-fm = callPackage ../applications/file-managers/felix-fm { };

  krusader = libsForQt5.callPackage ../applications/file-managers/krusader { };

  lesscpy = callPackage ../development/compilers/lesscpy { };

  lf = callPackage ../applications/file-managers/lf { };

  ctpv = callPackage ../applications/file-managers/lf/ctpv.nix { };

  mc = callPackage ../applications/file-managers/mc {
    inherit (darwin) autoSignDarwinBinariesHook;
  };

  mucommander = callPackage ../applications/file-managers/mucommander { };

  nnn = callPackage ../applications/file-managers/nnn { };

  noice = callPackage ../applications/file-managers/noice { };

  pcmanfm = callPackage ../applications/file-managers/pcmanfm { };

  portfolio-filemanager = callPackage ../applications/file-managers/portfolio-filemanager { };

  potreeconverter = callPackage ../applications/graphics/potreeconverter { };

  ranger = callPackage ../applications/file-managers/ranger { };

  sfm = callPackage ../applications/file-managers/sfm { };

  shfm = callPackage ../applications/file-managers/shfm { };

  spaceFM = callPackage ../applications/file-managers/spacefm { };

  vifm = callPackage ../applications/file-managers/vifm { };

  vifm-full = vifm.override {
    mediaSupport = true;
    inherit lib udisks2 python3;
  };

  walk = callPackage ../applications/file-managers/walk { };

  xfe = callPackage ../applications/file-managers/xfe {
    fox = fox_1_6;
  };

  xplorer = callPackage ../applications/file-managers/xplorer { };

  johnny-reborn-engine = callPackage ../applications/misc/johnny-reborn { };

  johnny-reborn = callPackage ../applications/misc/johnny-reborn/with-data.nix { };

  ### APPLICATIONS/TERMINAL-EMULATORS

  alacritty = callPackage ../applications/terminal-emulators/alacritty {
    inherit (darwin.apple_sdk_11_0.frameworks) AppKit CoreGraphics CoreServices CoreText Foundation OpenGL;
  };

  blackbox-terminal = callPackage ../applications/terminal-emulators/blackbox-terminal { };

  contour = qt6.callPackage ../applications/terminal-emulators/contour {
    inherit (darwin.apple_sdk_11_0.libs) utmp;
    inherit (darwin) sigtool;
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
    catch2 = catch2_3;
    fmt = fmt_9;
  };

  cool-retro-term = libsForQt5.callPackage ../applications/terminal-emulators/cool-retro-term { };

  darktile = callPackage ../applications/terminal-emulators/darktile { };

  dterm = callPackage ../applications/terminal-emulators/dterm { };

  foot = callPackage ../applications/terminal-emulators/foot { };

  guake = callPackage ../applications/terminal-emulators/guake { };

  hyper = callPackage ../applications/terminal-emulators/hyper { };

  iterm2 = callPackage ../applications/terminal-emulators/iterm2 { };

  kitty = darwin.apple_sdk_11_0.callPackage ../applications/terminal-emulators/kitty {
    harfbuzz = harfbuzz.override { withCoreText = stdenv.isDarwin; };
    inherit (darwin.apple_sdk_11_0) Libsystem;
    inherit (darwin.apple_sdk_11_0.frameworks)
      Cocoa
      Kernel
      UniformTypeIdentifiers
      UserNotifications
    ;
  };

  kitty-themes  = callPackage ../applications/terminal-emulators/kitty/themes.nix { };

  lxterminal = callPackage ../applications/terminal-emulators/lxterminal { };

  microcom = callPackage ../applications/terminal-emulators/microcom { };

  mlterm = darwin.apple_sdk_11_0.callPackage ../applications/terminal-emulators/mlterm { };
  mlterm-wayland = mlterm.override {
    enableX11 = false;
  };

  mrxvt = callPackage ../applications/terminal-emulators/mrxvt { };

  rxvt = callPackage ../applications/terminal-emulators/rxvt { };

  rxvt-unicode = callPackage ../applications/terminal-emulators/rxvt-unicode/wrapper.nix { };

  rxvt-unicode-emoji = rxvt-unicode.override {
    rxvt-unicode-unwrapped = rxvt-unicode-unwrapped-emoji;
  };

  rxvt-unicode-plugins = import ../applications/terminal-emulators/rxvt-unicode-plugins { inherit callPackage; };

  rxvt-unicode-unwrapped = callPackage ../applications/terminal-emulators/rxvt-unicode { };

  rxvt-unicode-unwrapped-emoji = rxvt-unicode-unwrapped.override {
    emojiSupport = true;
  };

  sakura = callPackage ../applications/terminal-emulators/sakura { };

  scriv = callPackage ../applications/version-management/scriv { };

  st = callPackage ../applications/terminal-emulators/st {
    conf = config.st.conf or null;
    patches = config.st.patches or [];
    extraLibs = config.st.extraLibs or [];
  };
  xst = callPackage ../applications/terminal-emulators/st/xst.nix { };
  lukesmithxyz-st = callPackage ../applications/terminal-emulators/st/lukesmithxyz-st { };
  mcaimi-st = callPackage ../applications/terminal-emulators/st/mcaimi-st.nix { };
  siduck76-st = callPackage ../applications/terminal-emulators/st/siduck76-st.nix { };

  stupidterm = callPackage ../applications/terminal-emulators/stupidterm {
    gtk = gtk3;
  };

  terminator = callPackage ../applications/terminal-emulators/terminator { };

  termite = callPackage ../applications/terminal-emulators/termite/wrapper.nix {
    termite = termite-unwrapped;
  };
  termite-unwrapped = callPackage ../applications/terminal-emulators/termite { };

  termonad = callPackage ../applications/terminal-emulators/termonad { };

  tym = callPackage ../applications/terminal-emulators/tym { };

  wayst = callPackage ../applications/terminal-emulators/wayst { };

  wezterm = darwin.apple_sdk_11_0.callPackage ../applications/terminal-emulators/wezterm {
    inherit (darwin.apple_sdk_11_0.frameworks) Cocoa CoreGraphics Foundation UserNotifications System;
  };

  x3270 = callPackage ../applications/terminal-emulators/x3270 { };

  xterm = callPackage ../applications/terminal-emulators/xterm { };

  xtermcontrol = callPackage ../applications/terminal-emulators/xtermcontrol { };

  yaft = callPackage ../applications/terminal-emulators/yaft { };

  writefreely = callPackage ../applications/misc/writefreely { };

  iqueue = callPackage ../development/libraries/iqueue { };

  lifecycled = callPackage ../tools/misc/lifecycled { };

  lil-pwny = callPackage ../tools/security/lil-pwny { };

  lilo = callPackage ../tools/misc/lilo { };

  logseq = callPackage ../applications/misc/logseq { };

  natls = callPackage ../tools/misc/natls { };

  notion-app-enhanced = callPackage ../applications/office/notion-app-enhanced { };

  pikchr = callPackage ../tools/graphics/pikchr { };

  poop = callPackage ../tools/misc/poop { };

  popl = callPackage ../development/libraries/popl { };

  popsicle = callPackage ../tools/misc/popsicle { };

  terminal-colors = callPackage ../applications/misc/terminal-colors { };

  termsyn = callPackage ../data/fonts/termsyn { };

  tvnamer = callPackage ../tools/misc/tvnamer { };

  twine = with python3Packages; toPythonApplication twine;

  abracadabra = qt6Packages.callPackage ../applications/radio/abracadabra { };

  accelergy = callPackage ../applications/science/computer-architecture/accelergy { };

  aldo = callPackage ../applications/radio/aldo { };

  alglib = callPackage ../development/libraries/alglib { };

  almanah = callPackage ../applications/misc/almanah { };

  alpine-make-vm-image = callPackage ../tools/virtualization/alpine-make-vm-image { };

  amazon-ec2-utils = callPackage ../tools/admin/amazon-ec2-utils { };

  amazon-ecs-cli = callPackage ../tools/virtualization/amazon-ecs-cli { };

  amazon-qldb-shell = callPackage ../development/tools/amazon-qldb-shell {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  amber = callPackage ../tools/text/amber {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  amber-secret = callPackage ../tools/security/amber {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  amberol = callPackage ../applications/audio/amberol { };

  inherit (callPackages ../development/tools/ammonite {})
    ammonite_2_12
    ammonite_2_13;
  ammonite = if scala == scala_2_12 then ammonite_2_12 else ammonite_2_13;

  amp = callPackage ../applications/editors/amp { };

  ams = callPackage ../applications/audio/ams { };

  amtterm = callPackage ../tools/system/amtterm { };

  analog = callPackage ../tools/admin/analog { };

  android-backup-extractor = callPackage ../tools/backup/android-backup-extractor { };

  android-tools = lowPrio (darwin.apple_sdk_11_0.callPackage ../tools/misc/android-tools { });

  anew = callPackage ../tools/text/anew { };

  anewer = callPackage ../tools/text/anewer { };

  angie = callPackage ../servers/http/angie {
    zlib = zlib-ng.override { withZlibCompat = true; };
    withPerl = false;
    # We don't use `with` statement here on purpose!
    # See https://github.com/NixOS/nixpkgs/pull/10474#discussion_r42369334
    modules = [ nginxModules.rtmp nginxModules.dav nginxModules.moreheaders ];
  };

  angieQuic = callPackage ../servers/http/angie {
    zlib = zlib-ng.override { withZlibCompat = true; };
    withPerl = false;
    withQuic = true;
    # We don't use `with` statement here on purpose!
    # See https://github.com/NixOS/nixpkgs/pull/10474#discussion_r42369334
    modules = [ nginxModules.rtmp nginxModules.dav nginxModules.moreheaders ];
    # Use latest quictls to allow http3 support
    openssl = quictls;
  };

  angie-console-light = callPackage ../servers/http/angie/console-light.nix { };

  angle-grinder = callPackage ../tools/text/angle-grinder { };

  ansifilter = callPackage ../tools/text/ansifilter { };

  antora = callPackage ../development/tools/documentation/antora { };

  apfs-fuse = callPackage ../tools/filesystems/apfs-fuse { };

  apk-tools = callPackage ../tools/package-management/apk-tools {
    lua = lua5_3;
  };

  apkid = callPackage ../development/tools/apkid { };

  apkleaks = callPackage ../tools/security/apkleaks { };

  apksigcopier = callPackage ../development/tools/apksigcopier { };

  apksigner = callPackage ../development/tools/apksigner { };

  apktool = callPackage ../development/tools/apktool { };

  appimage-run = callPackage ../tools/package-management/appimage-run { };
  appimage-run-tests = callPackage ../tools/package-management/appimage-run/test.nix {
    appimage-run = appimage-run.override {
      appimage-run-tests = null; /* break boostrap cycle for passthru.tests */
    };
  };

  appimagekit = callPackage ../tools/package-management/appimagekit { };

  apt-cacher-ng = callPackage ../servers/http/apt-cacher-ng { };

  aptly = callPackage ../tools/misc/aptly { };

  ArchiSteamFarm = callPackage ../applications/misc/ArchiSteamFarm { };

  archivebox = callPackage ../applications/misc/archivebox { };

  archivemount = callPackage ../tools/filesystems/archivemount { };

  archivy = callPackage ../applications/misc/archivy { };

  arandr = callPackage ../tools/X11/arandr { };

  arangodb = callPackage ../servers/nosql/arangodb { };

  arcanist = callPackage ../development/tools/misc/arcanist { php = php81; };

  arduino = arduino-core.override { withGui = true; };

  arduino-ci = callPackage ../development/embedded/arduino/arduino-ci { };

  arduino-cli = callPackage ../development/embedded/arduino/arduino-cli { };

  arduino-core = callPackage ../development/embedded/arduino/arduino-core/chrootenv.nix { };
  arduino-core-unwrapped = callPackage ../development/embedded/arduino/arduino-core { };

  arduino-language-server = callPackage ../development/embedded/arduino/arduino-language-server { };

  arduino-mk = callPackage ../development/embedded/arduino/arduino-mk { };

  arduinoOTA = callPackage ../development/embedded/arduino/arduinoOTA { };

  apio = python3Packages.callPackage ../development/embedded/fpga/apio { };

  apitrace = libsForQt5.callPackage ../applications/graphics/apitrace { };

  arguments = callPackage ../development/libraries/arguments { };

  argus = callPackage ../tools/networking/argus { };

  argus-clients = callPackage ../tools/networking/argus-clients { };

  argyllcms = callPackage ../tools/graphics/argyllcms { };

  arj = callPackage ../tools/archivers/arj {
    stdenv = gccStdenv;
  };

  arp-scan = callPackage ../tools/misc/arp-scan { };

  inherit (callPackages ../data/fonts/arphic {})
    arphic-ukai arphic-uming;

  artyFX = callPackage ../applications/audio/artyFX { };

  oggvideotools = callPackage ../tools/misc/oggvideotools { };

  owl-lisp = callPackage ../development/compilers/owl-lisp { };

  ascii = callPackage ../tools/text/ascii { };

  asciinema = callPackage ../tools/misc/asciinema { };

  asciinema-agg = callPackage ../tools/misc/asciinema-agg {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  asciinema-scenario = callPackage ../tools/misc/asciinema-scenario { };

  asciiquarium = callPackage ../applications/misc/asciiquarium { };

  ashuffle = callPackage ../applications/audio/ashuffle { };

  ashpd-demo = callPackage ../development/tools/ashpd-demo { };

  astc-encoder = callPackage ../tools/graphics/astc-encoder { };

  asymptote = libsForQt5.callPackage ../tools/graphics/asymptote { };

  async = callPackage ../development/tools/async { };

  atheme = callPackage ../servers/irc/atheme { };

  atinout = callPackage ../tools/networking/atinout { };

  atomicparsley = callPackage ../tools/video/atomicparsley {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  audiowaveform = callPackage ../tools/audio/audiowaveform { };

  authenticator = callPackage ../applications/misc/authenticator { };

  authelia = callPackage ../servers/authelia { };

  authentik-outposts = recurseIntoAttrs (callPackages ../by-name/au/authentik/outposts.nix { });

  autoflake = with python3.pkgs; toPythonApplication autoflake;

  autospotting = callPackage ../applications/misc/autospotting { };

  avfs = callPackage ../tools/filesystems/avfs { };

  aliyun-cli = callPackage ../tools/admin/aliyun-cli { };

  aws-encryption-sdk-cli = callPackage ../tools/admin/aws-encryption-sdk-cli { };

  aws-iam-authenticator = callPackage ../tools/security/aws-iam-authenticator { };

  awscli = callPackage ../tools/admin/awscli { };

  awscli2 = callPackage ../tools/admin/awscli2 { };

  okta-aws-cli = callPackage ../tools/admin/okta-aws-cli { };

  awsebcli = callPackage ../tools/virtualization/awsebcli { };

  awslimitchecker = callPackage ../tools/admin/awslimitchecker { };

  awslogs = callPackage ../tools/admin/awslogs { };

  awsume = python3Packages.callPackage ../tools/admin/awsume { };

  aws-assume-role = callPackage ../tools/admin/aws-assume-role { };

  aws-lambda-rie = callPackage ../tools/admin/aws-lambda-runtime-interface-emulator { };

  aws-env = callPackage ../tools/admin/aws-env { };

  aws-google-auth = python3Packages.callPackage ../tools/admin/aws-google-auth { };

  aws-mfa = python3Packages.callPackage ../tools/admin/aws-mfa { };

  aws-nuke = callPackage ../tools/admin/aws-nuke { };

  aws-rotate-key = callPackage ../tools/admin/aws-rotate-key { };

  aws-sam-cli = callPackage ../development/tools/aws-sam-cli { };

  aws-sso-cli = callPackage ../tools/admin/aws-sso-cli { };

  aws-sso-creds = callPackage ../tools/admin/aws-sso-creds { };

  aws-vault = callPackage ../tools/admin/aws-vault { };

  iamy = callPackage ../tools/admin/iamy { };

  iam-policy-json-to-terraform = callPackage ../tools/misc/iam-policy-json-to-terraform { };

  azeret-mono = callPackage ../data/fonts/azeret-mono { };

  azure-cli = callPackage ../tools/admin/azure-cli { };

  azure-cli-extensions = recurseIntoAttrs azure-cli.extensions;

  azure-functions-core-tools = callPackage ../development/tools/azure-functions-core-tools { };

  azure-static-sites-client = callPackage ../development/tools/azure-static-sites-client { };

  azure-storage-azcopy = callPackage ../development/tools/azcopy { };

  bark = callPackage ../tools/audio/bark { };

  bashblog = callPackage ../tools/text/bashblog { };

  berglas = callPackage ../tools/admin/berglas { };

  betterdiscordctl = callPackage ../tools/misc/betterdiscordctl { };

  betterdiscord-installer = callPackage ../tools/misc/betterdiscord-installer { };

  binocle = callPackage ../applications/misc/binocle {
    inherit (darwin.apple_sdk.frameworks) AppKit CoreFoundation CoreGraphics CoreVideo Foundation Metal QuartzCore;
  };

  bitwise = callPackage ../tools/misc/bitwise { };

  blisp = darwin.apple_sdk_11_0.callPackage ../development/embedded/blisp {
    inherit (darwin.apple_sdk_11_0.frameworks) IOKit;
  };

  brakeman = callPackage ../development/tools/analysis/brakeman { };

  brewtarget = libsForQt5.callPackage ../applications/misc/brewtarget { } ;

  bootspec = callPackage ../tools/misc/bootspec { };

  # Derivation's result is not used by nixpkgs. Useful for validation for
  # regressions of bootstrapTools on hydra and on ofborg. Example:
  #     pkgsCross.aarch64-multiplatform.freshBootstrapTools.build
  freshBootstrapTools = if stdenv.hostPlatform.isDarwin then
    callPackage ../stdenv/darwin/make-bootstrap-tools.nix {
      localSystem = stdenv.buildPlatform;
      crossSystem =
        if stdenv.buildPlatform == stdenv.hostPlatform then null else stdenv.hostPlatform;
    }
  else if stdenv.hostPlatform.isLinux then
    callPackage ../stdenv/linux/make-bootstrap-tools.nix {}
  else throw "freshBootstrapTools: unknown hostPlatform ${stdenv.hostPlatform.config}";

  boxes = callPackage ../tools/text/boxes { };

  boxxy = callPackage ../tools/misc/boxxy { };

  boundary = callPackage ../tools/networking/boundary { };

  chamber = callPackage ../tools/admin/chamber {  };

  chaos = callPackage ../tools/networking/chaos {  };

  charm = callPackage ../applications/misc/charm { };

  chars = callPackage ../tools/text/chars {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  checkip = callPackage ../tools/networking/checkip { };

  crystfel = callPackage ../applications/science/physics/crystfel { };

  crystfel-headless = callPackage ../applications/science/physics/crystfel { withGui = false; };

  cyberchef = callPackage ../tools/misc/cyberchef { };

  cw = callPackage ../tools/admin/cw { };

  ec2-api-tools = callPackage ../tools/virtualization/ec2-api-tools { };

  ec2-ami-tools = callPackage ../tools/virtualization/ec2-ami-tools { };

  ec2-metadata-mock = callPackage ../development/tools/ec2-metadata-mock { };

  exoscale-cli = callPackage ../tools/admin/exoscale-cli { };

  altermime = callPackage ../tools/networking/altermime { };

  alttab = callPackage ../tools/X11/alttab { };

  amule = callPackage ../tools/networking/p2p/amule { };

  amule-daemon = amule.override {
    monolithic = false;
    enableDaemon = true;
  };

  amule-gui = amule.override {
    monolithic = false;
    client = true;
  };

  amule-web = amule.override {
    monolithic = false;
    httpServer = true;
  };

  antennas = callPackage ../servers/antennas { };

  apg = callPackage ../tools/security/apg { };

  apt-dater = callPackage ../tools/package-management/apt-dater {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  asusctl = callPackage ../applications/system/asusctl { };

  autorevision = callPackage ../tools/misc/autorevision { };

  automirror = callPackage ../tools/misc/automirror { };

  barman = callPackage ../tools/misc/barman { };

  bartib = callPackage ../tools/misc/bartib { };

  base16-universal-manager = callPackage ../applications/misc/base16-universal-manager { };

  base16384 = callPackage ../tools/text/base16384 { };

  bashate = python3Packages.callPackage ../development/tools/bashate { };

  bash-my-aws = callPackage ../tools/admin/bash-my-aws { };

  bashcards = callPackage ../tools/misc/bashcards { };

  bazarr = callPackage ../servers/bazarr { };

  bisq-desktop = callPackage ../applications/blockchains/bisq-desktop {
    openjdk11 = openjdk11.override { enableJavaFX = true; };
  };

  bic = callPackage ../development/interpreters/bic { };

  biscuit-cli = callPackage ../tools/security/biscuit-cli { };

  inherit (callPackages ../tools/security/bitwarden-directory-connector { }) bitwarden-directory-connector-cli bitwarden-directory-connector;

  bitwarden-menu = python3Packages.callPackage ../applications/misc/bitwarden-menu { };

  inherit (nodePackages) concurrently;

  bklk = callPackage ../applications/misc/bklk { };

  bkt = callPackage ../tools/misc/bkt {  };

  bkyml = callPackage ../tools/misc/bkyml { };

  blackmagic-desktop-video = callPackage ../tools/video/blackmagic-desktop-video { };

  blocksat-cli = with python3Packages; toPythonApplication blocksat-cli;

  bmap-tools = callPackage ../tools/misc/bmap-tools { };

  bolliedelayxt-lv2 = callPackage ../applications/audio/bolliedelayxt.lv2 { };

  bonnmotion = callPackage ../development/tools/misc/bonnmotion { };

  bonnie = callPackage ../tools/filesystems/bonnie { };

  boron = callPackage ../development/interpreters/boron { };

  botamusique = callPackage ../tools/audio/botamusique { };

  boulder = callPackage ../tools/admin/boulder { };

  btrfs-heatmap = callPackage ../tools/filesystems/btrfs-heatmap { };

  bucklespring = bucklespring-x11;
  bucklespring-libinput = callPackage ../applications/audio/bucklespring { };
  bucklespring-x11 = callPackage ../applications/audio/bucklespring { legacy = true; };

  buildbotPackages = recurseIntoAttrs (python3.pkgs.callPackage ../development/tools/continuous-integration/buildbot { });
  inherit (buildbotPackages) buildbot buildbot-ui buildbot-full buildbot-plugins buildbot-worker;

  bunyan-rs = callPackage ../development/tools/bunyan-rs { };

  calcure = callPackage ../applications/misc/calcure { };

  callaudiod = callPackage ../applications/audio/callaudiod { };

  castopod = callPackage ../applications/audio/castopod { };

  calls = callPackage ../applications/networking/calls { };

  castnow = callPackage ../tools/networking/castnow { };

  castty = callPackage ../tools/misc/castty { };

  certigo = callPackage ../tools/admin/certigo { };

  certipy = with python3Packages; toPythonApplication certipy-ad;

  catcli = python3Packages.callPackage ../tools/filesystems/catcli { };

  changelogger = callPackage ../tools/misc/changelogger { };

  chipsec = callPackage ../tools/security/chipsec {
    kernel = null;
    withDriver = false;
  };

  chroma = callPackage ../tools/text/chroma { };

  clair = callPackage ../tools/admin/clair { };

  clairvoyance = callPackage ../tools/security/clairvoyance { };

  cloudfox = callPackage ../tools/security/cloudfox { };

  cloudhunter = callPackage ../tools/security/cloudhunter { };

  cloudsmith-cli = callPackage ../development/tools/cloudsmith-cli { };

  codeql = callPackage ../development/tools/analysis/codeql { };

  fedora-backgrounds = callPackage ../data/misc/fedora-backgrounds { };

  ccextractor = callPackage ../applications/video/ccextractor { };

  cconv = callPackage ../tools/text/cconv { };

  go-check = callPackage ../development/tools/check { };

  go-cve-search = callPackage ../tools/security/go-cve-search { };

  go-dork = callPackage ../tools/security/go-dork { };

  chkcrontab = callPackage ../tools/admin/chkcrontab { };

  claws = callPackage ../tools/misc/claws { };

  cloud-custodian = callPackage ../tools/networking/cloud-custodian  { };

  coconut = with python3Packages; toPythonApplication coconut;

  cod = callPackage ../tools/misc/cod { };

  codespell = callPackage ../development/tools/codespell { };

  conjure = callPackage ../applications/graphics/conjure { };

  coolreader = libsForQt5.callPackage ../applications/misc/coolreader { };

  corsair = with python3Packages; toPythonApplication corsair-scan;

  cosign = callPackage ../tools/security/cosign {
    inherit (darwin.apple_sdk.frameworks) PCSC;
  };

  coze = callPackage ../tools/security/coze { } ;

  cozy = callPackage ../applications/audio/cozy { };

  cpptoml = callPackage ../development/libraries/cpptoml { };

  cpuid = callPackage ../os-specific/linux/cpuid { };

  msr = callPackage ../os-specific/linux/msr { };

  ctre = callPackage ../development/libraries/ctre { };

  ctrtool = callPackage ../tools/archivers/ctrtool { };

  crowbar = callPackage ../tools/security/crowbar { };

  crumbs = callPackage ../applications/misc/crumbs { };

  crc32c = callPackage ../development/libraries/crc32c { };

  crcpp = callPackage ../development/libraries/crcpp { };

  cudd = callPackage ../development/libraries/cudd { };

  inherit (cue) writeCueValidator;

  cuelsp = callPackage ../development/tools/cuelsp { };

  cyclonedds = callPackage ../development/libraries/cyclonedds { };

  cyclone-scheme = callPackage ../development/interpreters/cyclone { };

  cyclonedx-gomod = callPackage ../tools/security/cyclonedx-gomod { };

  cyclonedx-python = callPackage ../tools/misc/cyclonedx-python { };

  dcap = callPackage ../tools/networking/dcap { };

  deltachat-cursed = callPackage ../applications/networking/instant-messengers/deltachat-cursed { };

  delayarchitect = callPackage ../applications/audio/delayarchitect { };

  dell-command-configure = callPackage ../tools/system/dell-command-configure { };

  deltachat-desktop = callPackage ../applications/networking/instant-messengers/deltachat-desktop {
    electron = electron_30;
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  deskew = callPackage ../applications/graphics/deskew { };

  detect-secrets = with python3Packages; toPythonApplication detect-secrets;

  deterministic-uname = callPackage ../build-support/deterministic-uname { };

  deterministic-host-uname = deterministic-uname.override { forPlatform = stdenv.hostPlatform; };

  dfmt = callPackage ../tools/text/dfmt { };

  diopser = callPackage ../applications/audio/diopser { };

  diskonaut = callPackage ../tools/misc/diskonaut { };

  diskus = callPackage ../tools/misc/diskus {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  distrobox = callPackage ../applications/virtualization/distrobox { };

  djmount = callPackage ../tools/filesystems/djmount { };

  dgsh = callPackage ../shells/dgsh { };

  dkimpy = with python3Packages; toPythonApplication dkimpy;

  dl-librescore = callPackage ../tools/audio/dl-librescore { };

  dontgo403 = callPackage ../tools/security/dontgo403 { };

  dpt-rp1-py = callPackage ../tools/misc/dpt-rp1-py { };

  doona = callPackage ../tools/security/doona { };

  dotter = callPackage ../tools/misc/dotter {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  droidcam = callPackage ../applications/video/droidcam { };

  droidmote = callPackage ../tools/inputmethods/droidmote { };

  eartag = callPackage ../applications/audio/eartag { };

  ecdsautils = callPackage ../tools/security/ecdsautils { };

  echidna = haskell.lib.compose.justStaticExecutables (haskellPackages.callPackage (../tools/security/echidna) { });

  sedutil = callPackage ../tools/security/sedutil { };

  emplace = callPackage ../tools/package-management/emplace { };

  enchive = callPackage ../tools/security/enchive { };

  enjarify = callPackage ../tools/misc/enjarify { };

  enpass = callPackage ../tools/security/enpass { };

  esbuild = callPackage ../development/tools/esbuild { };

  esbuild_netlify = callPackage ../development/tools/esbuild/netlify.nix { };

  essentia-extractor = callPackage ../tools/audio/essentia-extractor { };

  esh = callPackage ../tools/text/esh { };

  ezstream = callPackage ../tools/audio/ezstream { };

  libfx2 = with python3Packages; toPythonApplication fx2;

  fastmod = callPackage ../tools/text/fastmod {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  fedifetcher = callPackage ../tools/misc/fedifetcher { };

  flavours = callPackage ../applications/misc/flavours { };

  flirc = libsForQt5.callPackage ../applications/video/flirc {
    readline = readline70;
  };

  flood = callPackage ../applications/networking/p2p/flood { };

  flood-for-transmission = callPackage ../applications/networking/p2p/flood-for-transmission { };

  font-config-info = callPackage ../tools/misc/font-config-info { };

  foxdot = with python3Packages; toPythonApplication foxdot;

  fspy = callPackage ../applications/misc/fspy { };

  fluffychat = callPackage  ../applications/networking/instant-messengers/fluffychat { };

  fluffychat-web = fluffychat.override { targetFlutterPlatform = "web"; };

  fxlinuxprintutil = callPackage ../tools/misc/fxlinuxprintutil { };

  gbl = callPackage ../tools/archivers/gbl {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  genann = callPackage ../development/libraries/genann { };

  genpass = callPackage ../tools/security/genpass {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security;
  };

  genymotion = callPackage ../development/mobile/genymotion { };

  galene = callPackage ../servers/web-apps/galene { };

  galaxy-buds-client = callPackage ../applications/audio/galaxy-buds-client { };

  gamecube-tools = callPackage ../development/tools/gamecube-tools { };

  gammaray = qt6Packages.callPackage ../development/tools/gammaray { };

  gams = callPackage ../tools/misc/gams (config.gams or {});

  gem = callPackage ../applications/audio/pd-plugins/gem { };

  github-changelog-generator = callPackage ../development/tools/github-changelog-generator { };

  github-commenter = callPackage ../development/tools/github-commenter { };

  github-copilot-intellij-agent = callPackage ../development/tools/github-copilot-intellij-agent { };

  github-to-sqlite = with python3Packages; toPythonApplication github-to-sqlite;

  gistyc = with python3Packages; toPythonApplication gistyc;

  gjs = callPackage ../development/libraries/gjs { };

  gjo = callPackage ../tools/text/gjo { };

  glances = python3Packages.callPackage ../applications/system/glances { };

  glasgow = callPackage ../tools/misc/glasgow { };

  glasstty-ttf = callPackage ../data/fonts/glasstty-ttf { };

  glaxnimate = libsForQt5.callPackage ../applications/video/glaxnimate { };

  gluesql = callPackage ../servers/sql/gluesql { };

  gmni = callPackage ../applications/networking/browsers/gmni { };

  gmnisrv = callPackage ../servers/gemini/gmnisrv { };

  gmnitohtml = callPackage ../applications/misc/gmnitohtml { };

  go2tv = darwin.apple_sdk_11_0.callPackage ../applications/video/go2tv {
    inherit (darwin.apple_sdk_11_0.frameworks) Carbon Cocoa Kernel UserNotifications;
  };
  go2tv-lite = go2tv.override { withGui = false; };

  go2rtc = callPackage ../tools/video/go2rtc { };

  goimapnotify = callPackage ../tools/networking/goimapnotify { };

  gojsontoyaml = callPackage ../development/tools/gojsontoyaml { };

  gomatrix = callPackage ../applications/misc/gomatrix { };

  gomapenum = callPackage ../tools/security/gomapenum { };

  gopacked = callPackage ../applications/misc/gopacked { };

  goperf = callPackage ../development/tools/goperf { };

  gucci = callPackage ../tools/text/gucci { };

  guglielmo = libsForQt5.callPackage ../applications/radio/guglielmo { };

  grc = python3Packages.callPackage ../tools/misc/grc { };

  green-pdfviewer = callPackage ../applications/misc/green-pdfviewer {
    SDL = SDL_sixel;
  };

  gremlin-console = callPackage ../applications/misc/gremlin-console {
    openjdk = openjdk11;
  };

  gremlin-server = callPackage ../applications/misc/gremlin-server {
    openjdk = openjdk11;
  };

  grex = callPackage ../tools/misc/grex {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  gcsfuse = callPackage ../tools/filesystems/gcsfuse { };

  glyr = callPackage ../tools/audio/glyr { };

  google-amber = callPackage ../tools/graphics/amber {
    inherit (darwin) cctools;
  };

  hakrawler = callPackage ../tools/security/hakrawler { };

  harsh = callPackage ../applications/misc/harsh { };

  harvid = callPackage ../tools/video/harvid { };

  headset = callPackage ../applications/audio/headset { };

  hilbish = callPackage ../shells/hilbish { };

  hime = callPackage ../tools/inputmethods/hime { };

  himitsu = callPackage ../tools/security/himitsu { };

  himitsu-firefox = callPackage ../tools/security/himitsu-firefox { };

  hinit = haskell.lib.compose.justStaticExecutables haskellPackages.hinit;

  hostctl = callPackage ../tools/system/hostctl { };

  hp2p = callPackage ../tools/networking/hp2p { };

  hpe-ltfs = callPackage ../tools/backup/hpe-ltfs { };

  http2tcp = callPackage ../tools/networking/http2tcp { };

  httperf = callPackage ../tools/networking/httperf { };

  hwi = with python3Packages; toPythonApplication hwi;

  ili2c = callPackage ../tools/misc/ili2c { };

  imagelol = callPackage ../tools/compression/imagelol { };

  imageworsener = callPackage ../tools/graphics/imageworsener { };

  imgpatchtools = callPackage ../development/mobile/imgpatchtools { };

  imgcrypt = callPackage ../applications/virtualization/imgcrypt { };

  ipgrep = callPackage ../tools/networking/ipgrep { };

  ipp-usb = callPackage ../os-specific/linux/ipp-usb { };

  itchiodl = callPackage ../games/itchiodl { };

  itd = callPackage ../applications/misc/itd { };

  kavita = callPackage ../servers/web-apps/kavita { };

  keka = callPackage ../tools/compression/keka { };

  kord = callPackage ../applications/misc/kord { };

  lastpass-cli = callPackage ../tools/security/lastpass-cli { };

  leetcode-cli = callPackage ../applications/misc/leetcode-cli { };

  lesspass-cli = callPackage ../tools/security/lesspass-cli { };

  livebook = callPackage ../servers/web-apps/livebook {
    elixir = elixir_1_16;
    beamPackages = beamPackages.extend (self: super: { elixir = elixir_1_16; });
  };

  lsix = callPackage ../tools/graphics/lsix { };

  mdr = callPackage ../tools/misc/mdr { };

  mobilecoin-wallet = callPackage ../applications/misc/mobilecoin-wallet { };

  pacparser = callPackage ../tools/networking/pacparser { };

  pairdrop = callPackage ../applications/misc/pairdrop { };

  opencbm = callPackage ../tools/misc/opencbm { };

  parquet-tools = callPackage ../tools/misc/parquet-tools { };

  pass = callPackage ../tools/security/pass { };

  passage = callPackage ../tools/security/passage { };

  passphrase2pgp = callPackage ../tools/security/passphrase2pgp { };

  pass-nodmenu = callPackage ../tools/security/pass {
    dmenuSupport = false;
    pass = pass-nodmenu;
  };

  pass-wayland = callPackage ../tools/security/pass {
    waylandSupport = true;
    pass = pass-wayland;
  };

  passExtensions = recurseIntoAttrs pass.extensions;

  pbpctrl = callPackage ../applications/audio/pbpctrl { };

  inherd-quake = callPackage ../applications/misc/inherd-quake {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  xjadeo = callPackage ../tools/video/xjadeo { };

  asc-key-to-qr-code-gif = callPackage ../tools/security/asc-key-to-qr-code-gif { };

  go-audit = callPackage ../tools/system/go-audit { };

  gopass = callPackage ../tools/security/gopass { };

  gopass-hibp = callPackage ../tools/security/gopass/hibp.nix { };

  gopass-jsonapi = callPackage ../tools/security/gopass/jsonapi.nix { };

  git-credential-gopass = callPackage ../tools/security/gopass/git-credential.nix { };

  gopass-summon-provider = callPackage ../tools/security/gopass/summon.nix { };

  gosh = callPackage ../tools/security/gosh { };

  gospider = callPackage ../tools/security/gospider { };

  browserpass = callPackage ../tools/security/browserpass { };

  passff-host = callPackage ../tools/security/passff-host { };

  oracle-instantclient = callPackage ../development/libraries/oracle-instantclient { };

  goku = callPackage ../os-specific/darwin/goku { };

  grandperspective = callPackage ../os-specific/darwin/grandperspective { };

  hexfiend = callPackage ../os-specific/darwin/hexfiend { };

  grb = callPackage ../applications/misc/grb { };

  kerf   = kerf_1; /* kerf2 is WIP */
  kerf_1 = callPackage ../development/interpreters/kerf {
    stdenv = clangStdenv;
    inherit (darwin.apple_sdk.frameworks)
      Accelerate CoreGraphics CoreVideo
    ;
  };

  kwakd = callPackage ../servers/kwakd { };

  kwm = callPackage ../os-specific/darwin/kwm { };

  khd = callPackage ../os-specific/darwin/khd {
    inherit (darwin.apple_sdk.frameworks) Carbon Cocoa;
  };

  kjv = callPackage ../applications/misc/kjv { };

  lukesmithxyz-bible-kjv = callPackage ../applications/misc/kjv/lukesmithxyz-kjv.nix { };

  luigi = callPackage ../applications/networking/cluster/luigi { };

  m-cli = callPackage ../os-specific/darwin/m-cli { };

  pebble = callPackage ../tools/admin/pebble { };

  play-with-mpv = callPackage ../tools/video/play-with-mpv { };

  plausible = callPackage ../servers/web-apps/plausible {
    elixir = elixir_1_14;
    beamPackages = beamPackages.extend (self: super: { elixir = elixir_1_14; });
  };

  pam-reattach = callPackage ../os-specific/darwin/pam-reattach { };

  reattach-to-user-namespace = callPackage ../os-specific/darwin/reattach-to-user-namespace { };

  qes = callPackage ../os-specific/darwin/qes {
    inherit (darwin.apple_sdk.frameworks) Carbon;
  };

  waydroid = callPackage ../os-specific/linux/waydroid { };

  wgo = callPackage ../development/tools/wgo { };

  wiiload = callPackage ../development/tools/wiiload { };

  winhelpcgi = callPackage ../development/tools/winhelpcgi { };

  wiimms-iso-tools = callPackage ../tools/filesystems/wiimms-iso-tools { };

  xc = callPackage ../development/tools/xc { };

  ios-webkit-debug-proxy = callPackage ../development/mobile/ios-webkit-debug-proxy { };

  xcodeenv = callPackage ../development/mobile/xcodeenv { };

  xcodes = swiftPackages.callPackage ../development/tools/xcodes {
    inherit (swiftPackages.apple_sdk.frameworks) CryptoKit LocalAuthentication;
    inherit (swiftPackages.apple_sdk) libcompression;
  };

  gomobile = callPackage ../development/mobile/gomobile { };

  ssh-agents = callPackage ../tools/networking/ssh-agents { };

  ssh-import-id = python3Packages.callPackage ../tools/admin/ssh-import-id { };

  ssh-key-confirmer = callPackage ../tools/networking/ssh-key-confirmer { };

  ssh-mitm = callPackage ../tools/security/ssh-mitm { };

  sshchecker = callPackage ../tools/security/sshchecker { };

  sshs = callPackage ../development/tools/sshs { };

  supergfxctl = callPackage ../applications/system/supergfxctl { };

  titanium = callPackage ../development/mobile/titanium { };

  titanium-alloy = callPackage ../development/mobile/titanium-alloy { };

  titaniumenv = callPackage ../development/mobile/titaniumenv { };

  abootimg = callPackage ../development/mobile/abootimg { };

  adbfs-rootless = callPackage ../development/mobile/adbfs-rootless { };

  adb-sync = callPackage ../development/mobile/adb-sync {
    inherit (androidenv.androidPkgs_9_0) platform-tools;
  };

  amoco = callPackage ../tools/security/amoco { };

  anbox = callPackage ../os-specific/linux/anbox {
    protobuf = protobuf_21;
  };

  androidenv = callPackage ../development/mobile/androidenv { };

  androidndkPkgs = androidndkPkgs_21;
  androidndkPkgs_21 = (callPackage ../development/androidndk-pkgs {})."21";
  androidndkPkgs_23b = (callPackage ../development/androidndk-pkgs {})."23b";
  androidndkPkgs_24 = (callPackage ../development/androidndk-pkgs {})."24";

  androidsdk_9_0 = androidenv.androidPkgs_9_0.androidsdk;

  webos = recurseIntoAttrs {
    cmake-modules = callPackage ../development/mobile/webos/cmake-modules.nix { };

    novacom = callPackage ../development/mobile/webos/novacom.nix { };
    novacomd = callPackage ../development/mobile/webos/novacomd.nix { };
  };

  anevicon = callPackage ../tools/networking/anevicon {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  aoc-cli = callPackage ../tools/misc/aoc-cli {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  apprise = with python3Packages; toPythonApplication apprise;

  aptdec = callPackage ../development/libraries/aptdec { };

  aria2 = callPackage ../tools/networking/aria2 {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  as-tree = callPackage ../tools/misc/as-tree { };

  asmfmt = callPackage ../development/tools/asmfmt { };

  asmrepl = callPackage ../development/interpreters/asmrepl { };

  aspcud = callPackage ../tools/misc/aspcud { };

  at = callPackage ../tools/system/at { };

  atftp = callPackage ../tools/networking/atftp { };

  atlas = callPackage ../development/tools/database/atlas { };

  authoscope = callPackage ../tools/security/authoscope {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  autogen = callPackage ../development/tools/misc/autogen { };

  autojump = callPackage ../tools/misc/autojump { };

  automysqlbackup = callPackage ../tools/backup/automysqlbackup { };

  autorandr = callPackage ../tools/misc/autorandr { };

  avahi = callPackage ../development/libraries/avahi (config.avahi or {});

  avahi-compat = callPackage ../development/libraries/avahi ((config.avahi or {}) // {
    withLibdnssdCompat = true;
  });

  avro-c = callPackage ../development/libraries/avro-c { };

  avro-cpp = callPackage ../development/libraries/avro-c++ { };

  aws_mturk_clt = callPackage ../tools/misc/aws-mturk-clt { };

  awsls = callPackage ../tools/admin/awsls { };

  awsrm = callPackage ../tools/admin/awsrm { };

  awstats = callPackage ../tools/system/awstats { };

  awsweeper = callPackage ../tools/admin/awsweeper { };

  axel = callPackage ../tools/networking/axel {
    libssl = openssl;
  };

  b3sum = callPackage ../tools/security/b3sum { };

  backblaze-b2 = callPackage ../development/tools/backblaze-b2 { };

  bandwhich = callPackage ../tools/networking/bandwhich {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  badrobot = callPackage ../tools/security/badrobot { };

  bao = callPackage ../tools/security/bao { };

  bar = callPackage ../tools/system/bar { };

  base16-builder = callPackage ../misc/base16-builder { };

  basex = callPackage ../tools/text/xml/basex { };

  bashplotlib = callPackage ../tools/misc/bashplotlib { };

  babeld = callPackage ../tools/networking/babeld { };

  babelfish = callPackage ../shells/fish/babelfish.nix { };

  baboossh = callPackage ../tools/security/baboossh { };

  badchars = python3Packages.callPackage ../tools/security/badchars { };

  badvpn = callPackage ../tools/networking/badvpn { };

  barcode = callPackage ../tools/graphics/barcode { };

  bashmount = callPackage ../tools/filesystems/bashmount { };

  bat-extras = recurseIntoAttrs (callPackages ../tools/misc/bat-extras { });

  beats = callPackage ../tools/misc/beats { };

  BeatSaberModManager = callPackage ../games/BeatSaberModManager/default.nix { };

  beauty-line-icon-theme = callPackage ../data/icons/beauty-line-icon-theme {
    inherit (plasma5Packages) breeze-icons;
  };

  beautysh = with python3.pkgs; toPythonApplication beautysh;

  bc = callPackage ../tools/misc/bc { };

  gavin-bc = callPackage ../tools/misc/gavin-bc { };

  bdf2psf = callPackage ../tools/misc/bdf2psf { };

  bdf2sfd = callPackage ../tools/misc/bdf2sfd { };

  bdfresize = callPackage ../tools/misc/bdfresize { };

  bcache-tools = callPackage ../tools/filesystems/bcache-tools { };

  bchunk = callPackage ../tools/cd-dvd/bchunk { };

  inherit (callPackages ../misc/logging/beats/7.x.nix { })
    auditbeat7
    filebeat7
    heartbeat7
    metricbeat7
    packetbeat7;

  auditbeat = auditbeat7;
  filebeat = filebeat7;
  heartbeat = heartbeat7;
  metricbeat = metricbeat7;
  packetbeat = packetbeat7;

  bfr = callPackage ../tools/misc/bfr { };

  bfscripts = callPackage ../tools/misc/bfscripts { };

  bibtool = callPackage ../tools/misc/bibtool { };

  bibutils = callPackage ../tools/misc/bibutils { };

  bibtex2html = callPackage ../tools/misc/bibtex2html { };

  bicon = callPackage ../applications/misc/bicon { };

  biliass = with python3.pkgs; toPythonApplication biliass;

  bindfs = callPackage ../tools/filesystems/bindfs { };

  binwalk = with python3Packages; toPythonApplication binwalk;

  birdtray = libsForQt5.callPackage ../applications/misc/birdtray { };

  blitz = callPackage ../development/libraries/blitz { };

  blockbook = callPackage ../servers/blockbook { };

  blockhash = callPackage ../tools/graphics/blockhash { };

  bluemix-cli = callPackage ../tools/admin/bluemix-cli { };

  bluewalker = callPackage ../tools/bluetooth/bluewalker { };

  blur-effect = callPackage ../tools/graphics/blur-effect { };

  bootiso = callPackage ../tools/cd-dvd/bootiso { };

  butane = callPackage ../development/tools/butane { };

  buttercup-desktop = callPackage ../tools/security/buttercup-desktop { };

  charles = charles4;
  inherit (callPackages ../applications/networking/charles {})
    charles3
    charles4
  ;

  quaternion-qt5 = libsForQt5.callPackage ../applications/networking/instant-messengers/quaternion { };
  quaternion-qt6 = qt6Packages.callPackage ../applications/networking/instant-messengers/quaternion { };
  quaternion = quaternion-qt6;

  tensor = libsForQt5.callPackage ../applications/networking/instant-messengers/tensor { };

  libtensorflow = python3.pkgs.tensorflow.libtensorflow;

  libtorch-bin = callPackage ../development/libraries/science/math/libtorch/bin.nix {
    inherit (config) cudaSupport;
  };

  tensorflow-lite = callPackage ../development/libraries/science/math/tensorflow-lite { };

  tiny-cuda-nn = callPackage ../development/libraries/science/math/tiny-cuda-nn { };

  tezos-rust-libs = callPackage ../development/libraries/tezos-rust-libs { };

  behave = with python3Packages; toPythonApplication behave;

  behdad-fonts = callPackage ../data/fonts/behdad-fonts { };

  bfetch = callPackage ../tools/misc/bfetch { };

  bless = callPackage ../applications/editors/bless { };

  blink = darwin.apple_sdk_11_0.callPackage ../applications/emulators/blink { };

  blink1-tool = callPackage ../tools/misc/blink1-tool { };

  blis = callPackage ../development/libraries/science/math/blis { };

  bliss = callPackage ../applications/science/math/bliss { };

  blobfuse = callPackage ../tools/filesystems/blobfuse { };

  blockdiag = with python3Packages; toPythonApplication blockdiag;

  bmon = callPackage ../tools/misc/bmon { };

  boca = callPackage ../development/libraries/boca { };

  bubblewrap = callPackage ../tools/admin/bubblewrap { };

  borgbackup = callPackage ../tools/backup/borgbackup { };

  borgmatic = callPackage ../tools/backup/borgmatic { };

  borg-sans-mono = callPackage ../data/fonts/borg-sans-mono { };

  boringtun = callPackage ../tools/networking/boringtun { };

  book-summary = callPackage ../tools/text/book-summary { };

  bookstack = callPackage ../servers/web-apps/bookstack { };

  boomerang = libsForQt5.callPackage ../development/tools/boomerang { };

  boost-build = callPackage ../development/tools/boost-build { };

  boot = callPackage ../development/tools/build-managers/boot { };

  bowtie = callPackage ../applications/science/biology/bowtie { };

  bowtie2 = callPackage ../applications/science/biology/bowtie2 { };

  boxfs = callPackage ../tools/filesystems/boxfs { };

  bozohttpd = callPackage ../servers/http/bozohttpd { };
  bozohttpd-minimal = callPackage ../servers/http/bozohttpd { minimal = true; };

  brasero-original = lowPrio (callPackage ../tools/cd-dvd/brasero { });

  brasero = callPackage ../tools/cd-dvd/brasero/wrapper.nix { };

  brigand = callPackage ../development/libraries/brigand { };

  brltty = callPackage ../tools/misc/brltty { };

  brook = callPackage ../tools/networking/brook { };

  broot = callPackage ../tools/misc/broot {
    inherit (darwin.apple_sdk.frameworks) Foundation Security;
  };

  bruteforce-luks = callPackage ../tools/security/bruteforce-luks { };

  brutespray = callPackage ../tools/security/brutespray { };

  breakpointHook = assert stdenv.buildPlatform.isLinux;
    makeSetupHook {
      name = "breakpoint-hook";
    } ../build-support/setup-hooks/breakpoint-hook.sh;

  btrfs-progs = callPackage ../tools/filesystems/btrfs-progs { };

  btrfs-snap = callPackage ../tools/filesystems/btrfs-snap { };

  ssdfs-utils = callPackage ../tools/filesystems/ssdfs-utils { };

  btlejack = python3Packages.callPackage ../applications/radio/btlejack { };

  btrbk = callPackage ../tools/backup/btrbk { };

  buildpack = callPackage ../development/tools/buildpack { };

  bonk = callPackage ../tools/misc/bonk { };

  bottom-rs = callPackage ../tools/misc/bottom-rs { };

  bsp-layout = callPackage ../tools/misc/bsp-layout { };

  buildtorrent = callPackage ../tools/misc/buildtorrent { };

  bundletool = callPackage ../development/tools/bundletool { };

  bwm_ng = callPackage ../tools/networking/bwm-ng { };

  bwbasic = callPackage ../development/interpreters/bwbasic { };

  bws = callPackage ../tools/security/bws { };

  bsh = fetchurl {
    url = "http://www.beanshell.org/bsh-2.0b5.jar";
    hash = "sha256-YjIZlWOAc1SzvLWs6z3BNlAvAixrDvdDmHqD9m/uWlw=";
  };

  btfs = callPackage ../os-specific/linux/btfs { };

  buildah = callPackage ../development/tools/buildah/wrapper.nix { };
  buildah-unwrapped = callPackage ../development/tools/buildah { };

  buildkit = callPackage ../development/tools/buildkit { };

  bukubrow = callPackage ../tools/networking/bukubrow { };

  burpsuite = callPackage ../tools/networking/burpsuite { };

  ciano = callPackage ../applications/graphics/ciano {
    inherit (pantheon) granite;
    python = python3;
    gtk = gtk3;
  };

  c3d = callPackage ../applications/graphics/c3d {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  cue2pops = callPackage ../tools/cd-dvd/cue2pops { };

  cabal2nix-unwrapped = haskell.lib.compose.justStaticExecutables
    (haskellPackages.generateOptparseApplicativeCompletions [ "cabal2nix" ] haskellPackages.cabal2nix);

  cabal2nix = symlinkJoin {
    inherit (cabal2nix-unwrapped) name meta;
    nativeBuildInputs = [ buildPackages.makeWrapper ];
    paths = [ cabal2nix-unwrapped ];
    postBuild = ''
      wrapProgram $out/bin/cabal2nix \
        --prefix PATH ":" "${lib.makeBinPath [ nix nix-prefetch-scripts ]}"
    '';
  };

  stack2nix = with haskell.lib; overrideCabal (justStaticExecutables haskellPackages.stack2nix) (_: {
    executableToolDepends = [ makeWrapper ];
    postInstall = ''
      wrapProgram $out/bin/stack2nix \
        --prefix PATH ":" "${git}/bin:${cabal-install}/bin"
    '';
  });

  traefik = callPackage ../servers/traefik { };

  traefik-certs-dumper = callPackage ../tools/misc/traefik-certs-dumper { };

  caffeine = callPackage ../tools/misc/caffeine { };

  calamares = libsForQt5.callPackage ../tools/misc/calamares {
    python = python3;
    boost = boost.override { enablePython = true; python = python3; };
  };
  calamares-nixos = lowPrio (calamares.override { nixos-extensions = true; });
  calamares-nixos-extensions = callPackage ../tools/misc/calamares-nixos-extensions { };

  calendar-cli = callPackage ../tools/networking/calendar-cli { };

  cameradar = callPackage ../tools/security/cameradar { };

  candle = libsForQt5.callPackage ../applications/misc/candle { };

  capstone = callPackage ../development/libraries/capstone { };
  capstone_4 = callPackage ../development/libraries/capstone/4.nix { };

  keystone = callPackage ../development/libraries/keystone { };

  casync = callPackage ../applications/networking/sync/casync {
    sphinx = buildPackages.python3Packages.sphinx;
  };

  cataract          = callPackage ../applications/misc/cataract { };
  cataract-unstable = callPackage ../applications/misc/cataract/unstable.nix { };

  catch = callPackage ../development/libraries/catch { };

  catch2 = callPackage ../development/libraries/catch2 { };

  catch2_3 = callPackage ../development/libraries/catch2/3.nix { };

  catdoc = callPackage ../tools/text/catdoc { };

  catdocx = callPackage ../tools/text/catdocx { };

  catclock = callPackage ../applications/misc/catclock { };

  cardpeek = callPackage ../applications/misc/cardpeek { inherit (darwin.apple_sdk.frameworks) PCSC; };

  cde = callPackage ../tools/package-management/cde { };

  cdxgen = callPackage ../tools/security/cdxgen { };

  ceres-solver = callPackage ../development/libraries/ceres-solver {
    gflags = null; # only required for examples/tests
  };

  cbor-diag = callPackage ../development/tools/cbor-diag { };

  cassowary = callPackage ../tools/networking/cassowary { };

  croc = callPackage ../tools/networking/croc { };

  cbftp = callPackage ../tools/networking/cbftp { };

  cddl = callPackage ../development/tools/cddl { };

  cedille = callPackage ../applications/science/logic/cedille
                          { inherit (haskellPackages) alex happy Agda ghcWithPackages;
                          };

  cel-go = callPackage ../development/interpreters/cel-go { };

  cfdyndns = callPackage ../applications/networking/dyndns/cfdyndns { };

  cf-terraforming = callPackage ../tools/misc/cf-terraforming { };

  charliecloud = callPackage ../applications/virtualization/charliecloud { };

  chelf = callPackage ../tools/misc/chelf { };

  chisel = callPackage ../tools/networking/chisel { };

  cht-sh = callPackage ../tools/misc/cht.sh { };

  cinny = callPackage ../applications/networking/instant-messengers/cinny { };

  cinny-desktop = callPackage ../applications/networking/instant-messengers/cinny-desktop { };

  ckbcomp = callPackage ../tools/X11/ckbcomp { };

  clac = callPackage ../tools/misc/clac { };

  map-cmd = callPackage ../tools/misc/map { };

  clash-geoip = callPackage ../data/misc/clash-geoip { };

  clevercsv = with python3Packages; toPythonApplication clevercsv;

  clevis = callPackage ../tools/security/clevis {
    asciidoc = asciidoc-full;
  };

  cli53 = callPackage ../tools/admin/cli53 { };

  cli-visualizer = callPackage ../applications/misc/cli-visualizer { };

  cliam = callPackage ../tools/security/cliam { };

  clickclack = callPackage ../tools/misc/clickclack { };

  clickgen = with python3Packages; toPythonApplication clickgen;

  clog-cli = callPackage ../development/tools/clog-cli { };

  cloud-init = python3.pkgs.callPackage ../tools/virtualization/cloud-init { inherit systemd; };

  cloudbrute = callPackage ../tools/security/cloudbrute { };

  cloudflared = callPackage ../applications/networking/cloudflared {
    # https://github.com/cloudflare/cloudflared/issues/1151#issuecomment-1888819250
    buildGoModule = buildGoModule.override {
      go = go_1_21.overrideAttrs {
        pname = "cloudflare-go";
        version = "0-unstable-2023-12-06";
        src = fetchFromGitHub {
          owner = "cloudflare";
          repo = "go";
          rev = "34129e47042e214121b6bbff0ded4712debed18e";
          sha256 = "sha256-RA9KTY4cSxIt7dyJgAFQPemc6YBgcSwc/hqB4JHPxng=";
        };
      };
    };
  };

  cloudflare-dyndns = callPackage ../applications/networking/cloudflare-dyndns { };

  cloudmonkey = callPackage ../tools/virtualization/cloudmonkey { };

  clib = callPackage ../tools/package-management/clib { };

  clingo = callPackage ../applications/science/logic/potassco/clingo.nix { };

  clingcon = callPackage ../applications/science/logic/potassco/clingcon.nix { };

  clprover = callPackage ../applications/science/logic/clprover/clprover.nix { };

  clusterctl = callPackage ../applications/networking/cluster/clusterctl { };

  coloredlogs = with python3Packages; toPythonApplication coloredlogs;

  colpack = callPackage ../applications/science/math/colpack { };

  cz-cli = callPackage ../applications/version-management/cz-cli { };

  comma = callPackage ../tools/package-management/comma { };

  commitizen = callPackage ../applications/version-management/commitizen { };

  common-licenses = callPackage ../data/misc/common-licenses { };

  compactor = callPackage ../applications/networking/compactor {
    protobuf = protobuf_21;
  };

  consul-alerts = callPackage ../servers/monitoring/consul-alerts { };

  consul-template = callPackage ../tools/system/consul-template { };

  copyright-update = callPackage ../tools/text/copyright-update { };

  inherit (callPackages ../tools/misc/coreboot-utils { })
    msrtool
    cbmem
    ifdtool
    intelmetool
    cbfstool
    nvramtool
    superiotool
    ectool
    inteltool
    amdfwtool
    acpidump-all
    coreboot-utils;

  coreboot-configurator = libsForQt5.callPackage ../tools/misc/coreboot-configurator { };

  corosync = callPackage ../servers/corosync { };

  cowsay = callPackage ../tools/misc/cowsay { };

  fw-ectool = callPackage ../os-specific/linux/fw-ectool { };

  chayang = callPackage ../tools/wayland/chayang { };

  cherrytree = callPackage ../applications/misc/cherrytree { };

  chntpw = callPackage ../tools/security/chntpw { };

  chopchop = callPackage ../tools/security/chopchop { };

  clapboard = callPackage ../tools/wayland/clapboard { };

  cliphist = callPackage ../tools/wayland/cliphist { };

  clipman = callPackage ../tools/wayland/clipman { };

  kabeljau = callPackage ../games/kabeljau { };

  kanshi = callPackage ../tools/wayland/kanshi { };

  oguri = callPackage  ../tools/wayland/oguri { };

  shikane = callPackage ../tools/wayland/shikane { };

  shotman = callPackage ../tools/wayland/shotman { };

  sway-unwrapped = callPackage ../by-name/sw/sway-unwrapped/package.nix {
    wlroots = wlroots_0_17;
  };

  swaykbdd = callPackage ../tools/wayland/swaykbdd { };

  swayr = callPackage ../tools/wayland/swayr { };

  swayrbar = callPackage ../tools/wayland/swayrbar { };

  swaysome = callPackage ../tools/wayland/swaysome { };

  swaytools = python3Packages.callPackage ../tools/wayland/swaytools { };

  wayland-utils = callPackage ../tools/wayland/wayland-utils { };

  wayland-proxy-virtwl = callPackage ../tools/wayland/wayland-proxy-virtwl { };

  waylogout = callPackage ../tools/wayland/waylogout { };

  waynergy = callPackage ../tools/wayland/waynergy { };

  wayout = callPackage ../tools/wayland/wayout { };

  way-displays = callPackage ../tools/wayland/way-displays { };

  wev = callPackage ../tools/wayland/wev { };

  wdomirror = callPackage ../tools/wayland/wdomirror { };

  wdt = callPackage ../applications/networking/sync/wdt { };

  wlay = callPackage ../tools/wayland/wlay { };

  wl-clipboard = callPackage ../tools/wayland/wl-clipboard { };

  wl-clipboard-x11 = callPackage ../tools/wayland/wl-clipboard-x11 { };

  wl-color-picker = callPackage ../tools/wayland/wl-color-picker { };

  wl-mirror = callPackage ../tools/wayland/wl-mirror { };

  wl-screenrec = callPackage ../tools/wayland/wl-screenrec {
    ffmpeg = ffmpeg_5;
  };

  wleave = callPackage ../tools/wayland/wleave { };

  wlopm = callPackage ../tools/wayland/wlopm { };

  wlprop = callPackage ../tools/wayland/wlprop { };

  wlr-randr = callPackage ../tools/wayland/wlr-randr { };

  wlr-which-key = callPackage ../tools/wayland/wlr-which-key { };

  wlrctl = callPackage ../tools/wayland/wlrctl { };

  wlsunset = callPackage ../tools/wayland/wlsunset { };

  wl-gammactl = callPackage ../tools/wayland/wl-gammactl { };

  wluma = callPackage ../tools/wayland/wluma {  };

  wob = callPackage ../tools/wayland/wob { };

  wshowkeys = callPackage ../tools/wayland/wshowkeys { };

  wtype = callPackage ../tools/wayland/wtype { };

  cambalache = callPackage ../development/tools/cambalache { };

  cambrinary = python3Packages.callPackage ../applications/misc/cambrinary { };

  changedetection-io = callPackage ../servers/web-apps/changedetection-io { };

  clini = callPackage ../tools/misc/clini { };

  clipster = callPackage ../tools/misc/clipster { };

  clockify = callPackage ../applications/office/clockify { };

  contrast = callPackage ../applications/accessibility/contrast { };

  cplex = callPackage ../applications/science/math/cplex (config.cplex or {});

  cpulimit = callPackage ../tools/misc/cpulimit { };

  code-minimap = callPackage ../tools/misc/code-minimap { };

  codesearch = callPackage ../tools/text/codesearch { };

  codec2 = callPackage ../development/libraries/codec2 { };

  contacts = callPackage ../tools/misc/contacts {
    inherit (darwin.apple_sdk.frameworks) Foundation AddressBook;
  };

  colorls = callPackage ../tools/system/colorls { };

  coloursum = callPackage ../tools/text/coloursum {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  compsize = callPackage ../os-specific/linux/compsize { };

  comrak = callPackage ../tools/text/comrak { };

  cot = with python3Packages; toPythonApplication cot;

  coturn = callPackage ../servers/coturn { };

  coursier = callPackage ../development/tools/coursier { };

  cppclean = callPackage ../development/tools/cppclean { };

  credhub-cli = callPackage ../tools/admin/credhub-cli { };

  crex = callPackage ../tools/misc/crex { };

  cri-tools = callPackage ../tools/virtualization/cri-tools { };

  crip = callPackage ../applications/audio/crip { };

  crosvm = callPackage ../applications/virtualization/crosvm { };

  crossplane = with python3Packages; toPythonApplication crossplane;

  crowdsec = callPackage ../tools/security/crowdsec { };

  crunch = callPackage ../tools/security/crunch { };

  crudini = callPackage ../tools/misc/crudini { };

  csv2odf = callPackage ../applications/office/csv2odf { };

  csv2parquet = callPackage ../tools/misc/csv2parquet { };

  csvdiff = callPackage ../tools/text/csvdiff { };

  csview = callPackage ../tools/text/csview { };

  csvquote = callPackage ../tools/text/csvquote { };

  csvtool = callPackage ../development/ocaml-modules/csv/csvtool.nix { };

  csv2latex = callPackage ../tools/misc/csv2latex { };

  csvs-to-sqlite = callPackage ../tools/misc/csvs-to-sqlite { };

  cutemarked-ng = libsForQt5.callPackage ../applications/office/cutemarked-ng { };

  dab_lib = callPackage ../development/libraries/dab_lib { };

  dabet = callPackage ../tools/misc/dabet { };

  dabtools = callPackage ../applications/radio/dabtools { };

  daemontools = callPackage ../tools/admin/daemontools { };

  dale = callPackage ../development/compilers/dale { };

  damon = callPackage ../tools/admin/damon { };

  dancing-script = callPackage ../data/fonts/dancing-script { };

  dante = callPackage ../servers/dante { };

  dasel = callPackage ../applications/misc/dasel { };

  dasher = callPackage ../applications/accessibility/dasher { };

  dataclass-wizard = with python3Packages; toPythonApplication dataclass-wizard;

  datafusion-cli = callPackage ../development/misc/datafusion { };

  datamash = callPackage ../tools/misc/datamash { };

  datasette = with python3Packages; toPythonApplication datasette;

  howard-hinnant-date = callPackage ../development/libraries/howard-hinnant-date { };

  datefmt = callPackage ../tools/misc/datefmt { };

  datefudge = callPackage ../tools/system/datefudge { };

  dateutils = callPackage ../tools/misc/dateutils { };

  datovka = libsForQt5.callPackage ../applications/networking/datovka { };

  dave = callPackage ../servers/http/dave { };

  dconf = callPackage ../development/libraries/dconf { };

  ddate = callPackage ../tools/misc/ddate { };

  ddosify = callPackage ../development/tools/ddosify { };

  dduper = callPackage ../tools/filesystems/dduper { };

  deck = callPackage ../applications/networking/deck { };

  dedup = callPackage ../tools/backup/dedup { };

  deheader = callPackage ../development/tools/misc/deheader { };

  dehydrated = callPackage ../tools/admin/dehydrated { };

  deja-dup = callPackage ../applications/backup/deja-dup { };

  dejsonlz4 = callPackage ../tools/compression/dejsonlz4 { };

  desync = callPackage ../applications/networking/sync/desync { };

  devdocs-desktop = callPackage ../applications/misc/devdocs-desktop { };

  devmem2 = callPackage ../os-specific/linux/devmem2 { };

  dbus-broker = callPackage ../os-specific/linux/dbus-broker { };

  dblab = callPackage ../development/tools/database/dblab { };

  dbx = callPackage ../applications/misc/dbx { };

  ioport = callPackage ../os-specific/linux/ioport { };

  diagrams-builder = callPackage ../tools/graphics/diagrams-builder {
    inherit (haskellPackages) ghcWithPackages diagrams-builder;
  };

  dialect = callPackage ../applications/misc/dialect { };

  dialogbox = libsForQt5.callPackage ../tools/misc/dialogbox { };

  dieharder = callPackage ../tools/security/dieharder { };

  diesel-cli = callPackage ../development/tools/diesel-cli {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  digitemp = callPackage ../tools/misc/digitemp { };

  dijo = callPackage ../tools/misc/dijo {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  ding = callPackage ../applications/misc/ding {
    aspellDicts_de = aspellDicts.de;
    aspellDicts_en = aspellDicts.en;
  };

  dirb = callPackage ../tools/networking/dirb { };

  direnv = callPackage ../tools/misc/direnv { };

  dirstalk = callPackage ../tools/security/dirstalk { };

  h = callPackage ../tools/misc/h { };

  discount = callPackage ../tools/text/discount { };

  discourse = callPackage ../servers/web-apps/discourse { };

  discourseAllPlugins = discourse.override {
    plugins = lib.filter (p: p ? pluginName) (builtins.attrValues discourse.plugins);
  };

  discourse-mail-receiver = callPackage ../servers/web-apps/discourse/mail_receiver { };

  discocss = callPackage ../tools/misc/discocss { };

  disfetch = callPackage ../tools/misc/disfetch { };

  disk-inventory-x = callPackage ../tools/filesystems/disk-inventory-x { };

  diskscan = callPackage ../tools/misc/diskscan { };

  disorderfs = callPackage ../tools/filesystems/disorderfs {
    asciidoc = asciidoc-full;
  };

  dislocker = callPackage ../tools/filesystems/dislocker { };

  distgen = callPackage ../development/tools/distgen { };

  distrobuilder = callPackage ../tools/virtualization/distrobuilder { };

  ditaa = callPackage ../tools/graphics/ditaa { };

  dino = callPackage ../applications/networking/instant-messengers/dino {
    inherit (gst_all_1) gstreamer gst-plugins-base gst-plugins-bad gst-vaapi;
    gst-plugins-good = gst_all_1.gst-plugins-good.override { gtkSupport = true; };
  };

  djot-js = callPackage ../tools/text/djot-js { };

  djv = callPackage ../applications/graphics/djv { };

  dnschef = python3Packages.callPackage ../tools/networking/dnschef { };

  doitlive = callPackage ../tools/misc/doitlive { };

  dokuwiki = callPackage ../servers/web-apps/dokuwiki { };

  dolibarr = callPackage ../servers/web-apps/dolibarr { };

  dosage = callPackage ../applications/graphics/dosage { };

  dotenv-linter = callPackage ../development/tools/analysis/dotenv-linter {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  dotool = callPackage ../tools/dotool { };

  inherit (ocamlPackages) dot-merlin-reader;

  dozenal = callPackage ../applications/misc/dozenal { };

  dpic = callPackage ../tools/graphics/dpic { };

  dsdcc = callPackage ../development/libraries/dsdcc { };

  dstp = callPackage ../development/tools/dstp { };

  dsvpn = callPackage ../applications/networking/dsvpn { };

  dwt1-shell-color-scripts = callPackage ../tools/misc/dwt1-shell-color-scripts { };

  dtrx = callPackage ../tools/compression/dtrx { };

  dua = callPackage ../tools/misc/dua { };

  duf = callPackage ../tools/misc/duf { };

  dum = callPackage ../development/tools/dum { };

  inherit (ocaml-ng.ocamlPackages_4_10) dune_1;
  inherit (ocamlPackages) dune_2 dune_3 dune-release;

  duperemove = callPackage ../tools/filesystems/duperemove { };

  dvc = with python3.pkgs; toPythonApplication dvc;

  dvc-with-remotes = dvc.override {
    enableGoogle = true;
    enableAWS = true;
    enableAzure = true;
    enableSSH = true;
  };

  dynamic-colors = callPackage ../tools/misc/dynamic-colors { };

  dynamic-wallpaper = callPackage ../tools/graphics/dynamic-wallpaper { };

  dyncall = callPackage ../development/libraries/dyncall { };

  dyndnsc = callPackage ../applications/networking/dyndns/dyndnsc { };

  earthly = callPackage ../development/tools/earthly { };

  earlybird = callPackage ../tools/security/earlybird { };

  easyabc = callPackage ../applications/audio/easyabc { };

  easyaudiosync = qt6Packages.callPackage ../applications/audio/easyaudiosync {};

  easycrypt = callPackage ../applications/science/logic/easycrypt {
    why3 = pkgs.why3.override { ideSupport = false; };
  };

  easycrypt-runtest = callPackage ../applications/science/logic/easycrypt/runtest.nix { };

  easyocr = with python3.pkgs; toPythonApplication easyocr;

  ecasound = callPackage ../applications/audio/ecasound { };

  edac-utils = callPackage ../os-specific/linux/edac-utils { };

  eddy = libsForQt5.callPackage ../applications/graphics/eddy { };

  eggdrop = callPackage ../tools/networking/eggdrop { };

  egglog = callPackage ../applications/science/logic/egglog { };

  ego = callPackage ../tools/misc/ego { };

  ekam = callPackage ../development/tools/build-managers/ekam { };

  electronplayer = callPackage ../applications/video/electronplayer/electronplayer.nix { };

  electron-mail = callPackage ../applications/networking/mailreaders/electron-mail { };

  element = callPackage ../applications/science/chemistry/element { };

  element-desktop = callPackage ../applications/networking/instant-messengers/element/element-desktop.nix {
    inherit (darwin.apple_sdk.frameworks) Security AppKit CoreServices;
    electron = electron_30;
  };
  element-desktop-wayland = writeScriptBin "element-desktop" ''
    #!/bin/sh
    NIXOS_OZONE_WL=1 exec ${element-desktop}/bin/element-desktop "$@"
  '';

  element-web-unwrapped = callPackage ../applications/networking/instant-messengers/element/element-web.nix { };

  element-web = callPackage ../applications/networking/instant-messengers/element/element-web-wrapper.nix {
    conf = config.element-web.conf or { };
  };

  elementary-xfce-icon-theme = callPackage ../data/icons/elementary-xfce-icon-theme { };

  ell = callPackage ../os-specific/linux/ell { };

  elm-github-install = callPackage ../tools/package-management/elm-github-install { };

  elogind = callPackage ../applications/misc/elogind { };

  elpa = callPackage ../development/libraries/elpa { };

  enca = callPackage ../tools/text/enca { };

  encpipe = callPackage ../tools/misc/encpipe { };

  ent = callPackage ../tools/misc/ent { };

  entwine = callPackage ../applications/graphics/entwine { };

  envconsul = callPackage ../tools/system/envconsul { };

  envsubst = callPackage ../tools/misc/envsubst { };

  envfs = callPackage ../tools/filesystems/envfs { };

  er-patcher = callPackage ../tools/games/er-patcher { };

  erdtree = callPackage ../tools/system/erdtree { };

  errcheck = callPackage ../development/tools/errcheck { };

  eschalot = callPackage ../tools/security/eschalot { };

  espanso = callPackage ../applications/office/espanso {
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa Foundation IOKit Kernel AVFoundation Carbon QTKit AVKit WebKit;
  };
  espanso-wayland = espanso.override {
    x11Support = false;
    waylandSupport = true;
    espanso = espanso-wayland;
  };

  esphome = callPackage ../tools/misc/esphome { };

  esptool = callPackage ../tools/misc/esptool { };

  esptool-ck = callPackage ../tools/misc/esptool-ck { };

  ephemeralpg = callPackage ../development/tools/database/ephemeralpg { };

  et = callPackage ../applications/misc/et { };

  ejson = callPackage ../development/tools/ejson { };

  eternal-terminal = callPackage ../tools/networking/eternal-terminal { };

  extism-cli = callPackage ../development/tools/extism-cli { };

  extrude = callPackage ../tools/security/extrude { };

  fastly = callPackage ../misc/fastly {
    # If buildGoModule is overridden, provide a matching version of the go attribute
  };

  f2 = callPackage ../tools/misc/f2 { };

  f3 = callPackage ../tools/filesystems/f3 { };

  f3d = callPackage ../applications/graphics/f3d {
    inherit (darwin.apple_sdk.frameworks) Cocoa OpenGL;
  };

  facedetect = callPackage ../tools/graphics/facedetect { };

  faketty = callPackage ../tools/misc/faketty { };

  fasd = callPackage ../tools/misc/fasd { };

  fastJson = callPackage ../development/libraries/fastjson { };

  fast-cli = nodePackages.fast-cli;

  fast-cpp-csv-parser = callPackage ../development/libraries/fast-cpp-csv-parser { };

  fast-ssh = callPackage ../tools/networking/fast-ssh {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  faudio = callPackage ../development/libraries/faudio { };

  fblog = callPackage ../development/tools/fblog { };

  fd = callPackage ../tools/misc/fd { };

  fdroidcl = pkgs.callPackage ../development/mobile/fdroidcl { };

  fdroidserver = python3Packages.callPackage ../development/tools/fdroidserver { };

  fetch-scm = callPackage ../tools/misc/fetch-scm { };

  fiano = callPackage ../tools/misc/fiano { };

  filebench = callPackage ../tools/misc/filebench { };

  filebot = callPackage ../applications/video/filebot { };

  fileshare = callPackage ../servers/fileshare { };

  fileshelter = callPackage ../servers/web-apps/fileshelter { };

  fioctl = callPackage ../tools/admin/fioctl { };

  firecracker = callPackage ../applications/virtualization/firecracker { };

  firectl = callPackage ../applications/virtualization/firectl { };

  firestarter = callPackage ../applications/misc/firestarter { };

  fits-cloudctl = callPackage ../tools/admin/fits-cloudctl { };

  flitter = callPackage ../tools/misc/flitter { };

  flowgger = callPackage ../tools/misc/flowgger {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  frangipanni = callPackage ../tools/text/frangipanni { };

  frawk = callPackage ../tools/text/frawk { };

  fre = callPackage ../tools/misc/fre { };

  frei = callPackage ../tools/misc/frei { };

  frogmouth = callPackage ../tools/text/frogmouth { };

  fselect = callPackage ../tools/misc/fselect { };

  fsmon = callPackage ../tools/misc/fsmon { };

  fst = callPackage ../tools/text/fst { };

  fsql = callPackage ../tools/misc/fsql { };

  kitty-img = callPackage ../tools/misc/kitty-img { };

  ### TOOLS/TYPESETTING/TEX

  advi = callPackage ../tools/typesetting/tex/advi {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  auctex = callPackage ../tools/typesetting/tex/auctex { };

  blahtexml = callPackage ../tools/typesetting/tex/blahtexml { };

  catdvi = callPackage ../tools/typesetting/tex/catdvi { };

  dblatex = callPackage ../tools/typesetting/tex/dblatex { };

  dblatexFull = dblatex.override { enableAllFeatures = true; };

  gladtex = callPackage ../tools/typesetting/tex/gladtex { };

  latex2mathml = with python3Packages; toPythonApplication latex2mathml;

  latexrun = callPackage ../tools/typesetting/tex/latexrun { };

  lkproof = callPackage ../tools/typesetting/tex/lkproof { };

  mftrace = callPackage ../tools/typesetting/tex/mftrace { };

  # Keep the old PGF since some documents don't render properly with
  # the new one.
  pgf1 = callPackage ../tools/typesetting/tex/pgf-tikz/pgf-1.x.nix { };

  pgf2 = callPackage ../tools/typesetting/tex/pgf-tikz/pgf-2.x.nix { };

  pgf3 = callPackage ../tools/typesetting/tex/pgf-tikz/pgf-3.x.nix { };

  pgf = pgf2;

  pgf-pie = callPackage ../tools/typesetting/tex/pgf-tikz/pgf-pie.nix { };

  pgf-umlcd = callPackage ../tools/typesetting/tex/pgf-tikz/pgf-umlcd.nix { };

  pgf-umlsd = callPackage ../tools/typesetting/tex/pgf-tikz/pgf-umlsd.nix { };

  pgfplots = callPackage ../tools/typesetting/tex/pgf-tikz/pgfplots.nix { };

  pplatex = callPackage ../tools/typesetting/tex/pplatex { };

  tetex = callPackage ../tools/typesetting/tex/tetex { libpng = libpng12; };

  tex-match = callPackage ../tools/typesetting/tex/tex-match { };

  texFunctions = callPackage ../tools/typesetting/tex/nix pkgs;

  # TeX Live; see https://nixos.org/nixpkgs/manual/#sec-language-texlive
  texlive = callPackage ../tools/typesetting/tex/texlive { };
  inherit (texlive.schemes) texliveBasic texliveBookPub texliveConTeXt texliveFull texliveGUST texliveInfraOnly texliveMedium texliveMinimal texliveSmall texliveTeTeX;
  texlivePackages = recurseIntoAttrs (lib.mapAttrs (_: v: v.build) texlive.pkgs);

  fop = callPackage ../tools/typesetting/fop { };

  fondu = callPackage ../tools/misc/fondu {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  fpp = callPackage ../tools/misc/fpp { };

  fsmark = callPackage ../tools/misc/fsmark { };

  futhark = haskell.lib.compose.justStaticExecutables haskellPackages.futhark;

  fx = callPackage ../development/tools/fx { };

  tllist = callPackage ../development/libraries/tllist { };

  fcft = callPackage ../development/libraries/fcft { };

  fuzzel = callPackage ../applications/misc/fuzzel { };

  fuzzylite = callPackage ../development/libraries/fuzzylite { };

  flashfocus = callPackage ../misc/flashfocus { };

  qt-video-wlr = libsForQt5.callPackage ../applications/misc/qt-video-wlr { };

  fwup = callPackage ../tools/misc/fwup {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration;
  };

  fx-cast-bridge = callPackage ../tools/misc/fx-cast-bridge { };

  fzf-obc = callPackage ../shells/bash/fzf-obc { };

  fzf-zsh = callPackage ../shells/zsh/fzf-zsh { };

  fzy = callPackage ../tools/misc/fzy { };

  g2o = libsForQt5.callPackage ../development/libraries/g2o { };

  gb-backup = callPackage ../tools/backup/gamerbackup { };

  gdrive = callPackage ../applications/networking/gdrive { };

  gdrive3 = callPackage ../applications/networking/gdrive3 { };

  gdu = callPackage ../tools/system/gdu { };

  gfxreconstruct = callPackage ../tools/graphics/gfxreconstruct { };

  go-chromecast = callPackage ../applications/video/go-chromecast { };

  go-containerregistry = callPackage ../development/tools/go-containerregistry { };
  inherit (go-containerregistry) crane gcrane;

  go-rice = callPackage ../tools/misc/go.rice { };

  go-2fa = callPackage ../tools/security/2fa { };

  go-neb = callPackage ../applications/networking/instant-messengers/go-neb { };

  google-cursor = callPackage ../data/icons/google-cursor { };

  geckodriver = callPackage ../development/tools/geckodriver {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  geekbench_4 = callPackage ../tools/misc/geekbench/4.nix { };
  geekbench_5 = callPackage ../tools/misc/geekbench/5.nix { };
  geekbench_6 = callPackage ../tools/misc/geekbench/6.nix { };
  geekbench = geekbench_6;

  gencfsm = callPackage ../tools/security/gencfsm { };

  genromfs = callPackage ../tools/filesystems/genromfs { };

  gh-ost = callPackage ../tools/misc/gh-ost { };

  ghauri = callPackage ../tools/security/ghauri { };

  ghdorker = callPackage ../tools/security/ghdorker { };

  ghidra = darwin.apple_sdk_11_0.callPackage ../tools/security/ghidra/build.nix {
    protobuf = protobuf_21;
  };

  ghidra-extensions = recurseIntoAttrs (callPackage ../tools/security/ghidra/extensions.nix { });

  ghidra-bin = callPackage ../tools/security/ghidra { };

  gh2md = callPackage ../tools/backup/gh2md { };

  gif-for-cli = callPackage ../tools/misc/gif-for-cli { };

  giph = callPackage ../applications/video/giph { };

  gir-rs = callPackage ../development/tools/gir { };

  girsh = callPackage ../tools/networking/girsh { };

  gist = callPackage ../tools/text/gist { };

  gitjacker = callPackage ../tools/security/gitjacker { };

  gixy = callPackage ../tools/admin/gixy { };

  glpaper = callPackage ../development/tools/glpaper { };

  gllvm = callPackage ../development/tools/gllvm { };

  globalarrays = callPackage ../development/libraries/globalarrays { };

  glock = callPackage ../development/tools/glock { };

  glslviewer = callPackage ../development/tools/glslviewer {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  gpg-tui = callPackage ../tools/security/gpg-tui {
    inherit (darwin.apple_sdk.frameworks) AppKit Foundation;
    inherit (darwin) libobjc libresolv;
  };

  goa = callPackage ../development/tools/goa { };

  goawk = callPackage ../tools/text/goawk { };

  gohai = callPackage ../tools/system/gohai { };

  gorilla-bin = callPackage ../tools/security/gorilla-bin { };

  gorilla-cli = callPackage ../tools/llm/gorilla-cli { };

  godu = callPackage ../tools/misc/godu { };

  gosu = callPackage ../tools/misc/gosu { };

  gotify-cli = callPackage ../tools/misc/gotify-cli { };

  gotify-desktop = callPackage ../tools/misc/gotify-desktop { };

  gping = callPackage ../tools/networking/gping {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  gpu-burn = callPackage ../applications/misc/gpu-burn {
    # gpu-burn doesn't build on gcc11. CUDA 11.3 is the last version to use
    # pre-gcc11, in particular gcc9.
    stdenv = gcc9Stdenv;
  };

  gpu-viewer = callPackage ../applications/misc/gpu-viewer { };

  greg = callPackage ../applications/audio/greg {
    pythonPackages = python3Packages;
  };

  grim = callPackage ../tools/graphics/grim { };

  grit = callPackage ../tools/misc/grit { };

  grobi = callPackage ../tools/X11/grobi { };

  gscan2pdf = callPackage ../applications/graphics/gscan2pdf {
    # needs this fork of libtiff, because original libtiff
    # stopped packaging required tools with version 4.6
    libtiff = libtiff_t;
  };

  gsctl = callPackage ../applications/misc/gsctl { };

  gsocket = callPackage ../tools/networking/gsocket { };

  gthree = callPackage ../development/libraries/gthree { };

  gtg = callPackage ../applications/office/gtg { };

  gti = callPackage ../tools/misc/gti { };

  has = callPackage ../applications/misc/has { };

  hdate = callPackage ../applications/misc/hdate { };

  headache = callPackage ../development/tools/headache { };

  heatseeker = callPackage ../tools/misc/heatseeker { };

  hebcal = callPackage ../tools/misc/hebcal { };

  hebbot = callPackage ../servers/matrix-hebbot {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  hexio = callPackage ../development/tools/hexio { };

  hexyl = callPackage ../tools/misc/hexyl { };

  hid-listen = callPackage ../tools/misc/hid-listen { };

  hid-tools = callPackage ../tools/misc/hid-tools { };

  hidrd = callPackage ../tools/misc/hidrd { };

  hiksink = callPackage ../tools/misc/hiksink {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  hocr-tools = with python3Packages; toPythonApplication hocr-tools;

  homepage-dashboard = callPackage ../servers/homepage-dashboard {
    inherit (darwin) cctools;
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  hostsblock = callPackage ../tools/misc/hostsblock { };

  hopper = qt5.callPackage ../development/tools/analysis/hopper { };

  hr = callPackage ../applications/misc/hr { };

  huniq = callPackage ../tools/text/huniq { };

  hunt = callPackage ../tools/misc/hunt { };

  hypr = callPackage ../applications/window-managers/hyprwm/hypr {
    cairo = cairo.override { xcbSupport = true; };  };

  hyprland-autoname-workspaces = callPackage ../applications/misc/hyprland-autoname-workspaces { };

  hyprland-per-window-layout = callPackage ../tools/wayland/hyprland-per-window-layout { };

  hyprland-protocols = callPackage ../applications/window-managers/hyprwm/hyprland-protocols { };

  hyprpaper = callPackage ../applications/window-managers/hyprwm/hyprpaper { };

  hyprpicker = callPackage ../applications/window-managers/hyprwm/hyprpicker {
    wlroots = wlroots_0_16;
  };

  hyprshade = python311Packages.callPackage ../applications/window-managers/hyprwm/hyprshade { };

  hyprlandPlugins = recurseIntoAttrs (callPackage ../applications/window-managers/hyprwm/hyprland-plugins { });

  hysteria = callPackage ../tools/networking/hysteria { };

  hyx = callPackage ../tools/text/hyx { };

  icdiff = callPackage ../tools/text/icdiff { };

  ictree = callPackage ../tools/misc/ictree { };

  inchi = callPackage ../development/libraries/inchi { };

  icon-slicer = callPackage ../tools/X11/icon-slicer { };

  ifm = callPackage ../tools/graphics/ifm { };

  ink = callPackage ../tools/misc/ink { };

  inklecate = callPackage ../development/compilers/inklecate { };

  intensity-normalization = with python3Packages; toPythonApplication intensity-normalization;

  interactsh = callPackage ../tools/misc/interactsh { };

  interlock = callPackage ../servers/interlock { };

  invoice = callPackage ../tools/misc/invoice { };

  invoiceplane = callPackage ../servers/web-apps/invoiceplane { };

  iotas = callPackage ../applications/office/iotas { };

  iotools = callPackage ../tools/misc/iotools { };

  irpf = callPackage ../applications/finance/irpf { };

  jellycli = callPackage ../applications/audio/jellycli { };

  jellyfin-ffmpeg = callPackage ../development/libraries/jellyfin-ffmpeg { };

  jellyfin-media-player = libsForQt5.callPackage ../applications/video/jellyfin-media-player {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Cocoa CoreAudio MediaPlayer;
  };

  jellyfin-mpv-shim = python3Packages.callPackage ../applications/video/jellyfin-mpv-shim { };

  jellyseerr = callPackage ../servers/jellyseerr { };

  jiten = callPackage ../applications/misc/jiten { };

  kanjidraw = callPackage ../applications/misc/kanjidraw { };

  jotta-cli = callPackage ../applications/misc/jotta-cli { };

  joycond = callPackage ../os-specific/linux/joycond { };

  joystickwake = callPackage ../tools/games/joystickwake { };

  juce = callPackage ../development/misc/juce {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
  };

  jumppad = callPackage ../tools/virtualization/jumppad { };

  jwt-cli = callPackage ../tools/security/jwt-cli {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  jwt-hack = callPackage ../tools/security/jwt-hack { } ;

  jwx = callPackage ../tools/security/jwx { } ;

  kapacitor = callPackage ../servers/monitoring/kapacitor { };

  karma = callPackage ../servers/monitoring/karma { };

  kaldi = callPackage ../tools/audio/kaldi {
    inherit (darwin.apple_sdk.frameworks) Accelerate;
  };

  kalker = callPackage ../tools/misc/kalker { };

  kisslicer = callPackage ../tools/misc/kisslicer { };

  klaus = with python3Packages; toPythonApplication klaus;

  klipper = callPackage ../servers/klipper { };

  klipper-firmware = callPackage ../servers/klipper/klipper-firmware.nix { gcc-arm-embedded = gcc-arm-embedded-13; };

  klipper-flash = callPackage ../servers/klipper/klipper-flash.nix { };

  klipper-genconf = callPackage ../servers/klipper/klipper-genconf.nix { };

  klipper-estimator = callPackage ../applications/misc/klipper-estimator {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  klipperscreen = callPackage ../applications/misc/klipperscreen { };

  klog = qt5.callPackage ../applications/radio/klog { };

  knxd = callPackage ../servers/knxd { };

  komga = callPackage ../servers/komga { };

  komorebi = callPackage ../applications/graphics/komorebi { };

  konsave = callPackage ../applications/misc/konsave { };

  krapslog = callPackage ../tools/misc/krapslog { };

  krelay = callPackage ../applications/networking/cluster/krelay { };

  krill = callPackage ../servers/krill {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  lapce = callPackage ../applications/editors/lapce {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) Security CoreServices ApplicationServices Carbon AppKit;
  };

  lcdproc = callPackage ../servers/monitoring/lcdproc { };

  languagetool = callPackage ../tools/text/languagetool {  };

  languagetool-rust = callPackage ../tools/text/languagetool-rust {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ldtk = callPackage ../applications/editors/ldtk { };

  lepton = callPackage ../tools/graphics/lepton { };

  lepton-eda = callPackage ../applications/science/electronics/lepton-eda { };

  lexend = callPackage ../data/fonts/lexend { };

  lexicon = with python3Packages; toPythonApplication dns-lexicon;

  lenpaste = callPackage ../servers/lenpaste { };

  lief = callPackage ../development/libraries/lief {
    python = python3;
  };

  libnbd = callPackage ../development/libraries/libnbd { };

  libndtypes = callPackage ../development/libraries/libndtypes { };

  libnss-mysql = callPackage ../os-specific/linux/libnss-mysql { };

  libnvme = callPackage ../os-specific/linux/libnvme { };

  libnvidia-container = callPackage ../applications/virtualization/libnvidia-container { };

  librenms = callPackage ../servers/monitoring/librenms { };

  libxnd = callPackage ../development/libraries/libxnd { };

  libz = callPackage ../development/libraries/libz { };

  libzbc = callPackage ../os-specific/linux/libzbc { };

  libzbd = callPackage ../os-specific/linux/libzbd { };

  lifeograph = callPackage ../applications/editors/lifeograph { };

  limitcpu = callPackage ../tools/misc/limitcpu { };

  lineselect = callPackage ../tools/misc/lineselect { };

  link-grammar = callPackage ../tools/text/link-grammar { };

  linuxptp = callPackage ../os-specific/linux/linuxptp { };

  linuxdoc-tools = callPackage ../tools/text/sgml/linuxdoc-tools { };

  lisgd = callPackage ../tools/inputmethods/lisgd { };

  lite = callPackage ../applications/editors/lite { };

  lite-xl = callPackage ../applications/editors/lite-xl {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  loadwatch = callPackage ../tools/system/loadwatch { };

  loccount = callPackage ../development/tools/misc/loccount { };

  long-shebang = callPackage ../misc/long-shebang { };

  lssecret = callPackage ../misc/lssecret {};

  lowdown = callPackage ../tools/typesetting/lowdown { };

  numatop = callPackage ../os-specific/linux/numatop { };

  numworks-udev-rules = callPackage ../os-specific/linux/numworks-udev-rules { };

  num-utils = callPackage ../tools/text/num-utils { };

  iio-sensor-proxy = callPackage ../os-specific/linux/iio-sensor-proxy { };

  ipvsadm = callPackage ../os-specific/linux/ipvsadm { };

  ir-standard-fonts = callPackage ../data/fonts/ir-standard-fonts { };

  kaggle = with python3Packages; toPythonApplication kaggle;

  lynis = callPackage ../tools/security/lynis { };

  maigret = callPackage ../tools/security/maigret { };

  maliit-framework = libsForQt5.callPackage ../applications/misc/maliit-framework { };

  maliit-keyboard = libsForQt5.callPackage ../applications/misc/maliit-keyboard { };

  mandown = callPackage ../tools/misc/mandown { };

  mantra = callPackage ../tools/security/mantra { };

  mapcidr = callPackage ../tools/misc/mapcidr { };

  maple-mono = (callPackage ../data/fonts/maple-font { }).Mono;
  maple-mono-NF = (callPackage ../data/fonts/maple-font { }).NF;
  maple-mono-SC-NF = (callPackage ../data/fonts/maple-font { }).SC-NF;
  maple-mono-otf = (callPackage ../data/fonts/maple-font { }).opentype;
  maple-mono-woff2 = (callPackage ../data/fonts/maple-font { }).woff2;
  maple-mono-autohint = (callPackage ../data/fonts/maple-font { }).autohint;

  marl = callPackage ../development/libraries/marl { };

  marlin-calc = callPackage ../tools/misc/marlin-calc { };

  marp-cli = callPackage ../tools/typesetting/marp { };

  masscan = callPackage ../tools/security/masscan { };

  massren = callPackage ../tools/misc/massren { };

  mat2 = with python3.pkgs; toPythonApplication mat2;

  maxcso = callPackage ../tools/archivers/maxcso { };

  measureme = callPackage ../development/tools/rust/measureme { };

  medusa = callPackage ../tools/security/medusa { };

  megasync = libsForQt5.callPackage ../applications/misc/megasync {
    ffmpeg = ffmpeg_4;
  };

  megacmd = callPackage ../applications/misc/megacmd { };

  merriweather = callPackage ../data/fonts/merriweather { };

  merriweather-sans = callPackage ../data/fonts/merriweather-sans { };

  # while building documentation meson may want to run binaries for host
  # which needs an emulator
  # example of an error which this fixes
  # [Errno 8] Exec format error: './gdk3-scan'
  mesonEmulatorHook =
    makeSetupHook
      {
        name = "mesonEmulatorHook";
        substitutions = {
          crossFile = writeText "cross-file.conf" ''
              [binaries]
              exe_wrapper = ${lib.escapeShellArg (stdenv.targetPlatform.emulator pkgs)}
            '';
        };
      }
      # The throw is moved into the `makeSetupHook` derivation, so that its
      # outer level, but not its outPath can still be evaluated if the condition
      # doesn't hold. This ensures that splicing still can work correctly.
      (if (!stdenv.hostPlatform.canExecute stdenv.targetPlatform) then
        ../by-name/me/meson/emulator-hook.sh
       else
         throw "mesonEmulatorHook may only be added to nativeBuildInputs when the target binaries can't be executed; however you are attempting to use it in a situation where ${stdenv.hostPlatform.config} can execute ${stdenv.targetPlatform.config}. Consider only adding mesonEmulatorHook according to a conditional based canExecute in your package expression.");

  meson-tools = callPackage ../misc/meson-tools { };

  metabase = callPackage ../servers/metabase {
    jdk11 = jdk11_headless;
  };

  micropad = callPackage ../applications/office/micropad {
    electron = electron_27;
  };

  midicsv = callPackage ../tools/audio/midicsv { };

  miniscript = callPackage ../applications/blockchains/miniscript { };

  miniserve = callPackage ../tools/misc/miniserve { };

  mkspiffs = callPackage ../tools/filesystems/mkspiffs { };

  mkspiffs-presets = recurseIntoAttrs (callPackages ../tools/filesystems/mkspiffs/presets.nix { });

  mlarchive2maildir = callPackage ../applications/networking/mailreaders/mlarchive2maildir { };

  mmctl = callPackage ../tools/misc/mmctl { };

  moar = callPackage ../tools/misc/moar { };

  mobilizon = callPackage ../servers/mobilizon {
    elixir = elixir_1_15;
    beamPackages = beamPackages.extend (self: super: { elixir = elixir_1_15; });
    mobilizon-frontend = callPackage ../servers/mobilizon/frontend.nix { };
  };

  molly-brown = callPackage ../servers/gemini/molly-brown { };

  monetdb = callPackage ../servers/sql/monetdb { };

  monado = callPackage ../applications/graphics/monado {
    inherit (gst_all_1) gstreamer gst-plugins-base;
  };

  mons = callPackage ../tools/misc/mons { };

  monsoon = callPackage ../tools/security/monsoon { };

  moodle-dl = callPackage ../tools/networking/moodle-dl { };

  moon-phases = callPackage ../tools/misc/moon-phases { };

  moonraker = callPackage ../servers/moonraker { };

  morsel = callPackage ../tools/text/morsel { };

  mousetweaks = callPackage ../applications/accessibility/mousetweaks { };

  mp3blaster = callPackage ../applications/audio/mp3blaster { };

  mp3cat = callPackage ../tools/audio/mp3cat { };

  mp3fs = callPackage ../tools/filesystems/mp3fs { };

  mpdas = callPackage ../tools/audio/mpdas { };

  mpdcron = callPackage ../tools/audio/mpdcron { };

  mpdris2 = callPackage ../tools/audio/mpdris2 { };

  mpd-discord-rpc = callPackage ../tools/audio/mpd-discord-rpc { };

  mpd-mpris = callPackage ../tools/audio/mpd-mpris { };

  mpd-notification = callPackage ../tools/audio/mpd-notification { };

  mpd-sima = python3Packages.callPackage ../tools/audio/mpd-sima { };

  mpris-scrobbler = callPackage ../tools/audio/mpris-scrobbler { };

  mpd-touch-screen-gui = callPackage ../applications/audio/mpd-touch-screen-gui { };

  mq-cli = callPackage ../tools/system/mq-cli { };

  mrkd = callPackage ../tools/text/mrkd { };

  naproche = callPackage ../applications/science/logic/naproche { };

  nautilus-open-any-terminal = callPackage ../tools/misc/nautilus-open-any-terminal { };

  n2n = callPackage ../tools/networking/n2n { };

  nest-cli = callPackage ../development/tools/nest-cli { };

  nextdns = callPackage ../applications/networking/nextdns { };

  nexttrace = callPackage ../tools/networking/nexttrace { };

  ngadmin = callPackage ../applications/networking/ngadmin { };

  nfdump = callPackage ../tools/networking/nfdump { };

  nfstrace = callPackage ../tools/networking/nfstrace { };

  nixel = callPackage ../tools/nix/nixel { };

  nix-output-monitor = callPackage ../tools/nix/nix-output-monitor { };

  nix-template = callPackage ../tools/package-management/nix-template {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  nltk-data = callPackage ../tools/text/nltk_data { };

  seaborn-data = callPackage ../tools/misc/seaborn-data { };

  nodepy-runtime = with python3.pkgs; toPythonApplication nodepy-runtime;

  nixpkgs-pytools = with python3.pkgs; toPythonApplication nixpkgs-pytools;

  nostr-rs-relay = callPackage ../servers/nostr-rs-relay { };

  notemap = callPackage ../tools/networking/notemap { };

  noteshrink = callPackage ../tools/misc/noteshrink { };

  noti = callPackage ../tools/misc/noti {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  notify = callPackage ../tools/misc/notify { };

  notifymuch = callPackage ../applications/misc/notifymuch { };

  nmh = callPackage ../tools/networking/nmh { };

  npins = callPackage ../tools/nix/npins { };

  nrsc5 = callPackage ../applications/misc/nrsc5 { };

  nsync = callPackage ../development/libraries/nsync { };

  nsz = with python3.pkgs; toPythonApplication nsz;

  nwipe = callPackage ../tools/security/nwipe { };

  nx2elf = callPackage ../tools/compression/nx2elf { };

  nx-libs = callPackage ../tools/X11/nx-libs { };

  nyx = callPackage ../tools/networking/nyx { };

  oci-cli = callPackage ../tools/admin/oci-cli { };

  oci-seccomp-bpf-hook = callPackage ../os-specific/linux/oci-seccomp-bpf-hook { };

  ocrmypdf = with python3.pkgs; toPythonApplication ocrmypdf;

  ocrfeeder = callPackage ../applications/graphics/ocrfeeder { };

  onboard = callPackage ../applications/misc/onboard { };

  oneshot = callPackage ../tools/networking/oneshot { };

  orjail = callPackage ../tools/security/orjail { };

  online-judge-template-generator = python3Packages.callPackage ../tools/misc/online-judge-template-generator { };

  online-judge-tools = with python3.pkgs; toPythonApplication online-judge-tools;

  onnxruntime = callPackage ../development/libraries/onnxruntime {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  xkbd = callPackage ../applications/misc/xkbd { };

  libpsm2 = callPackage ../os-specific/linux/libpsm2 { };

  optar = callPackage ../tools/graphics/optar { };


  ockam = callPackage ../tools/networking/ockam {
    inherit (darwin.apple_sdk.frameworks) AppKit Security;
  };

  odoo = callPackage ../applications/finance/odoo { };
  odoo15 = callPackage ../applications/finance/odoo/odoo15.nix { };

  odafileconverter = libsForQt5.callPackage ../applications/graphics/odafileconverter { };

  oil-buku = callPackage ../applications/misc/oil-buku { };

  ondir = callPackage ../tools/misc/ondir { };

  opam-publish = callPackage ../development/tools/ocaml/opam-publish { };

  osdlyrics = callPackage ../applications/audio/osdlyrics { };

  ossutil = callPackage ../tools/admin/ossutil { };

  ospd-openvas = callPackage ../tools/security/ospd-openvas { };

  osv-detector = callPackage ../tools/security/osv-detector { };

  osv-scanner = callPackage ../tools/security/osv-scanner { };

  otf2bdf = callPackage ../applications/misc/otf2bdf { };

  pastel = callPackage ../applications/misc/pastel {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  inherit (ocamlPackages) patdiff;

  patool = with python3Packages; toPythonApplication patool;

  pbgopy = callPackage ../tools/text/pbgopy { };

  pbzx = callPackage ../tools/compression/pbzx { };

  pc = callPackage ../tools/misc/pc { };

  pcb2gcode = callPackage ../tools/misc/pcb2gcode { };

  pcp = callPackage ../tools/misc/pcp { };

  pev = callPackage ../development/tools/analysis/pev { };

  phd2 = callPackage ../applications/science/astronomy/phd2 { };

  phoronix-test-suite = callPackage ../tools/misc/phoronix-test-suite { };

  photon = callPackage ../tools/networking/photon { };

  photofield = callPackage ../servers/photofield { };

  photoprism = callPackage ../servers/photoprism { };

  piglit = callPackage ../tools/graphics/piglit { };

  pika = callPackage ../applications/graphics/pika { };

  pkg = callPackage ../tools/package-management/pkg { };

  playerctl = callPackage ../tools/audio/playerctl { };

  pn = callPackage ../tools/text/pn { };

  pocket-casts = callPackage ../applications/audio/pocket-casts {
    electron = electron_27;
  };

  portal = callPackage ../tools/misc/portal { };

  pouf = callPackage ../tools/misc/pouf { };

  poweralertd = callPackage ../tools/misc/poweralertd { };

  pscale = callPackage ../development/tools/pscale { };

  pulsarctl = callPackage ../tools/admin/pulsarctl { };

  psstop = callPackage ../tools/system/psstop { };

  precice = callPackage ../development/libraries/precice { };

  precice-config-visualizer = callPackage ../tools/misc/precice-config-visualizer { };

  proxmox-backup-client = callPackage ../applications/backup/proxmox-backup-client { };

  pueue = darwin.apple_sdk_11_0.callPackage ../applications/misc/pueue {
    inherit (darwin.apple_sdk_11_0) Libsystem;
    inherit (darwin.apple_sdk_11_0.frameworks) SystemConfiguration;
  };

  pixcat = with python3Packages; toPythonApplication pixcat;

  pixiecore = callPackage ../tools/networking/pixiecore { };

  waitron = callPackage ../tools/networking/waitron { };

  pyCA = callPackage ../applications/video/pyca { };

  pyznap = python3Packages.callPackage ../tools/backup/pyznap { };

  procs = darwin.apple_sdk_11_0.callPackage ../tools/admin/procs {
    inherit (darwin.apple_sdk_11_0.frameworks) Security;
    inherit (darwin.apple_sdk_11_0) Libsystem;
  };

  procyon = callPackage ../tools/misc/procyon { };

  profetch = callPackage ../tools/misc/profetch { };

  psrecord = python3Packages.callPackage ../tools/misc/psrecord { };

  pyenv = callPackage ../development/tools/pyenv { };

  r53-ddns = callPackage ../applications/networking/r53-ddns { };

  rare = python3Packages.callPackage ../games/rare { };

  rblake2sum = callPackage ../tools/security/rblake2sum {
      inherit (darwin.apple_sdk.frameworks) Security;
  };

  rblake3sum = callPackage ../tools/security/rblake3sum {
      inherit (darwin.apple_sdk.frameworks) Security;
  };

  reg = callPackage ../tools/virtualization/reg { };

  rex = callPackage ../tools/system/rex { };

  rivercarro = callPackage ../applications/misc/rivercarro { };

  river-luatile = callPackage ../applications/misc/river-luatile{ };

  rmapi = callPackage ../applications/misc/remarkable/rmapi { };

  rmate-sh = callPackage ../tools/misc/rmate-sh { };

  rmview = libsForQt5.callPackage ../applications/misc/remarkable/rmview { };

  rm-improved = callPackage ../applications/misc/rm-improved { };

  remarkable-mouse = python3Packages.callPackage ../applications/misc/remarkable/remarkable-mouse { };

  restream = callPackage ../applications/misc/remarkable/restream { };

  ropgadget = with python3Packages; toPythonApplication ropgadget;

  scour = with python3Packages; toPythonApplication scour;

  see = callPackage ../tools/misc/see { };

  s2png = callPackage ../tools/graphics/s2png { };

  sfz = callPackage ../tools/misc/sfz { };

  sgp4 = callPackage ../development/libraries/sgp4 { };

  shab = callPackage ../tools/text/shab { };

  sheldon = callPackage ../tools/misc/sheldon {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  shell-hist = callPackage ../tools/misc/shell-hist { };

  shellhub-agent = callPackage ../applications/networking/shellhub-agent { };

  shellclear = callPackage ../tools/security/shellclear { };

  shellnoob = callPackage ../tools/security/shellnoob { };

  sheesy-cli = callPackage ../tools/security/sheesy-cli {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  simdjson = callPackage ../development/libraries/simdjson { };

  shisho = callPackage ../tools/security/shisho { };

  sigslot = callPackage ../development/libraries/sigslot { };

  siglo = callPackage ../applications/misc/siglo { };

  simg2img = callPackage ../tools/filesystems/simg2img { };

  snazy = callPackage ../development/tools/snazy { };

  snippetpixie = callPackage ../tools/text/snippetpixie { };

  snowcat = callPackage ../tools/security/snowcat { };

  socklog = callPackage ../tools/system/socklog { };

  soju = callPackage ../applications/networking/soju { };

  spacevim = callPackage ../applications/editors/spacevim { };

  srvc = callPackage ../applications/version-management/srvc { };

  ssmsh = callPackage ../tools/admin/ssmsh { };

  stacs = callPackage ../tools/security/stacs { };

  stagit = callPackage ../development/tools/stagit { };

  starboard = callPackage ../applications/networking/cluster/starboard { };

  statserial = callPackage ../tools/misc/statserial { };

  step-ca = callPackage ../tools/security/step-ca {
    inherit (darwin.apple_sdk.frameworks) PCSC;
  };

  step-kms-plugin = callPackage ../tools/security/step-kms-plugin { };

  string-machine = callPackage ../applications/audio/string-machine { };

  stripe-cli = callPackage ../tools/admin/stripe-cli { };

  bash-supergenpass = callPackage ../tools/security/bash-supergenpass { };

  swappy = callPackage ../applications/misc/swappy { gtk = gtk3; };

  swapspace = callPackage ../tools/admin/swapspace { };

  swego = callPackage ../servers/swego { };

  sydbox = callPackage ../os-specific/linux/sydbox { };

  synth = callPackage ../tools/misc/synth {
    inherit (darwin.apple_sdk.frameworks) AppKit Security;
  };

  syscall_limiter = callPackage ../os-specific/linux/syscall_limiter { };

  skeema = callPackage ../tools/system/skeema { };

  svt-av1 = callPackage ../tools/video/svt-av1 { };

  inherit (callPackages ../servers/rainloop { })
    rainloop-community
    rainloop-standard;

  rav1e = callPackage ../tools/video/rav1e {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  raven-reader = callPackage ../applications/networking/newsreaders/raven-reader { };

  razergenie = libsForQt5.callPackage ../applications/misc/razergenie { };

  replay-sorcery = callPackage ../tools/video/replay-sorcery { };

  recyclarr = callPackage ../tools/video/recyclarr { };

  tsduck = callPackage ../tools/video/tsduck { };

  turso-cli = callPackage ../development/tools/turso-cli {};

  ripasso-cursive = callPackage ../tools/security/ripasso/cursive.nix {
    inherit (darwin.apple_sdk.frameworks) AppKit Security;
  };

  ripe-atlas-tools = callPackage ../tools/networking/ripe-atlas-tools { };

  roundcube = callPackage ../servers/roundcube { };

  roundcubePlugins = dontRecurseIntoAttrs (callPackage ../servers/roundcube/plugins { });

  routersploit = callPackage ../tools/security/routersploit { };

  routinator = callPackage ../servers/routinator {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  rsbep = callPackage ../tools/backup/rsbep { };

  rsbkb = callPackage ../tools/text/rsbkb { };

  rsign2 = callPackage ../tools/security/rsign2 { };

  rsyslog = callPackage ../tools/system/rsyslog {
    withHadoop = false; # Currently Broken
    withKsi = false; # Currently Broken
  };

  rsyslog-light = rsyslog.override {
    withKrb5 = false;
    withSystemd = false;
    withJemalloc = false;
    withMysql = false;
    withPostgres = false;
    withDbi = false;
    withNetSnmp = false;
    withUuid = false;
    withCurl = false;
    withGnutls = false;
    withGcrypt = false;
    withLognorm = false;
    withMaxminddb = false;
    withOpenssl = false;
    withRelp = false;
    withKsi = false;
    withLogging = false;
    withNet = false;
    withHadoop = false;
    withRdkafka = false;
    withMongo = false;
    withCzmq = false;
    withRabbitmq = false;
    withHiredis = false;
  };

  rtrtr = callPackage ../servers/rtrtr {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  xlogo = callPackage ../tools/X11/xlogo { };

  xmlbeans = callPackage ../tools/misc/xmlbeans { };

  xmlsort = perlPackages.XMLFilterSort;

  xmousepasteblock = callPackage ../tools/X11/xmousepasteblock { };

  mcrypt = callPackage ../tools/misc/mcrypt { };

  mongoaudit = callPackage ../tools/security/mongoaudit { };

  mongodb-compass = callPackage ../tools/misc/mongodb-compass { };

  mongodb-tools = callPackage ../tools/misc/mongodb-tools { };

  moosefs = callPackage ../tools/filesystems/moosefs { };

  mountain-duck = callPackage ../tools/filesystems/mountain-duck { };

  mox = callPackage ../servers/mail/mox { };

  mozlz4a = callPackage ../tools/compression/mozlz4a { };

  msr-tools = callPackage ../os-specific/linux/msr-tools { };

  mstflint = callPackage ../tools/misc/mstflint { };

  msldapdump = callPackage ../tools/security/msldapdump { };

  mslink = callPackage ../tools/misc/mslink { };

  mceinject = callPackage ../os-specific/linux/mceinject { };

  mcelog = callPackage ../os-specific/linux/mcelog {
    util-linux = util-linuxMinimal;
  };

  sqldef = callPackage ../development/tools/sqldef { };

  sqlint = callPackage ../development/tools/sqlint { };

  squawk = callPackage ../development/tools/squawk { };

  antibody = callPackage ../shells/zsh/antibody { };

  antidote = callPackage ../shells/zsh/antidote { };

  antigen = callPackage ../shells/zsh/antigen { };

  apparix = callPackage ../tools/misc/apparix { };

  apc-temp-fetch = with python3.pkgs; callPackage ../tools/networking/apc-temp-fetch { };

  arping = callPackage ../tools/networking/arping { };

  arpoison = callPackage ../tools/networking/arpoison { };

  asciidoc = callPackage ../tools/typesetting/asciidoc {
    inherit (python3.pkgs) pygments matplotlib numpy aafigure recursive-pth-loader;
    w3m = w3m-batch;
    enableStandardFeatures = false;
  };

  asciidoc-full = asciidoc.override {
    enableStandardFeatures = true;
  };

  asciidoc-full-with-plugins = asciidoc.override {
    enableStandardFeatures = true;
    enableExtraPlugins = true;
  };

  asciidoctor = callPackage ../tools/typesetting/asciidoctor { };

  asciidoctor-with-extensions = callPackage ../tools/typesetting/asciidoctor-with-extensions { };

  asciidoctorj = callPackage ../tools/typesetting/asciidoctorj { };

  asunder = callPackage ../applications/audio/asunder { };

  autossh = callPackage ../tools/networking/autossh { };

  assh = callPackage ../tools/networking/assh { };

  b2sum = callPackage ../tools/security/b2sum {
    inherit (llvmPackages) openmp;
  };

  bacula = callPackage ../tools/backup/bacula {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation IOKit;
  };

  bacon = callPackage ../development/tools/bacon {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  bats = callPackage ../development/interpreters/bats { };

  bbe = callPackage ../tools/misc/bbe { };

  bkcrack = callPackage ../tools/security/bkcrack { };

  bdsync = callPackage ../tools/backup/bdsync { };

  beamerpresenter = beamerpresenter-mupdf;

  beamerpresenter-mupdf = qt6Packages.callPackage ../applications/office/beamerpresenter {
    useMupdf = true;
    usePoppler = false;
  };

  beamerpresenter-poppler = qt6Packages.callPackage ../applications/office/beamerpresenter {
    useMupdf = false;
    usePoppler = true;
  };

  beanstalkd = callPackage ../servers/beanstalkd { };

  beetsPackages = lib.recurseIntoAttrs (callPackage ../tools/audio/beets { });
  inherit (beetsPackages) beets beets-unstable;

  bento4 = callPackage ../tools/video/bento4 { };

  bepasty = callPackage ../tools/misc/bepasty { };

  bettercap = callPackage ../tools/security/bettercap { };

  bfs = callPackage ../tools/system/bfs { };

  bgs = callPackage ../tools/X11/bgs { };

  bibclean = callPackage ../tools/typesetting/bibclean { };

  biber = callPackage ../tools/typesetting/biber { };

  biber-for-tectonic = callPackage ../tools/typesetting/tectonic/biber.nix { };

  biber-ms = callPackage ../tools/typesetting/biber-ms { };

  biblatex-check = callPackage ../tools/typesetting/biblatex-check { };

  binlore = callPackage ../development/tools/analysis/binlore { };

  birdfont = callPackage ../tools/misc/birdfont { };
  xmlbird = callPackage ../tools/misc/birdfont/xmlbird.nix { stdenv = gccStdenv; };

  blueberry = callPackage ../tools/bluetooth/blueberry { };

  blueman = callPackage ../tools/bluetooth/blueman { };

  bmrsa = callPackage ../tools/security/bmrsa/11.nix { };

  bogofilter = callPackage ../tools/misc/bogofilter { };

  bom = callPackage ../tools/admin/bom { };

  bomber-go = callPackage ../development/tools/bomber-go { };

  bomutils = callPackage ../tools/archivers/bomutils { };

  boofuzz= callPackage ../tools/security/boofuzz { };

  briar-desktop = callPackage ../applications/networking/instant-messengers/briar-desktop { };

  bsdbuild = callPackage ../development/tools/misc/bsdbuild { };

  bsdiff = callPackage ../tools/compression/bsdiff { };

  btar = callPackage ../tools/backup/btar { };

  bumpver = callPackage ../applications/version-management/bumpver { };

  bup = callPackage ../tools/backup/bup { };

  bupstash = darwin.apple_sdk_11_0.callPackage ../tools/backup/bupstash { };

  burp = callPackage ../tools/backup/burp { };

  buku = callPackage ../applications/misc/buku { };

  byzanz = callPackage ../applications/video/byzanz { };

  algolia-cli = callPackage ../development/tools/algolia-cli { };

  anydesk = callPackage ../applications/networking/remote/anydesk { };

  anystyle-cli = callPackage ../tools/misc/anystyle-cli { };

  atool = callPackage ../tools/archivers/atool { };

  bash_unit = callPackage ../tools/misc/bash_unit { };

  boltbrowser = callPackage ../tools/misc/boltbrowser { };

  bsc = callPackage ../tools/compression/bsc {
    inherit (llvmPackages) openmp;
  };

  bzip2 = callPackage ../tools/compression/bzip2 { };

  bzip2_1_1 = callPackage ../tools/compression/bzip2/1_1.nix { };

  bzip3 = callPackage ../tools/compression/bzip3 {
    stdenv = clangStdenv;
  };

  cabextract = callPackage ../tools/archivers/cabextract { };

  cadaver = callPackage ../tools/networking/cadaver { };

  davix = callPackage ../tools/networking/davix {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  davix-copy = davix.override { enableThirdPartyCopy = true; };

  cantata = libsForQt5.callPackage ../applications/audio/cantata { };

  cantoolz = callPackage ../tools/networking/cantoolz { };

  can-utils = callPackage ../os-specific/linux/can-utils { };

  cannelloni = callPackage ../os-specific/linux/cannelloni { };

  caudec = callPackage ../applications/audio/caudec { };

  ccd2iso = callPackage ../tools/cd-dvd/ccd2iso { };

  ccid = callPackage ../tools/security/ccid { };

  ccrypt = callPackage ../tools/security/ccrypt { };

  ccze = callPackage ../tools/misc/ccze { };

  cdi2iso = callPackage ../tools/cd-dvd/cdi2iso { };

  cdist = python3Packages.callPackage ../tools/admin/cdist { };

  cdimgtools = callPackage ../tools/cd-dvd/cdimgtools { };

  cdpr = callPackage ../tools/networking/cdpr { };

  cdrdao = callPackage ../tools/cd-dvd/cdrdao {
    inherit (darwin.apple_sdk.frameworks) CoreServices IOKit;
  };

  cdrkit = callPackage ../tools/cd-dvd/cdrkit { };

  cdrtools = callPackage ../tools/cd-dvd/cdrtools {
    stdenv = if stdenv.isDarwin then llvmPackages_14.stdenv else stdenv;
    inherit (darwin.apple_sdk.frameworks) Carbon IOKit;
  };

  cemu-ti = qt5.callPackage ../applications/science/math/cemu-ti { };

  cider = callPackage ../applications/audio/cider { };

  iat = callPackage ../tools/cd-dvd/iat { };

  isolyzer = callPackage ../tools/cd-dvd/isolyzer { };

  isomd5sum = callPackage ../tools/cd-dvd/isomd5sum { };

  mdf2iso = callPackage ../tools/cd-dvd/mdf2iso { };

  nrg2iso = callPackage ../tools/cd-dvd/nrg2iso { };

  ceph-csi = callPackage ../tools/filesystems/ceph-csi { };

  libceph = ceph.lib;
  inherit (callPackages ../tools/filesystems/ceph {
    lua = lua5_4; # Ceph currently requires >= 5.3

    # To see which `fmt` version Ceph upstream recommends, check its `src/fmt` submodule.
    #
    # Ceph does not currently build with `fmt_10`; see https://github.com/NixOS/nixpkgs/issues/281027#issuecomment-1899128557
    # If we want to switch for that before upstream fixes it, use this patch:
    # https://github.com/NixOS/nixpkgs/pull/281858#issuecomment-1899648638
    fmt = fmt_9;
  })
    ceph
    ceph-client;
  ceph-dev = ceph;

  cfdg = callPackage ../tools/graphics/cfdg { };

  cglm = callPackage ../development/libraries/cglm { };

  cgreen = callPackage ../development/libraries/cgreen { };

  chain-bench = callPackage ../tools/security/chain-bench { };

  chainsaw = callPackage ../tools/security/chainsaw { };

  checkinstall = callPackage ../tools/package-management/checkinstall { };

  checkmake = callPackage ../development/tools/checkmake { };

  cherrybomb = callPackage ../tools/security/cherrybomb { };

  chit = callPackage ../development/tools/chit { };

  chkrootkit = callPackage ../tools/security/chkrootkit { };

  chrony = callPackage ../tools/networking/chrony { };

  chunkfs = callPackage ../tools/filesystems/chunkfs { };

  chunksync = callPackage ../tools/backup/chunksync { };

  cicero-tui = callPackage ../tools/misc/cicero-tui { };

  cilium-cli = callPackage ../applications/networking/cluster/cilium {
    buildGoModule = buildGo122Module;
  };

  cjdns = callPackage ../tools/networking/cjdns { };
  cjdns-tools = callPackage ../tools/admin/cjdns-tools { };

  cjson = callPackage ../development/libraries/cjson { };

  cksfv = callPackage ../tools/networking/cksfv { };

  clementine = libsForQt5.callPackage ../applications/audio/clementine {
    gst_plugins =
      with gst_all_1; [ gst-plugins-base gst-plugins-good gst-plugins-ugly gst-libav ];
    protobuf = protobuf_21;
  };

  mellowplayer = libsForQt5.callPackage ../applications/audio/mellowplayer { };

  ciopfs = callPackage ../tools/filesystems/ciopfs { };

  circleci-cli = callPackage ../development/tools/misc/circleci-cli { };

  circus = with python310Packages; toPythonApplication circus;

  cirrus-cli = callPackage ../development/tools/continuous-integration/cirrus-cli { };

  cirrusgo = callPackage ../tools/security/cirrusgo { };

  inherit (callPackage ../applications/networking/remote/citrix-workspace { })
    citrix_workspace_23_09_0
    citrix_workspace_23_11_0
    citrix_workspace_24_02_0
  ;
  citrix_workspace = citrix_workspace_24_02_0;

  clima = callPackage ../tools/text/clima { };

  cmigemo = callPackage ../tools/text/cmigemo { };

  cmospwd = callPackage ../tools/security/cmospwd { };

  cmst = libsForQt5.callPackage ../tools/networking/cmst { };

  cmt = callPackage ../applications/audio/cmt { };

  crlfsuite = callPackage ../tools/security/crlfsuite { };

  crlfuzz = callPackage ../tools/security/crlfuzz { };

  hedgedoc = callPackage ../servers/web-apps/hedgedoc { };

  colord = callPackage ../tools/misc/colord { };

  colord-gtk = callPackage ../tools/misc/colord-gtk { };

  colord-gtk4 = colord-gtk.override { withGtk4 = true; };

  colordiff = callPackage ../tools/text/colordiff { };

  connect = callPackage ../tools/networking/connect { };

  conspy = callPackage ../os-specific/linux/conspy { };

  connmanFull = connman.override {
    # TODO: Why is this in `connmanFull` and not the default build? See TODO in
    # nixos/modules/services/networking/connman.nix (near the assertions)
    enableNetworkManagerCompatibility = true;
    enableHh2serialGps = true;
    enableL2tp = true;
    enableIospm = true;
    enableTist = true;
  };

  connmanMinimal = connman.override {
    # enableDatafiles = false; # If disabled, configuration and data files are not installed
    # enableEthernet = false; # If disabled no ethernet connection can be performed
    # enableWifi = false; # If disabled no WiFi connection can be performed
    enableBluetooth = false;
    enableClient = false;
    enableDundee = false;
    enableGadget = false;
    enableLoopback = false;
    enableNeard = false;
    enableOfono = false;
    enableOpenconnect = false;
    enableOpenvpn = false;
    enablePacrunner = false;
    enablePolkit = false;
    enablePptp = false;
    enableStats = false;
    enableTools = false;
    enableVpnc = false;
    enableWireguard = false;
    enableWispr = false;
  };

  convertlit = callPackage ../tools/text/convertlit { };

  collectd = callPackage ../tools/system/collectd {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  collectd-data = callPackage ../tools/system/collectd/data.nix { };

  colormake = callPackage ../development/tools/build-managers/colormake { };

  cpuminer = callPackage ../tools/misc/cpuminer { };

  crabz = callPackage ../tools/compression/crabz { };

  ethash = callPackage ../development/libraries/ethash { };

  cuetools = callPackage ../tools/cd-dvd/cuetools { };

  u3-tool = callPackage ../tools/filesystems/u3-tool { };

  unifdef = callPackage ../development/tools/misc/unifdef { };

  unify = with python3Packages; toPythonApplication unify;

  unionfs-fuse = callPackage ../tools/filesystems/unionfs-fuse { };

  unparam = callPackage ../tools/misc/unparam {
    buildGoModule = buildGo121Module;
  };

  inherit (nodePackages) uppy-companion;

  usb-modeswitch = callPackage ../development/tools/misc/usb-modeswitch { };
  usb-modeswitch-data = callPackage ../development/tools/misc/usb-modeswitch/data.nix { };

  usbsdmux = callPackage ../development/tools/misc/usbsdmux { };

  usbview = callPackage ../tools/misc/usbview { };

  usbimager = callPackage ../tools/misc/usbimager { };

  uwuify = callPackage ../tools/misc/uwuify { };

  anthy = callPackage ../tools/inputmethods/anthy { };

  evdevremapkeys = callPackage ../tools/inputmethods/evdevremapkeys { };

  evsieve = callPackage ../tools/inputmethods/evsieve { };

  eyedropper = callPackage ../applications/graphics/eyedropper { };

  persistent-evdev = python3Packages.callPackage ../servers/persistent-evdev { };

  evscript = callPackage ../tools/inputmethods/evscript { };

  keyd = callPackage ../tools/inputmethods/keyd { };

  keymapper = callPackage ../tools/inputmethods/keymapper { };

  twitch-tui = callPackage ../applications/networking/instant-messengers/twitch-tui {
    inherit (darwin.apple_sdk_11_0.frameworks) Security CoreServices SystemConfiguration;
  };

  gebaar-libinput = callPackage ../tools/inputmethods/gebaar-libinput { };

  kime = callPackage ../tools/inputmethods/kime { };

  libpinyin = callPackage ../development/libraries/libpinyin { };

  inherit (import ../development/libraries/libsbsms pkgs)
    libsbsms
    libsbsms_2_0_2
    libsbsms_2_3_0
  ;

  libskk = callPackage ../development/libraries/libskk {
    inherit (gnome) gnome-common;
  };

  m17-cxx-demod = callPackage ../applications/radio/m17-cxx-demod { };

  m17n_db = callPackage ../tools/inputmethods/m17n-db { };

  m17n_lib = callPackage ../tools/inputmethods/m17n-lib { };

  libotf = callPackage ../tools/inputmethods/m17n-lib/otf.nix { };

  netbird = callPackage ../tools/networking/netbird {
    inherit (darwin.apple_sdk_11_0.frameworks) Cocoa IOKit Kernel UserNotifications WebKit;
  };

  netbird-ui = netbird.override {
    ui = true;
  };

  netevent = callPackage ../tools/inputmethods/netevent { };

  netplan = callPackage ../tools/admin/netplan { };

  skktools = callPackage ../tools/inputmethods/skk/skktools { };
  skk-dicts = callPackage ../tools/inputmethods/skk/skk-dicts { };

  libkkc-data = callPackage ../data/misc/libkkc-data { };

  libkkc = callPackage ../tools/inputmethods/libkkc { };

  ibus = callPackage ../tools/inputmethods/ibus { };

  ibus-engines = recurseIntoAttrs {
    anthy = callPackage ../tools/inputmethods/ibus-engines/ibus-anthy { };

    bamboo = callPackage ../tools/inputmethods/ibus-engines/ibus-bamboo { };

    cangjie = callPackage ../tools/inputmethods/ibus-engines/ibus-cangjie { };

    hangul = callPackage ../tools/inputmethods/ibus-engines/ibus-hangul { };

    kkc = callPackage ../tools/inputmethods/ibus-engines/ibus-kkc { };

    libpinyin = callPackage ../tools/inputmethods/ibus-engines/ibus-libpinyin { };

    libthai = callPackage ../tools/inputmethods/ibus-engines/ibus-libthai { };

    m17n = callPackage ../tools/inputmethods/ibus-engines/ibus-m17n { };

    mozc = callPackage ../tools/inputmethods/ibus-engines/ibus-mozc { };

    openbangla-keyboard = libsForQt5.callPackage ../applications/misc/openbangla-keyboard { withIbusSupport = true; };

    rime = callPackage ../tools/inputmethods/ibus-engines/ibus-rime { };

    table = callPackage ../tools/inputmethods/ibus-engines/ibus-table { };

    table-chinese = callPackage ../tools/inputmethods/ibus-engines/ibus-table-chinese {
      ibus-table = ibus-engines.table;
    };

    table-others = callPackage ../tools/inputmethods/ibus-engines/ibus-table-others {
      ibus-table = ibus-engines.table;
    };

    uniemoji = callPackage ../tools/inputmethods/ibus-engines/ibus-uniemoji { };

    typing-booster-unwrapped = callPackage ../tools/inputmethods/ibus-engines/ibus-typing-booster { };

    typing-booster = callPackage ../tools/inputmethods/ibus-engines/ibus-typing-booster/wrapper.nix {
      typing-booster = ibus-engines.typing-booster-unwrapped;
    };
  };

  ibus-with-plugins = callPackage ../tools/inputmethods/ibus/wrapper.nix { };

  ibus-theme-tools = callPackage ../tools/misc/ibus-theme-tools { };

  interception-tools = callPackage ../tools/inputmethods/interception-tools { };
  interception-tools-plugins = recurseIntoAttrs {
    caps2esc = callPackage ../tools/inputmethods/interception-tools/caps2esc.nix { };
    dual-function-keys = callPackage ../tools/inputmethods/interception-tools/dual-function-keys.nix { };
  };

  age = callPackage ../tools/security/age { };

  agebox = callPackage ../tools/security/agebox { };

  age-plugin-ledger = callPackage ../tools/security/age-plugin-ledger {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  age-plugin-tpm = callPackage ../tools/security/age-plugin-tpm { };

  age-plugin-yubikey = darwin.apple_sdk_11_0.callPackage ../tools/security/age-plugin-yubikey {
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation PCSC IOKit;
  };

  artim-dark = callPackage ../data/themes/artim-dark { };

  bbin = callPackage ../development/tools/bbin { };

  blacken-docs = callPackage ../tools/misc/blacken-docs { };

  bore = callPackage ../tools/networking/bore {
    inherit (darwin) Libsystem;
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration;
  };

  brotli = callPackage ../tools/compression/brotli { };

  biodiff = callPackage ../development/tools/biodiff { };

  biome = callPackage ../development/tools/biome { };

  biosdevname = callPackage ../tools/networking/biosdevname { };

  bluetooth_battery = python3Packages.callPackage ../applications/misc/bluetooth_battery { };

  boa = callPackage ../development/web/boa { };

  bombardier = callPackage ../tools/networking/bombardier { };

  caeml = callPackage ../applications/networking/mailreaders/caeml { };

  calyx-vpn = libsForQt5.callPackage ../tools/networking/bitmask-vpn {
    provider = "calyx";
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security;
  };

  cask-server = libsForQt5.callPackage ../applications/misc/cask-server { };

  cc2538-bsl = callPackage ../tools/misc/cc2538-bsl { };

  code-browser-qt = libsForQt5.callPackage ../applications/editors/code-browser { withQt = true; };
  code-browser-gtk2 = callPackage ../applications/editors/code-browser { withGtk2 = true; };
  code-browser-gtk = callPackage ../applications/editors/code-browser { withGtk3 = true; };

  certdump = callPackage ../tools/security/certdump { };

  certstrap = callPackage ../tools/security/certstrap { };

  cffconvert = python3Packages.toPythonApplication python3Packages.cffconvert;

  cfssl = callPackage ../tools/security/cfssl { };

  cfs-zen-tweaks = callPackage ../os-specific/linux/cfs-zen-tweaks { };

  chafa = callPackage ../tools/misc/chafa {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  changetower = callPackage ../tools/networking/changetower { };

  chatblade = callPackage ../applications/misc/chatblade { };

  checkbashisms = callPackage ../development/tools/misc/checkbashisms { };

  checkmate = callPackage ../development/tools/checkmate { };

  circup = callPackage ../development/tools/circup { };

  civetweb = callPackage ../development/libraries/civetweb { };

  ckb-next = libsForQt5.callPackage ../tools/misc/ckb-next { };

  clamav = callPackage ../tools/security/clamav {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation;
  };

  client-ip-echo = callPackage ../servers/misc/client-ip-echo { };

  cloc = callPackage ../tools/misc/cloc { };

  clolcat = callPackage ../tools/misc/clolcat { };

  cloog = callPackage ../development/libraries/cloog {
    isl = isl_0_14;
  };

  cloog_0_18_0 = callPackage ../development/libraries/cloog/0.18.0.nix {
    isl = isl_0_11;
  };

  cloogppl = callPackage ../development/libraries/cloog-ppl { };

  cloud-utils = callPackage ../tools/misc/cloud-utils { };

  cloudlist = callPackage ../tools/security/cloudlist { };

  cmdpack = callPackages ../tools/misc/cmdpack { };

  cobalt = callPackage ../applications/misc/cobalt {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  cobang = python3Packages.callPackage ../applications/misc/cobang {
    gst-plugins-good = gst_all_1.gst-plugins-good.override { gtkSupport = true; };
  };

  cocoapods = callPackage ../development/tools/cocoapods { };

  cocoapods-beta = lowPrio (callPackage ../development/tools/cocoapods { beta = true; });

  cocom = callPackage ../tools/networking/cocom {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  codeberg-pages = callPackage ../development/tools/continuous-integration/codeberg-pages { };

  codebraid = callPackage ../tools/misc/codebraid { };

  codemov = callPackage ../tools/misc/codemov { };

  codevis = callPackage ../tools/misc/codevis { };

  compass = callPackage ../development/tools/compass { };

  cone = callPackage ../development/compilers/cone {
    llvmPackages = llvmPackages_13;
  };

  conda = callPackage ../tools/package-management/conda { };

  conduktor = callPackage ../applications/misc/conduktor { };

  console-bridge = callPackage ../development/libraries/console-bridge { };

  convbin = callPackage ../tools/misc/convbin { };

  convimg = callPackage ../tools/misc/convimg { };

  convfont = callPackage ../tools/misc/convfont { };

  convmv = callPackage ../tools/misc/convmv { };

  cpcfs = callPackage ../tools/filesystems/cpcfs { };

  coreutils =  callPackage ../tools/misc/coreutils { };

  # The coreutils above are built with dependencies from
  # bootstrapping. We cannot override it here, because that pulls in
  # openssl from the previous stage as well.
  coreutils-full = callPackage ../tools/misc/coreutils { minimal = false; };
  coreutils-prefixed = coreutils.override { withPrefix = true; singleBinary = false; };

  corkscrew = callPackage ../tools/networking/corkscrew { };

  cowpatty = callPackage ../tools/security/cowpatty { };

  cpio = callPackage ../tools/archivers/cpio { };

  cpufetch = callPackage ../tools/misc/cpufetch { };

  crackmapexec = callPackage ../tools/security/crackmapexec { };

  crackxls = callPackage ../tools/security/crackxls { };

  crd2pulumi = callPackage ../development/tools/crd2pulumi { };

  create-cycle-app = nodePackages.create-cycle-app;

  createrepo_c = callPackage ../tools/package-management/createrepo_c { };

  cromfs = callPackage ../tools/archivers/cromfs { };

  cron = callPackage ../tools/system/cron { };

  ctlptl = callPackage ../development/tools/ctlptl { };

  dumpnar = callPackage ../tools/archivers/dumpnar { };

  snooze = callPackage ../tools/system/snooze { };

  cudaPackages_10_0 = callPackage ./cuda-packages.nix { cudaVersion = "10.0"; };
  cudaPackages_10_1 = callPackage ./cuda-packages.nix { cudaVersion = "10.1"; };
  cudaPackages_10_2 = callPackage ./cuda-packages.nix { cudaVersion = "10.2"; };
  cudaPackages_10 = recurseIntoAttrs cudaPackages_10_2;

  cudaPackages_11_0 = callPackage ./cuda-packages.nix { cudaVersion = "11.0"; };
  cudaPackages_11_1 = callPackage ./cuda-packages.nix { cudaVersion = "11.1"; };
  cudaPackages_11_2 = callPackage ./cuda-packages.nix { cudaVersion = "11.2"; };
  cudaPackages_11_3 = callPackage ./cuda-packages.nix { cudaVersion = "11.3"; };
  cudaPackages_11_4 = callPackage ./cuda-packages.nix { cudaVersion = "11.4"; };
  cudaPackages_11_5 = callPackage ./cuda-packages.nix { cudaVersion = "11.5"; };
  cudaPackages_11_6 = callPackage ./cuda-packages.nix { cudaVersion = "11.6"; };
  cudaPackages_11_7 = callPackage ./cuda-packages.nix { cudaVersion = "11.7"; };
  cudaPackages_11_8 = callPackage ./cuda-packages.nix { cudaVersion = "11.8"; };
  cudaPackages_11 = recurseIntoAttrs cudaPackages_11_8;

  cudaPackages_12_0 = callPackage ./cuda-packages.nix { cudaVersion = "12.0"; };
  cudaPackages_12_1 = callPackage ./cuda-packages.nix { cudaVersion = "12.1"; };
  cudaPackages_12_2 = callPackage ./cuda-packages.nix { cudaVersion = "12.2"; };
  cudaPackages_12_3 = callPackage ./cuda-packages.nix { cudaVersion = "12.3"; };
  cudaPackages_12 = cudaPackages_12_2; # Latest supported by cudnn

  cudaPackages = recurseIntoAttrs cudaPackages_12;

  # TODO: move to alias
  cudatoolkit = cudaPackages.cudatoolkit;
  cudatoolkit_11 = cudaPackages_11.cudatoolkit;

  curie = callPackage ../data/fonts/curie { };

  curlFull = curl.override {
    ldapSupport = true;
    gsaslSupport = true;
    rtmpSupport = true;
    pslSupport = true;
    websocketSupport = true;
  };

  curlHTTP3 = curl.override {
    openssl = quictls;
    http3Support = true;
  };

  curl = curlMinimal.override ({
    idnSupport = true;
    pslSupport = true;
    zstdSupport = true;
  } // lib.optionalAttrs (!stdenv.hostPlatform.isStatic) {
    brotliSupport = true;
  });

  curlMinimal = callPackage ../tools/networking/curl { };

  curlWithGnuTls = curl.override { gnutlsSupport = true; opensslSupport = false; };

  curl-impersonate = darwin.apple_sdk_11_0.callPackage ../tools/networking/curl-impersonate { };
  curl-impersonate-ff = curl-impersonate.curl-impersonate-ff;
  curl-impersonate-chrome = curl-impersonate.curl-impersonate-chrome;

  curlie = callPackage ../tools/networking/curlie { };

  trurl = callPackage ../tools/networking/trurl { };

  curv = callPackage ../applications/graphics/curv { };

  cunit = callPackage ../tools/misc/cunit { };
  bcunit = callPackage ../tools/misc/bcunit { };

  curlftpfs = callPackage ../tools/filesystems/curlftpfs { };

  tcp-cutter = callPackage ../tools/networking/tcp-cutter { };

  cwebbin = callPackage ../development/tools/misc/cwebbin { };

  cve-bin-tool = python3Packages.callPackage ../tools/security/cve-bin-tool { };

  cvs-fast-export = callPackage ../applications/version-management/cvs-fast-export { };

  dadadodo = callPackage ../tools/text/dadadodo { };

  daemon = callPackage ../tools/system/daemon { };

  daemonize = callPackage ../tools/system/daemonize { };

  danger-gitlab = callPackage ../applications/version-management/danger-gitlab { };

  daq = callPackage ../applications/networking/ids/daq { };

  dar = callPackage ../tools/backup/dar {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation;
  };

  darkhttpd = callPackage ../servers/http/darkhttpd { };

  darkstat = callPackage ../tools/networking/darkstat { };

  dav1d = callPackage ../development/libraries/dav1d { };

  davfs2 = callPackage ../tools/filesystems/davfs2 { };

  dbench = callPackage ../development/tools/misc/dbench { };

  dclxvi = callPackage ../development/libraries/dclxvi { };

  dconf2nix = callPackage ../development/tools/haskell/dconf2nix { };

  dcraw = callPackage ../tools/graphics/dcraw { };

  dcs = callPackage ../tools/text/dcs { };

  dcfldd = callPackage ../tools/system/dcfldd { };

  debian-devscripts = callPackage ../tools/misc/debian-devscripts { };

  debian-goodies = callPackage ../applications/misc/debian-goodies { };

  debootstrap = callPackage ../tools/misc/debootstrap { };

  debugedit = callPackage ../development/tools/misc/debugedit { };

  decode-spam-headers = callPackage ../tools/networking/decode-spam-headers { };

  deer = callPackage ../shells/zsh/zsh-deer { };

  deno = callPackage ../development/web/deno { };

  deqp-runner = callPackage ../tools/graphics/deqp-runner { };

  detox = callPackage ../tools/misc/detox { };

  devilspie2 = callPackage ../applications/misc/devilspie2 {
    gtk = gtk3;
  };

  dex = callPackage ../tools/X11/dex { };

  ddccontrol = callPackage ../tools/misc/ddccontrol { };

  ddccontrol-db = callPackage ../data/misc/ddccontrol-db { };

  ddcui = libsForQt5.callPackage ../applications/misc/ddcui { };

  ddcutil = callPackage ../tools/misc/ddcutil { };

  ddclient = callPackage ../tools/networking/ddclient { };

  dd_rescue = callPackage ../tools/system/dd_rescue { };

  ddh = callPackage ../tools/system/ddh { };

  ddrescue = callPackage ../tools/system/ddrescue { };

  ddrescueview = callPackage ../tools/system/ddrescueview { };

  ddrutility = callPackage ../tools/system/ddrutility { };

  inherit (callPackages ../applications/networking/p2p/deluge { })
    deluge-gtk
    deluged
    deluge;

  deluge-2_x = deluge;

  der-ascii = callPackage ../tools/security/der-ascii { };

  desktop-file-utils = callPackage ../tools/misc/desktop-file-utils { };

  dfc  = callPackage ../tools/system/dfc { };

  dfrs  = callPackage ../tools/system/dfrs { };

  dev86 = callPackage ../development/compilers/dev86 { };

  diskrsync = callPackage ../tools/backup/diskrsync { };

  djbdns = callPackage ../tools/networking/djbdns { };

  dnscrypt-proxy = callPackage ../tools/networking/dnscrypt-proxy { };

  dnscrypt-wrapper = callPackage ../tools/networking/dnscrypt-wrapper { };

  dnscontrol = callPackage ../applications/networking/dnscontrol { };

  dnsenum = callPackage ../tools/security/dnsenum { };

  dnsmasq = callPackage ../tools/networking/dnsmasq { };

  dnsmon-go = callPackage ../tools/networking/dnsmon-go { };

  dnsmonster = callPackage ../tools/networking/dnsmonster { };

  dnspeep = callPackage ../tools/security/dnspeep { };

  dnsproxy = callPackage ../tools/networking/dnsproxy { };

  dnsperf = callPackage ../tools/networking/dnsperf { };

  dnsrecon = callPackage ../tools/security/dnsrecon { };

  dnstop = callPackage ../tools/networking/dnstop { };

  dnsviz = python3Packages.callPackage ../tools/networking/dnsviz { };

  dnsx = callPackage ../tools/security/dnsx { };

  dhcpdump = callPackage ../tools/networking/dhcpdump { };

  dhcpcd = callPackage ../tools/networking/dhcpcd { };

  dhcping = callPackage ../tools/networking/dhcping { };

  dotacat = callPackage ../tools/misc/dotacat { };

  di = callPackage ../tools/system/di { };

  diction = callPackage ../tools/text/diction { };

  diffoci = callPackage ../tools/misc/diffoci { };

  diffoscope = callPackage ../tools/misc/diffoscope {
    jdk = jdk8;
  };

  diffoscopeMinimal = diffoscope.override {
    enableBloat = false;
  };

  diffr = callPackage ../tools/text/diffr {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  diffsitter = callPackage ../tools/text/diffsitter { };

  diffstat = callPackage ../tools/text/diffstat { };

  difftastic = callPackage ../tools/text/difftastic { };

  diffutils = callPackage ../tools/text/diffutils { };

  dsp = callPackage ../tools/audio/dsp { };

  dirdiff = callPackage ../tools/text/dirdiff { };

  dwdiff = callPackage ../applications/misc/dwdiff { };

  picoscope = callPackage ../applications/science/electronics/picoscope { };

  picotts = callPackage ../tools/audio/picotts { };

  wgetpaste = callPackage ../tools/text/wgetpaste { };

  dismap = callPackage ../tools/security/dismap { };

  dismember = callPackage ../tools/security/dismember { };

  dirvish  = callPackage ../tools/backup/dirvish { };

  dleyna-connector-dbus = callPackage ../development/libraries/dleyna-connector-dbus { };

  dleyna-core = callPackage ../development/libraries/dleyna-core { };

  dleyna-renderer = callPackage ../development/libraries/dleyna-renderer { };

  dleyna-server = callPackage ../development/libraries/dleyna-server { };

  dmd = callPackage ../by-name/dm/dmd/package.nix ({
    inherit (darwin.apple_sdk.frameworks) Foundation;
  } // lib.optionalAttrs stdenv.isLinux {
    # https://github.com/NixOS/nixpkgs/pull/206907#issuecomment-1527034123
    stdenv = gcc11Stdenv;
  });

  dmg2img = callPackage ../tools/misc/dmg2img { };

  docbook2odf = callPackage ../tools/typesetting/docbook2odf { };

  doas = callPackage ../tools/security/doas { };

  doas-sudo-shim = callPackage ../tools/security/doas-sudo-shim { };

  docbook2x = callPackage ../tools/typesetting/docbook2x { };

  docbook2mdoc = callPackage ../tools/misc/docbook2mdoc { };

  docear = callPackage ../applications/office/docear { };

  dockbarx = callPackage ../applications/misc/dockbarx { };

  doctave = callPackage ../applications/misc/doctave {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  dog = callPackage ../tools/system/dog { };

  dogdns = callPackage ../tools/networking/dogdns {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  dq = callPackage ../tools/networking/dq { };

  dool = callPackage ../tools/system/dool { };

  dosfstools = callPackage ../tools/filesystems/dosfstools { };

  dotnetfx35 = callPackage ../development/libraries/dotnetfx35 { };

  dotnetfx40 = callPackage ../development/libraries/dotnetfx40 { };

  domoticz = callPackage ../servers/domoticz { };

  dorkscout = callPackage ../tools/security/dorkscout { };

  sl1-to-photon = python3Packages.callPackage ../applications/misc/sl1-to-photon { };

  drive = callPackage ../applications/networking/drive { };

  driftnet = callPackage ../tools/networking/driftnet { };

  driftctl = callPackage ../applications/networking/cluster/driftctl { };

  eks-node-viewer = callPackage ../applications/networking/cluster/eks-node-viewer { };

  drill = callPackage ../tools/networking/drill {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  drone = callPackage ../development/tools/continuous-integration/drone { };
  drone-oss = callPackage ../development/tools/continuous-integration/drone {
    enableUnfree = false;
  };

  drone-cli = callPackage ../development/tools/continuous-integration/drone-cli { };

  drone-runner-exec = callPackage ../development/tools/continuous-integration/drone-runner-exec { };

  drone-runner-docker = callPackage ../development/tools/continuous-integration/drone-runner-docker { };

  drone-runner-ssh = callPackage ../development/tools/continuous-integration/drone-runner-ssh { };

  dropbear = callPackage ../tools/networking/dropbear { };

  dsview = qt6Packages.callPackage ../applications/science/electronics/dsview { };

  dtach = callPackage ../tools/misc/dtach { };

  dtc = callPackage ../development/compilers/dtc { };

  dt-schema = callPackage ../development/tools/dt-schema { };

  dub = callPackage ../development/tools/build-managers/dub { };

  inherit (import ../build-support/dlang/dub-support.nix { inherit callPackage; })
    buildDubPackage dub-to-nix;

  duc = callPackage ../tools/misc/duc { };

  duff = callPackage ../tools/filesystems/duff {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  dump_syms = callPackage ../development/tools/dump_syms {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  dumpasn1 = callPackage ../tools/security/dumpasn1 { };

  dumptorrent = callPackage ../tools/misc/dumptorrent { };

  duo-unix = callPackage ../tools/security/duo-unix { };

  dupe-krill = callPackage ../tools/filesystems/dupe-krill { };

  duplicacy = callPackage ../tools/backup/duplicacy { };

  duplicati = callPackage ../tools/backup/duplicati { };

  duplicity = callPackage ../tools/backup/duplicity { };

  duply = callPackage ../tools/backup/duply { };

  dvd-vr = callPackage ../tools/cd-dvd/dvd-vr { };

  dvdisaster = callPackage ../tools/cd-dvd/dvdisaster { };

  dvdplusrwtools = callPackage ../tools/cd-dvd/dvd+rw-tools { };

  dvgrab = callPackage ../tools/video/dvgrab { };

  dvtm = callPackage ../tools/misc/dvtm {
    # if you prefer a custom config, write the config.h in dvtm.config.h
    # and enable
    # customConfig = builtins.readFile ./dvtm.config.h;
  };

  dvtm-unstable = callPackage ../tools/misc/dvtm/unstable.nix { };

  erofs-utils = callPackage ../tools/filesystems/erofs-utils { };

  e2tools = callPackage ../tools/filesystems/e2tools { };

  e2fsprogs = callPackage ../tools/filesystems/e2fsprogs { };

  easyrsa = callPackage ../tools/networking/easyrsa { };

  easysnap = callPackage ../tools/backup/easysnap { };

  ebook_tools = callPackage ../tools/text/ebook-tools { };

  ecryptfs = callPackage ../tools/security/ecryptfs { };

  ecryptfs-helper = callPackage ../tools/security/ecryptfs/helper.nix { };

  edgetpu-compiler = callPackage ../development/libraries/science/robotics/edgetpu-compiler { };

  edid-decode = callPackage ../tools/misc/edid-decode { };

  edid-generator = callPackage ../tools/misc/edid-generator { };

  edir = callPackage ../tools/misc/edir { };

  editres = callPackage ../tools/graphics/editres { };

  edit = callPackage ../applications/editors/edit { };

  edk2 = callPackage ../development/compilers/edk2 { };

  edk2-uefi-shell = callPackage ../tools/misc/edk2-uefi-shell { };

  edl = callPackage ../development/embedded/edl { };

  edlib = callPackage ../development/libraries/science/biology/edlib { };

  eff = callPackage ../development/interpreters/eff { };

  eflite = callPackage ../applications/audio/eflite { };

  eid-mw = callPackage ../tools/security/eid-mw {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  emote = callPackage ../tools/inputmethods/emote { };

  engauge-digitizer = libsForQt5.callPackage ../applications/science/math/engauge-digitizer { };

  epub2txt2 = callPackage ../tools/text/epub2txt2 { };

  epubcheck = callPackage ../tools/text/epubcheck { };

  evil-winrm = callPackage ../tools/security/evil-winrm { };

  evtx = callPackage ../tools/security/evtx { };

  luckybackup = libsForQt5.callPackage ../tools/backup/luckybackup {
    ssh = openssh;
  };

  kics = callPackage ../tools/admin/kics { };

  kramdown-asciidoc = callPackage ../tools/typesetting/kramdown-asciidoc { };

  lychee = callPackage ../tools/networking/lychee {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  magic-vlsi = callPackage ../applications/science/electronics/magic-vlsi { };

  mcrcon = callPackage ../tools/networking/mcrcon { };

  mozillavpn = qt6Packages.callPackage ../tools/networking/mozillavpn { };

  mozwire = callPackage ../tools/networking/mozwire {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  pax = callPackage ../tools/archivers/pax {
    inherit (pkgs.darwin.apple_sdk.libs) utmp;
  };

  rar2fs = callPackage ../tools/filesystems/rar2fs { };

  rocmPackages = rocmPackages_6;
  rocmPackages_5 = recurseIntoAttrs (callPackage ../development/rocm-modules/5 { });
  rocmPackages_6 = recurseIntoAttrs (callPackage ../development/rocm-modules/6 { });

  rune = callPackage ../development/interpreters/rune { };

  s9fes = callPackage ../development/interpreters/s9fes { };

  s-tar = callPackage ../tools/archivers/s-tar { };

  solo2-cli = callPackage ../tools/security/solo2-cli {
    inherit (darwin.apple_sdk.frameworks) PCSC IOKit CoreFoundation AppKit;
  };

  sonota = callPackage ../tools/misc/sonota { };

  sonobuoy = callPackage ../applications/networking/cluster/sonobuoy { };

  stratisd = callPackage ../tools/filesystems/stratisd { };

  stratis-cli = callPackage ../tools/filesystems/stratis-cli { };

  strawberry-qt5 = libsForQt5.callPackage ../applications/audio/strawberry { };

  strawberry-qt6 = qt6Packages.callPackage ../applications/audio/strawberry { };

  strawberry = strawberry-qt5;

  schleuder = callPackage ../tools/security/schleuder { };

  schleuder-cli = callPackage ../tools/security/schleuder/cli { };

  tealdeer = callPackage ../tools/misc/tealdeer {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  teamocil = callPackage ../tools/misc/teamocil { };

  the-way = callPackage ../development/tools/the-way { };

  timeline = callPackage ../applications/office/timeline { };

  tsm-client = callPackage ../tools/backup/tsm-client { };
  tsm-client-withGui = callPackage ../tools/backup/tsm-client { enableGui = true; };

  tracker = callPackage ../development/libraries/tracker { };

  tracy = callPackage ../development/tools/tracy { };

  trivy = callPackage ../tools/admin/trivy { };

  trompeloeil = callPackage ../development/libraries/trompeloeil { };

  updog = callPackage ../tools/networking/updog { };

  uudeview = callPackage ../tools/misc/uudeview { };

  uusi = haskell.lib.compose.justStaticExecutables haskellPackages.uusi;

  uutils-coreutils = callPackage ../tools/misc/uutils-coreutils {
    inherit (python3Packages) sphinx;
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  uutils-coreutils-noprefix = uutils-coreutils.override { prefix = null; };

  volctl = callPackage ../tools/audio/volctl { };

  volk = if (stdenv.isDarwin && stdenv.isAarch64) then
    (callPackage ../development/libraries/volk/2.5.0.nix { })
  else
    (callPackage ../development/libraries/volk { })
  ;

  vorta = qt6Packages.callPackage ../applications/backup/vorta { };

  vowpal-wabbit = callPackage ../applications/science/machine-learning/vowpal-wabbit { };

  vt-cli = callPackage ../tools/security/vt-cli { };

  wakeonlan = callPackage ../tools/networking/wakeonlan { };

  wallutils = callPackage ../tools/graphics/wallutils { };

  wayback = callPackage ../tools/misc/wayback { };

  worker-build = callPackage ../development/tools/worker-build {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  inherit (nodePackages) wrangler;

  wrangler_1 = callPackage ../development/tools/wrangler_1 {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation CoreServices Security;
  };

  snowflake = callPackage ../tools/networking/snowflake { };

  wsl-open = callPackage ../tools/misc/wsl-open { };

  wsl-vpnkit = callPackage ../tools/networking/wsl-vpnkit { };

  xkcdpass = with python3Packages; toPythonApplication xkcdpass;

  xjobs = callPackage ../tools/misc/xjobs { };

  xob = callPackage ../tools/X11/xob { };

  z-lua = callPackage ../tools/misc/z-lua { };

  zabbix-cli = callPackage ../tools/misc/zabbix-cli { };

  zabbixctl = callPackage ../tools/misc/zabbixctl { };

  zee = callPackage ../applications/editors/zee {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  zeek = darwin.apple_sdk_11_0.callPackage ../applications/networking/ids/zeek { };

  zeekscript = callPackage ../tools/security/zeekscript { };

  zoekt = callPackage ../tools/text/zoekt {
    buildGoModule = buildGo121Module;
  };

  zonemaster-cli = perlPackages.ZonemasterCLI;

  zotero-translation-server = callPackage ../tools/misc/zotero-translation-server { };

  zoxide = callPackage ../tools/misc/zoxide { };

  zzuf = callPackage ../tools/security/zzuf { };

  ### DEVELOPMENT / EMSCRIPTEN

  buildEmscriptenPackage = callPackage ../development/em-modules/generic { };

  carp = callPackage ../development/compilers/carp { };

  cholmod-extra = callPackage ../development/libraries/science/math/cholmod-extra { };

  choose = callPackage ../tools/text/choose { };

  d2 = callPackage ../tools/text/d2 { };

  ddev = callPackage ../applications/virtualization/ddev { };

  easeprobe = callPackage ../tools/misc/easeprobe { };

  emscripten = callPackage ../development/compilers/emscripten {
    llvmPackages = llvmPackages_17;
  };

  emscriptenPackages = recurseIntoAttrs (callPackage ./emscripten-packages.nix { });

  emscriptenStdenv = stdenv // { mkDerivation = buildEmscriptenPackage; };

  efibootmgr = callPackage ../tools/system/efibootmgr { };

  efivar = callPackage ../tools/system/efivar { };

  eget = callPackage ../tools/misc/eget { };

  evemu = callPackage ../tools/system/evemu { };

  # The latest version used by elasticsearch, logstash, kibana and the the beats from elastic.
  # When updating make sure to update all plugins or they will break!
  elk7Version = "7.17.16";

  elasticsearch7 = callPackage ../servers/search/elasticsearch/7.x.nix {
    util-linux = util-linuxMinimal;
    jre_headless = jdk11_headless; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };
  elasticsearch = elasticsearch7;

  elasticsearchPlugins = recurseIntoAttrs (
    callPackage ../servers/search/elasticsearch/plugins.nix {}
  );

  embree = callPackage ../development/libraries/embree { };
  embree2 = callPackage ../development/libraries/embree/2.x.nix { };

  emborg = python3Packages.callPackage ../development/python-modules/emborg { };

  emblem = callPackage ../applications/graphics/emblem { };

  emem = callPackage ../applications/misc/emem { };

  empty = callPackage ../tools/misc/empty { };

  emulsion = callPackage ../applications/graphics/emulsion {
    inherit (darwin.apple_sdk.frameworks) AppKit CoreGraphics CoreServices Foundation OpenGL;
  };

  emulsion-palette = callPackage ../applications/graphics/emulsion-palette { };

  emv = callPackage ../tools/misc/emv { };

  enblend-enfuse = callPackage ../tools/graphics/enblend-enfuse { };

  enc = callPackage ../tools/security/enc { };

  endlessh = callPackage ../servers/endlessh { };

  ericw-tools = callPackage ../applications/misc/ericw-tools { };

  cryfs = callPackage ../tools/filesystems/cryfs { };

  encfs = callPackage ../tools/filesystems/encfs {
    tinyxml2 = tinyxml-2;
  };

  enscript = callPackage ../tools/text/enscript { };

  ensemble-chorus = callPackage ../applications/audio/ensemble-chorus { };

  entr = callPackage ../tools/misc/entr { };

  envchain = callPackage ../tools/misc/envchain { inherit (darwin.apple_sdk.frameworks) Security; };

  eot_utilities = callPackage ../tools/misc/eot-utilities { };

  eplot = callPackage ../tools/graphics/eplot { };

  epstool = callPackage ../tools/graphics/epstool { };

  escrotum = callPackage ../tools/graphics/escrotum { };

  esshader = callPackage ../tools/graphics/esshader { };

  ethercalc = callPackage ../servers/web-apps/ethercalc { };

  ethtool = callPackage ../tools/misc/ethtool { };

  ets = callPackage ../tools/misc/ets { };

  ettercap = callPackage ../applications/networking/sniffers/ettercap { };

  evcc = callPackage ../servers/home-automation/evcc { };

  eventstat = callPackage ../os-specific/linux/eventstat { };

  evillimiter = python3Packages.callPackage ../tools/networking/evillimiter { };

  evtest = callPackage ../applications/misc/evtest { };

  evtest-qt = libsForQt5.callPackage ../applications/misc/evtest-qt { };

  eva = callPackage ../tools/misc/eva { };

  exactaudiocopy = callPackage ../applications/audio/exact-audio-copy { };

  exempi = callPackage ../development/libraries/exempi { };

  executor = with python3Packages; toPythonApplication executor;

  exif = callPackage ../tools/graphics/exif { };

  exifprobe = callPackage ../tools/graphics/exifprobe { };

  exiftags = callPackage ../tools/graphics/exiftags { };

  exiftool = perlPackages.ImageExifTool;

  ext4magic = callPackage ../tools/filesystems/ext4magic { };

  extract_url = callPackage ../applications/misc/extract_url { };

  extundelete = callPackage ../tools/filesystems/extundelete { };

  expect = callPackage ../tools/misc/expect { };

  expected-lite = callPackage ../development/libraries/expected-lite { };

  exportarr = callPackage ../servers/monitoring/prometheus/exportarr { };

  expliot = callPackage ../tools/security/expliot { };

  f2fs-tools = callPackage ../tools/filesystems/f2fs-tools { };

  Fabric = with python3Packages; toPythonApplication fabric;

  fail2ban = callPackage ../tools/security/fail2ban { };

  fakeroot = callPackage ../tools/system/fakeroot { };

  fakeroute = callPackage ../tools/networking/fakeroute { };

  fakechroot = callPackage ../tools/system/fakechroot { };

  fancy-motd = callPackage ../tools/system/fancy-motd { };

  fastpbkdf2 = callPackage ../development/libraries/fastpbkdf2 { };

  fanficfare = callPackage ../tools/text/fanficfare { };

  fastd = callPackage ../tools/networking/fastd { };

  fatsort = callPackage ../tools/filesystems/fatsort { };

  chewing-editor = libsForQt5.callPackage ../applications/misc/chewing-editor { };

  fcitx5 = callPackage ../tools/inputmethods/fcitx5 { };

  fcitx5-bamboo = callPackage ../tools/inputmethods/fcitx5/fcitx5-bamboo.nix { };

  fcitx5-mozc = libsForQt5.callPackage ../tools/inputmethods/fcitx5/fcitx5-mozc.nix {
    abseil-cpp = abseil-cpp.override {
      cxxStandard = "17";
    };
  };

  fcitx5-skk = qt6Packages.callPackage ../tools/inputmethods/fcitx5/fcitx5-skk.nix { };

  fcitx5-anthy = callPackage ../tools/inputmethods/fcitx5/fcitx5-anthy.nix { };

  fcitx5-chewing = callPackage ../tools/inputmethods/fcitx5/fcitx5-chewing.nix { };

  fcitx5-lua = callPackage ../tools/inputmethods/fcitx5/fcitx5-lua.nix { lua = lua5_3; };

  fcitx5-m17n = callPackage ../tools/inputmethods/fcitx5/fcitx5-m17n.nix { };

  fcitx5-openbangla-keyboard = libsForQt5.callPackage ../applications/misc/openbangla-keyboard { withFcitx5Support = true; };

  fcitx5-gtk = callPackage ../tools/inputmethods/fcitx5/fcitx5-gtk.nix { };

  fcitx5-hangul = callPackage ../tools/inputmethods/fcitx5/fcitx5-hangul.nix { };

  fcitx5-rime = callPackage ../tools/inputmethods/fcitx5/fcitx5-rime.nix { };

  fcitx5-table-extra = callPackage ../tools/inputmethods/fcitx5/fcitx5-table-extra.nix { };

  fcitx5-table-other = callPackage ../tools/inputmethods/fcitx5/fcitx5-table-other.nix { };

  fcppt = callPackage ../development/libraries/fcppt { };

  fcrackzip = callPackage ../tools/security/fcrackzip { };

  fcron = callPackage ../tools/system/fcron { };

  fdm = callPackage ../tools/networking/fdm { };

  fdtools = callPackage ../tools/misc/fdtools { };

  featherpad = qt5.callPackage ../applications/editors/featherpad { };

  fend = callPackage ../tools/misc/fend { };

  ferm = callPackage ../tools/networking/ferm { };

  feroxbuster = callPackage ../tools/security/feroxbuster {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  ffsend = callPackage ../tools/misc/ffsend {
    inherit (darwin.apple_sdk.frameworks) Security AppKit;
  };

  fgallery = callPackage ../tools/graphics/fgallery { };

  flannel = callPackage ../tools/networking/flannel { };
  cni-plugin-flannel = callPackage ../tools/networking/flannel/plugin.nix { };

  flashbench = callPackage ../os-specific/linux/flashbench { };

  flatpak = callPackage ../development/libraries/flatpak { };

  flatpak-builder = callPackage ../development/tools/flatpak-builder {
    binutils = binutils-unwrapped;
  };

  fltrdr = callPackage ../tools/misc/fltrdr {
    icu = icu63;
  };

  fluent-bit = darwin.apple_sdk_11_0.callPackage ../tools/misc/fluent-bit { };

  fluent-reader = callPackage ../applications/networking/feedreaders/fluent-reader { };

  flux = callPackage ../development/compilers/flux { };

  fierce = callPackage ../tools/security/fierce { };

  figlet = callPackage ../tools/misc/figlet { };

  file = callPackage ../tools/misc/file {
    inherit (windows) libgnurx;
  };

  filegive = callPackage ../tools/networking/filegive { };

  fileschanged = callPackage ../tools/misc/fileschanged { };

  filet = callPackage ../applications/misc/filet { };

  findex = callPackage ../applications/misc/findex { };

  findomain = callPackage ../tools/networking/findomain { };

  findutils = callPackage ../tools/misc/findutils { };

  findup = callPackage ../tools/misc/findup { };

  fingerprintx = callPackage ../tools/security/fingerprintx { };

  bsd-finger = callPackage ../tools/networking/bsd-finger { };
  bsd-fingerd = bsd-finger.override({ buildClient = false; });

  iprange = callPackage ../applications/networking/firehol/iprange.nix { };

  firehol = callPackage ../applications/networking/firehol { };

  fio = callPackage ../tools/system/fio { };

  flamerobin = callPackage ../applications/misc/flamerobin { };

  flashrom = callPackage ../tools/misc/flashrom { };

  flent = python3Packages.callPackage ../applications/networking/flent { };

  flexoptix-app = callPackage ../tools/misc/flexoptix-app { };

  flpsed = callPackage ../applications/editors/flpsed { };

  fluentd = callPackage ../tools/misc/fluentd { };

  flvstreamer = callPackage ../tools/networking/flvstreamer { };

  frr = callPackage ../servers/frr { };

  gemstash = callPackage ../development/tools/gemstash { };

  hmetis = pkgsi686Linux.callPackage ../applications/science/math/hmetis { };

  libbsd = callPackage ../development/libraries/libbsd { };

  libbladeRF = callPackage ../development/libraries/libbladeRF { };

  libbtbb = callPackage ../development/libraries/libbtbb {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  lpd8editor = libsForQt5.callPackage ../applications/audio/lpd8editor {};

  lp_solve = callPackage ../applications/science/math/lp_solve {
    inherit (darwin) cctools autoSignDarwinBinariesHook;
  };

  fabric-installer = callPackage ../tools/games/minecraft/fabric-installer { };

  faraday-agent-dispatcher = callPackage ../tools/security/faraday-agent-dispatcher { };

  faraday-cli = callPackage ../tools/security/faraday-cli { };

  fastlane = callPackage ../tools/admin/fastlane { };

  fatresize = callPackage ../tools/filesystems/fatresize { };

  fdk_aac = callPackage ../development/libraries/fdk-aac { };

  fdk-aac-encoder = callPackage ../applications/audio/fdkaac { };

  fead = callPackage ../applications/misc/fead { };

  feedgnuplot = callPackage ../tools/graphics/feedgnuplot { };

  fermyon-spin = callPackage ../development/tools/fermyon-spin { };

  fbcat = callPackage ../tools/misc/fbcat { };

  fbjni = callPackage ../development/libraries/fbjni { };

  fbv = callPackage ../tools/graphics/fbv { };

  fbvnc = callPackage ../tools/admin/fbvnc { };

  fim = callPackage ../tools/graphics/fim { };

  flamegraph = callPackage ../development/tools/flamegraph { };

  flawfinder = callPackage ../development/tools/flawfinder { };

  flintlock = callPackage ../applications/virtualization/flintlock { };

  flip-link = callPackage ../development/tools/flip-link { };

  flips = callPackage ../tools/compression/flips { };

  flowblade = callPackage ../applications/video/flowblade { };

  fontforge = lowPrio (callPackage ../tools/misc/fontforge {
    inherit (darwin.apple_sdk.frameworks) Carbon Cocoa;
    python = python3;
  });
  fontforge-gtk = fontforge.override {
    withSpiro = true;
    withGTK = true;
    gtk3 = gtk3-x11;
    inherit (darwin.apple_sdk.frameworks) Carbon Cocoa;
  };

  fontforge-fonttools = callPackage ../tools/misc/fontforge/fontforge-fonttools.nix { };

  fontmatrix = libsForQt5.callPackage ../applications/graphics/fontmatrix { };

  foremost = callPackage ../tools/system/foremost { };

  forktty = callPackage ../os-specific/linux/forktty { };

  fortran-fpm = callPackage ../tools/package-management/fortran-fpm { };

  fortune = callPackage ../tools/misc/fortune { };

  fox = callPackage ../development/libraries/fox {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  fox_1_6 = callPackage ../development/libraries/fox/fox-1.6.nix {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  fpart = callPackage ../tools/misc/fpart { };

  fpattern = callPackage ../development/libraries/fpattern { };

  fping = callPackage ../tools/networking/fping { };

  fpm = callPackage ../tools/package-management/fpm { };

  fprintd = callPackage ../tools/security/fprintd { };

  fprintd-tod = callPackage ../tools/security/fprintd/tod.nix { };

  ferdi = callPackage ../applications/networking/instant-messengers/ferdi {
    mkFranzDerivation = callPackage ../applications/networking/instant-messengers/franz/generic.nix { };
  };

  ferdium = callPackage ../applications/networking/instant-messengers/ferdium {
    mkFranzDerivation = callPackage ../applications/networking/instant-messengers/franz/generic.nix { };
  };

  fq = callPackage ../development/tools/fq { };

  franz = callPackage ../applications/networking/instant-messengers/franz {
    mkFranzDerivation = callPackage ../applications/networking/instant-messengers/franz/generic.nix { };
  };

  freac = callPackage ../applications/audio/freac { };

  freebind = callPackage ../tools/networking/freebind { };

  freeipmi = callPackage ../tools/system/freeipmi { };

  freetalk = callPackage ../applications/networking/instant-messengers/freetalk { };

  freetds = callPackage ../development/libraries/freetds { };

  freetube = callPackage ../applications/video/freetube { };

  freeze = callPackage ../tools/security/freeze { };

  freqtweak = callPackage ../applications/audio/freqtweak {
    wxGTK = wxGTK32;
  };

  frescobaldi = python3Packages.callPackage ../misc/frescobaldi { };

  freshfetch = callPackage ../tools/misc/freshfetch {
    inherit (darwin.apple_sdk.frameworks) AppKit CoreFoundation DiskArbitration Foundation IOKit;
  };

  frigate = callPackage ../applications/video/frigate { };

  frostwire = callPackage ../applications/networking/p2p/frostwire { };
  frostwire-bin = callPackage ../applications/networking/p2p/frostwire/frostwire-bin.nix { };

  ftgl = callPackage ../development/libraries/ftgl { };

  ftop = callPackage ../os-specific/linux/ftop { };

  ftxui = callPackage ../development/libraries/ftxui { };

  fsarchiver = callPackage ../tools/archivers/fsarchiver { };

  fstl = qt5.callPackage ../applications/graphics/fstl { };

  fswebcam = callPackage ../os-specific/linux/fswebcam { };

  fulcio = callPackage ../tools/security/fulcio { };

  fuseiso = callPackage ../tools/filesystems/fuseiso { };

  fdbPackages = dontRecurseIntoAttrs (callPackage ../servers/foundationdb { });

  inherit (fdbPackages)
    foundationdb71
  ;

  foundationdb = foundationdb71;

  fuse-7z-ng = callPackage ../tools/filesystems/fuse-7z-ng { };

  fuse-archive = callPackage ../tools/filesystems/fuse-archive { };

  fuse-ext2 = darwin.apple_sdk_11_0.callPackage ../tools/filesystems/fuse-ext2 { };

  fuse-overlayfs = callPackage ../tools/filesystems/fuse-overlayfs { };

  fusee-interfacee-tk = callPackage ../applications/misc/fusee-interfacee-tk { };

  fusee-launcher = callPackage ../development/tools/fusee-launcher { };

  fusee-nano = callPackage ../development/tools/fusee-nano { };

  fverb = callPackage ../applications/audio/fverb { };

  fwknop = callPackage ../tools/security/fwknop {
    texinfo = texinfo6_7; # Uses @setcontentsaftertitlepage, removed in 6.8.
  };

  elastic = callPackage ../applications/misc/elastic { };

  exfat = callPackage ../tools/filesystems/exfat { };

  exfatprogs = callPackage ../tools/filesystems/exfatprogs { };

  dos2unix = callPackage ../tools/text/dos2unix { };

  uni2ascii = callPackage ../tools/text/uni2ascii { };

  uniscribe = callPackage ../tools/text/uniscribe { };

  calcoo = callPackage ../applications/misc/calcoo { };

  galculator = callPackage ../applications/misc/galculator {
    gtk = gtk3;
  };

  fornalder = callPackage ../applications/version-management/fornalder { };

  galen = callPackage ../development/tools/galen { };

  gallery-dl = python3Packages.callPackage ../applications/misc/gallery-dl { };

  gandi-cli = python3Packages.callPackage ../tools/networking/gandi-cli { };

  gandom-fonts = callPackage ../data/fonts/gandom-fonts { };

  gaphor = python3Packages.callPackage ../tools/misc/gaphor { };

  inherit (callPackages ../tools/filesystems/garage {
    inherit (darwin.apple_sdk.frameworks) Security;
  })
    garage
      garage_0_8 garage_0_9
      garage_0_8_7 garage_0_9_4
      garage_1_0_0 garage_1_x;

  garmintools = callPackage ../development/libraries/garmintools { };

  gau = callPackage ../tools/security/gau { };

  gauge-unwrapped = callPackage ../development/tools/gauge { };
  gauge = callPackage ../development/tools/gauge/wrapper.nix { };
  gaugePlugins = recurseIntoAttrs (callPackage ../development/tools/gauge/plugins {});

  gawd = python3Packages.toPythonApplication python3Packages.gawd;

  gawk = callPackage ../tools/text/gawk {
    inherit (darwin) locale;
  };

  gawk-with-extensions = callPackage ../tools/text/gawk/gawk-with-extensions.nix {
    extensions = gawkextlib.full;
  };
  gawkextlib = callPackage ../tools/text/gawk/gawkextlib.nix { };

  gawkInteractive = gawk.override { interactive = true; };

  gatk = callPackage ../applications/science/biology/gatk {} ;

  gbdfed = callPackage ../tools/misc/gbdfed {
    gtk = gtk2-x11;
  };

  gcfflasher = callPackage ../applications/misc/gcfflasher { };

  gdmap = callPackage ../tools/system/gdmap { };

  gdtoolkit = callPackage ../development/tools/gdtoolkit { };

  gef = callPackage ../development/tools/misc/gef { };

  gelasio = callPackage ../data/fonts/gelasio { };

  gemget = callPackage ../tools/networking/gemget { };

  gen-oath-safe = callPackage ../tools/security/gen-oath-safe { };

  genext2fs = callPackage ../tools/filesystems/genext2fs { };

  gengetopt = callPackage ../development/tools/misc/gengetopt { };

  genimage = callPackage ../tools/filesystems/genimage { };

  genmap = callPackage ../applications/science/biology/genmap { };

  geonkick = callPackage ../applications/audio/geonkick { };

  gerrit = callPackage ../applications/version-management/gerrit { };

  geteltorito = callPackage ../tools/misc/geteltorito { };

  getmail6 = callPackage ../tools/networking/getmail6 { };

  getopt = callPackage ../tools/misc/getopt { };

  getoptions = callPackage ../tools/misc/getoptions { };

  gexiv2 = callPackage ../development/libraries/gexiv2 { };

  gftp = callPackage ../applications/networking/ftp/gftp {
    gtk = gtk2;
  };

  gfs2-utils = callPackage ../tools/filesystems/gfs2-utils { };

  gfbgraph = callPackage ../development/libraries/gfbgraph { };

  ggobi = callPackage ../tools/graphics/ggobi { };

  ggshield = callPackage ../tools/security/ggshield { };

  ghost = callPackage ../tools/security/ghost { };

  ghostunnel = callPackage ../tools/networking/ghostunnel { };

  ghz = callPackage ../tools/networking/ghz { };

  gi-crystal = callPackage ../development/tools/gi-crystal { };

  gibberish-detector = with python3Packages; toPythonApplication gibberish-detector;

  gibo = callPackage ../tools/misc/gibo { };

  gifsicle = callPackage ../tools/graphics/gifsicle { };

  gifski = callPackage ../tools/graphics/gifski {
    ffmpeg = ffmpeg_5;
  };

  github-backup = callPackage ../tools/misc/github-backup { };

  github-runner = callPackage ../development/tools/continuous-integration/github-runner {
     inherit (darwin) autoSignDarwinBinariesHook;
  };

  gitkraken = callPackage ../applications/version-management/gitkraken { };

  gitlab = callPackage ../applications/version-management/gitlab { };
  gitlab-ee = callPackage ../applications/version-management/gitlab {
    gitlabEnterprise = true;
  };

  gitlab-clippy = callPackage ../development/tools/rust/gitlab-clippy { };

  gitlab-container-registry = callPackage ../applications/version-management/gitlab/gitlab-container-registry { };

  gitlab-pages = callPackage ../applications/version-management/gitlab/gitlab-pages { };

  gitlab-runner = callPackage ../development/tools/continuous-integration/gitlab-runner { };

  gitlab-shell = callPackage ../applications/version-management/gitlab/gitlab-shell { };

  gitlab-triage = callPackage ../applications/version-management/gitlab-triage { };

  gitlab-workhorse = callPackage ../applications/version-management/gitlab/gitlab-workhorse { };

  gitlab-elasticsearch-indexer = callPackage ../applications/version-management/gitlab/gitlab-elasticsearch-indexer { };

  gitleaks = callPackage ../tools/security/gitleaks { };

  gitaly = callPackage ../applications/version-management/gitlab/gitaly { };

  gitqlient = libsForQt5.callPackage ../applications/version-management/gitqlient { };

  gogs = callPackage ../applications/version-management/gogs { };

  git-latexdiff = callPackage ../tools/typesetting/git-latexdiff { };

  gitea = callPackage ../applications/version-management/gitea { };

  gitea-actions-runner = callPackage ../development/tools/continuous-integration/gitea-actions-runner { };

  gokart = callPackage ../development/tools/gokart { };

  gl2ps = callPackage ../development/libraries/gl2ps { };

  glusterfs = callPackage ../tools/filesystems/glusterfs { };

  glmark2 = callPackage ../tools/graphics/glmark2 { };

  glogg = libsForQt5.callPackage ../tools/text/glogg { };

  glxinfo = callPackage ../tools/graphics/glxinfo { };

  gmrender-resurrect = callPackage ../tools/networking/gmrender-resurrect {
    inherit (gst_all_1) gstreamer gst-plugins-base gst-plugins-good gst-plugins-bad gst-plugins-ugly gst-libav;
  };

  gnirehtet = callPackage ../tools/networking/gnirehtet { };

  gnome-builder = callPackage ../applications/editors/gnome-builder { };

  gnome-desktop = callPackage ../development/libraries/gnome-desktop { };

  gnome-decoder = callPackage ../applications/graphics/gnome-decoder {
     inherit (gst_all_1) gstreamer gst-plugins-base;
     gst-plugins-bad = gst_all_1.gst-plugins-bad.override { enableZbar = true; };
  };

  gnome-epub-thumbnailer = callPackage ../applications/misc/gnome-epub-thumbnailer { };

  gnome-extension-manager = callPackage ../applications/misc/gnome-extension-manager { };

  gnome-feeds = callPackage ../applications/networking/feedreaders/gnome-feeds { };

  gnome-frog = callPackage ../applications/misc/gnome-frog { };

  gnome-keysign = callPackage ../tools/security/gnome-keysign { };

  gnome-secrets = callPackage ../applications/misc/gnome-secrets { };

  gnome-solanum = callPackage ../applications/misc/gnome-solanum { };

  gnome-podcasts = callPackage ../applications/audio/gnome-podcasts { };

  gnome-photos = callPackage ../applications/graphics/gnome-photos { };

  gnome-randr = callPackage ../tools/wayland/gnome-randr { };

  gnuapl = callPackage ../development/interpreters/gnu-apl { };

  gnu-shepherd = callPackage ../misc/gnu-shepherd { };

  dapl = callPackage ../development/interpreters/dzaima-apl {
    buildNativeImage = false;
    stdenv = stdenvNoCC;
    jdk = jre;
  };
  dapl-native = callPackage ../development/interpreters/dzaima-apl {
    buildNativeImage = true;
    jdk = graalvm-ce;
  };

  gnucap-full = gnucap.withPlugins(p: [ p.verilog ]);

  gnu-cim = callPackage ../development/compilers/gnu-cim { };

  gnu-cobol = callPackage ../development/compilers/gnu-cobol { };

  gnuclad = callPackage ../applications/graphics/gnuclad { };

  gnufdisk = callPackage ../tools/system/fdisk {
    guile = guile_1_8;
  };

  gnugrep = callPackage ../tools/text/gnugrep { };

  gnulib = callPackage ../development/tools/gnulib { };

  gnupatch = callPackage ../tools/text/gnupatch { };

  gnupg1orig = callPackage ../tools/security/gnupg/1.nix { };
  gnupg1compat = callPackage ../tools/security/gnupg/1compat.nix { };
  gnupg1 = gnupg1compat;    # use config.packageOverrides if you prefer original gnupg1

  gnupg22 = callPackage ../tools/security/gnupg/22.nix {
    pinentry = if stdenv.isDarwin then pinentry_mac else pinentry-gtk2;
    libgcrypt = libgcrypt_1_8;
  };

  gnupg24 = callPackage ../tools/security/gnupg/24.nix {
    pinentry = if stdenv.isDarwin then pinentry_mac else pinentry-gtk2;
  };
  gnupg = gnupg24;

  gnupg-pkcs11-scd = callPackage ../tools/security/gnupg-pkcs11-scd { };

  gnuplot = libsForQt5.callPackage ../tools/graphics/gnuplot {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  gnuplot_qt = gnuplot.override { withQt = true; };

  # must have AquaTerm installed separately
  gnuplot_aquaterm = gnuplot.override { aquaterm = true; };

  gnu-pw-mgr = callPackage ../tools/security/gnu-pw-mgr { };

  gnused = callPackage ../tools/text/gnused { };

  gnutar = callPackage ../tools/archivers/gnutar { };

  goaccess = callPackage ../tools/misc/goaccess { };

  gocryptfs = callPackage ../tools/filesystems/gocryptfs { };

  godot_4 = callPackage ../development/tools/godot/4 { };

  godot3 = callPackage ../development/tools/godot/3 { };

  godot3-export-templates = callPackage ../development/tools/godot/3/export-templates.nix { };

  godot3-headless = callPackage ../development/tools/godot/3/headless.nix { };

  godot3-debug-server = callPackage ../development/tools/godot/3/debug-server.nix { };

  godot3-server = callPackage ../development/tools/godot/3/server.nix { };

  godot3-mono = callPackage ../development/tools/godot/3/mono {};

  godot3-mono-export-templates = callPackage ../development/tools/godot/3/mono/export-templates.nix { };

  godot3-mono-headless = callPackage ../development/tools/godot/3/mono/headless.nix { };

  godot3-mono-debug-server = callPackage ../development/tools/godot/3/mono/debug-server.nix { };

  godot3-mono-server = callPackage ../development/tools/godot/3/mono/server.nix { };

  goeland = callPackage ../applications/networking/feedreaders/goeland { };

  go-mtpfs = callPackage ../tools/filesystems/go-mtpfs { };

  goofys = callPackage ../tools/filesystems/goofys { };

  go-sct = callPackage ../tools/X11/go-sct { };

  # rename to upower-notify?
  go-upower-notify = callPackage ../tools/misc/upower-notify { };

  goattracker = callPackage ../applications/audio/goattracker { };

  goattracker-stereo = callPackage ../applications/audio/goattracker {
    isStereo = true;
  };

  google-app-engine-go-sdk = callPackage ../development/tools/google-app-engine-go-sdk { };

  google-authenticator = callPackage ../os-specific/linux/google-authenticator { };

  google-cloud-sdk = callPackage ../tools/admin/google-cloud-sdk {
    python = python3;
  };
  google-cloud-sdk-gce = google-cloud-sdk.override {
    python = python3;
    with-gce = true;
  };

  google-cloud-bigtable-tool = callPackage ../tools/misc/google-cloud-bigtable-tool { };

  google-cloud-sql-proxy = callPackage ../tools/misc/google-cloud-sql-proxy { };

  google-fonts = callPackage ../data/fonts/google-fonts { };

  google-clasp = callPackage ../development/tools/google-clasp { };

  google-compute-engine = with python3.pkgs; toPythonApplication google-compute-engine;

  google-guest-oslogin = callPackage ../tools/virtualization/google-guest-oslogin { };

  google-cloud-cpp = callPackage ../development/libraries/google-cloud-cpp { };

  google-java-format = callPackage ../development/tools/google-java-format { };

  google-guest-agent = callPackage ../tools/virtualization/google-guest-agent { };

  google-guest-configs = callPackage ../tools/virtualization/google-guest-configs { };

  gdown = with python3Packages; toPythonApplication gdown;

  gopro = callPackage ../tools/video/gopro { };

  goreleaser = callPackage ../tools/misc/goreleaser { };

  goreplay = callPackage ../tools/networking/goreplay { };

  gost = callPackage ../tools/networking/gost { };

  gource = callPackage ../applications/version-management/gource { };

  govc = callPackage ../tools/virtualization/govc { };

  goverlay = callPackage ../tools/graphics/goverlay {
    inherit (qt5) wrapQtAppsHook;
    inherit (plasma5Packages) breeze-qt5;
  };

  gpart = callPackage ../tools/filesystems/gpart { };

  gparted = callPackage ../tools/misc/gparted { };

  gpt2tc = callPackage ../tools/text/gpt2tc { };

  gptcommit = callPackage ../development/tools/gptcommit {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  gptman = callPackage ../tools/system/gptman { };

  ldmtool = callPackage ../tools/misc/ldmtool { };

  gphotos-sync = callPackage ../tools/backup/gphotos-sync { };

  gpodder = callPackage ../applications/audio/gpodder { };

  gpp = callPackage ../development/tools/gpp { };

  gnuastro = callPackage ../applications/science/astronomy/gnuastro { };

  gpredict = callPackage ../applications/science/astronomy/gpredict {
    hamlib = hamlib_4;
  };

  gprof2dot = with python3Packages; toPythonApplication gprof2dot;

  gprojector = callPackage ../applications/science/astronomy/gprojector { };

  gptfdisk = callPackage ../tools/system/gptfdisk { };

  gql = callPackage ../applications/version-management/gql { };

  grafterm = callPackage ../tools/misc/grafterm { };

  gradience = callPackage ../applications/misc/gradience { };

  grafx2 = callPackage ../applications/graphics/grafx2 { };

  grails = callPackage ../development/web/grails { jdk = null; };

  graylog-5_1 = callPackage ../tools/misc/graylog/5.1.nix { };

  graylog-5_2 = callPackage ../tools/misc/graylog/5.2.nix { };

  graylog-6_0 = callPackage ../tools/misc/graylog/6.0.nix { };

  graylogPlugins = recurseIntoAttrs (
    callPackage ../tools/misc/graylog/plugins.nix { }
  );

  graphviz = callPackage ../tools/graphics/graphviz {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Foundation;
  };

  graphviz-nox = callPackage ../tools/graphics/graphviz {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Foundation;
    withXorg = false;
  };

  grass-sass = callPackage ../tools/misc/grass-sass { };

  gridtracker = callPackage ../applications/radio/gridtracker { };

  grig = callPackage ../applications/radio/grig { };

  grin = callPackage ../tools/text/grin { };

  gyb = callPackage ../tools/backup/gyb { };

  halftone = callPackage ../applications/graphics/halftone { };

  harminv = callPackage ../development/libraries/science/chemistry/harminv { };

  igrep = callPackage ../tools/text/igrep {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ripgrep = callPackage ../tools/text/ripgrep {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ripgrep-all = callPackage ../tools/text/ripgrep-all {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  grive2 = callPackage ../tools/filesystems/grive2 { };

  groff = callPackage ../tools/text/groff { };

  gromit-mpx = callPackage ../tools/graphics/gromit-mpx {
    gtk = gtk3;
    libappindicator = libappindicator-gtk3;
  };

  gron = callPackage ../development/tools/gron { };

  groonga = callPackage ../servers/search/groonga { };

  grpcurl = callPackage ../tools/networking/grpcurl { };

  grpc_cli = callPackage ../tools/networking/grpc_cli { };

  grpc-gateway = callPackage ../development/tools/grpc-gateway { };

  grpcui = callPackage ../tools/networking/grpcui { };

  grpc-tools = callPackage ../development/tools/misc/grpc-tools { };

  grpc-client-cli = callPackage ../development/tools/misc/grpc-client-cli { };

  grub2 = callPackage ../tools/misc/grub/default.nix { };

  grub2_efi = grub2.override {
    efiSupport = true;
  };

  grub2_light = grub2.override {
    zfsSupport = false;
  };

  grub2_xen = grub2.override {
    xenSupport = true;
  };

  grub2_pvgrub_image = callPackage ../tools/misc/grub/pvgrub_image { };

  grub4dos = callPackage ../tools/misc/grub4dos {
    stdenv = stdenv_32bit;
  };

  gruut = with python3.pkgs; toPythonApplication gruut;

  gruut-ipa = with python3.pkgs; toPythonApplication gruut-ipa;

  gx = callPackage ../tools/package-management/gx { };
  gx-go = callPackage ../tools/package-management/gx/go { };

  efitools = callPackage ../tools/security/efitools { };

  sbctl = callPackage ../tools/security/sbctl { };

  sbsigntool = callPackage ../tools/security/sbsigntool { };

  sonic-server = callPackage ../servers/search/sonic-server { };

  gsmartcontrol = callPackage ../tools/misc/gsmartcontrol { };

  gsmlib = callPackage ../development/libraries/gsmlib
    { autoreconfHook = buildPackages.autoreconfHook269; };

  gssdp = callPackage ../development/libraries/gssdp { };

  gssdp_1_6 = callPackage ../development/libraries/gssdp/1.6.nix { };

  gssdp-tools = callPackage ../development/libraries/gssdp/tools.nix { };

  gt5 = callPackage ../tools/system/gt5 { };

  gtest = callPackage ../development/libraries/gtest { };

  gbenchmark = callPackage ../development/libraries/gbenchmark { };

  gkraken = callPackage ../tools/system/gkraken { };

  gtkdatabox = callPackage ../development/libraries/gtkdatabox { };

  gtdialog = callPackage ../development/libraries/gtdialog { };

  gtkd = callPackage ../development/libraries/gtkd { dcompiler = ldc; };

  gtkgnutella = callPackage ../tools/networking/p2p/gtk-gnutella { };

  gtkperf = callPackage ../development/tools/misc/gtkperf { };

  gtkradiant = callPackage ../applications/misc/gtkradiant { };

  gtk-frdp = callPackage ../development/libraries/gtk-frdp { };

  gtk-vnc = callPackage ../tools/admin/gtk-vnc { };

  gup = callPackage ../development/tools/build-managers/gup { };

  gupnp = callPackage ../development/libraries/gupnp { };

  gupnp_1_6 = callPackage ../development/libraries/gupnp/1.6.nix { };

  gupnp-av = callPackage ../development/libraries/gupnp-av { };

  gupnp-dlna = callPackage ../development/libraries/gupnp-dlna { };

  gupnp-igd = callPackage ../development/libraries/gupnp-igd { };

  gupnp-tools = callPackage ../tools/networking/gupnp-tools { };

  gvm-tools = with python3.pkgs; toPythonApplication gvm-tools;

  gvpe = callPackage ../tools/networking/gvpe { };

  gvolicon = callPackage ../tools/audio/gvolicon { };

  gvproxy = callPackage ../tools/networking/gvproxy { };

  gyroflow = qt6Packages.callPackage ../applications/video/gyroflow {
    ffmpeg = ffmpeg_6;
  };

  gzip = callPackage ../tools/compression/gzip { };

  gzrt = callPackage ../tools/compression/gzrt { };

  httplab = callPackage ../tools/networking/httplab { };

  lucky-cli = callPackage ../development/web/lucky-cli { };

  mac-fdisk = callPackage ../tools/system/mac-fdisk { };

  partclone = callPackage ../tools/backup/partclone { };

  partimage = callPackage ../tools/backup/partimage { };

  pdisk = callPackage ../tools/system/pdisk {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation IOKit;
  };

  pgf_graphics = callPackage ../tools/graphics/pgf { };

  pgbackrest = callPackage ../tools/backup/pgbackrest { };

  pgformatter = callPackage ../development/tools/pgformatter { };

  pgloader = callPackage ../development/tools/pgloader { };

  pgtop = callPackage ../development/tools/pgtop { };

  pigz = callPackage ../tools/compression/pigz { };

  pixz = callPackage ../tools/compression/pixz { };

  plog = callPackage ../development/libraries/plog {};

  plplot = callPackage ../development/libraries/plplot {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  pxattr = callPackage ../tools/archivers/pxattr { };

  pxz = callPackage ../tools/compression/pxz { };

  haguichi = callPackage ../tools/networking/haguichi { };

  hans = callPackage ../tools/networking/hans { };

  h2 = callPackage ../servers/h2 { };

  h5utils = callPackage ../tools/misc/h5utils {
    libmatheval = null;
    hdf4 = null;
  };

  haproxy = callPackage ../tools/networking/haproxy { };

  hackertyper = callPackage ../tools/misc/hackertyper { };

  hackneyed = callPackage ../data/icons/hackneyed { };

  haveged = callPackage ../tools/security/haveged { };

  habitat = callPackage ../applications/networking/cluster/habitat { };

  hashcash = callPackage ../tools/security/hashcash { };

  hashcat = callPackage ../tools/security/hashcat {
    inherit (darwin.apple_sdk.frameworks) Foundation IOKit Metal OpenCL;
  };

  hashcat-utils = callPackage ../tools/security/hashcat-utils { };

  hashrat = callPackage ../tools/security/hashrat { };

  hash_extender = callPackage ../tools/security/hash_extender { };

  hash-identifier = callPackage ../tools/security/hash-identifier { };

  hash-slinger = callPackage ../tools/security/hash-slinger { };

  haskell-language-server = callPackage ../development/tools/haskell/haskell-language-server/withWrapper.nix { };

  hassil = with python3Packages; toPythonApplication hassil;

  hasmail = callPackage ../applications/networking/mailreaders/hasmail { };

  haste-client = callPackage ../tools/misc/haste-client { };

  haste-server = callPackage ../servers/haste-server { };

  hal-hardware-analyzer = libsForQt5.callPackage ../applications/science/electronics/hal-hardware-analyzer {
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  half = callPackage ../development/libraries/half { };

  halibut = callPackage ../tools/typesetting/halibut { };

  halide = callPackage ../development/compilers/halide {
    llvmPackages = llvmPackages_16;
  };

  hareThirdParty = recurseIntoAttrs (callPackage ./hare-third-party.nix { });

  ham = pkgs.perlPackages.ham;

  hamsket = callPackage ../applications/networking/instant-messengers/hamsket { };

  hardinfo = callPackage ../tools/system/hardinfo { };

  harmonia = callPackage ../tools/package-management/harmonia { };

  hayagriva = callPackage ../tools/typesetting/hayagriva { };

  hcl2json = callPackage ../applications/misc/hcl2json { };

  hcledit = callPackage ../tools/text/hcledit { };

  hcxtools = callPackage ../tools/security/hcxtools { };

  hcxdumptool = callPackage ../tools/security/hcxdumptool { };

  hdapsd = callPackage ../os-specific/linux/hdapsd { };

  hdaps-gl = callPackage ../tools/misc/hdaps-gl { };

  hddfancontrol = callPackage ../tools/system/hddfancontrol { };

  hddtemp = callPackage ../tools/misc/hddtemp { };

  hdf4 = callPackage ../tools/misc/hdf4 { };

  hdf5 = callPackage ../tools/misc/hdf5 {
    fortranSupport = false;
    fortran = gfortran;
  };

  hdf5_1_10 = callPackage ../tools/misc/hdf5/1.10.nix { };

  hdf5-mpi = hdf5.override {
    mpiSupport = true;
    cppSupport = false;
  };

  hdf5-cpp = hdf5.override { cppSupport = true; };

  hdf5-fortran = hdf5.override { fortranSupport = true; };

  hdf5-threadsafe = hdf5.override { threadsafe = true; };

  hdf5-blosc = callPackage ../development/libraries/hdf5-blosc { };

  hdfview = callPackage ../tools/misc/hdfview { };

  hecate = callPackage ../applications/editors/hecate { };

  heaptrack = libsForQt5.callPackage ../development/tools/profiling/heaptrack { };

  heatshrink = callPackage ../tools/compression/heatshrink { };

  heimdall = libsForQt5.callPackage ../tools/misc/heimdall { };

  heimdall-gui = heimdall.override { enableGUI = true; };

  headscale = callPackage ../servers/headscale { };

  health = callPackage ../applications/misc/health { };

  healthchecks = callPackage ../servers/web-apps/healthchecks { };

  heisenbridge = callPackage ../servers/heisenbridge { };

  helio-workstation = callPackage ../applications/audio/helio-workstation { };

  hevea = callPackage ../tools/typesetting/hevea { };

  hex = callPackage ../tools/misc/hex { };

  hexd = callPackage ../tools/misc/hexd { };
  pixd = callPackage ../tools/misc/pixd { };

  hexgui = callPackage ../games/hexgui { };

  hey = callPackage ../tools/networking/hey { };

  heygpt = callPackage ../tools/llm/heygpt { };

  hhpc = callPackage ../tools/misc/hhpc { };

  hiera-eyaml = callPackage ../tools/system/hiera-eyaml { };

  hivemind = callPackage ../applications/misc/hivemind { };

  hobbits = libsForQt5.callPackage ../tools/graphics/hobbits { };

  hfsprogs = callPackage ../tools/filesystems/hfsprogs { };

  highlight = callPackage ../tools/text/highlight ({
    lua = lua5;
  });

  hockeypuck = callPackage ../servers/hockeypuck/server.nix { };

  hockeypuck-web = callPackage ../servers/hockeypuck/web.nix { };

  holochain-launcher = callPackage ../applications/misc/holochain-launcher { };

  homesick = callPackage ../tools/misc/homesick { };

  honcho = callPackage ../tools/system/honcho { };

  horst = callPackage ../tools/networking/horst { };

  host = bind.host;

  hostess = callPackage ../development/tools/hostess { };

  hostname-debian = callPackage ../tools/networking/hostname-debian { };

  hotdoc = python3Packages.callPackage ../development/tools/hotdoc { };

  hotpatch = callPackage ../development/libraries/hotpatch { };

  hotspot = libsForQt5.callPackage ../development/tools/analysis/hotspot { };

  hpccm = with python3Packages; toPythonApplication hpccm;

  hping = callPackage ../tools/networking/hping { };

  hqplayer-desktop = libsForQt5.callPackage ../applications/audio/hqplayer-desktop { };

  html-proofer = callPackage ../tools/misc/html-proofer { };

  htmlq = callPackage ../development/tools/htmlq {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  htpdate = callPackage ../tools/networking/htpdate { };

  http-prompt = callPackage ../tools/networking/http-prompt { };

  http-getter = callPackage ../applications/networking/flent/http-getter.nix { };

  httpdump = callPackage ../tools/security/httpdump { };

  httpie = with python3Packages; toPythonApplication httpie;

  httping = callPackage ../tools/networking/httping { };

  httplz = callPackage ../tools/networking/httplz { };

  httpfs2 = callPackage ../tools/filesystems/httpfs { };

  httprobe = callPackage ../tools/networking/httprobe { };

  httpstat = callPackage ../tools/networking/httpstat { };

  httptunnel = callPackage ../tools/networking/httptunnel { };

  httpx = callPackage ../tools/security/httpx { };

  hue-plus = libsForQt5.callPackage ../applications/misc/hue-plus { };

  hurl = callPackage ../tools/networking/hurl { };

  hubicfuse = callPackage ../tools/filesystems/hubicfuse { };

  humanfriendly = with python3Packages; toPythonApplication humanfriendly;

  hwinfo = callPackage ../tools/system/hwinfo { };

  hw-probe = perlPackages.callPackage ../tools/system/hw-probe { };

  hybridreverb2 = callPackage ../applications/audio/hybridreverb2 { };

  hylafaxplus = callPackage ../servers/hylafaxplus {
    # needs this fork of libtiff, because original libtiff
    # stopped packaging required tools with version 4.6
    libtiff = libtiff_t;
  };

  hyphen = callPackage ../development/libraries/hyphen { };

  i2c-tools = callPackage ../os-specific/linux/i2c-tools { };

  i2pd = callPackage ../tools/networking/i2pd { };

  iannix = libsForQt5.callPackage ../applications/audio/iannix { };

  iaito = libsForQt5.callPackage ../tools/security/iaito { };

  jamulus = libsForQt5.callPackage ../applications/audio/jamulus { };

  ultrablue-server = callPackage ../os-specific/linux/ultrablue-server { };

  ibm-sw-tpm2 = callPackage ../tools/security/ibm-sw-tpm2 { };

  ibniz = callPackage ../tools/graphics/ibniz { };

  icebreaker = callPackage ../servers/icebreaker { };

  icecast = callPackage ../servers/icecast { };

  icemon = libsForQt5.callPackage ../applications/networking/icemon { };

  icepeak = haskell.lib.compose.justStaticExecutables haskellPackages.icepeak;

  iceshelf = callPackage ../tools/backup/iceshelf { };

  darkice = callPackage ../tools/audio/darkice { };

  bc-decaf = callPackage ../development/libraries/bc-decaf { };

  deckmaster = callPackage ../applications/misc/deckmaster { };

  deco = callPackage ../applications/misc/deco { };

  decoder = callPackage ../tools/security/decoder { };

  icoutils = callPackage ../tools/graphics/icoutils { };

  idutils = callPackage ../tools/misc/idutils { };

  idle3tools = callPackage ../tools/system/idle3tools { };

  ifcopenshell = with python3Packages; toPythonApplication ifcopenshell;

  iftop = callPackage ../tools/networking/iftop { };

  ifwifi = callPackage ../tools/networking/ifwifi {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ifuse = callPackage ../tools/filesystems/ifuse { };
  ideviceinstaller = callPackage ../tools/misc/ideviceinstaller { };
  idevicerestore = callPackage ../tools/misc/idevicerestore { };

  inherit (callPackages ../tools/filesystems/irods rec {
    stdenv = llvmPackages_13.libcxxStdenv;
    libcxx = llvmPackages_13.libcxx;
    boost = boost178.override { inherit stdenv; };
    fmt = fmt_8.override { inherit stdenv; };
    nanodbc_llvm = nanodbc.override { inherit stdenv; };
    avro-cpp_llvm = avro-cpp.override { inherit stdenv boost; };
  })
    irods
    irods-icommands;

  igmpproxy = callPackage ../tools/networking/igmpproxy { };

  ihaskell = callPackage ../development/tools/haskell/ihaskell/wrapper.nix {
    inherit (haskellPackages) ghcWithPackages;

    jupyter = python3.withPackages (ps: [ ps.jupyter ps.notebook ]);

    packages = config.ihaskell.packages or (_: []);
  };

  ijq = callPackage ../development/tools/ijq { };

  iruby = callPackage ../applications/editors/jupyter-kernels/iruby { };

  ike-scan = callPackage ../tools/security/ike-scan { };

  ilspycmd = callPackage ../development/tools/ilspycmd {
    inherit (darwin) autoSignDarwinBinariesHook;
  };

  imapsync = callPackage ../tools/networking/imapsync { };

  imgur-screenshot = callPackage ../tools/graphics/imgur-screenshot { };

  imgurbash2 = callPackage ../tools/graphics/imgurbash2 { };

  in-formant = qt6Packages.callPackage ../applications/audio/in-formant { };

  inadyn = callPackage ../tools/networking/inadyn { };

  incus-lts = callPackage ../by-name/in/incus/lts.nix { };

  incron = callPackage ../tools/system/incron { };

  indexed-bzip2 = with python3Packages; toPythonApplication indexed-bzip2;

  industrializer = callPackage ../applications/audio/industrializer { };

  inetutils = callPackage ../tools/networking/inetutils { };

  inferno = callPackage ../development/tools/inferno { };

  infisical = callPackage ../development/tools/infisical { };

  inform6 = darwin.apple_sdk_11_0.callPackage ../development/compilers/inform6 { };

  inform7 = callPackage ../development/compilers/inform7 { };

  infamousPlugins = callPackage ../applications/audio/infamousPlugins { };

  innernet = callPackage ../tools/networking/innernet {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  innoextract = callPackage ../tools/archivers/innoextract { };

  input-remapper = python3Packages.callPackage ../tools/inputmethods/input-remapper { };

  input-utils = callPackage ../os-specific/linux/input-utils { };

  inql = callPackage ../tools/security/inql { };

  intel-media-sdk = callPackage ../development/libraries/intel-media-sdk { };

  intermodal = callPackage ../tools/misc/intermodal { };

  internetarchive = with python3Packages; toPythonApplication internetarchive;

  invidious = callPackage ../servers/invidious { };

  invoice2data  = callPackage ../tools/text/invoice2data  { };

  inxi = callPackage ../tools/system/inxi { };

  iodine = callPackage ../tools/networking/iodine { };

  ioc-scan = callPackage ../tools/security/ioc-scan { };

  ioccheck = callPackage ../tools/security/ioccheck { };

  iocextract = with python3Packages; toPythonApplication iocextract;

  iocsearcher = with python3Packages; toPythonApplication iocsearcher;

  ioping = callPackage ../tools/system/ioping { };

  ior = callPackage ../tools/system/ior { };

  ioztat = callPackage ../tools/filesystems/ioztat { };

  ip2location = callPackage ../tools/networking/ip2location { };

  ip2unix = callPackage ../tools/networking/ip2unix { };

  ipad_charge = callPackage ../tools/misc/ipad_charge { };

  iperf2 = callPackage ../tools/networking/iperf/2.nix { };
  iperf3 = callPackage ../tools/networking/iperf/3.nix { };
  iperf = iperf3;

  iperf3d = callPackage ../tools/networking/iperf3d { };

  ipfetch = callPackage ../tools/networking/ipfetch { };

  ipfs-cluster = callPackage ../applications/networking/ipfs-cluster { };

  ipfs-upload-client = callPackage ../applications/networking/ipfs-upload-client { };

  ipget = callPackage ../applications/networking/ipget { };

  i-pi = with python3Packages; toPythonApplication i-pi;

  iptsd = callPackage ../applications/misc/iptsd { };

  ipmitool = callPackage ../tools/system/ipmitool { };

  ipmiutil = callPackage ../tools/system/ipmiutil { };

  ipatool = callPackage ../applications/misc/ipatool { };

  ipmicfg = callPackage ../applications/misc/ipmicfg { };

  ipmiview = callPackage ../applications/misc/ipmiview { };

  ipcalc = callPackage ../tools/networking/ipcalc { };

  netmask = callPackage ../tools/networking/netmask { };

  netifd = callPackage ../tools/networking/netifd { };

  ipinfo = callPackage ../tools/networking/ipinfo { };

  ipscan = callPackage ../tools/security/ipscan { };
  # ipscan is commonly known under the name angryipscanner
  angryipscanner = ipscan;

  ipv6calc = callPackage ../tools/networking/ipv6calc { };

  ipxe = callPackage ../tools/misc/ipxe { };

  irker = callPackage ../servers/irker { };

  iroh = callPackage ../applications/networking/iroh { };

  ised = callPackage ../tools/misc/ised { };

  isl = isl_0_20;
  isl_0_11 = callPackage ../development/libraries/isl/0.11.1.nix { };
  isl_0_14 = callPackage ../development/libraries/isl/0.14.1.nix { };
  isl_0_17 = callPackage ../development/libraries/isl/0.17.1.nix { };
  isl_0_20 = callPackage ../development/libraries/isl/0.20.0.nix { };
  isl_0_24 = callPackage ../development/libraries/isl/0.24.0.nix { };

  ispike = callPackage ../development/libraries/science/robotics/ispike { };

  isrcsubmit = callPackage ../tools/audio/isrcsubmit { };

  isync = callPackage ../tools/networking/isync {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  itm-tools = callPackage ../development/tools/misc/itm-tools { };

  ite-backlight = callPackage ../misc/ite-backlight { };

  iwgtk = callPackage ../tools/networking/iwgtk { };

  ix = callPackage ../tools/misc/ix { };

  jaaa = callPackage ../applications/audio/jaaa { };

  jackett = callPackage ../servers/jackett { };

  jade = callPackage ../tools/text/sgml/jade { };

  jadx = callPackage ../tools/security/jadx { };

  jamesdsp = libsForQt5.callPackage ../applications/audio/jamesdsp { };
  jamesdsp-pulse = libsForQt5.callPackage ../applications/audio/jamesdsp {
    usePipewire = false;
    usePulseaudio = true;
  };

  jaq = callPackage ../development/tools/jaq { };

  jasmin-compiler = callPackage ../development/compilers/jasmin-compiler { };

  jazzy = callPackage ../development/tools/jazzy { };

  jc = with python3Packages; toPythonApplication jc;

  jcli = callPackage ../development/tools/misc/jcli { };

  jd-cli = callPackage ../tools/security/jd-cli { };

  jd-diff-patch = callPackage ../development/tools/jd-diff-patch { };

  jd-gui = callPackage ../tools/security/jd-gui { };

  jdiskreport = callPackage ../tools/misc/jdiskreport { };

  jekyll = callPackage ../applications/misc/jekyll { };

  jello = with python3Packages; toPythonApplication jello;

  jen = callPackage ../tools/text/jen { };

  jet = callPackage ../development/tools/jet { };

  jf = callPackage ../development/tools/jf { };

  jfmt = callPackage ../development/tools/jfmt { };

  jfsutils = callPackage ../tools/filesystems/jfsutils { };

  jhead = callPackage ../tools/graphics/jhead { };

  jid = callPackage ../development/tools/jid { };

  jing = res.jing-trang;
  jing-trang = callPackage ../tools/text/xml/jing-trang {
    jdk_headless = jdk8_headless; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  jira-cli-go = callPackage ../development/tools/jira-cli-go { };

  jirafeau = callPackage ../servers/web-apps/jirafeau { };

  jitterentropy = callPackage ../development/libraries/jitterentropy { };

  jl = haskellPackages.jl;

  jless = callPackage ../development/tools/jless {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  jmespath = callPackage ../development/tools/jmespath { };

  juicefs = callPackage ../tools/filesystems/juicefs { };

  juicity = callPackage ../tools/networking/juicity { };

  jmtpfs = callPackage ../tools/filesystems/jmtpfs { };

  jnettop = callPackage ../tools/networking/jnettop { };

  jsvc = callPackage ../tools/system/jsvc { };

  junkie = callPackage ../tools/networking/junkie { };

  go-jira = callPackage ../applications/misc/go-jira { };

  jogl = callPackage ../by-name/jo/jogl/package.nix {
    stdenv = if stdenv.isDarwin && stdenv.isx86_64 then overrideSDK stdenv "11.0" else stdenv;
  };

  john = callPackage ../tools/security/john { };

  joomscan = callPackage ../tools/security/joomscan { };

  joplin = nodePackages.joplin;

  joplin-desktop = callPackage ../applications/misc/joplin-desktop { };

  jot = callPackage ../applications/misc/jot { };

  jotdown = callPackage ../tools/text/jotdown { };

  journaldriver = callPackage ../tools/misc/journaldriver { };

  jp = callPackage ../development/tools/jp { };

  jp2a = callPackage ../applications/misc/jp2a { };

  jpeg-archive = callPackage ../applications/graphics/jpeg-archive { };

  jpegexiforient = callPackage ../tools/graphics/jpegexiforient { };

  jpeginfo = callPackage ../applications/graphics/jpeginfo { };

  jpegoptim = callPackage ../applications/graphics/jpegoptim { };

  jpegrescan = callPackage ../applications/graphics/jpegrescan { };

  jpylyzer = with python3Packages; toPythonApplication jpylyzer;

  jq = callPackage ../development/tools/jq { };

  jiq = callPackage ../development/tools/misc/jiq { };

  jql = callPackage ../development/tools/jql { };

  jqp = callPackage ../development/tools/jqp { };

  jo = callPackage ../development/tools/jo { };

  jrnl = callPackage ../applications/misc/jrnl { };

  jsawk = callPackage ../tools/text/jsawk { };

  jsbeautifier = with python3Packages; toPythonApplication jsbeautifier;

  jscoverage = callPackage ../development/tools/misc/jscoverage { };

  jsduck = callPackage ../development/tools/jsduck { };

  jsluice = callPackage ../tools/security/jsluice { };

  json-schema-for-humans = with python3Packages; toPythonApplication json-schema-for-humans;

  jsonfmt = callPackage ../development/tools/misc/jsonfmt { };

  jsonwatch = callPackage ../tools/misc/jsonwatch {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  jsubfinder = callPackage ../tools/security/jsubfinder { };

  jtag-remote-server = callPackage ../development/embedded/jtag-remote-server { };

  jtc = callPackage ../development/tools/jtc { };

  jumpapp = callPackage ../tools/X11/jumpapp { };

  jove = callPackage ../applications/editors/jove { };

  jugglinglab = callPackage ../tools/misc/jugglinglab { };

  jupp = callPackage ../applications/editors/jupp { };

  jupyter = callPackage ../applications/editors/jupyter { };

  jupyter-all = jupyter.override {
    definitions = {
      clojure = clojupyter.definition;
      octave = octave-kernel.definition;
      # wolfram = wolfram-for-jupyter-kernel.definition; # unfree
    };
  };

  jupyter-console = callPackage ../applications/editors/jupyter/console.nix { };

  jupyter-kernel = callPackage ../applications/editors/jupyter/kernel.nix { };

  justify = callPackage ../tools/text/justify { };

  jwhois = callPackage ../tools/networking/jwhois { };

  k2pdfopt = callPackage ../applications/misc/k2pdfopt { };

  kargo = callPackage ../tools/misc/kargo { };

  kazam = callPackage ../applications/video/kazam { };

  kalibrate-rtl = callPackage ../applications/radio/kalibrate-rtl { };

  kalibrate-hackrf = callPackage ../applications/radio/kalibrate-hackrf { };

  wrapKakoune = kakoune: attrs: callPackage ../applications/editors/kakoune/wrapper.nix (attrs // { inherit kakoune; });
  kakounePlugins = recurseIntoAttrs (callPackage ../applications/editors/kakoune/plugins { });

  kakoune-unwrapped = callPackage ../applications/editors/kakoune { };
  kakoune = wrapKakoune kakoune-unwrapped {
    plugins = [ ];  # override with the list of desired plugins
  };
  kakouneUtils = callPackage ../applications/editors/kakoune/plugins/kakoune-utils.nix { };

  kaffeine = libsForQt5.callPackage ../applications/video/kaffeine { };

  kakoune-lsp = callPackage ../by-name/ka/kakoune-lsp/package.nix {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security SystemConfiguration;
  };

  kakoune-cr = callPackage ../tools/misc/kakoune-cr { crystal = crystal_1_2; };

  kaniko = callPackage ../applications/networking/cluster/kaniko { };

  katana = callPackage ../tools/security/katana { };

  kbdd = callPackage ../applications/window-managers/kbdd { };

  kbs2 = callPackage ../tools/security/kbs2 {
    inherit (darwin.apple_sdk.frameworks) AppKit SystemConfiguration;
  };

  kdash = callPackage ../development/tools/kdash {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  kdoctor = callPackage ../development/tools/kdoctor { };

  kdigger = callPackage ../tools/security/kdigger { };

  kdiskmark = libsForQt5.callPackage ../tools/filesystems/kdiskmark { };

  keepalived = callPackage ../tools/networking/keepalived { };

  keepwn = callPackage ../tools/security/keepwn { };

  kestrel = callPackage ../tools/security/kestrel { };

  kexec-tools = callPackage ../os-specific/linux/kexec-tools {
    # clangStdenv fails with
    # purgatory/arch/i386/entry32-16.S:23:2: error: unknown directive
    #  .arch i386
    #  ^
    # purgatory/arch/i386/entry32-16.S:115:11: error: unknown token in expression
    #  ljmp %cs:*(realdest - entry16)
    #           ^
    # make: *** [Makefile:128: purgatory/arch/i386/entry32-16.o] Error 1
    stdenv = gccStdenv;
  };

  keepkey-agent = with python3Packages; toPythonApplication keepkey-agent;

  keepmenu = callPackage ../applications/misc/keepmenu { };

  kent = callPackage ../applications/science/biology/kent { };

  keybase = darwin.apple_sdk_11_0.callPackage ../tools/security/keybase {
    # Reasoning for the inherited apple_sdk.frameworks:
    # 1. specific compiler errors about: AVFoundation, AudioToolbox, MediaToolbox
    # 2. the rest are added from here: https://github.com/keybase/client/blob/68bb8c893c5214040d86ea36f2f86fbb7fac8d39/go/chat/attachments/preview_darwin.go#L7
    #      #cgo LDFLAGS: -framework AVFoundation -framework CoreFoundation -framework ImageIO -framework CoreMedia  -framework Foundation -framework CoreGraphics -lobjc
    #    with the exception of CoreFoundation, due to the warning in https://github.com/NixOS/nixpkgs/blob/master/pkgs/os-specific/darwin/apple-sdk/frameworks.nix#L25
    inherit (darwin.apple_sdk_11_0.frameworks) AVFoundation AudioToolbox ImageIO CoreMedia Foundation CoreGraphics MediaToolbox;
  };

  kbfs = callPackage ../tools/security/keybase/kbfs.nix { };

  keybase-gui = callPackage ../tools/security/keybase/gui.nix { };

  keychain = callPackage ../tools/misc/keychain { };

  keyfuzz = callPackage ../tools/inputmethods/keyfuzz { };

  keymapp = callPackage ../applications/misc/keymapp { };

  keyscope = callPackage ../tools/security/keyscope {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration Foundation IOKit Security;
  };

  keystore-explorer = callPackage ../applications/misc/keystore-explorer {
    jdk = jdk11;
  };

  kluctl = callPackage ../applications/networking/cluster/kluctl { };

  kibi = callPackage ../applications/editors/kibi { };

  kio-fuse = libsForQt5.callPackage ../tools/filesystems/kio-fuse { };

  kismet = callPackage ../applications/networking/sniffers/kismet { };

  kiterunner = callPackage ../tools/security/kiterunner { };

  klick = callPackage ../applications/audio/klick { };

  klystrack = callPackage ../applications/audio/klystrack { };

  knit = callPackage ../development/tools/build-managers/knit { };

  knockpy = callPackage ../tools/security/knockpy { };

  knowsmore = callPackage ../tools/security/knowsmore { };

  kool = callPackage ../development/tools/misc/kool { };

  kore = callPackage ../development/web/kore { };

  krakenx = callPackage ../tools/system/krakenx { };

  kpcli = callPackage ../tools/security/kpcli { };

  kphotoalbum = libsForQt5.callPackage ../applications/graphics/kphotoalbum { };

  kratos = callPackage ../applications/misc/kratos { };

  krename = libsForQt5.callPackage ../applications/misc/krename { };

  krunner-pass = libsForQt5.callPackage ../tools/security/krunner-pass { };

  krunner-translator = libsForQt5.callPackage ../tools/misc/krunner-translator { };

  krunvm = callPackage ../applications/virtualization/krunvm {
    inherit (darwin) sigtool;
  };

  kronometer = libsForQt5.callPackage ../tools/misc/kronometer { };

  krop = callPackage ../applications/graphics/krop { };

  kdiff3 = libsForQt5.callPackage ../tools/text/kdiff3 { };

  kube-router = callPackage ../applications/networking/cluster/kube-router { };

  kubedock = callPackage ../development/tools/kubedock { };

  kubepug = callPackage ../development/tools/kubepug { };

  kubeshark = callPackage ../applications/networking/cluster/kubeshark { };

  kubergrunt = callPackage ../applications/networking/cluster/kubergrunt { };

  kubo-migrator-all-fs-repo-migrations = callPackage ../applications/networking/kubo-migrator/all-migrations.nix { };
  kubo-migrator-unwrapped = callPackage ../applications/networking/kubo-migrator/unwrapped.nix { };
  kubo-migrator = callPackage ../applications/networking/kubo-migrator { };

  kwalletcli = libsForQt5.callPackage ../tools/security/kwalletcli { };

  peruse = libsForQt5.callPackage ../tools/misc/peruse { };

  ksmoothdock = libsForQt5.callPackage ../applications/misc/ksmoothdock { };

  kstars = libsForQt5.callPackage ../applications/science/astronomy/kstars { };

  kytea = callPackage ../tools/text/kytea { };

  kyverno = callPackage ../applications/networking/cluster/kyverno { };

  k6 = callPackage ../development/tools/k6 { };

  l2md = callPackage ../tools/text/l2md { };

  lalezar-fonts = callPackage ../data/fonts/lalezar-fonts { };

  lalrpop = callPackage ../development/tools/lalrpop { };

  last-resort = callPackage ../data/fonts/last-resort { };

  ldc = callPackage ../development/compilers/ldc { };

  ligo =
    let ocaml_p = ocaml-ng.ocamlPackages_4_14_janeStreet_0_15; in
    callPackage ../development/compilers/ligo {
    coq = coq_8_13.override {
      customOCamlPackages = ocaml_p;
    };
    ocamlPackages = ocaml_p;
  };

  lego = callPackage ../tools/admin/lego { };

  leocad = libsForQt5.callPackage ../applications/graphics/leocad { };

  lha = callPackage ../tools/archivers/lha { };

  lhasa = callPackage ../tools/compression/lhasa { };

  libcsptr = callPackage ../development/libraries/libcsptr { };

  libgovirt = callPackage ../applications/virtualization/libgovirt { };

  libscrypt = callPackage ../development/libraries/libscrypt { };

  libcloudproviders = callPackage ../development/libraries/libcloudproviders { };

  libcoap = callPackage ../applications/networking/libcoap {
    autoconf = buildPackages.autoconf269;
  };

  libcryptui = callPackage ../development/libraries/libcryptui {
    autoreconfHook = buildPackages.autoreconfHook269;
    gtk3 = if stdenv.isDarwin then gtk3-x11 else gtk3;
  };

  libshumate = callPackage ../development/libraries/libshumate { };

  libsmartcols = callPackage ../development/libraries/libsmartcols { };

  libsmi = callPackage ../development/libraries/libsmi { };

  libgen-cli = callPackage ../tools/misc/libgen-cli { };

  libpff = callPackage ../tools/misc/libpff {};

  licensor = callPackage ../tools/misc/licensor { };

  lesspipe = callPackage ../tools/misc/lesspipe { };

  liquidsoap = callPackage ../tools/audio/liquidsoap/full.nix {
    ffmpeg = ffmpeg-full;
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  linuxwave = callPackage ../tools/audio/linuxwave { };

  lksctp-tools = callPackage ../os-specific/linux/lksctp-tools { };

  lldpd = callPackage ../tools/networking/lldpd { };

  llm = with python3Packages; toPythonApplication llm;

  lnav = callPackage ../tools/misc/lnav { };

  lnch = callPackage ../tools/misc/lnch { };

  lnx = callPackage ../servers/search/lnx {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration Foundation;
  };

  loadlibrary = callPackage ../tools/misc/loadlibrary { };

  loc = callPackage ../development/misc/loc { };

  lockfileProgs = callPackage ../tools/misc/lockfile-progs { };

  loganalyzer = libsForQt5.callPackage ../development/tools/loganalyzer { };

  logstash7 = callPackage ../tools/misc/logstash/7.x.nix {
    # https://www.elastic.co/support/matrix#logstash-and-jvm
    jre = jdk11_headless;
  };
  logstash7-oss = callPackage ../tools/misc/logstash/7.x.nix {
    enableUnfree = false;
    # https://www.elastic.co/support/matrix#logstash-and-jvm
    jre = jdk11_headless;
  };
  logstash = logstash7;

  logstash-contrib = callPackage ../tools/misc/logstash/contrib.nix { };

  lolcat = callPackage ../tools/misc/lolcat { };

  lottieconverter = callPackage ../tools/misc/lottieconverter { };

  loudgain = callPackage ../tools/audio/loudgain/default.nix { };

  lpcnetfreedv = callPackage ../development/libraries/lpcnetfreedv { };

  lsd = callPackage ../tools/misc/lsd { };

  lsdvd = callPackage ../tools/cd-dvd/lsdvd { };

  lsyncd = callPackage ../applications/networking/sync/lsyncd {
    inherit (darwin) xnu;
    lua = lua5_2_compat;
  };

  lttoolbox = callPackage ../applications/misc/lttoolbox { };

  ltwheelconf = callPackage ../applications/misc/ltwheelconf { };

  lunatask = callPackage ../applications/misc/lunatask { };

  lvmsync = callPackage ../tools/backup/lvmsync { };

  kapp = callPackage ../tools/networking/kapp { };

  kdbg = libsForQt5.callPackage ../development/tools/misc/kdbg { };

  kristall = libsForQt5.callPackage ../applications/networking/browsers/kristall { };

  lagrange = callPackage ../applications/networking/browsers/lagrange {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };
  lagrange-tui = lagrange.override { enableTUI = true; };

  kzipmix = pkgsi686Linux.callPackage ../tools/compression/kzipmix { };

  ma1sd = callPackage ../servers/ma1sd { };

  mailcatcher = callPackage ../development/web/mailcatcher { };

  makebootfat = callPackage ../tools/misc/makebootfat { };

  mani = callPackage ../development/tools/mani { };

  manifest-tool = callPackage ../development/tools/manifest-tool { };

  mask = callPackage ../development/tools/mask { };

  maskromtool = qt6Packages.callPackage ../tools/graphics/maskromtool { };

  mastotool = callPackage ../tools/misc/mastotool { };

  mathpix-snipping-tool = callPackage ../tools/misc/mathpix-snipping-tool { };

  matrix-conduit = callPackage ../servers/matrix-conduit { };

  matrix-sliding-sync = callPackage ../servers/matrix-synapse/sliding-sync { };

  matrix-synapse = callPackage ../servers/matrix-synapse/wrapper.nix { };
  matrix-synapse-unwrapped = callPackage ../servers/matrix-synapse/default.nix { };
  matrix-synapse-plugins = recurseIntoAttrs matrix-synapse-unwrapped.plugins;
  matrix-synapse-tools = recurseIntoAttrs matrix-synapse-unwrapped.tools;

  matrix-appservice-irc = callPackage ../servers/matrix-synapse/matrix-appservice-irc { };

  matrix-appservice-slack = callPackage ../servers/matrix-synapse/matrix-appservice-slack {
    matrix-sdk-crypto-nodejs = matrix-sdk-crypto-nodejs-0_1_0-beta_3;
    nodejs = nodejs_18;
  };

  matrix-appservice-discord = callPackage ../servers/matrix-appservice-discord {
    matrix-sdk-crypto-nodejs = matrix-sdk-crypto-nodejs-0_1_0-beta_3;
  };

  matrix-corporal = callPackage ../servers/matrix-corporal { };

  matrix-hookshot = callPackage ../servers/matrix-synapse/matrix-hookshot { };

  maubot = with python3Packages; toPythonApplication maubot;

  mautrix-discord = callPackage ../servers/mautrix-discord { };

  mautrix-facebook = callPackage ../servers/mautrix-facebook { };

  mautrix-googlechat = callPackage ../servers/mautrix-googlechat { };

  mautrix-signal = recurseIntoAttrs (callPackage ../servers/mautrix-signal { });

  mautrix-telegram = recurseIntoAttrs (callPackage ../servers/mautrix-telegram { });

  mautrix-whatsapp = callPackage ../servers/mautrix-whatsapp { };

  mcfly = callPackage ../tools/misc/mcfly { };

  m2r = with python3Packages; toPythonApplication m2r;

  md2gemini = with python3.pkgs; toPythonApplication md2gemini;

  mdbook = callPackage ../tools/text/mdbook {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-emojicodes = callPackage ../tools/text/mdbook-emojicodes { };

  mdbook-epub = callPackage ../tools/text/mdbook-epub {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-cmdrun = callPackage ../tools/text/mdbook-cmdrun { };

  mdbook-d2 = callPackage ../tools/text/mdbook-d2 { };

  mdbook-pagetoc = callPackage ../tools/text/mdbook-pagetoc { };

  mdbook-graphviz = callPackage ../tools/text/mdbook-graphviz {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-i18n-helpers = callPackage ../tools/text/mdbook-i18n-helpers { };

  mdbook-katex = callPackage ../tools/text/mdbook-katex {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-kroki-preprocessor = callPackage ../tools/text/mdbook-kroki-preprocessor { };

  mdbook-linkcheck = callPackage ../tools/text/mdbook-linkcheck {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  mdbook-open-on-gh = callPackage ../tools/text/mdbook-open-on-gh { };

  mdbook-man = callPackage ../tools/text/mdbook-man { };

  mdbook-mermaid = callPackage ../tools/text/mdbook-mermaid {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-pdf = callPackage ../tools/text/mdbook-pdf {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-plantuml = callPackage ../tools/text/mdbook-plantuml {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-toc = callPackage ../tools/text/mdbook-toc {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-admonish = callPackage ../tools/text/mdbook-admonish {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdbook-footnote = callPackage ../tools/text/mdbook-footnote {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  mdcat = callPackage ../tools/text/mdcat {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
    inherit (python3Packages) ansi2html;
  };

  medfile = callPackage ../development/libraries/medfile {
    hdf5 = hdf5.override { usev110Api = true; };
  };

  meilisearch = callPackage ../servers/search/meilisearch {
    inherit (darwin.apple_sdk_11_0.frameworks) Security SystemConfiguration;
  };

  memtester = callPackage ../tools/system/memtester { };

  mesa-demos = callPackage ../tools/graphics/mesa-demos { };

  mhonarc = perlPackages.MHonArc;

  minica = callPackage ../tools/security/minica { };

  minidlna = callPackage ../tools/networking/minidlna { };

  miniplayer = callPackage ../applications/audio/miniplayer { };

  minipro = callPackage ../tools/misc/minipro { };

  minisign = callPackage ../tools/security/minisign { };

  ministat = callPackage ../tools/misc/ministat { };

  minizign = callPackage ../tools/security/minizign { };

  mitm6 = callPackage ../tools/security/mitm6 { };

  mjolnir = callPackage ../servers/mjolnir {
    matrix-sdk-crypto-nodejs = matrix-sdk-crypto-nodejs-0_1_0-beta_3;
  };

  mmutils = callPackage ../tools/X11/mmutils { };

  mmv = callPackage ../tools/misc/mmv { };

  mmv-go = callPackage ../tools/misc/mmv-go { };

  mob = callPackage ../applications/misc/mob { };

  most = callPackage ../tools/misc/most { };

  motion = callPackage ../applications/video/motion { };

  mozphab = callPackage ../applications/misc/mozphab { };

  mtail = callPackage ../servers/monitoring/mtail { };

  mujmap = callPackage ../applications/networking/mujmap {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  multitail = callPackage ../tools/misc/multitail { };

  mvebu64boot = callPackage ../tools/misc/mvebu64boot { };

  mx-puppet-discord = callPackage ../servers/mx-puppet-discord { };

  mx-takeover = callPackage ../tools/security/mx-takeover { };

  mxt-app = callPackage ../misc/mxt-app { };

  naabu = callPackage ../tools/security/naabu { };

  nagstamon = callPackage ../tools/misc/nagstamon {
    pythonPackages = python3Packages;
  };

  nanoemoji = python3Packages.callPackage ../tools/misc/nanoemoji { };

  nagelfar = callPackage ../development/tools/nagelfar { };

  nats-top = callPackage ../tools/system/nats-top { };

  natscli = callPackage ../tools/system/natscli { };

  nsc = callPackage ../tools/system/nsc { };

  nbench = callPackage ../tools/misc/nbench { };

  nbtscanner = callPackage ../tools/security/nbtscanner {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  nbutools = callPackage ../tools/security/nbutools { };

  ncrack = callPackage ../tools/security/ncrack { };

  netexec = python3Packages.callPackage ../tools/security/netexec { };

  nerdctl = callPackage ../applications/networking/cluster/nerdctl { };

  netdata = callPackage ../tools/system/netdata {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation IOKit;
    protobuf = protobuf_21;
  };
  netdataCloud = netdata.override {
    withCloud = !stdenv.isDarwin;
    withCloudUi = true;
  };
  # Exposed here so the bots can auto-upgrade it
  netdata-go-plugins = callPackage ../tools/system/netdata/go.d.plugin.nix { };

  netsurf = recurseIntoAttrs (callPackage ../applications/networking/browsers/netsurf { });
  netsurf-browser = netsurf.browser;

  netperf = callPackage ../applications/networking/netperf { };

  netsniff-ng = callPackage ../tools/networking/netsniff-ng { };

  networkminer = callPackage ../tools/security/networkminer { };

  nixpacks = callPackage ../applications/virtualization/nixpacks { };

  nkeys = callPackage ../tools/system/nkeys { };

  nyxt = callPackage ../applications/networking/browsers/nyxt { };

  nfpm = callPackage ../tools/package-management/nfpm { };

  nginx-config-formatter = callPackage ../tools/misc/nginx-config-formatter { };

  ninka = callPackage ../development/tools/misc/ninka { };

  nixnote2 = libsForQt5.callPackage ../applications/misc/nixnote2 { };

  nodenv = callPackage ../development/tools/nodenv { };

  nodejs = hiPrio nodejs_20;
  nodejs-slim = nodejs-slim_20;
  corepack = hiPrio corepack_20;

  nodejs_18 = callPackage ../development/web/nodejs/v18.nix { };
  nodejs-slim_18 = callPackage ../development/web/nodejs/v18.nix { enableNpm = false; };
  corepack_18 = hiPrio (callPackage ../development/web/nodejs/corepack.nix { nodejs = nodejs_18; });

  nodejs_20 = callPackage ../development/web/nodejs/v20.nix { };
  nodejs-slim_20 = callPackage ../development/web/nodejs/v20.nix { enableNpm = false; };
  corepack_20 = hiPrio (callPackage ../development/web/nodejs/corepack.nix { nodejs = nodejs_20; });

  nodejs_22 = callPackage ../development/web/nodejs/v22.nix { };
  nodejs-slim_22 = callPackage ../development/web/nodejs/v22.nix { enableNpm = false; };
  corepack_22 = hiPrio (callPackage ../development/web/nodejs/corepack.nix { nodejs = nodejs_22; });

  # Update this when adding the newest nodejs major version!
  nodejs_latest = nodejs_22;
  nodejs-slim_latest = nodejs-slim_22;
  corepack_latest = hiPrio corepack_22;

  buildNpmPackage = callPackage ../build-support/node/build-npm-package { };

  npmHooks = callPackage ../build-support/node/build-npm-package/hooks { };

  inherit (callPackages ../build-support/node/fetch-npm-deps { })
    fetchNpmDeps prefetch-npm-deps;

  importNpmLock = callPackages ../build-support/node/import-npm-lock { };

  nodePackages_latest = dontRecurseIntoAttrs nodejs_latest.pkgs // { __attrsFailEvaluation = true; };

  nodePackages = dontRecurseIntoAttrs nodejs.pkgs // { __attrsFailEvaluation = true; };

  node2nix = nodePackages.node2nix;

  notesnook = callPackage ../applications/misc/notesnook { };

  oddjob = callPackage ../os-specific/linux/oddjob { };

  openipmi = callPackage ../tools/system/openipmi { };

  ox = callPackage ../applications/editors/ox { };

  oxigraph = callPackage ../servers/oxigraph {
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  file-rename = callPackage ../tools/filesystems/file-rename { };

  kcollectd = libsForQt5.callPackage ../tools/misc/kcollectd { };

  kea = callPackage ../tools/networking/kea { };

  keama = callPackage ../tools/networking/keama { };

  ktailctl = kdePackages.callPackage ../applications/networking/ktailctl {};

  ispell = callPackage ../tools/text/ispell { };

  iodash = callPackage ../development/libraries/iodash { };

  jbofihe = callPackage ../tools/text/jbofihe { };

  jbrowse = callPackage ../applications/science/biology/jbrowse { };

  jumanpp = callPackage ../tools/text/jumanpp { };

  jump = callPackage ../tools/system/jump { };

  latex2html = callPackage ../tools/misc/latex2html { };

  lazycli = callPackage ../tools/misc/lazycli { };

  lavat = callPackage ../tools/misc/lavat { };

  lcdf-typetools = callPackage ../tools/misc/lcdf-typetools { };

  lcsync = callPackage ../applications/networking/sync/lcsync { };

  ldapdomaindump = with python3Packages; toPythonApplication ldapdomaindump;

  ldapmonitor = callPackage ../tools/security/ldapmonitor { };

  ldapnomnom = callPackage ../tools/security/ldapnomnom { };

  ldapvi = callPackage ../tools/misc/ldapvi { };

  ldeep = callPackage ../tools/security/ldeep { };

  ldns = callPackage ../development/libraries/ldns { };

  leafpad = callPackage ../applications/editors/leafpad { };

  l3afpad = callPackage ../applications/editors/l3afpad { };

  leanify = callPackage ../tools/misc/leanify { };

  leatherman = callPackage ../development/libraries/leatherman { };

  lact = callPackage ../tools/system/lact { };

  ledit = callPackage ../tools/misc/ledit { };

  ledmon = callPackage ../tools/system/ledmon { };

  leela = callPackage ../tools/graphics/leela { };

  lemmeknow = callPackage ../tools/misc/lemmeknow { };

  lemmy-help = callPackage ../tools/misc/lemmy-help { };

  lerpn = callPackage ../tools/misc/lerpn { };

  lethe = callPackage ../tools/security/lethe {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  lftp = callPackage ../tools/networking/lftp { };

  libck = callPackage ../development/libraries/libck { };

  libcork = callPackage ../development/libraries/libcork { };

  libconfig = callPackage ../development/libraries/libconfig { };

  libcmis = callPackage ../development/libraries/libcmis { };

  libee = callPackage ../development/libraries/libee { };

  libepc = callPackage ../development/libraries/libepc { };

  liberfa = callPackage ../development/libraries/liberfa { };

  libestr = callPackage ../development/libraries/libestr { };

  libevdev = callPackage ../development/libraries/libevdev { };

  liberio = callPackage ../development/libraries/liberio { };

  libevdevplus = callPackage ../development/libraries/libevdevplus { };

  libfann = callPackage ../development/libraries/libfann { };

  libfsm = callPackage ../development/libraries/libfsm { };

  libgaminggear = callPackage ../development/libraries/libgaminggear { };

  libhandy = callPackage ../development/libraries/libhandy { };

  # Needed for apps that still depend on the unstable verison of the library (not libhandy-1)
  libhandy_0 = callPackage ../development/libraries/libhandy/0.x.nix { };

  libgmpris = callPackage ../development/libraries/libgmpris { };

  libgumath = callPackage ../development/libraries/libgumath { };

  libinsane = callPackage ../development/libraries/libinsane { };

  libint = callPackage ../development/libraries/libint { };
  libintPsi4 = callPackage ../development/libraries/libint {
    enableFortran = false;
    enableSSE = false;
    maxAm = 6;
    eriDeriv = 2;
    eri3Deriv = 2;
    eri2Deriv = 2;
    eriAm = [ 6 5 4 ];
    eri3Am = [ 6 5 4 ];
    eri2Am = [ 6 5 4 ];
    eriOptAm = [ 3 2 2 ];
    eri3OptAm = [ 3 2 2 ];
    eri2OptAm = [ 3 2 2 ];
    enableOneBody = true;
    oneBodyDerivOrd = 2;
    enableGeneric = false;
    enableContracted = false;
    cartGaussOrd = "standard";
    shGaussOrd = "gaussian";
    eri2PureSh = false;
    eri3PureSh = false;
  };

  libipfix = callPackage ../development/libraries/libipfix { };

  libirc = libsForQt5.callPackage ../development/libraries/libirc { };

  libircclient = callPackage ../development/libraries/libircclient { };

  libiscsi = callPackage ../development/libraries/libiscsi { };

  libisds = callPackage ../development/libraries/libisds { };

  libite = callPackage ../development/libraries/libite { };

  liblangtag = callPackage ../development/libraries/liblangtag {
    inherit (gnome) gnome-common;
  };

  liblouis = callPackage ../development/libraries/liblouis { };

  liboauth = callPackage ../development/libraries/liboauth { };

  libr3 = callPackage ../development/libraries/libr3 { };

  libraspberrypi = callPackage ../development/libraries/libraspberrypi { };

  libsidplayfp = callPackage ../development/libraries/libsidplayfp { };

  libspf2 = callPackage ../development/libraries/libspf2 { };

  libsrs2 = callPackage ../development/libraries/libsrs2 { };

  libtermkey = callPackage ../development/libraries/libtermkey { };

  libtelnet = callPackage ../development/libraries/libtelnet { };

  libtirpc = callPackage ../development/libraries/ti-rpc { };

  libtickit = callPackage ../development/libraries/libtickit { };

  libtins = callPackage ../development/libraries/libtins { };

  libtree = callPackage ../development/tools/misc/libtree { };

  libshout = callPackage ../development/libraries/libshout { };

  libqb = callPackage ../development/libraries/libqb { };

  libqmi = callPackage ../development/libraries/libqmi { };

  libqrtr-glib = callPackage ../development/libraries/libqrtr-glib { };

  libmbim = callPackage ../development/libraries/libmbim { };

  libmongocrypt = darwin.apple_sdk_11_0.callPackage ../development/libraries/libmongocrypt { };

  libmesode = callPackage ../development/libraries/libmesode { };

  libmsym = callPackage ../development/libraries/science/chemistry/libmsym { };

  libnabo = callPackage ../development/libraries/libnabo { };

  libngspice = callPackage ../development/libraries/libngspice { };

  libnixxml = callPackage ../development/libraries/libnixxml { };

  libplctag = callPackage ../development/libraries/libplctag { };

  libpointmatcher = callPackage ../development/libraries/libpointmatcher { };

  libportal = callPackage ../development/libraries/libportal { };
  libportal-gtk3 = libportal.override { variant = "gtk3"; };
  libportal-gtk4 = libportal.override { variant = "gtk4"; };
  libportal-qt5 = libportal.override { variant = "qt5"; };

  libmicrodns = callPackage ../development/libraries/libmicrodns { };

  libnids = callPackage ../tools/networking/libnids { };

  rtorrent = callPackage ../applications/networking/p2p/rakshasa-rtorrent {
    libtorrent = callPackage ../applications/networking/p2p/rakshasa-rtorrent/libtorrent.nix { };
  };

  jesec-rtorrent = callPackage ../applications/networking/p2p/jesec-rtorrent {
    libtorrent = callPackage ../applications/networking/p2p/jesec-rtorrent/libtorrent.nix { };
  };

  libmpack = callPackage ../development/libraries/libmpack { };

  libiberty = callPackage ../development/libraries/libiberty { };

  libucl = callPackage ../development/libraries/libucl { };

  libxc = callPackage ../development/libraries/libxc { };

  libxcomp = callPackage ../development/libraries/libxcomp { };

  libxl = callPackage ../development/libraries/libxl { };

  libx86emu = callPackage ../development/libraries/libx86emu { };

  libzim = callPackage ../development/libraries/libzim { };


  libzmf = callPackage ../development/libraries/libzmf { };

  libreddit = callPackage ../servers/libreddit {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  librespeed-cli = callPackage ../tools/misc/librespeed-cli { };

  libreswan = callPackage ../tools/networking/libreswan { };

  librest = callPackage ../development/libraries/librest { };

  librest_1_0 = callPackage ../development/libraries/librest/1.0.nix { };

  libwebsockets = callPackage ../development/libraries/libwebsockets { };

  libyafaray = callPackage ../tools/graphics/libyafaray { };

  licensee = callPackage ../tools/package-management/licensee { };

  lidarr = callPackage ../servers/lidarr { };

  inherit ({
    limesuite = callPackage ../applications/radio/limesuite {
      inherit (darwin.apple_sdk.frameworks) GLUT;
    };
    limesuiteWithGui = limesuite.override {
      withGui = true;
    };
  })
  limesuite
  limesuiteWithGui;

  limesurvey = callPackage ../servers/limesurvey { };

  linux-exploit-suggester = callPackage ../tools/security/linux-exploit-suggester { };

  linux-gpib = callPackage ../applications/science/electronics/linux-gpib/user.nix { };

  linuxquota = callPackage ../tools/misc/linuxquota { };

  lipl = callPackage ../tools/misc/lipl { };

  liquidctl = with python3Packages; toPythonApplication liquidctl;

  lmp = callPackage ../tools/security/lmp { };

  localproxy = callPackage ../applications/networking/localproxy { };

  localstack = with python3Packages; toPythonApplication localstack;

  localtime = callPackage ../tools/system/localtime { };

  log4j-detect = callPackage ../tools/security/log4j-detect { };

  log4j-scan = callPackage ../tools/security/log4j-scan { };

  log4j-sniffer = callPackage ../tools/security/log4j-sniffer { };

  log4j-vuln-scanner = callPackage ../tools/security/log4j-vuln-scanner { };

  log4jcheck = callPackage ../tools/security/log4jcheck { };

  log4shell-detector = callPackage ../tools/security/log4shell-detector { };

  logcheck = callPackage ../tools/system/logcheck { };

  logmap = callPackage ../tools/security/logmap { };

  logmein-hamachi = callPackage ../tools/networking/logmein-hamachi { };

  logkeys = callPackage ../tools/security/logkeys { };

  logrotate = callPackage ../tools/system/logrotate { };

  logstalgia = callPackage ../tools/graphics/logstalgia { };

  lokalise2-cli = callPackage ../tools/misc/lokalise2-cli { };

  loki = callPackage ../development/libraries/loki { };

  longview = callPackage ../servers/monitoring/longview { };

  lorien = callPackage ../applications/graphics/lorien { };

  lout = callPackage ../tools/typesetting/lout { };

  lr = callPackage ../tools/system/lr { };

  lrzip = callPackage ../tools/compression/lrzip { };

  lsb-release = callPackage ../os-specific/linux/lsb-release { };

  # lsh installs `bin/nettle-lfib-stream' and so does Nettle.  Give the
  # former a lower priority than Nettle.
  lsh = lowPrio (callPackage ../tools/networking/lsh { });

  lunatic = callPackage ../development/interpreters/lunatic { };

  lux = callPackage ../tools/video/lux { };

  lv = callPackage ../tools/text/lv { };

  lv_img_conv = callPackage ../development/tools/lv_img_conv  {
    inherit (darwin.apple_sdk.frameworks) CoreText;
  };

  lwc = callPackage ../tools/misc/lwc { };

  lxd-image-server = callPackage ../tools/virtualization/lxd-image-server { };

  lzfse = callPackage ../tools/compression/lzfse { };

  lzham = callPackage ../tools/compression/lzham { };

  lzip = callPackage ../tools/compression/lzip { };

  lziprecover = callPackage ../tools/compression/lziprecover { };

  xz = callPackage ../tools/compression/xz { };

  lz4 = callPackage ../tools/compression/lz4 { };

  lzbench = callPackage ../tools/compression/lzbench { };

  lzop = callPackage ../tools/compression/lzop { };

  lzwolf = callPackage ../games/lzwolf { SDL2_mixer = SDL2_mixer_2_0; };

  macchanger = callPackage ../os-specific/linux/macchanger { };

  macchina = callPackage ../tools/misc/macchina { };

  madlang = haskell.lib.compose.justStaticExecutables haskellPackages.madlang;

  maeparser = callPackage ../development/libraries/maeparser { };

  mailcheck = callPackage ../applications/networking/mailreaders/mailcheck { };

  maildrop = callPackage ../tools/networking/maildrop { };

  mailhog = callPackage ../servers/mail/mailhog { };

  mailnag = callPackage ../applications/networking/mailreaders/mailnag {
    availablePlugins = {
      # More are listed here: https://github.com/pulb/mailnag/#desktop-integration
      # Use the attributes here as arguments to `plugins` list
      goa = callPackage ../applications/networking/mailreaders/mailnag/goa-plugin.nix { };
    };
  };
  mailnagWithPlugins = mailnag.withPlugins(
    builtins.attrValues mailnag.availablePlugins
  );
  bubblemail = callPackage ../applications/networking/mailreaders/bubblemail { };

  mailpit = callPackage ../servers/mail/mailpit {
    libtool = if stdenv.isDarwin then darwin.cctools else libtool;
  };

  mailsend = callPackage ../tools/networking/mailsend { };

  mailutils = callPackage ../tools/networking/mailutils {
    sasl = gsasl;
  };

  matrix-sdk-crypto-nodejs = callPackage ../development/libraries/matrix-sdk-crypto-nodejs { };
  matrix-sdk-crypto-nodejs-0_1_0-beta_3 = callPackage ../development/libraries/matrix-sdk-crypto-nodejs/beta3.nix { };

  email = callPackage ../tools/networking/email { };

  maim = callPackage ../tools/graphics/maim { };

  mairix = callPackage ../tools/text/mairix { };

  makemkv = libsForQt5.callPackage ../applications/video/makemkv { };

  makerpm = callPackage ../development/tools/makerpm { };

  makefile2graph = callPackage ../development/tools/analysis/makefile2graph { };

  man = man-db;

  man-db = callPackage ../tools/misc/man-db { };

  mandoc = callPackage ../tools/misc/mandoc { };

  mangareader = libsForQt5.callPackage ../applications/graphics/mangareader { };

  mangohud = callPackage ../tools/graphics/mangohud {
    libXNVCtrl = linuxPackages.nvidia_x11.settings.libXNVCtrl;
    mangohud32 = pkgsi686Linux.mangohud;
    inherit (python3Packages) mako;
  };

  marimo = with python3Packages; toPythonApplication marimo;

  marktext = callPackage ../applications/misc/marktext { };

  mars-mips = callPackage ../development/tools/mars-mips { };

  maui-shell = libsForQt5.callPackage ../applications/window-managers/maui-shell { };

  mawk = callPackage ../tools/text/mawk { };

  mb2md = callPackage ../tools/text/mb2md { };

  mbuffer = callPackage ../tools/misc/mbuffer { };

  mecab =
    let
      mecab-nodic = callPackage ../tools/text/mecab/nodic.nix { };
    in
    callPackage ../tools/text/mecab {
      mecab-ipadic = callPackage ../tools/text/mecab/ipadic.nix {
        inherit mecab-nodic;
      };
    };

  mediawiki = callPackage ../servers/web-apps/mediawiki { };

  mediawriter = callPackage ../tools/system/mediawriter { };

  memtier-benchmark = callPackage ../tools/networking/memtier-benchmark { };

  memtest86-efi = callPackage ../tools/misc/memtest86-efi { };

  memtest86plus = callPackage ../tools/misc/memtest86+ { };

  mbutil = python310Packages.callPackage ../applications/misc/mbutil { };

  mcabber = callPackage ../applications/networking/instant-messengers/mcabber { };

  mcron = callPackage ../tools/system/mcron { };

  mcstatus = with python3Packages; toPythonApplication mcstatus;

  mdbtools = callPackage ../tools/misc/mdbtools { };

  mdk = callPackage ../development/tools/mdk { };

  mdk4 = callPackage ../tools/networking/mdk4 { };

  mdp = callPackage ../applications/misc/mdp { };

  megacli = callPackage ../tools/misc/megacli { };

  megatools = callPackage ../tools/networking/megatools { };

  memo = callPackage ../applications/misc/memo { };

  mencal = callPackage ../applications/misc/mencal { } ;

  metamorphose2 = callPackage ../applications/misc/metamorphose2 { };

  metar = callPackage ../applications/misc/metar { };

  mfcuk = callPackage ../tools/security/mfcuk { };

  mfoc = callPackage ../tools/security/mfoc { };

  microbin = callPackage ../servers/microbin { };

  microdnf = callPackage ../tools/package-management/microdnf { };

  microplane = callPackage ../tools/misc/microplane { };

  microserver = callPackage ../servers/microserver { };

  midisheetmusic = callPackage ../applications/audio/midisheetmusic { };

  mikutter = callPackage ../applications/networking/instant-messengers/mikutter { };

  mimeo = callPackage ../tools/misc/mimeo { };

  mimetic = callPackage ../development/libraries/mimetic { };

  minio-client = callPackage ../tools/networking/minio-client { };

  minio-certgen = callPackage ../tools/security/minio-certgen { };

  minissdpd = callPackage ../tools/networking/minissdpd { };

  miniupnpc = callPackage ../tools/networking/miniupnpc { };

  miniupnpd = callPackage ../tools/networking/miniupnpd { };

  miniupnpd-nftables = callPackage ../tools/networking/miniupnpd { firewall = "nftables"; };

  miniball = callPackage ../development/libraries/miniball { };

  minijail = callPackage ../tools/system/minijail { };

  minijail-tools = python3.pkgs.callPackage ../tools/system/minijail/tools.nix { };

  minilibx = callPackage ../development/libraries/minilibx { };

  minixml = callPackage ../development/libraries/minixml { };

  mir-qualia = callPackage ../tools/text/mir-qualia {
    pythonPackages = python3Packages;
  };

  mirakurun = callPackage ../applications/video/mirakurun { };

  miredo = callPackage ../tools/networking/miredo { };

  mirrorbits = callPackage ../servers/mirrorbits { };

  mitmproxy = with python3Packages; toPythonApplication mitmproxy;

  mitmproxy2swagger = callPackage ../tools/security/mitmproxy2swagger { };

  mjpegtools = callPackage ../tools/video/mjpegtools { };

  mjpegtoolsFull = mjpegtools.override {
    withMinimal = false;
  };

  mkclean = callPackage ../applications/video/mkclean { };

  mkcue = callPackage ../tools/cd-dvd/mkcue { };

  mkp224o = callPackage ../tools/security/mkp224o { };

  mkpasswd = hiPrio (callPackage ../tools/security/mkpasswd { });

  mkrand = callPackage ../tools/security/mkrand { };

  mktemp = callPackage ../tools/security/mktemp { };

  mktorrent = callPackage ../tools/misc/mktorrent { };

  mloader = callPackage ../tools/misc/mloader { };

  mmake = callPackage ../tools/misc/mmake { };

  mmixware = callPackage ../development/tools/mmixware { };

  modemmanager = callPackage ../tools/networking/modemmanager { };

  modem-manager-gui = callPackage ../applications/networking/modem-manager-gui { };

  modsecurity_standalone = callPackage ../tools/security/modsecurity { };

  modsecurity-crs = callPackage ../tools/security/modsecurity-crs { };

  molecule = with python3Packages; toPythonApplication molecule;

  molly-guard = callPackage ../os-specific/linux/molly-guard { };

  molotov = callPackage ../applications/video/molotov { };

  monit = callPackage ../tools/system/monit { };

  monocraft = callPackage ../data/fonts/monocraft { };

  monolith = callPackage ../tools/backup/monolith {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  moreutils = callPackage ../tools/misc/moreutils {
    docbook-xsl = docbook_xsl;
  };

  mole = callPackage ../tools/networking/mole { };

  morgen = callPackage ../applications/office/morgen {
    electron = electron_28;
  };

  mosh = callPackage ../tools/networking/mosh { };

  motrix = callPackage ../tools/networking/motrix { };

  mount-zip = callPackage ../tools/filesystems/mount-zip { };

  mpage = callPackage ../tools/text/mpage { };

  mprime = callPackage ../tools/misc/mprime { };

  mpw = callPackage ../tools/security/mpw { };

  mqtt_cpp = callPackage ../development/libraries/mqtt_cpp { };

  mr = callPackage ../applications/version-management/mr { };

  mrsh = callPackage ../shells/mrsh { };

  mrtg = callPackage ../tools/misc/mrtg { };

  mscgen = callPackage ../tools/graphics/mscgen { };

  msfpc = callPackage ../tools/security/msfpc { };

  melt = callPackage ../tools/security/melt { };

  metabigor = callPackage ../tools/security/metabigor { };

  metasploit = callPackage ../tools/security/metasploit { };

  mhost = callPackage ../applications/networking/mhost {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  ms-sys = callPackage ../tools/misc/ms-sys { };

  mtdutils = callPackage ../tools/filesystems/mtdutils { };

  mtools = callPackage ../tools/filesystems/mtools { };

  mtr = callPackage ../tools/networking/mtr { };

  mtr-exporter = callPackage ../tools/networking/mtr-exporter { };

  mtr-gui = callPackage ../tools/networking/mtr { withGtk = true; };

  mtx = callPackage ../tools/backup/mtx { };

  mt-st = callPackage ../tools/backup/mt-st { };

  mubeng = callPackage ../tools/networking/mubeng { };

  muffet = callPackage ../tools/networking/muffet { };

  multipass = qt6Packages.callPackage ../tools/virtualization/multipass { };

  multitime = callPackage ../tools/misc/multitime { };

  sta = callPackage ../tools/misc/sta { };

  multitran = recurseIntoAttrs (let callPackage = newScope pkgs.multitran; in {
    multitrandata = callPackage ../tools/text/multitran/data { };

    libbtree = callPackage ../tools/text/multitran/libbtree { };

    libmtsupport = callPackage ../tools/text/multitran/libmtsupport { };

    libfacet = callPackage ../tools/text/multitran/libfacet { };

    libmtquery = callPackage ../tools/text/multitran/libmtquery { };

    mtutils = callPackage ../tools/text/multitran/mtutils { };
  });

  munge = callPackage ../tools/security/munge { };

  mutagen = callPackage ../tools/misc/mutagen { };

  mutagen-compose = callPackage ../tools/misc/mutagen-compose { };

  mycli = callPackage ../tools/admin/mycli { };

  mycrypto = callPackage ../applications/blockchains/mycrypto { };

  mydumper = callPackage ../tools/backup/mydumper { };

  mylvmbackup = callPackage ../tools/backup/mylvmbackup { };

  mysql2pgsql = callPackage ../tools/misc/mysql2pgsql { };

  mysqltuner = callPackage ../tools/misc/mysqltuner { };

  mytetra = libsForQt5.callPackage ../applications/office/mytetra { };

  nabi = callPackage ../tools/inputmethods/nabi { };

  nahid-fonts = callPackage ../data/fonts/nahid-fonts { };

  namazu = callPackage ../tools/text/namazu { };

  nasty = callPackage ../tools/security/nasty { };

  nat-traverse = callPackage ../tools/networking/nat-traverse { };

  navi = callPackage ../applications/misc/navi { };

  navilu-font = callPackage ../data/fonts/navilu { stdenv = stdenvNoCC; };

  nbd = callPackage ../tools/networking/nbd { };
  xnbd = callPackage ../tools/networking/xnbd { };

  ndjbdns = callPackage ../tools/networking/ndjbdns { };

  ndppd = callPackage ../applications/networking/ndppd { };

  nearcore = callPackage ../applications/blockchains/nearcore { };

  nebula = callPackage ../tools/networking/nebula { };

  nemiver = callPackage ../development/tools/nemiver { };

  neo-cowsay = callPackage ../tools/misc/neo-cowsay { };

  neofetch = callPackage ../tools/misc/neofetch { };

  nerdfix = callPackage ../tools/text/nerdfix { };

  nerdfonts = callPackage ../data/fonts/nerdfonts { };

  netatalk = callPackage ../tools/filesystems/netatalk { };

  netavark = callPackage ../tools/networking/netavark { };

  netcdf = callPackage ../development/libraries/netcdf { };

  netcdf-mpi = netcdf.override {
    hdf5 = hdf5-mpi.override { usev110Api = true; };
  };

  netcdfcxx4 = callPackage ../development/libraries/netcdf-cxx4 { };

  netcdffortran = callPackage ../development/libraries/netcdf-fortran {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation CoreServices SystemConfiguration;
  };

  networking-ts-cxx = callPackage ../development/libraries/networking-ts-cxx { };

  nco = callPackage ../development/libraries/nco { };

  ncftp = callPackage ../tools/networking/ncftp { };

  ncgopher = callPackage ../applications/networking/ncgopher { };

  ncompress = callPackage ../tools/compression/ncompress { };

  ndisc6 = callPackage ../tools/networking/ndisc6 { };

  netassert = callPackage ../tools/networking/netassert { };

  netboot = callPackage ../tools/networking/netboot { };

  netbootxyz-efi = callPackage ../tools/misc/netbootxyz-efi { };

  inherit (callPackage ../servers/web-apps/netbox { })
    netbox netbox_3_6 netbox_3_7;

  netbox2netshot = callPackage ../tools/admin/netbox2netshot { };

  netcat = libressl.nc.overrideAttrs (old: {
    meta = old.meta // {
      mainProgram = "nc";
    };
  });

  netcat-gnu = callPackage ../tools/networking/netcat { };

  netcat-openbsd = callPackage ../tools/networking/netcat-openbsd { };

  netdiscover = callPackage ../tools/networking/netdiscover { };

  nethogs = callPackage ../tools/networking/nethogs { };

  nethoscope = callPackage ../tools/networking/nethoscope { };

  netkittftp = callPackage ../tools/networking/netkit/tftp { };

  netlify-cli = callPackage ../development/web/netlify-cli { };

  netlistsvg = callPackage ../applications/science/logic/netlistsvg { };

  netpbm = callPackage ../tools/graphics/netpbm { };

  netrw = callPackage ../tools/networking/netrw { };

  netselect = callPackage ../tools/networking/netselect { };

  networkmanager = callPackage ../tools/networking/networkmanager { };

  networkmanager-iodine = callPackage ../tools/networking/networkmanager/iodine { };

  networkmanager-openvpn = callPackage ../tools/networking/networkmanager/openvpn { };

  networkmanager-l2tp = callPackage ../tools/networking/networkmanager/l2tp { };

  networkmanager-vpnc = callPackage ../tools/networking/networkmanager/vpnc { };

  networkmanager-openconnect = callPackage ../tools/networking/networkmanager/openconnect { };

  networkmanager-fortisslvpn = callPackage ../tools/networking/networkmanager/fortisslvpn { };

  networkmanager_strongswan = callPackage ../tools/networking/networkmanager/strongswan { };

  networkmanager-sstp = callPackage ../tools/networking/networkmanager/sstp { };

  networkmanagerapplet = callPackage ../tools/networking/networkmanager/applet { };

  libnma = callPackage ../tools/networking/networkmanager/libnma { };

  libnma-gtk4 = libnma.override { withGtk4 = true; };

  networkmanager_dmenu = callPackage ../tools/networking/networkmanager/dmenu  { };

  nm-tray = libsForQt5.callPackage ../tools/networking/networkmanager/tray.nix { };

  newsboat = callPackage ../applications/networking/feedreaders/newsboat {
    inherit (darwin.apple_sdk.frameworks) Security Foundation;
  };

  grocy = callPackage ../servers/grocy { };

  inherit (callPackages ../servers/nextcloud {})
    nextcloud27 nextcloud28 nextcloud29;

  nextcloud27Packages = callPackage ../servers/nextcloud/packages {
    apps = lib.importJSON ../servers/nextcloud/packages/27.json;
  };
  nextcloud28Packages = callPackage ../servers/nextcloud/packages {
    apps = lib.importJSON ../servers/nextcloud/packages/28.json;
  };
  nextcloud29Packages = callPackage ../servers/nextcloud/packages {
    apps = lib.importJSON ../servers/nextcloud/packages/29.json;
  };


  nextcloud-client = libsForQt5.callPackage ../applications/networking/nextcloud-client { };

  nextcloud-news-updater = callPackage ../servers/nextcloud/news-updater.nix { };

  nextcloud-notify_push = callPackage ../servers/nextcloud/notify_push.nix { };

  ndstool = callPackage ../tools/archivers/ndstool { };

  nfs-ganesha = callPackage ../servers/nfs-ganesha { };

  nflz = callPackage ../misc/nflz { };

  ngrep = callPackage ../tools/networking/ngrep { };

  ngrok = callPackage ../tools/networking/ngrok { };

  nifi = callPackage ../servers/web-apps/nifi { };

  noip = callPackage ../tools/networking/noip { };

  inherit (callPackages ../applications/networking/cluster/nomad { })
    nomad
    nomad_1_4
    nomad_1_5
    nomad_1_6
    nomad_1_7
    ;

  nomad-autoscaler = callPackage ../applications/networking/cluster/nomad-autoscaler { };

  nomad-driver-podman = callPackage ../applications/networking/cluster/nomad-driver-podman { };

  nomad-pack = callPackage ../applications/networking/cluster/nomad-pack { };

  nova = callPackage ../applications/networking/cluster/nova { };

  nomino = callPackage ../tools/misc/nomino { };

  nb = callPackage ../tools/misc/nb { };

  nbqa = callPackage ../tools/misc/nbqa { };

  kb = callPackage ../tools/misc/kb { };

  notable = callPackage ../applications/misc/notable { };

  npm-check = callPackage ../development/tools/npm-check { };

  nth = with python3Packages; toPythonApplication name-that-hash;

  ntlmrecon = callPackage ../tools/security/ntlmrecon { };

  numberstation = callPackage ../applications/misc/numberstation { };

  nvchecker = with python3Packages; toPythonApplication (
    nvchecker.overridePythonAttrs (oldAttrs: {
      propagatedBuildInputs = oldAttrs.propagatedBuildInputs
        ++ lib.flatten (builtins.attrValues oldAttrs.optional-dependencies);
    })
  );

  nvfetcher = haskell.lib.compose.justStaticExecutables haskellPackages.nvfetcher;


  miller = callPackage ../tools/text/miller { };

  milu = callPackage ../applications/misc/milu { };

  mkgmap = callPackage ../applications/misc/mkgmap { };

  mkgmap-splitter = callPackage ../applications/misc/mkgmap/splitter { };

  mpack = callPackage ../tools/networking/mpack { };

  mtm = callPackage ../tools/misc/mtm { };

  op-geth = callPackage ../applications/blockchains/optimism/geth.nix {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  optimism = callPackage ../applications/blockchains/optimism { };

  pa_applet = callPackage ../tools/audio/pa-applet { };

  pandoc-acro = python3Packages.callPackage ../tools/misc/pandoc-acro { };

  pandoc-imagine = python3Packages.callPackage ../tools/misc/pandoc-imagine { };

  pandoc-include = python3Packages.callPackage ../tools/misc/pandoc-include { };

  pandoc-drawio-filter = python3Packages.callPackage ../tools/misc/pandoc-drawio-filter { };

  pandoc-katex = callPackage ../tools/misc/pandoc-katex { };

  pandoc-plantuml-filter = python3Packages.callPackage ../tools/misc/pandoc-plantuml-filter { };

  # pandoc-*nos is a filter suite, where pandoc-xnos has all functionality and the others are used for only specific functionality
  pandoc-eqnos = python3Packages.callPackage ../tools/misc/pandoc-eqnos { };
  pandoc-fignos = python3Packages.callPackage ../tools/misc/pandoc-fignos { };
  pandoc-secnos = python3Packages.callPackage ../tools/misc/pandoc-secnos { };
  pandoc-tablenos = python3Packages.callPackage ../tools/misc/pandoc-tablenos { };

  panicparse = callPackage ../tools/misc/panicparse {};

  panoply = callPackage ../tools/misc/panoply { };

  patray = callPackage ../tools/audio/patray { };

  pathvector = callPackage ../tools/networking/pathvector { };

  pasystray = callPackage ../tools/audio/pasystray { };

  pcmsolver = callPackage ../development/libraries/pcmsolver { };

  pegasus-frontend = libsForQt5.callPackage ../games/pegasus-frontend {};

  pgbadger = perlPackages.callPackage ../tools/misc/pgbadger { };

  pffft = callPackage ../development/libraries/pffft { };

  phash = callPackage ../development/libraries/phash { };

  pnmixer = callPackage ../tools/audio/pnmixer { };

  present-cli = callPackage ../tools/misc/present-cli { };

  pulsemixer = callPackage ../tools/audio/pulsemixer { };

  pwsafe = callPackage ../applications/misc/pwsafe { };

  pw-viz = callPackage ../applications/misc/pw-viz { };

  napi-rs-cli = callPackage ../development/tools/napi-rs-cli { };

  neil = callPackage ../development/tools/neil { };

  niff = callPackage ../tools/package-management/niff { };

  nifskope = libsForQt5.callPackage ../tools/graphics/nifskope { };

  nilfs-utils = callPackage ../tools/filesystems/nilfs-utils { };

  nitrogen = callPackage ../tools/X11/nitrogen { };

  smart-wallpaper = callPackage ../tools/X11/smart-wallpaper { };

  nms = callPackage ../tools/misc/nms { };

  nomachine-client = callPackage ../tools/admin/nomachine-client { };

  notify-desktop = callPackage ../tools/misc/notify-desktop { };

  nkf = callPackage ../tools/text/nkf { };

  nlopt = callPackage ../development/libraries/nlopt { octave = null; };

  npapi_sdk = callPackage ../development/libraries/npapi-sdk { };

  npiet = callPackage ../development/interpreters/npiet { };

  npth = callPackage ../development/libraries/npth { };

  nmap = callPackage ../tools/security/nmap { };

  nmap-formatter = callPackage ../tools/security/nmap-formatter { };

  nmapsi4 = libsForQt5.callPackage ../tools/security/nmap/qt.nix { };

  noise-repellent = callPackage ../applications/audio/noise-repellent { };

  noisetorch = callPackage ../applications/audio/noisetorch { };

  notary = callPackage ../tools/security/notary { };

  notation = callPackage ../tools/security/notation { };

  notify-osd = callPackage ../applications/misc/notify-osd { };

  notes = qt6Packages.callPackage ../applications/office/notes {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  notes-up = callPackage ../applications/office/notes-up { };

  notify-osd-customizable = callPackage ../applications/misc/notify-osd-customizable { };

  nox = callPackage ../tools/package-management/nox { };

  nq = callPackage ../tools/system/nq { };

  nsjail = callPackage ../tools/security/nsjail {
    protobuf = protobuf_21;
  };

  nss_pam_ldapd = callPackage ../tools/networking/nss-pam-ldapd { };

  ntfs3g = callPackage ../tools/filesystems/ntfs-3g {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration;
  };

  # ntfsprogs are merged into ntfs-3g
  ntfsprogs = pkgs.ntfs3g;

  ntfy = callPackage ../tools/misc/ntfy { };

  ntfy-sh = callPackage ../tools/misc/ntfy-sh { };

  ntirpc = callPackage ../development/libraries/ntirpc { };

  ntopng = callPackage ../tools/networking/ntopng { };

  ntp = callPackage ../tools/networking/ntp { };

  numdiff = callPackage ../tools/text/numdiff { };

  numlockx = callPackage ../tools/X11/numlockx { };

  nurl = callPackage ../tools/misc/nurl { };

  nttcp = callPackage ../tools/networking/nttcp { };

  ntttcp = callPackage ../tools/networking/ntttcp { };

  nuttcp = callPackage ../tools/networking/nuttcp { };

  nssmdns = callPackage ../tools/networking/nss-mdns { };

  nvfancontrol = callPackage ../tools/misc/nvfancontrol {
    libXNVCtrl = linuxPackages.nvidia_x11.settings.libXNVCtrl;
  };

  nvimpager = callPackage ../tools/misc/nvimpager { };

  nwdiag = with python3Packages; toPythonApplication nwdiag;

  nxdomain = python3.pkgs.callPackage ../tools/networking/nxdomain { };

  nxpmicro-mfgtools = callPackage ../development/tools/misc/nxpmicro-mfgtools { };

  nyancat = callPackage ../tools/misc/nyancat { };

  nylon = callPackage ../tools/networking/nylon { };

  nym = darwin.apple_sdk_11_0.callPackage ../applications/networking/nym {
    inherit (darwin.apple_sdk.frameworks) Security CoreServices;
  };

  nzbget = callPackage ../tools/networking/nzbget { };

  nzbhydra2 = callPackage ../servers/nzbhydra2 {
    # You need Java (at least 8, at most 15)
    # https://github.com/theotherp/nzbhydra2/issues/697
    # https://github.com/theotherp/nzbhydra2/#how-to-run
    jre = openjdk11;
  };

  oapi-codegen = callPackage ../tools/networking/oapi-codegen { };

  oath-toolkit = callPackage ../tools/security/oath-toolkit { };

  oatpp = callPackage ../development/libraries/oatpp { };

  obex_data_server = callPackage ../tools/bluetooth/obex-data-server { };

  obexd = callPackage ../tools/bluetooth/obexd { };

  obfs4 = callPackage ../tools/networking/obfs4 { };

  ocproxy = callPackage ../tools/networking/ocproxy { };

  ocserv = callPackage ../tools/networking/ocserv { };

  octofetch = callPackage ../tools/misc/octofetch {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  oha = callPackage ../tools/networking/oha { };

  onetun = callPackage ../tools/networking/onetun {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  opencorsairlink = callPackage ../tools/misc/opencorsairlink { };

  openfpgaloader = callPackage ../development/embedded/fpga/openfpgaloader { };

  openfortivpn = callPackage ../tools/networking/openfortivpn { };

  openobserve = darwin.apple_sdk_11_0.callPackage ../servers/monitoring/openobserve {
    apple_sdk = darwin.apple_sdk_11_0;
  };

  obexfs = callPackage ../tools/bluetooth/obexfs { };

  obexftp = callPackage ../tools/bluetooth/obexftp { };

  objconv = callPackage ../development/tools/misc/objconv { };

  odpic = callPackage ../development/libraries/odpic { };

  odt2txt = callPackage ../tools/text/odt2txt { };

  odyssey = callPackage ../tools/misc/odyssey { };

  offensive-azure = callPackage ../tools/security/offensive-azure { };

  offlineimap = callPackage ../tools/networking/offlineimap { };

  offzip = callPackage ../tools/compression/offzip { };

  ofono-phonesim = libsForQt5.callPackage ../development/tools/ofono-phonesim { };

  ogdf = callPackage ../development/libraries/ogdf { };

  oh-my-zsh = callPackage ../shells/zsh/oh-my-zsh { };

  ola = callPackage ../applications/misc/ola {
    protobuf = protobuf_21;
  };

  olive-editor = qt6Packages.callPackage ../applications/video/olive-editor {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation;
  };

  ombi = callPackage ../servers/ombi { };

  ome_zarr = with python3Packages; toPythonApplication ome-zarr;

  omping = callPackage ../applications/networking/omping { };

  onefetch = callPackage ../tools/misc/onefetch {
    inherit (darwin) libresolv;
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security;
  };

  onioncircuits = callPackage ../tools/security/onioncircuits { };

  onlykey-agent = callPackage ../tools/security/onlykey-agent { };

  onlykey-cli = callPackage ../tools/security/onlykey-cli { };

  onlykey = callPackage ../tools/security/onlykey { node_webkit = nwjs; };

  ooniprobe-cli = callPackage ../tools/networking/ooniprobe-cli { };

  openapi-generator-cli = callPackage ../tools/networking/openapi-generator-cli { jre = pkgs.jre_headless; };

  openboard = libsForQt5.callPackage ../applications/graphics/openboard { };

  opencc = callPackage ../tools/text/opencc { };

  opencl-info = callPackage ../tools/system/opencl-info { };

  opencryptoki = callPackage ../tools/security/opencryptoki { };

  opendbx = callPackage ../development/libraries/opendbx { };

  opendht = callPackage ../development/libraries/opendht  {
    inherit (darwin.apple_sdk.frameworks) Security;
    restinio = restinio_0_6;
  };

  opendkim = callPackage ../development/libraries/opendkim { };

  opendylan = callPackage ../development/compilers/opendylan {
    opendylan-bootstrap = opendylan_bin;
  };

  openfec = callPackage ../development/libraries/openfec { };

  ophis = python3Packages.callPackage ../development/compilers/ophis { };

  opendylan_bin = callPackage ../development/compilers/opendylan/bin.nix { };

  open-ecard = callPackage ../tools/security/open-ecard { };

  open-interpreter = with python3Packages; toPythonApplication open-interpreter;

  openjade = callPackage ../tools/text/sgml/openjade { };

  openhantek6022 = libsForQt5.callPackage ../applications/science/electronics/openhantek6022 { };

  openimagedenoise = callPackage ../development/libraries/openimagedenoise { };

  openmvg = callPackage ../applications/science/misc/openmvg {
    inherit (llvmPackages) openmp;
  };

  openmvs = callPackage ../applications/science/misc/openmvs {
    inherit (llvmPackages) openmp;
  };

  openntpd = callPackage ../tools/networking/openntpd { };

  openntpd_nixos = openntpd.override {
    privsepUser = "ntp";
    privsepPath = "/var/empty";
  };

  openobex = callPackage ../tools/bluetooth/openobex { };

  openresolv = callPackage ../tools/networking/openresolv { };

  openrefine = callPackage ../applications/science/misc/openrefine { jdk = jdk17; };

  openrgb = libsForQt5.callPackage ../applications/misc/openrgb { };

  openrgb-with-all-plugins = openrgb.withPlugins [
    openrgb-plugin-effects
    openrgb-plugin-hardwaresync
  ];

  openrgb-plugin-effects = libsForQt5.callPackage ../applications/misc/openrgb-plugins/effects { };

  openrgb-plugin-hardwaresync = libsForQt5.callPackage ../applications/misc/openrgb-plugins/hardwaresync { };

  opensbi = callPackage ../misc/opensbi { };

  opensc = callPackage ../tools/security/opensc {
    inherit (darwin.apple_sdk.frameworks) Carbon PCSC;
  };

  openseachest = callPackage ../tools/system/openseachest { };

  opensm = callPackage ../tools/networking/opensm { };

  tinyssh = callPackage ../tools/networking/tinyssh { };

  tinystatus = callPackage ../tools/networking/tinystatus { };

  toastify = darwin.apple_sdk_11_0.callPackage ../tools/misc/toastify {};

  tuc = callPackage ../tools/text/tuc { };

  opensshPackages = dontRecurseIntoAttrs (callPackage ../tools/networking/openssh {});

  openssh = opensshPackages.openssh.override {
    etcDir = "/etc/ssh";
  };

  openssh_hpn = opensshPackages.openssh_hpn.override {
    etcDir = "/etc/ssh";
  };

  openssh_gssapi = opensshPackages.openssh_gssapi.override {
    etcDir = "/etc/ssh";
  };

  ssh-copy-id = callPackage ../tools/networking/openssh/copyid.nix { };

  sshd-openpgp-auth = callPackage ../by-name/ss/ssh-openpgp-auth/daemon.nix { };

  opensp = callPackage ../tools/text/sgml/opensp { };

  opentofu = callPackage ../applications/networking/cluster/opentofu { };

  opentrack = libsForQt5.callPackage ../applications/misc/opentrack { };

  opentracker = callPackage ../applications/networking/p2p/opentracker { };

  alttpr-opentracker = callPackage ../tools/games/opentracker { };

  opentsdb = callPackage ../tools/misc/opentsdb { };

  openvpn = callPackage ../tools/networking/openvpn {};

  openvpn3 = callPackage ../tools/networking/openvpn3 { };

  openvpn_learnaddress = callPackage ../tools/networking/openvpn/openvpn_learnaddress.nix { };

  openvpn-auth-ldap = callPackage ../tools/networking/openvpn/openvpn-auth-ldap.nix {
    stdenv = clangStdenv;
  };

  namespaced-openvpn = python3Packages.callPackage ../tools/networking/namespaced-openvpn { };

  oq = callPackage ../development/tools/oq { };

  out-of-tree = callPackage ../development/tools/out-of-tree { };

  oppai-ng = callPackage ../tools/misc/oppai-ng { };

  operator-sdk = callPackage ../development/tools/operator-sdk { };

  oscclip = callPackage ../tools/misc/oscclip { };

  oui = callPackage ../tools/networking/oui { };

  owncast = callPackage ../servers/owncast { };

  owntracks-recorder = callPackage ../servers/owntracks-recorder { };

  update-dotdee = with python3Packages; toPythonApplication update-dotdee;

  update-nix-fetchgit = haskell.lib.compose.justStaticExecutables haskellPackages.update-nix-fetchgit;

  update-resolv-conf = callPackage ../tools/networking/openvpn/update-resolv-conf.nix { };

  update-systemd-resolved = callPackage ../tools/networking/openvpn/update-systemd-resolved.nix { };

  opae = callPackage ../development/libraries/opae { };

  open-pdf-sign = callPackage ../tools/misc/open-pdf-sign { };

  opentelemetry-collector = callPackage ../tools/misc/opentelemetry-collector { };
  opentelemetry-collector-contrib = callPackage ../tools/misc/opentelemetry-collector/contrib.nix { };

  openvswitch-lts = callPackage ../by-name/op/openvswitch/lts.nix { };

  ovn-lts = callPackage ../by-name/ov/ovn/lts.nix { };

  optifinePackages = callPackage ../tools/games/minecraft/optifine { };

  optifine = optifinePackages.optifine-latest;

  optipng = callPackage ../tools/graphics/optipng { };

  olsrd = callPackage ../tools/networking/olsrd { };

  oonf-olsrd2 = callPackage ../tools/networking/oonf-olsrd2 { };

  opl3bankeditor = libsForQt5.callPackage ../tools/audio/opl3bankeditor { };
  opn2bankeditor = libsForQt5.callPackage ../tools/audio/opl3bankeditor/opn2bankeditor.nix { };

  orangefs = callPackage ../tools/filesystems/orangefs {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  org-stats = callPackage ../tools/misc/org-stats { };

  orz = callPackage ../tools/compression/orz { };

  os-prober = callPackage ../tools/misc/os-prober { };

  oshka = callPackage ../development/tools/oshka { };

  osl = libsForQt5.callPackage ../development/compilers/osl {
    boost = boost179;
    libclang = llvmPackages_15.libclang;
    clang = clang_15;
    llvm = llvm_15;
    openexr = openexr_3;
  };

  osqp = callPackage ../development/libraries/science/math/osqp { };

  ossec-agent = callPackage ../tools/security/ossec/agent.nix { };

  ossec-server = callPackage ../tools/security/ossec/server.nix { };

  osslsigncode = callPackage ../development/tools/osslsigncode { };

  ostree = callPackage ../tools/misc/ostree { };

  ostree-rs-ext = callPackage ../tools/misc/ostree-rs-ext { };

  otel-cli = callPackage ../tools/misc/otel-cli { };

  otfcc = callPackage ../tools/misc/otfcc { };

  otpclient = callPackage ../applications/misc/otpclient { };

  otpw = callPackage ../os-specific/linux/otpw { };

  ovftool = callPackage ../tools/virtualization/ovftool { };

  overcommit = callPackage ../development/tools/overcommit { };

  overmind = callPackage ../applications/misc/overmind { };

  oxker = callPackage ../applications/misc/oxker { };

  oxlint = callPackage ../development/tools/oxlint { };

  ovh-ttyrec = callPackage ../tools/misc/ovh-ttyrec { };

  ovito = libsForQt5.callPackage ../applications/graphics/ovito {
    inherit (darwin.apple_sdk.frameworks) VideoDecodeAcceleration;
  };

  oxefmsynth = callPackage ../applications/audio/oxefmsynth { };

  oxidized = callPackage ../tools/admin/oxidized { };

  oxipng = callPackage ../tools/graphics/oxipng { };

  payload_dumper = callPackage ../tools/archivers/payload_dumper { };

  payload-dumper-go = callPackage ../tools/archivers/payload-dumper-go { };

  p2pvc = callPackage ../applications/video/p2pvc { };

  p3x-onenote = callPackage ../applications/office/p3x-onenote { };

  p4c = callPackage ../development/compilers/p4c {
    protobuf = protobuf_21;
  };

  p7zip = callPackage ../tools/archivers/p7zip { };

  packagekit = callPackage ../tools/package-management/packagekit { };

  packetdrill = callPackage ../tools/networking/packetdrill { };

  pacman = callPackage ../tools/package-management/pacman { };

  paco = callPackage ../development/compilers/paco { };

  padthv1 = libsForQt5.callPackage ../applications/audio/padthv1 { };

  padbuster = callPackage ../tools/security/padbuster { };

  page = callPackage ../tools/misc/page { };

  pageedit = libsForQt5.callPackage ../applications/office/PageEdit {
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  pagefind = libsForQt5.callPackage ../applications/misc/pagefind { };

  paging-calculator  = callPackage ../development/tools/paging-calculator { };

  pagmo2 = callPackage ../development/libraries/pagmo2 { };

  paho-mqtt-c = callPackage ../development/libraries/paho-mqtt-c { };

  paho-mqtt-cpp = callPackage ../development/libraries/paho-mqtt-cpp { };

  pakcs = callPackage ../development/compilers/pakcs {
    # Doesn't compile with GHC 9.0 due to whitespace syntax changes
    # see also https://github.com/NixOS/nixpkgs/issues/166108
    haskellPackages = haskell.packages.ghc810;
  };

  pal = callPackage ../tools/misc/pal { };

  pandoc = callPackage ../development/tools/pandoc { };

  pandoc-lua-filters = callPackage ../tools/misc/pandoc-lua-filters { };

  pamtester = callPackage ../tools/security/pamtester { };

  paperless-ngx = callPackage ../applications/office/paperless-ngx { };

  paperoni = callPackage ../tools/text/paperoni { };

  paperwork = callPackage ../applications/office/paperwork/paperwork-gtk.nix { };

  papeer = callPackage ../tools/text/papeer { };

  papertrail = callPackage ../tools/text/papertrail { };

  pappl = callPackage ../applications/printing/pappl { };

  par2cmdline = callPackage ../tools/networking/par2cmdline { };

  parallel = callPackage ../tools/misc/parallel { };

  parallel-full = callPackage ../tools/misc/parallel/wrapper.nix { };

  parastoo-fonts = callPackage ../data/fonts/parastoo-fonts { };

  parcellite = callPackage ../tools/misc/parcellite {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  parrot = callPackage ../applications/audio/parrot {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  patchutils = callPackage ../tools/text/patchutils { };

  patchutils_0_3_3 = callPackage ../tools/text/patchutils/0.3.3.nix { };

  patchutils_0_4_2 = callPackage ../tools/text/patchutils/0.4.2.nix { };

  parted = callPackage ../tools/misc/parted { };

  passh = callPackage ../tools/networking/passh { };

  paulstretch = callPackage ../applications/audio/paulstretch { };

  pazi = callPackage ../tools/misc/pazi { };

  peep = callPackage ../tools/misc/peep { };

  pell = callPackage ../applications/misc/pell { };

  perccli = callPackage ../tools/misc/perccli { };

  perceptualdiff = callPackage ../tools/graphics/perceptualdiff { };

  inherit (import ../servers/sql/percona-server pkgs) percona-server_lts percona-server_innovation;
  percona-server = percona-server_lts;
  inherit (import ../tools/backup/percona-xtrabackup pkgs) percona-xtrabackup_lts percona-xtrabackup_innovation;
  percona-xtrabackup = percona-xtrabackup_lts;

  pick = callPackage ../tools/misc/pick { };

  pipe-rename = callPackage ../tools/misc/pipe-rename { };

  pipecontrol = libsForQt5.callPackage ../applications/audio/pipecontrol { };

  pipectl = callPackage ../tools/misc/pipectl { };

  pitivi = callPackage ../applications/video/pitivi { };

  prism = callPackage ../applications/video/prism { };

  pulumi = callPackage ../tools/admin/pulumi { };

  pulumiPackages = recurseIntoAttrs (
    callPackage ../tools/admin/pulumi-packages { }
  );

  pulumi-bin = callPackage ../tools/admin/pulumi-bin { };

  p0f = callPackage ../tools/security/p0f { };

  pngloss = callPackage ../tools/graphics/pngloss { };

  pngout = callPackage ../tools/graphics/pngout { };

  patch = gnupatch;

  patchage = callPackage ../applications/audio/patchage { };

  patchance = python3Packages.callPackage ../applications/audio/patchance { };

  patatt = callPackage ../development/tools/patatt { };

  pcapfix = callPackage ../tools/networking/pcapfix { };

  pbzip2 = callPackage ../tools/compression/pbzip2 { };

  pcimem = callPackage ../os-specific/linux/pcimem { };

  pciutils = callPackage ../tools/system/pciutils {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  otpauth = callPackage ../tools/security/otpauth { };

  pcsclite = callPackage ../tools/security/pcsclite {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  pcscliteWithPolkit = pcsclite.override {
    pname = "pcsclite-with-polkit";
    polkitSupport = true;
  };

  pcsc-cyberjack = callPackage ../tools/security/pcsc-cyberjack { };

  pcsc-safenet = callPackage ../tools/security/pcsc-safenet { };

  pcsc-scm-scl011 = callPackage ../tools/security/pcsc-scm-scl011 { };

  pcsc-tools = callPackage ../tools/security/pcsc-tools {
    inherit (pkgs.darwin.apple_sdk.frameworks) PCSC;
  };

  ifdnfc = callPackage ../tools/security/ifdnfc { };

  pdd = python3Packages.callPackage ../tools/misc/pdd { };

  pdf2djvu = callPackage ../tools/typesetting/pdf2djvu { };

  pdfcrack = callPackage ../tools/security/pdfcrack { };

  pdfposter = callPackage ../applications/misc/pdfposter { };

  pdfsandwich = callPackage ../tools/typesetting/pdfsandwich { };

  pdftag = callPackage ../tools/graphics/pdftag { };

  pdf2svg = callPackage ../tools/graphics/pdf2svg { };

  pdftoipe = callPackage ../tools/graphics/pdftoipe { };

  pdfminer = with python3Packages; toPythonApplication pdfminer-six;

  pdf-quench = callPackage ../applications/misc/pdf-quench { };

  pdfarranger = callPackage ../applications/misc/pdfarranger { };

  briss = callPackage ../tools/graphics/briss { };

  brickd = callPackage ../servers/brickd { };

  bully = callPackage ../tools/networking/bully { };

  pcapc = callPackage ../tools/networking/pcapc { };

  pdm = callPackage ../tools/package-management/pdm { };

  pdnsd = callPackage ../tools/networking/pdnsd { };

  peco = callPackage ../tools/text/peco { };

  percollate = callPackage ../tools/text/percollate { };

  pg_activity = callPackage ../development/tools/database/pg_activity { };

  pg_checksums = callPackage ../development/tools/database/pg_checksums { };

  pg_flame = callPackage ../tools/misc/pg_flame { };

  pg_top = callPackage ../tools/misc/pg_top { };

  pgcenter = callPackage ../tools/misc/pgcenter { };

  pgmetrics = callPackage ../tools/misc/pgmetrics { };

  pgsync = callPackage ../development/tools/database/pgsync { };

  pdsh = callPackage ../tools/networking/pdsh {
    rsh = true;          # enable internal rsh implementation
    ssh = openssh;
  };

  pfetch = callPackage ../tools/misc/pfetch { };

  pfetch-rs = callPackage ../tools/misc/pfetch-rs { };

  pfstools = libsForQt5.callPackage ../tools/graphics/pfstools { };

  phoc = callPackage ../applications/misc/phoc {
    wlroots = wlroots_0_17;
  };

  phockup = callPackage ../applications/misc/phockup { };

  phodav = callPackage ../tools/networking/phodav { };

  photon-rss = callPackage ../applications/networking/feedreaders/photon { };

  pim6sd = callPackage ../servers/pim6sd { };

  piper-phonemize = callPackage ../development/libraries/piper-phonemize { };
  piper-train = callPackage ../tools/audio/piper/train.nix { };
  piper-tts = callPackage ../tools/audio/piper { };

  phosh = callPackage ../applications/window-managers/phosh { };

  phosh-mobile-settings = callPackage ../applications/window-managers/phosh/phosh-mobile-settings.nix { };

  piknik = callPackage ../tools/networking/piknik { };

  inherit (callPackages ../tools/security/pinentry { })
    pinentry-curses
    pinentry-emacs
    pinentry-gtk2
    pinentry-gnome3
    pinentry-qt
    pinentry-tty
    pinentry-all;

  pinentry_mac = callPackage ../tools/security/pinentry/mac.nix {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  pinentry-bemenu = callPackage ../tools/security/pinentry-bemenu { };

  pinentry-rofi = callPackage ../tools/security/pinentry-rofi { };

  pingtcp = callPackage ../tools/networking/pingtcp { };

  pingu = callPackage ../tools/networking/pingu { };

  pinnwand = callPackage ../servers/pinnwand { };

  pinsel = callPackage ../tools/misc/pinsel { };

  piping-server-rust = callPackage ../servers/piping-server-rust {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  pirate-get = callPackage ../tools/networking/pirate-get { };

  pipr = callPackage ../applications/misc/pipr { };

  pipreqs = callPackage ../tools/misc/pipreqs { };

  pius = callPackage ../tools/security/pius { };

  pixiewps = callPackage ../tools/networking/pixiewps { };

  pinyin-tool = callPackage ../tools/text/pinyin-tool {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  pk2cmd = callPackage ../tools/misc/pk2cmd { };

  plantuml = callPackage ../tools/misc/plantuml { };

  plantuml-c4 = callPackage ../tools/misc/plantuml/plantuml-c4.nix { };

  plantuml-server = callPackage ../tools/misc/plantuml-server { };

  plan9port = darwin.apple_sdk_11_0.callPackage ../tools/system/plan9port {
    inherit (darwin.apple_sdk_11_0.frameworks) Carbon Cocoa IOKit Metal QuartzCore;
    inherit (darwin) DarwinTools;
  };

  platformioPackages = dontRecurseIntoAttrs (callPackage ../development/embedded/platformio { });
  platformio = if stdenv.isLinux then platformioPackages.platformio-chrootenv else platformioPackages.platformio-core;
  platformio-core = platformioPackages.platformio-core;

  platinum-searcher = callPackage ../tools/text/platinum-searcher { };

  playbar2 = libsForQt5.callPackage ../applications/audio/playbar2 { };

  playwright = with python3Packages; toPythonApplication playwright;

  playwright-driver = callPackage ../development/web/playwright/driver.nix { };
  playwright-test = callPackage ../development/web/playwright-test/wrapped.nix { };

  please = callPackage ../tools/security/please { };

  plecost = callPackage ../tools/security/plecost { };

  plujain-ramp = callPackage ../applications/audio/plujain-ramp { };

  inherit (callPackage ../servers/plik { })
    plik plikd;

  plex = callPackage ../servers/plex { };

  plexRaw = callPackage ../servers/plex/raw.nix { };

  psitransfer = callPackage ../servers/psitransfer { };

  tab = callPackage ../tools/text/tab { };

  tabview = with python3Packages; toPythonApplication tabview;

  tautulli = python3Packages.callPackage ../servers/tautulli { };

  pleroma = callPackage ../servers/pleroma {
    elixir = elixir_1_14;
    beamPackages = beamPackages.extend (self: super: { elixir = elixir_1_14; });
  };

  plfit = callPackage ../tools/misc/plfit {
    python = null;
  };

  ploticus = callPackage ../tools/graphics/ploticus { };

  plotinus = callPackage ../tools/misc/plotinus { };

  plots = callPackage ../applications/misc/plots { };

  plotutils = callPackage ../tools/graphics/plotutils { };

  plowshare = callPackage ../tools/misc/plowshare { };

  pmenu = callPackage ../tools/X11/pmenu { };

  pngcheck = callPackage ../tools/graphics/pngcheck { };

  pngcrush = callPackage ../tools/graphics/pngcrush { };

  pngnq = callPackage ../tools/graphics/pngnq { };

  pngoptimizer = callPackage ../tools/graphics/pngoptimizer { };

  pngtoico = callPackage ../tools/graphics/pngtoico { };

  pngpaste = callPackage ../os-specific/darwin/pngpaste {
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa;
  };

  pngtools = callPackage ../tools/graphics/pngtools { };

  pngpp = callPackage ../development/libraries/png++ { };

  pngquant = callPackage ../tools/graphics/pngquant { };

  po4a = perlPackages.Po4a;

  poac = callPackage ../development/tools/poac {
    inherit (llvmPackages_14) stdenv;
  };

  podiff = callPackage ../tools/text/podiff { };

  pocketbase = callPackage ../servers/pocketbase { };

  podman = callPackage ../applications/virtualization/podman { };

  podman-compose = python3Packages.callPackage ../applications/virtualization/podman-compose { };

  podman-tui = callPackage ../applications/virtualization/podman-tui { };

  podman-desktop = callPackage ../applications/virtualization/podman-desktop {};

  pods = callPackage ../applications/virtualization/pods { };

  pod2mdoc = callPackage ../tools/misc/pod2mdoc { };

  poedit = callPackage ../tools/text/poedit {
    wxGTK32 = wxGTK32.override { withWebKit = true; };
  };

  polaris = callPackage ../servers/polaris { };

  polaris-web = callPackage ../servers/polaris/web.nix { };

  polipo = callPackage ../servers/polipo { };

  polkit_gnome = callPackage ../tools/security/polkit-gnome { };

  poly2tri-c = callPackage ../development/libraries/poly2tri-c { };

  polypane = callPackage ../applications/networking/browsers/polypane { };

  pomsky = callPackage ../tools/text/pomsky { };

  ponysay = callPackage ../tools/misc/ponysay { };

  popfile = callPackage ../tools/text/popfile { };

  poretools = callPackage ../applications/science/biology/poretools { };

  porsmo = callPackage ../applications/misc/porsmo { };

  pantum-driver = callPackage ../misc/drivers/pantum-driver { };

  posteid-seed-extractor = callPackage ../tools/security/posteid-seed-extractor { };

  postscript-lexmark = callPackage ../misc/drivers/postscript-lexmark { };

  povray = callPackage ../tools/graphics/povray {
    # https://github.com/POV-Ray/povray/issues/460
    # https://github.com/NixOS/nixpkgs/issues/311017
    stdenv = gcc12Stdenv;
  };

  power-profiles-daemon = callPackage ../os-specific/linux/power-profiles-daemon { };

  ppl = callPackage ../development/libraries/ppl { };

  pplite = callPackage ../development/libraries/pplite { };

  ppp = callPackage ../tools/networking/ppp { };

  pptp = callPackage ../tools/networking/pptp { };

  pptpd = callPackage ../tools/networking/pptpd { };

  pre-commit = callPackage ../tools/misc/pre-commit { };

  pre-commit-hook-ensure-sops = callPackage ../tools/misc/pre-commit-hook-ensure-sops { };

  pretender = callPackage ../tools/security/pretender { };

  prettierd = callPackage ../development/tools/prettierd { };

  pretty-simple = callPackage ../development/tools/pretty-simple { };

  prettyping = callPackage ../tools/networking/prettyping { };

  pritunl-client = callPackage ../tools/networking/pritunl-client { };

  pritunl-ssh = callPackage ../tools/networking/pritunl-ssh { };

  profile-cleaner = callPackage ../tools/misc/profile-cleaner { };

  profile-sync-daemon = callPackage ../tools/misc/profile-sync-daemon { };

  projectable = callPackage ../applications/file-managers/projectable { };

  projectlibre = callPackage ../applications/misc/projectlibre {
    jre = jre8;
    jdk = jdk8;
  };

  projectm = libsForQt5.callPackage ../applications/audio/projectm { };

  proot = callPackage ../tools/system/proot { };

  protoscope = callPackage ../development/tools/protoscope { };

  prototypejs = callPackage ../development/libraries/prototypejs { };

  proxmark3 = libsForQt5.callPackage ../tools/security/proxmark3/default.nix {
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation AppKit;
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  proxychains = callPackage ../tools/networking/proxychains { };

  proxychains-ng = callPackage ../tools/networking/proxychains-ng { };

  proxify = callPackage ../tools/networking/proxify { };

  proxysql = callPackage ../servers/sql/proxysql { };

  prs = callPackage ../tools/security/prs { };

  psw = callPackage ../tools/misc/psw { };

  pws = callPackage ../tools/misc/pws { };

  cntlm = callPackage ../tools/networking/cntlm { };

  cnping = callPackage ../tools/networking/cnping { };

  past-time = callPackage ../tools/misc/past-time { };

  pastebinit = callPackage ../tools/misc/pastebinit { };

  pmacct = callPackage ../tools/networking/pmacct { };

  pmix = callPackage ../development/libraries/pmix { };

  polygraph = callPackage ../tools/networking/polygraph { };

  pr-tracker = callPackage ../servers/pr-tracker { };

  progress = callPackage ../tools/misc/progress { };

  ps3-disc-dumper = callPackage ../tools/games/ps3-disc-dumper { };

  ps3iso-utils = callPackage ../tools/games/ps3iso-utils { };

  ps3netsrv = callPackage ../servers/ps3netsrv { };

  pscircle = callPackage ../os-specific/linux/pscircle { };

  psitop = callPackage ../applications/system/psitop { };

  psmisc = callPackage ../os-specific/linux/psmisc { };

  pssh = callPackage ../tools/networking/pssh { };

  pspg = callPackage ../tools/misc/pspg { };

  pstoedit = callPackage ../tools/graphics/pstoedit { };

  psutils = callPackage ../tools/typesetting/psutils { };

  psudohash = callPackage ../tools/security/psudohash { };

  psensor = callPackage ../tools/system/psensor {
    libXNVCtrl = linuxPackages.nvidia_x11.settings.libXNVCtrl;
  };

  pubs = callPackage ../tools/misc/pubs { };

  pulldown-cmark = callPackage ../tools/typesetting/pulldown-cmark { };

  pulumictl = callPackage ../development/tools/pulumictl { };

  pure-prompt = callPackage ../shells/zsh/pure-prompt { };

  pv = callPackage ../tools/misc/pv { };

  pwgen = callPackage ../tools/security/pwgen { };

  pwgen-secure = callPackage ../tools/security/pwgen-secure { };

  pwnat = callPackage ../tools/networking/pwnat { };

  pwninit = callPackage ../development/tools/misc/pwninit {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  pycflow2dot = with python3.pkgs; toPythonApplication pycflow2dot;

  pydf = callPackage ../applications/misc/pydf { };

  pyinfra = with python3Packages; toPythonApplication pyinfra;

  pylint = with python3Packages; toPythonApplication pylint;

  pympress = callPackage ../applications/office/pympress { };

  pyocd = with python3Packages; toPythonApplication pyocd;

  pypass = with python3Packages; toPythonApplication pypass;

  teapot = callPackage ../applications/office/teapot { };

  ticktick = callPackage ../applications/office/ticktick { };

  pyditz = callPackage ../applications/misc/pyditz {
    pythonPackages = python3Packages;
  };

  py-spy = darwin.apple_sdk_11_0.callPackage ../development/tools/py-spy { };

  pydeps = with python3Packages; toPythonApplication pydeps;

  pysentation = callPackage ../applications/misc/pysentation { };

  python-launcher = callPackage ../development/tools/misc/python-launcher { };

  pytrainer = callPackage ../applications/misc/pytrainer { };

  pywal = with python3Packages; toPythonApplication pywal;

  pystring = callPackage ../development/libraries/pystring { };

  raysession = python3Packages.callPackage ../applications/audio/raysession {};

  revolt-desktop = callPackage ../applications/networking/instant-messengers/revolt-desktop { };

  rbw = callPackage ../tools/security/rbw { };

  remarshal = with python3Packages; toPythonApplication remarshal;

  rehex = darwin.apple_sdk_11_0.callPackage ../applications/editors/rehex {
    inherit (darwin.apple_sdk_11_0.frameworks) Carbon Cocoa IOKit;
  };

  rio = callPackage ../applications/terminal-emulators/rio { };

  rig = callPackage ../tools/misc/rig { };

  ripdrag = callPackage ../tools/misc/ripdrag { };

  riseup-vpn = libsForQt5.callPackage ../tools/networking/bitmask-vpn {
    provider = "riseup";
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security;
  };

  rnm = callPackage ../tools/filesystems/rnm { };

  rocket = libsForQt5.callPackage ../tools/graphics/rocket { };

  rtabmap = libsForQt5.callPackage ../applications/video/rtabmap/default.nix {
    pcl = pcl.override { vtk = vtkWithQt5; };
  };

  rtaudio = callPackage ../development/libraries/audio/rtaudio {
    jack = libjack2;
    inherit (darwin.apple_sdk.frameworks) CoreAudio;
  };

  rtmidi = callPackage ../development/libraries/audio/rtmidi {
    jack = libjack2;
    inherit (darwin.apple_sdk.frameworks) CoreMIDI CoreAudio CoreServices;
  };

  openmpi = callPackage ../development/libraries/openmpi { };

  ouch = callPackage ../tools/compression/ouch { };

  mpi = openmpi; # this attribute should used to build MPI applications
  mpiCheckPhaseHook = callPackage ../build-support/setup-hooks/mpi-check-hook { };

  ucc = callPackage ../development/libraries/ucc { };

  ucx = callPackage ../development/libraries/ucx { };

  openmodelica = recurseIntoAttrs (callPackage ../applications/science/misc/openmodelica {});

  prowlarr = callPackage ../servers/prowlarr { };

  qarte = libsForQt5.callPackage ../applications/video/qarte { };

  qdrant = darwin.apple_sdk_11_0.callPackage ../servers/search/qdrant {
    inherit (darwin.apple_sdk_11_0.frameworks) Security SystemConfiguration;
  };

  qlcplus = libsForQt5.callPackage ../applications/misc/qlcplus { };

  qlog = qt6Packages.callPackage ../applications/radio/qlog { };

  qnial = callPackage ../development/interpreters/qnial { };

  quickbms = pkgsi686Linux.callPackage ../tools/archivers/quickbms { };

  q-text-as-data = callPackage ../tools/misc/q-text-as-data { };

  qalculate-gtk = callPackage ../applications/science/math/qalculate-gtk { };

  qalculate-qt = qt6Packages.callPackage ../applications/science/math/qalculate-qt { };

  qastools = libsForQt5.callPackage ../tools/audio/qastools { };

  qdigidoc = libsForQt5.callPackage ../tools/security/qdigidoc { } ;

  qhull = callPackage ../development/libraries/qhull { };

  qjournalctl = libsForQt5.callPackage ../applications/system/qjournalctl { };

  qjoypad = libsForQt5.callPackage ../tools/misc/qjoypad { };

  qmk = callPackage ../tools/misc/qmk { };

  qmk_hid = callPackage ../tools/misc/qmk_hid { };

  qmarkdowntextedit = libsForQt5.callPackage  ../development/libraries/qmarkdowntextedit { };

  qodem = callPackage ../tools/networking/qodem { };

  qosmic = libsForQt5.callPackage ../applications/graphics/qosmic { };

  qovery-cli = callPackage ../tools/admin/qovery-cli { };

  qownnotes = qt6Packages.callPackage ../applications/office/qownnotes {
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  qpdf = callPackage ../development/libraries/qpdf { };

  qprint = callPackage ../tools/text/qprint { };

  qrcp = callPackage ../tools/networking/qrcp { };

  qrscan = callPackage ../tools/misc/qrscan { };

  qtikz = libsForQt5.callPackage ../applications/graphics/ktikz { };

  qtspim = libsForQt5.callPackage ../development/tools/misc/qtspim { };

  quadrafuzz = callPackage ../applications/audio/quadrafuzz { };

  quickfix = callPackage ../development/libraries/quickfix { };

  quickjs = callPackage ../development/interpreters/quickjs { };

  quickserve = callPackage ../tools/networking/quickserve { };

  quictls = callPackage ../development/libraries/quictls { };

  quicktun = callPackage ../tools/networking/quicktun { };

  quickwit = callPackage ../servers/search/quickwit {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  quilt = callPackage ../development/tools/quilt { };

  raider = callPackage ../applications/misc/raider { };

  railway = callPackage ../development/tools/railway {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  quota = if stdenv.isLinux then linuxquota else unixtools.quota;

  qvge = libsForQt5.callPackage ../applications/graphics/qvge { };

  qview = libsForQt5.callPackage ../applications/graphics/qview { };

  wayback_machine_downloader = callPackage ../applications/networking/wayback_machine_downloader { };

  wiggle = callPackage ../development/tools/wiggle { };

  radamsa = callPackage ../tools/security/radamsa { };

  radarr = callPackage ../servers/radarr { };

  radeon-profile = libsForQt5.callPackage ../tools/misc/radeon-profile { };

  radsecproxy = callPackage ../tools/networking/radsecproxy { };

  radvd = callPackage ../tools/networking/radvd { };

  rain = callPackage ../development/tools/rain { };

  rainbowstream = with python3.pkgs; toPythonApplication rainbowstream;

  rambox = callPackage ../applications/networking/instant-messengers/rambox { };

  ramfetch = callPackage ../tools/misc/ramfetch { };

  rapidgzip = with python3Packages; toPythonApplication rapidgzip;

  rar = callPackage ../tools/archivers/rar { };

  rarcrack = callPackage ../tools/security/rarcrack { };

  rare-regex = callPackage ../tools/text/rare-regex { };

  ratman = callPackage ../tools/networking/ratman { };

  ratarmount = with python3Packages; toPythonApplication ratarmount;

  ratools = callPackage ../tools/networking/ratools { };

  ratt = callPackage ../applications/misc/ratt { };

  rc-9front = callPackage ../shells/rc-9front { };

  rcon = callPackage ../tools/networking/rcon { };

  rconc = callPackage ../tools/networking/rconc { };

  rdap = callPackage ../tools/networking/rdap { };

  rdbtools = callPackage ../development/tools/rdbtools { python = python3; };

  rdma-core = callPackage ../os-specific/linux/rdma-core { };

  rdrview = callPackage ../tools/networking/rdrview { };

  real_time_config_quick_scan = callPackage ../applications/audio/real_time_config_quick_scan { };

  realesrgan-ncnn-vulkan = callPackage ../tools/graphics/realesrgan-ncnn-vulkan { };

  react-native-debugger = callPackage ../development/tools/react-native-debugger { };

  readarr = callPackage ../servers/readarr { };

  read-edid = callPackage ../os-specific/linux/read-edid { };

  readstat = callPackage ../applications/science/math/readstat { };

  redir = callPackage ../tools/networking/redir { };

  redpanda-client = callPackage ../servers/redpanda { };

  redpanda-server = redpanda-client.server;

  redsocks = callPackage ../tools/networking/redsocks { };

  rep = callPackage ../development/tools/rep { };

  repseek = callPackage ../applications/science/biology/repseek { };

  reredirect = callPackage ../tools/misc/reredirect { };

  retext = qt6Packages.callPackage ../applications/editors/retext { };

  rewrk = callPackage ../tools/networking/rewrk { };

  inherit (callPackage ../tools/security/rekor { })
    rekor-cli
    rekor-server;

  rhai-doc = callPackage ../development/tools/misc/rhai-doc { };

  rich-cli = callPackage ../misc/rich-cli { };

  richgo = callPackage ../development/tools/richgo {  };

  risor = callPackage ../development/interpreters/risor { };

  rlci = callPackage ../development/interpreters/rlci { };

  rst2pdf = with python3Packages; toPythonApplication rst2pdf;

  rstcheck = with python3Packages; toPythonApplication rstcheck;

  rstfmt = callPackage ../development/tools/rstfmt { };

  rt = callPackage ../servers/rt { };

  rtmpdump = callPackage ../tools/video/rtmpdump { };
  rtmpdump_gnutls = rtmpdump.override { gnutlsSupport = true; opensslSupport = false; };

  rtptools = callPackage ../tools/networking/rtptools { };

  rtss = callPackage ../development/tools/misc/rtss { };

  realvnc-vnc-viewer = callPackage ../tools/admin/realvnc-vnc-viewer { };

  re-isearch = callPackage ../applications/search/re-isearch { };

  reason-shell = callPackage ../applications/science/misc/reason-shell { };

  reaverwps = callPackage ../tools/networking/reaver-wps { };

  reaverwps-t6x = callPackage ../tools/networking/reaver-wps-t6x { };

  rx = callPackage ../applications/graphics/rx { };

  qt-box-editor = libsForQt5.callPackage ../applications/misc/qt-box-editor { };

  readability-cli = callPackage ../tools/text/readability-cli { };

  recutils = callPackage ../tools/misc/recutils { };

  recoll = libsForQt5.callPackage ../applications/search/recoll { };

  recoll-nox = recoll.override { withGui = false; };

  redoc-cli = callPackage ../development/tools/redoc-cli { };

  reflex = callPackage ../development/tools/reflex { };

  reiser4progs = callPackage ../tools/filesystems/reiser4progs { };

  reiserfsprogs = callPackage ../tools/filesystems/reiserfsprogs { };

  relic = callPackage ../development/tools/relic { };

  remind = callPackage ../tools/misc/remind { };

  remmina = darwin.apple_sdk_11_0.callPackage ../applications/networking/remote/remmina { };

  rename = callPackage ../tools/misc/rename { };

  renameutils = callPackage ../tools/misc/renameutils { };

  renderdoc = libsForQt5.callPackage ../development/tools/renderdoc { };

  repgrep = callPackage ../tools/text/repgrep { };

  replace = callPackage ../tools/text/replace { };

  resvg = callPackage ../tools/graphics/resvg { };

  reckon = callPackage ../tools/text/reckon { };

  recoverjpeg = callPackage ../tools/misc/recoverjpeg { };

  reftools = callPackage ../development/tools/reftools { };

  redwax-tool = callPackage ../tools/security/redwax-tool { };

  regpg = callPackage ../tools/security/regpg { };

  remote-touchpad = callPackage ../tools/inputmethods/remote-touchpad { };

  remote-exec = python3Packages.callPackage ../tools/misc/remote-exec { };

  reposurgeon = callPackage ../applications/version-management/reposurgeon { };

  reptor = with python3.pkgs; toPythonApplication reptor;

  reptyr = callPackage ../os-specific/linux/reptyr { };

  rescuetime = libsForQt5.callPackage ../applications/misc/rescuetime { };

  inherit (callPackage ../development/misc/resholve { })
    resholve;

  restool = callPackage ../os-specific/linux/restool { };

  reuse = with python3.pkgs; toPythonApplication reuse;

  reveal-md = callPackage ../tools/text/reveal-md { };

  rewritefs = callPackage ../os-specific/linux/rewritefs { };

  rdiff-backup = callPackage ../tools/backup/rdiff-backup { };

  rdfind = callPackage ../tools/filesystems/rdfind { };

  rhash = callPackage ../tools/security/rhash { };

  rhoas = callPackage ../tools/admin/rhoas { };

  riemann_c_client = callPackage ../tools/misc/riemann-c-client { };
  riemann-tools = callPackage ../tools/misc/riemann-tools { };

  ripmime = callPackage ../tools/networking/ripmime { };

  rkflashtool = callPackage ../tools/misc/rkflashtool { };

  rkvm = callPackage ../tools/misc/rkvm { };

  rkrlv2 = callPackage ../applications/audio/rkrlv2 { };

  rmlint = callPackage ../tools/misc/rmlint {
    inherit (python3Packages) sphinx;
  };

  rmw = callPackage ../tools/misc/rmw { };

  rng-tools = callPackage ../tools/security/rng-tools { };

  rnnoise = callPackage ../development/libraries/rnnoise { };

  # Use `apple_sdk_11_0` because `apple_sdk.libs` does not provide `simd`
  rnnoise-plugin = darwin.apple_sdk_11_0.callPackage ../development/libraries/rnnoise-plugin {
    inherit (darwin.apple_sdk_11_0.frameworks) WebKit MetalKit CoreAudioKit;
    inherit (darwin.apple_sdk_11_0.libs) simd;
  };

  rnote = callPackage ../applications/graphics/rnote {
    inherit (gst_all_1) gstreamer;
    inherit (darwin.apple_sdk.frameworks) AudioUnit;
  };

  rnp = callPackage ../tools/security/rnp { };

  rnr = callPackage ../tools/text/rnr { };

  rnv = callPackage ../tools/text/xml/rnv { };

  roam-research = callPackage ../applications/office/roam-research { };

  rosie = callPackage ../tools/text/rosie { };

  rounded-mgenplus = callPackage ../data/fonts/rounded-mgenplus { };

  roundup = callPackage ../tools/misc/roundup { };

  routino = callPackage ../tools/misc/routino { };

  rq = callPackage ../development/tools/rq { };

  rsnapshot = callPackage ../tools/backup/rsnapshot { };

  rlwrap = callPackage ../tools/misc/rlwrap { };

  rmtrash = callPackage ../tools/misc/rmtrash { };

  roc-toolkit = callPackage ../development/libraries/audio/roc-toolkit { };

  rockbox-utility = libsForQt5.callPackage ../tools/misc/rockbox-utility { };

  rosegarden = libsForQt5.callPackage ../applications/audio/rosegarden { };

  rosenpass = callPackage ../tools/networking/rosenpass  { };

  rosenpass-tools = callPackage ../tools/networking/rosenpass/tools.nix  { };

  rot8 = callPackage ../tools/misc/rot8 { };

  rowhammer-test = callPackage ../tools/system/rowhammer-test { };

  rpPPPoE = callPackage ../tools/networking/rp-pppoe { };

  rpi-imager = libsForQt5.callPackage ../tools/misc/rpi-imager { };

  rpiboot = callPackage ../development/misc/rpiboot { };

  rpm = callPackage ../tools/package-management/rpm {
    python = python3;
    lua = lua5_4;
  };

  rpm-ostree = callPackage ../tools/misc/rpm-ostree { };

  rpm2targz = callPackage ../tools/archivers/rpm2targz { };

  rpmextract = callPackage ../tools/archivers/rpmextract { };

  rrdtool = callPackage ../tools/misc/rrdtool { };

  rscw = callPackage ../applications/radio/rscw { };

  rset = callPackage ../tools/admin/rset { };

  rshijack = callPackage ../tools/networking/rshijack { };

  rshim-user-space = callPackage ../tools/misc/rshim-user-space { };

  rsibreak = libsForQt5.callPackage ../applications/misc/rsibreak { };

  rslint = callPackage ../development/tools/rslint { };

  rss-bridge-cli = callPackage ../applications/misc/rss-bridge-cli { };

  rss2email = callPackage ../applications/networking/feedreaders/rss2email {
    pythonPackages = python3Packages;
  };

  feed2imap-go = callPackage ../applications/networking/feedreaders/feed2imap-go { };

  rsstail = callPackage ../applications/networking/feedreaders/rsstail { };

  rtz = callPackage ../tools/misc/rtz { };

  rubber = callPackage ../tools/typesetting/rubber { };

  rubocop = rubyPackages.rubocop;

  ruby-lsp = rubyPackages.ruby-lsp;

  runningx = callPackage ../tools/X11/runningx { };

  rund = callPackage ../development/tools/rund { };

  runme = callPackage ../development/tools/misc/runme { };

  runzip = callPackage ../tools/archivers/runzip { };

  unzoo = callPackage ../tools/archivers/unzoo { };

  ruplacer = callPackage ../tools/text/ruplacer {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  rust-code-analysis = callPackage ../development/tools/rust-code-analysis { };

  rust-motd = callPackage ../tools/misc/rust-motd {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  rust-petname = callPackage ../tools/text/rust-petname { };

  rustcat = callPackage ../tools/networking/rustcat {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  rustdesk = callPackage ../applications/networking/remote/rustdesk { };

  rustfilt = callPackage ../development/tools/rust/rustfilt { };

  rustscan = callPackage ../tools/security/rustscan {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  rustdesk-server = callPackage ../servers/rustdesk-server {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  rustypaste = callPackage ../servers/rustypaste { };

  rustypaste-cli = callPackage ../tools/misc/rustypaste-cli { };

  rustywind = callPackage ../development/tools/misc/rustywind { };

  rw = callPackage ../tools/misc/rw { };

  rwc = callPackage ../tools/system/rwc { };

  rwedid = callPackage ../tools/video/rwedid { };

  rxp = callPackage ../tools/text/xml/rxp { };

  rzip = callPackage ../tools/compression/rzip { };

  s-tui = callPackage ../tools/system/s-tui { };

  s3backer = callPackage ../tools/filesystems/s3backer { };

  s3bro = callPackage ../tools/admin/s3bro { };

  s3fs = darwin.apple_sdk_11_0.callPackage ../tools/filesystems/s3fs { };

  s3cmd = python3Packages.callPackage ../tools/networking/s3cmd { };

  s3rs = callPackage ../tools/networking/s3rs {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  s3-credentials = with python3Packages; toPythonApplication s3-credentials;

  s4cmd = callPackage ../tools/networking/s4cmd { };

  s5 = callPackage ../tools/security/s5 { };

  s5cmd = callPackage ../tools/networking/s5cmd { };

  sacad = callPackage ../tools/misc/sacad { };

  sad = callPackage ../tools/text/sad { };

  safecopy = callPackage ../tools/system/safecopy { };

  sacd = callPackage ../tools/cd-dvd/sacd { };

  safe = callPackage ../tools/security/safe { };

  safety-cli = with python3.pkgs; toPythonApplication safety;

  safe-rm = callPackage ../tools/system/safe-rm { };

  safecloset = callPackage ../applications/misc/safecloset { };

  safeeyes = callPackage ../applications/misc/safeeyes { };

  sagoin = callPackage ../tools/misc/sagoin { };

  sahel-fonts = callPackage ../data/fonts/sahel-fonts { };

  saldl = callPackage ../tools/networking/saldl { };

  salt = callPackage ../tools/admin/salt { };

  samim-fonts = callPackage ../data/fonts/samim-fonts { };

  saml2aws = callPackage ../tools/security/saml2aws {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  sammler = callPackage ../tools/security/sammler { };

  samplicator = callPackage ../tools/networking/samplicator { };

  sanctity = callPackage ../tools/misc/sanctity { };

  sandboxfs = callPackage ../tools/filesystems/sandboxfs { };

  sanjuuni = callPackage ../tools/graphics/sanjuuni { };

  sasquatch = callPackage ../tools/filesystems/sasquatch { };

  sasview = libsForQt5.callPackage ../applications/science/misc/sasview { };

  sbs = callPackage ../tools/X11/sbs { };

  schemes = callPackage ../applications/misc/schemes { };

  scanbd = callPackage ../tools/graphics/scanbd { };

  scdl = callPackage ../tools/misc/scdl { };

  scorecard = callPackage ../tools/security/scorecard { };

  scream = callPackage ../applications/audio/scream { };

  screen = callPackage ../tools/misc/screen {
    inherit (darwin.apple_sdk.libs) utmp;
  };

  scrcpy = callPackage ../misc/scrcpy { };

  screen-message = callPackage ../tools/X11/screen-message { };

  screenkey = callPackage ../applications/video/screenkey { };

  scrub = callPackage ../tools/misc/scrub { };

  scfbuild = python3.pkgs.callPackage ../tools/misc/scfbuild { };

  scraper = callPackage ../tools/text/scraper { };

  scriptaculous = callPackage ../development/libraries/scriptaculous { };

  script-directory = callPackage ../tools/misc/script-directory { };

  scrot = callPackage ../tools/graphics/scrot { };

  scrypt = callPackage ../tools/security/scrypt { };

  sd = callPackage ../tools/text/sd {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  sd-mux-ctrl = callPackage ../tools/misc/sd-mux-ctrl { };

  sd-switch = callPackage ../os-specific/linux/sd-switch { };

  sdate = callPackage ../tools/misc/sdate { };

  sdat2img = callPackage ../tools/filesystems/sdat2img { };

  sdcv = callPackage ../applications/misc/sdcv { };

  sdl-jstest = callPackage ../tools/misc/sdl-jstest { };

  senpai = callPackage ../applications/networking/irc/senpai { };

  skim = callPackage ../tools/misc/skim { };

  seashells = callPackage ../applications/misc/seashells { };

  seaweedfs = callPackage ../applications/networking/seaweedfs { };

  sec = callPackage ../tools/admin/sec { };

  seccure = callPackage ../tools/security/seccure { };

  secp256k1 = callPackage ../tools/security/secp256k1 { };

  securefs = darwin.apple_sdk_11_0.callPackage ../tools/filesystems/securefs { };

  seehecht = callPackage ../tools/text/seehecht { };

  selectdefaultapplication = libsForQt5.callPackage ../applications/misc/selectdefaultapplication { };

  semantic-release = callPackage ../development/tools/semantic-release {
    inherit (darwin) cctools;
  };

  semgrep = python3.pkgs.callPackage ../tools/security/semgrep { };
  semgrep-core = callPackage ../tools/security/semgrep/semgrep-core.nix { };

  setroot = callPackage  ../tools/X11/setroot { };

  setserial = callPackage ../tools/system/setserial { };

  setzer = callPackage ../applications/editors/setzer { };

  seqdiag = with python3Packages; toPythonApplication seqdiag;

  sequoia-sqv = callPackage ../tools/security/sequoia-sqv { };

  sequoia-sqop = callPackage ../tools/security/sequoia-sqop { };

  sequoia-chameleon-gnupg = callPackage ../tools/security/sequoia-chameleon-gnupg { };

  sewer = callPackage ../tools/admin/sewer { };

  sexpp = callPackage ../development/libraries/sexpp { };

  sfeed = callPackage ../tools/misc/sfeed { };

  sftpman = callPackage ../tools/filesystems/sftpman { };

  sftpgo = callPackage ../servers/sftpgo { };

  screenfetch = callPackage ../tools/misc/screenfetch { };

  sg3_utils = callPackage ../tools/system/sg3_utils { };

  sha1collisiondetection = callPackage ../tools/security/sha1collisiondetection { };

  shadowsocks-libev = callPackage ../tools/networking/shadowsocks-libev { };

  shadered = callPackage ../development/tools/shadered { };

  go-shadowsocks2 = callPackage ../tools/networking/go-shadowsocks2 { };

  shabnam-fonts = callPackage ../data/fonts/shabnam-fonts { };

  shadowsocks-rust = callPackage ../tools/networking/shadowsocks-rust {
    inherit (darwin.apple_sdk.frameworks) Security CoreServices;
  };

  shadowsocks-v2ray-plugin = callPackage ../tools/networking/shadowsocks-v2ray-plugin { };

  sharutils = callPackage ../tools/archivers/sharutils { };

  shell2http = callPackage ../servers/misc/shell2http { };

  shelldap = callPackage ../tools/misc/shelldap { };

  shellify = haskellPackages.shellify.bin;

  shellspec = callPackage ../tools/misc/shellspec { };

  schema2ldif = callPackage ../tools/text/schema2ldif { };

  schemacrawler = callPackage ../development/tools/schemacrawler { };

  sharedown = callPackage ../tools/misc/sharedown { };

  shen-sbcl = callPackage ../development/interpreters/shen-sbcl { };

  shen-sources = callPackage ../development/interpreters/shen-sources { };

  shiv = with python3Packages; toPythonApplication shiv;

  shim-unsigned = callPackage ../tools/misc/shim { };

  shocco = callPackage ../tools/text/shocco { };

  shopify-cli = callPackage ../development/web/shopify-cli { };

  shopify-themekit = callPackage ../development/web/shopify-themekit { };

  shorewall = callPackage ../tools/networking/shorewall { };

  shotwell = callPackage ../applications/graphics/shotwell { };

  shout = nodePackages.shout;

  shrikhand = callPackage ../data/fonts/shrikhand { };

  shunit2 = callPackage ../tools/misc/shunit2 { };

  sic = callPackage ../applications/networking/irc/sic { };

  siege = callPackage ../tools/networking/siege { };

  sieve-connect = callPackage ../applications/networking/sieve-connect { };

  sigal = callPackage ../applications/misc/sigal { };

  sigil = libsForQt5.callPackage ../applications/editors/sigil { };

  signalbackup-tools = darwin.apple_sdk_11_0.callPackage
    ../applications/networking/instant-messengers/signalbackup-tools { };

  signald = callPackage ../applications/networking/instant-messengers/signald { };

  signaldctl = callPackage ../applications/networking/instant-messengers/signaldctl { };

  signal-cli = callPackage ../applications/networking/instant-messengers/signal-cli { };

  inherit (callPackage ../applications/networking/instant-messengers/signal-desktop {}) signal-desktop signal-desktop-beta;

  slither-analyzer = with python3Packages; toPythonApplication slither-analyzer;

  sigma-cli = callPackage ../tools/security/sigma-cli { };

  signify = callPackage ../tools/security/signify { };

  # aka., pgp-tools
  signing-party = callPackage ../tools/security/signing-party { };

  sigtop = callPackage ../tools/backup/sigtop { };

  silc_client = callPackage ../applications/networking/instant-messengers/silc-client { };

  silc_server = callPackage ../servers/silc-server { };

  sile = callPackage ../tools/typesetting/sile {
    lua = lua5_3;
  };

  silenthound = callPackage ../tools/security/silenthound { };

  silice = callPackage ../development/compilers/silice { };

  silver-searcher = callPackage ../tools/text/silver-searcher { };

  simple-mtpfs = callPackage ../tools/filesystems/simple-mtpfs { };

  simpleproxy = callPackage ../tools/networking/simpleproxy { };

  simplescreenrecorder = libsForQt5.callPackage ../applications/video/simplescreenrecorder { };

  sipexer = callPackage ../tools/networking/sipexer { };

  sipsak = callPackage ../tools/networking/sipsak { };

  sipvicious = callPackage ../tools/security/sipvicious { };

  siril = callPackage ../applications/science/astronomy/siril { };

  sisco.lv2 = callPackage ../applications/audio/sisco.lv2 { };

  sixpair = callPackage ../tools/misc/sixpair { };

  skippy-xd = callPackage ../tools/X11/skippy-xd { };

  sks = callPackage ../servers/sks {
    ocamlPackages = ocaml-ng.ocamlPackages_4_12;
  };

  skydns = callPackage ../servers/skydns { };

  sing-box = callPackage ../tools/networking/sing-box { };

  sing-geosite = callPackage ../data/misc/sing-geosite { };

  sing-geoip = callPackage ../data/misc/sing-geoip { };

  sipcalc = callPackage ../tools/networking/sipcalc { };

  skribilo = callPackage ../tools/typesetting/skribilo { };

  skytemple = callPackage ../applications/misc/skytemple { };

  sleuthkit = callPackage ../tools/system/sleuthkit { };

  slides = callPackage ../applications/misc/slides { };

  slippy = callPackage ../applications/misc/slippy { };

  slirp4netns = callPackage ../tools/networking/slirp4netns { };

  slowlorust = callPackage ../tools/networking/slowlorust {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  slsa-verifier = callPackage ../tools/security/slsa-verifier { };

  slsnif = callPackage ../tools/misc/slsnif { };

  slstatus = callPackage ../applications/misc/slstatus {
    conf = config.slstatus.conf or null;
  };

  smartdns = callPackage ../tools/networking/smartdns { };

  smartmontools = callPackage ../tools/system/smartmontools {
    inherit (darwin.apple_sdk.frameworks) IOKit ApplicationServices;
  };

  smarty3 = callPackage ../development/libraries/smarty3 { };
  smarty3-i18n = callPackage ../development/libraries/smarty3-i18n { };

  smbnetfs = callPackage ../tools/filesystems/smbnetfs { };

  smenu = callPackage ../tools/misc/smenu { };

  boost-sml = callPackage ../development/libraries/boost-ext/boost-sml { };

  smu = callPackage ../tools/text/smu { };

  smug = callPackage ../tools/misc/smug { };

  smpq = callPackage ../by-name/sm/smpq/package.nix {
    stormlib = stormlib.overrideAttrs (old: {
      version = "9.22";
      src = fetchFromGitHub {
        owner = "ladislav-zezula";
        repo = "StormLib";
        rev = "v9.22";
        hash = "sha256-jFUfxLzuRHAvFE+q19i6HfGcL6GX4vKL1g7l7LOhjeU=";
      };
    });
  };

  snabb = callPackage ../tools/networking/snabb { };

  snallygaster = callPackage ../tools/security/snallygaster { };

  snapcast = darwin.apple_sdk_11_0.callPackage ../applications/audio/snapcast {
    inherit (darwin.apple_sdk_11_0.frameworks) IOKit AudioToolbox;
    pulseaudioSupport = config.pulseaudio or stdenv.isLinux;
  };

  snapdragon-profiler = callPackage ../tools/graphics/snapdragon-profiler { };

  snet = callPackage ../tools/networking/snet { };

  sng = callPackage ../tools/graphics/sng { };

  snmpcheck = callPackage ../tools/networking/snmpcheck { };

  snobol4 = callPackage ../development/interpreters/snobol4 { };

  snort = callPackage ../applications/networking/ids/snort { };

  so = callPackage ../development/tools/so {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  soapui = callPackage ../applications/networking/soapui {
    jdk = if stdenv.isDarwin
      then (jdk11.override { enableJavaFX = true; })
      else jdk11;
  };

  sockdump = callPackage ../tools/networking/sockdump { };

  specup = haskellPackages.specup.bin;

  spglib = callPackage ../development/libraries/spglib {
    inherit (llvmPackages) openmp;
  };

  spicy = callPackage ../development/tools/spicy { };

  spire = callPackage ../tools/security/spire { };
  # to match naming of other package repositories
  spire-agent = spire.agent;
  spire-server = spire.server;

  spoof-mac = python3Packages.callPackage ../tools/networking/spoof-mac { };

  sqlboiler = callPackage ../development/tools/sqlboiler { };

  sshed = callPackage ../tools/networking/sshed { };

  sshguard = callPackage ../tools/security/sshguard { };

  sshping = callPackage ../tools/networking/sshping { };

  ssh-chat = callPackage ../applications/networking/instant-messengers/ssh-chat { };

  ssh-to-age = callPackage ../tools/security/ssh-to-age { };

  ssh-to-pgp = callPackage ../tools/security/ssh-to-pgp { };

  suricata = callPackage ../applications/networking/ids/suricata {
    python = python3;
    libbpf = libbpf_0;
  };

  sof-firmware = callPackage ../os-specific/linux/firmware/sof-firmware { };

  softhsm = callPackage ../tools/security/softhsm {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  sonar-scanner-cli = callPackage ../tools/security/sonar-scanner-cli { };

  snapshot = callPackage ../applications/graphics/snapshot { };

  solvespace = callPackage ../applications/graphics/solvespace { };

  sonarr = callPackage ../servers/sonarr { };

  sonata = callPackage ../applications/audio/sonata { };

  sony-headphones-client = callPackage ../applications/audio/sony-headphones-client { };

  soundconverter = callPackage ../applications/audio/soundconverter { };

  soundkonverter = libsForQt5.soundkonverter;

  soundwireserver = callPackage ../applications/audio/soundwireserver { };

  sozu = callPackage ../servers/sozu { };

  spacer = callPackage ../tools/misc/spacer { };

  sparrow-unwrapped = callPackage ../applications/blockchains/sparrow {
    openimajgrabber = callPackage ../applications/blockchains/sparrow/openimajgrabber.nix {};
    openjdk = openjdk21.override { enableJavaFX = true; };
  };

  sparrow = callPackage ../applications/blockchains/sparrow/fhsenv.nix { };

  sparsehash = callPackage ../development/libraries/sparsehash { };

  spectre-meltdown-checker = callPackage ../tools/security/spectre-meltdown-checker { };

  spiped = callPackage ../tools/networking/spiped { };

  sqlite3-to-mysql = callPackage ../tools/misc/sqlite3-to-mysql { };

  sqls = callPackage ../applications/misc/sqls { };

  stdman = callPackage ../data/documentation/stdman { };

  stderred = callPackage ../tools/misc/stderred { };

  steck = callPackage ../servers/pinnwand/steck.nix { };

  stenc = callPackage ../tools/backup/stenc { };

  stm32loader = with python3Packages; toPythonApplication stm32loader;

  stremio = qt5.callPackage ../applications/video/stremio { };

  sunwait = callPackage ../applications/misc/sunwait { };

  surface-control = callPackage ../applications/misc/surface-control { };

  syntex = callPackage ../tools/graphics/syntex { };

  synapse-admin = callPackage ../tools/admin/synapse-admin { };

  sl = callPackage ../tools/misc/sl { };

  socat = callPackage ../tools/networking/socat { };

  sockperf = callPackage ../tools/networking/sockperf { };

  softnet = callPackage ../tools/networking/softnet { };

  solaar = callPackage ../applications/misc/solaar { };

  solanum = callPackage ../servers/irc/solanum {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  solc-select = with python3Packages; toPythonApplication solc-select;

  sourceHighlight = callPackage ../tools/text/source-highlight { };

  somebar = callPackage ../applications/misc/somebar { };

  spacebar = callPackage ../os-specific/darwin/spacebar {
    inherit (darwin.apple_sdk.frameworks)
      Carbon Cocoa ScriptingBridge SkyLight;
  };

  speech-denoiser = callPackage ../applications/audio/speech-denoiser { };

  splot = haskell.lib.compose.justStaticExecutables haskellPackages.splot;

  spotdl = callPackage ../tools/audio/spotdl { };

  squashfsTools = callPackage ../tools/filesystems/squashfs { };

  squashfs-tools-ng = darwin.apple_sdk_11_0.callPackage ../tools/filesystems/squashfs-tools-ng { };

  squashfuse = callPackage ../tools/filesystems/squashfuse { };

  srt-live-server = callPackage ../applications/video/srt-live-server { };

  srt-to-vtt-cl = callPackage ../tools/cd-dvd/srt-to-vtt-cl { };

  srtrelay = callPackage ../applications/video/srtrelay { };

  srsran = callPackage ../applications/radio/srsran {  };

  sourcehut = callPackage ../applications/version-management/sourcehut { };

  sshfs-fuse = callPackage ../tools/filesystems/sshfs-fuse { };
  sshfs = sshfs-fuse; # added 2017-08-14

  sshlatex = callPackage ../tools/typesetting/sshlatex { };

  sshuttle = callPackage ../tools/security/sshuttle { };

  inherit (callPackages ../tools/misc/sshx { })
    sshx
    sshx-server;

  ssldump = callPackage ../tools/networking/ssldump { };

  sslsplit = callPackage ../tools/networking/sslsplit { };

  sstp = callPackage ../tools/networking/sstp { };

  strip-nondeterminism = perlPackages.strip-nondeterminism;

  structorizer = callPackage ../applications/graphics/structorizer { };

  su-exec = callPackage ../tools/security/su-exec { };

  subberthehut = callPackage ../tools/misc/subberthehut { };

  subedit = callPackage ../tools/text/subedit { };

  subsurface = libsForQt5.callPackage ../applications/misc/subsurface { };

  sudo = callPackage ../tools/security/sudo { };

  sudo-rs = callPackage ../tools/security/sudo-rs { };

  suidChroot = callPackage ../tools/system/suid-chroot { };

  sundtek = callPackage ../misc/drivers/sundtek { };

  sunxi-tools = callPackage ../development/tools/sunxi-tools { };

  sumorobot-manager = python3Packages.callPackage ../applications/science/robotics/sumorobot-manager { };

  supertag = callPackage ../tools/filesystems/supertag { };

  svgbob = callPackage ../tools/graphics/svgbob { };

  svgcleaner = callPackage ../tools/graphics/svgcleaner { };

  svu = callPackage ../tools/misc/svu { };

  ssb-patchwork = callPackage ../applications/networking/ssb-patchwork { };

  ssdeep = callPackage ../tools/security/ssdeep { };

  ssh-ident = callPackage ../tools/networking/ssh-ident { };

  sshpass = callPackage ../tools/networking/sshpass { };

  sslscan = callPackage ../tools/security/sslscan {
    openssl = openssl.override { withZlib = true; };
  };

  sslmate = callPackage ../development/tools/sslmate { };

  sslmate-agent = callPackage ../development/tools/sslmate-agent { };

  sshocker = callPackage ../tools/security/sshocker { };

  sshoogr = callPackage ../tools/networking/sshoogr { };

  ssocr = callPackage ../applications/misc/ssocr { };

  sss-cli = callPackage ../tools/security/sss-cli { };

  ssss = callPackage ../tools/security/ssss { };

  stabber = callPackage ../misc/stabber { };

  stacer = libsForQt5.callPackage ../tools/system/stacer { };

  starcharts = callPackage ../servers/misc/starcharts { };

  staticjinja = with python3.pkgs; toPythonApplication staticjinja;

  stevenblack-blocklist  = callPackage ../tools/networking/stevenblack-blocklist { };

  stress = callPackage ../tools/system/stress { };

  stress-ng = callPackage ../tools/system/stress-ng { };

  stressapptest = callPackage ../tools/system/stressapptest { };

  stoken = callPackage ../tools/security/stoken (config.stoken or {});

  storeBackup = callPackage ../tools/backup/store-backup { };

  stow = callPackage ../tools/misc/stow { };
  xstow = callPackage ../tools/misc/xstow { };

  stun = callPackage ../tools/networking/stun { };

  stunnel = callPackage ../tools/networking/stunnel { };

  stuntman = callPackage ../tools/networking/stuntman { };

  stutter = haskell.lib.compose.justStaticExecutables haskellPackages.stutter;

  strongswan    = callPackage ../tools/networking/strongswan { };
  strongswanTNC = strongswan.override { enableTNC = true; };
  strongswanNM  = strongswan.override { enableNetworkManager = true; };

  stylish = callPackage ../applications/misc/stylish { };

  stylish-haskell = haskell.lib.compose.justStaticExecutables haskellPackages.stylish-haskell;

  su = shadow.su;

  subjs = callPackage ../tools/security/subjs { };

  subsonic = callPackage ../servers/misc/subsonic { };

  subfinder = callPackage ../tools/networking/subfinder { };

  subzerod = with python3Packages; toPythonApplication subzerod;

  suckit = callPackage ../tools/networking/suckit {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  surfraw = callPackage ../tools/networking/surfraw { };

  swagger-cli = callPackage ../tools/networking/swagger-cli { };

  swagger-codegen = callPackage ../tools/networking/swagger-codegen { };

  swagger-codegen3 = callPackage ../tools/networking/swagger-codegen3 { };

  swaggerhole = callPackage ../tools/security/swaggerhole { };

  swapview = callPackage ../os-specific/linux/swapview { };

  swc = callPackage ../development/tools/swc { };

  swtpm = callPackage ../tools/security/swtpm { };

  svnfs = callPackage ../tools/filesystems/svnfs { };

  svtplay-dl = callPackage ../tools/misc/svtplay-dl { };

  sycl-info = callPackage ../development/libraries/sycl-info { };

  symengine = callPackage ../development/libraries/symengine { };

  synaesthesia = callPackage ../applications/audio/synaesthesia { };

  sysbench = callPackage ../development/tools/misc/sysbench { };

  systemc = callPackage ../applications/science/electronics/systemc { };

  system-config-printer = callPackage ../tools/misc/system-config-printer {
    autoreconfHook = buildPackages.autoreconfHook269;
    libxml2 = libxml2Python;
  };

  systembus-notify = callPackage ../applications/misc/systembus-notify { };

  stricat = callPackage ../tools/security/stricat { };

  staruml = callPackage ../tools/misc/staruml { };

  stone-phaser = callPackage ../applications/audio/stone-phaser { };

  systrayhelper = callPackage ../tools/misc/systrayhelper { };

  Sylk = callPackage ../applications/networking/Sylk { };

  privoxy = callPackage ../tools/networking/privoxy {
    w3m = w3m-batch;
  };

  swaks = callPackage ../tools/networking/swaks { };

  swiftbar = callPackage ../os-specific/darwin/swiftbar { };

  swiften = callPackage ../development/libraries/swiften { };

  squeekboard = callPackage ../applications/accessibility/squeekboard { };

  systemdgenie = libsForQt5.callPackage ../applications/system/systemdgenie { };

  t = callPackage ../tools/misc/t { };

  tabnine = callPackage ../development/tools/tabnine { };

  tab-rs = callPackage ../tools/misc/tab-rs {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  tagtime = callPackage ../applications/misc/tagtime { };

  tailer = callPackage ../tools/misc/tailer { };

  tandoor-recipes = callPackage ../applications/misc/tandoor-recipes { };

  tango = callPackage ../applications/misc/tango { };

  tangram = callPackage ../applications/networking/instant-messengers/tangram { };

  t1utils = callPackage ../tools/misc/t1utils { };

  talkfilters = callPackage ../misc/talkfilters { };

  znapzend = callPackage ../tools/backup/znapzend { };

  tar2ext4 = callPackage ../tools/filesystems/tar2ext4 { };

  targetcli = callPackage ../os-specific/linux/targetcli { };

  target-isns = callPackage ../os-specific/linux/target-isns { };

  tarsnap = callPackage ../tools/backup/tarsnap { };

  tarsnapper = callPackage ../tools/backup/tarsnapper { };

  tarantool = callPackage ../servers/tarantool { };

  tarssh = callPackage ../servers/tarssh { };

  tartube = callPackage ../applications/video/tartube { };

  tartube-yt-dlp = callPackage ../applications/video/tartube {
    youtube-dl = yt-dlp;
  };

  tayga = callPackage ../tools/networking/tayga { };

  tcb = callPackage ../tools/security/tcb { };

  tcpcrypt = callPackage ../tools/security/tcpcrypt { };

  tcptraceroute = callPackage ../tools/networking/tcptraceroute { };

  tboot = callPackage ../tools/security/tboot { };

  tagutil = callPackage ../applications/audio/tagutil { };

  tcpdump = callPackage ../tools/networking/tcpdump { };

  tcpflow = callPackage ../tools/networking/tcpflow { };

  tcpkali = callPackage ../applications/networking/tcpkali { };

  tcpreplay = callPackage ../tools/networking/tcpreplay {
    inherit (darwin.apple_sdk.frameworks) Carbon CoreServices;
  };

  tdns-cli = callPackage ../tools/networking/tdns-cli { };

  tea = callPackage ../tools/misc/tea { };

  teavpn2 = callPackage ../tools/networking/teavpn2 { };

  inherit (nodePackages) teck-programmer;

  ted = callPackage ../tools/typesetting/ted { };

  teamviewer = libsForQt5.callPackage ../applications/networking/remote/teamviewer { };

  teehee = callPackage ../applications/editors/teehee { };

  teip = callPackage ../tools/text/teip { };

  telegraf = callPackage ../servers/monitoring/telegraf { };

  inherit (callPackages ../servers/teleport {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security AppKit;
  }) teleport_14 teleport_15 teleport;

  telepresence = callPackage ../tools/networking/telepresence {
    pythonPackages = python3Packages;
  };

  telepresence2 = callPackage ../tools/networking/telepresence2 { };

  teler = callPackage ../tools/security/teler { };

  termcolor = callPackage ../development/libraries/termcolor { };

  termscp = callPackage ../tools/networking/termscp {
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa Foundation Security;
  };

  termius = callPackage ../applications/networking/termius { };

  termplay = callPackage ../tools/misc/termplay { };

  tetrd = callPackage ../applications/networking/tetrd { };

  tewisay = callPackage ../tools/misc/tewisay { };

  texmacs = libsForQt5.callPackage ../applications/editors/texmacs {
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
    extraFonts = true;
  };

  texmaker = libsForQt5.callPackage ../applications/editors/texmaker { };

  texstudio = qt6Packages.callPackage ../applications/editors/texstudio { };

  textadept = libsForQt5.callPackage ../applications/editors/textadept { };

  texworks = qt6Packages.callPackage ../applications/editors/texworks { };

  tf2pulumi = callPackage ../development/tools/tf2pulumi { };

  thc-hydra = callPackage ../tools/security/thc-hydra { };

  thc-ipv6 = callPackage ../tools/security/thc-ipv6 { };

  thedesk = callPackage ../applications/misc/thedesk { };

  theharvester = callPackage ../tools/security/theharvester { };

  thelounge = callPackage ../applications/networking/irc/thelounge { };

  theLoungePlugins = with lib; let
    pkgs = filterAttrs (name: _: hasPrefix "thelounge-" name) nodePackages;
    getPackagesWithPrefix = prefix: mapAttrs' (name: pkg: nameValuePair (removePrefix ("thelounge-" + prefix + "-") name) pkg)
      (filterAttrs (name: _: hasPrefix ("thelounge-" + prefix + "-") name) pkgs);
  in
  recurseIntoAttrs {
    plugins = recurseIntoAttrs (getPackagesWithPrefix "plugin");
    themes = recurseIntoAttrs (getPackagesWithPrefix "theme");
  };

  thefuck = python3Packages.callPackage ../tools/misc/thefuck { };

  theme-sh = callPackage ../tools/misc/theme-sh { };

  thiefmd = callPackage ../applications/editors/thiefmd { };

  thin-provisioning-tools = callPackage ../tools/misc/thin-provisioning-tools {  };

  thinkpad-scripts = python3.pkgs.callPackage ../tools/misc/thinkpad-scripts { };

  threatest = callPackage ../tools/security/threatest {  };

  threema-desktop = callPackage ../applications/networking/instant-messengers/threema-desktop { };

  thumbdrives = callPackage ../applications/system/thumbdrives { };

  tidy-viewer = callPackage ../tools/text/tidy-viewer { };

  tiled = libsForQt5.callPackage ../applications/editors/tiled { };

  tiledb = callPackage ../development/libraries/tiledb { };

  timemachine = callPackage ../applications/audio/timemachine { };

  timelapse-deflicker = callPackage ../applications/graphics/timelapse-deflicker { };

  timelens = callPackage ../applications/video/timelens { };

  timetrap = callPackage ../applications/office/timetrap { };

  timetagger = callPackage ../servers/timetagger { };

  timetagger_cli = callPackage ../tools/misc/timetagger_cli { };

  timezonemap = callPackage ../development/libraries/timezonemap { };

  tzupdate = callPackage ../applications/misc/tzupdate { };

  tinc = callPackage ../tools/networking/tinc { };

  tie = callPackage ../development/tools/misc/tie { };

  tidb = callPackage ../servers/tidb { };

  tikzit = libsForQt5.callPackage ../tools/typesetting/tikzit { };

  tinc_pre = callPackage ../tools/networking/tinc/pre.nix { };

  tinycbor = callPackage ../development/libraries/tinycbor { };

  tinyfecvpn = callPackage ../tools/networking/tinyfecvpn { };

  tinyobjloader = callPackage ../development/libraries/tinyobjloader { };

  tinyprog = callPackage ../development/embedded/fpga/tinyprog { };

  tinyproxy = callPackage ../tools/networking/tinyproxy { };

  time-decode = callPackage ../tools/misc/time-decode { };

  timer = callPackage ../tools/misc/timer { };

  tio = callPackage ../tools/misc/tio { };

  tiv = callPackage ../applications/misc/tiv { };

  tkman = callPackage ../tools/misc/tkman { };

  tldr = callPackage ../tools/misc/tldr { };

  tldr-hs = haskellPackages.tldr;

  tlsclient = callPackage ../tools/admin/tlsclient { };

  tlsx = callPackage ../tools/security/tlsx { };

  tmate = callPackage ../tools/misc/tmate { };

  tmate-ssh-server = callPackage ../servers/tmate-ssh-server { };

  tml = callPackage ../tools/text/tml { };

  tmpwatch = callPackage ../tools/misc/tmpwatch  { };

  tmpmail = callPackage ../applications/networking/tmpmail { };

  tmux = callPackage ../tools/misc/tmux { };

  tmux-cssh = callPackage ../tools/misc/tmux-cssh { };

  tmuxp = callPackage ../tools/misc/tmuxp { };

  tmuxinator = callPackage ../tools/misc/tmuxinator { };

  tmux-mem-cpu-load = callPackage ../tools/misc/tmux-mem-cpu-load { };

  tmux-sessionizer = callPackage ../tools/misc/tmux-sessionizer {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  tmux-xpanes = callPackage ../tools/misc/tmux-xpanes { };

  tmuxPlugins = recurseIntoAttrs (callPackage ../misc/tmux-plugins {
    pkgs = pkgs.__splicedPackages;
  });

  tmsu = callPackage ../tools/filesystems/tmsu { };

  tncattach = callPackage ../applications/radio/tncattach { };

  to-html = callPackage ../tools/text/to-html { };

  toilet = callPackage ../tools/misc/toilet { };

  tokei = callPackage ../development/tools/misc/tokei {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  tokio-console = callPackage ../development/tools/tokio-console { };

  toml2json = callPackage ../development/tools/toml2json { };

  toml2nix = callPackage ../development/tools/toml2nix { };

  topfew = callPackage ../tools/text/topfew { };

  topfew-rs = callPackage ../tools/text/topfew-rs { };

  topgrade = callPackage ../tools/misc/topgrade {
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa Foundation;
  };

  topiary = callPackage ../development/tools/misc/topiary { };

  todo = callPackage ../tools/misc/todo { };

  toolbox = callPackage ../applications/virtualization/toolbox { };

  tor = callPackage ../tools/security/tor { };

  touchegg = callPackage ../tools/inputmethods/touchegg { };

  torrent7z = callPackage ../tools/archivers/torrent7z { };

  torsocks = callPackage ../tools/security/tor/torsocks.nix { };

  toss = callPackage ../tools/networking/toss { };

  tox-node = callPackage ../tools/networking/tox-node { };

  toxvpn = callPackage ../tools/networking/toxvpn { };

  toybox = darwin.apple_sdk_11_0.callPackage ../tools/misc/toybox { };

  trackma = callPackage ../tools/misc/trackma { };

  trackma-curses = trackma.override { withCurses = true; };

  trackma-gtk = trackma.override { withGTK = true; };

  trackma-qt = trackma.override { withQT = true; };

  tran = callPackage ../tools/networking/tran { };

  trayscale = callPackage ../applications/networking/trayscale { };

  tpmmanager = libsForQt5.callPackage ../applications/misc/tpmmanager { };

  tpm-quote-tools = callPackage ../tools/security/tpm-quote-tools { };

  tpm-tools = callPackage ../tools/security/tpm-tools { };

  tpm-luks = callPackage ../tools/security/tpm-luks { };

  tpm2-abrmd = callPackage ../tools/security/tpm2-abrmd { };

  tpm2-pkcs11 = callPackage ../misc/tpm2-pkcs11 { };

  tpm2-tools = callPackage ../tools/security/tpm2-tools { };

  trezor-udev-rules = callPackage ../os-specific/linux/trezor-udev-rules { };

  trezorctl = with python3Packages; toPythonApplication trezor;

  trezord = callPackage ../servers/trezord {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  trezor-agent = with python3Packages; toPythonApplication trezor-agent;

  trezor-suite = callPackage ../applications/blockchains/trezor-suite { };

  trunk = callPackage ../development/tools/trunk {
    inherit (darwin.apple_sdk.frameworks) CoreServices SystemConfiguration;
  };

  trunk-io = callPackage ../development/tools/trunk-io { };

  trunk-ng = callPackage ../by-name/tr/trunk-ng/package.nix {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  tthsum = callPackage ../applications/misc/tthsum { };

  ttdl = callPackage ../applications/misc/ttdl { };

  ttp = with python3.pkgs; toPythonApplication ttp;

  trace-cmd = callPackage ../os-specific/linux/trace-cmd { };

  kernelshark = libsForQt5.callPackage ../os-specific/linux/trace-cmd/kernelshark.nix { };

  traceroute = callPackage ../tools/networking/traceroute { };

  tracebox = callPackage ../tools/networking/tracebox { };

  tracee = callPackage ../tools/security/tracee {
    clang = clang_14;
  };

  tracefilegen = callPackage ../development/tools/analysis/garcosim/tracefilegen { };

  tracefilesim = callPackage ../development/tools/analysis/garcosim/tracefilesim { };

  transifex-cli = callPackage ../applications/misc/transifex-cli { };

  translatelocally = callPackage ../applications/misc/translatelocally { };

  translatelocally-models = recurseIntoAttrs (callPackages ../misc/translatelocally-models { });

  translate-shell = callPackage ../applications/misc/translate-shell { };

  translatepy = with python3.pkgs; toPythonApplication translatepy;

  trash-cli = callPackage ../tools/misc/trash-cli { };

  trashy = callPackage ../tools/misc/trashy { };

  trdl-client = callPackage ../tools/misc/trdl-client { };

  trenchbroom = libsForQt5.callPackage ../applications/misc/trenchbroom { };

  trickle = callPackage ../tools/networking/trickle { };

  node-manta = callPackage ../tools/admin/manta { };

  triton = callPackage ../tools/admin/triton { };

  triggerhappy = callPackage ../tools/inputmethods/triggerhappy { };

  inherit (callPackage ../applications/office/trilium {})
    trilium-desktop
    trilium-server
    ;

  trippy = callPackage ../tools/networking/trippy { };

  trousers = callPackage ../tools/security/trousers { };

  trueseeing = callPackage ../tools/security/trueseeing { };

  trx = callPackage ../tools/audio/trx { };

  tryton = callPackage ../applications/office/tryton { };

  trytond = with python3Packages; toPythonApplication trytond;

  tun2socks = callPackage ../tools/networking/tun2socks { };

  tuntox = callPackage ../tools/networking/tuntox { };

  tunwg = callPackage ../tools/networking/tunwg { };

  ttf2pt1 = callPackage ../tools/misc/ttf2pt1 { };

  ttfautohint = libsForQt5.callPackage ../tools/misc/ttfautohint {
    autoreconfHook = buildPackages.autoreconfHook269;
  };
  ttfautohint-nox = ttfautohint.override { enableGUI = false; };

  tty-clock = callPackage ../tools/misc/tty-clock { };

  tty-share = callPackage ../applications/misc/tty-share { };

  ttyplot = callPackage ../tools/misc/ttyplot { };

  ttygif = callPackage ../tools/misc/ttygif { };

  ttylog = callPackage ../tools/misc/ttylog { };

  twm = callPackage ../tools/misc/twm {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  txtpbfmt = callPackage ../development/tools/txtpbfmt { };

  ipbt = callPackage ../tools/misc/ipbt { };

  tuckr = callPackage ../applications/misc/tuckr { };

  tuhi = callPackage ../applications/misc/tuhi { };

  tui-journal = callPackage ../applications/misc/tui-journal { };

  tuir = callPackage ../applications/misc/tuir { };

  tuifeed = callPackage ../applications/networking/feedreaders/tuifeed {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  russ = callPackage ../applications/networking/feedreaders/russ { };

  tunnelto = callPackage ../tools/networking/tunnelto {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  tuptime = callPackage ../tools/system/tuptime { };

  turbo = callPackage ../tools/misc/turbo/wrapper.nix { };

  turbo-unwrapped = callPackage ../tools/misc/turbo {
    inherit (darwin.apple_sdk_11_0.frameworks) IOKit CoreServices CoreFoundation;
  };

  turses = callPackage ../applications/networking/instant-messengers/turses { };

  tutanota-desktop = callPackage ../applications/networking/mailreaders/tutanota-desktop { };

  tv = callPackage ../tools/text/tv { };

  tvm = callPackage ../development/compilers/tvm { };

  oysttyer = callPackage ../applications/networking/instant-messengers/oysttyer { };

  ttfb = callPackage ../development/tools/ttfb { };

  twilight = callPackage ../tools/graphics/twilight {
    libX11 = xorg.libX11;
  };

  twitch-chat-downloader = python3Packages.callPackage ../applications/misc/twitch-chat-downloader { };

  twitterBootstrap = callPackage ../development/web/twitter-bootstrap { };

  twspace-crawler = callPackage ../tools/misc/twspace-crawler { };

  twspace-dl = callPackage ../tools/misc/twspace-dl { };

  twtxt = python3Packages.callPackage ../applications/networking/twtxt { };

  twurl = callPackage ../tools/misc/twurl { };

  txt2man = callPackage ../tools/misc/txt2man { };

  txt2tags = callPackage ../tools/text/txt2tags { };

  txtw = callPackage ../tools/misc/txtw { };

  tydra = callPackage ../tools/misc/tydra { };

  typesense = callPackage ../servers/search/typesense { };

  typos = callPackage ../development/tools/typos { };

  tz = callPackage ../tools/misc/tz { };

  u9fs = callPackage ../servers/u9fs { };

  ua = callPackage ../tools/networking/ua { };

  uair = callPackage ../tools/misc/uair { };

  ubidump = python3Packages.callPackage ../tools/filesystems/ubidump { };

  ubi_reader = callPackage ../tools/filesystems/ubi_reader { };

  ubpm = libsForQt5.callPackage ../applications/misc/ubpm { };

  ubridge = callPackage ../tools/networking/ubridge { };

  ubertooth = callPackage ../applications/radio/ubertooth { };

  ucarp = callPackage ../servers/ucarp { };

  ucl = callPackage ../development/libraries/ucl { };

  ucspi-tcp = callPackage ../tools/networking/ucspi-tcp { };

  udftools = callPackage ../tools/filesystems/udftools { };

  udp2raw = callPackage ../tools/networking/udp2raw { };

  udpreplay = callPackage ../tools/networking/udpreplay { };

  udpt = callPackage ../servers/udpt { };

  udptunnel = callPackage ../tools/networking/udptunnel { };

  udpx = callPackage ../tools/security/udpx { };

  uftrace = callPackage ../development/tools/uftrace { };

  uftpd = callPackage ../servers/ftp/uftpd {};

  uget = callPackage ../tools/networking/uget { };

  uget-integrator = callPackage ../tools/networking/uget-integrator { };

  ugrep = callPackage ../tools/text/ugrep { };

  ugs = callPackage ../tools/misc/ugs { };

  uhk-agent = callPackage ../os-specific/linux/uhk-agent { };

  uhk-udev-rules = callPackage ../os-specific/linux/uhk-udev-rules { };

  uif2iso = callPackage ../tools/cd-dvd/uif2iso { };

  uivonim = callPackage ../applications/editors/uivonim { };

  ulid = callPackage ../tools/misc/ulid { };

  umlet = callPackage ../tools/misc/umlet { };

  unetbootin = libsForQt5.callPackage ../tools/cd-dvd/unetbootin { };

  unfs3 = callPackage ../servers/unfs3 { };

  unfurl = callPackage ../tools/text/unfurl { };

  unixbench = callPackage ../development/tools/misc/unixbench { };

  unoconv = callPackage ../tools/text/unoconv { };

  unrtf = callPackage ../tools/text/unrtf { };

  unrpa = with python3Packages; toPythonApplication unrpa;

  untex = callPackage ../tools/text/untex { };

  untrunc-anthwlock = callPackage ../tools/video/untrunc-anthwlock { };

  unzrip = callPackage ../tools/compression/unzrip { };

  up = callPackage ../tools/misc/up { };

  upterm = callPackage ../tools/misc/upterm { };

  upx = callPackage ../tools/compression/upx { };

  uq = callPackage ../misc/uq { };

  uqmi = callPackage ../tools/networking/uqmi { };

  urdfdom = callPackage ../development/libraries/urdfdom { };

  urdfdom-headers = callPackage ../development/libraries/urdfdom-headers { };

  urlencode = callPackage ../tools/misc/urlencode {};

  uriparser = callPackage ../development/libraries/uriparser { };

  urlscan = callPackage ../applications/misc/urlscan { };

  url-parser = callPackage ../tools/misc/url-parser { };

  urn-timer = callPackage ../tools/misc/urn-timer { };

  ursadb = callPackage ../servers/ursadb { };

  usbmuxd = callPackage ../tools/misc/usbmuxd { };

  usbmuxd2 = callPackage ../tools/misc/usbmuxd2 { };

  usort = with python3Packages; toPythonApplication usort;

  ustreamer = callPackage ../applications/video/ustreamer { };

  usync = callPackage ../applications/misc/usync { };

  uwc = callPackage ../tools/text/uwc { };

  uwsgi = callPackage ../servers/uwsgi { };

  uwufetch = callPackage ../tools/misc/uwufetch { };

  v2ray = callPackage ../tools/networking/v2ray { };

  v2raya = callPackage ../tools/networking/v2raya { };

  v2ray-domain-list-community = callPackage ../data/misc/v2ray-domain-list-community { };

  v2ray-geoip = callPackage ../data/misc/v2ray-geoip { };

  vacuum = libsForQt5.callPackage ../applications/networking/instant-messengers/vacuum {};

  validator-nu = callPackage ../tools/text/validator-nu { };

  vampire = callPackage ../applications/science/logic/vampire { };

  variety = callPackage ../applications/misc/variety { };

  vdmfec = callPackage ../applications/backup/vdmfec { };

  vk-bootstrap = callPackage ../development/libraries/vk-bootstrap { };

  vk-cli = callPackage ../applications/networking/instant-messengers/vk-cli { };

  vk-messenger = callPackage ../applications/networking/instant-messengers/vk-messenger { };

  volatility3 = callPackage ../tools/security/volatility3 { };

  vbetool = callPackage ../tools/system/vbetool { };

  vcsi = callPackage ../tools/video/vcsi { };

  vde2 = callPackage ../tools/networking/vde2 { };

  vboot_reference = callPackage ../tools/system/vboot_reference { };

  vcftools = callPackage ../applications/science/biology/vcftools { };

  vcmi = libsForQt5.callPackage ../games/vcmi { };

  vcsh = callPackage ../applications/version-management/vcsh { };

  vcs_query = callPackage ../tools/misc/vcs_query { };

  vcstool = callPackage ../development/tools/vcstool { };

  vectorscan = callPackage ../development/libraries/vectorscan { };

  verco = callPackage ../applications/version-management/verco { };

  verilator = callPackage ../applications/science/electronics/verilator { };

  verilog = callPackage ../applications/science/electronics/verilog { };

  versus = callPackage ../applications/networking/versus { };

  vexctl = callPackage ../tools/security/vexctl { };

  vgrep = callPackage ../tools/text/vgrep { };

  vhd2vl = callPackage ../applications/science/electronics/vhd2vl { };

  vhdl-ls = callPackage ../development/tools/language-servers/vhdl-ls { };

  video2midi = callPackage ../tools/audio/video2midi {
    pythonPackages = python3Packages;
  };

  video-trimmer = callPackage ../applications/video/video-trimmer { };

  via = callPackage ../tools/misc/via { };

  vial = callPackage ../tools/misc/vial { };

  viking = callPackage ../applications/misc/viking { };

  vim-vint = callPackage ../development/tools/vim-vint { };

  vimer = callPackage ../tools/misc/vimer { };

  vimpager = callPackage ../tools/misc/vimpager { };
  vimpager-latest = callPackage ../tools/misc/vimpager/latest.nix { };

  vimwiki-markdown = python3Packages.callPackage ../tools/misc/vimwiki-markdown { };

  visidata = (newScope python3Packages) ../applications/misc/visidata {
  };

  vit = callPackage ../applications/misc/vit { };

  viu = callPackage ../tools/graphics/viu { };

  vix = callPackage ../tools/misc/vix { };

  vkbasalt = callPackage ../tools/graphics/vkbasalt {
    vkbasalt32 = pkgsi686Linux.vkbasalt;
  };

  vkbasalt-cli = callPackage ../tools/graphics/vkbasalt-cli { };

  vkmark = callPackage ../tools/graphics/vkmark { };

  vncrec = callPackage ../tools/video/vncrec { };

  vo-amrwbenc = callPackage ../development/libraries/vo-amrwbenc { };

  vo-aacenc = callPackage ../development/libraries/vo-aacenc { };

  vobcopy = callPackage ../tools/cd-dvd/vobcopy { };

  vobsub2srt = callPackage ../tools/cd-dvd/vobsub2srt { };

  void = callPackage ../tools/misc/void { };

  volume_key = callPackage ../development/libraries/volume-key { };

  vorbisgain = callPackage ../tools/misc/vorbisgain { };

  vpnc = callPackage ../tools/networking/vpnc { };

  vpnc-scripts = callPackage ../tools/networking/vpnc-scripts { };

  vpn-slice = python3Packages.callPackage ../tools/networking/vpn-slice { };

  vp = callPackage ../applications/misc/vp {
    # Enable next line for console graphics. Note that
    # it requires `sixel` enabled terminals such as mlterm
    # or xterm -ti 340
    SDL = SDL_sixel;
  };

  vtm = callPackage ../tools/misc/vtm { };

  witness = callPackage ../tools/security/witness { };

  openconnectPackages = callPackage ../tools/networking/openconnect { };

  inherit (openconnectPackages) openconnect openconnect_openssl;

  globalprotect-openconnect = libsForQt5.callPackage ../tools/networking/globalprotect-openconnect { };

  ding-libs = callPackage ../tools/misc/ding-libs { };

  sssd = callPackage ../os-specific/linux/sssd {
    inherit (perlPackages) Po4a;
  };

  sentry-cli = callPackage ../development/tools/sentry-cli {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security SystemConfiguration;
  };

  sentry-native = callPackage ../development/libraries/sentry-native { };

  twilio-cli = callPackage ../development/tools/twilio-cli { };

  waifu2x-converter-cpp = callPackage ../tools/graphics/waifu2x-converter-cpp {
    inherit (darwin.apple_sdk.frameworks) OpenCL;
  };

  wakapi = callPackage ../tools/misc/wakapi { };

  wambo = callPackage ../development/tools/wambo { };

  weather = callPackage ../applications/misc/weather { };

  wego = callPackage ../applications/misc/wego { };

  wal_e = callPackage ../tools/backup/wal-e { };

  wander = callPackage ../tools/admin/wander { };

  watchexec = callPackage ../tools/misc/watchexec {
    inherit (darwin.apple_sdk.frameworks) Cocoa AppKit;
  };

  watchlog = callPackage ../tools/misc/watchlog { };

  watchman = darwin.apple_sdk_11_0.callPackage ../development/tools/watchman {
    inherit (darwin.apple_sdk_11_0.frameworks) CoreServices;
  };

  wavefunctioncollapse = callPackage ../tools/graphics/wavefunctioncollapse { };

  wbox = callPackage ../tools/networking/wbox { };

  webassemblyjs-cli = nodePackages."@webassemblyjs/cli-1.11.1";
  webassemblyjs-repl = nodePackages."@webassemblyjs/repl-1.11.1";
  wasm-strip = nodePackages."@webassemblyjs/wasm-strip";
  wasm-text-gen = nodePackages."@webassemblyjs/wasm-text-gen-1.11.1";
  wast-refmt = nodePackages."@webassemblyjs/wast-refmt-1.11.1";

  wasm-bindgen-cli = callPackage ../development/tools/wasm-bindgen-cli {
    inherit (darwin.apple_sdk.frameworks) Security;
    nodejs = nodejs_latest;
  };

  wasm-tools = callPackage ../tools/misc/wasm-tools { };

  wasmedge = callPackage ../development/tools/wasmedge {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else llvmPackages.stdenv;
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation;
  };

  wasmi = callPackage ../development/tools/wasmi { };

  wasmserve = callPackage ../development/tools/wasmserve {};

  welkin = callPackage ../tools/graphics/welkin { };

  wemux = callPackage ../tools/misc/wemux { };

  wf-recorder = callPackage ../applications/video/wf-recorder { };

  whatip = callPackage ../tools/networking/whatip { };

  whatsapp-chat-exporter = callPackage ../tools/misc/whatsapp-chat-exporter { };

  whatweb = callPackage ../tools/security/whatweb { };

  whipper = callPackage ../applications/audio/whipper { };

  whitebophir = callPackage ../servers/web-apps/whitebophir { };

  whois = callPackage ../tools/networking/whois { };

  wifish = callPackage ../tools/networking/wifish { };

  wimboot = callPackage ../tools/misc/wimboot { };

  wire = callPackage ../development/tools/wire { };

  wireguard-tools = callPackage ../tools/networking/wireguard-tools { };

  wireguard-vanity-address = callPackage ../tools/networking/wireguard-vanity-address {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  wiringpi = callPackage ../os-specific/linux/wiringpi { };

  wg-friendly-peer-names = callPackage ../tools/networking/wg-friendly-peer-names { };

  wg-netmanager = callPackage ../tools/networking/wg-netmanager {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  wgautomesh = callPackage ../tools/networking/wgautomesh { };

  woff2 = callPackage ../development/web/woff2 { };

  woodpecker-agent = callPackage ../development/tools/continuous-integration/woodpecker/agent.nix { };

  woodpecker-cli = callPackage ../development/tools/continuous-integration/woodpecker/cli.nix { };

  woodpecker-pipeline-transform = callPackage ../development/tools/continuous-integration/woodpecker-pipeline-transform { };

  woodpecker-plugin-git = callPackage ../development/tools/continuous-integration/woodpecker-plugin-git { };

  woodpecker-server = callPackage ../development/tools/continuous-integration/woodpecker/server.nix { };

  woof = callPackage ../tools/misc/woof { };

  wootility = callPackage ../tools/misc/wootility { };

  wormhole-william = callPackage ../tools/networking/wormhole-william { };

  wpscan = callPackage ../tools/security/wpscan { };

  write-good = callPackage ../tools/text/write-good { };

  wsmancli = callPackage ../tools/system/wsmancli { };

  wstunnel = haskell.lib.compose.justStaticExecutables haskellPackages.wstunnel;

  wolfebin = callPackage ../tools/networking/wolfebin { };

  wthrr = callPackage ../applications/misc/wthrr { };

  xautoclick = callPackage ../applications/misc/xautoclick { };

  xl2tpd = callPackage ../tools/networking/xl2tpd { };

  xe = callPackage ../tools/system/xe { };

  xray = callPackage ../tools/networking/xray { };

  xteve = callPackage ../servers/xteve { };

  termbook = callPackage ../tools/text/termbook { };

  testdisk = libsForQt5.callPackage ../tools/system/testdisk { };

  testdisk-qt = testdisk.override { enableQt = true; };

  textql = callPackage ../development/tools/textql { };

  html2text = callPackage ../tools/text/html2text { };

  html-tidy = callPackage ../tools/text/html-tidy { };

  html-xml-utils = callPackage ../tools/text/xml/html-xml-utils { };

  htmldoc = callPackage ../tools/typesetting/htmldoc {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration Foundation;
  };

  htmltest = callPackage ../development/tools/htmltest { };

  rcm = callPackage ../tools/misc/rcm { };

  td = callPackage ../tools/misc/td { };

  tdfgo = callPackage ../tools/misc/tdfgo { };

  tftp-hpa = callPackage ../tools/networking/tftp-hpa { };

  tigervnc = callPackage ../tools/admin/tigervnc { };

  tightvnc = callPackage ../tools/admin/tightvnc {
    fontDirectories = [ xorg.fontadobe75dpi xorg.fontmiscmisc xorg.fontcursormisc
      xorg.fontbhlucidatypewriter75dpi ];
  };

  time = callPackage ../tools/misc/time { };

  tweet-hs = haskell.lib.compose.justStaticExecutables haskellPackages.tweet-hs;

  tweeny = callPackage ../development/libraries/tweeny { };

  tkgate = callPackage ../applications/science/electronics/tkgate/1.x.nix { };

  tm = callPackage ../tools/system/tm { };

  tradcpp = callPackage ../development/tools/tradcpp { };

  traitor = callPackage ../tools/security/traitor { };

  tre = callPackage ../development/libraries/tre { };

  tremor-rs = darwin.apple_sdk_11_0.callPackage ../tools/misc/tremor-rs {
    inherit (darwin.apple_sdk_11_0.frameworks) Security;
  };

  tremor-language-server = callPackage ../tools/misc/tremor-rs/ls.nix { };

  truecrack = callPackage ../tools/security/truecrack { };
  truecrack-cuda = truecrack.override { cudaSupport = true; };

  ts = callPackage ../tools/system/ts { };

  ttmkfdir = callPackage ../tools/misc/ttmkfdir { };

  ttwatch = callPackage ../tools/misc/ttwatch { };

  turbovnc = callPackage ../tools/admin/turbovnc {
    # fontDirectories = [ xorg.fontadobe75dpi xorg.fontmiscmisc xorg.fontcursormisc xorg.fontbhlucidatypewriter75dpi ];
    libjpeg_turbo = libjpeg_turbo.override { enableJava = true; };
  };

  udunits = callPackage ../development/libraries/udunits { };

  ufmt = with python3Packages; toPythonApplication ufmt;

  uftp = callPackage ../servers/uftp { };

  uhttpmock = callPackage ../development/libraries/uhttpmock { };

  uim = callPackage ../tools/inputmethods/uim {
    autoconf = buildPackages.autoconf269;
  };

  uhub = callPackage ../servers/uhub { };

  unclutter = callPackage ../tools/misc/unclutter { };

  unclutter-xfixes = callPackage ../tools/misc/unclutter-xfixes { };

  unbound-with-systemd = unbound.override {
    withSystemd = true;
  };

  unbound-full = unbound.override {
    python = python3;
    withSystemd = true;
    withPythonModule = true;
    withDoH = true;
    withECS = true;
    withDNSCrypt = true;
    withDNSTAP = true;
    withTFO = true;
    withRedis = true;
  };

  unicorn = callPackage ../development/libraries/unicorn {
    inherit (darwin.apple_sdk.frameworks) IOKit;
    inherit (darwin) cctools;
  };

  unittest-cpp = callPackage ../development/libraries/unittest-cpp { };

  unnaturalscrollwheels = callPackage ../tools/inputmethods/unnaturalscrollwheels { };

  unrar-wrapper = python3Packages.callPackage ../tools/archivers/unrar-wrapper { };

  uptime-kuma = callPackage ../servers/monitoring/uptime-kuma { };

  vul = callPackage ../applications/misc/vul { };

  xar = callPackage ../tools/compression/xar { };

  xarchive = callPackage ../tools/archivers/xarchive { };

  xarchiver = callPackage ../tools/archivers/xarchiver { };

  xbanish = callPackage ../tools/X11/xbanish { };

  xbrightness = callPackage ../tools/X11/xbrightness { };

  xdg-launch = callPackage ../applications/misc/xdg-launch { };

  xdp-tools = callPackage ../tools/networking/xdp-tools {
    llvmPackages = llvmPackages_14;
  };

  xkbvalidate = callPackage ../tools/X11/xkbvalidate { };

  xkeysnail = callPackage ../tools/X11/xkeysnail { };

  xfstests = callPackage ../tools/misc/xfstests { };

  xprompt = callPackage ../tools/X11/xprompt { };

  xprintidle = callPackage ../tools/X11/xprintidle { };

  xprintidle-ng = callPackage ../tools/X11/xprintidle-ng { };

  xssstate = callPackage ../tools/X11/xssstate { };

  xscast = callPackage ../applications/video/xscast { };

  xsettingsd = callPackage ../tools/X11/xsettingsd { };

  xsensors = callPackage ../os-specific/linux/xsensors { };

  xspim = callPackage ../development/tools/misc/xspim { };

  xcrawl3r = callPackage ../tools/security/xcrawl3r { };

  xcruiser = callPackage ../applications/misc/xcruiser { };

  xwallpaper = callPackage ../tools/X11/xwallpaper { };

  gxkb = callPackage ../applications/misc/gxkb { };

  xxkb = callPackage ../applications/misc/xxkb { };

  ugarit = callPackage ../tools/backup/ugarit {
    inherit (chickenPackages_4) eggDerivation fetchegg;
  };

  ugarit-manifest-maker = callPackage ../tools/backup/ugarit-manifest-maker {
    inherit (chickenPackages_4) eggDerivation fetchegg;
  };

  ulogd = callPackage ../os-specific/linux/ulogd { };

  unar = callPackage ../tools/archivers/unar {
    inherit (darwin.apple_sdk.frameworks) Foundation AppKit;
    stdenv = clangStdenv;
  };

  unp = callPackage ../tools/archivers/unp { };

  unshield = callPackage ../tools/archivers/unshield { };

  unzip = callPackage ../tools/archivers/unzip { };

  unzipNLS = lowPrio (unzip.override { enableNLS = true; });

  undmg = callPackage ../tools/archivers/undmg { };

  uptimed = callPackage ../tools/system/uptimed { };

  upwork = callPackage ../applications/misc/upwork { };

  urjtag = callPackage ../tools/misc/urjtag { };

  urlhunter = callPackage ../tools/security/urlhunter { };

  urlwatch = callPackage ../tools/networking/urlwatch { };

  vals = callPackage ../tools/security/vals { };

  valum = callPackage ../development/web/valum { };

  inherit (callPackages ../servers/varnish { })
    varnish60 varnish74 varnish75;
  inherit (callPackages ../servers/varnish/packages.nix { })
    varnish60Packages varnish74Packages varnish75Packages;

  varnishPackages = varnish74Packages;
  varnish = varnishPackages.varnish;

  hitch = callPackage ../servers/hitch { };

  veracrypt = callPackage ../applications/misc/veracrypt {
    wxGTK = wxGTK32;
  };

  veryfasttree = callPackage ../applications/science/biology/veryfasttree { };

  viceroy = callPackage ../development/tools/viceroy {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  vlan = callPackage ../tools/networking/vlan { };

  vmtouch = callPackage ../tools/misc/vmtouch { };

  vncdo = with python3Packages; toPythonApplication vncdo;

  volumeicon = callPackage ../tools/audio/volumeicon { };

  # An alias to work around the splicing incidents
  # Related:
  # https://github.com/NixOS/nixpkgs/issues/204303
  # https://github.com/NixOS/nixpkgs/issues/211340
  # https://github.com/NixOS/nixpkgs/issues/227327
  wafHook = waf.hook;

  waf-tester = callPackage ../tools/security/waf-tester { };

  wagyu = callPackage ../tools/misc/wagyu {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  wakelan = callPackage ../tools/networking/wakelan { };

  wavemon = callPackage ../tools/networking/wavemon { };

  wdfs = callPackage ../tools/filesystems/wdfs { };

  web-eid-app = libsForQt5.callPackage ../tools/security/web-eid-app { };

  wdiff = callPackage ../tools/text/wdiff { };

  wdisplays = callPackage ../tools/graphics/wdisplays { };

  weaviate = callPackage ../servers/search/weaviate { };

  webalizer = callPackage ../tools/networking/webalizer { };

  webmesh = callPackage ../servers/webmesh { };

  wget = callPackage ../tools/networking/wget { };

  wget2 = callPackage ../tools/networking/wget2 { };

  wgpu-utils = callPackage ../tools/graphics/wgpu-utils {
    inherit (darwin.apple_sdk.frameworks) QuartzCore;
  };

  wg-bond = callPackage ../applications/networking/wg-bond { };

  wgcf = callPackage ../applications/networking/wgcf { };

  which = callPackage ../tools/system/which { };

  whsniff = callPackage ../applications/networking/sniffers/whsniff { };

  wio = callPackage ../by-name/wi/wio/package.nix {
    wlroots = wlroots_0_17;
  };

  wiiuse = callPackage ../development/libraries/wiiuse {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) Foundation IOBluetooth;
  };

  wipefreespace = callPackage ../tools/filesystems/wipefreespace { };

  woeusb = callPackage ../tools/misc/woeusb { };

  woeusb-ng = callPackage ../tools/misc/woeusb-ng { };

  chase = callPackage ../tools/system/chase { };

  wimlib = callPackage ../tools/archivers/wimlib { };

  wipe = callPackage ../tools/security/wipe { };

  wireguard-go = callPackage ../tools/networking/wireguard-go { };

  wkhtmltopdf = libsForQt5.callPackage ../tools/graphics/wkhtmltopdf { };

  wkhtmltopdf-bin = callPackage ../tools/graphics/wkhtmltopdf-bin {
    openssl = openssl_1_1;
  };

  wml = callPackage ../development/web/wml { };

  wol = callPackage ../tools/networking/wol { };

  wolf-shaper = callPackage ../applications/audio/wolf-shaper { };

  wpgtk = callPackage ../tools/X11/wpgtk { };

  wrap = callPackage ../tools/text/wrap { };

  wring = nodePackages.wring;

  wrk = callPackage ../tools/networking/wrk { };

  wrk2 = callPackage ../tools/networking/wrk2 { };

  wsysmon = callPackage ../tools/system/wsysmon { };

  wuzz = callPackage ../tools/networking/wuzz { };

  wv = callPackage ../tools/misc/wv { };

  wv2 = callPackage ../tools/misc/wv2 { };

  wvkbd = callPackage ../applications/accessibility/wvkbd { };

  wyrd = callPackage ../tools/misc/wyrd {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14_unsafe_string;
  };

  x86info = callPackage ../os-specific/linux/x86info { };

  x11_ssh_askpass = callPackage ../tools/networking/x11-ssh-askpass { };

  xbursttools = callPackage ../tools/misc/xburst-tools {
    # It needs a cross compiler for mipsel to build the firmware it will
    # load into the Ben Nanonote
    gccCross = pkgsCross.ben-nanonote.buildPackages.gccWithoutTargetLibc;
    autoconf = buildPackages.autoconf269;
  };

  clipnotify = callPackage ../tools/misc/clipnotify { };

  clipboard-jh = callPackage ../tools/misc/clipboard-jh { };

  clipbuzz = callPackage ../tools/misc/clipbuzz {
    zig = zig_0_12;
  };

  xclip = callPackage ../tools/misc/xclip { };

  xcur2png = callPackage ../tools/graphics/xcur2png { };

  xcwd = callPackage ../tools/X11/xcwd { };

  xtitle = callPackage ../tools/misc/xtitle { };

  xdelta = callPackage ../tools/compression/xdelta { };
  xdeltaUnstable = callPackage ../tools/compression/xdelta/unstable.nix { };

  xdot = with python3Packages; toPythonApplication xdot;

  xdummy = callPackage ../tools/misc/xdummy { };

  xdg-ninja = callPackage ../tools/misc/xdg-ninja { };

  xdxf2slob = callPackage ../tools/misc/xdxf2slob { };

  xe-guest-utilities = callPackage ../tools/virtualization/xe-guest-utilities { };

  xen-guest-agent = callPackage ../tools/virtualization/xen-guest-agent { };

  xflux = callPackage ../tools/misc/xflux { };
  xflux-gui = python3Packages.callPackage ../tools/misc/xflux/gui.nix { };

  xfsdump = callPackage ../tools/filesystems/xfsdump { };

  xfsprogs = callPackage ../tools/filesystems/xfsprogs { };
  libxfs = xfsprogs.dev;

  xml2 = callPackage ../tools/text/xml/xml2 { };

  xmlformat = callPackage ../tools/text/xml/xmlformat { };

  xmlroff = callPackage ../tools/typesetting/xmlroff { };

  xmloscopy = callPackage ../tools/text/xml/xmloscopy { };

  xmlstarlet = callPackage ../tools/text/xml/xmlstarlet { };

  xmlto = callPackage ../tools/typesetting/xmlto {
    w3m = w3m-batch;
  };

  xiccd = callPackage ../tools/misc/xiccd { };

  xidlehook = callPackage ../tools/X11/xidlehook {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  xprite-editor = callPackage ../tools/misc/xprite-editor {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  xq-xml = callPackage ../tools/text/xml/xq { };

  xsecurelock = callPackage ../tools/X11/xsecurelock { };

  xsel = callPackage ../tools/misc/xsel { };

  xsv = callPackage ../tools/text/xsv {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  xtreemfs = callPackage ../tools/filesystems/xtreemfs {
    boost = boost179;
  };

  xorriso = libisoburn;

  xurls = callPackage ../tools/text/xurls { };

  xxv = callPackage ../tools/misc/xxv { };

  xvfb-run = callPackage ../tools/misc/xvfb-run {
    inherit (texFunctions) fontsConf;

    # xvfb-run is used by a bunch of things to run tests
    # and doesn't support hardware accelerated rendering
    # so remove it from the rebuild heavy path for mesa
    xorgserver = xorg.xorgserver.overrideAttrs(old: {
      buildInputs = lib.filter (pkg: lib.getName pkg != "mesa") old.buildInputs;
      configureFlags = old.configureFlags ++ [
        "--disable-glamor"
        "--disable-glx"
        "--disable-dri"
        "--disable-dri2"
        "--disable-dri3"
      ];
    });
  };

  xvkbd = callPackage ../tools/X11/xvkbd { };

  xwinmosaic = callPackage ../tools/X11/xwinmosaic { };

  xwinwrap = callPackage ../tools/X11/xwinwrap { };

  yajsv = callPackage ../tools/misc/yajsv { };

  yallback = callPackage ../development/tools/analysis/yallback { };

  yapf = with python3Packages; toPythonApplication yapf;

  yarn = callPackage ../development/tools/yarn  { };

  yarn-berry = callPackage ../development/tools/yarn-berry { };

  yarn2nix-moretea = callPackage ../development/tools/yarn2nix-moretea/yarn2nix { pkgs = pkgs.__splicedPackages; };

  inherit (yarn2nix-moretea)
    yarn2nix
    mkYarnPackage
    mkYarnModules
    fixup_yarn_lock;

  yarr = callPackage ../applications/networking/feedreaders/yarr { };

  yascreen = callPackage ../development/libraries/yascreen { };

  yasr = callPackage ../applications/audio/yasr { };

  yank = callPackage ../tools/misc/yank { };

  yamlfix = with python3Packages; toPythonApplication yamlfix;

  yamlfmt = callPackage ../development/tools/yamlfmt { };

  yamllint = with python3Packages; toPythonApplication yamllint;

  yamlpath = callPackage ../development/tools/yamlpath { };

  yaml-merge = callPackage ../tools/text/yaml-merge { };

  yeshup = callPackage ../tools/system/yeshup { };

  ytfzf = callPackage ../tools/misc/ytfzf { };

  # To expose more packages for Yi, override the extraPackages arg.
  yi = callPackage ../applications/editors/yi/wrapper.nix {
    haskellPackages = haskell.packages.ghc810;
  };

  yaydl = callPackage ../tools/video/yaydl {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  yx = callPackage ../tools/text/yx { };

  zarchive = callPackage ../tools/archivers/zarchive { };

  zprint = callPackage ../development/tools/zprint { };

  yle-dl = callPackage ../tools/misc/yle-dl { };

  you-get = callPackage ../tools/misc/you-get { };

  zasm = callPackage ../development/compilers/zasm { };

  zbackup = callPackage ../tools/backup/zbackup {
    protobuf = protobuf_21;
  };

  zbar = libsForQt5.callPackage ../tools/graphics/zbar {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  zbctl = callPackage ../tools/admin/zbctl { };

  zdelta = callPackage ../tools/compression/zdelta { };

  zed = callPackage ../development/tools/zed { };

  zellij = callPackage ../tools/misc/zellij {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration Foundation;
  };

  zenith = callPackage ../tools/system/zenith {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  # Nvidia support does not require any propietary libraries, so CI can build it.
  # Note that when enabling this unconditionally, non-nvidia users will always have an empty "GPU" section.
  zenith-nvidia = callPackage ../tools/system/zenith {
    inherit (darwin.apple_sdk.frameworks) IOKit;
    nvidiaSupport = true;
  };

  zerotierone = callPackage ../tools/networking/zerotierone { };

  zerofree = callPackage ../tools/filesystems/zerofree { };

  zet = callPackage ../tools/text/zet { };

  zfp = callPackage ../tools/compression/zfp { };

  zfs-autobackup = callPackage ../tools/backup/zfs-autobackup { };

  zfsbackup = callPackage ../tools/backup/zfsbackup { };

  zfstools = callPackage ../tools/filesystems/zfstools { };

  zfsnap = callPackage ../tools/backup/zfsnap { };

  zpool-auto-expand-partitions = callPackage ../tools/filesystems/zpool-auto-expand-partitions { };

  zile = callPackage ../applications/editors/zile { };

  zinnia = callPackage ../tools/inputmethods/zinnia { };
  tegaki-zinnia-japanese = callPackage ../tools/inputmethods/tegaki-zinnia-japanese { };

  zim-tools = callPackage ../tools/text/zim-tools { };

  zimfw = callPackage ../shells/zsh/zimfw { };

  zld = callPackage ../development/tools/zld { };

  par = callPackage ../tools/text/par { };

  zip = callPackage ../tools/archivers/zip { };

  zincsearch = callPackage ../servers/search/zincsearch {
    buildGoModule = buildGo121Module;
  };

  zkfuse = callPackage ../tools/filesystems/zkfuse { };

  zpaq = callPackage ../tools/archivers/zpaq { };
  zpaqd = callPackage ../tools/archivers/zpaq/zpaqd.nix { };

  zplug = callPackage ../shells/zsh/zplug { };

  zps = callPackage ../tools/system/zps { };

  zi = callPackage ../shells/zsh/zi { };

  zinit = callPackage ../shells/zsh/zinit {} ;

  zint = qt6Packages.callPackage ../development/libraries/zint { };

  zs-apc-spdu-ctl = callPackage ../tools/networking/zs-apc-spdu-ctl { };

  zs-wait4host = callPackage ../tools/networking/zs-wait4host { };

  zstxtns-utils = callPackage ../tools/text/zstxtns-utils { };

  zsh-abbr = callPackage ../shells/zsh/zsh-abbr { };

  zsh-autoenv = callPackage ../tools/misc/zsh-autoenv { };

  zsh-autopair = callPackage ../shells/zsh/zsh-autopair { };

  zsh-bd = callPackage ../shells/zsh/zsh-bd { };

  zsh-better-npm-completion = callPackage ../shells/zsh/zsh-better-npm-completion { };

  zsh-clipboard = callPackage ../shells/zsh/zsh-clipboard { };

  zsh-defer = callPackage ../shells/zsh/zsh-defer { };

  zsh-edit = callPackage ../shells/zsh/zsh-edit { };

  zsh-git-prompt = callPackage ../shells/zsh/zsh-git-prompt { };

  zsh-history = callPackage ../shells/zsh/zsh-history { };

  zsh-history-search-multi-word = callPackage ../shells/zsh/zsh-history-search-multi-word { };

  zsh-history-substring-search = callPackage ../shells/zsh/zsh-history-substring-search { };

  zsh-navigation-tools = callPackage ../tools/misc/zsh-navigation-tools { };

  zsh-nix-shell = callPackage ../shells/zsh/zsh-nix-shell { };

  zsh-syntax-highlighting = callPackage ../shells/zsh/zsh-syntax-highlighting { };

  zsh-system-clipboard = callPackage ../shells/zsh/zsh-system-clipboard { };

  zsh-f-sy-h = callPackage ../shells/zsh/zsh-f-sy-h { };

  zsh-fast-syntax-highlighting = callPackage ../shells/zsh/zsh-fast-syntax-highlighting { };

  zsh-forgit = callPackage ../shells/zsh/zsh-forgit { };

  zsh-fzf-tab = callPackage ../shells/zsh/zsh-fzf-tab { };

  zsh-autocomplete = callPackage ../shells/zsh/zsh-autocomplete { };

  zsh-autosuggestions = callPackage ../shells/zsh/zsh-autosuggestions { };

  zsh-powerlevel10k = callPackage ../shells/zsh/zsh-powerlevel10k { };

  zsh-powerlevel9k = callPackage ../shells/zsh/zsh-powerlevel9k { };

  zsh-command-time = callPackage ../shells/zsh/zsh-command-time { };

  zsh-fzf-history-search = callPackage ../shells/zsh/zsh-fzf-history-search { };

  zsh-vi-mode = callPackage ../shells/zsh/zsh-vi-mode { };

  zsh-you-should-use = callPackage ../shells/zsh/zsh-you-should-use { };

  zsh-z = callPackage ../shells/zsh/zsh-z { };

  zssh = callPackage ../tools/networking/zssh { };

  zstd = callPackage ../tools/compression/zstd {
    cmake = buildPackages.cmakeMinimal;
  };

  zsv = callPackage ../development/tools/zsv { };

  zxing = callPackage ../tools/graphics/zxing { };

  zkar = callPackage ../tools/security/zkar { };

  zlint = callPackage ../tools/security/zlint { };

  zmap = callPackage ../tools/security/zmap { };

  zmusic = callPackage ../development/libraries/zmusic { };

  zpool-iostat-viz = callPackage ../tools/filesystems/zpool-iostat-viz { };


  ### SHELLS

  runtimeShell = "${runtimeShellPackage}${runtimeShellPackage.shellPath}";
  runtimeShellPackage = bash;

  agdsn-zsh-config = callPackage ../shells/zsh/agdsn-zsh-config { };

  any-nix-shell = callPackage ../shells/any-nix-shell { };

  nix-your-shell = callPackage ../shells/nix-your-shell { };

  bash = lowPrio (callPackage ../shells/bash/5.nix { });
  # WARNING: this attribute is used by nix-shell so it shouldn't be removed/renamed
  bashInteractive = callPackage ../shells/bash/5.nix {
    interactive = true;
    withDocs = true;
  };
  bashInteractiveFHS = callPackage ../shells/bash/5.nix {
    interactive = true;
    withDocs = true;
    forFHSEnv = true;
  };

  bash-completion = callPackage ../shells/bash/bash-completion { };

  gradle-completion = callPackage ../shells/zsh/gradle-completion { };

  nix-bash-completions = callPackage ../shells/bash/nix-bash-completions { };

  yarn-bash-completion = callPackage ../shells/bash/yarn-completion { };

  blesh = callPackage ../shells/bash/blesh { };

  undistract-me = callPackage ../shells/bash/undistract-me { };

  carapace = callPackage ../shells/carapace { };

  dash = callPackage ../shells/dash { };

  dasht = callPackage ../tools/misc/dasht { };

  dashing = callPackage ../tools/misc/dashing { };

  es = callPackage ../shells/es { };

  fish = callPackage ../shells/fish { };

  wrapFish = callPackage ../shells/fish/wrapper.nix { };

  fishPlugins = recurseIntoAttrs (callPackage ../shells/fish/plugins { });

  fzf-git-sh = callPackage ../shells/fzf-git-sh { };

  hishtory = callPackage ../shells/hishtory { };

  ion = callPackage ../shells/ion { };

  jush = callPackage ../shells/jush { };

  ksh = callPackage ../shells/ksh { };

  liquidprompt = callPackage ../shells/liquidprompt { };

  murex = callPackage ../shells/murex { };

  oh = callPackage ../shells/oh { };

  oksh = callPackage ../shells/oksh { };

  scponly = callPackage ../shells/scponly { };

  rush = callPackage ../shells/rush { };

  zsh = callPackage ../shells/zsh { };

  nix-zsh-completions = callPackage ../shells/zsh/nix-zsh-completions { };

  zsh-completions = callPackage ../shells/zsh/zsh-completions { };

  zsh-prezto = callPackage ../shells/zsh/zsh-prezto { };

  yash = callPackage ../shells/yash { };

  grml-zsh-config = callPackage ../shells/zsh/grml-zsh-config { };

  powerline = with python3Packages; toPythonApplication powerline;

  ### DEVELOPMENT / COMPILERS

  temurin-bin-21 = javaPackages.compiler.temurin-bin.jdk-21;
  temurin-jre-bin-21 = javaPackages.compiler.temurin-bin.jre-21;

  temurin-bin-20 = javaPackages.compiler.temurin-bin.jdk-20;
  temurin-jre-bin-20 = javaPackages.compiler.temurin-bin.jre-20;

  temurin-bin-19 = javaPackages.compiler.temurin-bin.jdk-19;
  temurin-jre-bin-19 = javaPackages.compiler.temurin-bin.jre-19;

  temurin-bin-18 = javaPackages.compiler.temurin-bin.jdk-18;
  temurin-jre-bin-18 = javaPackages.compiler.temurin-bin.jre-18;

  temurin-bin-17 = javaPackages.compiler.temurin-bin.jdk-17;
  temurin-jre-bin-17 = javaPackages.compiler.temurin-bin.jre-17;

  temurin-bin-16 = javaPackages.compiler.temurin-bin.jdk-16;
  temurin-bin-11 = javaPackages.compiler.temurin-bin.jdk-11;
  temurin-jre-bin-11 = javaPackages.compiler.temurin-bin.jre-11;
  temurin-bin-8 = javaPackages.compiler.temurin-bin.jdk-8;
  temurin-jre-bin-8 = javaPackages.compiler.temurin-bin.jre-8;

  temurin-bin = temurin-bin-21;
  temurin-jre-bin = temurin-jre-bin-21;

  semeru-bin-17 = javaPackages.compiler.semeru-bin.jdk-17;
  semeru-jre-bin-17 = javaPackages.compiler.semeru-bin.jre-17;
  semeru-bin-16 = javaPackages.compiler.semeru-bin.jdk-16;
  semeru-jre-bin-16 = javaPackages.compiler.semeru-bin.jre-16;
  semeru-bin-11 = javaPackages.compiler.semeru-bin.jdk-11;
  semeru-jre-bin-11 = javaPackages.compiler.semeru-bin.jre-11;
  semeru-bin-8 = javaPackages.compiler.semeru-bin.jdk-8;
  semeru-jre-bin-8 = javaPackages.compiler.semeru-bin.jre-8;

  semeru-bin = semeru-bin-17;
  semeru-jre-bin = semeru-jre-bin-17;

  adoptopenjdk-bin-17-packages-linux = import ../development/compilers/adoptopenjdk-bin/jdk17-linux.nix { inherit stdenv lib; };
  adoptopenjdk-bin-17-packages-darwin = import ../development/compilers/adoptopenjdk-bin/jdk17-darwin.nix { inherit lib; };

  adoptopenjdk-hotspot-bin-16 = javaPackages.compiler.adoptopenjdk-16.jdk-hotspot;
  adoptopenjdk-jre-hotspot-bin-16 = javaPackages.compiler.adoptopenjdk-16.jre-hotspot;
  adoptopenjdk-openj9-bin-16 = javaPackages.compiler.adoptopenjdk-16.jdk-openj9;
  adoptopenjdk-jre-openj9-bin-16 = javaPackages.compiler.adoptopenjdk-16.jre-openj9;

  adoptopenjdk-hotspot-bin-15 = javaPackages.compiler.adoptopenjdk-15.jdk-hotspot;
  adoptopenjdk-jre-hotspot-bin-15 = javaPackages.compiler.adoptopenjdk-15.jre-hotspot;
  adoptopenjdk-openj9-bin-15 = javaPackages.compiler.adoptopenjdk-15.jdk-openj9;
  adoptopenjdk-jre-openj9-bin-15 = javaPackages.compiler.adoptopenjdk-15.jre-openj9;

  adoptopenjdk-hotspot-bin-11 = javaPackages.compiler.adoptopenjdk-11.jdk-hotspot;
  adoptopenjdk-jre-hotspot-bin-11 = javaPackages.compiler.adoptopenjdk-11.jre-hotspot;
  adoptopenjdk-openj9-bin-11 = javaPackages.compiler.adoptopenjdk-11.jdk-openj9;
  adoptopenjdk-jre-openj9-bin-11 = javaPackages.compiler.adoptopenjdk-11.jre-openj9;

  adoptopenjdk-hotspot-bin-8 = javaPackages.compiler.adoptopenjdk-8.jdk-hotspot;
  adoptopenjdk-jre-hotspot-bin-8 = javaPackages.compiler.adoptopenjdk-8.jre-hotspot;
  adoptopenjdk-openj9-bin-8 = javaPackages.compiler.adoptopenjdk-8.jdk-openj9;
  adoptopenjdk-jre-openj9-bin-8 = javaPackages.compiler.adoptopenjdk-8.jre-openj9;

  adoptopenjdk-bin = adoptopenjdk-hotspot-bin-11;
  adoptopenjdk-jre-bin = adoptopenjdk-jre-hotspot-bin-11;

  adoptopenjdk-icedtea-web = callPackage ../development/compilers/adoptopenjdk-icedtea-web {
    jdk = jdk8;
  };

  alan = callPackage ../development/compilers/alan { };

  alan_2 = callPackage ../development/compilers/alan/2.nix { };

  alarm-clock-applet = callPackage ../tools/misc/alarm-clock-applet { };

  ante = callPackage ../development/compilers/ante { };

  armips = callPackage ../development/compilers/armips {
    stdenv = gcc10Stdenv;
  };

  arachne-pnr = callPackage ../development/compilers/arachne-pnr { };

  asciigraph = callPackage ../tools/text/asciigraph { };

  autocorrect = callPackage ../tools/text/autocorrect {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  as31 = callPackage ../development/compilers/as31 { };

  asl = callPackage ../development/compilers/asl { };

  aspectj = callPackage ../development/compilers/aspectj { };

  atasm = callPackage ../development/compilers/atasm { };

  ats = callPackage ../development/compilers/ats { };
  ats2 = callPackage ../development/compilers/ats2 { };

  ats-acc = callPackage ../development/tools/ats-acc { };

  august = callPackage ../development/compilers/august { };

  avra = callPackage ../development/compilers/avra { };

  ballerina = callPackage ../development/compilers/ballerina {
    openjdk = openjdk17_headless;
  };

  bfc = callPackage ../development/compilers/bfc { };

  bigloo = callPackage ../development/compilers/bigloo { };

  binaryen = callPackage ../development/compilers/binaryen {
    nodejs = nodejs-slim;
    inherit (python3Packages) filecheck;
  };

  blueprint-compiler = callPackage ../development/compilers/blueprint { };

  bluespec = callPackage ../development/compilers/bluespec {
    gmp-static = gmp.override { withStatic = true; };
  };

  bun = callPackage ../development/web/bun { };

  cakelisp = callPackage ../development/compilers/cakelisp { };

  ciao = callPackage ../development/compilers/ciao { };

  codon = callPackage ../development/compilers/codon {
    inherit (llvmPackages_14) lld stdenv;
  };

  colm = callPackage ../development/compilers/colm { };

  colmap = libsForQt5.callPackage ../applications/science/misc/colmap { inherit (config) cudaSupport; };
  colmapWithCuda = colmap.override { cudaSupport = true; };

  opensplatWithCuda = opensplat.override { cudaSupport = true; };

  chickenPackages_4 = recurseIntoAttrs (callPackage ../development/compilers/chicken/4 { });
  chickenPackages_5 = recurseIntoAttrs (callPackage ../development/compilers/chicken/5 { });
  chickenPackages = dontRecurseIntoAttrs chickenPackages_5;

  inherit (chickenPackages_5)
    fetchegg
    eggDerivation
    chicken
    egg2nix;

  cc65 = callPackage ../development/compilers/cc65 { };

  cdb = callPackage ../development/tools/database/cdb {
    stdenv = gccStdenv;
  };

  chez = callPackage ../development/compilers/chez {
    inherit (darwin) cctools;
  };

  chez-racket = callPackage ../development/compilers/chez-racket { };

  chez-srfi = callPackage ../development/chez-modules/chez-srfi { };

  chez-mit = callPackage ../development/chez-modules/chez-mit { };

  chez-scmutils = callPackage ../development/chez-modules/chez-scmutils { };

  chez-matchable = callPackage ../development/chez-modules/chez-matchable { };

  libclang = llvmPackages.libclang;
  clang-manpages = llvmPackages.clang-manpages;

  clang-sierraHack = clang.override {
    name = "clang-wrapper-with-reexport-hack";
    bintools = darwin.binutils.override {
      useMacosReexportHack = true;
    };
  };

  clang = llvmPackages.clang;
  clang_12 = llvmPackages_12.clang;
  clang_13 = llvmPackages_13.clang;
  clang_14 = llvmPackages_14.clang;
  clang_15 = llvmPackages_15.clang;
  clang_16 = llvmPackages_16.clang;
  clang_17 = llvmPackages_17.clang;

  clang-tools = llvmPackages.clang-tools;

  clang-analyzer = callPackage ../development/tools/analysis/clang-analyzer {
    llvmPackages = llvmPackages_14;
    inherit (llvmPackages_14) clang;
  };

  clazy = callPackage ../development/tools/analysis/clazy {
    llvmPackages = llvmPackages_14;
    stdenv = llvmPackages_14.stdenv;
  };

  #Use this instead of stdenv to build with clang
  clangStdenv = if stdenv.cc.isClang then stdenv else lowPrio llvmPackages.stdenv;
  clang-sierraHack-stdenv = overrideCC stdenv buildPackages.clang-sierraHack;
  libcxxStdenv = if stdenv.isDarwin then stdenv else lowPrio llvmPackages.libcxxStdenv;

  clean = callPackage ../development/compilers/clean { };

  clickable = python3Packages.callPackage ../development/tools/clickable { };

  closurecompiler = callPackage ../development/compilers/closure { };

  cmdstan = callPackage ../development/compilers/cmdstan { };

  coffeescript = callPackage ../development/compilers/coffeescript { };

  comby = callPackage ../development/tools/comby {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  inherit (coqPackages) compcert;

  computecpp-unwrapped = callPackage ../development/compilers/computecpp { };
  computecpp = wrapCCWith rec {
    cc = computecpp-unwrapped;
    extraPackages = [
      llvmPackages.compiler-rt
    ];
    extraBuildCommands = ''
      wrap compute $wrapper $ccPath/compute
      wrap compute++ $wrapper $ccPath/compute++
      export named_cc=compute
      export named_cxx=compute++

      rsrc="$out/resource-root"
      mkdir -p "$rsrc/lib"
      ln -s "${cc}/lib" "$rsrc/include"
      echo "-resource-dir=$rsrc" >> $out/nix-support/cc-cflags
    '';
  };

  copper = callPackage ../development/compilers/copper { };

  corretto11 = javaPackages.compiler.corretto11;
  corretto17 = javaPackages.compiler.corretto17;
  corretto19 = javaPackages.compiler.corretto19;
  corretto21 = javaPackages.compiler.corretto21;

  cotton = callPackage ../development/tools/cotton {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  critcl = callPackage ../development/compilers/critcl {
    tcllib = tcllib.override { withCritcl = false; };
  };

  inherit (darwin.apple_sdk_11_0.callPackage ../development/compilers/crystal { })
    crystal_1_2
    crystal_1_7
    crystal_1_8
    crystal_1_9
    crystal;

  crystal2nix = callPackage ../development/compilers/crystal2nix { };

  crystalline = callPackage ../development/tools/language-servers/crystalline {
    llvmPackages = llvmPackages_15;
  };

  icr = callPackage ../development/tools/icr { };

  scry = callPackage ../development/tools/scry { crystal = crystal_1_2; };

  dasm = callPackage ../development/compilers/dasm { };

  dbmate = callPackage ../development/tools/database/dbmate { };

  dbmonster = callPackage ../tools/security/dbmonster { };

  devpi-client = python3Packages.callPackage ../development/tools/devpi-client { };

  devpi-server = python3Packages.callPackage ../development/tools/devpi-server { };

  dictu = callPackage ../development/compilers/dictu { };

  eli = callPackage ../development/compilers/eli { };

  erg = callPackage ../development/compilers/erg { };

  elm2nix = haskell.lib.compose.justStaticExecutables haskellPackages.elm2nix;

  elmPackages = recurseIntoAttrs (callPackage ../development/compilers/elm { });

  apache-flex-sdk = callPackage ../development/compilers/apache-flex-sdk { };

  fasm = pkgsi686Linux.callPackage ../development/compilers/fasm {
    inherit (stdenv) isx86_64;
  };
  fasm-bin = callPackage ../development/compilers/fasm/bin.nix { };

  fasmg = callPackage ../development/compilers/fasmg { };

  fbc = if stdenv.hostPlatform.isDarwin then
    callPackage ../development/compilers/fbc/mac-bin.nix { }
  else
    callPackage ../development/compilers/fbc { };

  filecheck = with python3Packages; toPythonApplication filecheck;

  firrtl = callPackage ../development/compilers/firrtl { };

  flasm = callPackage ../development/compilers/flasm { };

  flyctl = callPackage ../development/web/flyctl { };

  fluidd = callPackage ../applications/misc/fluidd { };

  flutterPackages = recurseIntoAttrs (callPackage ../development/compilers/flutter { });
  flutter = flutterPackages.stable;
  flutter322 = flutterPackages.v3_22;
  flutter319 = flutterPackages.v3_19;
  flutter316 = flutterPackages.v3_16;
  flutter313 = flutterPackages.v3_13;

  fnm = callPackage ../development/tools/fnm {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration Foundation Security;
  };

  fnlfmt = callPackage ../development/tools/fnlfmt { };

  fpc = callPackage ../development/compilers/fpc { };

  g203-led = callPackage ../tools/misc/g203-led { };

  gambit = callPackage ../development/compilers/gambit { };
  gambit-unstable = callPackage ../development/compilers/gambit/unstable.nix { };
  gambit-support = callPackage ../development/compilers/gambit/gambit-support.nix { };
  gerbil = callPackage ../development/compilers/gerbil { };
  gerbil-unstable = callPackage ../development/compilers/gerbil/unstable.nix { };
  gerbil-support = callPackage ../development/compilers/gerbil/gerbil-support.nix { };
  gerbilPackages-unstable = pkgs.gerbil-support.gerbilPackages-unstable; # NB: don't recurseIntoAttrs for (unstable!) libraries
  glow-lang = pkgs.gerbilPackages-unstable.glow-lang;

  gbforth = callPackage ../development/compilers/gbforth { };

  default-gcc-version =
    if (with stdenv.targetPlatform; isVc4 || libc == "relibc") then 6
    else 13;
  gcc = pkgs.${"gcc${toString default-gcc-version}"};
  gccFun = callPackage ../development/compilers/gcc;
  gcc-unwrapped = gcc.cc;

  wrapNonDeterministicGcc = stdenv: ccWrapper:
    if ccWrapper.isGNU then ccWrapper.overrideAttrs(old: {
      env = old.env // {
        cc = old.env.cc.override {
          reproducibleBuild = false;
          profiledCompiler = with stdenv; (!isDarwin && hostPlatform.isx86);
        };
      };
    }) else ccWrapper;

  gccStdenv =
    if stdenv.cc.isGNU
    then stdenv
    else stdenv.override {
      cc = buildPackages.gcc;
      allowedRequisites = null;
      # Remove libcxx/libcxxabi, and add clang for AS if on darwin (it uses
      # clang's internal assembler).
      extraBuildInputs = lib.optional stdenv.hostPlatform.isDarwin clang.cc;
    };

  gcc49Stdenv = overrideCC gccStdenv buildPackages.gcc49;
  gcc6Stdenv = overrideCC gccStdenv buildPackages.gcc6;
  gcc7Stdenv = overrideCC gccStdenv buildPackages.gcc7;
  gcc8Stdenv = overrideCC gccStdenv buildPackages.gcc8;
  gcc9Stdenv = overrideCC gccStdenv buildPackages.gcc9;
  gcc10Stdenv = overrideCC gccStdenv buildPackages.gcc10;
  gcc11Stdenv = overrideCC gccStdenv buildPackages.gcc11;
  gcc12Stdenv = overrideCC gccStdenv buildPackages.gcc12;
  gcc13Stdenv = overrideCC gccStdenv buildPackages.gcc13;
  gcc14Stdenv = overrideCC gccStdenv buildPackages.gcc14;

  # This is not intended for use in nixpkgs but for providing a faster-running
  # compiler to nixpkgs users by building gcc with reproducibility-breaking
  # profile-guided optimizations
  fastStdenv = overrideCC gccStdenv (wrapNonDeterministicGcc gccStdenv buildPackages.gcc_latest);

  wrapCCMulti = cc:
    if stdenv.targetPlatform.system == "x86_64-linux" then let
      # Binutils with glibc multi
      bintools = cc.bintools.override {
        libc = glibc_multi;
      };
    in lowPrio (wrapCCWith {
      cc = cc.cc.override {
        stdenv = overrideCC stdenv (wrapCCWith {
          cc = cc.cc;
          inherit bintools;
          libc = glibc_multi;
        });
        profiledCompiler = false;
        enableMultilib = true;
      };
      libc = glibc_multi;
      inherit bintools;
      extraBuildCommands = ''
        echo "dontMoveLib64=1" >> $out/nix-support/setup-hook
      '';
  }) else throw "Multilib ${cc.name} not supported for ‘${stdenv.targetPlatform.system}’";

  wrapClangMulti = clang:
    if stdenv.targetPlatform.system == "x86_64-linux" then
      callPackage ../development/compilers/llvm/multi.nix {
        inherit clang;
        gcc32 = pkgsi686Linux.gcc;
        gcc64 = pkgs.gcc;
      }
    else throw "Multilib ${clang.cc.name} not supported for '${stdenv.targetPlatform.system}'";

  gcc_multi = wrapCCMulti gcc;
  clang_multi = wrapClangMulti clang;

  gccMultiStdenv = overrideCC stdenv buildPackages.gcc_multi;
  clangMultiStdenv = overrideCC stdenv buildPackages.clang_multi;
  multiStdenv = if stdenv.cc.isClang then clangMultiStdenv else gccMultiStdenv;

  gcc_debug = lowPrio (wrapCC (gcc.cc.overrideAttrs {
    dontStrip = true;
  }));

  gccCrossLibcStdenv = overrideCC stdenvNoCC buildPackages.gccWithoutTargetLibc;

  crossLibcStdenv =
    if stdenv.hostPlatform.useLLVM or false || stdenv.hostPlatform.isDarwin
    then overrideCC stdenv buildPackages.llvmPackages.clangNoLibc
    else gccCrossLibcStdenv;

  # The GCC used to build libc for the target platform. Normal gccs will be
  # built with, and use, that cross-compiled libc.
  gccWithoutTargetLibc = assert stdenv.targetPlatform != stdenv.hostPlatform; let
    libcCross1 = binutilsNoLibc.libc;
    in wrapCCWith {
      cc = gccFun {
        # copy-pasted
        inherit noSysDirs;
        majorMinorVersion = toString default-gcc-version;

        reproducibleBuild = true;
        profiledCompiler = false;

        isl = if !stdenv.isDarwin then isl_0_20 else null;

        withoutTargetLibc = true;
        langCC = false;
        libcCross = libcCross1;
        targetPackages.stdenv.cc.bintools = binutilsNoLibc;
        enableShared =
          stdenv.targetPlatform.hasSharedLibraries

          # temporarily disabled due to breakage;
          # see https://github.com/NixOS/nixpkgs/pull/243249
          && !stdenv.targetPlatform.isWindows
          && !(stdenv.targetPlatform.useLLVM or false)
        ;
      };
      bintools = binutilsNoLibc;
      libc = libcCross1;
      extraPackages = [];
  };

  inherit (callPackage ../development/compilers/gcc/all.nix { inherit noSysDirs; })
    gcc48 gcc49 gcc6 gcc7 gcc8 gcc9 gcc10 gcc11 gcc12 gcc13 gcc14;

  gcc_latest = gcc14;

  # Use the same GCC version as the one from stdenv by default
  gfortran = wrapCC (gcc.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran48 = wrapCC (gcc48.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran49 = wrapCC (gcc49.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran6 = wrapCC (gcc6.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran7 = wrapCC (gcc7.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran8 = wrapCC (gcc8.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran9 = wrapCC (gcc9.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran10 = wrapCC (gcc10.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran11 = wrapCC (gcc11.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran12 = wrapCC (gcc12.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran13 = wrapCC (gcc13.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  gfortran14 = wrapCC (gcc14.cc.override {
    name = "gfortran";
    langFortran = true;
    langCC = false;
    langC = false;
    profiledCompiler = false;
  });

  libgccjit = gcc.cc.override {
    name = "libgccjit";
    langFortran = false;
    langCC = false;
    langC = false;
    profiledCompiler = false;
    langJit = true;
    enableLTO = false;
  };

  gcj = gcj6;
  gcj6 = wrapCC (gcc6.cc.override {
    name = "gcj";
    langJava = true;
    langFortran = false;
    langCC = false;
    langC = false;
    profiledCompiler = false;
    inherit zip unzip zlib boehmgc gettext pkg-config perl;
    inherit (gnome2) libart_lgpl;
  });

  gnat = gnat12; # When changing this, update also gnatPackages

  gnat11 = wrapCC (gcc11.cc.override {
    name = "gnat";
    langC = true;
    langCC = false;
    langAda = true;
    profiledCompiler = false;
    # As per upstream instructions building a cross compiler
    # should be done with a (native) compiler of the same version.
    # If we are cross-compiling GNAT, we may as well do the same.
    gnat-bootstrap =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
      then buildPackages.gnat-bootstrap11
      else buildPackages.gnat11;
    stdenv =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
         && stdenv.buildPlatform.isDarwin
         && stdenv.buildPlatform.isx86_64
      then overrideCC stdenv gnat-bootstrap11
      else stdenv;
  });

  gnat12 = wrapCC (gcc12.cc.override {
    name = "gnat";
    langC = true;
    langCC = false;
    langAda = true;
    profiledCompiler = false;
    # As per upstream instructions building a cross compiler
    # should be done with a (native) compiler of the same version.
    # If we are cross-compiling GNAT, we may as well do the same.
    gnat-bootstrap =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
      then buildPackages.gnat-bootstrap12
      else buildPackages.gnat12;
    stdenv =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
         && stdenv.buildPlatform.isDarwin
         && stdenv.buildPlatform.isx86_64
      then overrideCC stdenv gnat-bootstrap12
      else stdenv;
  });

  gnat13 = wrapCC (gcc13.cc.override {
    name = "gnat";
    langC = true;
    langCC = false;
    langAda = true;
    profiledCompiler = false;
    # As per upstream instructions building a cross compiler
    # should be done with a (native) compiler of the same version.
    # If we are cross-compiling GNAT, we may as well do the same.
    gnat-bootstrap =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
      then buildPackages.gnat-bootstrap12
      else buildPackages.gnat13;
    stdenv =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
         && stdenv.buildPlatform.isDarwin
         && stdenv.buildPlatform.isx86_64
      then overrideCC stdenv gnat-bootstrap12
      else stdenv;
  });

  gnat14 = wrapCC (gcc14.cc.override {
    name = "gnat";
    langC = true;
    langCC = false;
    langAda = true;
    profiledCompiler = false;
    # As per upstream instructions building a cross compiler
    # should be done with a (native) compiler of the same version.
    # If we are cross-compiling GNAT, we may as well do the same.
    gnat-bootstrap =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
      then buildPackages.gnat-bootstrap12
      else buildPackages.gnat13;
    stdenv =
      if stdenv.hostPlatform == stdenv.targetPlatform
         && stdenv.buildPlatform == stdenv.hostPlatform
         && stdenv.buildPlatform.isDarwin
         && stdenv.buildPlatform.isx86_64
      then overrideCC stdenv gnat-bootstrap12
      else stdenv;
  });

  gnat-bootstrap = gnat-bootstrap12;
  gnat-bootstrap11 = wrapCC (callPackage ../development/compilers/gnat-bootstrap { majorVersion = "11"; });
  gnat-bootstrap12 = wrapCCWith ({
    cc = callPackage ../development/compilers/gnat-bootstrap { majorVersion = "12"; };
  } // lib.optionalAttrs (stdenv.hostPlatform.isDarwin) {
    bintools = bintoolsDualAs;
  });

  gnat12Packages = recurseIntoAttrs (callPackage ./ada-packages.nix { gnat = buildPackages.gnat12; });
  gnat13Packages = recurseIntoAttrs (callPackage ./ada-packages.nix { gnat = buildPackages.gnat13; });
  gnatPackages   = gnat12Packages;

  inherit (gnatPackages)
    gprbuild
    gnatprove;

  gnu-smalltalk = callPackage ../development/compilers/gnu-smalltalk { };

  gccgo = wrapCC (gcc.cc.override {
    name = "gccgo";
    langCC = true; #required for go.
    langC = true;
    langGo = true;
    langJit = true;
    profiledCompiler = false;
  } // {
    # not supported on darwin: https://github.com/golang/go/issues/463
    meta.broken = stdenv.hostPlatform.isDarwin;
  });

  gccgo12 = wrapCC (gcc12.cc.override {
    name = "gccgo";
    langCC = true; #required for go.
    langC = true;
    langGo = true;
    langJit = true;
    profiledCompiler = false;
  } // {
    # not supported on darwin: https://github.com/golang/go/issues/463
    meta.broken = stdenv.hostPlatform.isDarwin;
  });

  gccgo13 = wrapCC (gcc13.cc.override {
    name = "gccgo";
    langCC = true; #required for go.
    langC = true;
    langGo = true;
    langJit = true;
    profiledCompiler = false;
  } // {
    # not supported on darwin: https://github.com/golang/go/issues/463
    meta.broken = stdenv.hostPlatform.isDarwin;
  });

  gccgo14 = wrapCC (gcc14.cc.override {
    name = "gccgo";
    langCC = true; #required for go.
    langC = true;
    langGo = true;
    langJit = true;
    profiledCompiler = false;
  } // {
    # not supported on darwin: https://github.com/golang/go/issues/463
    meta.broken = stdenv.hostPlatform.isDarwin;
  });

  ghdl = ghdl-mcode;

  ghdl-mcode = callPackage ../development/compilers/ghdl {
    backend = "mcode";
  };

  ghdl-llvm = callPackage ../development/compilers/ghdl {
    backend = "llvm";
    inherit (llvmPackages_15) llvm;
  };

  gcc-arm-embedded-6 = callPackage ../development/compilers/gcc-arm-embedded/6 { };
  gcc-arm-embedded-7 = callPackage ../development/compilers/gcc-arm-embedded/7 { };
  gcc-arm-embedded-8 = callPackage ../development/compilers/gcc-arm-embedded/8 { };
  gcc-arm-embedded-9 = callPackage ../development/compilers/gcc-arm-embedded/9 { };
  gcc-arm-embedded-10 = callPackage ../development/compilers/gcc-arm-embedded/10 { };
  gcc-arm-embedded-11 = callPackage ../development/compilers/gcc-arm-embedded/11 { };
  gcc-arm-embedded-12 = callPackage ../development/compilers/gcc-arm-embedded/12 { };
  gcc-arm-embedded-13 = callPackage ../development/compilers/gcc-arm-embedded/13 { };
  gcc-arm-embedded = gcc-arm-embedded-12;

  # It would be better to match the default gcc so that there are no linking errors
  # when using C/C++ libraries in D packages, but right now versions >= 12 are broken.
  gdc = gdc11;
  gdc11 = wrapCC (gcc11.cc.override {
    name = "gdc";
    langCC = false;
    langC = false;
    langD = true;
    profiledCompiler = false;
  });

  gforth = callPackage ../development/compilers/gforth { };

  gleam = callPackage ../development/compilers/gleam {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  gmqcc = callPackage ../development/compilers/gmqcc { };

  gtk-server = callPackage ../development/interpreters/gtk-server { };

  # Haskell and GHC

  haskell = callPackage ./haskell-packages.nix { };

  haskellPackages = dontRecurseIntoAttrs
    # Prefer native-bignum to avoid linking issues with gmp
    # GHC 9.6 rts can't be built statically with hadrian, so we need to use 9.4
    # until 9.8 is ready
    (if stdenv.hostPlatform.isStatic then haskell.packages.native-bignum.ghc94
    # JS backend can't use gmp
    else if stdenv.hostPlatform.isGhcjs then haskell.packages.native-bignum.ghc96
    else haskell.packages.ghc96)
  // { __recurseIntoDerivationForReleaseJobs = true; };

  # haskellPackages.ghc is build->host (it exposes the compiler used to build the
  # set, similarly to stdenv.cc), but pkgs.ghc should be host->target to be more
  # consistent with the gcc, gnat, clang etc. derivations
  #
  # We use targetPackages.haskellPackages.ghc if available since this also has
  # the withPackages wrapper available. In the final cross-compiled package set
  # however, targetPackages won't be populated, so we need to fall back to the
  # plain, cross-compiled compiler (which is only theoretical at the moment).
  ghc = targetPackages.haskellPackages.ghc or
    # Prefer native-bignum to avoid linking issues with gmp
    # Use 9.4 for static over broken 9.6
    (if stdenv.targetPlatform.isStatic then haskell.compiler.native-bignum.ghc94
    # JS backend can't use GMP
    else if stdenv.targetPlatform.isGhcjs then haskell.compiler.native-bignum.ghc96
    else haskell.compiler.ghc96);

  alex = haskell.lib.compose.justStaticExecutables haskellPackages.alex;

  happy = haskell.lib.compose.justStaticExecutables haskellPackages.happy;

  hscolour = haskell.lib.compose.justStaticExecutables haskellPackages.hscolour;

  cabal-install = haskell.lib.compose.justStaticExecutables haskellPackages.cabal-install;

  stack = haskell.lib.compose.justStaticExecutables haskellPackages.stack;

  hlint = haskell.lib.compose.justStaticExecutables haskellPackages.hlint;

  krank = haskell.lib.compose.justStaticExecutables haskellPackages.krank;

  stylish-cabal = haskell.lib.compose.justStaticExecutables haskellPackages.stylish-cabal;

  lhs2tex = haskellPackages.lhs2tex;

  all-cabal-hashes = callPackage ../data/misc/hackage { };

  purescript = callPackage ../development/compilers/purescript/purescript { };

  psc-package = callPackage ../development/compilers/purescript/psc-package { };

  purescript-psa = nodePackages.purescript-psa;

  purenix = haskell.lib.compose.justStaticExecutables haskellPackages.purenix;

  spago = callPackage ../development/tools/purescript/spago { };

  pulp = nodePackages.pulp;

  pscid = nodePackages.pscid;

  coreboot-toolchain = recurseIntoAttrs (callPackage ../development/tools/misc/coreboot-toolchain { });

  remarkable-toolchain = callPackage ../development/tools/misc/remarkable/remarkable-toolchain { };

  remarkable2-toolchain = callPackage ../development/tools/misc/remarkable/remarkable2-toolchain { };

  spicedb     = callPackage ../servers/spicedb { };
  spicedb-zed = callPackage ../servers/spicedb/zed.nix { };

  tacacsplus = callPackage ../servers/tacacsplus { };

  tamarin-prover =
    (haskellPackages.callPackage ../applications/science/logic/tamarin-prover {
      # NOTE: do not use the haskell packages 'graphviz' and 'maude'
      inherit maude which;
      graphviz = graphviz-nox;
    });

  inherit (callPackage ../development/compilers/haxe {
    inherit (darwin.apple_sdk.frameworks) Security;
  })
    haxe_4_3
    haxe_4_1
    haxe_4_0
    ;

  haxe = haxe_4_3;
  haxePackages = recurseIntoAttrs (callPackage ./haxe-packages.nix { });
  inherit (haxePackages) hxcpp;

  hop = callPackage ../development/compilers/hop { };

  hop-cli = callPackage ../tools/admin/hop-cli {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security SystemConfiguration;
  };

  falcon = callPackage ../development/interpreters/falcon {
    stdenv = gcc10Stdenv;
  };

  fsharp = callPackage ../development/compilers/fsharp { };

  fstar = callPackage ../development/compilers/fstar {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
    z3 = z3_4_8_5;
  };

  dotnetPackages = recurseIntoAttrs (callPackage ./dotnet-packages.nix {});

  glslang = callPackage ../development/compilers/glslang { };

  gnostic = callPackage ../development/compilers/gnostic { };

  go-junit-report = callPackage ../development/tools/go-junit-report { };

  gocover-cobertura = callPackage ../development/tools/gocover-cobertura { };

  goblob = callPackage ../tools/security/goblob { };

  gogetdoc = callPackage ../development/tools/gogetdoc { };

  gox = callPackage ../development/tools/gox { };

  goxlr-utility = callPackage ../tools/audio/goxlr-utility {};

  gprolog = callPackage ../development/compilers/gprolog { };

  gwe = callPackage ../tools/misc/gwe {
    nvidia_x11 = linuxPackages.nvidia_x11;
  };

  gwt240 = callPackage ../development/compilers/gwt/2.4.0.nix { };

  hvm = darwin.apple_sdk_11_0.callPackage ../development/compilers/hvm { };

  iay = callPackage ../tools/misc/iay {
    inherit (darwin.apple_sdk.frameworks) AppKit Security Foundation Cocoa;
  };

  idrisPackages = dontRecurseIntoAttrs (callPackage ../development/idris-modules {
    idris-no-deps = haskellPackages.idris;
    pkgs = pkgs.__splicedPackages;
  });

  idris = idrisPackages.with-packages [ idrisPackages.base ] ;

  idris2Packages = recurseIntoAttrs (callPackage ../development/compilers/idris2 { });

  inherit (idris2Packages) idris2;

  inherit (callPackage ../development/tools/database/indradb { })
    indradb-server
    indradb-client;

  instawow = callPackage ../games/instawow/default.nix { };

  intel-graphics-compiler = callPackage ../development/compilers/intel-graphics-compiler { };

  intercal = callPackage ../development/compilers/intercal { };

  irony-server = callPackage ../development/tools/irony-server {
    # The repository of irony to use -- must match the version of the employed emacs
    # package.  Wishing we could merge it into one irony package, to avoid this issue,
    # but its emacs-side expression is autogenerated, and we can't hook into it (other
    # than peek into its version).
    inherit (emacs.pkgs.melpaStablePackages) irony;
  };

  heptagon = callPackage ../development/compilers/heptagon {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  holo-build = callPackage ../tools/package-management/holo-build { };

  hugs = callPackage ../development/interpreters/hugs { };

  inherit (javaPackages) openjfx11 openjfx15 openjfx17 openjfx19 openjfx20 openjfx21 openjfx22;
  openjfx = openjfx17;

  openjdk8-bootstrap = javaPackages.compiler.openjdk8-bootstrap;
  openjdk8 = javaPackages.compiler.openjdk8;
  openjdk8_headless = javaPackages.compiler.openjdk8.headless;
  jdk8 = openjdk8;
  jdk8_headless = openjdk8_headless;
  jre8 = openjdk8.jre;
  jre8_headless = openjdk8_headless.jre;

  openjdk11-bootstrap = javaPackages.compiler.openjdk11-bootstrap;
  openjdk11 = javaPackages.compiler.openjdk11;
  openjdk11_headless = javaPackages.compiler.openjdk11.headless;
  jdk11 = openjdk11;
  jdk11_headless = openjdk11_headless;

  openjdk17-bootstrap = javaPackages.compiler.openjdk17-bootstrap;
  openjdk17 = javaPackages.compiler.openjdk17;
  openjdk17_headless = javaPackages.compiler.openjdk17.headless;
  jdk17 = openjdk17;
  jdk17_headless = openjdk17_headless;

  openjdk16-bootstrap = javaPackages.compiler.openjdk16-bootstrap;

  openjdk19 = javaPackages.compiler.openjdk19;
  openjdk19_headless = javaPackages.compiler.openjdk19.headless;
  jdk19 = openjdk19;
  jdk19_headless = openjdk19_headless;

  openjdk20 = javaPackages.compiler.openjdk20;
  openjdk20_headless = javaPackages.compiler.openjdk20.headless;
  jdk20 = openjdk20;
  jdk20_headless = openjdk20_headless;

  openjdk21 = javaPackages.compiler.openjdk21;
  openjdk21_headless = javaPackages.compiler.openjdk21.headless;
  jdk21 = openjdk21;
  jdk21_headless = openjdk21_headless;

  openjdk22 = javaPackages.compiler.openjdk22;
  openjdk22_headless = javaPackages.compiler.openjdk22.headless;
  jdk22 = openjdk22;
  jdk22_headless = openjdk22_headless;

  /* default JDK */
  jdk = jdk21;
  jdk_headless = jdk21_headless;

  # Since the introduction of the Java Platform Module System in Java 9, Java
  # no longer ships a separate JRE package.
  #
  # If you are building a 'minimal' system/image, you are encouraged to use
  # 'jre_minimal' to build a bespoke JRE containing only the modules you need.
  #
  # For a general-purpose system, 'jre' defaults to the full JDK:
  jre = jdk;
  jre_headless = jdk_headless;

  jre17_minimal = callPackage ../development/compilers/openjdk/jre.nix {
    jdk = jdk17;
  };
  jre_minimal = callPackage ../development/compilers/openjdk/jre.nix { };

  openjdk = jdk;
  openjdk_headless = jdk_headless;

  graalvmCEPackages =
    recurseIntoAttrs (callPackage ../development/compilers/graalvm/community-edition { });
  graalvm-ce = graalvmCEPackages.graalvm-ce;
  buildGraalvmNativeImage = (callPackage ../build-support/build-graalvm-native-image {
    graalvmDrv = graalvm-ce;
  }).override;

  openshot-qt = libsForQt5.callPackage ../applications/video/openshot-qt { };

  lingua-franca = callPackage ../development/compilers/lingua-franca { };

  openspin = callPackage ../development/compilers/openspin { };

  oraclejdk = jdkdistro true false;

  oraclejdk8 = oraclejdk8distro true false;

  oraclejre = lowPrio (jdkdistro false false);

  oraclejre8 = lowPrio (oraclejdk8distro false false);

  jrePlugin = jre8Plugin;

  jre8Plugin = lowPrio (oraclejdk8distro false true);

  jdkdistro = oraclejdk8distro;

  oraclejdk8distro = installjdk: pluginSupport:
    (callPackage ../development/compilers/oraclejdk/jdk8-linux.nix {
      inherit installjdk pluginSupport;
    });

  oraclejdk11 = callPackage ../development/compilers/oraclejdk/jdk11-linux.nix { };

  jasmin = callPackage ../development/compilers/jasmin { };

  java-service-wrapper = callPackage ../tools/system/java-service-wrapper { };

  jna = callPackage ../development/java-modules/jna { };

  javacard-devkit = pkgsi686Linux.callPackage ../development/compilers/javacard-devkit { };

  juniper = callPackage ../development/compilers/juniper { };

  inherit (callPackage ../development/compilers/julia { })
    julia_16-bin
    julia_19-bin
    julia_110-bin
    julia_19
    julia_110;

  julia-lts = julia_16-bin;
  julia-stable = julia_110;
  julia = julia-stable;

  julia-lts-bin = julia_16-bin;
  julia-stable-bin = julia_110-bin;
  julia-bin = julia-stable-bin;

  jwasm =  callPackage ../development/compilers/jwasm { };

  kind2 = darwin.apple_sdk_11_0.callPackage ../development/compilers/kind2 { };

  kingstvis = callPackage ../applications/science/electronics/kingstvis { };

  knightos-genkfs = callPackage ../development/tools/knightos/genkfs { };

  regenkfs = callPackage ../development/tools/knightos/regenkfs { };

  knightos-kcc = callPackage ../development/tools/knightos/kcc { };

  knightos-kimg = callPackage ../development/tools/knightos/kimg { };

  knightos-kpack = callPackage ../development/tools/knightos/kpack { };

  knightos-mkrom = callPackage ../development/tools/knightos/mkrom { };

  remkrom = callPackage ../development/tools/knightos/remkrom { };

  knightos-patchrom = callPackage ../development/tools/knightos/patchrom { };

  knightos-mktiupgrade = callPackage ../development/tools/knightos/mktiupgrade { };

  knightos-scas = callPackage ../development/tools/knightos/scas { };

  knightos-z80e = callPackage ../development/tools/knightos/z80e { };

  koka = haskell.lib.compose.justStaticExecutables (haskellPackages.callPackage ../development/compilers/koka { });

  kotlin = callPackage ../development/compilers/kotlin { };
  kotlin-native = callPackage ../development/compilers/kotlin/native.nix { };

  lazarus = callPackage ../development/compilers/fpc/lazarus.nix {
    fpc = fpc;
  };

  lazarus-qt = libsForQt5.callPackage ../development/compilers/fpc/lazarus.nix {
    fpc = fpc;
    withQt = true;
  };

  lessc = nodePackages.less;

  liquibase = callPackage ../development/tools/database/liquibase { };

  lizardfs = callPackage ../tools/filesystems/lizardfs { };

  lobster = callPackage ../development/compilers/lobster {
    inherit (darwin.apple_sdk.frameworks)
      CoreFoundation Cocoa AudioToolbox OpenGL Foundation ForceFeedback;
  };

  lld = llvmPackages.lld;
  lld_12 = llvmPackages_12.lld;
  lld_13 = llvmPackages_13.lld;
  lld_14 = llvmPackages_14.lld;
  lld_15 = llvmPackages_15.lld;
  lld_16 = llvmPackages_16.lld;
  lld_17 = llvmPackages_17.lld;

  lldb = llvmPackages.lldb;
  lldb_12 = llvmPackages_12.lldb;
  lldb_13 = llvmPackages_13.lldb;
  lldb_14 = llvmPackages_14.lldb;
  lldb_15 = llvmPackages_15.lldb;
  lldb_16 = llvmPackages_16.lldb;
  lldb_17 = llvmPackages_17.lldb;

  llvm = llvmPackages.llvm;
  llvm_12 = llvmPackages_12.llvm;
  llvm_13 = llvmPackages_13.llvm;
  llvm_14 = llvmPackages_14.llvm;
  llvm_15 = llvmPackages_15.llvm;
  llvm_16 = llvmPackages_16.llvm;
  llvm_17 = llvmPackages_17.llvm;

  mlir_16 = llvmPackages_16.mlir;
  mlir_17 = llvmPackages_17.mlir;

  libllvm = llvmPackages.libllvm;
  llvm-manpages = llvmPackages.llvm-manpages;

  llvmPackages = let
    # This returns the minimum supported version for the platform. The
    # assumption is that or any later version is good.
    choose = platform:
      /**/ if platform.isDarwin then 16
      else if platform.isFreeBSD then 16
      else if platform.isOpenBSD then 18
      else if platform.isAndroid then 12
      else if platform.isLinux then 17
      else if platform.isWasm then 16
      # For unknown systems, assume the latest version is required.
      else 17;
    # We take the "max of the mins". Why? Since those are lower bounds of the
    # supported version set, this is like intersecting those sets and then
    # taking the min bound of that.
    minSupported = toString (lib.trivial.max (choose stdenv.hostPlatform) (choose
      stdenv.targetPlatform));
  in pkgs.${"llvmPackages_${minSupported}"};

  llvmPackages_12 = recurseIntoAttrs (callPackage ../development/compilers/llvm/12 ({
    inherit (stdenvAdapters) overrideCC;
    buildLlvmTools = buildPackages.llvmPackages_12.tools;
    targetLlvmLibraries = targetPackages.llvmPackages_12.libraries or llvmPackages_12.libraries;
    targetLlvm = targetPackages.llvmPackages_12.llvm or llvmPackages_12.llvm;
  }));

  inherit (rec {
    llvmPackagesSet = recurseIntoAttrs (callPackages ../development/compilers/llvm { });

    llvmPackages_13 = llvmPackagesSet."13";
    llvmPackages_14 = llvmPackagesSet."14";
    llvmPackages_15 = llvmPackagesSet."15";
    llvmPackages_16 = llvmPackagesSet."16";
    llvmPackages_17 = llvmPackagesSet."17";

    llvmPackages_18 = llvmPackagesSet."18";
    clang_18 = llvmPackages_18.clang;
    lld_18 = llvmPackages_18.lld;
    lldb_18 = llvmPackages_18.lldb;
    llvm_18 = llvmPackages_18.llvm;

    llvmPackages_git = llvmPackagesSet.git;
  }) llvmPackages_13
    llvmPackages_14
    llvmPackages_15
    llvmPackages_16
    llvmPackages_17
    llvmPackages_18
    clang_18
    lld_18
    lldb_18
    llvm_18
    llvmPackages_git;

  lorri = callPackage ../tools/misc/lorri {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  lunarml = callPackage ../development/compilers/lunarml { };

  manticore = callPackage ../development/compilers/manticore { };

  marst = callPackage ../development/compilers/marst { };

  mercury = callPackage ../development/compilers/mercury {
    jdk_headless = openjdk8_headless; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  microscheme = callPackage ../development/compilers/microscheme { };

  minimacy = callPackage ../development/compilers/minimacy { };

  mint = callPackage ../development/compilers/mint { crystal = crystal_1_9; };

  mitama-cpp-result = callPackage ../development/libraries/mitama-cpp-result { };

  mitscheme = callPackage ../development/compilers/mit-scheme {
    texinfo = texinfo6;
  };

  mitschemeX11 = mitscheme.override {
    enableX11 = true;
  };

  miranda = callPackage ../development/compilers/miranda { };

  mlkit = callPackage ../development/compilers/mlkit {};

  inherit (callPackage ../development/compilers/mlton {})
    mlton20130715
    mlton20180207Binary
    mlton20180207
    mlton20210117
    mltonHEAD;

  mlton = mlton20210117;

  mono = mono6;

  mono4 = lowPrio (callPackage ../development/compilers/mono/4.nix {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) Foundation;
  });

  mono5 = callPackage ../development/compilers/mono/5.nix {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  mono6 = callPackage ../development/compilers/mono/6.nix {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  monoDLLFixer = callPackage ../build-support/mono-dll-fixer { };

  roslyn = callPackage ../development/compilers/roslyn { };

  msbuild = callPackage ../development/tools/build-managers/msbuild { };

  mosml = callPackage ../development/compilers/mosml { };

  mozart2 = callPackage ../development/compilers/mozart {
    emacs = emacs-nox;
    jre_headless = jre8_headless; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  mozart2-binary = callPackage ../development/compilers/mozart/binary.nix { };

  muonlang = callPackage ../development/compilers/muonlang { };

  inherit (callPackages ../development/compilers/nim
                        { inherit (darwin) Security;  }
          ) nim-unwrapped-1 nim-unwrapped-2 nim1 nim2;
  nim = nim2;
  buildNimPackage = callPackage ../development/compilers/nim/build-nim-package.nix { };
  nimOverrides = callPackage ./nim-overrides.nix { };

  neko = callPackage ../development/compilers/neko { };

  nextpnrWithGui = libsForQt5.callPackage ../by-name/ne/nextpnr/package.nix {
    enableGui = true;
    inherit (darwin.apple_sdk.frameworks) OpenGL;
  };

  acme = callPackage ../development/compilers/acme { };

  nasm = callPackage ../development/compilers/nasm { };

  nasmfmt = callPackage ../development/tools/nasmfmt { };

  nqc = callPackage ../development/compilers/nqc { };

  nvidia_cg_toolkit = callPackage ../development/compilers/nvidia-cg-toolkit { };

  obliv-c = callPackage ../development/compilers/obliv-c {
    stdenv = gcc10Stdenv;
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  ocaml-ng = callPackage ./ocaml-packages.nix { } // { __attrsFailEvaluation = true; };
  ocaml = ocamlPackages.ocaml;

  ocamlPackages = recurseIntoAttrs ocaml-ng.ocamlPackages;

  ocaml-crunch = ocamlPackages.crunch.bin;

  inherit (ocamlPackages)
    ocamlformat # latest version
    ocamlformat_0_19_0 ocamlformat_0_20_0 ocamlformat_0_20_1 ocamlformat_0_21_0
    ocamlformat_0_22_4 ocamlformat_0_23_0 ocamlformat_0_24_1 ocamlformat_0_25_1
    ocamlformat_0_26_0 ocamlformat_0_26_1 ocamlformat_0_26_2;

  inherit (ocamlPackages) odig;

  orc = callPackage ../development/compilers/orc { };

  orocos-kdl = callPackage ../development/libraries/orocos-kdl { };

  ber_metaocaml = callPackage ../development/compilers/ocaml/ber-metaocaml.nix { };

  ocaml_make = callPackage ../development/ocaml-modules/ocamlmake { };

  ocaml-top = callPackage ../development/tools/ocaml/ocaml-top { };

  ocsigen-i18n = callPackage ../development/tools/ocaml/ocsigen-i18n { };

  opaline = callPackage ../development/tools/ocaml/opaline { };

  opam = callPackage ../development/tools/ocaml/opam {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  opam-installer = callPackage ../development/tools/ocaml/opam/installer.nix { };

  opam2json = callPackage ../development/tools/ocaml/opam2json { };

  wrapWatcom = callPackage ../development/compilers/open-watcom/wrapper.nix { };
  open-watcom-v2-unwrapped = callPackage ../development/compilers/open-watcom/v2.nix { };
  open-watcom-v2 = wrapWatcom open-watcom-v2-unwrapped { };
  open-watcom-bin-unwrapped = callPackage ../development/compilers/open-watcom/bin.nix { };
  open-watcom-bin = wrapWatcom open-watcom-bin-unwrapped { };

  passerine = callPackage ../development/compilers/passerine { };

  pforth = callPackage ../development/compilers/pforth { };

  picat = callPackage ../development/compilers/picat { };

  ponyc = callPackage ../development/compilers/ponyc {
    # Upstream pony no longer supports GCC
    stdenv = llvmPackages.stdenv;
  };

  blaze = callPackage ../development/libraries/blaze { };

  pony-corral = callPackage ../development/compilers/ponyc/pony-corral.nix { };

  prqlc = callPackage ../development/tools/database/prqlc { };

  qbe = callPackage ../development/compilers/qbe { };

  replibyte = callPackage ../development/tools/database/replibyte {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  rgbds = callPackage ../development/compilers/rgbds { };

  rml = callPackage ../development/compilers/rml {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  rgxg = callPackage ../tools/text/rgxg { };

  rtags = callPackage ../development/tools/rtags {
    inherit (darwin) apple_sdk;
  };

  wrapRustcWith = { rustc-unwrapped, ... } @ args: callPackage ../build-support/rust/rustc-wrapper args;
  wrapRustc = rustc-unwrapped: wrapRustcWith { inherit rustc-unwrapped; };

  rust_1_77 = callPackage ../development/compilers/rust/1_77.nix {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security SystemConfiguration;
    llvm_17 = llvmPackages_17.libllvm;
  };
  rust_1_79 = callPackage ../development/compilers/rust/1_79.nix {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security SystemConfiguration;
    llvm_18 = llvmPackages_18.libllvm;
  };
  rust = rust_1_77;

  mrustc = callPackage ../development/compilers/mrustc { };
  mrustc-minicargo = callPackage ../development/compilers/mrustc/minicargo.nix { };
  mrustc-bootstrap = callPackage ../development/compilers/mrustc/bootstrap.nix {
    openssl = openssl_1_1;
  };

  rustPackages_1_77 = rust_1_77.packages.stable;
  rustPackages_1_79 = rust_1_79.packages.stable;
  rustPackages = rustPackages_1_77;

  inherit (rustPackages) cargo cargo-auditable cargo-auditable-cargo-wrapper clippy rustc rustPlatform;

  makeRustPlatform = callPackage ../development/compilers/rust/make-rust-platform.nix { };

  buildRustCrate = callPackage ../build-support/rust/build-rust-crate { };
  buildRustCrateHelpers = callPackage ../build-support/rust/build-rust-crate/helpers.nix { };

  cargo2junit = callPackage ../development/tools/rust/cargo2junit { };

  cargo-web = callPackage ../development/tools/rust/cargo-web {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  cargo-flamegraph = callPackage ../development/tools/rust/cargo-flamegraph {
    inherit (darwin.apple_sdk.frameworks) Security;
    inherit (linuxPackages) perf;
  };

  defaultCrateOverrides = callPackage ../build-support/rust/default-crate-overrides.nix { };

  cargo-about = callPackage ../development/tools/rust/cargo-about { };
  cargo-all-features = callPackage ../development/tools/rust/cargo-all-features { };
  cargo-apk = callPackage ../development/tools/rust/cargo-apk { };
  cargo-audit = callPackage ../development/tools/rust/cargo-audit {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };
  cargo-benchcmp = callPackage ../development/tools/rust/cargo-benchcmp { };
  cargo-binstall = callPackage ../development/tools/rust/cargo-binstall { };
  cargo-bisect-rustc = callPackage ../development/tools/rust/cargo-bisect-rustc { };
  cargo-bitbake = callPackage ../development/tools/rust/cargo-bitbake { };
  cargo-c = callPackage ../development/tools/rust/cargo-c {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security;
  };
  cargo-clone = callPackage ../development/tools/rust/cargo-clone {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security SystemConfiguration;
  };
  cargo-codspeed = callPackage ../development/tools/rust/cargo-codspeed {
    rustPlatform = makeRustPlatform {
      stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
      inherit rustc cargo;
    };
  };
  cargo-component = callPackage ../development/tools/rust/cargo-component { };
  cargo-cranky = callPackage ../development/tools/rust/cargo-cranky { };
  cargo-criterion = callPackage ../development/tools/rust/cargo-criterion { };
  cargo-cyclonedx = callPackage ../development/tools/rust/cargo-cyclonedx {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration CoreFoundation;
  };
  cargo-deadlinks = callPackage ../development/tools/rust/cargo-deadlinks {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-deb = callPackage ../development/tools/rust/cargo-deb { };
  cargo-docset = callPackage ../development/tools/rust/cargo-docset { };
  cargo-duplicates = callPackage ../development/tools/rust/cargo-duplicates { };
  cargo-edit = callPackage ../development/tools/rust/cargo-edit {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-leptos = callPackage ../development/tools/rust/cargo-leptos { };
  cargo-kcov = callPackage ../development/tools/rust/cargo-kcov { };
  cargo-guppy = callPackage ../development/tools/rust/cargo-guppy { };
  cargo-hack = callPackage ../development/tools/rust/cargo-hack { };
  cargo-license = callPackage ../development/tools/rust/cargo-license { };
  cargo-llvm-cov = callPackage ../development/tools/rust/cargo-llvm-cov { };
  cargo-llvm-lines = callPackage ../development/tools/rust/cargo-llvm-lines { };
  cargo-local-registry = callPackage ../development/tools/rust/cargo-local-registry { };
  cargo-lock = callPackage ../development/tools/rust/cargo-lock { };
  cargo-machete = callPackage ../development/tools/rust/cargo-machete { };
  cargo-outdated = callPackage ../development/tools/rust/cargo-outdated {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation CoreServices Security SystemConfiguration;
  };
  inherit (callPackages ../development/tools/rust/cargo-pgrx { })
    cargo-pgrx_0_10_2
    cargo-pgrx_0_11_2
    cargo-pgrx_0_11_3
    ;
  cargo-pgrx = cargo-pgrx_0_11_2;

  buildPgrxExtension = callPackage ../development/tools/rust/cargo-pgrx/buildPgrxExtension.nix {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-release = callPackage ../development/tools/rust/cargo-release { };
  cargo-rr = callPackage ../development/tools/rust/cargo-rr { };
  cargo-tarpaulin = callPackage ../development/tools/analysis/cargo-tarpaulin {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-update = callPackage ../development/tools/rust/cargo-update { };

  cargo-asm = callPackage ../development/tools/rust/cargo-asm {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-bazel = callPackage ../development/tools/rust/cargo-bazel {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-binutils = callPackage ../development/tools/rust/cargo-binutils { };
  cargo-bolero = callPackage ../development/tools/rust/cargo-bolero { };
  cargo-bundle = callPackage ../development/tools/rust/cargo-bundle { };
  cargo-bundle-licenses = callPackage ../development/tools/rust/cargo-bundle-licenses { };
  cargo-cache = callPackage ../development/tools/rust/cargo-cache {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-careful = callPackage ../development/tools/rust/cargo-careful { };
  cargo-chef = callPackage ../development/tools/rust/cargo-chef { };
  cargo-crev = callPackage ../development/tools/rust/cargo-crev {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration CoreFoundation;
  };
  cargo-cross = callPackage ../development/tools/rust/cargo-cross { };
  cargo-deny = callPackage ../development/tools/rust/cargo-deny { };
  cargo-depgraph = callPackage ../development/tools/rust/cargo-depgraph { };
  cargo-dephell = callPackage ../development/tools/rust/cargo-dephell { };
  cargo-diet = callPackage ../development/tools/rust/cargo-diet { };
  cargo-dist = callPackage ../development/tools/rust/cargo-dist { };
  cargo-espmonitor = callPackage ../development/tools/rust/cargo-espmonitor { };
  cargo-hakari = callPackage ../development/tools/rust/cargo-hakari { };
  cargo-feature = callPackage ../development/tools/rust/cargo-feature { };
  cargo-fund = callPackage ../development/tools/rust/cargo-fund {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-fuzz = callPackage ../development/tools/rust/cargo-fuzz { };

  cargo-hf2 = callPackage ../development/tools/rust/cargo-hf2 {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };
  cargo-info = callPackage ../development/tools/rust/cargo-info { };
  cargo-inspect = callPackage ../development/tools/rust/cargo-inspect {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-insta = callPackage ../development/tools/rust/cargo-insta { };
  cargo-lambda = callPackage ../development/tools/rust/cargo-lambda {
    zig = zig_0_12;
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };
  cargo-limit = callPackage ../development/tools/rust/cargo-limit { };
  cargo-modules = callPackage ../development/tools/rust/cargo-modules { };
  cargo-mommy = callPackage ../development/tools/rust/cargo-mommy { };
  cargo-msrv = callPackage ../development/tools/rust/cargo-msrv {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-mutants = callPackage ../development/tools/rust/cargo-mutants { };

  cargo-ndk = callPackage ../development/tools/rust/cargo-ndk {
    inherit (darwin.apple_sdk.frameworks) CoreGraphics Foundation;
  };

  cargo-nextest = callPackage ../development/tools/rust/cargo-nextest { };
  cargo-play = callPackage ../development/tools/rust/cargo-play { };
  cargo-rdme = callPackage ../by-name/ca/cargo-rdme/package.nix {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-readme = callPackage ../development/tools/rust/cargo-readme { };
  cargo-risczero = callPackage ../development/tools/rust/cargo-risczero { };
  cargo-run-bin = callPackage ../development/tools/rust/cargo-run-bin {};
  cargo-semver-checks = callPackage ../development/tools/rust/cargo-semver-checks { };

  cargo-show-asm = callPackage ../development/tools/rust/cargo-show-asm { };
  cargo-shuttle = callPackage ../development/tools/rust/cargo-shuttle { };

  cargo-sort = callPackage ../development/tools/rust/cargo-sort { };
  cargo-spellcheck = callPackage ../development/tools/rust/cargo-spellcheck {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };
  cargo-supply-chain = callPackage ../development/tools/rust/cargo-supply-chain { };
  cargo-sweep = callPackage ../development/tools/rust/cargo-sweep { };
  cargo-sync-readme = callPackage ../development/tools/rust/cargo-sync-readme { };
  cargo-tally = callPackage ../development/tools/rust/cargo-tally { };
  cargo-temp = callPackage ../development/tools/rust/cargo-temp { };
  cargo-toml-lint = callPackage ../development/tools/rust/cargo-toml-lint { };
  cargo-udeps = callPackage ../development/tools/rust/cargo-udeps {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security SystemConfiguration;
  };
  cargo-ui = callPackage ../development/tools/rust/cargo-ui { };
  cargo-unused-features = callPackage ../development/tools/rust/cargo-unused-features { };

  cargo-tauri = callPackage ../development/tools/rust/cargo-tauri { };
  cargo-mobile2 = callPackage ../development/tools/rust/cargo-mobile2 { };

  cargo-valgrind = callPackage ../development/tools/rust/cargo-valgrind { };
  cargo-vet = callPackage ../development/tools/rust/cargo-vet {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  cargo-watch = callPackage ../development/tools/rust/cargo-watch {
    inherit (darwin.apple_sdk.frameworks) Foundation Cocoa;
  };
  cargo-wipe = callPackage ../development/tools/rust/cargo-wipe { };
  cargo-workspaces = callPackage ../development/tools/rust/cargo-workspaces { };
  cargo-xbuild = callPackage ../development/tools/rust/cargo-xbuild { };
  cargo-generate = callPackage ../development/tools/rust/cargo-generate { };
  cargo-bootimage = callPackage ../development/tools/rust/bootimage { };

  cargo-whatfeatures = callPackage ../development/tools/rust/cargo-whatfeatures {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  cargo-public-api = callPackage ../development/tools/rust/cargo-public-api { };

  cargo-zigbuild = callPackage ../development/tools/rust/cargo-zigbuild {
    zig = buildPackages.zig_0_12;
  };

  cauwugo = callPackage ../development/tools/rust/cauwugo { };

  crate2nix = callPackage ../development/tools/rust/crate2nix { };

  critcmp = callPackage ../development/tools/rust/critcmp { };

  devspace = callPackage ../development/tools/misc/devspace { };

  djlint = callPackage ../development/tools/djlint { };

  leptosfmt = callPackage ../development/tools/rust/leptosfmt { };

  maturin = callPackage ../development/tools/rust/maturin {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  panamax = callPackage ../development/tools/rust/panamax { };

  ograc = callPackage ../development/tools/rust/ograc { };

  opensmalltalk-vm = callPackage ../development/compilers/opensmalltalk-vm { };

  opensycl = darwin.apple_sdk_11_0.callPackage ../development/compilers/opensycl { };
  opensyclWithRocm = opensycl.override { rocmSupport = true; };

  pest-ide-tools = callPackage ../development/tools/misc/pest-ide-tools { };

  ravedude = callPackage ../development/tools/rust/ravedude { };

  ra-multiplex = callPackage ../development/tools/rust/ra-multiplex {};

  rhack = callPackage ../development/tools/rust/rhack { };
  roogle = callPackage ../development/tools/rust/roogle { };
  rustfmt = rustPackages.rustfmt;
  rust-analyzer-unwrapped = callPackage ../development/tools/rust/rust-analyzer {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };
  rust-analyzer = callPackage ../development/tools/rust/rust-analyzer/wrapper.nix { };
  rust-audit-info = callPackage ../development/tools/rust/rust-audit-info { };
  rust-bindgen-unwrapped = callPackage ../development/tools/rust/bindgen/unwrapped.nix { };
  rust-bindgen = callPackage ../development/tools/rust/bindgen { };
  rust-cbindgen = callPackage ../development/tools/rust/cbindgen {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  rust-script = callPackage ../development/tools/rust/rust-script { };
  rustup = callPackage ../development/tools/rust/rustup {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };
  rustup-toolchain-install-master = callPackage ../development/tools/rust/rustup-toolchain-install-master {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  rusty-man = callPackage ../development/tools/rust/rusty-man { };
  rustycli = callPackage ../development/tools/rust/rustycli { };
  specr-transpile = callPackage ../development/tools/rust/specr-transpile { };
  typeshare = callPackage ../development/tools/rust/typeshare { };

  sagittarius-scheme = callPackage ../development/compilers/sagittarius-scheme { };

  roswell = callPackage ../development/tools/roswell { };

  scala_2_10 = callPackage ../development/compilers/scala/2.x.nix { majorVersion = "2.10"; jre = jdk8; };
  scala_2_11 = callPackage ../development/compilers/scala/2.x.nix { majorVersion = "2.11"; jre = jdk8; };
  scala_2_12 = callPackage ../development/compilers/scala/2.x.nix { majorVersion = "2.12"; };
  scala_2_13 = callPackage ../development/compilers/scala/2.x.nix { majorVersion = "2.13"; };
  scala_3 = callPackage ../development/compilers/scala { };

  scala = scala_2_13;
  scala-runners = callPackage ../development/compilers/scala-runners {
    coursier = coursier.override { jre = jdk8; };
  };

  scalafix = callPackage ../development/tools/scalafix {
    jre = jre8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };
  scalafmt = callPackage ../development/tools/scalafmt { };

  scryer-prolog = callPackage ../development/compilers/scryer-prolog { };

  seren = callPackage ../applications/networking/instant-messengers/seren { };

  serialdv = callPackage ../development/libraries/serialdv {  };

  serpent = callPackage ../development/compilers/serpent { };

  shmig = callPackage ../development/tools/database/shmig { };

  sleek = callPackage ../development/tools/database/sleek { };

  smlfmt = callPackage ../development/tools/smlfmt { };

  # smlnjBootstrap should be redundant, now that smlnj works on Darwin natively
  smlnjBootstrap = callPackage ../development/compilers/smlnj/bootstrap.nix { };
  smlnj = callPackage ../development/compilers/smlnj {
    inherit (darwin) Libsystem;
  };

  smlpkg = callPackage ../tools/package-management/smlpkg { };

  solc = callPackage ../development/compilers/solc { };

  souffle = callPackage ../development/compilers/souffle { };

  spasm-ng = callPackage ../development/compilers/spasm-ng { };

  spirv-llvm-translator = callPackage ../development/compilers/spirv-llvm-translator { };

  sqldeveloper = callPackage ../development/tools/database/sqldeveloper {
    jdk = oraclejdk;
  };

  sqlfluff = callPackage ../development/tools/database/sqlfluff { };

  sqlx-cli = callPackage ../development/tools/rust/sqlx-cli {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration CoreFoundation Security;
  };

  squeak = callPackage ../development/compilers/squeak {
    stdenv = clangStdenv;
  };

  squirrel-sql = callPackage ../development/tools/database/squirrel-sql {
    drivers = [ jtds_jdbc mssql_jdbc mysql_jdbc postgresql_jdbc ];
  };

  surrealdb-migrations = callPackage ../development/tools/database/surrealdb-migrations {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  stalin = callPackage ../development/compilers/stalin { };

  metaBuildEnv = callPackage ../development/compilers/meta-environment/meta-build-env { };

  svd2rust = callPackage ../development/tools/rust/svd2rust { };

  svdtools = callPackage ../development/embedded/svdtools { };

  swiftPackages = recurseIntoAttrs (callPackage ../development/compilers/swift { });
  inherit (swiftPackages) swift swiftpm sourcekit-lsp swift-format;

  swiftpm2nix = callPackage ../development/tools/swiftpm2nix { };

  swiProlog = callPackage ../development/compilers/swi-prolog {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  swiPrologWithGui = swiProlog.override { withGui = true; };

  tbb_2020_3 = callPackage ../development/libraries/tbb/2020_3.nix { };
  tbb_2021_5 = callPackage ../development/libraries/tbb/2021_5.nix { } ;
  tbb_2021_11 = callPackage ../development/libraries/tbb { };
  # many packages still fail with latest version
  tbb = tbb_2020_3;

  terra = callPackage ../development/compilers/terra {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) Cocoa Foundation;
  };

  teyjus = callPackage ../development/compilers/teyjus {
    inherit (ocaml-ng.ocamlPackages_4_14) buildDunePackage;
  };

  thrust = callPackage ../development/tools/thrust {
    gconf = gnome2.GConf;
  };

  tinycc = darwin.apple_sdk_11_0.callPackage ../development/compilers/tinycc { };

  tinygo = callPackage ../development/compilers/tinygo {
    llvmPackages = llvmPackages_17;
  };

  tinyscheme = callPackage ../development/interpreters/tinyscheme { };

  tbox = callPackage ../development/libraries/tbox { };

  typescript = callPackage ../development/compilers/typescript { };

  bupc = callPackage ../development/compilers/bupc { };

  ubports-click = python3Packages.callPackage ../development/tools/click { };

  uasm = callPackage ../development/compilers/uasm { };

  urn = callPackage ../development/compilers/urn { };

  urweb = callPackage ../development/compilers/urweb {
    icu = icu67;
  };

  urbackup-client = callPackage ../applications/backup/urbackup-client { };

  vlang = callPackage ../development/compilers/vlang { };

  vala-lint = callPackage ../development/tools/vala-lint { };

  vcard = python3Packages.toPythonApplication python3Packages.vcard;

  inherit (callPackage ../development/compilers/vala { })
    vala_0_56
    vala;

  vyper = with python3Packages; toPythonApplication vyper;

  wazero = callPackage ../development/interpreters/wazero { };

  wcc = callPackage ../development/compilers/wcc { };

  wla-dx = callPackage ../development/compilers/wla-dx { };

  wrapCCWith =
    { cc
    , # This should be the only bintools runtime dep with this sort of logic. The
      # Others should instead delegate to the next stage's choice with
      # `targetPackages.stdenv.cc.bintools`. This one is different just to
      # provide the default choice, avoiding infinite recursion.
      # See the bintools attribute for the logic and reasoning. We need to provide
      # a default here, since eval will hit this function when bootstrapping
      # stdenv where the bintools attribute doesn't exist, but will never actually
      # be evaluated -- callPackage ends up being too eager.
      bintools ? pkgs.bintools
    , libc ? bintools.libc
    , # libc++ from the default LLVM version is bound at the top level, but we
      # want the C++ library to be explicitly chosen by the caller, and null by
      # default.
      libcxx ? null
    , extraPackages ? lib.optional (cc.isGNU or false && stdenv.targetPlatform.isMinGW) ((threadsCrossFor cc.version).package)
    , nixSupport ? {}
    , ...
    } @ extraArgs:
      callPackage ../build-support/cc-wrapper (let self = {
    nativeTools = stdenv.targetPlatform == stdenv.hostPlatform && stdenv.cc.nativeTools or false;
    nativeLibc = stdenv.targetPlatform == stdenv.hostPlatform && stdenv.cc.nativeLibc or false;
    nativePrefix = stdenv.cc.nativePrefix or "";
    noLibc = !self.nativeLibc && (self.libc == null);

    isGNU = cc.isGNU or false;
    isClang = cc.isClang or false;

    inherit cc bintools libc libcxx extraPackages nixSupport zlib;
  } // extraArgs; in self);

  wrapCC = cc: wrapCCWith {
    inherit cc;
  };

  wrapBintoolsWith =
    { bintools
    , libc ? if stdenv.targetPlatform != stdenv.hostPlatform then libcCross else stdenv.cc.libc
    , ...
    } @ extraArgs:
      callPackage ../build-support/bintools-wrapper (let self = {
    nativeTools = stdenv.targetPlatform == stdenv.hostPlatform && stdenv.cc.nativeTools or false;
    nativeLibc = stdenv.targetPlatform == stdenv.hostPlatform && stdenv.cc.nativeLibc or false;
    nativePrefix = stdenv.cc.nativePrefix or "";

    noLibc = (self.libc == null);

    inherit bintools libc;
    inherit (darwin) postLinkSignHook signingUtils;
  } // extraArgs; in self);

  yaml-language-server = callPackage  ../development/tools/language-servers/yaml-language-server { };

  # prolog
  yap = callPackage ../development/compilers/yap { };

  yasm = callPackage ../development/compilers/yasm { };

  yosys = callPackage ../development/compilers/yosys { };
  yosys-bluespec = callPackage ../development/compilers/yosys/plugins/bluespec.nix { };
  yosys-ghdl = callPackage ../development/compilers/yosys/plugins/ghdl.nix { };
  yosys-synlig = callPackage ../development/compilers/yosys/plugins/synlig.nix { };
  yosys-symbiflow = callPackage ../development/compilers/yosys/plugins/symbiflow.nix { };

  z88dk = callPackage ../development/compilers/z88dk { };

  zulip = callPackage ../applications/networking/instant-messengers/zulip { };

  zulip-term = callPackage ../applications/networking/instant-messengers/zulip-term { };

  zulu8 = callPackage ../development/compilers/zulu/8.nix { };
  zulu11 = callPackage ../development/compilers/zulu/11.nix { };
  zulu17 = callPackage ../development/compilers/zulu/17.nix { };
  zulu21 = callPackage ../development/compilers/zulu/21.nix { };
  zulu = zulu21;

  ### DEVELOPMENT / INTERPRETERS

  acl2 = callPackage ../development/interpreters/acl2 { };
  acl2-minimal = callPackage ../development/interpreters/acl2 { certifyBooks = false; };

  angelscript = callPackage ../development/interpreters/angelscript { };

  anko = callPackage ../development/interpreters/anko { };

  babashka-unwrapped = callPackage ../development/interpreters/babashka { };
  babashka = callPackage ../development/interpreters/babashka/wrapped.nix { };

  # BQN interpreters and compilers

  mbqn = callPackage ../development/interpreters/bqn/mlochbaum-bqn { };

  cbqn = cbqn-bootstrap.phase2;
  cbqn-replxx = cbqn-bootstrap.phase2-replxx;
  cbqn-standalone = cbqn-bootstrap.phase0;
  cbqn-standalone-replxx = cbqn-bootstrap.phase0-replxx;

  # Below, the classic self-bootstrapping process
  cbqn-bootstrap = lib.dontRecurseIntoAttrs {
    # Use clang to compile CBQN if we aren't already.
    # CBQN's upstream primarily targets and tests clang which means using gcc
    # will result in slower binaries and on some platforms failing/broken builds.
    # See https://github.com/dzaima/CBQN/issues/12.
    #
    # Known issues:
    #
    # * CBQN using gcc is broken at runtime on i686 due to
    #   https://gcc.gnu.org/bugzilla/show_bug.cgi?id=58416,
    # * CBQN uses some CPP macros gcc doesn't like for aarch64.
    stdenv = if !stdenv.cc.isClang then clangStdenv else stdenv;

    mbqn-source = buildPackages.mbqn.src;

    phase0 = callPackage ../development/interpreters/bqn/cbqn {
      inherit (cbqn-bootstrap) mbqn-source stdenv;
      genBytecode = false;
      bqn-path = null;
    };

    phase0-replxx = callPackage ../development/interpreters/bqn/cbqn {
      inherit (cbqn-bootstrap) mbqn-source stdenv;
      genBytecode = false;
      bqn-path = null;
      enableReplxx = true;
    };

    phase1 = callPackage ../development/interpreters/bqn/cbqn {
      inherit (cbqn-bootstrap) mbqn-source stdenv;
      genBytecode = true;
      bqn-path = "${buildPackages.cbqn-bootstrap.phase0}/bin/cbqn";
    };

    phase2 = callPackage ../development/interpreters/bqn/cbqn {
      inherit (cbqn-bootstrap) mbqn-source stdenv;
      genBytecode = true;
      bqn-path = "${buildPackages.cbqn-bootstrap.phase1}/bin/cbqn";
    };

    phase2-replxx = callPackage ../development/interpreters/bqn/cbqn {
      inherit (cbqn-bootstrap) mbqn-source stdenv;
      genBytecode = true;
      bqn-path = "${buildPackages.cbqn-bootstrap.phase1}/bin/cbqn";
      enableReplxx = true;
    };
  };

  dbqn = callPackage ../development/interpreters/bqn/dzaima-bqn {
    buildNativeImage = false;
    stdenv = stdenvNoCC;
    jdk = jre;
  };
  dbqn-native = callPackage ../development/interpreters/bqn/dzaima-bqn {
    buildNativeImage = true;
    jdk = graalvm-ce;
  };

  chibi = callPackage ../development/interpreters/chibi { };

  ceptre = callPackage ../development/interpreters/ceptre { };

  cg3 = callPackage ../development/interpreters/cg3 { };

  cling = callPackage ../development/interpreters/cling { };

  clips = callPackage ../development/interpreters/clips { };

  cliscord = callPackage ../misc/cliscord {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  clojupyter = callPackage ../applications/editors/jupyter-kernels/clojupyter {
    jre = jre8;
  };

  inherit (callPackage ../applications/editors/jupyter-kernels/xeus-cling { })
    cpp11-kernel cpp14-kernel cpp17-kernel cpp2a-kernel;
  xeus-cling = callPackage ../applications/editors/jupyter-kernels/xeus-cling/xeus-cling.nix { };

  clojure = callPackage ../development/interpreters/clojure {
    # set this to an LTS version of java
    jdk = jdk21;
  };

  clojure-lsp = callPackage ../development/tools/misc/clojure-lsp { };

  clooj = callPackage ../development/interpreters/clojure/clooj.nix { };

  dhall = haskell.lib.compose.justStaticExecutables haskellPackages.dhall;

  dhall-bash = haskell.lib.compose.justStaticExecutables haskellPackages.dhall-bash;

  dhall-docs = haskell.lib.compose.justStaticExecutables haskellPackages.dhall-docs;

  dhall-lsp-server = haskell.lib.compose.justStaticExecutables haskellPackages.dhall-lsp-server;

  dhall-json = haskell.lib.compose.justStaticExecutables haskellPackages.dhall-json;

  dhall-nix = haskell.lib.compose.justStaticExecutables haskellPackages.dhall-nix;

  dhall-nixpkgs = haskell.lib.compose.justStaticExecutables haskellPackages.dhall-nixpkgs;

  dhall-yaml = haskell.lib.compose.justStaticExecutables haskellPackages.dhall-yaml;

  dhallPackages = recurseIntoAttrs (callPackage ./dhall-packages.nix { });

  duktape = callPackage ../development/interpreters/duktape { };

  duckscript = callPackage ../development/tools/rust/duckscript {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  evcxr = callPackage ../development/interpreters/evcxr {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  beam = callPackage ./beam-packages.nix { };
  beam_nox = callPackage ./beam-packages.nix { beam = beam_nox; wxSupport = false; };
  beam_minimal = callPackage ./beam-packages.nix {
    beam = beam_minimal;
    wxSupport = false;
    systemdSupport = false;
  };

  inherit (beam.interpreters)
    erlang erlang_27 erlang_26 erlang_25 erlang_24
    erlang_odbc erlang_javac erlang_odbc_javac
    elixir elixir_1_16 elixir_1_15 elixir_1_14 elixir_1_13 elixir_1_12 elixir_1_11 elixir_1_10
    elixir-ls;

  erlang_nox = beam_nox.interpreters.erlang;

  inherit (beam.packages.erlang)
    ex_doc erlang-ls erlfmt elvis-erlang
    rebar rebar3 rebar3WithPlugins
    fetchHex
    lfe lfe_2_1;
  beamPackages = beam.packages.erlang // { __attrsFailEvaluation = true; };

  expr = callPackage ../development/interpreters/expr { };

  genemichaels = callPackage ../development/tools/rust/genemichaels { };

  gnudatalanguage = callPackage ../development/interpreters/gnudatalanguage {
    inherit (llvmPackages) openmp;
    inherit (darwin.apple_sdk.frameworks) Cocoa;
    # MPICH currently build on Darwin
    mpi = mpich;
  };

  gpython = callPackage ../development/interpreters/gpython { };

  graphql-client = callPackage ../development/tools/graphql-client {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  graphqlmap = callPackage ../tools/security/graphqlmap { };

  graphqurl = callPackage ../tools/networking/graphqurl { };

  groovy = callPackage ../development/interpreters/groovy { };

  inherit (callPackages ../applications/networking/cluster/hadoop {})
    hadoop_3_3
    hadoop_3_2
    hadoop2;
  hadoop3 = hadoop_3_3;
  hadoop = hadoop3;

  hashlink = callPackage ../development/interpreters/hashlink { };

  io = callPackage ../development/interpreters/io { };

  ivy = callPackage ../development/interpreters/ivy { };

  j = callPackage ../development/interpreters/j { };

  jacinda = haskell.lib.compose.justStaticExecutables haskellPackages.jacinda;

  janet = callPackage ../development/interpreters/janet { };

  jpm = callPackage ../development/interpreters/janet/jpm.nix { };

  jelly = callPackage ../development/interpreters/jelly { };

  jimtcl = callPackage ../development/interpreters/jimtcl { };

  jmeter = callPackage ../applications/networking/jmeter { };

  joker = callPackage ../development/interpreters/joker { };

  davmail = callPackage ../applications/networking/davmail {
    zulu = zulu11;
  };

  kamilalisp = callPackage ../development/interpreters/kamilalisp { };

  kanif = callPackage ../applications/networking/cluster/kanif { };

  kona = callPackage ../development/interpreters/kona { };

  konf = callPackage ../development/tools/konf { };

  lambda-lisp = callPackage ../development/interpreters/lambda-lisp { };
  lambda-lisp-blc = lambda-lisp;

  lolcode = callPackage ../development/interpreters/lolcode { };

  love_0_10 = callPackage ../development/interpreters/love/0.10.nix { };
  love_11 = callPackage ../development/interpreters/love/11.nix { };
  love = love_11;

  wabt = callPackage ../development/tools/wabt { };

  zuo = callPackage ../development/interpreters/zuo { };

  ### LUA interpreters
  emilua = callPackage ../development/interpreters/emilua { };

  luaInterpreters = callPackage ./../development/interpreters/lua-5 { };
  inherit (luaInterpreters) lua5_1 lua5_2 lua5_2_compat lua5_3 lua5_3_compat lua5_4 lua5_4_compat luajit_2_1 luajit_2_0 luajit_openresty;

  lua5 = lua5_2_compat;
  lua = lua5;

  lua51Packages = recurseIntoAttrs lua5_1.pkgs;
  lua52Packages = recurseIntoAttrs lua5_2.pkgs;
  lua53Packages = recurseIntoAttrs lua5_3.pkgs;
  lua54Packages = recurseIntoAttrs lua5_4.pkgs;
  luajitPackages = recurseIntoAttrs luajit.pkgs;

  luaPackages = lua52Packages;

  luajit = luajit_2_1;

  luarocks = luaPackages.luarocks;
  luarocks-nix = luaPackages.luarocks-nix;

  luarocks-packages-updater = callPackage ../development/lua-modules/updater {
    inherit (python3Packages) buildPythonApplication ;
  };


  luau = callPackage ../development/interpreters/luau { };

  lune = callPackage ../development/interpreters/lune { };

  toluapp = callPackage ../development/tools/toluapp {
    lua = lua5_1; # doesn't work with any other :(
  };

  ### END OF LUA

  ### CuboCore
  CuboCore = recurseIntoAttrs (import ./cubocore-packages.nix {
    inherit newScope lxqt lib libsForQt5;
  });

  ### End of CuboCore

  maude = callPackage ../development/interpreters/maude { };

  me_cleaner = callPackage ../tools/misc/me_cleaner { };

  mesos-dns = callPackage ../servers/mesos-dns { };

  metamath = callPackage ../development/interpreters/metamath { };

  minder = callPackage ../applications/misc/minder { };

  mujs = callPackage ../development/interpreters/mujs { };

  nelua = callPackage ../development/interpreters/nelua { };

  nextflow = callPackage ../development/interpreters/nextflow { };

  ngn-k = callPackage ../development/interpreters/ngn-k { };

  oak = callPackage ../development/interpreters/oak { };

  obb = callPackage ../development/interpreters/clojure/obb.nix { };

  octave = callPackage ../development/interpreters/octave { };

  octaveFull = octave.override {
    enableQt = true;
  };

  octave-kernel = callPackage ../applications/editors/jupyter-kernels/octave { };

  octavePackages = recurseIntoAttrs octave.pkgs;

  pachyderm = callPackage ../applications/networking/cluster/pachyderm { };


  # PHP interpreters, packages and extensions.
  #
  # Set default PHP interpreter, extensions and packages
  php = php82;
  phpExtensions = php.extensions;
  phpPackages = php.packages;

  # Import PHP83 interpreter, extensions and packages
  php83 = callPackage ../development/interpreters/php/8.3.nix {
    stdenv = if stdenv.cc.isClang then llvmPackages.stdenv else stdenv;
    pcre2 = pcre2.override {
      withJitSealloc = false; # See https://bugs.php.net/bug.php?id=78927 and https://bugs.php.net/bug.php?id=78630
    };
  };
  php83Extensions = recurseIntoAttrs php83.extensions;
  php83Packages = recurseIntoAttrs php83.packages;

  # Import PHP82 interpreter, extensions and packages
  php82 = callPackage ../development/interpreters/php/8.2.nix {
    stdenv = if stdenv.cc.isClang then llvmPackages.stdenv else stdenv;
    pcre2 = pcre2.override {
      withJitSealloc = false; # See https://bugs.php.net/bug.php?id=78927 and https://bugs.php.net/bug.php?id=78630
    };
  };
  php82Extensions = recurseIntoAttrs php82.extensions;
  php82Packages = recurseIntoAttrs php82.packages;

  # Import PHP81 interpreter, extensions and packages
  php81 = callPackage ../development/interpreters/php/8.1.nix {
    stdenv = if stdenv.cc.isClang then llvmPackages.stdenv else stdenv;
    pcre2 = pcre2.override {
      withJitSealloc = false; # See https://bugs.php.net/bug.php?id=78927 and https://bugs.php.net/bug.php?id=78630
    };
  };
  php81Extensions = recurseIntoAttrs php81.extensions;
  php81Packages = recurseIntoAttrs php81.packages;

  picoc = callPackage ../development/interpreters/picoc { };

  picolisp = callPackage ../development/interpreters/picolisp { };

  polyml = callPackage ../development/compilers/polyml { };
  polyml56 = callPackage ../development/compilers/polyml/5.6.nix { };
  polyml57 = callPackage ../development/compilers/polyml/5.7.nix { };

  # Python interpreters. All standard library modules are included except for tkinter, which is
  # available as `pythonPackages.tkinter` and can be used as any other Python package.
  # When switching these sets, please update docs at ../../doc/languages-frameworks/python.md
  python2 = python27;
  python3 = python311;

  # pythonPackages further below, but assigned here because they need to be in sync
  python2Packages = dontRecurseIntoAttrs python27Packages;
  python3Packages = dontRecurseIntoAttrs python311Packages;

  pypy = pypy2;
  pypy2 = pypy27;
  pypy3 = pypy39;

  # Python interpreter that is build with all modules, including tkinter.
  # These are for compatibility and should not be used inside Nixpkgs.
  python2Full = python2.override {
    self = python2Full;
    pythonAttr = "python2Full";
    x11Support = true;
  };
  python27Full = python27.override {
    self = python27Full;
    pythonAttr = "python27Full";
    x11Support = true;
  };
  python3Full = python3.override {
    self = python3Full;
    pythonAttr = "python3Full";
    bluezSupport = lib.meta.availableOn stdenv.hostPlatform bluez;
    x11Support = true;
  };
  python39Full = python39.override {
    self = python39Full;
    pythonAttr = "python39Full";
    bluezSupport = lib.meta.availableOn stdenv.hostPlatform bluez;
    x11Support = true;
  };
  python310Full = python310.override {
    self = python310Full;
    pythonAttr = "python310Full";
    bluezSupport = lib.meta.availableOn stdenv.hostPlatform bluez;
    x11Support = true;
  };
  python311Full = python311.override {
    self = python311Full;
    pythonAttr = "python311Full";
    bluezSupport = lib.meta.availableOn stdenv.hostPlatform bluez;
    x11Support = true;
  };
  python312Full = python312.override {
    self = python312Full;
    pythonAttr = "python312Full";
    bluezSupport = lib.meta.availableOn stdenv.hostPlatform bluez;
    x11Support = true;
  };
  python313Full = python313.override {
    self = python313Full;
    pythonAttr = "python313Full";
    bluezSupport = lib.meta.availableOn stdenv.hostPlatform bluez;
    x11Support = true;
  };

  # https://py-free-threading.github.io
  python313FreeThreading = python313.override {
    pythonAttr = "python313FreeThreading";
    enableGIL = false;
  };

  pythonInterpreters = callPackage ./../development/interpreters/python { };
  inherit (pythonInterpreters) python27 python39 python310 python311 python312 python313 python3Minimal pypy27 pypy310 pypy39 rustpython;

  # List of extensions with overrides to apply to all Python package sets.
  pythonPackagesExtensions = [ ];
  # Python package sets.
  python27Packages = python27.pkgs // { __attrsFailEvaluation = true; };
  python39Packages = python39.pkgs // { __attrsFailEvaluation = true; };
  python310Packages = python310.pkgs // { __attrsFailEvaluation = true; };
  python311Packages = recurseIntoAttrs python311.pkgs // { pythonPackages = python311.pkgs // { __attrsFailEvaluation = true; }; };
  python312Packages = recurseIntoAttrs python312.pkgs // { pythonPackages = python312.pkgs // { __attrsFailEvaluation = true; }; };
  python313Packages = python313.pkgs // { __attrsFailEvaluation = true; };
  pypyPackages = pypy.pkgs // { __attrsFailEvaluation = true; };
  pypy2Packages = pypy2.pkgs // { __attrsFailEvaluation = true; };
  pypy27Packages = pypy27.pkgs // { __attrsFailEvaluation = true; };
  pypy3Packages = pypy3.pkgs // { __attrsFailEvaluation = true; };
  pypy39Packages = pypy39.pkgs // { __attrsFailEvaluation = true; };
  pypy310Packages = pypy310.pkgs // { __attrsFailEvaluation = true; };

  py3c = callPackage ../development/libraries/py3c { };

  pythonManylinuxPackages = callPackage ./../development/interpreters/python/manylinux { };

  pythonCondaPackages = callPackage ./../development/interpreters/python/conda { };

  update-python-libraries = callPackage ../development/interpreters/python/update-python-libraries { };

  # Should eventually be moved inside Python interpreters.
  python-setup-hook = buildPackages.callPackage ../development/interpreters/python/setup-hook.nix { };

  pythonDocs = recurseIntoAttrs (callPackage ../development/interpreters/python/cpython/docs {});

  check-jsonschema = callPackage ../development/tools/check-jsonschema { };

  pypi-mirror = callPackage ../development/tools/pypi-mirror { };

  svg2tikz = with python3.pkgs; toPythonApplication svg2tikz;

  svg2pdf = callPackage ../tools/graphics/svg2pdf { };

  pew = callPackage ../development/tools/pew { };

  poetry = callPackage ../tools/package-management/poetry { };

  poetryPlugins = recurseIntoAttrs poetry.plugins;

  poetry2conda = callPackage ../tools/package-management/poetry2conda { };

  pip-audit = callPackage ../development/tools/pip-audit { };

  pipenv = callPackage ../development/tools/pipenv { };

  pipx = with python3.pkgs; toPythonApplication pipx;

  pipewire = callPackage ../development/libraries/pipewire {
    # ffmpeg depends on SDL2 which depends on pipewire by default.
    # Break the cycle by depending on ffmpeg-headless.
    # Pipewire only uses libavcodec (via an SPA plugin), which isn't
    # affected by the *-headless changes.
    ffmpeg = ffmpeg-headless;
  };

  pipewire_0_2 = callPackage ../development/libraries/pipewire/0.2.nix { };
  wireplumber = callPackage ../development/libraries/pipewire/wireplumber.nix { };

  pw-volume = callPackage ../tools/audio/pw-volume { };

  pyradio = callPackage ../applications/audio/pyradio { };

  racket = callPackage ../development/interpreters/racket {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation;
  };
  racket_7_9 = callPackage ../development/interpreters/racket/racket_7_9.nix {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation;
  };
  racket-minimal = callPackage ../development/interpreters/racket/minimal.nix { };

  rakudo = callPackage ../development/interpreters/rakudo { };
  moarvm = darwin.apple_sdk_11_0.callPackage ../development/interpreters/rakudo/moarvm.nix {
    inherit (darwin.apple_sdk_11_0.frameworks) CoreServices ApplicationServices;
  };
  nqp = callPackage  ../development/interpreters/rakudo/nqp.nix { };
  zef = callPackage ../development/interpreters/rakudo/zef.nix { };

  rascal = callPackage ../development/interpreters/rascal { };

  red = callPackage ../development/interpreters/red { };

  regexploit = callPackage ../tools/security/regexploit { };

  regextester = callPackage ../applications/misc/regextester { };

  inherit (ocamlPackages) reason;

  buildRubyGem = callPackage ../development/ruby-modules/gem {
    inherit (darwin) libobjc;
  };
  defaultGemConfig = callPackage ../development/ruby-modules/gem-config {
    inherit (darwin) DarwinTools cctools autoSignDarwinBinariesHook;
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };
  bundix = callPackage ../development/ruby-modules/bundix { };
  bundler = callPackage ../development/ruby-modules/bundler { };
  bundlerEnv = callPackage ../development/ruby-modules/bundler-env { };
  bundlerApp = callPackage ../development/ruby-modules/bundler-app { };
  bundlerUpdateScript = callPackage ../development/ruby-modules/bundler-update-script { };

  bundler-audit = callPackage ../tools/security/bundler-audit { };

  sol2 = callPackage ../development/libraries/sol2 { };

  solargraph = rubyPackages.solargraph;

  rbenv = callPackage ../development/ruby-modules/rbenv { };

  rubyfmt = darwin.apple_sdk_11_0.callPackage ../development/tools/rubyfmt {
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation Security;
    inherit (darwin) libobjc;
  };

  inherit (callPackage ../development/interpreters/ruby {
    inherit (darwin) libobjc libunwind;
    inherit (darwin.apple_sdk.frameworks) Foundation;
  })
    mkRubyVersion
    mkRuby
    ruby_3_1
    ruby_3_2
    ruby_3_3;

  ruby = ruby_3_1;
  rubyPackages = rubyPackages_3_1;

  rubyPackages_3_1 = recurseIntoAttrs ruby_3_1.gems;
  rubyPackages_3_2 = recurseIntoAttrs ruby_3_2.gems;
  rubyPackages_3_3 = recurseIntoAttrs ruby_3_3.gems;

  mruby = callPackage ../development/compilers/mruby { };

  samplebrain = libsForQt5.callPackage ../applications/audio/samplebrain { };

  scsh = callPackage ../development/interpreters/scsh { };

  scheme48 = callPackage ../development/interpreters/scheme48 { };

  scheme-bytestructures = callPackage ../development/scheme-modules/scheme-bytestructures { };

  smiley-sans = callPackage ../data/fonts/smiley-sans { };

  inherit (callPackages ../applications/networking/cluster/spark { })
    spark_3_5 spark_3_4;
  spark3 = spark_3_5;
  spark = spark3;

  sparkleshare = callPackage ../applications/version-management/sparkleshare { };

  spidermonkey_78 = callPackage ../development/interpreters/spidermonkey/78.nix {
    inherit (darwin) libobjc;
  };
  spidermonkey_91 = callPackage ../development/interpreters/spidermonkey/91.nix {
    inherit (darwin) libobjc;
  };
  spidermonkey_102 = callPackage ../development/interpreters/spidermonkey/102.nix {
    inherit (darwin) libobjc;
  };
  spidermonkey_115 = callPackage ../development/interpreters/spidermonkey/115.nix {
    inherit (darwin) libobjc;
  };

  ssm-session-manager-plugin = callPackage ../applications/networking/cluster/ssm-session-manager-plugin { };

  starlark-rust = callPackage ../development/interpreters/starlark-rust { };

  supercollider = libsForQt5.callPackage ../development/interpreters/supercollider {
    fftw = fftwSinglePrec;
  };

  supercollider_scel = supercollider.override { useSCEL = true; };

  supercolliderPlugins = recurseIntoAttrs {
    sc3-plugins = callPackage ../development/interpreters/supercollider/plugins/sc3-plugins.nix {
      fftw = fftwSinglePrec;
    };
  };

  supercollider-with-plugins = callPackage ../development/interpreters/supercollider/wrapper.nix {
    plugins = [];
  };

  supercollider-with-sc3-plugins = supercollider-with-plugins.override {
    plugins = with supercolliderPlugins; [ sc3-plugins ];
  };

  taktuk = callPackage ../applications/networking/cluster/taktuk { };

  tcl = tcl-8_6;
  tcl-8_5 = callPackage ../development/interpreters/tcl/8.5.nix { };
  tcl-8_6 = callPackage ../development/interpreters/tcl/8.6.nix { };

  tclreadline = callPackage ../development/interpreters/tclreadline { };

  eltclsh = callPackage ../development/tools/eltclsh { };

  waagent = callPackage ../applications/networking/cluster/waagent { };

  wasm = ocamlPackages.wasm;

  wasm3 = callPackage ../development/interpreters/wasm3 { };

  yaegi = callPackage ../development/interpreters/yaegi { };

  yex-lang = callPackage ../development/interpreters/yex-lang { };

  ### DEVELOPMENT / MISC

  inherit (callPackages ../development/misc/h3 { }) h3_3 h3_4;

  h3 = h3_3;

  avrlibc      = callPackage ../development/misc/avr/libc { };
  avrlibcCross = callPackage ../development/misc/avr/libc {
    stdenv = crossLibcStdenv;
  };

  avr8burnomat = callPackage ../development/misc/avr8-burn-omat { };

  cppreference-doc = callPackage ../development/misc/cppreference-doc { };

  sourceFromHead = callPackage ../build-support/source-from-head-fun.nix { };

  jruby = callPackage ../development/interpreters/jruby { };

  jython = callPackage ../development/interpreters/jython { };

  gImageReader = callPackage ../applications/misc/gImageReader { };

  gimme-aws-creds = callPackage ../tools/admin/gimme-aws-creds { };

  gimoji = callPackage ../applications/misc/gimoji { };

  guile_1_8 = callPackage ../development/interpreters/guile/1.8.nix { };

  # Needed for autogen
  guile_2_0 = callPackage ../development/interpreters/guile/2.0.nix { };

  guile_2_2 = callPackage ../development/interpreters/guile/2.2.nix { };

  guile_3_0 = callPackage ../development/interpreters/guile/3.0.nix { };

  guile = guile_3_0;

  guile-cairo = callPackage ../development/guile-modules/guile-cairo { };

  guile-commonmark = callPackage ../development/guile-modules/guile-commonmark { };

  guile-config = callPackage ../development/guile-modules/guile-config { };

  guile-fibers = callPackage ../development/guile-modules/guile-fibers { };

  guile-gcrypt = callPackage ../development/guile-modules/guile-gcrypt { };

  guile-git = callPackage ../development/guile-modules/guile-git { };

  guile-gnutls = callPackage ../development/guile-modules/guile-gnutls { };

  guile-json = callPackage ../development/guile-modules/guile-json { };

  guile-lib = callPackage ../development/guile-modules/guile-lib { };

  guile-ncurses = callPackage ../development/guile-modules/guile-ncurses { };

  guile-opengl = callPackage ../development/guile-modules/guile-opengl { };

  guile-reader = callPackage ../development/guile-modules/guile-reader { };

  guile-sdl = callPackage ../development/guile-modules/guile-sdl {
    guile = guile_2_2;
  };

  guile-sdl2 = callPackage ../development/guile-modules/guile-sdl2 { };

  guile-sqlite3 = callPackage ../development/guile-modules/guile-sqlite3 { };

  guile-ssh = callPackage ../development/guile-modules/guile-ssh { };

  guile-xcb = callPackage ../development/guile-modules/guile-xcb {
    guile = guile_2_2;
  };

  inav-blackbox-tools = callPackage ../tools/misc/inav-blackbox-tools { };

  infracost = callPackage ../tools/misc/infracost { };

  msp430GccSupport = callPackage ../development/misc/msp430/gcc-support.nix { };

  msp430Newlib      = callPackage ../development/misc/msp430/newlib.nix { };
  msp430NewlibCross = callPackage ../development/misc/msp430/newlib.nix {
    newlib = newlibCross;
  };

  mspds = callPackage ../development/misc/msp430/mspds { };
  mspds-bin = callPackage ../development/misc/msp430/mspds/binary.nix { };

  mspdebug = callPackage ../development/misc/msp430/mspdebug.nix { };

  vc4-newlib = callPackage ../development/misc/vc4/newlib.nix { };

  or1k-newlib = callPackage ../development/misc/or1k/newlib.nix { };

  rappel = callPackage ../development/misc/rappel { };

  protege-distribution = callPackage ../development/web/protege-distribution { };

  publii = callPackage ../development/web/publii {};

  umr = callPackage ../development/misc/umr { };

  refurb = callPackage ../development/tools/refurb { };

  srandrd = callPackage ../tools/X11/srandrd { };

  sratoolkit = callPackage ../applications/science/biology/sratoolkit { };

  srecord = callPackage ../development/tools/misc/srecord { };

  srelay = callPackage ../tools/networking/srelay { };

  xidel = callPackage ../tools/text/xidel { };

  asdf-vm = callPackage ../tools/misc/asdf-vm { };

  mise = callPackage ../tools/misc/mise {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  ### DEVELOPMENT / TOOLS

  abi-compliance-checker = callPackage ../development/tools/misc/abi-compliance-checker { };

  abi-dumper = callPackage ../development/tools/misc/abi-dumper { };

  abuild = callPackage ../development/tools/abuild { };

  actiona = libsForQt5.callPackage ../applications/misc/actiona { };

  actionlint = callPackage ../development/tools/analysis/actionlint { };

  adreaper = callPackage ../tools/security/adreaper { };

  aeron = callPackage ../servers/aeron { };

  inherit (callPackage ../development/tools/alloy { })
    alloy5
    alloy6
    alloy;

  altair = callPackage ../development/tools/altair-graphql-client { };

  ameba = callPackage ../development/tools/ameba { };

  anybadge = with python3Packages; toPythonApplication anybadge;

  apgdiff = callPackage ../development/tools/database/apgdiff { };

  apkg = callPackage ../tools/package-management/apkg { };

  augeas = callPackage ../tools/system/augeas { };

  autoadb = callPackage ../misc/autoadb { };

  ansible = ansible_2_16;
  ansible_2_16 = python3Packages.toPythonApplication python3Packages.ansible-core;
  ansible_2_15 = python3Packages.toPythonApplication (python3Packages.ansible-core.overridePythonAttrs (oldAttrs: rec {
    version = "2.15.9";
    src = oldAttrs.src.override {
      inherit version;
      hash = "sha256-JfmxtaWvPAmGvTko7QhurduGdSf7XIOv7xoDz60080U=";
    };
  }));

  ansible-builder = with python3Packages; toPythonApplication ansible-builder;

  ansible-doctor = callPackage ../tools/admin/ansible/doctor.nix { };

  ansible-navigator = with python3Packages; toPythonApplication ansible-navigator;

  dbus-test-runner = callPackage ../development/tools/dbus-test-runner { };

  doq = callPackage ../development/tools/misc/doq { };

  espup = callPackage ../development/tools/espup { };

  karma-runner = callPackage ../development/tools/karma-runner { };

  teller = callPackage ../development/tools/teller { };

  yakut = python3Packages.callPackage ../development/tools/misc/yakut { };

  ### DEVELOPMENT / TOOLS / LANGUAGE-SERVERS

  ansible-language-server = callPackage ../development/tools/language-servers/ansible-language-server { };

  beancount-language-server = callPackage ../development/tools/language-servers/beancount-language-server { };

  buf-language-server = callPackage ../development/tools/language-servers/buf-language-server { };

  ccls = callPackage ../development/tools/language-servers/ccls {
    llvmPackages = llvmPackages_14;
  };

  docker-compose-language-service = callPackage ../development/tools/language-servers/docker-compose-language-service { };

  dockerfile-language-server-nodejs = callPackage ../development/tools/language-servers/dockerfile-language-server-nodejs { };

  dot-language-server = callPackage ../development/tools/language-servers/dot-language-server { };

  emmet-ls = callPackage ../development/tools/language-servers/emmet-ls { };

  emmet-language-server = callPackage ../development/tools/language-servers/emmet-language-server { };

  fortls = python3.pkgs.callPackage ../development/tools/language-servers/fortls { };

  fortran-language-server = python3.pkgs.callPackage ../development/tools/language-servers/fortran-language-server { };

  glslls = callPackage ../development/tools/language-servers/glslls { };

  gopls = callPackage ../development/tools/language-servers/gopls { };

  helm-ls = callPackage ../development/tools/language-servers/helm-ls { };

  javascript-typescript-langserver = callPackage ../development/tools/language-servers/javascript-typescript-langserver { };

  jsonnet-language-server = callPackage ../development/tools/language-servers/jsonnet-language-server { };

  kotlin-language-server = callPackage ../development/tools/language-servers/kotlin-language-server { };

  lua-language-server = darwin.apple_sdk_11_0.callPackage ../development/tools/language-servers/lua-language-server {
    inherit (darwin.apple_sdk_11_0.frameworks) CoreFoundation Foundation;
    inherit (darwin) ditto;
  };

  metals = callPackage ../development/tools/language-servers/metals { };

  millet = callPackage ../development/tools/language-servers/millet { };

  neocmakelsp = callPackage ../development/tools/language-servers/neocmakelsp { };

  nil = callPackage ../development/tools/language-servers/nil { };

  nixd = callPackage ../development/tools/language-servers/nixd {
    llvmPackages = llvmPackages_16;
    nix = nixVersions.nix_2_19;
  };

  openscad-lsp = callPackage ../development/tools/language-servers/openscad-lsp { };

  perlnavigator = callPackage ../development/tools/language-servers/perlnavigator { };

  postgres-lsp = callPackage ../development/tools/language-servers/postgres-lsp { };

  pylyzer = callPackage ../development/tools/language-servers/pylyzer { };

  ruff-lsp = python3Packages.callPackage ../development/tools/language-servers/ruff-lsp { };

  rune-languageserver = callPackage ../development/tools/language-servers/rune-languageserver { };

  svls = callPackage ../development/tools/language-servers/svls { };

  typst-lsp = callPackage ../development/tools/language-servers/typst-lsp { };

  vala-language-server = callPackage ../development/tools/language-servers/vala-language-server { };

  verible = callPackage ../development/tools/language-servers/verible { };

  vscode-langservers-extracted = callPackage ../development/tools/language-servers/vscode-langservers-extracted { };

  zls = callPackage ../development/tools/language-servers/zls { };

  ansible-later = callPackage ../tools/admin/ansible/later.nix { };

  ansible-lint = callPackage ../tools/admin/ansible/lint.nix { };

  antlr2 = callPackage ../development/tools/parsing/antlr/2.7.7.nix { };
  antlr3_4 = callPackage ../development/tools/parsing/antlr/3.4.nix { };
  antlr3_5 = callPackage ../development/tools/parsing/antlr/3.5.nix { };
  antlr3 = antlr3_5;

  inherit (callPackages ../development/tools/parsing/antlr/4.nix { })
    antlr4_8
    antlr4_9
    antlr4_10
    antlr4_11
    antlr4_12
    antlr4_13;

  antlr4 = antlr4_13;

  antlr = antlr4;

  apacheAnt = callPackage ../development/tools/build-managers/apache-ant { };
  ant = apacheAnt;

  apacheKafka = apacheKafka_3_5;
  apacheKafka_2_8 = callPackage ../servers/apache-kafka { majorVersion = "2.8"; };
  apacheKafka_3_0 = callPackage ../servers/apache-kafka { majorVersion = "3.0"; };
  apacheKafka_3_1 = callPackage ../servers/apache-kafka { majorVersion = "3.1"; };
  apacheKafka_3_2 = callPackage ../servers/apache-kafka { majorVersion = "3.2"; };
  apacheKafka_3_3 = callPackage ../servers/apache-kafka { majorVersion = "3.3"; };
  apacheKafka_3_4 = callPackage ../servers/apache-kafka { majorVersion = "3.4"; };
  apacheKafka_3_5 = callPackage ../servers/apache-kafka { majorVersion = "3.5"; };

  apng2gif = callPackage ../tools/graphics/apng2gif { };

  gif2apng = callPackage ../tools/graphics/gif2apng { };

  apngopt = callPackage ../tools/graphics/apngopt { };

  kt = callPackage ../tools/misc/kt { };

  argbash = callPackage ../development/tools/misc/argbash { };

  arpa2cm = callPackage ../development/tools/build-managers/arpa2cm { };

  arpa2common = callPackage ../development/libraries/arpa2common { };

  asn2quickder = python3Packages.callPackage ../development/tools/asn2quickder { };

  astyle = callPackage ../development/tools/misc/astyle { };

  automaticcomponenttoolkit = callPackage ../development/tools/misc/automaticcomponenttoolkit { };

  aviator = callPackage ../development/tools/misc/aviator { };

  awf = callPackage ../development/tools/misc/awf { };

  aws-adfs = with python3Packages; toPythonApplication aws-adfs;

  electron-source = callPackage ../development/tools/electron { };

  inherit (callPackages ../development/tools/electron/binary { })
    electron_24-bin
    electron_27-bin
    electron_28-bin
    electron_29-bin
    electron_30-bin;

  inherit (callPackages ../development/tools/electron/chromedriver { })
    electron-chromedriver_29
    electron-chromedriver_30
    electron-chromedriver_31;

  electron_24 = electron_24-bin;
  electron_27 = if lib.meta.availableOn stdenv.hostPlatform electron-source.electron_27 then electron-source.electron_27 else electron_27-bin;
  electron_28 = if lib.meta.availableOn stdenv.hostPlatform electron-source.electron_28 then electron-source.electron_28 else electron_28-bin;
  electron_29 = if lib.meta.availableOn stdenv.hostPlatform electron-source.electron_29 then electron-source.electron_29 else electron_29-bin;
  electron_30 = if lib.meta.availableOn stdenv.hostPlatform electron-source.electron_30 then electron-source.electron_30 else electron_30-bin;
  electron = electron_29;
  electron-bin = electron_29-bin;

  autobuild = callPackage ../development/tools/misc/autobuild { };

  autoconf-archive = callPackage ../development/tools/misc/autoconf-archive { };

  autoconf = callPackage ../development/tools/misc/autoconf { };
  autoconf213 = callPackage ../development/tools/misc/autoconf/2.13.nix { };
  autoconf264 = callPackage ../development/tools/misc/autoconf/2.64.nix { };
  autoconf269 = callPackage ../development/tools/misc/autoconf/2.69.nix { };
  autoconf271 = callPackage ../development/tools/misc/autoconf/2.71.nix { };

  acr  = callPackage ../development/tools/misc/acr { };

  autocutsel = callPackage ../tools/X11/autocutsel{ };

  automake = automake116x;

  automake111x = callPackage ../development/tools/misc/automake/automake-1.11.x.nix { };

  automake115x = callPackage ../development/tools/misc/automake/automake-1.15.x.nix { };

  automake116x = callPackage ../development/tools/misc/automake/automake-1.16.x.nix { };

  avrdude = callPackage ../development/embedded/avrdude { };

  b4 = callPackage ../development/tools/b4 { };

  babeltrace = callPackage ../development/tools/misc/babeltrace { };

  bam = callPackage ../development/tools/build-managers/bam { };

  bandit = with python3Packages; toPythonApplication bandit;

  bazel = bazel_6;

  bazel_5 = callPackage ../development/tools/build-managers/bazel/bazel_5 {
    inherit (darwin) cctools sigtool;
    inherit (darwin.apple_sdk.frameworks) CoreFoundation CoreServices Foundation;
    buildJdk = jdk11_headless;
    runJdk = jdk11_headless;
    stdenv = if stdenv.cc.isClang then llvmPackages.stdenv
      else if stdenv.cc.isGNU then gcc12Stdenv
      else stdenv;
    bazel_self = bazel_5;
  };

  bazel_6 = darwin.apple_sdk_11_0.callPackage ../development/tools/build-managers/bazel/bazel_6 {
    inherit (darwin) cctools sigtool;
    inherit (darwin.apple_sdk_11_0.frameworks) CoreFoundation CoreServices Foundation;
    buildJdk = jdk11_headless;
    runJdk = jdk11_headless;
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv
      else if stdenv.cc.isClang then llvmPackages.stdenv
      else if stdenv.cc.isGNU then gcc12Stdenv
      else stdenv;
    bazel_self = bazel_6;
  };

  bazel_7 = darwin.apple_sdk_11_0.callPackage ../development/tools/build-managers/bazel/bazel_7 {
    inherit (darwin) cctools sigtool;
    inherit (darwin.apple_sdk_11_0.frameworks) CoreFoundation CoreServices Foundation IOKit;
    buildJdk = jdk17_headless;
    runJdk = jdk17_headless;
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv
      else if stdenv.cc.isClang then llvmPackages.stdenv
      else stdenv;
    bazel_self = bazel_7;
  };

  bazel-buildtools = callPackage ../development/tools/build-managers/bazel/buildtools { };
  buildifier = bazel-buildtools;
  buildozer = bazel-buildtools;
  unused_deps = bazel-buildtools;

  bazel-remote = callPackage ../development/tools/build-managers/bazel/bazel-remote { };

  bazel-watcher = callPackage ../development/tools/bazel-watcher { };

  bazel-gazelle = callPackage ../development/tools/bazel-gazelle { };

  bazel-kazel = callPackage ../development/tools/bazel-kazel { };

  bazelisk = callPackage ../development/tools/bazelisk { };

  rebazel = callPackage ../development/tools/rebazel {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  buildBazelPackage = darwin.apple_sdk_11_0.callPackage ../build-support/build-bazel-package { };

  bear = callPackage ../development/tools/build-managers/bear { };

  bingrep = callPackage ../development/tools/analysis/bingrep { };

  binutils-unwrapped = callPackage ../development/tools/misc/binutils {
    autoreconfHook = autoreconfHook269;
    # FHS sys dirs presumably only have stuff for the build platform
    noSysDirs = (stdenv.targetPlatform != stdenv.hostPlatform) || noSysDirs;
  };
  binutils-unwrapped-all-targets = callPackage ../development/tools/misc/binutils {
    autoreconfHook = if targetPlatform.isiOS then autoreconfHook269 else autoreconfHook;
    # FHS sys dirs presumably only have stuff for the build platform
    noSysDirs = (stdenv.targetPlatform != stdenv.hostPlatform) || noSysDirs;
    withAllTargets = true;
  };
  binutils = wrapBintoolsWith {
    bintools = binutils-unwrapped;
  };
  binutils_nogold = lowPrio (wrapBintoolsWith {
    bintools = binutils-unwrapped.override {
      enableGold = false;
    };
  });
  binutilsNoLibc = wrapBintoolsWith {
    bintools = binutils-unwrapped;
    libc = preLibcCrossHeaders;
  };

  libbfd = callPackage ../development/tools/misc/binutils/libbfd.nix { };

  libopcodes = callPackage ../development/tools/misc/binutils/libopcodes.nix { };

  # Held back 2.38 release. Remove once all dependencies are ported to 2.39.
  binutils-unwrapped_2_38 = callPackage ../development/tools/misc/binutils/2.38 {
    autoreconfHook = autoreconfHook269;
    # FHS sys dirs presumably only have stuff for the build platform
    noSysDirs = (stdenv.targetPlatform != stdenv.hostPlatform) || noSysDirs;
  };

  libbfd_2_38 = callPackage ../development/tools/misc/binutils/2.38/libbfd.nix {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  libopcodes_2_38 = callPackage ../development/tools/misc/binutils/2.38/libopcodes.nix {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  # Here we select the default bintools implementations to be used.  Note when
  # cross compiling these are used not for this stage but the *next* stage.
  # That is why we choose using this stage's target platform / next stage's
  # host platform.
  #
  # Because this is the *next* stages choice, it's a bit non-modular to put
  # here. In theory, bootstraping is supposed to not be a chain but at tree,
  # where each stage supports many "successor" stages, like multiple possible
  # futures. We don't have a better alternative, but with this downside in
  # mind, please be judicious when using this attribute. E.g. for building
  # things in *this* stage you should use probably `stdenv.cc.bintools` (from a
  # default or alternate `stdenv`), at build time, and try not to "force" a
  # specific bintools at runtime at all.
  #
  # In other words, try to only use this in wrappers, and only use those
  # wrappers from the next stage.
  bintools-unwrapped = let
    inherit (stdenv.targetPlatform) linker;
  in     if linker == "lld"     then llvmPackages.bintools-unwrapped
    else if linker == "cctools" then darwin.binutils-unwrapped
    else if linker == "bfd"     then binutils-unwrapped
    else if linker == "gold"    then binutils-unwrapped.override { enableGoldDefault = true; }
    else null;
  bintoolsNoLibc = wrapBintoolsWith {
    bintools = bintools-unwrapped;
    libc = preLibcCrossHeaders;
  };
  bintools = wrapBintoolsWith {
    bintools = bintools-unwrapped;
  };

  bintoolsDualAs = wrapBintoolsWith {
    bintools = darwin.binutilsDualAs-unwrapped;
    wrapGas = true;
  };

  bison = callPackage ../development/tools/parsing/bison { };

  bisoncpp = callPackage ../development/tools/parsing/bisonc++ { };

  black = with python3Packages; toPythonApplication black;

  blackfire = callPackage ../development/tools/misc/blackfire { };

  black-macchiato = with python3Packages; toPythonApplication black-macchiato;

  blackmagic = callPackage ../development/embedded/blackmagic { };

  bloaty = callPackage ../development/tools/bloaty { };

  bloomrpc = callPackage ../development/web/bloomrpc { };

  bloop = callPackage ../development/tools/build-managers/bloop { };

  bossa = callPackage ../development/embedded/bossa { };

  bossa-arduino = callPackage ../development/embedded/bossa/arduino.nix { };

  bob = callPackage ../development/tools/build-managers/bob { };

  buck = callPackage ../development/tools/build-managers/buck { };

  buck2 = callPackage ../development/tools/build-managers/buck2 { };

  build2 = callPackage ../development/tools/build-managers/build2 {
    # Break cycle by using self-contained toolchain for bootstrapping
    build2 = buildPackages.callPackage ../development/tools/build-managers/build2/bootstrap.nix { };
  };

  # Dependency of build2, must also break cycle for this
  libbutl = callPackage ../development/libraries/libbutl {
    build2 = build2.bootstrap;
    inherit (darwin) DarwinTools;
  };

  libbpkg = callPackage ../development/libraries/libbpkg { };
  libodb = callPackage ../development/libraries/libodb { };
  libodb-sqlite = callPackage ../development/libraries/libodb-sqlite { };
  bdep = callPackage ../development/tools/build-managers/build2/bdep.nix { };

  bore-cli = callPackage ../tools/networking/bore-cli/default.nix {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  bpkg = callPackage ../development/tools/build-managers/build2/bpkg.nix { };

  buildkite-agent = callPackage ../development/tools/continuous-integration/buildkite-agent { };

  buildkite-agent-metrics = callPackage ../servers/monitoring/buildkite-agent-metrics { };

  buildkite-cli = callPackage ../development/tools/continuous-integration/buildkite-cli { };

  buildkite-test-collector-rust  = callPackage ../development/tools/continuous-integration/buildkite-test-collector-rust {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  bump = callPackage ../development/tools/github/bump { };

  libbpf = callPackage ../os-specific/linux/libbpf { };
  libbpf_0 = callPackage ../os-specific/linux/libbpf/0.x.nix { };

  bundlewrap = with python3.pkgs; toPythonApplication bundlewrap;

  bpftools = callPackage ../os-specific/linux/bpftools { };

  bpm-tools = callPackage ../tools/audio/bpm-tools { };

  byacc = callPackage ../development/tools/parsing/byacc { };

  cadre = callPackage ../development/tools/cadre { };

  carto = callPackage ../development/tools/carto { };

  catnip = callPackage ../tools/audio/catnip { };

  catnip-gtk4 = callPackage ../tools/audio/catnip-gtk4 { };

  cbrowser = callPackage ../development/tools/misc/cbrowser { };

  cc-tool = callPackage ../development/embedded/cc-tool { };

  # Wrapper that works as gcc or g++
  # It can be used by setting in nixpkgs config like this, for example:
  #    replaceStdenv = { pkgs }: pkgs.ccacheStdenv;
  # But if you build in chroot, you should have that path in chroot
  # If instantiated directly, it will use $HOME/.ccache as the cache directory,
  # i.e. /homeless-shelter/.ccache using the Nix daemon.
  # You should specify a different directory using an override in
  # packageOverrides to set extraConfig.
  #
  # Example using Nix daemon (i.e. multiuser Nix install or on NixOS):
  #    packageOverrides = pkgs: {
  #     ccacheWrapper = pkgs.ccacheWrapper.override {
  #       extraConfig = ''
  #         export CCACHE_COMPRESS=1
  #         export CCACHE_DIR=/var/cache/ccache
  #         export CCACHE_UMASK=007
  #       '';
  #     };
  # You can use a different directory, but whichever directory you choose
  # should be owned by user root, group nixbld with permissions 0770.
  ccacheWrapper = makeOverridable ({ extraConfig, cc }:
    cc.override {
      cc = ccache.links {
        inherit extraConfig;
        unwrappedCC = cc.cc;
      };
    }) {
      extraConfig = "";
      inherit (stdenv) cc;
    };

  ccacheStdenv = lowPrio (makeOverridable ({ stdenv, ... } @ extraArgs:
    overrideCC stdenv (buildPackages.ccacheWrapper.override ({
      inherit (stdenv) cc;
    } // lib.optionalAttrs (builtins.hasAttr "extraConfig" extraArgs) {
      extraConfig = extraArgs.extraConfig;
    }))) {
      inherit stdenv;
    });

  cccc = callPackage ../development/tools/analysis/cccc { };

  cgdb = callPackage ../development/tools/misc/cgdb { };

  cheat = callPackage ../applications/misc/cheat { };

  matter-compiler = callPackage ../development/compilers/matter-compiler { };

  cfr = callPackage ../development/tools/java/cfr { };

  cfripper = callPackage ../tools/security/cfripper { };

  checkra1n = callPackage ../development/mobile/checkra1n { };

  checkstyle = callPackage ../development/tools/analysis/checkstyle { };

  chromedriver = callPackage ../development/tools/selenium/chromedriver { };

  chromium-xorg-conf = callPackage ../os-specific/linux/chromium-xorg-conf { };

  chrpath = callPackage ../development/tools/misc/chrpath { };

  chruby = callPackage ../development/tools/misc/chruby { rubies = null; };

  chruby-fish = callPackage ../development/tools/misc/chruby-fish { };

  cl-launch = callPackage ../development/tools/misc/cl-launch { };

  clean-css-cli = callPackage ../development/tools/clean-css-cli { };

  cloud-nuke = callPackage ../development/tools/cloud-nuke { };

  cloudcompare = libsForQt5.callPackage ../applications/graphics/cloudcompare { };

  cloudlog = callPackage ../applications/radio/cloudlog { };

  cloudflare-warp = callPackage ../tools/networking/cloudflare-warp { };

  cloudfoundry-cli = callPackage ../applications/networking/cluster/cloudfoundry-cli { };

  coan = callPackage ../development/tools/analysis/coan { };

  coder = callPackage ../development/tools/coder { };

  compile-daemon = callPackage ../development/tools/compile-daemon { };

  complexity = callPackage ../development/tools/misc/complexity { };

  complgen = callPackage ../development/tools/misc/complgen { };

  conan = callPackage ../development/tools/build-managers/conan { };

  cookiecutter = with python3Packages; toPythonApplication cookiecutter;

  cordova = callPackage ../development/mobile/cordova { };

  corrosion = callPackage ../development/tools/build-managers/corrosion { };

  corundum = callPackage ../development/tools/corundum { };

  confluencepot = callPackage ../servers/confluencepot { };

  confluent-platform = callPackage ../servers/confluent-platform { };

  ctags = callPackage ../development/tools/misc/ctags { };

  ctagsWrapped = callPackage ../development/tools/misc/ctags/wrapped.nix { };

  ctodo = callPackage ../applications/misc/ctodo { };

  cubiomes-viewer = libsForQt5.callPackage ../applications/misc/cubiomes-viewer { };

  ctmg = callPackage ../tools/security/ctmg { };

  # can't use override - it triggers infinite recursion
  cmakeMinimal = callPackage ../by-name/cm/cmake/package.nix {
    isMinimalBuild = true;
  };

  cmakeCurses = cmake.override {
    uiToolkits = [ "ncurses" ];
  };

  cmakeWithGui = cmake.override {
    uiToolkits = [ "ncurses" "qt5" ];
  };

  cmake-format = python3Packages.callPackage ../development/tools/cmake-format { };

  cobra-cli = callPackage ../development/tools/cobra-cli { };

  cmake-language-server = python3Packages.callPackage ../development/tools/misc/cmake-language-server {
    inherit cmake cmake-format;
  };

  # Does not actually depend on Qt 5
  inherit (plasma5Packages) extra-cmake-modules;

  coccinelle = callPackage ../development/tools/misc/coccinelle {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  cpptest = callPackage ../development/libraries/cpptest { };

  cppi = callPackage ../development/tools/misc/cppi { };

  cproto = callPackage ../development/tools/misc/cproto { };

  cflow = callPackage ../development/tools/misc/cflow { };

  cov-build = callPackage ../development/tools/analysis/cov-build { };

  cppcheck = callPackage ../development/tools/analysis/cppcheck { };

  cpplint = callPackage ../development/tools/analysis/cpplint { };

  credstash = with python3Packages; toPythonApplication credstash;

  creduce = callPackage ../development/tools/misc/creduce {
    inherit (llvmPackages_16) llvm libclang;
  };

  cscope = callPackage ../development/tools/misc/cscope { };

  csmith = callPackage ../development/tools/misc/csmith { };

  inherit (nodePackages) csslint;

  css-html-js-minify = with python3Packages; toPythonApplication css-html-js-minify;

  cvehound = callPackage ../development/tools/analysis/cvehound { };

  cvise = python3Packages.callPackage ../development/tools/misc/cvise {
    # cvise keeps up with fresh llvm releases and supports wide version range
    inherit (llvmPackages_14) llvm libclang;
  };

  cwltool = callPackage ../applications/science/misc/cwltool { };

  dbt = with python3Packages; toPythonApplication dbt-core;

  dprint = callPackage ../development/tools/dprint {
    inherit (darwin.apple_sdk_11_0.frameworks) CoreFoundation Security;
  };

  devbox = callPackage ../development/tools/devbox { };

  libcxx = llvmPackages.libcxx;

  librarian-puppet-go = callPackage ../development/tools/librarian-puppet-go { };

  libgcc = stdenv.cc.cc.libgcc or null;

  # This is for e.g. LLVM libraries on linux.
  gccForLibs =
    if stdenv.targetPlatform == stdenv.hostPlatform && targetPackages.stdenv.cc.isGNU
    # Can only do this is in the native case, otherwise we might get infinite
    # recursion if `targetPackages.stdenv.cc.cc` itself uses `gccForLibs`.
      then targetPackages.stdenv.cc.cc
    else gcc.cc;

  libstdcxx5 = callPackage ../development/libraries/gcc/libstdc++/5.nix { };

  libsigrok = callPackage ../development/tools/libsigrok {
    python = python3;
  };

  libsigrokdecode = callPackage ../development/tools/libsigrokdecode { };

  sqlcl = callPackage ../development/tools/database/sqlcl { };

  sigrok-firmware-fx2lafw = callPackage ../development/tools/sigrok-firmware-fx2lafw { };

  datree = callPackage ../development/tools/datree { };

  detekt = callPackage ../development/tools/detekt { };

  dcadec = callPackage ../development/tools/dcadec { };

  dejagnu = callPackage ../development/tools/misc/dejagnu { };

  devd = callPackage ../development/tools/devd { };

  devtodo = callPackage ../development/tools/devtodo { };

  d-spy = callPackage ../development/tools/misc/d-spy { };

  dfu-programmer = callPackage ../development/tools/misc/dfu-programmer { };

  dfu-util = callPackage ../development/tools/misc/dfu-util { };

  ddd = callPackage ../development/tools/misc/ddd { };

  lattice-diamond = callPackage ../development/embedded/fpga/lattice-diamond { };

  direvent = callPackage ../development/tools/misc/direvent { };

  distcc = callPackage ../development/tools/misc/distcc {
    libiberty_static = libiberty.override { staticBuild = true; };
  };

  # distccWrapper: wrapper that works as gcc or g++
  # It can be used by setting in nixpkgs config like this, for example:
  #    replaceStdenv = { pkgs }: pkgs.distccStdenv;
  # But if you build in chroot, a default 'nix' will create
  # a new net namespace, and won't have network access.
  # You can use an override in packageOverrides to set extraConfig:
  #    packageOverrides = pkgs: {
  #     distccWrapper = pkgs.distccWrapper.override {
  #       extraConfig = ''
  #         DISTCC_HOSTS="myhost1 myhost2"
  #       '';
  #     };
  #
  distccWrapper = makeOverridable ({ extraConfig ? "" }:
    wrapCC (distcc.links extraConfig)) { };
  distccStdenv = lowPrio (overrideCC stdenv buildPackages.distccWrapper);

  distccMasquerade = if stdenv.isDarwin
    then null
    else callPackage ../development/tools/misc/distcc/masq.nix {
      gccRaw = gcc.cc;
      binutils = binutils;
    };

  dive = callPackage ../development/tools/dive { };

  dioxus-cli = callPackage ../development/tools/rust/dioxus-cli { };

  doclifter = callPackage ../development/tools/misc/doclifter { };

  docutils = with python3Packages; toPythonApplication docutils;

  doctl = callPackage ../development/tools/doctl { };

  doit = with python3Packages; toPythonApplication doit;

  dolt = callPackage ../servers/sql/dolt { };

  dot2tex = with python3.pkgs; toPythonApplication dot2tex;

  doxygen = darwin.apple_sdk_11_0.callPackage ../development/tools/documentation/doxygen {
    qt5 = null;
    inherit (darwin.apple_sdk_11_0.frameworks) CoreServices;
  };

  doxygen_gui = lowPrio (doxygen.override { inherit qt5; });

  drake = callPackage ../development/tools/build-managers/drake { };

  drip = callPackage ../development/tools/drip { };

  drm_info = callPackage ../development/tools/drm_info { };

  dura = callPackage ../development/tools/misc/dura {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  dwfv = callPackage ../applications/science/electronics/dwfv { };

  dwz = callPackage ../development/tools/misc/dwz { };

  eask = callPackage ../development/tools/eask { };

  easypdkprog = callPackage ../development/embedded/easypdkprog { };

  editorconfig-checker = callPackage ../development/tools/misc/editorconfig-checker { };

  editorconfig-core-c = callPackage ../development/tools/misc/editorconfig-core-c { };

  edb = libsForQt5.callPackage ../development/tools/misc/edb { };

  eggdbus = callPackage ../development/tools/misc/eggdbus { };

  effitask = callPackage ../applications/misc/effitask { };

  efm-langserver = callPackage ../development/tools/efm-langserver { };

  egypt = callPackage ../development/tools/analysis/egypt { };

  electron-fiddle = callPackage ../development/tools/electron-fiddle { };

  elf2uf2-rs = darwin.apple_sdk_11_0.callPackage ../development/embedded/elf2uf2-rs {
    inherit (darwin.apple_sdk_11_0.frameworks) CoreFoundation DiskArbitration Foundation;
  };

  elfinfo = callPackage ../development/tools/misc/elfinfo { };

  elfkickers = callPackage ../development/tools/misc/elfkickers { };

  elfutils = callPackage ../development/tools/misc/elfutils { };

  eliot-tree = callPackage ../development/tools/eliot-tree { };

  emma = callPackage ../development/tools/analysis/emma { };

  ent-go = callPackage ../development/tools/ent { };

  epm = callPackage ../development/tools/misc/epm { };

  eresi = callPackage ../development/tools/analysis/eresi { };

  evmdis = callPackage ../development/tools/analysis/evmdis { };

  eweb = callPackage ../development/tools/literate-programming/eweb { };

  explain = callPackage ../development/tools/explain { };

  ezno = callPackage ../development/tools/misc/ezno { };

  func = callPackage ../applications/networking/cluster/func { };

  funnelweb = callPackage ../development/tools/literate-programming/funnelweb { };

  license_finder = callPackage ../development/tools/license_finder { };

  license-scanner = callPackage ../development/tools/license-scanner { };

  Literate = callPackage ../development/tools/literate-programming/Literate { };

  md-tangle = callPackage ../development/tools/literate-programming/md-tangle { };

  # NOTE: Override and set useIcon = false to use Awk instead of Icon.
  noweb = callPackage ../development/tools/literate-programming/noweb { };

  nuweb = callPackage ../development/tools/literate-programming/nuweb { };

  eztrace = callPackage ../development/tools/profiling/EZTrace { };

  faas-cli = callPackage ../development/tools/faas-cli { };

  fastddsgen = callPackage ../development/tools/fastddsgen { };

  fastgron = callPackage ../development/tools/fastgron { };

  fatcat = callPackage ../development/tools/fatcat { };

  findbugs = callPackage ../development/tools/analysis/findbugs { };

  findnewest = callPackage ../development/tools/misc/findnewest { };

  firebase-tools = callPackage ../development/tools/firebase-tools { };

  flootty = callPackage ../development/tools/flootty { };

  fffuu = haskell.lib.compose.justStaticExecutables (haskellPackages.callPackage ../tools/misc/fffuu { });

  ffuf = callPackage ../tools/security/ffuf { };

  flow = callPackage ../development/tools/analysis/flow {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };

  fly = callPackage ../development/tools/continuous-integration/fly { };

  foreman = callPackage ../tools/system/foreman { };
  goreman = callPackage ../tools/system/goreman { };

  fprettify = callPackage ../development/tools/fprettify { };

  framac = callPackage ../development/tools/analysis/frama-c { };

  frame = callPackage ../development/libraries/frame { };

  framesh = callPackage ../applications/blockchains/framesh { };

  frp = callPackage ../tools/networking/frp { };

  fsatrace = callPackage ../development/tools/misc/fsatrace { };

  fswatch = callPackage ../development/tools/misc/fswatch {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  fsearch = callPackage ../tools/misc/fsearch { };

  fujprog = callPackage ../development/embedded/fpga/fujprog {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  fundoc = callPackage ../development/tools/fundoc { };

  funzzy = callPackage ../development/tools/misc/funzzy { };

  gede = libsForQt5.callPackage ../development/tools/misc/gede { };

  gdbgui = python3Packages.callPackage ../development/tools/misc/gdbgui { };

  pifpaf = callPackage ../development/tools/pifpaf { };

  pmd = callPackage ../development/tools/analysis/pmd {
    openjdk = openjdk8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  jdepend = callPackage ../development/tools/analysis/jdepend {
    jdk = jdk8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  flex_2_5_35 = callPackage ../development/tools/parsing/flex/2.5.35.nix { };
  flex = callPackage ../development/tools/parsing/flex { };

  re-flex = callPackage ../development/tools/parsing/re-flex { };

  flexibee = callPackage ../applications/office/flexibee { };

  flexcpp = callPackage ../development/tools/parsing/flexc++ { };

  geis = callPackage ../development/libraries/geis { };

  gi-docgen = callPackage ../development/tools/documentation/gi-docgen { };

  git2-cpp = callPackage ../development/libraries/git2-cpp { };

  github-release = callPackage ../development/tools/github/github-release { };

  global = callPackage ../development/tools/misc/global { };

  gnome-doc-utils = callPackage ../development/tools/documentation/gnome-doc-utils { };

  gnome-desktop-testing = callPackage ../development/tools/gnome-desktop-testing { };

  gnome-firmware = callPackage ../applications/misc/gnome-firmware { };

  gnome-tecla = callPackage ../applications/misc/gnome-tecla { };

  gnome-usage = callPackage ../applications/misc/gnome-usage { };

  gnome-inform7 = callPackage ../applications/editors/gnome-inform7 { };

  gnome-latex = callPackage ../applications/editors/gnome-latex { };

  gnome-network-displays = callPackage ../applications/networking/gnome-network-displays { };

  gnome-multi-writer = callPackage ../applications/misc/gnome-multi-writer { };

  gnome-online-accounts = callPackage ../development/libraries/gnome-online-accounts { };

  gnome-video-effects = callPackage ../development/libraries/gnome-video-effects { };

  gnum4 = callPackage ../development/tools/misc/gnum4 { };
  m4 = gnum4;

  om4 = callPackage ../development/tools/misc/om4 { };

  gnumake = callPackage ../development/tools/build-managers/gnumake { };
  gnumake42 = callPackage ../development/tools/build-managers/gnumake/4.2 { };

  go-licenses = callPackage ../development/tools/misc/go-licenses  { };

  gob2 = callPackage ../development/tools/misc/gob2 { };

  gocd-agent = callPackage ../development/tools/continuous-integration/gocd-agent { };

  gocd-server = callPackage ../development/tools/continuous-integration/gocd-server { };

  gopatch = callPackage ../development/tools/misc/gopatch { };

  gotify-server = callPackage ../servers/gotify { };

  gotty = callPackage ../servers/gotty { };

  gputils = callPackage ../development/embedded/gputils { };

  gpuvis = callPackage ../development/tools/misc/gpuvis { };

  gqlint = callPackage ../development/tools/gqlint { };

  gradle-packages = import ../development/tools/build-managers/gradle {
    inherit jdk11 jdk17 jdk21;
  };
  gradleGen = gradle-packages.gen;
  gradle_6 = callPackage gradle-packages.gradle_6 { };
  gradle_7 = callPackage gradle-packages.gradle_7 { };
  gradle_8 = callPackage gradle-packages.gradle_8 { };
  gradle = gradle_8;

  grcov = callPackage ../development/tools/misc/grcov { };

  gperf = callPackage ../development/tools/misc/gperf { };
  # 3.1 changed some parameters from int to size_t, leading to mismatches.
  gperf_3_0 = callPackage ../development/tools/misc/gperf/3.0.x.nix { };

  grail = callPackage ../development/libraries/grail { };

  graphene = callPackage ../development/libraries/graphene { };

  griffe = with python3Packages; toPythonApplication griffe;

  gtk-doc = callPackage ../development/tools/documentation/gtk-doc { };

  gtkdialog = callPackage ../development/tools/misc/gtkdialog { };

  crowdin-cli = callPackage ../tools/text/crowdin-cli { };

  gtranslator = callPackage ../tools/text/gtranslator { };

  gtree = callPackage ../tools/text/gtree {
    buildGoModule = buildGo122Module;
  };

  guff = callPackage ../tools/graphics/guff { };

  guile-hall = callPackage ../development/tools/guile/guile-hall { };

  gwrap = callPackage ../development/tools/guile/g-wrap {
    guile = guile_2_2;
  };

  hadolint = haskell.lib.compose.justStaticExecutables haskellPackages.hadolint;

  halfempty = callPackage ../development/tools/halfempty { };

  hcloud = callPackage ../development/tools/hcloud { };

  hclfmt = callPackage ../development/tools/hclfmt { };

  help2man = callPackage ../development/tools/misc/help2man { };

  heroku = callPackage ../development/tools/heroku { };

  highlight-assertions = callPackage ../development/tools/misc/highlight-assertions { };

  confluent-cli = callPackage ../development/tools/confluent-cli { };

  html-minifier = callPackage ../development/tools/html-minifier { };

  htmlhint = callPackage ../development/tools/htmlhint { };

  htmlunit-driver = callPackage ../development/tools/selenium/htmlunit-driver { };

  hyenae = callPackage ../tools/networking/hyenae { };

  iaca_2_1 = callPackage ../development/tools/iaca/2.1.nix { };
  iaca_3_0 = callPackage ../development/tools/iaca/3.0.nix { };
  iaca = iaca_3_0;

  icestorm = callPackage ../development/embedded/fpga/icestorm { };

  icmake = callPackage ../development/tools/build-managers/icmake { };

  iconnamingutils = callPackage ../development/tools/misc/icon-naming-utils { };

  ikos = callPackage ../development/tools/analysis/ikos {
    inherit (llvmPackages_14) stdenv clang llvm;
    tbb = tbb_2021_11;
  };

  img = callPackage ../development/tools/img { };

  include-what-you-use = callPackage ../development/tools/analysis/include-what-you-use {
    llvmPackages = llvmPackages_18;
  };

  indent = callPackage ../development/tools/misc/indent { };

  inotify-tools = callPackage ../development/tools/misc/inotify-tools { };

  inherit (callPackage ../applications/misc/inochi2d { })
    inochi-creator inochi-session;

  intel-gpu-tools = callPackage ../development/tools/misc/intel-gpu-tools { };

  insomnia = callPackage ../development/web/insomnia { };

  ihp-new = callPackage ../development/web/ihp-new { };

  iozone = callPackage ../development/tools/misc/iozone { };

  itstool = callPackage ../development/tools/misc/itstool { };

  jacoco = callPackage ../development/tools/analysis/jacoco { };

  jake = callPackage ../development/tools/jake { };

  inherit (callPackage ../development/tools/build-managers/jam { })
    jam
    ftjam;

  javacc = callPackage ../development/tools/parsing/javacc {
    # Upstream doesn't support anything newer than Java 8.
    # https://github.com/javacc/javacc/blob/c708628423b71ce8bc3b70143fa5b6a2b7362b3a/README.md#building-javacc-from-source
    jdk = jdk8;
    jre = jre8;
  };

  jbake = callPackage ../development/tools/jbake { };

  jbang = callPackage ../development/tools/jbang { };

  jikespg = callPackage ../development/tools/parsing/jikespg { };

  jenkins = callPackage ../development/tools/continuous-integration/jenkins { };

  jenkins-job-builder = with python3Packages; toPythonApplication jenkins-job-builder;

  jpexs = callPackage ../development/tools/jpexs { };

  k0sctl = callPackage ../applications/networking/cluster/k0sctl { };

  k2tf = callPackage ../development/tools/misc/k2tf { };

  kafka-delta-ingest = callPackage ../development/tools/kafka-delta-ingest { };

  kamid = callPackage ../servers/ftp/kamid { };

  karate = callPackage ../development/tools/karate { };

  kati = callPackage ../development/tools/build-managers/kati { };

  kafkactl = callPackage ../development/tools/kafkactl { };

  kcat = callPackage ../development/tools/kcat { };

  kcc = libsForQt5.callPackage ../applications/graphics/kcc { };

  kcgi = callPackage ../development/web/kcgi { };

  kcov = callPackage ../development/tools/analysis/kcov { };

  kind = callPackage ../development/tools/kind { };

  khronos-ocl-icd-loader = callPackage ../development/libraries/khronos-ocl-icd-loader {  };

  ko = callPackage ../development/tools/ko { };

  konstraint = callPackage ../development/tools/konstraint { };

  krankerl = callPackage ../development/tools/krankerl { };

  krew = callPackage ../development/tools/krew { };

  kube-bench = callPackage ../tools/security/kube-bench { };

  kube-hunter = callPackage ../tools/security/kube-hunter { };

  kubeaudit = callPackage ../tools/security/kubeaudit { };

  kubectx = callPackage ../development/tools/kubectx { };

  kube-linter = callPackage ../development/tools/kube-linter { };

  kube-prompt = callPackage ../development/tools/kube-prompt { };

  kubeclarity = callPackage ../tools/security/kubeclarity { };

  kubemq-community = callPackage ../servers/kubemq-community { };

  kubeone = callPackage ../applications/networking/cluster/kubeone { };

  kubeprompt = callPackage ../development/tools/kubeprompt { };

  kubernetes-polaris = callPackage ../tools/security/kubernetes-polaris { };

  kubescape = callPackage ../tools/security/kubescape { };

  kubesec = callPackage ../tools/security/kubesec { };

  kubespy = callPackage ../applications/networking/cluster/kubespy { };

  kubeswitch = callPackage ../development/tools/kubeswitch { };

  kubie = callPackage ../development/tools/kubie {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  kustomize = callPackage ../development/tools/kustomize { };

  kustomize_3 = callPackage ../development/tools/kustomize/3.nix { };

  kustomize_4 = callPackage ../development/tools/kustomize/4.nix { };

  kustomize-sops = callPackage ../development/tools/kustomize/kustomize-sops.nix { };

  ktlint = callPackage ../development/tools/ktlint { };

  kythe = callPackage ../development/tools/kythe { };

  lazygit = callPackage ../development/tools/lazygit { };

  laminar = callPackage ../development/tools/continuous-integration/laminar { };

  lcov = callPackage ../development/tools/analysis/lcov { };

  leiningen = callPackage ../development/tools/build-managers/leiningen { };

  lemon = callPackage ../development/tools/parsing/lemon { };

  lenmus = callPackage ../applications/misc/lenmus { };

  libtool = libtool_2;

  libtool_1_5 = callPackage ../development/tools/misc/libtool { };

  libtool_2 = callPackage ../development/tools/misc/libtool/libtool2.nix { };

  libwhich = callPackage ../development/tools/misc/libwhich { };

  libwtk-sdl2 = callPackage ../development/libraries/libwtk-sdl2 { };

  linuxkit = callPackage ../development/tools/misc/linuxkit {
    inherit (darwin.apple_sdk_11_0.frameworks) Cocoa Virtualization;
    inherit (darwin) sigtool;
  };

  listenbrainz-mpd = callPackage ../applications/audio/listenbrainz-mpd  {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration CoreFoundation;
  };

  lit = with python3Packages; toPythonApplication lit;

  litecli = callPackage ../development/tools/database/litecli { };

  litefs = callPackage ../development/tools/database/litefs { };

  litestream = callPackage ../development/tools/database/litestream { };

  ls-lint = callPackage ../development/tools/ls-lint { };

  lsof = callPackage ../development/tools/misc/lsof { };

  ltrace = callPackage ../development/tools/misc/ltrace { };

  lttng-tools = callPackage ../development/tools/misc/lttng-tools { };

  lttng-ust = callPackage ../development/tools/misc/lttng-ust { };

  lttng-ust_2_12 = callPackage ../development/tools/misc/lttng-ust/2.12.nix { };

  lttv = callPackage ../development/tools/misc/lttv { };

  luaformatter = callPackage ../development/tools/luaformatter { };

  lurk = callPackage ../development/tools/lurk { };

  maizzle = callPackage ../development/tools/maizzle { };

  malt = callPackage ../development/tools/profiling/malt { };

  marksman = callPackage ../development/tools/marksman { };

  massif-visualizer = libsForQt5.callPackage ../development/tools/analysis/massif-visualizer { };

  mastodon-archive = callPackage ../tools/backup/mastodon-archive { };

  math-preview = callPackage ../tools/text/math-review { };

  maven = maven3;
  maven3 = callPackage ../development/tools/build-managers/apache-maven { };

  mavproxy = python3Packages.callPackage ../applications/science/robotics/mavproxy { };

  go-md2man = callPackage ../development/tools/misc/go-md2man { };

  mage = callPackage ../development/tools/build-managers/mage { };

  mbed-cli = callPackage ../development/tools/mbed-cli { };

  mdl = callPackage ../development/tools/misc/mdl { };

  meraki-cli = python3Packages.callPackage ../tools/admin/meraki-cli { };

  mermerd = callPackage ../development/tools/database/mermerd { };

  python-matter-server = with python3Packages; toPythonApplication (
    python-matter-server.overridePythonAttrs (oldAttrs: {
      propagatedBuildInputs = oldAttrs.propagatedBuildInputs ++ oldAttrs.passthru.optional-dependencies.server;
    })
  );

  minify = callPackage ../development/web/minify { };

  minizinc = callPackage ../development/tools/minizinc { };
  minizincide = qt6Packages.callPackage ../development/tools/minizinc/ide.nix {
    inherit (darwin.apple_sdk_11_0.frameworks) Cocoa;
  };

  mkcert = callPackage ../development/tools/misc/mkcert { };

  mkdocs = with python3Packages; toPythonApplication mkdocs;

  mmtf-cpp = callPackage ../development/libraries/science/chemistry/mmtf-cpp { };

  mockgen = callPackage ../development/tools/mockgen { };

  mockoon = callPackage ../tools/networking/mockoon { };

  modd = callPackage ../development/tools/modd { };

  mold = callPackage ../by-name/mo/mold/package.nix {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
    tbb = tbb_2021_11;
  };

  mold-wrapped = wrapBintoolsWith {
    bintools = mold;
    extraBuildCommands = ''
      wrap ${targetPackages.stdenv.cc.bintools.targetPrefix}ld.mold ${../build-support/bintools-wrapper/ld-wrapper.sh} ${mold}/bin/ld.mold
      wrap ${targetPackages.stdenv.cc.bintools.targetPrefix}mold ${../build-support/bintools-wrapper/ld-wrapper.sh} ${mold}/bin/mold
    '';
  };

  moon = callPackage ../development/tools/build-managers/moon/default.nix { };

  msgpack-tools = callPackage ../development/tools/msgpack-tools { };

  msgpuck = callPackage ../development/libraries/msgpuck { };

  msitools = callPackage ../development/tools/misc/msitools { };

  haskell-ci = haskell.lib.compose.justStaticExecutables haskellPackages.haskell-ci;

  nailgun = callPackage ../development/tools/nailgun { };

  nap = callPackage ../development/tools/nap { };

  nex = callPackage ../development/tools/parsing/nex { };

  ninja = callPackage ../development/tools/build-managers/ninja { };

  nimbo = with python3Packages; callPackage ../applications/misc/nimbo { };

  gn = callPackage ../development/tools/build-managers/gn { };
  gn1924 = callPackage ../development/tools/build-managers/gn/rev1924.nix { };

  nixbang = callPackage ../development/tools/misc/nixbang {
    pythonPackages = python3Packages;
  };

  nix-bisect = callPackage ../development/tools/misc/nix-bisect { };

  nix-build-uncached = callPackage ../development/tools/misc/nix-build-uncached { };

  nexus = callPackage ../development/tools/repository-managers/nexus {
    jre_headless = jre8_headless;
  };

  nmrpflash = callPackage ../development/embedded/nmrpflash { };

  norminette = callPackage ../development/tools/norminette { };

  nwjs = callPackage ../development/tools/nwjs { };

  nwjs-sdk = callPackage ../development/tools/nwjs {
    sdk = true;
  };

  nrf5-sdk = callPackage ../development/libraries/nrf5-sdk { };

  nrfutil = callPackage ../development/tools/misc/nrfutil { };

  obelisk = callPackage ../development/tools/ocaml/obelisk { menhir = ocamlPackages.menhir; };

  obuild = callPackage ../development/tools/ocaml/obuild { };

  omake = callPackage ../development/tools/ocaml/omake { };

  omniorb = callPackage ../development/tools/omniorb { };

  openai = with python3Packages; toPythonApplication openai;

  openai-whisper = with python3.pkgs; toPythonApplication openai-whisper;

  openai-whisper-cpp = darwin.apple_sdk_11_0.callPackage ../tools/audio/openai-whisper-cpp {
    inherit (darwin.apple_sdk_11_0.frameworks) Accelerate CoreGraphics CoreML CoreVideo MetalKit;
  };

  opengrok = callPackage ../development/tools/misc/opengrok { };

  openocd = callPackage ../development/embedded/openocd { };

  openocd-rp2040 = openocd.overrideAttrs (old: {
    pname = "openocd-rp2040";
    src = fetchFromGitHub {
      owner = "raspberrypi";
      repo = "openocd";
      rev = "4d87f6dcae77d3cbcd8ac3f7dc887adf46ffa504";
      hash = "sha256-bBqVoHsnNoaC2t8hqcduI8GGlO0VDMUovCB0HC+rxvc=";
      # openocd disables the vendored libraries that use submodules and replaces them with nix versions.
      # this works out as one of the submodule sources seems to be flakey.
      fetchSubmodules = false;
    };
    nativeBuildInputs = old.nativeBuildInputs ++ [
      autoreconfHook
    ];
  });

  oprofile = callPackage ../development/tools/profiling/oprofile {
    libiberty_static = libiberty.override { staticBuild = true; };
  };

  package-project-cmake = callPackage ../development/tools/package-project-cmake { };

  pactorio = callPackage ../development/tools/pactorio {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  panopticon = callPackage ../development/tools/analysis/panopticon { };

  parinfer-rust = callPackage ../development/tools/parinfer-rust { };

  parse-cli-bin = callPackage ../development/tools/parse-cli-bin { };

  patchelf = callPackage ../development/tools/misc/patchelf { };

  patchelfUnstable = lowPrio (callPackage ../development/tools/misc/patchelf/unstable.nix { });

  patsh = callPackage ../development/tools/misc/patsh { };

  pax-rs = callPackage ../development/tools/pax-rs { };

  perfect-hash = callPackage ../development/tools/misc/perfect-hash { };

  peg = callPackage ../development/tools/parsing/peg { };

  pgcli = with pkgs.python3Packages; toPythonApplication pgcli;

  picotool = callPackage ../development/tools/picotool { };

  planus = callPackage ../development/tools/misc/planus { };

  pmccabe = callPackage ../development/tools/misc/pmccabe { };

  pkgconf-unwrapped = libpkgconf;

  pkgconf = callPackage ../build-support/pkg-config-wrapper {
    pkg-config = pkgconf-unwrapped;
    baseBinName = "pkgconf";
  };

  pkg-config-unwrapped = callPackage ../development/tools/misc/pkg-config { };
  pkg-config = callPackage ../build-support/pkg-config-wrapper {
    pkg-config = pkg-config-unwrapped;
  };

  pkg-configUpstream = lowPrio (pkg-config.override (old: {
    pkg-config = old.pkg-config.override {
      vanilla = true;
    };
  }));

  pnpm-lock-export = callPackage ../development/web/pnpm-lock-export { };

  portableService = callPackage ../build-support/portable-service { };

  polar = callPackage ../tools/misc/polar { };

  inherit (nodePackages) postcss-cli;

  postiats-utilities = callPackage ../development/tools/postiats-utilities { };

  postman = callPackage ../development/web/postman { };

  pprof = callPackage ../development/tools/profiling/pprof { };

  pqrs = callPackage ../development/tools/pqrs { };

  pyprof2calltree = with python3Packages; toPythonApplication pyprof2calltree;

  prelink = callPackage ../development/tools/misc/prelink { };

  premake3 = callPackage ../development/tools/misc/premake/3.nix { };

  premake4 = callPackage ../development/tools/misc/premake { };

  premake5 = callPackage ../development/tools/misc/premake/5.nix {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  premake = premake4;

  process-compose = callPackage ../applications/misc/process-compose { };

  process-viewer = callPackage ../applications/misc/process-viewer { };

  procodile = callPackage ../tools/system/procodile { };

  protox = callPackage ../development/tools/misc/protox { };

  proxmove = callPackage ../tools/admin/proxmove { };

  pry = callPackage ../development/tools/pry { };

  pup = callPackage ../development/tools/pup { };

  puppeteer-cli = callPackage ../tools/graphics/puppeteer-cli { };

  pyrseas = callPackage ../development/tools/database/pyrseas { };

  pycritty = with python3Packages; toPythonApplication pycritty;

  pylint-exit = callPackage ../development/tools/pylint-exit { };

  qc = callPackage ../development/tools/qc { };

  qtcreator = qt6Packages.callPackage ../development/tools/qtcreator {
    inherit (linuxPackages) perf;
    stdenv = llvmPackages.stdenv;
  };

  qxmledit = libsForQt5.callPackage ../applications/editors/qxmledit {} ;

  radare2 = callPackage ../development/tools/analysis/radare2 ({
    lua = lua5;
  } // (config.radare or {}));

  ran = callPackage ../servers/http/ran { };

  rathole = callPackage ../tools/networking/rathole {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  retry = callPackage ../tools/system/retry { };

  rizin = pkgs.callPackage ../development/tools/analysis/rizin { };

  rizinPlugins = recurseIntoAttrs rizin.plugins;

  cutter = qt6.callPackage ../development/tools/analysis/rizin/cutter.nix { };

  cutterPlugins = recurseIntoAttrs cutter.plugins;

  ragel = ragelStable;

  randoop = callPackage ../development/tools/analysis/randoop { };

  inherit (callPackages ../development/tools/parsing/ragel { }) ragelStable ragelDev;

  hammer = callPackage ../development/tools/parsing/hammer { };

  rcodesign = darwin.apple_sdk_11_0.callPackage ../development/tools/rcodesign {};

  rdocker = callPackage ../development/tools/rdocker { };

  react-static = callPackage ../development/tools/react-static { };

  redis-dump = callPackage ../development/tools/redis-dump { };

  redis-plus-plus = callPackage ../development/libraries/redis-plus-plus { };

  redisinsight = callPackage ../development/tools/redisinsight { };

  redo = callPackage ../development/tools/build-managers/redo { };

  redo-apenwarr = callPackage ../development/tools/build-managers/redo-apenwarr { };

  redo-c = callPackage ../development/tools/build-managers/redo-c { };

  redo-sh = callPackage ../development/tools/build-managers/redo-sh { };

  regclient = callPackage ../development/tools/regclient { };
  inherit (regclient) regbot regctl regsync;

  regex-cli = callPackage ../development/tools/misc/regex-cli { };

  reno = callPackage ../development/tools/reno { };

  re2c = callPackage ../development/tools/parsing/re2c { };

  remake = callPackage ../development/tools/build-managers/remake { };

  replace-secret = callPackage ../build-support/replace-secret/replace-secret.nix { };

  replacement = callPackage ../development/tools/misc/replacement { };

  inherit (callPackage ../development/tools/replay-io { })
    replay-io replay-node-cli;

  requestly = callPackage ../tools/networking/requestly { };

  reshape = callPackage ../development/tools/reshape { } ;

  retdec = callPackage ../development/tools/analysis/retdec { };

  reviewdog = callPackage ../development/tools/misc/reviewdog { };

  revive = callPackage ../development/tools/revive { };

  riff = callPackage ../development/tools/misc/riff { };

  riffdiff = callPackage ../tools/text/riffdiff {};

  rman = callPackage ../development/tools/misc/rman { };

  rnginline = with python3Packages; toPythonApplication rnginline;

  rolespec = callPackage ../development/tools/misc/rolespec { };

  rr = callPackage ../development/tools/analysis/rr { };

  rsass = callPackage ../development/tools/misc/rsass { };

  rsonpath = callPackage ../development/tools/misc/rsonpath { };

  rufo = callPackage ../development/tools/rufo { };

  rye = darwin.apple_sdk_11_0.callPackage ../development/tools/rye {
    inherit (darwin.apple_sdk_11_0) Libsystem;
    inherit (darwin.apple_sdk_11_0.frameworks) CoreServices SystemConfiguration;
  };

  samurai = callPackage ../development/tools/build-managers/samurai { };

  muon = callPackage ../development/tools/build-managers/muon { };
  muonStandalone = muon.override {
    embedSamurai = true;
    buildDocs = false;
  };

  saleae-logic = callPackage ../development/tools/misc/saleae-logic { };

  saleae-logic-2 = callPackage ../development/tools/misc/saleae-logic-2 { };

  samply = callPackage ../development/tools/misc/samply { };

  sauce-connect = callPackage ../development/tools/sauce-connect { };

  sawjap = callPackage ../development/tools/java/sawjap { };

  sbomnix = python3.pkgs.callPackage ../tools/security/sbomnix { };

  sd-local = callPackage ../development/tools/sd-local { };

  seer = libsForQt5.callPackage ../development/tools/misc/seer { };

  selenium-server-standalone = callPackage ../development/tools/selenium/server { };

  selendroid = callPackage ../development/tools/selenium/selendroid { };

  sem = callPackage ../development/tools/sem { };

  semver-tool = callPackage ../development/tools/misc/semver-tool { };

  semantik = libsForQt5.callPackage ../applications/office/semantik { };

  sca2d = callPackage ../development/tools/sca2d {  };

  scons = scons_4_5_2;
  scons_3_1_2 = callPackage ../development/tools/build-managers/scons/3.1.2.nix { };
  scons_4_1_0 = callPackage ../development/tools/build-managers/scons/4.1.0.nix { };
  scons_4_5_2 = callPackage ../development/tools/build-managers/scons/4.5.2.nix { };

  mill = callPackage ../development/tools/build-managers/mill { };

  sbt = callPackage ../development/tools/build-managers/sbt { };
  sbt-with-scala-native = callPackage ../development/tools/build-managers/sbt/scala-native.nix { };
  simpleBuildTool = sbt;

  sbt-extras = callPackage ../development/tools/build-managers/sbt-extras { };

  scala-cli = callPackage ../development/tools/build-managers/scala-cli { };

  scss-lint = callPackage ../development/tools/scss-lint { };

  segger-ozone = callPackage ../development/tools/misc/segger-ozone { };

  selene = callPackage ../development/tools/selene { };

  shadowenv = callPackage ../tools/misc/shadowenv {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  shake = haskell.lib.compose.justStaticExecutables haskellPackages.shake;

  shallot = callPackage ../tools/misc/shallot { };

  inherit (callPackage ../development/tools/build-managers/shards { })
    shards_0_17
    shards;

  shellcheck = callPackage ../development/tools/shellcheck {
    inherit (__splicedPackages.haskellPackages) ShellCheck;
  };


  # Minimal shellcheck executable for package checks.
  # Use shellcheck which does not include docs, as
  # pandoc takes long to build and documentation isn't needed for just running the cli
  shellcheck-minimal = haskell.lib.compose.justStaticExecutables shellcheck.unwrapped;

  shellharden = callPackage ../development/tools/shellharden { };

  schemaspy = callPackage ../development/tools/database/schemaspy { };

  scenebuilder = callPackage ../development/tools/scenebuilder { };

  scenic-view = callPackage ../development/tools/scenic-view { };

  shncpd = callPackage ../tools/networking/shncpd { };

  sigrok-cli = callPackage ../development/tools/sigrok-cli { };

  simpleTpmPk11 = callPackage ../tools/security/simple-tpm-pk11 { };

  slimerjs = callPackage ../development/tools/slimerjs { };

  slint-lsp = callPackage ../development/tools/misc/slint-lsp {
    inherit (darwin.apple_sdk_11_0.frameworks) AppKit CoreGraphics CoreServices CoreText Foundation OpenGL;
  };

  sloccount = callPackage ../development/tools/misc/sloccount { };

  sloc = nodePackages.sloc;

  smatch = callPackage ../development/tools/analysis/smatch { };

  smc = callPackage ../tools/misc/smc { };

  snakemake = callPackage ../applications/science/misc/snakemake { };

  snore = callPackage ../tools/misc/snore { };

  snyk = callPackage ../development/tools/analysis/snyk { };

  snzip = callPackage ../tools/archivers/snzip { };

  snowman = qt5.callPackage ../development/tools/analysis/snowman { };

  sparse = callPackage ../development/tools/analysis/sparse {
    llvm = llvm_14;
  };

  speedtest-cli = with python3Packages; toPythonApplication speedtest-cli;

  spicy-parser-generator = callPackage ../development/tools/parsing/spicy { };

  spin = callPackage ../development/tools/analysis/spin { };

  spirv-headers = callPackage ../development/libraries/spirv-headers { };
  spirv-tools = callPackage ../development/tools/spirv-tools { };

  splint = callPackage ../development/tools/analysis/splint {
    flex = flex_2_5_35;
  };

  spoofer = callPackage ../tools/networking/spoofer {
    protobuf = protobuf_21;
  };

  spoofer-gui = callPackage ../tools/networking/spoofer {
    withGUI = true;
    protobuf = protobuf_21;
  };

  spooles = callPackage ../development/libraries/science/math/spooles { };

  spr = callPackage ../development/tools/spr {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  spruce = callPackage ../development/tools/misc/spruce { };

  sqlboiler-crdb = callPackage ../development/tools/database/sqlboiler-crdb { };

  sqlc = callPackage ../development/tools/database/sqlc { };

  sqlcheck = callPackage ../development/tools/database/sqlcheck { };

  sqlcmd = callPackage ../development/tools/database/sqlcmd { };

  sqlitebrowser = libsForQt5.callPackage ../development/tools/database/sqlitebrowser { };

  sqlite-utils = with python3Packages; toPythonApplication sqlite-utils;

  sqlite-web = callPackage ../development/tools/database/sqlite-web { };

  sqlmap = with python3Packages; toPythonApplication sqlmap;

  sqlpage = callPackage ../servers/sqlpage { };

  src-cli = callPackage ../development/tools/misc/src-cli { };

  sselp = callPackage ../tools/X11/sselp{ };

  statix = callPackage ../tools/nix/statix { };

  stm32cubemx = callPackage ../development/embedded/stm32/stm32cubemx { };

  stm32flash = callPackage ../development/embedded/stm32/stm32flash { };

  stm8flash = callPackage ../development/embedded/stm8/stm8flash { };

  strace = callPackage ../development/tools/misc/strace { };

  strace-analyzer = callPackage ../development/tools/misc/strace-analyzer { };

  stylelint = callPackage ../development/tools/analysis/stylelint { };

  stylua = callPackage ../development/tools/stylua { };

  summon = callPackage ../development/tools/summon { };

  supabase-cli = callPackage ../development/tools/supabase-cli { };

  surge-cli = callPackage ../development/tools/surge-cli { };

  svlint = callPackage ../development/tools/analysis/svlint { };

  swarm = callPackage ../development/tools/analysis/swarm { };

  swiftformat = callPackage ../development/tools/swiftformat { };

  swiftshader = callPackage ../development/libraries/swiftshader { };

  systemfd = callPackage ../development/tools/systemfd { };

  swig1 = callPackage ../development/tools/misc/swig { };
  swig2 = callPackage ../development/tools/misc/swig/2.x.nix { };
  swig3 = callPackage ../development/tools/misc/swig/3.x.nix { };
  swig4 = callPackage ../development/tools/misc/swig/4.nix { };
  swig = swig3;
  swigWithJava = swig;

  c2ffi = callPackage ../development/tools/misc/c2ffi { };

  c0 = callPackage ../development/compilers/c0 {
    stdenv = if stdenv.isDarwin then gccStdenv else stdenv;
  };

  c3c = callPackage ../development/compilers/c3c { };

  swfmill = callPackage ../tools/video/swfmill { };

  swftools = callPackage ../tools/video/swftools {
    stdenv = gccStdenv;
  };

  szyszka = callPackage ../tools/misc/szyszka { };

  taplo = callPackage ../development/tools/taplo {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  taoup = callPackage ../tools/misc/taoup { };

  tarmac = callPackage ../development/tools/tarmac {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  tcptrack = callPackage ../development/tools/misc/tcptrack { };

  teensy-cmake-macros = callPackage ../development/embedded/teensy-cmake-macros { };

  teensyduino = arduino-core.override { withGui = true; withTeensyduino = true; };

  teensy-loader-cli = callPackage ../development/embedded/teensy-loader-cli { };

  tytools = libsForQt5.callPackage ../development/embedded/tytools { };

  terracognita = callPackage ../development/tools/misc/terracognita { };

  terraform-lsp = callPackage ../development/tools/misc/terraform-lsp { };
  terraform-ls = callPackage ../development/tools/misc/terraform-ls {
    buildGoModule = buildGo122Module;
  };

  terraformer = callPackage ../development/tools/misc/terraformer { };

  terramate = callPackage ../development/tools/misc/terramate { };

  terrascan = callPackage ../tools/security/terrascan { };

  terser = callPackage ../development/tools/misc/terser { };

  tesh = callPackage ../tools/text/tesh {};

  texinfo413 = callPackage ../development/tools/misc/texinfo/4.13a.nix { };
  texinfo4 = texinfo413;
  texinfo5 = callPackage ../development/tools/misc/texinfo/5.2.nix { };
  texinfo6_5 = callPackage ../development/tools/misc/texinfo/6.5.nix { }; # needed for allegro
  texinfo6_7 = callPackage ../development/tools/misc/texinfo/6.7.nix { }; # needed for gpm, iksemel and fwknop
  texinfo6 = callPackage ../development/tools/misc/texinfo/6.8.nix { };
  texinfo7 = callPackage ../development/tools/misc/texinfo/7.0.nix { };
  texinfo = texinfo7;
  texinfoInteractive = texinfo.override { interactive = true; };

  texi2html = callPackage ../development/tools/misc/texi2html { };

  texi2mdoc = callPackage ../tools/misc/texi2mdoc { };

  texlab = callPackage ../development/tools/misc/texlab {
    inherit (darwin.apple_sdk.frameworks) Security CoreServices;
  };

  time-ghc-modules = callPackage ../development/tools/time-ghc-modules { };

  tflint = callPackage ../development/tools/analysis/tflint { };

  tflint-plugins = recurseIntoAttrs (
    callPackage ../development/tools/analysis/tflint-plugins { }
  );

  tfsec = callPackage ../development/tools/analysis/tfsec { };

  todoist = callPackage ../applications/misc/todoist { };

  todoist-electron = callPackage ../applications/misc/todoist-electron { };

  toil = callPackage ../applications/science/misc/toil { };

  travis = callPackage ../development/tools/misc/travis { };

  tree-sitter = makeOverridable (callPackage ../development/tools/parsing/tree-sitter) {
    inherit (darwin.apple_sdk.frameworks) Security CoreServices;
  };

  tree-sitter-grammars = recurseIntoAttrs tree-sitter.builtGrammars;

  trellis = callPackage ../development/embedded/fpga/trellis { };

  ttags = callPackage ../development/tools/misc/ttags { };

  ttyd = callPackage ../servers/ttyd { };

  turbogit = callPackage ../development/tools/turbogit {
    libgit2 = libgit2.overrideAttrs rec {
      version = "1.3.0";
      src = pkgs.fetchFromGitHub {
        owner = "libgit2";
        repo = "libgit2";
        rev = "v${version}";
        hash = "sha256-7atNkOBzX+nU1gtFQEaE+EF1L+eex+Ajhq2ocoJY920=";
      };
      patches = [];
      # tests fail on old version
      doCheck = false;
      meta = libgit2.meta // {
        maintainers = [];
        knownVulnerabilities = [ "CVE-2024-24575" "CVE-2024-24577" "CVE-2022-29187" "CVE 2022-24765" ];
      };
    };
  };

  tweak = callPackage ../applications/editors/tweak { };

  typical = callPackage ../development/tools/misc/typical { };

  tyson = callPackage ../development/tools/misc/tyson { };

  uddup = callPackage ../tools/security/uddup { };

  udis86 = callPackage  ../development/tools/udis86 { };

  uefi-firmware-parser = callPackage ../development/tools/analysis/uefi-firmware-parser { };

  uhd = callPackage ../applications/radio/uhd { };
  uhdMinimal = uhd.override {
    enableUtils = false;
    enablePythonApi = false;
  };

  uisp = callPackage ../development/embedded/uisp { };

  wch-isp = callPackage ../development/embedded/wch-isp { };

  uncrustify = callPackage ../development/tools/misc/uncrustify { };

  universal-ctags = callPackage ../development/tools/misc/universal-ctags { };

  unused = callPackage ../development/tools/misc/unused { };

  vagrant = callPackage ../development/tools/vagrant { };

  bashdb = callPackage ../development/tools/misc/bashdb { };

  gdb = callPackage ../development/tools/misc/gdb {
    guile = null;
  };

  gdbHostCpuOnly = gdb.override { hostCpuOnly = true; };

  gf = callPackage ../development/tools/misc/gf { };

  java-language-server = callPackage ../development/tools/java/java-language-server { };

  jprofiler = callPackage ../development/tools/java/jprofiler {
    jdk = jdk11;
  };

  jhiccup = callPackage ../development/tools/java/jhiccup { };

  valgrind = callPackage ../development/tools/analysis/valgrind {
    inherit (buildPackages.darwin) xnu bootstrap_cmds cctools;
  };
  valgrind-light = res.valgrind.override { gdb = null; };

  qcachegrind = libsForQt5.callPackage ../development/tools/analysis/qcachegrind { };

  visualvm = callPackage ../development/tools/java/visualvm { };

  volta = callPackage ../development/tools/volta { };

  vultr = callPackage ../development/tools/vultr { };

  vultr-cli = callPackage ../development/tools/vultr-cli { };

  vulnix = callPackage ../tools/security/vulnix { };

  vtable-dumper = callPackage ../development/tools/misc/vtable-dumper { };

  wails = callPackage ../development/tools/wails {
    stdenv = gccStdenv;
  };

  wasmer-pack = callPackage ../development/tools/misc/wasmer-pack { };

  whatsapp-for-linux = callPackage ../applications/networking/instant-messengers/whatsapp-for-linux { };

  whatstyle = callPackage ../development/tools/misc/whatstyle {
    inherit (llvmPackages) clang-unwrapped;
  };

  watson-ruby = callPackage ../development/tools/misc/watson-ruby { };

  web-ext = callPackage ../development/tools/web-ext { };

  webdis = callPackage ../development/tools/database/webdis { };

  xmake = callPackage ../development/tools/build-managers/xmake {
    lua = lua5_4;
  };

  xc3sprog = callPackage ../development/embedded/xc3sprog { };

  xcb-imdkit = callPackage ../development/libraries/xcb-imdkit { };

  xcode-install = callPackage ../development/tools/xcode-install { };

  xcodebuild = callPackage ../development/tools/xcbuild/wrapper.nix {
    inherit (darwin.apple_sdk.frameworks) CoreServices CoreGraphics ImageIO;
  };
  xcbuild = xcodebuild;
  xcbuildHook = makeSetupHook {
    name = "xcbuild-hook";
    propagatedBuildInputs = [ xcbuild ];
  } ../development/tools/xcbuild/setup-hook.sh  ;

  xcpretty = callPackage ../development/tools/xcpretty { };

  xeus = callPackage ../development/libraries/xeus { };

  xeus-zmq = callPackage ../development/libraries/xeus-zmq { };

  xmlindent = callPackage ../development/web/xmlindent { };

  xpwn = callPackage ../development/mobile/xpwn { };

  xxdiff = libsForQt5.callPackage ../development/tools/misc/xxdiff { };

  xxe-pe = callPackage ../applications/editors/xxe-pe { };

  xxdiff-tip = xxdiff;

  xxgdb = callPackage ../development/tools/misc/xxgdb { };

  yaml2json = callPackage ../development/tools/yaml2json { };

  yams = callPackage ../applications/audio/yams { };

  ycmd = callPackage ../by-name/yc/ycmd/package.nix {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
    python = python3;
  };

  yo = callPackage ../development/tools/yo { };

  yodl = callPackage ../development/tools/misc/yodl { };

  yq = python3.pkgs.toPythonApplication python3.pkgs.yq;

  yq-go = callPackage ../development/tools/yq-go { };

  ytt = callPackage ../development/tools/ytt { };

  zydis = callPackage ../development/libraries/zydis { };

  grabserial = callPackage ../development/tools/grabserial { };

  mypy = with python3Packages; toPythonApplication mypy;

  mypy-protobuf = with python3Packages; toPythonApplication mypy-protobuf;

  nsis = callPackage ../development/tools/nsis { };

  tockloader = callPackage ../development/tools/misc/tockloader { };

  zon2nix = callPackage ../tools/nix/zon2nix { };

  ztags = callPackage ../development/tools/misc/ztags { };

  ### DEVELOPMENT / LIBRARIES

  aalib = callPackage ../development/libraries/aalib { };

  abseil-cpp_202103 = callPackage ../development/libraries/abseil-cpp/202103.nix {
    # If abseil-cpp doesn’t have a deployment target of 10.13+, arrow-cpp crashes in libgrpc.dylib.
    stdenv = if stdenv.isDarwin && stdenv.isx86_64
      then overrideSDK stdenv { darwinMinVersion = "10.13"; }
      else stdenv;
  };
  abseil-cpp_202206 = callPackage ../development/libraries/abseil-cpp/202206.nix {
    # If abseil-cpp doesn’t have a deployment target of 10.13+, arrow-cpp crashes in libgrpc.dylib.
    stdenv = if stdenv.isDarwin && stdenv.isx86_64
      then overrideSDK stdenv { darwinMinVersion = "10.13"; }
      else stdenv;
  };
  abseil-cpp_202301 = callPackage ../development/libraries/abseil-cpp/202301.nix {
    # If abseil-cpp doesn’t have a deployment target of 10.13+, arrow-cpp crashes in libgrpc.dylib.
    stdenv = if stdenv.isDarwin && stdenv.isx86_64
      then overrideSDK stdenv { darwinMinVersion = "10.13"; }
      else stdenv;
  };
  abseil-cpp_202308 = callPackage ../development/libraries/abseil-cpp/202308.nix {
    # If abseil-cpp doesn’t have a deployment target of 10.13+, arrow-cpp crashes in libgrpc.dylib.
    stdenv = if stdenv.isDarwin && stdenv.isx86_64
      then overrideSDK stdenv { darwinMinVersion = "10.13"; }
      else stdenv;
  };
  abseil-cpp_202401 = callPackage ../development/libraries/abseil-cpp/202401.nix {
    # If abseil-cpp doesn’t have a deployment target of 10.13+, arrow-cpp crashes in libgrpc.dylib.
    stdenv = if stdenv.isDarwin && stdenv.isx86_64
      then overrideSDK stdenv { darwinMinVersion = "10.13"; }
      else stdenv;
  };
  abseil-cpp = abseil-cpp_202401;

  accountsservice = callPackage ../development/libraries/accountsservice { };

  acl = callPackage ../development/libraries/acl { };

  acltoolkit = callPackage ../tools/security/acltoolkit { };

  acsccid = callPackage ../tools/security/acsccid { };

  activemq = callPackage ../development/libraries/apache-activemq { };

  adns = callPackage ../development/libraries/adns { };

  adslib = callPackage ../development/libraries/adslib { };

  aemu = callPackage ../development/libraries/aemu { };

  afflib = callPackage ../development/libraries/afflib { };

  aften = callPackage ../development/libraries/aften { };

  alure = callPackage ../development/libraries/alure { };

  alure2 = callPackage ../development/libraries/alure2 { };

  agg = callPackage ../development/libraries/agg {
    stdenv = gccStdenv;
  };

  agkozak-zsh-prompt = callPackage ../shells/zsh/agkozak-zsh-prompt { };

  alass = callPackage ../applications/video/alass { };

  allegro = allegro4;
  allegro4 = callPackage ../development/libraries/allegro { };
  allegro5 = callPackage ../development/libraries/allegro/5.nix { };

  amdvlk = callPackage ../development/libraries/amdvlk { };

  amf-headers = callPackage ../development/libraries/amf-headers { };

  aml = callPackage ../development/libraries/aml { };

  amrnb = callPackage ../development/libraries/amrnb { };

  amrwb = callPackage ../development/libraries/amrwb { };

  ansi2html = with python3.pkgs; toPythonApplication ansi2html;

  antic = callPackage ../development/libraries/antic { };

  anttweakbar = callPackage ../development/libraries/AntTweakBar { };

  appstream = callPackage ../development/libraries/appstream { };

  appstream-glib = callPackage ../development/libraries/appstream-glib { };

  apr = callPackage ../development/libraries/apr {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  aprutil = callPackage ../development/libraries/apr-util { };

  aravis = callPackage ../development/libraries/aravis { };

  arb = callPackage ../development/libraries/arb { };

  argparse = callPackage ../development/libraries/argparse { };

  argparse-manpage = with python3Packages; toPythonApplication argparse-manpage;

  argp-standalone = callPackage ../development/libraries/argp-standalone { };

  aribb25 = callPackage ../development/libraries/aribb25 {
    inherit (darwin.apple_sdk.frameworks) PCSC;
  };

  armadillo = callPackage ../development/libraries/armadillo { };

  arrayfire = callPackage ../development/libraries/arrayfire {
    cudaPackages = cudaPackages_12;
  };

  arrow-glib = callPackage ../development/libraries/arrow-glib { };

  arsenal = callPackage ../tools/security/arsenal { };

  assimp = callPackage ../development/libraries/assimp { };

  asio_1_10 = callPackage ../development/libraries/asio/1.10.nix { };
  asio = callPackage ../development/libraries/asio { };

  asmjit = callPackage ../development/libraries/asmjit { };

  aspell = callPackage ../development/libraries/aspell { };

  aspellDicts = recurseIntoAttrs (callPackages ../development/libraries/aspell/dictionaries.nix {});

  aspellWithDicts = callPackage ../development/libraries/aspell/aspell-with-dicts.nix {
    aspell = aspell.override { searchNixProfiles = false; };
  };

  attr = callPackage ../development/libraries/attr { };

  at-spi2-core = callPackage ../development/libraries/at-spi2-core { };

  # Not moved to aliases while we decide if we should split the package again.
  at-spi2-atk = at-spi2-core;

  aqbanking = callPackage ../development/libraries/aqbanking { };

  aubio = callPackage ../development/libraries/aubio { };

  audiality2 = callPackage ../development/libraries/audiality2 { };

  audiofile = callPackage ../development/libraries/audiofile {
    inherit (darwin.apple_sdk.frameworks) AudioUnit CoreServices;
  };

  aws-c-auth = callPackage ../development/libraries/aws-c-auth { };

  aws-c-cal = callPackage ../development/libraries/aws-c-cal {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  aws-c-common = callPackage ../development/libraries/aws-c-common { };

  aws-c-compression = callPackage ../development/libraries/aws-c-compression { };

  aws-c-event-stream = callPackage ../development/libraries/aws-c-event-stream { };

  aws-c-http = callPackage ../development/libraries/aws-c-http { };

  aws-c-io = callPackage ../development/libraries/aws-c-io {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  aws-c-mqtt = callPackage ../development/libraries/aws-c-mqtt { };

  aws-c-s3 = callPackage ../development/libraries/aws-c-s3 { };

  aws-c-sdkutils = callPackage ../development/libraries/aws-c-sdkutils { };

  aws-checksums = callPackage ../development/libraries/aws-checksums { };

  aws-crt-cpp = callPackage ../development/libraries/aws-crt-cpp { };

  aws-sdk-cpp = callPackage ../development/libraries/aws-sdk-cpp {
    inherit (darwin.apple_sdk.frameworks) CoreAudio AudioToolbox;
  };

  ayatana-ido = callPackage ../development/libraries/ayatana-ido { };

  ayatana-webmail = callPackage ../applications/networking/mailreaders/ayatana-webmail { };

  azmq = callPackage ../development/libraries/azmq { };

  babl = callPackage ../development/libraries/babl { };

  backlight-auto = callPackage ../by-name/ba/backlight-auto/package.nix {
    zig = zig_0_11;
  };

  backward-cpp = callPackage ../development/libraries/backward-cpp { };

  bamf = callPackage ../development/libraries/bamf { };

  inherit (callPackages ../development/libraries/bashup-events { }) bashup-events32 bashup-events44;

  bcg729 = callPackage ../development/libraries/bcg729 { };

  bctoolbox = callPackage ../development/libraries/bctoolbox { };

  bc-soci = callPackage ../development/libraries/soci/bc-soci.nix { };

  bearssl = callPackage ../development/libraries/bearssl { };

  beecrypt = callPackage ../development/libraries/beecrypt { };

  belcard = callPackage ../development/libraries/belcard { };

  belr = callPackage ../development/libraries/belr { };

  bencode = callPackage ../development/libraries/bencode { };

  bencodetools = callPackage ../development/libraries/bencodetools { };

  belle-sip = callPackage ../development/libraries/belle-sip { };


  bicpl = callPackage ../development/libraries/science/biology/bicpl { };

  bicgl = callPackage ../development/libraries/science/biology/bicgl { inherit (darwin.apple_sdk.frameworks) GLUT; };

  # TODO(@Ericson2314): Build bionic libc from source
  bionic = if stdenv.hostPlatform.useAndroidPrebuilt
    then pkgs."androidndkPkgs_${stdenv.hostPlatform.ndkVer}".libraries
    else callPackage ../os-specific/linux/bionic-prebuilt { };

  boolstuff = callPackage ../development/libraries/boolstuff { };

  inherit (callPackage ../development/libraries/boost { inherit (buildPackages) boost-build; })
    boost175
    boost177
    boost178
    boost179
    boost180
    boost181
    boost182
    boost183
    boost184
    boost185
  ;

  boost = boost181;

  boost_process = callPackage ../development/libraries/boost-process { };

  bosh-cli = callPackage ../applications/networking/cluster/bosh-cli { };

  botan2 = callPackage ../development/libraries/botan/2.0.nix {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  botan3 = callPackage ../development/libraries/botan/3.0.nix {
    inherit (darwin.apple_sdk.frameworks) CoreServices Security;
  };

  box2d = callPackage ../development/libraries/box2d {
    inherit (darwin.apple_sdk.frameworks) Carbon Cocoa Kernel OpenGL;
  };

  boxfort = callPackage ../development/libraries/boxfort { };

  brunsli = callPackage ../development/libraries/brunsli { };

  buddy = callPackage ../development/libraries/buddy { };

  buildkit-nix = callPackage ../applications/virtualization/buildkit-nix { };

  bulletml = callPackage ../development/libraries/bulletml { };

  bwidget = callPackage ../development/libraries/bwidget { };

  bzrtp = callPackage ../development/libraries/bzrtp { };

  c-ares = callPackage ../development/libraries/c-ares { };

  c-aresMinimal = callPackage ../development/libraries/c-ares {
    withCMake = false;
  };

  inherit (callPackages ../development/libraries/c-blosc { })
    c-blosc c-blosc2;

  cachix = lib.getBin haskellPackages.cachix;

  calcium = callPackage ../development/libraries/calcium { };

  cubeb = callPackage ../development/libraries/audio/cubeb {
    inherit (darwin.apple_sdk.frameworks) AudioUnit CoreAudio CoreServices;
  };

  hercules-ci-agent = callPackage ../development/tools/continuous-integration/hercules-ci-agent { };

  hci = callPackage ../development/tools/continuous-integration/hci { };

  isa-l = callPackage ../development/libraries/isa-l { };

  niv = lib.getBin (haskell.lib.compose.justStaticExecutables haskellPackages.niv);

  ormolu = haskellPackages.ormolu.bin;

  capnproto = callPackage ../development/libraries/capnproto { };

  capnproto-java = callPackage ../development/tools/capnproto-java { };

  capnproto-rust = callPackage ../development/tools/capnproto-rust { };

  captive-browser = callPackage ../applications/networking/browsers/captive-browser { };

  catboost = callPackage ../development/libraries/catboost {
    # https://github.com/catboost/catboost/issues/2540
    cudaPackages = cudaPackages_11;
  };

  ndn-cxx = callPackage ../development/libraries/ndn-cxx { };

  ndn-tools = callPackage ../tools/networking/ndn-tools { };

  nfd = callPackage ../servers/nfd { };

  cddlib = callPackage ../development/libraries/cddlib { };

  cdk-go = callPackage ../tools/security/cdk-go { };

  cdo = callPackage ../development/libraries/cdo { };

  cista = callPackage ../development/libraries/cista { };

  cjose = callPackage ../development/libraries/cjose { };

  scmccid = callPackage ../development/libraries/scmccid { };

  ccrtp = callPackage ../development/libraries/ccrtp { };

  cctag = callPackage ../development/libraries/cctag {
    stdenv = clangStdenv;
    tbb = tbb_2021_11;
  };

  cctz = callPackage ../development/libraries/cctz {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  ceedling = callPackage ../development/tools/ceedling { };

  celt = callPackage ../development/libraries/celt { };
  celt_0_7 = callPackage ../development/libraries/celt/0.7.nix { };
  celt_0_5_1 = callPackage ../development/libraries/celt/0.5.1.nix { };

  cegui = callPackage ../development/libraries/cegui { };

  certbot = python3.pkgs.toPythonApplication python3.pkgs.certbot;

  certbot-full = certbot.withPlugins (cp: with cp; [
    certbot-dns-cloudflare
    certbot-dns-google
    certbot-dns-ovh
    certbot-dns-rfc2136
    certbot-dns-route53
  ]);

  caf = callPackage ../development/libraries/caf { };

  # CGAL 5 has API changes
  cgal_4 = callPackage ../development/libraries/CGAL/4.nix { };
  cgal_5 = callPackage ../development/libraries/CGAL { };
  cgal = cgal_5;

  cgui = callPackage ../development/libraries/cgui { };

  charls = callPackage ../development/libraries/charls { };

  check = callPackage ../development/libraries/check {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  chipmunk = callPackage ../development/libraries/chipmunk { };

  chmlib = callPackage ../development/libraries/chmlib { };

  chromaprint = callPackage ../development/libraries/chromaprint { };

  cl = callPackage ../development/libraries/cl { };

  clanlib = callPackage ../development/libraries/clanlib { };

  clap = callPackage ../development/libraries/clap { };

  classads = callPackage ../development/libraries/classads { };

  clfft = callPackage ../development/libraries/clfft { };

  clipp  = callPackage ../development/libraries/clipp { };

  clipper = callPackage ../development/libraries/clipper { };

  clipper2 = callPackage ../development/libraries/clipper2 { };

  cln = callPackage ../development/libraries/cln { };

  clucene_core_2 = callPackage ../development/libraries/clucene-core/2.x.nix { };

  clucene_core = clucene_core_2;

  clutter = callPackage ../development/libraries/clutter { };

  clutter-gst = callPackage ../development/libraries/clutter-gst {
  };

  clutter-gtk = callPackage ../development/libraries/clutter-gtk { };

  cminpack = callPackage ../development/libraries/cminpack { };

  cmocka = callPackage ../development/libraries/cmocka { };

  cmrt = callPackage ../development/libraries/cmrt { };

  codecserver = callPackage ../applications/audio/codecserver {
    protobuf = protobuf_21;
  };

  coeurl = callPackage ../development/libraries/coeurl { };

  coercer = callPackage ../tools/security/coercer { };

  cogl = callPackage ../development/libraries/cogl {
    inherit (darwin.apple_sdk.frameworks) OpenGL;
  };

  coin3d = callPackage ../development/libraries/coin3d { };

  soxt = callPackage ../development/libraries/soxt { };

  CoinMP = callPackage ../development/libraries/CoinMP { };

  coinlive = callPackage ../tools/misc/coinlive {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  cointop = callPackage ../applications/misc/cointop { };

  coloquinte = callPackage ../development/libraries/science/electronics/coloquinte { };

  cog = callPackage ../development/web/cog { };

  inherit (cosmopolitan) cosmocc;

  python-cosmopolitan = callPackage ../development/interpreters/python-cosmopolitan { };

  ctpp2 = callPackage ../development/libraries/ctpp2 { };

  ctpl = callPackage ../development/libraries/ctpl { };

  cppe = callPackage ../development/libraries/science/chemistry/cppe { };

  cppdb = callPackage ../development/libraries/cppdb { };

  cpp-utilities = callPackage ../development/libraries/cpp-utilities { };

  cpp-hocon = callPackage ../development/libraries/cpp-hocon { };

  cpp-ipfs-http-client = callPackage ../development/libraries/cpp-ipfs-http-client { };

  cpp-netlib = callPackage ../development/libraries/cpp-netlib { };

  cpp-jwt = callPackage ../development/libraries/cpp-jwt { };

  ctranslate2 = callPackage ../development/libraries/ctranslate2 rec {
    stdenv = if withCUDA then gcc11Stdenv else pkgs.stdenv;
    withCUDA = pkgs.config.cudaSupport;
    withCuDNN = withCUDA && (cudaPackages ? cudnn);
    cudaPackages = pkgs.cudaPackages;
  };

  ubus = callPackage ../development/libraries/ubus { };

  uci = callPackage ../development/libraries/uci { };

  uclient = callPackage ../development/libraries/uclient { };

  ustream-ssl = callPackage ../development/libraries/ustream-ssl { ssl_implementation = openssl; };

  ustream-ssl-wolfssl = callPackage ../development/libraries/ustream-ssl { ssl_implementation = wolfssl; additional_buildInputs = [ openssl ]; };

  ustream-ssl-mbedtls = callPackage ../development/libraries/ustream-ssl {
    ssl_implementation = mbedtls_2;
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
  };

  uri = callPackage ../development/libraries/uri { };

  cppcms = callPackage ../development/libraries/cppcms { };

  cppcodec = callPackage ../development/libraries/cppcodec { };

  cppunit = callPackage ../development/libraries/cppunit { };

  cpputest = callPackage ../development/libraries/cpputest { };

  cracklib = callPackage ../development/libraries/cracklib { };

  cre2 = callPackage ../development/libraries/cre2 { };

  croaring = callPackage ../development/libraries/croaring { };

  crocoddyl = callPackage ../development/libraries/crocoddyl { };

  crossguid = callPackage ../development/libraries/crossguid { };

  cryptopp = callPackage ../development/libraries/crypto++ { };

  cryptominisat = callPackage ../applications/science/logic/cryptominisat { };

  csdr = callPackage ../applications/radio/csdr { };

  ctypes_sh = callPackage ../development/libraries/ctypes_sh { };

  curlpp = callPackage ../development/libraries/curlpp { };

  cutee = callPackage ../development/libraries/cutee { };

  cxxtools = callPackage ../development/libraries/cxxtools { };

  cwiid = callPackage ../development/libraries/cwiid { };

  cxx-prettyprint = callPackage ../development/libraries/cxx-prettyprint { };

  cxxopts = callPackage ../development/libraries/cxxopts { };

  cxxtest = python3Packages.callPackage ../development/libraries/cxxtest { };

  cypress = callPackage ../development/web/cypress { };

  cyrus_sasl = callPackage ../development/libraries/cyrus-sasl { };

  cyrus-sasl-xoauth2 = callPackage ../development/libraries/cyrus-sasl-xoauth2 { };

  # Make bdb5 the default as it is the last release under the custom
  # bsd-like license
  db = db5;
  db4 = db48;
  db48 = callPackage ../development/libraries/db/db-4.8.nix { };
  db5 = db53;
  db53 = callPackage ../development/libraries/db/db-5.3.nix { };
  db6 = db60;
  db60 = callPackage ../development/libraries/db/db-6.0.nix { };
  db62 = callPackage ../development/libraries/db/db-6.2.nix { };

  dbxml = callPackage ../development/libraries/dbxml { };

  dbus = callPackage ../development/libraries/dbus { };
  dbus_cplusplus  = callPackage ../development/libraries/dbus-cplusplus { };
  dbus-glib       = callPackage ../development/libraries/dbus-glib { };
  dbus_java       = callPackage ../development/libraries/java/dbus-java { };

  dbus-sharp-1_0 = callPackage ../development/libraries/dbus-sharp/dbus-sharp-1.0.nix { };
  dbus-sharp-2_0 = callPackage ../development/libraries/dbus-sharp { };

  dbus-sharp-glib-1_0 = callPackage ../development/libraries/dbus-sharp-glib/dbus-sharp-glib-1.0.nix { };
  dbus-sharp-glib-2_0 = callPackage ../development/libraries/dbus-sharp-glib { };

  makeDBusConf = { suidHelper, serviceDirectories, apparmor ? "disabled" }:
    callPackage ../development/libraries/dbus/make-dbus-conf.nix {
      inherit suidHelper serviceDirectories apparmor;
    };

  dee = callPackage ../development/libraries/dee {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  dhex = callPackage ../applications/editors/dhex { };

  double-conversion = callPackage ../development/libraries/double-conversion { };

  dclib = callPackage ../development/libraries/dclib { };

  digiham = callPackage ../applications/radio/digiham { };

  directfb = callPackage ../development/libraries/directfb { };

  discord-rpc = callPackage ../development/libraries/discord-rpc {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  discord-sh = callPackage ../tools/networking/discord-sh { };

  dlib = callPackage ../development/libraries/dlib { };

  doctest = callPackage ../development/libraries/doctest { };

  docopt_cpp = callPackage ../development/libraries/docopt_cpp { };

  docopts = callPackage ../development/tools/misc/docopts { };

  dotconf = callPackage ../development/libraries/dotconf { };

  draco = callPackage ../development/libraries/draco {
    tinygltf = callPackage ../development/libraries/draco/tinygltf.nix { };
  };

  # Multi-arch "drivers" which we want to build for i686.
  driversi686Linux = recurseIntoAttrs {
    inherit (pkgsi686Linux)
      amdvlk
      intel-media-driver
      intel-vaapi-driver
      mesa
      libvdpau-va-gl
      vaapiVdpau
      glxinfo
      vdpauinfo;
  };

  drogon = callPackage ../development/libraries/drogon { };

  dssi = callPackage ../development/libraries/dssi { };

  duckdb = callPackage ../development/libraries/duckdb { };

  easyloggingpp = callPackage ../development/libraries/easyloggingpp { };

  eccodes = callPackage ../development/libraries/eccodes {
    pythonPackages = python3Packages;
    stdenv = if stdenv.isDarwin then gccStdenv else stdenv;
  };

  eclib = callPackage ../development/libraries/eclib { };

  editline = callPackage ../development/libraries/editline { };

  edencommon = darwin.apple_sdk_11_0.callPackage ../development/libraries/edencommon { };

  eigen = callPackage ../development/libraries/eigen { };

  eigen2 = callPackage ../development/libraries/eigen/2.0.nix { };

  eigenmath = callPackage ../applications/science/math/eigenmath { };

  vapoursynth = callPackage ../development/libraries/vapoursynth {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices;
  };

  vapoursynth-editor = libsForQt5.callPackage ../development/libraries/vapoursynth/editor.nix { };

  vapoursynth-mvtools = callPackage ../development/libraries/vapoursynth-mvtools { };

  vmmlib = callPackage ../development/libraries/vmmlib {
    inherit (darwin.apple_sdk.frameworks) Accelerate CoreGraphics CoreVideo;
  };

  eglexternalplatform = callPackage ../development/libraries/eglexternalplatform { };

  egl-wayland = callPackage ../development/libraries/egl-wayland { };

  elastix = callPackage ../development/libraries/science/biology/elastix {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  elfio = callPackage ../development/libraries/elfio { };

  emanote = haskell.lib.compose.justStaticExecutables haskellPackages.emanote;

  enchant2 = callPackage ../development/libraries/enchant/2.x.nix { };
  enchant = enchant2;

  enet = callPackage ../development/libraries/enet { };

  entt = callPackage ../development/libraries/entt { };

  epoll-shim = callPackage ../development/libraries/epoll-shim { };

  libepoxy = callPackage ../development/libraries/libepoxy {
    inherit (darwin.apple_sdk.frameworks) Carbon OpenGL;
  };

  libesmtp = callPackage ../development/libraries/libesmtp { };

  liberasurecode = callPackage ../applications/misc/liberasurecode { };

  example-robot-data = callPackage ../development/libraries/example-robot-data { };

  exiv2 = callPackage ../development/libraries/exiv2 { };

  expat = callPackage ../development/libraries/expat { };

  exprtk = callPackage ../development/libraries/exprtk { };

  eventlog = callPackage ../development/libraries/eventlog { };

  faac = callPackage ../development/libraries/faac { };

  faad2 = callPackage ../development/libraries/faad2 { };

  factor-lang-scope = callPackage ../development/compilers/factor-lang/scope.nix { };
  factor-lang = factor-lang-scope.interpreter;

  far2l = callPackage ../applications/misc/far2l {
    inherit (darwin.apple_sdk.frameworks) IOKit Carbon Cocoa AudioToolbox OpenGL System;
  };

  farbfeld = callPackage ../development/libraries/farbfeld { };

  farstream = callPackage ../development/libraries/farstream {
    inherit (gst_all_1)
      gstreamer gst-plugins-base gst-plugins-good gst-plugins-bad
      gst-libav;
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  fbthrift = darwin.apple_sdk_11_0.callPackage ../development/libraries/fbthrift { };

  fb303 = darwin.apple_sdk_11_0.callPackage ../development/libraries/fb303 { };

  fcgi = callPackage ../development/libraries/fcgi { };

  fcl = callPackage ../development/libraries/fcl { };

  febio = callPackage ../development/libraries/febio { };

  ffcast = callPackage ../tools/X11/ffcast { };

  fflas-ffpack = callPackage ../development/libraries/fflas-ffpack { };

  forge = callPackage ../development/libraries/forge { };

  linbox = callPackage ../development/libraries/linbox { };

  inherit (callPackage ../development/libraries/ffmpeg { })
    ffmpeg_4
    ffmpeg_4-headless
    ffmpeg_4-full
    ffmpeg_5
    ffmpeg_5-headless
    ffmpeg_5-full
    ffmpeg_6
    ffmpeg_6-headless
    ffmpeg_6-full
    ffmpeg_7
    ffmpeg_7-headless
    ffmpeg_7-full
    ffmpeg
    ffmpeg-headless
    ffmpeg-full;

  ffmpegthumbnailer = callPackage ../development/libraries/ffmpegthumbnailer { };

  ffmpeg-normalize = python3Packages.callPackage ../applications/video/ffmpeg-normalize { };

  ffms = callPackage ../development/libraries/ffms { };

  fftw = callPackage ../development/libraries/fftw { };
  fftwSinglePrec = fftw.override { precision = "single"; };
  fftwFloat = fftwSinglePrec; # the configure option is just an alias
  fftwLongDouble = fftw.override { precision = "long-double"; };
  # Need gcc >= 4.6.0 to build with FFTW with quad precision, but Darwin defaults to Clang
  fftwQuad = fftw.override {
    precision = "quad-precision";
    stdenv = gccStdenv;
  };
  fftwMpi = fftw.override { enableMpi = true; };

  filter-audio = callPackage ../development/libraries/filter-audio { };

  filtron = callPackage ../servers/filtron { };

  fizz = darwin.apple_sdk_11_0.callPackage ../development/libraries/fizz { };

  flann = callPackage ../development/libraries/flann { };

  flatcc = callPackage ../development/libraries/flatcc { };

  flint = callPackage ../development/libraries/flint { };

  flint3 = callPackage ../development/libraries/flint/3.nix { };

  flite = callPackage ../development/libraries/flite { };

  fltk13 = callPackage ../development/libraries/fltk {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Carbon Cocoa OpenGL;
  };
  fltk14 = callPackage ../development/libraries/fltk/1.4.nix {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Carbon Cocoa OpenGL;
  };
  fltk13-minimal = fltk13.override {
    withGL = false;
    withCairo = false;
    withPango = false;
    withExamples = false;
    withDocs = false;
  };
  fltk14-minimal = fltk14.override {
    withGL = false;
    withCairo = false;
    withPango = false;
    withExamples = false;
    withDocs = false;
  };
  fltk = fltk13;
  fltk-minimal = fltk13-minimal;

  flyway = callPackage ../development/tools/flyway { };

  inherit (callPackages ../development/libraries/fmt { }) fmt_8 fmt_9 fmt_10;

  fmt = fmt_10;

  fplll = callPackage ../development/libraries/fplll { };
  fplll_20160331 = callPackage ../development/libraries/fplll/20160331.nix { };

  freeimage = callPackage ../development/libraries/freeimage {
    inherit (darwin) autoSignDarwinBinariesHook;
  };

  freeipa = callPackage ../os-specific/linux/freeipa {
    kerberos = krb5.override {
      withVerto = true;
    };
    sasl = cyrus_sasl;
    samba = samba4.override {
      enableLDAP = true;
    };
  };

  freetts = callPackage ../development/libraries/freetts { };

  frog = res.languageMachines.frog;

  fstrcmp = callPackage ../development/libraries/fstrcmp { };

  fstrm = callPackage ../development/libraries/fstrm { };

  cfitsio = callPackage ../development/libraries/science/astronomy/cfitsio { };

  fontconfig = callPackage ../development/libraries/fontconfig {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation;
  };

  folly = darwin.apple_sdk_11_0.callPackage ../development/libraries/folly { };

  folks = callPackage ../development/libraries/folks { };

  fortify-headers = callPackage ../development/libraries/fortify-headers { };

  makeFontsConf = callPackage ../development/libraries/fontconfig/make-fonts-conf.nix { };

  makeFontsCache = let fontconfig_ = fontconfig; in {fontconfig ? fontconfig_, fontDirectories}:
    callPackage ../development/libraries/fontconfig/make-fonts-cache.nix {
      inherit fontconfig fontDirectories;
    };

  f2c = callPackage ../development/tools/f2c { };

  freealut = callPackage ../development/libraries/freealut { };

  freeglut = callPackage ../development/libraries/freeglut { };

  freenect = callPackage ../development/libraries/freenect {
    inherit (darwin.apple_sdk.frameworks) Cocoa GLUT;
  };

  freetype = callPackage ../development/libraries/freetype { };

  freexl = callPackage ../development/libraries/freexl { };

  frei0r = callPackage ../development/libraries/frei0r { };

  fribidi = callPackage ../development/libraries/fribidi { };

  frozen = callPackage ../development/libraries/frozen { };

  funambol = callPackage ../development/libraries/funambol { };

  function-runner = callPackage ../development/web/function-runner { };

  functionalplus = callPackage ../development/libraries/functionalplus { };

  galer = callPackage ../tools/security/galer { };

  gallia = callPackage ../tools/security/gallia { };

  gamenetworkingsockets = callPackage ../development/libraries/gamenetworkingsockets {
    protobuf = protobuf_21;
  };

  game-music-emu = callPackage ../development/libraries/audio/game-music-emu { };

  gamin = callPackage ../development/libraries/gamin { };
  fam = gamin; # added 2018-04-25

  ganv = callPackage ../development/libraries/ganv { };

  garble = callPackage ../development/tools/garble { };

  gcab = callPackage ../development/libraries/gcab { };

  gcovr = with python3Packages; toPythonApplication gcovr;

  gcr = callPackage ../development/libraries/gcr { };

  gcr_4 = callPackage ../development/libraries/gcr/4.nix { };

  gdl = callPackage ../development/libraries/gdl { };

  gdome2 = callPackage ../development/libraries/gdome2 { };

  gdbm = callPackage ../development/libraries/gdbm { };

  gecode_3 = callPackage ../development/libraries/gecode/3.nix { };
  gecode_6 = qt5.callPackage ../development/libraries/gecode { };
  gecode = gecode_6;

  geph = recurseIntoAttrs (callPackages ../applications/networking/geph { });

  gephi = callPackage ../applications/science/misc/gephi { };

  gegl = callPackage ../development/libraries/gegl {
    inherit (darwin.apple_sdk.frameworks) OpenCL;
  };

  gensio = darwin.apple_sdk_11_0.callPackage ../development/libraries/gensio { };

  geoclue2 = callPackage ../development/libraries/geoclue { };

  geoclue2-with-demo-agent = geoclue2.override { withDemoAgent = true; };

  geocode-glib = callPackage ../development/libraries/geocode-glib { };

  geocode-glib_2 = geocode-glib.override {
    libsoup = libsoup_3;
  };

  geoipWithDatabase = makeOverridable (callPackage ../development/libraries/geoip) {
    drvName = "geoip-tools";
    geoipDatabase = geolite-legacy;
  };

  geogram = callPackage ../development/libraries/geogram { };

  geographiclib = callPackage ../development/libraries/geographiclib { };

  geoip = callPackage ../development/libraries/geoip { };

  geoipjava = callPackage ../development/libraries/java/geoipjava { };

  geomyidae = callPackage ../applications/networking/gopher/geomyidae { };

  geos = callPackage ../development/libraries/geos { };

  geos_3_9 = callPackage ../development/libraries/geos/3.9.nix { };

  geos_3_11 = callPackage ../development/libraries/geos/3.11.nix { };

  getdata = callPackage ../development/libraries/getdata { };

  inherit (callPackages ../development/libraries/getdns { })
    getdns stubby;

  gettext = callPackage ../development/libraries/gettext { };

  gf2x = callPackage ../development/libraries/gf2x { };

  gfxstream = callPackage ../development/libraries/gfxstream { };

  gd = callPackage ../development/libraries/gd {
    automake = automake115x;
  };

  gdal = callPackage ../development/libraries/gdal { };

  gdalMinimal = callPackage ../development/libraries/gdal {
    useMinimalFeatures = true;
  };

  gdcm = callPackage ../development/libraries/gdcm {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Cocoa;
  };

  ggz_base_libs = callPackage ../development/libraries/ggz_base_libs { };

  gifticlib = callPackage ../development/libraries/science/biology/gifticlib { };

  gio-sharp = callPackage ../development/libraries/gio-sharp { };

  givaro = callPackage ../development/libraries/givaro { };
  givaro_3 = callPackage ../development/libraries/givaro/3.nix { };
  givaro_3_7 = callPackage ../development/libraries/givaro/3.7.nix { };

  ghc_filesystem = callPackage ../development/libraries/ghc_filesystem { };

  ghp-import = with python3Packages; toPythonApplication ghp-import;

  ghcid = haskellPackages.ghcid.bin;

  gr-framework = callPackage ../by-name/gr/gr-framework/package.nix {
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  graphia = qt6Packages.callPackage ../applications/science/misc/graphia { };

  graphinder = callPackage ../tools/security/graphinder { };

  hnswlib = callPackage ../development/libraries/hnswlib { };

  httplib = callPackage ../development/libraries/httplib { };

  icon-lang = callPackage ../development/interpreters/icon-lang { };

  libgit2 = callPackage ../development/libraries/libgit2 {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  libgit2-glib = callPackage ../development/libraries/libgit2-glib { };

  libhsts = callPackage ../development/libraries/libhsts { };

  glbinding = callPackage ../development/libraries/glbinding { };

  gle = callPackage ../development/libraries/gle { };

  glew = callPackage ../development/libraries/glew {
    inherit (darwin.apple_sdk.frameworks) OpenGL;
  };
  glew110 = callPackage ../development/libraries/glew/1.10.nix {
    inherit (darwin.apple_sdk.frameworks) AGL OpenGL;
  };
  glew-egl = callPackage ../development/libraries/glew {
    inherit (darwin.apple_sdk.frameworks) OpenGL;
    enableEGL = true;
  };

  glfw = glfw3;
  glfw-wayland-minecraft = callPackage ../development/libraries/glfw/3.x-wayland-minecraft.nix {};
  glfw2 = callPackage ../development/libraries/glfw/2.x.nix { };
  glfw3 = callPackage ../development/libraries/glfw/3.x.nix {
    inherit (darwin.apple_sdk.frameworks) Carbon Cocoa Kernel OpenGL;
  };

  glibc = callPackage ../development/libraries/glibc {
    stdenv = gccStdenv; # doesn't compile without gcc
  };

  mtrace = callPackage ../development/libraries/glibc/mtrace.nix { };

  # Provided by libc on Operating Systems that use the Extensible Linker Format.
  elf-header = if stdenv.hostPlatform.isElf then null else elf-header-real;

  elf-header-real = callPackage ../development/libraries/elf-header { };

  glibc_memusage = callPackage ../development/libraries/glibc {
    withGd = true;
  };

  # Being redundant to avoid cycles on boot. TODO: find a better way
  glibcCross = callPackage ../development/libraries/glibc {
    stdenv = gccCrossLibcStdenv; # doesn't compile without gcc
    libgcc = callPackage ../development/libraries/gcc/libgcc {
      gcc = gccCrossLibcStdenv.cc;
      glibc = glibcCross.override { libgcc = null; };
      stdenvNoLibs = gccCrossLibcStdenv;
    };
  };

  muslCross = musl.override {
    stdenv = crossLibcStdenv;
  };

  # These are used when buiding compiler-rt / libgcc, prior to building libc.
  preLibcCrossHeaders = let
    inherit (stdenv.targetPlatform) libc;
  in     if stdenv.targetPlatform.isMinGW then targetPackages.windows.mingw_w64_headers or windows.mingw_w64_headers
    else if libc == "nblibc" then targetPackages.netbsdCross.headers or netbsdCross.headers
    else if libc == "libSystem" && stdenv.targetPlatform.isAarch64 then targetPackages.darwin.LibsystemCross or darwin.LibsystemCross
    else null;

  # We can choose:
  libcCrossChooser = name:
    # libc is hackily often used from the previous stage. This `or`
    # hack fixes the hack, *sigh*.
    /**/ if name == null then null
    else if name == "glibc" then targetPackages.glibcCross or glibcCross
    else if name == "bionic" then targetPackages.bionic or bionic
    else if name == "uclibc" then targetPackages.uclibcCross or uclibcCross
    else if name == "avrlibc" then targetPackages.avrlibcCross or avrlibcCross
    else if name == "newlib" && stdenv.targetPlatform.isMsp430 then targetPackages.msp430NewlibCross or msp430NewlibCross
    else if name == "newlib" && stdenv.targetPlatform.isVc4 then targetPackages.vc4-newlib or vc4-newlib
    else if name == "newlib" && stdenv.targetPlatform.isOr1k then targetPackages.or1k-newlib or or1k-newlib
    else if name == "newlib" then targetPackages.newlibCross or newlibCross
    else if name == "newlib-nano" then targetPackages.newlib-nanoCross or newlib-nanoCross
    else if name == "musl" then targetPackages.muslCross or muslCross
    else if name == "msvcrt" then targetPackages.windows.mingw_w64 or windows.mingw_w64
    else if name == "ucrt" then targetPackages.windows.mingw_w64 or windows.mingw_w64
    else if name == "libSystem" then
      if stdenv.targetPlatform.useiOSPrebuilt
      then targetPackages.darwin.iosSdkPkgs.libraries or darwin.iosSdkPkgs.libraries
      else targetPackages.darwin.LibsystemCross or (throw "don't yet have a `targetPackages.darwin.LibsystemCross for ${stdenv.targetPlatform.config}`")
    else if name == "fblibc" then targetPackages.freebsd.libc or freebsd.libc
    else if name == "oblibc" then targetPackages.openbsd.libc or openbsd.libc
    else if name == "nblibc" then targetPackages.netbsd.libc or netbsd.libc
    else if name == "wasilibc" then targetPackages.wasilibc or wasilibc
    else if name == "relibc" then targetPackages.relibc or relibc
    else throw "Unknown libc ${name}";

  libcCross = assert stdenv.targetPlatform != stdenv.buildPlatform; libcCrossChooser stdenv.targetPlatform.libc;

  threadsCross = threadsCrossFor null;
  threadsCrossFor = cc_version:
    lib.optionalAttrs (stdenv.targetPlatform.isMinGW && !(stdenv.targetPlatform.useLLVM or false)) {
      # other possible values: win32 or posix
      model = "mcf";
      # For win32 or posix set this to null
      package =
        if cc_version == null || lib.versionAtLeast cc_version "13"
        then targetPackages.windows.mcfgthreads or windows.mcfgthreads
        else targetPackages.windows.mcfgthreads_pre_gcc_13 or windows.mcfgthreads_pre_gcc_13;
    };

  wasilibc = callPackage ../development/libraries/wasilibc {
    stdenv = crossLibcStdenv;
  };

  relibc = callPackage ../development/libraries/relibc { };

  # Only supported on Linux and only on glibc
  glibcLocales =
    if stdenv.hostPlatform.isLinux && stdenv.hostPlatform.isGnu
    then callPackage ../development/libraries/glibc/locales.nix { }
    else null;
  glibcLocalesUtf8 =
    if stdenv.hostPlatform.isLinux && stdenv.hostPlatform.isGnu
    then callPackage ../development/libraries/glibc/locales.nix { allLocales = false; }
    else null;

  glibcInfo = callPackage ../development/libraries/glibc/info.nix { };

  glibc_multi = callPackage ../development/libraries/glibc/multi.nix {
    # The buildPackages is required for cross-compilation. The pkgsi686Linux set
    # has target and host always set to the same value based on target platform
    # of the current set. We need host to be same as build to correctly get i686
    # variant of glibc.
    glibc32 = pkgsi686Linux.buildPackages.glibc;
  };

  glm = callPackage ../development/libraries/glm { };

  glog = callPackage ../development/libraries/glog { };

  gloox = callPackage ../development/libraries/gloox { };

  glpk = callPackage ../development/libraries/glpk { };

  glsurf = callPackage ../applications/science/math/glsurf {
    ocamlPackages = ocaml-ng.ocamlPackages_4_14_unsafe_string;
  };

  glui = callPackage ../development/libraries/glui { };

  gmime2 = callPackage ../development/libraries/gmime/2.nix { };
  gmime3 = callPackage ../development/libraries/gmime/3.nix { };
  gmime = gmime2;

  gmm = callPackage ../development/libraries/gmm { };

  gmp4 = callPackage ../development/libraries/gmp/4.3.2.nix { }; # required by older GHC versions
  gmp5 = callPackage ../development/libraries/gmp/5.1.x.nix { };
  gmp6 = callPackage ../development/libraries/gmp/6.x.nix { };
  gmp = gmp6;
  gmpxx = gmp.override { cxx = true; };

  #GMP ex-satellite, so better keep it near gmp
  mpfr = callPackage ../development/libraries/mpfr { };

  mpfi = callPackage ../development/libraries/mpfi { };

  mpdecimal = callPackage ../development/libraries/mpdecimal { };

  mpfshell = callPackage ../development/tools/mpfshell { };

  # A GMP fork
  mpir = callPackage ../development/libraries/mpir { };

  gl3w = callPackage ../development/libraries/gl3w { };

  gns3Packages = dontRecurseIntoAttrs (callPackage ../applications/networking/gns3 { });
  gns3-gui = gns3Packages.guiStable;
  gns3-server = gns3Packages.serverStable;

  gobject-introspection = callPackage ../development/libraries/gobject-introspection/wrapper.nix { };

  gobject-introspection-unwrapped = callPackage ../development/libraries/gobject-introspection {
    nixStoreDir = config.nix.storeDir or builtins.storeDir;
    inherit (darwin) cctools;
  };

  goocanvas = callPackage ../development/libraries/goocanvas { };
  goocanvas2 = callPackage ../development/libraries/goocanvas/2.x.nix { };
  goocanvas3 = callPackage ../development/libraries/goocanvas/3.x.nix { };
  goocanvasmm2 = callPackage ../development/libraries/goocanvasmm { };

  gflags = callPackage ../development/libraries/gflags { };

  gperftools = callPackage ../development/libraries/gperftools { };

  grilo = callPackage ../development/libraries/grilo { };

  grilo-plugins = callPackage ../development/libraries/grilo-plugins { };

  grpc = darwin.apple_sdk_11_0.callPackage ../development/libraries/grpc {
    stdenv = if stdenv.isDarwin && stdenv.isx86_64
      then overrideSDK darwin.apple_sdk_11_0.stdenv { darwinMinVersion = "10.13"; }
      else stdenv;
  };

  gsettings-qt = libsForQt5.callPackage ../development/libraries/gsettings-qt { };

  gst_all_1 = recurseIntoAttrs(callPackage ../development/libraries/gstreamer {
    callPackage = newScope (gst_all_1 // { libav = pkgs.ffmpeg-headless; });
    inherit (darwin.apple_sdk.frameworks) AudioToolbox AVFoundation Cocoa CoreFoundation CoreMedia CoreServices CoreVideo DiskArbitration Foundation IOKit MediaToolbox OpenGL Security SystemConfiguration VideoToolbox;
  });

  gusb = callPackage ../development/libraries/gusb { };

  qxmpp = libsForQt5.callPackage ../development/libraries/qxmpp { };

  gnet = callPackage ../development/libraries/gnet { };

  gnu-config = callPackage ../development/libraries/gnu-config { };

  gnu-efi = if stdenv.hostPlatform.isEfi
              then callPackage ../development/libraries/gnu-efi { }
            else null;

  gnutls = callPackage ../development/libraries/gnutls {
    inherit (darwin.apple_sdk.frameworks) Security;
    util-linux = util-linuxMinimal; # break the cyclic dependency
    autoconf = buildPackages.autoconf269;
  };

  gpac = callPackage ../applications/video/gpac { };

  gpgme = callPackage ../development/libraries/gpgme { };

  pgpdump = callPackage ../tools/security/pgpdump { };

  pgpkeyserver-lite = callPackage ../servers/web-apps/pgpkeyserver-lite { };

  pgweb = callPackage ../development/tools/database/pgweb { };

  granted = callPackage ../tools/admin/granted { };

  grantlee = libsForQt5.callPackage ../development/libraries/grantlee { };

  gsasl = callPackage ../development/libraries/gsasl { };

  gsl = callPackage ../development/libraries/gsl { };

  gsl-lite = callPackage ../development/libraries/gsl-lite { };

  gsm = callPackage ../development/libraries/gsm { };

  gsoap = callPackage ../development/libraries/gsoap { };

  gsound = callPackage ../development/libraries/gsound { };

  gss = callPackage ../development/libraries/gss { };

  gtkimageview = callPackage ../development/libraries/gtkimageview { };

  glib = callPackage ../development/libraries/glib (let
    glib-untested = glib.overrideAttrs { doCheck = false; };
  in {
    # break dependency cycles
    # these things are only used for tests, they don't get into the closure
    shared-mime-info = shared-mime-info.override { glib = glib-untested; };
    desktop-file-utils = desktop-file-utils.override { glib = glib-untested; };
    dbus = dbus.override { enableSystemd = false; };
  });

  glibmm = callPackage ../development/libraries/glibmm { };

  glibmm_2_68 = callPackage ../development/libraries/glibmm/2.68.nix {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  glib-networking = callPackage ../development/libraries/glib-networking { };

  glib-testing = callPackage ../development/libraries/glib-testing { };

  glirc = haskell.lib.compose.justStaticExecutables haskellPackages.glirc;

  gom = callPackage ../development/libraries/gom { };

  ace = callPackage ../development/libraries/ace { };

  # Not moved to aliases while we decide if we should split the package again.
  atk = at-spi2-core;

  atkmm = callPackage ../development/libraries/atkmm { };

  atkmm_2_36 = callPackage ../development/libraries/atkmm/2.36.nix { };

  pixman = callPackage ../development/libraries/pixman { };

  cairo = callPackage ../development/libraries/cairo { };

  cairo-lang = callPackage ../development/compilers/cairo { };

  cairomm = callPackage ../development/libraries/cairomm { };

  cairomm_1_16 = callPackage ../development/libraries/cairomm/1.16.nix {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices;
  };

  pango = callPackage ../development/libraries/pango {
    harfbuzz = harfbuzz.override { withCoreText = stdenv.isDarwin; };
  };

  pangolin = callPackage ../development/libraries/pangolin {
    inherit (darwin.apple_sdk.frameworks) Carbon Cocoa;
  };

  pangomm = callPackage ../development/libraries/pangomm {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices;
  };

  pangomm_2_48 = callPackage ../development/libraries/pangomm/2.48.nix {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices;
  };

  pangomm_2_42 = callPackage ../development/libraries/pangomm/2.42.nix {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices;
  };

  gdk-pixbuf = callPackage ../development/libraries/gdk-pixbuf { };

  gdk-pixbuf-xlib = callPackage ../development/libraries/gdk-pixbuf/xlib.nix { };

  gnome-menus = callPackage ../development/libraries/gnome-menus { };

  gnote = callPackage ../applications/office/gnote { };

  gtk2 = callPackage ../development/libraries/gtk/2.x.nix {
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa;
  };

  gtk2-x11 = gtk2.override {
    cairo = cairo.override { x11Support = true; };
    pango = pango.override { cairo = cairo.override { x11Support = true; }; x11Support = true; };
    gdktarget = "x11";
  };

  gtkextra = callPackage ../development/libraries/gtkextra { };

  gtk3 = callPackage ../development/libraries/gtk/3.x.nix {
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa QuartzCore;
  };

  gtk4 = callPackage ../development/libraries/gtk/4.x.nix {
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa;
  };


  # On darwin gtk uses cocoa by default instead of x11.
  gtk3-x11 = gtk3.override {
    cairo = cairo.override { x11Support = true; };
    pango = pango.override { cairo = cairo.override { x11Support = true; }; x11Support = true; };
    x11Support = true;
  };

  gtkmm2 = callPackage ../development/libraries/gtkmm/2.x.nix { };
  gtkmm3 = callPackage ../development/libraries/gtkmm/3.x.nix { };
  gtkmm4 = callPackage ../development/libraries/gtkmm/4.x.nix { };

  gtk_engines = callPackage ../development/libraries/gtk-engines { };

  gtk-engine-bluecurve = callPackage ../development/libraries/gtk-engine-bluecurve { };

  gtk-engine-murrine = callPackage ../development/libraries/gtk-engine-murrine { };

  gtk-sharp-2_0 = callPackage ../development/libraries/gtk-sharp/2.0.nix { };

  gtk-sharp-3_0 = callPackage ../development/libraries/gtk-sharp/3.0.nix { };

  gtk-sharp-beans = callPackage ../development/libraries/gtk-sharp-beans { };

  gtk-mac-integration = callPackage ../development/libraries/gtk-mac-integration {
    gtk = gtk3;
  };

  gtk-mac-integration-gtk2 = gtk-mac-integration.override {
    gtk = gtk2;
  };

  gtk-mac-integration-gtk3 = gtk-mac-integration;

  gtk-mac-bundler = callPackage ../development/tools/gtk-mac-bundler { };

  gtksourceview = gtksourceview3;

  gtksourceview3 = callPackage ../development/libraries/gtksourceview/3.x.nix { };

  gtksourceview4 = callPackage ../development/libraries/gtksourceview/4.x.nix { };

  gtksourceview5 = callPackage ../development/libraries/gtksourceview/5.x.nix { };

  gtksourceviewmm = callPackage ../development/libraries/gtksourceviewmm { };

  gtksourceviewmm4 = callPackage ../development/libraries/gtksourceviewmm/4.x.nix { };

  gtkspell2 = callPackage ../development/libraries/gtkspell { };

  gtkspell3 = callPackage ../development/libraries/gtkspell/3.nix { };

  gtkspellmm = callPackage ../development/libraries/gtkspellmm { };

  gtk-layer-shell = callPackage ../development/libraries/gtk-layer-shell { };

  gtk4-layer-shell = callPackage ../development/libraries/gtk4-layer-shell { };

  gts = callPackage ../development/libraries/gts { };

  gumbo = callPackage ../development/libraries/gumbo { };

  gvfs = callPackage ../development/libraries/gvfs { };

  gvm-libs = callPackage ../development/libraries/gvm-libs { };

  gwenhywfar = callPackage ../development/libraries/aqbanking/gwenhywfar.nix { };

  hamlib = hamlib_3;
  hamlib_3 = callPackage ../development/libraries/hamlib { };
  hamlib_4 = callPackage ../development/libraries/hamlib/4.nix { };

  heimdal = callPackage ../development/libraries/kerberos/heimdal.nix {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security SystemConfiguration;
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  harfbuzz = callPackage ../development/libraries/harfbuzz {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices CoreText;
  };

  harfbuzzFull = harfbuzz.override {
    withCoreText = stdenv.isDarwin;
    withGraphite2 = true;
    withIcu = true;
  };

  hawknl = callPackage ../development/libraries/hawknl { };

  hax11 = callPackage ../development/libraries/hax11 { };

  haxor-news = callPackage ../applications/misc/haxor-news { };

  hdt = callPackage ../misc/hdt { };

  hfinger = callPackage ../tools/security/hfinger { };

  herqq = libsForQt5.callPackage ../development/libraries/herqq { };

  hidapi = callPackage ../development/libraries/hidapi {
    inherit (darwin.apple_sdk.frameworks) Cocoa IOKit;
  };

  highfive = callPackage ../development/libraries/highfive { };

  highfive-mpi = highfive.override { hdf5 = hdf5-mpi; };

  hiredis = callPackage ../development/libraries/hiredis { };

  hiredis-vip = callPackage ../development/libraries/hiredis-vip { };

  hivex = callPackage ../development/libraries/hivex {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  hmat-oss = callPackage ../development/libraries/hmat-oss { };

  hound = callPackage ../development/tools/misc/hound { };

  hpp-fcl = callPackage ../development/libraries/hpp-fcl { };

  hpx = callPackage ../development/libraries/hpx {
    boost = boost179;
    asio = asio.override { boost = boost179; };
  };

  hspell = callPackage ../development/libraries/hspell { };

  hspellDicts = callPackage ../development/libraries/hspell/dicts.nix { };

  hsqldb = callPackage ../development/libraries/java/hsqldb { };

  hstr = callPackage ../applications/misc/hstr { };

  hstsparser = callPackage ../tools/security/hstsparser { };

  htmlcxx = callPackage ../development/libraries/htmlcxx { };

  http-parser = callPackage ../development/libraries/http-parser { };

  httpref = callPackage ../development/tools/misc/httpref { };

  hubble = callPackage ../applications/networking/cluster/hubble { };

  hunspell = callPackage ../development/libraries/hunspell { };

  hunspellDicts = recurseIntoAttrs (callPackages ../development/libraries/hunspell/dictionaries.nix {});

  hunspellDictsChromium = recurseIntoAttrs (callPackages ../development/libraries/hunspell/dictionaries-chromium.nix {});

  hunspellWithDicts = dicts: callPackage ../development/libraries/hunspell/wrapper.nix { inherit dicts; };

  hwloc = callPackage ../development/libraries/hwloc { };

  hydra_unstable = callPackage ../development/tools/misc/hydra/unstable.nix { nix = nixVersions.nix_2_20; };

  hydra-cli = callPackage ../development/tools/misc/hydra-cli { };

  hydraAntLogger = callPackage ../development/libraries/java/hydra-ant-logger { };

  hydra-check = with python3.pkgs; toPythonApplication hydra-check;

  hyena = callPackage ../development/libraries/hyena { };

  hyperscan = callPackage ../development/libraries/hyperscan { };

  icu-versions = callPackages ../development/libraries/icu { };
  inherit (icu-versions)
    icu58
    icu60
    icu63
    icu64
    icu66
    icu67
    icu68
    icu69
    icu70
    icu71
    icu72
    icu73
    icu74
  ;

  icu = icu73;

  id3lib = callPackage ../development/libraries/id3lib { };

  idasen = with python3Packages; toPythonApplication idasen;

  ikill = callPackage ../tools/misc/ikill { };

  ilbc = callPackage ../development/libraries/ilbc { };

  ilmbase = callPackage ../development/libraries/ilmbase { };

  imgui = callPackage ../development/libraries/imgui { };

  imtui = callPackage ../development/libraries/imtui { };

  immer = callPackage ../development/libraries/immer { };

  imv = callPackage ../applications/graphics/imv { };

  iml = callPackage ../development/libraries/iml { };

  imlib2 = callPackage ../development/libraries/imlib2 { };
  imlib2Full = imlib2.override {
    # Compilation error on Darwin with librsvg. For more information see:
    # https://github.com/NixOS/nixpkgs/pull/166452#issuecomment-1090725613
    svgSupport = !stdenv.isDarwin;
    heifSupport = !stdenv.isDarwin;
    webpSupport = true;
    jxlSupport = true;
    psSupport = true;
  };
  imlib2-nox = imlib2.override {
    x11Support = false;
  };

  imlibsetroot = callPackage ../applications/graphics/imlibsetroot { libXinerama = xorg.libXinerama; } ;

  impy = callPackage ../development/libraries/impy { };

  ineffassign = callPackage ../development/tools/ineffassign { };

  ijs = callPackage ../development/libraries/ijs { };

  itktcl  = callPackage ../development/libraries/itktcl { };
  incrtcl = callPackage ../development/libraries/incrtcl { };

  indicator-application-gtk2 = callPackage ../development/libraries/indicator-application/gtk2.nix { };
  indicator-application-gtk3 = callPackage ../development/libraries/indicator-application/gtk3.nix { };

  indicator-sound-switcher = callPackage ../applications/audio/indicator-sound-switcher { };

  indilib = darwin.apple_sdk_11_0.callPackage ../development/libraries/science/astronomy/indilib { };
  indi-full = callPackage ../development/libraries/science/astronomy/indilib/indi-full.nix { };

  inih = callPackage ../development/libraries/inih { };

  iniparser = callPackage ../development/libraries/iniparser { };

  initool = callPackage ../development/tools/initool { };

  intel-gmmlib = callPackage ../development/libraries/intel-gmmlib { };

  intel-media-driver = callPackage ../development/libraries/intel-media-driver { };

  intltool = callPackage ../development/tools/misc/intltool { };

  ios-cross-compile = callPackage ../development/compilers/ios-cross-compile/9.2.nix { };

  ip2location-c = callPackage ../development/libraries/ip2location-c { };

  irrlicht = if !stdenv.isDarwin then
    callPackage ../development/libraries/irrlicht { }
  else callPackage ../development/libraries/irrlicht/mac.nix {
    inherit (darwin.apple_sdk.frameworks) Cocoa OpenGL IOKit;
  };

  irrlichtmt = callPackage ../development/libraries/irrlichtmt {
    inherit  (darwin.apple_sdk.frameworks) Cocoa Kernel;
  };

  isocodes = callPackage ../development/libraries/iso-codes { };

  iso-flags = callPackage ../data/icons/iso-flags { };

  isoimagewriter = libsForQt5.callPackage ../tools/misc/isoimagewriter {};

  isort = with python3Packages; toPythonApplication isort;

  ispc = callPackage ../development/compilers/ispc {
    xcode = darwin.xcode_14;
    llvmPackages = llvmPackages_15;
  };

  isso = callPackage ../servers/isso {
    nodejs = nodejs_20;
  };

  itk_5_2 = callPackage ../development/libraries/itk/5.2.x.nix {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  itk_5 = callPackage ../development/libraries/itk/5.x.nix {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  itk = itk_5;

  jama = callPackage ../development/libraries/jama { };

  jansson = callPackage ../development/libraries/jansson { };

  jarowinkler-cpp = callPackage ../development/libraries/jarowinkler-cpp { };

  jbig2dec = callPackage ../development/libraries/jbig2dec { };

  jbig2enc = callPackage ../development/libraries/jbig2enc { };

  jcal = callPackage ../development/libraries/jcal { };

  jbigkit = callPackage ../development/libraries/jbigkit { };

  jefferson = callPackage ../tools/filesystems/jefferson { };

  jemalloc = callPackage ../development/libraries/jemalloc { };

  rust-jemalloc-sys = callPackage ../development/libraries/jemalloc/rust.nix { };
  rust-jemalloc-sys-unprefixed = rust-jemalloc-sys.override { unprefixed = true; };

  jose = callPackage ../development/libraries/jose { };

  jpcre2 = callPackage ../development/libraries/jpcre2 { };

  jshon = callPackage ../development/tools/parsing/jshon { };

  json2hcl = callPackage ../development/tools/json2hcl { };

  json2tsv = callPackage ../development/tools/json2tsv { };

  json2yaml = haskell.lib.compose.justStaticExecutables haskellPackages.json2yaml;

  json-glib = callPackage ../development/libraries/json-glib { };

  json_c = callPackage ../development/libraries/json-c { };

  jsoncpp = callPackage ../development/libraries/jsoncpp { };

  json-fortran = callPackage ../development/libraries/json-fortran { };

  jsonnet = callPackage ../development/compilers/jsonnet { };

  jsonnet-bundler = callPackage ../development/tools/jsonnet-bundler { };

  jrsonnet = callPackage ../development/compilers/jrsonnet { };

  go-jsonnet = callPackage ../development/compilers/go-jsonnet { };

  jsonrpc-glib = callPackage ../development/libraries/jsonrpc-glib { };

  jxrlib = callPackage ../development/libraries/jxrlib { };

  libjson = callPackage ../development/libraries/libjson { };

  libjodycode = callPackage ../development/libraries/libjodycode {
    # missing aligned_alloc()
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  libb64 = callPackage ../development/libraries/libb64 { };

  judy = callPackage ../development/libraries/judy { };

  kcp = callPackage ../development/libraries/kcp { };

  kddockwidgets = libsForQt5.callPackage ../development/libraries/kddockwidgets { };

  keybinder = callPackage ../development/libraries/keybinder {
    automake = automake111x;
    lua = lua5_1;
  };

  keybinder3 = callPackage ../development/libraries/keybinder3 {
    gtk3 = if stdenv.isDarwin then gtk3-x11 else gtk3;
    automake = automake111x;
  };

  krb5 = callPackage ../development/libraries/kerberos/krb5.nix {
    inherit (buildPackages.darwin) bootstrap_cmds;
  };
  libkrb5 = krb5.override { type = "lib"; };

  kronosnet = callPackage ../development/libraries/kronosnet { };

  ktextaddons = libsForQt5.callPackage ../development/libraries/ktextaddons {};

  l-smash = callPackage ../development/libraries/l-smash {
    stdenv = gccStdenv;
  };

  languageMachines = recurseIntoAttrs (import ../development/libraries/languagemachines/packages.nix {
    inherit pkgs;
  });

  lasem = callPackage ../development/libraries/lasem { };

  lasso = callPackage ../development/libraries/lasso { };

  LAStools = callPackage ../development/libraries/LAStools { };

  LASzip = callPackage ../development/libraries/LASzip { };
  LASzip2 = callPackage ../development/libraries/LASzip/LASzip2.nix { };

  laurel = callPackage ../servers/monitoring/laurel/default.nix { };

  lcm = callPackage ../development/libraries/lcm { };

  lcms = lcms2;

  lcms1 = callPackage ../development/libraries/lcms { };

  lcms2 = callPackage ../development/libraries/lcms2 { };

  lcrq = callPackage ../development/libraries/lcrq { };

  ldacbt = callPackage ../development/libraries/ldacbt { };

  ldb = callPackage ../development/libraries/ldb { };

  lensfun = callPackage ../development/libraries/lensfun { };

  lesbar = callPackage ../applications/window-managers/lesbar { };

  lesstif = callPackage ../development/libraries/lesstif { };

  leveldb = callPackage ../development/libraries/leveldb { };

  lmdb = callPackage ../development/libraries/lmdb { };

  lmdbxx = callPackage ../development/libraries/lmdbxx { };

  lemon-graph = callPackage ../development/libraries/lemon-graph { };

  levmar = callPackage ../development/libraries/levmar { };

  leptonica = callPackage ../development/libraries/leptonica { };

  lerc = callPackage ../development/libraries/lerc { };

  lib2geom = callPackage ../development/libraries/lib2geom {
    stdenv = if stdenv.cc.isClang then llvmPackages_13.stdenv else stdenv;
  };

  lib3ds = callPackage ../development/libraries/lib3ds { };

  lib3mf = callPackage ../development/libraries/lib3mf { };

  libAfterImage = callPackage ../development/libraries/libAfterImage { };

  libaacs = callPackage ../development/libraries/libaacs { };

  libaal = callPackage ../development/libraries/libaal { };

  libabigail = callPackage ../development/libraries/libabigail { };

  libaccounts-glib = callPackage ../development/libraries/libaccounts-glib { };

  libacr38u = callPackage ../tools/security/libacr38u {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  libad9361 = callPackage ../development/libraries/libad9361 { };

  libadwaita = callPackage ../development/libraries/libadwaita {
    inherit (pkgs.darwin.apple_sdk.frameworks) AppKit Foundation;
  };

  libaec = callPackage ../development/libraries/libaec { };

  libagar = callPackage ../development/libraries/libagar { };
  libagar_test = callPackage ../development/libraries/libagar/libagar_test.nix { };

  libao = callPackage ../development/libraries/libao {
    usePulseAudio = config.pulseaudio or (lib.meta.availableOn stdenv.hostPlatform libpulseaudio);
    inherit (darwin.apple_sdk.frameworks) CoreAudio CoreServices AudioUnit;
  };

  libaosd = callPackage ../development/libraries/libaosd { };

  libabw = callPackage ../development/libraries/libabw { };

  libamqpcpp = callPackage ../development/libraries/libamqpcpp { };

  libantlr3c = callPackage ../development/libraries/libantlr3c { };

  libaom = callPackage ../development/libraries/libaom { };

  libappindicator-gtk2 = libappindicator.override { gtkVersion = "2"; };
  libappindicator-gtk3 = libappindicator.override { gtkVersion = "3"; };
  libappindicator = callPackage ../development/libraries/libappindicator { };

  libayatana-appindicator = callPackage ../development/libraries/libayatana-appindicator { };

  libargs = callPackage ../development/libraries/libargs { };

  libarchive = callPackage ../development/libraries/libarchive { };

  libarchive-qt = libsForQt5.callPackage ../development/libraries/libarchive-qt { };

  libaribcaption = callPackage ../by-name/li/libaribcaption/package.nix {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices CoreFoundation CoreGraphics CoreText;
  };

  libasn1c = callPackage ../servers/osmocom/libasn1c/default.nix { };

  libasr = callPackage ../development/libraries/libasr { };

  libass = callPackage ../development/libraries/libass { };

  libast = callPackage ../development/libraries/libast { };

  libassuan = callPackage ../development/libraries/libassuan { };

  libasyncns = callPackage ../development/libraries/libasyncns { };

  libatomic_ops = callPackage ../development/libraries/libatomic_ops { };

  libaudclient = callPackage ../development/libraries/libaudclient { };

  libaudec = callPackage ../development/libraries/libaudec { };

  libav = libav_11; # branch 11 is API-compatible with branch 10
  libav_all = callPackages ../development/libraries/libav { };
  inherit (libav_all) libav_0_8 libav_11 libav_12;

  libavc1394 = callPackage ../development/libraries/libavc1394 { };

  libavif = callPackage ../development/libraries/libavif { };

  libayatana-common = callPackage ../development/libraries/libayatana-common { };

  libb2 = callPackage ../development/libraries/libb2 { };

  libbacktrace = callPackage ../development/libraries/libbacktrace { };

  libbap = callPackage ../development/libraries/libbap {
    inherit (ocaml-ng.ocamlPackages_4_14) bap ocaml findlib ctypes ctypes-foreign;
  };

  libbaseencode = callPackage ../development/libraries/libbaseencode { };

  libbass = (callPackage ../development/libraries/audio/libbass { }).bass;
  libbass_fx = (callPackage ../development/libraries/audio/libbass { }).bass_fx;

  libbde = callPackage ../development/libraries/libbde { };

  libbdplus = callPackage ../development/libraries/libbdplus { };

  libblockdev = callPackage ../development/libraries/libblockdev { };

  libblocksruntime = callPackage ../development/libraries/libblocksruntime { };

  libbluray = callPackage ../development/libraries/libbluray {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration;
  };

  libbs2b = callPackage ../development/libraries/audio/libbs2b { };

  libbytesize = callPackage ../development/libraries/libbytesize { };

  libcaca = callPackage ../development/libraries/libcaca { };

  libcacard = callPackage ../development/libraries/libcacard { };

  libcamera-qcam = callPackage ../by-name/li/libcamera/package.nix { withQcam = true; };

  libcanberra = callPackage ../development/libraries/libcanberra {
    inherit (darwin.apple_sdk.frameworks) Carbon CoreServices AppKit;
  };
  libcanberra-gtk2 = pkgs.libcanberra.override {
    gtkSupport = "gtk2";
  };
  libcanberra-gtk3 = pkgs.libcanberra.override {
    gtkSupport = "gtk3";
  };

  libcanberra_kde = if (config.kde_runtime.libcanberraWithoutGTK or true)
    then pkgs.libcanberra
    else pkgs.libcanberra-gtk2;

  libcaption = callPackage ../development/libraries/libcaption { };

  libcbor = callPackage ../development/libraries/libcbor { };

  libccd = callPackage ../development/libraries/libccd { };

  libcec = callPackage ../development/libraries/libcec { };

  libcec_platform = callPackage ../development/libraries/libcec/platform.nix { };

  libcef = callPackage ../development/libraries/libcef { };

  libcello = callPackage ../development/libraries/libcello { };

  libcerf = callPackage ../development/libraries/libcerf { };

  libcdada = callPackage ../development/libraries/libcdada { };

  libcdaudio = callPackage ../development/libraries/libcdaudio { };

  libcddb = callPackage ../development/libraries/libcddb { };

  libcdio = callPackage ../development/libraries/libcdio {
    inherit (darwin.apple_sdk.frameworks) Carbon IOKit;
  };

  libcdio-paranoia = callPackage ../development/libraries/libcdio-paranoia {
    inherit (darwin.apple_sdk.frameworks) DiskArbitration IOKit;
  };

  libcdr = callPackage ../development/libraries/libcdr { lcms = lcms2; };

  libchamplain = callPackage ../development/libraries/libchamplain { };

  libchamplain_libsoup3 = libchamplain.override { withLibsoup3 = true; };

  libchardet = callPackage ../development/libraries/libchardet { };

  libchewing = callPackage ../development/libraries/libchewing { };

  libchipcard = callPackage ../development/libraries/aqbanking/libchipcard.nix { };

  libcrafter = callPackage ../development/libraries/libcrafter { };

  libcrossguid = callPackage ../development/libraries/libcrossguid { };

  libcs50 = callPackage ../development/libraries/libcs50 { };

  libuchardet = callPackage ../development/libraries/libuchardet { };

  libchop = callPackage ../development/libraries/libchop { };

  libcifpp = callPackage ../development/libraries/libcifpp { };

  libcint = callPackage ../development/libraries/libcint { };

  libcli = callPackage ../development/libraries/libcli { };

  libclthreads = callPackage ../development/libraries/libclthreads  { };

  libclxclient = callPackage ../development/libraries/libclxclient  { };

  libconfuse = callPackage ../development/libraries/libconfuse { };

  libcangjie = callPackage ../development/libraries/libcangjie { };

  libcollectdclient = callPackage ../development/libraries/libcollectdclient { };

  libcomps = callPackage ../tools/package-management/libcomps { python = python3; };

  libcpr = callPackage ../development/libraries/libcpr { };

  libcredis = callPackage ../development/libraries/libcredis { };

  libctb = callPackage ../development/libraries/libctb { };

  libctemplate = callPackage ../development/libraries/libctemplate { };

  libctl = callPackage ../development/libraries/libctl { };

  libcotp = callPackage ../development/libraries/libcotp { };

  libcouchbase = callPackage ../development/libraries/libcouchbase { };

  libcue = callPackage ../development/libraries/libcue { };

  libcutl = callPackage ../development/libraries/libcutl { };

  libcxxrt = callPackage ../development/libraries/libcxxrt {
    stdenv = if stdenv.hostPlatform.useLLVM or false
             then overrideCC stdenv buildPackages.llvmPackages.tools.clangNoLibcxx
             else stdenv;
  };

  libdaemon = callPackage ../development/libraries/libdaemon { };

  libdatovka = callPackage ../development/libraries/libdatovka { };

  libdatrie = callPackage ../development/libraries/libdatrie { };

  libdazzle = callPackage ../development/libraries/libdazzle { };

  libdbi = callPackage ../development/libraries/libdbi { };

  libdbiDriversBase = libdbiDrivers.override {
    libmysqlclient = null;
    sqlite = null;
  };

  libdbiDrivers = callPackage ../development/libraries/libdbi-drivers { };

  libunity = callPackage ../development/libraries/libunity { };

  libdbusmenu = callPackage ../development/libraries/libdbusmenu { };
  libdbusmenu-gtk2 = libdbusmenu.override { gtkVersion = "2"; };
  libdbusmenu-gtk3 = libdbusmenu.override { gtkVersion = "3"; };

  libdc1394 = callPackage ../development/libraries/libdc1394 {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  libde265 = callPackage ../development/libraries/libde265 { };

  libdeflate = darwin.apple_sdk_11_0.callPackage ../development/libraries/libdeflate { };

  libdeltachat = callPackage ../development/libraries/libdeltachat {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security SystemConfiguration;
  };

  libdevil = callPackage ../development/libraries/libdevil {
    inherit (darwin.apple_sdk.frameworks) OpenGL;
  };

  libdevil-nox = callPackage ../development/libraries/libdevil {
    inherit (darwin.apple_sdk.frameworks) OpenGL;
    withXorg = false;
  };

  libdecor = callPackage ../development/libraries/libdecor { };

  libdex = callPackage ../development/libraries/libdex { };

  libdicom = callPackage ../development/libraries/libdicom { };

  libdigidocpp = callPackage ../development/libraries/libdigidocpp { };

  libdiscid = callPackage ../development/libraries/libdiscid { };

  libdisplay-info = callPackage ../development/libraries/libdisplay-info { };

  libdivecomputer = callPackage ../development/libraries/libdivecomputer { };

  libdivsufsort = callPackage ../development/libraries/libdivsufsort { };

  libdmtx = callPackage ../development/libraries/libdmtx { };

  libdmapsharing = callPackage ../development/libraries/libdmapsharing { };

  libdnet = callPackage ../development/libraries/libdnet { };

  libdnf = callPackage ../tools/package-management/libdnf { python = python3; };

  libdovi = callPackage ../development/libraries/libdovi { };

  libdrm = callPackage ../development/libraries/libdrm { };

  libdv = callPackage ../development/libraries/libdv { };

  libdvbcsa = callPackage ../development/libraries/libdvbcsa { };

  libdvbpsi = callPackage ../development/libraries/libdvbpsi { };

  libdwg = callPackage ../development/libraries/libdwg { };

  libdvdcss = callPackage ../development/libraries/libdvdcss {
    inherit (darwin) IOKit;
  };

  libdvdnav = callPackage ../development/libraries/libdvdnav { };
  libdvdnav_4_2_1 = callPackage ../development/libraries/libdvdnav/4.2.1.nix {
    libdvdread = libdvdread_4_9_9;
  };

  libdvdread = callPackage ../development/libraries/libdvdread { };
  libdvdread_4_9_9 = callPackage ../development/libraries/libdvdread/4.9.9.nix { };

  libdwarf = callPackage ../development/libraries/libdwarf { };
  dwarfdump = libdwarf.bin;

  libe57format = callPackage ../development/libraries/libe57format { };

  libeatmydata = callPackage ../development/libraries/libeatmydata { };

  libeb = callPackage ../development/libraries/libeb { };

  libebml = callPackage ../development/libraries/libebml { };

  libebur128 = callPackage ../development/libraries/libebur128 { };

  libei = callPackage ../development/libraries/libei { };

  libelf = callPackage ../development/libraries/libelf { };

  libelfin = callPackage ../development/libraries/libelfin { };

  libetpan = callPackage ../development/libraries/libetpan { };

  libexecinfo = callPackage ../development/libraries/libexecinfo { };

  libfaketime = callPackage ../development/libraries/libfaketime { };

  libfakekey = callPackage ../development/libraries/libfakekey { };

  libfido2 = callPackage ../development/libraries/libfido2 {};

  libfilezilla = darwin.apple_sdk_11_0.callPackage ../development/libraries/libfilezilla {
    inherit (darwin.apple_sdk_11_0.frameworks) ApplicationServices;
  };

  libfishsound = callPackage ../development/libraries/libfishsound { };

  libfm = callPackage ../development/libraries/libfm { };
  libfm-extra = libfm.override {
    extraOnly = true;
  };

  libfprint = callPackage ../development/libraries/libfprint { };

  libfprint-tod = callPackage ../development/libraries/libfprint-tod { };

  libfprint-2-tod1-goodix = callPackage ../development/libraries/libfprint-2-tod1-goodix { };

  libfprint-2-tod1-goodix-550a = callPackage ../development/libraries/libfprint-2-tod1-goodix-550a { };

  libfprint-2-tod1-vfs0090 = callPackage ../development/libraries/libfprint-2-tod1-vfs0090 { };

  libfprint-2-tod1-elan = callPackage ../development/libraries/libfprint-2-tod1-elan { };

  libfpx = callPackage ../development/libraries/libfpx { };

  libgadu = callPackage ../development/libraries/libgadu { };

  libgbinder = callPackage ../development/libraries/libgbinder { };

  libgda = callPackage ../development/libraries/libgda { };

  libgda6 = callPackage ../development/libraries/libgda/6.x.nix { };

  libgdamm = callPackage ../development/libraries/libgdamm { };

  libgdata = callPackage ../development/libraries/libgdata { };

  libgee = callPackage ../development/libraries/libgee { };

  libgepub = callPackage ../development/libraries/libgepub { };

  libgig = callPackage ../development/libraries/libgig { };

  libglibutil = callPackage ../development/libraries/libglibutil { };

  libgnome-keyring = callPackage ../development/libraries/libgnome-keyring { };
  libgnome-keyring3 = gnome.libgnome-keyring;

  libgnome-games-support = callPackage ../development/libraries/libgnome-games-support { };
  libgnome-games-support_2_0 = callPackage ../development/libraries/libgnome-games-support/2.0.nix { };

  libgnomekbd = callPackage ../development/libraries/libgnomekbd { };

  libglvnd = callPackage ../development/libraries/libglvnd { };

  libgnurl = callPackage ../development/libraries/libgnurl { };

  libgourou = callPackage ../development/libraries/libgourou { };

  libgringotts = callPackage ../development/libraries/libgringotts { };

  libgrss = callPackage ../development/libraries/libgrss {
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation AppKit;
  };

  libgweather = callPackage ../development/libraries/libgweather { };

  libgxps = callPackage ../development/libraries/libgxps { };

  libiio = callPackage ../development/libraries/libiio {
    inherit (darwin.apple_sdk.frameworks) CFNetwork CoreServices;
    python = python3;
  };

  libinjection = callPackage ../development/libraries/libinjection { };

  libinklevel = callPackage ../development/libraries/libinklevel { };

  libkcapi = callPackage ../development/libraries/libkcapi { };

  libnats-c = callPackage ../development/libraries/libnats-c { };

  liburing = callPackage ../development/libraries/liburing { };

  librseq = callPackage ../development/libraries/librseq { };

  libseccomp = callPackage ../development/libraries/libseccomp { };

  libsecret = callPackage ../development/libraries/libsecret { };

  libsegfault = callPackage ../development/libraries/libsegfault { };

  libserdes = callPackage ../development/libraries/libserdes { };

  libserialport = callPackage ../development/libraries/libserialport { };

  libsignal-protocol-c = callPackage ../development/libraries/libsignal-protocol-c { };

  libsignon-glib = callPackage ../development/libraries/libsignon-glib { };

  libsoundio = callPackage ../development/libraries/libsoundio {
    inherit (darwin.apple_sdk.frameworks) AudioUnit;
  };

  libspelling = callPackage ../development/libraries/libspelling { };

  libsystemtap = callPackage ../development/libraries/libsystemtap { };

  libgtop = callPackage ../development/libraries/libgtop { };

  libLAS = callPackage ../development/libraries/libLAS { };

  liblaxjson = callPackage ../development/libraries/liblaxjson { };

  liblo = callPackage ../development/libraries/liblo { };

  liblscp = callPackage ../development/libraries/liblscp { };

  libe-book = callPackage ../development/libraries/libe-book { };

  libemf2svg = callPackage ../development/libraries/libemf2svg { };

  libev = callPackage ../development/libraries/libev { };

  libevent = callPackage ../development/libraries/libevent { };

  libewf = callPackage ../development/libraries/libewf { };

  libexif = callPackage ../development/libraries/libexif { };

  libexosip = callPackage ../development/libraries/exosip { };

  libexsid = callPackage ../development/libraries/libexsid { };

  libextractor = callPackage ../development/libraries/libextractor {
    libmpeg2 = mpeg2dec;
  };

  libexttextcat = callPackage ../development/libraries/libexttextcat { };

  libf2c = callPackage ../development/libraries/libf2c { };

  libfabric = callPackage ../development/libraries/libfabric { };

  libfive = qt6Packages.callPackage ../development/libraries/libfive {
    python = python3;
  };

  libfixposix = callPackage ../development/libraries/libfixposix { };

  libff = callPackage ../development/libraries/libff { };

  libffcall = callPackage ../development/libraries/libffcall { };

  libffi = callPackage ../development/libraries/libffi { };
  libffi_3_3 = callPackage ../development/libraries/libffi/3.3.nix { };
  libffiBoot = libffi.override {
    doCheck = false;
  };

  libfreeaptx = callPackage ../development/libraries/libfreeaptx { };

  libfreefare = callPackage ../development/libraries/libfreefare {
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
    inherit (darwin) libobjc;
  };

  libftdi = callPackage ../development/libraries/libftdi {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  libftdi1 = callPackage ../development/libraries/libftdi/1.x.nix { };

  libfyaml = callPackage ../development/libraries/libfyaml { };

  libgcrypt = callPackage ../development/libraries/libgcrypt { };

  libgcrypt_1_8 = callPackage ../development/libraries/libgcrypt/1.8.nix { };

  libgdiplus = callPackage ../development/libraries/libgdiplus {
      inherit (darwin.apple_sdk.frameworks) Carbon;
  };

  libgnt = callPackage ../development/libraries/libgnt { };

  libgpg-error = callPackage ../development/libraries/libgpg-error { };

  # https://git.gnupg.org/cgi-bin/gitweb.cgi?p=libgpg-error.git;a=blob;f=README;h=fd6e1a83f55696c1f7a08f6dfca08b2d6b7617ec;hb=70058cd9f944d620764e57c838209afae8a58c78#l118
  libgpg-error-gen-posix-lock-obj = libgpg-error.override {
    genPosixLockObjOnly = true;
  };

  libgphoto2 = callPackage ../development/libraries/libgphoto2 { };

  libgpiod = callPackage ../development/libraries/libgpiod { };

  libgpod = callPackage ../development/libraries/libgpod {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  libgssglue = callPackage ../development/libraries/libgssglue { };

  libgudev = callPackage ../development/libraries/libgudev { };

  libguestfs-appliance = callPackage ../development/libraries/libguestfs/appliance.nix { };
  libguestfs = callPackage ../development/libraries/libguestfs {
    autoreconfHook = buildPackages.autoreconfHook264;
    ocamlPackages = ocaml-ng.ocamlPackages_4_14;
  };
  libguestfs-with-appliance = libguestfs.override {
    appliance = libguestfs-appliance;
    autoreconfHook = buildPackages.autoreconfHook264;
  };


  libhangul = callPackage ../development/libraries/libhangul { };

  libharu = callPackage ../development/libraries/libharu { };

  libhdhomerun = callPackage ../development/libraries/libhdhomerun { };

  libheif = callPackage ../development/libraries/libheif { };

  libhttpseverywhere = callPackage ../development/libraries/libhttpseverywhere { };

  libhugetlbfs = callPackage ../development/libraries/libhugetlbfs { };

  libhv = callPackage ../development/libraries/libhv {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  libhwy = callPackage ../development/libraries/libhwy { };

  libHX = callPackage ../development/libraries/libHX { };

  libibmad = callPackage ../development/libraries/libibmad { };

  libibumad = callPackage ../development/libraries/libibumad { };

  libical = callPackage ../development/libraries/libical { };

  libicns = callPackage ../development/libraries/libicns { };

  libieee1284 = callPackage ../development/libraries/libieee1284 { };

  libimobiledevice = callPackage ../development/libraries/libimobiledevice {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration CoreFoundation;
  };

  libimobiledevice-glue = callPackage ../development/libraries/libimobiledevice-glue { };

  libindicator-gtk2 = libindicator.override { gtkVersion = "2"; };
  libindicator-gtk3 = libindicator.override { gtkVersion = "3"; };
  libindicator = callPackage ../development/libraries/libindicator { };

  libayatana-indicator = callPackage ../development/libraries/libayatana-indicator { };

  libinotify-kqueue = callPackage ../development/libraries/libinotify-kqueue { };

  libiodbc = callPackage ../development/libraries/libiodbc {
    inherit (darwin.apple_sdk.frameworks) Carbon;
  };

  libirecovery = callPackage ../development/libraries/libirecovery { };

  libivykis = callPackage ../development/libraries/libivykis { };

  libkqueue = callPackage ../development/libraries/libkqueue { };

  liblastfmSF = callPackage ../development/libraries/liblastfmSF { };

  liblcf = callPackage ../development/libraries/liblcf { };

  liblc3 = callPackage ../development/libraries/liblc3 { };

  libliftoff = callPackage ../development/libraries/libliftoff { };

  liblqr1 = callPackage ../development/libraries/liblqr-1 {
    inherit (darwin.apple_sdk.frameworks) Carbon AppKit;
  };

  liblockfile = callPackage ../development/libraries/liblockfile { };

  liblogging = callPackage ../development/libraries/liblogging { };

  liblognorm = callPackage ../development/libraries/liblognorm { };

  libltc = callPackage ../development/libraries/libltc { };

  liblxi = callPackage ../development/libraries/liblxi { };

  libmaxminddb = callPackage ../development/libraries/libmaxminddb { };

  libmcfp = callPackage ../development/libraries/libmcfp { };

  libmcrypt = callPackage ../development/libraries/libmcrypt { };

  libmediaart = callPackage ../development/libraries/libmediaart { };

  libmhash = callPackage ../development/libraries/libmhash { };

  libmodbus = callPackage ../development/libraries/libmodbus { };

  libmtp = callPackage ../development/libraries/libmtp { };

  libmypaint = callPackage ../development/libraries/libmypaint { };

  libmysofa = callPackage ../development/libraries/audio/libmysofa { };

  libmysqlconnectorcpp = callPackage ../development/libraries/libmysqlconnectorcpp { };

  libnatpmp = callPackage ../development/libraries/libnatpmp { };

  libnatspec = callPackage ../development/libraries/libnatspec { };

  libndp = callPackage ../development/libraries/libndp { };

  libnfc = callPackage ../development/libraries/libnfc { };

  libnfs = callPackage ../development/libraries/libnfs { };

  libnice = callPackage ../development/libraries/libnice { };

  libnitrokey = callPackage ../development/libraries/libnitrokey { };

  libnsl = callPackage ../development/libraries/libnsl { };

  liboping = callPackage ../development/libraries/liboping { };

  libplist = callPackage ../development/libraries/libplist { };

  libqtdbusmock = libsForQt5.callPackage ../development/libraries/libqtdbusmock {
    inherit (lomiri) cmake-extras;
  };

  libqtdbustest = libsForQt5.callPackage ../development/libraries/libqtdbustest {
    inherit (lomiri) cmake-extras;
  };

  libre = callPackage ../development/libraries/libre {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration;
  };

  librecast = callPackage ../development/libraries/librecast { };

  libredwg = callPackage ../development/libraries/libredwg { };

  librem = callPackage ../development/libraries/librem { };

  libremidi = callPackage ../development/libraries/libremidi {
      inherit (darwin.apple_sdk.frameworks)
        CoreAudio
        CoreFoundation
        CoreMIDI
        CoreServices;
  };

  libremines = qt6.callPackage ../games/libremines { };

  librelp = callPackage ../development/libraries/librelp { };

  librepo = callPackage ../tools/package-management/librepo {
    python = python3;
  };

  libresample = callPackage ../development/libraries/libresample { };

  librevenge = callPackage ../development/libraries/librevenge { };

  librime = callPackage ../development/libraries/librime { };

  librsb = callPackage ../development/libraries/librsb {
    # Taken from https://build.opensuse.org/package/view_file/science/librsb/librsb.spec
    memHierarchy = "L3:16/64/8192K,L2:16/64/2048K,L1:8/64/16K";
  };

  librtprocess = callPackage ../development/libraries/librtprocess { };

  libsamplerate = callPackage ../development/libraries/libsamplerate {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Carbon CoreServices;
  };

  libsieve = callPackage ../development/libraries/libsieve { };

  libsixel = callPackage ../development/libraries/libsixel { };

  libsolv = callPackage ../development/libraries/libsolv { };

  libspectre = callPackage ../development/libraries/libspectre { };

  libspecbleach = callPackage ../development/libraries/audio/libspecbleach { };

  libspnav = callPackage ../development/libraries/libspnav { };

  libgsf = callPackage ../development/libraries/libgsf { };

  # GNU libc provides libiconv so systems with glibc don't need to
  # build libiconv separately. Additionally, Apple forked/repackaged
  # libiconv, so build and use the upstream one with a compatible ABI,
  # and BSDs include libiconv in libc.
  #
  # We also provide `libiconvReal`, which will always be a standalone libiconv,
  # just in case you want it regardless of platform.
  libiconv =
    if lib.elem stdenv.hostPlatform.libc [ "glibc" "musl" "nblibc" "wasilibc" "fblibc" ]
      then libcIconv (if stdenv.hostPlatform != stdenv.buildPlatform
        then libcCross
        else stdenv.cc.libc)
    else if stdenv.hostPlatform.isDarwin
      then libiconv-darwin
    else libiconvReal;

  libcIconv = libc: let
    inherit (libc) pname version;
    libcDev = lib.getDev libc;
  in runCommand "${pname}-iconv-${version}" { strictDeps = true; } ''
    mkdir -p $out/include
    ln -sv ${libcDev}/include/iconv.h $out/include
  '';

  libiconvReal = callPackage ../development/libraries/libiconv { };

  iconv =
    if lib.elem stdenv.hostPlatform.libc [ "glibc" "musl" ] then
      lib.getBin stdenv.cc.libc
    else if stdenv.hostPlatform.isDarwin then
      lib.getBin libiconv
    else if stdenv.hostPlatform.isFreeBSD then
      lib.getBin freebsd.iconv
    else
      lib.getBin libiconvReal;

  # On non-GNU systems we need GNU Gettext for libintl.
  libintl = if stdenv.hostPlatform.libc != "glibc" then gettext else null;

  libid3tag = callPackage ../development/libraries/libid3tag { };

  libidn = callPackage ../development/libraries/libidn { };

  libidn2 = callPackage ../development/libraries/libidn2 { };

  idnkit = callPackage ../development/libraries/idnkit { };

  libiec61883 = callPackage ../development/libraries/libiec61883 { };

  libimagequant = callPackage ../development/libraries/libimagequant { };

  libime = callPackage ../development/libraries/libime { };

  libinfinity = callPackage ../development/libraries/libinfinity { };

  libinput = callPackage ../development/libraries/libinput {
    graphviz = graphviz-nox;
  };

  libinput-gestures = callPackage ../tools/inputmethods/libinput-gestures { };

  libinstpatch = callPackage ../development/libraries/audio/libinstpatch { };

  libipt = callPackage ../development/libraries/libipt { };

  libiptcdata = callPackage ../development/libraries/libiptcdata { };

  libjcat = callPackage ../development/libraries/libjcat { };

  libjpeg_original = callPackage ../development/libraries/libjpeg { };
  # also known as libturbojpeg
  libjpeg_turbo = callPackage ../development/libraries/libjpeg-turbo { };
  libjpeg = libjpeg_turbo;
  libjpeg8 = libjpeg_turbo.override { enableJpeg8 = true; };

  libjson-rpc-cpp = callPackage ../development/libraries/libjson-rpc-cpp { };

  libjwt = callPackage ../development/libraries/libjwt { };

  libjxl = callPackage ../development/libraries/libjxl { };

  libkate = callPackage ../development/libraries/libkate { };

  libkeyfinder = callPackage ../development/libraries/audio/libkeyfinder { };

  libks = callPackage ../development/libraries/libks { };

  libksba = callPackage ../development/libraries/libksba { };

  libksi = callPackage ../development/libraries/libksi { };

  liblinear = callPackage ../development/libraries/liblinear { };

  libmad = callPackage ../development/libraries/libmad { };

  malcontent = callPackage ../development/libraries/malcontent { };

  malcontent-ui = callPackage ../development/libraries/malcontent/ui.nix { };

  libmanette = callPackage ../development/libraries/libmanette { };

  libmatchbox = callPackage ../development/libraries/libmatchbox { };

  libmatheval = callPackage ../development/libraries/libmatheval { };

  libmatthew_java = callPackage ../development/libraries/java/libmatthew-java {
    jdk = jdk8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  libmatroska = callPackage ../development/libraries/libmatroska { };

  libmd = callPackage ../development/libraries/libmd { };

  libmemcached = callPackage ../development/libraries/libmemcached { };

  inherit
    ({
      libmicrohttpd_0_9_77 = callPackage ../development/libraries/libmicrohttpd/0.9.77.nix { };
      libmicrohttpd_1_0 = callPackage ../development/libraries/libmicrohttpd/1.0.nix { };
    })
    libmicrohttpd_0_9_77
    libmicrohttpd_1_0
    ;

  libmicrohttpd = libmicrohttpd_0_9_77;

  libmikmod = callPackage ../development/libraries/libmikmod {
    inherit (darwin.apple_sdk.frameworks) CoreAudio;
  };

  libmilter = callPackage ../development/libraries/libmilter { };

  libminc = callPackage ../development/libraries/libminc { };

  libmkv = callPackage ../development/libraries/libmkv { };

  libnut = callPackage ../development/libraries/libnut { };

  libmms = callPackage ../development/libraries/libmms { };

  libmowgli = callPackage ../development/libraries/libmowgli { };

  libmng = callPackage ../development/libraries/libmng { };

  libmnl = callPackage ../development/libraries/libmnl { };

  libmodplug = callPackage ../development/libraries/libmodplug { };

  libmodule = callPackage ../development/libraries/libmodule { };

  libmpcdec = callPackage ../development/libraries/libmpcdec { };

  libmp3splt = callPackage ../development/libraries/libmp3splt { };

  libmrss = callPackage ../development/libraries/libmrss { };

  libmspack = callPackage ../development/libraries/libmspack { };

  libmusicbrainz3 = callPackage ../development/libraries/libmusicbrainz { };

  libmusicbrainz5 = callPackage ../development/libraries/libmusicbrainz/5.x.nix { };

  libmusicbrainz = libmusicbrainz3;

  libmwaw = callPackage ../development/libraries/libmwaw { };

  libmx = callPackage ../development/libraries/libmx { };

  libndctl = callPackage ../development/libraries/libndctl { };

  libnest2d = callPackage ../development/libraries/libnest2d { };

  libnet = callPackage ../development/libraries/libnet { };

  libnetfilter_acct = callPackage ../development/libraries/libnetfilter_acct { };

  libnetfilter_conntrack = callPackage ../development/libraries/libnetfilter_conntrack { };

  libnetfilter_cthelper = callPackage ../development/libraries/libnetfilter_cthelper { };

  libnetfilter_cttimeout = callPackage ../development/libraries/libnetfilter_cttimeout { };

  libnetfilter_log = callPackage ../development/libraries/libnetfilter_log { };

  libnetfilter_queue = callPackage ../development/libraries/libnetfilter_queue { };

  libnfnetlink = callPackage ../development/libraries/libnfnetlink { };

  libnftnl = callPackage ../development/libraries/libnftnl { };

  libnova = callPackage ../development/libraries/science/astronomy/libnova { };

  libnxml = callPackage ../development/libraries/libnxml { };

  libodfgen = callPackage ../development/libraries/libodfgen { };

  libofa = callPackage ../development/libraries/libofa { };

  libofx = callPackage ../development/libraries/libofx { };

  libogg = callPackage ../development/libraries/libogg { };

  liboggz = callPackage ../development/libraries/liboggz { };

  liboil = callPackage ../development/libraries/liboil { };

  libomxil-bellagio = callPackage ../development/libraries/libomxil-bellagio { };

  liboop = callPackage ../development/libraries/liboop { };

  libopenaptx = callPackage ../development/libraries/libopenaptx { };

  libopenglrecorder = callPackage ../development/libraries/libopenglrecorder { };

  libopus = callPackage ../development/libraries/libopus { };

  libopusenc = callPackage ../development/libraries/libopusenc { };

  liboqs = callPackage ../development/libraries/liboqs { };

  libosinfo = callPackage ../development/libraries/libosinfo { };

  libosip = callPackage ../development/libraries/osip { };

  libosmium = callPackage ../development/libraries/libosmium { };

  libosmoabis = callPackage ../servers/osmocom/libosmoabis { };

  libosmocore = callPackage ../servers/osmocom/libosmocore { };

  libosmo-netif = callPackage ../servers/osmocom/libosmo-netif { };

  libosmo-sccp = callPackage ../servers/osmocom/libosmo-sccp { };

  libosmscout = libsForQt5.callPackage ../development/libraries/libosmscout { };

  libotr = callPackage ../development/libraries/libotr { };

  libow = callPackage ../development/libraries/libow { };

  libp11 = callPackage ../development/libraries/libp11 { };

  libpam-wrapper = callPackage ../development/libraries/libpam-wrapper { };

  libpanel = callPackage ../development/libraries/libpanel { };

  libpar2 = callPackage ../development/libraries/libpar2 { };

  libpcap = callPackage ../development/libraries/libpcap { };

  libpeas = callPackage ../development/libraries/libpeas { };
  libpeas2 = callPackage ../development/libraries/libpeas/2.x.nix { };

  libpg_query = callPackage ../development/libraries/libpg_query { };

  libpipeline = callPackage ../development/libraries/libpipeline { };

  libpgf = callPackage ../development/libraries/libpgf { };

  libphonenumber = callPackage ../development/libraries/libphonenumber {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  libplacebo = callPackage ../development/libraries/libplacebo { };

  libpng = callPackage ../development/libraries/libpng { };
  libpng12 = callPackage ../development/libraries/libpng/12.nix { };

  libpostal = callPackage ../development/libraries/libpostal { };

  libpaper = callPackage ../development/libraries/libpaper { };

  libpfm = callPackage ../development/libraries/libpfm { };

  libpqxx = callPackage ../development/libraries/libpqxx { };
  libpqxx_6 = callPackage ../development/libraries/libpqxx/6.nix { };

  inherit (callPackages ../development/libraries/prometheus-client-c {
    stdenv = gccStdenv; # Required for darwin
  }) libprom libpromhttp;

  libproxy = callPackage ../development/libraries/libproxy { };

  libpseudo = callPackage ../development/libraries/libpseudo { };

  libpsl = callPackage ../development/libraries/libpsl { };

  libpst = callPackage ../development/libraries/libpst { };

  libptytty = callPackage ../development/libraries/libptytty { };

  libpulsar = callPackage ../development/libraries/libpulsar {
    protobuf = protobuf_21;
  };

  libpwquality = callPackage ../development/libraries/libpwquality {
    python = python3;
  };

  libqalculate = callPackage ../development/libraries/libqalculate { };

  libqt5pas = libsForQt5.callPackage ../development/compilers/fpc/libqt5pas.nix { };

  librclone = callPackage ../development/libraries/librclone { };

  libroxml = callPackage ../development/libraries/libroxml { };

  librsvg = callPackage ../development/libraries/librsvg {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) ApplicationServices Foundation;
  };

  librsync = callPackage ../development/libraries/librsync { };

  librttopo = callPackage ../development/libraries/librttopo { };

  libs3 = callPackage ../development/libraries/libs3 { };

  libschrift = callPackage ../development/libraries/libschrift { };

  libsciter = callPackage ../development/libraries/libsciter { };

  libsearpc = callPackage ../development/libraries/libsearpc { };

  libsigcxx = callPackage ../development/libraries/libsigcxx { };

  libsigcxx30 = callPackage ../development/libraries/libsigcxx/3.0.nix { };

  libsigsegv = callPackage ../development/libraries/libsigsegv { };

  libslirp = callPackage ../development/libraries/libslirp { };

  libsndfile = callPackage ../development/libraries/libsndfile {
    inherit (darwin.apple_sdk.frameworks) Carbon AudioToolbox;
  };

  libsnark = callPackage ../development/libraries/libsnark { };

  libsodium = callPackage ../development/libraries/libsodium { };

  libsoup = callPackage ../development/libraries/libsoup { };

  libsoup_3 = callPackage ../development/libraries/libsoup/3.x.nix { };

  libspectrum = callPackage ../development/libraries/libspectrum { };

  libspiro = callPackage ../development/libraries/libspiro { };

  libspng = callPackage ../development/libraries/libspng { };

  libssh = callPackage ../development/libraries/libssh { };

  libssh2 = callPackage ../development/libraries/libssh2 { };

  libstartup_notification = callPackage ../development/libraries/startup-notification { };

  libstemmer = callPackage ../development/libraries/libstemmer { };

  libstroke = callPackage ../development/libraries/libstroke { };

  libspatialaudio = callPackage ../development/libraries/libspatialaudio { };

  libspatialindex = callPackage ../development/libraries/libspatialindex { };

  libspatialite = callPackage ../development/libraries/libspatialite { };

  libstatgrab = callPackage ../development/libraries/libstatgrab {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  libsvm = callPackage ../development/libraries/libsvm { };

  libtar = callPackage ../development/libraries/libtar { };

  libtasn1 = callPackage ../development/libraries/libtasn1 { };

  libtcod = callPackage ../development/libraries/libtcod { };

  libthai = callPackage ../development/libraries/libthai { };

  libtheora = callPackage ../development/libraries/libtheora { };

  libthreadar = callPackage ../development/libraries/libthreadar { };

  libticables2 = callPackage ../development/libraries/libticables2 { };

  libticalcs2 = callPackage ../development/libraries/libticalcs2 {
    inherit (darwin) libobjc;
  };

  libticonv = callPackage ../development/libraries/libticonv { };

  libtifiles2 = callPackage ../development/libraries/libtifiles2 { };

  inherit
    ({
      libtiff = callPackage ../development/libraries/libtiff { };
      libtiff_t = callPackage ../development/libraries/libtiff/libtiff_t.nix { };
    })
    libtiff
    libtiff_t
    ;

  libtiger = callPackage ../development/libraries/libtiger { };

  libtommath = callPackage ../development/libraries/libtommath { };

  libtomcrypt = callPackage ../development/libraries/libtomcrypt { };

  libtorrent-rasterbar-2_0_x = callPackage ../development/libraries/libtorrent-rasterbar {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration;
    python = python3;
  };

  libtorrent-rasterbar-1_2_x = callPackage ../development/libraries/libtorrent-rasterbar/1.2.nix {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration;
    python = python3;
  };

  libtorrent-rasterbar = libtorrent-rasterbar-2_0_x;

  libtpms = callPackage ../tools/security/libtpms { };

  libtap = callPackage ../development/libraries/libtap { };

  libtsm = callPackage ../development/libraries/libtsm { };

  libsv = callPackage ../development/libraries/libsv { };

  libgeotiff = callPackage ../development/libraries/libgeotiff { };

  libu2f-host = callPackage ../development/libraries/libu2f-host { };

  libu2f-server = callPackage ../development/libraries/libu2f-server { };

  libubox-nossl = callPackage ../development/libraries/libubox { };

  libubox = callPackage ../development/libraries/libubox { with_ustream_ssl = true; };

  libubox-wolfssl = callPackage ../development/libraries/libubox { with_ustream_ssl = true; ustream-ssl = ustream-ssl-wolfssl; };

  libubox-mbedtls = callPackage ../development/libraries/libubox { with_ustream_ssl = true; ustream-ssl = ustream-ssl-mbedtls; };

  libudev-zero = callPackage ../development/libraries/libudev-zero { };

  libudfread = callPackage ../development/libraries/libudfread { };

  libuecc = callPackage ../development/libraries/libuecc { };

  libuev = callPackage ../development/libraries/libuev {};

  libui = callPackage ../development/libraries/libui {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  libuinputplus = callPackage ../development/libraries/libuinputplus { };

  libuiohook = callPackage ../development/libraries/libuiohook {
    inherit (darwin.apple_sdk.frameworks) AppKit ApplicationServices Carbon;
  };

  libunistring = callPackage ../development/libraries/libunistring { };

  libupnp = callPackage ../development/libraries/pupnp { };

  libwhereami = callPackage ../development/libraries/libwhereami { };

  giflib     = callPackage ../development/libraries/giflib { };

  libunarr = callPackage ../development/libraries/libunarr { };

  libunibreak = callPackage ../development/libraries/libunibreak { };

  libuninameslist = callPackage ../development/libraries/libuninameslist { };

  libunique = callPackage ../development/libraries/libunique { };
  libunique3 = callPackage ../development/libraries/libunique/3.x.nix { };

  liburcu = callPackage ../development/libraries/liburcu { };

  libjaylink = callPackage ../development/libraries/libjaylink { };

  libusb-compat-0_1 = callPackage ../development/libraries/libusb-compat/0.1.nix { };

  libusb1 = callPackage ../development/libraries/libusb1 {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  libusbgx = callPackage ../development/libraries/libusbgx { };

  libusbmuxd = callPackage ../development/libraries/libusbmuxd { };

  libusbsio = callPackage ../development/libraries/libusbsio { };

  libucontext = callPackage ../development/libraries/libucontext { };

  libutempter = callPackage ../development/libraries/libutempter { };

  libuldaq = callPackage ../development/libraries/libuldaq { };

  libunicode = callPackage ../development/libraries/libunicode {
    catch2 = catch2_3;
    fmt = fmt_9;
  };

  libunwind =
    if stdenv.isDarwin then darwin.libunwind
    else if stdenv.hostPlatform.system == "riscv32-linux" then llvmPackages_14.libunwind
    else callPackage ../development/libraries/libunwind { };

  libuv = darwin.apple_sdk_11_0.callPackage ../development/libraries/libuv { };

  libuvc = callPackage ../development/libraries/libuvc { };

  libv4l = lowPrio (v4l-utils.override {
    withUtils = false;
  });

  libva-minimal = callPackage ../development/libraries/libva { minimal = true; };
  libva = libva-minimal.override { minimal = false; };
  libva-utils = callPackage ../development/libraries/libva/utils.nix { };

  libva1 = callPackage ../development/libraries/libva/1.nix { };
  libva1-minimal = libva1.override { minimal = true; };

  libvarlink = callPackage ../development/libraries/libvarlink { };

  libvdpau = callPackage ../development/libraries/libvdpau { };

  libvdwxc = callPackage ../development/libraries/science/chemistry/libvdwxc { };

  libmodulemd = callPackage ../development/libraries/libmodulemd { };

  libvdpau-va-gl = callPackage ../development/libraries/libvdpau-va-gl { };

  libversion = callPackage ../development/libraries/libversion { };

  libverto = callPackage ../development/libraries/libverto { };

  libvgm = callPackage ../development/libraries/libvgm {
    inherit (darwin.apple_sdk.frameworks) CoreAudio AudioToolbox;
  };

  libvirt = callPackage ../development/libraries/libvirt {
    inherit (darwin.apple_sdk.frameworks) Carbon AppKit;
  };

  libvirt-glib = callPackage ../development/libraries/libvirt-glib { };

  libvisio = callPackage ../development/libraries/libvisio { };

  libvisio2svg = callPackage ../development/libraries/libvisio2svg { };

  libvisual = callPackage ../development/libraries/libvisual { };

  libvmaf = callPackage ../development/libraries/libvmaf { };

  libvncserver = callPackage ../development/libraries/libvncserver {
    inherit (darwin.apple_sdk.frameworks) Carbon;
  };

  libviper = callPackage ../development/libraries/libviper { };

  libviperfx = callPackage ../development/libraries/libviperfx { };

  libvpx = callPackage ../development/libraries/libvpx { };
  libvpx_1_8 = callPackage ../development/libraries/libvpx/1_8.nix { };

  libvterm = callPackage ../development/libraries/libvterm { };
  libvterm-neovim = callPackage ../development/libraries/libvterm-neovim { };

  libvorbis = callPackage ../development/libraries/libvorbis { };

  libvori = callPackage ../development/libraries/libvori { };

  libwbxml = callPackage ../development/libraries/libwbxml { };

  libwebcam = callPackage ../os-specific/linux/libwebcam { };

  libwebp = callPackage ../development/libraries/libwebp { };

  libwmf = callPackage ../development/libraries/libwmf { };

  libwnck = callPackage ../development/libraries/libwnck { };
  libwnck2 = callPackage ../development/libraries/libwnck/2.nix { };

  libwpd = callPackage ../development/libraries/libwpd { };

  libwpd_08 = callPackage ../development/libraries/libwpd/0.8.nix { };

  libwps = callPackage ../development/libraries/libwps { };

  libwpg = callPackage ../development/libraries/libwpg { };

  libx86 = callPackage ../development/libraries/libx86 { };

  libxcrypt = callPackage ../development/libraries/libxcrypt {
    fetchurl = stdenv.fetchurlBoot;
    perl = buildPackages.perl.override {
      enableCrypt = false;
      fetchurl = stdenv.fetchurlBoot;
    };
  };
  libxcrypt-legacy = libxcrypt.override { enableHashes = "all"; };

  libxdg_basedir = callPackage ../development/libraries/libxdg-basedir { };

  libxisf = callPackage ../development/libraries/science/astronomy/libxisf { };

  libxkbcommon = libxkbcommon_8;
  libxkbcommon_8 = callPackage ../development/libraries/libxkbcommon { };

  libxklavier = callPackage ../development/libraries/libxklavier { };

  libxls = callPackage ../development/libraries/libxls { };

  libxlsxwriter = callPackage ../development/libraries/libxlsxwriter { };

  libxmi = callPackage ../development/libraries/libxmi { };

  libxml2 = callPackage ../development/libraries/libxml2 {
    python = python3;
  };

  libxml2Python = let
    inherit (python3.pkgs) libxml2;
  in pkgs.buildEnv { # slightly hacky
    name = "libxml2+py-${res.libxml2.version}";
    paths = with libxml2; [ dev bin py ];
    # Avoid update.nix/tests conflicts with libxml2.
    passthru = builtins.removeAttrs libxml2.passthru [ "updateScript" "tests" ];
    # the hook to find catalogs is hidden by buildEnv
    postBuild = ''
      mkdir "$out/nix-support"
      cp '${libxml2.dev}/nix-support/propagated-build-inputs' "$out/nix-support/"
    '';
  };

  libxmlb = callPackage ../development/libraries/libxmlb { };

  libxmlxx = callPackage ../development/libraries/libxmlxx { };
  libxmlxx3 = callPackage ../development/libraries/libxmlxx/v3.nix { };

  libxmp = callPackage ../development/libraries/libxmp { };

  libxslt = callPackage ../development/libraries/libxslt {
    python = python3;
  };

  libxsmm = callPackage ../development/libraries/libxsmm { };

  libixp = callPackage ../development/libraries/libixp { };

  libwpe = callPackage ../development/libraries/libwpe { };

  libwpe-fdo = callPackage ../development/libraries/libwpe/fdo.nix { };

  libyaml = callPackage ../development/libraries/libyaml { };

  yaml-cpp = callPackage ../development/libraries/yaml-cpp { };

  yaml-cpp_0_3 = callPackage ../development/libraries/yaml-cpp/0.3.0.nix { };

  yas = callPackage ../development/libraries/yas { };

  libcyaml = callPackage ../development/libraries/libcyaml { };

  rang = callPackage ../development/libraries/rang { };

  libykclient = callPackage ../development/libraries/libykclient { };

  libykneomgr = callPackage ../development/libraries/libykneomgr { };

  libytnef = callPackage ../development/libraries/libytnef { };

  libyubikey = callPackage ../development/libraries/libyubikey { };

  libyuv = callPackage ../development/libraries/libyuv { };

  libzapojit = callPackage ../development/libraries/libzapojit { };

  libzen = callPackage ../development/libraries/libzen { };

  libzip = callPackage ../development/libraries/libzip { };

  libzdb = callPackage ../development/libraries/libzdb { };

  libwacom = callPackage ../development/libraries/libwacom { };

  libwacom-surface = callPackage ../development/libraries/libwacom/surface.nix { };

  lightlocker = callPackage ../misc/screensavers/light-locker { };

  lightspark = callPackage ../misc/lightspark { };

  opentracing-cpp = callPackage ../development/libraries/opentracing-cpp { };

  ligolo-ng = callPackage ../tools/networking/ligolo-ng { };

  linenoise = callPackage ../development/libraries/linenoise { };

  linenoise-ng = callPackage ../development/libraries/linenoise-ng { };

  lirc = callPackage ../development/libraries/lirc { };

  liquid-dsp = callPackage ../development/libraries/liquid-dsp {
    inherit (darwin) autoSignDarwinBinariesHook cctools;
  };

  liquidfun = callPackage ../development/libraries/liquidfun { };

  litehtml = callPackage ../development/libraries/litehtml { };

  llhttp = callPackage ../development/libraries/llhttp { };

  log4cpp = callPackage ../development/libraries/log4cpp { };

  log4cxx = callPackage ../development/libraries/log4cxx { };

  log4cplus = callPackage ../development/libraries/log4cplus { };

  log4shib = callPackage ../development/libraries/log4shib { };

  loudmouth = callPackage ../development/libraries/loudmouth { };

  lrdf = callPackage ../development/libraries/lrdf { };

  luabind = callPackage ../development/libraries/luabind { lua = lua5_1; };

  luabind_luajit = luabind.override { lua = luajit; };

  luabridge = callPackage ../development/libraries/luabridge { };

  luksmeta = callPackage ../development/libraries/luksmeta {
    asciidoc = asciidoc-full;
  };

  lyra = callPackage ../development/libraries/lyra { };

  lzo = callPackage ../development/libraries/lzo { };

  opencl-clang = callPackage ../development/libraries/opencl-clang { };

  magic-enum = callPackage ../development/libraries/magic-enum { };

  mapnik = callPackage ../development/libraries/mapnik {
    harfbuzz = harfbuzz.override {
      withIcu = true;
    };
  };

  marisa = callPackage ../development/libraries/marisa { };

  mathgl = callPackage ../development/libraries/mathgl { };

  matio = callPackage ../development/libraries/matio { };

  matterhorn = haskell.lib.compose.justStaticExecutables haskellPackages.matterhorn;

  maxflow = callPackage ../development/libraries/maxflow { };

  mbedtls_2 = callPackage ../development/libraries/mbedtls/2.nix { };
  mbedtls = callPackage ../development/libraries/mbedtls/3.nix { };

  mdctags = callPackage ../development/tools/misc/mdctags { };

  mdds = callPackage ../development/libraries/mdds { };

  mediastreamer = libsForQt5.callPackage ../development/libraries/mediastreamer { };

  mediastreamer-openh264 = callPackage ../development/libraries/mediastreamer/msopenh264.nix { };

  memorymapping = callPackage ../development/libraries/memorymapping { };
  memorymappingHook = makeSetupHook {
    name = "memorymapping-hook";
    propagatedBuildInputs = [ memorymapping ];
  } ../development/libraries/memorymapping/setup-hook.sh;

  memray = callPackage ../development/tools/memray { };

  memstream = callPackage ../development/libraries/memstream { };
  memstreamHook = makeSetupHook {
    name = "memstream-hook";
    propagatedBuildInputs = [ memstream ];
  } ../development/libraries/memstream/setup-hook.sh;

  menu-cache = callPackage ../development/libraries/menu-cache { };

  mergerfs = callPackage ../tools/filesystems/mergerfs { };

  mergerfs-tools = callPackage ../tools/filesystems/mergerfs/tools.nix { };

  meshoptimizer = callPackage ../development/libraries/meshoptimizer { };

  mctc-lib = callPackage ../development/libraries/science/chemistry/mctc-lib { };

  mstore = callPackage ../development/libraries/science/chemistry/mstore { };

  multicharge = callPackage ../development/libraries/science/chemistry/multicharge { };

  test-drive = callPackage ../development/libraries/test-drive { };

  dftd4 = callPackage ../development/libraries/science/chemistry/dftd4 { };

  simple-dftd3 = callPackage ../development/libraries/science/chemistry/simple-dftd3 { };

  tblite = callPackage ../development/libraries/science/chemistry/tblite { };

  toml-f = callPackage ../development/libraries/toml-f { };

  dbcsr = callPackage ../development/libraries/science/math/dbcsr { };

  taco = callPackage ../development/libraries/taco { };

  ## libGL/libGLU/Mesa stuff

  # Default libGL implementation, should provide headers and
  # libGL.so/libEGL.so/... to link agains them. Android NDK provides
  # an OpenGL implementation, we can just use that.
  libGL = if stdenv.hostPlatform.useAndroidPrebuilt then stdenv
          else callPackage ../development/libraries/mesa/stubs.nix {
            inherit (darwin.apple_sdk.frameworks) OpenGL;
          };

  # Default libGLU
  libGLU = mesa_glu;

  mesa = darwin.apple_sdk_11_0.callPackage ../development/libraries/mesa {
    inherit (darwin.apple_sdk_11_0.frameworks) OpenGL;
    inherit (darwin.apple_sdk_11_0.libs) Xplugin;
  };
  mesa_i686 = pkgsi686Linux.mesa; # make it build on Hydra

  mesa_glu =  callPackage ../development/libraries/mesa-glu {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices;
  };

  ## End libGL/libGLU/Mesa stuff

  meterbridge = callPackage ../applications/audio/meterbridge { };

  mhddfs = callPackage ../tools/filesystems/mhddfs { };

  microsoft-gsl = callPackage ../development/libraries/microsoft-gsl { };

  microsoft-edge = callPackage (import ../applications/networking/browsers/microsoft-edge).stable { };
  microsoft-edge-beta = callPackage (import ../applications/networking/browsers/microsoft-edge).beta { };
  microsoft-edge-dev = callPackage (import ../applications/networking/browsers/microsoft-edge).dev { };

  micronucleus = callPackage ../development/tools/misc/micronucleus { };

  markdown-anki-decks = callPackage ../tools/misc/markdown-anki-decks { };

  mdk-sdk = callPackage ../development/libraries/mdk-sdk { };

  mdslides = callPackage ../tools/misc/mdslides { };

  micropython = callPackage ../development/interpreters/micropython { };

  MIDIVisualizer = darwin.apple_sdk_11_0.callPackage ../applications/audio/midi-visualizer {
    inherit (darwin.apple_sdk_11_0.frameworks) AppKit Cocoa Carbon CoreAudio CoreMIDI CoreServices Kernel;
  };

  mimalloc = callPackage ../development/libraries/mimalloc { };

  miniaudio = callPackage ../development/libraries/miniaudio { };

  miniz = callPackage ../development/libraries/miniz { };

  minizip = callPackage ../development/libraries/minizip { };

  minizip-ng = callPackage ../development/libraries/minizip-ng { };

  mkvtoolnix = qt6Packages.callPackage ../applications/video/mkvtoolnix {
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  mkvtoolnix-cli = mkvtoolnix.override {
    withGUI = false;
  };

  mlc = callPackage ../tools/system/mlc { };

  mlt = darwin.apple_sdk_11_0.callPackage ../development/libraries/mlt { };

  mlib = callPackage ../development/libraries/mlib { };

  mlv-app = libsForQt5.callPackage ../applications/video/mlv-app { };

  mono-addins = callPackage ../development/libraries/mono-addins { };

  movine = callPackage ../development/tools/database/movine { };

  movit = callPackage ../development/libraries/movit { };

  moserial = callPackage ../tools/misc/moserial { };

  mosquitto = callPackage ../servers/mqtt/mosquitto { };

  nanomq = callPackage ../servers/mqtt/nanomq { };

  mps = callPackage ../development/libraries/mps { };

  libmpeg2 = callPackage ../development/libraries/libmpeg2 { };

  mpeg2dec = libmpeg2;

  mqtt-benchmark = callPackage ../tools/networking/mqtt-benchmark { };

  mqttmultimeter = callPackage ../tools/networking/mqttmultimeter { };

  msgpack = callPackage ../development/libraries/msgpack { };

  msgpack-c = callPackage ../development/libraries/msgpack-c { };

  msgpack-cxx = callPackage ../development/libraries/msgpack-cxx { };

  msoffcrypto-tool = with python3.pkgs; toPythonApplication msoffcrypto-tool;

  msilbc = callPackage ../development/libraries/msilbc { };

  mp4v2 = callPackage ../development/libraries/mp4v2 { };

  libmpc = callPackage ../development/libraries/libmpc { };

  mpich = callPackage ../development/libraries/mpich {
    ch4backend = libfabric;
  };

  mpich-pmix = mpich.override { pmixSupport = true; withPm = [ ]; };

  mstpd = callPackage ../os-specific/linux/mstpd { };

  mtdev = callPackage ../development/libraries/mtdev { };

  mtpfs = callPackage ../tools/filesystems/mtpfs { };

  mtxclient = callPackage ../development/libraries/mtxclient { };

  mu = callPackage ../tools/networking/mu { };

  mueval = callPackage ../development/tools/haskell/mueval { };

  mujoco = callPackage ../applications/science/robotics/mujoco { };

  muparserx = callPackage ../development/libraries/muparserx { };

  mutest = callPackage ../development/libraries/mutest { };

  mvapich = callPackage ../development/libraries/mvapich { };

  mvfst = darwin.apple_sdk_11_0.callPackage ../development/libraries/mvfst { };

  mygpoclient = with python3.pkgs; toPythonApplication mygpoclient;

  mygui = callPackage ../development/libraries/mygui {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  mythes = callPackage ../development/libraries/mythes { };

  nanodbc = callPackage ../development/libraries/nanodbc { };

  nanoflann = callPackage ../development/libraries/nanoflann { };

  nanomsg = callPackage ../development/libraries/nanomsg { };

  nanosvg = callPackage ../development/libraries/nanosvg { };

  nanovna-saver = libsForQt5.callPackage ../applications/science/electronics/nanovna-saver { };

  nanotts = callPackage ../tools/audio/nanotts { };

  ncnn = callPackage ../development/libraries/ncnn { };

  ndpi = callPackage ../development/libraries/ndpi { };

  nemo-qml-plugin-dbus = libsForQt5.callPackage ../development/libraries/nemo-qml-plugin-dbus { };

  netflix = callPackage ../applications/video/netflix { };

  nifticlib = callPackage ../development/libraries/science/biology/nifticlib { };

  notify-sharp = callPackage ../development/libraries/notify-sharp { };

  notcurses = callPackage ../development/libraries/notcurses { };

  ncurses5 = ncurses.override {
    abiVersion = "5";
  };
  ncurses6 = ncurses.override {
    abiVersion = "6";
  };
  ncurses =
    if stdenv.hostPlatform.useiOSPrebuilt
    then null
    else callPackage ../development/libraries/ncurses { };

  ndi = callPackage ../development/libraries/ndi { };

  neardal = callPackage ../development/libraries/neardal { };

  neatvnc = callPackage ../development/libraries/neatvnc { };

  neon = callPackage ../development/libraries/neon { };

  nettle = import ../development/libraries/nettle { inherit callPackage fetchurl; };

  newman = callPackage ../development/web/newman { };

  newt = callPackage ../development/libraries/newt { python = python3; };

  nghttp2 = callPackage ../development/libraries/nghttp2 { };
  libnghttp2 = nghttp2.lib;

  nghttp3 = callPackage ../development/libraries/nghttp3 { inherit (darwin.apple_sdk.frameworks) CoreServices; };

  ngtcp2 = callPackage ../development/libraries/ngtcp2 { inherit (darwin.apple_sdk.frameworks) CoreServices; };
  ngtcp2-gnutls = callPackage ../development/libraries/ngtcp2/gnutls.nix { };

  nix-plugins = callPackage ../development/libraries/nix-plugins { };

  nika-fonts = callPackage ../data/fonts/nika-fonts { };

  nikto = callPackage ../tools/networking/nikto { };

  nlohmann_json = callPackage ../development/libraries/nlohmann_json { };

  nng = callPackage ../development/libraries/nng { };

  nntp-proxy = callPackage ../applications/networking/nntp-proxy { };

  non = callPackage ../applications/audio/non { };

  ntl = callPackage ../development/libraries/ntl { };

  nspr = callPackage ../development/libraries/nspr {
    inherit (darwin.apple_sdk.frameworks) CoreServices;
  };

  nss_latest = callPackage ../development/libraries/nss/latest.nix { };
  nss_esr = callPackage ../development/libraries/nss/esr.nix { };
  nss = nss_esr;
  nssTools = nss.tools;

  nss_wrapper = callPackage ../development/libraries/nss_wrapper { };

  ntbtls = callPackage ../development/libraries/ntbtls { };

  ntk = callPackage ../development/libraries/audio/ntk { };

  numcpp = callPackage ../development/libraries/numcpp { };

  nuraft = callPackage ../development/libraries/nuraft { };

  nuspell = callPackage ../development/libraries/nuspell { };
  nuspellWithDicts = dicts: callPackage ../development/libraries/nuspell/wrapper.nix { inherit dicts; };

  # splicing magic
  nv-codec-headers-versions = callPackages ../development/libraries/nv-codec-headers { };
  inherit (nv-codec-headers-versions)
    nv-codec-headers-9
    nv-codec-headers-10
    nv-codec-headers-11
    nv-codec-headers-12
  ;
  # A default nv-codec-headers to make people happy
  nv-codec-headers = nv-codec-headers-versions.nv-codec-headers-9;

  nvidiaCtkPackages =
    callPackage ../by-name/nv/nvidia-container-toolkit/packages.nix
      { };
  inherit (nvidiaCtkPackages)
    nvidia-docker
    nvidia-podman
    ;

  nvidia-vaapi-driver = lib.hiPrio (callPackage ../development/libraries/nvidia-vaapi-driver { });

  nvidia-optical-flow-sdk = callPackage ../development/libraries/nvidia-optical-flow-sdk { };

  nvidia-system-monitor-qt = libsForQt5.callPackage ../tools/system/nvidia-system-monitor-qt { };

  nvitop = callPackage ../tools/system/nvitop { };

  nvtopPackages = recurseIntoAttrs (import ../tools/system/nvtop { inherit callPackage; });

  ocl-icd = callPackage ../development/libraries/ocl-icd { };

  ode = callPackage ../development/libraries/ode { };

  inherit (callPackages ../development/libraries/ogre { })
    ogre_13 ogre_14;

  ogre = ogre_14;

  olm = callPackage ../development/libraries/olm { };

  one_gadget = callPackage ../development/tools/misc/one_gadget { };

  oneDNN = callPackage ../development/libraries/oneDNN { };

  oneDNN_2 = callPackage ../development/libraries/oneDNN/2.nix { };

  onedrive = callPackage ../applications/networking/sync/onedrive { };

  oneko = callPackage ../applications/misc/oneko { };

  oniguruma = callPackage ../development/libraries/oniguruma { };

  oobicpl = callPackage ../development/libraries/science/biology/oobicpl { };

  ookla-speedtest = callPackage ../tools/networking/ookla-speedtest { };

  openalSoft = callPackage ../development/libraries/openal-soft {
    inherit (darwin.apple_sdk.frameworks) CoreServices AudioUnit AudioToolbox;
  };
  openal = openalSoft;

  openbabel = openbabel3;

  openbabel2 = callPackage ../development/libraries/openbabel/2.nix { };

  openbabel3 = callPackage ../development/libraries/openbabel {
    python = python3;
  };

  opencascade-occt = callPackage ../development/libraries/opencascade-occt { };

  opencascade-occt_7_6 = opencascade-occt.overrideAttrs rec {
    pname = "opencascade-occt";
    version = "7.6.2";
    commit = "V${builtins.replaceStrings ["."] ["_"] version}";
    src = fetchurl {
      name = "occt-${commit}.tar.gz";
      url = "https://git.dev.opencascade.org/gitweb/?p=occt.git;a=snapshot;h=${commit};sf=tgz";
      hash = "sha256-n3KFrN/mN1SVXfuhEUAQ1fJzrCvhiclxfEIouyj9Z18=";
    };
  };

  opencl-headers = callPackage ../development/libraries/opencl-headers { };

  opencl-clhpp = callPackage ../development/libraries/opencl-clhpp { };

  opencollada = callPackage ../development/libraries/opencollada { };

  opencore-amr = callPackage ../development/libraries/opencore-amr { };

  opencsg = callPackage ../development/libraries/opencsg {
    inherit (qt5) qmake;
    inherit (darwin.apple_sdk.frameworks) GLUT;
  };

  openct = callPackage ../development/libraries/openct { };

  opencv2 = callPackage ../development/libraries/opencv {
    inherit (darwin.apple_sdk.frameworks) AVFoundation Cocoa QTKit Accelerate;
    ffmpeg = ffmpeg_4;
  };

  opencv3 = callPackage ../development/libraries/opencv/3.x.nix {
    inherit (darwin.apple_sdk.frameworks)
      AVFoundation Cocoa VideoDecodeAcceleration CoreMedia MediaToolbox Accelerate;
    ffmpeg = ffmpeg_4;
  };

  opencv4 = callPackage ../development/libraries/opencv/4.x.nix {
    inherit (darwin.apple_sdk.frameworks)
      AVFoundation Cocoa VideoDecodeAcceleration CoreMedia MediaToolbox Accelerate;
    pythonPackages = python3Packages;
  };

  opencv4WithoutCuda = opencv4.override {
    enableCuda = false;
  };

  opencv = opencv4;

  imath = callPackage ../development/libraries/imath { };

  openexr = openexr_2;
  openexr_2 = callPackage ../development/libraries/openexr { };
  openexr_3 = callPackage ../development/libraries/openexr/3.nix { };

  openexrid-unstable = callPackage ../development/libraries/openexrid-unstable { };

  openldap = callPackage ../development/libraries/openldap { };

  opencolorio = darwin.apple_sdk_11_0.callPackage ../development/libraries/opencolorio {
    inherit (darwin.apple_sdk_11_0.frameworks) Carbon GLUT Cocoa;
  };
  opencolorio_1 = callPackage ../development/libraries/opencolorio/1.x.nix { };

  opendmarc = callPackage ../development/libraries/opendmarc { };

  ois = callPackage ../development/libraries/ois {
    inherit (darwin.apple_sdk.frameworks) Cocoa IOKit Kernel;
  };

  openjpeg = callPackage ../development/libraries/openjpeg { };

  openpa = callPackage ../development/libraries/openpa { };

  openpgp-card-tools = callPackage ../tools/security/openpgp-card-tools {
    inherit (darwin.apple_sdk.frameworks) PCSC;
  };

  opensaml-cpp = callPackage ../development/libraries/opensaml-cpp { };

  openscenegraph = callPackage ../development/libraries/openscenegraph {
    inherit (darwin.apple_sdk.frameworks) AGL Accelerate Carbon Cocoa Foundation;
  };

  openslide = callPackage ../development/libraries/openslide { };

  openslp = callPackage ../development/libraries/openslp { };

  openstackclient = with python3Packages; toPythonApplication python-openstackclient;
  glanceclient = with python3Packages; toPythonApplication python-glanceclient;
  heatclient = with python3Packages; toPythonApplication python-heatclient;
  ironicclient = with python3Packages; toPythonApplication python-ironicclient;
  manilaclient = with python3Packages; toPythonApplication python-manilaclient;

  openvdb = callPackage ../development/libraries/openvdb { };

  openvr = callPackage ../by-name/op/openvr/package.nix {
    inherit (darwin.apple_sdk.frameworks) Foundation AppKit;
  };

  inherit (callPackages ../development/libraries/libressl { })
    libressl_3_6
    libressl_3_7
    libressl_3_8
    libressl_3_9;

  libressl = libressl_3_9;

  boringssl = callPackage ../development/libraries/boringssl { };

  wolfssl = darwin.apple_sdk_11_0.callPackage ../development/libraries/wolfssl {
    inherit (darwin.apple_sdk_11_0.frameworks) Security;
  };

  openssl = openssl_3;

  openssl_legacy = openssl.override {
    conf = ../development/libraries/openssl/3.0/legacy.cnf;
  };

  inherit (callPackages ../development/libraries/openssl { })
    openssl_1_1
    openssl_3
    openssl_3_2
    openssl_3_3;

  opensubdiv = callPackage ../development/libraries/opensubdiv { };

  opensupaplex = callPackage ../games/opensupaplex { };

  open-wbo = callPackage ../applications/science/logic/open-wbo { };

  openwsman = callPackage ../development/libraries/openwsman { };

  ortp = callPackage ../development/libraries/ortp { };

  openhmd = callPackage ../development/libraries/openhmd { };

  openwebrx = callPackage ../applications/radio/openwebrx {
    inherit (python3Packages)
    buildPythonPackage buildPythonApplication setuptools pycsdr pydigiham;
  };

  oras = callPackage ../development/tools/oras { };

  orcania = callPackage ../development/libraries/orcania { };

  orogene = callPackage ../development/tools/misc/orogene { };

  osm-gps-map = callPackage ../development/libraries/osm-gps-map { };

  osmid = callPackage ../applications/audio/osmid { };

  osinfo-db = callPackage ../data/misc/osinfo-db { };
  osinfo-db-tools = callPackage ../tools/misc/osinfo-db-tools { };

  pacemaker = callPackage ../misc/logging/pacemaker { };

  p11-kit = callPackage ../development/libraries/p11-kit { };

  paperkey = callPackage ../tools/security/paperkey { };

  parsero = callPackage ../tools/security/parsero { };

  pcaudiolib = callPackage ../development/libraries/pcaudiolib { };

  pcg_c = callPackage ../development/libraries/pcg-c { };

  pcl = libsForQt5.callPackage ../development/libraries/pcl {
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
    inherit (darwin.apple_sdk_11_0.frameworks) Cocoa AGL OpenGL;
  };

  pcre = callPackage ../development/libraries/pcre { };
  pcre16 = res.pcre.override { variant = "pcre16"; };
  # pcre32 seems unused
  pcre-cpp = res.pcre.override { variant = "cpp"; };

  pcre2 = callPackage ../development/libraries/pcre2 { };

  pdal = callPackage ../development/libraries/pdal { };

  pdfhummus = libsForQt5.callPackage ../development/libraries/pdfhummus { };

  pe-parse = callPackage ../development/libraries/pe-parse { };

  phetch = callPackage ../applications/networking/gopher/phetch {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  inherit (callPackage ../development/libraries/physfs {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  })
    physfs_2
    physfs;

  pico-sdk = callPackage ../development/libraries/pico-sdk { };

  pinocchio = callPackage ../development/libraries/pinocchio { };

  pipelight = callPackage ../tools/misc/pipelight {
    stdenv = stdenv_32bit;
    wine-staging = pkgsi686Linux.wine-staging;
  };

  pkcs11helper = callPackage ../development/libraries/pkcs11helper { };

  pkgdiff = callPackage ../tools/misc/pkgdiff { };

  pkgtop = callPackage ../tools/misc/pkgtop { };

  place-cursor-at = haskell.lib.compose.justStaticExecutables haskellPackages.place-cursor-at;

  platform-folders = callPackage ../development/libraries/platform-folders { };

  plib = callPackage ../development/libraries/plib { };

  poco = callPackage ../development/libraries/poco { };

  podofo = callPackage ../development/libraries/podofo { };

  podofo010 = callPackage ../development/libraries/podofo/0.10.x.nix { };

  polkit = callPackage ../development/libraries/polkit { };

  poppler = callPackage ../development/libraries/poppler { lcms = lcms2; };

  poppler_gi = lowPrio (poppler.override {
    introspectionSupport = true;
  });

  poppler_min = poppler.override { # TODO: maybe reduce even more
    minimal = true;
    suffix = "min";
  };

  poppler_utils = poppler.override {
    suffix = "utils";
    utils = true;
  };

  popt = callPackage ../development/libraries/popt { };

  portaudio = callPackage ../development/libraries/portaudio {
    inherit (darwin.apple_sdk.frameworks) AudioToolbox AudioUnit CoreAudio CoreServices Carbon;
  };

  portmidi = callPackage ../development/libraries/portmidi {
    inherit (darwin.apple_sdk.frameworks) Carbon CoreAudio CoreFoundation CoreMIDI CoreServices;
  };

  presage = callPackage ../development/libraries/presage { };

  present = callPackage ../misc/present { };

  prime-server = callPackage ../development/libraries/prime-server { };

  proj = callPackage ../development/libraries/proj { };

  proj_7 = callPackage ../development/libraries/proj/7.nix { };

  proj-datumgrid = callPackage ../development/libraries/proj-datumgrid { };

  proselint = callPackage ../tools/text/proselint {
    inherit (python3Packages)
    buildPythonApplication click future six;
  };

  prospector = callPackage ../development/tools/prospector { };

  protobuf = protobuf_24;

  inherit
    ({
      protobuf_26 = callPackage ../development/libraries/protobuf/26.nix { };
      protobuf_25 = callPackage ../development/libraries/protobuf/25.nix { };
      protobuf_24 = callPackage ../development/libraries/protobuf/24.nix { };
      protobuf_23 = callPackage ../development/libraries/protobuf/23.nix {
        abseil-cpp = abseil-cpp_202301;
      };
      protobuf_21 = callPackage ../development/libraries/protobuf/21.nix {
        abseil-cpp = abseil-cpp_202103;
      };
      protobuf3_20 = callPackage ../development/libraries/protobuf/3.20.nix {
        abseil-cpp = abseil-cpp_202103;
      };
    })
    protobuf_26
    protobuf_25
    protobuf_24
    protobuf_23
    protobuf_21
    protobuf3_20
    ;

  protobufc = callPackage ../development/libraries/protobufc { };

  protolock = callPackage ../development/libraries/protolock { };

  protozero = callPackage ../development/libraries/protozero { };

  flatbuffers = callPackage ../development/libraries/flatbuffers { };

  nanopbMalloc = callPackage ../by-name/na/nanopb/package.nix { enableMalloc = true; };

  gnupth = callPackage ../development/libraries/pth { };
  pth = if stdenv.hostPlatform.isMusl then npth else gnupth;

  pslib = callPackage ../development/libraries/pslib { };

  pstreams = callPackage ../development/libraries/pstreams { };

  pufferpanel = callPackage ../servers/pufferpanel { };

  pugixml = callPackage ../development/libraries/pugixml { };

  pylode = callPackage ../misc/pylode { };

  python-qt = libsForQt5.callPackage ../development/libraries/python-qt { };

  pyotherside = libsForQt5.callPackage ../development/libraries/pyotherside { };

  plasma-hud = callPackage ../misc/plasma-hud { };

  re2 = callPackage ../development/libraries/re2 { };

  qbs = libsForQt5.callPackage ../development/tools/build-managers/qbs { };

  qdjango = libsForQt5.callPackage ../development/libraries/qdjango { };

  qmenumodel = libsForQt5.callPackage ../development/libraries/qmenumodel {
    inherit (lomiri) cmake-extras;
  };

  qoi = callPackage ../development/libraries/qoi { };

  qolibri = libsForQt5.callPackage ../applications/misc/qolibri { };

  quarto = callPackage ../development/libraries/quarto { };

  quartoMinimal = callPackage ../development/libraries/quarto { rWrapper = null; python3 = null; };

  qt5 = recurseIntoAttrs (makeOverridable
    (import ../development/libraries/qt-5/5.15) {
      inherit (__splicedPackages)
        makeScopeWithSplicing' generateSplicesForMkScope lib fetchurl fetchpatch fetchgit fetchFromGitHub makeSetupHook makeWrapper
        bison cups dconf harfbuzz libGL perl gtk3 python3
        llvmPackages_15 overrideSDK overrideLibcxx
        darwin;
      inherit (__splicedPackages.gst_all_1) gstreamer gst-plugins-base;
      inherit config;
      stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
    });

  libsForQt5 = (recurseIntoAttrs (import ./qt5-packages.nix {
    inherit lib __splicedPackages makeScopeWithSplicing' generateSplicesForMkScope pkgsHostTarget;
  })) // { __recurseIntoDerivationForReleaseJobs = true; };

  # plasma5Packages maps to the Qt5 packages set that is used to build the plasma5 desktop
  plasma5Packages = libsForQt5;

  qtEnv = qt5.env;
  qt5Full = qt5.full;

  qt6 = recurseIntoAttrs (callPackage ../development/libraries/qt-6 { });

  qt6Packages = recurseIntoAttrs (import ./qt6-packages.nix {
    inherit lib __splicedPackages makeScopeWithSplicing' generateSplicesForMkScope pkgsHostTarget kdePackages;
    stdenv = if stdenv.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  });

  quark-engine = callPackage ../tools/security/quark-engine { };

  quantlib = callPackage ../development/libraries/quantlib { };

  quesoglc = callPackage ../development/libraries/quesoglc { };

  quickder = callPackage ../development/libraries/quickder { };

  quickmem = callPackage ../development/libraries/quickmem { };

  quicksynergy = callPackage ../applications/misc/quicksynergy { };

  quill = callPackage ../tools/security/quill {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  qv2ray = libsForQt5.callPackage ../applications/networking/qv2ray { };

  rabbitmq-c = callPackage ../development/libraries/rabbitmq-c { };

  raft-canonical = callPackage ../development/libraries/raft-canonical { };

  range-v3 = callPackage ../development/libraries/range-v3 { };

  rabbitmq-java-client = callPackage ../development/libraries/rabbitmq-java-client {
    jre = jre8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
    jdk = jdk8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  rapidcheck = callPackage ../development/libraries/rapidcheck { };

  rapidfuzz-cpp = callPackage ../development/libraries/rapidfuzz-cpp { };

  rapidjson = callPackage ../development/libraries/rapidjson { };

  rapidjson-unstable = callPackage ../development/libraries/rapidjson/unstable.nix { };

  rapidxml = callPackage ../development/libraries/rapidxml { };

  rapidyaml = callPackage ../development/libraries/rapidyaml {};

  raul = callPackage ../development/libraries/audio/raul { };

  raylib = callPackage ../development/libraries/raylib { };

  readline = readline82;

  readline70 = callPackage ../development/libraries/readline/7.0.nix { };

  readline82 = callPackage ../development/libraries/readline/8.2.nix { };

  readmdict = with python3Packages; toPythonApplication readmdict;

  readosm = callPackage ../development/libraries/readosm { };

  recastnavigation = callPackage ../development/libraries/recastnavigation { };

  rinutils = callPackage ../development/libraries/rinutils { };

  rtrlib = callPackage ../development/libraries/rtrlib { };

  kissfft = callPackage ../development/libraries/kissfft { };
  kissfftFloat = kissfft.override {
    datatype = "float";
  };

  lambdabot = callPackage ../development/tools/haskell/lambdabot {
    haskellLib = haskell.lib.compose;
  };

  lambda-mod-zsh-theme = callPackage ../shells/zsh/lambda-mod-zsh-theme { };

  librdf_raptor = callPackage ../development/libraries/librdf/raptor.nix { };

  librdf_raptor2 = callPackage ../development/libraries/librdf/raptor2.nix { };

  librdf_rasqal = callPackage ../development/libraries/librdf/rasqal.nix { };

  librdf_redland = callPackage ../development/libraries/librdf/redland.nix { };
  redland = librdf_redland; # added 2018-04-25

  libsmf = callPackage ../development/libraries/audio/libsmf { };

  lilv = callPackage ../development/libraries/audio/lilv { };

  lv2 = callPackage ../development/libraries/audio/lv2 { };

  lvtk = callPackage ../development/libraries/audio/lvtk { };

  qm-dsp = callPackage ../development/libraries/audio/qm-dsp { };

  qradiolink = callPackage ../applications/radio/qradiolink {
    protobuf = protobuf_21;
  };

  qrupdate = callPackage ../development/libraries/qrupdate { };

  qadwaitadecorations-qt6 = callPackage ../by-name/qa/qadwaitadecorations/package.nix {
    useQt6 = true;
  };

  qgnomeplatform = libsForQt5.callPackage ../development/libraries/qgnomeplatform { };

  qgnomeplatform-qt6 = qt6Packages.callPackage ../development/libraries/qgnomeplatform {
    useQt6 = true;
  };

  randomx = darwin.apple_sdk_11_0.callPackage ../development/libraries/randomx { };

  remodel = callPackage ../development/tools/remodel {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  retro-gtk = callPackage ../development/libraries/retro-gtk { };

  resolv_wrapper = callPackage ../development/libraries/resolv_wrapper { };

  restish = callPackage ../tools/networking/restish { };

  rhino = callPackage ../development/libraries/java/rhino {
    javac = jdk8;
    jvm = jre8;
  };

  rlog = callPackage ../development/libraries/rlog { };

  rlottie = callPackage ../development/libraries/rlottie { };

  rocksdb = callPackage ../development/libraries/rocksdb { };

  rocksdb_8_11 = rocksdb.overrideAttrs rec {
    pname = "rocksdb";
    version = "8.11.4";
    src = fetchFromGitHub {
      owner = "facebook";
      repo = pname;
      rev = "v${version}";
      hash = "sha256-ZrU7G3xeimF3H2LRGBDHOq936u5pH/3nGecM4XEoWc8=";
    };
  };

  rocksdb_8_3 = rocksdb.overrideAttrs rec {
    pname = "rocksdb";
    version = "8.3.2";
    src = fetchFromGitHub {
      owner = "facebook";
      repo = pname;
      rev = "v${version}";
      hash = "sha256-mfIRQ8nkUbZ3Bugy3NAvOhcfzFY84J2kBUIUBcQ2/Qg=";
    };
  };

  rocksdb_7_10 = rocksdb.overrideAttrs rec {
    pname = "rocksdb";
    version = "7.10.2";
    src = fetchFromGitHub {
      owner = "facebook";
      repo = pname;
      rev = "v${version}";
      hash = "sha256-U2ReSrJwjAXUdRmwixC0DQXht/h/6rV8SOf5e2NozIs=";
    };
  };

  rocksdb_6_23 = rocksdb.overrideAttrs rec {
    pname = "rocksdb";
    version = "6.23.3";
    src = fetchFromGitHub {
      owner = "facebook";
      repo = pname;
      rev = "v${version}";
     hash = "sha256-SsDqhjdCdtIGNlsMj5kfiuS3zSGwcxi4KV71d95h7yk=";
   };
  };

  rotate-backups = callPackage ../tools/backup/rotate-backups { };

  rote = callPackage ../development/libraries/rote { };

  round = callPackage ../applications/graphics/round { };

  ronn = callPackage ../development/tools/ronn { };

  router = callPackage ../servers/http/router { };

  rover = callPackage ../development/tools/rover { };

  rqlite = callPackage ../servers/sql/rqlite { };

  rshell = python3.pkgs.callPackage ../development/embedded/rshell { };

  rtlcss = callPackage ../development/libraries/rtlcss { };

  rttr = callPackage ../development/libraries/rttr { };

  rubberband = callPackage ../development/libraries/rubberband { };

  rutabaga_gfx = callPackage ../development/libraries/rutabaga_gfx { };

  rure = callPackage ../development/libraries/rure { };

  rustc-demangle = callPackage ../development/libraries/rustc-demangle { };

  rustls-ffi = callPackage ../development/libraries/rustls-ffi {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  s2geometry = callPackage ../development/libraries/s2geometry { };

  safefile = callPackage ../development/libraries/safefile { };

  sbc = callPackage ../development/libraries/sbc { };

  schroedinger = callPackage ../development/libraries/schroedinger {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  scope-lite = callPackage ../development/libraries/scope-lite { };

  SDL_classic = callPackage ../development/libraries/SDL ({
    inherit (darwin.apple_sdk.frameworks) OpenGL CoreAudio CoreServices AudioUnit Kernel Cocoa GLUT;
  } // lib.optionalAttrs stdenv.hostPlatform.isAndroid {
    # libGLU doesn’t work with Android’s SDL
    libGLU = null;
  });

  SDL_compat = callPackage ../development/libraries/SDL_compat {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
    inherit (darwin) autoSignDarwinBinariesHook;
  };

  SDL = SDL_classic;

  SDL_audiolib = callPackage ../development/libraries/SDL_audiolib { };

  SDL_sixel = callPackage ../development/libraries/SDL_sixel { };

  SDL_gfx = callPackage ../development/libraries/SDL_gfx { };

  SDL_gpu = callPackage ../development/libraries/SDL_gpu { };

  SDL_image = callPackage ../development/libraries/SDL_image { };

  SDL_mixer = callPackage ../development/libraries/SDL_mixer { };

  SDL_net = callPackage ../development/libraries/SDL_net { };

  SDL_Pango = callPackage ../development/libraries/SDL_Pango { };

  SDL_sound = callPackage ../development/libraries/SDL_sound { };

  SDL_stretch= callPackage ../development/libraries/SDL_stretch { };

  SDL_ttf = callPackage ../development/libraries/SDL_ttf { };

  SDL2 = callPackage ../development/libraries/SDL2 {
    inherit (darwin.apple_sdk.frameworks) AudioUnit Cocoa CoreAudio CoreServices ForceFeedback OpenGL;
  };

  SDL2_image = callPackage ../development/libraries/SDL2_image {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };
  # Pinned for pygame, toppler
  SDL2_image_2_0 = SDL2_image.overrideAttrs (oldAttrs: {
    version = "2.0.5";
    src = fetchurl {
      inherit (oldAttrs.src) url;
      hash = "sha256-vdX24CZoL31+G+C2BRsgnaL0AqLdi9HEvZwlrSYxCNA";
    };
  });
  # Pinned for hedgewars:
  #   https://github.com/NixOS/nixpkgs/pull/274185#issuecomment-1856764786
  SDL2_image_2_6 = SDL2_image.overrideAttrs (oldAttrs: {
    version = "2.6.3";
    src = fetchurl {
      inherit (oldAttrs.src) url;
      hash = "sha256-kxyb5b8dfI+um33BV4KLfu6HTiPH8ktEun7/a0g2MSw=";
    };
  });

  SDL2_gfx = callPackage ../development/libraries/SDL2_gfx { };

  SDL2_sound = callPackage ../development/libraries/SDL2_sound {
    inherit (darwin.apple_sdk.frameworks) AudioToolbox CoreAudio;
  };

  SDL2_ttf = callPackage ../development/libraries/SDL2_ttf { };

  sdrplay = callPackage ../applications/radio/sdrplay { };

  sdrpp = callPackage ../applications/radio/sdrpp {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  sigdigger = libsForQt5.callPackage ../applications/radio/sigdigger { };

  sigutils = callPackage ../applications/radio/sigutils { };

  sblim-sfcc = callPackage ../development/libraries/sblim-sfcc { };

  selinux-sandbox = callPackage ../os-specific/linux/selinux-sandbox { };

  sealcurses = callPackage ../development/libraries/sealcurses { };

  seasocks = callPackage ../development/libraries/seasocks { };

  serd = callPackage ../development/libraries/serd { };

  serf = callPackage ../development/libraries/serf { };

  sfsexp = callPackage ../development/libraries/sfsexp { };

  shhmsg = callPackage ../development/libraries/shhmsg { };

  shhopt = callPackage ../development/libraries/shhopt { };

  shine = callPackage ../development/libraries/shine { };

  graphite2 = callPackage ../development/libraries/silgraphite/graphite2.nix { };

  s2n-tls = callPackage ../development/libraries/s2n-tls { };

  simavr = callPackage ../development/tools/simavr {
    avrgcc = pkgsCross.avr.buildPackages.gcc;
    avrlibc = pkgsCross.avr.libcCross;
    inherit (darwin.apple_sdk.frameworks) GLUT;
  };

  simgear = callPackage ../development/libraries/simgear { };

  simp_le = callPackage ../tools/admin/simp_le { };

  simpleitk = callPackage ../development/libraries/simpleitk { lua = lua5_4; };

  sfml = callPackage ../development/libraries/sfml {
    inherit (darwin.apple_sdk.frameworks) IOKit Foundation AppKit OpenAL;
  };
  csfml = callPackage ../development/libraries/csfml { };

  shapelib = callPackage ../development/libraries/shapelib { };

  sharness = callPackage ../development/libraries/sharness { };

  shibboleth-sp = callPackage ../development/libraries/shibboleth-sp { };

  skaffold = callPackage ../development/tools/skaffold { };

  skjold = callPackage ../development/tools/skjold { };

  skawarePackages = recurseIntoAttrs (callPackage ../development/skaware-packages { });

  inherit (skawarePackages)
    execline
    execline-man-pages
    mdevd
    nsss
    s6
    s6-dns
    s6-linux-init
    s6-linux-utils
    s6-man-pages
    s6-networking
    s6-networking-man-pages
    s6-portable-utils
    s6-portable-utils-man-pages
    s6-rc
    s6-rc-man-pages
    sdnotify-wrapper
    skalibs
    skalibs_2_10
    tipidee
    utmps;

  kgt = callPackage ../development/tools/kgt {
    inherit (skawarePackages) cleanPackaging;
  };

  nettee = callPackage ../tools/networking/nettee {
    inherit (skawarePackages) cleanPackaging;
  };

  shaq = callPackage ../tools/audio/shaq { };

  slang = callPackage ../development/libraries/slang { };

  slibGuile = callPackage ../development/libraries/slib {
    scheme = guile;
  };

  smpeg = callPackage ../development/libraries/smpeg { };

  smpeg2 = callPackage ../development/libraries/smpeg2 { };

  snappy = callPackage ../development/libraries/snappy { };

  snac2 = darwin.apple_sdk_11_0.callPackage ../servers/snac2 { };

  snappymail = callPackage ../servers/snappymail { };

  snow = callPackage ../tools/security/snow { };

  snowcrash = callPackage ../tools/security/snowcrash { };

  soapyairspy = callPackage ../applications/radio/soapyairspy {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  soapyaudio = callPackage ../applications/radio/soapyaudio {
    inherit (darwin.apple_sdk.frameworks) Accelerate CoreAudio;
  };

  soapybladerf = callPackage ../applications/radio/soapybladerf {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  soapyhackrf = callPackage ../applications/radio/soapyhackrf {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  soapysdr = callPackage ../applications/radio/soapysdr { };

  soapyremote = callPackage ../applications/radio/soapyremote { };

  soapysdrplay = callPackage ../applications/radio/soapysdrplay { };

  soapysdr-with-plugins = callPackage ../applications/radio/soapysdr {
    extraPackages = [
      limesuite
      soapyairspy
      soapyaudio
      soapybladerf
      soapyhackrf
      soapyremote
      soapyrtlsdr
      soapyuhd
    ];
  };

  soapyrtlsdr = callPackage ../applications/radio/soapyrtlsdr {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  soapyuhd = callPackage ../applications/radio/soapyuhd {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.frameworks) IOKit Security;
  };

  social-engineer-toolkit = callPackage ../tools/security/social-engineer-toolkit { };

  socket_wrapper = callPackage ../development/libraries/socket_wrapper { };

  sofia_sip = callPackage ../development/libraries/sofia-sip {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration;
  };

  soil = callPackage ../development/libraries/soil {
    inherit (darwin.apple_sdk.frameworks) Carbon;
  };

  sokol = callPackage ../development/libraries/sokol { };

  sonic = callPackage ../development/libraries/sonic { };

  sonivox = callPackage ../development/libraries/sonivox { };

  sope = callPackage ../development/libraries/sope { };

  sord = callPackage ../development/libraries/sord { };

  soundtouch = callPackage ../development/libraries/soundtouch { };

  span-lite = callPackage ../development/libraries/span-lite { };

  spandsp = callPackage ../development/libraries/spandsp { };
  spandsp3 = callPackage ../development/libraries/spandsp/3.nix { };

  spaceship-prompt = callPackage ../shells/zsh/spaceship-prompt { };

  sparrow3d = callPackage ../development/libraries/sparrow3d {};

  spdk = callPackage ../development/libraries/spdk { };

  speechd = callPackage ../development/libraries/speechd { };

  speech-tools = callPackage ../development/libraries/speech-tools {
    inherit (darwin.apple_sdk.frameworks) CoreServices AudioUnit Cocoa;
  };

  speex = callPackage ../development/libraries/speex {
    fftw = fftwFloat;
  };

  speexdsp = callPackage ../development/libraries/speexdsp {
    fftw = fftwFloat;
  };

  sphinx = with python3Packages; toPythonApplication sphinx;

  sphinx-autobuild = with python3Packages; toPythonApplication sphinx-autobuild;

  sphinx-serve = with python3Packages; toPythonApplication sphinx-serve;

  inherit (python3Packages) sphinxHook;

  sphinxsearch = callPackage ../servers/search/sphinxsearch { };

  spice = callPackage ../development/libraries/spice { };

  spice-gtk = callPackage ../development/libraries/spice-gtk { };

  spice-protocol = callPackage ../development/libraries/spice-protocol { };

  spice-up = callPackage ../applications/office/spice-up { };

  spirv-cross = callPackage ../tools/graphics/spirv-cross { };

  splat = callPackage ../applications/radio/splat { };

  suscan = callPackage ../applications/radio/suscan { };

  suwidgets = libsForQt5.callPackage ../applications/radio/suwidgets { };

  sratom = callPackage ../development/libraries/audio/sratom { };

  srm = callPackage ../tools/security/srm { };

  srt = callPackage ../development/libraries/srt { };

  srtp = callPackage ../development/libraries/srtp { };

  stargate-libcds = callPackage ../development/libraries/stargate-libcds { };

  stb = callPackage ../development/libraries/stb { };

  steghide = callPackage ../tools/graphics/steghide { };

  stegsolve = callPackage ../tools/graphics/stegsolve { };

  stxxl = callPackage ../development/libraries/stxxl { };

  sv-lang = callPackage ../applications/science/electronics/sv-lang { };

  sqld = callPackage ../servers/sql/sqld { };

  sqlite = lowPrio (callPackage ../development/libraries/sqlite { });

  unqlite = lowPrio (callPackage ../development/libraries/unqlite { });

  inherit (callPackage ../development/libraries/sqlite/tools.nix {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  }) sqlite-analyzer sqldiff;

  sqlar = callPackage ../development/libraries/sqlite/sqlar.nix { };

  sqlitecpp = callPackage ../development/libraries/sqlitecpp { };

  sqlite-interactive = (sqlite.override { interactive = true; }).bin;

  sqlite-jdbc = callPackage ../servers/sql/sqlite/jdbc { };

  sregex = callPackage ../development/libraries/sregex { };

  dqlite = callPackage ../development/libraries/dqlite { };

  sqlcipher = callPackage ../development/libraries/sqlcipher { };

  standardnotes = callPackage ../applications/editors/standardnotes { };

  stfl = callPackage ../development/libraries/stfl { };

  stlink = callPackage ../development/tools/misc/stlink { };
  stlink-gui = callPackage ../development/tools/misc/stlink { withGUI = true; };

  stegseek = callPackage ../tools/security/stegseek { };

  streamlink = callPackage ../applications/video/streamlink { };
  streamlink-twitch-gui-bin = callPackage ../applications/video/streamlink-twitch-gui/bin.nix { };

  structuresynth = libsForQt5.callPackage ../development/libraries/structuresynth { };

  sub-batch = callPackage ../applications/video/sub-batch { };

  subdl = callPackage ../applications/video/subdl { };

  subtitleedit = callPackage ../applications/video/subtitleedit { };

  subtitleeditor = callPackage ../applications/video/subtitleeditor { };

  subxt = callPackage ../development/tools/subxt { };

  suil = darwin.apple_sdk_11_0.callPackage ../development/libraries/audio/suil { };

  sundials = callPackage ../development/libraries/sundials {
    python = python3;
  };

  sutils = callPackage ../tools/misc/sutils { };

  svrcore = callPackage ../development/libraries/svrcore { };

  svxlink = libsForQt5.callPackage ../applications/radio/svxlink { };

  swiftclient = with python3Packages; toPythonApplication python-swiftclient;

  biblesync = callPackage ../development/libraries/biblesync { };

  szip = callPackage ../development/libraries/szip { };

  t1lib = callPackage ../development/libraries/t1lib { };

  tachyon = callPackage ../development/libraries/tachyon {
    inherit (darwin.apple_sdk.frameworks) Carbon;
  };

  tageditor = libsForQt5.callPackage ../applications/audio/tageditor { };

  ta-lib = callPackage ../development/libraries/ta-lib { };

  taglib = callPackage ../development/libraries/taglib { };

  taglib_extras = callPackage ../development/libraries/taglib-extras { };

  taglib-sharp = callPackage ../development/libraries/taglib-sharp { };

  tanidvr = callPackage ../applications/video/tanidvr { };

  talloc = callPackage ../development/libraries/talloc { };

  tagparser = callPackage ../development/libraries/tagparser { };

  taskflow = callPackage ../development/libraries/taskflow { };

  tclap = tclap_1_2;

  tclap_1_2 = callPackage ../development/libraries/tclap/1.2.nix { };

  tclap_1_4 = callPackage ../development/libraries/tclap/1.4.nix { };

  tcllib = callPackage ../development/libraries/tcllib { };

  tcltls = callPackage ../development/libraries/tcltls { };

  tclx = callPackage ../development/libraries/tclx { };

  tcl-fcgi = callPackage ../development/libraries/tcl-fcgi { };

  tdb = callPackage ../development/libraries/tdb { };

  tdlib = callPackage ../development/libraries/tdlib { };

  tectonic = callPackage ../tools/typesetting/tectonic/wrapper.nix { };

  tectonic-unwrapped = callPackage ../tools/typesetting/tectonic {
    harfbuzz = harfbuzzFull;
  };

  termbench-pro = callPackage ../development/libraries/termbench-pro { fmt = fmt_8; };

  telepathy-glib = callPackage ../development/libraries/telepathy/glib { };

  telepathy-farstream = callPackage ../development/libraries/telepathy/farstream { };

  termbox = callPackage ../development/libraries/termbox { };

  tevent = callPackage ../development/libraries/tevent { };

  tet = callPackage ../development/tools/misc/tet { };

  texpresso = callPackage ../tools/typesetting/tex/texpresso {
    texpresso-tectonic = callPackage ../tools/typesetting/tex/texpresso/tectonic.nix { };
  };

  text-engine = callPackage ../development/libraries/text-engine { };

  the-foundation = callPackage ../development/libraries/the-foundation { };

  theft = callPackage ../development/libraries/theft { };

  thrift = callPackage ../development/libraries/thrift { };

  tidyp = callPackage ../development/libraries/tidyp { };

  tidal-hifi = callPackage ../applications/audio/tidal-hifi { };

  tinycdb = callPackage ../development/libraries/tinycdb { };

  tinyxml = tinyxml2;

  tinyxml2 = callPackage ../development/libraries/tinyxml/2.6.2.nix { };

  tiscamera = callPackage ../os-specific/linux/tiscamera { };

  tivodecode = callPackage ../applications/video/tivodecode { };

  tix = callPackage ../development/libraries/tix { };

  tk = tk-8_6;

  tk-8_6 = callPackage ../development/libraries/tk/8.6.nix { };
  tk-8_5 = callPackage ../development/libraries/tk/8.5.nix { tcl = tcl-8_5; };

  tkimg = callPackage ../development/libraries/tkimg { };

  tkrzw = callPackage ../development/libraries/tkrzw { };

  tl-expected = callPackage ../development/libraries/tl-expected { };

  tnt = callPackage ../development/libraries/tnt { };

  tntnet = callPackage ../development/libraries/tntnet { };

  tntdb = callPackage ../development/libraries/tntdb { };

  kyotocabinet = callPackage ../development/libraries/kyotocabinet { };

  tomlc99 = callPackage ../development/libraries/tomlc99 { };

  tomlcpp = callPackage ../development/libraries/tomlcpp { };

  toml11 = callPackage ../development/libraries/toml11 { };

  tomlplusplus = callPackage ../development/libraries/tomlplusplus { };

  tokyocabinet = callPackage ../development/libraries/tokyo-cabinet { };

  tokyotyrant = callPackage ../development/libraries/tokyo-tyrant { };

  totem-pl-parser = callPackage ../development/libraries/totem-pl-parser { };

  tpm2-tss = callPackage ../development/libraries/tpm2-tss {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  tremor = callPackage ../development/libraries/tremor { };

  triehash = callPackage ../tools/misc/triehash { };

  trillian = callPackage ../tools/misc/trillian { };

  twolame = callPackage ../development/libraries/twolame { };

  udns = callPackage ../development/libraries/udns { };

  uid_wrapper = callPackage ../development/libraries/uid_wrapper { };

  umockdev = callPackage ../development/libraries/umockdev { };

  unconvert = callPackage ../development/tools/unconvert {
    buildGoModule = buildGo121Module;
  };

  uncover = callPackage ../tools/security/uncover { };

  unibilium = callPackage ../development/libraries/unibilium { };

  unicap = callPackage ../development/libraries/unicap { };

  unicon-lang = callPackage ../development/interpreters/unicon-lang { };

  updfparser = callPackage ../development/libraries/updfparser { };

  tsocks = callPackage ../development/libraries/tsocks { };

  unixODBC = callPackage ../development/libraries/unixODBC { };

  unixODBCDrivers = recurseIntoAttrs (callPackages ../development/libraries/unixODBCDrivers { });

  usrsctp = callPackage ../development/libraries/usrsctp { };

  usbredir = callPackage ../development/libraries/usbredir { };

  usbrip = callPackage ../tools/security/usbrip { };

  uthash = callPackage ../development/libraries/uthash { };

  uthenticode = callPackage ../development/libraries/uthenticode { };

  ucommon = callPackage ../development/libraries/ucommon { };

  v8 = callPackage ../development/libraries/v8 {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
  };

  intel-vaapi-driver = callPackage ../development/libraries/intel-vaapi-driver { };

  vaapi-intel-hybrid = callPackage ../development/libraries/vaapi-intel-hybrid { };

  vaapiVdpau = callPackage ../development/libraries/vaapi-vdpau { };

  vale = callPackage ../tools/text/vale { };

  valeStyles = recurseIntoAttrs (callPackages ../tools/text/vale/styles.nix { });

  valhalla = callPackage ../development/libraries/valhalla {
    boost = boost.override { enablePython = true; python = python3; };
    protobuf = protobuf_21.override {
      abseil-cpp = abseil-cpp_202103.override {
        cxxStandard = "17";
      };
    };
  };

  vamp-plugin-sdk = callPackage ../development/libraries/audio/vamp-plugin-sdk { };

  vc = callPackage ../development/libraries/vc { };

  vc_0_7 = callPackage ../development/libraries/vc/0.7.nix { };

  vcdimager = callPackage ../development/libraries/vcdimager { };

  vcg = callPackage ../development/libraries/vcg { };

  vencord-web-extension = callPackage ../by-name/ve/vencord/package.nix { buildWebExtension = true; };

  vid-stab = callPackage ../development/libraries/vid-stab {
    inherit (llvmPackages) openmp;
  };

  vigra = callPackage ../development/libraries/vigra {
    hdf5 = hdf5.override { usev110Api = true; };
  };

  vkd3d = callPackage ../development/libraries/vkd3d {};

  vkd3d-proton = callPackage ../development/libraries/vkd3d-proton {};

  vkdisplayinfo = callPackage ../tools/graphics/vkdisplayinfo { };

  vkdt = callPackage ../applications/graphics/vkdt { };

  vlock = callPackage ../misc/screensavers/vlock { };

  virtualpg = callPackage ../development/libraries/virtualpg { };

  vmime = callPackage ../development/libraries/vmime { };

  vrb = callPackage ../development/libraries/vrb { };

  vrpn = callPackage ../development/libraries/vrpn { };

  vsqlite = callPackage ../development/libraries/vsqlite { };

  vte = callPackage ../development/libraries/vte { };

  vte-gtk4 = vte.override {
    gtkVersion = "4";
  };

  vtk_9 = libsForQt5.callPackage ../development/libraries/vtk/9.x.nix {
    inherit (darwin) libobjc;
    inherit (darwin.apple_sdk.libs) xpc;
    inherit (darwin.apple_sdk.frameworks) AGL Cocoa CoreServices DiskArbitration
                                          IOKit CFNetwork Security ApplicationServices
                                          CoreText IOSurface ImageIO OpenGL GLUT;
  };

  vtk_9_withQt5 = vtk_9.override { enableQt = true; };

  vtk = vtk_9;
  vtkWithQt5 = vtk_9_withQt5;

  vulkan-caps-viewer = libsForQt5.callPackage ../tools/graphics/vulkan-caps-viewer { };

  vulkan-cts = callPackage ../tools/graphics/vulkan-cts { };

  vulkan-helper = callPackage ../tools/graphics/vulkan-helper { };

  vulkan-extension-layer = callPackage ../tools/graphics/vulkan-extension-layer { };
  vulkan-headers = callPackage ../development/libraries/vulkan-headers { };
  vulkan-loader = callPackage ../development/libraries/vulkan-loader { inherit (darwin) moltenvk; };
  vulkan-tools = callPackage ../tools/graphics/vulkan-tools {
    inherit (darwin) moltenvk;
    inherit (darwin.apple_sdk.frameworks) AppKit Cocoa;
  };
  vulkan-tools-lunarg = callPackage ../tools/graphics/vulkan-tools-lunarg { };
  vulkan-utility-libraries = callPackage ../development/libraries/vulkan-utility-libraries { };
  vulkan-validation-layers = callPackage ../development/tools/vulkan-validation-layers { };

  vxl = callPackage ../development/libraries/vxl { };

  waffle = callPackage ../development/libraries/waffle { };

  wally-cli = callPackage ../development/tools/wally-cli { };
  zsa-udev-rules = callPackage ../os-specific/linux/zsa-udev-rules { };

  wangle = darwin.apple_sdk_11_0.callPackage ../development/libraries/wangle { };

  wapp = callPackage ../development/libraries/wapp { };

  wavpack = callPackage ../development/libraries/wavpack { };

  wayland = darwin.apple_sdk_11_0.callPackage ../development/libraries/wayland { };
  wayland-scanner = wayland.bin;

  wayland-protocols = callPackage ../development/libraries/wayland/protocols.nix { };

  waylandpp = callPackage ../development/libraries/waylandpp {
    graphviz = graphviz-nox;
  };

  wcslib = callPackage ../development/libraries/science/astronomy/wcslib { };

  webkitgtk = callPackage ../development/libraries/webkitgtk {
    harfbuzz = harfbuzzFull;
    inherit (gst_all_1) gst-plugins-base gst-plugins-bad;
    inherit (darwin) apple_sdk;
  };

  webkitgtk_4_1 = webkitgtk.override {
    libsoup = libsoup_3;
  };

  webkitgtk_6_0 = webkitgtk.override {
    libsoup = libsoup_3;
    gtk3 = gtk4;
  };

  webp-pixbuf-loader = callPackage ../development/libraries/webp-pixbuf-loader { };

  websocketpp = callPackage ../development/libraries/websocket++ { };

  wfa2-lib = callPackage ../development/libraries/wfa2-lib { };

  webrtc-audio-processing_1 = callPackage ../development/libraries/webrtc-audio-processing { };
  webrtc-audio-processing_0_3 = callPackage ../development/libraries/webrtc-audio-processing/0.3.nix { };
  # bump when majoring of packages have updated
  webrtc-audio-processing = webrtc-audio-processing_0_3;

  whereami = callPackage ../development/libraries/whereami { };

  wildmidi = callPackage ../development/libraries/wildmidi {
    inherit (darwin.apple_sdk.frameworks) OpenAL CoreAudioKit;
  };

  wiredtiger = callPackage ../development/libraries/wiredtiger { };

  wlr-protocols = callPackage ../development/libraries/wlroots/protocols.nix { };

  wt = wt4;
  inherit (libsForQt5.callPackage ../development/libraries/wt { })
    wt4;

  wxformbuilder = callPackage ../development/tools/wxformbuilder { };

  wxGTK31 = callPackage ../development/libraries/wxwidgets/wxGTK31.nix {
    inherit (darwin.stubs) setfile;
    inherit (darwin.apple_sdk.frameworks) AGL Carbon Cocoa Kernel QTKit AVFoundation AVKit WebKit;
  };

  wxGTK32 = callPackage ../development/libraries/wxwidgets/wxGTK32.nix {
    inherit (darwin.stubs) setfile;
    inherit (darwin.apple_sdk.frameworks) AGL Carbon Cocoa Kernel QTKit AVFoundation AVKit WebKit;
  };

  wxSVG = callPackage ../development/libraries/wxSVG {
    wxGTK = wxGTK32;
  };

  wtk = callPackage ../development/libraries/wtk { };

  x264 = callPackage ../development/libraries/x264 { };

  x265 = callPackage ../development/libraries/x265 { };

  xandikos = callPackage ../servers/xandikos { };

  inherit (callPackages ../development/libraries/xapian { })
    xapian_1_4;
  xapian = xapian_1_4;

  xapian-omega = callPackage ../development/libraries/xapian/tools/omega {
    libmagic = file;
  };

  xavs = callPackage ../development/libraries/xavs { };

  Xaw3d = callPackage ../development/libraries/Xaw3d { };

  xbase = callPackage ../development/libraries/xbase { };

  xcb-util-cursor = xorg.xcbutilcursor;
  xcb-util-cursor-HEAD = callPackage ../development/libraries/xcb-util-cursor/HEAD.nix { };

  xcbutilxrm = callPackage ../servers/x11/xorg/xcb-util-xrm.nix { };

  xdo = callPackage ../tools/misc/xdo { };

  xdiskusage = callPackage ../tools/misc/xdiskusage { };

  xed = callPackage ../development/libraries/xed { };

  xedit = callPackage ../applications/editors/xedit { };

  xautolock = callPackage ../misc/screensavers/xautolock { };

  xercesc = callPackage ../development/libraries/xercesc { };

  xalanc = callPackage ../development/libraries/xalanc { };

  xgboost = callPackage ../development/libraries/xgboost { };

  xgboostWithCuda = xgboost.override { cudaSupport = true; };

  xgeometry-select = callPackage ../tools/X11/xgeometry-select { };

  xmake-core-sv = callPackage ../development/libraries/xmake-core-sv { };

  xmlrpc_c = callPackage ../development/libraries/xmlrpc-c { };

  xmlsec = callPackage ../development/libraries/xmlsec { };

  xml-security-c = callPackage ../development/libraries/xml-security-c { };

  xlslib = callPackage ../development/libraries/xlslib { };

  xsimd = callPackage ../development/libraries/xsimd { };

  xtensor = callPackage ../development/libraries/xtensor { };

  xtl = callPackage ../development/libraries/xtl { };

  xvidcore = callPackage ../development/libraries/xvidcore { };

  xxHash = callPackage ../development/libraries/xxHash { };

  xylib = callPackage ../development/libraries/xylib { };

  yajl = callPackage ../development/libraries/yajl { };

  yder = callPackage ../development/libraries/yder { };

  yojimbo = callPackage ../development/libraries/yojimbo { };

  yubico-pam = callPackage ../development/libraries/yubico-pam {
    inherit (darwin.apple_sdk.frameworks) CoreServices SystemConfiguration;
  };

  yubico-piv-tool = callPackage ../tools/misc/yubico-piv-tool {
    inherit (darwin.apple_sdk.frameworks) PCSC;
  };

  yubihsm-connector = callPackage ../tools/security/yubihsm-connector { };

  yubikey-manager = callPackage ../tools/misc/yubikey-manager { };

  yubikey-manager-qt = libsForQt5.callPackage ../tools/misc/yubikey-manager-qt { };

  yubikey-personalization = callPackage ../tools/misc/yubikey-personalization { };

  yubikey-personalization-gui = libsForQt5.callPackage ../tools/misc/yubikey-personalization-gui { };

  yubikey-agent = callPackage ../tools/security/yubikey-agent { };

  yubikey-touch-detector = callPackage ../tools/security/yubikey-touch-detector { };

  yubihsm-shell = callPackage ../tools/security/yubihsm-shell { };

  yubioath-flutter = callPackage ../applications/misc/yubioath-flutter { };

  yyjson = callPackage ../development/libraries/yyjson { };

  zchunk = callPackage ../development/libraries/zchunk { };

  zeitgeist = callPackage ../development/libraries/zeitgeist { };

  zlib = callPackage ../development/libraries/zlib { };

  zlib-ng = callPackage ../development/libraries/zlib-ng { };

  libdynd = callPackage ../development/libraries/libdynd { };

  zlog = callPackage ../development/libraries/zlog { };

  zeromq4 = callPackage ../development/libraries/zeromq/4.x.nix { };
  zeromq = zeromq4;

  cppzmq = callPackage ../development/libraries/cppzmq { };

  czmq = callPackage ../development/libraries/czmq { };

  zmqpp = callPackage ../development/libraries/zmqpp { };

  libzra = callPackage ../development/libraries/libzra { };

  # requires a newer Apple SDK
  zig_0_9 = darwin.apple_sdk_11_0.callPackage ../development/compilers/zig/0.9.nix {
    llvmPackages = llvmPackages_13;
  };
  # requires a newer Apple SDK
  zig_0_10 = darwin.apple_sdk_11_0.callPackage ../development/compilers/zig/0.10.nix {
    llvmPackages = llvmPackages_15;
  };
  # requires a newer Apple SDK
  zig_0_11 = darwin.apple_sdk_11_0.callPackage ../development/compilers/zig/0.11.nix {
    llvmPackages = llvmPackages_16;
  };
  # requires a newer Apple SDK
  zig_0_12 = darwin.apple_sdk_11_0.callPackage ../development/compilers/zig/0.12.nix {
    llvmPackages = llvmPackages_17;
  };
  zig = zig_0_12;

  zig-shell-completions = callPackage ../development/compilers/zig/shell-completions.nix { };

  zimlib = callPackage ../development/libraries/zimlib { };

  zita-convolver = callPackage ../development/libraries/audio/zita-convolver { };

  zita-alsa-pcmi = callPackage ../development/libraries/audio/zita-alsa-pcmi { };

  zita-resampler = callPackage ../development/libraries/audio/zita-resampler { };

  zitadel-tools = callPackage ../tools/misc/zitadel-tools { };

  zix = callPackage ../development/libraries/audio/zix { };

  zziplib = callPackage ../development/libraries/zziplib { };

  glpng = callPackage ../development/libraries/glpng { };

  gsignond = callPackage ../development/libraries/gsignond {
    plugins = [];
  };

  gsignondPlugins = recurseIntoAttrs {
    sasl = callPackage ../development/libraries/gsignond/plugins/sasl.nix { };
    oauth = callPackage ../development/libraries/gsignond/plugins/oauth.nix { };
    lastfm = callPackage ../development/libraries/gsignond/plugins/lastfm.nix { };
    mail = callPackage ../development/libraries/gsignond/plugins/mail.nix { };
  };

  plumed = callPackage ../development/libraries/science/chemistry/plumed { };

  ### DEVELOPMENT / LIBRARIES / AGDA

  agdaPackages = callPackage ./agda-packages.nix {
    inherit (haskellPackages) Agda;
  };
  agda = agdaPackages.agda;

  ### DEVELOPMENT / LIBRARIES / BASH

  bash-preexec = callPackage ../development/libraries/bash/bash-preexec { };

  ### DEVELOPMENT / LIBRARIES / JAVA

  commonsBcel = callPackage ../development/libraries/java/commons/bcel { };

  commonsBsf = callPackage ../development/libraries/java/commons/bsf { };

  commonsCompress = callPackage ../development/libraries/java/commons/compress { };

  commonsDaemon = callPackage ../development/libraries/java/commons/daemon { };

  commonsFileUpload = callPackage ../development/libraries/java/commons/fileupload { };

  commonsLang = callPackage ../development/libraries/java/commons/lang { };

  commonsLogging = callPackage ../development/libraries/java/commons/logging { };

  commonsIo = callPackage ../development/libraries/java/commons/io { };

  commonsMath = callPackage ../development/libraries/java/commons/math { };

  fastjar = callPackage ../development/tools/java/fastjar { };

  httpunit = callPackage ../development/libraries/java/httpunit { };

  javaCup = callPackage ../development/libraries/java/cup {
    jdk = jdk8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  jdom = callPackage ../development/libraries/java/jdom { };

  jffi = callPackage ../development/libraries/java/jffi { };

  jflex = callPackage ../development/libraries/java/jflex { };

  lombok = callPackage ../development/libraries/java/lombok { };

  lucene = callPackage ../development/libraries/java/lucene { };

  lucenepp = callPackage ../development/libraries/lucene++ { };

  mockobjects = callPackage ../development/libraries/java/mockobjects { };

  saxonb = saxonb_8_8;

  inherit (callPackages ../development/libraries/java/saxon {
    jre = jre_headless;
    jre8 = jre8_headless;
  })
    saxon
    saxonb_8_8
    saxonb_9_1
    saxon_9-he
    saxon-he;

  smack = callPackage ../development/libraries/java/smack { };

  swt = callPackage ../development/libraries/java/swt { };
  swt_jdk8 = callPackage ../development/libraries/java/swt {
    jdk = jdk8;
  };


  ### DEVELOPMENT / LIBRARIES / JAVASCRIPT

  nodehun = callPackage ../development/tools/nodehun { };

  yuicompressor = callPackage ../development/tools/yuicompressor { };

  ### DEVELOPMENT / BOWER MODULES (JAVASCRIPT)

  buildBowerComponents = callPackage ../development/bower-modules/generic { };

  ### DEVELOPMENT / GO

  # the unversioned attributes should always point to the same go version
  go = go_1_22;
  buildGoModule = buildGo122Module;
  buildGoPackage = buildGo122Package;

  # requires a newer Apple SDK
  go_1_21 = darwin.apple_sdk_11_0.callPackage ../development/compilers/go/1.21.nix {
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation Security;
  };
  buildGo121Module = darwin.apple_sdk_11_0.callPackage ../build-support/go/module.nix {
    go = buildPackages.go_1_21;
  };
  buildGo121Package = darwin.apple_sdk_11_0.callPackage ../build-support/go/package.nix {
    go = buildPackages.go_1_21;
  };

  # requires a newer Apple SDK
  go_1_22 = darwin.apple_sdk_11_0.callPackage ../development/compilers/go/1.22.nix {
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation Security;
  };
  buildGo122Module = darwin.apple_sdk_11_0.callPackage ../build-support/go/module.nix {
    go = buildPackages.go_1_22;
  };
  buildGo122Package = darwin.apple_sdk_11_0.callPackage ../build-support/go/package.nix {
    go = buildPackages.go_1_22;
  };

  leaps = callPackage ../development/tools/leaps { };

  ### DEVELOPMENT / JAVA MODULES

  javaPackages = recurseIntoAttrs (callPackage ./java-packages.nix { });

  ### DEVELOPMENT / LISP MODULES

  asdf = callPackage ../development/lisp-modules/asdf {
    texLive = null;
  };

  # QuickLisp minimal version
  asdf_2_26 = callPackage ../development/lisp-modules/asdf/2.26.nix {
    texLive = null;
  };
  # Currently most popular
  asdf_3_1 = callPackage ../development/lisp-modules/asdf/3.1.nix {
    texLive = null;
  };
  # Latest
  asdf_3_3 = callPackage ../development/lisp-modules/asdf/3.3.nix {
    texLive = null;
  };

  wrapLisp = callPackage ../development/lisp-modules/nix-cl.nix {};

  # Armed Bear Common Lisp
  abcl = wrapLisp {
    pkg = callPackage ../development/compilers/abcl { };
    faslExt = "abcl";
  };

  # Clozure Common Lisp
  ccl = wrapLisp {
    pkg = callPackage ../development/compilers/ccl {
      inherit (buildPackages.darwin) bootstrap_cmds;
    };
    faslExt = "lx64fsl";
  };

  # Clasp Common Lisp
  clasp-common-lisp = wrapLisp {
    pkg = callPackage ../development/compilers/clasp { };
    faslExt = "fasp";
  };

  # CLISP
  clisp = wrapLisp {
    pkg = callPackage ../development/interpreters/clisp { };
    faslExt = "fas";
    flags = ["-E" "UTF-8"];
  };

  wrapLispi686Linux = pkgsi686Linux.callPackage ../development/lisp-modules/nix-cl.nix {};

  # CMU Common Lisp
  cmucl_binary = wrapLispi686Linux {
    pkg = pkgsi686Linux.callPackage ../development/compilers/cmucl/binary.nix { };
    faslExt = "sse2f";
    program = "lisp";
  };

  # Embeddable Common Lisp
  ecl = wrapLisp {
    pkg = callPackage ../development/compilers/ecl { };
    faslExt = "fas";
  };
  ecl_16_1_2 = wrapLisp {
    pkg = callPackage ../development/compilers/ecl/16.1.2.nix { };
    faslExt = "fas";
  };

  # GNU Common Lisp
  gcl = wrapLisp {
    pkg = callPackage ../development/compilers/gcl { };
    faslExt = "o";
  };

  # ManKai Common Lisp
  mkcl = wrapLisp {
    pkg = callPackage ../development/compilers/mkcl {};
    faslExt = "fas";
  };

  # Steel Bank Common Lisp
  sbcl_2_4_3 = wrapLisp {
    pkg = callPackage ../development/compilers/sbcl { version = "2.4.3"; };
    faslExt = "fasl";
    flags = [ "--dynamic-space-size" "3000" ];
  };
  sbcl_2_4_4 = wrapLisp {
    pkg = callPackage ../development/compilers/sbcl { version = "2.4.4"; };
    faslExt = "fasl";
    flags = [ "--dynamic-space-size" "3000" ];
  };
  sbcl = sbcl_2_4_4;

  sbclPackages = recurseIntoAttrs sbcl.pkgs;

  ### DEVELOPMENT / PERL MODULES

  perlInterpreters = import ../development/interpreters/perl { inherit callPackage; };
  inherit (perlInterpreters) perl536 perl538;

  perl536Packages = recurseIntoAttrs perl536.pkgs;
  perl538Packages = recurseIntoAttrs perl538.pkgs;

  perl = perl538;
  perlPackages = perl538Packages;

  ack = perlPackages.ack;

  perlcritic = perlPackages.PerlCritic;

  sqitchMysql = (callPackage ../development/tools/misc/sqitch {
    mysqlSupport = true;
  }).overrideAttrs { pname = "sqitch-mysql"; };

  sqitchPg = (callPackage ../development/tools/misc/sqitch {
    postgresqlSupport = true;
  }).overrideAttrs { pname = "sqitch-pg"; };

  ### DEVELOPMENT / R MODULES

  R = darwin.apple_sdk_11_0.callPackage ../applications/science/math/R {
    # TODO: split docs into a separate output
    withRecommendedPackages = false;
    inherit (darwin.apple_sdk_11_0.frameworks) Cocoa Foundation;
    inherit (darwin) libobjc;
  };

  rWrapper = callPackage ../development/r-modules/wrapper.nix {
    recommendedPackages = with rPackages; [
      boot class cluster codetools foreign KernSmooth lattice MASS
      Matrix mgcv nlme nnet rpart spatial survival
    ];
    # Override this attribute to register additional libraries.
    packages = [];
  };

  radianWrapper = callPackage ../development/r-modules/wrapper-radian.nix {
    recommendedPackages = with rPackages; [
      boot class cluster codetools foreign KernSmooth lattice MASS
      Matrix mgcv nlme nnet rpart spatial survival
    ];
    radian = python3Packages.radian;
    # Override this attribute to register additional libraries.
    packages = [];
    # Override this attribute if you want to expose R with the same set of
    # packages as specified in radian
    wrapR = false;
  };

  rstudioWrapper = libsForQt5.callPackage ../development/r-modules/wrapper-rstudio.nix {
    recommendedPackages = with rPackages; [
      boot class cluster codetools foreign KernSmooth lattice MASS
      Matrix mgcv nlme nnet rpart spatial survival
    ];
    # Override this attribute to register additional libraries.
    packages = [];
  };

  rstudioServerWrapper = rstudioWrapper.override { rstudio = rstudio-server; };

  rPackages = (dontRecurseIntoAttrs (callPackage ../development/r-modules {
    overrides = (config.rPackageOverrides or (_: {})) pkgs;
  })) // { __attrsFailEvaluation = true; };

  ### SERVERS

  _389-ds-base = callPackage ../servers/ldap/389 { };

  _5etools = callPackage ../servers/web-apps/5etools { };

  adguardhome = callPackage ../servers/adguardhome { };

  alerta = callPackage ../servers/monitoring/alerta/client.nix { };

  alerta-server = callPackage ../servers/monitoring/alerta { };

  allmark = callPackage ../servers/allmark { };

  alps = callPackage ../servers/alps { };

  anuko-time-tracker = callPackage ../servers/web-apps/anuko-time-tracker { };

  apache-directory-server = callPackage ../servers/ldap/apache-directory-server { };

  apacheHttpd_2_4 = callPackage ../servers/http/apache-httpd/2.4.nix {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };
  apacheHttpd = apacheHttpd_2_4;

  apacheHttpdPackagesFor = apacheHttpd: self: let callPackage = newScope self; in {
    inherit apacheHttpd;

    mod_auth_mellon = callPackage ../servers/http/apache-modules/mod_auth_mellon { };

    # Redwax collection
    mod_ca = callPackage ../servers/http/apache-modules/mod_ca { };
    mod_crl = callPackage ../servers/http/apache-modules/mod_crl { };
    mod_csr = callPackage ../servers/http/apache-modules/mod_csr { };
    mod_cspnonce = callPackage ../servers/http/apache-modules/mod_cspnonce { };
    mod_ocsp = callPackage ../servers/http/apache-modules/mod_ocsp{ };
    mod_scep = callPackage ../servers/http/apache-modules/mod_scep { };
    mod_pkcs12 = callPackage ../servers/http/apache-modules/mod_pkcs12 { };
    mod_spkac= callPackage ../servers/http/apache-modules/mod_spkac { };
    mod_timestamp = callPackage ../servers/http/apache-modules/mod_timestamp { };

    mod_dnssd = callPackage ../servers/http/apache-modules/mod_dnssd { };


    mod_perl = callPackage ../servers/http/apache-modules/mod_perl { };

    mod_fastcgi = callPackage ../servers/http/apache-modules/mod_fastcgi { };

    mod_python = callPackage ../servers/http/apache-modules/mod_python { };

    mod_tile = callPackage ../servers/http/apache-modules/mod_tile { };

    mod_wsgi3 = callPackage ../servers/http/apache-modules/mod_wsgi { };

    mod_itk = callPackage ../servers/http/apache-modules/mod_itk { };

    mod_mbtiles = callPackage ../servers/http/apache-modules/mod_mbtiles { };

    php = pkgs.php.override { inherit apacheHttpd; };

    subversion = pkgs.subversion.override { httpServer = true; inherit apacheHttpd; };
  } // lib.optionalAttrs config.allowAliases {
    mod_evasive = throw "mod_evasive is not supported on Apache httpd 2.4";
    mod_wsgi  = self.mod_wsgi2;
    mod_wsgi2 = throw "mod_wsgi2 has been removed since Python 2 is EOL. Use mod_wsgi3 instead";
  };

  apacheHttpdPackages_2_4 = recurseIntoAttrs (apacheHttpdPackagesFor apacheHttpd_2_4 apacheHttpdPackages_2_4);
  apacheHttpdPackages = apacheHttpdPackages_2_4;

  appdaemon = callPackage ../servers/home-assistant/appdaemon.nix { };

  asouldocs = callPackage ../servers/asouldocs { };

  atlassian-bamboo = callPackage ../servers/atlassian/bamboo.nix { };
  atlassian-confluence = callPackage ../servers/atlassian/confluence.nix { };
  atlassian-crowd = callPackage ../servers/atlassian/crowd.nix { };
  atlassian-jira = callPackage ../servers/atlassian/jira.nix { };

  cadvisor = callPackage ../servers/monitoring/cadvisor { };

  cassandra_3_0 = callPackage ../servers/nosql/cassandra/3.0.nix {
    jre = jre8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
    python = python2;
  };
  cassandra_3_11 = callPackage ../servers/nosql/cassandra/3.11.nix {
    jre = jre8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
    python = python2;
  };
  cassandra_4 = callPackage ../servers/nosql/cassandra/4.nix {
    # Effective Cassandra 4.0.2 there is full Java 11 support
    #  -- https://cassandra.apache.org/doc/latest/cassandra/new/java11.html
    jre = pkgs.jdk11_headless;
    python = python3;
  };
  cassandra = cassandra_4;

  cassandra-cpp-driver = callPackage ../development/libraries/cassandra-cpp-driver/default.nix { };

  apache-jena = callPackage ../servers/nosql/apache-jena/binary.nix {
    java = jre;
  };

  apache-jena-fuseki = callPackage ../servers/nosql/apache-jena/fuseki-binary.nix {
    java = jre;
  };

  apcupsd = callPackage ../servers/apcupsd { };

  inherit (callPackages ../servers/asterisk { })
    asterisk asterisk-stable asterisk-lts
    asterisk_18 asterisk_20;

  asterisk-module-sccp = callPackage ../servers/asterisk/sccp { };

  asterisk-ldap = lowPrio (asterisk.override { ldapSupport = true; });

  baserow = callPackage ../servers/baserow { };

  bftpd = callPackage ../servers/ftp/bftpd { };

  bind = callPackage ../servers/dns/bind { };
  dnsutils = bind.dnsutils;
  dig = lib.addMetaAttrs { mainProgram = "dig"; } bind.dnsutils;

  bird = callPackage ../servers/bird { };

  bird-lg = callPackage ../servers/bird-lg { };

  birdwatcher = callPackage ../servers/birdwatcher { };

  bloat = callPackage ../servers/bloat { };

  bosun = callPackage ../servers/monitoring/bosun { };

  cayley = callPackage ../servers/cayley { };

  cgiserver = callPackage ../servers/http/cgiserver { };

  charybdis = callPackage ../servers/irc/charybdis {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  clamsmtp = callPackage ../servers/mail/clamsmtp { };

  clickhouse = callPackage ../servers/clickhouse {
    llvmPackages = llvmPackages_16;
  };

  clickhouse-cli = with python3Packages; toPythonApplication clickhouse-cli;

  clickhouse-backup = callPackage ../development/tools/database/clickhouse-backup { };

  codeowners = callPackage ../development/tools/codeowners { };

  couchdb3 = callPackage ../servers/http/couchdb/3.nix { };

  dcnnt = python3Packages.callPackage ../servers/dcnnt { };

  deconz = qt5.callPackage ../servers/deconz { };

  dendrite = callPackage ../servers/dendrite { };

  dex-oidc = callPackage ../servers/dex { };

  dex2jar = callPackage ../development/tools/java/dex2jar { };

  doh-proxy-rust = callPackage ../servers/dns/doh-proxy-rust {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  dgraph = callPackage ../servers/dgraph { };

  dico = callPackage ../servers/dico { };

  dict = callPackage ../servers/dict {
    libmaa = callPackage ../servers/dict/libmaa.nix { };
  };

  dictdDBs = recurseIntoAttrs (callPackages ../servers/dict/dictd-db.nix {});

  dictDBCollector = callPackage ../servers/dict/dictd-db-collector.nix { };

  diod = callPackage ../servers/diod { lua = lua5_1; };

  directx-headers = callPackage ../development/libraries/directx-headers { };

  directx-shader-compiler = callPackage ../tools/graphics/directx-shader-compiler {
    # https://github.com/NixOS/nixpkgs/issues/216294
    stdenv = if stdenv.cc.isGNU && stdenv.isi686 then gcc11Stdenv else stdenv;
  };

  dkimproxy = callPackage ../servers/mail/dkimproxy { };

  dmarc-metrics-exporter = callPackage ../servers/monitoring/prometheus/dmarc-metrics-exporter { };

  dmlive = callPackage ../applications/video/dmlive {
    inherit (darwin) configd;
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  do-agent = callPackage ../servers/monitoring/do-agent { };

  dodgy = with python3Packages; toPythonApplication dodgy;

  dovecot = callPackage ../servers/mail/dovecot { };
  dovecot_pigeonhole = callPackage ../servers/mail/dovecot/plugins/pigeonhole { };
  dovecot_fts_xapian = callPackage ../servers/mail/dovecot/plugins/fts_xapian { };

  dspam = callPackage ../servers/mail/dspam { };

  engelsystem = callPackage ../servers/web-apps/engelsystem { };

  envoy = callPackage ../servers/http/envoy {
    jdk = openjdk11_headless;
    gn = gn1924;
  };

  ergochat = callPackage ../servers/irc/ergochat { };

  etcd = etcd_3_5;
  etcd_3_4 = callPackage ../servers/etcd/3.4.nix { };
  etcd_3_5 = callPackage ../servers/etcd/3.5 { };

  ejabberd = callPackage ../servers/xmpp/ejabberd { erlang = erlang_24; };

  exhaustive = callPackage ../development/tools/exhaustive { };

  hyp = callPackage ../servers/http/hyp { };

  peering-manager = callPackage ../servers/web-apps/peering-manager { };

  podgrab = callPackage ../servers/misc/podgrab { };

  portunus = callPackage ../servers/portunus { };

  prosody = callPackage ../servers/xmpp/prosody {
    withExtraLibs = [];
    withExtraLuaPackages = _: [];
  };

  prosody-filer = callPackage ../servers/xmpp/prosody-filer { };

  biboumi = callPackage ../servers/xmpp/biboumi { };

  elasticmq-server-bin = callPackage ../servers/elasticmq-server-bin {
    jre = jre8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
    jdk = jdk8; # TODO: remove override https://github.com/NixOS/nixpkgs/pull/89731
  };

  eventstore = callPackage ../servers/nosql/eventstore { };

  exabgp = callPackage ../servers/networking/exabgp { };

  rustus = callPackage ../servers/networking/rustus {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  exim = callPackage ../servers/mail/exim { };

  fastnetmon-advanced = callPackage ../servers/fastnetmon-advanced { };

  fcgiwrap = callPackage ../servers/fcgiwrap { };

  fedigroups = callPackage ../servers/fedigroups {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ferretdb = callPackage ../servers/nosql/ferretdb { };

  felix = callPackage ../servers/felix { };

  felix_remoteshell = callPackage ../servers/felix/remoteshell.nix { };

  inherit (callPackages ../servers/firebird { }) firebird_4 firebird_3 firebird_2_5 firebird;

  freeradius = callPackage ../servers/freeradius { };

  freshrss = callPackage ../servers/web-apps/freshrss { };

  freeswitch = callPackage ../servers/sip/freeswitch {
    inherit (darwin.apple_sdk.frameworks) SystemConfiguration;
  };

  fusionInventory = callPackage ../servers/monitoring/fusion-inventory { };

  gamehub = callPackage ../games/gamehub { };

  gatling = callPackage ../servers/http/gatling { };

  glabels = callPackage ../applications/graphics/glabels { };

  glabels-qt = libsForQt5.callPackage ../applications/graphics/glabels-qt { };

  nats-server = callPackage ../servers/nats-server { };

  go-autoconfig = callPackage ../servers/go-autoconfig { };

  go-cqhttp = callPackage ../servers/go-cqhttp { };

  gofish = callPackage ../servers/gopher/gofish { };

  grafana = callPackage ../servers/monitoring/grafana { };
  grafanaPlugins = callPackages ../servers/monitoring/grafana/plugins { };

  grafana-agent = callPackage ../servers/monitoring/grafana-agent {
    buildGoModule = buildGo122Module;
  };

  grafana-loki = callPackage ../servers/monitoring/loki { };
  promtail = callPackage ../servers/monitoring/loki/promtail.nix { };

  mimir = callPackage ../servers/monitoring/mimir { };

  phlare = callPackage ../servers/monitoring/phlare { };

  grafana_reporter = callPackage ../servers/monitoring/grafana-reporter { };

  grafana-image-renderer = callPackage ../servers/monitoring/grafana-image-renderer { };

  grafana-dash-n-grab = callPackage ../servers/monitoring/grafana-dash-n-grab { };

  gerbera = callPackage ../servers/gerbera { };

  gobetween = callPackage ../servers/gobetween { };

  gobgpd = callPackage ../servers/misc/gobgpd { };

  graph-cli = callPackage ../tools/graphics/graph-cli { };

  h2o = callPackage ../servers/http/h2o { };

  haka = callPackage ../tools/security/haka { };

  hashi-ui = callPackage ../servers/hashi-ui { };

  hashi-up = callPackage ../applications/networking/cluster/hashi-up { };

  hasura-graphql-engine = haskell.lib.compose.justStaticExecutables haskell.packages.ghc810.graphql-engine;

  hasura-cli = callPackage ../servers/hasura/cli.nix { };

  inherit (callPackage ../servers/hbase {}) hbase_2_4 hbase_2_5 hbase_3_0;
  hbase2 = hbase_2_5;
  hbase3 = hbase_3_0;
  hbase = hbase2; # when updating, point to the latest stable release

  headphones = callPackage ../servers/headphones { };

  hiawatha = callPackage ../servers/http/hiawatha { };

  hiraeth = callPackage ../servers/hiraeth { };

  hoard = callPackage ../tools/misc/hoard { };

  home-assistant = callPackage ../servers/home-assistant { };

  buildHomeAssistantComponent = callPackage ../servers/home-assistant/build-custom-component { };
  home-assistant-custom-components = lib.recurseIntoAttrs
    (callPackage ../servers/home-assistant/custom-components {
      inherit (home-assistant.python.pkgs) callPackage;
    });
  home-assistant-custom-lovelace-modules = lib.recurseIntoAttrs
    (callPackage ../servers/home-assistant/custom-lovelace-modules {});

  home-assistant-cli = callPackage ../servers/home-assistant/cli.nix { };

  home-assistant-component-tests = recurseIntoAttrs home-assistant.tests.components;

  honk = callPackage ../servers/honk { };

  hqplayerd = callPackage ../servers/hqplayerd { };

  https-dns-proxy = callPackage ../servers/dns/https-dns-proxy { };

  hydron = callPackage ../servers/hydron { };

  hyprspace = callPackage ../applications/networking/hyprspace {
    inherit (darwin) iproute2mac;
  };

  ic-keysmith = callPackage ../tools/security/ic-keysmith { };

  icecream = callPackage ../servers/icecream { };

  icingaweb2-ipl = callPackage ../servers/icingaweb2/ipl.nix { };
  icingaweb2-thirdparty = callPackage ../servers/icingaweb2/thirdparty.nix { };
  icingaweb2 = callPackage ../servers/icingaweb2 { };
  icingaweb2Modules = {
    theme-april = callPackage ../servers/icingaweb2/theme-april { };
    theme-lsd = callPackage ../servers/icingaweb2/theme-lsd { };
    theme-particles = callPackage ../servers/icingaweb2/theme-particles { };
    theme-snow = callPackage ../servers/icingaweb2/theme-snow { };
    theme-spring = callPackage ../servers/icingaweb2/theme-spring { };
  };

  inspircd = callPackage ../servers/irc/inspircd { };

  inspircdMinimal = inspircd.override { extraModules = []; };

  imaginary = callPackage ../servers/imaginary { };

  imgproxy = callPackage ../servers/imgproxy { };

  irccat = callPackage ../servers/irc/irccat { };

  ircdog = callPackage ../applications/networking/irc/ircdog { };

  ircdHybrid = callPackage ../servers/irc/ircd-hybrid { };

  janus-gateway = callPackage ../servers/janus-gateway { };

  janusgraph = callPackage ../servers/nosql/janusgraph { };

  jboss = callPackage ../servers/http/jboss { };

  jboss_mysql_jdbc = callPackage ../servers/http/jboss/jdbc/mysql { };

  jetty = jetty_12;
  jetty_12 = callPackage ../servers/http/jetty/12.x.nix { };
  jetty_11 = callPackage ../servers/http/jetty/11.x.nix { };

  jibri = callPackage ../servers/jibri { };

  jicofo = callPackage ../servers/jicofo { };

  jitsi-excalidraw = callPackage ../servers/jitsi-excalidraw { };

  jigasi = callPackage ../servers/jigasi { };

  jitsi-meet = callPackage ../servers/web-apps/jitsi-meet { };

  jitsi-meet-prosody = callPackage ../misc/jitsi-meet-prosody { };

  jitsi-videobridge = callPackage ../servers/jitsi-videobridge { };

  kamailio = callPackage ../servers/sip/kamailio { };

  kapowbang = callPackage ../servers/kapowbang { };

  keycloak = callPackage ../servers/keycloak { };

  knot-dns = callPackage ../servers/dns/knot-dns { };
  knot-resolver = callPackage ../servers/dns/knot-resolver {
    systemd = systemdMinimal; # in closure already anyway
  };

  rdkafka = callPackage ../development/libraries/rdkafka { };

  leafnode = callPackage ../servers/news/leafnode { };

  leafnode1 = callPackage ../servers/news/leafnode/1.nix { };

  lemmy-server = callPackage ../servers/web-apps/lemmy/server.nix {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  lemmy-ui = callPackage ../servers/web-apps/lemmy/ui.nix {
    nodejs = nodejs_18;
  };

  lightgbm = callPackage ../development/libraries/lightgbm { };

  lighttpd = callPackage ../servers/http/lighttpd { };

  linx-server = callPackage ../servers/web-apps/linx-server { };

  livepeer = callPackage ../servers/livepeer { };

  lldap = callPackage ../servers/ldap/lldap { };

  lwan = callPackage ../servers/http/lwan { };

  labelImg = callPackage ../applications/science/machine-learning/labelimg { };

  mackerel-agent = callPackage ../servers/monitoring/mackerel-agent { };

  mailmanPackages = callPackage ../servers/mail/mailman { };
  inherit (mailmanPackages) mailman mailman-hyperkitty;
  mailman-web = mailmanPackages.web;

  mailman-rss = callPackage ../tools/misc/mailman-rss { };

  listadmin = callPackage ../applications/networking/listadmin { };

  maker-panel = callPackage ../tools/misc/maker-panel { };

  mastodon = callPackage ../servers/mastodon {
    nodejs-slim = nodejs-slim_20;
    ruby = ruby_3_2;
  };

  gotosocial = callPackage ../servers/gotosocial { };

  materialize = callPackage ../servers/sql/materialize {
    inherit (buildPackages.darwin) bootstrap_cmds;
    inherit (darwin.apple_sdk.frameworks) DiskArbitration Foundation;
  };

  matrix-alertmanager = callPackage ../servers/monitoring/matrix-alertmanager { };

  mattermost = callPackage ../servers/mattermost { };
  matterircd = callPackage ../servers/mattermost/matterircd.nix { };
  matterbridge = callPackage ../servers/matterbridge { };

  mattermost-desktop = callPackage ../applications/networking/instant-messengers/mattermost-desktop { };

  memcached = callPackage ../servers/memcached { };

  merecat = callPackage ../servers/http/merecat { };

  meteor = callPackage ../servers/meteor { };

  micronaut = callPackage ../development/tools/micronaut { };

  minio = callPackage ../servers/minio { };
  # Keep around to allow people to migrate their data from the old legacy fs format
  # https://github.com/minio/minio/releases/tag/RELEASE.2022-10-29T06-21-33Z
  minio_legacy_fs = callPackage ../servers/minio/legacy_fs.nix { };

  mkchromecast = libsForQt5.callPackage ../applications/networking/mkchromecast { };

  inherit (callPackages ../servers/mpd {
    inherit (darwin.apple_sdk.frameworks) AudioToolbox AudioUnit;
  }) mpd mpd-small mpdWithFeatures;

  mpdscribble = callPackage ../tools/misc/mpdscribble { };

  mtprotoproxy = python3.pkgs.callPackage ../servers/mtprotoproxy { };

  micro-httpd = callPackage ../servers/http/micro-httpd { };

  miniHttpd = callPackage ../servers/http/mini-httpd { };

  mlflow-server = callPackage ../servers/mlflow-server { };

  mlmmj = callPackage ../servers/mail/mlmmj { };

  moodle = callPackage ../servers/web-apps/moodle { };

  moodle-utils = callPackage ../servers/web-apps/moodle/moodle-utils.nix { };

  morty = callPackage ../servers/web-apps/morty { };

  inherit (callPackage ../applications/networking/mullvad { })
    mullvad;

  mullvad-vpn = callPackage ../applications/networking/mullvad-vpn { };

  mullvad-closest = with python3Packages; toPythonApplication mullvad-closest;

  mycorrhiza = callPackage ../servers/mycorrhiza { };

  napalm = with python3Packages; toPythonApplication (
    napalm.overridePythonAttrs (attrs: {
      # add community frontends that depend on the napalm python package
      propagatedBuildInputs = attrs.propagatedBuildInputs ++ [
        napalm-hp-procurve
      ];
    })
  );

  nas = callPackage ../servers/nas { };

  nats-streaming-server = callPackage ../servers/nats-streaming-server { };

  neard = callPackage ../servers/neard { };

  networkaudiod = callPackage ../servers/networkaudiod { };

  unifiedpush-common-proxies = callPackage ../servers/unifiedpush-common-proxies { };

  unit = callPackage ../servers/http/unit { };

  ncdns = callPackage ../servers/dns/ncdns { };

  nginx = nginxStable;

  nginx-doc = callPackage ../data/documentation/nginx-doc { };

  nginxQuic = callPackage ../servers/http/nginx/quic.nix {
    zlib = zlib-ng.override { withZlibCompat = true; };
    withPerl = false;
    # We don't use `with` statement here on purpose!
    # See https://github.com/NixOS/nixpkgs/pull/10474#discussion_r42369334
    modules = [ nginxModules.rtmp nginxModules.dav nginxModules.moreheaders ];
    # Use latest boringssl to allow http3 support
    openssl = quictls;
  };

  nginxStable = callPackage ../servers/http/nginx/stable.nix {
    zlib = zlib-ng.override { withZlibCompat = true; };
    withPerl = false;
    # We don't use `with` statement here on purpose!
    # See https://github.com/NixOS/nixpkgs/pull/10474#discussion_r42369334
    modules = [ nginxModules.rtmp nginxModules.dav nginxModules.moreheaders ];
  };

  nginxMainline = callPackage ../servers/http/nginx/mainline.nix {
    zlib = zlib-ng.override { withZlibCompat = true; };
    withKTLS = true;
    withPerl = false;
    # We don't use `with` statement here on purpose!
    # See https://github.com/NixOS/nixpkgs/pull/10474#discussion_r42369334
    modules = [ nginxModules.dav nginxModules.moreheaders ];
  };

  nginxModules = recurseIntoAttrs (callPackage ../servers/http/nginx/modules.nix { });

  # We should move to dynmaic modules and create a nginxFull package with all modules
  nginxShibboleth = nginxStable.override {
    modules = [ nginxModules.rtmp nginxModules.dav nginxModules.moreheaders nginxModules.shibboleth ];
  };

  libmodsecurity = callPackage ../tools/security/libmodsecurity {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  ngircd = callPackage ../servers/irc/ngircd { };

  nix-binary-cache = callPackage ../servers/http/nix-binary-cache { };

  nix-tour = callPackage ../applications/misc/nix-tour { };

  noseyparker = callPackage ../tools/security/noseyparker { };

  nosqli = callPackage ../tools/security/nosqli { };

  nsd = callPackage ../servers/dns/nsd (config.nsd or {});

  nsq = callPackage ../servers/nsq { };

  oauth2-proxy = callPackage ../servers/oauth2-proxy { };

  olaris-server = callPackage ../servers/olaris { };

  onlyoffice-documentserver = callPackage ../servers/onlyoffice-documentserver { };

  outline = callPackage ../servers/web-apps/outline (lib.fix (super: {
    yarn = yarn.override { inherit (super) nodejs; };
    nodejs = nodejs_20;
  }));

  openbgpd = callPackage ../servers/openbgpd { };

  openafs = callPackage ../servers/openafs/1.8 { };

  openresty = callPackage ../servers/http/openresty {
    withPerl = false;
    modules = [];
  };

  opensmtpd = callPackage ../servers/mail/opensmtpd { };
  opensmtpd-extras = callPackage ../servers/mail/opensmtpd/extras.nix { };
  opensmtpd-filter-rspamd = callPackage ../servers/mail/opensmtpd/filter-rspamd.nix { };
  opensmtpd-filter-dkimsign = callPackage ../servers/mail/opensmtpd/filter-dkimsign { };
  libopensmtpd = callPackage ../servers/mail/opensmtpd/libopensmtpd { };

  openxr-loader = callPackage ../development/libraries/openxr-loader { };

  osrm-backend = callPackage ../servers/osrm-backend { };

  oven-media-engine = callPackage ../servers/misc/oven-media-engine { };

  p910nd = callPackage ../servers/p910nd { };

  petidomo = callPackage ../servers/mail/petidomo { };

  pict-rs = callPackage ../servers/web-apps/pict-rs { };

  pict-rs_0_3 = callPackage ../servers/web-apps/pict-rs/0.3.nix {
    inherit (darwin.apple_sdk.frameworks) Security;
    ffmpeg = ffmpeg_4;
  };

  popa3d = callPackage ../servers/mail/popa3d { };

  postfix = callPackage ../servers/mail/postfix { };

  postfixadmin = callPackage ../servers/postfixadmin { };

  postsrsd = callPackage ../servers/mail/postsrsd { };

  rspamd = callPackage ../servers/mail/rspamd { };

  pfixtools = callPackage ../servers/mail/postfix/pfixtools.nix { };

  pflogsumm = callPackage ../servers/mail/postfix/pflogsumm.nix { };

  pomerium = callPackage ../servers/http/pomerium { };
  pomerium-cli = callPackage ../tools/security/pomerium-cli { };

  postgrey = callPackage ../servers/mail/postgrey { };

  pshs = callPackage ../servers/http/pshs { };

  quark = callPackage ../servers/http/quark { };

  smtprelay = callPackage ../servers/mail/smtprelay { };

  soft-serve = callPackage ../servers/soft-serve { };

  sympa = callPackage ../servers/mail/sympa { };

  system-sendmail = lowPrio (callPackage ../servers/mail/system-sendmail { });

  # PulseAudio daemons

  hsphfpd = callPackage ../servers/pulseaudio/hsphfpd.nix { };

  pulseaudio = callPackage ../servers/pulseaudio {
    inherit (darwin.apple_sdk.frameworks) CoreServices AudioUnit Cocoa CoreAudio;
  };

  qpaeq = libsForQt5.callPackage ../servers/pulseaudio/qpaeq.nix { };

  pulseaudioFull = pulseaudio.override {
    x11Support = true;
    jackaudioSupport = true;
    airtunesSupport = true;
    bluetoothSupport = true;
    advancedBluetoothCodecs = true;
    remoteControlSupport = true;
    zeroconfSupport = true;
  };

  libpulseaudio = pulseaudio.override {
    libOnly = true;
  };

  apulse = callPackage ../misc/apulse { };

  libpressureaudio = callPackage ../misc/apulse/pressureaudio.nix { };

  libcardiacarrest = callPackage ../misc/libcardiacarrest { };

  easyeffects = callPackage ../applications/audio/easyeffects {
    # Fix crashes with speexdsp effects
    speexdsp = speexdsp.override { withFftw3 = false; };
  };

  pulseeffects-legacy = callPackage ../applications/audio/pulseeffects-legacy { };

  tomcat_connectors = callPackage ../servers/http/apache-modules/tomcat-connectors { };

  tomcat-native = callPackage ../servers/http/tomcat/tomcat-native.nix { };

  pies = callPackage ../servers/pies { };

  rpcbind = callPackage ../servers/rpcbind { };

  rpcsvc-proto = callPackage ../tools/misc/rpcsvc-proto { };

  libmysqlclient = libmysqlclient_3_3;
  libmysqlclient_3_1 = mariadb-connector-c_3_1;
  libmysqlclient_3_2 = mariadb-connector-c_3_2;
  libmysqlclient_3_3 = mariadb-connector-c_3_3;
  mariadb-connector-c = mariadb-connector-c_3_3;
  mariadb-connector-c_3_1 = callPackage ../servers/sql/mariadb/connector-c/3_1.nix { };
  mariadb-connector-c_3_2 = callPackage ../servers/sql/mariadb/connector-c/3_2.nix { };
  mariadb-connector-c_3_3 = callPackage ../servers/sql/mariadb/connector-c/3_3.nix { };

  mariadb-galera = callPackage ../servers/sql/mariadb/galera { };

  inherit (import ../servers/sql/mariadb pkgs)
    mariadb_105
    mariadb_106
    mariadb_1011
    mariadb_110
  ;
  mariadb = mariadb_1011;
  mariadb-embedded = mariadb.override { withEmbedded = true; };

  mongodb = hiPrio mongodb-6_0;

  mongodb-5_0 = callPackage ../servers/nosql/mongodb/5.0.nix {
    sasl = cyrus_sasl;
    boost = boost179.override { enableShared = false; };
    inherit (darwin) cctools;
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security;
  };

  mongodb-6_0 = darwin.apple_sdk_11_0.callPackage ../servers/nosql/mongodb/6.0.nix {
    sasl = cyrus_sasl;
    boost = boost178.override { enableShared = false; };
    inherit (darwin) cctools;
    inherit (darwin.apple_sdk.frameworks) CoreFoundation Security;
    stdenv = if stdenv.isDarwin then
      darwin.apple_sdk_11_0.stdenv.override (old: {
        hostPlatform = old.hostPlatform // { darwinMinVersion = "10.14"; };
        buildPlatform = old.buildPlatform // { darwinMinVersion = "10.14"; };
        targetPlatform = old.targetPlatform // { darwinMinVersion = "10.14"; };
      }) else
      if stdenv.cc.isClang then llvmPackages.stdenv else stdenv;
  };

  immudb = callPackage ../servers/nosql/immudb { };

  influxdb = callPackage ../servers/nosql/influxdb { };
  influxdb2-server = callPackage ../servers/nosql/influxdb2 { };
  influxdb2-cli = callPackage ../servers/nosql/influxdb2/cli.nix { };
  influxdb2-token-manipulator = callPackage ../servers/nosql/influxdb2/token-manipulator.nix { };
  influxdb2-provision = callPackage ../servers/nosql/influxdb2/provision.nix { };
  # For backwards compatibility with older versions of influxdb2,
  # which bundled the server and CLI into the same derivation. Will be
  # removed in a few releases.
  influxdb2 = callPackage ../servers/nosql/influxdb2/combined.nix { };

  mysql80 = callPackage ../servers/sql/mysql/8.0.x.nix {
    inherit (darwin) cctools developer_cmds DarwinTools;
    inherit (darwin.apple_sdk.frameworks) CoreServices;
    boost = boost177; # Configure checks for specific version.
    icu = icu69;
    protobuf = protobuf_21;
  };

  mysql_jdbc = callPackage ../servers/sql/mysql/jdbc { };

  mssql_jdbc = callPackage ../servers/sql/mssql/jdbc { };
  jtds_jdbc = callPackage ../servers/sql/mssql/jdbc/jtds.nix { };

  azuredatastudio = callPackage ../applications/misc/azuredatastudio { };

  miniflux = callPackage ../servers/miniflux {
    buildGoModule = buildGo122Module;
  };

  mir = callPackage ../servers/mir { };

  miriway = callPackage ../applications/window-managers/miriway { };

  icinga2 = callPackage ../servers/monitoring/icinga2 { };

  icinga2-agent = callPackage ../servers/monitoring/icinga2 {
    nameSuffix = "-agent";
    withMysql = false;
    withNotification = false;
    withIcingadb = false;
  };

  munin = callPackage ../servers/monitoring/munin { };

  nagios = callPackage ../servers/monitoring/nagios { };

  monitoring-plugins = callPackage ../servers/monitoring/plugins { };

  inherit (callPackage ../servers/monitoring/plugins/labs_consol_de.nix { })
    check-mssql-health
    check-nwc-health
    check-ups-health;

  check-openvpn = callPackage ../servers/monitoring/plugins/openvpn.nix { };

  check_smartmon = callPackage ../servers/monitoring/nagios/plugins/smartmon.nix { };

  checkSSLCert = callPackage ../servers/monitoring/nagios/plugins/check_ssl_cert.nix { };

  check_systemd = callPackage ../servers/monitoring/nagios/plugins/check_systemd.nix { };

  check_zfs = callPackage ../servers/monitoring/nagios/plugins/zfs.nix { };

  neo4j = callPackage ../servers/nosql/neo4j { };

  neo4j-desktop = callPackage ../applications/misc/neo4j-desktop { };

  check-esxi-hardware = callPackage ../servers/monitoring/plugins/esxi.nix { };

  net-snmp = callPackage ../servers/monitoring/net-snmp { };

  newrelic-sysmond = callPackage ../servers/monitoring/newrelic-sysmond { };

  nullidentdmod = callPackage ../servers/identd/nullidentdmod { };

  riemann = callPackage ../servers/monitoring/riemann { };
  riemann-dash = callPackage ../servers/monitoring/riemann-dash { };

  unpfs = callPackage ../servers/unpfs { };

  oidentd = callPackage ../servers/identd/oidentd { };

  softether = callPackage ../servers/softether { };

  qboot = pkgsi686Linux.callPackage ../applications/virtualization/qboot { };

  rust-hypervisor-firmware = callPackage ../applications/virtualization/rust-hypervisor-firmware { };

  OVMF = callPackage ../applications/virtualization/OVMF {
    inherit (python3Packages) pexpect;
  };
  OVMFFull = callPackage ../applications/virtualization/OVMF {
    inherit (python3Packages) pexpect;
    secureBoot = true;
    httpSupport = true;
    tpmSupport = true;
    tlsSupport = true;
    msVarsTemplate = stdenv.isx86_64 || stdenv.isAarch64;
  };

  ops = callPackage ../applications/virtualization/ops { };

  vmfs-tools = callPackage ../tools/filesystems/vmfs-tools { };

  patroni = callPackage ../servers/sql/patroni { pythonPackages = python3Packages; };

  pgbouncer = callPackage ../servers/sql/pgbouncer { };

  pgcat = callPackage ../servers/sql/pgcat {};

  pgpool = callPackage ../servers/sql/pgpool { };

  tang = callPackage ../servers/tang {
    asciidoc = asciidoc-full;
  };

  promscale = callPackage ../servers/monitoring/prometheus/promscale { };

  timescaledb-parallel-copy = callPackage ../development/tools/database/timescaledb-parallel-copy { };

  timescaledb-tune = callPackage ../development/tools/database/timescaledb-tune { };

  inherit (import ../servers/sql/postgresql pkgs)
    postgresql_12
    postgresql_13
    postgresql_14
    postgresql_15
    postgresql_16

    postgresql_12_jit
    postgresql_13_jit
    postgresql_14_jit
    postgresql_15_jit
    postgresql_16_jit
  ;
  postgresql = postgresql_15;
  postgresql_jit = postgresql_15_jit;
  postgresqlPackages = recurseIntoAttrs postgresql.pkgs;
  postgresqlJitPackages = recurseIntoAttrs postgresql_jit.pkgs;
  postgresql12Packages = recurseIntoAttrs postgresql_12.pkgs;
  postgresql13Packages = recurseIntoAttrs postgresql_13.pkgs;
  postgresql14Packages = recurseIntoAttrs postgresql_14.pkgs;
  postgresql16Packages = recurseIntoAttrs postgresql_16.pkgs;
  postgresql12JitPackages = recurseIntoAttrs postgresql_12_jit.pkgs;
  postgresql13JitPackages = recurseIntoAttrs postgresql_13_jit.pkgs;
  postgresql14JitPackages = recurseIntoAttrs postgresql_14_jit.pkgs;
  postgresql15JitPackages = recurseIntoAttrs postgresql_15_jit.pkgs;
  postgresql16JitPackages = recurseIntoAttrs postgresql_16_jit.pkgs;
  postgresql15Packages = postgresqlPackages;

  postgresql_jdbc = callPackage ../development/java-modules/postgresql_jdbc { };

  postgresqlTestHook = callPackage ../build-support/setup-hooks/postgresql-test-hook { };

  postgrest = haskellPackages.postgrest.bin;

  redshift_jdbc = callPackage ../development/java-modules/redshift_jdbc { };

  liquibase_redshift_extension = callPackage ../development/java-modules/liquibase_redshift_extension { };

  promql-cli = callPackage ../tools/misc/promql-cli { };

  prom2json = callPackage ../servers/monitoring/prometheus/prom2json.nix { };
  prometheus = callPackage ../servers/monitoring/prometheus { };
  prometheus-alertmanager = callPackage ../servers/monitoring/prometheus/alertmanager.nix { };
  prometheus-apcupsd-exporter = callPackage ../servers/monitoring/prometheus/apcupsd-exporter.nix { };
  prometheus-artifactory-exporter = callPackage ../servers/monitoring/prometheus/artifactory-exporter.nix { };
  prometheus-atlas-exporter = callPackage ../servers/monitoring/prometheus/atlas-exporter.nix { };
  prometheus-aws-s3-exporter = callPackage ../servers/monitoring/prometheus/aws-s3-exporter.nix { };
  prometheus-bind-exporter = callPackage ../servers/monitoring/prometheus/bind-exporter.nix { };
  prometheus-bird-exporter = callPackage ../servers/monitoring/prometheus/bird-exporter.nix { };
  prometheus-bitcoin-exporter = callPackage ../servers/monitoring/prometheus/bitcoin-exporter.nix { };
  prometheus-blackbox-exporter = callPackage ../servers/monitoring/prometheus/blackbox-exporter.nix { };
  prometheus-cloudflare-exporter = callPackage ../servers/monitoring/prometheus/cloudflare-exporter.nix { };
  prometheus-collectd-exporter = callPackage ../servers/monitoring/prometheus/collectd-exporter.nix { };
  prometheus-consul-exporter = callPackage ../servers/monitoring/prometheus/consul-exporter.nix { };
  prometheus-dcgm-exporter = callPackage ../servers/monitoring/prometheus/dcgm-exporter { };
  prometheus-dnsmasq-exporter = callPackage ../servers/monitoring/prometheus/dnsmasq-exporter.nix { };
  prometheus-dovecot-exporter = callPackage ../servers/monitoring/prometheus/dovecot-exporter.nix { };
  prometheus-domain-exporter = callPackage ../servers/monitoring/prometheus/domain-exporter.nix { };
  prometheus-fastly-exporter = callPackage ../servers/monitoring/prometheus/fastly-exporter.nix { };
  prometheus-flow-exporter = callPackage ../servers/monitoring/prometheus/flow-exporter.nix { };
  prometheus-fritzbox-exporter = callPackage ../servers/monitoring/prometheus/fritzbox-exporter.nix { };
  prometheus-gitlab-ci-pipelines-exporter = callPackage ../servers/monitoring/prometheus/gitlab-ci-pipelines-exporter.nix { };
  prometheus-graphite-exporter = callPackage ../servers/monitoring/prometheus/graphite-exporter.nix { };
  prometheus-haproxy-exporter = callPackage ../servers/monitoring/prometheus/haproxy-exporter.nix { };
  prometheus-idrac-exporter = callPackage ../servers/monitoring/prometheus/idrac-exporter.nix { };
  prometheus-imap-mailstat-exporter = callPackage ../servers/monitoring/prometheus/imap-mailstat-exporter.nix { };
  prometheus-influxdb-exporter = callPackage ../servers/monitoring/prometheus/influxdb-exporter.nix { };
  prometheus-ipmi-exporter = callPackage ../servers/monitoring/prometheus/ipmi-exporter.nix { };
  prometheus-jitsi-exporter = callPackage ../servers/monitoring/prometheus/jitsi-exporter.nix { };
  prometheus-jmx-httpserver = callPackage ../servers/monitoring/prometheus/jmx-httpserver.nix {  };
  prometheus-json-exporter = callPackage ../servers/monitoring/prometheus/json-exporter.nix { };
  prometheus-junos-czerwonk-exporter = callPackage ../servers/monitoring/prometheus/junos-czerwonk-exporter.nix { };
  prometheus-kea-exporter = callPackage ../servers/monitoring/prometheus/kea-exporter.nix { };
  prometheus-keylight-exporter = callPackage ../servers/monitoring/prometheus/keylight-exporter.nix { };
  prometheus-knot-exporter = callPackage ../servers/monitoring/prometheus/knot-exporter.nix { };
  prometheus-lnd-exporter = callPackage ../servers/monitoring/prometheus/lnd-exporter.nix { };
  prometheus-mail-exporter = callPackage ../servers/monitoring/prometheus/mail-exporter.nix { };
  prometheus-mikrotik-exporter = callPackage ../servers/monitoring/prometheus/mikrotik-exporter.nix { };
  prometheus-minio-exporter = callPackage ../servers/monitoring/prometheus/minio-exporter { };
  prometheus-modemmanager-exporter = callPackage ../servers/monitoring/prometheus/modemmanager-exporter.nix { };
  prometheus-mongodb-exporter = callPackage ../servers/monitoring/prometheus/mongodb-exporter.nix { };
  prometheus-mysqld-exporter = callPackage ../servers/monitoring/prometheus/mysqld-exporter.nix { };
  prometheus-nats-exporter = callPackage ../servers/monitoring/prometheus/nats-exporter.nix { };
  prometheus-nextcloud-exporter = callPackage ../servers/monitoring/prometheus/nextcloud-exporter.nix { };
  prometheus-nginx-exporter = callPackage ../servers/monitoring/prometheus/nginx-exporter.nix { };
  prometheus-nginxlog-exporter = callPackage ../servers/monitoring/prometheus/nginxlog-exporter.nix { };
  prometheus-node-exporter = callPackage ../servers/monitoring/prometheus/node-exporter.nix {
    inherit (darwin.apple_sdk.frameworks) CoreFoundation IOKit;
  };
  prometheus-nut-exporter = callPackage ../servers/monitoring/prometheus/nut-exporter.nix { };
  prometheus-openldap-exporter = callPackage ../servers/monitoring/prometheus/openldap-exporter.nix { } ;
  prometheus-pgbouncer-exporter = callPackage ../servers/monitoring/prometheus/pgbouncer-exporter.nix { };
  prometheus-php-fpm-exporter = callPackage ../servers/monitoring/prometheus/php-fpm-exporter.nix { };
  prometheus-pihole-exporter = callPackage ../servers/monitoring/prometheus/pihole-exporter.nix {  };
  prometheus-ping-exporter = callPackage ../servers/monitoring/prometheus/ping-exporter.nix {  };
  prometheus-postfix-exporter = callPackage ../servers/monitoring/prometheus/postfix-exporter.nix { };
  prometheus-postgres-exporter = callPackage ../servers/monitoring/prometheus/postgres-exporter.nix { };
  prometheus-process-exporter = callPackage ../servers/monitoring/prometheus/process-exporter.nix { };
  prometheus-pushgateway = callPackage ../servers/monitoring/prometheus/pushgateway.nix { };
  prometheus-pve-exporter = callPackage ../servers/monitoring/prometheus/pve-exporter.nix { };
  prometheus-redis-exporter = callPackage ../servers/monitoring/prometheus/redis-exporter.nix { };
  prometheus-rabbitmq-exporter = callPackage ../servers/monitoring/prometheus/rabbitmq-exporter.nix { };
  prometheus-rtl_433-exporter = callPackage ../servers/monitoring/prometheus/rtl_433-exporter.nix { };
  prometheus-sabnzbd-exporter = callPackage ../servers/monitoring/prometheus/sabnzbd-exporter.nix { };
  prometheus-sachet = callPackage ../servers/monitoring/prometheus/sachet.nix { };
  prometheus-script-exporter = callPackage ../servers/monitoring/prometheus/script-exporter.nix { };
  prometheus-shelly-exporter = callPackage ../servers/monitoring/prometheus/shelly-exporter.nix { };
  prometheus-smartctl-exporter = callPackage ../servers/monitoring/prometheus/smartctl-exporter { };
  prometheus-smokeping-prober = callPackage ../servers/monitoring/prometheus/smokeping-prober.nix { };
  prometheus-snmp-exporter = callPackage ../servers/monitoring/prometheus/snmp-exporter.nix { };
  prometheus-statsd-exporter = callPackage ../servers/monitoring/prometheus/statsd-exporter.nix { };
  prometheus-surfboard-exporter = callPackage ../servers/monitoring/prometheus/surfboard-exporter.nix { };
  prometheus-sql-exporter = callPackage ../servers/monitoring/prometheus/sql-exporter.nix { };
  prometheus-systemd-exporter = callPackage ../servers/monitoring/prometheus/systemd-exporter.nix { };
  prometheus-tor-exporter = callPackage ../servers/monitoring/prometheus/tor-exporter.nix { };
  prometheus-unbound-exporter = callPackage ../servers/monitoring/prometheus/unbound-exporter.nix { };
  prometheus-v2ray-exporter = callPackage ../servers/monitoring/prometheus/v2ray-exporter.nix { };
  prometheus-varnish-exporter = callPackage ../servers/monitoring/prometheus/varnish-exporter.nix { };
  prometheus-wireguard-exporter = callPackage ../servers/monitoring/prometheus/wireguard-exporter.nix {
    inherit (darwin.apple_sdk.frameworks) Security;
  };
  prometheus-zfs-exporter = callPackage ../servers/monitoring/prometheus/zfs-exporter.nix { };
  prometheus-xmpp-alerts = callPackage ../servers/monitoring/prometheus/xmpp-alerts.nix { };

  prometheus-cpp = callPackage ../development/libraries/prometheus-cpp { };

  proximity-sort = callPackage ../tools/misc/proximity-sort { };

  psql2csv = callPackage ../tools/misc/psql2csv { };

  psqlodbc = callPackage ../development/libraries/psqlodbc { };

  public-inbox = perlPackages.callPackage ../servers/mail/public-inbox { };

  pure-ftpd = callPackage ../servers/ftp/pure-ftpd { };

  spf-engine = python3.pkgs.callPackage ../servers/mail/spf-engine { };

  pypiserver = with python3Packages; toPythonApplication pypiserver;

  qpid-cpp = callPackage ../servers/amqp/qpid-cpp { };

  qremotecontrol-server = libsForQt5.callPackage ../servers/misc/qremotecontrol-server { };

  questdb = callPackage ../servers/nosql/questdb { };

  rabbitmq-server = callPackage ../servers/amqp/rabbitmq-server {
    inherit (darwin.apple_sdk.frameworks) AppKit Carbon Cocoa;
    elixir = elixir_1_14;
  };

  radicale2 = callPackage ../servers/radicale/2.x.nix { };
  radicale3 = radicale;

  qcal = callPackage ../tools/networking/qcal/default.nix { };

  qcard = callPackage ../tools/networking/qcard { };

  rake = callPackage ../development/tools/build-managers/rake { };

  rakkess = callPackage ../development/tools/rakkess { };

  redis = callPackage ../servers/nosql/redis { };

  redli = callPackage ../tools/networking/redli { };

  redstore = callPackage ../servers/http/redstore { };

  repro-get = callPackage ../tools/package-management/repro-get { };

  restic = callPackage ../tools/backup/restic { };

  restic-integrity = callPackage ../applications/backup/restic-integrity { };
  restic-rest-server = callPackage ../tools/backup/restic/rest-server.nix { };

  rethinkdb = callPackage ../servers/nosql/rethinkdb {
    stdenv = clangStdenv;
    libtool = darwin.cctools;
    protobuf = protobuf_21;
  };

  rippled = callPackage ../servers/rippled {
    boost = boost177;
  };

  rippled-validator-keys-tool = callPackage ../servers/rippled/validator-keys-tool.nix { };

  roon-bridge = callPackage ../servers/roon-bridge { };

  rpiplay = callPackage ../servers/rpiplay { };

  roon-server = callPackage ../servers/roon-server { };

  rustic-rs = callPackage ../tools/backup/rustic-rs {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  supervise = callPackage ../tools/system/supervise { };

  spamassassin = callPackage ../servers/mail/spamassassin { };

  deadpixi-sam-unstable = callPackage ../applications/editors/deadpixi-sam { };

  samba4 = darwin.apple_sdk_11_0.callPackage ../servers/samba/4.x.nix { };

  samba = samba4;

  samba4Full = lowPrio (samba4.override {
    enableLDAP = true;
    enablePrinting = true;
    enableMDNS = true;
    enableDomainController = true;
    enableRegedit = true;
    enableCephFS = !stdenv.hostPlatform.isAarch64;
  });

  sambaFull = samba4Full;

  sampler = callPackage ../applications/misc/sampler { };

  scalene = with python3Packages; toPythonApplication scalene;

  scalr-cli = callPackage ../tools/admin/scalr-cli { };

  scaphandre = callPackage ../servers/scaphandre { };

  shairplay = callPackage ../servers/shairplay { avahi = avahi-compat; };

  shairport-sync = callPackage ../servers/shairport-sync { };

  sharing = callPackage ../servers/sharing { };

  showoff = callPackage ../servers/http/showoff { };

  serfdom = callPackage ../servers/serf { };

  seyren = callPackage ../servers/monitoring/seyren { };

  ruby-zoom = callPackage ../tools/text/ruby-zoom { };

  sensu = callPackage ../servers/monitoring/sensu { };

  inherit (callPackages ../servers/monitoring/sensu-go { })
    sensu-go-agent
    sensu-go-backend
    sensu-go-cli;

  check-wmiplus = callPackage ../servers/monitoring/plugins/wmiplus { };

  shishi = callPackage ../servers/shishi {
      pam = if stdenv.isLinux then pam else null;
      # see also openssl, which has/had this same trick
  };

  sickgear = callPackage ../servers/sickbeard/sickgear.nix { };

  sipwitch = callPackage ../servers/sip/sipwitch { };

  smcroute = callPackage ../servers/smcroute { };

  snipe-it = callPackage ../by-name/sn/snipe-it/package.nix {
    php = php81;
  };

  sogo = callPackage ../servers/web-apps/sogo { };

  spacecookie =
    haskell.lib.compose.justStaticExecutables haskellPackages.spacecookie;

  spawn_fcgi = callPackage ../servers/http/spawn-fcgi { };

  spring-boot-cli = callPackage ../development/tools/spring-boot-cli { };

  squid = callPackage ../servers/squid { };

  duckling-proxy = callPackage ../servers/duckling-proxy { };

  sslh = callPackage ../servers/sslh { };

  thttpd = callPackage ../servers/http/thttpd { };

  stalwart-mail_0_6 = (stalwart-mail.override { rocksdb_8_11 = rocksdb_8_3; }).overrideAttrs (old: rec {
    pname = "stalwart-mail_0_6";
    version = "0.6.0";
    src = fetchFromGitHub {
      owner = "stalwartlabs";
      repo = "mail-server";
      rev = "v${version}";
      hash = "sha256-OHwUWSUW6ovLQTxnuUrolQGhxbhp4YqKSH+ZTpe2WXc=";
      fetchSubmodules = true;
    };
    cargoDeps = old.cargoDeps.overrideAttrs (_: {
      inherit src;
      name = "${pname}-${version}-vendor.tar.gz";
      outputHash = "sha256-mW3OXQj6DcIMO1YlTG3G+a1ORRcuvp5/h7BU+b4QbnE=";
    });
  });

  static-web-server = callPackage ../servers/static-web-server { };

  stone = callPackage ../servers/stone { };

  storm = callPackage ../servers/computing/storm { };

  switcheroo-control = callPackage ../os-specific/linux/switcheroo-control { };

  slurm = callPackage ../servers/computing/slurm { };

  slurm-spank-stunnel = callPackage ../servers/computing/slurm-spank-stunnel { };

  slurm-spank-x11 = callPackage ../servers/computing/slurm-spank-x11 { };

  systemd-journal2gelf = callPackage ../tools/system/systemd-journal2gelf { };

  tailscale = callPackage ../servers/tailscale {
    buildGoModule = buildGo122Module;
  };

  tailscale-systray = callPackage ../applications/misc/tailscale-systray { };

  tailspin = callPackage ../tools/misc/tailspin { };

  thanos = callPackage ../servers/monitoring/thanos { };

  trafficserver = callPackage ../servers/http/trafficserver { };

  inherit (callPackages ../servers/http/tomcat { })
    tomcat9
    tomcat10;

  tomcat_mysql_jdbc = callPackage ../servers/http/tomcat/jdbc/mysql { };

  torque = callPackage ../servers/computing/torque {
    autoreconfHook = buildPackages.autoreconfHook269;
  };

  tt-rss = callPackage ../servers/tt-rss { };
  tt-rss-plugin-feediron = callPackage ../servers/tt-rss/plugin-feediron { };
  tt-rss-plugin-ff-instagram = callPackage ../servers/tt-rss/plugin-ff-instagram { };
  tt-rss-plugin-auth-ldap = callPackage ../servers/tt-rss/plugin-auth-ldap { };
  tt-rss-theme-feedly = callPackage ../servers/tt-rss/theme-feedly { };

  rss-bridge = callPackage ../servers/web-apps/rss-bridge { };

  selfoss = callPackage ../servers/web-apps/selfoss { };

  shaarli = callPackage ../servers/web-apps/shaarli { };

  shiori = callPackage ../servers/web-apps/shiori { };

  slskd = callPackage ../servers/web-apps/slskd { };

  inherit (callPackages ../servers/web-apps/matomo {})
    matomo
    matomo_5
    matomo-beta;

  axis2 = callPackage ../servers/http/tomcat/axis2 { };

  inherit (callPackages ../servers/unifi { })
    unifi7
    unifi8;

  unifi = unifi7;

  unifi-protect-backup = callPackage ../applications/backup/unifi-protect-backup { };

  unifi-video = callPackage ../servers/unifi-video { };

  unpackerr = callPackage ../servers/unpackerr {
    inherit (darwin.apple_sdk.frameworks) Cocoa WebKit;
  };

  unstructured-api = callPackage ../servers/unstructured-api { };

  urserver = callPackage ../servers/urserver { };

  vouch-proxy = callPackage ../servers/vouch-proxy { };

  virtiofsd = callPackage ../servers/misc/virtiofsd { };

  virtualenv = with python3Packages; toPythonApplication virtualenv;

  virtualenv-clone = with python3Packages; toPythonApplication virtualenv-clone;

  vmagent = callPackage ../servers/monitoring/vmagent { };

  vsftpd = callPackage ../servers/ftp/vsftpd { };

  wallabag = callPackage ../servers/web-apps/wallabag { };

  webdav = callPackage ../servers/webdav { };

  webdav-server-rs = callPackage ../servers/webdav-server-rs { };

  webmetro = callPackage ../servers/webmetro { };

  wesher = callPackage ../servers/wesher { };

  wishlist = callPackage ../servers/wishlist { };

  wsdd = callPackage ../servers/wsdd { };

  webhook = callPackage ../servers/http/webhook { };

  xinetd = callPackage ../servers/xinetd { };

  zookeeper = callPackage ../servers/zookeeper { };

  zookeeper_mt = callPackage ../development/libraries/zookeeper_mt { };

  xqilla = callPackage ../development/tools/xqilla { };

  xquartz = callPackage ../servers/x11/xquartz { };

  quartz-wm = callPackage ../servers/x11/quartz-wm {
    stdenv = clangStdenv;
    inherit (darwin.apple_sdk.frameworks) AppKit Foundation;
    inherit (darwin.apple_sdk.libs) Xplugin;
  };

  xorg = let
    # Use `lib.callPackageWith __splicedPackages` rather than plain `callPackage`
    # so as not to have the newly bound xorg items already in scope,  which would
    # have created a cycle.
    overrides = lib.callPackageWith __splicedPackages ../servers/x11/xorg/overrides.nix {
      inherit (darwin.apple_sdk.frameworks) ApplicationServices Carbon Cocoa;
      inherit (darwin.apple_sdk.libs) Xplugin;
      inherit (buildPackages.darwin) bootstrap_cmds;
      udev = if stdenv.isLinux then udev else null;
      libdrm = if stdenv.isLinux then libdrm else null;
    };

    generatedPackages = lib.callPackageWith __splicedPackages ../servers/x11/xorg/default.nix { };

    xorgPackages = makeScopeWithSplicing' {
      otherSplices = generateSplicesForMkScope "xorg";
      f = lib.extends overrides generatedPackages;
    };

  in recurseIntoAttrs xorgPackages;

  xorg-autoconf = callPackage ../development/tools/misc/xorg-autoconf { };

  xwayland = callPackage ../servers/x11/xorg/xwayland.nix { };

  xwaylandvideobridge = libsForQt5.callPackage ../tools/wayland/xwaylandvideobridge { };

  yaws = callPackage ../servers/http/yaws { };

  zabbixFor = version: rec {
    agent = (callPackages ../servers/monitoring/zabbix/agent.nix {}).${version};
    proxy-mysql = (callPackages ../servers/monitoring/zabbix/proxy.nix { mysqlSupport = true; }).${version};
    proxy-pgsql = (callPackages ../servers/monitoring/zabbix/proxy.nix { postgresqlSupport = true; }).${version};
    proxy-sqlite = (callPackages ../servers/monitoring/zabbix/proxy.nix { sqliteSupport = true; }).${version};
    server-mysql = (callPackages ../servers/monitoring/zabbix/server.nix { mysqlSupport = true; }).${version};
    server-pgsql = (callPackages ../servers/monitoring/zabbix/server.nix { postgresqlSupport = true; }).${version};
    web = (callPackages ../servers/monitoring/zabbix/web.nix {}).${version};
    agent2 = (callPackages ../servers/monitoring/zabbix/agent2.nix {}).${version};

    # backwards compatibility
    server = server-pgsql;
  };

  zabbix60 = recurseIntoAttrs (zabbixFor "v60");
  zabbix50 = recurseIntoAttrs (zabbixFor "v50");

  zabbix = zabbix60;

  zipkin = callPackage ../servers/monitoring/zipkin { };

  ### SERVERS / GEOSPATIAL

  fit-trackee = callPackage ../servers/geospatial/fit-trackee { };

  geoserver = callPackage ../servers/geospatial/geoserver { };

  mapcache = callPackage ../servers/geospatial/mapcache { };

  mapproxy = callPackage ../servers/geospatial/mapproxy { };

  mapserver = callPackage ../servers/geospatial/mapserver { };

  martin = callPackage ../servers/geospatial/martin {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  mbtileserver = callPackage ../servers/geospatial/mbtileserver { };

  pg_featureserv = callPackage ../servers/geospatial/pg_featureserv { };

  pg_tileserv = callPackage ../servers/geospatial/pg_tileserv { };

  tile38 = callPackage ../servers/geospatial/tile38 { };

  ### OS-SPECIFIC

  airbuddy = callPackage ../os-specific/darwin/airbuddy { };

  afuse = callPackage ../os-specific/linux/afuse { };

  autofs5 = callPackage ../os-specific/linux/autofs { };

  _915resolution = callPackage ../os-specific/linux/915resolution { };

  nfs-utils = callPackage ../os-specific/linux/nfs-utils { };

  acpi = callPackage ../os-specific/linux/acpi { };

  acpid = callPackage ../os-specific/linux/acpid { };

  acpitool = callPackage ../os-specific/linux/acpitool { };

  aldente = callPackage ../os-specific/darwin/aldente { };

  alfred = callPackage ../os-specific/linux/batman-adv/alfred.nix { };

  alertmanager-irc-relay = callPackage ../servers/monitoring/alertmanager-irc-relay { };

  alsa-utils = callPackage ../by-name/al/alsa-utils/package.nix {
    fftw = fftwFloat;
  };

  apparency = callPackage ../os-specific/darwin/apparency { };

  arm-trusted-firmware = callPackage ../misc/arm-trusted-firmware { };
  inherit (arm-trusted-firmware)
    buildArmTrustedFirmware
    armTrustedFirmwareTools
    armTrustedFirmwareAllwinner
    armTrustedFirmwareAllwinnerH616
    armTrustedFirmwareAllwinnerH6
    armTrustedFirmwareQemu
    armTrustedFirmwareRK3328
    armTrustedFirmwareRK3399
    armTrustedFirmwareRK3588
    armTrustedFirmwareS905
    ;

  mlxbf-bootimages = callPackage ../misc/mlxbf-bootimages { };

  microcodeAmd = callPackage ../os-specific/linux/microcode/amd.nix { };

  microcodeIntel = callPackage ../os-specific/linux/microcode/intel.nix { };

  iucode-tool = callPackage ../os-specific/linux/microcode/iucode-tool.nix { };

  inherit (callPackages ../os-specific/linux/apparmor { })
    libapparmor apparmor-utils apparmor-bin-utils apparmor-parser apparmor-pam
    apparmor-profiles apparmor-kernel-patches apparmorRulesFromClosure;

  aseq2json = callPackage ../os-specific/linux/aseq2json { };

  atop = callPackage ../os-specific/linux/atop { };

  audit = callPackage ../os-specific/linux/audit { };

  ath9k-htc-blobless-firmware = callPackage ../os-specific/linux/firmware/ath9k { };
  ath9k-htc-blobless-firmware-unstable =
    callPackage ../os-specific/linux/firmware/ath9k { enableUnstable = true; };

  bartender = callPackage ../os-specific/darwin/bartender { };

  b43Firmware_5_1_138 = callPackage ../os-specific/linux/firmware/b43-firmware/5.1.138.nix { };

  b43Firmware_6_30_163_46 = callPackage ../os-specific/linux/firmware/b43-firmware/6.30.163.46.nix { };

  b43FirmwareCutter = callPackage ../os-specific/linux/firmware/b43-firmware-cutter { };

  epilys-bb = callPackage ../tools/system/epilys-bb { };

  below = callPackage ../os-specific/linux/below { };

  bt-fw-converter = callPackage ../os-specific/linux/firmware/bt-fw-converter { };

  brillo = callPackage ../os-specific/linux/brillo { };

  broadcom-bt-firmware = callPackage ../os-specific/linux/firmware/broadcom-bt-firmware { };

  batctl = callPackage ../os-specific/linux/batman-adv/batctl.nix { };

  beefi = callPackage ../os-specific/linux/beefi { };

  blktrace = callPackage ../os-specific/linux/blktrace { };

  bluez5 = bluez;
  bluez5-experimental = bluez-experimental;

  bluez-experimental = bluez.override {
    enableExperimental = true;
  };

  bolt = callPackage ../os-specific/linux/bolt { };

  bpf-linker = callPackage ../development/tools/bpf-linker { };

  bpftune = callPackage ../os-specific/linux/bpftune { };

  bpfmon = callPackage ../os-specific/linux/bpfmon { };

  bridge-utils = callPackage ../os-specific/linux/bridge-utils { };

  busybox = callPackage ../os-specific/linux/busybox { };
  busybox-sandbox-shell = callPackage ../os-specific/linux/busybox/sandbox-shell.nix {
    # musl roadmap has RISC-V support projected for 1.1.20
    busybox = if !stdenv.hostPlatform.isRiscV && !stdenv.hostPlatform.isLoongArch64 && stdenv.hostPlatform.libc != "bionic"
              then pkgsStatic.busybox
              else busybox;
  };

  cachefilesd = callPackage ../os-specific/linux/cachefilesd { };

  cariddi = callPackage ../tools/security/cariddi { };

  checkpolicy = callPackage ../os-specific/linux/checkpolicy { };

  checksec = callPackage ../os-specific/linux/checksec { };

  cifs-utils = callPackage ../os-specific/linux/cifs-utils { };

  cm-rgb = python3Packages.callPackage ../tools/system/cm-rgb { };

  cpustat = callPackage ../os-specific/linux/cpustat { };

  cockroachdb-bin = callPackage ../servers/sql/cockroachdb/cockroachdb-bin.nix { };

  coconutbattery = callPackage ../os-specific/darwin/coconutbattery { };

  conky = callPackage ../os-specific/linux/conky ({
    lua = lua5_4;
    inherit (linuxPackages.nvidia_x11.settings) libXNVCtrl;
  } // config.conky or {});

  conntrack-tools = callPackage ../os-specific/linux/conntrack-tools { };

  coredns = callPackage ../servers/dns/coredns { };

  corerad = callPackage ../tools/networking/corerad { };

  cpufrequtils = callPackage ../os-specific/linux/cpufrequtils { };

  cpupower-gui = python3Packages.callPackage ../os-specific/linux/cpupower-gui {
    inherit (pkgs) meson;
  };

  criu = callPackage ../os-specific/linux/criu { };

  cryptomator = callPackage ../tools/security/cryptomator {
    jdk = jdk21.override { enableJavaFX = true; };
  };

  cryptsetup = callPackage ../os-specific/linux/cryptsetup { };

  cramfsprogs = callPackage ../os-specific/linux/cramfsprogs { };

  cramfsswap = callPackage ../os-specific/linux/cramfsswap { };

  cshatag = callPackage ../os-specific/linux/cshatag { };

  # Darwin package set
  #
  # Even though this is a set of packages not single package, use `callPackage`
  # not `callPackages` so the per-package callPackages don't have their
  # `.override` clobbered. C.F. `llvmPackages` which does the same.
  darwin = recurseIntoAttrs (callPackage ./darwin-packages.nix { });

  defaultbrowser = callPackage ../os-specific/darwin/defaultbrowser {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  disk_indicator = callPackage ../os-specific/linux/disk-indicator { };

  displaylink = callPackage ../os-specific/linux/displaylink {
    inherit (linuxPackages) evdi;
  };

  dmidecode = callPackage ../os-specific/linux/dmidecode { };

  dmtcp = callPackage ../os-specific/linux/dmtcp { };

  directvnc = callPackage ../os-specific/linux/directvnc { };

  dmraid = callPackage ../os-specific/linux/dmraid { lvm2 = lvm2_dmeventd; };

  drbd = callPackage ../os-specific/linux/drbd/utils.nix { };

  dropwatch = callPackage ../os-specific/linux/dropwatch { };

  dsd = callPackage ../applications/radio/dsd { };

  dstat = callPackage ../os-specific/linux/dstat { };

  esdm = callPackage ../os-specific/linux/esdm { };

  evdev-proto = callPackage ../os-specific/bsd/freebsd/evdev-proto { };

  fscryptctl = callPackage ../os-specific/linux/fscryptctl { };
  # unstable until the first 1.x release
  fscrypt-experimental = callPackage ../os-specific/linux/fscrypt { };

  fsverity-utils = callPackage ../os-specific/linux/fsverity-utils { };

  fwanalyzer = callPackage ../tools/filesystems/fwanalyzer { };

  fwupd-efi = callPackage ../os-specific/linux/firmware/fwupd-efi { };

  firmware-manager = callPackage ../os-specific/linux/firmware/firmware-manager { };

  firmware-updater = callPackage ../os-specific/linux/firmware/firmware-updater { };

  fwts = callPackage ../os-specific/linux/fwts { };

  gobi_loader = callPackage ../os-specific/linux/gobi_loader { };

  libossp_uuid = callPackage ../development/libraries/libossp-uuid { };

  libuuid = if stdenv.isLinux
    then util-linuxMinimal
    else null;

  light = callPackage ../os-specific/linux/light { };

  lightum = callPackage ../os-specific/linux/lightum { };

  ebtables = callPackage ../os-specific/linux/ebtables { };

  elegant-sddm = libsForQt5.callPackage ../data/themes/elegant-sddm { };

  error-inject = callPackages ../os-specific/linux/error-inject { };

  extrace = callPackage ../os-specific/linux/extrace { };

  facetimehd-calibration = callPackage ../os-specific/linux/firmware/facetimehd-calibration { };

  facetimehd-firmware = callPackage ../os-specific/linux/firmware/facetimehd-firmware { };

  fan2go = callPackage ../os-specific/linux/fan2go { };

  fanctl = callPackage ../os-specific/linux/fanctl { };

  fatrace = callPackage ../os-specific/linux/fatrace { };

  ffado = libsForQt5.callPackage ../os-specific/linux/ffado { };
  libffado = ffado;

  fbterm = callPackage ../os-specific/linux/fbterm { };

  firejail = callPackage ../os-specific/linux/firejail { };

  fnotifystat = callPackage ../os-specific/linux/fnotifystat { };

  forkstat = callPackage ../os-specific/linux/forkstat { };

  freefall = callPackage ../os-specific/linux/freefall {
    inherit (linuxPackages) kernel;
  };

  fusePackages = dontRecurseIntoAttrs (callPackage ../os-specific/linux/fuse {
    util-linux = util-linuxMinimal;
  });
  fuse = fuse2;
  fuse2 = lowPrio (if stdenv.isDarwin then macfuse-stubs else fusePackages.fuse_2);
  fuse3 = fusePackages.fuse_3;
  fuse-common = hiPrio fusePackages.fuse_3.common;

  fxload = callPackage ../os-specific/linux/fxload { };

  gfxtablet = callPackage ../os-specific/linux/gfxtablet { };

  gmailctl = callPackage ../applications/networking/gmailctl { };

  gomp = callPackage ../applications/version-management/gomp { };

  gomplate = callPackage ../development/tools/gomplate { };

  gpm = callPackage ../servers/gpm {
    withNcurses = false; # Keep curses disabled for lack of value

    # latest 6.8 mysteriously fails to parse '@headings single':
    #   https://lists.gnu.org/archive/html/bug-texinfo/2021-09/msg00011.html
    texinfo = buildPackages.texinfo6_7;
  };

  gpm-ncurses = gpm.override { withNcurses = true; };

  gpu-switch = callPackage ../os-specific/linux/gpu-switch { };

  gradm = callPackage ../os-specific/linux/gradm { };

  gt = callPackage ../os-specific/linux/gt { };

  gtop = callPackage ../tools/system/gtop { };

  hd-idle = callPackage ../os-specific/linux/hd-idle { };

  hdparm = callPackage ../os-specific/linux/hdparm { };

  health-check = callPackage ../os-specific/linux/health-check { };

  hibernate = callPackage ../os-specific/linux/hibernate { };

  hostapd = callPackage ../os-specific/linux/hostapd { };

  htop = callPackage ../tools/system/htop {
    inherit (darwin) IOKit;
  };

  htop-vim = callPackage ../tools/system/htop/htop-vim.nix { };

  humility = callPackage ../development/tools/rust/humility {
    inherit (darwin.apple_sdk.frameworks) AppKit;
  };

  btop = darwin.apple_sdk_11_0.callPackage ../tools/system/btop { };

  nmon = callPackage ../os-specific/linux/nmon { };

  hwdata = callPackage ../os-specific/linux/hwdata { };

  i7z = qt5.callPackage ../os-specific/linux/i7z { };

  pcm = callPackage ../os-specific/linux/pcm { };

  ifmetric = callPackage ../os-specific/linux/ifmetric { };

  ima-evm-utils = callPackage ../os-specific/linux/ima-evm-utils { };

  intel2200BGFirmware = callPackage ../os-specific/linux/firmware/intel2200BGFirmware { };

  intel-cmt-cat = callPackage ../os-specific/linux/intel-cmt-cat { };

  intel-compute-runtime = callPackage ../os-specific/linux/intel-compute-runtime { };

  intel-ocl = callPackage ../os-specific/linux/intel-ocl { };

  level-zero = callPackage ../development/libraries/level-zero { };

  iomelt = callPackage ../os-specific/linux/iomelt { };

  iotop = callPackage ../os-specific/linux/iotop { };
  iotop-c = callPackage ../os-specific/linux/iotop-c { };

  iproute2 = callPackage ../os-specific/linux/iproute { };

  ipu6-camera-bins = callPackage ../os-specific/linux/firmware/ipu6-camera-bins {};

  ipu6-camera-hal = callPackage ../development/libraries/ipu6-camera-hal {};

  ipu6ep-camera-hal = callPackage ../development/libraries/ipu6-camera-hal {
    ipuVersion = "ipu6ep";
  };

  ipu6epmtl-camera-hal = callPackage ../development/libraries/ipu6-camera-hal {
    ipuVersion = "ipu6epmtl";
  };

  ivsc-firmware = callPackage ../os-specific/linux/firmware/ivsc-firmware { };

  iputils = hiPrio (callPackage ../os-specific/linux/iputils { });
  # hiPrio for collisions with inetutils (ping)

  iptables = callPackage ../os-specific/linux/iptables { };
  iptables-legacy = callPackage ../os-specific/linux/iptables { nftablesCompat = false; };
  iptables-nftables-compat = iptables;

  iptstate = callPackage ../os-specific/linux/iptstate { } ;

  ipset = callPackage ../os-specific/linux/ipset { };

  irqbalance = callPackage ../os-specific/linux/irqbalance { };

  itpp = callPackage ../development/libraries/science/math/itpp { };

  iw = callPackage ../os-specific/linux/iw { };

  iwd = callPackage ../os-specific/linux/iwd { };

  jool-cli = callPackage ../os-specific/linux/jool/cli.nix { };

  juju = callPackage ../applications/networking/juju { };

  jujuutils = callPackage ../os-specific/linux/jujuutils { };

  karabiner-elements = callPackage ../os-specific/darwin/karabiner-elements { };

  kbd = callPackage ../os-specific/linux/kbd { };

  kbdlight = callPackage ../os-specific/linux/kbdlight { };

  kinect-audio-setup = callPackage ../os-specific/linux/kinect-audio-setup { };

  kmscon = callPackage ../os-specific/linux/kmscon { };

  kmscube = callPackage ../os-specific/linux/kmscube { };

  kmsxx = callPackage ../development/libraries/kmsxx { };

  latencytop = callPackage ../os-specific/linux/latencytop { };

  ldm = callPackage ../os-specific/linux/ldm { };

  libaio = callPackage ../os-specific/linux/libaio { };

  libajantv2 = callPackage ../development/libraries/libajantv2 { };

  libargon2 = callPackage ../development/libraries/libargon2 { };

  libatasmart = callPackage ../os-specific/linux/libatasmart { };

  libax25 = callPackage ../development/libraries/libax25 { };

  libcgroup = callPackage ../os-specific/linux/libcgroup { };

  libdatachannel = callPackage ../development/libraries/libdatachannel { };

  libkrun = callPackage ../development/libraries/libkrun { };

  libkrun-sev = libkrun.override { sevVariant = true; };

  libkrunfw = callPackage ../development/libraries/libkrunfw { };

  libnl = callPackage ../os-specific/linux/libnl { };

  libnl-tiny = callPackage ../os-specific/linux/libnl-tiny { };

  libtraceevent = callPackage ../os-specific/linux/libtraceevent { };

  libtracefs = callPackage ../os-specific/linux/libtracefs { };

  lieer = callPackage ../applications/networking/lieer { };

  linuxConsoleTools = callPackage ../os-specific/linux/consoletools { };

  linux-wifi-hotspot = callPackage ../os-specific/linux/linux-wifi-hotspot { };

  linthesia = callPackage ../games/linthesia/default.nix { };

  libreelec-dvb-firmware = callPackage ../os-specific/linux/firmware/libreelec-dvb-firmware { };

  openiscsi = callPackage ../os-specific/linux/open-iscsi { };

  open-isns = callPackage ../os-specific/linux/open-isns { };

  osx-cpu-temp = callPackage ../os-specific/darwin/osx-cpu-temp {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  macfuse-stubs = callPackage ../os-specific/darwin/macfuse {
    inherit (darwin) libtapi;
    inherit (darwin.apple_sdk.frameworks) DiskArbitration;
  };

  osxsnarf = callPackage ../os-specific/darwin/osxsnarf { };

  plistwatch = callPackage ../os-specific/darwin/plistwatch { };

  power-calibrate = callPackage ../os-specific/linux/power-calibrate { };

  powercap = callPackage ../os-specific/linux/powercap { };

  powerstat = callPackage ../os-specific/linux/powerstat { };

  projecteur = libsForQt5.callPackage ../os-specific/linux/projecteur { };

  smemstat = callPackage ../os-specific/linux/smemstat { };

  tgt = callPackage ../tools/networking/tgt { };

  lkl = callPackage ../applications/virtualization/lkl { };
  lklWithFirewall = callPackage ../applications/virtualization/lkl { firewallSupport = true; };

  inherit (callPackages ../os-specific/linux/kernel-headers { inherit (pkgsBuildBuild) elf-header; })
    linuxHeaders makeLinuxHeaders;

  klibc = callPackage ../os-specific/linux/klibc { };

  klibcShrunk = lowPrio (callPackage ../os-specific/linux/klibc/shrunk.nix { });

  linuxKernel = recurseIntoAttrs (callPackage ./linux-kernels.nix { });

  inherit (linuxKernel) buildLinux linuxConfig kernelPatches;

  linuxPackagesFor = linuxKernel.packagesFor;

  hardenedLinuxPackagesFor = linuxKernel.hardenedPackagesFor;

  linuxManualConfig = linuxKernel.manualConfig;

  linuxPackages_custom = linuxKernel.customPackage;

  # This serves as a test for linuxPackages_custom
  linuxPackages_custom_tinyconfig_kernel = let
    base = linuxPackages.kernel;
    tinyLinuxPackages = linuxKernel.customPackage {
      inherit (base) version modDirVersion src;
      allowImportFromDerivation = false;
      configfile = linuxConfig {
        makeTarget = "tinyconfig";
        src = base.src;
      };
    };
    in tinyLinuxPackages.kernel;

  # The current default kernel / kernel modules.
  linuxPackages = linuxKernel.packageAliases.linux_default;
  linux = linuxPackages.kernel;

  linuxPackages_latest = linuxKernel.packageAliases.linux_latest;
  linux_latest = linuxPackages_latest.kernel;

  # Testing (rc) kernel
  linuxPackages_testing = linuxKernel.packages.linux_testing;
  linux_testing = linuxKernel.kernels.linux_testing;

  # Realtime kernel
  linuxPackages-rt = linuxKernel.packageAliases.linux_rt_default;
  linuxPackages-rt_latest = linuxKernel.packageAliases.linux_rt_latest;
  linux-rt = linuxPackages-rt.kernel;
  linux-rt_latest = linuxPackages-rt_latest.kernel;

  # hardened kernels
  linuxPackages_hardened = linuxKernel.packages.linux_hardened;
  linux_hardened = linuxPackages_hardened.kernel;
  linuxPackages_4_19_hardened = linuxKernel.packages.linux_4_19_hardened;
  linux_4_19_hardened = linuxPackages_4_19_hardened.kernel;
  linuxPackages_5_4_hardened = linuxKernel.packages.linux_5_4_hardened;
  linux_5_4_hardened = linuxKernel.kernels.linux_5_4_hardened;
  linuxPackages_5_10_hardened = linuxKernel.packages.linux_5_10_hardened;
  linux_5_10_hardened = linuxKernel.kernels.linux_5_10_hardened;
  linuxPackages_5_15_hardened = linuxKernel.packages.linux_5_15_hardened;
  linux_5_15_hardened = linuxKernel.kernels.linux_5_15_hardened;
  linuxPackages_6_1_hardened = linuxKernel.packages.linux_6_1_hardened;
  linux_6_1_hardened = linuxKernel.kernels.linux_6_1_hardened;
  linuxPackages_6_6_hardened = linuxKernel.packages.linux_6_6_hardened;
  linux_6_6_hardened = linuxKernel.kernels.linux_6_6_hardened;
  linuxPackages_6_8_hardened = linuxKernel.packages.linux_6_8_hardened;
  linux_6_8_hardened = linuxKernel.kernels.linux_6_8_hardened;
  linuxPackages_6_9_hardened = linuxKernel.packages.linux_6_9_hardened;
  linux_6_9_hardened = linuxKernel.kernels.linux_6_9_hardened;

  # GNU Linux-libre kernels
  linuxPackages-libre = linuxKernel.packages.linux_libre;
  linux-libre = linuxPackages-libre.kernel;
  linuxPackages_latest-libre = linuxKernel.packages.linux_latest_libre;
  linux_latest-libre = linuxPackages_latest-libre.kernel;

  # zen-kernel
  linuxPackages_zen = linuxKernel.packages.linux_zen;
  linux_zen = linuxPackages_zen.kernel;
  linuxPackages_lqx = linuxKernel.packages.linux_lqx;
  linux_lqx = linuxPackages_lqx.kernel;

  # XanMod kernel
  linuxPackages_xanmod = linuxKernel.packages.linux_xanmod;
  linux_xanmod = linuxKernel.kernels.linux_xanmod;
  linuxPackages_xanmod_stable = linuxKernel.packages.linux_xanmod_stable;
  linux_xanmod_stable = linuxKernel.kernels.linux_xanmod_stable;
  linuxPackages_xanmod_latest = linuxKernel.packages.linux_xanmod_latest;
  linux_xanmod_latest = linuxKernel.kernels.linux_xanmod_latest;

  linux-doc = callPackage ../os-specific/linux/kernel/htmldocs.nix { };

  cryptodev = linuxPackages.cryptodev;

  dpdk = callPackage ../os-specific/linux/dpdk { };

  keyutils = callPackage ../os-specific/linux/keyutils { };

  libselinux = callPackage ../os-specific/linux/libselinux { };

  libsemanage = callPackage ../os-specific/linux/libsemanage {
    python = python3;
  };

  librasterlite2 = callPackage ../development/libraries/librasterlite2 {
    inherit (darwin.apple_sdk.frameworks) ApplicationServices;
  };

  libraqm = callPackage ../development/libraries/libraqm { };

  libraw = callPackage ../development/libraries/libraw { };

  libraw1394 = callPackage ../development/libraries/libraw1394 { };

  librealsense = darwin.apple_sdk_11_0.callPackage ../development/libraries/librealsense { };

  librealsenseWithCuda = callPackage ../development/libraries/librealsense {
    cudaSupport = true;
    # librealsenseWithCuda doesn't build on gcc11. CUDA 11.3 is the last version
    # to use pre-gcc11, in particular gcc9.
    stdenv = gcc9Stdenv;
  };

  librealsenseWithoutCuda = callPackage ../development/libraries/librealsense {
    cudaSupport = false;
  };

  librealsense-gui = callPackage ../development/libraries/librealsense {
    enableGUI = true;
  };

  libsass = callPackage ../development/libraries/libsass { };

  libsepol = callPackage ../os-specific/linux/libsepol { };

  libsmbios = callPackage ../os-specific/linux/libsmbios { };

  libsurvive = callPackage ../development/libraries/libsurvive { };

  lm_sensors = callPackage ../os-specific/linux/lm-sensors { };

  lockdep = callPackage ../os-specific/linux/lockdep { };

  lsirec = callPackage ../os-specific/linux/lsirec { };

  lsiutil = callPackage ../os-specific/linux/lsiutil { };

  kaitai-struct-compiler = callPackage ../development/compilers/kaitai-struct-compiler { };

  kmod = callPackage ../os-specific/linux/kmod { };

  kmod-blacklist-ubuntu = callPackage ../os-specific/linux/kmod-blacklist-ubuntu { };

  kmod-debian-aliases = callPackage ../os-specific/linux/kmod-debian-aliases { };

  libcap = callPackage ../os-specific/linux/libcap { };

  libcap_ng = callPackage ../os-specific/linux/libcap-ng { };

  libnotify = callPackage ../development/libraries/libnotify { };

  libvolume_id = callPackage ../os-specific/linux/libvolume_id { };

  lsscsi = callPackage ../os-specific/linux/lsscsi { };

  lvm2 = callPackage ../os-specific/linux/lvm2/2_03.nix {
    # break the cyclic dependency:
    # util-linux (non-minimal) depends (optionally, but on by default) on systemd,
    # systemd (optionally, but on by default) on cryptsetup and cryptsetup depends on lvm2
    util-linux = util-linuxMinimal;
  };

  lvm2_dmeventd = lvm2.override {
    enableDmeventd = true;
    enableCmdlib = true;
  };
  lvm2_vdo = lvm2_dmeventd.override {
    enableVDO = true;
  };

  maddy = callPackage ../servers/maddy { };

  mbelib = callPackage ../development/libraries/audio/mbelib { };

  mbpfan = callPackage ../os-specific/linux/mbpfan { };

  mdadm = mdadm4;
  mdadm4 = callPackage ../os-specific/linux/mdadm { };

  mdevctl = callPackage ../os-specific/linux/mdevctl { };

  metastore = callPackage ../os-specific/linux/metastore { };

  minimal-bootstrap = recurseIntoAttrs (import ../os-specific/linux/minimal-bootstrap {
    inherit (stdenv) buildPlatform hostPlatform;
    inherit lib config;
    fetchurl = import ../build-support/fetchurl/boot.nix {
      inherit (stdenv.buildPlatform) system;
    };
    checkMeta = callPackage ../stdenv/generic/check-meta.nix { };
  });
  minimal-bootstrap-sources = callPackage ../os-specific/linux/minimal-bootstrap/stage0-posix/bootstrap-sources.nix { };
  make-minimal-bootstrap-sources = callPackage ../os-specific/linux/minimal-bootstrap/stage0-posix/make-bootstrap-sources.nix {
    inherit (stdenv) hostPlatform;
  };

  mingetty = callPackage ../os-specific/linux/mingetty { };

  miraclecast = callPackage ../os-specific/linux/miraclecast { };

  mkinitcpio-nfs-utils = callPackage ../os-specific/linux/mkinitcpio-nfs-utils { };

  mmc-utils = callPackage ../os-specific/linux/mmc-utils { };

  aggregateModules = modules:
    callPackage ../os-specific/linux/kmod/aggregator.nix {
      inherit (buildPackages) kmod;
      inherit modules;
    };

  multipart-parser-c = callPackage ../development/libraries/multipart-parser-c { };

  multipath-tools = callPackage ../os-specific/linux/multipath-tools { };

  musl = callPackage ../os-specific/linux/musl { };

  musl-fts = callPackage ../os-specific/linux/musl-fts { };
  musl-obstack = callPackage ../os-specific/linux/musl-obstack { };

  nsh = callPackage ../shells/nsh { };

  nushell = darwin.apple_sdk_11_0.callPackage ../shells/nushell {
    inherit (darwin.apple_sdk_11_0) Libsystem;
    inherit (darwin.apple_sdk_11_0.frameworks) AppKit Security;
  };

  nushellFull = nushell.override { additionalFeatures = p: p ++ ["dataframe"]; };

  nu_scripts = callPackage ../shells/nushell/nu_scripts { };

  nushellPlugins = recurseIntoAttrs (callPackage ../shells/nushell/plugins {
    inherit (darwin.apple_sdk.frameworks) Security;
    inherit (darwin.apple_sdk_11_0.frameworks) IOKit CoreFoundation Foundation;
  });

  nettools = if stdenv.isLinux
    then callPackage ../os-specific/linux/net-tools { }
    else unixtools.nettools;

  nftables = callPackage ../os-specific/linux/nftables { };

  noah = callPackage ../os-specific/darwin/noah {
    inherit (darwin.apple_sdk.frameworks) Hypervisor;
  };

  numactl = callPackage ../os-specific/linux/numactl { };

  numad = callPackage ../os-specific/linux/numad { };

  nvme-cli = callPackage ../os-specific/linux/nvme-cli { };

  nvmet-cli = callPackage ../os-specific/linux/nvmet-cli { };

  system76-firmware = callPackage ../os-specific/linux/firmware/system76-firmware { };

  system76-keyboard-configurator = callPackage ../applications/misc/system76-keyboard-configurator { };

  ocf-resource-agents = callPackage ../os-specific/linux/ocf-resource-agents { };

  open-vm-tools = callPackage ../applications/virtualization/open-vm-tools { };
  open-vm-tools-headless = open-vm-tools.override { withX = false; };

  oxtools = callPackage ../os-specific/linux/oxtools { };

  air = callPackage ../development/tools/air { };

  delve = callPackage ../development/tools/delve { };

  dep2nix = callPackage ../development/tools/dep2nix { };

  easyjson = callPackage ../development/tools/easyjson { };

  iferr = callPackage ../development/tools/iferr { };

  gci = callPackage ../development/tools/gci { };

  gcov2lcov = callPackage ../development/tools/gcov2lcov { };

  ginkgo = callPackage ../development/tools/ginkgo { };

  gdlv = darwin.apple_sdk_11_0.callPackage ../development/tools/gdlv {
    inherit (darwin.apple_sdk_11_0.frameworks) Foundation CoreGraphics Metal AppKit;
  };

  go-bindata = callPackage ../development/tools/go-bindata { };

  go-bindata-assetfs = callPackage ../development/tools/go-bindata-assetfs { };

  go-minimock = callPackage ../development/tools/go-minimock { };

  go-protobuf = callPackage ../development/tools/go-protobuf { };

  go-symbols = callPackage ../development/tools/go-symbols { };

  go-toml = callPackage ../development/tools/go-toml { };

  go-outline = callPackage ../development/tools/go-outline { };

  gocode-gomod = callPackage ../development/tools/gocode-gomod { };

  goconst = callPackage ../development/tools/goconst { };

  goconvey = callPackage ../development/tools/goconvey { };

  go-callvis = callPackage ../development/tools/go-callvis { };

  gotags = callPackage ../development/tools/gotags { };

  go-task = callPackage ../development/tools/go-task { };

  golines = callPackage ../development/tools/golines { };

  golint = callPackage ../development/tools/golint { };

  golangci-lint = callPackage ../development/tools/golangci-lint { };

  golangci-lint-langserver = callPackage ../development/tools/golangci-lint-langserver { };

  gocyclo = callPackage ../development/tools/gocyclo { };

  godef = callPackage ../development/tools/godef { };

  goimports-reviser = callPackage ../development/tools/goimports-reviser { };

  gopkgs = callPackage ../development/tools/gopkgs { };

  gosec = callPackage ../development/tools/gosec { };

  gotemplate = callPackage ../development/tools/gotemplate { };

  gotraceui = callPackage ../development/tools/gotraceui { };

  govers = callPackage ../development/tools/govers { };

  goverview = callPackage ../tools/security/goverview { };

  go-tools = callPackage ../development/tools/go-tools {
    buildGoModule = buildGo122Module;
  };

  gotest = callPackage ../development/tools/gotest { };

  gotestfmt = callPackage ../development/tools/gotestfmt { };

  gotools = callPackage ../development/tools/gotools { };

  gotop = callPackage ../tools/system/gotop {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  go-migrate = callPackage ../development/tools/go-migrate { };

  go-jet = callPackage ../development/tools/go-jet { };

  go-mockery = callPackage ../development/tools/go-mockery { };

  go-mod-graph-chart = callPackage ../development/tools/go-mod-graph-chart { };

  gomacro = callPackage ../development/tools/gomacro { };

  gomodifytags = callPackage ../development/tools/gomodifytags { };

  gops = callPackage ../development/tools/gops { };

  gore = callPackage ../development/tools/gore { };

  goresym = callPackage ../development/tools/goresym { };

  gotests = callPackage ../development/tools/gotests { };

  gotestsum = callPackage ../development/tools/gotestsum { };

  gqlgenc = callPackage ../development/tools/gqlgenc { };

  impl = callPackage ../development/tools/impl { };

  moq = callPackage ../development/tools/moq { };

  quicktemplate = callPackage ../development/tools/quicktemplate { };

  linux_logo = callPackage ../tools/misc/linux-logo { };

  linux-pam = callPackage ../os-specific/linux/pam { };

  nss_ldap = callPackage ../os-specific/linux/nss_ldap { };

  octomap = callPackage ../development/libraries/octomap { };

  odin = callPackage ../development/compilers/odin {
    inherit (pkgs.darwin.apple_sdk_11_0) MacOSX-SDK;
    inherit (pkgs.darwin.apple_sdk_11_0.frameworks) Security;
  };

  odp-dpdk = callPackage ../os-specific/linux/odp-dpdk { };

  odroid-xu3-bootloader = callPackage ../tools/misc/odroid-xu3-bootloader { };

  ofono = callPackage ../tools/networking/ofono { };

  okapi = callPackage ../development/libraries/okapi {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ols = callPackage ../development/tools/ols { };

  openpam = callPackage ../development/libraries/openpam { };

  openbsm = callPackage ../development/libraries/openbsm { };

  pagemon = callPackage ../os-specific/linux/pagemon { };

  pam = if stdenv.isLinux then linux-pam else openpam;

  # pam_bioapi ( see http://www.thinkwiki.org/wiki/How_to_enable_the_fingerprint_reader )

  pam_ccreds = callPackage ../os-specific/linux/pam_ccreds { };

  pam_dp9ik = callPackage ../os-specific/linux/pam_dp9ik { };

  pam_gnupg = callPackage ../os-specific/linux/pam_gnupg { };

  pam_krb5 = callPackage ../os-specific/linux/pam_krb5 { };

  pam_ldap = callPackage ../os-specific/linux/pam_ldap { };

  pam_mktemp = callPackage ../os-specific/linux/pam_mktemp { };

  pam_mount = callPackage ../os-specific/linux/pam_mount { };

  pam_mysql = callPackage ../os-specific/linux/pam_mysql { };

  pam_p11 = callPackage ../os-specific/linux/pam_p11 { };

  pam_pgsql = callPackage ../os-specific/linux/pam_pgsql { };

  pam_rssh = callPackage ../os-specific/linux/pam_rssh { };

  pam_ssh_agent_auth = callPackage ../os-specific/linux/pam_ssh_agent_auth { };

  pam_tmpdir = callPackage ../os-specific/linux/pam_tmpdir { };

  pam_u2f = callPackage ../os-specific/linux/pam_u2f { };

  pam_ussh = callPackage ../os-specific/linux/pam_ussh { };

  paxctl = callPackage ../os-specific/linux/paxctl { };

  paxtest = callPackage ../os-specific/linux/paxtest { };

  pax-utils = callPackage ../os-specific/linux/pax-utils { };

  pcmciaUtils = callPackage ../os-specific/linux/pcmciautils { };

  pcstat = callPackage ../tools/system/pcstat { };

  perf-tools = callPackage ../os-specific/linux/perf-tools { };

  pgrok = callPackage ../tools/networking/pgrok { };

  picoprobe-udev-rules = callPackage ../os-specific/linux/picoprobe-udev-rules { };

  pipes = callPackage ../misc/screensavers/pipes { };

  pipes-rs = callPackage ../misc/screensavers/pipes-rs { };

  pipework = callPackage ../os-specific/linux/pipework { };

  pktgen = callPackage ../os-specific/linux/pktgen { };

  plymouth = callPackage ../os-specific/linux/plymouth { };

  pmount = callPackage ../os-specific/linux/pmount { };

  pmutils = callPackage ../os-specific/linux/pm-utils { };

  policycoreutils = callPackage ../os-specific/linux/policycoreutils { };

  semodule-utils = callPackage ../os-specific/linux/semodule-utils { };

  pdns = callPackage ../servers/dns/pdns { };

  powerdns-admin = callPackage ../applications/networking/powerdns-admin { };

  dnsdist = callPackage ../servers/dns/dnsdist { };

  powertop = callPackage ../os-specific/linux/powertop { };

  pps-tools = callPackage ../os-specific/linux/pps-tools { };

  procps = if stdenv.isLinux
    then callPackage ../os-specific/linux/procps-ng { }
    else unixtools.procps;

  procdump = callPackage ../os-specific/linux/procdump { };

  prototool = callPackage ../development/tools/prototool { };

  qemu_kvm = lowPrio (qemu.override { hostCpuOnly = true; });
  qemu_full = lowPrio (qemu.override { smbdSupport = true; cephSupport = true; glusterfsSupport = true; });

  # See `xenPackages` source for explanations.
  # Building with `xen` instead of `xen-slim` is possible, but makes no sense.
  qemu_xen = lowPrio (qemu.override { hostCpuOnly = true; xenSupport = true; xen = xen-slim; });
  qemu_xen-light = lowPrio (qemu.override { hostCpuOnly = true; xenSupport = true; xen = xen-light; });
  qemu_xen_4_15 = lowPrio (qemu.override { hostCpuOnly = true; xenSupport = true; xen = xen_4_15-slim; });
  qemu_xen_4_15-light = lowPrio (qemu.override { hostCpuOnly = true; xenSupport = true; xen = xen_4_15-light; });

  qemu_test = lowPrio (qemu.override { hostCpuOnly = true; nixosTestRunner = true; });

  quick-lint-js = callPackage ../development/tools/quick-lint-js { };

  linux-firmware = callPackage ../os-specific/linux/firmware/linux-firmware { };

  qmk-udev-rules = callPackage ../os-specific/linux/qmk-udev-rules { };

  radeontools = callPackage ../os-specific/linux/radeontools { };

  radeontop = callPackage ../os-specific/linux/radeontop { };

  rasdaemon = callPackage ../os-specific/linux/rasdaemon { };

  raspberrypifw = callPackage ../os-specific/linux/firmware/raspberrypi { };
  raspberrypiWirelessFirmware = callPackage ../os-specific/linux/firmware/raspberrypi-wireless { };

  raspberrypi-armstubs = callPackage ../os-specific/linux/firmware/raspberrypi/armstubs.nix { };

  reap = callPackage ../os-specific/linux/reap { };

  rectangle = callPackage ../os-specific/darwin/rectangle { };

  regionset = callPackage ../os-specific/linux/regionset { };

  rfkill_udev = callPackage ../os-specific/linux/rfkill/udev.nix { };

  riscv-pk = callPackage ../misc/riscv-pk { };

  ristate = callPackage ../tools/misc/ristate { };

  roccat-tools = callPackage ../os-specific/linux/roccat-tools { };

  rojo = callPackage ../development/tools/rojo { };

  mediamtx = callPackage ../servers/mediamtx { };

  rtkit = callPackage ../os-specific/linux/rtkit { };

  rt-tests = callPackage ../os-specific/linux/rt-tests { };

  rt5677-firmware = callPackage ../os-specific/linux/firmware/rt5677 { };

  rtl8192su-firmware = callPackage ../os-specific/linux/firmware/rtl8192su-firmware { };

  rtl8761b-firmware = callPackage ../os-specific/linux/firmware/rtl8761b-firmware { };

  rtw88-firmware = callPackage ../os-specific/linux/firmware/rtw88-firmware { };

  rvvm = callPackage ../applications/virtualization/rvvm { };

  s3ql = callPackage ../tools/backup/s3ql { };

  sasutils = callPackage ../os-specific/linux/sasutils { };

  sass = callPackage ../development/tools/sass { };

  sassc = callPackage ../development/tools/sassc { };

  scanmem = callPackage ../tools/misc/scanmem { };

  schedtool = callPackage ../os-specific/linux/schedtool { };

  sddm-chili-theme = libsForQt5.callPackage ../data/themes/chili-sddm { };

  sdparm = callPackage ../os-specific/linux/sdparm { };

  sdrangel = libsForQt5.callPackage ../applications/radio/sdrangel {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
  };

  setools = callPackage ../os-specific/linux/setools { };

  seturgent = callPackage ../os-specific/linux/seturgent { };

  sgx-azure-dcap-client = callPackage ../os-specific/linux/sgx/azure-dcap-client { };

  sgx-sdk = callPackage ../os-specific/linux/sgx/sdk { };

  sgx-ssl = callPackage ../os-specific/linux/sgx/ssl { };

  sgx-psw = callPackage ../os-specific/linux/sgx/psw {
    protobuf = protobuf_21;
  };

  shadow = callPackage ../os-specific/linux/shadow { };

  shortcat = callPackage ../os-specific/darwin/shortcat { };

  sinit = callPackage ../os-specific/linux/sinit {
    rcinit = "/etc/rc.d/rc.init";
    rcshutdown = "/etc/rc.d/rc.shutdown";
  };

  skopeo = callPackage ../development/tools/skopeo { };

  srisum = callPackage ../tools/misc/srisum { };

  smem = callPackage ../os-specific/linux/smem { };

  smimesign = callPackage ../os-specific/darwin/smimesign { };

  statik = callPackage ../development/tools/statik { };

  statifier = callPackage ../os-specific/linux/statifier { };

  steamos-devkit = callPackage ../development/tools/steamos-devkit { };

  swiftdefaultapps = callPackage ../os-specific/darwin/swiftdefaultapps { };

  sysdig = callPackage ../os-specific/linux/sysdig {
    kernel = null;
  }; # sysdig is a client, for a driver look at linuxPackagesFor

  sysfsutils = callPackage ../os-specific/linux/sysfsutils { };

  sysprof = callPackage ../development/tools/profiling/sysprof { };

  libsysprof-capture = callPackage ../development/tools/profiling/sysprof/capture.nix { };

  sysklogd = callPackage ../os-specific/linux/sysklogd { };

  syslinux = callPackage ../os-specific/linux/syslinux { };

  sysstat = callPackage ../os-specific/linux/sysstat { };

  systemd = callPackage ../os-specific/linux/systemd {
    # break some cyclic dependencies
    util-linux = util-linuxMinimal;
    # provide a super minimal gnupg used for systemd-machined
    gnupg = gnupg.override {
      enableMinimal = true;
      guiSupport = false;
    };
  };
  systemdMinimal = systemd.override {
    pname = "systemd-minimal";
    withAcl = false;
    withAnalyze = false;
    withApparmor = false;
    withAudit = false;
    withCompression = false;
    withCoredump = false;
    withCryptsetup = false;
    withRepart = false;
    withDocumentation = false;
    withEfi = false;
    withFido2 = false;
    withHostnamed = false;
    withHomed = false;
    withHwdb = false;
    withImportd = false;
    withIptables = false;
    withLibBPF = false;
    withLibidn2 = false;
    withLocaled = false;
    withLogind = false;
    withMachined = false;
    withNetworkd = false;
    withNss = false;
    withOomd = false;
    withPCRE2 = false;
    withPam = false;
    withPolkit = false;
    withPortabled = false;
    withRemote = false;
    withResolved = false;
    withShellCompletions = false;
    withSysupdate = false;
    withSysusers = false;
    withTimedated = false;
    withTimesyncd = false;
    withTpm2Tss = false;
    withUserDb = false;
    withUkify = false;
    withBootloader = false;
    withPasswordQuality = false;
    withVmspawn = false;
    withQrencode = false;
  };
  systemdLibs = systemdMinimal.override {
    pname = "systemd-minimal-libs";
    buildLibsOnly = true;
  };
  # We do not want to include ukify in the normal systemd attribute as it
  # relies on Python at runtime.
  systemdUkify = systemd.override {
    withUkify = true;
  };

  udev =
    if (with stdenv.hostPlatform; isLinux && isStatic) then libudev-zero
    else systemdLibs;

  systemd-wait = callPackage ../os-specific/linux/systemd-wait { };

  sysvinit = callPackage ../os-specific/linux/sysvinit { };

  sysvtools = sysvinit.override {
    withoutInitTools = true;
  };

  # FIXME: `tcp-wrapper' is actually not OS-specific.
  tcp_wrappers = callPackage ../os-specific/linux/tcp-wrappers { };

  teck-udev-rules = callPackage ../os-specific/linux/teck-udev-rules { };

  tiptop = callPackage ../os-specific/linux/tiptop { };

  tpacpi-bat = callPackage ../os-specific/linux/tpacpi-bat { };

  trickster = callPackage ../servers/trickster/trickster.nix { };

  trinity = callPackage ../os-specific/linux/trinity { };

  trino-cli = callPackage ../development/tools/database/trino-cli { };

  trinsic-cli = callPackage ../tools/admin/trinsic-cli {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  trust-dns = callPackage ../servers/dns/trust-dns { };

  trustymail = callPackage ../tools/security/trustymail { };

  tunctl = callPackage ../os-specific/linux/tunctl { };

  twa = callPackage ../tools/networking/twa { };

  twingate = callPackage ../applications/networking/twingate { };

  # Upstream U-Boots:
  inherit (callPackage ../misc/uboot {})
    buildUBoot
    ubootTools
    ubootA20OlinuxinoLime
    ubootA20OlinuxinoLime2EMMC
    ubootBananaPi
    ubootBananaPim3
    ubootBananaPim64
    ubootAmx335xEVM
    ubootClearfog
    ubootCubieboard2
    ubootGuruplug
    ubootJetsonTK1
    ubootLibreTechCC
    ubootNanoPCT4
    ubootNanoPCT6
    ubootNovena
    ubootOdroidC2
    ubootOdroidXU3
    ubootOlimexA64Olinuxino
    ubootOlimexA64Teres1
    ubootOrangePi3
    ubootOrangePi5
    ubootOrangePiPc
    ubootOrangePiZeroPlus2H5
    ubootOrangePiZero
    ubootOrangePiZero2
    ubootPcduino3Nano
    ubootPine64
    ubootPine64LTS
    ubootPinebook
    ubootPinebookPro
    ubootQemuAarch64
    ubootQemuArm
    ubootQemuRiscv64Smode
    ubootQemuX86
    ubootRaspberryPi
    ubootRaspberryPi2
    ubootRaspberryPi3_32bit
    ubootRaspberryPi3_64bit
    ubootRaspberryPi4_32bit
    ubootRaspberryPi4_64bit
    ubootRaspberryPiZero
    ubootRock4CPlus
    ubootRock5ModelB
    ubootRock64
    ubootRock64v2
    ubootRockPi4
    ubootRockPro64
    ubootROCPCRK3399
    ubootSheevaplug
    ubootSopine
    ubootTuringRK1
    ubootUtilite
    ubootWandboard
    ;

  # Upstream Barebox:
  inherit (callPackage ../misc/barebox {})
    buildBarebox
    bareboxTools;

  uclibc-ng-cross = uclibc-ng.override {
    stdenv = crossLibcStdenv;
  };

  # Aliases
  uclibc = uclibc-ng;
  uclibcCross = uclibc-ng-cross;

  eudev = callPackage ../by-name/eu/eudev/package.nix {
    util-linux = util-linuxMinimal;
  };

  libudev0-shim = callPackage ../os-specific/linux/libudev0-shim { };

  udisks2 = callPackage ../os-specific/linux/udisks/2-default.nix { };
  udisks = udisks2;

  ugtrain = callPackage ../tools/misc/ugtrain { };

  unscd = callPackage ../os-specific/linux/unscd { };

  untie = callPackage ../os-specific/linux/untie { };

  upower = callPackage ../os-specific/linux/upower { };

  upscayl = callPackage ../applications/graphics/upscayl { };

  usbguard = callPackage ../os-specific/linux/usbguard { };

  usbguard-notifier = callPackage ../os-specific/linux/usbguard-notifier { };

  usbrelay = callPackage ../os-specific/linux/usbrelay { };
  usbrelayd = callPackage ../os-specific/linux/usbrelay/daemon.nix { };

  usbtop = callPackage ../os-specific/linux/usbtop { };

  usbutils = callPackage ../os-specific/linux/usbutils { };

  userhosts = callPackage ../tools/networking/userhosts { };

  usermount = callPackage ../os-specific/linux/usermount { };

  util-linux = callPackage ../os-specific/linux/util-linux { };

  util-linuxMinimal = util-linux.override {
    nlsSupport = false;
    ncursesSupport = false;
    systemdSupport = false;
    translateManpages = false;
  };

  v4l-utils = qt5.callPackage ../os-specific/linux/v4l-utils { };

  v4l2-relayd = callPackage ../os-specific/linux/v4l2-relayd { };

  vendir = callPackage ../development/tools/vendir { };

  vndr = callPackage ../development/tools/vndr { };

  vdo = callPackage ../os-specific/linux/vdo { };

  windows = callPackages ../os-specific/windows {};

  wirelesstools = callPackage ../os-specific/linux/wireless-tools { };

  wooting-udev-rules = callPackage ../os-specific/linux/wooting-udev-rules { };

  teensy-udev-rules = callPackage ../os-specific/linux/teensy-udev-rules { };

  sensible-side-buttons = callPackage ../os-specific/darwin/sensible-side-buttons { };

  wpa_supplicant = callPackage ../os-specific/linux/wpa_supplicant { };

  wpa_supplicant_ro_ssids = wpa_supplicant.override {
    readOnlyModeSSIDs = true;
  };

  wpa_supplicant_gui = libsForQt5.callPackage ../os-specific/linux/wpa_supplicant/gui.nix { };

  xf86_input_cmt = callPackage ../os-specific/linux/xf86-input-cmt { };

  xf86_input_wacom = callPackage ../os-specific/linux/xf86-input-wacom { };

  xf86_video_nested = callPackage ../os-specific/linux/xf86-video-nested { };

  xilinx-bootgen = callPackage ../tools/misc/xilinx-bootgen { };

  xorg_sys_opengl = callPackage ../os-specific/linux/opengl/xorg-sys { };

  zd1211fw = callPackage ../os-specific/linux/firmware/zd1211 { };

  zenmonitor = callPackage ../os-specific/linux/zenmonitor { };

  inherit
    ({
      zfs_2_1 = callPackage ../os-specific/linux/zfs/2_1.nix {
        configFile = "user";
      };
      zfs_2_2 = callPackage ../os-specific/linux/zfs/2_2.nix {
        configFile = "user";
      };
      zfs_unstable = callPackage ../os-specific/linux/zfs/unstable.nix {
        configFile = "user";
      };
    })
    zfs_2_1
    zfs_2_2
    zfs_unstable;
  zfs = zfs_2_2;

  ### DATA

  _3270font = callPackage ../data/fonts/3270font { };

  adapta-backgrounds = callPackage ../data/misc/adapta-backgrounds { };

  adapta-gtk-theme = callPackage ../data/themes/adapta { };

  adapta-kde-theme = callPackage ../data/themes/adapta-kde { };

  adementary-theme = callPackage ../data/themes/adementary { };

  adi1090x-plymouth-themes = callPackage ../data/themes/adi1090x-plymouth-themes { };

  adwaita-qt = libsForQt5.callPackage ../data/themes/adwaita-qt { };

  adwaita-qt6 = qt6Packages.callPackage ../data/themes/adwaita-qt {
    useQt6 = true;
  };

  adw-gtk3 = callPackage ../data/themes/adw-gtk3 { };

  agave = callPackage ../data/fonts/agave { };

  aileron = callPackage ../data/fonts/aileron { };

  alacritty-theme = callPackage ../data/themes/alacritty-theme { };

  albatross = callPackage ../data/themes/albatross { };

  alegreya = callPackage ../data/fonts/alegreya { };

  alegreya-sans = callPackage ../data/fonts/alegreya-sans { };

  alice = callPackage ../data/fonts/alice { };

  alkalami = callPackage ../data/fonts/alkalami { };

  amber-theme = callPackage ../data/themes/amber { };

  amiri = callPackage ../data/fonts/amiri { };

  anarchism = callPackage ../data/documentation/anarchism { };

  ananicy = callPackage ../misc/ananicy { };

  andagii = callPackage ../data/fonts/andagii { };

  andika = callPackage ../data/fonts/andika { };

  androguard = with python3.pkgs; toPythonApplication androguard;

  android-udev-rules = callPackage ../os-specific/linux/android-udev-rules { };

  andromeda-gtk-theme = libsForQt5.callPackage ../data/themes/andromeda-gtk-theme { };

  ankacoder = callPackage ../data/fonts/ankacoder { };
  ankacoder-condensed = callPackage ../data/fonts/ankacoder/condensed.nix { };

  annapurna-sil = callPackage ../data/fonts/annapurna-sil { };

  anonymousPro = callPackage ../data/fonts/anonymous-pro { };

  ant-theme = callPackage ../data/themes/ant-theme/ant.nix { };

  ant-bloody-theme = callPackage ../data/themes/ant-theme/ant-bloody.nix { };

  dracula-theme = callPackage ../data/themes/dracula-theme { };

  ant-nebula-theme = callPackage ../data/themes/ant-theme/ant-nebula.nix { };

  arc-icon-theme = callPackage ../data/icons/arc-icon-theme { };

  arc-kde-theme = callPackage ../data/themes/arc-kde { };

  arc-theme = callPackage ../data/themes/arc { };

  arkpandora_ttf = callPackage ../data/fonts/arkpandora { };

  aurulent-sans = callPackage ../data/fonts/aurulent-sans { };

  ayu-theme-gtk = callPackage ../data/themes/ayu-theme-gtk { };

  b612  = callPackage ../data/fonts/b612 { };

  babelstone-han = callPackage ../data/fonts/babelstone-han { };

  baekmuk-ttf = callPackage ../data/fonts/baekmuk-ttf { };

  bakoma_ttf = callPackage ../data/fonts/bakoma-ttf { };

  barlow = callPackage ../data/fonts/barlow { };

  base16-schemes = callPackage ../data/themes/base16-schemes { };

  bgnet = callPackage ../data/documentation/bgnet { };

  bibata-extra-cursors = callPackage ../data/icons/bibata-cursors/extra.nix { };
  bibata-cursors-translucent = callPackage ../data/icons/bibata-cursors/translucent.nix { };

  apple-cursor = callPackage ../data/icons/apple-cursor { };

  blackbird = callPackage ../data/themes/blackbird { };

  blackout = callPackage ../data/fonts/blackout { };

  breath-theme = libsForQt5.callPackage ../data/themes/breath-theme { };

  brise = callPackage ../data/misc/brise { };

  apl386 = callPackage ../data/fonts/apl386 { };

  bqn386 = callPackage ../data/fonts/bqn386 { };

  cacert = callPackage ../data/misc/cacert { };

  caladea = callPackage ../data/fonts/caladea { };

  canta-theme = callPackage ../data/themes/canta { };

  cantarell-fonts = callPackage ../data/fonts/cantarell-fonts { };

  capitaine-cursors = callPackage ../data/icons/capitaine-cursors { };

  capitaine-cursors-themed = callPackage ../data/icons/capitaine-cursors-themed { };

  carlito = callPackage ../data/fonts/carlito { };

  cascadia-code = callPackage ../data/fonts/cascadia-code { };

  ccsymbols = callPackage ../data/fonts/ccsymbols { };

  charis-sil = callPackage ../data/fonts/charis-sil { };

  changie = callPackage ../development/tools/changie { };

  cherry = callPackage ../data/fonts/cherry { };

  chonburi-font = callPackage ../data/fonts/chonburi { };

  chunk = callPackage ../data/fonts/chunk { };

  cldr-annotations = callPackage ../data/misc/cldr-annotations { };

  clearlooks-phenix = callPackage ../data/themes/clearlooks-phenix { };

  cnspec = callPackage ../tools/security/cnspec {
    buildGoModule = buildGo122Module;
  };

  cnstrokeorder = callPackage ../data/fonts/cnstrokeorder { };

  colloid-gtk-theme = callPackage ../data/themes/colloid-gtk-theme { };

  colloid-icon-theme = callPackage ../data/icons/colloid-icon-theme { };

  comfortaa = callPackage ../data/fonts/comfortaa { };

  colloid-kde = libsForQt5.callPackage ../data/themes/colloid-kde { };

  comic-mono = callPackage ../data/fonts/comic-mono { };

  comic-neue = callPackage ../data/fonts/comic-neue { };

  comic-relief = callPackage ../data/fonts/comic-relief { };

  comixcursors = callPackage ../data/icons/comixcursors { };

  courier-prime = callPackage ../data/fonts/courier-prime { };

  cozette = callPackage ../data/fonts/cozette { };

  culmus = callPackage ../data/fonts/culmus { };

  clearlyU = callPackage ../data/fonts/clearlyU { };

  cm_unicode = callPackage ../data/fonts/cm-unicode { };

  creep = callPackage ../data/fonts/creep { };

  crimson = callPackage ../data/fonts/crimson { };

  crimson-pro = callPackage ../data/fonts/crimson-pro { };

  dejavu_fonts = lowPrio (callPackage ../data/fonts/dejavu-fonts {});

  # solve collision for nix-env before https://github.com/NixOS/nix/pull/815
  dejavu_fontsEnv = buildEnv {
    name = dejavu_fonts.name;
    paths = [ dejavu_fonts.out ];
  };

  dina-font = callPackage ../data/fonts/dina { };

  dm-sans = callPackage ../data/fonts/dm-sans { };
  dns-root-data = callPackage ../data/misc/dns-root-data { };

  docbook5 = callPackage ../data/sgml+xml/schemas/docbook-5.0 { };

  docbook_sgml_dtd_31 = callPackage ../data/sgml+xml/schemas/sgml-dtd/docbook/3.1.nix { };

  docbook_sgml_dtd_41 = callPackage ../data/sgml+xml/schemas/sgml-dtd/docbook/4.1.nix { };

  docbook_xml_dtd_412 = callPackage ../data/sgml+xml/schemas/xml-dtd/docbook/4.1.2.nix { };

  docbook_xml_dtd_42 = callPackage ../data/sgml+xml/schemas/xml-dtd/docbook/4.2.nix { };

  docbook_xml_dtd_43 = callPackage ../data/sgml+xml/schemas/xml-dtd/docbook/4.3.nix { };

  docbook_xml_dtd_44 = callPackage ../data/sgml+xml/schemas/xml-dtd/docbook/4.4.nix { };

  docbook_xml_dtd_45 = callPackage ../data/sgml+xml/schemas/xml-dtd/docbook/4.5.nix { };

  docbook_xml_ebnf_dtd = callPackage ../data/sgml+xml/schemas/xml-dtd/docbook-ebnf { };

  inherit (callPackages ../data/sgml+xml/stylesheets/xslt/docbook-xsl { })
    docbook-xsl-nons
    docbook-xsl-ns;

  # TODO: move this to aliases
  docbook_xsl = docbook-xsl-nons;
  docbook_xsl_ns = docbook-xsl-ns;

  documentation-highlighter = callPackage ../misc/documentation-highlighter { };

  mmdoc = callPackage ../tools/text/mmdoc { };

  documize-community = callPackage ../servers/documize-community { };

  doulos-sil = callPackage ../data/fonts/doulos-sil { };

  cabin = callPackage ../data/fonts/cabin { };

  camingo-code = callPackage ../data/fonts/camingo-code { };

  combinatorial_designs = callPackage ../data/misc/combinatorial_designs { };

  commit-mono = callPackage ../data/fonts/commit-mono { };

  cooper-hewitt = callPackage ../data/fonts/cooper-hewitt { };

  d2coding = callPackage ../data/fonts/d2coding { };

  dosis = callPackage ../data/fonts/dosis { };

  dosemu_fonts = callPackage ../data/fonts/dosemu-fonts { };

  dotcolon-fonts = callPackage ../data/fonts/dotcolon-fonts { };

  dracula-icon-theme = callPackage ../data/icons/dracula-icon-theme { };

  e17gtk = callPackage ../data/themes/e17gtk { };

  eb-garamond = callPackage ../data/fonts/eb-garamond { };

  edukai = callPackage ../data/fonts/edukai { };

  eduli = callPackage ../data/fonts/eduli { };

  epapirus-icon-theme = papirus-icon-theme.override { withElementary = true; };

  moeli = eduli;

  edusong = callPackage ../data/fonts/edusong { };

  efont-unicode = callPackage ../data/fonts/efont-unicode { };

  elliptic_curves = callPackage ../data/misc/elliptic_curves { };

  equilux-theme = callPackage ../data/themes/equilux-theme { };

  eunomia = callPackage ../data/fonts/eunomia { };

  fa_1 = callPackage ../data/fonts/fa_1 { };

  f1_8 = callPackage ../data/fonts/f1_8 { };

  f5_6 = callPackage ../data/fonts/f5_6 { };

  faba-icon-theme = callPackage ../data/icons/faba-icon-theme { };

  faba-mono-icons = callPackage ../data/icons/faba-mono-icons { };

  ferrum = callPackage ../data/fonts/ferrum { };

  fg-virgil = callPackage ../data/fonts/fg-virgil { };

  fixedsys-excelsior = callPackage ../data/fonts/fixedsys-excelsior { };

  graphs = callPackage ../data/misc/graphs { };

  emacs-all-the-icons-fonts = callPackage ../data/fonts/emacs-all-the-icons-fonts { };

  emojione = callPackage ../data/fonts/emojione {
    inherit (nodePackages) svgo;
  };

  emojipick = callPackage ../applications/misc/emojipick { };

  encode-sans = callPackage ../data/fonts/encode-sans { };

  envypn-font = callPackage ../data/fonts/envypn-font { };

  ezra-sil = callPackage ../data/fonts/ezra-sil { };

  fantasque-sans-mono = callPackage ../data/fonts/fantasque-sans-mono { };

  fanwood = callPackage ../data/fonts/fanwood { };

  fira-code = callPackage ../data/fonts/fira-code { };
  fira-code-symbols = callPackage ../data/fonts/fira-code/symbols.nix { };
  fira-code-nerdfont = nerdfonts.override {
    fonts = [ "FiraCode" ];
  };

  fira-go = callPackage ../data/fonts/fira-go { };

  flat-remix-icon-theme = callPackage ../data/icons/flat-remix-icon-theme {
    inherit (plasma5Packages) breeze-icons;
  };
  flat-remix-gtk = callPackage ../data/themes/flat-remix-gtk { };
  flat-remix-gnome = callPackage ../data/themes/flat-remix-gnome { };

  fluent-gtk-theme = callPackage ../data/themes/fluent-gtk-theme { };

  fluent-icon-theme = callPackage ../data/icons/fluent-icon-theme { };

  font-awesome_4 = (callPackage ../data/fonts/font-awesome { }).v4;
  font-awesome_5 = (callPackage ../data/fonts/font-awesome { }).v5;
  font-awesome_6 = (callPackage ../data/fonts/font-awesome { }).v6;
  font-awesome = font-awesome_6;

  fraunces = callPackage ../data/fonts/fraunces { };

  freefont_ttf = callPackage ../data/fonts/freefont-ttf { };

  freepats = callPackage ../data/misc/freepats { };

  game-devices-udev-rules = callPackage ../os-specific/linux/game-devices-udev-rules { };

  garamond-libre = callPackage ../data/fonts/garamond-libre { };

  g15daemon = callPackage ../os-specific/linux/g15daemon { };

  galatia-sil = callPackage ../data/fonts/galatia-sil { };

  gentium = callPackage ../data/fonts/gentium { };

  gentium-book-basic = callPackage ../data/fonts/gentium-book-basic { };

  geolite-legacy = callPackage ../data/misc/geolite-legacy { };

  gohufont = callPackage ../data/fonts/gohufont { };

  gnome-user-docs = callPackage ../data/documentation/gnome-user-docs { };

  gsettings-desktop-schemas = callPackage ../development/libraries/gsettings-desktop-schemas { };

  gnome-icon-theme = callPackage ../data/icons/gnome-icon-theme { };

  go-font = callPackage ../data/fonts/go-font { };

  goudy-bookletter-1911 = callPackage ../data/fonts/goudy-bookletter-1911 { };

  graphite-gtk-theme = callPackage ../data/themes/graphite-gtk-theme { };

  graphite-kde-theme = libsForQt5.callPackage ../data/themes/graphite-kde-theme { };

  greybird = callPackage ../data/themes/greybird { };

  gruvbox-dark-gtk = callPackage ../data/themes/gruvbox-dark-gtk { };

  palenight-theme = callPackage ../data/themes/gtk-theme-framework { theme = "palenight"; };

  amarena-theme = callPackage ../data/themes/gtk-theme-framework { theme = "amarena"; };

  gruvterial-theme = callPackage ../data/themes/gtk-theme-framework { theme = "gruvterial"; };

  oceanic-theme = callPackage ../data/themes/gtk-theme-framework { theme = "oceanic"; };

  spacx-gtk-theme = callPackage ../data/themes/gtk-theme-framework { theme = "spacx"; };

  inherit
    ({
      gruppled-black-cursors = callPackage ../data/icons/gruppled-cursors { theme = "gruppled_black"; };
      gruppled-black-lite-cursors = callPackage ../data/icons/gruppled-lite-cursors {
        theme = "gruppled_black_lite";
      };
      gruppled-white-cursors = callPackage ../data/icons/gruppled-cursors { theme = "gruppled_white"; };
      gruppled-white-lite-cursors = callPackage ../data/icons/gruppled-lite-cursors {
        theme = "gruppled_white_lite";
      };
    })
    gruppled-black-cursors
    gruppled-black-lite-cursors
    gruppled-white-cursors
    gruppled-white-lite-cursors
    ;

  gruvbox-dark-icons-gtk = callPackage ../data/icons/gruvbox-dark-icons-gtk {
    inherit (plasma5Packages) breeze-icons;
  };

  gubbi-font = callPackage ../data/fonts/gubbi { };

  gyre-fonts = callPackage ../data/fonts/gyre { };

  hack-font = callPackage ../data/fonts/hack { };

  hackgen-font = callPackage ../data/fonts/hackgen { };

  hackgen-nf-font = callPackage ../data/fonts/hackgen/nerdfont.nix { };

  helvetica-neue-lt-std = callPackage ../data/fonts/helvetica-neue-lt-std { };

  helvum = callPackage ../applications/audio/helvum { };

  hetzner-kube = callPackage ../applications/networking/cluster/hetzner-kube { };

  hicolor-icon-theme = callPackage ../data/icons/hicolor-icon-theme { };

  hannom = callPackage ../data/fonts/hannom { };

  hanazono = callPackage ../data/fonts/hanazono { };

  hermit = callPackage ../data/fonts/hermit { };

  hubot-sans = callPackage ../data/fonts/hubot-sans { };

  humanity-icon-theme = callPackage ../data/icons/humanity-icon-theme { };

  hyperscrypt-font = callPackage ../data/fonts/hyperscrypt { };

  ia-writer-duospace = callPackage ../data/fonts/ia-writer-duospace { };

  i-dot-ming = callPackage ../data/fonts/i-dot-ming { };

  ibm-plex = callPackage ../data/fonts/ibm-plex { };

  iconpack-jade = callPackage ../data/icons/iconpack-jade { };

  iconpack-obsidian = callPackage ../data/icons/iconpack-obsidian { };

  inconsolata = callPackage ../data/fonts/inconsolata { };

  inconsolata-lgc = callPackage ../data/fonts/inconsolata/lgc.nix { };

  inconsolata-nerdfont = nerdfonts.override {
    fonts = [ "Inconsolata" ];
  };

  intel-one-mono = callPackage ../data/fonts/intel-one-mono {};

  input-fonts = callPackage ../data/fonts/input-fonts { };

  inriafonts = callPackage ../data/fonts/inriafonts { };

  iosevka = callPackage ../data/fonts/iosevka { };
  iosevka-bin = callPackage ../data/fonts/iosevka/bin.nix { };
  iosevka-comfy = recurseIntoAttrs (callPackages ../data/fonts/iosevka/comfy.nix {});

  ipafont = callPackage ../data/fonts/ipafont { };
  ipaexfont = callPackage ../data/fonts/ipaexfont { };

  iwona = callPackage ../data/fonts/iwona { };

  jetbrains-mono = callPackage ../data/fonts/jetbrains-mono { };

  jost = callPackage ../data/fonts/jost { };

  joypixels = callPackage ../data/fonts/joypixels { };

  junction-font = callPackage ../data/fonts/junction { };

  junicode = callPackage ../data/fonts/junicode { };

  julia-mono = callPackage ../data/fonts/julia-mono { };

  juno-theme = callPackage ../data/themes/juno { };

  kanit-font = callPackage ../data/fonts/kanit { };

  kanji-stroke-order-font = callPackage ../data/fonts/kanji-stroke-order-font { };

  kacst = callPackage ../data/fonts/kacst { };

  kawkab-mono-font = callPackage ../data/fonts/kawkab-mono { };

  kde-rounded-corners = kdePackages.callPackage ../data/themes/kwin-decorations/kde-rounded-corners { };

  khmeros = callPackage ../data/fonts/khmeros { };

  knewave = callPackage ../data/fonts/knewave { };

  kochi-substitute = callPackage ../data/fonts/kochi-substitute { };

  kochi-substitute-naga10 = callPackage ../data/fonts/kochi-substitute-naga10 { };

  kode-mono = callPackage ../data/fonts/kode-mono { };

  kopia = callPackage ../tools/backup/kopia { };

  kora-icon-theme = callPackage ../data/icons/kora-icon-theme {
    inherit (gnome) adwaita-icon-theme;
    inherit (libsForQt5.kdeFrameworks) breeze-icons;
  };

  koreader = callPackage ../applications/misc/koreader { };

  kreative-square-fonts = callPackage ../data/fonts/kreative-square-fonts { };

  la-capitaine-icon-theme = callPackage ../data/icons/la-capitaine-icon-theme {
    inherit (plasma5Packages) breeze-icons;
    inherit (pantheon) elementary-icon-theme;
  };

  layan-gtk-theme = callPackage ../data/themes/layan-gtk-theme { };

  layan-kde = libsForQt5.callPackage ../data/themes/layan-kde { };

  lao = callPackage ../data/fonts/lao { };

  lato = callPackage ../data/fonts/lato { };

  league-gothic = callPackage ../data/fonts/league-gothic { };

  league-of-moveable-type = callPackage ../data/fonts/league-of-moveable-type { };

  league-script-number-one = callPackage ../data/fonts/league-script-number-one { };

  league-spartan = callPackage ../data/fonts/league-spartan { };

  ledger-udev-rules = callPackage ../os-specific/linux/ledger-udev-rules { };

  libGDSII = callPackage ../development/libraries/science/chemistry/libGDSII { };

  inherit (callPackages ../data/fonts/liberation-fonts { })
    liberation_ttf_v1
    liberation_ttf_v2
    ;
  liberation_ttf = liberation_ttf_v2;

  liberation-sans-narrow = callPackage ../data/fonts/liberation-sans-narrow { };

  libevdevc = callPackage ../os-specific/linux/libevdevc { };

  libgestures = callPackage ../os-specific/linux/libgestures { };

  liberastika = callPackage ../data/fonts/liberastika { };

  libertine = callPackage ../data/fonts/libertine { };

  libertinus = callPackage ../data/fonts/libertinus { };

  libratbag = callPackage ../os-specific/linux/libratbag { };

  librist = callPackage ../development/libraries/librist { };

  libre-baskerville = callPackage ../data/fonts/libre-baskerville { };

  libre-bodoni = callPackage ../data/fonts/libre-bodoni { };

  libre-caslon = callPackage ../data/fonts/libre-caslon { };

  libre-franklin = callPackage ../data/fonts/libre-franklin { };

  lightly-qt = libsForQt5.callPackage ../data/themes/lightly-qt { };

  lightly-boehs = libsForQt5.callPackage ../data/themes/lightly-boehs { };

  linden-hill = callPackage ../data/fonts/linden-hill { };

  line-awesome = callPackage ../data/fonts/line-awesome { };

  linearicons-free = callPackage ../data/fonts/linearicons-free { };

  linja-pi-pu-lukin = callPackage ../data/fonts/linja-pi-pu-lukin { };

  linja-sike = callPackage ../data/fonts/linja-sike { };

  linux-manual = callPackage ../data/documentation/linux-manual { };

  lklug-sinhala = callPackage ../data/fonts/lklug-sinhala { };

  lmmath = callPackage ../data/fonts/lmmath { };

  lmodern = callPackage ../data/fonts/lmodern { };

  # ltunifi and solaar both provide udev rules but solaar's rules are more
  # up-to-date so we simply use that instead of having to maintain our own rules
  logitech-udev-rules = solaar.udev;

  # lohit-fonts.assamese lohit-fonts.bengali lohit-fonts.devanagari lohit-fonts.gujarati lohit-fonts.gurmukhi
  # lohit-fonts.kannada lohit-fonts.malayalam lohit-fonts.marathi lohit-fonts.nepali lohit-fonts.odia
  # lohit-fonts.tamil-classical lohit-fonts.tamil lohit-fonts.telugu
  # lohit-fonts.kashmiri lohit-fonts.konkani lohit-fonts.maithili lohit-fonts.sindhi
  lohit-fonts = recurseIntoAttrs ( callPackages ../data/fonts/lohit-fonts { } );

  lounge-gtk-theme = callPackage ../data/themes/lounge { };

  luculent = callPackage ../data/fonts/luculent { };

  luna-icons = callPackage ../data/icons/luna-icons {
    inherit (plasma5Packages) breeze-icons;
  };

  lxgw-wenkai = callPackage ../data/fonts/lxgw-wenkai { };

  maia-icon-theme = libsForQt5.callPackage ../data/icons/maia-icon-theme { };

  mailcap = callPackage ../data/misc/mailcap { };

  marathi-cursive = callPackage ../data/fonts/marathi-cursive { };

  man-pages = callPackage ../data/documentation/man-pages { };

  man-pages-posix = callPackage ../data/documentation/man-pages-posix { };

  manrope = callPackage ../data/fonts/manrope { };

  martian-mono = callPackage ../data/fonts/martian-mono { };

  marwaita = callPackage ../data/themes/marwaita { };

  marwaita-manjaro = callPackage ../data/themes/marwaita-manjaro { };

  marwaita-peppermint = callPackage ../data/themes/marwaita-peppermint { };

  marwaita-pop_os = callPackage ../data/themes/marwaita-pop_os { };

  marwaita-ubuntu = callPackage ../data/themes/marwaita-ubuntu { };

  matcha-gtk-theme = callPackage ../data/themes/matcha { };

  materia-theme = callPackage ../data/themes/materia-theme { };

  materia-kde-theme = callPackage ../data/themes/materia-kde { };

  material-design-icons = callPackage ../data/fonts/material-design-icons { };

  material-icons = callPackage ../data/fonts/material-icons { };

  material-symbols = callPackage ../data/fonts/material-symbols { };

  material-kwin-decoration = libsForQt5.callPackage ../data/themes/material-kwin-decoration { };

  meslo-lg = callPackage ../data/fonts/meslo-lg { };

  meslo-lgs-nf = callPackage ../data/fonts/meslo-lgs-nf { };

  migmix = callPackage ../data/fonts/migmix { };

  migu = callPackage ../data/fonts/migu { };

  miscfiles = callPackage ../data/misc/miscfiles { };

  media-player-info = callPackage ../data/misc/media-player-info { };

  medio = callPackage ../data/fonts/medio { };

  melete = callPackage ../data/fonts/melete { };

  mno16 = callPackage ../data/fonts/mno16 { };

  mnist = callPackage ../data/machine-learning/mnist { };

  mobile-broadband-provider-info = callPackage ../data/misc/mobile-broadband-provider-info { };

  mojave-gtk-theme = callPackage ../data/themes/mojave {
    inherit (gnome) gnome-shell;
  };

  moka-icon-theme = callPackage ../data/icons/moka-icon-theme { };

  mona-sans = callPackage ../data/fonts/mona-sans { };

  monoid = callPackage ../data/fonts/monoid { };

  mononoki = callPackage ../data/fonts/mononoki { };

  montserrat = callPackage ../data/fonts/montserrat { };

  mph_2b_damase = callPackage ../data/fonts/mph-2b-damase { };

  mplus-outline-fonts = recurseIntoAttrs (callPackage ../data/fonts/mplus-outline-fonts { });

  mro-unicode = callPackage ../data/fonts/mro-unicode { };

  mustache-spec = callPackage ../data/documentation/mustache-spec { };

  mustache-tcl = callPackage ../development/libraries/mustache-tcl { };

  mustache-go = callPackage ../development/tools/mustache-go { };

  mustache-hpp = callPackage ../development/libraries/mustache-hpp { };

  myrddin = callPackage ../development/compilers/myrddin { };

  myrica = callPackage ../data/fonts/myrica { };

  nacelle = callPackage ../data/fonts/nacelle { };

  nafees = callPackage ../data/fonts/nafees { };

  nanum = callPackage ../data/fonts/nanum {  };

  nanum-gothic-coding = callPackage ../data/fonts/nanum-gothic-coding {  };

  nasin-nanpa = callPackage ../data/fonts/nasin-nanpa { };

  national-park-typeface = callPackage ../data/fonts/national-park { };

  nordic = libsForQt5.callPackage ../data/themes/nordic { };

  nordzy-cursor-theme = callPackage ../data/icons/nordzy-cursor-theme { };

  nordzy-icon-theme = callPackage ../data/icons/nordzy-icon-theme { };

  noto-fonts-lgc-plus = callPackage ../by-name/no/noto-fonts/package.nix {
    suffix = "-lgc-plus";
    variants = [
      "Noto Sans"
      "Noto Serif"
      "Noto Sans Mono"
      "Noto Music"
      "Noto Sans Symbols"
      "Noto Sans Symbols 2"
      "Noto Sans Math"
    ];
    longDescription = ''
      This package provides the Noto Fonts, but only for latin, greek
      and cyrillic scripts, as well as some extra fonts.
    '';
  };

  nuclear = callPackage ../applications/audio/nuclear { };

  nuclei = callPackage ../tools/security/nuclei { };

  nullmailer = callPackage ../servers/mail/nullmailer {
    stdenv = gccStdenv;
  };

  numix-icon-theme = callPackage ../data/icons/numix-icon-theme {
    inherit (gnome) adwaita-icon-theme;
    inherit (plasma5Packages) breeze-icons;
  };

  numix-icon-theme-circle = callPackage ../data/icons/numix-icon-theme-circle { };

  numix-icon-theme-square = callPackage ../data/icons/numix-icon-theme-square { };

  numix-cursor-theme = callPackage ../data/icons/numix-cursor-theme { };

  numix-gtk-theme = callPackage ../data/themes/numix { };

  numix-solarized-gtk-theme = callPackage ../data/themes/numix-solarized { };

  numix-sx-gtk-theme = callPackage ../data/themes/numix-sx { };

  office-code-pro = callPackage ../data/fonts/office-code-pro { };

  _0xproto = callPackage ../data/fonts/0xproto { };

  oldstandard = callPackage ../data/fonts/oldstandard { };

  oldsindhi = callPackage ../data/fonts/oldsindhi { };

  omni-gtk-theme = callPackage ../data/themes/omni-gtk-theme { };

  onestepback = callPackage ../data/themes/onestepback { };

  open-dyslexic = callPackage ../data/fonts/open-dyslexic { };

  open-sans = callPackage ../data/fonts/open-sans { };

  openmoji-color = callPackage ../data/fonts/openmoji { fontFormats = ["glyf_colr_0"]; };

  openmoji-black = callPackage ../data/fonts/openmoji { fontFormats = ["glyf"]; };

  openzone-cursors = callPackage ../data/themes/openzone { };

  oranchelo-icon-theme = callPackage ../data/icons/oranchelo-icon-theme { };

  orbitron = callPackage ../data/fonts/orbitron { };

  orbuculum = callPackage ../development/embedded/orbuculum { };

  orchis-theme = callPackage ../data/themes/orchis-theme { };

  orion = callPackage ../data/themes/orion { };

  ostrich-sans = callPackage ../data/fonts/ostrich-sans { };

  overpass = callPackage ../data/fonts/overpass { };

  oxygenfonts = callPackage ../data/fonts/oxygenfonts { };

  paper-gtk-theme = callPackage ../data/themes/paper-gtk { };

  paper-icon-theme = callPackage ../data/icons/paper-icon-theme { };

  papirus-icon-theme = callPackage ../data/icons/papirus-icon-theme {
    inherit (pantheon) elementary-icon-theme;
    inherit (plasma5Packages) breeze-icons;
  };

  papirus-maia-icon-theme = callPackage ../data/icons/papirus-maia-icon-theme {
    inherit (plasma5Packages) breeze-icons;
  };

  papirus-folders = callPackage ../data/misc/papirus-folders { };

  papirus-nord = callPackage ../data/icons/papirus-nord { };

  plasma-overdose-kde-theme = callPackage ../data/themes/plasma-overdose-kde-theme { };

  papis = with python3Packages; toPythonApplication papis;

  paperlike-go = callPackage ../tools/misc/paperlike-go { };

  paps = callPackage ../tools/misc/paps { };

  pecita = callPackage ../data/fonts/pecita { };

  paratype-pt-mono = callPackage ../data/fonts/paratype-pt/mono.nix { };
  paratype-pt-sans = callPackage ../data/fonts/paratype-pt/sans.nix { };
  paratype-pt-serif = callPackage ../data/fonts/paratype-pt/serif.nix { };

  pari-galdata = callPackage ../data/misc/pari-galdata { };

  pari-seadata-small = callPackage ../data/misc/pari-seadata-small { };

  penna = callPackage ../data/fonts/penna { };

  pitch-black = callPackage ../data/themes/pitch-black { };

  plano-theme = callPackage ../data/themes/plano { };

  plata-theme = callPackage ../data/themes/plata {
    inherit (mate) marco;
  };

  poly = callPackage ../data/fonts/poly { };

  polychromatic = libsForQt5.callPackage ../applications/misc/polychromatic { };

  polytopes_db = callPackage ../data/misc/polytopes_db { };

  pop-gtk-theme = callPackage ../data/themes/pop-gtk { };

  pop-icon-theme = callPackage ../data/icons/pop-icon-theme {
    inherit (gnome) adwaita-icon-theme;
  };

  powerline-fonts = callPackage ../data/fonts/powerline-fonts { };

  powerline-symbols = callPackage ../data/fonts/powerline-symbols { };

  powerline-go = callPackage ../tools/misc/powerline-go { };

  powerline-rs = callPackage ../tools/misc/powerline-rs {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  prociono = callPackage ../data/fonts/prociono { };

  profont = callPackage ../data/fonts/profont { };

  proggyfonts = callPackage ../data/fonts/proggyfonts { };

  public-sans  = callPackage ../data/fonts/public-sans { };

  publicsuffix-list = callPackage ../data/misc/publicsuffix-list { };

  phinger-cursors = callPackage ../data/icons/phinger-cursors { };

  qogir-icon-theme = callPackage ../data/icons/qogir-icon-theme { };

  qogir-kde = libsForQt5.callPackage ../data/themes/qogir-kde { };

  qogir-theme = callPackage ../data/themes/qogir { };

  quintom-cursor-theme = callPackage ../data/icons/quintom-cursor-theme { };

  redhat-official-fonts = callPackage ../data/fonts/redhat-official { };

  rime-data = callPackage ../data/misc/rime-data { };

  roapi-http = callPackage ../servers/roapi/http.nix { };

  rose-pine-gtk-theme = callPackage ../data/themes/rose-pine-gtk { };
  rose-pine-icon-theme = callPackage ../data/icons/rose-pine { };

  route159 = callPackage ../data/fonts/route159 { };

  ruwudu = callPackage ../data/fonts/ruwudu { };

  sampradaya = callPackage ../data/fonts/sampradaya { };

  sarabun-font = callPackage ../data/fonts/sarabun { };

  savepagenow = callPackage ../tools/misc/savepagenow { };

  scheme-manpages = callPackage ../data/documentation/scheme-manpages { };

  scowl = callPackage ../data/misc/scowl { };

  seshat = callPackage ../data/fonts/seshat { };

  session-desktop = callPackage ../applications/networking/instant-messengers/session-desktop { };

  shaderc = callPackage ../development/compilers/shaderc {
    inherit (darwin) autoSignDarwinBinariesHook cctools;
  };

  shades-of-gray-theme = callPackage ../data/themes/shades-of-gray { };

  sierra-breeze-enhanced = libsForQt5.callPackage ../data/themes/kwin-decorations/sierra-breeze-enhanced { useQt5 = true; };

  simp1e-cursors = callPackage ../data/icons/simp1e-cursors { };

  sjasmplus = callPackage ../development/compilers/sjasmplus { };

  skeu = callPackage ../data/themes/skeu { };

  sweet-nova = callPackage ../data/themes/sweet-nova { };

  shared-mime-info = callPackage ../data/misc/shared-mime-info { };

  shared_desktop_ontologies = callPackage ../data/misc/shared-desktop-ontologies { };

  scheherazade = callPackage ../data/fonts/scheherazade { version = "2.100"; };

  scheherazade-new = callPackage ../data/fonts/scheherazade { };

  signwriting = callPackage ../data/fonts/signwriting { };

  sierra-gtk-theme = callPackage ../data/themes/sierra { };

  sil-abyssinica = callPackage ../data/fonts/sil-abyssinica { };

  sil-padauk = callPackage ../data/fonts/sil-padauk { };

  snap7 = callPackage ../development/libraries/snap7 { };

  sniglet = callPackage ../data/fonts/sniglet { };

  snowblind = callPackage ../data/themes/snowblind { };

  solarc-gtk-theme = callPackage ../data/themes/solarc { };

  sorts-mill-goudy = callPackage ../data/fonts/sorts-mill-goudy { };

  soundfont-fluid = callPackage ../data/soundfonts/fluid { };

  soundfont-generaluser = callPackage ../data/soundfonts/generaluser { };

  soundfont-ydp-grand = callPackage ../data/soundfonts/ydp-grand { };

  spdx-license-list-data = callPackage ../data/misc/spdx-license-list-data { };

  stdmanpages = callPackage ../data/documentation/std-man-pages { };

  star-history = callPackage ../tools/misc/star-history { };

  starfetch = callPackage ../tools/misc/starfetch { };

  starry = callPackage ../tools/misc/starry { };

  starship = callPackage ../tools/misc/starship {
    inherit (darwin.apple_sdk.frameworks) Security Foundation Cocoa;
  };

  stig = callPackage ../applications/networking/p2p/stig { };

  stix-otf = callPackage ../data/fonts/stix-otf { };

  stix-two = callPackage ../data/fonts/stix-two { };

  super-tiny-icons = callPackage ../data/icons/super-tiny-icons { };

  inherit (callPackages ../data/fonts/gdouros { })
    aegan aegyptus akkadian assyrian eemusic maya symbola textfonts unidings;

  iana-etc = callPackage ../data/misc/iana-etc { };

  poppler_data = callPackage ../data/misc/poppler-data { };

  inherit (callPackages ../data/fonts/pretendard { })
    pretendard
    pretendard-gov
    pretendard-jp
    pretendard-std;

  quattrocento = callPackage ../data/fonts/quattrocento { };

  quattrocento-sans = callPackage ../data/fonts/quattrocento-sans { };

  quivira = callPackage ../data/fonts/quivira { };

  raleway = callPackage ../data/fonts/raleway { };

  recursive = callPackage ../data/fonts/recursive { };

  reversal-icon-theme = callPackage ../data/icons/reversal-icon-theme {
    inherit (gnome) adwaita-icon-theme;
  };

  rubik = callPackage ../data/fonts/rubik { };

  rhodium-libre = callPackage ../data/fonts/rhodium-libre { };

  rictydiminished-with-firacode = callPackage ../data/fonts/rictydiminished-with-firacode { };

  roboto = callPackage ../data/fonts/roboto { };

  roboto-mono = callPackage ../data/fonts/roboto-mono { };

  roboto-serif = callPackage ../data/fonts/roboto-serif { };

  roboto-slab = callPackage ../data/fonts/roboto-slab { };

  hasklig = callPackage ../data/fonts/hasklig { };

  maligned = callPackage ../development/tools/maligned { };

  inter = callPackage ../data/fonts/inter { };

  open-fonts = callPackage ../data/fonts/open-fonts { };

  scientifica = callPackage ../data/fonts/scientifica { };

  siji = callPackage ../data/fonts/siji { };

  sound-theme-freedesktop = callPackage ../data/misc/sound-theme-freedesktop { };

  source-code-pro = callPackage ../data/fonts/source-code-pro { };

  source-sans = callPackage ../data/fonts/source-sans { };
  source-sans-pro = callPackage ../data/fonts/source-sans-pro { };

  source-serif = callPackage ../data/fonts/source-serif { };
  source-serif-pro = callPackage ../data/fonts/source-serif-pro { };

  source-han-code-jp = callPackage ../data/fonts/source-han-code-jp { };

  sourceHanPackages = dontRecurseIntoAttrs (callPackage ../data/fonts/source-han { });
  source-han-sans = sourceHanPackages.sans;
  source-han-serif = sourceHanPackages.serif;
  source-han-mono = sourceHanPackages.mono;
  source-han-sans-vf-otf = sourceHanPackages.sans-vf-otf;
  source-han-sans-vf-ttf = sourceHanPackages.sans-vf-ttf;
  source-han-serif-vf-otf = sourceHanPackages.serif-vf-otf;
  source-han-serif-vf-ttf = sourceHanPackages.serif-vf-ttf;

  spleen = callPackage ../data/fonts/spleen { };

  stilo-themes = callPackage ../data/themes/stilo { };

  sudo-font = callPackage ../data/fonts/sudo { };

  inherit (callPackages ../data/fonts/tai-languages { }) tai-ahom;

  takao = callPackage ../data/fonts/takao { };

  taskspooler = callPackage ../tools/system/taskspooler { };

  tamsyn = callPackage ../data/fonts/tamsyn { };

  tamzen = callPackage ../data/fonts/tamzen { };

  tango-icon-theme = callPackage ../data/icons/tango-icon-theme {
    gtk = res.gtk2;
  };

  tau-hydrogen = callPackage ../data/icons/tau-hydrogen { };

  theme-jade1 = callPackage ../data/themes/jade1 { };

  theme-obsidian2 = callPackage ../data/themes/obsidian2 { };

  themes = name: callPackage (../data/misc/themes + ("/" + name + ".nix")) { };

  theano = callPackage ../data/fonts/theano { };

  the-neue-black = callPackage ../data/fonts/the-neue-black { };

  tela-circle-icon-theme = callPackage ../data/icons/tela-circle-icon-theme {
    inherit (gnome) adwaita-icon-theme;
    inherit (libsForQt5) breeze-icons;
  };

  tela-icon-theme = callPackage ../data/icons/tela-icon-theme { };

  template-glib = callPackage ../development/libraries/template-glib { };

  tempora_lgc = callPackage ../data/fonts/tempora-lgc { };

  tenderness = callPackage ../data/fonts/tenderness { };

  terminus_font = callPackage ../data/fonts/terminus-font { };

  terminus_font_ttf = callPackage ../data/fonts/terminus-font-ttf { };

  terminus-nerdfont = nerdfonts.override {
    fonts = [ "Terminus" ];
  };

  tex-gyre = callPackages ../data/fonts/tex-gyre { };

  tex-gyre-math = callPackages ../data/fonts/tex-gyre-math { };

  theme-vertex = callPackage ../data/themes/vertex { };

  tibetan-machine = callPackage ../data/fonts/tibetan-machine { };

  times-newer-roman = callPackage ../data/fonts/times-newer-roman { };

  tipa = callPackage ../data/fonts/tipa { };

  ttf_bitstream_vera = callPackage ../data/fonts/ttf-bitstream-vera { };

  ttf-envy-code-r = callPackage ../data/fonts/ttf-envy-code-r { };

  ttf-tw-moe = callPackage ../data/fonts/ttf-tw-moe { };

  twemoji-color-font = callPackage ../data/fonts/twemoji-color-font { };

  twitter-color-emoji = callPackage ../data/fonts/twitter-color-emoji { };

  tzdata = callPackage ../data/misc/tzdata { };

  ubuntu-themes = callPackage ../data/themes/ubuntu-themes { };

  ubuntu_font_family = callPackage ../data/fonts/ubuntu-font-family { };

  ucs-fonts = callPackage ../data/fonts/ucs-fonts { };

  bront_fonts = callPackage ../data/fonts/bront { };

  ultimate-oldschool-pc-font-pack = callPackage ../data/fonts/ultimate-oldschool-pc-font-pack { };

  ultralist = callPackage ../applications/misc/ultralist { };

  undefined-medium = callPackage ../data/fonts/undefined-medium { };

  unfonts-core = callPackage ../data/fonts/unfonts-core { };

  uni-vga = callPackage ../data/fonts/uni-vga { };

  unicode-character-database = callPackage ../data/misc/unicode-character-database { };

  unicode-emoji = callPackage ../data/misc/unicode-emoji { };

  unihan-database = callPackage ../data/misc/unihan-database { };

  unifont = callPackage ../data/fonts/unifont { };

  unifont_upper = callPackage ../data/fonts/unifont_upper { };

  unscii = callPackage ../data/fonts/unscii { };

  utterly-nord-plasma = libsForQt5.callPackage ../data/themes/utterly-nord-plasma { };

  utterly-round-plasma-style = callPackage ../data/themes/utterly-round-plasma-style { };

  uw-ttyp0 = callPackage ../data/fonts/uw-ttyp0 { };

  u001-font = callPackage ../data/fonts/u001 { };

  vanilla-dmz = callPackage ../data/icons/vanilla-dmz { };

  vdrsymbols = callPackage ../data/fonts/vdrsymbols { };

  vegur = callPackage ../data/fonts/vegur { };

  vegeta = callPackage ../tools/networking/vegeta { };

  venta = callPackage ../data/themes/venta { };

  victor-mono = callPackage ../data/fonts/victor-mono { };

  vimix-cursor-theme = callPackage ../data/icons/vimix-cursor-theme { };

  vimix-gtk-themes = callPackage ../data/themes/vimix {
    inherit (gnome) gnome-shell;
  };

  vimix-icon-theme = callPackage ../data/icons/vimix-icon-theme { };

  vistafonts = callPackage ../data/fonts/vista-fonts { };

  vistafonts-chs = callPackage ../data/fonts/vista-fonts-chs { };

  vistafonts-cht = callPackage ../data/fonts/vista-fonts-cht { };

  vollkorn = callPackage ../data/fonts/vollkorn { };

  weather-icons = callPackage ../data/fonts/weather-icons { };

  whitesur-cursors = callPackage ../data/icons/whitesur-cursors { };

  whitesur-gtk-theme = callPackage ../data/themes/whitesur {
    inherit (gnome) gnome-shell;
  };

  whitesur-icon-theme = callPackage ../data/icons/whitesur-icon-theme { };

  whitesur-kde = libsForQt5.callPackage ../data/themes/whitesur-kde { };

  wireless-regdb = callPackage ../data/misc/wireless-regdb { };

  work-sans  = callPackage ../data/fonts/work-sans { };

  wqy_microhei = callPackage ../data/fonts/wqy-microhei { };

  wqy_zenhei = callPackage ../data/fonts/wqy-zenhei { };

  xhtml1 = callPackage ../data/sgml+xml/schemas/xml-dtd/xhtml1 { };

  xits-math = callPackage ../data/fonts/xits-math { };

  xkcd-font = callPackage ../data/fonts/xkcd-font { };

  xkeyboard_config = xorg.xkeyboardconfig;

  xlsx2csv = with python3Packages; toPythonApplication xlsx2csv;

  xorg-rgb = callPackage ../data/misc/xorg-rgb { };

  yanone-kaffeesatz = callPackage ../data/fonts/yanone-kaffeesatz { };

  yaru-theme = callPackage ../data/themes/yaru { };

  yaru-remix-theme = callPackage ../data/themes/yaru-remix { };

  zafiro-icons = callPackage ../data/icons/zafiro-icons {
    inherit (plasma5Packages) breeze-icons;
  };

  zeal-qt5 = libsForQt5.callPackage ../data/documentation/zeal { };
  zeal = zeal-qt5;
  zeal-qt6 = qt6Packages.callPackage ../data/documentation/zeal {
    qtx11extras = null; # Because it does not exist in qt6
  };

  zilla-slab = callPackage ../data/fonts/zilla-slab { };

  zuki-themes = callPackage ../data/themes/zuki { };

  ### APPLICATIONS / GIS

  gmt = callPackage ../applications/gis/gmt {
    inherit (darwin.apple_sdk.frameworks)
      Accelerate CoreGraphics CoreVideo;
  };

  gshhg-gmt = callPackage ../applications/gis/gmt/gshhg.nix { };

  dcw-gmt = callPackage ../applications/gis/gmt/dcw.nix { };

  grass = callPackage ../applications/gis/grass {
    stdenv = if stdenv.isDarwin then overrideSDK stdenv "11.0" else stdenv;
  };

  qgis-ltr = callPackage ../applications/gis/qgis/ltr.nix { };

  qgis = callPackage ../applications/gis/qgis { };

  qmapshack = libsForQt5.callPackage ../applications/gis/qmapshack { };

  spatialite_gui = callPackage ../applications/gis/spatialite-gui {
    inherit (darwin.apple_sdk.frameworks) Carbon Cocoa IOKit;
    wxGTK = wxGTK32;
  };

  spatialite_tools = callPackage ../applications/gis/spatialite-tools { };

  udig = callPackage ../applications/gis/udig { };

  whitebox-tools = callPackage ../applications/gis/whitebox-tools {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  zombietrackergps = libsForQt5.callPackage ../applications/gis/zombietrackergps { };

  ### APPLICATIONS

  _2bwm = callPackage ../applications/window-managers/2bwm {
    patches = config."2bwm".patches or [];
  };

  a2jmidid = callPackage ../applications/audio/a2jmidid { };

  aacgain = callPackage ../applications/audio/aacgain { };

  aaxtomp3 = callPackage ../applications/audio/aaxtomp3 { };

  abcde = callPackage ../applications/audio/abcde {
    inherit (python3Packages) eyed3;
  };

  abiword = callPackage ../applications/office/abiword { };

  abook = callPackage ../applications/misc/abook { };

  acd-cli = callPackage ../applications/networking/sync/acd_cli {
    inherit (python3Packages)
      buildPythonApplication appdirs colorama python-dateutil
      requests requests-toolbelt setuptools sqlalchemy fusepy;
  };

  inherit (qt6Packages.callPackage ../applications/office/activitywatch { })
    aw-qt
    aw-server-rust
    aw-watcher-afk
    aw-watcher-window;

  activitywatch = callPackage ../applications/office/activitywatch/wrapper.nix { };

  aw-watcher-window-wayland = callPackage ../tools/wayland/aw-watcher-window-wayland {  };

  adobe-reader = pkgsi686Linux.callPackage ../applications/misc/adobe-reader { };

  adl = callPackage ../applications/video/adl { };

  appvm = callPackage ../applications/virtualization/appvm { };

  anilibria-winmaclinux = libsForQt5.callPackage ../applications/video/anilibria-winmaclinux { };

  masterpdfeditor4 = libsForQt5.callPackage ../applications/misc/masterpdfeditor4 { };

  master_me = callPackage ../applications/audio/master_me {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  pdfstudio2021 = callPackage ../applications/misc/pdfstudio { year = "2021"; };

  pdfstudio2022 = callPackage ../applications/misc/pdfstudio { year = "2022"; };

  pdfstudio2023 = callPackage ../applications/misc/pdfstudio { year = "2023"; };

  pdfstudioviewer = callPackage ../applications/misc/pdfstudio { program = "pdfstudioviewer"; };

  abaddon = callPackage ../applications/networking/instant-messengers/abaddon { };

  acpic = callPackage ../applications/misc/acpic/default.nix { };

  adguardian = callPackage ../applications/networking/adguardian { };

  aeolus = callPackage ../applications/audio/aeolus { };
  aeolus-stops = callPackage ../applications/audio/aeolus/stops.nix { };

  aewan = callPackage ../applications/editors/aewan { };

  agedu = callPackage ../tools/misc/agedu { };

  agenda = callPackage ../applications/office/agenda { };

  ahoviewer = callPackage ../applications/graphics/ahoviewer { };

  airwave = libsForQt5.callPackage ../applications/audio/airwave { };

  akira-unstable = callPackage ../applications/graphics/akira { };

  alembic = callPackage ../development/libraries/alembic {
    openexr = openexr_3;
  };

  alfaview = callPackage ../applications/networking/instant-messengers/alfaview { };

  alchemy = callPackage ../applications/graphics/alchemy { };

  alock = callPackage ../misc/screensavers/alock { };

  alot = callPackage ../applications/networking/mailreaders/alot { };

  alpine = callPackage ../applications/networking/mailreaders/alpine { };

  msgviewer = callPackage ../applications/networking/mailreaders/msgviewer { };

  amarok = libsForQt5.callPackage ../applications/audio/amarok { };
  amarok-kf5 = amarok; # for compatibility

  AMB-plugins = callPackage ../applications/audio/AMB-plugins { };

  ams-lv2 = callPackage ../applications/audio/ams-lv2 { };

  androidStudioPackages = recurseIntoAttrs
    (callPackage ../applications/editors/android-studio { });
  android-studio = androidStudioPackages.stable;

  antfs-cli = callPackage ../applications/misc/antfs-cli { };

  antimony = libsForQt5.callPackage ../applications/graphics/antimony { };

  antiword = callPackage ../applications/office/antiword { };

  anup = callPackage ../applications/misc/anup {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  anytone-emu = callPackage ../applications/radio/anytone-emu { };

  ao = libfive;

  apache-directory-studio = callPackage ../applications/networking/apache-directory-studio { };

  apkeep = callPackage ../tools/misc/apkeep {
    inherit (darwin.apple_sdk.frameworks) Security SystemConfiguration;
  };

  apngasm = callPackage ../applications/graphics/apngasm { };
  apngasm_2 = callPackage ../applications/graphics/apngasm/2.nix { };

  appcleaner = callPackage ../applications/misc/appcleaner { };

  appeditor = callPackage ../applications/misc/appeditor { };

  appgate-sdp = callPackage ../applications/networking/appgate-sdp { };

  ardour = callPackage ../applications/audio/ardour { };
  ardour_7 = callPackage ../applications/audio/ardour/7.nix { };

  arelle = with python3Packages; toPythonApplication arelle;

  argo = callPackage ../applications/networking/cluster/argo { };

  argocd = callPackage ../applications/networking/cluster/argocd { };

  argocd-autopilot = callPackage ../applications/networking/cluster/argocd-autopilot { };

  argo-rollouts = callPackage ../applications/networking/cluster/argo-rollouts { };

  ario = callPackage ../applications/audio/ario { };

  arion = callPackage ../applications/virtualization/arion { };

  arkade = callPackage ../applications/networking/cluster/arkade { };

  artem = callPackage ../applications/graphics/artem { };

  asuka = callPackage ../applications/networking/browsers/asuka {
    inherit (darwin.apple_sdk.frameworks) Security;
  };

  ascii-image-converter = callPackage ../applications/graphics/ascii-image-converter { };

  artha = callPackage ../applications/misc/artha { };

  atlassian-cli = callPackage ../applications/office/atlassian-cli { };

  asap = callPackage ../tools/audio/asap { };

  aseprite = callPackage ../applications/editors/aseprite { };

  assign-lb-ip = callPackage ../applications/networking/cluster/assign-lb-ip { };

  astroid = callPackage ../applications/networking/mailreaders/astroid {
    vim = vim-full.override { features = "normal"; };
    protobuf = protobuf_21;
  };

  aucatctl = callPackage ../applications/audio/aucatctl { };

  audacious = libsForQt5.callPackage ../applications/audio/audacious { };
  audacious-plugins = libsForQt5.callPackage ../applications/audio/audacious/plugins.nix {
    # Avoid circular dependency
    audacious = audacious.override { audacious-plugins = null; };
  };
  audaciousQt5 = audacious;

  audacity = callPackage ../applications/audio/audacity {
    inherit (darwin.apple_sdk.frameworks) AppKit CoreAudioKit;
  };

  audio-recorder = callPackage ../applications/audio/audio-recorder { };

  auto-multiple-choice = callPackage ../applications/misc/auto-multiple-choice { };

  autokey = callPackage ../applications/office/autokey { };

  autotalent = callPackage ../applications/audio/autotalent { };

  autotrace = callPackage ../applications/graphics/autotrace { };

  av-98 = callPackage ../applications/networking/browsers/av-98 { };

  avalanchego = callPackage ../applications/networking/avalanchego {
    inherit (darwin.apple_sdk.frameworks) IOKit;
  };

  avizo = callPackage ../applications/misc/avizo { };

  avocode = callPackage ../applications/graphics/avocode { };

  ax25-apps = callPackage ../applications/radio/ax25-apps { };

  ax25-tools = callPackage ../applications/radio/ax25-tools { };

  azpainter = callPackage ../applications/graphics/azpainter { };

  bambootracker = libsForQt5.callPackage ../applications/audio/bambootracker {
    stdenv = if stdenv.hostPlatform.isDarwin then
      darwin.apple_sdk_11_0.stdenv
    else
      stdenv;
  };
  bambootracker-qt6 = qt6Packages.callPackage ../applications/audio/bambootracker {
    stdenv = if stdenv.hostPlatform.isDarwin then
      darwin.apple_sdk_11_0.stdenv
    else
      stdenv;
  };

  blocky = callPackage ../applications/networking/blocky { };

  brutefir = callPackage ../applications/audio/brutefir { };

  cheesecutter = callPackage ../applications/audio/cheesecutter { };

  cutecapture = callPackage ../applications/video/cutecapture { };

  milkytracker = callPackage ../applications/audio/milkytracker {
    inherit (darwin.apple_sdk.frameworks) Cocoa CoreAudio Foundation;
  };

  ptcollab = callPackage ../by-name/pt/ptcollab/package.nix {
    stdenv = if stdenv.hostPlatform.isDarwin then darwin.apple_sdk_11_0.stdenv else stdenv;
  };

  schismtracker = callPackage ../applications/audio/schismtracker {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  jnetmap = callPackage ../applications/networking/jnetmap { };

  jxplorer  = callPackage ../applications/networking/jxplorer {};

  joincap = callPackage ../tools/security/joincap { };

  json-plot = callPackage ../applications/graphics/json-plot { };

  libbitcoin = callPackage ../tools/misc/libbitcoin/libbitcoin.nix {
    boost = boost175; # fatal error: 'boost/interprocess/detail/posix_time_types_wrk.hpp' file not found
  };
  libbitcoin-protocol = callPackage ../tools/misc/libbitcoin/libbitcoin-protocol.nix {
    boost = boost175;
  };
  libbitcoin-client   = callPackage ../tools/misc/libbitcoin/libbitcoin-client.nix {
    boost = boost175;
  };
  libbitcoin-network  = callPackage ../tools/misc/libbitcoin/libbitcoin-network.nix {
    boost = boost175;
  };
  libbitcoin-explorer = callPackage ../tools/misc/libbitcoin/libbitcoin-explorer.nix {
    boost = boost175;
  };


  armcord = callPackage ../applications/networking/instant-messengers/armcord { };

  aumix = callPackage ../applications/audio/aumix {
    gtkGUI = false;
  };

  autopanosiftc = callPackage ../applications/graphics/autopanosiftc { };

  ausweisapp = qt6Packages.callPackage ../applications/misc/ausweisapp { };

  avidemux = libsForQt5.callPackage ../applications/video/avidemux { };

  avrdudess = callPackage ../applications/misc/avrdudess { };

  awesome = callPackage ../applications/window-managers/awesome {
    cairo = cairo.override { xcbSupport = true; };
    inherit (texFunctions) fontsConf;
  };

  awesomebump = libsForQt5.callPackage ../applications/graphics/awesomebump { };

  inherit (gnome) baobab;

  badwolf = callPackage ../applications/networking/browsers/badwolf { };

  backintime-common = callPackage ../applications/networking/sync/backintime/common.nix { };

  backintime-qt = libsForQt5.callPackage ../applications/networking/sync/backintime/qt.nix { };

  backintime = backintime-qt;

  balsa = callPackage ../applications/networking/mailreaders/balsa { };

  bandwidth = callPackage ../tools/misc/bandwidth { };

  banking = callPackage ../applications/office/banking { };

  barrier = libsForQt5.callPackage ../applications/misc/barrier { };

  bashSnippets = callPackage ../applications/misc/bashSnippets { };

  batik = callPackage ../applications/graphics/batik { };

  batsignal = callPackage ../applications/misc/batsignal { };

  baudline = callPackage ../applications/audio/baudline { };

  bb =  callPackage ../applications/misc/bb { };

  bchoppr = callPackage ../applications/audio/bchoppr { };

  bespokesynth = darwin.apple_sdk_11_0.callPackage ../applications/audio/bespokesynth {
    inherit (darwin.apple_sdk_11_0.frameworks) Accelerate Cocoa WebKit CoreServices CoreAudioKit IOBluetooth MetalKit;
  };

  bespokesynth-with-vst2 = bespokesynth.override {
    enableVST2 = true;
  };

  bfcal = libsForQt5.callPackage ../applications/misc/bfcal { };

  bibletime = libsForQt5.callPackage ../applications/misc/bibletime { };

  bino3d = libsForQt5.callPackage ../applications/video/bino3d {
    glew = glew110;
  };

  bitlbee = callPackage ../applications/networking/instant-messengers/bitlbee { };
  bitlbee-plugins = callPackage ../applications/networking/instant-messengers/bitlbee/plugins.nix { };

  bitlbee-discord = callPackage ../applications/networking/instant-messengers/bitlbee-discord { };

  bitlbee-facebook = callPackage ../applications/networking/instant-messengers/bitlbee-facebook { };

  bitlbee-steam = callPackage ../applications/networking/instant-messengers/bitlbee-steam { };

  bitlbee-mastodon = callPackage ../applications/networking/instant-messengers/bitlbee-mastodon { };

  bitmeter = callPackage ../applications/audio/bitmeter { };

  bitscope = recurseIntoAttrs
    (callPackage ../applications/science/electronics/bitscope/packages.nix { });

  bitwig-studio3 =  callPackage ../applications/audio/bitwig-studio/bitwig-studio3.nix { };
  bitwig-studio4 =  callPackage ../applications/audio/bitwig-studio/bitwig-studio4.nix {
    libjpeg = libjpeg8;
  };
  bitwig-studio5 =  callPackage ../applications/audio/bitwig-studio/bitwig-studio5.nix {
    libjpeg = libjpeg8;
  };

  bitwig-studio = bitwig-studio5;

  bgpdump = callPackage ../tools/networking/bgpdump { };

  bgpq3 = callPackage ../tools/networking/bgpq3 { };

  bgpq4 = callPackage ../tools/networking/bgpq4 { };

  blackbox = callPackage ../applications/version-management/blackbox {
    pinentry = pinentry-curses;
  };

  bleachbit = callPackage ../applications/misc/bleachbit { };

  blender = callPackage  ../applications/misc/blender {
    openexr = openexr_3;
    inherit (darwin.apple_sdk.frameworks) Cocoa CoreGraphics ForceFeedback OpenAL OpenGL;
  };

  blender-hip = blender.override { hipSupport = true; };

  blflash = callPackage ../tools/misc/blflash { };

  blogc = callPackage ../applications/misc/blogc { };

  bloodhound = callPackage ../applications/misc/bloodhound { };

  blucontrol = callPackage ../applications/misc/blucontrol/wrapper.nix {
    inherit (haskellPackages) ghcWithPackages;
  };

  bluefish = callPackage ../applications/editors/bluefish {
    gtk = gtk3;
  };

  bluej = callPackage ../applications/editors/bluej {
    openjdk = openjdk17.override {
      enableJavaFX = true;
      openjfx = openjfx17.override { withWebKit = true; };
    };
  };

  bluejeans-gui = callPackage ../applications/networking/instant-messengers/bluejeans { };

  bluemail = callPackage ../applications/networking/mailreaders/bluemail { };

  blugon = callPackage ../applications/misc/blugon { };

  bombadillo = callPackage ../applications/networking/browsers/bombadillo { };

  bombono = callPackage ../applications/video/bombono { };

  bonzomatic = callPackage ../applications/editors/bonzomatic { };

  booster = callPackage ../applications/system/booster { };

  bottles = callPackage ../applications/misc/bottles/fhsenv.nix { };

  bottles-unwrapped = callPackage ../applications/misc/bottles { };

  buzztrax = callPackage ../applications/audio/buzztrax { };

  brave = callPackage ../applications/networking/browsers/brave { };

  break-time = callPackage ../applications/misc/break-time { };

  breezy = with python3Packages; toPythonApplication breezy;

  notmuch-bower = callPackage ../applications/networking/mailreaders/notmuch-bower { };

  brig = callPackage ../applications/networking/brig { };

  bristol = callPackage ../applications/audio/bristol { };

  brlcad = callPackage ../applications/graphics/brlcad { };

  bjumblr = callPackage ../applications/audio/bjumblr { };

  bschaffl = callPackage ../applications/audio/bschaffl { };

  bsequencer = callPackage ../applications/audio/bsequencer { };

  bslizr = callPackage ../applications/audio/bslizr { };

  bshapr = callPackage ../applications/audio/bshapr { };

  bspwm = callPackage ../applications/window-managers/bspwm { };

  bvi = callPackage ../applications/editors/bvi { };

  bviplus = callPackage ../applications/editors/bviplus { };

  caerbannog = callPackage ../applications/misc/caerbannog { };

  cardo = callPackage ../data/fonts/cardo { };

  cage = callPackage ../applications/window-managers/cage {
    wlroots = wlroots_0_16;
  };

  calf = callPackage ../applications/audio/calf {
      inherit (gnome2) libglade;
  };

  calcmysky = qt6Packages.callPackage ../applications/science/astronomy/calcmysky { };

  calcurse = callPackage ../applications/misc/calcurse { };

  calculix = callPackage ../applications/science/math/calculix { };

  calibre = qt6Packages.callPackage ../applications/misc/calibre {
    podofo = podofo010;
  };

  calibre-web = callPackage ../servers/calibre-web { };

  # calico-felix and calico-node have not been packaged due to libbpf, linking issues
  inherit (callPackage ../applications/networking/cluster/calico {})
    calico-apiserver
    calico-app-policy
    calico-cni-plugin
    calico-kube-controllers
    calico-pod2daemon
    calico-typha
    calicoctl
    confd-calico;

  calligra = libsForQt5.callPackage ../applications/office/calligra { };

  perkeep = callPackage ../applications/misc/perkeep { };

  canto-curses = callPackage ../applications/networking/feedreaders/canto-curses { };

  canto-daemon = callPackage ../applications/networking/feedreaders/canto-daemon { };

  caprine-bin = callPackage ../applications/networking/instant-messengers/caprine-bin { };

  carddav-util = callPackage ../tools/networking/carddav-util { };

  cardinal = callPackage ../applications/audio/cardinal { };

  carla = libsForQt5.callPackage ../applications/audio/carla { };

  castor = callPackage ../applications/networking/browsers/castor { };

  catfs = callPackage ../os-specific/linux/catfs { };

  catgirl = callPackage ../applications/networking/irc/catgirl { };

  catimg = callPackage ../tools/misc/catimg { };

  catt = callPackage ../applications/video/catt { };

  cava = callPackage ../applications/audio/cava { };

  cb2bib = libsForQt5.callPackage ../applications/office/cb2bib { };

  cbatticon = callPackage ../applications/misc/cbatticon { };

  cbc = callPackage ../applications/science/math/cbc { };

  cddiscid = callPackage ../applications/audio/cd-discid {
    inherit (darwin) IOKit;
  };

  cdparanoia = cdparanoiaIII;

  cdparanoiaIII = callPackage ../applications/audio/cdparanoia {
    inherit (darwin) IOKit;
    inherit (darwin.apple_sdk.frameworks) Carbon;
  };

  celeste = callPackage ../applications/networking/sync/celeste { };

  cyan = callPackage ../applications/graphics/cyan { };

  centerim = callPackage ../applications/networking/instant-messengers/centerim { };

  chatty = callPackage ../applications/networking/instant-messengers/chatty { };

  chirp = callPackage ../applications/radio/chirp { };

  browsh = callPackage ../applications/networking/browsers/browsh { };

  brotab = callPackage ../tools/misc/brotab {
    python = python3;
  };

  bookworm = callPackage ../applications/office/bookworm { };

  boops = callPackage ../applications/audio/boops { };

  bumblebee-status = callPackage ../applications/window-managers/i3/bumblebee-status {
    python = python3;
  };

  cgif = callPackage ../tools/graphics/cgif { };

  ChowCentaur  = callPackage ../applications/audio/ChowCentaur { };

  ChowKick  = callPackage ../applications/audio/ChowKick { };

  ChowPhaser  = callPackage ../applications/audio/ChowPhaser { };

  CHOWTapeModel = callPackage ../applications/audio/CHOWTapeModel { };

  chromium = callPackage ../applications/networking/browsers/chromium (config.chromium or {});

  chuck = callPackage ../applications/audio/chuck {
    inherit (darwin) DarwinTools;
    inherit (darwin.apple_sdk.frameworks) AppKit Carbon CoreAudio CoreMIDI CoreServices Kernel MultitouchSupport;
  };

  cinelerra = callPackage ../applications/video/cinelerra { };

  cipher = callPackage ../applications/misc/cipher { };

  clapper = callPackage ../applications/video/clapper { };

  claws-mail = callPackage ../applications/networking/mailreaders/claws-mail { };

  cligh = python3Packages.callPackage ../development/tools/github/cligh { };

  clight = callPackage ../applications/misc/clight { };

  clight-gui = libsForQt5.callPackage ../applications/misc/clight/clight-gui.nix { };

  clightd = callPackage ../applications/misc/clight/clightd.nix { };

  clipgrab = libsForQt5.callPackage ../applications/video/clipgrab { };

  clipmenu = callPackage ../applications/misc/clipmenu { };

  clipit = callPackage ../applications/misc/clipit { };

  clipqr = callPackage ../applications/misc/clipqr { };

  cloud-hypervisor = callPackage ../applications/virtualization/cloud-hypervisor { };

  clp = callPackage ../applications/science/math/clp { };

  cmatrix = callPackage ../applications/misc/cmatrix { };

  pokemon-colorscripts-mac = callPackage ../applications/misc/pokemon-colorscripts-mac { };

  cmctl = callPackage ../applications/networking/cluster/cmctl { };

  cmus = callPackage ../applications/audio/cmus {
    inherit (darwin.apple_sdk.frameworks) AudioUnit CoreAudio VideoToolbox;
    libjack = libjack2;
  };

  cmusfm = callPackage ../applications/audio/cmusfm { };

  cni = callPackage ../applications/networking/cluster/cni { };
  cni-plugins = callPackage ../applications/networking/cluster/cni/plugins.nix { };

  dnsname-cni = callPackage ../applications/networking/cluster/dnsname-cni { };

  multus-cni = callPackage ../applications/networking/cluster/multus-cni { };

  cntr = callPackage ../applications/virtualization/cntr { };

  communi = libsForQt5.callPackage ../applications/networking/irc/communi { };

  complete-alias = callPackage ../tools/misc/complete-alias { };

  completely = callPackage ../tools/misc/completely { };

  confclerk = libsForQt5.callPackage ../applications/misc/confclerk { };

  copyq = qt6Packages.callPackage ../applications/misc/copyq { };

  corectrl = libsForQt5.callPackage ../applications/misc/corectrl { };

  coreth = callPackage ../applications/networking/coreth { };

  coriander = callPackage ../applications/video/coriander {
    inherit (gnome2) libgnomeui GConf;
  };

  cpeditor = libsForQt5.callPackage ../applications/editors/cpeditor { };

  csa = callPackage ../applications/audio/csa { };

  csound = callPackage ../applications/audio/csound {
    inherit (pkgs.darwin.apple_sdk.frameworks) Accelerate AudioUnit CoreAudio CoreMIDI;
  };

  csound-manual = callPackage ../applications/audio/csound/csound-manual { };

  csound-qt = libsForQt5.callPackage ../applications/audio/csound/csound-qt { };

  codeblocks = callPackage ../applications/editors/codeblocks { };
  codeblocksFull = codeblocks.override { contribPlugins = true; };

  cudatext-qt = callPackage ../applications/editors/cudatext { widgetset = "qt5"; };
  cudatext-gtk = callPackage ../applications/editors/cudatext { widgetset = "gtk2"; };
  cudatext = cudatext-qt;

  convos = callPackage ../applications/networking/irc/convos { };

  comical = callPackage ../applications/graphics/comical {
    inherit (darwin.apple_sdk.frameworks) Cocoa;
  };

  containerd = callPackage ../applications/virtualization/containerd { };

  container2wasm = callPackage ../development/tools/container2wasm { };

  convchain = callPackage ../tools/graphics/convchain { };

  cordless = callPackage ../applications/networking/instant-messengers/cordless { };

  coursera-dl = callPackage ../applications/misc/coursera-dl { };

  coyim = callPackage ../applications/networking/instant-messengers/coyim { };

  cozy-drive = callPackage ../applications/networking/cozy-drive { };

  cplay-ng = callPackage ../applications/audio/cplay-ng { };

  cqrlog = callPackage ../applications/radio/cqrlog {
    hamlib = hamlib_4;
  };

  crun = callPackage ../applications/virtualization/crun { };

  csdp = callPackage ../applications/science/math/csdp { };

  csv2svg = callPackage ../tools/graphics/csv2svg { };

  ctop = callPackage ../tools/system/ctop { };

  cubicsdr = callPackage ../applications/radio/cubicsdr {
    inherit (darwin.apple_sdk.frameworks) Cocoa WebKit;
  };

  cum = callPackage ../applications/misc/cum { };

  cuneiform = callPackage ../tools/graphics/cuneiform { };

  curseradio = callPackage ../applications/audio/curseradio { };

  curtail = callPackage ../applications/graphics/curtail { };

  cutecom = libsForQt5.callPackage ../tools/misc/cutecom { };

  cvs = callPackage ../applications/version-management/cvs { };

  cvsps = callPackage ../applications/version-management/cvsps { };

  cvsq = callPackage ../applications/version-management/cvsq { };

  cvs2svn = callPackage ../applications/version-management/cvs2svn { };

  cwm = callPackage ../applications/window-managers/cwm { };

  cyber = callPackage ../development/interpreters/cyber { };

  cyberduck = callPackage ../applications/networking/cyberduck { };

  cyclone = callPackage ../applications/audio/pd-plugins/cyclone  { };

  dablin = callPackage ../applications/radio/dablin { };

  daktilo = callPackage ../tools/misc/daktilo { };

  darcs = haskell.lib.compose.disableCabalFlag "library"
    (haskell.lib.compose.justStaticExecutables haskellPackages.darcs);

  darkman = callPackage ../applications/misc/darkman { };

  darktable = callPackage ../applications/graphics/darktable {
    lua = lua5_4;
    pugixml = pugixml.override { shared = true; };
    stdenv = if stdenv.isDarwin && stdenv.isx86_64 then overrideSDK llvmPackages_18.stdenv { darwinMinVersion = "10.14"; darwinSdkVersion = "11.0"; } else stdenv;
  };

  das_watchdog = callPackage ../tools/system/das_watchdog { };

  datadog-agent = callPackage ../tools/networking/dd-agent/datadog-agent.nix {
    pythonPackages = datadog-integrations-core { };
  };
  datadog-process-agent = callPackage ../tools/networking/dd-agent/datadog-process-agent.nix { };
  datadog-integrations-core = extras: callPackage ../tools/networking/dd-agent/integrations-core.nix {
    python = python3;
    extraIntegrations = extras;
  };

  dayon = callPackage ../applications/networking/remote/dayon { };

  ddgr = callPackage ../applications/misc/ddgr { };

  deadbeef = callPackage ../applications/audio/deadbeef { };

  deadbeefPlugins = {
    headerbar-gtk3 = callPackage ../applications/audio/deadbeef/plugins/headerbar-gtk3.nix { };
    lyricbar = callPackage ../applications/audio/deadbeef/plugins/lyricbar.nix { };
    mpris2 = callPackage ../applications/audio/deadbeef/plugins/mpris2.nix { };
    musical-spectrum = callPackage ../applications/audio/deadbeef/plugins/musical-spectrum.nix { };
    statusnotifier = callPackage ../applications/audio/deadbeef/plugins/statusnotifier.nix { };
    playlist-manager = callPackage ../applications/audio/deadbeef/plugins/playlist-manager.nix { };
  };

  deadbeef-with-plugins = callPackage ../applications/audio/deadbeef/wrapper.nix {
    plugins = [];
  };

  denaro = callPackage ../applications/finance/denaro { };

  inherit (callPackage ../development/tools/devpod { }) devpod devpod-desktop;

  dfasma = libsForQt5.callPackage ../applications/audio/dfasma { };

  dfilemanager = libsForQt5.callPackage ../applications/file-managers/dfilemanager { };

  dht = callPackage ../applications/networking/p2p/dht { };

  dia = callPackage ../applications/graphics/dia { };

  diebahn = callPackage ../applications/misc/diebahn { };

  digital = callPackage ../applications/science/electronics/digital { };

  direwolf = callPackage ../applications/radio/direwolf {
    hamlib = hamlib_4;
  };

  dirt = callPackage ../applications/audio/dirt { };

  distrho = callPackage ../applications/audio/distrho { };

  dit = callPackage ../applications/editors/dit { };

  djvulibre = callPackage ../applications/misc/djvulibre { };

  djvu2pdf = callPackage ../tools/typesetting/djvu2pdf { };

  djview = libsForQt5.callPackage ../applications/graphics/djview { };
  djview4 = djview;

  dmenu = callPackage ../applications/misc/dmenu { };
  dmenu-wayland = callPackage ../applications/misc/dmenu/wayland.nix { };

  dmenu-rs = callPackage ../applications/misc/dmenu-rs { };

  dmensamenu = callPackage ../applications/misc/dmensamenu {
    inherit (python3Packages) buildPythonApplication requests;
  };

  dmrconfig = callPackage ../applications/radio/dmrconfig { };

  dmtx-utils = callPackage ../tools/graphics/dmtx-utils {
    inherit (darwin.apple_sdk.frameworks) Foundation;
  };

  dnd-tools = callPackage ../applications/misc/dnd-tools { };

  dnglab = callPackage ../tools/graphics/dnglab { };

  inherit (callPackage ../applications/virtualization/docker {})
    docker_24 docker_25 docker_26 docker_27;

  docker = docker_25;
  docker-client = docker.override { clientOnly = true; };

  docker-gc = callPackage ../applications/virtualization/docker/gc.nix { };
  docker-machine-hyperkit = callPackage ../applications/networking/cluster/docker-machine/hyperkit.nix { };
  docker-machine-kvm2 = callPackage ../applications/networking/cluster/docker-machine/kvm2.nix { };

  docker-distribution = callPackage ../applications/virtualization/docker/distribution.nix { };

  dockfmt = callPackage ../development/tools/dockfmt { };

  afterburn = callPackage ../tools/admin/afterburn { };

  docker-buildx = callPackage ../applications/virtualization/docker/buildx.nix { };
  docker-compose = callPackage ../applications/virtualization/docker/compose.nix { };
  docker-compose_1 = python3Packages.callPackage ../applications/virtualization/docker/compose_1.nix { };
  docker-sbom = callPackage ../applications/virtualization/docker/sbom.nix { };


  amazon-ecr-credential-helper = callPackage ../tools/admin/amazon-ecr-credential-helper { };

  dk = callPackage ../applications/window-managers/dk { };

  docker-credential-helpers = callPackage ../tools/admin/docker-credential-helpers { };

  dockstarter = callPackage ../tools/virtualization/dockstarter { };

  doodle = callPackage ../applications/search/doodle { };

  dr14_tmeter = callPackage ../applications/audio/dr14_tmeter { };

  dracut = callPackage ../os-specific/linux/dracut { };

  dragonflydb = callPackage