{
  lib,
  stdenv,
  fetchFromGitHub,
  fetchpatch2,
  pkgsBuildHost,
  glslang,
  meson,
  ninja,
  pkg-config,
  windows,
  spirv-headers,
  vulkan-headers,
  SDL2,
  glfw,
  gitUpdater,
  sdl2Support ? (!stdenv.hostPlatform.isWindows),
  glfwSupport ? (!stdenv.hostPlatform.isWindows),
}:

assert stdenv.hostPlatform.isWindows -> !glfwSupport && !sdl2Support;

let
  inherit (stdenv) hostPlatform;

  libPrefix = lib.optionalString (!hostPlatform.isWindows) "lib";
  soVersion =
    version:
    if hostPlatform.isDarwin then
      ".${version}${hostPlatform.extensions.sharedLibrary}"
    else if hostPlatform.isWindows then
      hostPlatform.extensions.sharedLibrary
    else
      "${hostPlatform.extensions.sharedLibrary}.${version}";

  libglfw = "${libPrefix}glfw${soVersion "3"}";
  libSDL2 = "${libPrefix}SDL2${lib.optionalString (!hostPlatform.isWindows) "-2.0"}${soVersion "0"}";
in
stdenv.mkDerivation (finalAttrs: {
  pname = "dxvk";
  version = "2.5";

  src = fetchFromGitHub {
    owner = "doitsujin";
    repo = "dxvk";
    rev = "v${finalAttrs.version}";
    hash = "sha256-JwEKmzYnoDhad0VKadQYEqraMogXn7BFI9kmT9F+Ts0=";
    fetchSubmodules = true; # Needed for the DirectX headers and libdisplay-info
  };

  patches = [
    # Fixes `ERROR: Index 2 out of bounds of array of size 2` error on native builds.
    (fetchpatch2 {
      url = "https://github.com/doitsujin/dxvk/commit/e6209d28cd9c51ad371605db4c0de27547c0d28c.patch?full_index=1";
      hash = "sha256-zKO6YXOUKBnwcijzCldDzmDqXSkf+wWe9KmUlhsaTyM=";
    })
    # Fixes missing symbol error when building with GLFW support enabled.
    (fetchpatch2 {
      url = "https://github.com/doitsujin/dxvk/commit/62ef4854a6accf2b042f25f44db667abadc0b3fd.patch?full_index=1";
      hash = "sha256-oXybFqoD68gTLHO1kDHll4XCfpU+s9E+PmaXQFqEy/g=";
    })
  ];

  postPatch =
    ''
      substituteInPlace meson.build \
        --replace-fail "dependency('glfw'" "dependency('glfw3'"
      substituteInPlace subprojects/libdisplay-info/tool/gen-search-table.py \
        --replace-fail "/usr/bin/env python3" "${lib.getBin pkgsBuildHost.python3}/bin/python3"
    ''
    + lib.optionalString glfwSupport ''
      substituteInPlace src/wsi/glfw/wsi_platform_glfw.cpp \
        --replace-fail '${libglfw}' '${lib.getLib glfw}/lib/${libglfw}'
    ''
    + lib.optionalString sdl2Support ''
      substituteInPlace src/wsi/sdl2/wsi_platform_sdl2.cpp \
        --replace-fail '${libSDL2}' '${lib.getLib SDL2}/lib/${libSDL2}'
    '';

  strictDeps = true;

  nativeBuildInputs = [
    glslang
    meson
    ninja
  ] ++ lib.optionals (glfwSupport || sdl2Support) [ pkg-config ];

  buildInputs =
    [
      spirv-headers
      vulkan-headers
    ]
    ++ lib.optionals sdl2Support [ SDL2 ]
    ++ lib.optionals glfwSupport [ glfw ]
    ++ lib.optionals hostPlatform.isWindows [ windows.pthreads ];

  # Build with the Vulkan SDK in nixpkgs.
  preConfigure = ''
    rm -rf include/spirv/include include/vulkan/include
    mkdir -p include/spirv/include include/vulkan/include
  '';

  mesonBuildType = "release";

  doCheck = true;

  passthru.updateScript = gitUpdater { rev-prefix = "v"; };

  __structuredAttrs = true;

  meta = {
    description = "Vulkan-based translation layer for Direct3D 8/9/10/11";
    homepage = "https://github.com/doitsujin/dxvk";
    changelog = "https://github.com/doitsujin/dxvk/releases";
    maintainers = [ lib.maintainers.reckenrode ];
    license = lib.licenses.zlib;
    badPlatforms = lib.platforms.darwin;
    platforms = lib.platforms.windows ++ lib.platforms.unix;
    pkgConfigModules = [
      "dxvk-d3d10core"
      "dxvk-d3d11"
      "dxvk-d3d8"
      "dxvk-d3d9"
      "dxvk-dxgi"
    ];
  };
})
