{
  lib,
  autoreconfHook,
  dejagnu,
  mkAppleDerivation,
  stdenv,
  testers,
  texinfo,

  # test suite depends on dejagnu which cannot be used during bootstrapping
  # dejagnu also requires tcl which can't be built statically at the moment
  doCheck ? !(stdenv.hostPlatform.isStatic),
}:

mkAppleDerivation (finalAttrs: {
  releaseName = "libffi";

  outputs = [
    "out"
    "dev"
    "man"
    "info"
  ];

  patches = [
    # Clang 18 requires that no non-private symbols by defined after cfi_startproc. Apply the upstream libffi fix.
    ./patches/llvm-18-compatibility.patch
    # Fix a memory leak when using the trampoline dylib. See https://github.com/libffi/libffi/pull/621#discussion_r955298301.
    ./patches/fix-tramponline-memory-leak.patch
  ];

  # Make sure libffi is using the trampolines dylib in this package not the system one.
  postPatch = ''
    substituteInPlace src/closures.c --replace-fail /usr/lib "$out/lib"
  '';

  enableParallelBuilding = true;

  nativeBuildInputs = [
    autoreconfHook
    texinfo
  ];

  configurePlatforms = [
    "build"
    "host"
  ];

  configureFlags = [
    "--with-gcc-arch=generic" # no detection of -march= or -mtune=
    "--enable-builddir=build"
  ];

  postConfigure = ''
    # Use Apple’s configuration instead of the one generated by the `configure` script.
    cp darwin/include/fficonfig_${stdenv.hostPlatform.darwinArch}.h build/fficonfig.h
    cp darwin/include/ffitarget_${
      if stdenv.hostPlatform.isAarch64 then "arm64" else "x86"
    }.h build/include/ffitarget.h
    # Use `macCatalyst` instead of `iosmac` to avoid errors due to invalid availability annotations.
    substitute darwin/include/ffi.h build/include/ffi.h \
      --replace-fail iosmac macCatalyst
  '';

  postBuild = lib.optionalString stdenv.hostPlatform.isAarch64 ''
    $CC -Os -Wl,-allowable_client,! -Wl,-not_for_dyld_shared_cache -Wl,-no_compact_unwind \
      src/aarch64/trampoline.S -dynamiclib -o libffi-trampolines.dylib \
      -Iinclude -Ibuild -Ibuild/include \
      -install_name "$out/lib/libffi-trampoline.dylib" -Wl,-compatibility_version,1 -Wl,-current_version,1
  '';

  postInstall =
    # The Darwin SDK puts the headers in `include/ffi`. Add a symlink for compatibility.
    ''
      ln -s "$dev/include" "$dev/include/ffi"
    ''
    # Install the trampoline dylib since it is build manually.
    + lib.optionalString stdenv.hostPlatform.isAarch64 ''
      cp libffi-trampolines.dylib "$out/lib/libffi-trampolines.dylib"
    '';

  preCheck = ''
    # The tests use -O0 which is not compatible with -D_FORTIFY_SOURCE.
    NIX_HARDENING_ENABLE=''${NIX_HARDENING_ENABLE/fortify3/}
    NIX_HARDENING_ENABLE=''${NIX_HARDENING_ENABLE/fortify/}
  '';

  dontStrip = stdenv.hostPlatform != stdenv.buildPlatform; # Don't run the native `strip' when cross-compiling.

  inherit doCheck;

  nativeCheckInputs = [ dejagnu ];

  passthru = {
    tests = {
      pkg-config = testers.hasPkgConfigModules {
        package = finalAttrs.finalPackage;
      };
    };
  };

  meta = {
    description = "Foreign function call interface library";
    longDescription = ''
      The libffi library provides a portable, high level programming
      interface to various calling conventions.  This allows a
      programmer to call any function specified by a call interface
      description at run-time.

      FFI stands for Foreign Function Interface.  A foreign function
      interface is the popular name for the interface that allows code
      written in one language to call code written in another
      language.  The libffi library really only provides the lowest,
      machine dependent layer of a fully featured foreign function
      interface.  A layer must exist above libffi that handles type
      conversions for values passed between the two languages.
    '';
    homepage = "https://github.com/apple-oss-distributions/libffi/";
    license = lib.licenses.mit;
    pkgConfigModules = [ "libffi" ];
  };
})
