"""
This code provides access to 10000 points
of a uniform incremental grid on SO3.

The main purpose for this grid is to provide
a systematic method of sampling orientations
in an incremental fashion. The sample sequence
picks new points in such a fashion that it 'avoids'
previous picks. On average, the volume around all points
will be equal.

The sequence was based upon the code described here:

A. Yershova, S. Jain, S. M. LaValle, and J. C. Mitchell.

Generating Uniform Incremental Grids on SO(3) Using
the Hopf Fibration

International Journal of Robotics Research, November 2009.


Instead of recoding this method, I opted for storing a sample sequence
of 10000. At a later junction, a reimplementation of the provide method
could be worthwhile.

The code return a quaternion.

"""
from __future__ import absolute_import, division, print_function
from libtbx.forward_compatibility import object
from scitbx.array_family import flex
from six.moves import range
SO3_sequence = [(0.683013, 0.183013, -0.683013, -0.183013), (-0.183013, 0.683013, 0.183013, -0.683013), (0.5, 0.5, -0.5, -0.5), (-0.5, 0.5, 0.5, -0.5), (0.183013, 0.683013, -0.183013, -0.683013), (-0.683013, 0.183013, 0.683013, -0.183013), (0.683013, 0.183013, 0.683013, 0.183013), (-0.183013, 0.683013, -0.183013, 0.683013), (0.5, 0.5, 0.5, 0.5), (-0.5, 0.5, -0.5, 0.5), (0.183013, 0.683013, 0.183013, 0.683013), (-0.683013, 0.183013, -0.683013, 0.183013), (0.881766, 0.236268, -0.353553, 0.204124), (-0.236268, 0.881766, -0.204124, -0.353553), (0.645497, 0.645497, -0.408248, 4.9996e-17), (-0.645497, 0.645497, -7.4994e-17, -0.408248), (0.236268, 0.881766, -0.353553, -0.204124), (-0.881766, 0.236268, 0.204124, -0.353553), (0.394338, 0.105662, 0.790569, -0.456435), (-0.105662, 0.394338, 0.456435, 0.790569), (0.288675, 0.288675, 0.912871, -2.23589e-16), (-0.288675, 0.288675, 2.79486e-16, 0.912871), (0.105662, 0.394338, 0.790569, 0.456435), (-0.394338, 0.105662, -0.456435, 0.790569), (0.394338, 0.105662, -0.790569, 0.456435), (-0.105662, 0.394338, -0.456435, -0.790569), (0.288675, 0.288675, -0.912871, 1.11794e-16), (-0.288675, 0.288675, -1.67692e-16, -0.912871), (0.105662, 0.394338, -0.790569, -0.456435), (-0.394338, 0.105662, 0.456435, -0.790569), (0.881766, 0.236268, 0.353553, -0.204124), (-0.236268, 0.881766, 0.204124, 0.353553), (0.645497, 0.645497, 0.408248, -9.9992e-17), (-0.645497, 0.645497, 1.2499e-16, 0.408248), (0.236268, 0.881766, 0.353553, 0.204124), (-0.881766, 0.236268, -0.204124, 0.353553), (0.683013, 0.183013, -0.183013, 0.683013), (-0.183013, 0.683013, -0.683013, -0.183013), (0.5, 0.5, -0.5, 0.5), (-0.5, 0.5, -0.5, -0.5), (0.183013, 0.683013, -0.683013, 0.183013), (-0.683013, 0.183013, -0.183013, -0.683013), (0.683013, 0.183013, 0.183013, -0.683013), (-0.183013, 0.683013, 0.683013, 0.183013), (0.5, 0.5, 0.5, -0.5), (-0.5, 0.5, 0.5, 0.5), (0.183013, 0.683013, 0.683013, -0.183013), (-0.683013, 0.183013, 0.183013, 0.683013), (0.394338, 0.105662, -0.456435, -0.790569), (-0.105662, 0.394338, 0.790569, -0.456435), (0.288675, 0.288675, -1.67692e-16, -0.912871), (-0.288675, 0.288675, 0.912871, -2.23589e-16), (0.105662, 0.394338, 0.456435, -0.790569), (-0.394338, 0.105662, 0.790569, 0.456435), (0.881766, 0.236268, 0.204124, 0.353553), (-0.236268, 0.881766, -0.353553, 0.204124), (0.645497, 0.645497, 2.4998e-17, 0.408248), (-0.645497, 0.645497, -0.408248, 4.9996e-17), (0.236268, 0.881766, -0.204124, 0.353553), (-0.881766, 0.236268, -0.353553, -0.204124), (0.881766, 0.236268, -0.204124, -0.353553), (-0.236268, 0.881766, 0.353553, -0.204124), (0.645497, 0.645497, -7.4994e-17, -0.408248), (-0.645497, 0.645497, 0.408248, -9.9992e-17), (0.236268, 0.881766, 0.204124, -0.353553), (-0.881766, 0.236268, 0.353553, 0.204124), (0.394338, 0.105662, 0.456435, 0.790569), (-0.105662, 0.394338, -0.790569, 0.456435), (0.288675, 0.288675, 5.58972e-17, 0.912871), (-0.288675, 0.288675, -0.912871, 1.11794e-16), (0.105662, 0.394338, -0.456435, 0.790569), (-0.394338, 0.105662, -0.790569, -0.456435), (0.809511, 0.106574, -0.572411, -0.0753593), (-0.106574, 0.809511, 0.0753593, -0.572411), (0.64777, 0.497052, -0.458043, -0.351469), (-0.497052, 0.64777, 0.351469, -0.458043), (0.31246, 0.754344, -0.220942, -0.533402), (-0.754344, 0.31246, 0.533402, -0.220942), (0.572411, 0.0753593, 0.809511, 0.106574), (-0.0753593, 0.572411, -0.106574, 0.809511), (0.458043, 0.351469, 0.64777, 0.497052), (-0.351469, 0.458043, -0.497052, 0.64777), (0.220942, 0.533402, 0.31246, 0.754344), (-0.533402, 0.220942, -0.754344, 0.31246), (0.97057, 0.127778, -0.161943, 0.124263), (-0.127778, 0.97057, -0.124263, -0.161943), (0.776649, 0.595944, -0.202378, 0.0266435), (-0.595944, 0.776649, -0.0266435, -0.202378), (0.374626, 0.904427, -0.188586, -0.0781149), (-0.904427, 0.374626, 0.0781149, -0.188586), (0.572411, 0.0753593, 0.64777, -0.497052), (-0.0753593, 0.572411, 0.497052, 0.64777), (0.458043, 0.351469, 0.809511, -0.106574), (-0.351469, 0.458043, 0.106574, 0.809511), (0.220942, 0.533402, 0.754344, 0.31246), (-0.533402, 0.220942, -0.31246, 0.754344), (0.572411, 0.0753593, -0.64777, 0.497052), (-0.0753593, 0.572411, -0.497052, -0.64777), (0.458043, 0.351469, -0.809511, 0.106574), (-0.351469, 0.458043, -0.106574, -0.809511), (0.220942, 0.533402, -0.754344, -0.31246), (-0.533402, 0.220942, 0.31246, -0.754344), (0.97057, 0.127778, 0.161943, -0.124263), (-0.127778, 0.97057, 0.124263, 0.161943), (0.776649, 0.595944, 0.202378, -0.0266435), (-0.595944, 0.776649, 0.0266435, 0.202378), (0.374626, 0.904427, 0.188586, 0.0781149), (-0.904427, 0.374626, -0.0781149, 0.188586), (0.572411, 0.0753593, -0.106574, 0.809511), (-0.0753593, 0.572411, -0.809511, -0.106574), (0.458043, 0.351469, -0.497052, 0.64777), (-0.351469, 0.458043, -0.64777, -0.497052), (0.220942, 0.533402, -0.754344, 0.31246), (-0.533402, 0.220942, -0.31246, -0.754344), (0.809511, 0.106574, 0.0753593, -0.572411), (-0.106574, 0.809511, 0.572411, 0.0753593), (0.64777, 0.497052, 0.351469, -0.458043), (-0.497052, 0.64777, 0.458043, 0.351469), (0.31246, 0.754344, 0.533402, -0.220942), (-0.754344, 0.31246, 0.220942, 0.533402), (0.202378, 0.0266435, -0.595944, -0.776649), (-0.0266435, 0.202378, 0.776649, -0.595944), (0.161943, 0.124263, -0.127778, -0.97057), (-0.124263, 0.161943, 0.97057, -0.127778), (0.0781149, 0.188586, 0.374626, -0.904427), (-0.188586, 0.0781149, 0.904427, 0.374626), (0.809511, 0.106574, 0.351469, 0.458043), (-0.106574, 0.809511, -0.458043, 0.351469), (0.64777, 0.497052, 0.0753593, 0.572411), (-0.497052, 0.64777, -0.572411, 0.0753593), (0.31246, 0.754344, -0.220942, 0.533402), (-0.754344, 0.31246, -0.533402, -0.220942), (0.809511, 0.106574, -0.351469, -0.458043), (-0.106574, 0.809511, 0.458043, -0.351469), (0.64777, 0.497052, -0.0753593, -0.572411), (-0.497052, 0.64777, 0.572411, -0.0753593), (0.31246, 0.754344, 0.220942, -0.533402), (-0.754344, 0.31246, 0.533402, 0.220942), (0.202378, 0.0266435, 0.595944, 0.776649), (-0.0266435, 0.202378, -0.776649, 0.595944), (0.161943, 0.124263, 0.127778, 0.97057), (-0.124263, 0.161943, -0.97057, 0.127778), (0.0781149, 0.188586, -0.374626, 0.904427), (-0.188586, 0.0781149, -0.904427, -0.374626), (0.533402, 0.220942, -0.754344, -0.31246), (-0.220942, 0.533402, 0.31246, -0.754344), (0.351469, 0.458043, -0.497052, -0.64777), (-0.458043, 0.351469, 0.64777, -0.497052), (0.0753593, 0.572411, -0.106574, -0.809511), (-0.572411, 0.0753593, 0.809511, -0.106574), (0.754344, 0.31246, 0.533402, 0.220942), (-0.31246, 0.754344, -0.220942, 0.533402), (0.497052, 0.64777, 0.351469, 0.458043), (-0.64777, 0.497052, -0.458043, 0.351469), (0.106574, 0.809511, 0.0753593, 0.572411), (-0.809511, 0.106574, -0.572411, 0.0753593), (0.754344, 0.31246, -0.533402, 0.220942), (-0.31246, 0.754344, -0.220942, -0.533402), (0.497052, 0.64777, -0.572411, -0.0753593), (-0.64777, 0.497052, 0.0753593, -0.572411), (0.106574, 0.809511, -0.458043, -0.351469), (-0.809511, 0.106574, 0.351469, -0.458043), (0.188586, 0.0781149, 0.904427, -0.374626), (-0.0781149, 0.188586, 0.374626, 0.904427), (0.124263, 0.161943, 0.97057, 0.127778), (-0.161943, 0.124263, -0.127778, 0.97057), (0.0266435, 0.202378, 0.776649, 0.595944), (-0.202378, 0.0266435, -0.595944, 0.776649), (0.188586, 0.0781149, -0.904427, 0.374626), (-0.0781149, 0.188586, -0.374626, -0.904427), (0.124263, 0.161943, -0.97057, -0.127778), (-0.161943, 0.124263, 0.127778, -0.97057), (0.0266435, 0.202378, -0.776649, -0.595944), (-0.202378, 0.0266435, 0.595944, -0.776649), (0.754344, 0.31246, 0.533402, -0.220942), (-0.31246, 0.754344, 0.220942, 0.533402), (0.497052, 0.64777, 0.572411, 0.0753593), (-0.64777, 0.497052, -0.0753593, 0.572411), (0.106574, 0.809511, 0.458043, 0.351469), (-0.809511, 0.106574, -0.351469, 0.458043), (0.754344, 0.31246, -0.220942, 0.533402), (-0.31246, 0.754344, -0.533402, -0.220942), (0.497052, 0.64777, -0.458043, 0.351469), (-0.64777, 0.497052, -0.351469, -0.458043), (0.106574, 0.809511, -0.572411, 0.0753593), (-0.809511, 0.106574, -0.0753593, -0.572411), (0.533402, 0.220942, 0.31246, -0.754344), (-0.220942, 0.533402, 0.754344, 0.31246), (0.351469, 0.458043, 0.64777, -0.497052), (-0.458043, 0.351469, 0.497052, 0.64777), (0.0753593, 0.572411, 0.809511, -0.106574), (-0.572411, 0.0753593, 0.106574, 0.809511), (0.533402, 0.220942, -0.31246, -0.754344), (-0.220942, 0.533402, 0.754344, -0.31246), (0.351469, 0.458043, 0.106574, -0.809511), (-0.458043, 0.351469, 0.809511, 0.106574), (0.0753593, 0.572411, 0.497052, -0.64777), (-0.572411, 0.0753593, 0.64777, 0.497052), (0.904427, 0.374626, 0.0781149, 0.188586), (-0.374626, 0.904427, -0.188586, 0.0781149), (0.595944, 0.776649, -0.0266435, 0.202378), (-0.776649, 0.595944, -0.202378, -0.0266435), (0.127778, 0.97057, -0.124263, 0.161943), (-0.97057, 0.127778, -0.161943, -0.124263), (0.904427, 0.374626, -0.0781149, -0.188586), (-0.374626, 0.904427, 0.188586, -0.0781149), (0.595944, 0.776649, 0.0266435, -0.202378), (-0.776649, 0.595944, 0.202378, 0.0266435), (0.127778, 0.97057, 0.124263, -0.161943), (-0.97057, 0.127778, 0.161943, 0.124263), (0.533402, 0.220942, 0.31246, 0.754344), (-0.220942, 0.533402, -0.754344, 0.31246), (0.351469, 0.458043, -0.106574, 0.809511), (-0.458043, 0.351469, -0.809511, -0.106574), (0.0753593, 0.572411, -0.497052, 0.64777), (-0.572411, 0.0753593, -0.64777, -0.497052), (0.754344, 0.31246, -0.533402, -0.220942), (-0.31246, 0.754344, 0.220942, -0.533402), (0.497052, 0.64777, -0.351469, -0.458043), (-0.64777, 0.497052, 0.458043, -0.351469), (0.106574, 0.809511, -0.0753593, -0.572411), (-0.809511, 0.106574, 0.572411, -0.0753593), (0.533402, 0.220942, 0.754344, 0.31246), (-0.220942, 0.533402, -0.31246, 0.754344), (0.351469, 0.458043, 0.497052, 0.64777), (-0.458043, 0.351469, -0.64777, 0.497052), (0.0753593, 0.572411, 0.106574, 0.809511), (-0.572411, 0.0753593, -0.809511, 0.106574), (0.904427, 0.374626, -0.188586, 0.0781149), (-0.374626, 0.904427, -0.0781149, -0.188586), (0.595944, 0.776649, -0.202378, -0.0266435), (-0.776649, 0.595944, 0.0266435, -0.202378), (0.127778, 0.97057, -0.161943, -0.124263), (-0.97057, 0.127778, 0.124263, -0.161943), (0.533402, 0.220942, 0.754344, -0.31246), (-0.220942, 0.533402, 0.31246, 0.754344), (0.351469, 0.458043, 0.809511, 0.106574), (-0.458043, 0.351469, -0.106574, 0.809511), (0.0753593, 0.572411, 0.64777, 0.497052), (-0.572411, 0.0753593, -0.497052, 0.64777), (0.533402, 0.220942, -0.754344, 0.31246), (-0.220942, 0.533402, -0.31246, -0.754344), (0.351469, 0.458043, -0.809511, -0.106574), (-0.458043, 0.351469, 0.106574, -0.809511), (0.0753593, 0.572411, -0.64777, -0.497052), (-0.572411, 0.0753593, 0.497052, -0.64777), (0.904427, 0.374626, 0.188586, -0.0781149), (-0.374626, 0.904427, 0.0781149, 0.188586), (0.595944, 0.776649, 0.202378, 0.0266435), (-0.776649, 0.595944, -0.0266435, 0.202378), (0.127778, 0.97057, 0.161943, 0.124263), (-0.97057, 0.127778, -0.124263, 0.161943), (0.533402, 0.220942, -0.31246, 0.754344), (-0.220942, 0.533402, -0.754344, -0.31246), (0.351469, 0.458043, -0.64777, 0.497052), (-0.458043, 0.351469, -0.497052, -0.64777), (0.0753593, 0.572411, -0.809511, 0.106574), (-0.572411, 0.0753593, -0.106574, -0.809511), (0.754344, 0.31246, 0.220942, -0.533402), (-0.31246, 0.754344, 0.533402, 0.220942), (0.497052, 0.64777, 0.458043, -0.351469), (-0.64777, 0.497052, 0.351469, 0.458043), (0.106574, 0.809511, 0.572411, -0.0753593), (-0.809511, 0.106574, 0.0753593, 0.572411), (0.188586, 0.0781149, -0.374626, -0.904427), (-0.0781149, 0.188586, 0.904427, -0.374626), (0.124263, 0.161943, 0.127778, -0.97057), (-0.161943, 0.124263, 0.97057, 0.127778), (0.0266435, 0.202378, 0.595944, -0.776649), (-0.202378, 0.0266435, 0.776649, 0.595944), (0.754344, 0.31246, 0.220942, 0.533402), (-0.31246, 0.754344, -0.533402, 0.220942), (0.497052, 0.64777, -0.0753593, 0.572411), (-0.64777, 0.497052, -0.572411, -0.0753593), (0.106574, 0.809511, -0.351469, 0.458043), (-0.809511, 0.106574, -0.458043, -0.351469), (0.754344, 0.31246, -0.220942, -0.533402), (-0.31246, 0.754344, 0.533402, -0.220942), (0.497052, 0.64777, 0.0753593, -0.572411), (-0.64777, 0.497052, 0.572411, 0.0753593), (0.106574, 0.809511, 0.351469, -0.458043), (-0.809511, 0.106574, 0.458043, 0.351469), (0.188586, 0.0781149, 0.374626, 0.904427), (-0.0781149, 0.188586, -0.904427, 0.374626), (0.124263, 0.161943, -0.127778, 0.97057), (-0.161943, 0.124263, -0.97057, -0.127778), (0.0266435, 0.202378, -0.595944, 0.776649), (-0.202378, 0.0266435, -0.776649, -0.595944), (0.572411, 0.0753593, -0.809511, -0.106574), (-0.0753593, 0.572411, 0.106574, -0.809511), (0.458043, 0.351469, -0.64777, -0.497052), (-0.351469, 0.458043, 0.497052, -0.64777), (0.220942, 0.533402, -0.31246, -0.754344), (-0.533402, 0.220942, 0.754344, -0.31246), (0.809511, 0.106574, 0.572411, 0.0753593), (-0.106574, 0.809511, -0.0753593, 0.572411), (0.64777, 0.497052, 0.458043, 0.351469), (-0.497052, 0.64777, -0.351469, 0.458043), (0.31246, 0.754344, 0.220942, 0.533402), (-0.754344, 0.31246, -0.533402, 0.220942), (0.809511, 0.106574, -0.458043, 0.351469), (-0.106574, 0.809511, -0.351469, -0.458043), (0.64777, 0.497052, -0.572411, 0.0753593), (-0.497052, 0.64777, -0.0753593, -0.572411), (0.31246, 0.754344, -0.533402, -0.220942), (-0.754344, 0.31246, 0.220942, -0.533402), (0.202378, 0.0266435, 0.776649, -0.595944), (-0.0266435, 0.202378, 0.595944, 0.776649), (0.161943, 0.124263, 0.97057, -0.127778), (-0.124263, 0.161943, 0.127778, 0.97057), (0.0781149, 0.188586, 0.904427, 0.374626), (-0.188586, 0.0781149, -0.374626, 0.904427), (0.202378, 0.0266435, -0.776649, 0.595944), (-0.0266435, 0.202378, -0.595944, -0.776649), (0.161943, 0.124263, -0.97057, 0.127778), (-0.124263, 0.161943, -0.127778, -0.97057), (0.0781149, 0.188586, -0.904427, -0.374626), (-0.188586, 0.0781149, 0.374626, -0.904427), (0.809511, 0.106574, 0.458043, -0.351469), (-0.106574, 0.809511, 0.351469, 0.458043), (0.64777, 0.497052, 0.572411, -0.0753593), (-0.497052, 0.64777, 0.0753593, 0.572411), (0.31246, 0.754344, 0.533402, 0.220942), (-0.754344, 0.31246, -0.220942, 0.533402), (0.809511, 0.106574, -0.0753593, 0.572411), (-0.106574, 0.809511, -0.572411, -0.0753593), (0.64777, 0.497052, -0.351469, 0.458043), (-0.497052, 0.64777, -0.458043, -0.351469), (0.31246, 0.754344, -0.533402, 0.220942), (-0.754344, 0.31246, -0.220942, -0.533402), (0.572411, 0.0753593, 0.106574, -0.809511), (-0.0753593, 0.572411, 0.809511, 0.106574), (0.458043, 0.351469, 0.497052, -0.64777), (-0.351469, 0.458043, 0.64777, 0.497052), (0.220942, 0.533402, 0.754344, -0.31246), (-0.533402, 0.220942, 0.31246, 0.754344), (0.572411, 0.0753593, -0.497052, -0.64777), (-0.0753593, 0.572411, 0.64777, -0.497052), (0.458043, 0.351469, -0.106574, -0.809511), (-0.351469, 0.458043, 0.809511, -0.106574), (0.220942, 0.533402, 0.31246, -0.754344), (-0.533402, 0.220942, 0.754344, 0.31246), (0.97057, 0.127778, 0.124263, 0.161943), (-0.127778, 0.97057, -0.161943, 0.124263), (0.776649, 0.595944, 0.0266435, 0.202378), (-0.595944, 0.776649, -0.202378, 0.0266435), (0.374626, 0.904427, -0.0781149, 0.188586), (-0.904427, 0.374626, -0.188586, -0.0781149), (0.97057, 0.127778, -0.124263, -0.161943), (-0.127778, 0.97057, 0.161943, -0.124263), (0.776649, 0.595944, -0.0266435, -0.202378), (-0.595944, 0.776649, 0.202378, -0.0266435), (0.374626, 0.904427, 0.0781149, -0.188586), (-0.904427, 0.374626, 0.188586, 0.0781149), (0.572411, 0.0753593, 0.497052, 0.64777), (-0.0753593, 0.572411, -0.64777, 0.497052), (0.458043, 0.351469, 0.106574, 0.809511), (-0.351469, 0.458043, -0.809511, 0.106574), (0.220942, 0.533402, -0.31246, 0.754344), (-0.533402, 0.220942, -0.754344, -0.31246), (0.701057, 0.092296, -0.683013, 0.183013), (-0.092296, 0.701057, -0.183013, -0.683013), (0.560986, 0.430459, -0.683013, -0.183013), (-0.430459, 0.560986, 0.183013, -0.683013), (0.270598, 0.653281, -0.5, -0.5), (-0.653281, 0.270598, 0.5, -0.5), (0.701057, 0.092296, 0.683013, -0.183013), (-0.092296, 0.701057, 0.183013, 0.683013), (0.560986, 0.430459, 0.683013, 0.183013), (-0.430459, 0.560986, -0.183013, 0.683013), (0.270598, 0.653281, 0.5, 0.5), (-0.653281, 0.270598, -0.5, 0.5), (0.905061, 0.119154, -0.394338, 0.105662), (-0.119154, 0.905061, -0.105662, -0.394338), (0.724229, 0.555721, -0.394338, -0.105662), (-0.555721, 0.724229, 0.105662, -0.394338), (0.349341, 0.843383, -0.288675, -0.288675), (-0.843383, 0.349341, 0.288675, -0.288675), (0.404756, 0.0532871, 0.881766, -0.236268), (-0.0532871, 0.404756, 0.236268, 0.881766), (0.323885, 0.248526, 0.881766, 0.236268), (-0.248526, 0.323885, -0.236268, 0.881766), (0.15623, 0.377172, 0.645497, 0.645497), (-0.377172, 0.15623, -0.645497, 0.645497), (0.404756, 0.0532871, -0.881766, 0.236268), (-0.0532871, 0.404756, -0.236268, -0.881766), (0.323885, 0.248526, -0.881766, -0.236268), (-0.248526, 0.323885, 0.236268, -0.881766), (0.15623, 0.377172, -0.645497, -0.645497), (-0.377172, 0.15623, 0.645497, -0.645497), (0.905061, 0.119154, 0.394338, -0.105662), (-0.119154, 0.905061, 0.105662, 0.394338), (0.724229, 0.555721, 0.394338, 0.105662), (-0.555721, 0.724229, -0.105662, 0.394338), (0.349341, 0.843383, 0.288675, 0.288675), (-0.843383, 0.349341, -0.288675, 0.288675), (0.701057, 0.092296, 0.183013, 0.683013), (-0.092296, 0.701057, -0.683013, 0.183013), (0.560986, 0.430459, -0.183013, 0.683013), (-0.430459, 0.560986, -0.683013, -0.183013), (0.270598, 0.653281, -0.5, 0.5), (-0.653281, 0.270598, -0.5, -0.5), (0.701057, 0.092296, -0.183013, -0.683013), (-0.092296, 0.701057, 0.683013, -0.183013), (0.560986, 0.430459, 0.183013, -0.683013), (-0.430459, 0.560986, 0.683013, 0.183013), (0.270598, 0.653281, 0.5, -0.5), (-0.653281, 0.270598, 0.5, 0.5), (0.404756, 0.0532871, -0.236268, -0.881766), (-0.0532871, 0.404756, 0.881766, -0.236268), (0.323885, 0.248526, 0.236268, -0.881766), (-0.248526, 0.323885, 0.881766, 0.236268), (0.15623, 0.377172, 0.645497, -0.645497), (-0.377172, 0.15623, 0.645497, 0.645497), (0.905061, 0.119154, 0.105662, 0.394338), (-0.119154, 0.905061, -0.394338, 0.105662), (0.724229, 0.555721, -0.105662, 0.394338), (-0.555721, 0.724229, -0.394338, -0.105662), (0.349341, 0.843383, -0.288675, 0.288675), (-0.843383, 0.349341, -0.288675, -0.288675), (0.905061, 0.119154, -0.105662, -0.394338), (-0.119154, 0.905061, 0.394338, -0.105662), (0.724229, 0.555721, 0.105662, -0.394338), (-0.555721, 0.724229, 0.394338, 0.105662), (0.349341, 0.843383, 0.288675, -0.288675), (-0.843383, 0.349341, 0.288675, 0.288675), (0.404756, 0.0532871, 0.236268, 0.881766), (-0.0532871, 0.404756, -0.881766, 0.236268), (0.323885, 0.248526, -0.236268, 0.881766), (-0.248526, 0.323885, -0.881766, -0.236268), (0.15623, 0.377172, -0.645497, 0.645497), (-0.377172, 0.15623, -0.645497, -0.645497), (0.653281, 0.270598, -0.5, -0.5), (-0.270598, 0.653281, 0.5, -0.5), (0.430459, 0.560986, -0.183013, -0.683013), (-0.560986, 0.430459, 0.683013, -0.183013), (0.092296, 0.701057, 0.183013, -0.683013), (-0.701057, 0.092296, 0.683013, 0.183013), (0.653281, 0.270598, 0.5, 0.5), (-0.270598, 0.653281, -0.5, 0.5), (0.430459, 0.560986, 0.183013, 0.683013), (-0.560986, 0.430459, -0.683013, 0.183013), (0.092296, 0.701057, -0.183013, 0.683013), (-0.701057, 0.092296, -0.683013, -0.183013), (0.843383, 0.349341, -0.288675, 0.288675), (-0.349341, 0.843383, -0.288675, -0.288675), (0.555721, 0.724229, -0.394338, 0.105662), (-0.724229, 0.555721, -0.105662, -0.394338), (0.119154, 0.905061, -0.394338, -0.105662), (-0.905061, 0.119154, 0.105662, -0.394338), (0.377172, 0.15623, 0.645497, -0.645497), (-0.15623, 0.377172, 0.645497, 0.645497), (0.248526, 0.323885, 0.881766, -0.236268), (-0.323885, 0.248526, 0.236268, 0.881766), (0.0532871, 0.404756, 0.881766, 0.236268), (-0.404756, 0.0532871, -0.236268, 0.881766), (0.377172, 0.15623, -0.645497, 0.645497), (-0.15623, 0.377172, -0.645497, -0.645497), (0.248526, 0.323885, -0.881766, 0.236268), (-0.323885, 0.248526, -0.236268, -0.881766), (0.0532871, 0.404756, -0.881766, -0.236268), (-0.404756, 0.0532871, 0.236268, -0.881766), (0.843383, 0.349341, 0.288675, -0.288675), (-0.349341, 0.843383, 0.288675, 0.288675), (0.555721, 0.724229, 0.394338, -0.105662), (-0.724229, 0.555721, 0.105662, 0.394338), (0.119154, 0.905061, 0.394338, 0.105662), (-0.905061, 0.119154, -0.105662, 0.394338), (0.653281, 0.270598, -0.5, 0.5), (-0.270598, 0.653281, -0.5, -0.5), (0.430459, 0.560986, -0.683013, 0.183013), (-0.560986, 0.430459, -0.183013, -0.683013), (0.092296, 0.701057, -0.683013, -0.183013), (-0.701057, 0.092296, 0.183013, -0.683013), (0.653281, 0.270598, 0.5, -0.5), (-0.270598, 0.653281, 0.5, 0.5), (0.430459, 0.560986, 0.683013, -0.183013), (-0.560986, 0.430459, 0.183013, 0.683013), (0.092296, 0.701057, 0.683013, 0.183013), (-0.701057, 0.092296, -0.183013, 0.683013), (0.377172, 0.15623, -0.645497, -0.645497), (-0.15623, 0.377172, 0.645497, -0.645497), (0.248526, 0.323885, -0.236268, -0.881766), (-0.323885, 0.248526, 0.881766, -0.236268), (0.0532871, 0.404756, 0.236268, -0.881766), (-0.404756, 0.0532871, 0.881766, 0.236268), (0.843383, 0.349341, 0.288675, 0.288675), (-0.349341, 0.843383, -0.288675, 0.288675), (0.555721, 0.724229, 0.105662, 0.394338), (-0.724229, 0.555721, -0.394338, 0.105662), (0.119154, 0.905061, -0.105662, 0.394338), (-0.905061, 0.119154, -0.394338, -0.105662), (0.843383, 0.349341, -0.288675, -0.288675), (-0.349341, 0.843383, 0.288675, -0.288675), (0.555721, 0.724229, -0.105662, -0.394338), (-0.724229, 0.555721, 0.394338, -0.105662), (0.119154, 0.905061, 0.105662, -0.394338), (-0.905061, 0.119154, 0.394338, 0.105662), (0.377172, 0.15623, 0.645497, 0.645497), (-0.15623, 0.377172, -0.645497, 0.645497), (0.248526, 0.323885, 0.236268, 0.881766), (-0.323885, 0.248526, -0.881766, 0.236268), (0.0532871, 0.404756, -0.236268, 0.881766), (-0.404756, 0.0532871, -0.881766, -0.236268), (0.653281, 0.270598, -0.707107, 8.65956e-17), (-0.270598, 0.653281, -1.29893e-16, -0.707107), (0.430459, 0.560986, -0.612372, -0.353553), (-0.560986, 0.430459, 0.353553, -0.612372), (0.092296, 0.701057, -0.353553, -0.612372), (-0.701057, 0.092296, 0.612372, -0.353553), (0.653281, 0.270598, 0.707107, -1.73191e-16), (-0.270598, 0.653281, 2.16489e-16, 0.707107), (0.430459, 0.560986, 0.612372, 0.353553), (-0.560986, 0.430459, -0.353553, 0.612372), (0.092296, 0.701057, 0.353553, 0.612372), (-0.701057, 0.092296, -0.612372, 0.353553), (0.843383, 0.349341, -0.408248, 4.9996e-17), (-0.349341, 0.843383, -7.4994e-17, -0.408248), (0.555721, 0.724229, -0.353553, -0.204124), (-0.724229, 0.555721, 0.204124, -0.353553), (0.119154, 0.905061, -0.204124, -0.353553), (-0.905061, 0.119154, 0.353553, -0.204124), (0.377172, 0.15623, 0.912871, -2.23589e-16), (-0.15623, 0.377172, 2.79486e-16, 0.912871), (0.248526, 0.323885, 0.790569, 0.456435), (-0.323885, 0.248526, -0.456435, 0.790569), (0.0532871, 0.404756, 0.456435, 0.790569), (-0.404756, 0.0532871, -0.790569, 0.456435), (0.377172, 0.15623, -0.912871, 1.11794e-16), (-0.15623, 0.377172, -1.67692e-16, -0.912871), (0.248526, 0.323885, -0.790569, -0.456435), (-0.323885, 0.248526, 0.456435, -0.790569), (0.0532871, 0.404756, -0.456435, -0.790569), (-0.404756, 0.0532871, 0.790569, -0.456435), (0.843383, 0.349341, 0.408248, -9.9992e-17), (-0.349341, 0.843383, 1.2499e-16, 0.408248), (0.555721, 0.724229, 0.353553, 0.204124), (-0.724229, 0.555721, -0.204124, 0.353553), (0.119154, 0.905061, 0.204124, 0.353553), (-0.905061, 0.119154, -0.353553, 0.204124), (0.653281, 0.270598, 4.32978e-17, 0.707107), (-0.270598, 0.653281, -0.707107, 8.65956e-17), (0.430459, 0.560986, -0.353553, 0.612372), (-0.560986, 0.430459, -0.612372, -0.353553), (0.092296, 0.701057, -0.612372, 0.353553), (-0.701057, 0.092296, -0.353553, -0.612372), (0.653281, 0.270598, -1.29893e-16, -0.707107), (-0.270598, 0.653281, 0.707107, -1.73191e-16), (0.430459, 0.560986, 0.353553, -0.612372), (-0.560986, 0.430459, 0.612372, 0.353553), (0.092296, 0.701057, 0.612372, -0.353553), (-0.701057, 0.092296, 0.353553, 0.612372), (0.377172, 0.15623, -1.67692e-16, -0.912871), (-0.15623, 0.377172, 0.912871, -2.23589e-16), (0.248526, 0.323885, 0.456435, -0.790569), (-0.323885, 0.248526, 0.790569, 0.456435), (0.0532871, 0.404756, 0.790569, -0.456435), (-0.404756, 0.0532871, 0.456435, 0.790569), (0.843383, 0.349341, 2.4998e-17, 0.408248), (-0.349341, 0.843383, -0.408248, 4.9996e-17), (0.555721, 0.724229, -0.204124, 0.353553), (-0.724229, 0.555721, -0.353553, -0.204124), (0.119154, 0.905061, -0.353553, 0.204124), (-0.905061, 0.119154, -0.204124, -0.353553), (0.843383, 0.349341, -7.4994e-17, -0.408248), (-0.349341, 0.843383, 0.408248, -9.9992e-17), (0.555721, 0.724229, 0.204124, -0.353553), (-0.724229, 0.555721, 0.353553, 0.204124), (0.119154, 0.905061, 0.353553, -0.204124), (-0.905061, 0.119154, 0.204124, 0.353553), (0.377172, 0.15623, 5.58972e-17, 0.912871), (-0.15623, 0.377172, -0.912871, 1.11794e-16), (0.248526, 0.323885, -0.456435, 0.790569), (-0.323885, 0.248526, -0.790569, -0.456435), (0.0532871, 0.404756, -0.790569, 0.456435), (-0.404756, 0.0532871, -0.456435, -0.790569), (0.701057, 0.092296, -0.612372, -0.353553), (-0.092296, 0.701057, 0.353553, -0.612372), (0.560986, 0.430459, -0.353553, -0.612372), (-0.430459, 0.560986, 0.612372, -0.353553), (0.270598, 0.653281, -1.29893e-16, -0.707107), (-0.653281, 0.270598, 0.707107, -1.73191e-16), (0.701057, 0.092296, 0.612372, 0.353553), (-0.092296, 0.701057, -0.353553, 0.612372), (0.560986, 0.430459, 0.353553, 0.612372), (-0.430459, 0.560986, -0.612372, 0.353553), (0.270598, 0.653281, 4.32978e-17, 0.707107), (-0.653281, 0.270598, -0.707107, 8.65956e-17), (0.905061, 0.119154, -0.204124, 0.353553), (-0.119154, 0.905061, -0.353553, -0.204124), (0.724229, 0.555721, -0.353553, 0.204124), (-0.555721, 0.724229, -0.204124, -0.353553), (0.349341, 0.843383, -0.408248, 4.9996e-17), (-0.843383, 0.349341, -7.4994e-17, -0.408248), (0.404756, 0.0532871, 0.456435, -0.790569), (-0.0532871, 0.404756, 0.790569, 0.456435), (0.323885, 0.248526, 0.790569, -0.456435), (-0.248526, 0.323885, 0.456435, 0.790569), (0.15623, 0.377172, 0.912871, -2.23589e-16), (-0.377172, 0.15623, 2.79486e-16, 0.912871), (0.404756, 0.0532871, -0.456435, 0.790569), (-0.0532871, 0.404756, -0.790569, -0.456435), (0.323885, 0.248526, -0.790569, 0.456435), (-0.248526, 0.323885, -0.456435, -0.790569), (0.15623, 0.377172, -0.912871, 1.11794e-16), (-0.377172, 0.15623, -1.67692e-16, -0.912871), (0.905061, 0.119154, 0.204124, -0.353553), (-0.119154, 0.905061, 0.353553, 0.204124), (0.724229, 0.555721, 0.353553, -0.204124), (-0.555721, 0.724229, 0.204124, 0.353553), (0.349341, 0.843383, 0.408248, -9.9992e-17), (-0.843383, 0.349341, 1.2499e-16, 0.408248), (0.701057, 0.092296, -0.353553, 0.612372), (-0.092296, 0.701057, -0.612372, -0.353553), (0.560986, 0.430459, -0.612372, 0.353553), (-0.430459, 0.560986, -0.353553, -0.612372), (0.270598, 0.653281, -0.707107, 8.65956e-17), (-0.653281, 0.270598, -1.29893e-16, -0.707107), (0.701057, 0.092296, 0.353553, -0.612372), (-0.092296, 0.701057, 0.612372, 0.353553), (0.560986, 0.430459, 0.612372, -0.353553), (-0.430459, 0.560986, 0.353553, 0.612372), (0.270598, 0.653281, 0.707107, -1.73191e-16), (-0.653281, 0.270598, 2.16489e-16, 0.707107), (0.404756, 0.0532871, -0.790569, -0.456435), (-0.0532871, 0.404756, 0.456435, -0.790569), (0.323885, 0.248526, -0.456435, -0.790569), (-0.248526, 0.323885, 0.790569, -0.456435), (0.15623, 0.377172, -1.67692e-16, -0.912871), (-0.377172, 0.15623, 0.912871, -2.23589e-16), (0.905061, 0.119154, 0.353553, 0.204124), (-0.119154, 0.905061, -0.204124, 0.353553), (0.724229, 0.555721, 0.204124, 0.353553), (-0.555721, 0.724229, -0.353553, 0.204124), (0.349341, 0.843383, 2.4998e-17, 0.408248), (-0.843383, 0.349341, -0.408248, 4.9996e-17), (0.905061, 0.119154, -0.353553, -0.204124), (-0.119154, 0.905061, 0.204124, -0.353553), (0.724229, 0.555721, -0.204124, -0.353553), (-0.555721, 0.724229, 0.353553, -0.204124), (0.349341, 0.843383, -7.4994e-17, -0.408248), (-0.843383, 0.349341, 0.408248, -9.9992e-17), (0.404756, 0.0532871, 0.790569, 0.456435), (-0.0532871, 0.404756, -0.456435, 0.790569), (0.323885, 0.248526, 0.456435, 0.790569), (-0.248526, 0.323885, -0.790569, 0.456435), (0.15623, 0.377172, 5.58972e-17, 0.912871), (-0.377172, 0.15623, -0.912871, 1.11794e-16), (0.864171, 0.0566408, -0.498929, -0.0327016), (-0.0566408, 0.864171, 0.0327016, -0.498929), (0.720074, 0.481138, -0.415735, -0.277785), (-0.481138, 0.720074, 0.277785, -0.415735), (0.383033, 0.776715, -0.221144, -0.448436), (-0.776715, 0.383033, 0.448436, -0.221144), (0.498929, 0.0327016, 0.864171, 0.0566408), (-0.0327016, 0.498929, -0.0566408, 0.864171), (0.415735, 0.277785, 0.720074, 0.481138), (-0.277785, 0.415735, -0.481138, 0.720074), (0.221144, 0.448436, 0.383033, 0.776715), (-0.448436, 0.221144, -0.776715, 0.383033), (0.992648, 0.0650616, -0.0767343, 0.0672942), (-0.0650616, 0.992648, -0.0672942, -0.0767343), (0.827128, 0.552669, -0.100101, 0.0199113), (-0.552669, 0.827128, -0.0199113, -0.100101), (0.439979, 0.892189, -0.0966457, -0.0328068), (-0.892189, 0.439979, 0.0328068, -0.0966457), (0.644115, 0.0422175, 0.574227, -0.503584), (-0.0422175, 0.644115, 0.503584, 0.574227), (0.536711, 0.358619, 0.749087, -0.149003), (-0.358619, 0.536711, 0.149003, 0.749087), (0.285496, 0.578929, 0.72323, 0.245503), (-0.578929, 0.285496, -0.245503, 0.72323), (0.644115, 0.0422175, -0.574227, 0.503584), (-0.0422175, 0.644115, -0.503584, -0.574227), (0.536711, 0.358619, -0.749087, 0.149003), (-0.358619, 0.536711, -0.149003, -0.749087), (0.285496, 0.578929, -0.72323, -0.245503), (-0.578929, 0.285496, 0.245503, -0.72323), (0.992648, 0.0650616, 0.0767343, -0.0672942), (-0.0650616, 0.992648, 0.0672942, 0.0767343), (0.827128, 0.552669, 0.100101, -0.0199113), (-0.552669, 0.827128, 0.0199113, 0.100101), (0.439979, 0.892189, 0.0966457, 0.0328068), (-0.892189, 0.439979, -0.0328068, 0.0966457), (0.498929, 0.0327016, -0.0566408, 0.864171), (-0.0327016, 0.498929, -0.864171, -0.0566408), (0.415735, 0.277785, -0.481138, 0.720074), (-0.277785, 0.415735, -0.720074, -0.481138), (0.221144, 0.448436, -0.776715, 0.383033), (-0.448436, 0.221144, -0.383033, -0.776715), (0.864171, 0.0566408, 0.0327016, -0.498929), (-0.0566408, 0.864171, 0.498929, 0.0327016), (0.720074, 0.481138, 0.277785, -0.415735), (-0.481138, 0.720074, 0.415735, 0.277785), (0.383033, 0.776715, 0.448436, -0.221144), (-0.776715, 0.383033, 0.221144, 0.448436), (0.101844, 0.00667518, -0.655903, -0.747914), (-0.00667518, 0.101844, 0.747914, -0.655903), (0.0848615, 0.0567026, -0.194072, -0.975664), (-0.0567026, 0.0848615, 0.975664, -0.194072), (0.0451409, 0.0915367, 0.319761, -0.941985), (-0.0915367, 0.0451409, 0.941985, 0.319761), (0.762127, 0.0499525, 0.425606, 0.485311), (-0.0499525, 0.762127, -0.485311, 0.425606), (0.635045, 0.424324, 0.12593, 0.633094), (-0.424324, 0.635045, -0.633094, 0.12593), (0.337804, 0.684998, -0.207488, 0.611241), (-0.684998, 0.337804, -0.611241, -0.207488), (0.762127, 0.0499525, -0.425606, -0.485311), (-0.0499525, 0.762127, 0.485311, -0.425606), (0.635045, 0.424324, -0.12593, -0.633094), (-0.424324, 0.635045, 0.633094, -0.12593), (0.337804, 0.684998, 0.207488, -0.611241), (-0.684998, 0.337804, 0.611241, 0.207488), (0.101844, 0.00667518, 0.655903, 0.747914), (-0.00667518, 0.101844, -0.747914, 0.655903), (0.0848615, 0.0567026, 0.194072, 0.975664), (-0.0567026, 0.0848615, -0.975664, 0.194072), (0.0451409, 0.0915367, -0.319761, 0.941985), (-0.0915367, 0.0451409, -0.941985, -0.319761), (0.611241, 0.207488, -0.72323, -0.245503), (-0.207488, 0.611241, 0.245503, -0.72323), (0.425606, 0.485311, -0.503584, -0.574227), (-0.485311, 0.425606, 0.574227, -0.503584), (0.12593, 0.633094, -0.149003, -0.749087), (-0.633094, 0.12593, 0.749087, -0.149003), (0.72323, 0.245503, 0.611241, 0.207488), (-0.245503, 0.72323, -0.207488, 0.611241), (0.503584, 0.574227, 0.425606, 0.485311), (-0.574227, 0.503584, -0.485311, 0.425606), (0.149003, 0.749087, 0.12593, 0.633094), (-0.749087, 0.149003, -0.633094, 0.12593), (0.820066, 0.278375, -0.448436, 0.221144), (-0.278375, 0.820066, -0.221144, -0.448436), (0.57101, 0.651112, -0.498929, -0.0327016), (-0.651112, 0.57101, 0.0327016, -0.498929), (0.168953, 0.849385, -0.415735, -0.277785), (-0.849385, 0.168953, 0.277785, -0.415735), (0.289937, 0.0984203, 0.853797, -0.421046), (-0.0984203, 0.289937, 0.421046, 0.853797), (0.201883, 0.230203, 0.949933, 0.0622619), (-0.230203, 0.201883, -0.0622619, 0.949933), (0.059734, 0.300303, 0.791535, 0.528887), (-0.300303, 0.059734, -0.528887, 0.791535), (0.289937, 0.0984203, -0.853797, 0.421046), (-0.0984203, 0.289937, -0.421046, -0.853797), (0.201883, 0.230203, -0.949933, -0.0622619), (-0.230203, 0.201883, 0.0622619, -0.949933), (0.059734, 0.300303, -0.791535, -0.528887), (-0.300303, 0.059734, 0.528887, -0.791535), (0.820066, 0.278375, 0.448436, -0.221144), (-0.278375, 0.820066, 0.221144, 0.448436), (0.57101, 0.651112, 0.498929, 0.0327016), (-0.651112, 0.57101, -0.0327016, 0.498929), (0.168953, 0.849385, 0.415735, 0.277785), (-0.849385, 0.168953, -0.277785, 0.415735), (0.72323, 0.245503, -0.207488, 0.611241), (-0.245503, 0.72323, -0.611241, -0.207488), (0.503584, 0.574227, -0.485311, 0.425606), (-0.574227, 0.503584, -0.425606, -0.485311), (0.149003, 0.749087, -0.633094, 0.12593), (-0.749087, 0.149003, -0.12593, -0.633094), (0.611241, 0.207488, 0.245503, -0.72323), (-0.207488, 0.611241, 0.72323, 0.245503), (0.425606, 0.485311, 0.574227, -0.503584), (-0.485311, 0.425606, 0.503584, 0.574227), (0.12593, 0.633094, 0.749087, -0.149003), (-0.633094, 0.12593, 0.149003, 0.749087), (0.473465, 0.16072, -0.383033, -0.776715), (-0.16072, 0.473465, 0.776715, -0.383033), (0.329673, 0.37592, 0.0566408, -0.864171), (-0.37592, 0.329673, 0.864171, 0.0566408), (0.0975452, 0.490393, 0.481138, -0.720074), (-0.490393, 0.0975452, 0.720074, 0.481138), (0.901451, 0.306001, 0.135423, 0.27461), (-0.306001, 0.901451, -0.27461, 0.135423), (0.627679, 0.71573, -0.0200255, 0.305531), (-0.71573, 0.627679, -0.305531, -0.0200255), (0.18572, 0.93368, -0.170108, 0.254585), (-0.93368, 0.18572, -0.254585, -0.170108), (0.901451, 0.306001, -0.135423, -0.27461), (-0.306001, 0.901451, 0.27461, -0.135423), (0.627679, 0.71573, 0.0200255, -0.305531), (-0.71573, 0.627679, 0.305531, 0.0200255), (0.18572, 0.93368, 0.170108, -0.254585), (-0.93368, 0.18572, 0.254585, 0.170108), (0.473465, 0.16072, 0.383033, 0.776715), (-0.16072, 0.473465, -0.776715, 0.383033), (0.329673, 0.37592, -0.0566408, 0.864171), (-0.37592, 0.329673, -0.864171, -0.0566408), (0.0975452, 0.490393, -0.481138, 0.720074), (-0.490393, 0.0975452, -0.720074, -0.481138), (0.820066, 0.278375, -0.473465, -0.16072), (-0.278375, 0.820066, 0.16072, -0.473465), (0.57101, 0.651112, -0.329673, -0.37592), (-0.651112, 0.57101, 0.37592, -0.329673), (0.168953, 0.849385, -0.0975452, -0.490393), (-0.849385, 0.168953, 0.490393, -0.0975452), (0.473465, 0.16072, 0.820066, 0.278375), (-0.16072, 0.473465, -0.278375, 0.820066), (0.329673, 0.37592, 0.57101, 0.651112), (-0.37592, 0.329673, -0.651112, 0.57101), (0.0975452, 0.490393, 0.168953, 0.849385), (-0.490393, 0.0975452, -0.849385, 0.168953), (0.941985, 0.319761, -0.0915367, 0.0451409), (-0.319761, 0.941985, -0.0451409, -0.0915367), (0.655903, 0.747914, -0.101844, -0.00667518), (-0.747914, 0.655903, 0.00667518, -0.101844), (0.194072, 0.975664, -0.0848615, -0.0567026), (-0.975664, 0.194072, 0.0567026, -0.0848615), (0.611241, 0.207488, 0.684998, -0.337804), (-0.207488, 0.611241, 0.337804, 0.684998), (0.425606, 0.485311, 0.762127, 0.0499525), (-0.485311, 0.425606, -0.0499525, 0.762127), (0.12593, 0.633094, 0.635045, 0.424324), (-0.633094, 0.12593, -0.424324, 0.635045), (0.611241, 0.207488, -0.684998, 0.337804), (-0.207488, 0.611241, -0.337804, -0.684998), (0.425606, 0.485311, -0.762127, -0.0499525), (-0.485311, 0.425606, 0.0499525, -0.762127), (0.12593, 0.633094, -0.635045, -0.424324), (-0.633094, 0.12593, 0.424324, -0.635045), (0.941985, 0.319761, 0.0915367, -0.0451409), (-0.319761, 0.941985, 0.0451409, 0.0915367), (0.655903, 0.747914, 0.101844, 0.00667518), (-0.747914, 0.655903, -0.00667518, 0.101844), (0.194072, 0.975664, 0.0848615, 0.0567026), (-0.975664, 0.194072, -0.0567026, 0.0848615), (0.473465, 0.16072, -0.278375, 0.820066), (-0.16072, 0.473465, -0.820066, -0.278375), (0.329673, 0.37592, -0.651112, 0.57101), (-0.37592, 0.329673, -0.57101, -0.651112), (0.0975452, 0.490393, -0.849385, 0.168953), (-0.490393, 0.0975452, -0.168953, -0.849385), (0.820066, 0.278375, 0.16072, -0.473465), (-0.278375, 0.820066, 0.473465, 0.16072), (0.57101, 0.651112, 0.37592, -0.329673), (-0.651112, 0.57101, 0.329673, 0.37592), (0.168953, 0.849385, 0.490393, -0.0975452), (-0.849385, 0.168953, 0.0975452, 0.490393), (0.0966457, 0.0328068, -0.439979, -0.892189), (-0.0328068, 0.0966457, 0.892189, -0.439979), (0.0672942, 0.0767343, 0.0650616, -0.992648), (-0.0767343, 0.0672942, 0.992648, 0.0650616), (0.0199113, 0.100101, 0.552669, -0.827128), (-0.100101, 0.0199113, 0.827128, 0.552669), (0.72323, 0.245503, 0.285496, 0.578929), (-0.245503, 0.72323, -0.578929, 0.285496), (0.503584, 0.574227, -0.0422175, 0.644115), (-0.574227, 0.503584, -0.644115, -0.0422175), (0.149003, 0.749087, -0.358619, 0.536711), (-0.749087, 0.149003, -0.536711, -0.358619), (0.72323, 0.245503, -0.285496, -0.578929), (-0.245503, 0.72323, 0.578929, -0.285496), (0.503584, 0.574227, 0.0422175, -0.644115), (-0.574227, 0.503584, 0.644115, 0.0422175), (0.149003, 0.749087, 0.358619, -0.536711), (-0.749087, 0.149003, 0.536711, 0.358619), (0.0966457, 0.0328068, 0.439979, 0.892189), (-0.0328068, 0.0966457, -0.892189, 0.439979), (0.0672942, 0.0767343, -0.0650616, 0.992648), (-0.0767343, 0.0672942, -0.992648, -0.0650616), (0.0199113, 0.100101, -0.552669, 0.827128), (-0.100101, 0.0199113, -0.827128, -0.552669), (0.644115, 0.0422175, -0.762127, -0.0499525), (-0.0422175, 0.644115, 0.0499525, -0.762127), (0.536711, 0.358619, -0.635045, -0.424324), (-0.358619, 0.536711, 0.424324, -0.635045), (0.285496, 0.578929, -0.337804, -0.684998), (-0.578929, 0.285496, 0.684998, -0.337804), (0.762127, 0.0499525, 0.644115, 0.0422175), (-0.0499525, 0.762127, -0.0422175, 0.644115), (0.635045, 0.424324, 0.536711, 0.358619), (-0.424324, 0.635045, -0.358619, 0.536711), (0.337804, 0.684998, 0.285496, 0.578929), (-0.684998, 0.337804, -0.578929, 0.285496), (0.864171, 0.0566408, -0.37592, 0.329673), (-0.0566408, 0.864171, -0.329673, -0.37592), (0.720074, 0.481138, -0.490393, 0.0975452), (-0.481138, 0.720074, -0.0975452, -0.490393), (0.383033, 0.776715, -0.473465, -0.16072), (-0.776715, 0.383033, 0.16072, -0.473465), (0.305531, 0.0200255, 0.71573, -0.627679), (-0.0200255, 0.305531, 0.627679, 0.71573), (0.254585, 0.170108, 0.93368, -0.18572), (-0.170108, 0.254585, 0.18572, 0.93368), (0.135423, 0.27461, 0.901451, 0.306001), (-0.27461, 0.135423, -0.306001, 0.901451), (0.305531, 0.0200255, -0.71573, 0.627679), (-0.0200255, 0.305531, -0.627679, -0.71573), (0.254585, 0.170108, -0.93368, 0.18572), (-0.170108, 0.254585, -0.18572, -0.93368), (0.135423, 0.27461, -0.901451, -0.306001), (-0.27461, 0.135423, 0.306001, -0.901451), (0.864171, 0.0566408, 0.37592, -0.329673), (-0.0566408, 0.864171, 0.329673, 0.37592), (0.720074, 0.481138, 0.490393, -0.0975452), (-0.481138, 0.720074, 0.0975452, 0.490393), (0.383033, 0.776715, 0.473465, 0.16072), (-0.776715, 0.383033, -0.16072, 0.473465), (0.762127, 0.0499525, -0.0422175, 0.644115), (-0.0499525, 0.762127, -0.644115, -0.0422175), (0.635045, 0.424324, -0.358619, 0.536711), (-0.424324, 0.635045, -0.536711, -0.358619), (0.337804, 0.684998, -0.578929, 0.285496), (-0.684998, 0.337804, -0.285496, -0.578929), (0.644115, 0.0422175, 0.0499525, -0.762127), (-0.0422175, 0.644115, 0.762127, 0.0499525), (0.536711, 0.358619, 0.424324, -0.635045), (-0.358619, 0.536711, 0.635045, 0.424324), (0.285496, 0.578929, 0.684998, -0.337804), (-0.578929, 0.285496, 0.337804, 0.684998), (0.498929, 0.0327016, -0.57101, -0.651112), (-0.0327016, 0.498929, 0.651112, -0.57101), (0.415735, 0.277785, -0.168953, -0.849385), (-0.277785, 0.415735, 0.849385, -0.168953), (0.221144, 0.448436, 0.278375, -0.820066), (-0.448436, 0.221144, 0.820066, 0.278375), (0.949933, 0.0622619, 0.201883, 0.230203), (-0.0622619, 0.949933, -0.230203, 0.201883), (0.791535, 0.528887, 0.059734, 0.300303), (-0.528887, 0.791535, -0.300303, 0.059734), (0.421046, 0.853797, -0.0984203, 0.289937), (-0.853797, 0.421046, -0.289937, -0.0984203), (0.949933, 0.0622619, -0.201883, -0.230203), (-0.0622619, 0.949933, 0.230203, -0.201883), (0.791535, 0.528887, -0.059734, -0.300303), (-0.528887, 0.791535, 0.300303, -0.059734), (0.421046, 0.853797, 0.0984203, -0.289937), (-0.853797, 0.421046, 0.289937, 0.0984203), (0.498929, 0.0327016, 0.57101, 0.651112), (-0.0327016, 0.498929, -0.651112, 0.57101), (0.415735, 0.277785, 0.168953, 0.849385), (-0.277785, 0.415735, -0.849385, 0.168953), (0.221144, 0.448436, -0.278375, 0.820066), (-0.448436, 0.221144, -0.820066, -0.278375), (0.762127, 0.0499525, -0.611241, 0.207488), (-0.0499525, 0.762127, -0.207488, -0.611241), (0.635045, 0.424324, -0.633094, -0.12593), (-0.424324, 0.635045, 0.12593, -0.633094), (0.337804, 0.684998, -0.485311, -0.425606), (-0.684998, 0.337804, 0.425606, -0.485311), (0.644115, 0.0422175, 0.72323, -0.245503), (-0.0422175, 0.644115, 0.245503, 0.72323), (0.536711, 0.358619, 0.749087, 0.149003), (-0.358619, 0.536711, -0.149003, 0.749087), (0.285496, 0.578929, 0.574227, 0.503584), (-0.578929, 0.285496, -0.503584, 0.574227), (0.949933, 0.0622619, -0.300303, 0.059734), (-0.0622619, 0.949933, -0.059734, -0.300303), (0.791535, 0.528887, -0.289937, -0.0984203), (-0.528887, 0.791535, 0.0984203, -0.289937), (0.421046, 0.853797, -0.201883, -0.230203), (-0.853797, 0.421046, 0.230203, -0.201883), (0.498929, 0.0327016, 0.820066, -0.278375), (-0.0327016, 0.498929, 0.278375, 0.820066), (0.415735, 0.277785, 0.849385, 0.168953), (-0.277785, 0.415735, -0.168953, 0.849385), (0.221144, 0.448436, 0.651112, 0.57101), (-0.448436, 0.221144, -0.57101, 0.651112), (0.498929, 0.0327016, -0.820066, 0.278375), (-0.0327016, 0.498929, -0.278375, -0.820066), (0.415735, 0.277785, -0.849385, -0.168953), (-0.277785, 0.415735, 0.168953, -0.849385), (0.221144, 0.448436, -0.651112, -0.57101), (-0.448436, 0.221144, 0.57101, -0.651112), (0.949933, 0.0622619, 0.300303, -0.059734), (-0.0622619, 0.949933, 0.059734, 0.300303), (0.791535, 0.528887, 0.289937, 0.0984203), (-0.528887, 0.791535, -0.0984203, 0.289937), (0.421046, 0.853797, 0.201883, 0.230203), (-0.853797, 0.421046, -0.230203, 0.201883), (0.644115, 0.0422175, 0.245503, 0.72323), (-0.0422175, 0.644115, -0.72323, 0.245503), (0.536711, 0.358619, -0.149003, 0.749087), (-0.358619, 0.536711, -0.749087, -0.149003), (0.285496, 0.578929, -0.503584, 0.574227), (-0.578929, 0.285496, -0.574227, -0.503584), (0.762127, 0.0499525, -0.207488, -0.611241), (-0.0499525, 0.762127, 0.611241, -0.207488), (0.635045, 0.424324, 0.12593, -0.633094), (-0.424324, 0.635045, 0.633094, 0.12593), (0.337804, 0.684998, 0.425606, -0.485311), (-0.684998, 0.337804, 0.485311, 0.425606), (0.305531, 0.0200255, -0.18572, -0.93368), (-0.0200255, 0.305531, 0.93368, -0.18572), (0.254585, 0.170108, 0.306001, -0.901451), (-0.170108, 0.254585, 0.901451, 0.306001), (0.135423, 0.27461, 0.71573, -0.627679), (-0.27461, 0.135423, 0.627679, 0.71573), (0.864171, 0.0566408, 0.16072, 0.473465), (-0.0566408, 0.864171, -0.473465, 0.16072), (0.720074, 0.481138, -0.0975452, 0.490393), (-0.481138, 0.720074, -0.490393, -0.0975452), (0.383033, 0.776715, -0.329673, 0.37592), (-0.776715, 0.383033, -0.37592, -0.329673), (0.864171, 0.0566408, -0.16072, -0.473465), (-0.0566408, 0.864171, 0.473465, -0.16072), (0.720074, 0.481138, 0.0975452, -0.490393), (-0.481138, 0.720074, 0.490393, 0.0975452), (0.383033, 0.776715, 0.329673, -0.37592), (-0.776715, 0.383033, 0.37592, 0.329673), (0.305531, 0.0200255, 0.18572, 0.93368), (-0.0200255, 0.305531, -0.93368, 0.18572), (0.254585, 0.170108, -0.306001, 0.901451), (-0.170108, 0.254585, -0.901451, -0.306001), (0.135423, 0.27461, -0.71573, 0.627679), (-0.27461, 0.135423, -0.627679, -0.71573), (0.72323, 0.245503, -0.485311, -0.425606), (-0.245503, 0.72323, 0.425606, -0.485311), (0.503584, 0.574227, -0.207488, -0.611241), (-0.574227, 0.503584, 0.611241, -0.207488), (0.149003, 0.749087, 0.12593, -0.633094), (-0.749087, 0.149003, 0.633094, 0.12593), (0.611241, 0.207488, 0.574227, 0.503584), (-0.207488, 0.611241, -0.503584, 0.574227), (0.425606, 0.485311, 0.245503, 0.72323), (-0.485311, 0.425606, -0.72323, 0.245503), (0.12593, 0.633094, -0.149003, 0.749087), (-0.633094, 0.12593, -0.749087, -0.149003), (0.901451, 0.306001, -0.170108, 0.254585), (-0.306001, 0.901451, -0.254585, -0.170108), (0.627679, 0.71573, -0.27461, 0.135423), (-0.71573, 0.627679, -0.135423, -0.27461), (0.18572, 0.93368, -0.305531, -0.0200255), (-0.93368, 0.18572, 0.0200255, -0.305531), (0.473465, 0.16072, 0.57101, -0.651112), (-0.16072, 0.473465, 0.651112, 0.57101), (0.329673, 0.37592, 0.820066, -0.278375), (-0.37592, 0.329673, 0.278375, 0.820066), (0.0975452, 0.490393, 0.849385, 0.168953), (-0.490393, 0.0975452, -0.168953, 0.849385), (0.473465, 0.16072, -0.57101, 0.651112), (-0.16072, 0.473465, -0.651112, -0.57101), (0.329673, 0.37592, -0.820066, 0.278375), (-0.37592, 0.329673, -0.278375, -0.820066), (0.0975452, 0.490393, -0.849385, -0.168953), (-0.490393, 0.0975452, 0.168953, -0.849385), (0.901451, 0.306001, 0.170108, -0.254585), (-0.306001, 0.901451, 0.254585, 0.170108), (0.627679, 0.71573, 0.27461, -0.135423), (-0.71573, 0.627679, 0.135423, 0.27461), (0.18572, 0.93368, 0.305531, 0.0200255), (-0.93368, 0.18572, -0.0200255, 0.305531), (0.611241, 0.207488, -0.503584, 0.574227), (-0.207488, 0.611241, -0.574227, -0.503584), (0.425606, 0.485311, -0.72323, 0.245503), (-0.485311, 0.425606, -0.245503, -0.72323), (0.12593, 0.633094, -0.749087, -0.149003), (-0.633094, 0.12593, 0.149003, -0.749087), (0.72323, 0.245503, 0.425606, -0.485311), (-0.245503, 0.72323, 0.485311, 0.425606), (0.503584, 0.574227, 0.611241, -0.207488), (-0.574227, 0.503584, 0.207488, 0.611241), (0.149003, 0.749087, 0.633094, 0.12593), (-0.749087, 0.149003, -0.12593, 0.633094), (0.289937, 0.0984203, -0.791535, -0.528887), (-0.0984203, 0.289937, 0.528887, -0.791535), (0.201883, 0.230203, -0.421046, -0.853797), (-0.230203, 0.201883, 0.853797, -0.421046), (0.059734, 0.300303, 0.0622619, -0.949933), (-0.300303, 0.059734, 0.949933, 0.0622619), (0.820066, 0.278375, 0.37592, 0.329673), (-0.278375, 0.820066, -0.329673, 0.37592), (0.57101, 0.651112, 0.16072, 0.473465), (-0.651112, 0.57101, -0.473465, 0.16072), (0.168953, 0.849385, -0.0975452, 0.490393), (-0.849385, 0.168953, -0.490393, -0.0975452), (0.820066, 0.278375, -0.37592, -0.329673), (-0.278375, 0.820066, 0.329673, -0.37592), (0.57101, 0.651112, -0.16072, -0.473465), (-0.651112, 0.57101, 0.473465, -0.16072), (0.168953, 0.849385, 0.0975452, -0.490393), (-0.849385, 0.168953, 0.490393, 0.0975452), (0.289937, 0.0984203, 0.791535, 0.528887), (-0.0984203, 0.289937, -0.528887, 0.791535), (0.201883, 0.230203, 0.421046, 0.853797), (-0.230203, 0.201883, -0.853797, 0.421046), (0.059734, 0.300303, -0.0622619, 0.949933), (-0.300303, 0.059734, -0.949933, -0.0622619), (0.72323, 0.245503, -0.644115, 0.0422175), (-0.245503, 0.72323, -0.0422175, -0.644115), (0.503584, 0.574227, -0.578929, -0.285496), (-0.574227, 0.503584, 0.285496, -0.578929), (0.149003, 0.749087, -0.358619, -0.536711), (-0.749087, 0.149003, 0.536711, -0.358619), (0.611241, 0.207488, 0.762127, -0.0499525), (-0.207488, 0.611241, 0.0499525, 0.762127), (0.425606, 0.485311, 0.684998, 0.337804), (-0.485311, 0.425606, -0.337804, 0.684998), (0.12593, 0.633094, 0.424324, 0.635045), (-0.633094, 0.12593, -0.635045, 0.424324), (0.901451, 0.306001, -0.305531, -0.0200255), (-0.306001, 0.901451, 0.0200255, -0.305531), (0.627679, 0.71573, -0.254585, -0.170108), (-0.71573, 0.627679, 0.170108, -0.254585), (0.18572, 0.93368, -0.135423, -0.27461), (-0.93368, 0.18572, 0.27461, -0.135423), (0.473465, 0.16072, 0.864171, -0.0566408), (-0.16072, 0.473465, 0.0566408, 0.864171), (0.329673, 0.37592, 0.776715, 0.383033), (-0.37592, 0.329673, -0.383033, 0.776715), (0.0975452, 0.490393, 0.481138, 0.720074), (-0.490393, 0.0975452, -0.720074, 0.481138), (0.473465, 0.16072, -0.864171, 0.0566408), (-0.16072, 0.473465, -0.0566408, -0.864171), (0.329673, 0.37592, -0.776715, -0.383033), (-0.37592, 0.329673, 0.383033, -0.776715), (0.0975452, 0.490393, -0.481138, -0.720074), (-0.490393, 0.0975452, 0.720074, -0.481138), (0.901451, 0.306001, 0.305531, 0.0200255), (-0.306001, 0.901451, -0.0200255, 0.305531), (0.627679, 0.71573, 0.254585, 0.170108), (-0.71573, 0.627679, -0.170108, 0.254585), (0.18572, 0.93368, 0.135423, 0.27461), (-0.93368, 0.18572, -0.27461, 0.135423), (0.611241, 0.207488, 0.0499525, 0.762127), (-0.207488, 0.611241, -0.762127, 0.0499525), (0.425606, 0.485311, -0.337804, 0.684998), (-0.485311, 0.425606, -0.684998, -0.337804), (0.12593, 0.633094, -0.635045, 0.424324), (-0.633094, 0.12593, -0.424324, -0.635045), (0.72323, 0.245503, -0.0422175, -0.644115), (-0.245503, 0.72323, 0.644115, -0.0422175), (0.503584, 0.574227, 0.285496, -0.578929), (-0.574227, 0.503584, 0.578929, 0.285496), (0.149003, 0.749087, 0.536711, -0.358619), (-0.749087, 0.149003, 0.358619, 0.536711), (0.289937, 0.0984203, 0.0622619, -0.949933), (-0.0984203, 0.289937, 0.949933, 0.0622619), (0.201883, 0.230203, 0.528887, -0.791535), (-0.230203, 0.201883, 0.791535, 0.528887), (0.059734, 0.300303, 0.853797, -0.421046), (-0.300303, 0.059734, 0.421046, 0.853797), (0.820066, 0.278375, 0.0327016, 0.498929), (-0.278375, 0.820066, -0.498929, 0.0327016), (0.57101, 0.651112, -0.221144, 0.448436), (-0.651112, 0.57101, -0.448436, -0.221144), (0.168953, 0.849385, -0.415735, 0.277785), (-0.849385, 0.168953, -0.277785, -0.415735), (0.820066, 0.278375, -0.0327016, -0.498929), (-0.278375, 0.820066, 0.498929, -0.0327016), (0.57101, 0.651112, 0.221144, -0.448436), (-0.651112, 0.57101, 0.448436, 0.221144), (0.168953, 0.849385, 0.415735, -0.277785), (-0.849385, 0.168953, 0.277785, 0.415735), (0.289937, 0.0984203, -0.0622619, 0.949933), (-0.0984203, 0.289937, -0.949933, -0.0622619), (0.201883, 0.230203, -0.528887, 0.791535), (-0.230203, 0.201883, -0.791535, -0.528887), (0.059734, 0.300303, -0.853797, 0.421046), (-0.300303, 0.059734, -0.421046, -0.853797), (0.762127, 0.0499525, -0.578929, -0.285496), (-0.0499525, 0.762127, 0.285496, -0.578929), (0.635045, 0.424324, -0.358619, -0.536711), (-0.424324, 0.635045, 0.536711, -0.358619), (0.337804, 0.684998, -0.0422175, -0.644115), (-0.684998, 0.337804, 0.644115, -0.0422175), (0.644115, 0.0422175, 0.684998, 0.337804), (-0.0422175, 0.644115, -0.337804, 0.684998), (0.536711, 0.358619, 0.424324, 0.635045), (-0.358619, 0.536711, -0.635045, 0.424324), (0.285496, 0.578929, 0.0499525, 0.762127), (-0.578929, 0.285496, -0.762127, 0.0499525), (0.949933, 0.0622619, -0.0984203, 0.289937), (-0.0622619, 0.949933, -0.289937, -0.0984203), (0.791535, 0.528887, -0.230203, 0.201883), (-0.528887, 0.791535, -0.201883, -0.230203), (0.421046, 0.853797, -0.300303, 0.059734), (-0.853797, 0.421046, -0.059734, -0.300303), (0.498929, 0.0327016, 0.383033, -0.776715), (-0.0327016, 0.498929, 0.776715, 0.383033), (0.415735, 0.277785, 0.720074, -0.481138), (-0.277785, 0.415735, 0.481138, 0.720074), (0.221144, 0.448436, 0.864171, -0.0566408), (-0.448436, 0.221144, 0.0566408, 0.864171), (0.498929, 0.0327016, -0.383033, 0.776715), (-0.0327016, 0.498929, -0.776715, -0.383033), (0.415735, 0.277785, -0.720074, 0.481138), (-0.277785, 0.415735, -0.481138, -0.720074), (0.221144, 0.448436, -0.864171, 0.0566408), (-0.448436, 0.221144, -0.0566408, -0.864171), (0.949933, 0.0622619, 0.0984203, -0.289937), (-0.0622619, 0.949933, 0.289937, 0.0984203), (0.791535, 0.528887, 0.230203, -0.201883), (-0.528887, 0.791535, 0.201883, 0.230203), (0.421046, 0.853797, 0.300303, -0.059734), (-0.853797, 0.421046, 0.059734, 0.300303), (0.644115, 0.0422175, -0.337804, 0.684998), (-0.0422175, 0.644115, -0.684998, -0.337804), (0.536711, 0.358619, -0.635045, 0.424324), (-0.358619, 0.536711, -0.424324, -0.635045), (0.285496, 0.578929, -0.762127, 0.0499525), (-0.578929, 0.285496, -0.0499525, -0.762127), (0.762127, 0.0499525, 0.285496, -0.578929), (-0.0499525, 0.762127, 0.578929, 0.285496), (0.635045, 0.424324, 0.536711, -0.358619), (-0.424324, 0.635045, 0.358619, 0.536711), (0.337804, 0.684998, 0.644115, -0.0422175), (-0.684998, 0.337804, 0.0422175, 0.644115), (0.305531, 0.0200255, -0.901451, -0.306001), (-0.0200255, 0.305531, 0.306001, -0.901451), (0.254585, 0.170108, -0.627679, -0.71573), (-0.170108, 0.254585, 0.71573, -0.627679), (0.135423, 0.27461, -0.18572, -0.93368), (-0.27461, 0.135423, 0.93368, -0.18572), (0.864171, 0.0566408, 0.448436, 0.221144), (-0.0566408, 0.864171, -0.221144, 0.448436), (0.720074, 0.481138, 0.277785, 0.415735), (-0.481138, 0.720074, -0.415735, 0.277785), (0.383033, 0.776715, 0.0327016, 0.498929), (-0.776715, 0.383033, -0.498929, 0.0327016), (0.864171, 0.0566408, -0.448436, -0.221144), (-0.0566408, 0.864171, 0.221144, -0.448436), (0.720074, 0.481138, -0.277785, -0.415735), (-0.481138, 0.720074, 0.415735, -0.277785), (0.383033, 0.776715, -0.0327016, -0.498929), (-0.776715, 0.383033, 0.498929, -0.0327016), (0.305531, 0.0200255, 0.901451, 0.306001), (-0.0200255, 0.305531, -0.306001, 0.901451), (0.254585, 0.170108, 0.627679, 0.71573), (-0.170108, 0.254585, -0.71573, 0.627679), (0.135423, 0.27461, 0.18572, 0.93368), (-0.27461, 0.135423, -0.93368, 0.18572), (0.749087, 0.149003, -0.633094, -0.12593), (-0.149003, 0.749087, 0.12593, -0.633094), (0.574227, 0.503584, -0.485311, -0.425606), (-0.503584, 0.574227, 0.425606, -0.485311), (0.245503, 0.72323, -0.207488, -0.611241), (-0.72323, 0.245503, 0.611241, -0.207488), (0.633094, 0.12593, 0.749087, 0.149003), (-0.12593, 0.633094, -0.149003, 0.749087), (0.485311, 0.425606, 0.574227, 0.503584), (-0.425606, 0.485311, -0.503584, 0.574227), (0.207488, 0.611241, 0.245503, 0.72323), (-0.611241, 0.207488, -0.72323, 0.245503), (0.93368, 0.18572, -0.254585, 0.170108), (-0.18572, 0.93368, -0.170108, -0.254585), (0.71573, 0.627679, -0.305531, 0.0200255), (-0.627679, 0.71573, -0.0200255, -0.305531), (0.306001, 0.901451, -0.27461, -0.135423), (-0.901451, 0.306001, 0.135423, -0.27461), (0.490393, 0.0975452, 0.720074, -0.481138), (-0.0975452, 0.490393, 0.481138, 0.720074), (0.37592, 0.329673, 0.864171, -0.0566408), (-0.329673, 0.37592, 0.0566408, 0.864171), (0.16072, 0.473465, 0.776715, 0.383033), (-0.473465, 0.16072, -0.383033, 0.776715), (0.490393, 0.0975452, -0.720074, 0.481138), (-0.0975452, 0.490393, -0.481138, -0.720074), (0.37592, 0.329673, -0.864171, 0.0566408), (-0.329673, 0.37592, -0.0566408, -0.864171), (0.16072, 0.473465, -0.776715, -0.383033), (-0.473465, 0.16072, 0.383033, -0.776715), (0.93368, 0.18572, 0.254585, -0.170108), (-0.18572, 0.93368, 0.170108, 0.254585), (0.71573, 0.627679, 0.305531, -0.0200255), (-0.627679, 0.71573, 0.0200255, 0.305531), (0.306001, 0.901451, 0.27461, 0.135423), (-0.901451, 0.306001, -0.135423, 0.27461), (0.633094, 0.12593, -0.149003, 0.749087), (-0.12593, 0.633094, -0.749087, -0.149003), (0.485311, 0.425606, -0.503584, 0.574227), (-0.425606, 0.485311, -0.574227, -0.503584), (0.207488, 0.611241, -0.72323, 0.245503), (-0.611241, 0.207488, -0.245503, -0.72323), (0.749087, 0.149003, 0.12593, -0.633094), (-0.149003, 0.749087, 0.633094, 0.12593), (0.574227, 0.503584, 0.425606, -0.485311), (-0.503584, 0.574227, 0.485311, 0.425606), (0.245503, 0.72323, 0.611241, -0.207488), (-0.72323, 0.245503, 0.207488, 0.611241), (0.300303, 0.059734, -0.528887, -0.791535), (-0.059734, 0.300303, 0.791535, -0.528887), (0.230203, 0.201883, -0.0622619, -0.949933), (-0.201883, 0.230203, 0.949933, -0.0622619), (0.0984203, 0.289937, 0.421046, -0.853797), (-0.289937, 0.0984203, 0.853797, 0.421046), (0.849385, 0.168953, 0.277785, 0.415735), (-0.168953, 0.849385, -0.415735, 0.277785), (0.651112, 0.57101, 0.0327016, 0.498929), (-0.57101, 0.651112, -0.498929, 0.0327016), (0.278375, 0.820066, -0.221144, 0.448436), (-0.820066, 0.278375, -0.448436, -0.221144), (0.849385, 0.168953, -0.277785, -0.415735), (-0.168953, 0.849385, 0.415735, -0.277785), (0.651112, 0.57101, -0.0327016, -0.498929), (-0.57101, 0.651112, 0.498929, -0.0327016), (0.278375, 0.820066, 0.221144, -0.448436), (-0.820066, 0.278375, 0.448436, 0.221144), (0.300303, 0.059734, 0.528887, 0.791535), (-0.059734, 0.300303, -0.791535, 0.528887), (0.230203, 0.201883, 0.0622619, 0.949933), (-0.201883, 0.230203, -0.949933, 0.0622619), (0.0984203, 0.289937, -0.421046, 0.853797), (-0.289937, 0.0984203, -0.853797, -0.421046), (0.448436, 0.221144, -0.776715, -0.383033), (-0.221144, 0.448436, 0.383033, -0.776715), (0.277785, 0.415735, -0.481138, -0.720074), (-0.415735, 0.277785, 0.720074, -0.481138), (0.0327016, 0.498929, -0.0566408, -0.864171), (-0.498929, 0.0327016, 0.864171, -0.0566408), (0.776715, 0.383033, 0.448436, 0.221144), (-0.383033, 0.776715, -0.221144, 0.448436), (0.481138, 0.720074, 0.277785, 0.415735), (-0.720074, 0.481138, -0.415735, 0.277785), (0.0566408, 0.864171, 0.0327016, 0.498929), (-0.864171, 0.0566408, -0.498929, 0.0327016), (0.684998, 0.337804, -0.611241, 0.207488), (-0.337804, 0.684998, -0.207488, -0.611241), (0.424324, 0.635045, -0.633094, -0.12593), (-0.635045, 0.424324, 0.12593, -0.633094), (0.0499525, 0.762127, -0.485311, -0.425606), (-0.762127, 0.0499525, 0.425606, -0.485311), (0.0915367, 0.0451409, 0.941985, -0.319761), (-0.0451409, 0.0915367, 0.319761, 0.941985), (0.0567026, 0.0848615, 0.975664, 0.194072), (-0.0848615, 0.0567026, -0.194072, 0.975664), (0.00667518, 0.101844, 0.747914, 0.655903), (-0.101844, 0.00667518, -0.655903, 0.747914), (0.0915367, 0.0451409, -0.941985, 0.319761), (-0.0451409, 0.0915367, -0.319761, -0.941985), (0.0567026, 0.0848615, -0.975664, -0.194072), (-0.0848615, 0.0567026, 0.194072, -0.975664), (0.00667518, 0.101844, -0.747914, -0.655903), (-0.101844, 0.00667518, 0.655903, -0.747914), (0.684998, 0.337804, 0.611241, -0.207488), (-0.337804, 0.684998, 0.207488, 0.611241), (0.424324, 0.635045, 0.633094, 0.12593), (-0.635045, 0.424324, -0.12593, 0.633094), (0.0499525, 0.762127, 0.485311, 0.425606), (-0.762127, 0.0499525, -0.425606, 0.485311), (0.776715, 0.383033, -0.221144, 0.448436), (-0.383033, 0.776715, -0.448436, -0.221144), (0.481138, 0.720074, -0.415735, 0.277785), (-0.720074, 0.481138, -0.277785, -0.415735), (0.0566408, 0.864171, -0.498929, 0.0327016), (-0.864171, 0.0566408, -0.0327016, -0.498929), (0.448436, 0.221144, 0.383033, -0.776715), (-0.221144, 0.448436, 0.776715, 0.383033), (0.277785, 0.415735, 0.720074, -0.481138), (-0.415735, 0.277785, 0.481138, 0.720074), (0.0327016, 0.498929, 0.864171, -0.0566408), (-0.498929, 0.0327016, 0.0566408, 0.864171), (0.578929, 0.285496, -0.245503, -0.72323), (-0.285496, 0.578929, 0.72323, -0.245503), (0.358619, 0.536711, 0.149003, -0.749087), (-0.536711, 0.358619, 0.749087, 0.149003), (0.0422175, 0.644115, 0.503584, -0.574227), (-0.644115, 0.0422175, 0.574227, 0.503584), (0.892189, 0.439979, 0.0328068, 0.0966457), (-0.439979, 0.892189, -0.0966457, 0.0328068), (0.552669, 0.827128, -0.0199113, 0.100101), (-0.827128, 0.552669, -0.100101, -0.0199113), (0.0650616, 0.992648, -0.0672942, 0.0767343), (-0.992648, 0.0650616, -0.0767343, -0.0672942), (0.892189, 0.439979, -0.0328068, -0.0966457), (-0.439979, 0.892189, 0.0966457, -0.0328068), (0.552669, 0.827128, 0.0199113, -0.100101), (-0.827128, 0.552669, 0.100101, 0.0199113), (0.0650616, 0.992648, 0.0672942, -0.0767343), (-0.992648, 0.0650616, 0.0767343, 0.0672942), (0.578929, 0.285496, 0.245503, 0.72323), (-0.285496, 0.578929, -0.72323, 0.245503), (0.358619, 0.536711, -0.149003, 0.749087), (-0.536711, 0.358619, -0.749087, -0.149003), (0.0422175, 0.644115, -0.503584, 0.574227), (-0.644115, 0.0422175, -0.574227, -0.503584), (0.684998, 0.337804, -0.578929, -0.285496), (-0.337804, 0.684998, 0.285496, -0.578929), (0.424324, 0.635045, -0.358619, -0.536711), (-0.635045, 0.424324, 0.536711, -0.358619), (0.0499525, 0.762127, -0.0422175, -0.644115), (-0.762127, 0.0499525, 0.644115, -0.0422175), (0.578929, 0.285496, 0.684998, 0.337804), (-0.285496, 0.578929, -0.337804, 0.684998), (0.358619, 0.536711, 0.424324, 0.635045), (-0.536711, 0.358619, -0.635045, 0.424324), (0.0422175, 0.644115, 0.0499525, 0.762127), (-0.644115, 0.0422175, -0.762127, 0.0499525), (0.853797, 0.421046, -0.289937, 0.0984203), (-0.421046, 0.853797, -0.0984203, -0.289937), (0.528887, 0.791535, -0.300303, -0.059734), (-0.791535, 0.528887, 0.059734, -0.300303), (0.0622619, 0.949933, -0.230203, -0.201883), (-0.949933, 0.0622619, 0.201883, -0.230203), (0.448436, 0.221144, 0.820066, -0.278375), (-0.221144, 0.448436, 0.278375, 0.820066), (0.277785, 0.415735, 0.849385, 0.168953), (-0.415735, 0.277785, -0.168953, 0.849385), (0.0327016, 0.498929, 0.651112, 0.57101), (-0.498929, 0.0327016, -0.57101, 0.651112), (0.448436, 0.221144, -0.820066, 0.278375), (-0.221144, 0.448436, -0.278375, -0.820066), (0.277785, 0.415735, -0.849385, -0.168953), (-0.415735, 0.277785, 0.168953, -0.849385), (0.0327016, 0.498929, -0.651112, -0.57101), (-0.498929, 0.0327016, 0.57101, -0.651112), (0.853797, 0.421046, 0.289937, -0.0984203), (-0.421046, 0.853797, 0.0984203, 0.289937), (0.528887, 0.791535, 0.300303, 0.059734), (-0.791535, 0.528887, -0.059734, 0.300303), (0.0622619, 0.949933, 0.230203, 0.201883), (-0.949933, 0.0622619, -0.201883, 0.230203), (0.578929, 0.285496, -0.337804, 0.684998), (-0.285496, 0.578929, -0.684998, -0.337804), (0.358619, 0.536711, -0.635045, 0.424324), (-0.536711, 0.358619, -0.424324, -0.635045), (0.0422175, 0.644115, -0.762127, 0.0499525), (-0.644115, 0.0422175, -0.0499525, -0.762127), (0.684998, 0.337804, 0.285496, -0.578929), (-0.337804, 0.684998, 0.578929, 0.285496), (0.424324, 0.635045, 0.536711, -0.358619), (-0.635045, 0.424324, 0.358619, 0.536711), (0.0499525, 0.762127, 0.644115, -0.0422175), (-0.762127, 0.0499525, 0.0422175, 0.644115), (0.27461, 0.135423, -0.306001, -0.901451), (-0.135423, 0.27461, 0.901451, -0.306001), (0.170108, 0.254585, 0.18572, -0.93368), (-0.254585, 0.170108, 0.93368, 0.18572), (0.0200255, 0.305531, 0.627679, -0.71573), (-0.305531, 0.0200255, 0.71573, 0.627679), (0.776715, 0.383033, 0.16072, 0.473465), (-0.383033, 0.776715, -0.473465, 0.16072), (0.481138, 0.720074, -0.0975452, 0.490393), (-0.720074, 0.481138, -0.490393, -0.0975452), (0.0566408, 0.864171, -0.329673, 0.37592), (-0.864171, 0.0566408, -0.37592, -0.329673), (0.776715, 0.383033, -0.16072, -0.473465), (-0.383033, 0.776715, 0.473465, -0.16072), (0.481138, 0.720074, 0.0975452, -0.490393), (-0.720074, 0.481138, 0.490393, 0.0975452), (0.0566408, 0.864171, 0.329673, -0.37592), (-0.864171, 0.0566408, 0.37592, 0.329673), (0.27461, 0.135423, 0.306001, 0.901451), (-0.135423, 0.27461, -0.901451, 0.306001), (0.170108, 0.254585, -0.18572, 0.93368), (-0.254585, 0.170108, -0.93368, -0.18572), (0.0200255, 0.305531, -0.627679, 0.71573), (-0.305531, 0.0200255, -0.71573, -0.627679), (0.490393, 0.0975452, -0.849385, -0.168953), (-0.0975452, 0.490393, 0.168953, -0.849385), (0.37592, 0.329673, -0.651112, -0.57101), (-0.329673, 0.37592, 0.57101, -0.651112), (0.16072, 0.473465, -0.278375, -0.820066), (-0.473465, 0.16072, 0.820066, -0.278375), (0.849385, 0.168953, 0.490393, 0.0975452), (-0.168953, 0.849385, -0.0975452, 0.490393), (0.651112, 0.57101, 0.37592, 0.329673), (-0.57101, 0.651112, -0.329673, 0.37592), (0.278375, 0.820066, 0.16072, 0.473465), (-0.820066, 0.278375, -0.473465, 0.16072), (0.749087, 0.149003, -0.536711, 0.358619), (-0.149003, 0.749087, -0.358619, -0.536711), (0.574227, 0.503584, -0.644115, 0.0422175), (-0.503584, 0.574227, -0.0422175, -0.644115), (0.245503, 0.72323, -0.578929, -0.285496), (-0.72323, 0.245503, 0.285496, -0.578929), (0.100101, 0.0199113, 0.827128, -0.552669), (-0.0199113, 0.100101, 0.552669, 0.827128), (0.0767343, 0.0672942, 0.992648, -0.0650616), (-0.0672942, 0.0767343, 0.0650616, 0.992648), (0.0328068, 0.0966457, 0.892189, 0.439979), (-0.0966457, 0.0328068, -0.439979, 0.892189), (0.100101, 0.0199113, -0.827128, 0.552669), (-0.0199113, 0.100101, -0.552669, -0.827128), (0.0767343, 0.0672942, -0.992648, 0.0650616), (-0.0672942, 0.0767343, -0.0650616, -0.992648), (0.0328068, 0.0966457, -0.892189, -0.439979), (-0.0966457, 0.0328068, 0.439979, -0.892189), (0.749087, 0.149003, 0.536711, -0.358619), (-0.149003, 0.749087, 0.358619, 0.536711), (0.574227, 0.503584, 0.644115, -0.0422175), (-0.503584, 0.574227, 0.0422175, 0.644115), (0.245503, 0.72323, 0.578929, 0.285496), (-0.72323, 0.245503, -0.285496, 0.578929), (0.849385, 0.168953, -0.0975452, 0.490393), (-0.168953, 0.849385, -0.490393, -0.0975452), (0.651112, 0.57101, -0.329673, 0.37592), (-0.57101, 0.651112, -0.37592, -0.329673), (0.278375, 0.820066, -0.473465, 0.16072), (-0.820066, 0.278375, -0.16072, -0.473465), (0.490393, 0.0975452, 0.168953, -0.849385), (-0.0975452, 0.490393, 0.849385, 0.168953), (0.37592, 0.329673, 0.57101, -0.651112), (-0.329673, 0.37592, 0.651112, 0.57101), (0.16072, 0.473465, 0.820066, -0.278375), (-0.473465, 0.16072, 0.278375, 0.820066), (0.633094, 0.12593, -0.424324, -0.635045), (-0.12593, 0.633094, 0.635045, -0.424324), (0.485311, 0.425606, -0.0499525, -0.762127), (-0.425606, 0.485311, 0.762127, -0.0499525), (0.207488, 0.611241, 0.337804, -0.684998), (-0.611241, 0.207488, 0.684998, 0.337804), (0.975664, 0.194072, 0.0567026, 0.0848615), (-0.194072, 0.975664, -0.0848615, 0.0567026), (0.747914, 0.655903, 0.00667518, 0.101844), (-0.655903, 0.747914, -0.101844, 0.00667518), (0.319761, 0.941985, -0.0451409, 0.0915367), (-0.941985, 0.319761, -0.0915367, -0.0451409), (0.975664, 0.194072, -0.0567026, -0.0848615), (-0.194072, 0.975664, 0.0848615, -0.0567026), (0.747914, 0.655903, -0.00667518, -0.101844), (-0.655903, 0.747914, 0.101844, -0.00667518), (0.319761, 0.941985, 0.0451409, -0.0915367), (-0.941985, 0.319761, 0.0915367, 0.0451409), (0.633094, 0.12593, 0.424324, 0.635045), (-0.12593, 0.633094, -0.635045, 0.424324), (0.485311, 0.425606, 0.0499525, 0.762127), (-0.425606, 0.485311, -0.762127, 0.0499525), (0.207488, 0.611241, -0.337804, 0.684998), (-0.611241, 0.207488, -0.684998, -0.337804), (0.633094, 0.12593, -0.749087, 0.149003), (-0.12593, 0.633094, -0.149003, -0.749087), (0.485311, 0.425606, -0.72323, -0.245503), (-0.425606, 0.485311, 0.245503, -0.72323), (0.207488, 0.611241, -0.503584, -0.574227), (-0.611241, 0.207488, 0.574227, -0.503584), (0.749087, 0.149003, 0.633094, -0.12593), (-0.149003, 0.749087, 0.12593, 0.633094), (0.574227, 0.503584, 0.611241, 0.207488), (-0.503584, 0.574227, -0.207488, 0.611241), (0.245503, 0.72323, 0.425606, 0.485311), (-0.72323, 0.245503, -0.485311, 0.425606), (0.849385, 0.168953, -0.490393, 0.0975452), (-0.168953, 0.849385, -0.0975452, -0.490393), (0.651112, 0.57101, -0.473465, -0.16072), (-0.57101, 0.651112, 0.16072, -0.473465), (0.278375, 0.820066, -0.329673, -0.37592), (-0.820066, 0.278375, 0.37592, -0.329673), (0.300303, 0.059734, 0.949933, -0.0622619), (-0.059734, 0.300303, 0.0622619, 0.949933), (0.230203, 0.201883, 0.853797, 0.421046), (-0.201883, 0.230203, -0.421046, 0.853797), (0.0984203, 0.289937, 0.528887, 0.791535), (-0.289937, 0.0984203, -0.791535, 0.528887), (0.300303, 0.059734, -0.949933, 0.0622619), (-0.059734, 0.300303, -0.0622619, -0.949933), (0.230203, 0.201883, -0.853797, -0.421046), (-0.201883, 0.230203, 0.421046, -0.853797), (0.0984203, 0.289937, -0.528887, -0.791535), (-0.289937, 0.0984203, 0.791535, -0.528887), (0.849385, 0.168953, 0.490393, -0.0975452), (-0.168953, 0.849385, 0.0975452, 0.490393), (0.651112, 0.57101, 0.473465, 0.16072), (-0.57101, 0.651112, -0.16072, 0.473465), (0.278375, 0.820066, 0.329673, 0.37592), (-0.820066, 0.278375, -0.37592, 0.329673), (0.749087, 0.149003, 0.12593, 0.633094), (-0.149003, 0.749087, -0.633094, 0.12593), (0.574227, 0.503584, -0.207488, 0.611241), (-0.503584, 0.574227, -0.611241, -0.207488), (0.245503, 0.72323, -0.485311, 0.425606), (-0.72323, 0.245503, -0.425606, -0.485311), (0.633094, 0.12593, -0.149003, -0.749087), (-0.12593, 0.633094, 0.749087, -0.149003), (0.485311, 0.425606, 0.245503, -0.72323), (-0.425606, 0.485311, 0.72323, 0.245503), (0.207488, 0.611241, 0.574227, -0.503584), (-0.611241, 0.207488, 0.503584, 0.574227), (0.490393, 0.0975452, -0.168953, -0.849385), (-0.0975452, 0.490393, 0.849385, -0.168953), (0.37592, 0.329673, 0.278375, -0.820066), (-0.329673, 0.37592, 0.820066, 0.278375), (0.16072, 0.473465, 0.651112, -0.57101), (-0.473465, 0.16072, 0.57101, 0.651112), (0.93368, 0.18572, 0.0200255, 0.305531), (-0.18572, 0.93368, -0.305531, 0.0200255), (0.71573, 0.627679, -0.135423, 0.27461), (-0.627679, 0.71573, -0.27461, -0.135423), (0.306001, 0.901451, -0.254585, 0.170108), (-0.901451, 0.306001, -0.170108, -0.254585), (0.93368, 0.18572, -0.0200255, -0.305531), (-0.18572, 0.93368, 0.305531, -0.0200255), (0.71573, 0.627679, 0.135423, -0.27461), (-0.627679, 0.71573, 0.27461, 0.135423), (0.306001, 0.901451, 0.254585, -0.170108), (-0.901451, 0.306001, 0.170108, 0.254585), (0.490393, 0.0975452, 0.168953, 0.849385), (-0.0975452, 0.490393, -0.849385, 0.168953), (0.37592, 0.329673, -0.278375, 0.820066), (-0.329673, 0.37592, -0.820066, -0.278375), (0.16072, 0.473465, -0.651112, 0.57101), (-0.473465, 0.16072, -0.57101, -0.651112), (0.578929, 0.285496, -0.503584, -0.574227), (-0.285496, 0.578929, 0.574227, -0.503584), (0.358619, 0.536711, -0.149003, -0.749087), (-0.536711, 0.358619, 0.749087, -0.149003), (0.0422175, 0.644115, 0.245503, -0.72323), (-0.644115, 0.0422175, 0.72323, 0.245503), (0.684998, 0.337804, 0.425606, 0.485311), (-0.337804, 0.684998, -0.485311, 0.425606), (0.424324, 0.635045, 0.12593, 0.633094), (-0.635045, 0.424324, -0.633094, 0.12593), (0.0499525, 0.762127, -0.207488, 0.611241), (-0.762127, 0.0499525, -0.611241, -0.207488), (0.776715, 0.383033, -0.37592, 0.329673), (-0.383033, 0.776715, -0.329673, -0.37592), (0.481138, 0.720074, -0.490393, 0.0975452), (-0.720074, 0.481138, -0.0975452, -0.490393), (0.0566408, 0.864171, -0.473465, -0.16072), (-0.864171, 0.0566408, 0.16072, -0.473465), (0.27461, 0.135423, 0.627679, -0.71573), (-0.135423, 0.27461, 0.71573, 0.627679), (0.170108, 0.254585, 0.901451, -0.306001), (-0.254585, 0.170108, 0.306001, 0.901451), (0.0200255, 0.305531, 0.93368, 0.18572), (-0.305531, 0.0200255, -0.18572, 0.93368), (0.27461, 0.135423, -0.627679, 0.71573), (-0.135423, 0.27461, -0.71573, -0.627679), (0.170108, 0.254585, -0.901451, 0.306001), (-0.254585, 0.170108, -0.306001, -0.901451), (0.0200255, 0.305531, -0.93368, -0.18572), (-0.305531, 0.0200255, 0.18572, -0.93368), (0.776715, 0.383033, 0.37592, -0.329673), (-0.383033, 0.776715, 0.329673, 0.37592), (0.481138, 0.720074, 0.490393, -0.0975452), (-0.720074, 0.481138, 0.0975452, 0.490393), (0.0566408, 0.864171, 0.473465, 0.16072), (-0.864171, 0.0566408, -0.16072, 0.473465), (0.684998, 0.337804, -0.485311, 0.425606), (-0.337804, 0.684998, -0.425606, -0.485311), (0.424324, 0.635045, -0.633094, 0.12593), (-0.635045, 0.424324, -0.12593, -0.633094), (0.0499525, 0.762127, -0.611241, -0.207488), (-0.762127, 0.0499525, 0.207488, -0.611241), (0.578929, 0.285496, 0.574227, -0.503584), (-0.285496, 0.578929, 0.503584, 0.574227), (0.358619, 0.536711, 0.749087, -0.149003), (-0.536711, 0.358619, 0.149003, 0.749087), (0.0422175, 0.644115, 0.72323, 0.245503), (-0.644115, 0.0422175, -0.245503, 0.72323), (0.448436, 0.221144, -0.57101, -0.651112), (-0.221144, 0.448436, 0.651112, -0.57101), (0.277785, 0.415735, -0.168953, -0.849385), (-0.415735, 0.277785, 0.849385, -0.168953), (0.0327016, 0.498929, 0.278375, -0.820066), (-0.498929, 0.0327016, 0.820066, 0.278375), (0.853797, 0.421046, 0.230203, 0.201883), (-0.421046, 0.853797, -0.201883, 0.230203), (0.528887, 0.791535, 0.0984203, 0.289937), (-0.791535, 0.528887, -0.289937, 0.0984203), (0.0622619, 0.949933, -0.059734, 0.300303), (-0.949933, 0.0622619, -0.300303, -0.059734), (0.853797, 0.421046, -0.230203, -0.201883), (-0.421046, 0.853797, 0.201883, -0.230203), (0.528887, 0.791535, -0.0984203, -0.289937), (-0.791535, 0.528887, 0.289937, -0.0984203), (0.0622619, 0.949933, 0.059734, -0.300303), (-0.949933, 0.0622619, 0.300303, 0.059734), (0.448436, 0.221144, 0.57101, 0.651112), (-0.221144, 0.448436, -0.651112, 0.57101), (0.277785, 0.415735, 0.168953, 0.849385), (-0.415735, 0.277785, -0.849385, 0.168953), (0.0327016, 0.498929, -0.278375, 0.820066), (-0.498929, 0.0327016, -0.820066, -0.278375), (0.578929, 0.285496, -0.762127, -0.0499525), (-0.285496, 0.578929, 0.0499525, -0.762127), (0.358619, 0.536711, -0.635045, -0.424324), (-0.536711, 0.358619, 0.424324, -0.635045), (0.0422175, 0.644115, -0.337804, -0.684998), (-0.644115, 0.0422175, 0.684998, -0.337804), (0.684998, 0.337804, 0.644115, 0.0422175), (-0.337804, 0.684998, -0.0422175, 0.644115), (0.424324, 0.635045, 0.536711, 0.358619), (-0.635045, 0.424324, -0.358619, 0.536711), (0.0499525, 0.762127, 0.285496, 0.578929), (-0.762127, 0.0499525, -0.578929, 0.285496), (0.776715, 0.383033, -0.498929, -0.0327016), (-0.383033, 0.776715, 0.0327016, -0.498929), (0.481138, 0.720074, -0.415735, -0.277785), (-0.720074, 0.481138, 0.277785, -0.415735), (0.0566408, 0.864171, -0.221144, -0.448436), (-0.864171, 0.0566408, 0.448436, -0.221144), (0.27461, 0.135423, 0.93368, 0.18572), (-0.135423, 0.27461, -0.18572, 0.93368), (0.170108, 0.254585, 0.71573, 0.627679), (-0.254585, 0.170108, -0.627679, 0.71573), (0.0200255, 0.305531, 0.306001, 0.901451), (-0.305531, 0.0200255, -0.901451, 0.306001), (0.27461, 0.135423, -0.93368, -0.18572), (-0.135423, 0.27461, 0.18572, -0.93368), (0.170108, 0.254585, -0.71573, -0.627679), (-0.254585, 0.170108, 0.627679, -0.71573), (0.0200255, 0.305531, -0.306001, -0.901451), (-0.305531, 0.0200255, 0.901451, -0.306001), (0.776715, 0.383033, 0.498929, 0.0327016), (-0.383033, 0.776715, -0.0327016, 0.498929), (0.481138, 0.720074, 0.415735, 0.277785), (-0.720074, 0.481138, -0.277785, 0.415735), (0.0566408, 0.864171, 0.221144, 0.448436), (-0.864171, 0.0566408, -0.448436, 0.221144), (0.684998, 0.337804, -0.0422175, 0.644115), (-0.337804, 0.684998, -0.644115, -0.0422175), (0.424324, 0.635045, -0.358619, 0.536711), (-0.635045, 0.424324, -0.536711, -0.358619), (0.0499525, 0.762127, -0.578929, 0.285496), (-0.762127, 0.0499525, -0.285496, -0.578929), (0.578929, 0.285496, 0.0499525, -0.762127), (-0.285496, 0.578929, 0.762127, 0.0499525), (0.358619, 0.536711, 0.424324, -0.635045), (-0.536711, 0.358619, 0.635045, 0.424324), (0.0422175, 0.644115, 0.684998, -0.337804), (-0.644115, 0.0422175, 0.337804, 0.684998), (0.448436, 0.221144, 0.0566408, -0.864171), (-0.221144, 0.448436, 0.864171, 0.0566408), (0.277785, 0.415735, 0.481138, -0.720074), (-0.415735, 0.277785, 0.720074, 0.481138), (0.0327016, 0.498929, 0.776715, -0.383033), (-0.498929, 0.0327016, 0.383033, 0.776715), (0.853797, 0.421046, -0.059734, 0.300303), (-0.421046, 0.853797, -0.300303, -0.059734), (0.528887, 0.791535, -0.201883, 0.230203), (-0.791535, 0.528887, -0.230203, -0.201883), (0.0622619, 0.949933, -0.289937, 0.0984203), (-0.949933, 0.0622619, -0.0984203, -0.289937), (0.853797, 0.421046, 0.059734, -0.300303), (-0.421046, 0.853797, 0.300303, 0.059734), (0.528887, 0.791535, 0.201883, -0.230203), (-0.791535, 0.528887, 0.230203, 0.201883), (0.0622619, 0.949933, 0.289937, -0.0984203), (-0.949933, 0.0622619, 0.0984203, 0.289937), (0.448436, 0.221144, -0.0566408, 0.864171), (-0.221144, 0.448436, -0.864171, -0.0566408), (0.277785, 0.415735, -0.481138, 0.720074), (-0.415735, 0.277785, -0.720074, -0.481138), (0.0327016, 0.498929, -0.776715, 0.383033), (-0.498929, 0.0327016, -0.383033, -0.776715), (0.633094, 0.12593, -0.635045, -0.424324), (-0.12593, 0.633094, 0.424324, -0.635045), (0.485311, 0.425606, -0.337804, -0.684998), (-0.425606, 0.485311, 0.684998, -0.337804), (0.207488, 0.611241, 0.0499525, -0.762127), (-0.611241, 0.207488, 0.762127, 0.0499525), (0.749087, 0.149003, 0.536711, 0.358619), (-0.149003, 0.749087, -0.358619, 0.536711), (0.574227, 0.503584, 0.285496, 0.578929), (-0.503584, 0.574227, -0.578929, 0.285496), (0.245503, 0.72323, -0.0422175, 0.644115), (-0.72323, 0.245503, -0.644115, -0.0422175), (0.849385, 0.168953, -0.277785, 0.415735), (-0.168953, 0.849385, -0.415735, -0.277785), (0.651112, 0.57101, -0.448436, 0.221144), (-0.57101, 0.651112, -0.221144, -0.448436), (0.278375, 0.820066, -0.498929, -0.0327016), (-0.820066, 0.278375, 0.0327016, -0.498929), (0.300303, 0.059734, 0.421046, -0.853797), (-0.059734, 0.300303, 0.853797, 0.421046), (0.230203, 0.201883, 0.791535, -0.528887), (-0.201883, 0.230203, 0.528887, 0.791535), (0.0984203, 0.289937, 0.949933, -0.0622619), (-0.289937, 0.0984203, 0.0622619, 0.949933), (0.300303, 0.059734, -0.421046, 0.853797), (-0.059734, 0.300303, -0.853797, -0.421046), (0.230203, 0.201883, -0.791535, 0.528887), (-0.201883, 0.230203, -0.528887, -0.791535), (0.0984203, 0.289937, -0.949933, 0.0622619), (-0.289937, 0.0984203, -0.0622619, -0.949933), (0.849385, 0.168953, 0.277785, -0.415735), (-0.168953, 0.849385, 0.415735, 0.277785), (0.651112, 0.57101, 0.448436, -0.221144), (-0.57101, 0.651112, 0.221144, 0.448436), (0.278375, 0.820066, 0.498929, 0.0327016), (-0.820066, 0.278375, -0.0327016, 0.498929), (0.749087, 0.149003, -0.358619, 0.536711), (-0.149003, 0.749087, -0.536711, -0.358619), (0.574227, 0.503584, -0.578929, 0.285496), (-0.503584, 0.574227, -0.285496, -0.578929), (0.245503, 0.72323, -0.644115, -0.0422175), (-0.72323, 0.245503, 0.0422175, -0.644115), (0.633094, 0.12593, 0.424324, -0.635045), (-0.12593, 0.633094, 0.635045, 0.424324), (0.485311, 0.425606, 0.684998, -0.337804), (-0.425606, 0.485311, 0.337804, 0.684998), (0.207488, 0.611241, 0.762127, 0.0499525), (-0.611241, 0.207488, -0.0499525, 0.762127), (0.490393, 0.0975452, -0.720074, -0.481138), (-0.0975452, 0.490393, 0.481138, -0.720074), (0.37592, 0.329673, -0.383033, -0.776715), (-0.329673, 0.37592, 0.776715, -0.383033), (0.16072, 0.473465, 0.0566408, -0.864171), (-0.473465, 0.16072, 0.864171, 0.0566408), (0.93368, 0.18572, 0.27461, 0.135423), (-0.18572, 0.93368, -0.135423, 0.27461), (0.71573, 0.627679, 0.170108, 0.254585), (-0.627679, 0.71573, -0.254585, 0.170108), (0.306001, 0.901451, 0.0200255, 0.305531), (-0.901451, 0.306001, -0.305531, 0.0200255), (0.93368, 0.18572, -0.27461, -0.135423), (-0.18572, 0.93368, 0.135423, -0.27461), (0.71573, 0.627679, -0.170108, -0.254585), (-0.627679, 0.71573, 0.254585, -0.170108), (0.306001, 0.901451, -0.0200255, -0.305531), (-0.901451, 0.306001, 0.305531, -0.0200255), (0.490393, 0.0975452, 0.720074, 0.481138), (-0.0975452, 0.490393, -0.481138, 0.720074), (0.37592, 0.329673, 0.383033, 0.776715), (-0.329673, 0.37592, -0.776715, 0.383033), (0.16072, 0.473465, -0.0566408, 0.864171), (-0.473465, 0.16072, -0.864171, -0.0566408), (0.849385, 0.168953, -0.490393, -0.0975452), (-0.168953, 0.849385, 0.0975452, -0.490393), (0.651112, 0.57101, -0.37592, -0.329673), (-0.57101, 0.651112, 0.329673, -0.37592), (0.278375, 0.820066, -0.16072, -0.473465), (-0.820066, 0.278375, 0.473465, -0.16072), (0.490393, 0.0975452, 0.849385, 0.168953), (-0.0975452, 0.490393, -0.168953, 0.849385), (0.37592, 0.329673, 0.651112, 0.57101), (-0.329673, 0.37592, -0.57101, 0.651112), (0.16072, 0.473465, 0.278375, 0.820066), (-0.473465, 0.16072, -0.820066, 0.278375), (0.975664, 0.194072, -0.0848615, 0.0567026), (-0.194072, 0.975664, -0.0567026, -0.0848615), (0.747914, 0.655903, -0.101844, 0.00667518), (-0.655903, 0.747914, -0.00667518, -0.101844), (0.319761, 0.941985, -0.0915367, -0.0451409), (-0.941985, 0.319761, 0.0451409, -0.0915367), (0.633094, 0.12593, 0.635045, -0.424324), (-0.12593, 0.633094, 0.424324, 0.635045), (0.485311, 0.425606, 0.762127, -0.0499525), (-0.425606, 0.485311, 0.0499525, 0.762127), (0.207488, 0.611241, 0.684998, 0.337804), (-0.611241, 0.207488, -0.337804, 0.684998), (0.633094, 0.12593, -0.635045, 0.424324), (-0.12593, 0.633094, -0.424324, -0.635045), (0.485311, 0.425606, -0.762127, 0.0499525), (-0.425606, 0.485311, -0.0499525, -0.762127), (0.207488, 0.611241, -0.684998, -0.337804), (-0.611241, 0.207488, 0.337804, -0.684998), (0.975664, 0.194072, 0.0848615, -0.0567026), (-0.194072, 0.975664, 0.0567026, 0.0848615), (0.747914, 0.655903, 0.101844, -0.00667518), (-0.655903, 0.747914, 0.00667518, 0.101844), (0.319761, 0.941985, 0.0915367, 0.0451409), (-0.941985, 0.319761, -0.0451409, 0.0915367), (0.490393, 0.0975452, -0.168953, 0.849385), (-0.0975452, 0.490393, -0.849385, -0.168953), (0.37592, 0.329673, -0.57101, 0.651112), (-0.329673, 0.37592, -0.651112, -0.57101), (0.16072, 0.473465, -0.820066, 0.278375), (-0.473465, 0.16072, -0.278375, -0.820066), (0.849385, 0.168953, 0.0975452, -0.490393), (-0.168953, 0.849385, 0.490393, 0.0975452), (0.651112, 0.57101, 0.329673, -0.37592), (-0.57101, 0.651112, 0.37592, 0.329673), (0.278375, 0.820066, 0.473465, -0.16072), (-0.820066, 0.278375, 0.16072, 0.473465), (0.100101, 0.0199113, -0.552669, -0.827128), (-0.0199113, 0.100101, 0.827128, -0.552669), (0.0767343, 0.0672942, -0.0650616, -0.992648), (-0.0672942, 0.0767343, 0.992648, -0.0650616), (0.0328068, 0.0966457, 0.439979, -0.892189), (-0.0966457, 0.0328068, 0.892189, 0.439979), (0.749087, 0.149003, 0.358619, 0.536711), (-0.149003, 0.749087, -0.536711, 0.358619), (0.574227, 0.503584, 0.0422175, 0.644115), (-0.503584, 0.574227, -0.644115, 0.0422175), (0.245503, 0.72323, -0.285496, 0.578929), (-0.72323, 0.245503, -0.578929, -0.285496), (0.749087, 0.149003, -0.358619, -0.536711), (-0.149003, 0.749087, 0.536711, -0.358619), (0.574227, 0.503584, -0.0422175, -0.644115), (-0.503584, 0.574227, 0.644115, -0.0422175), (0.245503, 0.72323, 0.285496, -0.578929), (-0.72323, 0.245503, 0.578929, 0.285496), (0.100101, 0.0199113, 0.552669, 0.827128), (-0.0199113, 0.100101, -0.827128, 0.552669), (0.0767343, 0.0672942, 0.0650616, 0.992648), (-0.0672942, 0.0767343, -0.992648, 0.0650616), (0.0328068, 0.0966457, -0.439979, 0.892189), (-0.0966457, 0.0328068, -0.892189, -0.439979), (0.578929, 0.285496, -0.684998, -0.337804), (-0.285496, 0.578929, 0.337804, -0.684998), (0.358619, 0.536711, -0.424324, -0.635045), (-0.536711, 0.358619, 0.635045, -0.424324), (0.0422175, 0.644115, -0.0499525, -0.762127), (-0.644115, 0.0422175, 0.762127, -0.0499525), (0.684998, 0.337804, 0.578929, 0.285496), (-0.337804, 0.684998, -0.285496, 0.578929), (0.424324, 0.635045, 0.358619, 0.536711), (-0.635045, 0.424324, -0.536711, 0.358619), (0.0499525, 0.762127, 0.0422175, 0.644115), (-0.762127, 0.0499525, -0.644115, 0.0422175), (0.776715, 0.383033, -0.473465, 0.16072), (-0.383033, 0.776715, -0.16072, -0.473465), (0.481138, 0.720074, -0.490393, -0.0975452), (-0.720074, 0.481138, 0.0975452, -0.490393), (0.0566408, 0.864171, -0.37592, -0.329673), (-0.864171, 0.0566408, 0.329673, -0.37592), (0.27461, 0.135423, 0.901451, -0.306001), (-0.135423, 0.27461, 0.306001, 0.901451), (0.170108, 0.254585, 0.93368, 0.18572), (-0.254585, 0.170108, -0.18572, 0.93368), (0.0200255, 0.305531, 0.71573, 0.627679), (-0.305531, 0.0200255, -0.627679, 0.71573), (0.27461, 0.135423, -0.901451, 0.306001), (-0.135423, 0.27461, -0.306001, -0.901451), (0.170108, 0.254585, -0.93368, -0.18572), (-0.254585, 0.170108, 0.18572, -0.93368), (0.0200255, 0.305531, -0.71573, -0.627679), (-0.305531, 0.0200255, 0.627679, -0.71573), (0.776715, 0.383033, 0.473465, -0.16072), (-0.383033, 0.776715, 0.16072, 0.473465), (0.481138, 0.720074, 0.490393, 0.0975452), (-0.720074, 0.481138, -0.0975452, 0.490393), (0.0566408, 0.864171, 0.37592, 0.329673), (-0.864171, 0.0566408, -0.329673, 0.37592), (0.684998, 0.337804, -0.285496, 0.578929), (-0.337804, 0.684998, -0.578929, -0.285496), (0.424324, 0.635045, -0.536711, 0.358619), (-0.635045, 0.424324, -0.358619, -0.536711), (0.0499525, 0.762127, -0.644115, 0.0422175), (-0.762127, 0.0499525, -0.0422175, -0.644115), (0.578929, 0.285496, 0.337804, -0.684998), (-0.285496, 0.578929, 0.684998, 0.337804), (0.358619, 0.536711, 0.635045, -0.424324), (-0.536711, 0.358619, 0.424324, 0.635045), (0.0422175, 0.644115, 0.762127, -0.0499525), (-0.644115, 0.0422175, 0.0499525, 0.762127), (0.448436, 0.221144, -0.278375, -0.820066), (-0.221144, 0.448436, 0.820066, -0.278375), (0.277785, 0.415735, 0.168953, -0.849385), (-0.415735, 0.277785, 0.849385, 0.168953), (0.0327016, 0.498929, 0.57101, -0.651112), (-0.498929, 0.0327016, 0.651112, 0.57101), (0.853797, 0.421046, 0.0984203, 0.289937), (-0.421046, 0.853797, -0.289937, 0.0984203), (0.528887, 0.791535, -0.059734, 0.300303), (-0.791535, 0.528887, -0.300303, -0.059734), (0.0622619, 0.949933, -0.201883, 0.230203), (-0.949933, 0.0622619, -0.230203, -0.201883), (0.853797, 0.421046, -0.0984203, -0.289937), (-0.421046, 0.853797, 0.289937, -0.0984203), (0.528887, 0.791535, 0.059734, -0.300303), (-0.791535, 0.528887, 0.300303, 0.059734), (0.0622619, 0.949933, 0.201883, -0.230203), (-0.949933, 0.0622619, 0.230203, 0.201883), (0.448436, 0.221144, 0.278375, 0.820066), (-0.221144, 0.448436, -0.820066, 0.278375), (0.277785, 0.415735, -0.168953, 0.849385), (-0.415735, 0.277785, -0.849385, -0.168953), (0.0327016, 0.498929, -0.57101, 0.651112), (-0.498929, 0.0327016, -0.651112, -0.57101), (0.776715, 0.383033, -0.448436, -0.221144), (-0.383033, 0.776715, 0.221144, -0.448436), (0.481138, 0.720074, -0.277785, -0.415735), (-0.720074, 0.481138, 0.415735, -0.277785), (0.0566408, 0.864171, -0.0327016, -0.498929), (-0.864171, 0.0566408, 0.498929, -0.0327016), (0.448436, 0.221144, 0.776715, 0.383033), (-0.221144, 0.448436, -0.383033, 0.776715), (0.277785, 0.415735, 0.481138, 0.720074), (-0.415735, 0.277785, -0.720074, 0.481138), (0.0327016, 0.498929, 0.0566408, 0.864171), (-0.498929, 0.0327016, -0.864171, 0.0566408), (0.892189, 0.439979, -0.0966457, 0.0328068), (-0.439979, 0.892189, -0.0328068, -0.0966457), (0.552669, 0.827128, -0.100101, -0.0199113), (-0.827128, 0.552669, 0.0199113, -0.100101), (0.0650616, 0.992648, -0.0767343, -0.0672942), (-0.992648, 0.0650616, 0.0672942, -0.0767343), (0.578929, 0.285496, 0.72323, -0.245503), (-0.285496, 0.578929, 0.245503, 0.72323), (0.358619, 0.536711, 0.749087, 0.149003), (-0.536711, 0.358619, -0.149003, 0.749087), (0.0422175, 0.644115, 0.574227, 0.503584), (-0.644115, 0.0422175, -0.503584, 0.574227), (0.578929, 0.285496, -0.72323, 0.245503), (-0.285496, 0.578929, -0.245503, -0.72323), (0.358619, 0.536711, -0.749087, -0.149003), (-0.536711, 0.358619, 0.149003, -0.749087), (0.0422175, 0.644115, -0.574227, -0.503584), (-0.644115, 0.0422175, 0.503584, -0.574227), (0.892189, 0.439979, 0.0966457, -0.0328068), (-0.439979, 0.892189, 0.0328068, 0.0966457), (0.552669, 0.827128, 0.100101, 0.0199113), (-0.827128, 0.552669, -0.0199113, 0.100101), (0.0650616, 0.992648, 0.0767343, 0.0672942), (-0.992648, 0.0650616, -0.0672942, 0.0767343), (0.448436, 0.221144, -0.383033, 0.776715), (-0.221144, 0.448436, -0.776715, -0.383033), (0.277785, 0.415735, -0.720074, 0.481138), (-0.415735, 0.277785, -0.481138, -0.720074), (0.0327016, 0.498929, -0.864171, 0.0566408), (-0.498929, 0.0327016, -0.0566408, -0.864171), (0.776715, 0.383033, 0.221144, -0.448436), (-0.383033, 0.776715, 0.448436, 0.221144), (0.481138, 0.720074, 0.415735, -0.277785), (-0.720074, 0.481138, 0.277785, 0.415735), (0.0566408, 0.864171, 0.498929, -0.0327016), (-0.864171, 0.0566408, 0.0327016, 0.498929), (0.0915367, 0.0451409, -0.319761, -0.941985), (-0.0451409, 0.0915367, 0.941985, -0.319761), (0.0567026, 0.0848615, 0.194072, -0.975664), (-0.0848615, 0.0567026, 0.975664, 0.194072), (0.00667518, 0.101844, 0.655903, -0.747914), (-0.101844, 0.00667518, 0.747914, 0.655903), (0.684998, 0.337804, 0.207488, 0.611241), (-0.337804, 0.684998, -0.611241, 0.207488), (0.424324, 0.635045, -0.12593, 0.633094), (-0.635045, 0.424324, -0.633094, -0.12593), (0.0499525, 0.762127, -0.425606, 0.485311), (-0.762127, 0.0499525, -0.485311, -0.425606), (0.684998, 0.337804, -0.207488, -0.611241), (-0.337804, 0.684998, 0.611241, -0.207488), (0.424324, 0.635045, 0.12593, -0.633094), (-0.635045, 0.424324, 0.633094, 0.12593), (0.0499525, 0.762127, 0.425606, -0.485311), (-0.762127, 0.0499525, 0.485311, 0.425606), (0.0915367, 0.0451409, 0.319761, 0.941985), (-0.0451409, 0.0915367, -0.941985, 0.319761), (0.0567026, 0.0848615, -0.194072, 0.975664), (-0.0848615, 0.0567026, -0.975664, -0.194072), (0.00667518, 0.101844, -0.655903, 0.747914), (-0.101844, 0.00667518, -0.747914, -0.655903), (0.633094, 0.12593, -0.749087, -0.149003), (-0.12593, 0.633094, 0.149003, -0.749087), (0.485311, 0.425606, -0.574227, -0.503584), (-0.425606, 0.485311, 0.503584, -0.574227), (0.207488, 0.611241, -0.245503, -0.72323), (-0.611241, 0.207488, 0.72323, -0.245503), (0.749087, 0.149003, 0.633094, 0.12593), (-0.149003, 0.749087, -0.12593, 0.633094), (0.574227, 0.503584, 0.485311, 0.425606), (-0.503584, 0.574227, -0.425606, 0.485311), (0.245503, 0.72323, 0.207488, 0.611241), (-0.72323, 0.245503, -0.611241, 0.207488), (0.849385, 0.168953, -0.415735, 0.277785), (-0.168953, 0.849385, -0.277785, -0.415735), (0.651112, 0.57101, -0.498929, 0.0327016), (-0.57101, 0.651112, -0.0327016, -0.498929), (0.278375, 0.820066, -0.448436, -0.221144), (-0.820066, 0.278375, 0.221144, -0.448436), (0.300303, 0.059734, 0.791535, -0.528887), (-0.059734, 0.300303, 0.528887, 0.791535), (0.230203, 0.201883, 0.949933, -0.0622619), (-0.201883, 0.230203, 0.0622619, 0.949933), (0.0984203, 0.289937, 0.853797, 0.421046), (-0.289937, 0.0984203, -0.421046, 0.853797), (0.300303, 0.059734, -0.791535, 0.528887), (-0.059734, 0.300303, -0.528887, -0.791535), (0.230203, 0.201883, -0.949933, 0.0622619), (-0.201883, 0.230203, -0.0622619, -0.949933), (0.0984203, 0.289937, -0.853797, -0.421046), (-0.289937, 0.0984203, 0.421046, -0.853797), (0.849385, 0.168953, 0.415735, -0.277785), (-0.168953, 0.849385, 0.277785, 0.415735), (0.651112, 0.57101, 0.498929, -0.0327016), (-0.57101, 0.651112, 0.0327016, 0.498929), (0.278375, 0.820066, 0.448436, 0.221144), (-0.820066, 0.278375, -0.221144, 0.448436), (0.749087, 0.149003, -0.12593, 0.633094), (-0.149003, 0.749087, -0.633094, -0.12593), (0.574227, 0.503584, -0.425606, 0.485311), (-0.503584, 0.574227, -0.485311, -0.425606), (0.245503, 0.72323, -0.611241, 0.207488), (-0.72323, 0.245503, -0.207488, -0.611241), (0.633094, 0.12593, 0.149003, -0.749087), (-0.12593, 0.633094, 0.749087, 0.149003), (0.485311, 0.425606, 0.503584, -0.574227), (-0.425606, 0.485311, 0.574227, 0.503584), (0.207488, 0.611241, 0.72323, -0.245503), (-0.611241, 0.207488, 0.245503, 0.72323), (0.490393, 0.0975452, -0.481138, -0.720074), (-0.0975452, 0.490393, 0.720074, -0.481138), (0.37592, 0.329673, -0.0566408, -0.864171), (-0.329673, 0.37592, 0.864171, -0.0566408), (0.16072, 0.473465, 0.383033, -0.776715), (-0.473465, 0.16072, 0.776715, 0.383033), (0.93368, 0.18572, 0.170108, 0.254585), (-0.18572, 0.93368, -0.254585, 0.170108), (0.71573, 0.627679, 0.0200255, 0.305531), (-0.627679, 0.71573, -0.305531, 0.0200255), (0.306001, 0.901451, -0.135423, 0.27461), (-0.901451, 0.306001, -0.27461, -0.135423), (0.93368, 0.18572, -0.170108, -0.254585), (-0.18572, 0.93368, 0.254585, -0.170108), (0.71573, 0.627679, -0.0200255, -0.305531), (-0.627679, 0.71573, 0.305531, -0.0200255), (0.306001, 0.901451, 0.135423, -0.27461), (-0.901451, 0.306001, 0.27461, 0.135423), (0.490393, 0.0975452, 0.481138, 0.720074), (-0.0975452, 0.490393, -0.720074, 0.481138), (0.37592, 0.329673, 0.0566408, 0.864171), (-0.329673, 0.37592, -0.864171, 0.0566408), (0.16072, 0.473465, -0.383033, 0.776715), (-0.473465, 0.16072, -0.776715, -0.383033), (0.749087, 0.149003, -0.633094, 0.12593), (-0.149003, 0.749087, -0.12593, -0.633094), (0.574227, 0.503584, -0.611241, -0.207488), (-0.503584, 0.574227, 0.207488, -0.611241), (0.245503, 0.72323, -0.425606, -0.485311), (-0.72323, 0.245503, 0.485311, -0.425606), (0.633094, 0.12593, 0.749087, -0.149003), (-0.12593, 0.633094, 0.149003, 0.749087), (0.485311, 0.425606, 0.72323, 0.245503), (-0.425606, 0.485311, -0.245503, 0.72323), (0.207488, 0.611241, 0.503584, 0.574227), (-0.611241, 0.207488, -0.574227, 0.503584), (0.93368, 0.18572, -0.305531, 0.0200255), (-0.18572, 0.93368, -0.0200255, -0.305531), (0.71573, 0.627679, -0.27461, -0.135423), (-0.627679, 0.71573, 0.135423, -0.27461), (0.306001, 0.901451, -0.170108, -0.254585), (-0.901451, 0.306001, 0.254585, -0.170108), (0.490393, 0.0975452, 0.849385, -0.168953), (-0.0975452, 0.490393, 0.168953, 0.849385), (0.37592, 0.329673, 0.820066, 0.278375), (-0.329673, 0.37592, -0.278375, 0.820066), (0.16072, 0.473465, 0.57101, 0.651112), (-0.473465, 0.16072, -0.651112, 0.57101), (0.490393, 0.0975452, -0.849385, 0.168953), (-0.0975452, 0.490393, -0.168953, -0.849385), (0.37592, 0.329673, -0.820066, -0.278375), (-0.329673, 0.37592, 0.278375, -0.820066), (0.16072, 0.473465, -0.57101, -0.651112), (-0.473465, 0.16072, 0.651112, -0.57101), (0.93368, 0.18572, 0.305531, -0.0200255), (-0.18572, 0.93368, 0.0200255, 0.305531), (0.71573, 0.627679, 0.27461, 0.135423), (-0.627679, 0.71573, -0.135423, 0.27461), (0.306001, 0.901451, 0.170108, 0.254585), (-0.901451, 0.306001, -0.254585, 0.170108), (0.633094, 0.12593, 0.149003, 0.749087), (-0.12593, 0.633094, -0.749087, 0.149003), (0.485311, 0.425606, -0.245503, 0.72323), (-0.425606, 0.485311, -0.72323, -0.245503), (0.207488, 0.611241, -0.574227, 0.503584), (-0.611241, 0.207488, -0.503584, -0.574227), (0.749087, 0.149003, -0.12593, -0.633094), (-0.149003, 0.749087, 0.633094, -0.12593), (0.574227, 0.503584, 0.207488, -0.611241), (-0.503584, 0.574227, 0.611241, 0.207488), (0.245503, 0.72323, 0.485311, -0.425606), (-0.72323, 0.245503, 0.425606, 0.485311), (0.300303, 0.059734, -0.0622619, -0.949933), (-0.059734, 0.300303, 0.949933, -0.0622619), (0.230203, 0.201883, 0.421046, -0.853797), (-0.201883, 0.230203, 0.853797, 0.421046), (0.0984203, 0.289937, 0.791535, -0.528887), (-0.289937, 0.0984203, 0.528887, 0.791535), (0.849385, 0.168953, 0.0975452, 0.490393), (-0.168953, 0.849385, -0.490393, 0.0975452), (0.651112, 0.57101, -0.16072, 0.473465), (-0.57101, 0.651112, -0.473465, -0.16072), (0.278375, 0.820066, -0.37592, 0.329673), (-0.820066, 0.278375, -0.329673, -0.37592), (0.849385, 0.168953, -0.0975452, -0.490393), (-0.168953, 0.849385, 0.490393, -0.0975452), (0.651112, 0.57101, 0.16072, -0.473465), (-0.57101, 0.651112, 0.473465, 0.16072), (0.278375, 0.820066, 0.37592, -0.329673), (-0.820066, 0.278375, 0.329673, 0.37592), (0.300303, 0.059734, 0.0622619, 0.949933), (-0.059734, 0.300303, -0.949933, 0.0622619), (0.230203, 0.201883, -0.421046, 0.853797), (-0.201883, 0.230203, -0.853797, -0.421046), (0.0984203, 0.289937, -0.791535, 0.528887), (-0.289937, 0.0984203, -0.528887, -0.791535), (0.684998, 0.337804, -0.425606, -0.485311), (-0.337804, 0.684998, 0.485311, -0.425606), (0.424324, 0.635045, -0.12593, -0.633094), (-0.635045, 0.424324, 0.633094, -0.12593), (0.0499525, 0.762127, 0.207488, -0.611241), (-0.762127, 0.0499525, 0.611241, 0.207488), (0.578929, 0.285496, 0.503584, 0.574227), (-0.285496, 0.578929, -0.574227, 0.503584), (0.358619, 0.536711, 0.149003, 0.749087), (-0.536711, 0.358619, -0.749087, 0.149003), (0.0422175, 0.644115, -0.245503, 0.72323), (-0.644115, 0.0422175, -0.72323, -0.245503), (0.853797, 0.421046, -0.201883, 0.230203), (-0.421046, 0.853797, -0.230203, -0.201883), (0.528887, 0.791535, -0.289937, 0.0984203), (-0.791535, 0.528887, -0.0984203, -0.289937), (0.0622619, 0.949933, -0.300303, -0.059734), (-0.949933, 0.0622619, 0.059734, -0.300303), (0.448436, 0.221144, 0.651112, -0.57101), (-0.221144, 0.448436, 0.57101, 0.651112), (0.277785, 0.415735, 0.849385, -0.168953), (-0.415735, 0.277785, 0.168953, 0.849385), (0.0327016, 0.498929, 0.820066, 0.278375), (-0.498929, 0.0327016, -0.278375, 0.820066), (0.448436, 0.221144, -0.651112, 0.57101), (-0.221144, 0.448436, -0.57101, -0.651112), (0.277785, 0.415735, -0.849385, 0.168953), (-0.415735, 0.277785, -0.168953, -0.849385), (0.0327016, 0.498929, -0.820066, -0.278375), (-0.498929, 0.0327016, 0.278375, -0.820066), (0.853797, 0.421046, 0.201883, -0.230203), (-0.421046, 0.853797, 0.230203, 0.201883), (0.528887, 0.791535, 0.289937, -0.0984203), (-0.791535, 0.528887, 0.0984203, 0.289937), (0.0622619, 0.949933, 0.300303, 0.059734), (-0.949933, 0.0622619, -0.059734, 0.300303), (0.578929, 0.285496, -0.574227, 0.503584), (-0.285496, 0.578929, -0.503584, -0.574227), (0.358619, 0.536711, -0.749087, 0.149003), (-0.536711, 0.358619, -0.149003, -0.749087), (0.0422175, 0.644115, -0.72323, -0.245503), (-0.644115, 0.0422175, 0.245503, -0.72323), (0.684998, 0.337804, 0.485311, -0.425606), (-0.337804, 0.684998, 0.425606, 0.485311), (0.424324, 0.635045, 0.633094, -0.12593), (-0.635045, 0.424324, 0.12593, 0.633094), (0.0499525, 0.762127, 0.611241, 0.207488), (-0.762127, 0.0499525, -0.207488, 0.611241), (0.27461, 0.135423, -0.71573, -0.627679), (-0.135423, 0.27461, 0.627679, -0.71573), (0.170108, 0.254585, -0.306001, -0.901451), (-0.254585, 0.170108, 0.901451, -0.306001), (0.0200255, 0.305531, 0.18572, -0.93368), (-0.305531, 0.0200255, 0.93368, 0.18572), (0.776715, 0.383033, 0.329673, 0.37592), (-0.383033, 0.776715, -0.37592, 0.329673), (0.481138, 0.720074, 0.0975452, 0.490393), (-0.720074, 0.481138, -0.490393, 0.0975452), (0.0566408, 0.864171, -0.16072, 0.473465), (-0.864171, 0.0566408, -0.473465, -0.16072), (0.776715, 0.383033, -0.329673, -0.37592), (-0.383033, 0.776715, 0.37592, -0.329673), (0.481138, 0.720074, -0.0975452, -0.490393), (-0.720074, 0.481138, 0.490393, -0.0975452), (0.0566408, 0.864171, 0.16072, -0.473465), (-0.864171, 0.0566408, 0.473465, 0.16072), (0.27461, 0.135423, 0.71573, 0.627679), (-0.135423, 0.27461, -0.627679, 0.71573), (0.170108, 0.254585, 0.306001, 0.901451), (-0.254585, 0.170108, -0.901451, 0.306001), (0.0200255, 0.305531, -0.18572, 0.93368), (-0.305531, 0.0200255, -0.93368, -0.18572), (0.684998, 0.337804, -0.644115, -0.0422175), (-0.337804, 0.684998, 0.0422175, -0.644115), (0.424324, 0.635045, -0.536711, -0.358619), (-0.635045, 0.424324, 0.358619, -0.536711), (0.0499525, 0.762127, -0.285496, -0.578929), (-0.762127, 0.0499525, 0.578929, -0.285496), (0.578929, 0.285496, 0.762127, 0.0499525), (-0.285496, 0.578929, -0.0499525, 0.762127), (0.358619, 0.536711, 0.635045, 0.424324), (-0.536711, 0.358619, -0.424324, 0.635045), (0.0422175, 0.644115, 0.337804, 0.684998), (-0.644115, 0.0422175, -0.684998, 0.337804), (0.853797, 0.421046, -0.300303, -0.059734), (-0.421046, 0.853797, 0.059734, -0.300303), (0.528887, 0.791535, -0.230203, -0.201883), (-0.791535, 0.528887, 0.201883, -0.230203), (0.0622619, 0.949933, -0.0984203, -0.289937), (-0.949933, 0.0622619, 0.289937, -0.0984203), (0.448436, 0.221144, 0.864171, 0.0566408), (-0.221144, 0.448436, -0.0566408, 0.864171), (0.277785, 0.415735, 0.720074, 0.481138), (-0.415735, 0.277785, -0.481138, 0.720074), (0.0327016, 0.498929, 0.383033, 0.776715), (-0.498929, 0.0327016, -0.776715, 0.383033), (0.448436, 0.221144, -0.864171, -0.0566408), (-0.221144, 0.448436, 0.0566408, -0.864171), (0.277785, 0.415735, -0.720074, -0.481138), (-0.415735, 0.277785, 0.481138, -0.720074), (0.0327016, 0.498929, -0.383033, -0.776715), (-0.498929, 0.0327016, 0.776715, -0.383033), (0.853797, 0.421046, 0.300303, 0.059734), (-0.421046, 0.853797, -0.059734, 0.300303), (0.528887, 0.791535, 0.230203, 0.201883), (-0.791535, 0.528887, -0.201883, 0.230203), (0.0622619, 0.949933, 0.0984203, 0.289937), (-0.949933, 0.0622619, -0.289937, 0.0984203), (0.578929, 0.285496, -0.0499525, 0.762127), (-0.285496, 0.578929, -0.762127, -0.0499525), (0.358619, 0.536711, -0.424324, 0.635045), (-0.536711, 0.358619, -0.635045, -0.424324), (0.0422175, 0.644115, -0.684998, 0.337804), (-0.644115, 0.0422175, -0.337804, -0.684998), (0.684998, 0.337804, 0.0422175, -0.644115), (-0.337804, 0.684998, 0.644115, 0.0422175), (0.424324, 0.635045, 0.358619, -0.536711), (-0.635045, 0.424324, 0.536711, 0.358619), (0.0499525, 0.762127, 0.578929, -0.285496), (-0.762127, 0.0499525, 0.285496, 0.578929), (0.27461, 0.135423, 0.18572, -0.93368), (-0.135423, 0.27461, 0.93368, 0.18572), (0.170108, 0.254585, 0.627679, -0.71573), (-0.254585, 0.170108, 0.71573, 0.627679), (0.0200255, 0.305531, 0.901451, -0.306001), (-0.305531, 0.0200255, 0.306001, 0.901451), (0.776715, 0.383033, -0.0327016, 0.498929), (-0.383033, 0.776715, -0.498929, -0.0327016), (0.481138, 0.720074, -0.277785, 0.415735), (-0.720074, 0.481138, -0.415735, -0.277785), (0.0566408, 0.864171, -0.448436, 0.221144), (-0.864171, 0.0566408, -0.221144, -0.448436), (0.776715, 0.383033, 0.0327016, -0.498929), (-0.383033, 0.776715, 0.498929, 0.0327016), (0.481138, 0.720074, 0.277785, -0.415735), (-0.720074, 0.481138, 0.415735, 0.277785), (0.0566408, 0.864171, 0.448436, -0.221144), (-0.864171, 0.0566408, 0.221144, 0.448436), (0.27461, 0.135423, -0.18572, 0.93368), (-0.135423, 0.27461, -0.93368, -0.18572), (0.170108, 0.254585, -0.627679, 0.71573), (-0.254585, 0.170108, -0.71573, -0.627679), (0.0200255, 0.305531, -0.901451, 0.306001), (-0.305531, 0.0200255, -0.306001, -0.901451), (0.749087, 0.149003, -0.536711, -0.358619), (-0.149003, 0.749087, 0.358619, -0.536711), (0.574227, 0.503584, -0.285496, -0.578929), (-0.503584, 0.574227, 0.578929, -0.285496), (0.245503, 0.72323, 0.0422175, -0.644115), (-0.72323, 0.245503, 0.644115, 0.0422175), (0.633094, 0.12593, 0.635045, 0.424324), (-0.12593, 0.633094, -0.424324, 0.635045), (0.485311, 0.425606, 0.337804, 0.684998), (-0.425606, 0.485311, -0.684998, 0.337804), (0.207488, 0.611241, -0.0499525, 0.762127), (-0.611241, 0.207488, -0.762127, -0.0499525), (0.93368, 0.18572, -0.135423, 0.27461), (-0.18572, 0.93368, -0.27461, -0.135423), (0.71573, 0.627679, -0.254585, 0.170108), (-0.627679, 0.71573, -0.170108, -0.254585), (0.306001, 0.901451, -0.305531, 0.0200255), (-0.901451, 0.306001, -0.0200255, -0.305531), (0.490393, 0.0975452, 0.481138, -0.720074), (-0.0975452, 0.490393, 0.720074, 0.481138), (0.37592, 0.329673, 0.776715, -0.383033), (-0.329673, 0.37592, 0.383033, 0.776715), (0.16072, 0.473465, 0.864171, 0.0566408), (-0.473465, 0.16072, -0.0566408, 0.864171), (0.490393, 0.0975452, -0.481138, 0.720074), (-0.0975452, 0.490393, -0.720074, -0.481138), (0.37592, 0.329673, -0.776715, 0.383033), (-0.329673, 0.37592, -0.383033, -0.776715), (0.16072, 0.473465, -0.864171, -0.0566408), (-0.473465, 0.16072, 0.0566408, -0.864171), (0.93368, 0.18572, 0.135423, -0.27461), (-0.18572, 0.93368, 0.27461, 0.135423), (0.71573, 0.627679, 0.254585, -0.170108), (-0.627679, 0.71573, 0.170108, 0.254585), (0.306001, 0.901451, 0.305531, -0.0200255), (-0.901451, 0.306001, 0.0200255, 0.305531), (0.633094, 0.12593, -0.424324, 0.635045), (-0.12593, 0.633094, -0.635045, -0.424324), (0.485311, 0.425606, -0.684998, 0.337804), (-0.425606, 0.485311, -0.337804, -0.684998), (0.207488, 0.611241, -0.762127, -0.0499525), (-0.611241, 0.207488, 0.0499525, -0.762127), (0.749087, 0.149003, 0.358619, -0.536711), (-0.149003, 0.749087, 0.536711, 0.358619), (0.574227, 0.503584, 0.578929, -0.285496), (-0.503584, 0.574227, 0.285496, 0.578929), (0.245503, 0.72323, 0.644115, 0.0422175), (-0.72323, 0.245503, -0.0422175, 0.644115), (0.300303, 0.059734, -0.853797, -0.421046), (-0.059734, 0.300303, 0.421046, -0.853797), (0.230203, 0.201883, -0.528887, -0.791535), (-0.201883, 0.230203, 0.791535, -0.528887), (0.0984203, 0.289937, -0.0622619, -0.949933), (-0.289937, 0.0984203, 0.949933, -0.0622619), (0.849385, 0.168953, 0.415735, 0.277785), (-0.168953, 0.849385, -0.277785, 0.415735), (0.651112, 0.57101, 0.221144, 0.448436), (-0.57101, 0.651112, -0.448436, 0.221144), (0.278375, 0.820066, -0.0327016, 0.498929), (-0.820066, 0.278375, -0.498929, -0.0327016), (0.849385, 0.168953, -0.415735, -0.277785), (-0.168953, 0.849385, 0.277785, -0.415735), (0.651112, 0.57101, -0.221144, -0.448436), (-0.57101, 0.651112, 0.448436, -0.221144), (0.278375, 0.820066, 0.0327016, -0.498929), (-0.820066, 0.278375, 0.498929, 0.0327016), (0.300303, 0.059734, 0.853797, 0.421046), (-0.059734, 0.300303, -0.421046, 0.853797), (0.230203, 0.201883, 0.528887, 0.791535), (-0.201883, 0.230203, -0.791535, 0.528887), (0.0984203, 0.289937, 0.0622619, 0.949933), (-0.289937, 0.0984203, -0.949933, 0.0622619), (0.762127, 0.0499525, -0.644115, -0.0422175), (-0.0499525, 0.762127, 0.0422175, -0.644115), (0.635045, 0.424324, -0.536711, -0.358619), (-0.424324, 0.635045, 0.358619, -0.536711), (0.337804, 0.684998, -0.285496, -0.578929), (-0.684998, 0.337804, 0.578929, -0.285496), (0.644115, 0.0422175, 0.762127, 0.0499525), (-0.0422175, 0.644115, -0.0499525, 0.762127), (0.536711, 0.358619, 0.635045, 0.424324), (-0.358619, 0.536711, -0.424324, 0.635045), (0.285496, 0.578929, 0.337804, 0.684998), (-0.578929, 0.285496, -0.684998, 0.337804), (0.949933, 0.0622619, -0.230203, 0.201883), (-0.0622619, 0.949933, -0.201883, -0.230203), (0.791535, 0.528887, -0.300303, 0.059734), (-0.528887, 0.791535, -0.059734, -0.300303), (0.421046, 0.853797, -0.289937, -0.0984203), (-0.853797, 0.421046, 0.0984203, -0.289937), (0.498929, 0.0327016, 0.651112, -0.57101), (-0.0327016, 0.498929, 0.57101, 0.651112), (0.415735, 0.277785, 0.849385, -0.168953), (-0.277785, 0.415735, 0.168953, 0.849385), (0.221144, 0.448436, 0.820066, 0.278375), (-0.448436, 0.221144, -0.278375, 0.820066), (0.498929, 0.0327016, -0.651112, 0.57101), (-0.0327016, 0.498929, -0.57101, -0.651112), (0.415735, 0.277785, -0.849385, 0.168953), (-0.277785, 0.415735, -0.168953, -0.849385), (0.221144, 0.448436, -0.820066, -0.278375), (-0.448436, 0.221144, 0.278375, -0.820066), (0.949933, 0.0622619, 0.230203, -0.201883), (-0.0622619, 0.949933, 0.201883, 0.230203), (0.791535, 0.528887, 0.300303, -0.059734), (-0.528887, 0.791535, 0.059734, 0.300303), (0.421046, 0.853797, 0.289937, 0.0984203), (-0.853797, 0.421046, -0.0984203, 0.289937), (0.644115, 0.0422175, -0.0499525, 0.762127), (-0.0422175, 0.644115, -0.762127, -0.0499525), (0.536711, 0.358619, -0.424324, 0.635045), (-0.358619, 0.536711, -0.635045, -0.424324), (0.285496, 0.578929, -0.684998, 0.337804), (-0.578929, 0.285496, -0.337804, -0.684998), (0.762127, 0.0499525, 0.0422175, -0.644115), (-0.0499525, 0.762127, 0.644115, 0.0422175), (0.635045, 0.424324, 0.358619, -0.536711), (-0.424324, 0.635045, 0.536711, 0.358619), (0.337804, 0.684998, 0.578929, -0.285496), (-0.684998, 0.337804, 0.285496, 0.578929), (0.305531, 0.0200255, -0.627679, -0.71573), (-0.0200255, 0.305531, 0.71573, -0.627679), (0.254585, 0.170108, -0.18572, -0.93368), (-0.170108, 0.254585, 0.93368, -0.18572), (0.135423, 0.27461, 0.306001, -0.901451), (-0.27461, 0.135423, 0.901451, 0.306001), (0.864171, 0.0566408, 0.329673, 0.37592), (-0.0566408, 0.864171, -0.37592, 0.329673), (0.720074, 0.481138, 0.0975452, 0.490393), (-0.481138, 0.720074, -0.490393, 0.0975452), (0.383033, 0.776715, -0.16072, 0.473465), (-0.776715, 0.383033, -0.473465, -0.16072), (0.864171, 0.0566408, -0.329673, -0.37592), (-0.0566408, 0.864171, 0.37592, -0.329673), (0.720074, 0.481138, -0.0975452, -0.490393), (-0.481138, 0.720074, 0.490393, -0.0975452), (0.383033, 0.776715, 0.16072, -0.473465), (-0.776715, 0.383033, 0.473465, 0.16072), (0.305531, 0.0200255, 0.627679, 0.71573), (-0.0200255, 0.305531, -0.71573, 0.627679), (0.254585, 0.170108, 0.18572, 0.93368), (-0.170108, 0.254585, -0.93368, 0.18572), (0.135423, 0.27461, -0.306001, 0.901451), (-0.27461, 0.135423, -0.901451, -0.306001), (0.473465, 0.16072, -0.820066, -0.278375), (-0.16072, 0.473465, 0.278375, -0.820066), (0.329673, 0.37592, -0.57101, -0.651112), (-0.37592, 0.329673, 0.651112, -0.57101), (0.0975452, 0.490393, -0.168953, -0.849385), (-0.490393, 0.0975452, 0.849385, -0.168953), (0.820066, 0.278375, 0.473465, 0.16072), (-0.278375, 0.820066, -0.16072, 0.473465), (0.57101, 0.651112, 0.329673, 0.37592), (-0.651112, 0.57101, -0.37592, 0.329673), (0.168953, 0.849385, 0.0975452, 0.490393), (-0.849385, 0.168953, -0.490393, 0.0975452), (0.72323, 0.245503, -0.578929, 0.285496), (-0.245503, 0.72323, -0.285496, -0.578929), (0.503584, 0.574227, -0.644115, -0.0422175), (-0.574227, 0.503584, 0.0422175, -0.644115), (0.149003, 0.749087, -0.536711, -0.358619), (-0.749087, 0.149003, 0.358619, -0.536711), (0.0966457, 0.0328068, 0.892189, -0.439979), (-0.0328068, 0.0966457, 0.439979, 0.892189), (0.0672942, 0.0767343, 0.992648, 0.0650616), (-0.0767343, 0.0672942, -0.0650616, 0.992648), (0.0199113, 0.100101, 0.827128, 0.552669), (-0.100101, 0.0199113, -0.552669, 0.827128), (0.0966457, 0.0328068, -0.892189, 0.439979), (-0.0328068, 0.0966457, -0.439979, -0.892189), (0.0672942, 0.0767343, -0.992648, -0.0650616), (-0.0767343, 0.0672942, 0.0650616, -0.992648), (0.0199113, 0.100101, -0.827128, -0.552669), (-0.100101, 0.0199113, 0.552669, -0.827128), (0.72323, 0.245503, 0.578929, -0.285496), (-0.245503, 0.72323, 0.285496, 0.578929), (0.503584, 0.574227, 0.644115, 0.0422175), (-0.574227, 0.503584, -0.0422175, 0.644115), (0.149003, 0.749087, 0.536711, 0.358619), (-0.749087, 0.149003, -0.358619, 0.536711), (0.820066, 0.278375, -0.16072, 0.473465), (-0.278375, 0.820066, -0.473465, -0.16072), (0.57101, 0.651112, -0.37592, 0.329673), (-0.651112, 0.57101, -0.329673, -0.37592), (0.168953, 0.849385, -0.490393, 0.0975452), (-0.849385, 0.168953, -0.0975452, -0.490393), (0.473465, 0.16072, 0.278375, -0.820066), (-0.16072, 0.473465, 0.820066, 0.278375), (0.329673, 0.37592, 0.651112, -0.57101), (-0.37592, 0.329673, 0.57101, 0.651112), (0.0975452, 0.490393, 0.849385, -0.168953), (-0.490393, 0.0975452, 0.168953, 0.849385), (0.611241, 0.207488, -0.337804, -0.684998), (-0.207488, 0.611241, 0.684998, -0.337804), (0.425606, 0.485311, 0.0499525, -0.762127), (-0.485311, 0.425606, 0.762127, 0.0499525), (0.12593, 0.633094, 0.424324, -0.635045), (-0.633094, 0.12593, 0.635045, 0.424324), (0.941985, 0.319761, 0.0451409, 0.0915367), (-0.319761, 0.941985, -0.0915367, 0.0451409), (0.655903, 0.747914, -0.00667518, 0.101844), (-0.747914, 0.655903, -0.101844, -0.00667518), (0.194072, 0.975664, -0.0567026, 0.0848615), (-0.975664, 0.194072, -0.0848615, -0.0567026), (0.941985, 0.319761, -0.0451409, -0.0915367), (-0.319761, 0.941985, 0.0915367, -0.0451409), (0.655903, 0.747914, 0.00667518, -0.101844), (-0.747914, 0.655903, 0.101844, 0.00667518), (0.194072, 0.975664, 0.0567026, -0.0848615), (-0.975664, 0.194072, 0.0848615, 0.0567026), (0.611241, 0.207488, 0.337804, 0.684998), (-0.207488, 0.611241, -0.684998, 0.337804), (0.425606, 0.485311, -0.0499525, 0.762127), (-0.485311, 0.425606, -0.762127, -0.0499525), (0.12593, 0.633094, -0.424324, 0.635045), (-0.633094, 0.12593, -0.635045, -0.424324), (0.72323, 0.245503, -0.611241, -0.207488), (-0.245503, 0.72323, 0.207488, -0.611241), (0.503584, 0.574227, -0.425606, -0.485311), (-0.574227, 0.503584, 0.485311, -0.425606), (0.149003, 0.749087, -0.12593, -0.633094), (-0.749087, 0.149003, 0.633094, -0.12593), (0.611241, 0.207488, 0.72323, 0.245503), (-0.207488, 0.611241, -0.245503, 0.72323), (0.425606, 0.485311, 0.503584, 0.574227), (-0.485311, 0.425606, -0.574227, 0.503584), (0.12593, 0.633094, 0.149003, 0.749087), (-0.633094, 0.12593, -0.749087, 0.149003), (0.901451, 0.306001, -0.27461, 0.135423), (-0.306001, 0.901451, -0.135423, -0.27461), (0.627679, 0.71573, -0.305531, -0.0200255), (-0.71573, 0.627679, 0.0200255, -0.305531), (0.18572, 0.93368, -0.254585, -0.170108), (-0.93368, 0.18572, 0.170108, -0.254585), (0.473465, 0.16072, 0.776715, -0.383033), (-0.16072, 0.473465, 0.383033, 0.776715), (0.329673, 0.37592, 0.864171, 0.0566408), (-0.37592, 0.329673, -0.0566408, 0.864171), (0.0975452, 0.490393, 0.720074, 0.481138), (-0.490393, 0.0975452, -0.481138, 0.720074), (0.473465, 0.16072, -0.776715, 0.383033), (-0.16072, 0.473465, -0.383033, -0.776715), (0.329673, 0.37592, -0.864171, -0.0566408), (-0.37592, 0.329673, 0.0566408, -0.864171), (0.0975452, 0.490393, -0.720074, -0.481138), (-0.490393, 0.0975452, 0.481138, -0.720074), (0.901451, 0.306001, 0.27461, -0.135423), (-0.306001, 0.901451, 0.135423, 0.27461), (0.627679, 0.71573, 0.305531, 0.0200255), (-0.71573, 0.627679, -0.0200255, 0.305531), (0.18572, 0.93368, 0.254585, 0.170108), (-0.93368, 0.18572, -0.170108, 0.254585), (0.611241, 0.207488, -0.245503, 0.72323), (-0.207488, 0.611241, -0.72323, -0.245503), (0.425606, 0.485311, -0.574227, 0.503584), (-0.485311, 0.425606, -0.503584, -0.574227), (0.12593, 0.633094, -0.749087, 0.149003), (-0.633094, 0.12593, -0.149003, -0.749087), (0.72323, 0.245503, 0.207488, -0.611241), (-0.245503, 0.72323, 0.611241, 0.207488), (0.503584, 0.574227, 0.485311, -0.425606), (-0.574227, 0.503584, 0.425606, 0.485311), (0.149003, 0.749087, 0.633094, -0.12593), (-0.749087, 0.149003, 0.12593, 0.633094), (0.289937, 0.0984203, -0.421046, -0.853797), (-0.0984203, 0.289937, 0.853797, -0.421046), (0.201883, 0.230203, 0.0622619, -0.949933), (-0.230203, 0.201883, 0.949933, 0.0622619), (0.059734, 0.300303, 0.528887, -0.791535), (-0.300303, 0.059734, 0.791535, 0.528887), (0.820066, 0.278375, 0.221144, 0.448436), (-0.278375, 0.820066, -0.448436, 0.221144), (0.57101, 0.651112, -0.0327016, 0.498929), (-0.651112, 0.57101, -0.498929, -0.0327016), (0.168953, 0.849385, -0.277785, 0.415735), (-0.849385, 0.168953, -0.415735, -0.277785), (0.820066, 0.278375, -0.221144, -0.448436), (-0.278375, 0.820066, 0.448436, -0.221144), (0.57101, 0.651112, 0.0327016, -0.498929), (-0.651112, 0.57101, 0.498929, 0.0327016), (0.168953, 0.849385, 0.277785, -0.415735), (-0.849385, 0.168953, 0.415735, 0.277785), (0.289937, 0.0984203, 0.421046, 0.853797), (-0.0984203, 0.289937, -0.853797, 0.421046), (0.201883, 0.230203, -0.0622619, 0.949933), (-0.230203, 0.201883, -0.949933, -0.0622619), (0.059734, 0.300303, -0.528887, 0.791535), (-0.300303, 0.059734, -0.791535, -0.528887), (0.498929, 0.0327016, -0.864171, -0.0566408), (-0.0327016, 0.498929, 0.0566408, -0.864171), (0.415735, 0.277785, -0.720074, -0.481138), (-0.277785, 0.415735, 0.481138, -0.720074), (0.221144, 0.448436, -0.383033, -0.776715), (-0.448436, 0.221144, 0.776715, -0.383033), (0.864171, 0.0566408, 0.498929, 0.0327016), (-0.0566408, 0.864171, -0.0327016, 0.498929), (0.720074, 0.481138, 0.415735, 0.277785), (-0.481138, 0.720074, -0.277785, 0.415735), (0.383033, 0.776715, 0.221144, 0.448436), (-0.776715, 0.383033, -0.448436, 0.221144), (0.762127, 0.0499525, -0.485311, 0.425606), (-0.0499525, 0.762127, -0.425606, -0.485311), (0.635045, 0.424324, -0.633094, 0.12593), (-0.424324, 0.635045, -0.12593, -0.633094), (0.337804, 0.684998, -0.611241, -0.207488), (-0.684998, 0.337804, 0.207488, -0.611241), (0.101844, 0.00667518, 0.747914, -0.655903), (-0.00667518, 0.101844, 0.655903, 0.747914), (0.0848615, 0.0567026, 0.975664, -0.194072), (-0.0567026, 0.0848615, 0.194072, 0.975664), (0.0451409, 0.0915367, 0.941985, 0.319761), (-0.0915367, 0.0451409, -0.319761, 0.941985), (0.101844, 0.00667518, -0.747914, 0.655903), (-0.00667518, 0.101844, -0.655903, -0.747914), (0.0848615, 0.0567026, -0.975664, 0.194072), (-0.0567026, 0.0848615, -0.194072, -0.975664), (0.0451409, 0.0915367, -0.941985, -0.319761), (-0.0915367, 0.0451409, 0.319761, -0.941985), (0.762127, 0.0499525, 0.485311, -0.425606), (-0.0499525, 0.762127, 0.425606, 0.485311), (0.635045, 0.424324, 0.633094, -0.12593), (-0.424324, 0.635045, 0.12593, 0.633094), (0.337804, 0.684998, 0.611241, 0.207488), (-0.684998, 0.337804, -0.207488, 0.611241), (0.864171, 0.0566408, -0.0327016, 0.498929), (-0.0566408, 0.864171, -0.498929, -0.0327016), (0.720074, 0.481138, -0.277785, 0.415735), (-0.481138, 0.720074, -0.415735, -0.277785), (0.383033, 0.776715, -0.448436, 0.221144), (-0.776715, 0.383033, -0.221144, -0.448436), (0.498929, 0.0327016, 0.0566408, -0.864171), (-0.0327016, 0.498929, 0.864171, 0.0566408), (0.415735, 0.277785, 0.481138, -0.720074), (-0.277785, 0.415735, 0.720074, 0.481138), (0.221144, 0.448436, 0.776715, -0.383033), (-0.448436, 0.221144, 0.383033, 0.776715), (0.644115, 0.0422175, -0.503584, -0.574227), (-0.0422175, 0.644115, 0.574227, -0.503584), (0.536711, 0.358619, -0.149003, -0.749087), (-0.358619, 0.536711, 0.749087, -0.149003), (0.285496, 0.578929, 0.245503, -0.72323), (-0.578929, 0.285496, 0.72323, 0.245503), (0.992648, 0.0650616, 0.0672942, 0.0767343), (-0.0650616, 0.992648, -0.0767343, 0.0672942), (0.827128, 0.552669, 0.0199113, 0.100101), (-0.552669, 0.827128, -0.100101, 0.0199113), (0.439979, 0.892189, -0.0328068, 0.0966457), (-0.892189, 0.439979, -0.0966457, -0.0328068), (0.992648, 0.0650616, -0.0672942, -0.0767343), (-0.0650616, 0.992648, 0.0767343, -0.0672942), (0.827128, 0.552669, -0.0199113, -0.100101), (-0.552669, 0.827128, 0.100101, -0.0199113), (0.439979, 0.892189, 0.0328068, -0.0966457), (-0.892189, 0.439979, 0.0966457, 0.0328068), (0.644115, 0.0422175, 0.503584, 0.574227), (-0.0422175, 0.644115, -0.574227, 0.503584), (0.536711, 0.358619, 0.149003, 0.749087), (-0.358619, 0.536711, -0.749087, 0.149003), (0.285496, 0.578929, -0.245503, 0.72323), (-0.578929, 0.285496, -0.72323, -0.245503), (0.644115, 0.0422175, -0.72323, 0.245503), (-0.0422175, 0.644115, -0.245503, -0.72323), (0.536711, 0.358619, -0.749087, -0.149003), (-0.358619, 0.536711, 0.149003, -0.749087), (0.285496, 0.578929, -0.574227, -0.503584), (-0.578929, 0.285496, 0.503584, -0.574227), (0.762127, 0.0499525, 0.611241, -0.207488), (-0.0499525, 0.762127, 0.207488, 0.611241), (0.635045, 0.424324, 0.633094, 0.12593), (-0.424324, 0.635045, -0.12593, 0.633094), (0.337804, 0.684998, 0.485311, 0.425606), (-0.684998, 0.337804, -0.425606, 0.485311), (0.864171, 0.0566408, -0.473465, 0.16072), (-0.0566408, 0.864171, -0.16072, -0.473465), (0.720074, 0.481138, -0.490393, -0.0975452), (-0.481138, 0.720074, 0.0975452, -0.490393), (0.383033, 0.776715, -0.37592, -0.329673), (-0.776715, 0.383033, 0.329673, -0.37592), (0.305531, 0.0200255, 0.93368, -0.18572), (-0.0200255, 0.305531, 0.18572, 0.93368), (0.254585, 0.170108, 0.901451, 0.306001), (-0.170108, 0.254585, -0.306001, 0.901451), (0.135423, 0.27461, 0.627679, 0.71573), (-0.27461, 0.135423, -0.71573, 0.627679), (0.305531, 0.0200255, -0.93368, 0.18572), (-0.0200255, 0.305531, -0.18572, -0.93368), (0.254585, 0.170108, -0.901451, -0.306001), (-0.170108, 0.254585, 0.306001, -0.901451), (0.135423, 0.27461, -0.627679, -0.71573), (-0.27461, 0.135423, 0.71573, -0.627679), (0.864171, 0.0566408, 0.473465, -0.16072), (-0.0566408, 0.864171, 0.16072, 0.473465), (0.720074, 0.481138, 0.490393, 0.0975452), (-0.481138, 0.720074, -0.0975452, 0.490393), (0.383033, 0.776715, 0.37592, 0.329673), (-0.776715, 0.383033, -0.329673, 0.37592), (0.762127, 0.0499525, 0.207488, 0.611241), (-0.0499525, 0.762127, -0.611241, 0.207488), (0.635045, 0.424324, -0.12593, 0.633094), (-0.424324, 0.635045, -0.633094, -0.12593), (0.337804, 0.684998, -0.425606, 0.485311), (-0.684998, 0.337804, -0.485311, -0.425606), (0.644115, 0.0422175, -0.245503, -0.72323), (-0.0422175, 0.644115, 0.72323, -0.245503), (0.536711, 0.358619, 0.149003, -0.749087), (-0.358619, 0.536711, 0.749087, 0.149003), (0.285496, 0.578929, 0.503584, -0.574227), (-0.578929, 0.285496, 0.574227, 0.503584), (0.498929, 0.0327016, -0.278375, -0.820066), (-0.0327016, 0.498929, 0.820066, -0.278375), (0.415735, 0.277785, 0.168953, -0.849385), (-0.277785, 0.415735, 0.849385, 0.168953), (0.221144, 0.448436, 0.57101, -0.651112), (-0.448436, 0.221144, 0.651112, 0.57101), (0.949933, 0.0622619, 0.059734, 0.300303), (-0.0622619, 0.949933, -0.300303, 0.059734), (0.791535, 0.528887, -0.0984203, 0.289937), (-0.528887, 0.791535, -0.289937, -0.0984203), (0.421046, 0.853797, -0.230203, 0.201883), (-0.853797, 0.421046, -0.201883, -0.230203), (0.949933, 0.0622619, -0.059734, -0.300303), (-0.0622619, 0.949933, 0.300303, -0.059734), (0.791535, 0.528887, 0.0984203, -0.289937), (-0.528887, 0.791535, 0.289937, 0.0984203), (0.421046, 0.853797, 0.230203, -0.201883), (-0.853797, 0.421046, 0.201883, 0.230203), (0.498929, 0.0327016, 0.278375, 0.820066), (-0.0327016, 0.498929, -0.820066, 0.278375), (0.415735, 0.277785, -0.168953, 0.849385), (-0.277785, 0.415735, -0.849385, -0.168953), (0.221144, 0.448436, -0.57101, 0.651112), (-0.448436, 0.221144, -0.651112, -0.57101), (0.611241, 0.207488, -0.574227, -0.503584), (-0.207488, 0.611241, 0.503584, -0.574227), (0.425606, 0.485311, -0.245503, -0.72323), (-0.485311, 0.425606, 0.72323, -0.245503), (0.12593, 0.633094, 0.149003, -0.749087), (-0.633094, 0.12593, 0.749087, 0.149003), (0.72323, 0.245503, 0.485311, 0.425606), (-0.245503, 0.72323, -0.425606, 0.485311), (0.503584, 0.574227, 0.207488, 0.611241), (-0.574227, 0.503584, -0.611241, 0.207488), (0.149003, 0.749087, -0.12593, 0.633094), (-0.749087, 0.149003, -0.633094, -0.12593), (0.820066, 0.278375, -0.329673, 0.37592), (-0.278375, 0.820066, -0.37592, -0.329673), (0.57101, 0.651112, -0.473465, 0.16072), (-0.651112, 0.57101, -0.16072, -0.473465), (0.168953, 0.849385, -0.490393, -0.0975452), (-0.849385, 0.168953, 0.0975452, -0.490393), (0.289937, 0.0984203, 0.528887, -0.791535), (-0.0984203, 0.289937, 0.791535, 0.528887), (0.201883, 0.230203, 0.853797, -0.421046), (-0.230203, 0.201883, 0.421046, 0.853797), (0.059734, 0.300303, 0.949933, 0.0622619), (-0.300303, 0.059734, -0.0622619, 0.949933), (0.289937, 0.0984203, -0.528887, 0.791535), (-0.0984203, 0.289937, -0.791535, -0.528887), (0.201883, 0.230203, -0.853797, 0.421046), (-0.230203, 0.201883, -0.421046, -0.853797), (0.059734, 0.300303, -0.949933, -0.0622619), (-0.300303, 0.059734, 0.0622619, -0.949933), (0.820066, 0.278375, 0.329673, -0.37592), (-0.278375, 0.820066, 0.37592, 0.329673), (0.57101, 0.651112, 0.473465, -0.16072), (-0.651112, 0.57101, 0.16072, 0.473465), (0.168953, 0.849385, 0.490393, 0.0975452), (-0.849385, 0.168953, -0.0975452, 0.490393), (0.72323, 0.245503, -0.425606, 0.485311), (-0.245503, 0.72323, -0.485311, -0.425606), (0.503584, 0.574227, -0.611241, 0.207488), (-0.574227, 0.503584, -0.207488, -0.611241), (0.149003, 0.749087, -0.633094, -0.12593), (-0.749087, 0.149003, 0.12593, -0.633094), (0.611241, 0.207488, 0.503584, -0.574227), (-0.207488, 0.611241, 0.574227, 0.503584), (0.425606, 0.485311, 0.72323, -0.245503), (-0.485311, 0.425606, 0.245503, 0.72323), (0.12593, 0.633094, 0.749087, 0.149003), (-0.633094, 0.12593, -0.149003, 0.749087), (0.473465, 0.16072, -0.651112, -0.57101), (-0.16072, 0.473465, 0.57101, -0.651112), (0.329673, 0.37592, -0.278375, -0.820066), (-0.37592, 0.329673, 0.820066, -0.278375), (0.0975452, 0.490393, 0.168953, -0.849385), (-0.490393, 0.0975452, 0.849385, 0.168953), (0.901451, 0.306001, 0.254585, 0.170108), (-0.306001, 0.901451, -0.170108, 0.254585), (0.627679, 0.71573, 0.135423, 0.27461), (-0.71573, 0.627679, -0.27461, 0.135423), (0.18572, 0.93368, -0.0200255, 0.305531), (-0.93368, 0.18572, -0.305531, -0.0200255), (0.901451, 0.306001, -0.254585, -0.170108), (-0.306001, 0.901451, 0.170108, -0.254585), (0.627679, 0.71573, -0.135423, -0.27461), (-0.71573, 0.627679, 0.27461, -0.135423), (0.18572, 0.93368, 0.0200255, -0.305531), (-0.93368, 0.18572, 0.305531, 0.0200255), (0.473465, 0.16072, 0.651112, 0.57101), (-0.16072, 0.473465, -0.57101, 0.651112), (0.329673, 0.37592, 0.278375, 0.820066), (-0.37592, 0.329673, -0.820066, 0.278375), (0.0975452, 0.490393, -0.168953, 0.849385), (-0.490393, 0.0975452, -0.849385, -0.168953), (0.611241, 0.207488, -0.762127, 0.0499525), (-0.207488, 0.611241, -0.0499525, -0.762127), (0.425606, 0.485311, -0.684998, -0.337804), (-0.485311, 0.425606, 0.337804, -0.684998), (0.12593, 0.633094, -0.424324, -0.635045), (-0.633094, 0.12593, 0.635045, -0.424324), (0.72323, 0.245503, 0.644115, -0.0422175), (-0.245503, 0.72323, 0.0422175, 0.644115), (0.503584, 0.574227, 0.578929, 0.285496), (-0.574227, 0.503584, -0.285496, 0.578929), (0.149003, 0.749087, 0.358619, 0.536711), (-0.749087, 0.149003, -0.536711, 0.358619), (0.820066, 0.278375, -0.498929, 0.0327016), (-0.278375, 0.820066, -0.0327016, -0.498929), (0.57101, 0.651112, -0.448436, -0.221144), (-0.651112, 0.57101, 0.221144, -0.448436), (0.168953, 0.849385, -0.277785, -0.415735), (-0.849385, 0.168953, 0.415735, -0.277785), (0.289937, 0.0984203, 0.949933, 0.0622619), (-0.0984203, 0.289937, -0.0622619, 0.949933), (0.201883, 0.230203, 0.791535, 0.528887), (-0.230203, 0.201883, -0.528887, 0.791535), (0.059734, 0.300303, 0.421046, 0.853797), (-0.300303, 0.059734, -0.853797, 0.421046), (0.289937, 0.0984203, -0.949933, -0.0622619), (-0.0984203, 0.289937, 0.0622619, -0.949933), (0.201883, 0.230203, -0.791535, -0.528887), (-0.230203, 0.201883, 0.528887, -0.791535), (0.059734, 0.300303, -0.421046, -0.853797), (-0.300303, 0.059734, 0.853797, -0.421046), (0.820066, 0.278375, 0.498929, -0.0327016), (-0.278375, 0.820066, 0.0327016, 0.498929), (0.57101, 0.651112, 0.448436, 0.221144), (-0.651112, 0.57101, -0.221144, 0.448436), (0.168953, 0.849385, 0.277785, 0.415735), (-0.849385, 0.168953, -0.415735, 0.277785), (0.72323, 0.245503, 0.0422175, 0.644115), (-0.245503, 0.72323, -0.644115, 0.0422175), (0.503584, 0.574227, -0.285496, 0.578929), (-0.574227, 0.503584, -0.578929, -0.285496), (0.149003, 0.749087, -0.536711, 0.358619), (-0.749087, 0.149003, -0.358619, -0.536711), (0.611241, 0.207488, -0.0499525, -0.762127), (-0.207488, 0.611241, 0.762127, -0.0499525), (0.425606, 0.485311, 0.337804, -0.684998), (-0.485311, 0.425606, 0.684998, 0.337804), (0.12593, 0.633094, 0.635045, -0.424324), (-0.633094, 0.12593, 0.424324, 0.635045), (0.473465, 0.16072, -0.0566408, -0.864171), (-0.16072, 0.473465, 0.864171, -0.0566408), (0.329673, 0.37592, 0.383033, -0.776715), (-0.37592, 0.329673, 0.776715, 0.383033), (0.0975452, 0.490393, 0.720074, -0.481138), (-0.490393, 0.0975452, 0.481138, 0.720074), (0.901451, 0.306001, -0.0200255, 0.305531), (-0.306001, 0.901451, -0.305531, -0.0200255), (0.627679, 0.71573, -0.170108, 0.254585), (-0.71573, 0.627679, -0.254585, -0.170108), (0.18572, 0.93368, -0.27461, 0.135423), (-0.93368, 0.18572, -0.135423, -0.27461), (0.901451, 0.306001, 0.0200255, -0.305531), (-0.306001, 0.901451, 0.305531, 0.0200255), (0.627679, 0.71573, 0.170108, -0.254585), (-0.71573, 0.627679, 0.254585, 0.170108), (0.18572, 0.93368, 0.27461, -0.135423), (-0.93368, 0.18572, 0.135423, 0.27461), (0.473465, 0.16072, 0.0566408, 0.864171), (-0.16072, 0.473465, -0.864171, 0.0566408), (0.329673, 0.37592, -0.383033, 0.776715), (-0.37592, 0.329673, -0.776715, -0.383033), (0.0975452, 0.490393, -0.720074, 0.481138), (-0.490393, 0.0975452, -0.481138, -0.720074), (0.644115, 0.0422175, -0.684998, -0.337804), (-0.0422175, 0.644115, 0.337804, -0.684998), (0.536711, 0.358619, -0.424324, -0.635045), (-0.358619, 0.536711, 0.635045, -0.424324), (0.285496, 0.578929, -0.0499525, -0.762127), (-0.578929, 0.285496, 0.762127, -0.0499525), (0.762127, 0.0499525, 0.578929, 0.285496), (-0.0499525, 0.762127, -0.285496, 0.578929), (0.635045, 0.424324, 0.358619, 0.536711), (-0.424324, 0.635045, -0.536711, 0.358619), (0.337804, 0.684998, 0.0422175, 0.644115), (-0.684998, 0.337804, -0.644115, 0.0422175), (0.864171, 0.0566408, -0.221144, 0.448436), (-0.0566408, 0.864171, -0.448436, -0.221144), (0.720074, 0.481138, -0.415735, 0.277785), (-0.481138, 0.720074, -0.277785, -0.415735), (0.383033, 0.776715, -0.498929, 0.0327016), (-0.776715, 0.383033, -0.0327016, -0.498929), (0.305531, 0.0200255, 0.306001, -0.901451), (-0.0200255, 0.305531, 0.901451, 0.306001), (0.254585, 0.170108, 0.71573, -0.627679), (-0.170108, 0.254585, 0.627679, 0.71573), (0.135423, 0.27461, 0.93368, -0.18572), (-0.27461, 0.135423, 0.18572, 0.93368), (0.305531, 0.0200255, -0.306001, 0.901451), (-0.0200255, 0.305531, -0.901451, -0.306001), (0.254585, 0.170108, -0.71573, 0.627679), (-0.170108, 0.254585, -0.627679, -0.71573), (0.135423, 0.27461, -0.93368, 0.18572), (-0.27461, 0.135423, -0.18572, -0.93368), (0.864171, 0.0566408, 0.221144, -0.448436), (-0.0566408, 0.864171, 0.448436, 0.221144), (0.720074, 0.481138, 0.415735, -0.277785), (-0.481138, 0.720074, 0.277785, 0.415735), (0.383033, 0.776715, 0.498929, -0.0327016), (-0.776715, 0.383033, 0.0327016, 0.498929), (0.762127, 0.0499525, -0.285496, 0.578929), (-0.0499525, 0.762127, -0.578929, -0.285496), (0.635045, 0.424324, -0.536711, 0.358619), (-0.424324, 0.635045, -0.358619, -0.536711), (0.337804, 0.684998, -0.644115, 0.0422175), (-0.684998, 0.337804, -0.0422175, -0.644115), (0.644115, 0.0422175, 0.337804, -0.684998), (-0.0422175, 0.644115, 0.684998, 0.337804), (0.536711, 0.358619, 0.635045, -0.424324), (-0.358619, 0.536711, 0.424324, 0.635045), (0.285496, 0.578929, 0.762127, -0.0499525), (-0.578929, 0.285496, 0.0499525, 0.762127), (0.498929, 0.0327016, -0.776715, -0.383033), (-0.0327016, 0.498929, 0.383033, -0.776715), (0.415735, 0.277785, -0.481138, -0.720074), (-0.277785, 0.415735, 0.720074, -0.481138), (0.221144, 0.448436, -0.0566408, -0.864171), (-0.448436, 0.221144, 0.864171, -0.0566408), (0.949933, 0.0622619, 0.289937, 0.0984203), (-0.0622619, 0.949933, -0.0984203, 0.289937), (0.791535, 0.528887, 0.201883, 0.230203), (-0.528887, 0.791535, -0.230203, 0.201883), (0.421046, 0.853797, 0.059734, 0.300303), (-0.853797, 0.421046, -0.300303, 0.059734), (0.949933, 0.0622619, -0.289937, -0.0984203), (-0.0622619, 0.949933, 0.0984203, -0.289937), (0.791535, 0.528887, -0.201883, -0.230203), (-0.528887, 0.791535, 0.230203, -0.201883), (0.421046, 0.853797, -0.059734, -0.300303), (-0.853797, 0.421046, 0.300303, -0.059734), (0.498929, 0.0327016, 0.776715, 0.383033), (-0.0327016, 0.498929, -0.383033, 0.776715), (0.415735, 0.277785, 0.481138, 0.720074), (-0.277785, 0.415735, -0.720074, 0.481138), (0.221144, 0.448436, 0.0566408, 0.864171), (-0.448436, 0.221144, -0.864171, 0.0566408), (0.814748, 0.0534014, -0.572411, 0.0753593), (-0.0534014, 0.814748, -0.0753593, -0.572411), (0.678892, 0.453621, -0.533402, -0.220942), (-0.453621, 0.678892, 0.220942, -0.533402), (0.361127, 0.732294, -0.351469, -0.458043), (-0.732294, 0.361127, 0.458043, -0.351469), (0.576114, 0.0377605, 0.809511, -0.106574), (-0.0377605, 0.576114, 0.106574, 0.809511), (0.480049, 0.320759, 0.754344, 0.31246), (-0.320759, 0.480049, -0.31246, 0.754344), (0.255355, 0.51781, 0.497052, 0.64777), (-0.51781, 0.255355, -0.64777, 0.497052), (0.976849, 0.0640261, -0.193291, 0.0656136), (-0.0640261, 0.976849, -0.0656136, -0.193291), (0.813963, 0.543873, -0.200202, -0.0398226), (-0.543873, 0.813963, 0.0398226, -0.200202), (0.432976, 0.877989, -0.153469, -0.134588), (-0.877989, 0.432976, 0.134588, -0.153469), (0.576114, 0.0377605, 0.707107, -0.408248), (-0.0377605, 0.576114, 0.408248, 0.707107), (0.480049, 0.320759, 0.816497, -1.99984e-16), (-0.320759, 0.480049, 2.4998e-16, 0.816497), (0.255355, 0.51781, 0.707107, 0.408248), (-0.51781, 0.255355, -0.408248, 0.707107), (0.576114, 0.0377605, -0.707107, 0.408248), (-0.0377605, 0.576114, -0.408248, -0.707107), (0.480049, 0.320759, -0.816497, 9.9992e-17), (-0.320759, 0.480049, -1.49988e-16, -0.816497), (0.255355, 0.51781, -0.707107, -0.408248), (-0.51781, 0.255355, 0.408248, -0.707107), (0.976849, 0.0640261, 0.193291, -0.0656136), (-0.0640261, 0.976849, 0.0656136, 0.193291), (0.813963, 0.543873, 0.200202, 0.0398226), (-0.543873, 0.813963, -0.0398226, 0.200202), (0.432976, 0.877989, 0.153469, 0.134588), (-0.877989, 0.432976, -0.134588, 0.153469), (0.576114, 0.0377605, 0.106574, 0.809511), (-0.0377605, 0.576114, -0.809511, 0.106574), (0.480049, 0.320759, -0.31246, 0.754344), (-0.320759, 0.480049, -0.754344, -0.31246), (0.255355, 0.51781, -0.64777, 0.497052), (-0.51781, 0.255355, -0.497052, -0.64777), (0.814748, 0.0534014, -0.0753593, -0.572411), (-0.0534014, 0.814748, 0.572411, -0.0753593), (0.678892, 0.453621, 0.220942, -0.533402), (-0.453621, 0.678892, 0.533402, 0.220942), (0.361127, 0.732294, 0.458043, -0.351469), (-0.732294, 0.361127, 0.351469, 0.458043), (0.203687, 0.0133504, -0.314672, -0.926993), (-0.0133504, 0.203687, 0.926993, -0.314672), (0.169723, 0.113405, 0.190983, -0.960135), (-0.113405, 0.169723, 0.960135, 0.190983), (0.0902818, 0.183073, 0.645463, -0.73601), (-0.183073, 0.0902818, 0.73601, 0.645463), (0.814748, 0.0534014, 0.288675, 0.5), (-0.0534014, 0.814748, -0.5, 0.288675), (0.678892, 0.453621, 3.53525e-17, 0.57735), (-0.453621, 0.678892, -0.57735, 7.0705e-17), (0.361127, 0.732294, -0.288675, 0.5), (-0.732294, 0.361127, -0.5, -0.288675), (0.814748, 0.0534014, -0.288675, -0.5), (-0.0534014, 0.814748, 0.5, -0.288675), (0.678892, 0.453621, -1.06058e-16, -0.57735), (-0.453621, 0.678892, 0.57735, -1.4141e-16), (0.361127, 0.732294, 0.288675, -0.5), (-0.732294, 0.361127, 0.5, 0.288675), (0.203687, 0.0133504, 0.314672, 0.926993), (-0.0133504, 0.203687, -0.926993, 0.314672), (0.169723, 0.113405, -0.190983, 0.960135), (-0.113405, 0.169723, -0.960135, -0.190983), (0.0902818, 0.183073, -0.645463, 0.73601), (-0.183073, 0.0902818, -0.73601, -0.645463), (0.54671, 0.185583, -0.809511, -0.106574), (-0.185583, 0.54671, 0.106574, -0.809511), (0.380673, 0.434075, -0.64777, -0.497052), (-0.434075, 0.380673, 0.497052, -0.64777), (0.112635, 0.566257, -0.31246, -0.754344), (-0.566257, 0.112635, 0.754344, -0.31246), (0.773165, 0.262454, 0.572411, 0.0753593), (-0.262454, 0.773165, -0.0753593, 0.572411), (0.538354, 0.613875, 0.458043, 0.351469), (-0.613875, 0.538354, -0.351469, 0.458043), (0.159291, 0.800808, 0.220942, 0.533402), (-0.800808, 0.159291, -0.533402, 0.220942), (0.773165, 0.262454, -0.557678, 0.149429), (-0.262454, 0.773165, -0.149429, -0.557678), (0.538354, 0.613875, -0.557678, -0.149429), (-0.613875, 0.538354, 0.149429, -0.557678), (0.159291, 0.800808, -0.408248, -0.408248), (-0.800808, 0.159291, 0.408248, -0.408248), (0.193291, 0.0656136, 0.976849, -0.0640261), (-0.0656136, 0.193291, 0.0640261, 0.976849), (0.134588, 0.153469, 0.877989, 0.432976), (-0.153469, 0.134588, -0.432976, 0.877989), (0.0398226, 0.200202, 0.543873, 0.813963), (-0.200202, 0.0398226, -0.813963, 0.543873), (0.193291, 0.0656136, -0.976849, 0.0640261), (-0.0656136, 0.193291, -0.0640261, -0.976849), (0.134588, 0.153469, -0.877989, -0.432976), (-0.153469, 0.134588, 0.432976, -0.877989), (0.0398226, 0.200202, -0.543873, -0.813963), (-0.200202, 0.0398226, 0.813963, -0.543873), (0.773165, 0.262454, 0.557678, -0.149429), (-0.262454, 0.773165, 0.149429, 0.557678), (0.538354, 0.613875, 0.557678, 0.149429), (-0.613875, 0.538354, -0.149429, 0.557678), (0.159291, 0.800808, 0.408248, 0.408248), (-0.800808, 0.159291, -0.408248, 0.408248), (0.773165, 0.262454, -0.0753593, 0.572411), (-0.262454, 0.773165, -0.572411, -0.0753593), (0.538354, 0.613875, -0.351469, 0.458043), (-0.613875, 0.538354, -0.458043, -0.351469), (0.159291, 0.800808, -0.533402, 0.220942), (-0.800808, 0.159291, -0.220942, -0.533402), (0.54671, 0.185583, 0.106574, -0.809511), (-0.185583, 0.54671, 0.809511, 0.106574), (0.380673, 0.434075, 0.497052, -0.64777), (-0.434075, 0.380673, 0.64777, 0.497052), (0.112635, 0.566257, 0.754344, -0.31246), (-0.566257, 0.112635, 0.31246, 0.754344), (0.54671, 0.185583, -0.211325, -0.788675), (-0.185583, 0.54671, 0.788675, -0.211325), (0.380673, 0.434075, 0.211325, -0.788675), (-0.434075, 0.380673, 0.788675, 0.211325), (0.112635, 0.566257, 0.57735, -0.57735), (-0.566257, 0.112635, 0.57735, 0.57735), (0.926993, 0.314672, 0.0133504, 0.203687), (-0.314672, 0.926993, -0.203687, 0.0133504), (0.645463, 0.73601, -0.0902818, 0.183073), (-0.73601, 0.645463, -0.183073, -0.0902818), (0.190983, 0.960135, -0.169723, 0.113405), (-0.960135, 0.190983, -0.113405, -0.169723), (0.926993, 0.314672, -0.0133504, -0.203687), (-0.314672, 0.926993, 0.203687, -0.0133504), (0.645463, 0.73601, 0.0902818, -0.183073), (-0.73601, 0.645463, 0.183073, 0.0902818), (0.190983, 0.960135, 0.169723, -0.113405), (-0.960135, 0.190983, 0.113405, 0.169723), (0.54671, 0.185583, 0.211325, 0.788675), (-0.185583, 0.54671, -0.788675, 0.211325), (0.380673, 0.434075, -0.211325, 0.788675), (-0.434075, 0.380673, -0.788675, -0.211325), (0.112635, 0.566257, -0.57735, 0.57735), (-0.566257, 0.112635, -0.57735, -0.57735), (0.773165, 0.262454, -0.572411, -0.0753593), (-0.262454, 0.773165, 0.0753593, -0.572411), (0.538354, 0.613875, -0.458043, -0.351469), (-0.613875, 0.538354, 0.351469, -0.458043), (0.159291, 0.800808, -0.220942, -0.533402), (-0.800808, 0.159291, 0.533402, -0.220942), (0.54671, 0.185583, 0.809511, 0.106574), (-0.185583, 0.54671, -0.106574, 0.809511), (0.380673, 0.434075, 0.64777, 0.497052), (-0.434075, 0.380673, -0.497052, 0.64777), (0.112635, 0.566257, 0.31246, 0.754344), (-0.566257, 0.112635, -0.754344, 0.31246), (0.926993, 0.314672, -0.203687, 0.0133504), (-0.314672, 0.926993, -0.0133504, -0.203687), (0.645463, 0.73601, -0.183073, -0.0902818), (-0.73601, 0.645463, 0.0902818, -0.183073), (0.190983, 0.960135, -0.113405, -0.169723), (-0.960135, 0.190983, 0.169723, -0.113405), (0.54671, 0.185583, 0.788675, -0.211325), (-0.185583, 0.54671, 0.211325, 0.788675), (0.380673, 0.434075, 0.788675, 0.211325), (-0.434075, 0.380673, -0.211325, 0.788675), (0.112635, 0.566257, 0.57735, 0.57735), (-0.566257, 0.112635, -0.57735, 0.57735), (0.54671, 0.185583, -0.788675, 0.211325), (-0.185583, 0.54671, -0.211325, -0.788675), (0.380673, 0.434075, -0.788675, -0.211325), (-0.434075, 0.380673, 0.211325, -0.788675), (0.112635, 0.566257, -0.57735, -0.57735), (-0.566257, 0.112635, 0.57735, -0.57735), (0.926993, 0.314672, 0.203687, -0.0133504), (-0.314672, 0.926993, 0.0133504, 0.203687), (0.645463, 0.73601, 0.183073, 0.0902818), (-0.73601, 0.645463, -0.0902818, 0.183073), (0.190983, 0.960135, 0.113405, 0.169723), (-0.960135, 0.190983, -0.169723, 0.113405), (0.54671, 0.185583, -0.106574, 0.809511), (-0.185583, 0.54671, -0.809511, -0.106574), (0.380673, 0.434075, -0.497052, 0.64777), (-0.434075, 0.380673, -0.64777, -0.497052), (0.112635, 0.566257, -0.754344, 0.31246), (-0.566257, 0.112635, -0.31246, -0.754344), (0.773165, 0.262454, 0.0753593, -0.572411), (-0.262454, 0.773165, 0.572411, 0.0753593), (0.538354, 0.613875, 0.351469, -0.458043), (-0.613875, 0.538354, 0.458043, 0.351469), (0.159291, 0.800808, 0.533402, -0.220942), (-0.800808, 0.159291, 0.220942, 0.533402), (0.193291, 0.0656136, -0.0640261, -0.976849), (-0.0656136, 0.193291, 0.976849, -0.0640261), (0.134588, 0.153469, 0.432976, -0.877989), (-0.153469, 0.134588, 0.877989, 0.432976), (0.0398226, 0.200202, 0.813963, -0.543873), (-0.200202, 0.0398226, 0.543873, 0.813963), (0.773165, 0.262454, 0.149429, 0.557678), (-0.262454, 0.773165, -0.557678, 0.149429), (0.538354, 0.613875, -0.149429, 0.557678), (-0.613875, 0.538354, -0.557678, -0.149429), (0.159291, 0.800808, -0.408248, 0.408248), (-0.800808, 0.159291, -0.408248, -0.408248), (0.773165, 0.262454, -0.149429, -0.557678), (-0.262454, 0.773165, 0.557678, -0.149429), (0.538354, 0.613875, 0.149429, -0.557678), (-0.613875, 0.538354, 0.557678, 0.149429), (0.159291, 0.800808, 0.408248, -0.408248), (-0.800808, 0.159291, 0.408248, 0.408248), (0.193291, 0.0656136, 0.0640261, 0.976849), (-0.0656136, 0.193291, -0.976849, 0.0640261), (0.134588, 0.153469, -0.432976, 0.877989), (-0.153469, 0.134588, -0.877989, -0.432976), (0.0398226, 0.200202, -0.813963, 0.543873), (-0.200202, 0.0398226, -0.543873, -0.813963), (0.576114, 0.0377605, -0.809511, 0.106574), (-0.0377605, 0.576114, -0.106574, -0.809511), (0.480049, 0.320759, -0.754344, -0.31246), (-0.320759, 0.480049, 0.31246, -0.754344), (0.255355, 0.51781, -0.497052, -0.64777), (-0.51781, 0.255355, 0.64777, -0.497052), (0.814748, 0.0534014, 0.572411, -0.0753593), (-0.0534014, 0.814748, 0.0753593, 0.572411), (0.678892, 0.453621, 0.533402, 0.220942), (-0.453621, 0.678892, -0.220942, 0.533402), (0.361127, 0.732294, 0.351469, 0.458043), (-0.732294, 0.361127, -0.458043, 0.351469), (0.814748, 0.0534014, -0.5, 0.288675), (-0.0534014, 0.814748, -0.288675, -0.5), (0.678892, 0.453621, -0.57735, 7.0705e-17), (-0.453621, 0.678892, -1.06058e-16, -0.57735), (0.361127, 0.732294, -0.5, -0.288675), (-0.732294, 0.361127, 0.288675, -0.5), (0.203687, 0.0133504, 0.926993, -0.314672), (-0.0133504, 0.203687, 0.314672, 0.926993), (0.169723, 0.113405, 0.960135, 0.190983), (-0.113405, 0.169723, -0.190983, 0.960135), (0.0902818, 0.183073, 0.73601, 0.645463), (-0.183073, 0.0902818, -0.645463, 0.73601), (0.203687, 0.0133504, -0.926993, 0.314672), (-0.0133504, 0.203687, -0.314672, -0.926993), (0.169723, 0.113405, -0.960135, -0.190983), (-0.113405, 0.169723, 0.190983, -0.960135), (0.0902818, 0.183073, -0.73601, -0.645463), (-0.183073, 0.0902818, 0.645463, -0.73601), (0.814748, 0.0534014, 0.5, -0.288675), (-0.0534014, 0.814748, 0.288675, 0.5), (0.678892, 0.453621, 0.57735, -1.4141e-16), (-0.453621, 0.678892, 1.76763e-16, 0.57735), (0.361127, 0.732294, 0.5, 0.288675), (-0.732294, 0.361127, -0.288675, 0.5), (0.814748, 0.0534014, 0.0753593, 0.572411), (-0.0534014, 0.814748, -0.572411, 0.0753593), (0.678892, 0.453621, -0.220942, 0.533402), (-0.453621, 0.678892, -0.533402, -0.220942), (0.361127, 0.732294, -0.458043, 0.351469), (-0.732294, 0.361127, -0.351469, -0.458043), (0.576114, 0.0377605, -0.106574, -0.809511), (-0.0377605, 0.576114, 0.809511, -0.106574), (0.480049, 0.320759, 0.31246, -0.754344), (-0.320759, 0.480049, 0.754344, 0.31246), (0.255355, 0.51781, 0.64777, -0.497052), (-0.51781, 0.255355, 0.497052, 0.64777), (0.576114, 0.0377605, -0.408248, -0.707107), (-0.0377605, 0.576114, 0.707107, -0.408248), (0.480049, 0.320759, -1.49988e-16, -0.816497), (-0.320759, 0.480049, 0.816497, -1.99984e-16), (0.255355, 0.51781, 0.408248, -0.707107), (-0.51781, 0.255355, 0.707107, 0.408248), (0.976849, 0.0640261, 0.0656136, 0.193291), (-0.0640261, 0.976849, -0.193291, 0.0656136), (0.813963, 0.543873, -0.0398226, 0.200202), (-0.543873, 0.813963, -0.200202, -0.0398226), (0.432976, 0.877989, -0.134588, 0.153469), (-0.877989, 0.432976, -0.153469, -0.134588), (0.976849, 0.0640261, -0.0656136, -0.193291), (-0.0640261, 0.976849, 0.193291, -0.0656136), (0.813963, 0.543873, 0.0398226, -0.200202), (-0.543873, 0.813963, 0.200202, 0.0398226), (0.432976, 0.877989, 0.134588, -0.153469), (-0.877989, 0.432976, 0.153469, 0.134588), (0.576114, 0.0377605, 0.408248, 0.707107), (-0.0377605, 0.576114, -0.707107, 0.408248), (0.480049, 0.320759, 4.9996e-17, 0.816497), (-0.320759, 0.480049, -0.816497, 9.9992e-17), (0.255355, 0.51781, -0.408248, 0.707107), (-0.51781, 0.255355, -0.707107, -0.408248), (0.705593, 0.046247, -0.612372, 0.353553), (-0.046247, 0.705593, -0.353553, -0.612372), (0.587938, 0.392847, -0.707107, 8.65956e-17), (-0.392847, 0.587938, -1.29893e-16, -0.707107), (0.312745, 0.634185, -0.612372, -0.353553), (-0.634185, 0.312745, 0.353553, -0.612372), (0.705593, 0.046247, 0.612372, -0.353553), (-0.046247, 0.705593, 0.353553, 0.612372), (0.587938, 0.392847, 0.707107, -1.73191e-16), (-0.392847, 0.587938, 2.16489e-16, 0.707107), (0.312745, 0.634185, 0.612372, 0.353553), (-0.634185, 0.312745, -0.353553, 0.612372), (0.910916, 0.0597046, -0.404756, 0.0532871), (-0.0597046, 0.910916, -0.0532871, -0.404756), (0.759024, 0.507164, -0.377172, -0.15623), (-0.507164, 0.759024, 0.15623, -0.377172), (0.403752, 0.818729, -0.248526, -0.323885), (-0.818729, 0.403752, 0.323885, -0.248526), (0.407374, 0.0267007, 0.905061, -0.119154), (-0.0267007, 0.407374, 0.119154, 0.905061), (0.339446, 0.226811, 0.843383, 0.349341), (-0.226811, 0.339446, -0.349341, 0.843383), (0.180564, 0.366147, 0.555721, 0.724229), (-0.366147, 0.180564, -0.724229, 0.555721), (0.407374, 0.0267007, -0.905061, 0.119154), (-0.0267007, 0.407374, -0.119154, -0.905061), (0.339446, 0.226811, -0.843383, -0.349341), (-0.226811, 0.339446, 0.349341, -0.843383), (0.180564, 0.366147, -0.555721, -0.724229), (-0.366147, 0.180564, 0.724229, -0.555721), (0.910916, 0.0597046, 0.404756, -0.0532871), (-0.0597046, 0.910916, 0.0532871, 0.404756), (0.759024, 0.507164, 0.377172, 0.15623), (-0.507164, 0.759024, -0.15623, 0.377172), (0.403752, 0.818729, 0.248526, 0.323885), (-0.818729, 0.403752, -0.323885, 0.248526), (0.705593, 0.046247, 0.353553, 0.612372), (-0.046247, 0.705593, -0.612372, 0.353553), (0.587938, 0.392847, 4.32978e-17, 0.707107), (-0.392847, 0.587938, -0.707107, 8.65956e-17), (0.312745, 0.634185, -0.353553, 0.612372), (-0.634185, 0.312745, -0.612372, -0.353553), (0.705593, 0.046247, -0.353553, -0.612372), (-0.046247, 0.705593, 0.612372, -0.353553), (0.587938, 0.392847, -1.29893e-16, -0.707107), (-0.392847, 0.587938, 0.707107, -1.73191e-16), (0.312745, 0.634185, 0.353553, -0.612372), (-0.634185, 0.312745, 0.612372, 0.353553), (0.407374, 0.0267007, -0.119154, -0.905061), (-0.0267007, 0.407374, 0.905061, -0.119154), (0.339446, 0.226811, 0.349341, -0.843383), (-0.226811, 0.339446, 0.843383, 0.349341), (0.180564, 0.366147, 0.724229, -0.555721), (-0.366147, 0.180564, 0.555721, 0.724229), (0.910916, 0.0597046, 0.0532871, 0.404756), (-0.0597046, 0.910916, -0.404756, 0.0532871), (0.759024, 0.507164, -0.15623, 0.377172), (-0.507164, 0.759024, -0.377172, -0.15623), (0.403752, 0.818729, -0.323885, 0.248526), (-0.818729, 0.403752, -0.248526, -0.323885), (0.910916, 0.0597046, -0.0532871, -0.404756), (-0.0597046, 0.910916, 0.404756, -0.0532871), (0.759024, 0.507164, 0.15623, -0.377172), (-0.507164, 0.759024, 0.377172, 0.15623), (0.403752, 0.818729, 0.323885, -0.248526), (-0.818729, 0.403752, 0.248526, 0.323885), (0.407374, 0.0267007, 0.119154, 0.905061), (-0.0267007, 0.407374, -0.905061, 0.119154), (0.339446, 0.226811, -0.349341, 0.843383), (-0.226811, 0.339446, -0.843383, -0.349341), (0.180564, 0.366147, -0.724229, 0.555721), (-0.366147, 0.180564, -0.555721, -0.724229), (0.669581, 0.227292, -0.612372, -0.353553), (-0.227292, 0.669581, 0.353553, -0.612372), (0.466228, 0.531631, -0.353553, -0.612372), (-0.531631, 0.466228, 0.612372, -0.353553), (0.13795, 0.69352, -1.29893e-16, -0.707107), (-0.69352, 0.13795, 0.707107, -1.73191e-16), (0.669581, 0.227292, 0.612372, 0.353553), (-0.227292, 0.669581, -0.353553, 0.612372), (0.466228, 0.531631, 0.353553, 0.612372), (-0.531631, 0.466228, -0.612372, 0.353553), (0.13795, 0.69352, 4.32978e-17, 0.707107), (-0.69352, 0.13795, -0.707107, 8.65956e-17), (0.864425, 0.293433, -0.323885, 0.248526), (-0.293433, 0.864425, -0.248526, -0.323885), (0.601898, 0.686333, -0.404756, 0.0532871), (-0.686333, 0.601898, -0.0532871, -0.404756), (0.178092, 0.89533, -0.377172, -0.15623), (-0.89533, 0.178092, 0.15623, -0.377172), (0.386583, 0.131227, 0.724229, -0.555721), (-0.131227, 0.386583, 0.555721, 0.724229), (0.269177, 0.306937, 0.905061, -0.119154), (-0.306937, 0.269177, 0.119154, 0.905061), (0.0796453, 0.400404, 0.843383, 0.349341), (-0.400404, 0.0796453, -0.349341, 0.843383), (0.386583, 0.131227, -0.724229, 0.555721), (-0.131227, 0.386583, -0.555721, -0.724229), (0.269177, 0.306937, -0.905061, 0.119154), (-0.306937, 0.269177, -0.119154, -0.905061), (0.0796453, 0.400404, -0.843383, -0.349341), (-0.400404, 0.0796453, 0.349341, -0.843383), (0.864425, 0.293433, 0.323885, -0.248526), (-0.293433, 0.864425, 0.248526, 0.323885), (0.601898, 0.686333, 0.404756, -0.0532871), (-0.686333, 0.601898, 0.0532871, 0.404756), (0.178092, 0.89533, 0.377172, 0.15623), (-0.89533, 0.178092, -0.15623, 0.377172), (0.669581, 0.227292, -0.353553, 0.612372), (-0.227292, 0.669581, -0.612372, -0.353553), (0.466228, 0.531631, -0.612372, 0.353553), (-0.531631, 0.466228, -0.353553, -0.612372), (0.13795, 0.69352, -0.707107, 8.65956e-17), (-0.69352, 0.13795, -1.29893e-16, -0.707107), (0.669581, 0.227292, 0.353553, -0.612372), (-0.227292, 0.669581, 0.612372, 0.353553), (0.466228, 0.531631, 0.612372, -0.353553), (-0.531631, 0.466228, 0.353553, 0.612372), (0.13795, 0.69352, 0.707107, -1.73191e-16), (-0.69352, 0.13795, 2.16489e-16, 0.707107), (0.386583, 0.131227, -0.555721, -0.724229), (-0.131227, 0.386583, 0.724229, -0.555721), (0.269177, 0.306937, -0.119154, -0.905061), (-0.306937, 0.269177, 0.905061, -0.119154), (0.0796453, 0.400404, 0.349341, -0.843383), (-0.400404, 0.0796453, 0.843383, 0.349341), (0.864425, 0.293433, 0.248526, 0.323885), (-0.293433, 0.864425, -0.323885, 0.248526), (0.601898, 0.686333, 0.0532871, 0.404756), (-0.686333, 0.601898, -0.404756, 0.0532871), (0.178092, 0.89533, -0.15623, 0.377172), (-0.89533, 0.178092, -0.377172, -0.15623), (0.864425, 0.293433, -0.248526, -0.323885), (-0.293433, 0.864425, 0.323885, -0.248526), (0.601898, 0.686333, -0.0532871, -0.404756), (-0.686333, 0.601898, 0.404756, -0.0532871), (0.178092, 0.89533, 0.15623, -0.377172), (-0.89533, 0.178092, 0.377172, 0.15623), (0.386583, 0.131227, 0.555721, 0.724229), (-0.131227, 0.386583, -0.724229, 0.555721), (0.269177, 0.306937, 0.119154, 0.905061), (-0.306937, 0.269177, -0.905061, 0.119154), (0.0796453, 0.400404, -0.349341, 0.843383), (-0.400404, 0.0796453, -0.843383, -0.349341), (0.669581, 0.227292, -0.683013, 0.183013), (-0.227292, 0.669581, -0.183013, -0.683013), (0.466228, 0.531631, -0.683013, -0.183013), (-0.531631, 0.466228, 0.183013, -0.683013), (0.13795, 0.69352, -0.5, -0.5), (-0.69352, 0.13795, 0.5, -0.5), (0.669581, 0.227292, 0.683013, -0.183013), (-0.227292, 0.669581, 0.183013, 0.683013), (0.466228, 0.531631, 0.683013, 0.183013), (-0.531631, 0.466228, -0.183013, 0.683013), (0.13795, 0.69352, 0.5, 0.5), (-0.69352, 0.13795, -0.5, 0.5), (0.864425, 0.293433, -0.404756, -0.0532871), (-0.293433, 0.864425, 0.0532871, -0.404756), (0.601898, 0.686333, -0.323885, -0.248526), (-0.686333, 0.601898, 0.248526, -0.323885), (0.178092, 0.89533, -0.15623, -0.377172), (-0.89533, 0.178092, 0.377172, -0.15623), (0.386583, 0.131227, 0.905061, 0.119154), (-0.131227, 0.386583, -0.119154, 0.905061), (0.269177, 0.306937, 0.724229, 0.555721), (-0.306937, 0.269177, -0.555721, 0.724229), (0.0796453, 0.400404, 0.349341, 0.843383), (-0.400404, 0.0796453, -0.843383, 0.349341), (0.386583, 0.131227, -0.905061, -0.119154), (-0.131227, 0.386583, 0.119154, -0.905061), (0.269177, 0.306937, -0.724229, -0.555721), (-0.306937, 0.269177, 0.555721, -0.724229), (0.0796453, 0.400404, -0.349341, -0.843383), (-0.400404, 0.0796453, 0.843383, -0.349341), (0.864425, 0.293433, 0.404756, 0.0532871), (-0.293433, 0.864425, -0.0532871, 0.404756), (0.601898, 0.686333, 0.323885, 0.248526), (-0.686333, 0.601898, -0.248526, 0.323885), (0.178092, 0.89533, 0.15623, 0.377172), (-0.89533, 0.178092, -0.377172, 0.15623), (0.669581, 0.227292, 0.183013, 0.683013), (-0.227292, 0.669581, -0.683013, 0.183013), (0.466228, 0.531631, -0.183013, 0.683013), (-0.531631, 0.466228, -0.683013, -0.183013), (0.13795, 0.69352, -0.5, 0.5), (-0.69352, 0.13795, -0.5, -0.5), (0.669581, 0.227292, -0.183013, -0.683013), (-0.227292, 0.669581, 0.683013, -0.183013), (0.466228, 0.531631, 0.183013, -0.683013), (-0.531631, 0.466228, 0.683013, 0.183013), (0.13795, 0.69352, 0.5, -0.5), (-0.69352, 0.13795, 0.5, 0.5), (0.386583, 0.131227, 0.119154, -0.905061), (-0.131227, 0.386583, 0.905061, 0.119154), (0.269177, 0.306937, 0.555721, -0.724229), (-0.306937, 0.269177, 0.724229, 0.555721), (0.0796453, 0.400404, 0.843383, -0.349341), (-0.400404, 0.0796453, 0.349341, 0.843383), (0.864425, 0.293433, -0.0532871, 0.404756), (-0.293433, 0.864425, -0.404756, -0.0532871), (0.601898, 0.686333, -0.248526, 0.323885), (-0.686333, 0.601898, -0.323885, -0.248526), (0.178092, 0.89533, -0.377172, 0.15623), (-0.89533, 0.178092, -0.15623, -0.377172), (0.864425, 0.293433, 0.0532871, -0.404756), (-0.293433, 0.864425, 0.404756, 0.0532871), (0.601898, 0.686333, 0.248526, -0.323885), (-0.686333, 0.601898, 0.323885, 0.248526), (0.178092, 0.89533, 0.377172, -0.15623), (-0.89533, 0.178092, 0.15623, 0.377172), (0.386583, 0.131227, -0.119154, 0.905061), (-0.131227, 0.386583, -0.905061, -0.119154), (0.269177, 0.306937, -0.555721, 0.724229), (-0.306937, 0.269177, -0.724229, -0.555721), (0.0796453, 0.400404, -0.843383, 0.349341), (-0.400404, 0.0796453, -0.349341, -0.843383), (0.705593, 0.046247, -0.683013, -0.183013), (-0.046247, 0.705593, 0.183013, -0.683013), (0.587938, 0.392847, -0.5, -0.5), (-0.392847, 0.587938, 0.5, -0.5), (0.312745, 0.634185, -0.183013, -0.683013), (-0.634185, 0.312745, 0.683013, -0.183013), (0.705593, 0.046247, 0.683013, 0.183013), (-0.046247, 0.705593, -0.183013, 0.683013), (0.587938, 0.392847, 0.5, 0.5), (-0.392847, 0.587938, -0.5, 0.5), (0.312745, 0.634185, 0.183013, 0.683013), (-0.634185, 0.312745, -0.683013, 0.183013), (0.910916, 0.0597046, -0.248526, 0.323885), (-0.0597046, 0.910916, -0.323885, -0.248526), (0.759024, 0.507164, -0.377172, 0.15623), (-0.507164, 0.759024, -0.15623, -0.377172), (0.403752, 0.818729, -0.404756, -0.0532871), (-0.818729, 0.403752, 0.0532871, -0.404756), (0.407374, 0.0267007, 0.555721, -0.724229), (-0.0267007, 0.407374, 0.724229, 0.555721), (0.339446, 0.226811, 0.843383, -0.349341), (-0.226811, 0.339446, 0.349341, 0.843383), (0.180564, 0.366147, 0.905061, 0.119154), (-0.366147, 0.180564, -0.119154, 0.905061), (0.407374, 0.0267007, -0.555721, 0.724229), (-0.0267007, 0.407374, -0.724229, -0.555721), (0.339446, 0.226811, -0.843383, 0.349341), (-0.226811, 0.339446, -0.349341, -0.843383), (0.180564, 0.366147, -0.905061, -0.119154), (-0.366147, 0.180564, 0.119154, -0.905061), (0.910916, 0.0597046, 0.248526, -0.323885), (-0.0597046, 0.910916, 0.323885, 0.248526), (0.759024, 0.507164, 0.377172, -0.15623), (-0.507164, 0.759024, 0.15623, 0.377172), (0.403752, 0.818729, 0.404756, 0.0532871), (-0.818729, 0.403752, -0.0532871, 0.404756), (0.705593, 0.046247, -0.183013, 0.683013), (-0.046247, 0.705593, -0.683013, -0.183013), (0.587938, 0.392847, -0.5, 0.5), (-0.392847, 0.587938, -0.5, -0.5), (0.312745, 0.634185, -0.683013, 0.183013), (-0.634185, 0.312745, -0.183013, -0.683013), (0.705593, 0.046247, 0.183013, -0.683013), (-0.046247, 0.705593, 0.683013, 0.183013), (0.587938, 0.392847, 0.5, -0.5), (-0.392847, 0.587938, 0.5, 0.5), (0.312745, 0.634185, 0.683013, -0.183013), (-0.634185, 0.312745, 0.183013, 0.683013), (0.407374, 0.0267007, -0.724229, -0.555721), (-0.0267007, 0.407374, 0.555721, -0.724229), (0.339446, 0.226811, -0.349341, -0.843383), (-0.226811, 0.339446, 0.843383, -0.349341), (0.180564, 0.366147, 0.119154, -0.905061), (-0.366147, 0.180564, 0.905061, 0.119154), (0.910916, 0.0597046, 0.323885, 0.248526), (-0.0597046, 0.910916, -0.248526, 0.323885), (0.759024, 0.507164, 0.15623, 0.377172), (-0.507164, 0.759024, -0.377172, 0.15623), (0.403752, 0.818729, -0.0532871, 0.404756), (-0.818729, 0.403752, -0.404756, -0.0532871), (0.910916, 0.0597046, -0.323885, -0.248526), (-0.0597046, 0.910916, 0.248526, -0.323885), (0.759024, 0.507164, -0.15623, -0.377172), (-0.507164, 0.759024, 0.377172, -0.15623), (0.403752, 0.818729, 0.0532871, -0.404756), (-0.818729, 0.403752, 0.404756, 0.0532871), (0.407374, 0.0267007, 0.724229, 0.555721), (-0.0267007, 0.407374, -0.555721, 0.724229), (0.339446, 0.226811, 0.349341, 0.843383), (-0.226811, 0.339446, -0.843383, 0.349341), (0.180564, 0.366147, -0.119154, 0.905061), (-0.366147, 0.180564, -0.905061, -0.119154), (0.800808, 0.159291, -0.533402, -0.220942), (-0.159291, 0.800808, 0.220942, -0.533402), (0.613875, 0.538354, -0.351469, -0.458043), (-0.538354, 0.613875, 0.458043, -0.351469), (0.262454, 0.773165, -0.0753593, -0.572411), (-0.773165, 0.262454, 0.572411, -0.0753593), (0.566257, 0.112635, 0.754344, 0.31246), (-0.112635, 0.566257, -0.31246, 0.754344), (0.434075, 0.380673, 0.497052, 0.64777), (-0.380673, 0.434075, -0.64777, 0.497052), (0.185583, 0.54671, 0.106574, 0.809511), (-0.54671, 0.185583, -0.809511, 0.106574), (0.960135, 0.190983, -0.113405, 0.169723), (-0.190983, 0.960135, -0.169723, -0.113405), (0.73601, 0.645463, -0.183073, 0.0902818), (-0.645463, 0.73601, -0.0902818, -0.183073), (0.314672, 0.926993, -0.203687, -0.0133504), (-0.926993, 0.314672, 0.0133504, -0.203687), (0.566257, 0.112635, 0.57735, -0.57735), (-0.112635, 0.566257, 0.57735, 0.57735), (0.434075, 0.380673, 0.788675, -0.211325), (-0.380673, 0.434075, 0.211325, 0.788675), (0.185583, 0.54671, 0.788675, 0.211325), (-0.54671, 0.185583, -0.211325, 0.788675), (0.566257, 0.112635, -0.57735, 0.57735), (-0.112635, 0.566257, -0.57735, -0.57735), (0.434075, 0.380673, -0.788675, 0.211325), (-0.380673, 0.434075, -0.211325, -0.788675), (0.185583, 0.54671, -0.788675, -0.211325), (-0.54671, 0.185583, 0.211325, -0.788675), (0.960135, 0.190983, 0.113405, -0.169723), (-0.190983, 0.960135, 0.169723, 0.113405), (0.73601, 0.645463, 0.183073, -0.0902818), (-0.645463, 0.73601, 0.0902818, 0.183073), (0.314672, 0.926993, 0.203687, 0.0133504), (-0.926993, 0.314672, -0.0133504, 0.203687), (0.566257, 0.112635, -0.31246, 0.754344), (-0.112635, 0.566257, -0.754344, -0.31246), (0.434075, 0.380673, -0.64777, 0.497052), (-0.380673, 0.434075, -0.497052, -0.64777), (0.185583, 0.54671, -0.809511, 0.106574), (-0.54671, 0.185583, -0.106574, -0.809511), (0.800808, 0.159291, 0.220942, -0.533402), (-0.159291, 0.800808, 0.533402, 0.220942), (0.613875, 0.538354, 0.458043, -0.351469), (-0.538354, 0.613875, 0.351469, 0.458043), (0.262454, 0.773165, 0.572411, -0.0753593), (-0.773165, 0.262454, 0.0753593, 0.572411), (0.200202, 0.0398226, -0.813963, -0.543873), (-0.0398226, 0.200202, 0.543873, -0.813963), (0.153469, 0.134588, -0.432976, -0.877989), (-0.134588, 0.153469, 0.877989, -0.432976), (0.0656136, 0.193291, 0.0640261, -0.976849), (-0.193291, 0.0656136, 0.976849, 0.0640261), (0.800808, 0.159291, 0.408248, 0.408248), (-0.159291, 0.800808, -0.408248, 0.408248), (0.613875, 0.538354, 0.149429, 0.557678), (-0.538354, 0.613875, -0.557678, 0.149429), (0.262454, 0.773165, -0.149429, 0.557678), (-0.773165, 0.262454, -0.557678, -0.149429), (0.800808, 0.159291, -0.408248, -0.408248), (-0.159291, 0.800808, 0.408248, -0.408248), (0.613875, 0.538354, -0.149429, -0.557678), (-0.538354, 0.613875, 0.557678, -0.149429), (0.262454, 0.773165, 0.149429, -0.557678), (-0.773165, 0.262454, 0.557678, 0.149429), (0.200202, 0.0398226, 0.813963, 0.543873), (-0.0398226, 0.200202, -0.543873, 0.813963), (0.153469, 0.134588, 0.432976, 0.877989), (-0.134588, 0.153469, -0.877989, 0.432976), (0.0656136, 0.193291, -0.0640261, 0.976849), (-0.193291, 0.0656136, -0.976849, -0.0640261), (0.51781, 0.255355, -0.64777, -0.497052), (-0.255355, 0.51781, 0.497052, -0.64777), (0.320759, 0.480049, -0.31246, -0.754344), (-0.480049, 0.320759, 0.754344, -0.31246), (0.0377605, 0.576114, 0.106574, -0.809511), (-0.576114, 0.0377605, 0.809511, 0.106574), (0.732294, 0.361127, 0.458043, 0.351469), (-0.361127, 0.732294, -0.351469, 0.458043), (0.453621, 0.678892, 0.220942, 0.533402), (-0.678892, 0.453621, -0.533402, 0.220942), (0.0534014, 0.814748, -0.0753593, 0.572411), (-0.814748, 0.0534014, -0.572411, -0.0753593), (0.732294, 0.361127, -0.5, 0.288675), (-0.361127, 0.732294, -0.288675, -0.5), (0.453621, 0.678892, -0.57735, 7.0705e-17), (-0.678892, 0.453621, -1.06058e-16, -0.57735), (0.0534014, 0.814748, -0.5, -0.288675), (-0.814748, 0.0534014, 0.288675, -0.5), (0.183073, 0.0902818, 0.73601, -0.645463), (-0.0902818, 0.183073, 0.645463, 0.73601), (0.113405, 0.169723, 0.960135, -0.190983), (-0.169723, 0.113405, 0.190983, 0.960135), (0.0133504, 0.203687, 0.926993, 0.314672), (-0.203687, 0.0133504, -0.314672, 0.926993), (0.183073, 0.0902818, -0.73601, 0.645463), (-0.0902818, 0.183073, -0.645463, -0.73601), (0.113405, 0.169723, -0.960135, 0.190983), (-0.169723, 0.113405, -0.190983, -0.960135), (0.0133504, 0.203687, -0.926993, -0.314672), (-0.203687, 0.0133504, 0.314672, -0.926993), (0.732294, 0.361127, 0.5, -0.288675), (-0.361127, 0.732294, 0.288675, 0.5), (0.453621, 0.678892, 0.57735, -1.4141e-16), (-0.678892, 0.453621, 1.76763e-16, 0.57735), (0.0534014, 0.814748, 0.5, 0.288675), (-0.814748, 0.0534014, -0.288675, 0.5), (0.732294, 0.361127, -0.351469, 0.458043), (-0.361127, 0.732294, -0.458043, -0.351469), (0.453621, 0.678892, -0.533402, 0.220942), (-0.678892, 0.453621, -0.220942, -0.533402), (0.0534014, 0.814748, -0.572411, -0.0753593), (-0.814748, 0.0534014, 0.0753593, -0.572411), (0.51781, 0.255355, 0.497052, -0.64777), (-0.255355, 0.51781, 0.64777, 0.497052), (0.320759, 0.480049, 0.754344, -0.31246), (-0.480049, 0.320759, 0.31246, 0.754344), (0.0377605, 0.576114, 0.809511, 0.106574), (-0.576114, 0.0377605, -0.106574, 0.809511), (0.51781, 0.255355, -0.408248, -0.707107), (-0.255355, 0.51781, 0.707107, -0.408248), (0.320759, 0.480049, -1.49988e-16, -0.816497), (-0.480049, 0.320759, 0.816497, -1.99984e-16), (0.0377605, 0.576114, 0.408248, -0.707107), (-0.576114, 0.0377605, 0.707107, 0.408248), (0.877989, 0.432976, 0.134588, 0.153469), (-0.432976, 0.877989, -0.153469, 0.134588), (0.543873, 0.813963, 0.0398226, 0.200202), (-0.813963, 0.543873, -0.200202, 0.0398226), (0.0640261, 0.976849, -0.0656136, 0.193291), (-0.976849, 0.0640261, -0.193291, -0.0656136), (0.877989, 0.432976, -0.134588, -0.153469), (-0.432976, 0.877989, 0.153469, -0.134588), (0.543873, 0.813963, -0.0398226, -0.200202), (-0.813963, 0.543873, 0.200202, -0.0398226), (0.0640261, 0.976849, 0.0656136, -0.193291), (-0.976849, 0.0640261, 0.193291, 0.0656136), (0.51781, 0.255355, 0.408248, 0.707107), (-0.255355, 0.51781, -0.707107, 0.408248), (0.320759, 0.480049, 4.9996e-17, 0.816497), (-0.480049, 0.320759, -0.816497, 9.9992e-17), (0.0377605, 0.576114, -0.408248, 0.707107), (-0.576114, 0.0377605, -0.707107, -0.408248), (0.732294, 0.361127, -0.458043, -0.351469), (-0.361127, 0.732294, 0.351469, -0.458043), (0.453621, 0.678892, -0.220942, -0.533402), (-0.678892, 0.453621, 0.533402, -0.220942), (0.0534014, 0.814748, 0.0753593, -0.572411), (-0.814748, 0.0534014, 0.572411, 0.0753593), (0.51781, 0.255355, 0.64777, 0.497052), (-0.255355, 0.51781, -0.497052, 0.64777), (0.320759, 0.480049, 0.31246, 0.754344), (-0.480049, 0.320759, -0.754344, 0.31246), (0.0377605, 0.576114, -0.106574, 0.809511), (-0.576114, 0.0377605, -0.809511, -0.106574), (0.877989, 0.432976, -0.153469, 0.134588), (-0.432976, 0.877989, -0.134588, -0.153469), (0.543873, 0.813963, -0.200202, 0.0398226), (-0.813963, 0.543873, -0.0398226, -0.200202), (0.0640261, 0.976849, -0.193291, -0.0656136), (-0.976849, 0.0640261, 0.0656136, -0.193291), (0.51781, 0.255355, 0.707107, -0.408248), (-0.255355, 0.51781, 0.408248, 0.707107), (0.320759, 0.480049, 0.816497, -1.99984e-16), (-0.480049, 0.320759, 2.4998e-16, 0.816497), (0.0377605, 0.576114, 0.707107, 0.408248), (-0.576114, 0.0377605, -0.408248, 0.707107), (0.51781, 0.255355, -0.707107, 0.408248), (-0.255355, 0.51781, -0.408248, -0.707107), (0.320759, 0.480049, -0.816497, 9.9992e-17), (-0.480049, 0.320759, -1.49988e-16, -0.816497), (0.0377605, 0.576114, -0.707107, -0.408248), (-0.576114, 0.0377605, 0.408248, -0.707107), (0.877989, 0.432976, 0.153469, -0.134588), (-0.432976, 0.877989, 0.134588, 0.153469), (0.543873, 0.813963, 0.200202, -0.0398226), (-0.813963, 0.543873, 0.0398226, 0.200202), (0.0640261, 0.976849, 0.193291, 0.0656136), (-0.976849, 0.0640261, -0.0656136, 0.193291), (0.51781, 0.255355, -0.497052, 0.64777), (-0.255355, 0.51781, -0.64777, -0.497052), (0.320759, 0.480049, -0.754344, 0.31246), (-0.480049, 0.320759, -0.31246, -0.754344), (0.0377605, 0.576114, -0.809511, -0.106574), (-0.576114, 0.0377605, 0.106574, -0.809511), (0.732294, 0.361127, 0.351469, -0.458043), (-0.361127, 0.732294, 0.458043, 0.351469), (0.453621, 0.678892, 0.533402, -0.220942), (-0.678892, 0.453621, 0.220942, 0.533402), (0.0534014, 0.814748, 0.572411, 0.0753593), (-0.814748, 0.0534014, -0.0753593, 0.572411), (0.183073, 0.0902818, -0.645463, -0.73601), (-0.0902818, 0.183073, 0.73601, -0.645463), (0.113405, 0.169723, -0.190983, -0.960135), (-0.169723, 0.113405, 0.960135, -0.190983), (0.0133504, 0.203687, 0.314672, -0.926993), (-0.203687, 0.0133504, 0.926993, 0.314672), (0.732294, 0.361127, 0.288675, 0.5), (-0.361127, 0.732294, -0.5, 0.288675), (0.453621, 0.678892, 3.53525e-17, 0.57735), (-0.678892, 0.453621, -0.57735, 7.0705e-17), (0.0534014, 0.814748, -0.288675, 0.5), (-0.814748, 0.0534014, -0.5, -0.288675), (0.732294, 0.361127, -0.288675, -0.5), (-0.361127, 0.732294, 0.5, -0.288675), (0.453621, 0.678892, -1.06058e-16, -0.57735), (-0.678892, 0.453621, 0.57735, -1.4141e-16), (0.0534014, 0.814748, 0.288675, -0.5), (-0.814748, 0.0534014, 0.5, 0.288675), (0.183073, 0.0902818, 0.645463, 0.73601), (-0.0902818, 0.183073, -0.73601, 0.645463), (0.113405, 0.169723, 0.190983, 0.960135), (-0.169723, 0.113405, -0.960135, 0.190983), (0.0133504, 0.203687, -0.314672, 0.926993), (-0.203687, 0.0133504, -0.926993, -0.314672), (0.566257, 0.112635, -0.754344, -0.31246), (-0.112635, 0.566257, 0.31246, -0.754344), (0.434075, 0.380673, -0.497052, -0.64777), (-0.380673, 0.434075, 0.64777, -0.497052), (0.185583, 0.54671, -0.106574, -0.809511), (-0.54671, 0.185583, 0.809511, -0.106574), (0.800808, 0.159291, 0.533402, 0.220942), (-0.159291, 0.800808, -0.220942, 0.533402), (0.613875, 0.538354, 0.351469, 0.458043), (-0.538354, 0.613875, -0.458043, 0.351469), (0.262454, 0.773165, 0.0753593, 0.572411), (-0.773165, 0.262454, -0.572411, 0.0753593), (0.800808, 0.159291, -0.408248, 0.408248), (-0.159291, 0.800808, -0.408248, -0.408248), (0.613875, 0.538354, -0.557678, 0.149429), (-0.538354, 0.613875, -0.149429, -0.557678), (0.262454, 0.773165, -0.557678, -0.149429), (-0.773165, 0.262454, 0.149429, -0.557678), (0.200202, 0.0398226, 0.543873, -0.813963), (-0.0398226, 0.200202, 0.813963, 0.543873), (0.153469, 0.134588, 0.877989, -0.432976), (-0.134588, 0.153469, 0.432976, 0.877989), (0.0656136, 0.193291, 0.976849, 0.0640261), (-0.193291, 0.0656136, -0.0640261, 0.976849), (0.200202, 0.0398226, -0.543873, 0.813963), (-0.0398226, 0.200202, -0.813963, -0.543873), (0.153469, 0.134588, -0.877989, 0.432976), (-0.134588, 0.153469, -0.432976, -0.877989), (0.0656136, 0.193291, -0.976849, -0.0640261), (-0.193291, 0.0656136, 0.0640261, -0.976849), (0.800808, 0.159291, 0.408248, -0.408248), (-0.159291, 0.800808, 0.408248, 0.408248), (0.613875, 0.538354, 0.557678, -0.149429), (-0.538354, 0.613875, 0.149429, 0.557678), (0.262454, 0.773165, 0.557678, 0.149429), (-0.773165, 0.262454, -0.149429, 0.557678), (0.800808, 0.159291, -0.220942, 0.533402), (-0.159291, 0.800808, -0.533402, -0.220942), (0.613875, 0.538354, -0.458043, 0.351469), (-0.538354, 0.613875, -0.351469, -0.458043), (0.262454, 0.773165, -0.572411, 0.0753593), (-0.773165, 0.262454, -0.0753593, -0.572411), (0.566257, 0.112635, 0.31246, -0.754344), (-0.112635, 0.566257, 0.754344, 0.31246), (0.434075, 0.380673, 0.64777, -0.497052), (-0.380673, 0.434075, 0.497052, 0.64777), (0.185583, 0.54671, 0.809511, -0.106574), (-0.54671, 0.185583, 0.106574, 0.809511), (0.566257, 0.112635, -0.57735, -0.57735), (-0.112635, 0.566257, 0.57735, -0.57735), (0.434075, 0.380673, -0.211325, -0.788675), (-0.380673, 0.434075, 0.788675, -0.211325), (0.185583, 0.54671, 0.211325, -0.788675), (-0.54671, 0.185583, 0.788675, 0.211325), (0.960135, 0.190983, 0.169723, 0.113405), (-0.190983, 0.960135, -0.113405, 0.169723), (0.73601, 0.645463, 0.0902818, 0.183073), (-0.645463, 0.73601, -0.183073, 0.0902818), (0.314672, 0.926993, -0.0133504, 0.203687), (-0.926993, 0.314672, -0.203687, -0.0133504), (0.960135, 0.190983, -0.169723, -0.113405), (-0.190983, 0.960135, 0.113405, -0.169723), (0.73601, 0.645463, -0.0902818, -0.183073), (-0.645463, 0.73601, 0.183073, -0.0902818), (0.314672, 0.926993, 0.0133504, -0.203687), (-0.926993, 0.314672, 0.203687, 0.0133504), (0.566257, 0.112635, 0.57735, 0.57735), (-0.112635, 0.566257, -0.57735, 0.57735), (0.434075, 0.380673, 0.211325, 0.788675), (-0.380673, 0.434075, -0.788675, 0.211325), (0.185583, 0.54671, -0.211325, 0.788675), (-0.54671, 0.185583, -0.788675, -0.211325), (0.69352, 0.13795, -0.707107, 8.65956e-17), (-0.13795, 0.69352, -1.29893e-16, -0.707107), (0.531631, 0.466228, -0.612372, -0.353553), (-0.466228, 0.531631, 0.353553, -0.612372), (0.227292, 0.669581, -0.353553, -0.612372), (-0.669581, 0.227292, 0.612372, -0.353553), (0.69352, 0.13795, 0.707107, -1.73191e-16), (-0.13795, 0.69352, 2.16489e-16, 0.707107), (0.531631, 0.466228, 0.612372, 0.353553), (-0.466228, 0.531631, -0.353553, 0.612372), (0.227292, 0.669581, 0.353553, 0.612372), (-0.669581, 0.227292, -0.612372, 0.353553), (0.89533, 0.178092, -0.377172, 0.15623), (-0.178092, 0.89533, -0.15623, -0.377172), (0.686333, 0.601898, -0.404756, -0.0532871), (-0.601898, 0.686333, 0.0532871, -0.404756), (0.293433, 0.864425, -0.323885, -0.248526), (-0.864425, 0.293433, 0.248526, -0.323885), (0.400404, 0.0796453, 0.843383, -0.349341), (-0.0796453, 0.400404, 0.349341, 0.843383), (0.306937, 0.269177, 0.905061, 0.119154), (-0.269177, 0.306937, -0.119154, 0.905061), (0.131227, 0.386583, 0.724229, 0.555721), (-0.386583, 0.131227, -0.555721, 0.724229), (0.400404, 0.0796453, -0.843383, 0.349341), (-0.0796453, 0.400404, -0.349341, -0.843383), (0.306937, 0.269177, -0.905061, -0.119154), (-0.269177, 0.306937, 0.119154, -0.905061), (0.131227, 0.386583, -0.724229, -0.555721), (-0.386583, 0.131227, 0.555721, -0.724229), (0.89533, 0.178092, 0.377172, -0.15623), (-0.178092, 0.89533, 0.15623, 0.377172), (0.686333, 0.601898, 0.404756, 0.0532871), (-0.601898, 0.686333, -0.0532871, 0.404756), (0.293433, 0.864425, 0.323885, 0.248526), (-0.864425, 0.293433, -0.248526, 0.323885), (0.69352, 0.13795, 4.32978e-17, 0.707107), (-0.13795, 0.69352, -0.707107, 8.65956e-17), (0.531631, 0.466228, -0.353553, 0.612372), (-0.466228, 0.531631, -0.612372, -0.353553), (0.227292, 0.669581, -0.612372, 0.353553), (-0.669581, 0.227292, -0.353553, -0.612372), (0.69352, 0.13795, -1.29893e-16, -0.707107), (-0.13795, 0.69352, 0.707107, -1.73191e-16), (0.531631, 0.466228, 0.353553, -0.612372), (-0.466228, 0.531631, 0.612372, 0.353553), (0.227292, 0.669581, 0.612372, -0.353553), (-0.669581, 0.227292, 0.353553, 0.612372), (0.400404, 0.0796453, -0.349341, -0.843383), (-0.0796453, 0.400404, 0.843383, -0.349341), (0.306937, 0.269177, 0.119154, -0.905061), (-0.269177, 0.306937, 0.905061, 0.119154), (0.131227, 0.386583, 0.555721, -0.724229), (-0.386583, 0.131227, 0.724229, 0.555721), (0.89533, 0.178092, 0.15623, 0.377172), (-0.178092, 0.89533, -0.377172, 0.15623), (0.686333, 0.601898, -0.0532871, 0.404756), (-0.601898, 0.686333, -0.404756, -0.0532871), (0.293433, 0.864425, -0.248526, 0.323885), (-0.864425, 0.293433, -0.323885, -0.248526), (0.89533, 0.178092, -0.15623, -0.377172), (-0.178092, 0.89533, 0.377172, -0.15623), (0.686333, 0.601898, 0.0532871, -0.404756), (-0.601898, 0.686333, 0.404756, 0.0532871), (0.293433, 0.864425, 0.248526, -0.323885), (-0.864425, 0.293433, 0.323885, 0.248526), (0.400404, 0.0796453, 0.349341, 0.843383), (-0.0796453, 0.400404, -0.843383, 0.349341), (0.306937, 0.269177, -0.119154, 0.905061), (-0.269177, 0.306937, -0.905061, -0.119154), (0.131227, 0.386583, -0.555721, 0.724229), (-0.386583, 0.131227, -0.724229, -0.555721), (0.634185, 0.312745, -0.353553, -0.612372), (-0.312745, 0.634185, 0.612372, -0.353553), (0.392847, 0.587938, -1.29893e-16, -0.707107), (-0.587938, 0.392847, 0.707107, -1.73191e-16), (0.046247, 0.705593, 0.353553, -0.612372), (-0.705593, 0.046247, 0.612372, 0.353553), (0.634185, 0.312745, 0.353553, 0.612372), (-0.312745, 0.634185, -0.612372, 0.353553), (0.392847, 0.587938, 4.32978e-17, 0.707107), (-0.587938, 0.392847, -0.707107, 8.65956e-17), (0.046247, 0.705593, -0.353553, 0.612372), (-0.705593, 0.046247, -0.612372, -0.353553), (0.818729, 0.403752, -0.248526, 0.323885), (-0.403752, 0.818729, -0.323885, -0.248526), (0.507164, 0.759024, -0.377172, 0.15623), (-0.759024, 0.507164, -0.15623, -0.377172), (0.0597046, 0.910916, -0.404756, -0.0532871), (-0.910916, 0.0597046, 0.0532871, -0.404756), (0.366147, 0.180564, 0.555721, -0.724229), (-0.180564, 0.366147, 0.724229, 0.555721), (0.226811, 0.339446, 0.843383, -0.349341), (-0.339446, 0.226811, 0.349341, 0.843383), (0.0267007, 0.407374, 0.905061, 0.119154), (-0.407374, 0.0267007, -0.119154, 0.905061), (0.366147, 0.180564, -0.555721, 0.724229), (-0.180564, 0.366147, -0.724229, -0.555721), (0.226811, 0.339446, -0.843383, 0.349341), (-0.339446, 0.226811, -0.349341, -0.843383), (0.0267007, 0.407374, -0.905061, -0.119154), (-0.407374, 0.0267007, 0.119154, -0.905061), (0.818729, 0.403752, 0.248526, -0.323885), (-0.403752, 0.818729, 0.323885, 0.248526), (0.507164, 0.759024, 0.377172, -0.15623), (-0.759024, 0.507164, 0.15623, 0.377172), (0.0597046, 0.910916, 0.404756, 0.0532871), (-0.910916, 0.0597046, -0.0532871, 0.404756), (0.634185, 0.312745, -0.612372, 0.353553), (-0.312745, 0.634185, -0.353553, -0.612372), (0.392847, 0.587938, -0.707107, 8.65956e-17), (-0.587938, 0.392847, -1.29893e-16, -0.707107), (0.046247, 0.705593, -0.612372, -0.353553), (-0.705593, 0.046247, 0.353553, -0.612372), (0.634185, 0.312745, 0.612372, -0.353553), (-0.312745, 0.634185, 0.353553, 0.612372), (0.392847, 0.587938, 0.707107, -1.73191e-16), (-0.587938, 0.392847, 2.16489e-16, 0.707107), (0.046247, 0.705593, 0.612372, 0.353553), (-0.705593, 0.046247, -0.353553, 0.612372), (0.366147, 0.180564, -0.724229, -0.555721), (-0.180564, 0.366147, 0.555721, -0.724229), (0.226811, 0.339446, -0.349341, -0.843383), (-0.339446, 0.226811, 0.843383, -0.349341), (0.0267007, 0.407374, 0.119154, -0.905061), (-0.407374, 0.0267007, 0.905061, 0.119154), (0.818729, 0.403752, 0.323885, 0.248526), (-0.403752, 0.818729, -0.248526, 0.323885), (0.507164, 0.759024, 0.15623, 0.377172), (-0.759024, 0.507164, -0.377172, 0.15623), (0.0597046, 0.910916, -0.0532871, 0.404756), (-0.910916, 0.0597046, -0.404756, -0.0532871), (0.818729, 0.403752, -0.323885, -0.248526), (-0.403752, 0.818729, 0.248526, -0.323885), (0.507164, 0.759024, -0.15623, -0.377172), (-0.759024, 0.507164, 0.377172, -0.15623), (0.0597046, 0.910916, 0.0532871, -0.404756), (-0.910916, 0.0597046, 0.404756, 0.0532871), (0.366147, 0.180564, 0.724229, 0.555721), (-0.180564, 0.366147, -0.555721, 0.724229), (0.226811, 0.339446, 0.349341, 0.843383), (-0.339446, 0.226811, -0.843383, 0.349341), (0.0267007, 0.407374, -0.119154, 0.905061), (-0.407374, 0.0267007, -0.905061, -0.119154), (0.634185, 0.312745, -0.683013, -0.183013), (-0.312745, 0.634185, 0.183013, -0.683013), (0.392847, 0.587938, -0.5, -0.5), (-0.587938, 0.392847, 0.5, -0.5), (0.046247, 0.705593, -0.183013, -0.683013), (-0.705593, 0.046247, 0.683013, -0.183013), (0.634185, 0.312745, 0.683013, 0.183013), (-0.312745, 0.634185, -0.183013, 0.683013), (0.392847, 0.587938, 0.5, 0.5), (-0.587938, 0.392847, -0.5, 0.5), (0.046247, 0.705593, 0.183013, 0.683013), (-0.705593, 0.046247, -0.683013, 0.183013), (0.818729, 0.403752, -0.404756, 0.0532871), (-0.403752, 0.818729, -0.0532871, -0.404756), (0.507164, 0.759024, -0.377172, -0.15623), (-0.759024, 0.507164, 0.15623, -0.377172), (0.0597046, 0.910916, -0.248526, -0.323885), (-0.910916, 0.0597046, 0.323885, -0.248526), (0.366147, 0.180564, 0.905061, -0.119154), (-0.180564, 0.366147, 0.119154, 0.905061), (0.226811, 0.339446, 0.843383, 0.349341), (-0.339446, 0.226811, -0.349341, 0.843383), (0.0267007, 0.407374, 0.555721, 0.724229), (-0.407374, 0.0267007, -0.724229, 0.555721), (0.366147, 0.180564, -0.905061, 0.119154), (-0.180564, 0.366147, -0.119154, -0.905061), (0.226811, 0.339446, -0.843383, -0.349341), (-0.339446, 0.226811, 0.349341, -0.843383), (0.0267007, 0.407374, -0.555721, -0.724229), (-0.407374, 0.0267007, 0.724229, -0.555721), (0.818729, 0.403752, 0.404756, -0.0532871), (-0.403752, 0.818729, 0.0532871, 0.404756), (0.507164, 0.759024, 0.377172, 0.15623), (-0.759024, 0.507164, -0.15623, 0.377172), (0.0597046, 0.910916, 0.248526, 0.323885), (-0.910916, 0.0597046, -0.323885, 0.248526), (0.634185, 0.312745, -0.183013, 0.683013), (-0.312745, 0.634185, -0.683013, -0.183013), (0.392847, 0.587938, -0.5, 0.5), (-0.587938, 0.392847, -0.5, -0.5), (0.046247, 0.705593, -0.683013, 0.183013), (-0.705593, 0.046247, -0.183013, -0.683013), (0.634185, 0.312745, 0.183013, -0.683013), (-0.312745, 0.634185, 0.683013, 0.183013), (0.392847, 0.587938, 0.5, -0.5), (-0.587938, 0.392847, 0.5, 0.5), (0.046247, 0.705593, 0.683013, -0.183013), (-0.705593, 0.046247, 0.183013, 0.683013), (0.366147, 0.180564, -0.119154, -0.905061), (-0.180564, 0.366147, 0.905061, -0.119154), (0.226811, 0.339446, 0.349341, -0.843383), (-0.339446, 0.226811, 0.843383, 0.349341), (0.0267007, 0.407374, 0.724229, -0.555721), (-0.407374, 0.0267007, 0.555721, 0.724229), (0.818729, 0.403752, 0.0532871, 0.404756), (-0.403752, 0.818729, -0.404756, 0.0532871), (0.507164, 0.759024, -0.15623, 0.377172), (-0.759024, 0.507164, -0.377172, -0.15623), (0.0597046, 0.910916, -0.323885, 0.248526), (-0.910916, 0.0597046, -0.248526, -0.323885), (0.818729, 0.403752, -0.0532871, -0.404756), (-0.403752, 0.818729, 0.404756, -0.0532871), (0.507164, 0.759024, 0.15623, -0.377172), (-0.759024, 0.507164, 0.377172, 0.15623), (0.0597046, 0.910916, 0.323885, -0.248526), (-0.910916, 0.0597046, 0.248526, 0.323885), (0.366147, 0.180564, 0.119154, 0.905061), (-0.180564, 0.366147, -0.905061, 0.119154), (0.226811, 0.339446, -0.349341, 0.843383), (-0.339446, 0.226811, -0.843383, -0.349341), (0.0267007, 0.407374, -0.724229, 0.555721), (-0.407374, 0.0267007, -0.555721, -0.724229), (0.69352, 0.13795, -0.5, -0.5), (-0.13795, 0.69352, 0.5, -0.5), (0.531631, 0.466228, -0.183013, -0.683013), (-0.466228, 0.531631, 0.683013, -0.183013), (0.227292, 0.669581, 0.183013, -0.683013), (-0.669581, 0.227292, 0.683013, 0.183013), (0.69352, 0.13795, 0.5, 0.5), (-0.13795, 0.69352, -0.5, 0.5), (0.531631, 0.466228, 0.183013, 0.683013), (-0.466228, 0.531631, -0.683013, 0.183013), (0.227292, 0.669581, -0.183013, 0.683013), (-0.669581, 0.227292, -0.683013, -0.183013), (0.89533, 0.178092, -0.15623, 0.377172), (-0.178092, 0.89533, -0.377172, -0.15623), (0.686333, 0.601898, -0.323885, 0.248526), (-0.601898, 0.686333, -0.248526, -0.323885), (0.293433, 0.864425, -0.404756, 0.0532871), (-0.864425, 0.293433, -0.0532871, -0.404756), (0.400404, 0.0796453, 0.349341, -0.843383), (-0.0796453, 0.400404, 0.843383, 0.349341), (0.306937, 0.269177, 0.724229, -0.555721), (-0.269177, 0.306937, 0.555721, 0.724229), (0.131227, 0.386583, 0.905061, -0.119154), (-0.386583, 0.131227, 0.119154, 0.905061), (0.400404, 0.0796453, -0.349341, 0.843383), (-0.0796453, 0.400404, -0.843383, -0.349341), (0.306937, 0.269177, -0.724229, 0.555721), (-0.269177, 0.306937, -0.555721, -0.724229), (0.131227, 0.386583, -0.905061, 0.119154), (-0.386583, 0.131227, -0.119154, -0.905061), (0.89533, 0.178092, 0.15623, -0.377172), (-0.178092, 0.89533, 0.377172, 0.15623), (0.686333, 0.601898, 0.323885, -0.248526), (-0.601898, 0.686333, 0.248526, 0.323885), (0.293433, 0.864425, 0.404756, -0.0532871), (-0.864425, 0.293433, 0.0532871, 0.404756), (0.69352, 0.13795, -0.5, 0.5), (-0.13795, 0.69352, -0.5, -0.5), (0.531631, 0.466228, -0.683013, 0.183013), (-0.466228, 0.531631, -0.183013, -0.683013), (0.227292, 0.669581, -0.683013, -0.183013), (-0.669581, 0.227292, 0.183013, -0.683013), (0.69352, 0.13795, 0.5, -0.5), (-0.13795, 0.69352, 0.5, 0.5), (0.531631, 0.466228, 0.683013, -0.183013), (-0.466228, 0.531631, 0.183013, 0.683013), (0.227292, 0.669581, 0.683013, 0.183013), (-0.669581, 0.227292, -0.183013, 0.683013), (0.400404, 0.0796453, -0.843383, -0.349341), (-0.0796453, 0.400404, 0.349341, -0.843383), (0.306937, 0.269177, -0.555721, -0.724229), (-0.269177, 0.306937, 0.724229, -0.555721), (0.131227, 0.386583, -0.119154, -0.905061), (-0.386583, 0.131227, 0.905061, -0.119154), (0.89533, 0.178092, 0.377172, 0.15623), (-0.178092, 0.89533, -0.15623, 0.377172), (0.686333, 0.601898, 0.248526, 0.323885), (-0.601898, 0.686333, -0.323885, 0.248526), (0.293433, 0.864425, 0.0532871, 0.404756), (-0.864425, 0.293433, -0.404756, 0.0532871), (0.89533, 0.178092, -0.377172, -0.15623), (-0.178092, 0.89533, 0.15623, -0.377172), (0.686333, 0.601898, -0.248526, -0.323885), (-0.601898, 0.686333, 0.323885, -0.248526), (0.293433, 0.864425, -0.0532871, -0.404756), (-0.864425, 0.293433, 0.404756, -0.0532871), (0.400404, 0.0796453, 0.843383, 0.349341), (-0.0796453, 0.400404, -0.349341, 0.843383), (0.306937, 0.269177, 0.555721, 0.724229), (-0.269177, 0.306937, -0.724229, 0.555721), (0.131227, 0.386583, 0.119154, 0.905061), (-0.386583, 0.131227, -0.905061, 0.119154), (0.800808, 0.159291, -0.57735, 7.0705e-17), (-0.159291, 0.800808, -1.06058e-16, -0.57735), (0.613875, 0.538354, -0.5, -0.288675), (-0.538354, 0.613875, 0.288675, -0.5), (0.262454, 0.773165, -0.288675, -0.5), (-0.773165, 0.262454, 0.5, -0.288675), (0.566257, 0.112635, 0.816497, -1.99984e-16), (-0.112635, 0.566257, 2.4998e-16, 0.816497), (0.434075, 0.380673, 0.707107, 0.408248), (-0.380673, 0.434075, -0.408248, 0.707107), (0.185583, 0.54671, 0.408248, 0.707107), (-0.54671, 0.185583, -0.707107, 0.408248), (0.960135, 0.190983, -0.200202, 0.0398226), (-0.190983, 0.960135, -0.0398226, -0.200202), (0.73601, 0.645463, -0.193291, -0.0656136), (-0.645463, 0.73601, 0.0656136, -0.193291), (0.314672, 0.926993, -0.134588, -0.153469), (-0.926993, 0.314672, 0.153469, -0.134588), (0.566257, 0.112635, 0.754344, -0.31246), (-0.112635, 0.566257, 0.31246, 0.754344), (0.434075, 0.380673, 0.809511, 0.106574), (-0.380673, 0.434075, -0.106574, 0.809511), (0.185583, 0.54671, 0.64777, 0.497052), (-0.54671, 0.185583, -0.497052, 0.64777), (0.566257, 0.112635, -0.754344, 0.31246), (-0.112635, 0.566257, -0.31246, -0.754344), (0.434075, 0.380673, -0.809511, -0.106574), (-0.380673, 0.434075, 0.106574, -0.809511), (0.185583, 0.54671, -0.64777, -0.497052), (-0.54671, 0.185583, 0.497052, -0.64777), (0.960135, 0.190983, 0.200202, -0.0398226), (-0.190983, 0.960135, 0.0398226, 0.200202), (0.73601, 0.645463, 0.193291, 0.0656136), (-0.645463, 0.73601, -0.0656136, 0.193291), (0.314672, 0.926993, 0.134588, 0.153469), (-0.926993, 0.314672, -0.153469, 0.134588), (0.566257, 0.112635, 4.9996e-17, 0.816497), (-0.112635, 0.566257, -0.816497, 9.9992e-17), (0.434075, 0.380673, -0.408248, 0.707107), (-0.380673, 0.434075, -0.707107, -0.408248), (0.185583, 0.54671, -0.707107, 0.408248), (-0.54671, 0.185583, -0.408248, -0.707107), (0.800808, 0.159291, -1.06058e-16, -0.57735), (-0.159291, 0.800808, 0.57735, -1.4141e-16), (0.613875, 0.538354, 0.288675, -0.5), (-0.538354, 0.613875, 0.5, 0.288675), (0.262454, 0.773165, 0.5, -0.288675), (-0.773165, 0.262454, 0.288675, 0.5), (0.200202, 0.0398226, -0.190983, -0.960135), (-0.0398226, 0.200202, 0.960135, -0.190983), (0.153469, 0.134588, 0.314672, -0.926993), (-0.134588, 0.153469, 0.926993, 0.314672), (0.0656136, 0.193291, 0.73601, -0.645463), (-0.193291, 0.0656136, 0.645463, 0.73601), (0.800808, 0.159291, 0.220942, 0.533402), (-0.159291, 0.800808, -0.533402, 0.220942), (0.613875, 0.538354, -0.0753593, 0.572411), (-0.538354, 0.613875, -0.572411, -0.0753593), (0.262454, 0.773165, -0.351469, 0.458043), (-0.773165, 0.262454, -0.458043, -0.351469), (0.800808, 0.159291, -0.220942, -0.533402), (-0.159291, 0.800808, 0.533402, -0.220942), (0.613875, 0.538354, 0.0753593, -0.572411), (-0.538354, 0.613875, 0.572411, 0.0753593), (0.262454, 0.773165, 0.351469, -0.458043), (-0.773165, 0.262454, 0.458043, 0.351469), (0.200202, 0.0398226, 0.190983, 0.960135), (-0.0398226, 0.200202, -0.960135, 0.190983), (0.153469, 0.134588, -0.314672, 0.926993), (-0.134588, 0.153469, -0.926993, -0.314672), (0.0656136, 0.193291, -0.73601, 0.645463), (-0.193291, 0.0656136, -0.645463, -0.73601), (0.51781, 0.255355, -0.788675, -0.211325), (-0.255355, 0.51781, 0.211325, -0.788675), (0.320759, 0.480049, -0.57735, -0.57735), (-0.480049, 0.320759, 0.57735, -0.57735), (0.0377605, 0.576114, -0.211325, -0.788675), (-0.576114, 0.0377605, 0.788675, -0.211325), (0.732294, 0.361127, 0.557678, 0.149429), (-0.361127, 0.732294, -0.149429, 0.557678), (0.453621, 0.678892, 0.408248, 0.408248), (-0.678892, 0.453621, -0.408248, 0.408248), (0.0534014, 0.814748, 0.149429, 0.557678), (-0.814748, 0.0534014, -0.557678, 0.149429), (0.732294, 0.361127, -0.572411, 0.0753593), (-0.361127, 0.732294, -0.0753593, -0.572411), (0.453621, 0.678892, -0.533402, -0.220942), (-0.678892, 0.453621, 0.220942, -0.533402), (0.0534014, 0.814748, -0.351469, -0.458043), (-0.814748, 0.0534014, 0.458043, -0.351469), (0.183073, 0.0902818, 0.976849, 0.0640261), (-0.0902818, 0.183073, -0.0640261, 0.976849), (0.113405, 0.169723, 0.813963, 0.543873), (-0.169723, 0.113405, -0.543873, 0.813963), (0.0133504, 0.203687, 0.432976, 0.877989), (-0.203687, 0.0133504, -0.877989, 0.432976), (0.183073, 0.0902818, -0.976849, -0.0640261), (-0.0902818, 0.183073, 0.0640261, -0.976849), (0.113405, 0.169723, -0.813963, -0.543873), (-0.169723, 0.113405, 0.543873, -0.813963), (0.0133504, 0.203687, -0.432976, -0.877989), (-0.203687, 0.0133504, 0.877989, -0.432976), (0.732294, 0.361127, 0.572411, -0.0753593), (-0.361127, 0.732294, 0.0753593, 0.572411), (0.453621, 0.678892, 0.533402, 0.220942), (-0.678892, 0.453621, -0.220942, 0.533402), (0.0534014, 0.814748, 0.351469, 0.458043), (-0.814748, 0.0534014, -0.458043, 0.351469), (0.732294, 0.361127, -0.149429, 0.557678), (-0.361127, 0.732294, -0.557678, -0.149429), (0.453621, 0.678892, -0.408248, 0.408248), (-0.678892, 0.453621, -0.408248, -0.408248), (0.0534014, 0.814748, -0.557678, 0.149429), (-0.814748, 0.0534014, -0.149429, -0.557678), (0.51781, 0.255355, 0.211325, -0.788675), (-0.255355, 0.51781, 0.788675, 0.211325), (0.320759, 0.480049, 0.57735, -0.57735), (-0.480049, 0.320759, 0.57735, 0.57735), (0.0377605, 0.576114, 0.788675, -0.211325), (-0.576114, 0.0377605, 0.211325, 0.788675), (0.51781, 0.255355, -0.106574, -0.809511), (-0.255355, 0.51781, 0.809511, -0.106574), (0.320759, 0.480049, 0.31246, -0.754344), (-0.480049, 0.320759, 0.754344, 0.31246), (0.0377605, 0.576114, 0.64777, -0.497052), (-0.576114, 0.0377605, 0.497052, 0.64777), (0.877989, 0.432976, -0.0133504, 0.203687), (-0.432976, 0.877989, -0.203687, -0.0133504), (0.543873, 0.813963, -0.113405, 0.169723), (-0.813963, 0.543873, -0.169723, -0.113405), (0.0640261, 0.976849, -0.183073, 0.0902818), (-0.976849, 0.0640261, -0.0902818, -0.183073), (0.877989, 0.432976, 0.0133504, -0.203687), (-0.432976, 0.877989, 0.203687, 0.0133504), (0.543873, 0.813963, 0.113405, -0.169723), (-0.813963, 0.543873, 0.169723, 0.113405), (0.0640261, 0.976849, 0.183073, -0.0902818), (-0.976849, 0.0640261, 0.0902818, 0.183073), (0.51781, 0.255355, 0.106574, 0.809511), (-0.255355, 0.51781, -0.809511, 0.106574), (0.320759, 0.480049, -0.31246, 0.754344), (-0.480049, 0.320759, -0.754344, -0.31246), (0.0377605, 0.576114, -0.64777, 0.497052), (-0.576114, 0.0377605, -0.497052, -0.64777), (0.732294, 0.361127, -0.557678, -0.149429), (-0.361127, 0.732294, 0.149429, -0.557678), (0.453621, 0.678892, -0.408248, -0.408248), (-0.678892, 0.453621, 0.408248, -0.408248), (0.0534014, 0.814748, -0.149429, -0.557678), (-0.814748, 0.0534014, 0.557678, -0.149429), (0.51781, 0.255355, 0.788675, 0.211325), (-0.255355, 0.51781, -0.211325, 0.788675), (0.320759, 0.480049, 0.57735, 0.57735), (-0.480049, 0.320759, -0.57735, 0.57735), (0.0377605, 0.576114, 0.211325, 0.788675), (-0.576114, 0.0377605, -0.788675, 0.211325), (0.877989, 0.432976, -0.203687, -0.0133504), (-0.432976, 0.877989, 0.0133504, -0.203687), (0.543873, 0.813963, -0.169723, -0.113405), (-0.813963, 0.543873, 0.113405, -0.169723), (0.0640261, 0.976849, -0.0902818, -0.183073), (-0.976849, 0.0640261, 0.183073, -0.0902818), (0.51781, 0.255355, 0.809511, -0.106574), (-0.255355, 0.51781, 0.106574, 0.809511), (0.320759, 0.480049, 0.754344, 0.31246), (-0.480049, 0.320759, -0.31246, 0.754344), (0.0377605, 0.576114, 0.497052, 0.64777), (-0.576114, 0.0377605, -0.64777, 0.497052), (0.51781, 0.255355, -0.809511, 0.106574), (-0.255355, 0.51781, -0.106574, -0.809511), (0.320759, 0.480049, -0.754344, -0.31246), (-0.480049, 0.320759, 0.31246, -0.754344), (0.0377605, 0.576114, -0.497052, -0.64777), (-0.576114, 0.0377605, 0.64777, -0.497052), (0.877989, 0.432976, 0.203687, 0.0133504), (-0.432976, 0.877989, -0.0133504, 0.203687), (0.543873, 0.813963, 0.169723, 0.113405), (-0.813963, 0.543873, -0.113405, 0.169723), (0.0640261, 0.976849, 0.0902818, 0.183073), (-0.976849, 0.0640261, -0.183073, 0.0902818), (0.51781, 0.255355, -0.211325, 0.788675), (-0.255355, 0.51781, -0.788675, -0.211325), (0.320759, 0.480049, -0.57735, 0.57735), (-0.480049, 0.320759, -0.57735, -0.57735), (0.0377605, 0.576114, -0.788675, 0.211325), (-0.576114, 0.0377605, -0.211325, -0.788675), (0.732294, 0.361127, 0.149429, -0.557678), (-0.361127, 0.732294, 0.557678, 0.149429), (0.453621, 0.678892, 0.408248, -0.408248), (-0.678892, 0.453621, 0.408248, 0.408248), (0.0534014, 0.814748, 0.557678, -0.149429), (-0.814748, 0.0534014, 0.149429, 0.557678), (0.183073, 0.0902818, 0.0640261, -0.976849), (-0.0902818, 0.183073, 0.976849, 0.0640261), (0.113405, 0.169723, 0.543873, -0.813963), (-0.169723, 0.113405, 0.813963, 0.543873), (0.0133504, 0.203687, 0.877989, -0.432976), (-0.203687, 0.0133504, 0.432976, 0.877989), (0.732294, 0.361127, 0.0753593, 0.572411), (-0.361127, 0.732294, -0.572411, 0.0753593), (0.453621, 0.678892, -0.220942, 0.533402), (-0.678892, 0.453621, -0.533402, -0.220942), (0.0534014, 0.814748, -0.458043, 0.351469), (-0.814748, 0.0534014, -0.351469, -0.458043), (0.732294, 0.361127, -0.0753593, -0.572411), (-0.361127, 0.732294, 0.572411, -0.0753593), (0.453621, 0.678892, 0.220942, -0.533402), (-0.678892, 0.453621, 0.533402, 0.220942), (0.0534014, 0.814748, 0.458043, -0.351469), (-0.814748, 0.0534014, 0.351469, 0.458043), (0.183073, 0.0902818, -0.0640261, 0.976849), (-0.0902818, 0.183073, -0.976849, -0.0640261), (0.113405, 0.169723, -0.543873, 0.813963), (-0.169723, 0.113405, -0.813963, -0.543873), (0.0133504, 0.203687, -0.877989, 0.432976), (-0.203687, 0.0133504, -0.432976, -0.877989), (0.566257, 0.112635, -0.816497, 9.9992e-17), (-0.112635, 0.566257, -1.49988e-16, -0.816497), (0.434075, 0.380673, -0.707107, -0.408248), (-0.380673, 0.434075, 0.408248, -0.707107), (0.185583, 0.54671, -0.408248, -0.707107), (-0.54671, 0.185583, 0.707107, -0.408248), (0.800808, 0.159291, 0.57735, -1.4141e-16), (-0.159291, 0.800808, 1.76763e-16, 0.57735), (0.613875, 0.538354, 0.5, 0.288675), (-0.538354, 0.613875, -0.288675, 0.5), (0.262454, 0.773165, 0.288675, 0.5), (-0.773165, 0.262454, -0.5, 0.288675), (0.800808, 0.159291, -0.533402, 0.220942), (-0.159291, 0.800808, -0.220942, -0.533402), (0.613875, 0.538354, -0.572411, -0.0753593), (-0.538354, 0.613875, 0.0753593, -0.572411), (0.262454, 0.773165, -0.458043, -0.351469), (-0.773165, 0.262454, 0.351469, -0.458043), (0.200202, 0.0398226, 0.960135, -0.190983), (-0.0398226, 0.200202, 0.190983, 0.960135), (0.153469, 0.134588, 0.926993, 0.314672), (-0.134588, 0.153469, -0.314672, 0.926993), (0.0656136, 0.193291, 0.645463, 0.73601), (-0.193291, 0.0656136, -0.73601, 0.645463), (0.200202, 0.0398226, -0.960135, 0.190983), (-0.0398226, 0.200202, -0.190983, -0.960135), (0.153469, 0.134588, -0.926993, -0.314672), (-0.134588, 0.153469, 0.314672, -0.926993), (0.0656136, 0.193291, -0.645463, -0.73601), (-0.193291, 0.0656136, 0.73601, -0.645463), (0.800808, 0.159291, 0.533402, -0.220942), (-0.159291, 0.800808, 0.220942, 0.533402), (0.613875, 0.538354, 0.572411, 0.0753593), (-0.538354, 0.613875, -0.0753593, 0.572411), (0.262454, 0.773165, 0.458043, 0.351469), (-0.773165, 0.262454, -0.351469, 0.458043), (0.800808, 0.159291, 3.53525e-17, 0.57735), (-0.159291, 0.800808, -0.57735, 7.0705e-17), (0.613875, 0.538354, -0.288675, 0.5), (-0.538354, 0.613875, -0.5, -0.288675), (0.262454, 0.773165, -0.5, 0.288675), (-0.773165, 0.262454, -0.288675, -0.5), (0.566257, 0.112635, -1.49988e-16, -0.816497), (-0.112635, 0.566257, 0.816497, -1.99984e-16), (0.434075, 0.380673, 0.408248, -0.707107), (-0.380673, 0.434075, 0.707107, 0.408248), (0.185583, 0.54671, 0.707107, -0.408248), (-0.54671, 0.185583, 0.408248, 0.707107), (0.566257, 0.112635, -0.31246, -0.754344), (-0.112635, 0.566257, 0.754344, -0.31246), (0.434075, 0.380673, 0.106574, -0.809511), (-0.380673, 0.434075, 0.809511, 0.106574), (0.185583, 0.54671, 0.497052, -0.64777), (-0.54671, 0.185583, 0.64777, 0.497052), (0.960135, 0.190983, 0.0398226, 0.200202), (-0.190983, 0.960135, -0.200202, 0.0398226), (0.73601, 0.645463, -0.0656136, 0.193291), (-0.645463, 0.73601, -0.193291, -0.0656136), (0.314672, 0.926993, -0.153469, 0.134588), (-0.926993, 0.314672, -0.134588, -0.153469), (0.960135, 0.190983, -0.0398226, -0.200202), (-0.190983, 0.960135, 0.200202, -0.0398226), (0.73601, 0.645463, 0.0656136, -0.193291), (-0.645463, 0.73601, 0.193291, 0.0656136), (0.314672, 0.926993, 0.153469, -0.134588), (-0.926993, 0.314672, 0.134588, 0.153469), (0.566257, 0.112635, 0.31246, 0.754344), (-0.112635, 0.566257, -0.754344, 0.31246), (0.434075, 0.380673, -0.106574, 0.809511), (-0.380673, 0.434075, -0.809511, -0.106574), (0.185583, 0.54671, -0.497052, 0.64777), (-0.54671, 0.185583, -0.64777, -0.497052), (0.69352, 0.13795, -0.653281, 0.270598), (-0.13795, 0.69352, -0.270598, -0.653281), (0.531631, 0.466228, -0.701057, -0.092296), (-0.466228, 0.531631, 0.092296, -0.701057), (0.227292, 0.669581, -0.560986, -0.430459), (-0.669581, 0.227292, 0.430459, -0.560986), (0.69352, 0.13795, 0.653281, -0.270598), (-0.13795, 0.69352, 0.270598, 0.653281), (0.531631, 0.466228, 0.701057, 0.092296), (-0.466228, 0.531631, -0.092296, 0.701057), (0.227292, 0.669581, 0.560986, 0.430459), (-0.669581, 0.227292, -0.430459, 0.560986), (0.89533, 0.178092, -0.408248, 4.9996e-17), (-0.178092, 0.89533, -7.4994e-17, -0.408248), (0.686333, 0.601898, -0.353553, -0.204124), (-0.601898, 0.686333, 0.204124, -0.353553), (0.293433, 0.864425, -0.204124, -0.353553), (-0.864425, 0.293433, 0.353553, -0.204124), (0.400404, 0.0796453, 0.912871, -2.23589e-16), (-0.0796453, 0.400404, 2.79486e-16, 0.912871), (0.306937, 0.269177, 0.790569, 0.456435), (-0.269177, 0.306937, -0.456435, 0.790569), (0.131227, 0.386583, 0.456435, 0.790569), (-0.386583, 0.131227, -0.790569, 0.456435), (0.400404, 0.0796453, -0.912871, 1.11794e-16), (-0.0796453, 0.400404, -1.67692e-16, -0.912871), (0.306937, 0.269177, -0.790569, -0.456435), (-0.269177, 0.306937, 0.456435, -0.790569), (0.131227, 0.386583, -0.456435, -0.790569), (-0.386583, 0.131227, 0.790569, -0.456435), (0.89533, 0.178092, 0.408248, -9.9992e-17), (-0.178092, 0.89533, 1.2499e-16, 0.408248), (0.686333, 0.601898, 0.353553, 0.204124), (-0.601898, 0.686333, -0.204124, 0.353553), (0.293433, 0.864425, 0.204124, 0.353553), (-0.864425, 0.293433, -0.353553, 0.204124), (0.69352, 0.13795, 0.270598, 0.653281), (-0.13795, 0.69352, -0.653281, 0.270598), (0.531631, 0.466228, -0.092296, 0.701057), (-0.466228, 0.531631, -0.701057, -0.092296), (0.227292, 0.669581, -0.430459, 0.560986), (-0.669581, 0.227292, -0.560986, -0.430459), (0.69352, 0.13795, -0.270598, -0.653281), (-0.13795, 0.69352, 0.653281, -0.270598), (0.531631, 0.466228, 0.092296, -0.701057), (-0.466228, 0.531631, 0.701057, 0.092296), (0.227292, 0.669581, 0.430459, -0.560986), (-0.669581, 0.227292, 0.560986, 0.430459), (0.400404, 0.0796453, -1.67692e-16, -0.912871), (-0.0796453, 0.400404, 0.912871, -2.23589e-16), (0.306937, 0.269177, 0.456435, -0.790569), (-0.269177, 0.306937, 0.790569, 0.456435), (0.131227, 0.386583, 0.790569, -0.456435), (-0.386583, 0.131227, 0.456435, 0.790569), (0.89533, 0.178092, 2.4998e-17, 0.408248), (-0.178092, 0.89533, -0.408248, 4.9996e-17), (0.686333, 0.601898, -0.204124, 0.353553), (-0.601898, 0.686333, -0.353553, -0.204124), (0.293433, 0.864425, -0.353553, 0.204124), (-0.864425, 0.293433, -0.204124, -0.353553), (0.89533, 0.178092, -7.4994e-17, -0.408248), (-0.178092, 0.89533, 0.408248, -9.9992e-17), (0.686333, 0.601898, 0.204124, -0.353553), (-0.601898, 0.686333, 0.353553, 0.204124), (0.293433, 0.864425, 0.353553, -0.204124), (-0.864425, 0.293433, 0.204124, 0.353553), (0.400404, 0.0796453, 5.58972e-17, 0.912871), (-0.0796453, 0.400404, -0.912871, 1.11794e-16), (0.306937, 0.269177, -0.456435, 0.790569), (-0.269177, 0.306937, -0.790569, -0.456435), (0.131227, 0.386583, -0.790569, 0.456435), (-0.386583, 0.131227, -0.456435, -0.790569), (0.634185, 0.312745, -0.560986, -0.430459), (-0.312745, 0.634185, 0.430459, -0.560986), (0.392847, 0.587938, -0.270598, -0.653281), (-0.587938, 0.392847, 0.653281, -0.270598), (0.046247, 0.705593, 0.092296, -0.701057), (-0.705593, 0.046247, 0.701057, 0.092296), (0.634185, 0.312745, 0.560986, 0.430459), (-0.312745, 0.634185, -0.430459, 0.560986), (0.392847, 0.587938, 0.270598, 0.653281), (-0.587938, 0.392847, -0.653281, 0.270598), (0.046247, 0.705593, -0.092296, 0.701057), (-0.705593, 0.046247, -0.701057, -0.092296), (0.818729, 0.403752, -0.353553, 0.204124), (-0.403752, 0.818729, -0.204124, -0.353553), (0.507164, 0.759024, -0.408248, 4.9996e-17), (-0.759024, 0.507164, -7.4994e-17, -0.408248), (0.0597046, 0.910916, -0.353553, -0.204124), (-0.910916, 0.0597046, 0.204124, -0.353553), (0.366147, 0.180564, 0.790569, -0.456435), (-0.180564, 0.366147, 0.456435, 0.790569), (0.226811, 0.339446, 0.912871, -2.23589e-16), (-0.339446, 0.226811, 2.79486e-16, 0.912871), (0.0267007, 0.407374, 0.790569, 0.456435), (-0.407374, 0.0267007, -0.456435, 0.790569), (0.366147, 0.180564, -0.790569, 0.456435), (-0.180564, 0.366147, -0.456435, -0.790569), (0.226811, 0.339446, -0.912871, 1.11794e-16), (-0.339446, 0.226811, -1.67692e-16, -0.912871), (0.0267007, 0.407374, -0.790569, -0.456435), (-0.407374, 0.0267007, 0.456435, -0.790569), (0.818729, 0.403752, 0.353553, -0.204124), (-0.403752, 0.818729, 0.204124, 0.353553), (0.507164, 0.759024, 0.408248, -9.9992e-17), (-0.759024, 0.507164, 1.2499e-16, 0.408248), (0.0597046, 0.910916, 0.353553, 0.204124), (-0.910916, 0.0597046, -0.204124, 0.353553), (0.634185, 0.312745, -0.430459, 0.560986), (-0.312745, 0.634185, -0.560986, -0.430459), (0.392847, 0.587938, -0.653281, 0.270598), (-0.587938, 0.392847, -0.270598, -0.653281), (0.046247, 0.705593, -0.701057, -0.092296), (-0.705593, 0.046247, 0.092296, -0.701057), (0.634185, 0.312745, 0.430459, -0.560986), (-0.312745, 0.634185, 0.560986, 0.430459), (0.392847, 0.587938, 0.653281, -0.270598), (-0.587938, 0.392847, 0.270598, 0.653281), (0.046247, 0.705593, 0.701057, 0.092296), (-0.705593, 0.046247, -0.092296, 0.701057), (0.366147, 0.180564, -0.456435, -0.790569), (-0.180564, 0.366147, 0.790569, -0.456435), (0.226811, 0.339446, -1.67692e-16, -0.912871), (-0.339446, 0.226811, 0.912871, -2.23589e-16), (0.0267007, 0.407374, 0.456435, -0.790569), (-0.407374, 0.0267007, 0.790569, 0.456435), (0.818729, 0.403752, 0.204124, 0.353553), (-0.403752, 0.818729, -0.353553, 0.204124), (0.507164, 0.759024, 2.4998e-17, 0.408248), (-0.759024, 0.507164, -0.408248, 4.9996e-17), (0.0597046, 0.910916, -0.204124, 0.353553), (-0.910916, 0.0597046, -0.353553, -0.204124), (0.818729, 0.403752, -0.204124, -0.353553), (-0.403752, 0.818729, 0.353553, -0.204124), (0.507164, 0.759024, -7.4994e-17, -0.408248), (-0.759024, 0.507164, 0.408248, -9.9992e-17), (0.0597046, 0.910916, 0.204124, -0.353553), (-0.910916, 0.0597046, 0.353553, 0.204124), (0.366147, 0.180564, 0.456435, 0.790569), (-0.180564, 0.366147, -0.790569, 0.456435), (0.226811, 0.339446, 5.58972e-17, 0.912871), (-0.339446, 0.226811, -0.912871, 1.11794e-16), (0.0267007, 0.407374, -0.456435, 0.790569), (-0.407374, 0.0267007, -0.790569, -0.456435), (0.634185, 0.312745, -0.701057, 0.092296), (-0.312745, 0.634185, -0.092296, -0.701057), (0.392847, 0.587938, -0.653281, -0.270598), (-0.587938, 0.392847, 0.270598, -0.653281), (0.046247, 0.705593, -0.430459, -0.560986), (-0.705593, 0.046247, 0.560986, -0.430459), (0.634185, 0.312745, 0.701057, -0.092296), (-0.312745, 0.634185, 0.092296, 0.701057), (0.392847, 0.587938, 0.653281, 0.270598), (-0.587938, 0.392847, -0.270598, 0.653281), (0.046247, 0.705593, 0.430459, 0.560986), (-0.705593, 0.046247, -0.560986, 0.430459), (0.818729, 0.403752, -0.394338, -0.105662), (-0.403752, 0.818729, 0.105662, -0.394338), (0.507164, 0.759024, -0.288675, -0.288675), (-0.759024, 0.507164, 0.288675, -0.288675), (0.0597046, 0.910916, -0.105662, -0.394338), (-0.910916, 0.0597046, 0.394338, -0.105662), (0.366147, 0.180564, 0.881766, 0.236268), (-0.180564, 0.366147, -0.236268, 0.881766), (0.226811, 0.339446, 0.645497, 0.645497), (-0.339446, 0.226811, -0.645497, 0.645497), (0.0267007, 0.407374, 0.236268, 0.881766), (-0.407374, 0.0267007, -0.881766, 0.236268), (0.366147, 0.180564, -0.881766, -0.236268), (-0.180564, 0.366147, 0.236268, -0.881766), (0.226811, 0.339446, -0.645497, -0.645497), (-0.339446, 0.226811, 0.645497, -0.645497), (0.0267007, 0.407374, -0.236268, -0.881766), (-0.407374, 0.0267007, 0.881766, -0.236268), (0.818729, 0.403752, 0.394338, 0.105662), (-0.403752, 0.818729, -0.105662, 0.394338), (0.507164, 0.759024, 0.288675, 0.288675), (-0.759024, 0.507164, -0.288675, 0.288675), (0.0597046, 0.910916, 0.105662, 0.394338), (-0.910916, 0.0597046, -0.394338, 0.105662), (0.634185, 0.312745, 0.092296, 0.701057), (-0.312745, 0.634185, -0.701057, 0.092296), (0.392847, 0.587938, -0.270598, 0.653281), (-0.587938, 0.392847, -0.653281, -0.270598), (0.046247, 0.705593, -0.560986, 0.430459), (-0.705593, 0.046247, -0.430459, -0.560986), (0.634185, 0.312745, -0.092296, -0.701057), (-0.312745, 0.634185, 0.701057, -0.092296), (0.392847, 0.587938, 0.270598, -0.653281), (-0.587938, 0.392847, 0.653281, 0.270598), (0.046247, 0.705593, 0.560986, -0.430459), (-0.705593, 0.046247, 0.430459, 0.560986), (0.366147, 0.180564, 0.236268, -0.881766), (-0.180564, 0.366147, 0.881766, 0.236268), (0.226811, 0.339446, 0.645497, -0.645497), (-0.339446, 0.226811, 0.645497, 0.645497), (0.0267007, 0.407374, 0.881766, -0.236268), (-0.407374, 0.0267007, 0.236268, 0.881766), (0.818729, 0.403752, -0.105662, 0.394338), (-0.403752, 0.818729, -0.394338, -0.105662), (0.507164, 0.759024, -0.288675, 0.288675), (-0.759024, 0.507164, -0.288675, -0.288675), (0.0597046, 0.910916, -0.394338, 0.105662), (-0.910916, 0.0597046, -0.105662, -0.394338), (0.818729, 0.403752, 0.105662, -0.394338), (-0.403752, 0.818729, 0.394338, 0.105662), (0.507164, 0.759024, 0.288675, -0.288675), (-0.759024, 0.507164, 0.288675, 0.288675), (0.0597046, 0.910916, 0.394338, -0.105662), (-0.910916, 0.0597046, 0.105662, 0.394338), (0.366147, 0.180564, -0.236268, 0.881766), (-0.180564, 0.366147, -0.881766, -0.236268), (0.226811, 0.339446, -0.645497, 0.645497), (-0.339446, 0.226811, -0.645497, -0.645497), (0.0267007, 0.407374, -0.881766, 0.236268), (-0.407374, 0.0267007, -0.236268, -0.881766), (0.69352, 0.13795, -0.653281, -0.270598), (-0.13795, 0.69352, 0.270598, -0.653281), (0.531631, 0.466228, -0.430459, -0.560986), (-0.466228, 0.531631, 0.560986, -0.430459), (0.227292, 0.669581, -0.092296, -0.701057), (-0.669581, 0.227292, 0.701057, -0.092296), (0.69352, 0.13795, 0.653281, 0.270598), (-0.13795, 0.69352, -0.270598, 0.653281), (0.531631, 0.466228, 0.430459, 0.560986), (-0.466228, 0.531631, -0.560986, 0.430459), (0.227292, 0.669581, 0.092296, 0.701057), (-0.669581, 0.227292, -0.701057, 0.092296), (0.89533, 0.178092, -0.288675, 0.288675), (-0.178092, 0.89533, -0.288675, -0.288675), (0.686333, 0.601898, -0.394338, 0.105662), (-0.601898, 0.686333, -0.105662, -0.394338), (0.293433, 0.864425, -0.394338, -0.105662), (-0.864425, 0.293433, 0.105662, -0.394338), (0.400404, 0.0796453, 0.645497, -0.645497), (-0.0796453, 0.400404, 0.645497, 0.645497), (0.306937, 0.269177, 0.881766, -0.236268), (-0.269177, 0.306937, 0.236268, 0.881766), (0.131227, 0.386583, 0.881766, 0.236268), (-0.386583, 0.131227, -0.236268, 0.881766), (0.400404, 0.0796453, -0.645497, 0.645497), (-0.0796453, 0.400404, -0.645497, -0.645497), (0.306937, 0.269177, -0.881766, 0.236268), (-0.269177, 0.306937, -0.236268, -0.881766), (0.131227, 0.386583, -0.881766, -0.236268), (-0.386583, 0.131227, 0.236268, -0.881766), (0.89533, 0.178092, 0.288675, -0.288675), (-0.178092, 0.89533, 0.288675, 0.288675), (0.686333, 0.601898, 0.394338, -0.105662), (-0.601898, 0.686333, 0.105662, 0.394338), (0.293433, 0.864425, 0.394338, 0.105662), (-0.864425, 0.293433, -0.105662, 0.394338), (0.69352, 0.13795, -0.270598, 0.653281), (-0.13795, 0.69352, -0.653281, -0.270598), (0.531631, 0.466228, -0.560986, 0.430459), (-0.466228, 0.531631, -0.430459, -0.560986), (0.227292, 0.669581, -0.701057, 0.092296), (-0.669581, 0.227292, -0.092296, -0.701057), (0.69352, 0.13795, 0.270598, -0.653281), (-0.13795, 0.69352, 0.653281, 0.270598), (0.531631, 0.466228, 0.560986, -0.430459), (-0.466228, 0.531631, 0.430459, 0.560986), (0.227292, 0.669581, 0.701057, -0.092296), (-0.669581, 0.227292, 0.092296, 0.701057), (0.400404, 0.0796453, -0.645497, -0.645497), (-0.0796453, 0.400404, 0.645497, -0.645497), (0.306937, 0.269177, -0.236268, -0.881766), (-0.269177, 0.306937, 0.881766, -0.236268), (0.131227, 0.386583, 0.236268, -0.881766), (-0.386583, 0.131227, 0.881766, 0.236268), (0.89533, 0.178092, 0.288675, 0.288675), (-0.178092, 0.89533, -0.288675, 0.288675), (0.686333, 0.601898, 0.105662, 0.394338), (-0.601898, 0.686333, -0.394338, 0.105662), (0.293433, 0.864425, -0.105662, 0.394338), (-0.864425, 0.293433, -0.394338, -0.105662), (0.89533, 0.178092, -0.288675, -0.288675), (-0.178092, 0.89533, 0.288675, -0.288675), (0.686333, 0.601898, -0.105662, -0.394338), (-0.601898, 0.686333, 0.394338, -0.105662), (0.293433, 0.864425, 0.105662, -0.394338), (-0.864425, 0.293433, 0.394338, 0.105662), (0.400404, 0.0796453, 0.645497, 0.645497), (-0.0796453, 0.400404, -0.645497, 0.645497), (0.306937, 0.269177, 0.236268, 0.881766), (-0.269177, 0.306937, -0.881766, 0.236268), (0.131227, 0.386583, -0.236268, 0.881766), (-0.386583, 0.131227, -0.881766, -0.236268), (0.814748, 0.0534014, -0.557678, -0.149429), (-0.0534014, 0.814748, 0.149429, -0.557678), (0.678892, 0.453621, -0.408248, -0.408248), (-0.453621, 0.678892, 0.408248, -0.408248), (0.361127, 0.732294, -0.149429, -0.557678), (-0.732294, 0.361127, 0.557678, -0.149429), (0.576114, 0.0377605, 0.788675, 0.211325), (-0.0377605, 0.576114, -0.211325, 0.788675), (0.480049, 0.320759, 0.57735, 0.57735), (-0.320759, 0.480049, -0.57735, 0.57735), (0.255355, 0.51781, 0.211325, 0.788675), (-0.51781, 0.255355, -0.788675, 0.211325), (0.976849, 0.0640261, -0.0902818, 0.183073), (-0.0640261, 0.976849, -0.183073, -0.0902818), (0.813963, 0.543873, -0.169723, 0.113405), (-0.543873, 0.813963, -0.113405, -0.169723), (0.432976, 0.877989, -0.203687, 0.0133504), (-0.877989, 0.432976, -0.0133504, -0.203687), (0.576114, 0.0377605, 0.497052, -0.64777), (-0.0377605, 0.576114, 0.64777, 0.497052), (0.480049, 0.320759, 0.754344, -0.31246), (-0.320759, 0.480049, 0.31246, 0.754344), (0.255355, 0.51781, 0.809511, 0.106574), (-0.51781, 0.255355, -0.106574, 0.809511), (0.576114, 0.0377605, -0.497052, 0.64777), (-0.0377605, 0.576114, -0.64777, -0.497052), (0.480049, 0.320759, -0.754344, 0.31246), (-0.320759, 0.480049, -0.31246, -0.754344), (0.255355, 0.51781, -0.809511, -0.106574), (-0.51781, 0.255355, 0.106574, -0.809511), (0.976849, 0.0640261, 0.0902818, -0.183073), (-0.0640261, 0.976849, 0.183073, 0.0902818), (0.813963, 0.543873, 0.169723, -0.113405), (-0.543873, 0.813963, 0.113405, 0.169723), (0.432976, 0.877989, 0.203687, -0.0133504), (-0.877989, 0.432976, 0.0133504, 0.203687), (0.576114, 0.0377605, -0.211325, 0.788675), (-0.0377605, 0.576114, -0.788675, -0.211325), (0.480049, 0.320759, -0.57735, 0.57735), (-0.320759, 0.480049, -0.57735, -0.57735), (0.255355, 0.51781, -0.788675, 0.211325), (-0.51781, 0.255355, -0.211325, -0.788675), (0.814748, 0.0534014, 0.149429, -0.557678), (-0.0534014, 0.814748, 0.557678, 0.149429), (0.678892, 0.453621, 0.408248, -0.408248), (-0.453621, 0.678892, 0.408248, 0.408248), (0.361127, 0.732294, 0.557678, -0.149429), (-0.732294, 0.361127, 0.149429, 0.557678), (0.203687, 0.0133504, -0.877989, -0.432976), (-0.0133504, 0.203687, 0.432976, -0.877989), (0.169723, 0.113405, -0.543873, -0.813963), (-0.113405, 0.169723, 0.813963, -0.543873), (0.0902818, 0.183073, -0.0640261, -0.976849), (-0.183073, 0.0902818, 0.976849, -0.0640261), (0.814748, 0.0534014, 0.458043, 0.351469), (-0.0534014, 0.814748, -0.351469, 0.458043), (0.678892, 0.453621, 0.220942, 0.533402), (-0.453621, 0.678892, -0.533402, 0.220942), (0.361127, 0.732294, -0.0753593, 0.572411), (-0.732294, 0.361127, -0.572411, -0.0753593), (0.814748, 0.0534014, -0.458043, -0.351469), (-0.0534014, 0.814748, 0.351469, -0.458043), (0.678892, 0.453621, -0.220942, -0.533402), (-0.453621, 0.678892, 0.533402, -0.220942), (0.361127, 0.732294, 0.0753593, -0.572411), (-0.732294, 0.361127, 0.572411, 0.0753593), (0.203687, 0.0133504, 0.877989, 0.432976), (-0.0133504, 0.203687, -0.432976, 0.877989), (0.169723, 0.113405, 0.543873, 0.813963), (-0.113405, 0.169723, -0.813963, 0.543873), (0.0902818, 0.183073, 0.0640261, 0.976849), (-0.183073, 0.0902818, -0.976849, 0.0640261), (0.54671, 0.185583, -0.707107, -0.408248), (-0.185583, 0.54671, 0.408248, -0.707107), (0.380673, 0.434075, -0.408248, -0.707107), (-0.434075, 0.380673, 0.707107, -0.408248), (0.112635, 0.566257, -1.49988e-16, -0.816497), (-0.566257, 0.112635, 0.816497, -1.99984e-16), (0.773165, 0.262454, 0.5, 0.288675), (-0.262454, 0.773165, -0.288675, 0.5), (0.538354, 0.613875, 0.288675, 0.5), (-0.613875, 0.538354, -0.5, 0.288675), (0.159291, 0.800808, 3.53525e-17, 0.57735), (-0.800808, 0.159291, -0.57735, 7.0705e-17), (0.773165, 0.262454, -0.458043, 0.351469), (-0.262454, 0.773165, -0.351469, -0.458043), (0.538354, 0.613875, -0.572411, 0.0753593), (-0.613875, 0.538354, -0.0753593, -0.572411), (0.159291, 0.800808, -0.533402, -0.220942), (-0.800808, 0.159291, 0.220942, -0.533402), (0.193291, 0.0656136, 0.645463, -0.73601), (-0.0656136, 0.193291, 0.73601, 0.645463), (0.134588, 0.153469, 0.926993, -0.314672), (-0.153469, 0.134588, 0.314672, 0.926993), (0.0398226, 0.200202, 0.960135, 0.190983), (-0.200202, 0.0398226, -0.190983, 0.960135), (0.193291, 0.0656136, -0.645463, 0.73601), (-0.0656136, 0.193291, -0.73601, -0.645463), (0.134588, 0.153469, -0.926993, 0.314672), (-0.153469, 0.134588, -0.314672, -0.926993), (0.0398226, 0.200202, -0.960135, -0.190983), (-0.200202, 0.0398226, 0.190983, -0.960135), (0.773165, 0.262454, 0.458043, -0.351469), (-0.262454, 0.773165, 0.351469, 0.458043), (0.538354, 0.613875, 0.572411, -0.0753593), (-0.613875, 0.538354, 0.0753593, 0.572411), (0.159291, 0.800808, 0.533402, 0.220942), (-0.800808, 0.159291, -0.220942, 0.533402), (0.773165, 0.262454, -0.288675, 0.5), (-0.262454, 0.773165, -0.5, -0.288675), (0.538354, 0.613875, -0.5, 0.288675), (-0.613875, 0.538354, -0.288675, -0.5), (0.159291, 0.800808, -0.57735, 7.0705e-17), (-0.800808, 0.159291, -1.06058e-16, -0.57735), (0.54671, 0.185583, 0.408248, -0.707107), (-0.185583, 0.54671, 0.707107, 0.408248), (0.380673, 0.434075, 0.707107, -0.408248), (-0.434075, 0.380673, 0.408248, 0.707107), (0.112635, 0.566257, 0.816497, -1.99984e-16), (-0.566257, 0.112635, 2.4998e-16, 0.816497), (0.54671, 0.185583, -0.497052, -0.64777), (-0.185583, 0.54671, 0.64777, -0.497052), (0.380673, 0.434075, -0.106574, -0.809511), (-0.434075, 0.380673, 0.809511, -0.106574), (0.112635, 0.566257, 0.31246, -0.754344), (-0.566257, 0.112635, 0.754344, 0.31246), (0.926993, 0.314672, 0.153469, 0.134588), (-0.314672, 0.926993, -0.134588, 0.153469), (0.645463, 0.73601, 0.0656136, 0.193291), (-0.73601, 0.645463, -0.193291, 0.0656136), (0.190983, 0.960135, -0.0398226, 0.200202), (-0.960135, 0.190983, -0.200202, -0.0398226), (0.926993, 0.314672, -0.153469, -0.134588), (-0.314672, 0.926993, 0.134588, -0.153469), (0.645463, 0.73601, -0.0656136, -0.193291), (-0.73601, 0.645463, 0.193291, -0.0656136), (0.190983, 0.960135, 0.0398226, -0.200202), (-0.960135, 0.190983, 0.200202, 0.0398226), (0.54671, 0.185583, 0.497052, 0.64777), (-0.185583, 0.54671, -0.64777, 0.497052), (0.380673, 0.434075, 0.106574, 0.809511), (-0.434075, 0.380673, -0.809511, 0.106574), (0.112635, 0.566257, -0.31246, 0.754344), (-0.566257, 0.112635, -0.754344, -0.31246), (0.773165, 0.262454, -0.5, -0.288675), (-0.262454, 0.773165, 0.288675, -0.5), (0.538354, 0.613875, -0.288675, -0.5), (-0.613875, 0.538354, 0.5, -0.288675), (0.159291, 0.800808, -1.06058e-16, -0.57735), (-0.800808, 0.159291, 0.57735, -1.4141e-16), (0.54671, 0.185583, 0.707107, 0.408248), (-0.185583, 0.54671, -0.408248, 0.707107), (0.380673, 0.434075, 0.408248, 0.707107), (-0.434075, 0.380673, -0.707107, 0.408248), (0.112635, 0.566257, 4.9996e-17, 0.816497), (-0.566257, 0.112635, -0.816497, 9.9992e-17), (0.926993, 0.314672, -0.134588, 0.153469), (-0.314672, 0.926993, -0.153469, -0.134588), (0.645463, 0.73601, -0.193291, 0.0656136), (-0.73601, 0.645463, -0.0656136, -0.193291), (0.190983, 0.960135, -0.200202, -0.0398226), (-0.960135, 0.190983, 0.0398226, -0.200202), (0.54671, 0.185583, 0.64777, -0.497052), (-0.185583, 0.54671, 0.497052, 0.64777), (0.380673, 0.434075, 0.809511, -0.106574), (-0.434075, 0.380673, 0.106574, 0.809511), (0.112635, 0.566257, 0.754344, 0.31246), (-0.566257, 0.112635, -0.31246, 0.754344), (0.54671, 0.185583, -0.64777, 0.497052), (-0.185583, 0.54671, -0.497052, -0.64777), (0.380673, 0.434075, -0.809511, 0.106574), (-0.434075, 0.380673, -0.106574, -0.809511), (0.112635, 0.566257, -0.754344, -0.31246), (-0.566257, 0.112635, 0.31246, -0.754344), (0.926993, 0.314672, 0.134588, -0.153469), (-0.314672, 0.926993, 0.153469, 0.134588), (0.645463, 0.73601, 0.193291, -0.0656136), (-0.73601, 0.645463, 0.0656136, 0.193291), (0.190983, 0.960135, 0.200202, 0.0398226), (-0.960135, 0.190983, -0.0398226, 0.200202), (0.54671, 0.185583, -0.408248, 0.707107), (-0.185583, 0.54671, -0.707107, -0.408248), (0.380673, 0.434075, -0.707107, 0.408248), (-0.434075, 0.380673, -0.408248, -0.707107), (0.112635, 0.566257, -0.816497, 9.9992e-17), (-0.566257, 0.112635, -1.49988e-16, -0.816497), (0.773165, 0.262454, 0.288675, -0.5), (-0.262454, 0.773165, 0.5, 0.288675), (0.538354, 0.613875, 0.5, -0.288675), (-0.613875, 0.538354, 0.288675, 0.5), (0.159291, 0.800808, 0.57735, -1.4141e-16), (-0.800808, 0.159291, 1.76763e-16, 0.57735), (0.193291, 0.0656136, -0.73601, -0.645463), (-0.0656136, 0.193291, 0.645463, -0.73601), (0.134588, 0.153469, -0.314672, -0.926993), (-0.153469, 0.134588, 0.926993, -0.314672), (0.0398226, 0.200202, 0.190983, -0.960135), (-0.200202, 0.0398226, 0.960135, 0.190983), (0.773165, 0.262454, 0.351469, 0.458043), (-0.262454, 0.773165, -0.458043, 0.351469), (0.538354, 0.613875, 0.0753593, 0.572411), (-0.613875, 0.538354, -0.572411, 0.0753593), (0.159291, 0.800808, -0.220942, 0.533402), (-0.800808, 0.159291, -0.533402, -0.220942), (0.773165, 0.262454, -0.351469, -0.458043), (-0.262454, 0.773165, 0.458043, -0.351469), (0.538354, 0.613875, -0.0753593, -0.572411), (-0.613875, 0.538354, 0.572411, -0.0753593), (0.159291, 0.800808, 0.220942, -0.533402), (-0.800808, 0.159291, 0.533402, 0.220942), (0.193291, 0.0656136, 0.73601, 0.645463), (-0.0656136, 0.193291, -0.645463, 0.73601), (0.134588, 0.153469, 0.314672, 0.926993), (-0.153469, 0.134588, -0.926993, 0.314672), (0.0398226, 0.200202, -0.190983, 0.960135), (-0.200202, 0.0398226, -0.960135, -0.190983), (0.576114, 0.0377605, -0.788675, -0.211325), (-0.0377605, 0.576114, 0.211325, -0.788675), (0.480049, 0.320759, -0.57735, -0.57735), (-0.320759, 0.480049, 0.57735, -0.57735), (0.255355, 0.51781, -0.211325, -0.788675), (-0.51781, 0.255355, 0.788675, -0.211325), (0.814748, 0.0534014, 0.557678, 0.149429), (-0.0534014, 0.814748, -0.149429, 0.557678), (0.678892, 0.453621, 0.408248, 0.408248), (-0.453621, 0.678892, -0.408248, 0.408248), (0.361127, 0.732294, 0.149429, 0.557678), (-0.732294, 0.361127, -0.557678, 0.149429), (0.814748, 0.0534014, -0.351469, 0.458043), (-0.0534014, 0.814748, -0.458043, -0.351469), (0.678892, 0.453621, -0.533402, 0.220942), (-0.453621, 0.678892, -0.220942, -0.533402), (0.361127, 0.732294, -0.572411, -0.0753593), (-0.732294, 0.361127, 0.0753593, -0.572411), (0.203687, 0.0133504, 0.432976, -0.877989), (-0.0133504, 0.203687, 0.877989, 0.432976), (0.169723, 0.113405, 0.813963, -0.543873), (-0.113405, 0.169723, 0.543873, 0.813963), (0.0902818, 0.183073, 0.976849, -0.0640261), (-0.183073, 0.0902818, 0.0640261, 0.976849), (0.203687, 0.0133504, -0.432976, 0.877989), (-0.0133504, 0.203687, -0.877989, -0.432976), (0.169723, 0.113405, -0.813963, 0.543873), (-0.113405, 0.169723, -0.543873, -0.813963), (0.0902818, 0.183073, -0.976849, 0.0640261), (-0.183073, 0.0902818, -0.0640261, -0.976849), (0.814748, 0.0534014, 0.351469, -0.458043), (-0.0534014, 0.814748, 0.458043, 0.351469), (0.678892, 0.453621, 0.533402, -0.220942), (-0.453621, 0.678892, 0.220942, 0.533402), (0.361127, 0.732294, 0.572411, 0.0753593), (-0.732294, 0.361127, -0.0753593, 0.572411), (0.814748, 0.0534014, -0.149429, 0.557678), (-0.0534014, 0.814748, -0.557678, -0.149429), (0.678892, 0.453621, -0.408248, 0.408248), (-0.453621, 0.678892, -0.408248, -0.408248), (0.361127, 0.732294, -0.557678, 0.149429), (-0.732294, 0.361127, -0.149429, -0.557678), (0.576114, 0.0377605, 0.211325, -0.788675), (-0.0377605, 0.576114, 0.788675, 0.211325), (0.480049, 0.320759, 0.57735, -0.57735), (-0.320759, 0.480049, 0.57735, 0.57735), (0.255355, 0.51781, 0.788675, -0.211325), (-0.51781, 0.255355, 0.211325, 0.788675), (0.576114, 0.0377605, -0.64777, -0.497052), (-0.0377605, 0.576114, 0.497052, -0.64777), (0.480049, 0.320759, -0.31246, -0.754344), (-0.320759, 0.480049, 0.754344, -0.31246), (0.255355, 0.51781, 0.106574, -0.809511), (-0.51781, 0.255355, 0.809511, 0.106574), (0.976849, 0.0640261, 0.183073, 0.0902818), (-0.0640261, 0.976849, -0.0902818, 0.183073), (0.813963, 0.543873, 0.113405, 0.169723), (-0.543873, 0.813963, -0.169723, 0.113405), (0.432976, 0.877989, 0.0133504, 0.203687), (-0.877989, 0.432976, -0.203687, 0.0133504), (0.976849, 0.0640261, -0.183073, -0.0902818), (-0.0640261, 0.976849, 0.0902818, -0.183073), (0.813963, 0.543873, -0.113405, -0.169723), (-0.543873, 0.813963, 0.169723, -0.113405), (0.432976, 0.877989, -0.0133504, -0.203687), (-0.877989, 0.432976, 0.203687, -0.0133504), (0.576114, 0.0377605, 0.64777, 0.497052), (-0.0377605, 0.576114, -0.497052, 0.64777), (0.480049, 0.320759, 0.31246, 0.754344), (-0.320759, 0.480049, -0.754344, 0.31246), (0.255355, 0.51781, -0.106574, 0.809511), (-0.51781, 0.255355, -0.809511, -0.106574), (0.705593, 0.046247, -0.701057, 0.092296), (-0.046247, 0.705593, -0.092296, -0.701057), (0.587938, 0.392847, -0.653281, -0.270598), (-0.392847, 0.587938, 0.270598, -0.653281), (0.312745, 0.634185, -0.430459, -0.560986), (-0.634185, 0.312745, 0.560986, -0.430459), (0.705593, 0.046247, 0.701057, -0.092296), (-0.046247, 0.705593, 0.092296, 0.701057), (0.587938, 0.392847, 0.653281, 0.270598), (-0.392847, 0.587938, -0.270598, 0.653281), (0.312745, 0.634185, 0.430459, 0.560986), (-0.634185, 0.312745, -0.560986, 0.430459), (0.910916, 0.0597046, -0.353553, 0.204124), (-0.0597046, 0.910916, -0.204124, -0.353553), (0.759024, 0.507164, -0.408248, 4.9996e-17), (-0.507164, 0.759024, -7.4994e-17, -0.408248), (0.403752, 0.818729, -0.353553, -0.204124), (-0.818729, 0.403752, 0.204124, -0.353553), (0.407374, 0.0267007, 0.790569, -0.456435), (-0.0267007, 0.407374, 0.456435, 0.790569), (0.339446, 0.226811, 0.912871, -2.23589e-16), (-0.226811, 0.339446, 2.79486e-16, 0.912871), (0.180564, 0.366147, 0.790569, 0.456435), (-0.366147, 0.180564, -0.456435, 0.790569), (0.407374, 0.0267007, -0.790569, 0.456435), (-0.0267007, 0.407374, -0.456435, -0.790569), (0.339446, 0.226811, -0.912871, 1.11794e-16), (-0.226811, 0.339446, -1.67692e-16, -0.912871), (0.180564, 0.366147, -0.790569, -0.456435), (-0.366147, 0.180564, 0.456435, -0.790569), (0.910916, 0.0597046, 0.353553, -0.204124), (-0.0597046, 0.910916, 0.204124, 0.353553), (0.759024, 0.507164, 0.408248, -9.9992e-17), (-0.507164, 0.759024, 1.2499e-16, 0.408248), (0.403752, 0.818729, 0.353553, 0.204124), (-0.818729, 0.403752, -0.204124, 0.353553), (0.705593, 0.046247, 0.092296, 0.701057), (-0.046247, 0.705593, -0.701057, 0.092296), (0.587938, 0.392847, -0.270598, 0.653281), (-0.392847, 0.587938, -0.653281, -0.270598), (0.312745, 0.634185, -0.560986, 0.430459), (-0.634185, 0.312745, -0.430459, -0.560986), (0.705593, 0.046247, -0.092296, -0.701057), (-0.046247, 0.705593, 0.701057, -0.092296), (0.587938, 0.392847, 0.270598, -0.653281), (-0.392847, 0.587938, 0.653281, 0.270598), (0.312745, 0.634185, 0.560986, -0.430459), (-0.634185, 0.312745, 0.430459, 0.560986), (0.407374, 0.0267007, -0.456435, -0.790569), (-0.0267007, 0.407374, 0.790569, -0.456435), (0.339446, 0.226811, -1.67692e-16, -0.912871), (-0.226811, 0.339446, 0.912871, -2.23589e-16), (0.180564, 0.366147, 0.456435, -0.790569), (-0.366147, 0.180564, 0.790569, 0.456435), (0.910916, 0.0597046, 0.204124, 0.353553), (-0.0597046, 0.910916, -0.353553, 0.204124), (0.759024, 0.507164, 2.4998e-17, 0.408248), (-0.507164, 0.759024, -0.408248, 4.9996e-17), (0.403752, 0.818729, -0.204124, 0.353553), (-0.818729, 0.403752, -0.353553, -0.204124), (0.910916, 0.0597046, -0.204124, -0.353553), (-0.0597046, 0.910916, 0.353553, -0.204124), (0.759024, 0.507164, -7.4994e-17, -0.408248), (-0.507164, 0.759024, 0.408248, -9.9992e-17), (0.403752, 0.818729, 0.204124, -0.353553), (-0.818729, 0.403752, 0.353553, 0.204124), (0.407374, 0.0267007, 0.456435, 0.790569), (-0.0267007, 0.407374, -0.790569, 0.456435), (0.339446, 0.226811, 5.58972e-17, 0.912871), (-0.226811, 0.339446, -0.912871, 1.11794e-16), (0.180564, 0.366147, -0.456435, 0.790569), (-0.366147, 0.180564, -0.790569, -0.456435), (0.669581, 0.227292, -0.430459, -0.560986), (-0.227292, 0.669581, 0.560986, -0.430459), (0.466228, 0.531631, -0.092296, -0.701057), (-0.531631, 0.466228, 0.701057, -0.092296), (0.13795, 0.69352, 0.270598, -0.653281), (-0.69352, 0.13795, 0.653281, 0.270598), (0.669581, 0.227292, 0.430459, 0.560986), (-0.227292, 0.669581, -0.560986, 0.430459), (0.466228, 0.531631, 0.092296, 0.701057), (-0.531631, 0.466228, -0.701057, 0.092296), (0.13795, 0.69352, -0.270598, 0.653281), (-0.69352, 0.13795, -0.653281, -0.270598), (0.864425, 0.293433, -0.204124, 0.353553), (-0.293433, 0.864425, -0.353553, -0.204124), (0.601898, 0.686333, -0.353553, 0.204124), (-0.686333, 0.601898, -0.204124, -0.353553), (0.178092, 0.89533, -0.408248, 4.9996e-17), (-0.89533, 0.178092, -7.4994e-17, -0.408248), (0.386583, 0.131227, 0.456435, -0.790569), (-0.131227, 0.386583, 0.790569, 0.456435), (0.269177, 0.306937, 0.790569, -0.456435), (-0.306937, 0.269177, 0.456435, 0.790569), (0.0796453, 0.400404, 0.912871, -2.23589e-16), (-0.400404, 0.0796453, 2.79486e-16, 0.912871), (0.386583, 0.131227, -0.456435, 0.790569), (-0.131227, 0.386583, -0.790569, -0.456435), (0.269177, 0.306937, -0.790569, 0.456435), (-0.306937, 0.269177, -0.456435, -0.790569), (0.0796453, 0.400404, -0.912871, 1.11794e-16), (-0.400404, 0.0796453, -1.67692e-16, -0.912871), (0.864425, 0.293433, 0.204124, -0.353553), (-0.293433, 0.864425, 0.353553, 0.204124), (0.601898, 0.686333, 0.353553, -0.204124), (-0.686333, 0.601898, 0.204124, 0.353553), (0.178092, 0.89533, 0.408248, -9.9992e-17), (-0.89533, 0.178092, 1.2499e-16, 0.408248), (0.669581, 0.227292, -0.560986, 0.430459), (-0.227292, 0.669581, -0.430459, -0.560986), (0.466228, 0.531631, -0.701057, 0.092296), (-0.531631, 0.466228, -0.092296, -0.701057), (0.13795, 0.69352, -0.653281, -0.270598), (-0.69352, 0.13795, 0.270598, -0.653281), (0.669581, 0.227292, 0.560986, -0.430459), (-0.227292, 0.669581, 0.430459, 0.560986), (0.466228, 0.531631, 0.701057, -0.092296), (-0.531631, 0.466228, 0.092296, 0.701057), (0.13795, 0.69352, 0.653281, 0.270598), (-0.69352, 0.13795, -0.270598, 0.653281), (0.386583, 0.131227, -0.790569, -0.456435), (-0.131227, 0.386583, 0.456435, -0.790569), (0.269177, 0.306937, -0.456435, -0.790569), (-0.306937, 0.269177, 0.790569, -0.456435), (0.0796453, 0.400404, -1.67692e-16, -0.912871), (-0.400404, 0.0796453, 0.912871, -2.23589e-16), (0.864425, 0.293433, 0.353553, 0.204124), (-0.293433, 0.864425, -0.204124, 0.353553), (0.601898, 0.686333, 0.204124, 0.353553), (-0.686333, 0.601898, -0.353553, 0.204124), (0.178092, 0.89533, 2.4998e-17, 0.408248), (-0.89533, 0.178092, -0.408248, 4.9996e-17), (0.864425, 0.293433, -0.353553, -0.204124), (-0.293433, 0.864425, 0.204124, -0.353553), (0.601898, 0.686333, -0.204124, -0.353553), (-0.686333, 0.601898, 0.353553, -0.204124), (0.178092, 0.89533, -7.4994e-17, -0.408248), (-0.89533, 0.178092, 0.408248, -9.9992e-17), (0.386583, 0.131227, 0.790569, 0.456435), (-0.131227, 0.386583, -0.456435, 0.790569), (0.269177, 0.306937, 0.456435, 0.790569), (-0.306937, 0.269177, -0.790569, 0.456435), (0.0796453, 0.400404, 5.58972e-17, 0.912871), (-0.400404, 0.0796453, -0.912871, 1.11794e-16), (0.669581, 0.227292, -0.701057, -0.092296), (-0.227292, 0.669581, 0.092296, -0.701057), (0.466228, 0.531631, -0.560986, -0.430459), (-0.531631, 0.466228, 0.430459, -0.560986), (0.13795, 0.69352, -0.270598, -0.653281), (-0.69352, 0.13795, 0.653281, -0.270598), (0.669581, 0.227292, 0.701057, 0.092296), (-0.227292, 0.669581, -0.092296, 0.701057), (0.466228, 0.531631, 0.560986, 0.430459), (-0.531631, 0.466228, -0.430459, 0.560986), (0.13795, 0.69352, 0.270598, 0.653281), (-0.69352, 0.13795, -0.653281, 0.270598), (0.864425, 0.293433, -0.394338, 0.105662), (-0.293433, 0.864425, -0.105662, -0.394338), (0.601898, 0.686333, -0.394338, -0.105662), (-0.686333, 0.601898, 0.105662, -0.394338), (0.178092, 0.89533, -0.288675, -0.288675), (-0.89533, 0.178092, 0.288675, -0.288675), (0.386583, 0.131227, 0.881766, -0.236268), (-0.131227, 0.386583, 0.236268, 0.881766), (0.269177, 0.306937, 0.881766, 0.236268), (-0.306937, 0.269177, -0.236268, 0.881766), (0.0796453, 0.400404, 0.645497, 0.645497), (-0.400404, 0.0796453, -0.645497, 0.645497), (0.386583, 0.131227, -0.881766, 0.236268), (-0.131227, 0.386583, -0.236268, -0.881766), (0.269177, 0.306937, -0.881766, -0.236268), (-0.306937, 0.269177, 0.236268, -0.881766), (0.0796453, 0.400404, -0.645497, -0.645497), (-0.400404, 0.0796453, 0.645497, -0.645497), (0.864425, 0.293433, 0.394338, -0.105662), (-0.293433, 0.864425, 0.105662, 0.394338), (0.601898, 0.686333, 0.394338, 0.105662), (-0.686333, 0.601898, -0.105662, 0.394338), (0.178092, 0.89533, 0.288675, 0.288675), (-0.89533, 0.178092, -0.288675, 0.288675), (0.669581, 0.227292, -0.092296, 0.701057), (-0.227292, 0.669581, -0.701057, -0.092296), (0.466228, 0.531631, -0.430459, 0.560986), (-0.531631, 0.466228, -0.560986, -0.430459), (0.13795, 0.69352, -0.653281, 0.270598), (-0.69352, 0.13795, -0.270598, -0.653281), (0.669581, 0.227292, 0.092296, -0.701057), (-0.227292, 0.669581, 0.701057, 0.092296), (0.466228, 0.531631, 0.430459, -0.560986), (-0.531631, 0.466228, 0.560986, 0.430459), (0.13795, 0.69352, 0.653281, -0.270598), (-0.69352, 0.13795, 0.270598, 0.653281), (0.386583, 0.131227, -0.236268, -0.881766), (-0.131227, 0.386583, 0.881766, -0.236268), (0.269177, 0.306937, 0.236268, -0.881766), (-0.306937, 0.269177, 0.881766, 0.236268), (0.0796453, 0.400404, 0.645497, -0.645497), (-0.400404, 0.0796453, 0.645497, 0.645497), (0.864425, 0.293433, 0.105662, 0.394338), (-0.293433, 0.864425, -0.394338, 0.105662), (0.601898, 0.686333, -0.105662, 0.394338), (-0.686333, 0.601898, -0.394338, -0.105662), (0.178092, 0.89533, -0.288675, 0.288675), (-0.89533, 0.178092, -0.288675, -0.288675), (0.864425, 0.293433, -0.105662, -0.394338), (-0.293433, 0.864425, 0.394338, -0.105662), (0.601898, 0.686333, 0.105662, -0.394338), (-0.686333, 0.601898, 0.394338, 0.105662), (0.178092, 0.89533, 0.288675, -0.288675), (-0.89533, 0.178092, 0.288675, 0.288675), (0.386583, 0.131227, 0.236268, 0.881766), (-0.131227, 0.386583, -0.881766, 0.236268), (0.269177, 0.306937, -0.236268, 0.881766), (-0.306937, 0.269177, -0.881766, -0.236268), (0.0796453, 0.400404, -0.645497, 0.645497), (-0.400404, 0.0796453, -0.645497, -0.645497), (0.705593, 0.046247, -0.560986, -0.430459), (-0.046247, 0.705593, 0.430459, -0.560986), (0.587938, 0.392847, -0.270598, -0.653281), (-0.392847, 0.587938, 0.653281, -0.270598), (0.312745, 0.634185, 0.092296, -0.701057), (-0.634185, 0.312745, 0.701057, 0.092296), (0.705593, 0.046247, 0.560986, 0.430459), (-0.046247, 0.705593, -0.430459, 0.560986), (0.587938, 0.392847, 0.270598, 0.653281), (-0.392847, 0.587938, -0.653281, 0.270598), (0.312745, 0.634185, -0.092296, 0.701057), (-0.634185, 0.312745, -0.701057, -0.092296), (0.910916, 0.0597046, -0.105662, 0.394338), (-0.0597046, 0.910916, -0.394338, -0.105662), (0.759024, 0.507164, -0.288675, 0.288675), (-0.507164, 0.759024, -0.288675, -0.288675), (0.403752, 0.818729, -0.394338, 0.105662), (-0.818729, 0.403752, -0.105662, -0.394338), (0.407374, 0.0267007, 0.236268, -0.881766), (-0.0267007, 0.407374, 0.881766, 0.236268), (0.339446, 0.226811, 0.645497, -0.645497), (-0.226811, 0.339446, 0.645497, 0.645497), (0.180564, 0.366147, 0.881766, -0.236268), (-0.366147, 0.180564, 0.236268, 0.881766), (0.407374, 0.0267007, -0.236268, 0.881766), (-0.0267007, 0.407374, -0.881766, -0.236268), (0.339446, 0.226811, -0.645497, 0.645497), (-0.226811, 0.339446, -0.645497, -0.645497), (0.180564, 0.366147, -0.881766, 0.236268), (-0.366147, 0.180564, -0.236268, -0.881766), (0.910916, 0.0597046, 0.105662, -0.394338), (-0.0597046, 0.910916, 0.394338, 0.105662), (0.759024, 0.507164, 0.288675, -0.288675), (-0.507164, 0.759024, 0.288675, 0.288675), (0.403752, 0.818729, 0.394338, -0.105662), (-0.818729, 0.403752, 0.105662, 0.394338), (0.705593, 0.046247, -0.430459, 0.560986), (-0.046247, 0.705593, -0.560986, -0.430459), (0.587938, 0.392847, -0.653281, 0.270598), (-0.392847, 0.587938, -0.270598, -0.653281), (0.312745, 0.634185, -0.701057, -0.092296), (-0.634185, 0.312745, 0.092296, -0.701057), (0.705593, 0.046247, 0.430459, -0.560986), (-0.046247, 0.705593, 0.560986, 0.430459), (0.587938, 0.392847, 0.653281, -0.270598), (-0.392847, 0.587938, 0.270598, 0.653281), (0.312745, 0.634185, 0.701057, 0.092296), (-0.634185, 0.312745, -0.092296, 0.701057), (0.407374, 0.0267007, -0.881766, -0.236268), (-0.0267007, 0.407374, 0.236268, -0.881766), (0.339446, 0.226811, -0.645497, -0.645497), (-0.226811, 0.339446, 0.645497, -0.645497), (0.180564, 0.366147, -0.236268, -0.881766), (-0.366147, 0.180564, 0.881766, -0.236268), (0.910916, 0.0597046, 0.394338, 0.105662), (-0.0597046, 0.910916, -0.105662, 0.394338), (0.759024, 0.507164, 0.288675, 0.288675), (-0.507164, 0.759024, -0.288675, 0.288675), (0.403752, 0.818729, 0.105662, 0.394338), (-0.818729, 0.403752, -0.394338, 0.105662), (0.910916, 0.0597046, -0.394338, -0.105662), (-0.0597046, 0.910916, 0.105662, -0.394338), (0.759024, 0.507164, -0.288675, -0.288675), (-0.507164, 0.759024, 0.288675, -0.288675), (0.403752, 0.818729, -0.105662, -0.394338), (-0.818729, 0.403752, 0.394338, -0.105662), (0.407374, 0.0267007, 0.881766, 0.236268), (-0.0267007, 0.407374, -0.236268, 0.881766), (0.339446, 0.226811, 0.645497, 0.645497), (-0.226811, 0.339446, -0.645497, 0.645497), (0.180564, 0.366147, 0.236268, 0.881766), (-0.366147, 0.180564, -0.881766, 0.236268), (0.889545, 0.0194099, -0.456327, -0.00995709), (-0.0194099, 0.889545, 0.00995709, -0.456327), (0.760663, 0.461582, -0.390212, -0.236787), (-0.461582, 0.760663, 0.236787, -0.390212), (0.427963, 0.780073, -0.21954, -0.400169), (-0.780073, 0.427963, 0.400169, -0.21954), (0.456327, 0.00995709, 0.889545, 0.0194099), (-0.00995709, 0.456327, -0.0194099, 0.889545), (0.390212, 0.236787, 0.760663, 0.461582), (-0.236787, 0.390212, -0.461582, 0.760663), (0.21954, 0.400169, 0.427963, 0.780073), (-0.400169, 0.21954, -0.780073, 0.427963), (0.998459, 0.0217865, -0.036863, 0.0352886), (-0.0217865, 0.998459, -0.0352886, -0.036863), (0.853798, 0.518097, -0.0495686, 0.0121294), (-0.518097, 0.853798, -0.0121294, -0.0495686), (0.480362, 0.875584, -0.0489923, -0.01428), (-0.875584, 0.480362, 0.01428, -0.0489923), (0.676842, 0.0147687, 0.531645, -0.50894), (-0.0147687, 0.676842, 0.50894, 0.531645), (0.578778, 0.351211, 0.714888, -0.174932), (-0.351211, 0.578778, 0.174932, 0.714888), (0.325631, 0.593547, 0.706578, 0.205949), (-0.593547, 0.325631, -0.205949, 0.706578), (0.676842, 0.0147687, -0.531645, 0.50894), (-0.0147687, 0.676842, -0.50894, -0.531645), (0.578778, 0.351211, -0.714888, 0.174932), (-0.351211, 0.578778, -0.174932, -0.714888), (0.325631, 0.593547, -0.706578, -0.205949), (-0.593547, 0.325631, 0.205949, -0.706578), (0.998459, 0.0217865, 0.036863, -0.0352886), (-0.0217865, 0.998459, 0.0352886, 0.036863), (0.853798, 0.518097, 0.0495686, -0.0121294), (-0.518097, 0.853798, 0.0121294, 0.0495686), (0.480362, 0.875584, 0.0489923, 0.01428), (-0.875584, 0.480362, -0.01428, 0.0489923), (0.456327, 0.00995709, -0.0194099, 0.889545), (-0.00995709, 0.456327, -0.889545, -0.0194099), (0.390212, 0.236787, -0.461582, 0.760663), (-0.236787, 0.390212, -0.760663, -0.461582), (0.21954, 0.400169, -0.780073, 0.427963), (-0.400169, 0.21954, -0.427963, -0.780073), (0.889545, 0.0194099, 0.00995709, -0.456327), (-0.0194099, 0.889545, 0.456327, 0.00995709), (0.760663, 0.461582, 0.236787, -0.390212), (-0.461582, 0.760663, 0.390212, 0.236787), (0.427963, 0.780073, 0.400169, -0.21954), (-0.780073, 0.427963, 0.21954, 0.400169), (0.0510189, 0.00111324, -0.690612, -0.721423), (-0.00111324, 0.0510189, 0.721423, -0.690612), (0.043627, 0.0264735, -0.237376, -0.970076), (-0.0264735, 0.043627, 0.970076, -0.237376), (0.0245454, 0.0447403, 0.279464, -0.958799), (-0.0447403, 0.0245454, 0.958799, 0.279464), (0.735805, 0.0160553, 0.468157, 0.489043), (-0.0160553, 0.735805, -0.489043, 0.468157), (0.629198, 0.381807, 0.160914, 0.657602), (-0.381807, 0.629198, -0.657602, 0.160914), (0.353998, 0.645253, -0.189445, 0.649957), (-0.645253, 0.353998, -0.649957, -0.189445), (0.735805, 0.0160553, -0.468157, -0.489043), (-0.0160553, 0.735805, 0.489043, -0.468157), (0.629198, 0.381807, -0.160914, -0.657602), (-0.381807, 0.629198, 0.657602, -0.160914), (0.353998, 0.645253, 0.189445, -0.649957), (-0.645253, 0.353998, 0.649957, 0.189445), (0.0510189, 0.00111324, 0.690612, 0.721423), (-0.00111324, 0.0510189, -0.721423, 0.690612), (0.043627, 0.0264735, 0.237376, 0.970076), (-0.0264735, 0.043627, -0.970076, 0.237376), (0.0245454, 0.0447403, -0.279464, 0.958799), (-0.0447403, 0.0245454, -0.958799, -0.279464), (0.649957, 0.189445, -0.706578, -0.205949), (-0.189445, 0.649957, 0.205949, -0.706578), (0.468157, 0.489043, -0.50894, -0.531645), (-0.489043, 0.468157, 0.531645, -0.50894), (0.160914, 0.657602, -0.174932, -0.714888), (-0.657602, 0.160914, 0.714888, -0.174932), (0.706578, 0.205949, 0.649957, 0.189445), (-0.205949, 0.706578, -0.189445, 0.649957), (0.50894, 0.531645, 0.468157, 0.489043), (-0.531645, 0.50894, -0.489043, 0.468157), (0.174932, 0.714888, 0.160914, 0.657602), (-0.714888, 0.174932, -0.657602, 0.160914), (0.854211, 0.24898, -0.400169, 0.21954), (-0.24898, 0.854211, -0.21954, -0.400169), (0.615278, 0.642728, -0.456327, -0.00995709), (-0.642728, 0.615278, 0.00995709, -0.456327), (0.211483, 0.864258, -0.390212, -0.236787), (-0.864258, 0.211483, 0.236787, -0.390212), (0.342946, 0.0999598, 0.818881, -0.449254), (-0.0999598, 0.342946, 0.449254, 0.818881), (0.24702, 0.258041, 0.933799, 0.0203756), (-0.258041, 0.24702, -0.0203756, 0.933799), (0.0849055, 0.34698, 0.798506, 0.484545), (-0.34698, 0.0849055, -0.484545, 0.798506), (0.342946, 0.0999598, -0.818881, 0.449254), (-0.0999598, 0.342946, -0.449254, -0.818881), (0.24702, 0.258041, -0.933799, -0.0203756), (-0.258041, 0.24702, 0.0203756, -0.933799), (0.0849055, 0.34698, -0.798506, -0.484545), (-0.34698, 0.0849055, 0.484545, -0.798506), (0.854211, 0.24898, 0.400169, -0.21954), (-0.24898, 0.854211, 0.21954, 0.400169), (0.615278, 0.642728, 0.456327, 0.00995709), (-0.642728, 0.615278, -0.00995709, 0.456327), (0.211483, 0.864258, 0.390212, 0.236787), (-0.864258, 0.211483, -0.236787, 0.390212), (0.706578, 0.205949, -0.189445, 0.649957), (-0.205949, 0.706578, -0.649957, -0.189445), (0.50894, 0.531645, -0.489043, 0.468157), (-0.531645, 0.50894, -0.468157, -0.489043), (0.174932, 0.714888, -0.657602, 0.160914), (-0.714888, 0.174932, -0.160914, -0.657602), (0.649957, 0.189445, 0.205949, -0.706578), (-0.189445, 0.649957, 0.706578, 0.205949), (0.468157, 0.489043, 0.531645, -0.50894), (-0.489043, 0.468157, 0.50894, 0.531645), (0.160914, 0.657602, 0.714888, -0.174932), (-0.657602, 0.160914, 0.174932, 0.714888), (0.438201, 0.127724, -0.427963, -0.780073), (-0.127724, 0.438201, 0.780073, -0.427963), (0.315631, 0.329713, 0.0194099, -0.889545), (-0.329713, 0.315631, 0.889545, 0.0194099), (0.108488, 0.443355, 0.461582, -0.760663), (-0.443355, 0.108488, 0.760663, 0.461582), (0.896707, 0.261366, 0.171817, 0.313182), (-0.261366, 0.896707, -0.313182, 0.171817), (0.645888, 0.674703, -0.00779265, 0.357132), (-0.674703, 0.645888, -0.357132, -0.00779265), (0.222004, 0.907254, -0.185315, 0.305389), (-0.907254, 0.222004, -0.305389, -0.185315), (0.896707, 0.261366, -0.171817, -0.313182), (-0.261366, 0.896707, 0.313182, -0.171817), (0.645888, 0.674703, 0.00779265, -0.357132), (-0.674703, 0.645888, 0.357132, 0.00779265), (0.222004, 0.907254, 0.185315, -0.305389), (-0.907254, 0.222004, 0.305389, 0.185315), (0.438201, 0.127724, 0.427963, 0.780073), (-0.127724, 0.438201, -0.780073, 0.427963), (0.315631, 0.329713, -0.0194099, 0.889545), (-0.329713, 0.315631, -0.889545, -0.0194099), (0.108488, 0.443355, -0.461582, 0.760663), (-0.443355, 0.108488, -0.760663, -0.461582), (0.854211, 0.24898, -0.438201, -0.127724), (-0.24898, 0.854211, 0.127724, -0.438201), (0.615278, 0.642728, -0.315631, -0.329713), (-0.642728, 0.615278, 0.329713, -0.315631), (0.211483, 0.864258, -0.108488, -0.443355), (-0.864258, 0.211483, 0.443355, -0.108488), (0.438201, 0.127724, 0.854211, 0.24898), (-0.127724, 0.438201, -0.24898, 0.854211), (0.315631, 0.329713, 0.615278, 0.642728), (-0.329713, 0.315631, -0.642728, 0.615278), (0.108488, 0.443355, 0.211483, 0.864258), (-0.443355, 0.108488, -0.864258, 0.211483), (0.958799, 0.279464, -0.0447403, 0.0245454), (-0.279464, 0.958799, -0.0245454, -0.0447403), (0.690612, 0.721423, -0.0510189, -0.00111324), (-0.721423, 0.690612, 0.00111324, -0.0510189), (0.237376, 0.970076, -0.043627, -0.0264735), (-0.970076, 0.237376, 0.0264735, -0.043627), (0.649957, 0.189445, 0.645253, -0.353998), (-0.189445, 0.649957, 0.353998, 0.645253), (0.468157, 0.489043, 0.735805, 0.0160553), (-0.489043, 0.468157, -0.0160553, 0.735805), (0.160914, 0.657602, 0.629198, 0.381807), (-0.657602, 0.160914, -0.381807, 0.629198), (0.649957, 0.189445, -0.645253, 0.353998), (-0.189445, 0.649957, -0.353998, -0.645253), (0.468157, 0.489043, -0.735805, -0.0160553), (-0.489043, 0.468157, 0.0160553, -0.735805), (0.160914, 0.657602, -0.629198, -0.381807), (-0.657602, 0.160914, 0.381807, -0.629198), (0.958799, 0.279464, 0.0447403, -0.0245454), (-0.279464, 0.958799, 0.0245454, 0.0447403), (0.690612, 0.721423, 0.0510189, 0.00111324), (-0.721423, 0.690612, -0.00111324, 0.0510189), (0.237376, 0.970076, 0.043627, 0.0264735), (-0.970076, 0.237376, -0.0264735, 0.043627), (0.438201, 0.127724, -0.24898, 0.854211), (-0.127724, 0.438201, -0.854211, -0.24898), (0.315631, 0.329713, -0.642728, 0.615278), (-0.329713, 0.315631, -0.615278, -0.642728), (0.108488, 0.443355, -0.864258, 0.211483), (-0.443355, 0.108488, -0.211483, -0.864258), (0.854211, 0.24898, 0.127724, -0.438201), (-0.24898, 0.854211, 0.438201, 0.127724), (0.615278, 0.642728, 0.329713, -0.315631), (-0.642728, 0.615278, 0.315631, 0.329713), (0.211483, 0.864258, 0.443355, -0.108488), (-0.864258, 0.211483, 0.108488, 0.443355), (0.0489923, 0.01428, -0.480362, -0.875584), (-0.01428, 0.0489923, 0.875584, -0.480362), (0.0352886, 0.036863, 0.0217865, -0.998459), (-0.036863, 0.0352886, 0.998459, 0.0217865), (0.0121294, 0.0495686, 0.518097, -0.853798), (-0.0495686, 0.0121294, 0.853798, 0.518097), (0.706578, 0.205949, 0.325631, 0.593547), (-0.205949, 0.706578, -0.593547, 0.325631), (0.50894, 0.531645, -0.0147687, 0.676842), (-0.531645, 0.50894, -0.676842, -0.0147687), (0.174932, 0.714888, -0.351211, 0.578778), (-0.714888, 0.174932, -0.578778, -0.351211), (0.706578, 0.205949, -0.325631, -0.593547), (-0.205949, 0.706578, 0.593547, -0.325631), (0.50894, 0.531645, 0.0147687, -0.676842), (-0.531645, 0.50894, 0.676842, 0.0147687), (0.174932, 0.714888, 0.351211, -0.578778), (-0.714888, 0.174932, 0.578778, 0.351211), (0.0489923, 0.01428, 0.480362, 0.875584), (-0.01428, 0.0489923, -0.875584, 0.480362), (0.0352886, 0.036863, -0.0217865, 0.998459), (-0.036863, 0.0352886, -0.998459, -0.0217865), (0.0121294, 0.0495686, -0.518097, 0.853798), (-0.0495686, 0.0121294, -0.853798, -0.518097), (0.676842, 0.0147687, -0.735805, -0.0160553), (-0.0147687, 0.676842, 0.0160553, -0.735805), (0.578778, 0.351211, -0.629198, -0.381807), (-0.351211, 0.578778, 0.381807, -0.629198), (0.325631, 0.593547, -0.353998, -0.645253), (-0.593547, 0.325631, 0.645253, -0.353998), (0.735805, 0.0160553, 0.676842, 0.0147687), (-0.0160553, 0.735805, -0.0147687, 0.676842), (0.629198, 0.381807, 0.578778, 0.351211), (-0.381807, 0.629198, -0.351211, 0.578778), (0.353998, 0.645253, 0.325631, 0.593547), (-0.645253, 0.353998, -0.593547, 0.325631), (0.889545, 0.0194099, -0.329713, 0.315631), (-0.0194099, 0.889545, -0.315631, -0.329713), (0.760663, 0.461582, -0.443355, 0.108488), (-0.461582, 0.760663, -0.108488, -0.443355), (0.427963, 0.780073, -0.438201, -0.127724), (-0.780073, 0.427963, 0.127724, -0.438201), (0.357132, 0.00779265, 0.674703, -0.645888), (-0.00779265, 0.357132, 0.645888, 0.674703), (0.305389, 0.185315, 0.907254, -0.222004), (-0.185315, 0.305389, 0.222004, 0.907254), (0.171817, 0.313182, 0.896707, 0.261366), (-0.313182, 0.171817, -0.261366, 0.896707), (0.357132, 0.00779265, -0.674703, 0.645888), (-0.00779265, 0.357132, -0.645888, -0.674703), (0.305389, 0.185315, -0.907254, 0.222004), (-0.185315, 0.305389, -0.222004, -0.907254), (0.171817, 0.313182, -0.896707, -0.261366), (-0.313182, 0.171817, 0.261366, -0.896707), (0.889545, 0.0194099, 0.329713, -0.315631), (-0.0194099, 0.889545, 0.315631, 0.329713), (0.760663, 0.461582, 0.443355, -0.108488), (-0.461582, 0.760663, 0.108488, 0.443355), (0.427963, 0.780073, 0.438201, 0.127724), (-0.780073, 0.427963, -0.127724, 0.438201), (0.735805, 0.0160553, -0.0147687, 0.676842), (-0.0160553, 0.735805, -0.676842, -0.0147687), (0.629198, 0.381807, -0.351211, 0.578778), (-0.381807, 0.629198, -0.578778, -0.351211), (0.353998, 0.645253, -0.593547, 0.325631), (-0.645253, 0.353998, -0.325631, -0.593547), (0.676842, 0.0147687, 0.0160553, -0.735805), (-0.0147687, 0.676842, 0.735805, 0.0160553), (0.578778, 0.351211, 0.381807, -0.629198), (-0.351211, 0.578778, 0.629198, 0.381807), (0.325631, 0.593547, 0.645253, -0.353998), (-0.593547, 0.325631, 0.353998, 0.645253), (0.456327, 0.00995709, -0.615278, -0.642728), (-0.00995709, 0.456327, 0.642728, -0.615278), (0.390212, 0.236787, -0.211483, -0.864258), (-0.236787, 0.390212, 0.864258, -0.211483), (0.21954, 0.400169, 0.24898, -0.854211), (-0.400169, 0.21954, 0.854211, 0.24898), (0.933799, 0.0203756, 0.24702, 0.258041), (-0.0203756, 0.933799, -0.258041, 0.24702), (0.798506, 0.484545, 0.0849055, 0.34698), (-0.484545, 0.798506, -0.34698, 0.0849055), (0.449254, 0.818881, -0.0999598, 0.342946), (-0.818881, 0.449254, -0.342946, -0.0999598), (0.933799, 0.0203756, -0.24702, -0.258041), (-0.0203756, 0.933799, 0.258041, -0.24702), (0.798506, 0.484545, -0.0849055, -0.34698), (-0.484545, 0.798506, 0.34698, -0.0849055), (0.449254, 0.818881, 0.0999598, -0.342946), (-0.818881, 0.449254, 0.342946, 0.0999598), (0.456327, 0.00995709, 0.615278, 0.642728), (-0.00995709, 0.456327, -0.642728, 0.615278), (0.390212, 0.236787, 0.211483, 0.864258), (-0.236787, 0.390212, -0.864258, 0.211483), (0.21954, 0.400169, -0.24898, 0.854211), (-0.400169, 0.21954, -0.854211, -0.24898), (0.790381, 0.0172462, -0.570736, 0.221947), (-0.0172462, 0.790381, -0.221947, -0.570736), (0.675867, 0.410126, -0.605245, -0.0931562), (-0.410126, 0.675867, 0.0931562, -0.605245), (0.380255, 0.693113, -0.47758, -0.383298), (-0.693113, 0.380255, 0.383298, -0.47758), (0.612227, 0.0133588, 0.736817, -0.286532), (-0.0133588, 0.612227, 0.286532, 0.736817), (0.523524, 0.317682, 0.781368, 0.120264), (-0.317682, 0.523524, -0.120264, 0.781368), (0.294544, 0.536883, 0.616553, 0.494836), (-0.536883, 0.294544, -0.494836, 0.616553), (0.96667, 0.0210928, -0.252825, 0.0344079), (-0.0210928, 0.96667, -0.0344079, -0.252825), (0.826614, 0.501602, -0.236156, -0.0966142), (-0.501602, 0.826614, 0.0966142, -0.236156), (0.465068, 0.847707, -0.15621, -0.201749), (-0.847707, 0.465068, 0.201749, -0.15621), (0.539933, 0.0117814, 0.784402, -0.305037), (-0.0117814, 0.539933, 0.305037, 0.784402), (0.461705, 0.28017, 0.83183, 0.128031), (-0.28017, 0.461705, -0.128031, 0.83183), (0.259764, 0.473487, 0.656371, 0.526793), (-0.473487, 0.259764, -0.526793, 0.656371), (0.539933, 0.0117814, -0.784402, 0.305037), (-0.0117814, 0.539933, -0.305037, -0.784402), (0.461705, 0.28017, -0.83183, -0.128031), (-0.28017, 0.461705, 0.128031, -0.83183), (0.259764, 0.473487, -0.656371, -0.526793), (-0.473487, 0.259764, 0.526793, -0.656371), (0.96667, 0.0210928, 0.252825, -0.0344079), (-0.0210928, 0.96667, 0.0344079, 0.252825), (0.826614, 0.501602, 0.236156, 0.0966142), (-0.501602, 0.826614, -0.0966142, 0.236156), (0.465068, 0.847707, 0.15621, 0.201749), (-0.847707, 0.465068, -0.201749, 0.15621), (0.612227, 0.0133588, 0.286532, 0.736817), (-0.0133588, 0.612227, -0.736817, 0.286532), (0.523524, 0.317682, -0.120264, 0.781368), (-0.317682, 0.523524, -0.781368, -0.120264), (0.294544, 0.536883, -0.494836, 0.616553), (-0.536883, 0.294544, -0.616553, -0.494836), (0.790381, 0.0172462, -0.221947, -0.570736), (-0.0172462, 0.790381, 0.570736, -0.221947), (0.675867, 0.410126, 0.0931562, -0.605245), (-0.410126, 0.675867, 0.605245, 0.0931562), (0.380255, 0.693113, 0.383298, -0.47758), (-0.693113, 0.380255, 0.47758, 0.383298), (0.255094, 0.00556618, -0.130387, -0.958068), (-0.00556618, 0.255094, 0.958068, -0.130387), (0.218135, 0.132368, 0.366115, -0.894905), (-0.132368, 0.218135, 0.894905, 0.366115), (0.122727, 0.223701, 0.764518, -0.591953), (-0.223701, 0.122727, 0.591953, 0.764518), (0.841425, 0.01836, 0.195739, 0.503342), (-0.01836, 0.841425, -0.503342, 0.195739), (0.719516, 0.436613, -0.082156, 0.533776), (-0.436613, 0.719516, -0.533776, -0.082156), (0.404812, 0.737876, -0.338037, 0.421186), (-0.737876, 0.404812, -0.421186, -0.338037), (0.841425, 0.01836, -0.195739, -0.503342), (-0.01836, 0.841425, 0.503342, -0.195739), (0.719516, 0.436613, 0.082156, -0.533776), (-0.436613, 0.719516, 0.533776, 0.082156), (0.404812, 0.737876, 0.338037, -0.421186), (-0.737876, 0.404812, 0.421186, 0.338037), (0.255094, 0.00556618, 0.130387, 0.958068), (-0.00556618, 0.255094, -0.958068, 0.130387), (0.218135, 0.132368, -0.366115, 0.894905), (-0.132368, 0.218135, -0.894905, -0.366115), (0.122727, 0.223701, -0.764518, 0.591953), (-0.223701, 0.122727, -0.591953, -0.764518), (0.758986, 0.221224, -0.47758, -0.383298), (-0.221224, 0.758986, 0.383298, -0.47758), (0.546689, 0.571079, -0.221947, -0.570736), (-0.571079, 0.546689, 0.570736, -0.221947), (0.187907, 0.767913, 0.0931562, -0.605245), (-0.767913, 0.187907, 0.605245, 0.0931562), (0.587908, 0.17136, 0.616553, 0.494836), (-0.17136, 0.587908, -0.494836, 0.616553), (0.423464, 0.442356, 0.286532, 0.736817), (-0.442356, 0.423464, -0.736817, 0.286532), (0.145552, 0.594823, -0.120264, 0.781368), (-0.594823, 0.145552, -0.781368, -0.120264), (0.928272, 0.270567, -0.108841, 0.230776), (-0.270567, 0.928272, -0.230776, -0.108841), (0.668624, 0.698454, -0.209647, 0.145438), (-0.698454, 0.668624, -0.145438, -0.209647), (0.229819, 0.939191, -0.254279, 0.0211289), (-0.939191, 0.229819, -0.0211289, -0.254279), (0.518486, 0.151125, 0.526793, -0.656371), (-0.151125, 0.518486, 0.656371, 0.526793), (0.37346, 0.390121, 0.784402, -0.305037), (-0.390121, 0.37346, 0.305037, 0.784402), (0.128365, 0.524585, 0.83183, 0.128031), (-0.524585, 0.128365, -0.128031, 0.83183), (0.518486, 0.151125, -0.526793, 0.656371), (-0.151125, 0.518486, -0.656371, -0.526793), (0.37346, 0.390121, -0.784402, 0.305037), (-0.390121, 0.37346, -0.305037, -0.784402), (0.128365, 0.524585, -0.83183, -0.128031), (-0.524585, 0.128365, 0.128031, -0.83183), (0.928272, 0.270567, 0.108841, -0.230776), (-0.270567, 0.928272, 0.230776, 0.108841), (0.668624, 0.698454, 0.209647, -0.145438), (-0.698454, 0.668624, 0.145438, 0.209647), (0.229819, 0.939191, 0.254279, -0.0211289), (-0.939191, 0.229819, 0.0211289, 0.254279), (0.587908, 0.17136, -0.494836, 0.616553), (-0.17136, 0.587908, -0.616553, -0.494836), (0.423464, 0.442356, -0.736817, 0.286532), (-0.442356, 0.423464, -0.286532, -0.736817), (0.145552, 0.594823, -0.781368, -0.120264), (-0.594823, 0.145552, 0.120264, -0.781368), (0.758986, 0.221224, 0.383298, -0.47758), (-0.221224, 0.758986, 0.47758, 0.383298), (0.546689, 0.571079, 0.570736, -0.221947), (-0.571079, 0.546689, 0.221947, 0.570736), (0.187907, 0.767913, 0.605245, 0.0931562), (-0.767913, 0.187907, -0.0931562, 0.605245), (0.244962, 0.0713998, -0.874518, -0.412449), (-0.0713998, 0.244962, 0.412449, -0.874518), (0.176443, 0.184315, -0.55113, -0.794451), (-0.184315, 0.176443, 0.794451, -0.55113), (0.0606468, 0.247843, -0.0800673, -0.963579), (-0.247843, 0.0606468, 0.963579, -0.0800673), (0.808002, 0.235511, 0.421186, 0.338037), (-0.235511, 0.808002, -0.338037, 0.421186), (0.581995, 0.60796, 0.195739, 0.503342), (-0.60796, 0.581995, -0.503342, 0.195739), (0.200042, 0.817506, -0.082156, 0.533776), (-0.817506, 0.200042, -0.533776, -0.082156), (0.808002, 0.235511, -0.421186, -0.338037), (-0.235511, 0.808002, 0.338037, -0.421186), (0.581995, 0.60796, -0.195739, -0.503342), (-0.60796, 0.581995, 0.503342, -0.195739), (0.200042, 0.817506, 0.082156, -0.533776), (-0.817506, 0.200042, 0.533776, 0.082156), (0.244962, 0.0713998, 0.874518, 0.412449), (-0.0713998, 0.244962, -0.412449, 0.874518), (0.176443, 0.184315, 0.55113, 0.794451), (-0.184315, 0.176443, -0.794451, 0.55113), (0.0606468, 0.247843, 0.0800673, 0.963579), (-0.247843, 0.0606468, -0.963579, 0.0800673), (0.758986, 0.221224, -0.608733, 0.0666671), (-0.221224, 0.758986, -0.0666671, -0.608733), (0.546689, 0.571079, -0.560512, -0.246631), (-0.571079, 0.546689, 0.246631, -0.560512), (0.187907, 0.767913, -0.362102, -0.493844), (-0.767913, 0.187907, 0.493844, -0.362102), (0.587908, 0.17136, 0.785871, -0.0860668), (-0.17136, 0.587908, 0.0860668, 0.785871), (0.423464, 0.442356, 0.723617, 0.318399), (-0.442356, 0.423464, -0.318399, 0.723617), (0.145552, 0.594823, 0.467471, 0.63755), (-0.594823, 0.145552, -0.63755, 0.467471), (0.928272, 0.270567, -0.253115, -0.0322003), (-0.270567, 0.928272, 0.0322003, -0.253115), (0.668624, 0.698454, -0.203104, -0.154444), (-0.698454, 0.668624, 0.154444, -0.203104), (0.229819, 0.939191, -0.0986713, -0.235304), (-0.939191, 0.229819, 0.235304, -0.0986713), (0.518486, 0.151125, 0.836623, -0.0916251), (-0.151125, 0.518486, 0.0916251, 0.836623), (0.37346, 0.390121, 0.770349, 0.338962), (-0.390121, 0.37346, -0.338962, 0.770349), (0.128365, 0.524585, 0.497661, 0.678724), (-0.524585, 0.128365, -0.678724, 0.497661), (0.518486, 0.151125, -0.836623, 0.0916251), (-0.151125, 0.518486, -0.0916251, -0.836623), (0.37346, 0.390121, -0.770349, -0.338962), (-0.390121, 0.37346, 0.338962, -0.770349), (0.128365, 0.524585, -0.497661, -0.678724), (-0.524585, 0.128365, 0.678724, -0.497661), (0.928272, 0.270567, 0.253115, 0.0322003), (-0.270567, 0.928272, -0.0322003, 0.253115), (0.668624, 0.698454, 0.203104, 0.154444), (-0.698454, 0.668624, -0.154444, 0.203104), (0.229819, 0.939191, 0.0986713, 0.235304), (-0.939191, 0.229819, -0.235304, 0.0986713), (0.587908, 0.17136, 0.0860668, 0.785871), (-0.17136, 0.587908, -0.785871, 0.0860668), (0.423464, 0.442356, -0.318399, 0.723617), (-0.442356, 0.423464, -0.723617, -0.318399), (0.145552, 0.594823, -0.63755, 0.467471), (-0.594823, 0.145552, -0.467471, -0.63755), (0.758986, 0.221224, -0.0666671, -0.608733), (-0.221224, 0.758986, 0.608733, -0.0666671), (0.546689, 0.571079, 0.246631, -0.560512), (-0.571079, 0.546689, 0.560512, 0.246631), (0.187907, 0.767913, 0.493844, -0.362102), (-0.767913, 0.187907, 0.362102, 0.493844), (0.244962, 0.0713998, 0.122022, -0.95917), (-0.0713998, 0.244962, 0.95917, 0.122022), (0.176443, 0.184315, 0.585259, -0.769654), (-0.184315, 0.176443, 0.769654, 0.585259), (0.0606468, 0.247843, 0.891676, -0.373911), (-0.247843, 0.0606468, 0.373911, 0.891676), (0.808002, 0.235511, 0.0587948, 0.536852), (-0.235511, 0.808002, -0.536852, 0.0587948), (0.581995, 0.60796, -0.217508, 0.494325), (-0.60796, 0.581995, -0.494325, -0.217508), (0.200042, 0.817506, -0.43553, 0.319344), (-0.817506, 0.200042, -0.319344, -0.43553), (0.808002, 0.235511, -0.0587948, -0.536852), (-0.235511, 0.808002, 0.536852, -0.0587948), (0.581995, 0.60796, 0.217508, -0.494325), (-0.60796, 0.581995, 0.494325, 0.217508), (0.200042, 0.817506, 0.43553, -0.319344), (-0.817506, 0.200042, 0.319344, 0.43553), (0.244962, 0.0713998, -0.122022, 0.95917), (-0.0713998, 0.244962, -0.95917, -0.122022), (0.176443, 0.184315, -0.585259, 0.769654), (-0.184315, 0.176443, -0.769654, -0.585259), (0.0606468, 0.247843, -0.891676, 0.373911), (-0.247843, 0.0606468, -0.373911, -0.891676), (0.790381, 0.0172462, -0.560512, -0.246631), (-0.0172462, 0.790381, 0.246631, -0.560512), (0.675867, 0.410126, -0.362102, -0.493844), (-0.410126, 0.675867, 0.493844, -0.362102), (0.380255, 0.693113, -0.0666671, -0.608733), (-0.693113, 0.380255, 0.608733, -0.0666671), (0.612227, 0.0133588, 0.723617, 0.318399), (-0.0133588, 0.612227, -0.318399, 0.723617), (0.523524, 0.317682, 0.467471, 0.63755), (-0.317682, 0.523524, -0.63755, 0.467471), (0.294544, 0.536883, 0.0860668, 0.785871), (-0.536883, 0.294544, -0.785871, 0.0860668), (0.96667, 0.0210928, -0.0454032, 0.251083), (-0.0210928, 0.96667, -0.251083, -0.0454032), (0.826614, 0.501602, -0.164862, 0.194743), (-0.501602, 0.826614, -0.194743, -0.164862), (0.465068, 0.847707, -0.240146, 0.0862212), (-0.847707, 0.465068, -0.0862212, -0.240146), (0.539933, 0.0117814, 0.338962, -0.770349), (-0.0117814, 0.539933, 0.770349, 0.338962), (0.461705, 0.28017, 0.678724, -0.497661), (-0.28017, 0.461705, 0.497661, 0.678724), (0.259764, 0.473487, 0.836623, -0.0916251), (-0.473487, 0.259764, 0.0916251, 0.836623), (0.539933, 0.0117814, -0.338962, 0.770349), (-0.0117814, 0.539933, -0.770349, -0.338962), (0.461705, 0.28017, -0.678724, 0.497661), (-0.28017, 0.461705, -0.497661, -0.678724), (0.259764, 0.473487, -0.836623, 0.0916251), (-0.473487, 0.259764, -0.0916251, -0.836623), (0.96667, 0.0210928, 0.0454032, -0.251083), (-0.0210928, 0.96667, 0.251083, 0.0454032), (0.826614, 0.501602, 0.164862, -0.194743), (-0.501602, 0.826614, 0.194743, 0.164862), (0.465068, 0.847707, 0.240146, -0.0862212), (-0.847707, 0.465068, 0.0862212, 0.240146), (0.612227, 0.0133588, -0.318399, 0.723617), (-0.0133588, 0.612227, -0.723617, -0.318399), (0.523524, 0.317682, -0.63755, 0.467471), (-0.317682, 0.523524, -0.467471, -0.63755), (0.294544, 0.536883, -0.785871, 0.0860668), (-0.536883, 0.294544, -0.0860668, -0.785871), (0.790381, 0.0172462, 0.246631, -0.560512), (-0.0172462, 0.790381, 0.560512, 0.246631), (0.675867, 0.410126, 0.493844, -0.362102), (-0.410126, 0.675867, 0.362102, 0.493844), (0.380255, 0.693113, 0.608733, -0.0666671), (-0.693113, 0.380255, 0.0666671, 0.608733), (0.255094, 0.00556618, -0.951469, -0.172054), (-0.00556618, 0.255094, 0.172054, -0.951469), (0.218135, 0.132368, -0.73797, -0.624737), (-0.132368, 0.218135, 0.624737, -0.73797), (0.122727, 0.223701, -0.326732, -0.910023), (-0.223701, 0.122727, 0.910023, -0.326732), (0.841425, 0.01836, 0.494325, 0.217508), (-0.01836, 0.841425, -0.217508, 0.494325), (0.719516, 0.436613, 0.319344, 0.43553), (-0.436613, 0.719516, -0.43553, 0.319344), (0.404812, 0.737876, 0.0587948, 0.536852), (-0.737876, 0.404812, -0.536852, 0.0587948), (0.841425, 0.01836, -0.494325, -0.217508), (-0.01836, 0.841425, 0.217508, -0.494325), (0.719516, 0.436613, -0.319344, -0.43553), (-0.436613, 0.719516, 0.43553, -0.319344), (0.404812, 0.737876, -0.0587948, -0.536852), (-0.737876, 0.404812, 0.536852, -0.0587948), (0.255094, 0.00556618, 0.951469, 0.172054), (-0.00556618, 0.255094, -0.172054, 0.951469), (0.218135, 0.132368, 0.73797, 0.624737), (-0.132368, 0.218135, -0.624737, 0.73797), (0.122727, 0.223701, 0.326732, 0.910023), (-0.223701, 0.122727, -0.910023, 0.326732), (0.781368, 0.120264, -0.605245, -0.0931562), (-0.120264, 0.781368, 0.0931562, -0.605245), (0.616553, 0.494836, -0.47758, -0.383298), (-0.494836, 0.616553, 0.383298, -0.47758), (0.286532, 0.736817, -0.221947, -0.570736), (-0.736817, 0.286532, 0.570736, -0.221947), (0.605245, 0.0931562, 0.781368, 0.120264), (-0.0931562, 0.605245, -0.120264, 0.781368), (0.47758, 0.383298, 0.616553, 0.494836), (-0.383298, 0.47758, -0.494836, 0.616553), (0.221947, 0.570736, 0.286532, 0.736817), (-0.570736, 0.221947, -0.736817, 0.286532), (0.955647, 0.147088, -0.205769, 0.150876), (-0.147088, 0.955647, -0.150876, -0.205769), (0.75407, 0.605205, -0.253639, 0.0277779), (-0.605205, 0.75407, -0.0277779, -0.253639), (0.350441, 0.901159, -0.233546, -0.102763), (-0.901159, 0.350441, 0.102763, -0.233546), (0.533776, 0.082156, 0.678724, -0.497661), (-0.082156, 0.533776, 0.497661, 0.678724), (0.421186, 0.338037, 0.836623, -0.0916251), (-0.338037, 0.421186, 0.0916251, 0.836623), (0.195739, 0.503342, 0.770349, 0.338962), (-0.503342, 0.195739, -0.338962, 0.770349), (0.533776, 0.082156, -0.678724, 0.497661), (-0.082156, 0.533776, -0.497661, -0.678724), (0.421186, 0.338037, -0.836623, 0.0916251), (-0.338037, 0.421186, -0.0916251, -0.836623), (0.195739, 0.503342, -0.770349, -0.338962), (-0.503342, 0.195739, 0.338962, -0.770349), (0.955647, 0.147088, 0.205769, -0.150876), (-0.147088, 0.955647, 0.150876, 0.205769), (0.75407, 0.605205, 0.253639, -0.0277779), (-0.605205, 0.75407, 0.0277779, 0.253639), (0.350441, 0.901159, 0.233546, 0.102763), (-0.901159, 0.350441, -0.102763, 0.233546), (0.605245, 0.0931562, -0.120264, 0.781368), (-0.0931562, 0.605245, -0.781368, -0.120264), (0.47758, 0.383298, -0.494836, 0.616553), (-0.383298, 0.47758, -0.616553, -0.494836), (0.221947, 0.570736, -0.736817, 0.286532), (-0.570736, 0.221947, -0.286532, -0.736817), (0.781368, 0.120264, 0.0931562, -0.605245), (-0.120264, 0.781368, 0.605245, 0.0931562), (0.616553, 0.494836, 0.383298, -0.47758), (-0.494836, 0.616553, 0.47758, 0.383298), (0.286532, 0.736817, 0.570736, -0.221947), (-0.736817, 0.286532, 0.221947, 0.570736), (0.252186, 0.0388151, -0.571737, -0.779751), (-0.0388151, 0.252186, 0.779751, -0.571737), (0.198992, 0.159708, -0.105263, -0.961153), (-0.159708, 0.198992, 0.961153, -0.105263), (0.0924779, 0.237807, 0.389416, -0.885015), (-0.237807, 0.0924779, 0.885015, 0.389416), (0.83183, 0.128031, 0.319344, 0.43553), (-0.128031, 0.83183, -0.43553, 0.319344), (0.656371, 0.526793, 0.0587948, 0.536852), (-0.526793, 0.656371, -0.536852, 0.0587948), (0.305037, 0.784402, -0.217508, 0.494325), (-0.784402, 0.305037, -0.494325, -0.217508), (0.83183, 0.128031, -0.319344, -0.43553), (-0.128031, 0.83183, 0.43553, -0.319344), (0.656371, 0.526793, -0.0587948, -0.536852), (-0.526793, 0.656371, 0.536852, -0.0587948), (0.305037, 0.784402, 0.217508, -0.494325), (-0.784402, 0.305037, 0.494325, 0.217508), (0.252186, 0.0388151, 0.571737, 0.779751), (-0.0388151, 0.252186, -0.779751, 0.571737), (0.198992, 0.159708, 0.105263, 0.961153), (-0.159708, 0.198992, -0.961153, 0.105263), (0.0924779, 0.237807, -0.389416, 0.885015), (-0.237807, 0.0924779, -0.885015, -0.389416), (0.494325, 0.217508, -0.770349, -0.338962), (-0.217508, 0.494325, 0.338962, -0.770349), (0.319344, 0.43553, -0.497661, -0.678724), (-0.43553, 0.319344, 0.678724, -0.497661), (0.0587948, 0.536852, -0.0916251, -0.836623), (-0.536852, 0.0587948, 0.836623, -0.0916251), (0.770349, 0.338962, 0.494325, 0.217508), (-0.338962, 0.770349, -0.217508, 0.494325), (0.497661, 0.678724, 0.319344, 0.43553), (-0.678724, 0.497661, -0.43553, 0.319344), (0.0916251, 0.836623, 0.0587948, 0.536852), (-0.836623, 0.0916251, -0.536852, 0.0587948), (0.723617, 0.318399, -0.570736, 0.221947), (-0.318399, 0.723617, -0.221947, -0.570736), (0.467471, 0.63755, -0.605245, -0.0931562), (-0.63755, 0.467471, 0.0931562, -0.605245), (0.0860668, 0.785871, -0.47758, -0.383298), (-0.785871, 0.0860668, 0.383298, -0.47758), (0.140128, 0.0616577, 0.921021, -0.358166), (-0.0616577, 0.140128, 0.358166, 0.921021), (0.0905254, 0.123461, 0.97671, 0.15033), (-0.123461, 0.0905254, -0.15033, 0.97671), (0.0166668, 0.152183, 0.770691, 0.618545), (-0.152183, 0.0166668, -0.618545, 0.770691), (0.140128, 0.0616577, -0.921021, 0.358166), (-0.0616577, 0.140128, -0.358166, -0.921021), (0.0905254, 0.123461, -0.97671, -0.15033), (-0.123461, 0.0905254, 0.15033, -0.97671), (0.0166668, 0.152183, -0.770691, -0.618545), (-0.152183, 0.0166668, 0.618545, -0.770691), (0.723617, 0.318399, 0.570736, -0.221947), (-0.318399, 0.723617, 0.221947, 0.570736), (0.467471, 0.63755, 0.605245, 0.0931562), (-0.63755, 0.467471, -0.0931562, 0.605245), (0.0860668, 0.785871, 0.47758, 0.383298), (-0.785871, 0.0860668, -0.383298, 0.47758), (0.770349, 0.338962, -0.217508, 0.494325), (-0.338962, 0.770349, -0.494325, -0.217508), (0.497661, 0.678724, -0.43553, 0.319344), (-0.678724, 0.497661, -0.319344, -0.43553), (0.0916251, 0.836623, -0.536852, 0.0587948), (-0.836623, 0.0916251, -0.0587948, -0.536852), (0.494325, 0.217508, 0.338962, -0.770349), (-0.217508, 0.494325, 0.770349, 0.338962), (0.319344, 0.43553, 0.678724, -0.497661), (-0.43553, 0.319344, 0.497661, 0.678724), (0.0587948, 0.536852, 0.836623, -0.0916251), (-0.536852, 0.0587948, 0.0916251, 0.836623), (0.560512, 0.246631, -0.286532, -0.736817), (-0.246631, 0.560512, 0.736817, -0.286532), (0.362102, 0.493844, 0.120264, -0.781368), (-0.493844, 0.362102, 0.781368, 0.120264), (0.0666671, 0.608733, 0.494836, -0.616553), (-0.608733, 0.0666671, 0.616553, 0.494836), (0.904522, 0.397999, 0.0554868, 0.142684), (-0.397999, 0.904522, -0.142684, 0.0554868), (0.584339, 0.796938, -0.023289, 0.151311), (-0.796938, 0.584339, -0.151311, -0.023289), (0.107584, 0.982338, -0.0958246, 0.119395), (-0.982338, 0.107584, -0.119395, -0.0958246), (0.904522, 0.397999, -0.0554868, -0.142684), (-0.397999, 0.904522, 0.142684, -0.0554868), (0.584339, 0.796938, 0.023289, -0.151311), (-0.796938, 0.584339, 0.151311, 0.023289), (0.107584, 0.982338, 0.0958246, -0.119395), (-0.982338, 0.107584, 0.119395, 0.0958246), (0.560512, 0.246631, 0.286532, 0.736817), (-0.246631, 0.560512, -0.736817, 0.286532), (0.362102, 0.493844, -0.120264, 0.781368), (-0.493844, 0.362102, -0.781368, -0.120264), (0.0666671, 0.608733, -0.494836, 0.616553), (-0.608733, 0.0666671, -0.616553, -0.494836), (0.723617, 0.318399, -0.560512, -0.246631), (-0.318399, 0.723617, 0.246631, -0.560512), (0.467471, 0.63755, -0.362102, -0.493844), (-0.63755, 0.467471, 0.493844, -0.362102), (0.0860668, 0.785871, -0.0666671, -0.608733), (-0.785871, 0.0860668, 0.608733, -0.0666671), (0.560512, 0.246631, 0.723617, 0.318399), (-0.246631, 0.560512, -0.318399, 0.723617), (0.362102, 0.493844, 0.467471, 0.63755), (-0.493844, 0.362102, -0.63755, 0.467471), (0.0666671, 0.608733, 0.0860668, 0.785871), (-0.608733, 0.0666671, -0.785871, 0.0860668), (0.885015, 0.389416, -0.237807, 0.0924779), (-0.389416, 0.885015, -0.0924779, -0.237807), (0.571737, 0.779751, -0.252186, -0.0388151), (-0.779751, 0.571737, 0.0388151, -0.252186), (0.105263, 0.961153, -0.198992, -0.159708), (-0.961153, 0.105263, 0.159708, -0.198992), (0.494325, 0.217508, 0.784402, -0.305037), (-0.217508, 0.494325, 0.305037, 0.784402), (0.319344, 0.43553, 0.83183, 0.128031), (-0.43553, 0.319344, -0.128031, 0.83183), (0.0587948, 0.536852, 0.656371, 0.526793), (-0.536852, 0.0587948, -0.526793, 0.656371), (0.494325, 0.217508, -0.784402, 0.305037), (-0.217508, 0.494325, -0.305037, -0.784402), (0.319344, 0.43553, -0.83183, -0.128031), (-0.43553, 0.319344, 0.128031, -0.83183), (0.0587948, 0.536852, -0.656371, -0.526793), (-0.536852, 0.0587948, 0.526793, -0.656371), (0.885015, 0.389416, 0.237807, -0.0924779), (-0.389416, 0.885015, 0.0924779, 0.237807), (0.571737, 0.779751, 0.252186, 0.0388151), (-0.779751, 0.571737, -0.0388151, 0.252186), (0.105263, 0.961153, 0.198992, 0.159708), (-0.961153, 0.105263, -0.159708, 0.198992), (0.560512, 0.246631, -0.318399, 0.723617), (-0.246631, 0.560512, -0.723617, -0.318399), (0.362102, 0.493844, -0.63755, 0.467471), (-0.493844, 0.362102, -0.467471, -0.63755), (0.0666671, 0.608733, -0.785871, 0.0860668), (-0.608733, 0.0666671, -0.0860668, -0.785871), (0.723617, 0.318399, 0.246631, -0.560512), (-0.318399, 0.723617, 0.560512, 0.246631), (0.467471, 0.63755, 0.493844, -0.362102), (-0.63755, 0.467471, 0.362102, 0.493844), (0.0860668, 0.785871, 0.608733, -0.0666671), (-0.785871, 0.0860668, 0.0666671, 0.608733), (0.233546, 0.102763, -0.350441, -0.901159), (-0.102763, 0.233546, 0.901159, -0.350441), (0.150876, 0.205769, 0.147088, -0.955647), (-0.205769, 0.150876, 0.955647, 0.147088), (0.0277779, 0.253639, 0.605205, -0.75407), (-0.253639, 0.0277779, 0.75407, 0.605205), (0.770349, 0.338962, 0.195739, 0.503342), (-0.338962, 0.770349, -0.503342, 0.195739), (0.497661, 0.678724, -0.082156, 0.533776), (-0.678724, 0.497661, -0.533776, -0.082156), (0.0916251, 0.836623, -0.338037, 0.421186), (-0.836623, 0.0916251, -0.421186, -0.338037), (0.770349, 0.338962, -0.195739, -0.503342), (-0.338962, 0.770349, 0.503342, -0.195739), (0.497661, 0.678724, 0.082156, -0.533776), (-0.678724, 0.497661, 0.533776, 0.082156), (0.0916251, 0.836623, 0.338037, -0.421186), (-0.836623, 0.0916251, 0.421186, 0.338037), (0.233546, 0.102763, 0.350441, 0.901159), (-0.102763, 0.233546, -0.901159, 0.350441), (0.150876, 0.205769, -0.147088, 0.955647), (-0.205769, 0.150876, -0.955647, -0.147088), (0.0277779, 0.253639, -0.605205, 0.75407), (-0.253639, 0.0277779, -0.75407, -0.605205), (0.533776, 0.082156, -0.83183, -0.128031), (-0.082156, 0.533776, 0.128031, -0.83183), (0.421186, 0.338037, -0.656371, -0.526793), (-0.338037, 0.421186, 0.526793, -0.656371), (0.195739, 0.503342, -0.305037, -0.784402), (-0.503342, 0.195739, 0.784402, -0.305037), (0.83183, 0.128031, 0.533776, 0.082156), (-0.128031, 0.83183, -0.082156, 0.533776), (0.656371, 0.526793, 0.421186, 0.338037), (-0.526793, 0.656371, -0.338037, 0.421186), (0.305037, 0.784402, 0.195739, 0.503342), (-0.784402, 0.305037, -0.503342, 0.195739), (0.781368, 0.120264, -0.493844, 0.362102), (-0.120264, 0.781368, -0.362102, -0.493844), (0.616553, 0.494836, -0.608733, 0.0666671), (-0.494836, 0.616553, -0.0666671, -0.608733), (0.286532, 0.736817, -0.560512, -0.246631), (-0.736817, 0.286532, 0.246631, -0.560512), (0.151311, 0.023289, 0.796938, -0.584339), (-0.023289, 0.151311, 0.584339, 0.796938), (0.119395, 0.0958246, 0.982338, -0.107584), (-0.0958246, 0.119395, 0.107584, 0.982338), (0.0554868, 0.142684, 0.904522, 0.397999), (-0.142684, 0.0554868, -0.397999, 0.904522), (0.151311, 0.023289, -0.796938, 0.584339), (-0.023289, 0.151311, -0.584339, -0.796938), (0.119395, 0.0958246, -0.982338, 0.107584), (-0.0958246, 0.119395, -0.107584, -0.982338), (0.0554868, 0.142684, -0.904522, -0.397999), (-0.142684, 0.0554868, 0.397999, -0.904522), (0.781368, 0.120264, 0.493844, -0.362102), (-0.120264, 0.781368, 0.362102, 0.493844), (0.616553, 0.494836, 0.608733, -0.0666671), (-0.494836, 0.616553, 0.0666671, 0.608733), (0.286532, 0.736817, 0.560512, 0.246631), (-0.736817, 0.286532, -0.246631, 0.560512), (0.83183, 0.128031, -0.082156, 0.533776), (-0.128031, 0.83183, -0.533776, -0.082156), (0.656371, 0.526793, -0.338037, 0.421186), (-0.526793, 0.656371, -0.421186, -0.338037), (0.305037, 0.784402, -0.503342, 0.195739), (-0.784402, 0.305037, -0.195739, -0.503342), (0.533776, 0.082156, 0.128031, -0.83183), (-0.082156, 0.533776, 0.83183, 0.128031), (0.421186, 0.338037, 0.526793, -0.656371), (-0.338037, 0.421186, 0.656371, 0.526793), (0.195739, 0.503342, 0.784402, -0.305037), (-0.503342, 0.195739, 0.305037, 0.784402), (0.605245, 0.0931562, -0.467471, -0.63755), (-0.0931562, 0.605245, 0.63755, -0.467471), (0.47758, 0.383298, -0.0860668, -0.785871), (-0.383298, 0.47758, 0.785871, -0.0860668), (0.221947, 0.570736, 0.318399, -0.723617), (-0.570736, 0.221947, 0.723617, 0.318399), (0.97671, 0.15033, 0.0905254, 0.123461), (-0.15033, 0.97671, -0.123461, 0.0905254), (0.770691, 0.618545, 0.0166668, 0.152183), (-0.618545, 0.770691, -0.152183, 0.0166668), (0.358166, 0.921021, -0.0616577, 0.140128), (-0.921021, 0.358166, -0.140128, -0.0616577), (0.97671, 0.15033, -0.0905254, -0.123461), (-0.15033, 0.97671, 0.123461, -0.0905254), (0.770691, 0.618545, -0.0166668, -0.152183), (-0.618545, 0.770691, 0.152183, -0.0166668), (0.358166, 0.921021, 0.0616577, -0.140128), (-0.921021, 0.358166, 0.140128, 0.0616577), (0.605245, 0.0931562, 0.467471, 0.63755), (-0.0931562, 0.605245, -0.63755, 0.467471), (0.47758, 0.383298, 0.0860668, 0.785871), (-0.383298, 0.47758, -0.785871, 0.0860668), (0.221947, 0.570736, -0.318399, 0.723617), (-0.570736, 0.221947, -0.723617, -0.318399), (0.669124, 0.102988, -0.714888, 0.174932), (-0.102988, 0.669124, -0.174932, -0.714888), (0.527984, 0.423752, -0.706578, -0.205949), (-0.423752, 0.527984, 0.205949, -0.706578), (0.245372, 0.630972, -0.50894, -0.531645), (-0.630972, 0.245372, 0.531645, -0.50894), (0.727414, 0.11196, 0.657602, -0.160914), (-0.11196, 0.727414, 0.160914, 0.657602), (0.573979, 0.460667, 0.649957, 0.189445), (-0.460667, 0.573979, -0.189445, 0.649957), (0.266747, 0.685939, 0.468157, 0.489043), (-0.685939, 0.266747, -0.489043, 0.468157), (0.879401, 0.135353, -0.443355, 0.108488), (-0.135353, 0.879401, -0.108488, -0.443355), (0.693907, 0.556919, -0.438201, -0.127724), (-0.556919, 0.693907, 0.127724, -0.438201), (0.322482, 0.82926, -0.315631, -0.329713), (-0.82926, 0.322482, 0.329713, -0.315631), (0.35306, 0.0543411, 0.918275, -0.170784), (-0.0543411, 0.35306, 0.170784, 0.918275), (0.278588, 0.223591, 0.880641, 0.311234), (-0.223591, 0.278588, -0.311234, 0.880641), (0.129469, 0.332929, 0.607041, 0.709857), (-0.332929, 0.129469, -0.709857, 0.607041), (0.35306, 0.0543411, -0.918275, 0.170784), (-0.0543411, 0.35306, -0.170784, -0.918275), (0.278588, 0.223591, -0.880641, -0.311234), (-0.223591, 0.278588, 0.311234, -0.880641), (0.129469, 0.332929, -0.607041, -0.709857), (-0.332929, 0.129469, 0.709857, -0.607041), (0.879401, 0.135353, 0.443355, -0.108488), (-0.135353, 0.879401, 0.108488, 0.443355), (0.693907, 0.556919, 0.438201, 0.127724), (-0.556919, 0.693907, -0.127724, 0.438201), (0.322482, 0.82926, 0.315631, 0.329713), (-0.82926, 0.322482, -0.329713, 0.315631), (0.727414, 0.11196, 0.160914, 0.657602), (-0.11196, 0.727414, -0.657602, 0.160914), (0.573979, 0.460667, -0.189445, 0.649957), (-0.460667, 0.573979, -0.649957, -0.189445), (0.266747, 0.685939, -0.489043, 0.468157), (-0.685939, 0.266747, -0.468157, -0.489043), (0.669124, 0.102988, -0.174932, -0.714888), (-0.102988, 0.669124, 0.714888, -0.174932), (0.527984, 0.423752, 0.205949, -0.706578), (-0.423752, 0.527984, 0.706578, 0.205949), (0.245372, 0.630972, 0.531645, -0.50894), (-0.630972, 0.245372, 0.50894, 0.531645), (0.451123, 0.0694345, -0.211483, -0.864258), (-0.0694345, 0.451123, 0.864258, -0.211483), (0.355967, 0.285694, 0.24898, -0.854211), (-0.285694, 0.355967, 0.854211, 0.24898), (0.16543, 0.425401, 0.642728, -0.615278), (-0.425401, 0.16543, 0.615278, 0.642728), (0.923151, 0.142086, 0.0653166, 0.351195), (-0.142086, 0.923151, -0.351195, 0.0653166), (0.728429, 0.584626, -0.119032, 0.336802), (-0.584626, 0.728429, -0.336802, -0.119032), (0.338525, 0.870515, -0.271485, 0.232163), (-0.870515, 0.338525, -0.232163, -0.271485), (0.923151, 0.142086, -0.0653166, -0.351195), (-0.142086, 0.923151, 0.351195, -0.0653166), (0.728429, 0.584626, 0.119032, -0.336802), (-0.584626, 0.728429, 0.336802, 0.119032), (0.338525, 0.870515, 0.271485, -0.232163), (-0.870515, 0.338525, 0.232163, 0.271485), (0.451123, 0.0694345, 0.211483, 0.864258), (-0.0694345, 0.451123, -0.864258, 0.211483), (0.355967, 0.285694, -0.24898, 0.854211), (-0.285694, 0.355967, -0.854211, -0.24898), (0.16543, 0.425401, -0.642728, 0.615278), (-0.425401, 0.16543, -0.615278, -0.642728), (0.619669, 0.272661, -0.50894, -0.531645), (-0.272661, 0.619669, 0.531645, -0.50894), (0.400319, 0.545966, -0.174932, -0.714888), (-0.545966, 0.400319, 0.714888, -0.174932), (0.0737032, 0.672979, 0.205949, -0.706578), (-0.672979, 0.0737032, 0.706578, 0.205949), (0.673651, 0.296414, 0.468157, 0.489043), (-0.296414, 0.673651, -0.489043, 0.468157), (0.435192, 0.593527, 0.160914, 0.657602), (-0.593527, 0.435192, -0.657602, 0.160914), (0.0801239, 0.731606, -0.189445, 0.649957), (-0.731606, 0.0801239, -0.649957, -0.189445), (0.814404, 0.358346, -0.329713, 0.315631), (-0.358346, 0.814404, -0.315631, -0.329713), (0.526122, 0.717539, -0.443355, 0.108488), (-0.717539, 0.526122, -0.108488, -0.443355), (0.096865, 0.884468, -0.438201, -0.127724), (-0.884468, 0.096865, 0.127724, -0.438201), (0.326965, 0.143868, 0.637427, -0.682703), (-0.143868, 0.326965, 0.682703, 0.637427), (0.211226, 0.288076, 0.893379, -0.272525), (-0.288076, 0.211226, 0.272525, 0.893379), (0.0388891, 0.355094, 0.909951, 0.210676), (-0.355094, 0.0388891, -0.210676, 0.909951), (0.326965, 0.143868, -0.637427, 0.682703), (-0.143868, 0.326965, -0.682703, -0.637427), (0.211226, 0.288076, -0.893379, 0.272525), (-0.288076, 0.211226, -0.272525, -0.893379), (0.0388891, 0.355094, -0.909951, -0.210676), (-0.355094, 0.0388891, 0.210676, -0.909951), (0.814404, 0.358346, 0.329713, -0.315631), (-0.358346, 0.814404, 0.315631, 0.329713), (0.526122, 0.717539, 0.443355, -0.108488), (-0.717539, 0.526122, 0.108488, 0.443355), (0.096865, 0.884468, 0.438201, 0.127724), (-0.884468, 0.096865, -0.127724, 0.438201), (0.673651, 0.296414, -0.489043, 0.468157), (-0.296414, 0.673651, -0.468157, -0.489043), (0.435192, 0.593527, -0.657602, 0.160914), (-0.593527, 0.435192, -0.160914, -0.657602), (0.0801239, 0.731606, -0.649957, -0.189445), (-0.731606, 0.0801239, 0.189445, -0.649957), (0.619669, 0.272661, 0.531645, -0.50894), (-0.272661, 0.619669, 0.50894, 0.531645), (0.400319, 0.545966, 0.714888, -0.174932), (-0.545966, 0.400319, 0.174932, 0.714888), (0.0737032, 0.672979, 0.706578, 0.205949), (-0.672979, 0.0737032, -0.205949, 0.706578), (0.417781, 0.183828, -0.615278, -0.642728), (-0.183828, 0.417781, 0.642728, -0.615278), (0.269895, 0.36809, -0.211483, -0.864258), (-0.36809, 0.269895, 0.864258, -0.211483), (0.0496907, 0.453723, 0.24898, -0.854211), (-0.453723, 0.0496907, 0.854211, 0.24898), (0.85492, 0.376174, 0.2611, 0.243784), (-0.376174, 0.85492, -0.243784, 0.2611), (0.552296, 0.753236, 0.104227, 0.341674), (-0.753236, 0.552296, -0.341674, 0.104227), (0.101684, 0.92847, -0.0805733, 0.348012), (-0.92847, 0.101684, -0.348012, -0.0805733), (0.85492, 0.376174, -0.2611, -0.243784), (-0.376174, 0.85492, 0.243784, -0.2611), (0.552296, 0.753236, -0.104227, -0.341674), (-0.753236, 0.552296, 0.341674, -0.104227), (0.101684, 0.92847, 0.0805733, -0.348012), (-0.92847, 0.101684, 0.348012, 0.0805733), (0.417781, 0.183828, 0.615278, 0.642728), (-0.183828, 0.417781, -0.642728, 0.615278), (0.269895, 0.36809, 0.211483, 0.864258), (-0.36809, 0.269895, -0.864258, 0.211483), (0.0496907, 0.453723, -0.24898, 0.854211), (-0.453723, 0.0496907, -0.854211, -0.24898), (0.619669, 0.272661, -0.735805, -0.0160553), (-0.272661, 0.619669, 0.0160553, -0.735805), (0.400319, 0.545966, -0.629198, -0.381807), (-0.545966, 0.400319, 0.381807, -0.629198), (0.0737032, 0.672979, -0.353998, -0.645253), (-0.672979, 0.0737032, 0.645253, -0.353998), (0.673651, 0.296414, 0.676842, 0.0147687), (-0.296414, 0.673651, -0.0147687, 0.676842), (0.435192, 0.593527, 0.578778, 0.351211), (-0.593527, 0.435192, -0.351211, 0.578778), (0.0801239, 0.731606, 0.325631, 0.593547), (-0.731606, 0.0801239, -0.593547, 0.325631), (0.814404, 0.358346, -0.456327, -0.00995709), (-0.358346, 0.814404, 0.00995709, -0.456327), (0.526122, 0.717539, -0.390212, -0.236787), (-0.717539, 0.526122, 0.236787, -0.390212), (0.096865, 0.884468, -0.21954, -0.400169), (-0.884468, 0.096865, 0.400169, -0.21954), (0.326965, 0.143868, 0.931188, 0.072702), (-0.143868, 0.326965, -0.072702, 0.931188), (0.211226, 0.288076, 0.770081, 0.528556), (-0.288076, 0.211226, -0.528556, 0.770081), (0.0388891, 0.355094, 0.402632, 0.842783), (-0.355094, 0.0388891, -0.842783, 0.402632), (0.326965, 0.143868, -0.931188, -0.072702), (-0.143868, 0.326965, 0.072702, -0.931188), (0.211226, 0.288076, -0.770081, -0.528556), (-0.288076, 0.211226, 0.528556, -0.770081), (0.0388891, 0.355094, -0.402632, -0.842783), (-0.355094, 0.0388891, 0.842783, -0.402632), (0.814404, 0.358346, 0.456327, 0.00995709), (-0.358346, 0.814404, -0.00995709, 0.456327), (0.526122, 0.717539, 0.390212, 0.236787), (-0.717539, 0.526122, -0.236787, 0.390212), (0.096865, 0.884468, 0.21954, 0.400169), (-0.884468, 0.096865, -0.400169, 0.21954), (0.673651, 0.296414, -0.0147687, 0.676842), (-0.296414, 0.673651, -0.676842, -0.0147687), (0.435192, 0.593527, -0.351211, 0.578778), (-0.593527, 0.435192, -0.578778, -0.351211), (0.0801239, 0.731606, -0.593547, 0.325631), (-0.731606, 0.0801239, -0.325631, -0.593547), (0.619669, 0.272661, 0.0160553, -0.735805), (-0.272661, 0.619669, 0.735805, 0.0160553), (0.400319, 0.545966, 0.381807, -0.629198), (-0.545966, 0.400319, 0.629198, 0.381807), (0.0737032, 0.672979, 0.645253, -0.353998), (-0.672979, 0.0737032, 0.353998, 0.645253), (0.417781, 0.183828, 0.0194099, -0.889545), (-0.183828, 0.417781, 0.889545, 0.0194099), (0.269895, 0.36809, 0.461582, -0.760663), (-0.36809, 0.269895, 0.760663, 0.461582), (0.0496907, 0.453723, 0.780073, -0.427963), (-0.453723, 0.0496907, 0.427963, 0.780073), (0.85492, 0.376174, -0.027805, 0.356133), (-0.376174, 0.85492, -0.356133, -0.027805), (0.552296, 0.753236, -0.202147, 0.294518), (-0.753236, 0.552296, -0.294518, -0.202147), (0.101684, 0.92847, -0.322323, 0.153987), (-0.92847, 0.101684, -0.153987, -0.322323), (0.85492, 0.376174, 0.027805, -0.356133), (-0.376174, 0.85492, 0.356133, 0.027805), (0.552296, 0.753236, 0.202147, -0.294518), (-0.753236, 0.552296, 0.294518, 0.202147), (0.101684, 0.92847, 0.322323, -0.153987), (-0.92847, 0.101684, 0.153987, 0.322323), (0.417781, 0.183828, -0.0194099, 0.889545), (-0.183828, 0.417781, -0.889545, -0.0194099), (0.269895, 0.36809, -0.461582, 0.760663), (-0.36809, 0.269895, -0.760663, -0.461582), (0.0496907, 0.453723, -0.780073, 0.427963), (-0.453723, 0.0496907, -0.427963, -0.780073), (0.669124, 0.102988, -0.629198, -0.381807), (-0.102988, 0.669124, 0.381807, -0.629198), (0.527984, 0.423752, -0.353998, -0.645253), (-0.423752, 0.527984, 0.645253, -0.353998), (0.245372, 0.630972, 0.0160553, -0.735805), (-0.630972, 0.245372, 0.735805, 0.0160553), (0.727414, 0.11196, 0.578778, 0.351211), (-0.11196, 0.727414, -0.351211, 0.578778), (0.573979, 0.460667, 0.325631, 0.593547), (-0.460667, 0.573979, -0.593547, 0.325631), (0.266747, 0.685939, -0.0147687, 0.676842), (-0.685939, 0.266747, -0.676842, -0.0147687), (0.879401, 0.135353, -0.236787, 0.390212), (-0.135353, 0.879401, -0.390212, -0.236787), (0.693907, 0.556919, -0.400169, 0.21954), (-0.556919, 0.693907, -0.21954, -0.400169), (0.322482, 0.82926, -0.456327, -0.00995709), (-0.82926, 0.322482, 0.00995709, -0.456327), (0.35306, 0.0543411, 0.43901, -0.824418), (-0.0543411, 0.35306, 0.824418, 0.43901), (0.278588, 0.223591, 0.792403, -0.494462), (-0.223591, 0.278588, 0.494462, 0.792403), (0.129469, 0.332929, 0.933472, -0.032015), (-0.332929, 0.129469, 0.032015, 0.933472), (0.35306, 0.0543411, -0.43901, 0.824418), (-0.0543411, 0.35306, -0.824418, -0.43901), (0.278588, 0.223591, -0.792403, 0.494462), (-0.223591, 0.278588, -0.494462, -0.792403), (0.129469, 0.332929, -0.933472, 0.032015), (-0.332929, 0.129469, -0.032015, -0.933472), (0.879401, 0.135353, 0.236787, -0.390212), (-0.135353, 0.879401, 0.390212, 0.236787), (0.693907, 0.556919, 0.400169, -0.21954), (-0.556919, 0.693907, 0.21954, 0.400169), (0.322482, 0.82926, 0.456327, 0.00995709), (-0.82926, 0.322482, -0.00995709, 0.456327), (0.727414, 0.11196, -0.351211, 0.578778), (-0.11196, 0.727414, -0.578778, -0.351211), (0.573979, 0.460667, -0.593547, 0.325631), (-0.460667, 0.573979, -0.325631, -0.593547), (0.266747, 0.685939, -0.676842, -0.0147687), (-0.685939, 0.266747, 0.0147687, -0.676842), (0.669124, 0.102988, 0.381807, -0.629198), (-0.102988, 0.669124, 0.629198, 0.381807), (0.527984, 0.423752, 0.645253, -0.353998), (-0.423752, 0.527984, 0.353998, 0.645253), (0.245372, 0.630972, 0.735805, 0.0160553), (-0.630972, 0.245372, -0.0160553, 0.735805), (0.451123, 0.0694345, -0.760663, -0.461582), (-0.0694345, 0.451123, 0.461582, -0.760663), (0.355967, 0.285694, -0.427963, -0.780073), (-0.285694, 0.355967, 0.780073, -0.427963), (0.16543, 0.425401, 0.0194099, -0.889545), (-0.425401, 0.16543, 0.889545, 0.0194099), (0.923151, 0.142086, 0.3153, 0.1679), (-0.142086, 0.923151, -0.1679, 0.3153), (0.728429, 0.584626, 0.189107, 0.303055), (-0.584626, 0.728429, -0.303055, 0.189107), (0.338525, 0.870515, 0.0122442, 0.357007), (-0.870515, 0.338525, -0.357007, 0.0122442), (0.923151, 0.142086, -0.3153, -0.1679), (-0.142086, 0.923151, 0.1679, -0.3153), (0.728429, 0.584626, -0.189107, -0.303055), (-0.584626, 0.728429, 0.303055, -0.189107), (0.338525, 0.870515, -0.0122442, -0.357007), (-0.870515, 0.338525, 0.357007, -0.0122442), (0.451123, 0.0694345, 0.760663, 0.461582), (-0.0694345, 0.451123, -0.461582, 0.760663), (0.355967, 0.285694, 0.427963, 0.780073), (-0.285694, 0.355967, -0.780073, 0.427963), (0.16543, 0.425401, -0.0194099, 0.889545), (-0.425401, 0.16543, -0.889545, -0.0194099), (0.879401, 0.135353, -0.451123, -0.0694345), (-0.135353, 0.879401, 0.0694345, -0.451123), (0.693907, 0.556919, -0.355967, -0.285694), (-0.556919, 0.693907, 0.285694, -0.355967), (0.322482, 0.82926, -0.16543, -0.425401), (-0.82926, 0.322482, 0.425401, -0.16543), (0.451123, 0.0694345, 0.879401, 0.135353), (-0.0694345, 0.451123, -0.135353, 0.879401), (0.355967, 0.285694, 0.693907, 0.556919), (-0.285694, 0.355967, -0.556919, 0.693907), (0.16543, 0.425401, 0.322482, 0.82926), (-0.425401, 0.16543, -0.82926, 0.322482), (0.987074, 0.151925, -0.0411537, 0.0301751), (-0.151925, 0.987074, -0.0301751, -0.0411537), (0.778868, 0.625108, -0.0507277, 0.00555559), (-0.625108, 0.778868, -0.00555559, -0.0507277), (0.361966, 0.930794, -0.0467093, -0.0205526), (-0.930794, 0.361966, 0.0205526, -0.0467093), (0.669124, 0.102988, 0.593527, -0.435192), (-0.102988, 0.669124, 0.435192, 0.593527), (0.527984, 0.423752, 0.731606, -0.0801239), (-0.423752, 0.527984, 0.0801239, 0.731606), (0.245372, 0.630972, 0.673651, 0.296414), (-0.630972, 0.245372, -0.296414, 0.673651), (0.669124, 0.102988, -0.593527, 0.435192), (-0.102988, 0.669124, -0.435192, -0.593527), (0.527984, 0.423752, -0.731606, 0.0801239), (-0.423752, 0.527984, -0.0801239, -0.731606), (0.245372, 0.630972, -0.673651, -0.296414), (-0.630972, 0.245372, 0.296414, -0.673651), (0.987074, 0.151925, 0.0411537, -0.0301751), (-0.151925, 0.987074, 0.0301751, 0.0411537), (0.778868, 0.625108, 0.0507277, -0.00555559), (-0.625108, 0.778868, 0.00555559, 0.0507277), (0.361966, 0.930794, 0.0467093, 0.0205526), (-0.930794, 0.361966, -0.0205526, 0.0467093), (0.451123, 0.0694345, -0.135353, 0.879401), (-0.0694345, 0.451123, -0.879401, -0.135353), (0.355967, 0.285694, -0.556919, 0.693907), (-0.285694, 0.355967, -0.693907, -0.556919), (0.16543, 0.425401, -0.82926, 0.322482), (-0.425401, 0.16543, -0.322482, -0.82926), (0.879401, 0.135353, 0.0694345, -0.451123), (-0.135353, 0.879401, 0.451123, 0.0694345), (0.693907, 0.556919, 0.285694, -0.355967), (-0.556919, 0.693907, 0.355967, 0.285694), (0.322482, 0.82926, 0.425401, -0.16543), (-0.82926, 0.322482, 0.16543, 0.425401), (0.0504371, 0.00776301, -0.590539, -0.805394), (-0.00776301, 0.0504371, 0.805394, -0.590539), (0.0397983, 0.0319415, -0.108725, -0.992761), (-0.0319415, 0.0397983, 0.992761, -0.108725), (0.0184956, 0.0475613, 0.402222, -0.914119), (-0.0475613, 0.0184956, 0.914119, 0.402222), (0.727414, 0.11196, 0.400319, 0.545966), (-0.11196, 0.727414, -0.545966, 0.400319), (0.573979, 0.460667, 0.0737032, 0.672979), (-0.460667, 0.573979, -0.672979, 0.0737032), (0.266747, 0.685939, -0.272661, 0.619669), (-0.685939, 0.266747, -0.619669, -0.272661), (0.727414, 0.11196, -0.400319, -0.545966), (-0.11196, 0.727414, 0.545966, -0.400319), (0.573979, 0.460667, -0.0737032, -0.672979), (-0.460667, 0.573979, 0.672979, -0.0737032), (0.266747, 0.685939, 0.272661, -0.619669), (-0.685939, 0.266747, 0.619669, 0.272661), (0.0504371, 0.00776301, 0.590539, 0.805394), (-0.00776301, 0.0504371, -0.805394, 0.590539), (0.0397983, 0.0319415, 0.108725, 0.992761), (-0.0319415, 0.0397983, -0.992761, 0.108725), (0.0184956, 0.0475613, -0.402222, 0.914119), (-0.0475613, 0.0184956, -0.914119, -0.402222), (0.619669, 0.272661, -0.673651, -0.296414), (-0.272661, 0.619669, 0.296414, -0.673651), (0.400319, 0.545966, -0.435192, -0.593527), (-0.545966, 0.400319, 0.593527, -0.435192), (0.0737032, 0.672979, -0.0801239, -0.731606), (-0.672979, 0.0737032, 0.731606, -0.0801239), (0.673651, 0.296414, 0.619669, 0.272661), (-0.296414, 0.673651, -0.272661, 0.619669), (0.435192, 0.593527, 0.400319, 0.545966), (-0.593527, 0.435192, -0.545966, 0.400319), (0.0801239, 0.731606, 0.0737032, 0.672979), (-0.731606, 0.0801239, -0.672979, 0.0737032), (0.814404, 0.358346, -0.425401, 0.16543), (-0.358346, 0.814404, -0.16543, -0.425401), (0.526122, 0.717539, -0.451123, -0.0694345), (-0.717539, 0.526122, 0.0694345, -0.451123), (0.096865, 0.884468, -0.355967, -0.285694), (-0.884468, 0.096865, 0.285694, -0.355967), (0.326965, 0.143868, 0.870515, -0.338525), (-0.143868, 0.326965, 0.338525, 0.870515), (0.211226, 0.288076, 0.923151, 0.142086), (-0.288076, 0.211226, -0.142086, 0.923151), (0.0388891, 0.355094, 0.728429, 0.584626), (-0.355094, 0.0388891, -0.584626, 0.728429), (0.326965, 0.143868, -0.870515, 0.338525), (-0.143868, 0.326965, -0.338525, -0.870515), (0.211226, 0.288076, -0.923151, -0.142086), (-0.288076, 0.211226, 0.142086, -0.923151), (0.0388891, 0.355094, -0.728429, -0.584626), (-0.355094, 0.0388891, 0.584626, -0.728429), (0.814404, 0.358346, 0.425401, -0.16543), (-0.358346, 0.814404, 0.16543, 0.425401), (0.526122, 0.717539, 0.451123, 0.0694345), (-0.717539, 0.526122, -0.0694345, 0.451123), (0.096865, 0.884468, 0.355967, 0.285694), (-0.884468, 0.096865, -0.285694, 0.355967), (0.673651, 0.296414, -0.272661, 0.619669), (-0.296414, 0.673651, -0.619669, -0.272661), (0.435192, 0.593527, -0.545966, 0.400319), (-0.593527, 0.435192, -0.400319, -0.545966), (0.0801239, 0.731606, -0.672979, 0.0737032), (-0.731606, 0.0801239, -0.0737032, -0.672979), (0.619669, 0.272661, 0.296414, -0.673651), (-0.272661, 0.619669, 0.673651, 0.296414), (0.400319, 0.545966, 0.593527, -0.435192), (-0.545966, 0.400319, 0.435192, 0.593527), (0.0737032, 0.672979, 0.731606, -0.0801239), (-0.672979, 0.0737032, 0.0801239, 0.731606), (0.417781, 0.183828, -0.322482, -0.82926), (-0.183828, 0.417781, 0.82926, -0.322482), (0.269895, 0.36809, 0.135353, -0.879401), (-0.36809, 0.269895, 0.879401, 0.135353), (0.0496907, 0.453723, 0.556919, -0.693907), (-0.453723, 0.0496907, 0.693907, 0.556919), (0.85492, 0.376174, 0.129469, 0.332929), (-0.376174, 0.85492, -0.332929, 0.129469), (0.552296, 0.753236, -0.0543411, 0.35306), (-0.753236, 0.552296, -0.35306, -0.0543411), (0.101684, 0.92847, -0.223591, 0.278588), (-0.92847, 0.101684, -0.278588, -0.223591), (0.85492, 0.376174, -0.129469, -0.332929), (-0.376174, 0.85492, 0.332929, -0.129469), (0.552296, 0.753236, 0.0543411, -0.35306), (-0.753236, 0.552296, 0.35306, 0.0543411), (0.101684, 0.92847, 0.223591, -0.278588), (-0.92847, 0.101684, 0.278588, 0.223591), (0.417781, 0.183828, 0.322482, 0.82926), (-0.183828, 0.417781, -0.82926, 0.322482), (0.269895, 0.36809, -0.135353, 0.879401), (-0.36809, 0.269895, -0.879401, -0.135353), (0.0496907, 0.453723, -0.556919, 0.693907), (-0.453723, 0.0496907, -0.693907, -0.556919), (0.814404, 0.358346, -0.417781, -0.183828), (-0.358346, 0.814404, 0.183828, -0.417781), (0.526122, 0.717539, -0.269895, -0.36809), (-0.717539, 0.526122, 0.36809, -0.269895), (0.096865, 0.884468, -0.0496907, -0.453723), (-0.884468, 0.096865, 0.453723, -0.0496907), (0.417781, 0.183828, 0.814404, 0.358346), (-0.183828, 0.417781, -0.358346, 0.814404), (0.269895, 0.36809, 0.526122, 0.717539), (-0.36809, 0.269895, -0.717539, 0.526122), (0.0496907, 0.453723, 0.096865, 0.884468), (-0.453723, 0.0496907, -0.884468, 0.096865), (0.914119, 0.402222, -0.0475613, 0.0184956), (-0.402222, 0.914119, -0.0184956, -0.0475613), (0.590539, 0.805394, -0.0504371, -0.00776301), (-0.805394, 0.590539, 0.00776301, -0.0504371), (0.108725, 0.992761, -0.0397983, -0.0319415), (-0.992761, 0.108725, 0.0319415, -0.0397983), (0.619669, 0.272661, 0.685939, -0.266747), (-0.272661, 0.619669, 0.266747, 0.685939), (0.400319, 0.545966, 0.727414, 0.11196), (-0.545966, 0.400319, -0.11196, 0.727414), (0.0737032, 0.672979, 0.573979, 0.460667), (-0.672979, 0.0737032, -0.460667, 0.573979), (0.619669, 0.272661, -0.685939, 0.266747), (-0.272661, 0.619669, -0.266747, -0.685939), (0.400319, 0.545966, -0.727414, -0.11196), (-0.545966, 0.400319, 0.11196, -0.727414), (0.0737032, 0.672979, -0.573979, -0.460667), (-0.672979, 0.0737032, 0.460667, -0.573979), (0.914119, 0.402222, 0.0475613, -0.0184956), (-0.402222, 0.914119, 0.0184956, 0.0475613), (0.590539, 0.805394, 0.0504371, 0.00776301), (-0.805394, 0.590539, -0.00776301, 0.0504371), (0.108725, 0.992761, 0.0397983, 0.0319415), (-0.992761, 0.108725, -0.0319415, 0.0397983), (0.417781, 0.183828, -0.358346, 0.814404), (-0.183828, 0.417781, -0.814404, -0.358346), (0.269895, 0.36809, -0.717539, 0.526122), (-0.36809, 0.269895, -0.526122, -0.717539), (0.0496907, 0.453723, -0.884468, 0.096865), (-0.453723, 0.0496907, -0.096865, -0.884468), (0.814404, 0.358346, 0.183828, -0.417781), (-0.358346, 0.814404, 0.417781, 0.183828), (0.526122, 0.717539, 0.36809, -0.269895), (-0.717539, 0.526122, 0.269895, 0.36809), (0.096865, 0.884468, 0.453723, -0.0496907), (-0.884468, 0.096865, 0.0496907, 0.453723), (0.0467093, 0.0205526, -0.361966, -0.930794), (-0.0205526, 0.0467093, 0.930794, -0.361966), (0.0301751, 0.0411537, 0.151925, -0.987074), (-0.0411537, 0.0301751, 0.987074, 0.151925), (0.00555559, 0.0507277, 0.625108, -0.778868), (-0.0507277, 0.00555559, 0.778868, 0.625108), (0.673651, 0.296414, 0.245372, 0.630972), (-0.296414, 0.673651, -0.630972, 0.245372), (0.435192, 0.593527, -0.102988, 0.669124), (-0.593527, 0.435192, -0.669124, -0.102988), (0.0801239, 0.731606, -0.423752, 0.527984), (-0.731606, 0.0801239, -0.527984, -0.423752), (0.673651, 0.296414, -0.245372, -0.630972), (-0.296414, 0.673651, 0.630972, -0.245372), (0.435192, 0.593527, 0.102988, -0.669124), (-0.593527, 0.435192, 0.669124, 0.102988), (0.0801239, 0.731606, 0.423752, -0.527984), (-0.731606, 0.0801239, 0.527984, 0.423752), (0.0467093, 0.0205526, 0.361966, 0.930794), (-0.0205526, 0.0467093, -0.930794, 0.361966), (0.0301751, 0.0411537, -0.151925, 0.987074), (-0.0411537, 0.0301751, -0.987074, -0.151925), (0.00555559, 0.0507277, -0.625108, 0.778868), (-0.0507277, 0.00555559, -0.778868, -0.625108), (0.669124, 0.102988, -0.727414, -0.11196), (-0.102988, 0.669124, 0.11196, -0.727414), (0.527984, 0.423752, -0.573979, -0.460667), (-0.423752, 0.527984, 0.460667, -0.573979), (0.245372, 0.630972, -0.266747, -0.685939), (-0.630972, 0.245372, 0.685939, -0.266747), (0.727414, 0.11196, 0.669124, 0.102988), (-0.11196, 0.727414, -0.102988, 0.669124), (0.573979, 0.460667, 0.527984, 0.423752), (-0.460667, 0.573979, -0.423752, 0.527984), (0.266747, 0.685939, 0.245372, 0.630972), (-0.685939, 0.266747, -0.630972, 0.245372), (0.879401, 0.135353, -0.36809, 0.269895), (-0.135353, 0.879401, -0.269895, -0.36809), (0.693907, 0.556919, -0.453723, 0.0496907), (-0.556919, 0.693907, -0.0496907, -0.453723), (0.322482, 0.82926, -0.417781, -0.183828), (-0.82926, 0.322482, 0.183828, -0.417781), (0.35306, 0.0543411, 0.753236, -0.552296), (-0.0543411, 0.35306, 0.552296, 0.753236), (0.278588, 0.223591, 0.92847, -0.101684), (-0.223591, 0.278588, 0.101684, 0.92847), (0.129469, 0.332929, 0.85492, 0.376174), (-0.332929, 0.129469, -0.376174, 0.85492), (0.35306, 0.0543411, -0.753236, 0.552296), (-0.0543411, 0.35306, -0.552296, -0.753236), (0.278588, 0.223591, -0.92847, 0.101684), (-0.223591, 0.278588, -0.101684, -0.92847), (0.129469, 0.332929, -0.85492, -0.376174), (-0.332929, 0.129469, 0.376174, -0.85492), (0.879401, 0.135353, 0.36809, -0.269895), (-0.135353, 0.879401, 0.269895, 0.36809), (0.693907, 0.556919, 0.453723, -0.0496907), (-0.556919, 0.693907, 0.0496907, 0.453723), (0.322482, 0.82926, 0.417781, 0.183828), (-0.82926, 0.322482, -0.183828, 0.417781), (0.727414, 0.11196, -0.102988, 0.669124), (-0.11196, 0.727414, -0.669124, -0.102988), (0.573979, 0.460667, -0.423752, 0.527984), (-0.460667, 0.573979, -0.527984, -0.423752), (0.266747, 0.685939, -0.630972, 0.245372), (-0.685939, 0.266747, -0.245372, -0.630972), (0.669124, 0.102988, 0.11196, -0.727414), (-0.102988, 0.669124, 0.727414, 0.11196), (0.527984, 0.423752, 0.460667, -0.573979), (-0.423752, 0.527984, 0.573979, 0.460667), (0.245372, 0.630972, 0.685939, -0.266747), (-0.630972, 0.245372, 0.266747, 0.685939), (0.451123, 0.0694345, -0.526122, -0.717539), (-0.0694345, 0.451123, 0.717539, -0.526122), (0.355967, 0.285694, -0.096865, -0.884468), (-0.285694, 0.355967, 0.884468, -0.096865), (0.16543, 0.425401, 0.358346, -0.814404), (-0.425401, 0.16543, 0.814404, 0.358346), (0.923151, 0.142086, 0.211226, 0.288076), (-0.142086, 0.923151, -0.288076, 0.211226), (0.728429, 0.584626, 0.0388891, 0.355094), (-0.584626, 0.728429, -0.355094, 0.0388891), (0.338525, 0.870515, -0.143868, 0.326965), (-0.870515, 0.338525, -0.326965, -0.143868), (0.923151, 0.142086, -0.211226, -0.288076), (-0.142086, 0.923151, 0.288076, -0.211226), (0.728429, 0.584626, -0.0388891, -0.355094), (-0.584626, 0.728429, 0.355094, -0.0388891), (0.338525, 0.870515, 0.143868, -0.326965), (-0.870515, 0.338525, 0.326965, 0.143868), (0.451123, 0.0694345, 0.526122, 0.717539), (-0.0694345, 0.451123, -0.717539, 0.526122), (0.355967, 0.285694, 0.096865, 0.884468), (-0.285694, 0.355967, -0.884468, 0.096865), (0.16543, 0.425401, -0.358346, 0.814404), (-0.425401, 0.16543, -0.814404, -0.358346), (0.781368, 0.120264, -0.594823, 0.145552), (-0.120264, 0.781368, -0.145552, -0.594823), (0.616553, 0.494836, -0.587908, -0.17136), (-0.494836, 0.616553, 0.17136, -0.587908), (0.286532, 0.736817, -0.423464, -0.442356), (-0.736817, 0.286532, 0.442356, -0.423464), (0.605245, 0.0931562, 0.767913, -0.187907), (-0.0931562, 0.605245, 0.187907, 0.767913), (0.47758, 0.383298, 0.758986, 0.221224), (-0.383298, 0.47758, -0.221224, 0.758986), (0.221947, 0.570736, 0.546689, 0.571079), (-0.570736, 0.221947, -0.571079, 0.546689), (0.955647, 0.147088, -0.255153, 0.00111332), (-0.147088, 0.955647, -0.00111332, -0.255153), (0.75407, 0.605205, -0.221525, -0.126612), (-0.605205, 0.75407, 0.126612, -0.221525), (0.350441, 0.901159, -0.128541, -0.220412), (-0.901159, 0.350441, 0.220412, -0.128541), (0.533776, 0.082156, 0.817506, -0.200042), (-0.082156, 0.533776, 0.200042, 0.817506), (0.421186, 0.338037, 0.808002, 0.235511), (-0.338037, 0.421186, -0.235511, 0.808002), (0.195739, 0.503342, 0.581995, 0.60796), (-0.503342, 0.195739, -0.60796, 0.581995), (0.533776, 0.082156, -0.817506, 0.200042), (-0.082156, 0.533776, -0.200042, -0.817506), (0.421186, 0.338037, -0.808002, -0.235511), (-0.338037, 0.421186, 0.235511, -0.808002), (0.195739, 0.503342, -0.581995, -0.60796), (-0.503342, 0.195739, 0.60796, -0.581995), (0.955647, 0.147088, 0.255153, -0.00111332), (-0.147088, 0.955647, 0.00111332, 0.255153), (0.75407, 0.605205, 0.221525, 0.126612), (-0.605205, 0.75407, -0.126612, 0.221525), (0.350441, 0.901159, 0.128541, 0.220412), (-0.901159, 0.350441, -0.220412, 0.128541), (0.605245, 0.0931562, 0.187907, 0.767913), (-0.0931562, 0.605245, -0.767913, 0.187907), (0.47758, 0.383298, -0.221224, 0.758986), (-0.383298, 0.47758, -0.758986, -0.221224), (0.221947, 0.570736, -0.571079, 0.546689), (-0.570736, 0.221947, -0.546689, -0.571079), (0.781368, 0.120264, -0.145552, -0.594823), (-0.120264, 0.781368, 0.594823, -0.145552), (0.616553, 0.494836, 0.17136, -0.587908), (-0.494836, 0.616553, 0.587908, 0.17136), (0.286532, 0.736817, 0.442356, -0.423464), (-0.736817, 0.286532, 0.423464, 0.442356), (0.252186, 0.0388151, -0.00421888, -0.966891), (-0.0388151, 0.252186, 0.966891, -0.00421888), (0.198992, 0.159708, 0.479792, -0.839462), (-0.159708, 0.198992, 0.839462, 0.479792), (0.0924779, 0.237807, 0.835243, -0.487099), (-0.237807, 0.0924779, 0.487099, 0.835243), (0.83183, 0.128031, 0.128365, 0.524585), (-0.128031, 0.83183, -0.524585, 0.128365), (0.656371, 0.526793, -0.151125, 0.518486), (-0.526793, 0.656371, -0.518486, -0.151125), (0.305037, 0.784402, -0.390121, 0.37346), (-0.784402, 0.305037, -0.37346, -0.390121), (0.83183, 0.128031, -0.128365, -0.524585), (-0.128031, 0.83183, 0.524585, -0.128365), (0.656371, 0.526793, 0.151125, -0.518486), (-0.526793, 0.656371, 0.518486, 0.151125), (0.305037, 0.784402, 0.390121, -0.37346), (-0.784402, 0.305037, 0.37346, 0.390121), (0.252186, 0.0388151, 0.00421888, 0.966891), (-0.0388151, 0.252186, -0.966891, 0.00421888), (0.198992, 0.159708, -0.479792, 0.839462), (-0.159708, 0.198992, -0.839462, -0.479792), (0.0924779, 0.237807, -0.835243, 0.487099), (-0.237807, 0.0924779, -0.487099, -0.835243), (0.723617, 0.318399, -0.423464, -0.442356), (-0.318399, 0.723617, 0.442356, -0.423464), (0.467471, 0.63755, -0.145552, -0.594823), (-0.63755, 0.467471, 0.594823, -0.145552), (0.0860668, 0.785871, 0.17136, -0.587908), (-0.785871, 0.0860668, 0.587908, 0.17136), (0.560512, 0.246631, 0.546689, 0.571079), (-0.246631, 0.560512, -0.571079, 0.546689), (0.362102, 0.493844, 0.187907, 0.767913), (-0.493844, 0.362102, -0.767913, 0.187907), (0.0666671, 0.608733, -0.221224, 0.758986), (-0.608733, 0.0666671, -0.758986, -0.221224), (0.885015, 0.389416, -0.138032, 0.214595), (-0.389416, 0.885015, -0.214595, -0.138032), (0.571737, 0.779751, -0.226837, 0.116829), (-0.779751, 0.571737, -0.116829, -0.226837), (0.105263, 0.961153, -0.254861, -0.0122419), (-0.961153, 0.105263, 0.0122419, -0.254861), (0.494325, 0.217508, 0.60796, -0.581995), (-0.217508, 0.494325, 0.581995, 0.60796), (0.319344, 0.43553, 0.817506, -0.200042), (-0.43553, 0.319344, 0.200042, 0.817506), (0.0587948, 0.536852, 0.808002, 0.235511), (-0.536852, 0.0587948, -0.235511, 0.808002), (0.494325, 0.217508, -0.60796, 0.581995), (-0.217508, 0.494325, -0.581995, -0.60796), (0.319344, 0.43553, -0.817506, 0.200042), (-0.43553, 0.319344, -0.200042, -0.817506), (0.0587948, 0.536852, -0.808002, -0.235511), (-0.536852, 0.0587948, 0.235511, -0.808002), (0.885015, 0.389416, 0.138032, -0.214595), (-0.389416, 0.885015, 0.214595, 0.138032), (0.571737, 0.779751, 0.226837, -0.116829), (-0.779751, 0.571737, 0.116829, 0.226837), (0.105263, 0.961153, 0.254861, 0.0122419), (-0.961153, 0.105263, -0.0122419, 0.254861), (0.560512, 0.246631, -0.571079, 0.546689), (-0.246631, 0.560512, -0.546689, -0.571079), (0.362102, 0.493844, -0.767913, 0.187907), (-0.493844, 0.362102, -0.187907, -0.767913), (0.0666671, 0.608733, -0.758986, -0.221224), (-0.608733, 0.0666671, 0.221224, -0.758986), (0.723617, 0.318399, 0.442356, -0.423464), (-0.318399, 0.723617, 0.423464, 0.442356), (0.467471, 0.63755, 0.594823, -0.145552), (-0.63755, 0.467471, 0.145552, 0.594823), (0.0860668, 0.785871, 0.587908, 0.17136), (-0.785871, 0.0860668, -0.17136, 0.587908), (0.233546, 0.102763, -0.813201, -0.523068), (-0.102763, 0.233546, 0.523068, -0.813201), (0.150876, 0.205769, -0.442718, -0.859591), (-0.205769, 0.150876, 0.859591, -0.442718), (0.0277779, 0.253639, 0.0463901, -0.965787), (-0.253639, 0.0277779, 0.965787, 0.0463901), (0.770349, 0.338962, 0.37346, 0.390121), (-0.338962, 0.770349, -0.390121, 0.37346), (0.497661, 0.678724, 0.128365, 0.524585), (-0.678724, 0.497661, -0.524585, 0.128365), (0.0916251, 0.836623, -0.151125, 0.518486), (-0.836623, 0.0916251, -0.518486, -0.151125), (0.770349, 0.338962, -0.37346, -0.390121), (-0.338962, 0.770349, 0.390121, -0.37346), (0.497661, 0.678724, -0.128365, -0.524585), (-0.678724, 0.497661, 0.524585, -0.128365), (0.0916251, 0.836623, 0.151125, -0.518486), (-0.836623, 0.0916251, 0.518486, 0.151125), (0.233546, 0.102763, 0.813201, 0.523068), (-0.102763, 0.233546, -0.523068, 0.813201), (0.150876, 0.205769, 0.442718, 0.859591), (-0.205769, 0.150876, -0.859591, 0.442718), (0.0277779, 0.253639, -0.0463901, 0.965787), (-0.253639, 0.0277779, -0.965787, -0.0463901), (0.723617, 0.318399, -0.612227, -0.0133588), (-0.318399, 0.723617, 0.0133588, -0.612227), (0.467471, 0.63755, -0.523524, -0.317682), (-0.63755, 0.467471, 0.317682, -0.523524), (0.0860668, 0.785871, -0.294544, -0.536883), (-0.785871, 0.0860668, 0.536883, -0.294544), (0.560512, 0.246631, 0.790381, 0.0172462), (-0.246631, 0.560512, -0.0172462, 0.790381), (0.362102, 0.493844, 0.675867, 0.410126), (-0.493844, 0.362102, -0.410126, 0.675867), (0.0666671, 0.608733, 0.380255, 0.693113), (-0.608733, 0.0666671, -0.693113, 0.380255), (0.885015, 0.389416, -0.246747, -0.064963), (-0.389416, 0.885015, 0.064963, -0.246747), (0.571737, 0.779751, -0.181207, -0.179633), (-0.779751, 0.571737, 0.179633, -0.181207), (0.105263, 0.961153, -0.0671138, -0.24617), (-0.961153, 0.105263, 0.24617, -0.0671138), (0.494325, 0.217508, 0.841425, 0.01836), (-0.217508, 0.494325, -0.01836, 0.841425), (0.319344, 0.43553, 0.719516, 0.436613), (-0.43553, 0.319344, -0.436613, 0.719516), (0.0587948, 0.536852, 0.404812, 0.737876), (-0.536852, 0.0587948, -0.737876, 0.404812), (0.494325, 0.217508, -0.841425, -0.01836), (-0.217508, 0.494325, 0.01836, -0.841425), (0.319344, 0.43553, -0.719516, -0.436613), (-0.43553, 0.319344, 0.436613, -0.719516), (0.0587948, 0.536852, -0.404812, -0.737876), (-0.536852, 0.0587948, 0.737876, -0.404812), (0.885015, 0.389416, 0.246747, 0.064963), (-0.389416, 0.885015, -0.064963, 0.246747), (0.571737, 0.779751, 0.181207, 0.179633), (-0.779751, 0.571737, -0.179633, 0.181207), (0.105263, 0.961153, 0.0671138, 0.24617), (-0.961153, 0.105263, -0.24617, 0.0671138), (0.560512, 0.246631, -0.0172462, 0.790381), (-0.246631, 0.560512, -0.790381, -0.0172462), (0.362102, 0.493844, -0.410126, 0.675867), (-0.493844, 0.362102, -0.675867, -0.410126), (0.0666671, 0.608733, -0.693113, 0.380255), (-0.608733, 0.0666671, -0.380255, -0.693113), (0.723617, 0.318399, 0.0133588, -0.612227), (-0.318399, 0.723617, 0.612227, 0.0133588), (0.467471, 0.63755, 0.317682, -0.523524), (-0.63755, 0.467471, 0.523524, 0.317682), (0.0860668, 0.785871, 0.536883, -0.294544), (-0.785871, 0.0860668, 0.294544, 0.536883), (0.233546, 0.102763, 0.246175, -0.935037), (-0.102763, 0.233546, 0.935037, 0.246175), (0.150876, 0.205769, 0.680712, -0.686678), (-0.205769, 0.150876, 0.686678, 0.680712), (0.0277779, 0.253639, 0.932853, -0.254325), (-0.253639, 0.0277779, 0.254325, 0.932853), (0.770349, 0.338962, -0.0117814, 0.539933), (-0.338962, 0.770349, -0.539933, -0.0117814), (0.497661, 0.678724, -0.28017, 0.461705), (-0.678724, 0.497661, -0.461705, -0.28017), (0.0916251, 0.836623, -0.473487, 0.259764), (-0.836623, 0.0916251, -0.259764, -0.473487), (0.770349, 0.338962, 0.0117814, -0.539933), (-0.338962, 0.770349, 0.539933, 0.0117814), (0.497661, 0.678724, 0.28017, -0.461705), (-0.678724, 0.497661, 0.461705, 0.28017), (0.0916251, 0.836623, 0.473487, -0.259764), (-0.836623, 0.0916251, 0.259764, 0.473487), (0.233546, 0.102763, -0.246175, 0.935037), (-0.102763, 0.233546, -0.935037, -0.246175), (0.150876, 0.205769, -0.680712, 0.686678), (-0.205769, 0.150876, -0.686678, -0.680712), (0.0277779, 0.253639, -0.932853, 0.254325), (-0.253639, 0.0277779, -0.254325, -0.932853), (0.781368, 0.120264, -0.523524, -0.317682), (-0.120264, 0.781368, 0.317682, -0.523524), (0.616553, 0.494836, -0.294544, -0.536883), (-0.494836, 0.616553, 0.536883, -0.294544), (0.286532, 0.736817, 0.0133588, -0.612227), (-0.736817, 0.286532, 0.612227, 0.0133588), (0.605245, 0.0931562, 0.675867, 0.410126), (-0.0931562, 0.605245, -0.410126, 0.675867), (0.47758, 0.383298, 0.380255, 0.693113), (-0.383298, 0.47758, -0.693113, 0.380255), (0.221947, 0.570736, -0.0172462, 0.790381), (-0.570736, 0.221947, -0.790381, -0.0172462), (0.955647, 0.147088, -0.0777877, 0.243009), (-0.147088, 0.955647, -0.243009, -0.0777877), (0.75407, 0.605205, -0.18887, 0.171558), (-0.605205, 0.75407, -0.171558, -0.18887), (0.350441, 0.901159, -0.249346, 0.0541382), (-0.901159, 0.350441, -0.0541382, -0.249346), (0.533776, 0.082156, 0.436613, -0.719516), (-0.082156, 0.533776, 0.719516, 0.436613), (0.421186, 0.338037, 0.737876, -0.404812), (-0.338037, 0.421186, 0.404812, 0.737876), (0.195739, 0.503342, 0.841425, 0.01836), (-0.503342, 0.195739, -0.01836, 0.841425), (0.533776, 0.082156, -0.436613, 0.719516), (-0.082156, 0.533776, -0.719516, -0.436613), (0.421186, 0.338037, -0.737876, 0.404812), (-0.338037, 0.421186, -0.404812, -0.737876), (0.195739, 0.503342, -0.841425, -0.01836), (-0.503342, 0.195739, 0.01836, -0.841425), (0.955647, 0.147088, 0.0777877, -0.243009), (-0.147088, 0.955647, 0.243009, 0.0777877), (0.75407, 0.605205, 0.18887, -0.171558), (-0.605205, 0.75407, 0.171558, 0.18887), (0.350441, 0.901159, 0.249346, -0.0541382), (-0.901159, 0.350441, 0.0541382, 0.249346), (0.605245, 0.0931562, -0.410126, 0.675867), (-0.0931562, 0.605245, -0.675867, -0.410126), (0.47758, 0.383298, -0.693113, 0.380255), (-0.383298, 0.47758, -0.380255, -0.693113), (0.221947, 0.570736, -0.790381, -0.0172462), (-0.570736, 0.221947, 0.0172462, -0.790381), (0.781368, 0.120264, 0.317682, -0.523524), (-0.120264, 0.781368, 0.523524, 0.317682), (0.616553, 0.494836, 0.536883, -0.294544), (-0.494836, 0.616553, 0.294544, 0.536883), (0.286532, 0.736817, 0.612227, 0.0133588), (-0.736817, 0.286532, -0.0133588, 0.612227), (0.252186, 0.0388151, -0.920872, -0.294773), (-0.0388151, 0.252186, 0.294773, -0.920872), (0.198992, 0.159708, -0.650112, -0.715717), (-0.159708, 0.198992, 0.715717, -0.650112), (0.0924779, 0.237807, -0.205155, -0.944885), (-0.237807, 0.0924779, 0.944885, -0.205155), (0.83183, 0.128031, 0.461705, 0.28017), (-0.128031, 0.83183, -0.28017, 0.461705), (0.656371, 0.526793, 0.259764, 0.473487), (-0.526793, 0.656371, -0.473487, 0.259764), (0.305037, 0.784402, -0.0117814, 0.539933), (-0.784402, 0.305037, -0.539933, -0.0117814), (0.83183, 0.128031, -0.461705, -0.28017), (-0.128031, 0.83183, 0.28017, -0.461705), (0.656371, 0.526793, -0.259764, -0.473487), (-0.526793, 0.656371, 0.473487, -0.259764), (0.305037, 0.784402, 0.0117814, -0.539933), (-0.784402, 0.305037, 0.539933, 0.0117814), (0.252186, 0.0388151, 0.920872, 0.294773), (-0.0388151, 0.252186, -0.294773, 0.920872), (0.198992, 0.159708, 0.650112, 0.715717), (-0.159708, 0.198992, -0.715717, 0.650112), (0.0924779, 0.237807, 0.205155, 0.944885), (-0.237807, 0.0924779, -0.944885, 0.205155), (0.790381, 0.0172462, -0.612227, -0.0133588), (-0.0172462, 0.790381, 0.0133588, -0.612227), (0.675867, 0.410126, -0.523524, -0.317682), (-0.410126, 0.675867, 0.317682, -0.523524), (0.380255, 0.693113, -0.294544, -0.536883), (-0.693113, 0.380255, 0.536883, -0.294544), (0.612227, 0.0133588, 0.790381, 0.0172462), (-0.0133588, 0.612227, -0.0172462, 0.790381), (0.523524, 0.317682, 0.675867, 0.410126), (-0.317682, 0.523524, -0.410126, 0.675867), (0.294544, 0.536883, 0.380255, 0.693113), (-0.536883, 0.294544, -0.693113, 0.380255), (0.96667, 0.0210928, -0.184315, 0.176443), (-0.0210928, 0.96667, -0.176443, -0.184315), (0.826614, 0.501602, -0.247843, 0.0606468), (-0.501602, 0.826614, -0.0606468, -0.247843), (0.465068, 0.847707, -0.244962, -0.0713998), (-0.847707, 0.465068, 0.0713998, -0.244962), (0.539933, 0.0117814, 0.60796, -0.581995), (-0.0117814, 0.539933, 0.581995, 0.60796), (0.461705, 0.28017, 0.817506, -0.200042), (-0.28017, 0.461705, 0.200042, 0.817506), (0.259764, 0.473487, 0.808002, 0.235511), (-0.473487, 0.259764, -0.235511, 0.808002), (0.539933, 0.0117814, -0.60796, 0.581995), (-0.0117814, 0.539933, -0.581995, -0.60796), (0.461705, 0.28017, -0.817506, 0.200042), (-0.28017, 0.461705, -0.200042, -0.817506), (0.259764, 0.473487, -0.808002, -0.235511), (-0.473487, 0.259764, 0.235511, -0.808002), (0.96667, 0.0210928, 0.184315, -0.176443), (-0.0210928, 0.96667, 0.176443, 0.184315), (0.826614, 0.501602, 0.247843, -0.0606468), (-0.501602, 0.826614, 0.0606468, 0.247843), (0.465068, 0.847707, 0.244962, 0.0713998), (-0.847707, 0.465068, -0.0713998, 0.244962), (0.612227, 0.0133588, -0.0172462, 0.790381), (-0.0133588, 0.612227, -0.790381, -0.0172462), (0.523524, 0.317682, -0.410126, 0.675867), (-0.317682, 0.523524, -0.675867, -0.410126), (0.294544, 0.536883, -0.693113, 0.380255), (-0.536883, 0.294544, -0.380255, -0.693113), (0.790381, 0.0172462, 0.0133588, -0.612227), (-0.0172462, 0.790381, 0.612227, 0.0133588), (0.675867, 0.410126, 0.317682, -0.523524), (-0.410126, 0.675867, 0.523524, 0.317682), (0.380255, 0.693113, 0.536883, -0.294544), (-0.693113, 0.380255, 0.294544, 0.536883), (0.255094, 0.00556618, -0.668624, -0.698454), (-0.00556618, 0.255094, 0.698454, -0.668624), (0.218135, 0.132368, -0.229819, -0.939191), (-0.132368, 0.218135, 0.939191, -0.229819), (0.122727, 0.223701, 0.270567, -0.928272), (-0.223701, 0.122727, 0.928272, 0.270567), (0.841425, 0.01836, 0.37346, 0.390121), (-0.01836, 0.841425, -0.390121, 0.37346), (0.719516, 0.436613, 0.128365, 0.524585), (-0.436613, 0.719516, -0.524585, 0.128365), (0.404812, 0.737876, -0.151125, 0.518486), (-0.737876, 0.404812, -0.518486, -0.151125), (0.841425, 0.01836, -0.37346, -0.390121), (-0.01836, 0.841425, 0.390121, -0.37346), (0.719516, 0.436613, -0.128365, -0.524585), (-0.436613, 0.719516, 0.524585, -0.128365), (0.404812, 0.737876, 0.151125, -0.518486), (-0.737876, 0.404812, 0.518486, 0.151125), (0.255094, 0.00556618, 0.668624, 0.698454), (-0.00556618, 0.255094, -0.698454, 0.668624), (0.218135, 0.132368, 0.229819, 0.939191), (-0.132368, 0.218135, -0.939191, 0.229819), (0.122727, 0.223701, -0.270567, 0.928272), (-0.223701, 0.122727, -0.928272, -0.270567), (0.518486, 0.151125, -0.808002, -0.235511), (-0.151125, 0.518486, 0.235511, -0.808002), (0.37346, 0.390121, -0.581995, -0.60796), (-0.390121, 0.37346, 0.60796, -0.581995), (0.128365, 0.524585, -0.200042, -0.817506), (-0.524585, 0.128365, 0.817506, -0.200042), (0.808002, 0.235511, 0.518486, 0.151125), (-0.235511, 0.808002, -0.151125, 0.518486), (0.581995, 0.60796, 0.37346, 0.390121), (-0.60796, 0.581995, -0.390121, 0.37346), (0.200042, 0.817506, 0.128365, 0.524585), (-0.817506, 0.200042, -0.524585, 0.128365), (0.758986, 0.221224, -0.536883, 0.294544), (-0.221224, 0.758986, -0.294544, -0.536883), (0.546689, 0.571079, -0.612227, -0.0133588), (-0.571079, 0.546689, 0.0133588, -0.612227), (0.187907, 0.767913, -0.523524, -0.317682), (-0.767913, 0.187907, 0.317682, -0.523524), (0.146977, 0.0428399, 0.866392, -0.475319), (-0.0428399, 0.146977, 0.475319, 0.866392), (0.105866, 0.110589, 0.987977, 0.0215577), (-0.110589, 0.105866, -0.0215577, 0.987977), (0.0363881, 0.148706, 0.844834, 0.512658), (-0.148706, 0.0363881, -0.512658, 0.844834), (0.146977, 0.0428399, -0.866392, 0.475319), (-0.0428399, 0.146977, -0.475319, -0.866392), (0.105866, 0.110589, -0.987977, -0.0215577), (-0.110589, 0.105866, 0.0215577, -0.987977), (0.0363881, 0.148706, -0.844834, -0.512658), (-0.148706, 0.0363881, 0.512658, -0.844834), (0.758986, 0.221224, 0.536883, -0.294544), (-0.221224, 0.758986, 0.294544, 0.536883), (0.546689, 0.571079, 0.612227, 0.0133588), (-0.571079, 0.546689, -0.0133588, 0.612227), (0.187907, 0.767913, 0.523524, 0.317682), (-0.767913, 0.187907, -0.317682, 0.523524), (0.808002, 0.235511, -0.151125, 0.518486), (-0.235511, 0.808002, -0.518486, -0.151125), (0.581995, 0.60796, -0.390121, 0.37346), (-0.60796, 0.581995, -0.37346, -0.390121), (0.200042, 0.817506, -0.524585, 0.128365), (-0.817506, 0.200042, -0.128365, -0.524585), (0.518486, 0.151125, 0.235511, -0.808002), (-0.151125, 0.518486, 0.808002, 0.235511), (0.37346, 0.390121, 0.60796, -0.581995), (-0.390121, 0.37346, 0.581995, 0.60796), (0.128365, 0.524585, 0.817506, -0.200042), (-0.524585, 0.128365, 0.200042, 0.817506), (0.587908, 0.17136, -0.380255, -0.693113), (-0.17136, 0.587908, 0.693113, -0.380255), (0.423464, 0.442356, 0.0172462, -0.790381), (-0.442356, 0.423464, 0.790381, 0.0172462), (0.145552, 0.594823, 0.410126, -0.675867), (-0.594823, 0.145552, 0.675867, 0.410126), (0.948733, 0.27653, 0.0736361, 0.134221), (-0.27653, 0.948733, -0.134221, 0.0736361), (0.683361, 0.713849, -0.00333971, 0.153057), (-0.713849, 0.683361, -0.153057, -0.00333971), (0.234884, 0.959892, -0.0794206, 0.130881), (-0.959892, 0.234884, -0.130881, -0.0794206), (0.948733, 0.27653, -0.0736361, -0.134221), (-0.27653, 0.948733, 0.134221, -0.0736361), (0.683361, 0.713849, 0.00333971, -0.153057), (-0.713849, 0.683361, 0.153057, 0.00333971), (0.234884, 0.959892, 0.0794206, -0.130881), (-0.959892, 0.234884, 0.130881, 0.0794206), (0.587908, 0.17136, 0.380255, 0.693113), (-0.17136, 0.587908, -0.693113, 0.380255), (0.423464, 0.442356, -0.0172462, 0.790381), (-0.442356, 0.423464, -0.790381, -0.0172462), (0.145552, 0.594823, -0.410126, 0.675867), (-0.594823, 0.145552, -0.675867, -0.410126), (0.758986, 0.221224, -0.587908, -0.17136), (-0.221224, 0.758986, 0.17136, -0.587908), (0.546689, 0.571079, -0.423464, -0.442356), (-0.571079, 0.546689, 0.442356, -0.423464), (0.187907, 0.767913, -0.145552, -0.594823), (-0.767913, 0.187907, 0.594823, -0.145552), (0.587908, 0.17136, 0.758986, 0.221224), (-0.17136, 0.587908, -0.221224, 0.758986), (0.423464, 0.442356, 0.546689, 0.571079), (-0.442356, 0.423464, -0.571079, 0.546689), (0.145552, 0.594823, 0.187907, 0.767913), (-0.594823, 0.145552, -0.767913, 0.187907), (0.928272, 0.270567, -0.223701, 0.122727), (-0.270567, 0.928272, -0.122727, -0.223701), (0.668624, 0.698454, -0.255094, -0.00556618), (-0.698454, 0.668624, 0.00556618, -0.255094), (0.229819, 0.939191, -0.218135, -0.132368), (-0.939191, 0.229819, 0.132368, -0.218135), (0.518486, 0.151125, 0.737876, -0.404812), (-0.151125, 0.518486, 0.404812, 0.737876), (0.37346, 0.390121, 0.841425, 0.01836), (-0.390121, 0.37346, -0.01836, 0.841425), (0.128365, 0.524585, 0.719516, 0.436613), (-0.524585, 0.128365, -0.436613, 0.719516), (0.518486, 0.151125, -0.737876, 0.404812), (-0.151125, 0.518486, -0.404812, -0.737876), (0.37346, 0.390121, -0.841425, -0.01836), (-0.390121, 0.37346, 0.01836, -0.841425), (0.128365, 0.524585, -0.719516, -0.436613), (-0.524585, 0.128365, 0.436613, -0.719516), (0.928272, 0.270567, 0.223701, -0.122727), (-0.270567, 0.928272, 0.122727, 0.223701), (0.668624, 0.698454, 0.255094, 0.00556618), (-0.698454, 0.668624, -0.00556618, 0.255094), (0.229819, 0.939191, 0.218135, 0.132368), (-0.939191, 0.229819, -0.132368, 0.218135), (0.587908, 0.17136, -0.221224, 0.758986), (-0.17136, 0.587908, -0.758986, -0.221224), (0.423464, 0.442356, -0.571079, 0.546689), (-0.442356, 0.423464, -0.546689, -0.571079), (0.145552, 0.594823, -0.767913, 0.187907), (-0.594823, 0.145552, -0.187907, -0.767913), (0.758986, 0.221224, 0.17136, -0.587908), (-0.221224, 0.758986, 0.587908, 0.17136), (0.546689, 0.571079, 0.442356, -0.423464), (-0.571079, 0.546689, 0.423464, 0.442356), (0.187907, 0.767913, 0.594823, -0.145552), (-0.767913, 0.187907, 0.145552, 0.594823), (0.244962, 0.0713998, -0.465068, -0.847707), (-0.0713998, 0.244962, 0.847707, -0.465068), (0.176443, 0.184315, 0.0210928, -0.96667), (-0.184315, 0.176443, 0.96667, 0.0210928), (0.0606468, 0.247843, 0.501602, -0.826614), (-0.247843, 0.0606468, 0.826614, 0.501602), (0.808002, 0.235511, 0.259764, 0.473487), (-0.235511, 0.808002, -0.473487, 0.259764), (0.581995, 0.60796, -0.0117814, 0.539933), (-0.60796, 0.581995, -0.539933, -0.0117814), (0.200042, 0.817506, -0.28017, 0.461705), (-0.817506, 0.200042, -0.461705, -0.28017), (0.808002, 0.235511, -0.259764, -0.473487), (-0.235511, 0.808002, 0.473487, -0.259764), (0.581995, 0.60796, 0.0117814, -0.539933), (-0.60796, 0.581995, 0.539933, 0.0117814), (0.200042, 0.817506, 0.28017, -0.461705), (-0.817506, 0.200042, 0.461705, 0.28017), (0.244962, 0.0713998, 0.465068, 0.847707), (-0.0713998, 0.244962, -0.847707, 0.465068), (0.176443, 0.184315, -0.0210928, 0.96667), (-0.184315, 0.176443, -0.96667, -0.0210928), (0.0606468, 0.247843, -0.501602, 0.826614), (-0.247843, 0.0606468, -0.826614, -0.501602), (0.539933, 0.0117814, -0.841425, -0.01836), (-0.0117814, 0.539933, 0.01836, -0.841425), (0.461705, 0.28017, -0.719516, -0.436613), (-0.28017, 0.461705, 0.436613, -0.719516), (0.259764, 0.473487, -0.404812, -0.737876), (-0.473487, 0.259764, 0.737876, -0.404812), (0.841425, 0.01836, 0.539933, 0.0117814), (-0.01836, 0.841425, -0.0117814, 0.539933), (0.719516, 0.436613, 0.461705, 0.28017), (-0.436613, 0.719516, -0.28017, 0.461705), (0.404812, 0.737876, 0.259764, 0.473487), (-0.737876, 0.404812, -0.473487, 0.259764), (0.790381, 0.0172462, -0.442356, 0.423464), (-0.0172462, 0.790381, -0.423464, -0.442356), (0.675867, 0.410126, -0.594823, 0.145552), (-0.410126, 0.675867, -0.145552, -0.594823), (0.380255, 0.693113, -0.587908, -0.17136), (-0.693113, 0.380255, 0.17136, -0.587908), (0.153057, 0.00333971, 0.713849, -0.683361), (-0.00333971, 0.153057, 0.683361, 0.713849), (0.130881, 0.0794206, 0.959892, -0.234884), (-0.0794206, 0.130881, 0.234884, 0.959892), (0.0736361, 0.134221, 0.948733, 0.27653), (-0.134221, 0.0736361, -0.27653, 0.948733), (0.153057, 0.00333971, -0.713849, 0.683361), (-0.00333971, 0.153057, -0.683361, -0.713849), (0.130881, 0.0794206, -0.959892, 0.234884), (-0.0794206, 0.130881, -0.234884, -0.959892), (0.0736361, 0.134221, -0.948733, -0.27653), (-0.134221, 0.0736361, 0.27653, -0.948733), (0.790381, 0.0172462, 0.442356, -0.423464), (-0.0172462, 0.790381, 0.423464, 0.442356), (0.675867, 0.410126, 0.594823, -0.145552), (-0.410126, 0.675867, 0.145552, 0.594823), (0.380255, 0.693113, 0.587908, 0.17136), (-0.693113, 0.380255, -0.17136, 0.587908), (0.841425, 0.01836, -0.0117814, 0.539933), (-0.01836, 0.841425, -0.539933, -0.0117814), (0.719516, 0.436613, -0.28017, 0.461705), (-0.436613, 0.719516, -0.461705, -0.28017), (0.404812, 0.737876, -0.473487, 0.259764), (-0.737876, 0.404812, -0.259764, -0.473487), (0.539933, 0.0117814, 0.01836, -0.841425), (-0.0117814, 0.539933, 0.841425, 0.01836), (0.461705, 0.28017, 0.436613, -0.719516), (-0.28017, 0.461705, 0.719516, 0.436613), (0.259764, 0.473487, 0.737876, -0.404812), (-0.473487, 0.259764, 0.404812, 0.737876), (0.612227, 0.0133588, -0.546689, -0.571079), (-0.0133588, 0.612227, 0.571079, -0.546689), (0.523524, 0.317682, -0.187907, -0.767913), (-0.317682, 0.523524, 0.767913, -0.187907), (0.294544, 0.536883, 0.221224, -0.758986), (-0.536883, 0.294544, 0.758986, 0.221224), (0.987977, 0.0215577, 0.105866, 0.110589), (-0.0215577, 0.987977, -0.110589, 0.105866), (0.844834, 0.512658, 0.0363881, 0.148706), (-0.512658, 0.844834, -0.148706, 0.0363881), (0.475319, 0.866392, -0.0428399, 0.146977), (-0.866392, 0.475319, -0.146977, -0.0428399), (0.987977, 0.0215577, -0.105866, -0.110589), (-0.0215577, 0.987977, 0.110589, -0.105866), (0.844834, 0.512658, -0.0363881, -0.148706), (-0.512658, 0.844834, 0.148706, -0.0363881), (0.475319, 0.866392, 0.0428399, -0.146977), (-0.866392, 0.475319, 0.146977, 0.0428399), (0.612227, 0.0133588, 0.546689, 0.571079), (-0.0133588, 0.612227, -0.571079, 0.546689), (0.523524, 0.317682, 0.187907, 0.767913), (-0.317682, 0.523524, -0.767913, 0.187907), (0.294544, 0.536883, -0.221224, 0.758986), (-0.536883, 0.294544, -0.758986, -0.221224), (0.676842, 0.0147687, -0.685939, 0.266747), (-0.0147687, 0.676842, -0.266747, -0.685939), (0.578778, 0.351211, -0.727414, -0.11196), (-0.351211, 0.578778, 0.11196, -0.727414), (0.325631, 0.593547, -0.573979, -0.460667), (-0.593547, 0.325631, 0.460667, -0.573979), (0.735805, 0.0160553, 0.630972, -0.245372), (-0.0160553, 0.735805, 0.245372, 0.630972), (0.629198, 0.381807, 0.669124, 0.102988), (-0.381807, 0.629198, -0.102988, 0.669124), (0.353998, 0.645253, 0.527984, 0.423752), (-0.645253, 0.353998, -0.423752, 0.527984), (0.889545, 0.0194099, -0.425401, 0.16543), (-0.0194099, 0.889545, -0.16543, -0.425401), (0.760663, 0.461582, -0.451123, -0.0694345), (-0.461582, 0.760663, 0.0694345, -0.451123), (0.427963, 0.780073, -0.355967, -0.285694), (-0.780073, 0.427963, 0.285694, -0.355967), (0.357132, 0.00779265, 0.888127, -0.289182), (-0.00779265, 0.357132, 0.289182, 0.888127), (0.305389, 0.185315, 0.913732, 0.193624), (-0.185315, 0.305389, -0.193624, 0.913732), (0.171817, 0.313182, 0.694503, 0.624549), (-0.313182, 0.171817, -0.624549, 0.694503), (0.357132, 0.00779265, -0.888127, 0.289182), (-0.00779265, 0.357132, -0.289182, -0.888127), (0.305389, 0.185315, -0.913732, -0.193624), (-0.185315, 0.305389, 0.193624, -0.913732), (0.171817, 0.313182, -0.694503, -0.624549), (-0.313182, 0.171817, 0.624549, -0.694503), (0.889545, 0.0194099, 0.425401, -0.16543), (-0.0194099, 0.889545, 0.16543, 0.425401), (0.760663, 0.461582, 0.451123, 0.0694345), (-0.461582, 0.760663, -0.0694345, 0.451123), (0.427963, 0.780073, 0.355967, 0.285694), (-0.780073, 0.427963, -0.285694, 0.355967), (0.735805, 0.0160553, 0.245372, 0.630972), (-0.0160553, 0.735805, -0.630972, 0.245372), (0.629198, 0.381807, -0.102988, 0.669124), (-0.381807, 0.629198, -0.669124, -0.102988), (0.353998, 0.645253, -0.423752, 0.527984), (-0.645253, 0.353998, -0.527984, -0.423752), (0.676842, 0.0147687, -0.266747, -0.685939), (-0.0147687, 0.676842, 0.685939, -0.266747), (0.578778, 0.351211, 0.11196, -0.727414), (-0.351211, 0.578778, 0.727414, 0.11196), (0.325631, 0.593547, 0.460667, -0.573979), (-0.593547, 0.325631, 0.573979, 0.460667), (0.456327, 0.00995709, -0.322482, -0.82926), (-0.00995709, 0.456327, 0.82926, -0.322482), (0.390212, 0.236787, 0.135353, -0.879401), (-0.236787, 0.390212, 0.879401, 0.135353), (0.21954, 0.400169, 0.556919, -0.693907), (-0.400169, 0.21954, 0.693907, 0.556919), (0.933799, 0.0203756, 0.110598, 0.339665), (-0.0203756, 0.933799, -0.339665, 0.110598), (0.798506, 0.484545, -0.0740518, 0.349457), (-0.484545, 0.798506, -0.349457, -0.0740518), (0.449254, 0.818881, -0.238859, 0.265613), (-0.818881, 0.449254, -0.265613, -0.238859), (0.933799, 0.0203756, -0.110598, -0.339665), (-0.0203756, 0.933799, 0.339665, -0.110598), (0.798506, 0.484545, 0.0740518, -0.349457), (-0.484545, 0.798506, 0.349457, 0.0740518), (0.449254, 0.818881, 0.238859, -0.265613), (-0.818881, 0.449254, 0.265613, 0.238859), (0.456327, 0.00995709, 0.322482, 0.82926), (-0.00995709, 0.456327, -0.82926, 0.322482), (0.390212, 0.236787, -0.135353, 0.879401), (-0.236787, 0.390212, -0.879401, -0.135353), (0.21954, 0.400169, -0.556919, 0.693907), (-0.400169, 0.21954, -0.693907, -0.556919), (0.649957, 0.189445, -0.573979, -0.460667), (-0.189445, 0.649957, 0.460667, -0.573979), (0.468157, 0.489043, -0.266747, -0.685939), (-0.489043, 0.468157, 0.685939, -0.266747), (0.160914, 0.657602, 0.11196, -0.727414), (-0.657602, 0.160914, 0.727414, 0.11196), (0.706578, 0.205949, 0.527984, 0.423752), (-0.205949, 0.706578, -0.423752, 0.527984), (0.50894, 0.531645, 0.245372, 0.630972), (-0.531645, 0.50894, -0.630972, 0.245372), (0.174932, 0.714888, -0.102988, 0.669124), (-0.714888, 0.174932, -0.669124, -0.102988), (0.854211, 0.24898, -0.285694, 0.355967), (-0.24898, 0.854211, -0.355967, -0.285694), (0.615278, 0.642728, -0.425401, 0.16543), (-0.642728, 0.615278, -0.16543, -0.425401), (0.211483, 0.864258, -0.451123, -0.0694345), (-0.864258, 0.211483, 0.0694345, -0.451123), (0.342946, 0.0999598, 0.542863, -0.760063), (-0.0999598, 0.342946, 0.760063, 0.542863), (0.24702, 0.258041, 0.850165, -0.386802), (-0.258041, 0.24702, 0.386802, 0.850165), (0.0849055, 0.34698, 0.929665, 0.0901015), (-0.34698, 0.0849055, -0.0901015, 0.929665), (0.342946, 0.0999598, -0.542863, 0.760063), (-0.0999598, 0.342946, -0.760063, -0.542863), (0.24702, 0.258041, -0.850165, 0.386802), (-0.258041, 0.24702, -0.386802, -0.850165), (0.0849055, 0.34698, -0.929665, -0.0901015), (-0.34698, 0.0849055, 0.0901015, -0.929665), (0.854211, 0.24898, 0.285694, -0.355967), (-0.24898, 0.854211, 0.355967, 0.285694), (0.615278, 0.642728, 0.425401, -0.16543), (-0.642728, 0.615278, 0.16543, 0.425401), (0.211483, 0.864258, 0.451123, 0.0694345), (-0.864258, 0.211483, -0.0694345, 0.451123), (0.706578, 0.205949, -0.423752, 0.527984), (-0.205949, 0.706578, -0.527984, -0.423752), (0.50894, 0.531645, -0.630972, 0.245372), (-0.531645, 0.50894, -0.245372, -0.630972), (0.174932, 0.714888, -0.669124, -0.102988), (-0.714888, 0.174932, 0.102988, -0.669124), (0.649957, 0.189445, 0.460667, -0.573979), (-0.189445, 0.649957, 0.573979, 0.460667), (0.468157, 0.489043, 0.685939, -0.266747), (-0.489043, 0.468157, 0.266747, 0.685939), (0.160914, 0.657602, 0.727414, 0.11196), (-0.657602, 0.160914, -0.11196, 0.727414), (0.438201, 0.127724, -0.693907, -0.556919), (-0.127724, 0.438201, 0.556919, -0.693907), (0.315631, 0.329713, -0.322482, -0.82926), (-0.329713, 0.315631, 0.82926, -0.322482), (0.108488, 0.443355, 0.135353, -0.879401), (-0.443355, 0.108488, 0.879401, 0.135353), (0.896707, 0.261366, 0.290687, 0.207618), (-0.261366, 0.896707, -0.207618, 0.290687), (0.645888, 0.674703, 0.147933, 0.325146), (-0.674703, 0.645888, -0.325146, 0.147933), (0.222004, 0.907254, -0.0344594, 0.355551), (-0.907254, 0.222004, -0.355551, -0.0344594), (0.896707, 0.261366, -0.290687, -0.207618), (-0.261366, 0.896707, 0.207618, -0.290687), (0.645888, 0.674703, -0.147933, -0.325146), (-0.674703, 0.645888, 0.325146, -0.147933), (0.222004, 0.907254, 0.0344594, -0.355551), (-0.907254, 0.222004, 0.355551, 0.0344594), (0.438201, 0.127724, 0.693907, 0.556919), (-0.127724, 0.438201, -0.556919, 0.693907), (0.315631, 0.329713, 0.322482, 0.82926), (-0.329713, 0.315631, -0.82926, 0.322482), (0.108488, 0.443355, -0.135353, 0.879401), (-0.443355, 0.108488, -0.879401, -0.135353), (0.649957, 0.189445, -0.731606, 0.0801239), (-0.189445, 0.649957, -0.0801239, -0.731606), (0.468157, 0.489043, -0.673651, -0.296414), (-0.489043, 0.468157, 0.296414, -0.673651), (0.160914, 0.657602, -0.435192, -0.593527), (-0.657602, 0.160914, 0.593527, -0.435192), (0.706578, 0.205949, 0.672979, -0.0737032), (-0.205949, 0.706578, 0.0737032, 0.672979), (0.50894, 0.531645, 0.619669, 0.272661), (-0.531645, 0.50894, -0.272661, 0.619669), (0.174932, 0.714888, 0.400319, 0.545966), (-0.714888, 0.174932, -0.545966, 0.400319), (0.854211, 0.24898, -0.453723, 0.0496907), (-0.24898, 0.854211, -0.0496907, -0.453723), (0.615278, 0.642728, -0.417781, -0.183828), (-0.642728, 0.615278, 0.183828, -0.417781), (0.211483, 0.864258, -0.269895, -0.36809), (-0.864258, 0.211483, 0.36809, -0.269895), (0.342946, 0.0999598, 0.932711, -0.0494643), (-0.0999598, 0.342946, 0.0494643, 0.932711), (0.24702, 0.258041, 0.832483, 0.423518), (-0.258041, 0.24702, -0.423518, 0.832483), (0.0849055, 0.34698, 0.509193, 0.783019), (-0.34698, 0.0849055, -0.783019, 0.509193), (0.342946, 0.0999598, -0.932711, 0.0494643), (-0.0999598, 0.342946, -0.0494643, -0.932711), (0.24702, 0.258041, -0.832483, -0.423518), (-0.258041, 0.24702, 0.423518, -0.832483), (0.0849055, 0.34698, -0.509193, -0.783019), (-0.34698, 0.0849055, 0.783019, -0.509193), (0.854211, 0.24898, 0.453723, -0.0496907), (-0.24898, 0.854211, 0.0496907, 0.453723), (0.615278, 0.642728, 0.417781, 0.183828), (-0.642728, 0.615278, -0.183828, 0.417781), (0.211483, 0.864258, 0.269895, 0.36809), (-0.864258, 0.211483, -0.36809, 0.269895), (0.706578, 0.205949, 0.0737032, 0.672979), (-0.205949, 0.706578, -0.672979, 0.0737032), (0.50894, 0.531645, -0.272661, 0.619669), (-0.531645, 0.50894, -0.619669, -0.272661), (0.174932, 0.714888, -0.545966, 0.400319), (-0.714888, 0.174932, -0.400319, -0.545966), (0.649957, 0.189445, -0.0801239, -0.731606), (-0.189445, 0.649957, 0.731606, -0.0801239), (0.468157, 0.489043, 0.296414, -0.673651), (-0.489043, 0.468157, 0.673651, 0.296414), (0.160914, 0.657602, 0.593527, -0.435192), (-0.657602, 0.160914, 0.435192, 0.593527), (0.438201, 0.127724, -0.096865, -0.884468), (-0.127724, 0.438201, 0.884468, -0.096865), (0.315631, 0.329713, 0.358346, -0.814404), (-0.329713, 0.315631, 0.814404, 0.358346), (0.108488, 0.443355, 0.717539, -0.526122), (-0.443355, 0.108488, 0.526122, 0.717539), (0.896707, 0.261366, 0.0189177, 0.356716), (-0.261366, 0.896707, -0.356716, 0.0189177), (0.645888, 0.674703, -0.161975, 0.318384), (-0.674703, 0.645888, -0.318384, -0.161975), (0.222004, 0.907254, -0.299466, 0.194741), (-0.907254, 0.222004, -0.194741, -0.299466), (0.896707, 0.261366, -0.0189177, -0.356716), (-0.261366, 0.896707, 0.356716, -0.0189177), (0.645888, 0.674703, 0.161975, -0.318384), (-0.674703, 0.645888, 0.318384, 0.161975), (0.222004, 0.907254, 0.299466, -0.194741), (-0.907254, 0.222004, 0.194741, 0.299466), (0.438201, 0.127724, 0.096865, 0.884468), (-0.127724, 0.438201, -0.884468, 0.096865), (0.315631, 0.329713, -0.358346, 0.814404), (-0.329713, 0.315631, -0.814404, -0.358346), (0.108488, 0.443355, -0.717539, 0.526122), (-0.443355, 0.108488, -0.526122, -0.717539), (0.676842, 0.0147687, -0.673651, -0.296414), (-0.0147687, 0.676842, 0.296414, -0.673651), (0.578778, 0.351211, -0.435192, -0.593527), (-0.351211, 0.578778, 0.593527, -0.435192), (0.325631, 0.593547, -0.0801239, -0.731606), (-0.593547, 0.325631, 0.731606, -0.0801239), (0.735805, 0.0160553, 0.619669, 0.272661), (-0.0160553, 0.735805, -0.272661, 0.619669), (0.629198, 0.381807, 0.400319, 0.545966), (-0.381807, 0.629198, -0.545966, 0.400319), (0.353998, 0.645253, 0.0737032, 0.672979), (-0.645253, 0.353998, -0.672979, 0.0737032), (0.889545, 0.0194099, -0.183828, 0.417781), (-0.0194099, 0.889545, -0.417781, -0.183828), (0.760663, 0.461582, -0.36809, 0.269895), (-0.461582, 0.760663, -0.269895, -0.36809), (0.427963, 0.780073, -0.453723, 0.0496907), (-0.780073, 0.427963, -0.0496907, -0.453723), (0.357132, 0.00779265, 0.327646, -0.874668), (-0.00779265, 0.357132, 0.874668, 0.327646), (0.305389, 0.185315, 0.721084, -0.593661), (-0.185315, 0.305389, 0.593661, 0.721084), (0.171817, 0.313182, 0.921308, -0.153584), (-0.313182, 0.171817, 0.153584, 0.921308), (0.357132, 0.00779265, -0.327646, 0.874668), (-0.00779265, 0.357132, -0.874668, -0.327646), (0.305389, 0.185315, -0.721084, 0.593661), (-0.185315, 0.305389, -0.593661, -0.721084), (0.171817, 0.313182, -0.921308, 0.153584), (-0.313182, 0.171817, -0.153584, -0.921308), (0.889545, 0.0194099, 0.183828, -0.417781), (-0.0194099, 0.889545, 0.417781, 0.183828), (0.760663, 0.461582, 0.36809, -0.269895), (-0.461582, 0.760663, 0.269895, 0.36809), (0.427963, 0.780073, 0.453723, -0.0496907), (-0.780073, 0.427963, 0.0496907, 0.453723), (0.735805, 0.0160553, -0.272661, 0.619669), (-0.0160553, 0.735805, -0.619669, -0.272661), (0.629198, 0.381807, -0.545966, 0.400319), (-0.381807, 0.629198, -0.400319, -0.545966), (0.353998, 0.645253, -0.672979, 0.0737032), (-0.645253, 0.353998, -0.0737032, -0.672979), (0.676842, 0.0147687, 0.296414, -0.673651), (-0.0147687, 0.676842, 0.673651, 0.296414), (0.578778, 0.351211, 0.593527, -0.435192), (-0.351211, 0.578778, 0.435192, 0.593527), (0.325631, 0.593547, 0.731606, -0.0801239), (-0.593547, 0.325631, 0.0801239, 0.731606), (0.456327, 0.00995709, -0.814404, -0.358346), (-0.00995709, 0.456327, 0.358346, -0.814404), (0.390212, 0.236787, -0.526122, -0.717539), (-0.236787, 0.390212, 0.717539, -0.526122), (0.21954, 0.400169, -0.096865, -0.884468), (-0.400169, 0.21954, 0.884468, -0.096865), (0.933799, 0.0203756, 0.334517, 0.125309), (-0.0203756, 0.933799, -0.125309, 0.334517), (0.798506, 0.484545, 0.227046, 0.275779), (-0.484545, 0.798506, -0.275779, 0.227046), (0.449254, 0.818881, 0.0587382, 0.352355), (-0.818881, 0.449254, -0.352355, 0.0587382), (0.933799, 0.0203756, -0.334517, -0.125309), (-0.0203756, 0.933799, 0.125309, -0.334517), (0.798506, 0.484545, -0.227046, -0.275779), (-0.484545, 0.798506, 0.275779, -0.227046), (0.449254, 0.818881, -0.0587382, -0.352355), (-0.818881, 0.449254, 0.352355, -0.0587382), (0.456327, 0.00995709, 0.814404, 0.358346), (-0.00995709, 0.456327, -0.358346, 0.814404), (0.390212, 0.236787, 0.526122, 0.717539), (-0.236787, 0.390212, -0.717539, 0.526122), (0.21954, 0.400169, 0.096865, 0.884468), (-0.400169, 0.21954, -0.884468, 0.096865), (0.841425, 0.01836, -0.531857, 0.0937807), (-0.01836, 0.841425, -0.0937807, -0.531857), (0.719516, 0.436613, -0.507492, -0.184712), (-0.436613, 0.719516, 0.184712, -0.507492), (0.404812, 0.737876, -0.347145, -0.413711), (-0.737876, 0.404812, 0.413711, -0.347145), (0.539933, 0.0117814, 0.828839, -0.146147), (-0.0117814, 0.539933, 0.146147, 0.828839), (0.461705, 0.28017, 0.790869, 0.287853), (-0.28017, 0.461705, -0.287853, 0.790869), (0.259764, 0.473487, 0.540986, 0.644722), (-0.473487, 0.259764, -0.644722, 0.540986), (0.987977, 0.0215577, -0.148706, 0.0363881), (-0.0215577, 0.987977, -0.0363881, -0.148706), (0.844834, 0.512658, -0.146977, -0.0428399), (-0.512658, 0.844834, 0.0428399, -0.146977), (0.475319, 0.866392, -0.105866, -0.110589), (-0.866392, 0.475319, 0.110589, -0.105866), (0.612227, 0.0133588, 0.666759, -0.424773), (-0.0133588, 0.612227, 0.424773, 0.666759), (0.523524, 0.317682, 0.789817, -0.0344842), (-0.317682, 0.523524, 0.0344842, 0.789817), (0.294544, 0.536883, 0.701244, 0.365044), (-0.536883, 0.294544, -0.365044, 0.701244), (0.612227, 0.0133588, -0.666759, 0.424773), (-0.0133588, 0.612227, -0.424773, -0.666759), (0.523524, 0.317682, -0.789817, 0.0344842), (-0.317682, 0.523524, -0.0344842, -0.789817), (0.294544, 0.536883, -0.701244, -0.365044), (-0.536883, 0.294544, 0.365044, -0.701244), (0.987977, 0.0215577, 0.148706, -0.0363881), (-0.0215577, 0.987977, 0.0363881, 0.148706), (0.844834, 0.512658, 0.146977, 0.0428399), (-0.512658, 0.844834, -0.0428399, 0.146977), (0.475319, 0.866392, 0.105866, 0.110589), (-0.866392, 0.475319, -0.110589, 0.105866), (0.539933, 0.0117814, 0.146147, 0.828839), (-0.0117814, 0.539933, -0.828839, 0.146147), (0.461705, 0.28017, -0.287853, 0.790869), (-0.28017, 0.461705, -0.790869, -0.287853), (0.259764, 0.473487, -0.644722, 0.540986), (-0.473487, 0.259764, -0.540986, -0.644722), (0.841425, 0.01836, -0.0937807, -0.531857), (-0.01836, 0.841425, 0.531857, -0.0937807), (0.719516, 0.436613, 0.184712, -0.507492), (-0.436613, 0.719516, 0.507492, 0.184712), (0.404812, 0.737876, 0.413711, -0.347145), (-0.737876, 0.404812, 0.347145, 0.413711), (0.153057, 0.00333971, -0.234884, -0.959892), (-0.00333971, 0.153057, 0.959892, -0.234884), (0.130881, 0.0794206, 0.27653, -0.948733), (-0.0794206, 0.130881, 0.948733, 0.27653), (0.0736361, 0.134221, 0.713849, -0.683361), (-0.134221, 0.0736361, 0.683361, 0.713849), (0.790381, 0.0172462, 0.329027, 0.51647), (-0.0172462, 0.790381, -0.51647, 0.329027), (0.675867, 0.410126, 0.0267113, 0.61179), (-0.410126, 0.675867, -0.61179, 0.0267113), (0.380255, 0.693113, -0.282762, 0.543181), (-0.693113, 0.380255, -0.543181, -0.282762), (0.790381, 0.0172462, -0.329027, -0.51647), (-0.0172462, 0.790381, 0.51647, -0.329027), (0.675867, 0.410126, -0.0267113, -0.61179), (-0.410126, 0.675867, 0.61179, -0.0267113), (0.380255, 0.693113, 0.282762, -0.543181), (-0.693113, 0.380255, 0.543181, 0.282762), (0.153057, 0.00333971, 0.234884, 0.959892), (-0.00333971, 0.153057, -0.959892, 0.234884), (0.130881, 0.0794206, -0.27653, 0.948733), (-0.0794206, 0.130881, -0.948733, -0.27653), (0.0736361, 0.134221, -0.713849, 0.683361), (-0.134221, 0.0736361, -0.683361, -0.713849), (0.587908, 0.17136, -0.787561, -0.0689027), (-0.17136, 0.587908, 0.0689027, -0.787561), (0.423464, 0.442356, -0.647597, -0.453452), (-0.442356, 0.423464, 0.453452, -0.647597), (0.145552, 0.594823, -0.334109, -0.716499), (-0.594823, 0.145552, 0.716499, -0.334109), (0.758986, 0.221224, 0.610042, 0.0533718), (-0.221224, 0.758986, -0.0533718, 0.610042), (0.546689, 0.571079, 0.501626, 0.351242), (-0.571079, 0.546689, -0.351242, 0.501626), (0.187907, 0.767913, 0.2588, 0.554998), (-0.767913, 0.187907, -0.554998, 0.2588), (0.808002, 0.235511, -0.515066, 0.1624), (-0.235511, 0.808002, -0.1624, -0.515066), (0.581995, 0.60796, -0.52726, -0.116891), (-0.60796, 0.581995, 0.116891, -0.52726), (0.200042, 0.817506, -0.398175, -0.36486), (-0.817506, 0.200042, 0.36486, -0.398175), (0.244962, 0.0713998, 0.949931, -0.18035), (-0.0713998, 0.244962, 0.18035, 0.949931), (0.176443, 0.184315, 0.91284, 0.318778), (-0.184315, 0.176443, -0.318778, 0.91284), (0.0606468, 0.247843, 0.631153, 0.73249), (-0.247843, 0.0606468, -0.73249, 0.631153), (0.244962, 0.0713998, -0.949931, 0.18035), (-0.0713998, 0.244962, -0.18035, -0.949931), (0.176443, 0.184315, -0.91284, -0.318778), (-0.184315, 0.176443, 0.318778, -0.91284), (0.0606468, 0.247843, -0.631153, -0.73249), (-0.247843, 0.0606468, 0.73249, -0.631153), (0.808002, 0.235511, 0.515066, -0.1624), (-0.235511, 0.808002, 0.1624, 0.515066), (0.581995, 0.60796, 0.52726, 0.116891), (-0.60796, 0.581995, -0.116891, 0.52726), (0.200042, 0.817506, 0.398175, 0.36486), (-0.817506, 0.200042, -0.36486, 0.398175), (0.758986, 0.221224, -0.0533718, 0.610042), (-0.221224, 0.758986, -0.610042, -0.0533718), (0.546689, 0.571079, -0.351242, 0.501626), (-0.571079, 0.546689, -0.501626, -0.351242), (0.187907, 0.767913, -0.554998, 0.2588), (-0.767913, 0.187907, -0.2588, -0.554998), (0.587908, 0.17136, 0.0689027, -0.787561), (-0.17136, 0.587908, 0.787561, 0.0689027), (0.423464, 0.442356, 0.453452, -0.647597), (-0.442356, 0.423464, 0.647597, 0.453452), (0.145552, 0.594823, 0.716499, -0.334109), (-0.594823, 0.145552, 0.334109, 0.716499), (0.518486, 0.151125, -0.253082, -0.802672), (-0.151125, 0.518486, 0.802672, -0.253082), (0.37346, 0.390121, 0.182161, -0.821676), (-0.390121, 0.37346, 0.821676, 0.182161), (0.128365, 0.524585, 0.568594, -0.620511), (-0.524585, 0.128365, 0.620511, 0.568594), (0.928272, 0.270567, 0.0475926, 0.250677), (-0.270567, 0.928272, -0.250677, 0.0475926), (0.668624, 0.698454, -0.0841223, 0.240889), (-0.698454, 0.668624, -0.240889, -0.0841223), (0.229819, 0.939191, -0.193297, 0.166555), (-0.939191, 0.229819, -0.166555, -0.193297), (0.928272, 0.270567, -0.0475926, -0.250677), (-0.270567, 0.928272, 0.250677, -0.0475926), (0.668624, 0.698454, 0.0841223, -0.240889), (-0.698454, 0.668624, 0.240889, 0.0841223), (0.229819, 0.939191, 0.193297, -0.166555), (-0.939191, 0.229819, 0.166555, 0.193297), (0.518486, 0.151125, 0.253082, 0.802672), (-0.151125, 0.518486, -0.802672, 0.253082), (0.37346, 0.390121, -0.182161, 0.821676), (-0.390121, 0.37346, -0.821676, -0.182161), (0.128365, 0.524585, -0.568594, 0.620511), (-0.524585, 0.128365, -0.620511, -0.568594), (0.808002, 0.235511, -0.538007, -0.0470695), (-0.235511, 0.808002, 0.0470695, -0.538007), (0.581995, 0.60796, -0.442393, -0.309767), (-0.60796, 0.581995, 0.309767, -0.442393), (0.200042, 0.817506, -0.22824, -0.489462), (-0.817506, 0.200042, 0.489462, -0.22824), (0.518486, 0.151125, 0.838423, 0.0733525), (-0.151125, 0.518486, -0.0733525, 0.838423), (0.37346, 0.390121, 0.689419, 0.482737), (-0.390121, 0.37346, -0.482737, 0.689419), (0.128365, 0.524585, 0.355686, 0.762772), (-0.524585, 0.128365, -0.762772, 0.355686), (0.948733, 0.27653, -0.153057, -0.00333971), (-0.27653, 0.948733, 0.00333971, -0.153057), (0.683361, 0.713849, -0.130881, -0.0794206), (-0.713849, 0.683361, 0.0794206, -0.130881), (0.234884, 0.959892, -0.0736361, -0.134221), (-0.959892, 0.234884, 0.134221, -0.0736361), (0.587908, 0.17136, 0.753979, -0.237729), (-0.17136, 0.587908, 0.237729, 0.753979), (0.423464, 0.442356, 0.77183, 0.171111), (-0.442356, 0.423464, -0.171111, 0.77183), (0.145552, 0.594823, 0.582869, 0.534101), (-0.594823, 0.145552, -0.534101, 0.582869), (0.587908, 0.17136, -0.753979, 0.237729), (-0.17136, 0.587908, -0.237729, -0.753979), (0.423464, 0.442356, -0.77183, -0.171111), (-0.442356, 0.423464, 0.171111, -0.77183), (0.145552, 0.594823, -0.582869, -0.534101), (-0.594823, 0.145552, 0.534101, -0.582869), (0.948733, 0.27653, 0.153057, 0.00333971), (-0.27653, 0.948733, -0.00333971, 0.153057), (0.683361, 0.713849, 0.130881, 0.0794206), (-0.713849, 0.683361, -0.0794206, 0.130881), (0.234884, 0.959892, 0.0736361, 0.134221), (-0.959892, 0.234884, -0.134221, 0.0736361), (0.518486, 0.151125, -0.0733525, 0.838423), (-0.151125, 0.518486, -0.838423, -0.0733525), (0.37346, 0.390121, -0.482737, 0.689419), (-0.390121, 0.37346, -0.689419, -0.482737), (0.128365, 0.524585, -0.762772, 0.355686), (-0.524585, 0.128365, -0.355686, -0.762772), (0.808002, 0.235511, 0.0470695, -0.538007), (-0.235511, 0.808002, 0.538007, 0.0470695), (0.581995, 0.60796, 0.309767, -0.442393), (-0.60796, 0.581995, 0.442393, 0.309767), (0.200042, 0.817506, 0.489462, -0.22824), (-0.817506, 0.200042, 0.22824, 0.489462), (0.146977, 0.0428399, 0.0215577, -0.987977), (-0.0428399, 0.146977, 0.987977, 0.0215577), (0.105866, 0.110589, 0.512658, -0.844834), (-0.110589, 0.105866, 0.844834, 0.512658), (0.0363881, 0.148706, 0.866392, -0.475319), (-0.148706, 0.0363881, 0.475319, 0.866392), (0.758986, 0.221224, 0.184144, 0.58403), (-0.221224, 0.758986, -0.58403, 0.184144), (0.546689, 0.571079, -0.132542, 0.597857), (-0.571079, 0.546689, -0.597857, -0.132542), (0.187907, 0.767913, -0.413713, 0.451488), (-0.767913, 0.187907, -0.451488, -0.413713), (0.758986, 0.221224, -0.184144, -0.58403), (-0.221224, 0.758986, 0.58403, -0.184144), (0.546689, 0.571079, 0.132542, -0.597857), (-0.571079, 0.546689, 0.597857, 0.132542), (0.187907, 0.767913, 0.413713, -0.451488), (-0.767913, 0.187907, 0.451488, 0.413713), (0.146977, 0.0428399, -0.0215577, 0.987977), (-0.0428399, 0.146977, -0.987977, -0.0215577), (0.105866, 0.110589, -0.512658, 0.844834), (-0.110589, 0.105866, -0.844834, -0.512658), (0.0363881, 0.148706, -0.866392, 0.475319), (-0.148706, 0.0363881, -0.475319, -0.866392), (0.612227, 0.0133588, -0.778559, 0.137281), (-0.0133588, 0.612227, -0.137281, -0.778559), (0.523524, 0.317682, -0.742892, -0.270391), (-0.317682, 0.523524, 0.270391, -0.742892), (0.294544, 0.536883, -0.508168, -0.605611), (-0.536883, 0.294544, 0.605611, -0.508168), (0.790381, 0.0172462, 0.603069, -0.106337), (-0.0172462, 0.790381, 0.106337, 0.603069), (0.675867, 0.410126, 0.575442, 0.209444), (-0.410126, 0.675867, -0.209444, 0.575442), (0.380255, 0.693113, 0.393625, 0.469105), (-0.693113, 0.380255, -0.469105, 0.393625), (0.841425, 0.01836, -0.455483, 0.290175), (-0.01836, 0.841425, -0.290175, -0.455483), (0.719516, 0.436613, -0.539548, 0.0235572), (-0.436613, 0.719516, -0.0235572, -0.539548), (0.404812, 0.737876, -0.479041, -0.249373), (-0.737876, 0.404812, 0.249373, -0.479041), (0.255094, 0.00556618, 0.870885, -0.420065), (-0.00556618, 0.255094, 0.420065, 0.870885), (0.218135, 0.132368, 0.964241, 0.0716555), (-0.132368, 0.218135, -0.0716555, 0.964241), (0.122727, 0.223701, 0.79923, 0.544176), (-0.223701, 0.122727, -0.544176, 0.79923), (0.255094, 0.00556618, -0.870885, 0.420065), (-0.00556618, 0.255094, -0.420065, -0.870885), (0.218135, 0.132368, -0.964241, -0.0716555), (-0.132368, 0.218135, 0.0716555, -0.964241), (0.122727, 0.223701, -0.79923, -0.544176), (-0.223701, 0.122727, 0.544176, -0.79923), (0.841425, 0.01836, 0.455483, -0.290175), (-0.01836, 0.841425, 0.290175, 0.455483), (0.719516, 0.436613, 0.539548, -0.0235572), (-0.436613, 0.719516, 0.0235572, 0.539548), (0.404812, 0.737876, 0.479041, 0.249373), (-0.737876, 0.404812, -0.249373, 0.479041), (0.790381, 0.0172462, 0.106337, 0.603069), (-0.0172462, 0.790381, -0.603069, 0.106337), (0.675867, 0.410126, -0.209444, 0.575442), (-0.410126, 0.675867, -0.575442, -0.209444), (0.380255, 0.693113, -0.469105, 0.393625), (-0.693113, 0.380255, -0.393625, -0.469105), (0.612227, 0.0133588, -0.137281, -0.778559), (-0.0133588, 0.612227, 0.778559, -0.137281), (0.523524, 0.317682, 0.270391, -0.742892), (-0.317682, 0.523524, 0.742892, 0.270391), (0.294544, 0.536883, 0.605611, -0.508168), (-0.536883, 0.294544, 0.508168, 0.605611), (0.539933, 0.0117814, -0.452205, -0.70982), (-0.0117814, 0.539933, 0.70982, -0.452205), (0.461705, 0.28017, -0.0367112, -0.840824), (-0.28017, 0.461705, 0.840824, -0.0367112), (0.259764, 0.473487, 0.388619, -0.746531), (-0.473487, 0.259764, 0.746531, 0.388619), (0.96667, 0.0210928, 0.110851, 0.229818), (-0.0210928, 0.96667, -0.229818, 0.110851), (0.826614, 0.501602, -0.0189092, 0.254454), (-0.501602, 0.826614, -0.254454, -0.0189092), (0.465068, 0.847707, -0.143603, 0.210909), (-0.847707, 0.465068, -0.210909, -0.143603), (0.96667, 0.0210928, -0.110851, -0.229818), (-0.0210928, 0.96667, 0.229818, -0.110851), (0.826614, 0.501602, 0.0189092, -0.254454), (-0.501602, 0.826614, 0.254454, 0.0189092), (0.465068, 0.847707, 0.143603, -0.210909), (-0.847707, 0.465068, 0.210909, 0.143603), (0.539933, 0.0117814, 0.452205, 0.70982), (-0.0117814, 0.539933, -0.70982, 0.452205), (0.461705, 0.28017, 0.0367112, 0.840824), (-0.28017, 0.461705, -0.840824, 0.0367112), (0.259764, 0.473487, -0.388619, 0.746531), (-0.473487, 0.259764, -0.746531, -0.388619), (0.735805, 0.0160553, -0.570979, 0.363754), (-0.0160553, 0.735805, -0.363754, -0.570979), (0.629198, 0.381807, -0.676359, 0.0295305), (-0.381807, 0.629198, -0.0295305, -0.676359), (0.353998, 0.645253, -0.600509, -0.312605), (-0.645253, 0.353998, 0.312605, -0.600509), (0.676842, 0.0147687, 0.620719, -0.395442), (-0.0147687, 0.676842, 0.395442, 0.620719), (0.578778, 0.351211, 0.73528, -0.032103), (-0.351211, 0.578778, 0.032103, 0.73528), (0.325631, 0.593547, 0.652822, 0.339838), (-0.593547, 0.325631, -0.339838, 0.652822), (0.933799, 0.0203756, -0.355759, 0.0322425), (-0.0203756, 0.933799, -0.0322425, -0.355759), (0.798506, 0.484545, -0.324218, -0.149957), (-0.484545, 0.798506, 0.149957, -0.324218), (0.449254, 0.818881, -0.205802, -0.291975), (-0.818881, 0.449254, 0.291975, -0.205802), (0.456327, 0.00995709, 0.876239, -0.154505), (-0.00995709, 0.456327, 0.154505, 0.876239), (0.390212, 0.236787, 0.836098, 0.304315), (-0.236787, 0.390212, -0.304315, 0.836098), (0.21954, 0.400169, 0.571924, 0.681593), (-0.400169, 0.21954, -0.681593, 0.571924), (0.456327, 0.00995709, -0.876239, 0.154505), (-0.00995709, 0.456327, -0.154505, -0.876239), (0.390212, 0.236787, -0.836098, -0.304315), (-0.236787, 0.390212, 0.304315, -0.836098), (0.21954, 0.400169, -0.571924, -0.681593), (-0.400169, 0.21954, 0.681593, -0.571924), (0.933799, 0.0203756, 0.355759, -0.0322425), (-0.0203756, 0.933799, 0.0322425, 0.355759), (0.798506, 0.484545, 0.324218, 0.149957), (-0.484545, 0.798506, -0.149957, 0.324218), (0.449254, 0.818881, 0.205802, 0.291975), (-0.818881, 0.449254, -0.291975, 0.205802), (0.676842, 0.0147687, 0.395442, 0.620719), (-0.0147687, 0.676842, -0.620719, 0.395442), (0.578778, 0.351211, 0.032103, 0.73528), (-0.351211, 0.578778, -0.73528, 0.032103), (0.325631, 0.593547, -0.339838, 0.652822), (-0.593547, 0.325631, -0.652822, -0.339838), (0.735805, 0.0160553, -0.363754, -0.570979), (-0.0160553, 0.735805, 0.570979, -0.363754), (0.629198, 0.381807, -0.0295305, -0.676359), (-0.381807, 0.629198, 0.676359, -0.0295305), (0.353998, 0.645253, 0.312605, -0.600509), (-0.645253, 0.353998, 0.600509, 0.312605), (0.357132, 0.00779265, -0.0843049, -0.930209), (-0.00779265, 0.357132, 0.930209, -0.0843049), (0.305389, 0.185315, 0.392094, -0.847737), (-0.185315, 0.305389, 0.847737, 0.392094), (0.171817, 0.313182, 0.763432, -0.538115), (-0.313182, 0.171817, 0.538115, 0.763432), (0.889545, 0.0194099, 0.0792592, 0.449501), (-0.0194099, 0.889545, -0.449501, 0.0792592), (0.760663, 0.461582, -0.15611, 0.428909), (-0.461582, 0.760663, -0.428909, -0.15611), (0.427963, 0.780073, -0.34965, 0.293391), (-0.780073, 0.427963, -0.293391, -0.34965), (0.889545, 0.0194099, -0.0792592, -0.449501), (-0.0194099, 0.889545, 0.449501, -0.0792592), (0.760663, 0.461582, 0.15611, -0.428909), (-0.461582, 0.760663, 0.428909, 0.15611), (0.427963, 0.780073, 0.34965, -0.293391), (-0.780073, 0.427963, 0.293391, 0.34965), (0.357132, 0.00779265, 0.0843049, 0.930209), (-0.00779265, 0.357132, -0.930209, 0.0843049), (0.305389, 0.185315, -0.392094, 0.847737), (-0.185315, 0.305389, -0.847737, -0.392094), (0.171817, 0.313182, -0.763432, 0.538115), (-0.313182, 0.171817, -0.538115, -0.763432), (0.706578, 0.205949, -0.600509, -0.312605), (-0.205949, 0.706578, 0.312605, -0.600509), (0.50894, 0.531645, -0.363754, -0.570979), (-0.531645, 0.50894, 0.570979, -0.363754), (0.174932, 0.714888, -0.0295305, -0.676359), (-0.714888, 0.174932, 0.676359, -0.0295305), (0.649957, 0.189445, 0.652822, 0.339838), (-0.189445, 0.649957, -0.339838, 0.652822), (0.468157, 0.489043, 0.395442, 0.620719), (-0.489043, 0.468157, -0.620719, 0.395442), (0.160914, 0.657602, 0.032103, 0.73528), (-0.657602, 0.160914, -0.73528, 0.032103), (0.896707, 0.261366, -0.267097, 0.237199), (-0.261366, 0.896707, -0.237199, -0.267097), (0.645888, 0.674703, -0.349912, 0.0718721), (-0.674703, 0.645888, -0.0718721, -0.349912), (0.222004, 0.907254, -0.338969, -0.112713), (-0.907254, 0.222004, 0.112713, -0.338969), (0.438201, 0.127724, 0.681593, -0.571924), (-0.127724, 0.438201, 0.571924, 0.681593), (0.315631, 0.329713, 0.876239, -0.154505), (-0.329713, 0.315631, 0.154505, 0.876239), (0.108488, 0.443355, 0.836098, 0.304315), (-0.443355, 0.108488, -0.304315, 0.836098), (0.438201, 0.127724, -0.681593, 0.571924), (-0.127724, 0.438201, -0.571924, -0.681593), (0.315631, 0.329713, -0.876239, 0.154505), (-0.329713, 0.315631, -0.154505, -0.876239), (0.108488, 0.443355, -0.836098, -0.304315), (-0.443355, 0.108488, 0.304315, -0.836098), (0.896707, 0.261366, 0.267097, -0.237199), (-0.261366, 0.896707, 0.237199, 0.267097), (0.645888, 0.674703, 0.349912, -0.0718721), (-0.674703, 0.645888, 0.0718721, 0.349912), (0.222004, 0.907254, 0.338969, 0.112713), (-0.907254, 0.222004, -0.112713, 0.338969), (0.649957, 0.189445, -0.339838, 0.652822), (-0.189445, 0.649957, -0.652822, -0.339838), (0.468157, 0.489043, -0.620719, 0.395442), (-0.489043, 0.468157, -0.395442, -0.620719), (0.160914, 0.657602, -0.73528, 0.032103), (-0.657602, 0.160914, -0.032103, -0.73528), (0.706578, 0.205949, 0.312605, -0.600509), (-0.205949, 0.706578, 0.600509, 0.312605), (0.50894, 0.531645, 0.570979, -0.363754), (-0.531645, 0.50894, 0.363754, 0.570979), (0.174932, 0.714888, 0.676359, -0.0295305), (-0.714888, 0.174932, 0.0295305, 0.676359), (0.342946, 0.0999598, -0.620208, -0.698382), (-0.0999598, 0.342946, 0.698382, -0.620208), (0.24702, 0.258041, -0.187925, -0.914921), (-0.258041, 0.24702, 0.914921, -0.187925), (0.0849055, 0.34698, 0.294712, -0.886307), (-0.34698, 0.0849055, 0.886307, 0.294712), (0.854211, 0.24898, 0.293391, 0.34965), (-0.24898, 0.854211, -0.34965, 0.293391), (0.615278, 0.642728, 0.0792592, 0.449501), (-0.642728, 0.615278, -0.449501, 0.0792592), (0.211483, 0.864258, -0.15611, 0.428909), (-0.864258, 0.211483, -0.428909, -0.15611), (0.854211, 0.24898, -0.293391, -0.34965), (-0.24898, 0.854211, 0.34965, -0.293391), (0.615278, 0.642728, -0.0792592, -0.449501), (-0.642728, 0.615278, 0.449501, -0.0792592), (0.211483, 0.864258, 0.15611, -0.428909), (-0.864258, 0.211483, 0.428909, 0.15611), (0.342946, 0.0999598, 0.620208, 0.698382), (-0.0999598, 0.342946, -0.698382, 0.620208), (0.24702, 0.258041, 0.187925, 0.914921), (-0.258041, 0.24702, -0.914921, 0.187925), (0.0849055, 0.34698, -0.294712, 0.886307), (-0.34698, 0.0849055, -0.886307, -0.294712), (0.706578, 0.205949, -0.645669, 0.203579), (-0.205949, 0.706578, -0.203579, -0.645669), (0.50894, 0.531645, -0.660956, -0.14653), (-0.531645, 0.50894, 0.14653, -0.660956), (0.174932, 0.714888, -0.499139, -0.457377), (-0.714888, 0.174932, 0.457377, -0.499139), (0.649957, 0.189445, 0.701917, -0.221313), (-0.189445, 0.649957, 0.221313, 0.701917), (0.468157, 0.489043, 0.718534, 0.159295), (-0.489043, 0.468157, -0.159295, 0.718534), (0.160914, 0.657602, 0.542621, 0.497221), (-0.657602, 0.160914, -0.497221, 0.542621), (0.896707, 0.261366, -0.351982, -0.0609334), (-0.261366, 0.896707, 0.0609334, -0.351982), (0.645888, 0.674703, -0.274359, -0.228761), (-0.674703, 0.645888, 0.228761, -0.274359), (0.222004, 0.907254, -0.123221, -0.335292), (-0.907254, 0.222004, 0.335292, -0.123221), (0.438201, 0.127724, 0.886371, 0.0775474), (-0.127724, 0.438201, -0.0775474, 0.886371), (0.315631, 0.329713, 0.728846, 0.510343), (-0.329713, 0.315631, -0.510343, 0.728846), (0.108488, 0.443355, 0.376027, 0.806393), (-0.443355, 0.108488, -0.806393, 0.376027), (0.438201, 0.127724, -0.886371, -0.0775474), (-0.127724, 0.438201, 0.0775474, -0.886371), (0.315631, 0.329713, -0.728846, -0.510343), (-0.329713, 0.315631, 0.510343, -0.728846), (0.108488, 0.443355, -0.376027, -0.806393), (-0.443355, 0.108488, 0.806393, -0.376027), (0.896707, 0.261366, 0.351982, 0.0609334), (-0.261366, 0.896707, -0.0609334, 0.351982), (0.645888, 0.674703, 0.274359, 0.228761), (-0.674703, 0.645888, -0.228761, 0.274359), (0.222004, 0.907254, 0.123221, 0.335292), (-0.907254, 0.222004, -0.335292, 0.123221), (0.649957, 0.189445, 0.221313, 0.701917), (-0.189445, 0.649957, -0.701917, 0.221313), (0.468157, 0.489043, -0.159295, 0.718534), (-0.489043, 0.468157, -0.718534, -0.159295), (0.160914, 0.657602, -0.497221, 0.542621), (-0.657602, 0.160914, -0.542621, -0.497221), (0.706578, 0.205949, -0.203579, -0.645669), (-0.205949, 0.706578, 0.645669, -0.203579), (0.50894, 0.531645, 0.14653, -0.660956), (-0.531645, 0.50894, 0.660956, 0.14653), (0.174932, 0.714888, 0.457377, -0.499139), (-0.714888, 0.174932, 0.499139, 0.457377), (0.342946, 0.0999598, 0.159324, -0.920332), (-0.0999598, 0.342946, 0.920332, 0.159324), (0.24702, 0.258041, 0.598144, -0.71737), (-0.258041, 0.24702, 0.71737, 0.598144), (0.0849055, 0.34698, 0.876693, -0.322188), (-0.34698, 0.0849055, 0.322188, 0.876693), (0.854211, 0.24898, -0.039781, 0.454699), (-0.24898, 0.854211, -0.454699, -0.039781), (0.615278, 0.642728, -0.261801, 0.37389), (-0.642728, 0.615278, -0.37389, -0.261801), (0.211483, 0.864258, -0.413671, 0.192898), (-0.864258, 0.211483, -0.192898, -0.413671), (0.854211, 0.24898, 0.039781, -0.454699), (-0.24898, 0.854211, 0.454699, 0.039781), (0.615278, 0.642728, 0.261801, -0.37389), (-0.642728, 0.615278, 0.37389, 0.261801), (0.211483, 0.864258, 0.413671, -0.192898), (-0.864258, 0.211483, 0.192898, 0.413671), (0.342946, 0.0999598, -0.159324, 0.920332), (-0.0999598, 0.342946, -0.920332, -0.159324), (0.24702, 0.258041, -0.598144, 0.71737), (-0.258041, 0.24702, -0.71737, -0.598144), (0.0849055, 0.34698, -0.876693, 0.322188), (-0.34698, 0.0849055, -0.322188, -0.876693), (0.735805, 0.0160553, -0.660956, -0.14653), (-0.0160553, 0.735805, 0.14653, -0.660956), (0.629198, 0.381807, -0.499139, -0.457377), (-0.381807, 0.629198, 0.457377, -0.499139), (0.353998, 0.645253, -0.203579, -0.645669), (-0.645253, 0.353998, 0.645669, -0.203579), (0.676842, 0.0147687, 0.718534, 0.159295), (-0.0147687, 0.676842, -0.159295, 0.718534), (0.578778, 0.351211, 0.542621, 0.497221), (-0.351211, 0.578778, -0.497221, 0.542621), (0.325631, 0.593547, 0.221313, 0.701917), (-0.593547, 0.325631, -0.701917, 0.221313), (0.933799, 0.0203756, -0.196604, 0.298247), (-0.0203756, 0.933799, -0.298247, -0.196604), (0.798506, 0.484545, -0.319387, 0.159987), (-0.484545, 0.798506, -0.159987, -0.319387), (0.449254, 0.818881, -0.356591, -0.0211408), (-0.818881, 0.449254, 0.0211408, -0.356591), (0.456327, 0.00995709, 0.510343, -0.728846), (-0.00995709, 0.456327, 0.728846, 0.510343), (0.390212, 0.236787, 0.806393, -0.376027), (-0.236787, 0.390212, 0.376027, 0.806393), (0.21954, 0.400169, 0.886371, 0.0775474), (-0.400169, 0.21954, -0.0775474, 0.886371), (0.456327, 0.00995709, -0.510343, 0.728846), (-0.00995709, 0.456327, -0.728846, -0.510343), (0.390212, 0.236787, -0.806393, 0.376027), (-0.236787, 0.390212, -0.376027, -0.806393), (0.21954, 0.400169, -0.886371, -0.0775474), (-0.400169, 0.21954, 0.0775474, -0.886371), (0.933799, 0.0203756, 0.196604, -0.298247), (-0.0203756, 0.933799, 0.298247, 0.196604), (0.798506, 0.484545, 0.319387, -0.159987), (-0.484545, 0.798506, 0.159987, 0.319387), (0.449254, 0.818881, 0.356591, 0.0211408), (-0.818881, 0.449254, -0.0211408, 0.356591), (0.676842, 0.0147687, -0.159295, 0.718534), (-0.0147687, 0.676842, -0.718534, -0.159295), (0.578778, 0.351211, -0.497221, 0.542621), (-0.351211, 0.578778, -0.542621, -0.497221), (0.325631, 0.593547, -0.701917, 0.221313), (-0.593547, 0.325631, -0.221313, -0.701917), (0.735805, 0.0160553, 0.14653, -0.660956), (-0.0160553, 0.735805, 0.660956, 0.14653), (0.629198, 0.381807, 0.457377, -0.499139), (-0.381807, 0.629198, 0.499139, 0.457377), (0.353998, 0.645253, 0.645669, -0.203579), (-0.645253, 0.353998, 0.203579, 0.645669), (0.357132, 0.00779265, -0.77983, -0.514064), (-0.00779265, 0.357132, 0.514064, -0.77983), (0.305389, 0.185315, -0.418321, -0.835107), (-0.185315, 0.305389, 0.835107, -0.418321), (0.171817, 0.313182, 0.0552772, -0.932384), (-0.313182, 0.171817, 0.932384, 0.0552772), (0.889545, 0.0194099, 0.37389, 0.261801), (-0.0194099, 0.889545, -0.261801, 0.37389), (0.760663, 0.461582, 0.192898, 0.413671), (-0.461582, 0.760663, -0.413671, 0.192898), (0.427963, 0.780073, -0.039781, 0.454699), (-0.780073, 0.427963, -0.454699, -0.039781), (0.889545, 0.0194099, -0.37389, -0.261801), (-0.0194099, 0.889545, 0.261801, -0.37389), (0.760663, 0.461582, -0.192898, -0.413671), (-0.461582, 0.760663, 0.413671, -0.192898), (0.427963, 0.780073, 0.039781, -0.454699), (-0.780073, 0.427963, 0.454699, 0.039781), (0.357132, 0.00779265, 0.77983, 0.514064), (-0.00779265, 0.357132, -0.514064, 0.77983), (0.305389, 0.185315, 0.418321, 0.835107), (-0.185315, 0.305389, -0.835107, 0.418321), (0.171817, 0.313182, -0.0552772, 0.932384), (-0.313182, 0.171817, -0.932384, -0.0552772), (0.83183, 0.128031, -0.507492, -0.184712), (-0.128031, 0.83183, 0.184712, -0.507492), (0.656371, 0.526793, -0.347145, -0.413711), (-0.526793, 0.656371, 0.413711, -0.347145), (0.305037, 0.784402, -0.0937807, -0.531857), (-0.784402, 0.305037, 0.531857, -0.0937807), (0.533776, 0.082156, 0.790869, 0.287853), (-0.082156, 0.533776, -0.287853, 0.790869), (0.421186, 0.338037, 0.540986, 0.644722), (-0.338037, 0.421186, -0.644722, 0.540986), (0.195739, 0.503342, 0.146147, 0.828839), (-0.503342, 0.195739, -0.828839, 0.146147), (0.97671, 0.15033, -0.0616577, 0.140128), (-0.15033, 0.97671, -0.140128, -0.0616577), (0.770691, 0.618545, -0.123461, 0.0905254), (-0.618545, 0.770691, -0.0905254, -0.123461), (0.358166, 0.921021, -0.152183, 0.0166668), (-0.921021, 0.358166, -0.0166668, -0.152183), (0.605245, 0.0931562, 0.534101, -0.582869), (-0.0931562, 0.605245, 0.582869, 0.534101), (0.47758, 0.383298, 0.753979, -0.237729), (-0.383298, 0.47758, 0.237729, 0.753979), (0.221947, 0.570736, 0.77183, 0.171111), (-0.570736, 0.221947, -0.171111, 0.77183), (0.605245, 0.0931562, -0.534101, 0.582869), (-0.0931562, 0.605245, -0.582869, -0.534101), (0.47758, 0.383298, -0.753979, 0.237729), (-0.383298, 0.47758, -0.237729, -0.753979), (0.221947, 0.570736, -0.77183, -0.171111), (-0.570736, 0.221947, 0.171111, -0.77183), (0.97671, 0.15033, 0.0616577, -0.140128), (-0.15033, 0.97671, 0.140128, 0.0616577), (0.770691, 0.618545, 0.123461, -0.0905254), (-0.618545, 0.770691, 0.0905254, 0.123461), (0.358166, 0.921021, 0.152183, -0.0166668), (-0.921021, 0.358166, 0.0166668, 0.152183), (0.533776, 0.082156, -0.287853, 0.790869), (-0.082156, 0.533776, -0.790869, -0.287853), (0.421186, 0.338037, -0.644722, 0.540986), (-0.338037, 0.421186, -0.540986, -0.644722), (0.195739, 0.503342, -0.828839, 0.146147), (-0.503342, 0.195739, -0.146147, -0.828839), (0.83183, 0.128031, 0.184712, -0.507492), (-0.128031, 0.83183, 0.507492, 0.184712), (0.656371, 0.526793, 0.413711, -0.347145), (-0.526793, 0.656371, 0.347145, 0.413711), (0.305037, 0.784402, 0.531857, -0.0937807), (-0.784402, 0.305037, 0.0937807, 0.531857), (0.151311, 0.023289, -0.904522, -0.397999), (-0.023289, 0.151311, 0.397999, -0.904522), (0.119395, 0.0958246, -0.584339, -0.796938), (-0.0958246, 0.119395, 0.796938, -0.584339), (0.0554868, 0.142684, -0.107584, -0.982338), (-0.142684, 0.0554868, 0.982338, -0.107584), (0.781368, 0.120264, 0.451488, 0.413713), (-0.120264, 0.781368, -0.413713, 0.451488), (0.616553, 0.494836, 0.184144, 0.58403), (-0.494836, 0.616553, -0.58403, 0.184144), (0.286532, 0.736817, -0.132542, 0.597857), (-0.736817, 0.286532, -0.597857, -0.132542), (0.781368, 0.120264, -0.451488, -0.413713), (-0.120264, 0.781368, 0.413713, -0.451488), (0.616553, 0.494836, -0.184144, -0.58403), (-0.494836, 0.616553, 0.58403, -0.184144), (0.286532, 0.736817, 0.132542, -0.597857), (-0.736817, 0.286532, 0.597857, 0.132542), (0.151311, 0.023289, 0.904522, 0.397999), (-0.023289, 0.151311, -0.397999, 0.904522), (0.119395, 0.0958246, 0.584339, 0.796938), (-0.0958246, 0.119395, -0.796938, 0.584339), (0.0554868, 0.142684, 0.107584, 0.982338), (-0.142684, 0.0554868, -0.982338, 0.107584), (0.560512, 0.246631, -0.647597, -0.453452), (-0.246631, 0.560512, 0.453452, -0.647597), (0.362102, 0.493844, -0.334109, -0.716499), (-0.493844, 0.362102, 0.716499, -0.334109), (0.0666671, 0.608733, 0.0689027, -0.787561), (-0.608733, 0.0666671, 0.787561, 0.0689027), (0.723617, 0.318399, 0.501626, 0.351242), (-0.318399, 0.723617, -0.351242, 0.501626), (0.467471, 0.63755, 0.2588, 0.554998), (-0.63755, 0.467471, -0.554998, 0.2588), (0.0860668, 0.785871, -0.0533718, 0.610042), (-0.785871, 0.0860668, -0.610042, -0.0533718), (0.770349, 0.338962, -0.455483, 0.290175), (-0.338962, 0.770349, -0.290175, -0.455483), (0.497661, 0.678724, -0.539548, 0.0235572), (-0.678724, 0.497661, -0.0235572, -0.539548), (0.0916251, 0.836623, -0.479041, -0.249373), (-0.836623, 0.0916251, 0.249373, -0.479041), (0.233546, 0.102763, 0.74876, -0.611763), (-0.102763, 0.233546, 0.611763, 0.74876), (0.150876, 0.205769, 0.954327, -0.155422), (-0.205769, 0.150876, 0.155422, 0.954327), (0.0277779, 0.253639, 0.904182, 0.342564), (-0.253639, 0.0277779, -0.342564, 0.904182), (0.233546, 0.102763, -0.74876, 0.611763), (-0.102763, 0.233546, -0.611763, -0.74876), (0.150876, 0.205769, -0.954327, 0.155422), (-0.205769, 0.150876, -0.155422, -0.954327), (0.0277779, 0.253639, -0.904182, -0.342564), (-0.253639, 0.0277779, 0.342564, -0.904182), (0.770349, 0.338962, 0.455483, -0.290175), (-0.338962, 0.770349, 0.290175, 0.455483), (0.497661, 0.678724, 0.539548, -0.0235572), (-0.678724, 0.497661, 0.0235572, 0.539548), (0.0916251, 0.836623, 0.479041, 0.249373), (-0.836623, 0.0916251, -0.249373, 0.479041), (0.723617, 0.318399, -0.351242, 0.501626), (-0.318399, 0.723617, -0.501626, -0.351242), (0.467471, 0.63755, -0.554998, 0.2588), (-0.63755, 0.467471, -0.2588, -0.554998), (0.0860668, 0.785871, -0.610042, -0.0533718), (-0.785871, 0.0860668, 0.0533718, -0.610042), (0.560512, 0.246631, 0.453452, -0.647597), (-0.246631, 0.560512, 0.647597, 0.453452), (0.362102, 0.493844, 0.716499, -0.334109), (-0.493844, 0.362102, 0.334109, 0.716499), (0.0666671, 0.608733, 0.787561, 0.0689027), (-0.608733, 0.0666671, -0.0689027, 0.787561), (0.494325, 0.217508, -0.452205, -0.70982), (-0.217508, 0.494325, 0.70982, -0.452205), (0.319344, 0.43553, -0.0367112, -0.840824), (-0.43553, 0.319344, 0.840824, -0.0367112), (0.0587948, 0.536852, 0.388619, -0.746531), (-0.536852, 0.0587948, 0.746531, 0.388619), (0.885015, 0.389416, 0.161438, 0.19759), (-0.389416, 0.885015, -0.19759, 0.161438), (0.571737, 0.779751, 0.0410143, 0.251837), (-0.779751, 0.571737, -0.251837, 0.0410143), (0.105263, 0.961153, -0.0903992, 0.238605), (-0.961153, 0.105263, -0.238605, -0.0903992), (0.885015, 0.389416, -0.161438, -0.19759), (-0.389416, 0.885015, 0.19759, -0.161438), (0.571737, 0.779751, -0.0410143, -0.251837), (-0.779751, 0.571737, 0.251837, -0.0410143), (0.105263, 0.961153, 0.0903992, -0.238605), (-0.961153, 0.105263, 0.238605, 0.0903992), (0.494325, 0.217508, 0.452205, 0.70982), (-0.217508, 0.494325, -0.70982, 0.452205), (0.319344, 0.43553, 0.0367112, 0.840824), (-0.43553, 0.319344, -0.840824, 0.0367112), (0.0587948, 0.536852, -0.388619, 0.746531), (-0.536852, 0.0587948, -0.746531, -0.388619), (0.770349, 0.338962, -0.442393, -0.309767), (-0.338962, 0.770349, 0.309767, -0.442393), (0.497661, 0.678724, -0.22824, -0.489462), (-0.678724, 0.497661, 0.489462, -0.22824), (0.0916251, 0.836623, 0.0470695, -0.538007), (-0.836623, 0.0916251, 0.538007, 0.0470695), (0.494325, 0.217508, 0.689419, 0.482737), (-0.217508, 0.494325, -0.482737, 0.689419), (0.319344, 0.43553, 0.355686, 0.762772), (-0.43553, 0.319344, -0.762772, 0.355686), (0.0587948, 0.536852, -0.0733525, 0.838423), (-0.536852, 0.0587948, -0.838423, -0.0733525), (0.904522, 0.397999, -0.0958246, 0.119395), (-0.397999, 0.904522, -0.119395, -0.0958246), (0.584339, 0.796938, -0.142684, 0.0554868), (-0.796938, 0.584339, -0.0554868, -0.142684), (0.107584, 0.982338, -0.151311, -0.023289), (-0.982338, 0.107584, 0.023289, -0.151311), (0.560512, 0.246631, 0.666759, -0.424773), (-0.246631, 0.560512, 0.424773, 0.666759), (0.362102, 0.493844, 0.789817, -0.0344842), (-0.493844, 0.362102, 0.0344842, 0.789817), (0.0666671, 0.608733, 0.701244, 0.365044), (-0.608733, 0.0666671, -0.365044, 0.701244), (0.560512, 0.246631, -0.666759, 0.424773), (-0.246631, 0.560512, -0.424773, -0.666759), (0.362102, 0.493844, -0.789817, 0.0344842), (-0.493844, 0.362102, -0.0344842, -0.789817), (0.0666671, 0.608733, -0.701244, -0.365044), (-0.608733, 0.0666671, 0.365044, -0.701244), (0.904522, 0.397999, 0.0958246, -0.119395), (-0.397999, 0.904522, 0.119395, 0.0958246), (0.584339, 0.796938, 0.142684, -0.0554868), (-0.796938, 0.584339, 0.0554868, 0.142684), (0.107584, 0.982338, 0.151311, 0.023289), (-0.982338, 0.107584, -0.023289, 0.151311), (0.494325, 0.217508, -0.482737, 0.689419), (-0.217508, 0.494325, -0.689419, -0.482737), (0.319344, 0.43553, -0.762772, 0.355686), (-0.43553, 0.319344, -0.355686, -0.762772), (0.0587948, 0.536852, -0.838423, -0.0733525), (-0.536852, 0.0587948, 0.0733525, -0.838423), (0.770349, 0.338962, 0.309767, -0.442393), (-0.338962, 0.770349, 0.442393, 0.309767), (0.497661, 0.678724, 0.489462, -0.22824), (-0.678724, 0.497661, 0.22824, 0.489462), (0.0916251, 0.836623, 0.538007, 0.0470695), (-0.836623, 0.0916251, -0.0470695, 0.538007), (0.140128, 0.0616577, -0.770691, -0.618545), (-0.0616577, 0.140128, 0.618545, -0.770691), (0.0905254, 0.123461, -0.358166, -0.921021), (-0.123461, 0.0905254, 0.921021, -0.358166), (0.0166668, 0.152183, 0.15033, -0.97671), (-0.152183, 0.0166668, 0.97671, 0.15033), (0.723617, 0.318399, 0.329027, 0.51647), (-0.318399, 0.723617, -0.51647, 0.329027), (0.467471, 0.63755, 0.0267113, 0.61179), (-0.63755, 0.467471, -0.61179, 0.0267113), (0.0860668, 0.785871, -0.282762, 0.543181), (-0.785871, 0.0860668, -0.543181, -0.282762), (0.723617, 0.318399, -0.329027, -0.51647), (-0.318399, 0.723617, 0.51647, -0.329027), (0.467471, 0.63755, -0.0267113, -0.61179), (-0.63755, 0.467471, 0.61179, -0.0267113), (0.0860668, 0.785871, 0.282762, -0.543181), (-0.785871, 0.0860668, 0.543181, 0.282762), (0.140128, 0.0616577, 0.770691, 0.618545), (-0.0616577, 0.140128, -0.618545, 0.770691), (0.0905254, 0.123461, 0.358166, 0.921021), (-0.123461, 0.0905254, -0.921021, 0.358166), (0.0166668, 0.152183, -0.15033, 0.97671), (-0.152183, 0.0166668, -0.97671, -0.15033), (0.605245, 0.0931562, -0.742892, -0.270391), (-0.0931562, 0.605245, 0.270391, -0.742892), (0.47758, 0.383298, -0.508168, -0.605611), (-0.383298, 0.47758, 0.605611, -0.508168), (0.221947, 0.570736, -0.137281, -0.778559), (-0.570736, 0.221947, 0.778559, -0.137281), (0.781368, 0.120264, 0.575442, 0.209444), (-0.120264, 0.781368, -0.209444, 0.575442), (0.616553, 0.494836, 0.393625, 0.469105), (-0.494836, 0.616553, -0.469105, 0.393625), (0.286532, 0.736817, 0.106337, 0.603069), (-0.736817, 0.286532, -0.603069, 0.106337), (0.83183, 0.128031, -0.36486, 0.398175), (-0.128031, 0.83183, -0.398175, -0.36486), (0.656371, 0.526793, -0.515066, 0.1624), (-0.526793, 0.656371, -0.1624, -0.515066), (0.305037, 0.784402, -0.52726, -0.116891), (-0.784402, 0.305037, 0.116891, -0.52726), (0.252186, 0.0388151, 0.564911, -0.784711), (-0.0388151, 0.252186, 0.784711, 0.564911), (0.198992, 0.159708, 0.881583, -0.397124), (-0.159708, 0.198992, 0.397124, 0.881583), (0.0924779, 0.237807, 0.962035, 0.0968718), (-0.237807, 0.0924779, -0.0968718, 0.962035), (0.252186, 0.0388151, -0.564911, 0.784711), (-0.0388151, 0.252186, -0.784711, -0.564911), (0.198992, 0.159708, -0.881583, 0.397124), (-0.159708, 0.198992, -0.397124, -0.881583), (0.0924779, 0.237807, -0.962035, -0.0968718), (-0.237807, 0.0924779, 0.0968718, -0.962035), (0.83183, 0.128031, 0.36486, -0.398175), (-0.128031, 0.83183, 0.398175, 0.36486), (0.656371, 0.526793, 0.515066, -0.1624), (-0.526793, 0.656371, 0.1624, 0.515066), (0.305037, 0.784402, 0.52726, 0.116891), (-0.784402, 0.305037, -0.116891, 0.52726), (0.781368, 0.120264, -0.209444, 0.575442), (-0.120264, 0.781368, -0.575442, -0.209444), (0.616553, 0.494836, -0.469105, 0.393625), (-0.494836, 0.616553, -0.393625, -0.469105), (0.286532, 0.736817, -0.603069, 0.106337), (-0.736817, 0.286532, -0.106337, -0.603069), (0.605245, 0.0931562, 0.270391, -0.742892), (-0.0931562, 0.605245, 0.742892, 0.270391), (0.47758, 0.383298, 0.605611, -0.508168), (-0.383298, 0.47758, 0.508168, 0.605611), (0.221947, 0.570736, 0.778559, -0.137281), (-0.570736, 0.221947, 0.137281, 0.778559), (0.533776, 0.082156, -0.620511, -0.568594), (-0.082156, 0.533776, 0.568594, -0.620511), (0.421186, 0.338037, -0.253082, -0.802672), (-0.338037, 0.421186, 0.802672, -0.253082), (0.195739, 0.503342, 0.182161, -0.821676), (-0.503342, 0.195739, 0.821676, 0.182161), (0.955647, 0.147088, 0.207077, 0.149074), (-0.147088, 0.955647, -0.149074, 0.207077), (0.75407, 0.605205, 0.104797, 0.232641), (-0.605205, 0.75407, -0.232641, 0.104797), (0.350441, 0.901159, -0.0255635, 0.253871), (-0.901159, 0.350441, -0.253871, -0.0255635), (0.955647, 0.147088, -0.207077, -0.149074), (-0.147088, 0.955647, 0.149074, -0.207077), (0.75407, 0.605205, -0.104797, -0.232641), (-0.605205, 0.75407, 0.232641, -0.104797), (0.350441, 0.901159, 0.0255635, -0.253871), (-0.901159, 0.350441, 0.253871, 0.0255635), (0.533776, 0.082156, 0.620511, 0.568594), (-0.082156, 0.533776, -0.568594, 0.620511), (0.421186, 0.338037, 0.253082, 0.802672), (-0.338037, 0.421186, -0.802672, 0.253082), (0.195739, 0.503342, -0.182161, 0.821676), (-0.503342, 0.195739, -0.821676, -0.182161), (0.727414, 0.11196, -0.676359, 0.0295305), (-0.11196, 0.727414, -0.0295305, -0.676359), (0.573979, 0.460667, -0.600509, -0.312605), (-0.460667, 0.573979, 0.312605, -0.600509), (0.266747, 0.685939, -0.363754, -0.570979), (-0.685939, 0.266747, 0.570979, -0.363754), (0.669124, 0.102988, 0.73528, -0.032103), (-0.102988, 0.669124, 0.032103, 0.73528), (0.527984, 0.423752, 0.652822, 0.339838), (-0.423752, 0.527984, -0.339838, 0.652822), (0.245372, 0.630972, 0.395442, 0.620719), (-0.630972, 0.245372, -0.620719, 0.395442), (0.923151, 0.142086, -0.327855, 0.141827), (-0.142086, 0.923151, -0.141827, -0.327855), (0.728429, 0.584626, -0.354845, -0.0411018), (-0.584626, 0.728429, 0.0411018, -0.354845), (0.338525, 0.870515, -0.286754, -0.213018), (-0.870515, 0.338525, 0.213018, -0.286754), (0.451123, 0.0694345, 0.806393, -0.376027), (-0.0694345, 0.451123, 0.376027, 0.806393), (0.355967, 0.285694, 0.886371, 0.0775474), (-0.285694, 0.355967, -0.0775474, 0.886371), (0.16543, 0.425401, 0.728846, 0.510343), (-0.425401, 0.16543, -0.510343, 0.728846), (0.451123, 0.0694345, -0.806393, 0.376027), (-0.0694345, 0.451123, -0.376027, -0.806393), (0.355967, 0.285694, -0.886371, -0.0775474), (-0.285694, 0.355967, 0.0775474, -0.886371), (0.16543, 0.425401, -0.728846, -0.510343), (-0.425401, 0.16543, 0.510343, -0.728846), (0.923151, 0.142086, 0.327855, -0.141827), (-0.142086, 0.923151, 0.141827, 0.327855), (0.728429, 0.584626, 0.354845, 0.0411018), (-0.584626, 0.728429, -0.0411018, 0.354845), (0.338525, 0.870515, 0.286754, 0.213018), (-0.870515, 0.338525, -0.213018, 0.286754), (0.669124, 0.102988, 0.032103, 0.73528), (-0.102988, 0.669124, -0.73528, 0.032103), (0.527984, 0.423752, -0.339838, 0.652822), (-0.423752, 0.527984, -0.652822, -0.339838), (0.245372, 0.630972, -0.620719, 0.395442), (-0.630972, 0.245372, -0.395442, -0.620719), (0.727414, 0.11196, -0.0295305, -0.676359), (-0.11196, 0.727414, 0.676359, -0.0295305), (0.573979, 0.460667, 0.312605, -0.600509), (-0.460667, 0.573979, 0.600509, 0.312605), (0.266747, 0.685939, 0.570979, -0.363754), (-0.685939, 0.266747, 0.363754, 0.570979), (0.35306, 0.0543411, -0.370838, -0.857249), (-0.0543411, 0.35306, 0.857249, -0.370838), (0.278588, 0.223591, 0.107469, -0.927818), (-0.223591, 0.278588, 0.927818, 0.107469), (0.129469, 0.332929, 0.55698, -0.749779), (-0.332929, 0.129469, 0.749779, 0.55698), (0.879401, 0.135353, 0.192898, 0.413671), (-0.135353, 0.879401, -0.413671, 0.192898), (0.693907, 0.556919, -0.039781, 0.454699), (-0.556919, 0.693907, -0.454699, -0.039781), (0.322482, 0.82926, -0.261801, 0.37389), (-0.82926, 0.322482, -0.37389, -0.261801), (0.879401, 0.135353, -0.192898, -0.413671), (-0.135353, 0.879401, 0.413671, -0.192898), (0.693907, 0.556919, 0.039781, -0.454699), (-0.556919, 0.693907, 0.454699, 0.039781), (0.322482, 0.82926, 0.261801, -0.37389), (-0.82926, 0.322482, 0.37389, 0.261801), (0.35306, 0.0543411, 0.370838, 0.857249), (-0.0543411, 0.35306, -0.857249, 0.370838), (0.278588, 0.223591, -0.107469, 0.927818), (-0.223591, 0.278588, -0.927818, -0.107469), (0.129469, 0.332929, -0.55698, 0.749779), (-0.332929, 0.129469, -0.749779, -0.55698), (0.673651, 0.296414, -0.363754, -0.570979), (-0.296414, 0.673651, 0.570979, -0.363754), (0.435192, 0.593527, -0.0295305, -0.676359), (-0.593527, 0.435192, 0.676359, -0.0295305), (0.0801239, 0.731606, 0.312605, -0.600509), (-0.731606, 0.0801239, 0.600509, 0.312605), (0.619669, 0.272661, 0.395442, 0.620719), (-0.272661, 0.619669, -0.620719, 0.395442), (0.400319, 0.545966, 0.032103, 0.73528), (-0.545966, 0.400319, -0.73528, 0.032103), (0.0737032, 0.672979, -0.339838, 0.652822), (-0.672979, 0.0737032, -0.652822, -0.339838), (0.85492, 0.376174, -0.179572, 0.308801), (-0.376174, 0.85492, -0.308801, -0.179572), (0.552296, 0.753236, -0.309914, 0.177644), (-0.753236, 0.552296, -0.177644, -0.309914), (0.101684, 0.92847, -0.357216, -0.00111332), (-0.92847, 0.101684, 0.00111332, -0.357216), (0.417781, 0.183828, 0.510343, -0.728846), (-0.183828, 0.417781, 0.728846, 0.510343), (0.269895, 0.36809, 0.806393, -0.376027), (-0.36809, 0.269895, 0.376027, 0.806393), (0.0496907, 0.453723, 0.886371, 0.0775474), (-0.453723, 0.0496907, -0.0775474, 0.886371), (0.417781, 0.183828, -0.510343, 0.728846), (-0.183828, 0.417781, -0.728846, -0.510343), (0.269895, 0.36809, -0.806393, 0.376027), (-0.36809, 0.269895, -0.376027, -0.806393), (0.0496907, 0.453723, -0.886371, -0.0775474), (-0.453723, 0.0496907, 0.0775474, -0.886371), (0.85492, 0.376174, 0.179572, -0.308801), (-0.376174, 0.85492, 0.308801, 0.179572), (0.552296, 0.753236, 0.309914, -0.177644), (-0.753236, 0.552296, 0.177644, 0.309914), (0.101684, 0.92847, 0.357216, 0.00111332), (-0.92847, 0.101684, -0.00111332, 0.357216), (0.619669, 0.272661, -0.620719, 0.395442), (-0.272661, 0.619669, -0.395442, -0.620719), (0.400319, 0.545966, -0.73528, 0.032103), (-0.545966, 0.400319, -0.032103, -0.73528), (0.0737032, 0.672979, -0.652822, -0.339838), (-0.672979, 0.0737032, 0.339838, -0.652822), (0.673651, 0.296414, 0.570979, -0.363754), (-0.296414, 0.673651, 0.363754, 0.570979), (0.435192, 0.593527, 0.676359, -0.0295305), (-0.593527, 0.435192, 0.0295305, 0.676359), (0.0801239, 0.731606, 0.600509, 0.312605), (-0.731606, 0.0801239, -0.312605, 0.600509), (0.326965, 0.143868, -0.807427, -0.469529), (-0.143868, 0.326965, 0.469529, -0.807427), (0.211226, 0.288076, -0.464487, -0.810338), (-0.288076, 0.211226, 0.810338, -0.464487), (0.0388891, 0.355094, 0.00291102, -0.934017), (-0.355094, 0.0388891, 0.934017, 0.00291102), (0.814404, 0.358346, 0.37389, 0.261801), (-0.358346, 0.814404, -0.261801, 0.37389), (0.526122, 0.717539, 0.192898, 0.413671), (-0.717539, 0.526122, -0.413671, 0.192898), (0.096865, 0.884468, -0.039781, 0.454699), (-0.884468, 0.096865, -0.454699, -0.039781), (0.814404, 0.358346, -0.37389, -0.261801), (-0.358346, 0.814404, 0.261801, -0.37389), (0.526122, 0.717539, -0.192898, -0.413671), (-0.717539, 0.526122, 0.413671, -0.192898), (0.096865, 0.884468, 0.039781, -0.454699), (-0.884468, 0.096865, 0.454699, 0.039781), (0.326965, 0.143868, 0.807427, 0.469529), (-0.143868, 0.326965, -0.469529, 0.807427), (0.211226, 0.288076, 0.464487, 0.810338), (-0.288076, 0.211226, -0.810338, 0.464487), (0.0388891, 0.355094, -0.00291102, 0.934017), (-0.355094, 0.0388891, -0.934017, -0.00291102), (0.673651, 0.296414, -0.660956, -0.14653), (-0.296414, 0.673651, 0.14653, -0.660956), (0.435192, 0.593527, -0.499139, -0.457377), (-0.593527, 0.435192, 0.457377, -0.499139), (0.0801239, 0.731606, -0.203579, -0.645669), (-0.731606, 0.0801239, 0.645669, -0.203579), (0.619669, 0.272661, 0.718534, 0.159295), (-0.272661, 0.619669, -0.159295, 0.718534), (0.400319, 0.545966, 0.542621, 0.497221), (-0.545966, 0.400319, -0.497221, 0.542621), (0.0737032, 0.672979, 0.221313, 0.701917), (-0.672979, 0.0737032, -0.701917, 0.221313), (0.85492, 0.376174, -0.353392, 0.0521393), (-0.376174, 0.85492, -0.0521393, -0.353392), (0.552296, 0.753236, -0.332116, -0.131542), (-0.753236, 0.552296, 0.131542, -0.332116), (0.101684, 0.92847, -0.22185, -0.279976), (-0.92847, 0.101684, 0.279976, -0.22185), (0.417781, 0.183828, 0.876239, -0.154505), (-0.183828, 0.417781, 0.154505, 0.876239), (0.269895, 0.36809, 0.836098, 0.304315), (-0.36809, 0.269895, -0.304315, 0.836098), (0.0496907, 0.453723, 0.571924, 0.681593), (-0.453723, 0.0496907, -0.681593, 0.571924), (0.417781, 0.183828, -0.876239, 0.154505), (-0.183828, 0.417781, -0.154505, -0.876239), (0.269895, 0.36809, -0.836098, -0.304315), (-0.36809, 0.269895, 0.304315, -0.836098), (0.0496907, 0.453723, -0.571924, -0.681593), (-0.453723, 0.0496907, 0.681593, -0.571924), (0.85492, 0.376174, 0.353392, -0.0521393), (-0.376174, 0.85492, 0.0521393, 0.353392), (0.552296, 0.753236, 0.332116, 0.131542), (-0.753236, 0.552296, -0.131542, 0.332116), (0.101684, 0.92847, 0.22185, 0.279976), (-0.92847, 0.101684, -0.279976, 0.22185), (0.619669, 0.272661, -0.159295, 0.718534), (-0.272661, 0.619669, -0.718534, -0.159295), (0.400319, 0.545966, -0.497221, 0.542621), (-0.545966, 0.400319, -0.542621, -0.497221), (0.0737032, 0.672979, -0.701917, 0.221313), (-0.672979, 0.0737032, -0.221313, -0.701917), (0.673651, 0.296414, 0.14653, -0.660956), (-0.296414, 0.673651, 0.660956, 0.14653), (0.435192, 0.593527, 0.457377, -0.499139), (-0.593527, 0.435192, 0.499139, 0.457377), (0.0801239, 0.731606, 0.645669, -0.203579), (-0.731606, 0.0801239, 0.203579, 0.645669), (0.326965, 0.143868, -0.136329, -0.924018), (-0.143868, 0.326965, 0.924018, -0.136329), (0.211226, 0.288076, 0.343944, -0.868388), (-0.288076, 0.211226, 0.868388, 0.343944), (0.0388891, 0.355094, 0.732059, -0.580074), (-0.355094, 0.0388891, 0.580074, 0.732059), (0.814404, 0.358346, 0.0792592, 0.449501), (-0.358346, 0.814404, -0.449501, 0.0792592), (0.526122, 0.717539, -0.15611, 0.428909), (-0.717539, 0.526122, -0.428909, -0.15611), (0.096865, 0.884468, -0.34965, 0.293391), (-0.884468, 0.096865, -0.293391, -0.34965), (0.814404, 0.358346, -0.0792592, -0.449501), (-0.358346, 0.814404, 0.449501, -0.0792592), (0.526122, 0.717539, 0.15611, -0.428909), (-0.717539, 0.526122, 0.428909, 0.15611), (0.096865, 0.884468, 0.34965, -0.293391), (-0.884468, 0.096865, 0.293391, 0.34965), (0.326965, 0.143868, 0.136329, 0.924018), (-0.143868, 0.326965, -0.924018, 0.136329), (0.211226, 0.288076, -0.343944, 0.868388), (-0.288076, 0.211226, -0.868388, -0.343944), (0.0388891, 0.355094, -0.732059, 0.580074), (-0.355094, 0.0388891, -0.580074, -0.732059), (0.727414, 0.11196, -0.499139, -0.457377), (-0.11196, 0.727414, 0.457377, -0.499139), (0.573979, 0.460667, -0.203579, -0.645669), (-0.460667, 0.573979, 0.645669, -0.203579), (0.266747, 0.685939, 0.14653, -0.660956), (-0.685939, 0.266747, 0.660956, 0.14653), (0.669124, 0.102988, 0.542621, 0.497221), (-0.102988, 0.669124, -0.497221, 0.542621), (0.527984, 0.423752, 0.221313, 0.701917), (-0.423752, 0.527984, -0.701917, 0.221313), (0.245372, 0.630972, -0.159295, 0.718534), (-0.630972, 0.245372, -0.718534, -0.159295), (0.923151, 0.142086, -0.0935296, 0.344756), (-0.142086, 0.923151, -0.344756, -0.0935296), (0.728429, 0.584626, -0.253377, 0.251802), (-0.584626, 0.728429, -0.251802, -0.253377), (0.338525, 0.870515, -0.345332, 0.0913788), (-0.870515, 0.338525, -0.0913788, -0.345332), (0.451123, 0.0694345, 0.304315, -0.836098), (-0.0694345, 0.451123, 0.836098, 0.304315), (0.355967, 0.285694, 0.681593, -0.571924), (-0.285694, 0.355967, 0.571924, 0.681593), (0.16543, 0.425401, 0.876239, -0.154505), (-0.425401, 0.16543, 0.154505, 0.876239), (0.451123, 0.0694345, -0.304315, 0.836098), (-0.0694345, 0.451123, -0.836098, -0.304315), (0.355967, 0.285694, -0.681593, 0.571924), (-0.285694, 0.355967, -0.571924, -0.681593), (0.16543, 0.425401, -0.876239, 0.154505), (-0.425401, 0.16543, -0.154505, -0.876239), (0.923151, 0.142086, 0.0935296, -0.344756), (-0.142086, 0.923151, 0.344756, 0.0935296), (0.728429, 0.584626, 0.253377, -0.251802), (-0.584626, 0.728429, 0.251802, 0.253377), (0.338525, 0.870515, 0.345332, -0.0913788), (-0.870515, 0.338525, 0.0913788, 0.345332), (0.669124, 0.102988, -0.497221, 0.542621), (-0.102988, 0.669124, -0.542621, -0.497221), (0.527984, 0.423752, -0.701917, 0.221313), (-0.423752, 0.527984, -0.221313, -0.701917), (0.245372, 0.630972, -0.718534, -0.159295), (-0.630972, 0.245372, 0.159295, -0.718534), (0.727414, 0.11196, 0.457377, -0.499139), (-0.11196, 0.727414, 0.499139, 0.457377), (0.573979, 0.460667, 0.645669, -0.203579), (-0.460667, 0.573979, 0.203579, 0.645669), (0.266747, 0.685939, 0.660956, 0.14653), (-0.685939, 0.266747, -0.14653, 0.660956), (0.35306, 0.0543411, -0.901438, -0.244553), (-0.0543411, 0.35306, 0.244553, -0.901438), (0.278588, 0.223591, -0.658391, -0.662508), (-0.223591, 0.278588, 0.662508, -0.658391), (0.129469, 0.332929, -0.23893, -0.902944), (-0.332929, 0.129469, 0.902944, -0.23893), (0.879401, 0.135353, 0.428909, 0.15611), (-0.135353, 0.879401, -0.15611, 0.428909), (0.693907, 0.556919, 0.293391, 0.34965), (-0.556919, 0.693907, -0.34965, 0.293391), (0.322482, 0.82926, 0.0792592, 0.449501), (-0.82926, 0.322482, -0.449501, 0.0792592), (0.879401, 0.135353, -0.428909, -0.15611), (-0.135353, 0.879401, 0.15611, -0.428909), (0.693907, 0.556919, -0.293391, -0.34965), (-0.556919, 0.693907, 0.34965, -0.293391), (0.322482, 0.82926, -0.0792592, -0.449501), (-0.82926, 0.322482, 0.449501, -0.0792592), (0.35306, 0.0543411, 0.901438, 0.244553), (-0.0543411, 0.35306, -0.244553, 0.901438), (0.278588, 0.223591, 0.658391, 0.662508), (-0.223591, 0.278588, -0.662508, 0.658391), (0.129469, 0.332929, 0.23893, 0.902944), (-0.332929, 0.129469, -0.902944, 0.23893), (0.83183, 0.128031, -0.539548, 0.0235572), (-0.128031, 0.83183, -0.0235572, -0.539548), (0.656371, 0.526793, -0.479041, -0.249373), (-0.526793, 0.656371, 0.249373, -0.479041), (0.305037, 0.784402, -0.290175, -0.455483), (-0.784402, 0.305037, 0.455483, -0.290175), (0.533776, 0.082156, 0.840824, -0.0367112), (-0.082156, 0.533776, 0.0367112, 0.840824), (0.421186, 0.338037, 0.746531, 0.388619), (-0.338037, 0.421186, -0.388619, 0.746531), (0.195739, 0.503342, 0.452205, 0.70982), (-0.503342, 0.195739, -0.70982, 0.452205), (0.97671, 0.15033, -0.152183, 0.0166668), (-0.15033, 0.97671, -0.0166668, -0.152183), (0.770691, 0.618545, -0.140128, -0.0616577), (-0.618545, 0.770691, 0.0616577, -0.140128), (0.358166, 0.921021, -0.0905254, -0.123461), (-0.921021, 0.358166, 0.123461, -0.0905254), (0.605245, 0.0931562, 0.716499, -0.334109), (-0.0931562, 0.605245, 0.334109, 0.716499), (0.47758, 0.383298, 0.787561, 0.0689027), (-0.383298, 0.47758, -0.0689027, 0.787561), (0.221947, 0.570736, 0.647597, 0.453452), (-0.570736, 0.221947, -0.453452, 0.647597), (0.605245, 0.0931562, -0.716499, 0.334109), (-0.0931562, 0.605245, -0.334109, -0.716499), (0.47758, 0.383298, -0.787561, -0.0689027), (-0.383298, 0.47758, 0.0689027, -0.787561), (0.221947, 0.570736, -0.647597, -0.453452), (-0.570736, 0.221947, 0.453452, -0.647597), (0.97671, 0.15033, 0.152183, -0.0166668), (-0.15033, 0.97671, 0.0166668, 0.152183), (0.770691, 0.618545, 0.140128, 0.0616577), (-0.618545, 0.770691, -0.0616577, 0.140128), (0.358166, 0.921021, 0.0905254, 0.123461), (-0.921021, 0.358166, -0.123461, 0.0905254), (0.533776, 0.082156, 0.0367112, 0.840824), (-0.082156, 0.533776, -0.840824, 0.0367112), (0.421186, 0.338037, -0.388619, 0.746531), (-0.338037, 0.421186, -0.746531, -0.388619), (0.195739, 0.503342, -0.70982, 0.452205), (-0.503342, 0.195739, -0.452205, -0.70982), (0.83183, 0.128031, -0.0235572, -0.539548), (-0.128031, 0.83183, 0.539548, -0.0235572), (0.656371, 0.526793, 0.249373, -0.479041), (-0.526793, 0.656371, 0.479041, 0.249373), (0.305037, 0.784402, 0.455483, -0.290175), (-0.784402, 0.305037, 0.290175, 0.455483), (0.151311, 0.023289, -0.107584, -0.982338), (-0.023289, 0.151311, 0.982338, -0.107584), (0.119395, 0.0958246, 0.397999, -0.904522), (-0.0958246, 0.119395, 0.904522, 0.397999), (0.0554868, 0.142684, 0.796938, -0.584339), (-0.142684, 0.0554868, 0.584339, 0.796938), (0.781368, 0.120264, 0.2588, 0.554998), (-0.120264, 0.781368, -0.554998, 0.2588), (0.616553, 0.494836, -0.0533718, 0.610042), (-0.494836, 0.616553, -0.610042, -0.0533718), (0.286532, 0.736817, -0.351242, 0.501626), (-0.736817, 0.286532, -0.501626, -0.351242), (0.781368, 0.120264, -0.2588, -0.554998), (-0.120264, 0.781368, 0.554998, -0.2588), (0.616553, 0.494836, 0.0533718, -0.610042), (-0.494836, 0.616553, 0.610042, 0.0533718), (0.286532, 0.736817, 0.351242, -0.501626), (-0.736817, 0.286532, 0.501626, 0.351242), (0.151311, 0.023289, 0.107584, 0.982338), (-0.023289, 0.151311, -0.982338, 0.107584), (0.119395, 0.0958246, -0.397999, 0.904522), (-0.0958246, 0.119395, -0.904522, -0.397999), (0.0554868, 0.142684, -0.796938, 0.584339), (-0.142684, 0.0554868, -0.584339, -0.796938), (0.560512, 0.246631, -0.77183, -0.171111), (-0.246631, 0.560512, 0.171111, -0.77183), (0.362102, 0.493844, -0.582869, -0.534101), (-0.493844, 0.362102, 0.534101, -0.582869), (0.0666671, 0.608733, -0.237729, -0.753979), (-0.608733, 0.0666671, 0.753979, -0.237729), (0.723617, 0.318399, 0.597857, 0.132542), (-0.318399, 0.723617, -0.132542, 0.597857), (0.467471, 0.63755, 0.451488, 0.413713), (-0.63755, 0.467471, -0.413713, 0.451488), (0.0860668, 0.785871, 0.184144, 0.58403), (-0.785871, 0.0860668, -0.58403, 0.184144), (0.770349, 0.338962, -0.531857, 0.0937807), (-0.338962, 0.770349, -0.0937807, -0.531857), (0.497661, 0.678724, -0.507492, -0.184712), (-0.678724, 0.497661, 0.184712, -0.507492), (0.0916251, 0.836623, -0.347145, -0.413711), (-0.836623, 0.0916251, 0.413711, -0.347145), (0.233546, 0.102763, 0.965345, -0.0548163), (-0.102763, 0.233546, 0.0548163, 0.965345), (0.150876, 0.205769, 0.863421, 0.4352), (-0.205769, 0.150876, -0.4352, 0.863421), (0.0277779, 0.253639, 0.530145, 0.808605), (-0.253639, 0.0277779, -0.808605, 0.530145), (0.233546, 0.102763, -0.965345, 0.0548163), (-0.102763, 0.233546, -0.0548163, -0.965345), (0.150876, 0.205769, -0.863421, -0.4352), (-0.205769, 0.150876, 0.4352, -0.863421), (0.0277779, 0.253639, -0.530145, -0.808605), (-0.253639, 0.0277779, 0.808605, -0.530145), (0.770349, 0.338962, 0.531857, -0.0937807), (-0.338962, 0.770349, 0.0937807, 0.531857), (0.497661, 0.678724, 0.507492, 0.184712), (-0.678724, 0.497661, -0.184712, 0.507492), (0.0916251, 0.836623, 0.347145, 0.413711), (-0.836623, 0.0916251, -0.413711, 0.347145), (0.723617, 0.318399, -0.132542, 0.597857), (-0.318399, 0.723617, -0.597857, -0.132542), (0.467471, 0.63755, -0.413713, 0.451488), (-0.63755, 0.467471, -0.451488, -0.413713), (0.0860668, 0.785871, -0.58403, 0.184144), (-0.785871, 0.0860668, -0.184144, -0.58403), (0.560512, 0.246631, 0.171111, -0.77183), (-0.246631, 0.560512, 0.77183, 0.171111), (0.362102, 0.493844, 0.534101, -0.582869), (-0.493844, 0.362102, 0.582869, 0.534101), (0.0666671, 0.608733, 0.753979, -0.237729), (-0.608733, 0.0666671, 0.237729, 0.753979), (0.494325, 0.217508, -0.146147, -0.828839), (-0.217508, 0.494325, 0.828839, -0.146147), (0.319344, 0.43553, 0.287853, -0.790869), (-0.43553, 0.319344, 0.790869, 0.287853), (0.0587948, 0.536852, 0.644722, -0.540986), (-0.536852, 0.0587948, 0.540986, 0.644722), (0.885015, 0.389416, 0.0144655, 0.254745), (-0.389416, 0.885015, -0.254745, 0.0144655), (0.571737, 0.779751, -0.114845, 0.227848), (-0.779751, 0.571737, -0.227848, -0.114845), (0.105263, 0.961153, -0.213383, 0.1399), (-0.961153, 0.105263, -0.1399, -0.213383), (0.885015, 0.389416, -0.0144655, -0.254745), (-0.389416, 0.885015, 0.254745, -0.0144655), (0.571737, 0.779751, 0.114845, -0.227848), (-0.779751, 0.571737, 0.227848, 0.114845), (0.105263, 0.961153, 0.213383, -0.1399), (-0.961153, 0.105263, 0.1399, 0.213383), (0.494325, 0.217508, 0.146147, 0.828839), (-0.217508, 0.494325, -0.828839, 0.146147), (0.319344, 0.43553, -0.287853, 0.790869), (-0.43553, 0.319344, -0.790869, -0.287853), (0.0587948, 0.536852, -0.644722, 0.540986), (-0.536852, 0.0587948, -0.540986, -0.644722), (0.770349, 0.338962, -0.52726, -0.116891), (-0.338962, 0.770349, 0.116891, -0.52726), (0.497661, 0.678724, -0.398175, -0.36486), (-0.678724, 0.497661, 0.36486, -0.398175), (0.0916251, 0.836623, -0.1624, -0.515066), (-0.836623, 0.0916251, 0.515066, -0.1624), (0.494325, 0.217508, 0.821676, 0.182161), (-0.217508, 0.494325, -0.182161, 0.821676), (0.319344, 0.43553, 0.620511, 0.568594), (-0.43553, 0.319344, -0.568594, 0.620511), (0.0587948, 0.536852, 0.253082, 0.802672), (-0.536852, 0.0587948, -0.802672, 0.253082), (0.904522, 0.397999, -0.151311, -0.023289), (-0.397999, 0.904522, 0.023289, -0.151311), (0.584339, 0.796938, -0.119395, -0.0958246), (-0.796938, 0.584339, 0.0958246, -0.119395), (0.107584, 0.982338, -0.0554868, -0.142684), (-0.982338, 0.107584, 0.142684, -0.0554868), (0.560512, 0.246631, 0.778559, -0.137281), (-0.246631, 0.560512, 0.137281, 0.778559), (0.362102, 0.493844, 0.742892, 0.270391), (-0.493844, 0.362102, -0.270391, 0.742892), (0.0666671, 0.608733, 0.508168, 0.605611), (-0.608733, 0.0666671, -0.605611, 0.508168), (0.560512, 0.246631, -0.778559, 0.137281), (-0.246631, 0.560512, -0.137281, -0.778559), (0.362102, 0.493844, -0.742892, -0.270391), (-0.493844, 0.362102, 0.270391, -0.742892), (0.0666671, 0.608733, -0.508168, -0.605611), (-0.608733, 0.0666671, 0.605611, -0.508168), (0.904522, 0.397999, 0.151311, 0.023289), (-0.397999, 0.904522, -0.023289, 0.151311), (0.584339, 0.796938, 0.119395, 0.0958246), (-0.796938, 0.584339, -0.0958246, 0.119395), (0.107584, 0.982338, 0.0554868, 0.142684), (-0.982338, 0.107584, -0.142684, 0.0554868), (0.494325, 0.217508, -0.182161, 0.821676), (-0.217508, 0.494325, -0.821676, -0.182161), (0.319344, 0.43553, -0.568594, 0.620511), (-0.43553, 0.319344, -0.620511, -0.568594), (0.0587948, 0.536852, -0.802672, 0.253082), (-0.536852, 0.0587948, -0.253082, -0.802672), (0.770349, 0.338962, 0.116891, -0.52726), (-0.338962, 0.770349, 0.52726, 0.116891), (0.497661, 0.678724, 0.36486, -0.398175), (-0.678724, 0.497661, 0.398175, 0.36486), (0.0916251, 0.836623, 0.515066, -0.1624), (-0.836623, 0.0916251, 0.1624, 0.515066), (0.140128, 0.0616577, 0.15033, -0.97671), (-0.0616577, 0.140128, 0.97671, 0.15033), (0.0905254, 0.123461, 0.618545, -0.770691), (-0.123461, 0.0905254, 0.770691, 0.618545), (0.0166668, 0.152183, 0.921021, -0.358166), (-0.152183, 0.0166668, 0.358166, 0.921021), (0.723617, 0.318399, 0.106337, 0.603069), (-0.318399, 0.723617, -0.603069, 0.106337), (0.467471, 0.63755, -0.209444, 0.575442), (-0.63755, 0.467471, -0.575442, -0.209444), (0.0860668, 0.785871, -0.469105, 0.393625), (-0.785871, 0.0860668, -0.393625, -0.469105), (0.723617, 0.318399, -0.106337, -0.603069), (-0.318399, 0.723617, 0.603069, -0.106337), (0.467471, 0.63755, 0.209444, -0.575442), (-0.63755, 0.467471, 0.575442, 0.209444), (0.0860668, 0.785871, 0.469105, -0.393625), (-0.785871, 0.0860668, 0.393625, 0.469105), (0.140128, 0.0616577, -0.15033, 0.97671), (-0.0616577, 0.140128, -0.97671, -0.15033), (0.0905254, 0.123461, -0.618545, 0.770691), (-0.123461, 0.0905254, -0.770691, -0.618545), (0.0166668, 0.152183, -0.921021, 0.358166), (-0.152183, 0.0166668, -0.358166, -0.921021), (0.605245, 0.0931562, -0.789817, 0.0344842), (-0.0931562, 0.605245, -0.0344842, -0.789817), (0.47758, 0.383298, -0.701244, -0.365044), (-0.383298, 0.47758, 0.365044, -0.701244), (0.221947, 0.570736, -0.424773, -0.666759), (-0.570736, 0.221947, 0.666759, -0.424773), (0.781368, 0.120264, 0.61179, -0.0267113), (-0.120264, 0.781368, 0.0267113, 0.61179), (0.616553, 0.494836, 0.543181, 0.282762), (-0.494836, 0.616553, -0.282762, 0.543181), (0.286532, 0.736817, 0.329027, 0.51647), (-0.736817, 0.286532, -0.51647, 0.329027), (0.83183, 0.128031, -0.489462, 0.22824), (-0.128031, 0.83183, -0.22824, -0.489462), (0.656371, 0.526793, -0.538007, -0.0470695), (-0.526793, 0.656371, 0.0470695, -0.538007), (0.305037, 0.784402, -0.442393, -0.309767), (-0.784402, 0.305037, 0.309767, -0.442393), (0.252186, 0.0388151, 0.918264, -0.302798), (-0.0388151, 0.252186, 0.302798, 0.918264), (0.198992, 0.159708, 0.946639, 0.196901), (-0.159708, 0.198992, -0.196901, 0.946639), (0.0924779, 0.237807, 0.721363, 0.643841), (-0.237807, 0.0924779, -0.643841, 0.721363), (0.252186, 0.0388151, -0.918264, 0.302798), (-0.0388151, 0.252186, -0.302798, -0.918264), (0.198992, 0.159708, -0.946639, -0.196901), (-0.159708, 0.198992, 0.196901, -0.946639), (0.0924779, 0.237807, -0.721363, -0.643841), (-0.237807, 0.0924779, 0.643841, -0.721363), (0.83183, 0.128031, 0.489462, -0.22824), (-0.128031, 0.83183, 0.22824, 0.489462), (0.656371, 0.526793, 0.538007, 0.0470695), (-0.526793, 0.656371, -0.0470695, 0.538007), (0.305037, 0.784402, 0.442393, 0.309767), (-0.784402, 0.305037, -0.309767, 0.442393), (0.781368, 0.120264, 0.0267113, 0.61179), (-0.120264, 0.781368, -0.61179, 0.0267113), (0.616553, 0.494836, -0.282762, 0.543181), (-0.494836, 0.616553, -0.543181, -0.282762), (0.286532, 0.736817, -0.51647, 0.329027), (-0.736817, 0.286532, -0.329027, -0.51647), (0.605245, 0.0931562, -0.0344842, -0.789817), (-0.0931562, 0.605245, 0.789817, -0.0344842), (0.47758, 0.383298, 0.365044, -0.701244), (-0.383298, 0.47758, 0.701244, 0.365044), (0.221947, 0.570736, 0.666759, -0.424773), (-0.570736, 0.221947, 0.424773, 0.666759), (0.533776, 0.082156, -0.355686, -0.762772), (-0.082156, 0.533776, 0.762772, -0.355686), (0.421186, 0.338037, 0.0733525, -0.838423), (-0.338037, 0.421186, 0.838423, 0.0733525), (0.195739, 0.503342, 0.482737, -0.689419), (-0.503342, 0.195739, 0.689419, 0.482737), (0.955647, 0.147088, 0.0799054, 0.242321), (-0.147088, 0.955647, -0.242321, 0.0799054), (0.75407, 0.605205, -0.0519602, 0.249809), (-0.605205, 0.75407, -0.249809, -0.0519602), (0.350441, 0.901159, -0.169903, 0.19036), (-0.901159, 0.350441, -0.19036, -0.169903), (0.955647, 0.147088, -0.0799054, -0.242321), (-0.147088, 0.955647, 0.242321, -0.0799054), (0.75407, 0.605205, 0.0519602, -0.249809), (-0.605205, 0.75407, 0.249809, 0.0519602), (0.350441, 0.901159, 0.169903, -0.19036), (-0.901159, 0.350441, 0.19036, 0.169903), (0.533776, 0.082156, 0.355686, 0.762772), (-0.082156, 0.533776, -0.762772, 0.355686), (0.421186, 0.338037, -0.0733525, 0.838423), (-0.338037, 0.421186, -0.838423, -0.0733525), (0.195739, 0.503342, -0.482737, 0.689419), (-0.503342, 0.195739, -0.689419, -0.482737), (0.727414, 0.11196, -0.613573, 0.286114), (-0.11196, 0.727414, -0.286114, -0.613573), (0.573979, 0.460667, -0.674427, -0.0590047), (-0.460667, 0.573979, 0.0590047, -0.674427), (0.266747, 0.685939, -0.554569, -0.388313), (-0.685939, 0.266747, 0.388313, -0.554569), (0.669124, 0.102988, 0.667024, -0.311039), (-0.102988, 0.669124, 0.311039, 0.667024), (0.527984, 0.423752, 0.733179, 0.0641449), (-0.423752, 0.527984, -0.0641449, 0.733179), (0.245372, 0.630972, 0.60288, 0.422141), (-0.630972, 0.245372, -0.422141, 0.60288), (0.923151, 0.142086, -0.356924, -0.0144693), (-0.142086, 0.923151, 0.0144693, -0.356924), (0.728429, 0.584626, -0.301871, -0.190993), (-0.584626, 0.728429, 0.190993, -0.301871), (0.338525, 0.870515, -0.165931, -0.31634), (-0.870515, 0.338525, 0.31634, -0.165931), (0.451123, 0.0694345, 0.88891, -0.0388106), (-0.0694345, 0.451123, 0.0388106, 0.88891), (0.355967, 0.285694, 0.789224, 0.410844), (-0.285694, 0.355967, -0.410844, 0.789224), (0.16543, 0.425401, 0.478066, 0.750413), (-0.425401, 0.16543, -0.750413, 0.478066), (0.451123, 0.0694345, -0.88891, 0.0388106), (-0.0694345, 0.451123, -0.0388106, -0.88891), (0.355967, 0.285694, -0.789224, -0.410844), (-0.285694, 0.355967, 0.410844, -0.789224), (0.16543, 0.425401, -0.478066, -0.750413), (-0.425401, 0.16543, 0.750413, -0.478066), (0.923151, 0.142086, 0.356924, 0.0144693), (-0.142086, 0.923151, -0.0144693, 0.356924), (0.728429, 0.584626, 0.301871, 0.190993), (-0.584626, 0.728429, -0.190993, 0.301871), (0.338525, 0.870515, 0.165931, 0.31634), (-0.870515, 0.338525, -0.31634, 0.165931), (0.669124, 0.102988, 0.311039, 0.667024), (-0.102988, 0.669124, -0.667024, 0.311039), (0.527984, 0.423752, -0.0641449, 0.733179), (-0.423752, 0.527984, -0.733179, -0.0641449), (0.245372, 0.630972, -0.422141, 0.60288), (-0.630972, 0.245372, -0.60288, -0.422141), (0.727414, 0.11196, -0.286114, -0.613573), (-0.11196, 0.727414, 0.613573, -0.286114), (0.573979, 0.460667, 0.0590047, -0.674427), (-0.460667, 0.573979, 0.674427, 0.0590047), (0.266747, 0.685939, 0.388313, -0.554569), (-0.685939, 0.266747, 0.554569, 0.388313), (0.35306, 0.0543411, 0.037833, -0.933255), (-0.0543411, 0.35306, 0.933255, 0.037833), (0.278588, 0.223591, 0.499392, -0.789306), (-0.223591, 0.278588, 0.789306, 0.499392), (0.129469, 0.332929, 0.827139, -0.433863), (-0.332929, 0.129469, 0.433863, 0.827139), (0.879401, 0.135353, 0.0199094, 0.456001), (-0.135353, 0.879401, -0.456001, 0.0199094), (0.693907, 0.556919, -0.210758, 0.404863), (-0.556919, 0.693907, -0.404863, -0.210758), (0.322482, 0.82926, -0.384954, 0.245243), (-0.82926, 0.322482, -0.245243, -0.384954), (0.879401, 0.135353, -0.0199094, -0.456001), (-0.135353, 0.879401, 0.456001, -0.0199094), (0.693907, 0.556919, 0.210758, -0.404863), (-0.556919, 0.693907, 0.404863, 0.210758), (0.322482, 0.82926, 0.384954, -0.245243), (-0.82926, 0.322482, 0.245243, 0.384954), (0.35306, 0.0543411, -0.037833, 0.933255), (-0.0543411, 0.35306, -0.933255, -0.037833), (0.278588, 0.223591, -0.499392, 0.789306), (-0.223591, 0.278588, -0.789306, -0.499392), (0.129469, 0.332929, -0.827139, 0.433863), (-0.332929, 0.129469, -0.433863, -0.827139), (0.673651, 0.296414, -0.554569, -0.388313), (-0.296414, 0.673651, 0.388313, -0.554569), (0.435192, 0.593527, -0.286114, -0.613573), (-0.593527, 0.435192, 0.613573, -0.286114), (0.0801239, 0.731606, 0.0590047, -0.674427), (-0.731606, 0.0801239, 0.674427, 0.0590047), (0.619669, 0.272661, 0.60288, 0.422141), (-0.272661, 0.619669, -0.422141, 0.60288), (0.400319, 0.545966, 0.311039, 0.667024), (-0.545966, 0.400319, -0.667024, 0.311039), (0.0737032, 0.672979, -0.0641449, 0.733179), (-0.672979, 0.0737032, -0.733179, -0.0641449), (0.85492, 0.376174, -0.295772, 0.200307), (-0.376174, 0.85492, -0.200307, -0.295772), (0.552296, 0.753236, -0.3563, 0.0255845), (-0.753236, 0.552296, -0.0255845, -0.3563), (0.101684, 0.92847, -0.321357, -0.155993), (-0.92847, 0.101684, 0.155993, -0.321357), (0.417781, 0.183828, 0.750413, -0.478066), (-0.183828, 0.417781, 0.478066, 0.750413), (0.269895, 0.36809, 0.88891, -0.0388106), (-0.36809, 0.269895, 0.0388106, 0.88891), (0.0496907, 0.453723, 0.789224, 0.410844), (-0.453723, 0.0496907, -0.410844, 0.789224), (0.417781, 0.183828, -0.750413, 0.478066), (-0.183828, 0.417781, -0.478066, -0.750413), (0.269895, 0.36809, -0.88891, 0.0388106), (-0.36809, 0.269895, -0.0388106, -0.88891), (0.0496907, 0.453723, -0.789224, -0.410844), (-0.453723, 0.0496907, 0.410844, -0.789224), (0.85492, 0.376174, 0.295772, -0.200307), (-0.376174, 0.85492, 0.200307, 0.295772), (0.552296, 0.753236, 0.3563, -0.0255845), (-0.753236, 0.552296, 0.0255845, 0.3563), (0.101684, 0.92847, 0.321357, 0.155993), (-0.92847, 0.101684, -0.155993, 0.321357), (0.619669, 0.272661, -0.422141, 0.60288), (-0.272661, 0.619669, -0.60288, -0.422141), (0.400319, 0.545966, -0.667024, 0.311039), (-0.545966, 0.400319, -0.311039, -0.667024), (0.0737032, 0.672979, -0.733179, -0.0641449), (-0.672979, 0.0737032, 0.0641449, -0.733179), (0.673651, 0.296414, 0.388313, -0.554569), (-0.296414, 0.673651, 0.554569, 0.388313), (0.435192, 0.593527, 0.613573, -0.286114), (-0.593527, 0.435192, 0.286114, 0.613573), (0.0801239, 0.731606, 0.674427, 0.0590047), (-0.731606, 0.0801239, -0.0590047, 0.674427), (0.326965, 0.143868, -0.523745, -0.773361), (-0.143868, 0.326965, 0.773361, -0.523745), (0.211226, 0.288076, -0.0668963, -0.931623), (-0.288076, 0.211226, 0.931623, -0.0668963), (0.0388891, 0.355094, 0.407877, -0.840257), (-0.355094, 0.0388891, 0.840257, 0.407877), (0.814404, 0.358346, 0.245243, 0.384954), (-0.358346, 0.814404, -0.384954, 0.245243), (0.526122, 0.717539, 0.0199094, 0.456001), (-0.717539, 0.526122, -0.456001, 0.0199094), (0.096865, 0.884468, -0.210758, 0.404863), (-0.884468, 0.096865, -0.404863, -0.210758), (0.814404, 0.358346, -0.245243, -0.384954), (-0.358346, 0.814404, 0.384954, -0.245243), (0.526122, 0.717539, -0.0199094, -0.456001), (-0.717539, 0.526122, 0.456001, -0.0199094), (0.096865, 0.884468, 0.210758, -0.404863), (-0.884468, 0.096865, 0.404863, 0.210758), (0.326965, 0.143868, 0.523745, 0.773361), (-0.143868, 0.326965, -0.773361, 0.523745), (0.211226, 0.288076, 0.0668963, 0.931623), (-0.288076, 0.211226, -0.931623, 0.0668963), (0.0388891, 0.355094, -0.407877, 0.840257), (-0.355094, 0.0388891, -0.840257, -0.407877), (0.673651, 0.296414, -0.666718, 0.11756), (-0.296414, 0.673651, -0.11756, -0.666718), (0.435192, 0.593527, -0.636175, -0.231549), (-0.593527, 0.435192, 0.231549, -0.636175), (0.0801239, 0.731606, -0.435169, -0.518615), (-0.731606, 0.0801239, 0.518615, -0.435169), (0.619669, 0.272661, 0.724799, -0.127802), (-0.272661, 0.619669, 0.127802, 0.724799), (0.400319, 0.545966, 0.691595, 0.25172), (-0.545966, 0.400319, -0.25172, 0.691595), (0.0737032, 0.672979, 0.473079, 0.563793), (-0.672979, 0.0737032, -0.563793, 0.473079), (0.85492, 0.376174, -0.341017, -0.106355), (-0.376174, 0.85492, 0.106355, -0.341017), (0.552296, 0.753236, -0.242152, -0.262615), (-0.753236, 0.552296, 0.262615, -0.242152), (0.101684, 0.92847, -0.0784025, -0.348507), (-0.92847, 0.101684, 0.348507, -0.0784025), (0.417781, 0.183828, 0.868666, 0.192579), (-0.183828, 0.417781, -0.192579, 0.868666), (0.269895, 0.36809, 0.655997, 0.601111), (-0.36809, 0.269895, -0.601111, 0.655997), (0.0496907, 0.453723, 0.267555, 0.848576), (-0.453723, 0.0496907, -0.848576, 0.267555), (0.417781, 0.183828, -0.868666, -0.192579), (-0.183828, 0.417781, 0.192579, -0.868666), (0.269895, 0.36809, -0.655997, -0.601111), (-0.36809, 0.269895, 0.601111, -0.655997), (0.0496907, 0.453723, -0.267555, -0.848576), (-0.453723, 0.0496907, 0.848576, -0.267555), (0.85492, 0.376174, 0.341017, 0.106355), (-0.376174, 0.85492, -0.106355, 0.341017), (0.552296, 0.753236, 0.242152, 0.262615), (-0.753236, 0.552296, -0.262615, 0.242152), (0.101684, 0.92847, 0.0784025, 0.348507), (-0.92847, 0.101684, -0.348507, 0.0784025), (0.619669, 0.272661, 0.127802, 0.724799), (-0.272661, 0.619669, -0.724799, 0.127802), (0.400319, 0.545966, -0.25172, 0.691595), (-0.545966, 0.400319, -0.691595, -0.25172), (0.0737032, 0.672979, -0.563793, 0.473079), (-0.672979, 0.0737032, -0.473079, -0.563793), (0.673651, 0.296414, -0.11756, -0.666718), (-0.296414, 0.673651, 0.666718, -0.11756), (0.435192, 0.593527, 0.231549, -0.636175), (-0.593527, 0.435192, 0.636175, 0.231549), (0.0801239, 0.731606, 0.518615, -0.435169), (-0.731606, 0.0801239, 0.435169, 0.518615), (0.326965, 0.143868, 0.278088, -0.891663), (-0.143868, 0.326965, 0.891663, 0.278088), (0.211226, 0.288076, 0.686663, -0.633159), (-0.288076, 0.211226, 0.633159, 0.686663), (0.0388891, 0.355094, 0.911247, -0.205), (-0.355094, 0.0388891, 0.205, 0.911247), (0.814404, 0.358346, -0.0987907, 0.445616), (-0.358346, 0.814404, -0.445616, -0.0987907), (0.526122, 0.717539, -0.308363, 0.33652), (-0.717539, 0.526122, -0.33652, -0.308363), (0.096865, 0.884468, -0.43531, 0.137253), (-0.884468, 0.096865, -0.137253, -0.43531), (0.814404, 0.358346, 0.0987907, -0.445616), (-0.358346, 0.814404, 0.445616, 0.0987907), (0.526122, 0.717539, 0.308363, -0.33652), (-0.717539, 0.526122, 0.33652, 0.308363), (0.096865, 0.884468, 0.43531, -0.137253), (-0.884468, 0.096865, 0.137253, 0.43531), (0.326965, 0.143868, -0.278088, 0.891663), (-0.143868, 0.326965, -0.891663, -0.278088), (0.211226, 0.288076, -0.686663, 0.633159), (-0.288076, 0.211226, -0.633159, -0.686663), (0.0388891, 0.355094, -0.911247, 0.205), (-0.355094, 0.0388891, -0.205, -0.911247), (0.727414, 0.11196, -0.636175, -0.231549), (-0.11196, 0.727414, 0.231549, -0.636175), (0.573979, 0.460667, -0.435169, -0.518615), (-0.460667, 0.573979, 0.518615, -0.435169), (0.266747, 0.685939, -0.11756, -0.666718), (-0.685939, 0.266747, 0.666718, -0.11756), (0.669124, 0.102988, 0.691595, 0.25172), (-0.102988, 0.669124, -0.25172, 0.691595), (0.527984, 0.423752, 0.473079, 0.563793), (-0.423752, 0.527984, -0.563793, 0.473079), (0.245372, 0.630972, 0.127802, 0.724799), (-0.630972, 0.245372, -0.724799, 0.127802), (0.923151, 0.142086, -0.233851, 0.270033), (-0.142086, 0.923151, -0.270033, -0.233851), (0.728429, 0.584626, -0.337537, 0.11693), (-0.584626, 0.728429, -0.11693, -0.337537), (0.338525, 0.870515, -0.350781, -0.0675044), (-0.870515, 0.338525, 0.0675044, -0.350781), (0.451123, 0.0694345, 0.601111, -0.655997), (-0.0694345, 0.451123, 0.655997, 0.601111), (0.355967, 0.285694, 0.848576, -0.267555), (-0.285694, 0.355967, 0.267555, 0.848576), (0.16543, 0.425401, 0.868666, 0.192579), (-0.425401, 0.16543, -0.192579, 0.868666), (0.451123, 0.0694345, -0.601111, 0.655997), (-0.0694345, 0.451123, -0.655997, -0.601111), (0.355967, 0.285694, -0.848576, 0.267555), (-0.285694, 0.355967, -0.267555, -0.848576), (0.16543, 0.425401, -0.868666, -0.192579), (-0.425401, 0.16543, 0.192579, -0.868666), (0.923151, 0.142086, 0.233851, -0.270033), (-0.142086, 0.923151, 0.270033, 0.233851), (0.728429, 0.584626, 0.337537, -0.11693), (-0.584626, 0.728429, 0.11693, 0.337537), (0.338525, 0.870515, 0.350781, 0.0675044), (-0.870515, 0.338525, -0.0675044, 0.350781), (0.669124, 0.102988, -0.25172, 0.691595), (-0.102988, 0.669124, -0.691595, -0.25172), (0.527984, 0.423752, -0.563793, 0.473079), (-0.423752, 0.527984, -0.473079, -0.563793), (0.245372, 0.630972, -0.724799, 0.127802), (-0.630972, 0.245372, -0.127802, -0.724799), (0.727414, 0.11196, 0.231549, -0.636175), (-0.11196, 0.727414, 0.636175, 0.231549), (0.573979, 0.460667, 0.518615, -0.435169), (-0.460667, 0.573979, 0.435169, 0.518615), (0.266747, 0.685939, 0.666718, -0.11756), (-0.685939, 0.266747, 0.11756, 0.666718), (0.35306, 0.0543411, -0.706059, -0.611454), (-0.0543411, 0.35306, 0.611454, -0.706059), (0.278588, 0.223591, -0.305739, -0.882564), (-0.223591, 0.278588, 0.882564, -0.305739), (0.129469, 0.332929, 0.176505, -0.917192), (-0.332929, 0.129469, 0.917192, 0.176505), (0.879401, 0.135353, 0.33652, 0.308363), (-0.135353, 0.879401, -0.308363, 0.33652), (0.693907, 0.556919, 0.137253, 0.43531), (-0.556919, 0.693907, -0.43531, 0.137253), (0.322482, 0.82926, -0.0987907, 0.445616), (-0.82926, 0.322482, -0.445616, -0.0987907), (0.879401, 0.135353, -0.33652, -0.308363), (-0.135353, 0.879401, 0.308363, -0.33652), (0.693907, 0.556919, -0.137253, -0.43531), (-0.556919, 0.693907, 0.43531, -0.137253), (0.322482, 0.82926, 0.0987907, -0.445616), (-0.82926, 0.322482, 0.445616, 0.0987907), (0.35306, 0.0543411, 0.706059, 0.611454), (-0.0543411, 0.35306, -0.611454, 0.706059), (0.278588, 0.223591, 0.305739, 0.882564), (-0.223591, 0.278588, -0.882564, 0.305739), (0.129469, 0.332929, -0.176505, 0.917192), (-0.332929, 0.129469, -0.917192, -0.176505), (0.841425, 0.01836, -0.52726, -0.116891), (-0.01836, 0.841425, 0.116891, -0.52726), (0.719516, 0.436613, -0.398175, -0.36486), (-0.436613, 0.719516, 0.36486, -0.398175), (0.404812, 0.737876, -0.1624, -0.515066), (-0.737876, 0.404812, 0.515066, -0.1624), (0.539933, 0.0117814, 0.821676, 0.182161), (-0.0117814, 0.539933, -0.182161, 0.821676), (0.461705, 0.28017, 0.620511, 0.568594), (-0.28017, 0.461705, -0.568594, 0.620511), (0.259764, 0.473487, 0.253082, 0.802672), (-0.473487, 0.259764, -0.802672, 0.253082), (0.987977, 0.0215577, -0.0428399, 0.146977), (-0.0215577, 0.987977, -0.146977, -0.0428399), (0.844834, 0.512658, -0.110589, 0.105866), (-0.512658, 0.844834, -0.105866, -0.110589), (0.475319, 0.866392, -0.148706, 0.0363881), (-0.866392, 0.475319, -0.0363881, -0.148706), (0.612227, 0.0133588, 0.453452, -0.647597), (-0.0133588, 0.612227, 0.647597, 0.453452), (0.523524, 0.317682, 0.716499, -0.334109), (-0.317682, 0.523524, 0.334109, 0.716499), (0.294544, 0.536883, 0.787561, 0.0689027), (-0.536883, 0.294544, -0.0689027, 0.787561), (0.612227, 0.0133588, -0.453452, 0.647597), (-0.0133588, 0.612227, -0.647597, -0.453452), (0.523524, 0.317682, -0.716499, 0.334109), (-0.317682, 0.523524, -0.334109, -0.716499), (0.294544, 0.536883, -0.787561, -0.0689027), (-0.536883, 0.294544, 0.0689027, -0.787561), (0.987977, 0.0215577, 0.0428399, -0.146977), (-0.0215577, 0.987977, 0.146977, 0.0428399), (0.844834, 0.512658, 0.110589, -0.105866), (-0.512658, 0.844834, 0.105866, 0.110589), (0.475319, 0.866392, 0.148706, -0.0363881), (-0.866392, 0.475319, 0.0363881, 0.148706), (0.539933, 0.0117814, -0.182161, 0.821676), (-0.0117814, 0.539933, -0.821676, -0.182161), (0.461705, 0.28017, -0.568594, 0.620511), (-0.28017, 0.461705, -0.620511, -0.568594), (0.259764, 0.473487, -0.802672, 0.253082), (-0.473487, 0.259764, -0.253082, -0.802672), (0.841425, 0.01836, 0.116891, -0.52726), (-0.01836, 0.841425, 0.52726, 0.116891), (0.719516, 0.436613, 0.36486, -0.398175), (-0.436613, 0.719516, 0.398175, 0.36486), (0.404812, 0.737876, 0.515066, -0.1624), (-0.737876, 0.404812, 0.1624, 0.515066), (0.153057, 0.00333971, -0.948733, -0.27653), (-0.00333971, 0.153057, 0.27653, -0.948733), (0.130881, 0.0794206, -0.683361, -0.713849), (-0.0794206, 0.130881, 0.713849, -0.683361), (0.0736361, 0.134221, -0.234884, -0.959892), (-0.134221, 0.0736361, 0.959892, -0.234884), (0.790381, 0.0172462, 0.501626, 0.351242), (-0.0172462, 0.790381, -0.351242, 0.501626), (0.675867, 0.410126, 0.2588, 0.554998), (-0.410126, 0.675867, -0.554998, 0.2588), (0.380255, 0.693113, -0.0533718, 0.610042), (-0.693113, 0.380255, -0.610042, -0.0533718), (0.790381, 0.0172462, -0.501626, -0.351242), (-0.0172462, 0.790381, 0.351242, -0.501626), (0.675867, 0.410126, -0.2588, -0.554998), (-0.410126, 0.675867, 0.554998, -0.2588), (0.380255, 0.693113, 0.0533718, -0.610042), (-0.693113, 0.380255, 0.610042, 0.0533718), (0.153057, 0.00333971, 0.948733, 0.27653), (-0.00333971, 0.153057, -0.27653, 0.948733), (0.130881, 0.0794206, 0.683361, 0.713849), (-0.0794206, 0.130881, -0.713849, 0.683361), (0.0736361, 0.134221, 0.234884, 0.959892), (-0.134221, 0.0736361, -0.959892, 0.234884), (0.587908, 0.17136, -0.701244, -0.365044), (-0.17136, 0.587908, 0.365044, -0.701244), (0.423464, 0.442356, -0.424773, -0.666759), (-0.442356, 0.423464, 0.666759, -0.424773), (0.145552, 0.594823, -0.0344842, -0.789817), (-0.594823, 0.145552, 0.789817, -0.0344842), (0.758986, 0.221224, 0.543181, 0.282762), (-0.221224, 0.758986, -0.282762, 0.543181), (0.546689, 0.571079, 0.329027, 0.51647), (-0.571079, 0.546689, -0.51647, 0.329027), (0.187907, 0.767913, 0.0267113, 0.61179), (-0.767913, 0.187907, -0.61179, 0.0267113), (0.808002, 0.235511, -0.413711, 0.347145), (-0.235511, 0.808002, -0.347145, -0.413711), (0.581995, 0.60796, -0.531857, 0.0937807), (-0.60796, 0.581995, -0.0937807, -0.531857), (0.200042, 0.817506, -0.507492, -0.184712), (-0.817506, 0.200042, 0.184712, -0.507492), (0.244962, 0.0713998, 0.662504, -0.704262), (-0.0713998, 0.244962, 0.704262, 0.662504), (0.176443, 0.184315, 0.925876, -0.278657), (-0.184315, 0.176443, 0.278657, 0.925876), (0.0606468, 0.247843, 0.941161, 0.221614), (-0.247843, 0.0606468, -0.221614, 0.941161), (0.244962, 0.0713998, -0.662504, 0.704262), (-0.0713998, 0.244962, -0.704262, -0.662504), (0.176443, 0.184315, -0.925876, 0.278657), (-0.184315, 0.176443, -0.278657, -0.925876), (0.0606468, 0.247843, -0.941161, -0.221614), (-0.247843, 0.0606468, 0.221614, -0.941161), (0.808002, 0.235511, 0.413711, -0.347145), (-0.235511, 0.808002, 0.347145, 0.413711), (0.581995, 0.60796, 0.531857, -0.0937807), (-0.60796, 0.581995, 0.0937807, 0.531857), (0.200042, 0.817506, 0.507492, 0.184712), (-0.817506, 0.200042, -0.184712, 0.507492), (0.758986, 0.221224, -0.282762, 0.543181), (-0.221224, 0.758986, -0.543181, -0.282762), (0.546689, 0.571079, -0.51647, 0.329027), (-0.571079, 0.546689, -0.329027, -0.51647), (0.187907, 0.767913, -0.61179, 0.0267113), (-0.767913, 0.187907, -0.0267113, -0.61179), (0.587908, 0.17136, 0.365044, -0.701244), (-0.17136, 0.587908, 0.701244, 0.365044), (0.423464, 0.442356, 0.666759, -0.424773), (-0.442356, 0.423464, 0.424773, 0.666759), (0.145552, 0.594823, 0.789817, -0.0344842), (-0.594823, 0.145552, 0.0344842, 0.789817), (0.518486, 0.151125, -0.540986, -0.644722), (-0.151125, 0.518486, 0.644722, -0.540986), (0.37346, 0.390121, -0.146147, -0.828839), (-0.390121, 0.37346, 0.828839, -0.146147), (0.128365, 0.524585, 0.287853, -0.790869), (-0.524585, 0.128365, 0.790869, 0.287853), (0.928272, 0.270567, 0.185848, 0.174828), (-0.270567, 0.928272, -0.174828, 0.185848), (0.668624, 0.698454, 0.0735348, 0.244329), (-0.698454, 0.668624, -0.244329, 0.0735348), (0.229819, 0.939191, -0.0584817, 0.248363), (-0.939191, 0.229819, -0.248363, -0.0584817), (0.928272, 0.270567, -0.185848, -0.174828), (-0.270567, 0.928272, 0.174828, -0.185848), (0.668624, 0.698454, -0.0735348, -0.244329), (-0.698454, 0.668624, 0.244329, -0.0735348), (0.229819, 0.939191, 0.0584817, -0.248363), (-0.939191, 0.229819, 0.248363, 0.0584817), (0.518486, 0.151125, 0.540986, 0.644722), (-0.151125, 0.518486, -0.644722, 0.540986), (0.37346, 0.390121, 0.146147, 0.828839), (-0.390121, 0.37346, -0.828839, 0.146147), (0.128365, 0.524585, -0.287853, 0.790869), (-0.524585, 0.128365, -0.790869, -0.287853), (0.808002, 0.235511, -0.479041, -0.249373), (-0.235511, 0.808002, 0.249373, -0.479041), (0.581995, 0.60796, -0.290175, -0.455483), (-0.60796, 0.581995, 0.455483, -0.290175), (0.200042, 0.817506, -0.0235572, -0.539548), (-0.817506, 0.200042, 0.539548, -0.0235572), (0.518486, 0.151125, 0.746531, 0.388619), (-0.151125, 0.518486, -0.388619, 0.746531), (0.37346, 0.390121, 0.452205, 0.70982), (-0.390121, 0.37346, -0.70982, 0.452205), (0.128365, 0.524585, 0.0367112, 0.840824), (-0.524585, 0.128365, -0.840824, 0.0367112), (0.948733, 0.27653, -0.0794206, 0.130881), (-0.27653, 0.948733, -0.130881, -0.0794206), (0.683361, 0.713849, -0.134221, 0.0736361), (-0.713849, 0.683361, -0.0736361, -0.134221), (0.234884, 0.959892, -0.153057, -0.00333971), (-0.959892, 0.234884, 0.00333971, -0.153057), (0.587908, 0.17136, 0.605611, -0.508168), (-0.17136, 0.587908, 0.508168, 0.605611), (0.423464, 0.442356, 0.778559, -0.137281), (-0.442356, 0.423464, 0.137281, 0.778559), (0.145552, 0.594823, 0.742892, 0.270391), (-0.594823, 0.145552, -0.270391, 0.742892), (0.587908, 0.17136, -0.605611, 0.508168), (-0.17136, 0.587908, -0.508168, -0.605611), (0.423464, 0.442356, -0.778559, 0.137281), (-0.442356, 0.423464, -0.137281, -0.778559), (0.145552, 0.594823, -0.742892, -0.270391), (-0.594823, 0.145552, 0.270391, -0.742892), (0.948733, 0.27653, 0.0794206, -0.130881), (-0.27653, 0.948733, 0.130881, 0.0794206), (0.683361, 0.713849, 0.134221, -0.0736361), (-0.713849, 0.683361, 0.0736361, 0.134221), (0.234884, 0.959892, 0.153057, 0.00333971), (-0.959892, 0.234884, -0.00333971, 0.153057), (0.518486, 0.151125, -0.388619, 0.746531), (-0.151125, 0.518486, -0.746531, -0.388619), (0.37346, 0.390121, -0.70982, 0.452205), (-0.390121, 0.37346, -0.452205, -0.70982), (0.128365, 0.524585, -0.840824, 0.0367112), (-0.524585, 0.128365, -0.0367112, -0.840824), (0.808002, 0.235511, 0.249373, -0.479041), (-0.235511, 0.808002, 0.479041, 0.249373), (0.581995, 0.60796, 0.455483, -0.290175), (-0.60796, 0.581995, 0.290175, 0.455483), (0.200042, 0.817506, 0.539548, -0.0235572), (-0.817506, 0.200042, 0.0235572, 0.539548), (0.146977, 0.0428399, -0.844834, -0.512658), (-0.0428399, 0.146977, 0.512658, -0.844834), (0.105866, 0.110589, -0.475319, -0.866392), (-0.110589, 0.105866, 0.866392, -0.475319), (0.0363881, 0.148706, 0.0215577, -0.987977), (-0.148706, 0.0363881, 0.987977, 0.0215577), (0.758986, 0.221224, 0.393625, 0.469105), (-0.221224, 0.758986, -0.469105, 0.393625), (0.546689, 0.571079, 0.106337, 0.603069), (-0.571079, 0.546689, -0.603069, 0.106337), (0.187907, 0.767913, -0.209444, 0.575442), (-0.767913, 0.187907, -0.575442, -0.209444), (0.758986, 0.221224, -0.393625, -0.469105), (-0.221224, 0.758986, 0.469105, -0.393625), (0.546689, 0.571079, -0.106337, -0.603069), (-0.571079, 0.546689, 0.603069, -0.106337), (0.187907, 0.767913, 0.209444, -0.575442), (-0.767913, 0.187907, 0.575442, 0.209444), (0.146977, 0.0428399, 0.844834, 0.512658), (-0.0428399, 0.146977, -0.512658, 0.844834), (0.105866, 0.110589, 0.475319, 0.866392), (-0.110589, 0.105866, -0.866392, 0.475319), (0.0363881, 0.148706, -0.0215577, 0.987977), (-0.148706, 0.0363881, -0.987977, -0.0215577), (0.612227, 0.0133588, -0.77183, -0.171111), (-0.0133588, 0.612227, 0.171111, -0.77183), (0.523524, 0.317682, -0.582869, -0.534101), (-0.317682, 0.523524, 0.534101, -0.582869), (0.294544, 0.536883, -0.237729, -0.753979), (-0.536883, 0.294544, 0.753979, -0.237729), (0.790381, 0.0172462, 0.597857, 0.132542), (-0.0172462, 0.790381, -0.132542, 0.597857), (0.675867, 0.410126, 0.451488, 0.413713), (-0.410126, 0.675867, -0.413713, 0.451488), (0.380255, 0.693113, 0.184144, 0.58403), (-0.693113, 0.380255, -0.58403, 0.184144), (0.841425, 0.01836, -0.309767, 0.442393), (-0.01836, 0.841425, -0.442393, -0.309767), (0.719516, 0.436613, -0.489462, 0.22824), (-0.436613, 0.719516, -0.22824, -0.489462), (0.404812, 0.737876, -0.538007, -0.0470695), (-0.737876, 0.404812, 0.0470695, -0.538007), (0.255094, 0.00556618, 0.457653, -0.851733), (-0.00556618, 0.255094, 0.851733, 0.457653), (0.218135, 0.132368, 0.822206, -0.508796), (-0.132368, 0.218135, 0.508796, 0.822206), (0.122727, 0.223701, 0.966449, -0.0295277), (-0.223701, 0.122727, 0.0295277, 0.966449), (0.255094, 0.00556618, -0.457653, 0.851733), (-0.00556618, 0.255094, -0.851733, -0.457653), (0.218135, 0.132368, -0.822206, 0.508796), (-0.132368, 0.218135, -0.508796, -0.822206), (0.122727, 0.223701, -0.966449, 0.0295277), (-0.223701, 0.122727, -0.0295277, -0.966449), (0.841425, 0.01836, 0.309767, -0.442393), (-0.01836, 0.841425, 0.442393, 0.309767), (0.719516, 0.436613, 0.489462, -0.22824), (-0.436613, 0.719516, 0.22824, 0.489462), (0.404812, 0.737876, 0.538007, 0.0470695), (-0.737876, 0.404812, -0.0470695, 0.538007), (0.790381, 0.0172462, -0.132542, 0.597857), (-0.0172462, 0.790381, -0.597857, -0.132542), (0.675867, 0.410126, -0.413713, 0.451488), (-0.410126, 0.675867, -0.451488, -0.413713), (0.380255, 0.693113, -0.58403, 0.184144), (-0.693113, 0.380255, -0.184144, -0.58403), (0.612227, 0.0133588, 0.171111, -0.77183), (-0.0133588, 0.612227, 0.77183, 0.171111), (0.523524, 0.317682, 0.534101, -0.582869), (-0.317682, 0.523524, 0.582869, 0.534101), (0.294544, 0.536883, 0.753979, -0.237729), (-0.536883, 0.294544, 0.237729, 0.753979), (0.539933, 0.0117814, -0.689419, -0.482737), (-0.0117814, 0.539933, 0.482737, -0.689419), (0.461705, 0.28017, -0.355686, -0.762772), (-0.28017, 0.461705, 0.762772, -0.355686), (0.259764, 0.473487, 0.0733525, -0.838423), (-0.473487, 0.259764, 0.838423, 0.0733525), (0.96667, 0.0210928, 0.224764, 0.12077), (-0.0210928, 0.96667, -0.12077, 0.224764), (0.826614, 0.501602, 0.134266, 0.216972), (-0.501602, 0.826614, -0.216972, 0.134266), (0.465068, 0.847707, 0.00779206, 0.255036), (-0.847707, 0.465068, -0.255036, 0.00779206), (0.96667, 0.0210928, -0.224764, -0.12077), (-0.0210928, 0.96667, 0.12077, -0.224764), (0.826614, 0.501602, -0.134266, -0.216972), (-0.501602, 0.826614, 0.216972, -0.134266), (0.465068, 0.847707, -0.00779206, -0.255036), (-0.847707, 0.465068, 0.255036, -0.00779206), (0.539933, 0.0117814, 0.689419, 0.482737), (-0.0117814, 0.539933, -0.482737, 0.689419), (0.461705, 0.28017, 0.355686, 0.762772), (-0.28017, 0.461705, -0.762772, 0.355686), (0.259764, 0.473487, -0.0733525, 0.838423), (-0.473487, 0.259764, -0.838423, -0.0733525), (0.735805, 0.0160553, -0.666718, 0.11756), (-0.0160553, 0.735805, -0.11756, -0.666718), (0.629198, 0.381807, -0.636175, -0.231549), (-0.381807, 0.629198, 0.231549, -0.636175), (0.353998, 0.645253, -0.435169, -0.518615), (-0.645253, 0.353998, 0.518615, -0.435169), (0.676842, 0.0147687, 0.724799, -0.127802), (-0.0147687, 0.676842, 0.127802, 0.724799), (0.578778, 0.351211, 0.691595, 0.25172), (-0.351211, 0.578778, -0.25172, 0.691595), (0.325631, 0.593547, 0.473079, 0.563793), (-0.593547, 0.325631, -0.563793, 0.473079), (0.933799, 0.0203756, -0.306538, 0.183408), (-0.0203756, 0.933799, -0.183408, -0.306538), (0.798506, 0.484545, -0.357174, 0.0055664), (-0.484545, 0.798506, -0.0055664, -0.357174), (0.449254, 0.818881, -0.312105, -0.173766), (-0.818881, 0.449254, 0.173766, -0.312105), (0.456327, 0.00995709, 0.750413, -0.478066), (-0.00995709, 0.456327, 0.478066, 0.750413), (0.390212, 0.236787, 0.88891, -0.0388106), (-0.236787, 0.390212, 0.0388106, 0.88891), (0.21954, 0.400169, 0.789224, 0.410844), (-0.400169, 0.21954, -0.410844, 0.789224), (0.456327, 0.00995709, -0.750413, 0.478066), (-0.00995709, 0.456327, -0.478066, -0.750413), (0.390212, 0.236787, -0.88891, 0.0388106), (-0.236787, 0.390212, -0.0388106, -0.88891), (0.21954, 0.400169, -0.789224, -0.410844), (-0.400169, 0.21954, 0.410844, -0.789224), (0.933799, 0.0203756, 0.306538, -0.183408), (-0.0203756, 0.933799, 0.183408, 0.306538), (0.798506, 0.484545, 0.357174, -0.0055664), (-0.484545, 0.798506, 0.0055664, 0.357174), (0.449254, 0.818881, 0.312105, 0.173766), (-0.818881, 0.449254, -0.173766, 0.312105), (0.676842, 0.0147687, 0.127802, 0.724799), (-0.0147687, 0.676842, -0.724799, 0.127802), (0.578778, 0.351211, -0.25172, 0.691595), (-0.351211, 0.578778, -0.691595, -0.25172), (0.325631, 0.593547, -0.563793, 0.473079), (-0.593547, 0.325631, -0.473079, -0.563793), (0.735805, 0.0160553, -0.11756, -0.666718), (-0.0160553, 0.735805, 0.666718, -0.11756), (0.629198, 0.381807, 0.231549, -0.636175), (-0.381807, 0.629198, 0.636175, 0.231549), (0.353998, 0.645253, 0.518615, -0.435169), (-0.645253, 0.353998, 0.435169, 0.518615), (0.357132, 0.00779265, -0.479559, -0.801511), (-0.00779265, 0.357132, 0.801511, -0.479559), (0.305389, 0.185315, -0.0145545, -0.933908), (-0.185315, 0.305389, 0.933908, -0.0145545), (0.171817, 0.313182, 0.454349, -0.816065), (-0.313182, 0.171817, 0.816065, 0.454349), (0.889545, 0.0194099, 0.245243, 0.384954), (-0.0194099, 0.889545, -0.384954, 0.245243), (0.760663, 0.461582, 0.0199094, 0.456001), (-0.461582, 0.760663, -0.456001, 0.0199094), (0.427963, 0.780073, -0.210758, 0.404863), (-0.780073, 0.427963, -0.404863, -0.210758), (0.889545, 0.0194099, -0.245243, -0.384954), (-0.0194099, 0.889545, 0.384954, -0.245243), (0.760663, 0.461582, -0.0199094, -0.456001), (-0.461582, 0.760663, 0.456001, -0.0199094), (0.427963, 0.780073, 0.210758, -0.404863), (-0.780073, 0.427963, 0.404863, 0.210758), (0.357132, 0.00779265, 0.479559, 0.801511), (-0.00779265, 0.357132, -0.801511, 0.479559), (0.305389, 0.185315, 0.0145545, 0.933908), (-0.185315, 0.305389, -0.933908, 0.0145545), (0.171817, 0.313182, -0.454349, 0.816065), (-0.313182, 0.171817, -0.816065, -0.454349), (0.706578, 0.205949, -0.435169, -0.518615), (-0.205949, 0.706578, 0.518615, -0.435169), (0.50894, 0.531645, -0.11756, -0.666718), (-0.531645, 0.50894, 0.666718, -0.11756), (0.174932, 0.714888, 0.231549, -0.636175), (-0.714888, 0.174932, 0.636175, 0.231549), (0.649957, 0.189445, 0.473079, 0.563793), (-0.189445, 0.649957, -0.563793, 0.473079), (0.468157, 0.489043, 0.127802, 0.724799), (-0.489043, 0.468157, -0.724799, 0.127802), (0.160914, 0.657602, -0.25172, 0.691595), (-0.657602, 0.160914, -0.691595, -0.25172), (0.896707, 0.261366, -0.137729, 0.329598), (-0.261366, 0.896707, -0.329598, -0.137729), (0.645888, 0.674703, -0.284076, 0.216576), (-0.674703, 0.645888, -0.216576, -0.284076), (0.222004, 0.907254, -0.354305, 0.0455222), (-0.907254, 0.222004, -0.0455222, -0.354305), (0.438201, 0.127724, 0.410844, -0.789224), (-0.127724, 0.438201, 0.789224, 0.410844), (0.315631, 0.329713, 0.750413, -0.478066), (-0.329713, 0.315631, 0.478066, 0.750413), (0.108488, 0.443355, 0.88891, -0.0388106), (-0.443355, 0.108488, 0.0388106, 0.88891), (0.438201, 0.127724, -0.410844, 0.789224), (-0.127724, 0.438201, -0.789224, -0.410844), (0.315631, 0.329713, -0.750413, 0.478066), (-0.329713, 0.315631, -0.478066, -0.750413), (0.108488, 0.443355, -0.88891, 0.0388106), (-0.443355, 0.108488, -0.0388106, -0.88891), (0.896707, 0.261366, 0.137729, -0.329598), (-0.261366, 0.896707, 0.329598, 0.137729), (0.645888, 0.674703, 0.284076, -0.216576), (-0.674703, 0.645888, 0.216576, 0.284076), (0.222004, 0.907254, 0.354305, -0.0455222), (-0.907254, 0.222004, 0.0455222, 0.354305), (0.649957, 0.189445, -0.563793, 0.473079), (-0.189445, 0.649957, -0.473079, -0.563793), (0.468157, 0.489043, -0.724799, 0.127802), (-0.489043, 0.468157, -0.127802, -0.724799), (0.160914, 0.657602, -0.691595, -0.25172), (-0.657602, 0.160914, 0.25172, -0.691595), (0.706578, 0.205949, 0.518615, -0.435169), (-0.205949, 0.706578, 0.435169, 0.518615), (0.50894, 0.531645, 0.666718, -0.11756), (-0.531645, 0.50894, 0.11756, 0.666718), (0.174932, 0.714888, 0.636175, 0.231549), (-0.714888, 0.174932, -0.231549, 0.636175), (0.342946, 0.0999598, -0.861805, -0.360122), (-0.0999598, 0.342946, 0.360122, -0.861805), (0.24702, 0.258041, -0.566284, -0.742777), (-0.258041, 0.24702, 0.742777, -0.566284), (0.0849055, 0.34698, -0.119028, -0.926406), (-0.34698, 0.0849055, 0.926406, -0.119028), (0.854211, 0.24898, 0.404863, 0.210758), (-0.24898, 0.854211, -0.210758, 0.404863), (0.615278, 0.642728, 0.245243, 0.384954), (-0.642728, 0.615278, -0.384954, 0.245243), (0.211483, 0.864258, 0.0199094, 0.456001), (-0.864258, 0.211483, -0.456001, 0.0199094), (0.854211, 0.24898, -0.404863, -0.210758), (-0.24898, 0.854211, 0.210758, -0.404863), (0.615278, 0.642728, -0.245243, -0.384954), (-0.642728, 0.615278, 0.384954, -0.245243), (0.211483, 0.864258, -0.0199094, -0.456001), (-0.864258, 0.211483, 0.456001, -0.0199094), (0.342946, 0.0999598, 0.861805, 0.360122), (-0.0999598, 0.342946, -0.360122, 0.861805), (0.24702, 0.258041, 0.566284, 0.742777), (-0.258041, 0.24702, -0.742777, 0.566284), (0.0849055, 0.34698, 0.119028, 0.926406), (-0.34698, 0.0849055, -0.926406, 0.119028), (0.706578, 0.205949, -0.674427, -0.0590047), (-0.205949, 0.706578, 0.0590047, -0.674427), (0.50894, 0.531645, -0.554569, -0.388313), (-0.531645, 0.50894, 0.388313, -0.554569), (0.174932, 0.714888, -0.286114, -0.613573), (-0.714888, 0.174932, 0.613573, -0.286114), (0.649957, 0.189445, 0.733179, 0.0641449), (-0.189445, 0.649957, -0.0641449, 0.733179), (0.468157, 0.489043, 0.60288, 0.422141), (-0.489043, 0.468157, -0.422141, 0.60288), (0.160914, 0.657602, 0.311039, 0.667024), (-0.657602, 0.160914, -0.667024, 0.311039), (0.896707, 0.261366, -0.343563, 0.0978201), (-0.261366, 0.896707, -0.0978201, -0.343563), (0.645888, 0.674703, -0.346444, -0.0870667), (-0.674703, 0.645888, 0.0870667, -0.346444), (0.222004, 0.907254, -0.256496, -0.248624), (-0.907254, 0.222004, 0.248624, -0.256496), (0.438201, 0.127724, 0.848576, -0.267555), (-0.127724, 0.438201, 0.267555, 0.848576), (0.315631, 0.329713, 0.868666, 0.192579), (-0.329713, 0.315631, -0.192579, 0.868666), (0.108488, 0.443355, 0.655997, 0.601111), (-0.443355, 0.108488, -0.601111, 0.655997), (0.438201, 0.127724, -0.848576, 0.267555), (-0.127724, 0.438201, -0.267555, -0.848576), (0.315631, 0.329713, -0.868666, -0.192579), (-0.329713, 0.315631, 0.192579, -0.868666), (0.108488, 0.443355, -0.655997, -0.601111), (-0.443355, 0.108488, 0.601111, -0.655997), (0.896707, 0.261366, 0.343563, -0.0978201), (-0.261366, 0.896707, 0.0978201, 0.343563), (0.645888, 0.674703, 0.346444, 0.0870667), (-0.674703, 0.645888, -0.0870667, 0.346444), (0.222004, 0.907254, 0.256496, 0.248624), (-0.907254, 0.222004, -0.248624, 0.256496), (0.649957, 0.189445, -0.0641449, 0.733179), (-0.189445, 0.649957, -0.733179, -0.0641449), (0.468157, 0.489043, -0.422141, 0.60288), (-0.489043, 0.468157, -0.60288, -0.422141), (0.160914, 0.657602, -0.667024, 0.311039), (-0.657602, 0.160914, -0.311039, -0.667024), (0.706578, 0.205949, 0.0590047, -0.674427), (-0.205949, 0.706578, 0.674427, 0.0590047), (0.50894, 0.531645, 0.388313, -0.554569), (-0.531645, 0.50894, 0.554569, 0.388313), (0.174932, 0.714888, 0.613573, -0.286114), (-0.714888, 0.174932, 0.286114, 0.613573), (0.342946, 0.0999598, -0.255772, -0.898319), (-0.0999598, 0.342946, 0.898319, -0.255772), (0.24702, 0.258041, 0.227655, -0.905853), (-0.258041, 0.24702, 0.905853, 0.227655), (0.0849055, 0.34698, 0.650081, -0.670664), (-0.34698, 0.0849055, 0.670664, 0.650081), (0.854211, 0.24898, 0.137253, 0.43531), (-0.24898, 0.854211, -0.43531, 0.137253), (0.615278, 0.642728, -0.0987907, 0.445616), (-0.642728, 0.615278, -0.445616, -0.0987907), (0.211483, 0.864258, -0.308363, 0.33652), (-0.864258, 0.211483, -0.33652, -0.308363), (0.854211, 0.24898, -0.137253, -0.43531), (-0.24898, 0.854211, 0.43531, -0.137253), (0.615278, 0.642728, 0.0987907, -0.445616), (-0.642728, 0.615278, 0.445616, 0.0987907), (0.211483, 0.864258, 0.308363, -0.33652), (-0.864258, 0.211483, 0.33652, 0.308363), (0.342946, 0.0999598, 0.255772, 0.898319), (-0.0999598, 0.342946, -0.898319, 0.255772), (0.24702, 0.258041, -0.227655, 0.905853), (-0.258041, 0.24702, -0.905853, -0.227655), (0.0849055, 0.34698, -0.650081, 0.670664), (-0.34698, 0.0849055, -0.670664, -0.650081), (0.735805, 0.0160553, -0.554569, -0.388313), (-0.0160553, 0.735805, 0.388313, -0.554569), (0.629198, 0.381807, -0.286114, -0.613573), (-0.381807, 0.629198, 0.613573, -0.286114), (0.353998, 0.645253, 0.0590047, -0.674427), (-0.645253, 0.353998, 0.674427, 0.0590047), (0.676842, 0.0147687, 0.60288, 0.422141), (-0.0147687, 0.676842, -0.422141, 0.60288), (0.578778, 0.351211, 0.311039, 0.667024), (-0.351211, 0.578778, -0.667024, 0.311039), (0.325631, 0.593547, -0.0641449, 0.733179), (-0.593547, 0.325631, -0.733179, -0.0641449), (0.933799, 0.0203756, -0.0477298, 0.354014), (-0.0203756, 0.933799, -0.354014, -0.0477298), (0.798506, 0.484545, -0.218342, 0.28272), (-0.484545, 0.798506, -0.28272, -0.218342), (0.449254, 0.818881, -0.33045, 0.135672), (-0.818881, 0.449254, -0.135672, -0.33045), (0.456327, 0.00995709, 0.192579, -0.868666), (-0.00995709, 0.456327, 0.868666, 0.192579), (0.390212, 0.236787, 0.601111, -0.655997), (-0.236787, 0.390212, 0.655997, 0.601111), (0.21954, 0.400169, 0.848576, -0.267555), (-0.400169, 0.21954, 0.267555, 0.848576), (0.456327, 0.00995709, -0.192579, 0.868666), (-0.00995709, 0.456327, -0.868666, -0.192579), (0.390212, 0.236787, -0.601111, 0.655997), (-0.236787, 0.390212, -0.655997, -0.601111), (0.21954, 0.400169, -0.848576, 0.267555), (-0.400169, 0.21954, -0.267555, -0.848576), (0.933799, 0.0203756, 0.0477298, -0.354014), (-0.0203756, 0.933799, 0.354014, 0.0477298), (0.798506, 0.484545, 0.218342, -0.28272), (-0.484545, 0.798506, 0.28272, 0.218342), (0.449254, 0.818881, 0.33045, -0.135672), (-0.818881, 0.449254, 0.135672, 0.33045), (0.676842, 0.0147687, -0.422141, 0.60288), (-0.0147687, 0.676842, -0.60288, -0.422141), (0.578778, 0.351211, -0.667024, 0.311039), (-0.351211, 0.578778, -0.311039, -0.667024), (0.325631, 0.593547, -0.733179, -0.0641449), (-0.593547, 0.325631, 0.0641449, -0.733179), (0.735805, 0.0160553, 0.388313, -0.554569), (-0.0160553, 0.735805, 0.554569, 0.388313), (0.629198, 0.381807, 0.613573, -0.286114), (-0.381807, 0.629198, 0.286114, 0.613573), (0.353998, 0.645253, 0.674427, 0.0590047), (-0.645253, 0.353998, -0.0590047, 0.674427), (0.357132, 0.00779265, -0.925646, -0.1248), (-0.00779265, 0.357132, 0.1248, -0.925646), (0.305389, 0.185315, -0.739233, -0.570903), (-0.185315, 0.305389, 0.570903, -0.739233), (0.171817, 0.313182, -0.354743, -0.864033), (-0.313182, 0.171817, 0.864033, -0.354743), (0.889545, 0.0194099, 0.445616, 0.0987907), (-0.0194099, 0.889545, -0.0987907, 0.445616), (0.760663, 0.461582, 0.33652, 0.308363), (-0.461582, 0.760663, -0.308363, 0.33652), (0.427963, 0.780073, 0.137253, 0.43531), (-0.780073, 0.427963, -0.43531, 0.137253), (0.889545, 0.0194099, -0.445616, -0.0987907), (-0.0194099, 0.889545, 0.0987907, -0.445616), (0.760663, 0.461582, -0.33652, -0.308363), (-0.461582, 0.760663, 0.308363, -0.33652), (0.427963, 0.780073, -0.137253, -0.43531), (-0.780073, 0.427963, 0.43531, -0.137253), (0.357132, 0.00779265, 0.925646, 0.1248), (-0.00779265, 0.357132, -0.1248, 0.925646), (0.305389, 0.185315, 0.739233, 0.570903), (-0.185315, 0.305389, -0.570903, 0.739233), (0.171817, 0.313182, 0.354743, 0.864033), (-0.313182, 0.171817, -0.864033, 0.354743), (0.836623, 0.0916251, -0.536852, -0.0587948), (-0.0916251, 0.836623, 0.0587948, -0.536852), (0.678724, 0.497661, -0.43553, -0.319344), (-0.497661, 0.678724, 0.319344, -0.43553), (0.338962, 0.770349, -0.217508, -0.494325), (-0.770349, 0.338962, 0.494325, -0.217508), (0.536852, 0.0587948, 0.836623, 0.0916251), (-0.0587948, 0.536852, -0.0916251, 0.836623), (0.43553, 0.319344, 0.678724, 0.497661), (-0.319344, 0.43553, -0.497661, 0.678724), (0.217508, 0.494325, 0.338962, 0.770349), (-0.494325, 0.217508, -0.770349, 0.338962), (0.982338, 0.107584, -0.119395, 0.0958246), (-0.107584, 0.982338, -0.0958246, -0.119395), (0.796938, 0.584339, -0.151311, 0.023289), (-0.584339, 0.796938, -0.023289, -0.151311), (0.397999, 0.904522, -0.142684, -0.0554868), (-0.904522, 0.397999, 0.0554868, -0.142684), (0.608733, 0.0666671, 0.616553, -0.494836), (-0.0666671, 0.608733, 0.494836, 0.616553), (0.493844, 0.362102, 0.781368, -0.120264), (-0.362102, 0.493844, 0.120264, 0.781368), (0.246631, 0.560512, 0.736817, 0.286532), (-0.560512, 0.246631, -0.286532, 0.736817), (0.608733, 0.0666671, -0.616553, 0.494836), (-0.0666671, 0.608733, -0.494836, -0.616553), (0.493844, 0.362102, -0.781368, 0.120264), (-0.362102, 0.493844, -0.120264, -0.781368), (0.246631, 0.560512, -0.736817, -0.286532), (-0.560512, 0.246631, 0.286532, -0.736817), (0.982338, 0.107584, 0.119395, -0.0958246), (-0.107584, 0.982338, 0.0958246, 0.119395), (0.796938, 0.584339, 0.151311, -0.023289), (-0.584339, 0.796938, 0.023289, 0.151311), (0.397999, 0.904522, 0.142684, 0.0554868), (-0.904522, 0.397999, -0.0554868, 0.142684), (0.536852, 0.0587948, -0.0916251, 0.836623), (-0.0587948, 0.536852, -0.836623, -0.0916251), (0.43553, 0.319344, -0.497661, 0.678724), (-0.319344, 0.43553, -0.678724, -0.497661), (0.217508, 0.494325, -0.770349, 0.338962), (-0.494325, 0.217508, -0.338962, -0.770349), (0.836623, 0.0916251, 0.0587948, -0.536852), (-0.0916251, 0.836623, 0.536852, 0.0587948), (0.678724, 0.497661, 0.319344, -0.43553), (-0.497661, 0.678724, 0.43553, 0.319344), (0.338962, 0.770349, 0.494325, -0.217508), (-0.770349, 0.338962, 0.217508, 0.494325), (0.152183, 0.0166668, -0.618545, -0.770691), (-0.0166668, 0.152183, 0.770691, -0.618545), (0.123461, 0.0905254, -0.15033, -0.97671), (-0.0905254, 0.123461, 0.97671, -0.15033), (0.0616577, 0.140128, 0.358166, -0.921021), (-0.140128, 0.0616577, 0.921021, 0.358166), (0.785871, 0.0860668, 0.383298, 0.47758), (-0.0860668, 0.785871, -0.47758, 0.383298), (0.63755, 0.467471, 0.0931562, 0.605245), (-0.467471, 0.63755, -0.605245, 0.0931562), (0.318399, 0.723617, -0.221947, 0.570736), (-0.723617, 0.318399, -0.570736, -0.221947), (0.785871, 0.0860668, -0.383298, -0.47758), (-0.0860668, 0.785871, 0.47758, -0.383298), (0.63755, 0.467471, -0.0931562, -0.605245), (-0.467471, 0.63755, 0.605245, -0.0931562), (0.318399, 0.723617, 0.221947, -0.570736), (-0.723617, 0.318399, 0.570736, 0.221947), (0.152183, 0.0166668, 0.618545, 0.770691), (-0.0166668, 0.152183, -0.770691, 0.618545), (0.123461, 0.0905254, 0.15033, 0.97671), (-0.0905254, 0.123461, -0.97671, 0.15033), (0.0616577, 0.140128, -0.358166, 0.921021), (-0.140128, 0.0616577, -0.921021, -0.358166), (0.570736, 0.221947, -0.736817, -0.286532), (-0.221947, 0.570736, 0.286532, -0.736817), (0.383298, 0.47758, -0.494836, -0.616553), (-0.47758, 0.383298, 0.616553, -0.494836), (0.0931562, 0.605245, -0.120264, -0.781368), (-0.605245, 0.0931562, 0.781368, -0.120264), (0.736817, 0.286532, 0.570736, 0.221947), (-0.286532, 0.736817, -0.221947, 0.570736), (0.494836, 0.616553, 0.383298, 0.47758), (-0.616553, 0.494836, -0.47758, 0.383298), (0.120264, 0.781368, 0.0931562, 0.605245), (-0.781368, 0.120264, -0.605245, 0.0931562), (0.784402, 0.305037, -0.494325, 0.217508), (-0.305037, 0.784402, -0.217508, -0.494325), (0.526793, 0.656371, -0.536852, -0.0587948), (-0.656371, 0.526793, 0.0587948, -0.536852), (0.128031, 0.83183, -0.43553, -0.319344), (-0.83183, 0.128031, 0.319344, -0.43553), (0.237807, 0.0924779, 0.885015, -0.389416), (-0.0924779, 0.237807, 0.389416, 0.885015), (0.159708, 0.198992, 0.961153, 0.105263), (-0.198992, 0.159708, -0.105263, 0.961153), (0.0388151, 0.252186, 0.779751, 0.571737), (-0.252186, 0.0388151, -0.571737, 0.779751), (0.237807, 0.0924779, -0.885015, 0.389416), (-0.0924779, 0.237807, -0.389416, -0.885015), (0.159708, 0.198992, -0.961153, -0.105263), (-0.198992, 0.159708, 0.105263, -0.961153), (0.0388151, 0.252186, -0.779751, -0.571737), (-0.252186, 0.0388151, 0.571737, -0.779751), (0.784402, 0.305037, 0.494325, -0.217508), (-0.305037, 0.784402, 0.217508, 0.494325), (0.526793, 0.656371, 0.536852, 0.0587948), (-0.656371, 0.526793, -0.0587948, 0.536852), (0.128031, 0.83183, 0.43553, 0.319344), (-0.83183, 0.128031, -0.319344, 0.43553), (0.736817, 0.286532, -0.221947, 0.570736), (-0.286532, 0.736817, -0.570736, -0.221947), (0.494836, 0.616553, -0.47758, 0.383298), (-0.616553, 0.494836, -0.383298, -0.47758), (0.120264, 0.781368, -0.605245, 0.0931562), (-0.781368, 0.120264, -0.0931562, -0.605245), (0.570736, 0.221947, 0.286532, -0.736817), (-0.221947, 0.570736, 0.736817, 0.286532), (0.383298, 0.47758, 0.616553, -0.494836), (-0.47758, 0.383298, 0.494836, 0.616553), (0.0931562, 0.605245, 0.781368, -0.120264), (-0.605245, 0.0931562, 0.120264, 0.781368), (0.503342, 0.195739, -0.338962, -0.770349), (-0.195739, 0.503342, 0.770349, -0.338962), (0.338037, 0.421186, 0.0916251, -0.836623), (-0.421186, 0.338037, 0.836623, 0.0916251), (0.082156, 0.533776, 0.497661, -0.678724), (-0.533776, 0.082156, 0.678724, 0.497661), (0.901159, 0.350441, 0.102763, 0.233546), (-0.350441, 0.901159, -0.233546, 0.102763), (0.605205, 0.75407, -0.0277779, 0.253639), (-0.75407, 0.605205, -0.253639, -0.0277779), (0.147088, 0.955647, -0.150876, 0.205769), (-0.955647, 0.147088, -0.205769, -0.150876), (0.901159, 0.350441, -0.102763, -0.233546), (-0.350441, 0.901159, 0.233546, -0.102763), (0.605205, 0.75407, 0.0277779, -0.253639), (-0.75407, 0.605205, 0.253639, 0.0277779)]

class sample_orientations(object):
  def __init__(self):
    self.sequence = SO3_sequence
    self.max_n    = len(SO3_sequence)
    self.counter  = 0

  def __next__(self):
    assert self.counter < self.max_n
    tmp = self.sequence[self.counter]
    self.counter+=1
    return tmp

  def restart(self,start_point=0):
    self.counter = start_point


def tst():
  so3 = sample_orientations()
  for ii in range(10000):
    q = flex.double(next(so3))
    q = q*q
    assert abs(1-q)<1e-5

if __name__ == "__main__":
  tst()
  print("OK")
