#############################################################################
##
##  This  file  tests  the large finite fields.
##
#@local fieldpairs,fieldsizes,iF,iPI,iW,izs,zs,iFI,x,i,F
gap> START_TEST("ffeconway.tst");

#
# Disable various warnings which depend on state of prime number and 
# Conway Polynomial databases and on whether FactInt is loaded 
#
gap> iPI := InfoLevel(InfoPrimeInt);; 
gap> iF := InfoLevel(InfoFactor);;
gap> if IsBound(InfoFactInt) then iFI := InfoLevel(InfoFactInt); fi;
gap> iW := InfoLevel(InfoWarning);;
gap> SetInfoLevel(InfoPrimeInt,0);
gap> SetInfoLevel(InfoFactor,0);
gap> if IsBound(InfoFactInt) then SetInfoLevel(InfoFactInt,0); fi;
gap> SetInfoLevel(InfoWarning,0);

#
# A range of field sizes to hit various cases for the underlying vector 
# arithmetic.
#
gap> fieldsizes := [ [2,17], [2,32], [2,60], [2,76], [2,87], [3,11], 
> [3,20], [3,60], [17,4], [257,2], [257,11], [65521,2], [65537,2], 
> [268435399,2], [4294967291,2], [1152921504606846883,3] ];;
gap> Add( fieldsizes, [NextPrimeInt(2^64),2] );

#
# When we come to test cross-field, we need to make sure we don't try and
# create overlarge fields
# leave the output here so we trip here if the Conway polynomial database changes
#
gap> fieldpairs := Concatenation(List(fieldsizes, pd -> List(Filtered([1..pd[2]-1], 
>   d2 -> IsCheapConwayPolynomial(pd[1],Lcm(pd[2],d2))), d2 ->
>   [pd[1],pd[2],d2])));
[ [ 2, 17, 1 ], [ 2, 17, 2 ], [ 2, 17, 3 ], [ 2, 17, 4 ], [ 2, 17, 5 ], 
  [ 2, 17, 6 ], [ 2, 17, 7 ], [ 2, 32, 1 ], [ 2, 32, 2 ], [ 2, 32, 3 ], 
  [ 2, 32, 4 ], [ 2, 32, 6 ], [ 2, 32, 8 ], [ 2, 32, 12 ], [ 2, 32, 16 ], 
  [ 2, 32, 24 ], [ 2, 60, 1 ], [ 2, 60, 2 ], [ 2, 60, 3 ], [ 2, 60, 4 ], 
  [ 2, 60, 5 ], [ 2, 60, 6 ], [ 2, 60, 8 ], [ 2, 60, 10 ], [ 2, 60, 12 ], 
  [ 2, 60, 15 ], [ 2, 60, 20 ], [ 2, 60, 24 ], [ 2, 60, 30 ], [ 2, 60, 40 ], 
  [ 2, 76, 1 ], [ 2, 76, 2 ], [ 2, 76, 4 ], [ 2, 76, 19 ], [ 2, 76, 38 ], 
  [ 2, 87, 1 ], [ 2, 87, 3 ], [ 2, 87, 29 ], [ 3, 11, 1 ], [ 3, 11, 2 ], 
  [ 3, 11, 3 ], [ 3, 11, 4 ], [ 3, 11, 5 ], [ 3, 11, 6 ], [ 3, 11, 7 ], 
  [ 3, 20, 1 ], [ 3, 20, 2 ], [ 3, 20, 3 ], [ 3, 20, 4 ], [ 3, 20, 5 ], 
  [ 3, 20, 6 ], [ 3, 20, 8 ], [ 3, 20, 10 ], [ 3, 20, 12 ], [ 3, 20, 15 ], 
  [ 3, 60, 1 ], [ 3, 60, 2 ], [ 3, 60, 3 ], [ 3, 60, 4 ], [ 3, 60, 5 ], 
  [ 3, 60, 6 ], [ 3, 60, 10 ], [ 3, 60, 12 ], [ 3, 60, 15 ], [ 3, 60, 20 ], 
  [ 3, 60, 30 ], [ 17, 4, 1 ], [ 17, 4, 2 ], [ 17, 4, 3 ], [ 257, 2, 1 ], 
  [ 257, 11, 1 ], [ 65521, 2, 1 ], [ 65537, 2, 1 ], [ 268435399, 2, 1 ], 
  [ 4294967291, 2, 1 ], [ 1152921504606846883, 3, 1 ], 
  [ 18446744073709551629, 2, 1 ] ]
gap> IsCheapConwayPolynomial( 997, 1 );
true

#
# construct generating elements
#
gap> zs := List(fieldsizes, pd -> Z(pd[1],pd[2]));
[ z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z, z ]
gap> ForAll([1..Length(zs)] , i-> IsCoeffsModConwayPolRep(zs[i]));
true

#
# and another way
#
gap> zs = List(fieldsizes, pd -> Z(pd[1]^pd[2]));
true

#
# Construct some more interesting elements and test View, Print and Display
#
gap> izs := List(zs, Inverse);
[ z2+z16, z2+z3+z6+z8+z14+z31, 
  z+z2+z3+z4+z7+z11+z16+z18+z21+z24+z25+z29+z31+z32+z33+z35+z38+z40+z41+z43+z4\
4+z59, 
  1+z+z4+z13+z14+z18+z19+z22+z23+z24+z26+z28+z30+z32+z33+z34+z35+z36+z37+z75, 
  1+z2+z4+z6+z7+z9+z10+z11+z12+z13+z15+z19+z25+z27+z29+z86, z+2z10, 
  1+2z2+2z3+2z4+z7+z8+z9+z10+2z12+z19, 
  2z+2z2+2z3+z4+2z5+2z7+z9+2z10+z13+2z14+2z15+z19+z20+2z21+2z23+2z25+2z26+2z27\
+z29+z31+2z32+z33+z34+2z35+2z37+2z38+z39+2z40+z43+z59, 8+9z+11z3, 2+171z, 
  99+86z10, 61667+23125z, 21846+43691z, 89478467+89478466z, 
  2147483646+2147483645z, 1+576460752303423442z2, 2+9223372036854775814z ]
gap> Print(izs,"\n");
[ Z(2,17)^2+Z(2,17)^16, 
  Z(2,32)^2+Z(2,32)^3+Z(2,32)^6+Z(2,32)^8+Z(2,32)^14+Z(2,32)^31, 
  Z(2,60)+Z(2,60)^2+Z(2,60)^3+Z(2,60)^4+Z(2,60)^7+Z(2,60)^11+Z(2,60)^16+Z(2,60\
)^18+Z(2,60)^21+Z(2,60)^24+Z(2,60)^25+Z(2,60)^29+Z(2,60)^31+Z(2,60)^32+Z(2,60)\
^33+Z(2,60)^35+Z(2,60)^38+Z(2,60)^40+Z(2,60)^41+Z(2,60)^43+Z(2,60)^44+Z(2,60)^\
59, 
  Z(2)^0+Z(2,76)+Z(2,76)^4+Z(2,76)^13+Z(2,76)^14+Z(2,76)^18+Z(2,76)^19+Z(2,76)\
^22+Z(2,76)^23+Z(2,76)^24+Z(2,76)^26+Z(2,76)^28+Z(2,76)^30+Z(2,76)^32+Z(2,76)^\
33+Z(2,76)^34+Z(2,76)^35+Z(2,76)^36+Z(2,76)^37+Z(2,76)^75, 
  Z(2)^0+Z(2,87)^2+Z(2,87)^4+Z(2,87)^6+Z(2,87)^7+Z(2,87)^9+Z(2,87)^10+Z(2,87)^\
11+Z(2,87)^12+Z(2,87)^13+Z(2,87)^15+Z(2,87)^19+Z(2,87)^25+Z(2,87)^27+Z(2,87)^2\
9+Z(2,87)^86, Z(3,11)+2*Z(3,11)^10, 
  Z(3)^0+2*Z(3,20)^2+2*Z(3,20)^3+2*Z(3,20)^4+Z(3,20)^7+Z(3,20)^8+Z(3,20)^9+Z(3\
,20)^10+2*Z(3,20)^12+Z(3,20)^19, 
  2*Z(3,60)+2*Z(3,60)^2+2*Z(3,60)^3+Z(3,60)^4+2*Z(3,60)^5+2*Z(3,60)^7+Z(3,60)^\
9+2*Z(3,60)^10+Z(3,60)^13+2*Z(3,60)^14+2*Z(3,60)^15+Z(3,60)^19+Z(3,60)^20+2*Z(\
3,60)^21+2*Z(3,60)^23+2*Z(3,60)^25+2*Z(3,60)^26+2*Z(3,60)^27+Z(3,60)^29+Z(3,60\
)^31+2*Z(3,60)^32+Z(3,60)^33+Z(3,60)^34+2*Z(3,60)^35+2*Z(3,60)^37+2*Z(3,60)^38\
+Z(3,60)^39+2*Z(3,60)^40+Z(3,60)^43+Z(3,60)^59, 
  Z(17)^10+9*Z(17,4)+11*Z(17,4)^3, Z(257)^48+171*Z(257,2), 
  Z(257)^198+86*Z(257,11)^10, Z(65521)^1451+23125*Z(65521,2), 
  ZmodpZObj(21846,65537)+43691*Z(65537,2), 
  ZmodpZObj(89478467,268435399)+89478466*Z(268435399,2), 
  ZmodpZObj(2147483646,4294967291)+2147483645*Z(4294967291,2), 
  ZmodpZObj(1,1152921504606846883)+576460752303423442*Z(1152921504606846883,3)\
^2, 
  ZmodpZObj(2,18446744073709551629)+9223372036854775814*Z(18446744073709551629\
,2) ]
gap> for x in izs do
>    Display(x);
> od;
z2+z16
z2+z3+z6+z8+z14+z31
z+z2+z3+z4+z7+z11+z16+z18+z21+z24+z25+z29+z31+z32+z33+z35+z38+z40+z41+z43+z44+\
z59
1+z+z4+z13+z14+z18+z19+z22+z23+z24+z26+z28+z30+z32+z33+z34+z35+z36+z37+z75
1+z2+z4+z6+z7+z9+z10+z11+z12+z13+z15+z19+z25+z27+z29+z86
z+2z10
1+2z2+2z3+2z4+z7+z8+z9+z10+2z12+z19
2z+2z2+2z3+z4+2z5+2z7+z9+2z10+z13+2z14+2z15+z19+z20+2z21+2z23+2z25+2z26+2z27+z\
29+z31+2z32+z33+z34+2z35+2z37+2z38+z39+2z40+z43+z59
8+9z+11z3
2+171z
99+86z10
61667+23125z
21846+43691z
89478467+89478466z
2147483646+2147483645z
1+576460752303423442z2
2+9223372036854775814z

#
# Test arithmetic within the field
#
gap> zs + izs;
[ z+z2+z16, z+z2+z3+z6+z8+z14+z31, 
  z2+z3+z4+z7+z11+z16+z18+z21+z24+z25+z29+z31+z32+z33+z35+z38+z40+z41+z43+z44+\
z59, 1+z4+z13+z14+z18+z19+z22+z23+z24+z26+z28+z30+z32+z33+z34+z35+z36+z37+z75,
  1+z+z2+z4+z6+z7+z9+z10+z11+z12+z13+z15+z19+z25+z27+z29+z86, 2z+2z10, 
  1+z+2z2+2z3+2z4+z7+z8+z9+z10+2z12+z19, 
  2z2+2z3+z4+2z5+2z7+z9+2z10+z13+2z14+2z15+z19+z20+2z21+2z23+2z25+2z26+2z27+z2\
9+z31+2z32+z33+z34+2z35+2z37+2z38+z39+2z40+z43+z59, 8+10z+11z3, 2+172z, 
  99+z+86z10, 61667+23126z, 21846+43692z, 89478467+89478467z, 
  2147483646+2147483646z, 1+z+576460752303423442z2, 2+9223372036854775815z ]
gap> zs - izs;
[ z+z2+z16, z+z2+z3+z6+z8+z14+z31, 
  z2+z3+z4+z7+z11+z16+z18+z21+z24+z25+z29+z31+z32+z33+z35+z38+z40+z41+z43+z44+\
z59, 1+z4+z13+z14+z18+z19+z22+z23+z24+z26+z28+z30+z32+z33+z34+z35+z36+z37+z75,
  1+z+z2+z4+z6+z7+z9+z10+z11+z12+z13+z15+z19+z25+z27+z29+z86, z10, 
  2+z+z2+z3+z4+2z7+2z8+2z9+2z10+z12+2z19, 
  2z+z2+z3+2z4+z5+z7+2z9+z10+2z13+z14+z15+2z19+2z20+z21+z23+z25+z26+z27+2z29+2\
z31+z32+2z33+2z34+z35+z37+z38+2z39+z40+2z43+2z59, 9+9z+6z3, 255+87z, 
  158+z+171z10, 3854+42397z, 43691+21847z, 178956932+178956934z, 
  2147483645+2147483647z, 1152921504606846882+z+576460752303423441z2, 
  18446744073709551627+9223372036854775816z ]
gap> List(izs, x->x^2);
[ z+z15, z+z2+z5+z7+z13+z30, 
  1+z+z2+z3+z6+z10+z15+z17+z20+z23+z24+z28+z30+z31+z32+z34+z37+z39+z40+z42+z43\
+z58, z+z3+z4+z12+z14+z17+z19+z21+z24+z25+z26+z27+z28+z29+z30+z31+z37+z74+z75,
  1+z+z2+z3+z4+z5+z7+z8+z13+z14+z15+z18+z19+z24+z25+z26+z27+z28+z29+z85+z86, 
  1+2z9, 1+2z+z2+z3+2z4+z6+2z7+2z8+2z9+z10+2z11+2z12+z18+z19, 
  2+2z+2z2+z3+2z4+2z6+z8+2z9+z12+2z13+2z14+z18+z19+2z20+2z22+2z24+2z25+2z26+z2\
8+z30+2z31+z32+z33+2z34+2z36+2z37+z38+2z39+z42+z58, 5+4z+11z2+3z3, 175+85z, 
  35+86z9+33z10, 3174+50331z, 50973+58255z, 59652311+149130777z, 
  3221225468+3221225468z, 1+576460752303423442z+576460752303423442z2, 
  9223372036854775818+18446744073709551628z ]
gap> List([1..Length(zs)] , i-> zs[i]/izs[i]);
[ z2, z2, z2, z2, z2, z2, z2, z2, z2, 254+6z, z2, 65504+3z, 65534+z, 
  268435396+2z, 4294967289+z, z2, 18446744073709551627+4z ]
gap> List(izs, AdditiveInverse);
[ z2+z16, z2+z3+z6+z8+z14+z31, 
  z+z2+z3+z4+z7+z11+z16+z18+z21+z24+z25+z29+z31+z32+z33+z35+z38+z40+z41+z43+z4\
4+z59, 
  1+z+z4+z13+z14+z18+z19+z22+z23+z24+z26+z28+z30+z32+z33+z34+z35+z36+z37+z75, 
  1+z2+z4+z6+z7+z9+z10+z11+z12+z13+z15+z19+z25+z27+z29+z86, 2z+z10, 
  2+z2+z3+z4+2z7+2z8+2z9+2z10+z12+2z19, 
  z+z2+z3+2z4+z5+z7+2z9+z10+2z13+z14+z15+2z19+2z20+z21+z23+z25+z26+z27+2z29+2z\
31+z32+2z33+2z34+z35+z37+z38+2z39+z40+2z43+2z59, 9+8z+6z3, 255+86z, 
  158+171z10, 3854+42396z, 43691+21846z, 178956932+178956933z, 
  2147483645+2147483646z, 1152921504606846882+576460752303423441z2, 
  18446744073709551627+9223372036854775815z ]

#
# Test comparisons within the field
#
gap> ForAny([1..Length(zs)] , i-> zs[i] < zs[i]);
false
gap> List([1..Length(zs)] , i-> zs[i] < izs[i]);
[ false, false, true, true, true, true, true, true, true, true, true, true, 
  true, true, true, true, true ]
gap> List([1..Length(zs)] , i-> izs[i] < zs[i]);
[ true, true, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false ]

#
# Test arithmetic across fields
#
gap> List(fieldpairs, pdd -> Z(pdd[1],pdd[2])+Z(pdd[1],pdd[3]));
[ 1+z, z+z4+z9+z10+z15+z17+z19+z21+z23+z25+z26+z27+z29+z30+z33, 
  z2+z3+z6+z10+z13+z15+z16+z18+z19+z20+z24+z25+z35+z38+z39+z40+z41+z44+z46+z48\
+z49, 
  1+z2+z3+z4+z5+z7+z10+z11+z12+z13+z14+z16+z17+z18+z19+z21+z22+z23+z24+z25+z26\
+z30+z31+z32+z35+z37+z38+z40+z47+z48+z50+z51+z56+z58+z62+z63+z67, 
  z+z3+z7+z8+z9+z10+z13+z14+z17+z19+z20+z21+z22+z24+z25+z27+z28+z30+z32+z33+z3\
6+z38+z41+z42+z45+z46+z47+z48+z50+z51+z52+z54+z58+z59+z60+z61+z63+z65+z68+z69+\
z71+z73+z75+z76+z77+z82, 
  z2+z3+z5+z7+z10+z11+z13+z16+z18+z19+z20+z21+z22+z25+z26+z27+z28+z30+z31+z32+\
z34+z35+z37+z38+z39+z40+z44+z45+z47+z54+z55+z56+z58+z60+z61+z63+z65+z67+z69+z7\
3+z74+z79+z83+z87+z89+z96+z97+z99+z100, <<an element of GF(2, 119)>>, 1+z, 
  1+z3+z4+z5+z6+z8+z9+z14+z17+z18+z19+z20+z21+z24+z29+z30, 
  z+z5+z7+z9+z12+z14+z15+z16+z17+z20+z22+z23+z24+z25+z28+z31+z33+z36+z37+z38+z\
39+z42+z43+z44+z52+z56+z61+z62+z65+z67+z68+z69+z70+z71+z72+z73+z76+z77+z79+z80\
+z81+z82+z83+z85+z86+z89+z94, 
  z4+z5+z6+z8+z10+z12+z14+z15+z17+z18+z19+z20+z21+z22+z23+z24+z25+z29+z30, 
  1+z6+z11+z12+z13+z15+z18+z22+z23+z30+z31+z36+z37+z38+z39+z41+z44+z45+z47+z48\
+z50+z51+z52+z55+z56+z57+z58+z59+z60+z63+z64+z66+z68+z69+z74+z76+z81+z83+z84+z\
85+z86+z88+z89+z90+z91+z92+z93, 
  z+z2+z3+z6+z7+z10+z12+z14+z17+z19+z20+z21+z22+z29+z31, 
  z2+z9+z10+z11+z15+z16+z17+z18+z20+z23+z25+z27+z29+z33+z35+z36+z39+z40+z42+z4\
5+z47+z48+z52+z54+z55+z56+z57+z59+z62+z67+z68+z71+z74+z75+z76+z80+z81+z83+z85+\
z86+z89+z90+z92+z93, 
  z+z2+z4+z5+z6+z7+z10+z11+z14+z15+z16+z20+z23+z26+z28+z29+z30+z31, 
  1+z+z4+z6+z7+z9+z12+z14+z17+z19+z21+z22+z28+z30+z33+z34+z36+z38+z40+z41+z45+\
z48+z49+z51+z52+z53+z59+z61+z63+z67+z69+z70+z75+z78+z81+z83+z85+z86+z87+z90+z9\
3+z94, 1+z, 
  z2+z4+z5+z7+z9+z11+z14+z17+z19+z20+z21+z23+z24+z25+z27+z30+z35+z36+z38+z39+z\
41+z43+z50+z51+z53+z54+z55+z57+z59, 
  1+z+z5+z7+z8+z11+z12+z15+z17+z22+z23+z24+z27+z28+z29+z31+z34+z35+z37+z38+z39\
+z42+z47+z48+z51+z52+z53, 
  1+z2+z3+z4+z7+z8+z9+z10+z11+z16+z18+z19+z20+z21+z23+z26+z27+z31+z32+z34+z36+\
z37+z39+z40+z41+z42+z45+z46+z47+z49+z50+z51+z53+z54+z56+z57+z58, 
  z+z2+z3+z6+z7+z9+z11+z13+z17+z19+z21+z22+z23+z25+z27+z29+z30+z31+z36+z38+z39\
+z41+z45+z46+z48+z51+z53+z54+z55+z56+z59, 
  z2+z4+z6+z7+z9+z11+z12+z14+z16+z17+z19+z20+z21+z22+z25+z26+z31+z34+z35+z36+z\
37+z38+z44+z46+z48+z49+z51+z52+z57+z58+z59, <<an element of GF(2, 120)>>, 
  z+z3+z4+z6+z8+z10+z11+z15+z18+z19+z22+z24+z26+z28+z30+z31+z32+z33+z34+z36+z3\
7+z39+z40+z41+z42+z47+z48+z50+z53+z54+z56+z57+z59, 
  z4+z6+z7+z13+z14+z16+z17+z18+z19+z20+z21+z22+z24+z26+z28+z31+z33+z34+z35+z37\
+z39+z43+z47+z48+z51+z52+z54+z58, 
  1+z4+z6+z8+z9+z14+z16+z24+z28+z29+z32+z34+z38+z39+z40+z42+z47+z48+z50+z51+z5\
2+z54+z56+z57+z59, 
  1+z+z2+z3+z4+z7+z11+z14+z15+z16+z17+z18+z22+z25+z26+z29+z34+z35+z36+z37+z38+\
z40+z44+z45+z46+z47+z48+z50+z52+z55+z56+z57+z58, <<an element of GF(2, 120)>>,
  z2+z4+z7+z10+z13+z15+z18+z20+z23+z25+z26+z38+z39+z41+z43+z44+z46+z48+z50+z51\
+z55+z56+z57+z59, <<an element of GF(2, 120)>>, 1+z, 
  z3+z6+z7+z8+z10+z14+z15+z18+z20+z22+z23+z25+z26+z27+z30+z31+z32+z33+z35+z36+\
z38+z42+z45+z46+z47+z50+z51+z54+z55+z57+z58+z60+z62+z63+z64+z65+z66+z67+z69+z7\
1+z73+z75, 
  1+z2+z3+z4+z7+z8+z9+z17+z18+z19+z21+z26+z29+z30+z31+z32+z34+z35+z36+z39+z43+\
z44+z45+z46+z48+z49+z52+z53+z56+z57+z60+z63+z64+z65+z69+z70+z71+z73+z74+z75, 
  1+z2+z5+z6+z7+z8+z10+z12+z13+z15+z16+z18+z19+z20+z26+z28+z33+z35+z37+z40+z41\
+z43+z44+z45+z46+z48+z49+z52+z53+z54+z55+z57+z60+z62+z65+z67+z72+z73+z74, 
  z2+z3+z5+z6+z8+z9+z11+z19+z23+z27+z31+z32+z34+z35+z36+z38+z42+z46+z48+z49+z5\
0+z53+z58+z59+z60+z62+z63+z65+z66+z67+z68+z69+z70+z74, 1+z, 
  1+z2+z3+z5+z9+z15+z22+z25+z26+z27+z28+z29+z30+z31+z32+z34+z36+z40+z42+z43+z4\
4+z48+z51+z52+z55+z56+z57+z60+z61+z62+z63+z64+z65+z67+z68+z69+z70+z71+z72+z75+\
z77+z79+z81+z83+z84+z86, 
  z+z2+z6+z8+z9+z10+z14+z15+z16+z19+z20+z22+z23+z26+z28+z30+z31+z33+z34+z37+z4\
1+z46+z50+z51+z54+z55+z56+z57+z58+z59+z60+z61+z63+z64+z66+z68+z71+z72+z76+z78+\
z81+z82+z83+z84+z85, 2+z, 
  2+z+z2+2z3+z4+2z5+2z6+z7+2z9+2z11+z12+2z14+z16+z17+2z18+2z19+2z20+2z21, 
  1+2z+2z3+2z4+2z6+2z8+2z9+z11+z12+z14+2z15+2z17+z18+z19+2z21+2z23+z25+2z26+2z\
27+2z28+2z29+2z30+z32, 
  2z+2z3+z8+z9+z12+z15+2z17+2z19+z21+z22+z23+z24+z25+z30+z34+z35+z37+2z38+2z40\
+z41+2z42+z43, 
  2+z+z4+2z6+z7+2z9+z10+2z14+z16+2z19+z20+2z21+2z22+z23+2z25+z26+2z27+z28+2z29\
+z30+z31+2z34+2z35+z36+z38+z40+z41+z42+z44+2z45+2z47+z51+z52, 
  1+z+z4+z6+2z8+z9+z11+2z13+z14+z17+z19+z20+2z21+z23+z25+2z26+z27+z28+z29+z31+\
z32+z33+z34+z35+2z38+z40+z41+2z42+2z45+z46+2z47+z50+2z51+z52+2z57+z58+2z59+z60\
+z61+2z62+2z63+2z64+z65, 
  1+z2+2z4+z5+2z6+2z7+z8+2z9+2z10+z11+2z12+2z13+z14+z16+2z17+2z18+z21+z22+z24+\
2z27+2z28+2z29+2z32+z33+z35+z36+z40+2z41+2z42+z45+2z46+2z48+2z49+z51+z52+z53+2\
z54+z55+z56+z57+2z58+z59+z60+z61+z63+z64+2z65+2z66+2z67+z68+2z73+2z76, 2+z, 
  z+z2+2z3+2z4+z7+2z8+z9+z10+2z12+2z13+2z14+z15+2z17+2z18+z19, 
  2+z2+z3+2z5+z7+2z11+z12+z14+z15+z16+z18+z19+2z20+2z22+z25+z27+z29+2z30+z32+z\
33+2z34+z35+2z36+2z39+z40+z41+2z43+2z45+z46+2z48+2z49+2z50+z53+2z54+2z55+z56+2\
z57, 1+2z+z2+2z3+2z4+2z5+z6+z7+2z11+2z12+z13+z14+z15+2z16+z17+z18+z19, 
  2z+2z2+z4+z5+z6+2z8+z11+2z12+z16+z19, 
  2z+z2+z3+2z4+z6+2z7+2z8+2z9+2z10+2z12+2z13+z15+z16+z17+2z18+z19+z20+2z21+z24\
+z25+2z26+2z27+z28+2z29+z30+2z31+z32+z33+z35+z36+2z37+z39+2z40+2z42+z45+z46+z4\
7+2z48+2z50+z51+2z52+z53+z54+z55+2z57+2z58+2z59, 
  2+z+2z3+2z4+2z5+z6+2z7+2z8+z9+2z11+2z12+z13+2z15+z16+z19+z23+2z24+2z25+2z26+\
z27+2z28+z29+2z30+2z33+2z35+z39, 
  z+z3+2z4+2z7+z8+2z10+z11+2z12+z13+z14+z15+z17+z18+2z19, 
  2+z+2z2+2z5+2z6+z8+2z9+z10+z12+2z15+z18+z19+z20+z23+2z25+2z26+z27+2z28+2z29+\
2z30+z31+2z32+2z33+2z34+2z38+2z39+2z40+z42+z44+z45+2z46+z48+z49+2z50+2z52+z53+\
z54+2z55+2z56+z57+z58, 
  2+z+2z2+2z3+z4+z7+2z8+2z9+2z10+z11+z13+z14+z15+2z19+2z21+2z23+2z24+z25+2z29+\
2z31+z32+z34+z35+z36+z37+z38+z39+2z40+2z42+2z44+z45+2z46+z49+z50+z51+2z53+z54+\
2z55+2z57+2z58+2z59, 2+z, 
  2+2z+2z2+2z4+2z5+z6+2z7+z8+2z9+2z11+z13+2z14+2z15+z16+2z17+2z18+z19+2z20+z21\
+z22+2z23+2z25+2z27+z28+2z29+2z31+z35+2z36+z37+2z38+2z39+z40+2z42+z43+2z45+2z4\
6+z47+z49+z51+2z52+2z53+z55+2z58+z59, 
  1+z+2z2+2z3+z4+2z5+2z6+z7+z9+z10+z11+z12+2z16+z17+z18+z19+z21+z25+z26+2z27+z\
28+2z29+2z30+z31+z34+z35+z36+2z37+z38+2z39+z40+2z42+2z43+2z46+z47+2z48+2z49+z5\
2+2z53+2z55+z57+2z58+2z59, 
  1+z+z3+2z5+2z7+2z8+2z9+z10+z12+z13+2z14+2z19+z20+z22+2z23+z26+2z27+z28+2z29+\
z30+z32+2z33+z34+2z35+z37+2z38+2z40+z41+2z42+z43+2z44+2z45+z46+2z47+z48+z49+2z\
51+z53+2z56+z57+2z58+z59, 
  2+2z+2z3+z4+2z5+z7+z8+z9+z10+2z11+2z12+2z13+z14+z18+2z20+z24+z25+z28+2z29+2z\
30+2z31+z32+z33+z34+2z35+z36+2z37+z38+z40+z44+z46+z47+z49+z50+2z52+2z53+2z57+2\
z58+2z59, 
  2+2z2+2z3+2z7+2z8+2z11+2z12+2z13+2z14+2z16+2z17+2z18+z19+2z20+z22+z24+z25+2z\
28+z30+2z34+z35+z37+z38+z39+2z40+2z41+z42+2z45+2z46+2z47+2z48+z51+2z53+2z54+z5\
5+2z56+z57+z58+z59, 
  z+2z3+2z4+z5+z6+2z8+z10+z11+2z12+2z15+2z16+z19+z21+2z22+z25+2z29+2z30+2z31+2\
z32+z33+2z34+z38+2z39+z41+z43+2z48+2z49+z50+z53+2z54+2z55+2z56+2z57+z58+z59, 
  1+2z+z3+z4+2z5+z6+z8+2z10+2z11+z12+2z14+z15+z16+z17+z18+z19+2z20+z21+z22+z23\
+2z25+2z27+2z30+2z31+z32+z33+z34+2z36+2z37+2z39+2z40+2z41+z44+2z45+z47+z48+z49\
+2z53+2z54+2z55+z56+2z59, 
  1+2z+2z4+2z6+z7+2z8+z13+z16+z17+2z19+z20+z22+2z23+2z24+z25+z26+z27+z28+2z33+\
z35+2z38+z39+2z40+2z41+z42+2z44+2z45+z47+z49+2z50+z51+z52+2z54+2z55+2z56+z57+z\
58+z59, 
  1+z+2z2+2z3+2z4+z6+2z9+2z10+z11+z14+z15+2z16+2z17+2z20+2z21+2z22+2z26+2z27+2\
z28+2z29+2z31+z32+z33+z34+z36+z37+2z38+z41+z42+2z45+2z46+2z47+2z50+2z52+2z53+2\
z54+z56+z57+z58+z59, 
  1+z+2z2+z3+z4+2z6+2z7+2z9+z10+z11+2z12+z14+2z15+z16+z19+2z21+2z24+z26+2z28+2\
z29+2z30+z32+z33+z34+z35+2z37+2z40+z41+2z42+2z46+2z48+z49+2z50+2z53+z54+z55+z5\
6+z58+2z59, 3+z, 12+z2+9z3, 10+9z+13z3+11z4+5z5+10z6+15z7+5z8+13z9+6z10+9z11, 
  3+z, 3+z, 17+z, 3+z, 3+z, 2+z, 2+z, 2+z ]
gap> List(fieldpairs, pdd -> Z(pdd[1],pdd[2])-Z(pdd[1],pdd[3]));
[ 1+z, z+z4+z9+z10+z15+z17+z19+z21+z23+z25+z26+z27+z29+z30+z33, 
  z2+z3+z6+z10+z13+z15+z16+z18+z19+z20+z24+z25+z35+z38+z39+z40+z41+z44+z46+z48\
+z49, 
  1+z2+z3+z4+z5+z7+z10+z11+z12+z13+z14+z16+z17+z18+z19+z21+z22+z23+z24+z25+z26\
+z30+z31+z32+z35+z37+z38+z40+z47+z48+z50+z51+z56+z58+z62+z63+z67, 
  z+z3+z7+z8+z9+z10+z13+z14+z17+z19+z20+z21+z22+z24+z25+z27+z28+z30+z32+z33+z3\
6+z38+z41+z42+z45+z46+z47+z48+z50+z51+z52+z54+z58+z59+z60+z61+z63+z65+z68+z69+\
z71+z73+z75+z76+z77+z82, 
  z2+z3+z5+z7+z10+z11+z13+z16+z18+z19+z20+z21+z22+z25+z26+z27+z28+z30+z31+z32+\
z34+z35+z37+z38+z39+z40+z44+z45+z47+z54+z55+z56+z58+z60+z61+z63+z65+z67+z69+z7\
3+z74+z79+z83+z87+z89+z96+z97+z99+z100, <<an element of GF(2, 119)>>, 1+z, 
  1+z3+z4+z5+z6+z8+z9+z14+z17+z18+z19+z20+z21+z24+z29+z30, 
  z+z5+z7+z9+z12+z14+z15+z16+z17+z20+z22+z23+z24+z25+z28+z31+z33+z36+z37+z38+z\
39+z42+z43+z44+z52+z56+z61+z62+z65+z67+z68+z69+z70+z71+z72+z73+z76+z77+z79+z80\
+z81+z82+z83+z85+z86+z89+z94, 
  z4+z5+z6+z8+z10+z12+z14+z15+z17+z18+z19+z20+z21+z22+z23+z24+z25+z29+z30, 
  1+z6+z11+z12+z13+z15+z18+z22+z23+z30+z31+z36+z37+z38+z39+z41+z44+z45+z47+z48\
+z50+z51+z52+z55+z56+z57+z58+z59+z60+z63+z64+z66+z68+z69+z74+z76+z81+z83+z84+z\
85+z86+z88+z89+z90+z91+z92+z93, 
  z+z2+z3+z6+z7+z10+z12+z14+z17+z19+z20+z21+z22+z29+z31, 
  z2+z9+z10+z11+z15+z16+z17+z18+z20+z23+z25+z27+z29+z33+z35+z36+z39+z40+z42+z4\
5+z47+z48+z52+z54+z55+z56+z57+z59+z62+z67+z68+z71+z74+z75+z76+z80+z81+z83+z85+\
z86+z89+z90+z92+z93, 
  z+z2+z4+z5+z6+z7+z10+z11+z14+z15+z16+z20+z23+z26+z28+z29+z30+z31, 
  1+z+z4+z6+z7+z9+z12+z14+z17+z19+z21+z22+z28+z30+z33+z34+z36+z38+z40+z41+z45+\
z48+z49+z51+z52+z53+z59+z61+z63+z67+z69+z70+z75+z78+z81+z83+z85+z86+z87+z90+z9\
3+z94, 1+z, 
  z2+z4+z5+z7+z9+z11+z14+z17+z19+z20+z21+z23+z24+z25+z27+z30+z35+z36+z38+z39+z\
41+z43+z50+z51+z53+z54+z55+z57+z59, 
  1+z+z5+z7+z8+z11+z12+z15+z17+z22+z23+z24+z27+z28+z29+z31+z34+z35+z37+z38+z39\
+z42+z47+z48+z51+z52+z53, 
  1+z2+z3+z4+z7+z8+z9+z10+z11+z16+z18+z19+z20+z21+z23+z26+z27+z31+z32+z34+z36+\
z37+z39+z40+z41+z42+z45+z46+z47+z49+z50+z51+z53+z54+z56+z57+z58, 
  z+z2+z3+z6+z7+z9+z11+z13+z17+z19+z21+z22+z23+z25+z27+z29+z30+z31+z36+z38+z39\
+z41+z45+z46+z48+z51+z53+z54+z55+z56+z59, 
  z2+z4+z6+z7+z9+z11+z12+z14+z16+z17+z19+z20+z21+z22+z25+z26+z31+z34+z35+z36+z\
37+z38+z44+z46+z48+z49+z51+z52+z57+z58+z59, <<an element of GF(2, 120)>>, 
  z+z3+z4+z6+z8+z10+z11+z15+z18+z19+z22+z24+z26+z28+z30+z31+z32+z33+z34+z36+z3\
7+z39+z40+z41+z42+z47+z48+z50+z53+z54+z56+z57+z59, 
  z4+z6+z7+z13+z14+z16+z17+z18+z19+z20+z21+z22+z24+z26+z28+z31+z33+z34+z35+z37\
+z39+z43+z47+z48+z51+z52+z54+z58, 
  1+z4+z6+z8+z9+z14+z16+z24+z28+z29+z32+z34+z38+z39+z40+z42+z47+z48+z50+z51+z5\
2+z54+z56+z57+z59, 
  1+z+z2+z3+z4+z7+z11+z14+z15+z16+z17+z18+z22+z25+z26+z29+z34+z35+z36+z37+z38+\
z40+z44+z45+z46+z47+z48+z50+z52+z55+z56+z57+z58, <<an element of GF(2, 120)>>,
  z2+z4+z7+z10+z13+z15+z18+z20+z23+z25+z26+z38+z39+z41+z43+z44+z46+z48+z50+z51\
+z55+z56+z57+z59, <<an element of GF(2, 120)>>, 1+z, 
  z3+z6+z7+z8+z10+z14+z15+z18+z20+z22+z23+z25+z26+z27+z30+z31+z32+z33+z35+z36+\
z38+z42+z45+z46+z47+z50+z51+z54+z55+z57+z58+z60+z62+z63+z64+z65+z66+z67+z69+z7\
1+z73+z75, 
  1+z2+z3+z4+z7+z8+z9+z17+z18+z19+z21+z26+z29+z30+z31+z32+z34+z35+z36+z39+z43+\
z44+z45+z46+z48+z49+z52+z53+z56+z57+z60+z63+z64+z65+z69+z70+z71+z73+z74+z75, 
  1+z2+z5+z6+z7+z8+z10+z12+z13+z15+z16+z18+z19+z20+z26+z28+z33+z35+z37+z40+z41\
+z43+z44+z45+z46+z48+z49+z52+z53+z54+z55+z57+z60+z62+z65+z67+z72+z73+z74, 
  z2+z3+z5+z6+z8+z9+z11+z19+z23+z27+z31+z32+z34+z35+z36+z38+z42+z46+z48+z49+z5\
0+z53+z58+z59+z60+z62+z63+z65+z66+z67+z68+z69+z70+z74, 1+z, 
  1+z2+z3+z5+z9+z15+z22+z25+z26+z27+z28+z29+z30+z31+z32+z34+z36+z40+z42+z43+z4\
4+z48+z51+z52+z55+z56+z57+z60+z61+z62+z63+z64+z65+z67+z68+z69+z70+z71+z72+z75+\
z77+z79+z81+z83+z84+z86, 
  z+z2+z6+z8+z9+z10+z14+z15+z16+z19+z20+z22+z23+z26+z28+z30+z31+z33+z34+z37+z4\
1+z46+z50+z51+z54+z55+z56+z57+z58+z59+z60+z61+z63+z64+z66+z68+z71+z72+z76+z78+\
z81+z82+z83+z84+z85, 1+z, 
  2+2z+2z2+2z3+z4+2z5+z8+z9+2z10+z11+z12+z15+z16+2z18+2z19+2z20+z21, 
  1+2z+2z2+2z3+z4+z6+z8+2z11+z12+2z13+2z14+2z15+2z16+2z17+z18+z20+2z21+z22+z23\
+z24+2z25+z26+2z28+z30+2z31, 
  2z+z5+2z7+2z8+z9+z10+2z11+z12+z13+z14+z17+z18+2z19+2z20+2z23+2z26+2z28+2z31+\
2z33+2z35+z36+2z38+z39+z40+2z41+2z42+2z43, 
  1+z+z3+2z4+2z5+z6+2z9+z11+z12+z15+2z18+2z19+z20+2z21+z22+2z24+z25+2z27+z28+z\
29+2z34+z36+2z37+2z38+z40+2z41+2z43+2z45+z47+2z51+z52+2z54, 
  1+z+z3+z4+2z5+2z6+z9+z10+z11+z13+2z14+2z15+z16+2z17+2z18+z19+2z22+2z23+2z24+\
z26+z28+z30+z31+2z33+z34+z35+z37+2z38+z40+2z41+z42+z44+2z45+z46+2z47+z49+2z50+\
z51+2z52+z54+2z55+z56+z57+z58+z59+2z61+z62+z64, 
  2+z+z5+z6+2z9+2z11+2z13+z14+z16+2z20+z23+2z24+z27+z28+2z29+2z30+z31+2z32+2z3\
4+2z35+z36+2z37+z38+z39+2z40+2z41+z42+z45+2z46+z47+2z49+2z51+2z52+2z55+z56+2z5\
7+z58+z61+2z63+z64+2z65+2z67+2z69+2z70+z71+z72+z73+z75+2z76, 1+z, 
  z+2z2+z3+z4+2z7+z8+2z9+2z10+z12+z13+z14+2z15+z17+z18+2z19, 
  z4+z5+2z6+2z7+z9+z10+2z12+z14+z15+z17+2z18+2z19+2z20+z21+2z22+2z25+z26+z28+z\
30+z31+z32+z33+2z35+2z37+z38+z39+2z40+z41+2z42+z43+2z45+z47+z48+z49+2z50+z52+2\
z54+z55+z56+2z58+2z59, 
  2+2z2+z3+z4+z5+2z6+2z7+z11+z12+2z13+2z14+2z15+z16+2z17+2z18+2z19, 
  z2+2z4+2z5+2z6+z8+2z11+z12+2z16+2z19, 
  2+z+2z4+z6+z7+z8+2z9+2z10+2z11+z12+z13+2z14+z15+z18+2z19+2z21+z22+2z24+2z25+\
2z26+2z27+2z29+2z30+2z31+z32+z33+2z34+2z35+z36+z38+2z39+z40+2z41+z48+2z50+2z51\
+2z52+2z55+2z56, 
  2+z+2z7+2z8+2z9+2z10+2z11+2z12+z13+2z14+z15+2z16+2z17+z18+2z20+2z21+2z22+z24\
+2z25+2z26+2z28+z29+2z30+2z32+z33+z34+z36+2z38+z39, 
  z+2z3+z4+z7+2z8+z10+2z11+z12+2z13+2z14+2z15+2z17+2z18+z19, 
  2z+2z2+z3+z4+z5+2z8+2z9+2z11+2z12+2z14+z16+z17+2z18+2z19+z21+z22+2z23+z25+2z\
26+2z28+2z29+z30+2z36+2z37+2z38+z39+z40+2z41+z42+2z44+2z46+z47+2z48+2z49+2z50+\
2z52+z55+z57+z58+2z59, 
  2z+2z2+2z3+2z6+2z7+z8+2z9+2z10+z11+2z13+z14+z15+z16+z17+z19+z20+2z21+z22+z23\
+z24+2z25+z26+z27+z28+2z29+2z31+z32+2z33+z34+2z35+z36+z37+2z39+z40+2z41+z44+2z\
46+z47+2z49+2z51+z52+2z53+z55+2z56, 1+z, 
  1+z2+z4+z5+2z6+z7+2z8+z9+z11+2z13+z14+z15+2z16+z17+z18+2z19+z20+2z21+2z22+z2\
3+z25+z27+2z28+z29+z31+2z35+z36+2z37+z38+z39+2z40+z42+2z43+z45+z46+2z47+2z49+2\
z51+z52+z53+2z55+z58+2z59, 
  2+z+z2+z3+2z4+z5+z6+2z7+2z9+2z10+2z11+2z12+z16+2z17+2z18+2z19+2z21+2z25+2z26\
+z27+2z28+z29+z30+2z31+2z34+2z35+2z36+z37+2z38+z39+2z40+z42+z43+z46+2z47+z48+z\
49+2z52+z53+z55+2z57+z58+z59, 
  2+z+2z3+z5+z7+z8+z9+2z10+2z12+2z13+z14+z19+2z20+2z22+z23+2z26+z27+2z28+z29+2\
z30+2z32+z33+2z34+z35+2z37+z38+z40+2z41+z42+2z43+z44+z45+2z46+z47+2z48+2z49+z5\
1+2z53+z56+2z57+z58+2z59, 
  1+z3+2z4+z5+2z7+2z8+2z9+2z10+z11+z12+z13+2z14+2z18+z20+2z24+2z25+2z28+z29+z3\
0+z31+2z32+2z33+2z34+z35+2z36+z37+2z38+2z40+2z44+2z46+2z47+2z49+2z50+z52+z53+z\
57+z58+z59, 
  1+2z+z2+z3+z7+z8+z11+z12+z13+z14+z16+z17+z18+2z19+z20+2z22+2z24+2z25+z28+2z3\
0+z34+2z35+2z37+2z38+2z39+z40+z41+2z42+z45+z46+z47+z48+2z51+z53+z54+2z55+z56+2\
z57+2z58+2z59, 
  z+z3+z4+2z5+2z6+z8+2z10+2z11+z12+z15+z16+2z19+2z21+z22+2z25+z29+z30+z31+z32+\
2z33+z34+2z38+z39+2z41+2z43+z48+z49+2z50+2z53+z54+z55+z56+z57+2z58+2z59, 
  2+2z3+2z4+z5+2z6+2z8+z10+z11+2z12+z14+2z15+2z16+2z17+2z18+2z19+z20+2z21+2z22\
+2z23+z25+z27+z30+z31+2z32+2z33+2z34+z36+z37+z39+z40+z41+2z44+z45+2z47+2z48+2z\
49+z53+z54+z55+2z56+z59, 
  2+z4+z6+2z7+z8+2z13+2z16+2z17+z19+2z20+2z22+z23+z24+2z25+2z26+2z27+2z28+z33+\
2z35+z38+2z39+z40+z41+2z42+z44+z45+2z47+2z49+z50+2z51+2z52+z54+z55+z56+2z57+2z\
58+2z59, 
  2+z+z2+z3+z4+2z6+z9+z10+2z11+2z14+2z15+z16+z17+z20+z21+z22+z26+z27+z28+z29+z\
31+2z32+2z33+2z34+2z36+2z37+z38+2z41+2z42+z45+z46+z47+z50+z52+z53+z54+2z56+2z5\
7+2z58+2z59, 
  2+z+z2+2z3+2z4+z6+z7+z9+2z10+2z11+z12+2z14+z15+2z16+2z19+z21+z24+2z26+z28+z2\
9+z30+2z32+2z33+2z34+2z35+z37+z40+2z41+z42+z46+z48+2z49+z50+z53+2z54+2z55+2z56\
+2z58+z59, 14+z, 5+2z+16z2+8z3, 4z+13z2+8z3+4z4+8z6+16z7+9z8+14z9+16z10+11z11,
  254+z, 254+z, 65504+z, 65534+z, 268435396+z, 4294967289+z, 
  1152921504606846881+z, 18446744073709551627+z ]
gap> List(fieldpairs, pdd -> Z(pdd[1],pdd[2])*Z(pdd[1],pdd[3]));
[ z, 1+z2+z4+z9+z10+z13+z14+z15+z16+z18+z20+z21+z24+z25+z29+z30+z31+z33, 
  1+z+z3+z9+z10+z11+z14+z19+z20+z21+z22+z23+z25+z26+z33+z35+z36+z37+z41+z43+z4\
6+z47+z49, 
  z+z3+z4+z5+z7+z10+z14+z15+z17+z19+z26+z28+z31+z32+z37+z40+z41+z44+z45+z47+z4\
8+z51+z53+z55+z57+z60+z62+z64+z65+z66+z67, 
  1+z3+z5+z6+z7+z9+z10+z11+z12+z13+z14+z16+z18+z19+z22+z26+z28+z29+z31+z34+z42\
+z44+z47+z51+z55+z56+z58+z60+z64+z68+z71+z72+z73+z74+z75+z76+z77+z80+z81+z82+z\
84, 
  1+z+z4+z6+z8+z10+z11+z13+z15+z16+z17+z19+z20+z23+z26+z31+z37+z38+z39+z42+z45\
+z46+z50+z51+z52+z56+z57+z62+z63+z65+z66+z69+z71+z72+z73+z76+z81+z83+z84+z85+z\
87+z88+z91+z93+z98+z100+z101, 
  1+z2+z4+z6+z8+z11+z13+z16+z17+z20+z21+z23+z24+z27+z28+z29+z32+z35+z36+z38+z4\
0+z48+z49+z51+z52+z54+z55+z56+z58+z59+z61+z68+z69+z70+z72+z75+z77+z78+z79+z80+\
z81+z87+z91+z92+z96+z101+z102+z107+z109+z111+z112+z113+z115+z118, z, 
  z+z2+z4+z5+z6+z7+z9+z10+z15+z18+z19+z20+z21+z22+z25+z30+z31, 
  z2+z3+z8+z13+z17+z19+z22+z23+z24+z27+z29+z31+z33+z37+z39+z40+z42+z45+z48+z51\
+z52+z54+z55+z58+z59+z60+z63+z64+z68+z69+z70+z76+z78+z79+z81+z82+z83+z86+z87+z\
88+z89+z93+z94+z95, 
  z2+z5+z6+z7+z9+z11+z13+z15+z16+z18+z19+z20+z21+z22+z23+z24+z25+z26+z30+z31, 
  z+z3+z6+z8+z10+z16+z18+z19+z25+z26+z28+z32+z35+z36+z40+z41+z42+z43+z44+z45+z\
47+z48+z49+z50+z51+z52+z53+z54+z55+z56+z57+z58+z60+z65+z67+z69+z71+z72+z74+z77\
+z83+z84+z88+z89+z91+z93+z94+z95, 1+z8+z9+z11+z13+z18+z20+z21+z22+z23+z30, 
  z+z2+z3+z4+z5+z6+z7+z8+z10+z11+z13+z15+z18+z20+z21+z24+z26+z27+z28+z29+z30+z\
32+z37+z38+z39+z40+z43+z46+z47+z48+z49+z52+z53+z54+z57+z62+z66+z67+z68+z70+z74\
+z77+z79+z80+z82+z83+z86+z87+z89+z90+z91+z93+z94+z95, 
  1+z4+z5+z6+z8+z9+z11+z12+z16+z17+z21+z24+z27+z29+z30+z31, 
  1+z+z3+z5+z6+z10+z11+z13+z14+z17+z18+z19+z20+z23+z25+z27+z29+z33+z34+z41+z42\
+z43+z45+z46+z47+z50+z51+z53+z54+z55+z56+z57+z60+z61+z62+z64+z67+z68+z70+z73+z\
75+z77+z81+z82+z84+z88+z89+z94, z, 
  1+z4+z6+z10+z15+z17+z18+z19+z20+z21+z24+z28+z30+z31+z32+z33+z34+z37+z40+z41+\
z45+z51+z52+z54+z55+z56+z58, 
  z+z6+z8+z9+z12+z13+z16+z18+z23+z24+z25+z28+z29+z30+z32+z35+z36+z38+z39+z40+z\
43+z48+z49+z52+z53+z54, 
  z+z2+z3+z4+z5+z8+z9+z10+z11+z12+z17+z19+z20+z21+z22+z24+z27+z28+z32+z33+z35+\
z37+z38+z40+z41+z42+z43+z46+z47+z48+z50+z51+z52+z54+z55+z57+z58+z59, 
  1+z2+z5+z7+z10+z14+z17+z18+z19+z20+z23+z24+z25+z28+z31+z33+z34+z36+z37+z40+z\
41+z44+z45+z46+z47+z49+z52+z54+z55+z56+z57, 
  1+z4+z7+z10+z13+z15+z18+z19+z20+z21+z23+z25+z27+z30+z33+z34+z35+z37+z38+z41+\
z42+z44+z47+z49+z50+z52+z53+z58+z59, 
  1+z2+z3+z4+z5+z6+z7+z12+z16+z17+z19+z20+z22+z26+z27+z28+z29+z30+z33+z36+z37+\
z38+z40+z41+z45+z46+z47+z48+z52+z53+z54+z56+z57+z59+z62+z64+z69+z73+z75+z85+z8\
7+z89+z92+z95+z100+z101+z104+z105+z106+z108+z110+z113+z115+z118, 
  1+z2+z3+z7+z8+z9+z11+z16+z17+z20+z22+z23+z26+z27+z29+z30+z31+z35+z36+z37+z38\
+z39+z40+z43+z44+z45+z48+z49+z51+z54+z55+z57+z58, 
  z2+z5+z7+z8+z14+z15+z17+z18+z19+z20+z21+z22+z23+z25+z27+z29+z32+z34+z35+z36+\
z38+z40+z44+z48+z49+z52+z53+z55+z59, 
  1+z+z3+z4+z7+z8+z9+z10+z12+z15+z19+z22+z26+z29+z32+z34+z35+z36+z40+z42+z43+z\
44+z45+z48+z49+z51+z52+z53+z55+z57+z58, 
  z+z3+z4+z5+z8+z12+z15+z16+z17+z18+z19+z23+z26+z27+z30+z35+z36+z37+z38+z39+z4\
1+z45+z46+z47+z48+z49+z51+z53+z56+z57+z58+z59, <<an element of GF(2, 120)>>, 
  1+z4+z11+z12+z14+z16+z17+z21+z22+z24+z25+z27+z30+z32+z33+z34+z36+z40+z41+z47\
+z49+z51+z52+z56+z57+z58, <<an element of GF(2, 120)>>, z, 
  1+z+z4+z5+z7+z8+z9+z11+z14+z16+z20+z21+z25+z26+z28+z29+z32+z35+z38+z39+z43+z\
46+z47+z48+z51+z52+z55+z56+z58+z59+z61+z63+z64+z65+z66+z67+z68+z70+z72+z74, 
  1+z3+z4+z8+z9+z10+z14+z15+z18+z22+z23+z24+z25+z29+z30+z32+z34+z38+z40+z44+z4\
5+z46+z47+z49+z50+z53+z54+z57+z58+z61+z64+z65+z66+z70+z71+z72+z74+z75, 
  z+z2+z3+z6+z7+z8+z9+z11+z13+z14+z16+z17+z19+z20+z21+z27+z29+z34+z36+z38+z41+\
z42+z44+z45+z46+z47+z49+z50+z53+z54+z55+z56+z58+z61+z63+z66+z68+z73+z74+z75, 
  z2+z3+z4+z6+z7+z9+z10+z12+z20+z24+z28+z32+z33+z35+z36+z37+z39+z43+z47+z49+z5\
0+z51+z54+z59+z60+z61+z63+z64+z66+z67+z68+z69+z70+z71+z75, z, 
  1+z2+z4+z5+z6+z7+z8+z11+z12+z13+z14+z20+z23+z27+z29+z31+z32+z33+z35+z37+z41+\
z43+z44+z45+z49+z52+z53+z56+z57+z58+z61+z62+z63+z64+z65+z66+z68+z69+z70+z71+z7\
2+z73+z76+z78+z80+z82+z84+z85, 
  z3+z7+z9+z10+z11+z15+z16+z17+z20+z21+z23+z24+z27+z29+z31+z32+z34+z35+z38+z42\
+z47+z51+z52+z55+z56+z57+z58+z59+z60+z61+z62+z64+z65+z67+z69+z72+z73+z77+z79+z\
82+z83+z84+z85+z86, 2z, 1+z+2z2+2z3+z4+z5+z9+2z12+2z14+z17+z18, 
  1+2z+2z2+z3+2z4+z6+z7+2z8+2z9+2z10+z11+2z13+2z14+2z16+z18+z20+z21+2z22+2z24+\
2z25+z26+z27+2z31, 
  2z+2z2+z3+z6+2z7+z8+2z11+2z12+2z13+z16+2z17+z18+2z22+2z24+2z26+z27+z28+z29+2\
z30+2z31+2z32+z33+z37+z38+2z41, 
  2z2+2z4+z5+z6+2z9+2z11+z12+z13+z16+z17+z18+2z19+z21+z23+z24+z27+z28+2z31+z32\
+2z33+2z34+2z36+2z39+2z41+2z44+z45+2z46+2z47+2z48+2z51+2z52+2z53+2z54, 
  1+z+z2+2z3+z5+z6+2z8+z9+2z13+2z16+2z17+2z18+2z19+z20+z21+z22+2z23+z24+2z25+z\
26+2z27+z28+z33+2z34+2z35+z36+2z38+2z39+z40+2z41+z42+z44+z45+z47+z48+z49+z50+2\
z51+2z52+z53+2z54+2z55+z56+2z57+2z59+2z60+2z61+2z62+2z63, 
  1+2z+z5+2z8+2z9+z12+z13+2z16+2z17+z19+z20+z24+z25+z27+z29+z30+z31+z32+2z33+2\
z34+z35+2z38+z39+2z42+z44+z45+2z46+z48+z49+z50+z52+2z54+2z55+2z57+z58+z59+2z62\
+z64+2z66+z68+z69+z71+2z72+z73+z75, 2z, 1+2z+2z3+z9+2z14+2z15+z16+2z18+2z19, 
  2+2z+2z2+z6+z7+2z10+z17+z19+2z20+2z21+z24+2z25+2z27+z28+z29+2z30+z31+2z32+2z\
35+z36+2z37+z38+2z40+2z41+2z42+z44+2z46+z47+z49+z51+z52+z54+z56+2z57+2z58+z59,
  1+z2+2z3+2z6+z7+2z9+2z10+2z11+2z12+z14+z15+z16+2z17+z18+z19, 
  1+2z+z2+z4+2z5+z6+z7+2z8+z9+2z10+2z11+z12+z17, 
  1+z+2z2+2z3+z5+2z6+z7+2z10+2z11+2z12+2z13+z14+z15+z16+2z17+2z18+z20+2z21+2z2\
3+2z24+2z25+z28+2z30+z31+2z32+z34+2z36+z37+z41+2z42+2z43+2z44+z45+z47+z48+2z49\
+z53+z54+2z55+z56+2z59, 
  1+z+2z2+z3+2z6+z7+z10+2z11+z13+2z14+z15+z16+z17+2z18+z19+z21+z22+z23+z24+2z2\
5+z28+2z29+2z31+2z33+z35+z36+z38+2z39, 
  2+z+2z3+z5+2z9+z10+z12+z13+z14+z15+z16+z18+z19, 
  2+z+2z4+2z6+2z7+z8+2z10+z11+z12+z13+z16+z18+2z19+z20+2z22+z23+z25+2z26+2z28+\
z29+z30+2z31+2z32+2z33+z37+z38+z40+z41+z42+2z46+2z47+2z48+z50+z51+2z52+2z53+2z\
54+2z55+z56+z57, 
  2+z+2z2+2z4+2z6+2z7+2z9+2z11+2z12+2z13+z14+z15+2z16+2z17+2z18+2z20+2z21+z22+\
z23+2z24+2z25+z28+z29+z30+2z31+z32+2z33+2z34+2z35+z36+2z37+2z39+2z40+z41+2z42+\
2z43+2z44+2z45+z46+z47+2z48+z49+z50+z51+z52+z53+2z54+z57+2z58, 2z, 
  1+2z+2z2+z4+z5+z7+z9+z10+z11+2z12+z17+2z18+2z19+z21+2z22+z23+z27+z29+z30+z32\
+z33+2z34+2z35+2z36+2z37+2z38+z40+2z41+2z43+2z46+2z47+z48+z50+z52+2z53+2z54+z5\
6+2z59, 
  2+z+2z2+z3+z4+2z5+z6+2z7+2z10+z12+z13+z14+2z15+2z16+2z17+z18+z19+2z20+z21+2z\
24+z28+z29+2z31+2z32+2z33+z34+2z35+z37+z38+2z43+2z47+z48+2z49+2z50+z53+2z54+2z\
56+z58+2z59, 
  1+z+z2+z3+2z4+2z5+2z9+z10+2z11+z13+z16+z20+z22+z23+z26+2z27+z29+z30+z31+2z32\
+2z33+z34+2z38+2z40+z42+2z43+2z45+2z46+z47+2z48+z49+z50+2z52+z54+2z57+z58+2z59
    , 
  2+2z+2z3+z4+2z5+z6+z9+2z10+2z12+2z13+2z16+z19+z20+2z22+2z24+z25+2z27+2z28+z2\
9+2z31+2z34+2z35+z36+z37+z38+z40+z44+z45+z47+z48+z50+z51+2z53+2z54+2z58+2z59, 
  1+2z+2z5+z6+2z9+2z10+z11+2z12+2z13+z14+z16+2z17+2z18+2z19+z21+z22+z23+z24+z2\
5+2z26+z27+z28+2z29+2z30+z31+2z32+z33+2z34+z35+2z36+2z38+2z39+2z42+z43+2z44+2z\
46+2z47+2z48+2z49+z52+2z54+2z55+z56+2z57+z58+z59, 
  1+z2+z3+z5+2z6+z7+z8+2z9+2z10+2z11+z12+2z13+2z14+z15+2z17+2z21+2z22+2z23+z24\
+2z26+z27+z28+z30+2z31+z32+z35+z36+z38+2z39+z40+z41+z42+2z49+2z50+z51+z54+2z55\
+2z56+2z57+2z58+z59, 
  2+z+2z3+2z5+z6+z7+2z8+z9+z10+z11+2z12+z13+z14+z15+z17+z18+z19+2z20+z23+z26+2\
z27+z28+z30+2z31+2z34+2z35+2z36+2z37+z38+2z39+z41+2z42+z44+z45+2z46+z48+z49+z5\
0+2z54+2z55+2z56+z57, 
  1+z+2z2+z3+z4+z5+z6+2z7+2z8+2z9+2z10+z11+z15+z16+z17+z18+z20+z22+z23+2z25+2z\
26+2z27+2z28+z29+2z30+2z32+z33+z34+2z35+2z36+z38+2z42+z43+2z44+2z45+2z46+z48+z\
50+2z51+z52+z53+2z55+2z56+2z57+z58+z59, 
  1+z+z2+z5+z6+z7+z8+z10+z12+2z14+2z15+2z16+2z17+2z18+2z20+z21+2z23+z24+z26+2z\
29+z30+z32+2z33+z36+z37+2z38+2z40+z41+z42+z43+2z44+2z46+2z47+2z48+2z51+2z53+2z\
54+2z55+z57+z58+z59, 
  2+z+2z2+z3+2z5+2z6+2z7+z8+z12+2z13+z14+z16+z17+2z20+z21+z22+2z24+2z25+2z26+2\
z28+2z29+2z31+z32+2z34+2z35+z38+2z39+z40+z41+z42+2z43+z44+2z47+2z49+z50+2z51+2\
z54+z55+z56+z57+z59, 3z, 7+7z+4z2+z3, 
  5+12z+z2+11z3+3z5+8z6+z7+11z8+10z9+16z10+7z11, 3z, 3z, 17z, 3z, 3z, 2z, 2z, 
  2z ]
gap> List(fieldpairs, pdd -> Z(pdd[1],pdd[2])/Z(pdd[1],pdd[3]));
[ z, z+z4+z7+z8+z9+z10+z13+z14+z17+z18+z19+z23+z24+z25+z26+z27+z28+z29+z32, 
  1+z+z9+z10+z12+z13+z20+z21+z26+z28+z30+z31+z32+z33+z34+z37+z39+z46+z48+z49+z\
50, 
  z+z5+z6+z10+z11+z12+z15+z16+z20+z24+z25+z28+z34+z36+z37+z38+z40+z43+z44+z45+\
z46+z47+z48+z49+z53+z55+z57+z58+z59+z63+z64, 
  z+z2+z5+z8+z9+z13+z14+z15+z16+z20+z21+z22+z24+z26+z27+z28+z29+z30+z31+z32+z3\
3+z38+z41+z42+z44+z46+z48+z49+z50+z53+z55+z59+z62+z63+z65+z66+z68+z70+z72+z75+\
z77+z78+z79+z80+z82+z83, 
  1+z2+z7+z8+z9+z10+z11+z12+z13+z16+z17+z19+z20+z22+z27+z28+z29+z30+z34+z36+z3\
7+z39+z40+z41+z45+z47+z51+z52+z54+z56+z58+z59+z60+z63+z65+z67+z68+z69+z72+z74+\
z81+z84+z88+z89+z90+z93+z95+z97+z101, 
  1+z2+z3+z4+z8+z10+z11+z12+z14+z15+z16+z21+z22+z24+z32+z33+z34+z36+z39+z40+z4\
4+z50+z51+z52+z54+z62+z63+z65+z68+z70+z71+z74+z77+z80+z88+z89+z92+z94+z101+z10\
2+z104+z107+z108+z109+z110+z111+z118, z, 
  z2+z4+z5+z6+z7+z9+z10+z15+z18+z19+z20+z21+z22+z25+z30+z31, 
  1+z3+z8+z9+z10+z12+z13+z16+z17+z18+z20+z23+z24+z28+z29+z31+z34+z36+z38+z40+z\
42+z43+z46+z47+z48+z51+z55+z56+z57+z59+z63+z64+z65+z66+z69+z71+z72+z73+z77+z79\
+z81+z82+z83+z85+z86+z87+z88+z90+z91+z92+z93+z94+z95, 
  z+z2+z4+z6+z8+z9+z12+z16+z17+z18+z20+z21+z23+z24+z26+z28+z29+z30, 
  1+z3+z6+z7+z8+z10+z11+z13+z15+z16+z17+z18+z19+z21+z22+z29+z31+z32+z36+z37+z3\
8+z39+z40+z41+z42+z45+z46+z47+z48+z49+z50+z51+z53+z55+z58+z61+z62+z65+z67+z68+\
z72+z73+z76+z78+z81+z84+z87+z88+z92+z93+z94+z95, 
  z2+z7+z8+z11+z14+z16+z17+z19+z21+z24+z25+z27+z28+z30+z31, 
  1+z2+z3+z5+z9+z11+z13+z15+z16+z17+z18+z19+z20+z23+z26+z27+z33+z39+z40+z41+z4\
5+z46+z48+z49+z50+z51+z53+z55+z56+z59+z60+z61+z62+z63+z64+z65+z67+z68+z69+z70+\
z73+z77+z82+z83+z84+z88+z89+z90+z91+z93+z94+z95, 
  1+z3+z4+z5+z7+z10+z11+z13+z14+z15+z16+z18+z19+z20+z22+z23+z24+z26+z29+z30+z3\
1, 
  z+z2+z7+z10+z11+z13+z14+z18+z20+z21+z24+z25+z27+z30+z33+z34+z37+z38+z39+z41+\
z43+z44+z46+z48+z50+z51+z52+z55+z56+z58+z59+z63+z65+z66+z67+z68+z69+z71+z76+z7\
8+z80+z81+z83+z88+z89+z90+z91+z92+z94, z, 
  1+z+z4+z6+z10+z15+z17+z18+z19+z20+z21+z24+z28+z30+z31+z32+z33+z34+z37+z40+z4\
1+z45+z51+z52+z54+z55+z56+z58, 
  z+z2+z5+z6+z7+z8+z10+z11+z12+z13+z15+z17+z19+z20+z22+z23+z24+z25+z26+z27+z29\
+z32+z34+z38+z39+z41+z43+z50+z52+z56+z57, 
  1+z2+z3+z5+z6+z7+z8+z11+z14+z16+z19+z20+z22+z23+z26+z27+z30+z31+z32+z36+z42+\
z43+z44+z45+z47+z48+z50+z52+z58+z59, 
  1+z4+z6+z7+z8+z9+z10+z13+z17+z20+z21+z25+z26+z35+z37+z38+z44+z45+z47+z48+z56\
+z57+z58+z59, 
  1+z2+z4+z5+z6+z7+z9+z11+z16+z21+z22+z24+z26+z27+z31+z32+z33+z35+z36+z37+z45+\
z48+z49+z50+z51+z52+z53+z55+z57+z58, 
  1+z3+z5+z8+z9+z11+z13+z16+z17+z20+z21+z22+z26+z27+z28+z29+z32+z33+z35+z42+z4\
6+z47+z49+z50+z51+z52+z55+z60+z61+z62+z63+z66+z67+z69+z70+z72+z73+z74+z78+z80+\
z81+z82+z83+z87+z88+z92+z93+z96+z99+z100+z102+z103+z105+z106+z107+z108+z115+z1\
16, 
  z+z2+z3+z4+z6+z8+z9+z10+z11+z12+z13+z14+z15+z17+z20+z24+z25+z28+z30+z37+z40+\
z41+z42+z45+z46+z48+z50+z51+z54+z57+z58, 
  z2+z3+z5+z6+z7+z10+z11+z13+z15+z16+z21+z22+z26+z29+z30+z32+z33+z35+z36+z37+z\
38+z43+z45+z46+z47+z49+z50+z51+z53+z54+z56+z58, 
  z2+z3+z4+z6+z9+z12+z13+z14+z16+z24+z25+z26+z30+z31+z32+z34+z37+z40+z42+z43+z\
44+z45+z46+z49+z50, 
  z3+z8+z9+z13+z14+z17+z19+z22+z23+z24+z25+z28+z29+z32+z33+z36+z38+z40+z41+z42\
+z43+z44+z46+z47+z51+z53+z54+z57, 
  z2+z3+z4+z7+z12+z17+z19+z20+z21+z22+z23+z28+z29+z30+z31+z32+z34+z36+z37+z38+\
z41+z50+z51+z57+z58+z60+z61+z63+z66+z67+z68+z69+z73+z74+z75+z76+z79+z81+z82+z8\
3+z85+z87+z89+z93+z94+z97+z98+z99+z100+z101+z102+z104+z106+z107+z110+z112+z116\
+z117, 
  z2+z4+z5+z6+z8+z10+z11+z12+z16+z17+z18+z20+z21+z23+z24+z25+z27+z30+z34+z36+z\
38+z39+z41+z43+z44+z45+z48+z49+z51+z53+z54+z56+z59, 
  1+z+z2+z6+z11+z13+z17+z18+z22+z23+z24+z26+z31+z32+z33+z35+z40+z44+z45+z47+z4\
8+z49+z51+z55+z57+z59+z62+z66+z67+z69+z71+z74+z76+z77+z79+z81+z82+z83+z88+z89+\
z90+z93+z94+z96+z97+z98+z99+z102+z104+z105+z106+z108+z111+z112+z114+z118+z119,
  z, 
  1+z4+z5+z7+z8+z9+z11+z14+z16+z20+z21+z25+z26+z28+z29+z32+z35+z38+z39+z43+z46\
+z47+z48+z51+z52+z55+z56+z58+z59+z61+z63+z64+z65+z66+z67+z68+z70+z72+z74, 
  z2+z4+z7+z8+z9+z11+z12+z13+z15+z16+z17+z19+z22+z23+z25+z26+z31+z34+z35+z37+z\
38+z41+z42+z44+z45+z47+z50+z53+z54+z55+z56+z57+z58+z61+z62+z64+z67+z68+z69+z70\
+z71+z72+z73+z75, 
  1+z2+z3+z8+z9+z10+z11+z13+z17+z19+z21+z25+z26+z27+z28+z29+z31+z32+z35+z36+z4\
1+z43+z47+z51+z52+z59+z60+z61+z63+z64+z67+z69+z75, 
  z2+z3+z8+z9+z11+z14+z15+z16+z20+z21+z22+z24+z27+z28+z30+z33+z34+z39+z40+z41+\
z44+z46+z48+z50+z51+z53+z58+z61+z62+z63+z64+z65+z66+z67+z68+z69+z70+z71+z72+z7\
3+z74+z75, z, 
  1+z2+z4+z5+z6+z8+z11+z13+z15+z16+z18+z20+z24+z25+z27+z29+z30+z31+z33+z37+z40\
+z42+z44+z45+z48+z49+z50+z53+z54+z55+z58+z59+z62+z63+z65+z67+z68+z69+z70+z71+z\
72+z73+z74+z75+z77+z79+z80+z84+z86, 
  z2+z4+z7+z9+z12+z13+z18+z19+z21+z22+z24+z25+z28+z29+z30+z33+z34+z36+z39+z43+\
z45+z46+z50+z51+z52+z55+z57+z58+z60+z61+z62+z65+z66+z69+z71+z73+z74+z75+z77+z7\
8+z79+z80+z81+z82, 2z, 
  2+z+2z2+2z5+2z6+z7+z8+z9+2z10+z12+z14+z15+2z16+2z17+2z18+z19+z20, 
  2+2z5+2z6+2z7+2z8+2z9+2z11+2z13+2z14+2z16+2z19+z21+z22+z23+2z24+z25+2z26+z29\
+z30+2z31+2z32, 
  1+z+z2+2z3+z6+z7+z8+z9+2z10+2z11+z12+2z14+z15+z26+z27+2z29+z30+z31+2z33+2z34\
+z35+z36+z41+2z42, 
  2+2z+z2+2z3+z4+z5+z7+z8+z11+2z13+2z14+2z20+2z21+2z22+2z26+z28+z29+z30+z31+2z\
32+z33+2z34+2z35+2z38+z39+z40+2z41+2z42+2z43+z44+z46+z48+z49+2z52+2z53+2z54, 
  1+z+z2+z4+2z5+2z7+z8+z9+z10+2z11+z13+z14+2z16+2z18+2z19+z20+2z22+2z25+z26+2z\
28+z29+z30+2z31+2z32+z33+2z36+z38+2z39+2z42+2z43+z44+2z45+2z46+2z48+2z49+2z50+\
2z54+2z55+z57+z58+z60+2z65, 
  2+z+z3+2z4+2z5+2z6+2z7+z8+2z10+z11+z14+2z15+2z18+z19+z24+2z25+z26+z27+z30+z3\
1+2z32+2z33+2z35+z36+z38+z39+2z40+z41+2z43+2z45+z46+2z48+2z50+z52+2z53+z55+z59\
+2z60+z61+z63+z65+2z66+z67+2z68+z69+z70+z73+z75+z76, 2z, 
  1+z+2z3+z9+2z14+2z15+z16+2z18+2z19, 
  1+2z2+2z3+2z4+z5+2z6+2z8+2z9+z10+z13+2z16+2z17+z18+z19+2z20+z22+2z25+z27+2z2\
8+z29+2z31+z32+z33+z35+z37+2z40+z41+2z48+2z50+z52+z53+z54+2z55+2z57+z58, 
  2z+z3+z4+z5+z8+2z9+2z10+z11+2z12+2z14+2z16+2z18+2z19, 
  2+z3+2z4+2z6+2z7+2z10+z11+2z13+2z14+2z17+2z18+z19, 
  2+z3+2z4+2z6+2z8+z9+z10+2z12+z13+2z14+2z16+z19+z21+z22+2z23+z24+2z25+2z26+2z\
28+2z29+z30+z32+2z33+2z34+2z35+z37+z39+z41+2z42+z44+2z45+z46+2z47+z48+z49+z51+\
z52+2z53+2z54+2z55+z57+2z58+z59, 
  2+2z2+z3+2z4+z6+z8+z10+z11+2z13+z15+2z16+z17+z19+z20+2z22+z23+2z24+z25+2z26+\
2z27+z28+2z29+z30+z31+z32+2z33+z35+z36+2z37+2z39, 
  z+z2+2z3+2z5+2z7+2z8+2z9+z10+2z11+2z12+z13+z14+z15+z18, 
  z+2z3+2z4+z5+2z6+2z7+z8+2z12+2z15+z16+2z17+z18+2z19+2z20+z21+2z24+2z25+2z26+\
2z28+2z29+2z31+z33+z34+z35+z37+z38+z40+2z41+z42+2z43+z45+z46+z47+2z48+2z51+2z5\
3+z54+2z56+2z57+2z58+2z59, 
  1+z2+z3+2z5+z6+2z8+z11+2z12+z14+2z16+2z19+2z20+2z21+z22+z23+z25+z26+2z28+z29\
+z30+2z32+2z33+z34+z35+z36+2z38+2z39+z40+2z41+2z45+z47+z55+2z56+2z57, 2z, 
  1+z+2z2+z4+z5+z7+z9+z10+z11+2z12+z17+2z18+2z19+z21+2z22+z23+z27+z29+z30+z32+\
z33+2z34+2z35+2z36+2z37+2z38+z40+2z41+2z43+2z46+2z47+z48+z50+z52+2z53+2z54+z56\
+2z59, 
  2+2z+2z2+z4+z5+2z6+z8+2z12+2z13+2z15+z16+z17+2z18+2z19+2z20+z22+z23+z24+2z25\
+2z26+z27+2z29+2z30+2z32+z33+z34+2z35+z36+2z37+2z38+z39+2z40+2z41+z42+z43+z44+\
z45+z46+z50+z51+z54+z55+2z56+2z58+2z59, 
  2+z3+2z4+2z5+z9+2z10+z11+2z12+2z13+z14+2z17+z19+2z21+z22+z24+z25+2z26+z27+z2\
8+z29+z30+2z31+z32+z34+2z35+2z36+2z37+z39+z41+z42+2z43+z44+z45+z46+z48+z49+z50\
+z51+z52+2z53+2z54+2z55+z56+z59, 
  1+2z2+2z5+z6+z7+2z10+z11+2z12+2z13+2z14+2z16+z18+z19+z20+2z22+2z23+2z24+2z25\
+z26+2z27+2z28+2z30+z31+z32+2z33+2z34+2z35+2z36+2z37+2z38+z40+z41+2z43+z44+2z4\
5+2z46+z47+z48+z50+2z52+2z53+z55+2z58, 
  2z2+z3+z4+z6+z7+z11+z14+z16+2z17+2z18+z19+z20+z21+z22+z23+z26+2z27+2z28+z29+\
z30+2z31+2z32+2z34+2z35+2z36+z38+z39+z40+2z42+z45+2z46+2z49+2z51+z52+z53+2z54+\
2z55+2z56+2z57+z58+2z59, 
  2+z+2z5+z6+z7+z8+z9+z14+z15+z16+2z17+2z19+z20+2z21+z23+z24+2z25+2z26+2z28+2z\
30+2z32+z33+2z34+z36+2z37+z38+z40+z41+z42+2z43+z44+2z48+z50+2z52+2z53+2z54+z55\
+z56+2z58, 
  2+2z+z2+2z3+2z4+2z6+z9+2z11+z14+2z15+2z16+z17+2z18+2z19+2z20+2z21+z23+z24+z2\
5+z26+2z27+z29+2z30+2z31+z32+2z34+2z36+z37+2z38+2z39+z40+2z41+z42+z43+z44+2z45\
+2z46+z47+2z48+2z49+z50+2z51+z52+z55+z56+z57+z59, 
  z3+2z5+2z6+2z7+z9+z10+2z11+z12+z15+2z17+z18+z19+2z20+z21+z22+z24+z25+z27+z28\
+z30+z32+2z33+z34+2z35+2z36+2z37+z38+2z40+2z41+2z45+z46+z47+2z48+z49+2z50+z53+\
2z54+2z55+z56+z57+z59, 
  2+z+2z4+2z5+z6+z7+z9+z10+2z11+z12+2z13+z15+z16+2z17+2z22+z25+2z26+z29+2z32+z\
33+z35+2z37+2z38+z40+2z41+z44+z45+2z46+2z48+2z49+2z50+z52+z56+2z58+z59, 
  2z+2z2+2z4+z5+2z7+z8+2z9+2z10+2z12+z14+2z15+z16+z17+2z18+z19+z20+z21+2z22+z2\
5+2z26+z28+2z29+z30+z32+z33+2z34+z35+2z36+z39+z40+2z41+z42+z44+2z46+z47+2z48+z\
49+z51+2z52+z53+z54+z57+z58+2z59, 6z, 9+15z+10z2+11z3, 
  5+7z+14z2+13z3+10z4+10z5+4z6+13z7+5z8+12z9+3z10+7z11, 86z, 86z, 42396z, 
  21846z, 178956933z, 2147483646z, 576460752303423442z, 9223372036854775815z ]
gap> ForAll(fieldsizes, pd -> ForAll(DivisorsInt(pd[2]), d2 -> 
>     Z(pd[1],pd[2])^((pd[1]^pd[2]-1)/(pd[1]^d2-1)) = Z(pd[1],d2) ));
true

#
# Test comparisons across fields
#
gap> x := List(fieldpairs, pdd -> Z(pdd[1],pdd[2]) < Z(pdd[1],pdd[3]));
[ false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false ]
gap> x = List(fieldpairs, pdd -> not (Z(pdd[1],pdd[3]) < Z(pdd[1],pdd[2])));
true

# MinimalPolynomial
gap> for i in [1..Length(zs)] do
>  F := GF(Characteristic(zs[i]));
>  SetName(Indeterminate(F, 567), "t");
>  Print(i,": \c",MinimalPolynomial(F, zs[i], 567),"\n");
> od;
1: t^17+t^3+Z(2)^0
2: t^32+t^15+t^9+t^7+t^4+t^3+Z(2)^0
3: t^60+t^45+t^44+t^42+t^41+t^39+t^36+t^34+t^33+t^32+t^30+t^26+t^25+t^22+t^19+t^1\
7+t^12+t^8+t^5+t^4+t^3+t^2+Z(2)^0
4: t^76+t^38+t^37+t^36+t^35+t^34+t^33+t^31+t^29+t^27+t^25+t^24+t^23+t^20+t^19+t^1\
5+t^14+t^5+t^2+t+Z(2)^0
5: t^87+t^30+t^28+t^26+t^20+t^16+t^14+t^13+t^12+t^11+t^10+t^8+t^7+t^5+t^3+t+Z(2)^\
0
6: t^11-t^2+Z(3)^0
7: t^20-t^13+t^11+t^10+t^9+t^8-t^5-t^4-t^3+t-Z(3)^0
8: t^60+t^44-t^41+t^40-t^39-t^38-t^36+t^35+t^34-t^33+t^32+t^30-t^28-t^27-t^26-t^2\
4-t^22+t^21+t^20-t^16-t^15+t^14-t^11+t^10-t^8-t^6+t^5-t^4-t^3-t^2-Z(3)^0
9: t^4+Z(17)^11*t^2+Z(17)^3*t+Z(17)
10: t^2+Z(257)^177*t+Z(257)
11: t^11+Z(257)^199*t+Z(257)^129
12: t^2+Z(65521)^34212*t+Z(65521)
13: t^2-t+ZmodpZObj(3,65537)
14: t^2+ZmodpZObj(268435397,268435399)*t+ZmodpZObj(3,268435399)
15: t^2-t+ZmodpZObj(2,4294967291)
16: t^3+ZmodpZObj(2,1152921504606846883)*t+ZmodpZObj(1152921504606846881,115292150\
4606846883)
17: t^2+ZmodpZObj(18446744073709551625,18446744073709551629)*t+ZmodpZObj(2,1844674\
4073709551629)

#
gap> Display([[ zs[1], zs[1]^2], [0*zs[1], zs[1]^3]]);
z = Z( 2, 17); z2 = z^2, etc.
z  z2 
.  z3 

# AsInternalFFE
gap> AsInternalFFE(0*Z(2,10));
0*Z(2)
gap> AsInternalFFE(Z(2,10)^0);
Z(2)^0
gap> AsInternalFFE(Z(2,10));
Z(2^10)
gap> AsInternalFFE(0*Z(7,6));
0*Z(7)
gap> AsInternalFFE(Z(7,6)^0);
Z(7)^0
gap> AsInternalFFE(Z(7,6));
fail
gap> AsInternalFFE(0*Z(65537,2));
fail
gap> AsInternalFFE(Z(65537,2)^0);
fail
gap> AsInternalFFE(Z(65537,2));
fail

#
gap> SetInfoLevel(InfoPrimeInt,iPI);
gap> SetInfoLevel(InfoFactor,iF);
gap> if IsBound(InfoFactInt) then SetInfoLevel(InfoFactInt,iFI); fi;
gap> SetInfoLevel(InfoWarning,iW);
gap> STOP_TEST("ffeconway.tst");
