# ---------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# ---------------------------------------------------------

import os
import re
from io import open
from typing import Any, Match, cast

import pkg_resources
from setuptools import find_packages, setup

# Change the PACKAGE_NAME only to change folder and different name
PACKAGE_NAME = "azure-ai-ml"
PACKAGE_PPRINT_NAME = "Machine Learning"

# a-b-c => a/b/c
PACKAGE_FOLDER_PATH = PACKAGE_NAME.replace("-", "/")
# a-b-c => a.b.c
NAMESPACE_NAME = PACKAGE_NAME.replace("-", ".")

# Version extraction inspired from 'requests'
with open(os.path.join(PACKAGE_FOLDER_PATH, "_version.py"), "r") as fd:
    version = cast(Match[Any], re.search(r'^VERSION\s*=\s*[\'"]([^\'"]*)[\'"]', fd.read(), re.MULTILINE)).group(1)
if not version:
    raise RuntimeError("Cannot find version information")

with open("README.md", encoding="utf-8") as f:
    readme = f.read()
with open("CHANGELOG.md", encoding="utf-8") as f:
    changelog = f.read()

setup(
    name=PACKAGE_NAME,
    version=version,
    description="Microsoft Azure {} Client Library for Python".format(PACKAGE_PPRINT_NAME),
    long_description_content_type="text/markdown",
    long_description=readme + "\n\n" + changelog,
    license="MIT License",
    author="Microsoft Corporation",
    author_email="azuresdkengsysadmins@microsoft.com",
    url="https://github.com/Azure/azure-sdk-for-python",
    keywords="azure, azure sdk",
    classifiers=[
        "Development Status :: 5 - Production/Stable",
        "Programming Language :: Python",
        "Programming Language :: Python :: 3",
        "Programming Language :: Python :: 3 :: Only",
        "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10",
        "Programming Language :: Python :: 3.11",
        "Programming Language :: Python :: 3.12",
        "Programming Language :: Python :: 3.13",
        "Programming Language :: Python :: 3.14",
        "License :: OSI Approved :: MIT License",
    ],
    zip_safe=False,
    include_package_data=True,
    packages=find_packages(
        exclude=[
            "tests",
            # Exclude packages that will be covered by PEP420 or nspkg
            "azure",
            "azure.ai",
        ]
    ),
    python_requires=">=3.7",
    install_requires=[
        # NOTE: To avoid breaking changes in a major version bump, all dependencies should pin an upper bound if possible.
        "pyyaml>=5.1.0,<7.0.0",
        "azure-core>=1.23.0",
        "azure-mgmt-core>=1.3.0",
        "marshmallow>=3.5,<4.0.0",
        "jsonschema>=4.0.0,<5.0.0",
        "tqdm<5.0.0",
        # Used for PR 825138
        "strictyaml<2.0.0",
        # Used for PR 718512
        "colorama<1.0.0",
        "pyjwt<3.0.0",
        "azure-storage-blob>=12.10.0",
        "azure-storage-file-share",
        "azure-storage-file-datalake>=12.2.0",
        "pydash>=6.0.0,<9.0.0",
        "isodate<1.0.0",
        "azure-common>=1.1",
        "typing-extensions<5.0.0",
        "azure-monitor-opentelemetry",
    ],
    extras_require={
        # user can run `pip install azure-ai-ml[designer]` to install mldesigner along with this package
        # so user can submit @dsl.pipeline with @mldesigner.command_component inside it.
        "designer": [
            "mldesigner",
        ],
        # user can run `pip install azure-ai-ml[mount]` to install azureml-dataprep-rslex along with this package
        # so user can call data.mount() and datastore.mount() operations supported by it.
        "mount": [
            "azureml-dataprep-rslex>=2.22.0; python_version < '3.13'",
        ],
    },
    project_urls={
        "Bug Reports": "https://github.com/Azure/azure-sdk-for-python/issues",
        "Source": "https://github.com/Azure/azure-sdk-for-python",
    },
)
