% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/performance_roc.R
\name{performance_roc}
\alias{performance_roc}
\title{Simple ROC curve}
\usage{
performance_roc(x, ..., predictions, new_data)
}
\arguments{
\item{x}{A numeric vector, representing the outcome (0/1), or a model with
binomial outcome.}

\item{...}{One or more models with binomial outcome. In this case,
\code{new_data} is ignored.}

\item{predictions}{If \code{x} is numeric, a numeric vector of same length
as \code{x}, representing the actual predicted values.}

\item{new_data}{If \code{x} is a model, a data frame that is passed to
\code{predict()} as \code{newdata}-argument. If \code{NULL}, the ROC for
the full model is calculated.}
}
\value{
A data frame with three columns, the x/y-coordinate pairs for the ROC
curve (\code{Sensitivity} and \code{Specificity}), and a column with the
model name.
}
\description{
This function calculates a simple ROC curves of x/y coordinates based on
response and predictions of a binomial model.

It returns the area under the curve (AUC) as a percentage, which corresponds
to the probability that a randomly chosen observation of "condition 1" is
correctly classified by the model as having a higher probability of being
"condition 1" than a randomly chosen "condition 2" observation.

Applying \code{as.data.frame()} to the output returns a data frame containing the
following:
\itemize{
\item \code{Sensitivity} (that actually corresponds to \code{1 - Specificity}): It is the
False Positive Rate.
\item \code{Sensitivity}: It is the True Positive Rate, which is the proportion of
correctly classified "condition 1" observations.
}
}
\note{
There is also a \href{https://easystats.github.io/see/articles/performance.html}{\code{plot()}-method}
implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\examples{
library(bayestestR)
data(iris)

set.seed(123)
iris$y <- rbinom(nrow(iris), size = 1, .3)
folds <- sample(nrow(iris), size = nrow(iris) / 8, replace = FALSE)
test_data <- iris[folds, ]
train_data <- iris[-folds, ]

model <- glm(y ~ Sepal.Length + Sepal.Width, data = train_data, family = "binomial")
as.data.frame(performance_roc(model, new_data = test_data))
as.numeric(performance_roc(model))

roc <- performance_roc(model, new_data = test_data)
area_under_curve(roc$Specificity, roc$Sensitivity)

if (interactive()) {
  m1 <- glm(y ~ Sepal.Length + Sepal.Width, data = iris, family = "binomial")
  m2 <- glm(y ~ Sepal.Length + Petal.Width, data = iris, family = "binomial")
  m3 <- glm(y ~ Sepal.Length + Species, data = iris, family = "binomial")
  performance_roc(m1, m2, m3)

  # if you have `see` package installed, you can also plot comparison of
  # ROC curves for different models
  if (require("see")) plot(performance_roc(m1, m2, m3))
}
}
